/*
 This file is part of GNU Taler
 (C) 2020-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  AbsoluteTime,
  Duration,
  TransactionMajorState,
  TransactionMinorState,
  TransactionType,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import {
  createSimpleTestkudosEnvironmentV3,
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

const coinCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  feeDeposit: "TESTKUDOS:0",
  feeRefresh: "TESTKUDOS:0",
  feeRefund: "TESTKUDOS:0",
  feeWithdraw: "TESTKUDOS:0",
  rsaKeySize: 1024,
};

const coinConfigList: CoinConfig[] = [
  {
    ...coinCommon,
    name: "n1",
    value: "TESTKUDOS:1",
  },
];

const purse_expiration = AbsoluteTime.toProtocolTimestamp(
  AbsoluteTime.addDuration(
    AbsoluteTime.now(),
    Duration.fromSpec({ days: 2 }),
  ),
);

/**
 * Test peer push payments with a large number of coins.
 *
 * Since we use an artificallly large number of coins, this
 * test is a bit slower than other tests.
 */
export async function runPeerPushLargeTest(t: GlobalTestState) {
  // Set up test environment
  const [
    { walletClient: wallet1, bankClient, exchange },
    { walletClient: wallet2 },
  ] = await Promise.all([
    createSimpleTestkudosEnvironmentV3(t, coinConfigList, {
      additionalBankConfig(b) {
           
      },
    }),
    createWalletDaemonWithClient(t, {
      name: "w2"
    })
  ]);

  // Withdraw digital cash into the wallet.
  const withdrawRes = await withdrawViaBankV3(t, {
    walletClient: wallet1,
    bankClient,
    exchange,
    amount: "TESTKUDOS:300",
  });
  await withdrawRes.withdrawalFinishedCond;

  const check = await wallet1.call(
    WalletApiOperation.CheckPeerPushDebit,
    {
      amount: "TESTKUDOS:200",
    },
  );
  t.assertAmountEquals(check.amountEffective, "TESTKUDOS:200");

  const initiate = await wallet1.call(
    WalletApiOperation.InitiatePeerPushDebit,
    {
      partialContractTerms: {
        summary: "Hello World 🥺",
        amount: "TESTKUDOS:200",
        purse_expiration,
      },
    },
  );

  await wallet1.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: initiate.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.Ready
    },
  });

  const tx = await wallet1.call(
    WalletApiOperation.GetTransactionById,
    {
      transactionId: initiate.transactionId,
    },
  );
  t.assertDeepEqual(tx.type, TransactionType.PeerPushDebit);
  t.assertTrue(!!tx.talerUri);

  const prepare = await wallet2.call(
    WalletApiOperation.PreparePeerPushCredit,
    {
      talerUri: tx.talerUri,
    },
  );

  await wallet2.call(
    WalletApiOperation.ConfirmPeerPushCredit,
    {
      transactionId: prepare.transactionId,
    },
  );

  await Promise.all([
    wallet1.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: initiate.transactionId,
      txState: {
        major: TransactionMajorState.Done,
      },
    }),
    wallet2.call(WalletApiOperation.TestingWaitTransactionState, {
      transactionId: prepare.transactionId,
      txState: {
        major: TransactionMajorState.Done,
      },
    })
  ]);
}

runPeerPushLargeTest.suites = ["wallet", "slow"];
