/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  AbsoluteTime,
  Amounts,
  NotificationType,
  PreparePayResult,
  PreparePayResultType,
  TalerProtocolTimestamp,
  TransactionIdStr,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { useEffect } from "preact/hooks";
import { alertFromError, useAlertContext } from "../../context/alert.js";
import { useBackendContext } from "../../context/backend.js";
import { useAsyncAsHook } from "../../hooks/useAsyncAsHook.js";
import { Props, State } from "./index.js";

export function useComponentState({
  talerPayPullUri,
  onClose,
  goToWalletManualWithdraw,
  onSuccess,
}: Props): State {
  const api = useBackendContext();
  const { i18n } = useTranslationContext();
  const { pushAlertOnError } = useAlertContext();
  const hook = useAsyncAsHook(async () => {
    const p2p = await api.wallet.call(WalletApiOperation.PreparePeerPullDebit, {
      talerUri: talerPayPullUri,
    });
    const balance = await api.wallet.call(WalletApiOperation.GetBalances, {});
    return { p2p, balance };
  });

  useEffect(() =>
    api.listener.onUpdateNotification(
      [NotificationType.TransactionStateTransition],
      hook?.retry,
    ),
  );

  if (!hook) {
    return {
      status: "loading",
      error: undefined,
    };
  }
  if (hook.hasError) {
    return {
      status: "error",
      retry: {
        onClick: pushAlertOnError(async () => {
          hook.retry();
        }),
      },
      error: alertFromError(
        i18n,
        i18n.str`Could not load the transfer payment status`,
        hook,
      ),
    };
  }

  const { contractTerms, transactionId, amountEffective, amountRaw } =
    hook.response.p2p;

  const amountStr: string = contractTerms.amount;
  const amount = Amounts.parseOrThrow(amountStr);
  const effective = Amounts.parseOrThrow(amountEffective);
  // const raw = Amounts.parseOrThrow(amountRaw);
  // const summary: string | undefined = contractTerms.summary;

  const expiration: AbsoluteTime =
    contractTerms.purse_expiration ? AbsoluteTime.fromProtocolTimestamp(contractTerms.purse_expiration) : AbsoluteTime.never();

  const foundBalance = hook.response.balance.balances.find(
    (b) => Amounts.parseOrThrow(b.available).currency === amount.currency,
  );

  const baseResult = {
    shareUri: talerPayPullUri,
    transactionId,
    contractTerms,
    summary: contractTerms.summary,
    error: undefined,
    expiration,
    cancel: pushAlertOnError(onClose),
    goToWalletManualWithdraw,
    minimum_age: undefined,
  };

  if (!foundBalance) {
    return {
      status: "no-enough-balance",
      ...baseResult,
      effective: undefined,
      balanceDetails: undefined,
      amount,
      merchant: undefined,
      choices: undefined,
    };
  }

  async function accept(): Promise<void> {
    const resp = await api.wallet.call(
      WalletApiOperation.ConfirmPeerPullDebit,
      {
        transactionId,
      },
    );
    onSuccess(resp.transactionId);
  }

  return {
    status: "ready",
    ...baseResult,
    payHandler: {
      onClick: pushAlertOnError(accept),
    },
    effective,
    amount,
    merchant: undefined,
    choices: undefined,
  };
}
