/* Contains the basic I/O and setup routines.  */

/*  Copyright (C) 2001, 2002 Free Software Foundation, Inc.
 *
 *  Dap is a GNU program.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <sys/types.h>
#include <time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdlib.h>
#include "dap_make.h"
#include "externs.h"

/* dataobs structures contain the names and values of the variables.  
 * Also contain number of variables, etc.: see dap_make.h.  
 */

/* The values read from the current input line.  */
dataobs dap_obs[3];	/* normally use 0 for in and out; select by dap_ono */

/* The values read from the previous input line.  */
dataobs dap_prev[2];	/* only use 0 or 1 for input */

/* For saving values read from current input line.  */
static dataobs dosave;

static RFILE *rfile;		/* the array of ram files (see dap_make.h) */

/* A DFILE can be either a disk file or a ramfile: 0, 1, 2 are used for disk
 * files, > 2 for ramfiles.  
 */
static DFILE *dfile;

DFILE *dap_in[2];	/* input files */
DFILE *dap_out;		/* output file */
FILE *dap_lst;		/* .lst file */
FILE *dap_log;		/* .log file, for lines read, written */
FILE *dap_err;		/* .err file, for errors */
int dap_ono;		/* specifies which set of variable values (dataobs) to use */
int dap_delim;		/* delimiter for reading data files (not datasets) */
static int *fieldwd;	/* width of fixed width input data fields */
static int nfields;		/* number of fixed width input data fields */
static int lineno[2];		/* line number for up to 2 input datasets */
static int outline;		/* number of lines written */
static int pageno;		/* output page number */
static int eof[2];		/* flags: has EOF been reached on input file(s)? */
static long filepos[2];		/* file positions of input file(s) */

/* These are the types of input data files (not datasets) recognized.  */
#define TEXT 0		/* ordinary text, fields either fixed width or delimited */
#define DBASE 1		/* dBase */
#define DSET 2		/* dap dataset */

static int intype;	/* type of input file */
static int inlen;	/* line length for files with dBase files */
static int toolong;	/* flag: string data longer than specified encountered? */

char *dap_title = NULL;	/* title for .lst pages, caption for .ps pages */

char *dap_dapname;	/* name of program running */
char *dap_psname;	/* name of the .ps file */

extern int dap_dblhigh;	/* for coding doubles */
extern int dap_dbllow;	/* for coding doubles */

static int nmallocs = 0;
static int nfrees = 0;

char *dap_malloc(int nbytes, char *vars)
{
char *m;

nmallocs++;
if (!(m = malloc(nbytes)))
	{
	perror(dap_dapname);
	fputs(vars, stderr);
	putc('\n', stderr);
	exit(1);
	}
if (dap_memtrace)
	{
	fprintf(dap_log, "malloc %x\n", (unsigned int) m);
	fflush(dap_log);
	if (dap_mabort && m == dap_memtrace)
		abort();
	}
return m;
}

void dap_free(void *ptr)
{
nfrees++;
if (dap_memtrace)
	{
	fprintf(dap_log, "free %x\n", (unsigned int) ptr);
	fflush(dap_log);
	if (dap_fabort && ptr == dap_memtrace)
		abort();
	}
free(ptr);
}

/* void dap_param(); */

static void initdo(dataobs *dato)
{
int d;

dato->do_int = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
dato->do_il = (int **) dap_malloc(sizeof(int *) * dap_maxvar, "dap_maxvar");
dato->do_dbl = (double *) dap_malloc(sizeof(double) * dap_maxvar, "dap_maxvar");
dato->do_dl = (double **) dap_malloc(sizeof(double *) * dap_maxvar, "dap_maxvar");
dato->do_str = (char **) dap_malloc(sizeof(char *) * dap_maxvar, "dap_maxvar");
dato->do_sl = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
dato->do_nam = (char **) dap_malloc(sizeof(char *) * dap_maxvar, "dap_maxvar");
dato->do_len = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
dato->do_in = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
dato->do_out = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
for (d = 0; d < dap_maxvar; d++)
	{
	dato->do_str[d] = NULL;
	dato->do_nam[d] = NULL;
	}
}

int main(int argc, char **argv)
{
char *lstname;		/* name of .lst file */
char *logname;		/* name of .log file */
char *errname;		/* name of .err file */
int len;		/* length of lstname, logname, errname */
int v;			/* index to variables */
double testd;           /* to check endianness */

/* dap_param(); */
dap_dapname = argv[0];
len = strlen(argv[0]);
lstname = malloc(len + 5);	/* room for suffix */
strcpy(lstname, argv[0]);
if (len >= 4 && !strcmp(lstname + len - 4, ".dap"))
	strcpy(lstname + len - 3, "lst");
else
	{
	strcat(lstname, ".lst");
	len += 4;
	}
if (!(dap_lst = fopen(lstname, "a")))
	{
	perror(dap_dapname);
	exit(1);
	}
logname = malloc(len + 1);
strcpy(logname, lstname);
strcpy(logname + len - 3, "log");
if (!(dap_log = fopen(logname, "w")))
	{
	perror(dap_dapname);
	exit(1);
	}
errname = malloc(len + 1);
strcpy(errname, lstname);
strcpy(errname + len - 3, "err");
if (!(dap_err = fopen(errname, "w")))
	{
	perror(dap_dapname);
	exit(1);
	}
initdo(dap_obs);
initdo(dap_obs + 1);
initdo(dap_obs + 2);
initdo(dap_prev);
initdo(dap_prev + 1);
initdo(&dosave);
rfile = (RFILE *) dap_malloc(sizeof(RFILE) * dap_nrfiles, "dap_nrfiles");
dfile = (DFILE *) dap_malloc(sizeof(DFILE) * (dap_nrfiles + 3), "dap_nrfiles");
pageno = 1;
dap_psname = dap_malloc(len + 1, argv[0]);
strcpy(dap_psname, lstname);
strcpy(dap_psname + len - 3, "ps");
dap_initpict();
for (v = 0; v < dap_maxvar; v++)
	{
	for (dap_ono = 0; dap_ono < 3; dap_ono++)
		{
		dap_obs[dap_ono].do_nam[v] = NULL;
		dap_obs[dap_ono].do_dl[v] = NULL;
		dap_obs[dap_ono].do_il[v] = NULL;
		dap_obs[dap_ono].do_str[v] = NULL;
		dap_obs[dap_ono].do_sl[v] = 0;
		if (dap_ono < 2)
			dap_prev[dap_ono].do_str[v] = NULL;
		}
	dosave.do_str[v] = NULL;
	}
dap_ono = 0;
testd = -2.0;
if (!(*((int *) &testd)))
	{
	dap_dbllow = 0;
	dap_dblhigh = 1;
	}
else
	{
	dap_dbllow = 1;
	dap_dblhigh = 0;
	}
dap_main(argc, argv);
return 0;
}

/* Open a disk file dataset */
static FILE *dopen(char fname[], char mode[])
{
char *dname;
FILE *f;

dname = dap_malloc(strlen(fname) + strlen(dap_setdir) + 2, fname);
dap_name(dname, fname);
f = fopen(dname, mode);
dap_free(dname);
return f;
}

/* COMMENT */
static DFILE *dfopen(char *fname, char *mode)
{
static int rfileinit = 0;	/* flag: has the ramfile array been initialized? */
int f;
char truemode[2];

if (!rfileinit)
	{
	rfileinit = 1;
	for (f = 0; f < dap_nrfiles; f++)
		{
		dfile[3 + f].dfile_name = NULL;
		dfile[3 + f].dfile_ram = rfile + f;
		dfile[3 + f].dfile_disk = NULL;
		rfile[f].rfile_str = NULL;
		}
	for (f = 0; f < 3; f++)
		{
		dfile[f].dfile_disk = NULL;
		dfile[f].dfile_ram = NULL;
		}
	}
if (!fname)
	{
	fputs("(dfopen) no file name given\n", dap_err);
	exit(1);
	}
if (fname[0] == '<')
	{
	if (!strcmp(mode, "r"))
		{
		for (f = 0; f < dap_nrfiles; f++)
			{
			if (dfile[3 + f].dfile_name &&
					!strcmp(dfile[3 + f].dfile_name, fname + 1))
				{
				rfile[f].rfile_pos = rfile[f].rfile_str;
				return dfile + 3 + f;
				}
			}
		return NULL;
		}
	else if (!strcmp(mode, "w"))
		{
		for (f = 0; f < dap_nrfiles; f++)
			{
			if (dfile[3 + f].dfile_name &&
					!strcmp(dfile[3 + f].dfile_name, fname + 1))
				{
				rfile[f].rfile_end = rfile[f].rfile_str;
				rfile[f].rfile_pos = rfile[f].rfile_str;
				return dfile + 3 + f;
				}
			}
		for (f = 0; f < dap_nrfiles; f++)
			{
			if (!dfile[3 + f].dfile_name)
				{
				rfile[f].rfile_str = dap_malloc(dap_rfilesize,
									"dap_rfilesize");
				dfile[3 + f].dfile_name = dap_malloc(strlen(fname), fname);
				strcpy(dfile[3 + f].dfile_name, fname + 1);
				rfile[f].rfile_end = rfile[f].rfile_str;
				rfile[f].rfile_pos = rfile[f].rfile_str;
				return dfile + 3 + f;
				}
			}
		return NULL;
		}
	else if (!strcmp(mode, "a"))
		{
		for (f = 0; f < dap_nrfiles; f++)
			{
			if (!strcmp(dfile[3 + f].dfile_name, fname + 1))
				{
				rfile[f].rfile_pos = rfile[f].rfile_end;
				return dfile + 3 + f;
				}
			}
		return NULL;
		}
	else
		{
		fprintf(dap_err, "(dfopen) bad mode: %s\n", mode);
		exit(1);
		}
	}
for (f = 0; f < 3; f++)
	{
	if (!dfile[f].dfile_disk)
		{
		if (mode[1] == 'f')
			{
			truemode[0] = mode[0];
			truemode[1] = '\0';
			dfile[f].dfile_disk = fopen(fname, truemode);
			}
		else
			dfile[f].dfile_disk = dopen(fname, mode);
		if (dfile[f].dfile_disk)
			{
			dfile[f].dfile_name = dap_malloc(strlen(fname) + 1, fname);
			strcpy(dfile[f].dfile_name, fname);
			return dfile + f;
			}
		else
			return NULL;
		}
	}
return NULL;
}

static void dfclose(DFILE *fp)
{
if (fp->dfile_disk)
	{
	fclose(fp->dfile_disk);
	fp->dfile_disk = NULL;
	}
else
	fp->dfile_ram->rfile_pos = fp->dfile_ram->rfile_str;
}

static int dgetc(DFILE *fp)
{
int c;

if (fp->dfile_disk)
	c = getc(fp->dfile_disk);
else if (fp->dfile_ram->rfile_pos < fp->dfile_ram->rfile_end)
	c = *fp->dfile_ram->rfile_pos++;
else
	c = EOF;
return c;
}

static void undgetc(int c, DFILE *fp)
{
if (fp->dfile_disk)
	ungetc(c, fp->dfile_disk);
else if (fp->dfile_ram->rfile_pos > fp->dfile_ram->rfile_str)
	{
	--fp->dfile_ram->rfile_pos;
	*fp->dfile_ram->rfile_pos = c;
	}
else
	{
	fprintf(dap_err, "(undgetc) can't unget past beginning of file %s\n",
			fp->dfile_name);
	exit(1);
	}
}

void dap_putc(int c, DFILE *fp)
{
if (fp->dfile_disk)
	putc(c, fp->dfile_disk);
else if (fp->dfile_ram->rfile_pos < fp->dfile_ram->rfile_str + dap_rfilesize)
	{
	*fp->dfile_ram->rfile_pos++ = c;
	fp->dfile_ram->rfile_end++;
	}
else
	{
	fprintf(dap_err, "(dap_putc) too many characters: %s\n",
			fp->dfile_name);
	exit(1);
	}
}

static void dputs(char *s, char *suff, DFILE *fp)
{
if (fp->dfile_disk)
	{
	fputs(s, fp->dfile_disk);
	fputs(suff, fp->dfile_disk);
	}
else
	{
	while (*s)
		dap_putc(*s++, fp);
	while (*suff)
		dap_putc(*suff++, fp);
	}
}

static void dputi(int i, DFILE *fp)
{
static char *istr = NULL;
int s;

if (!istr)
	istr = dap_malloc(dap_intlen + 1, "dap_intlen");
if (fp->dfile_disk)
	fprintf(fp->dfile_disk, "%d", i);
else
	{
	sprintf(istr, "%d", i);
	for (s = 0; s < strlen(istr); s++)
		dap_putc(istr[s], fp);
	}
}

static void dflush(DFILE *fp)
{
fflush(fp->dfile_disk);
}

/* Append or replace suffix. If suff[0] == '<', then suff replaces
 * characters following last '.', otherwise is appended
 */
void dap_suffix(char dst[], char src[], char suff[])
{
int n, s;

for (n = 0; src[n]; n++)
	dst[n] = src[n];
if (suff[0] == '<')
	{
	while (--n >= 0)
		{
		if (dst[n] == '.')
			break;
		}
	if (n < 0)
		{
		fprintf(dap_err, "(dap_suffix) source name has no '.': %s\n", src);
		exit(1);
		}
	n++;
	s = 1;
	}
else
	s = 0;
for ( ; suff[s]; s++)
	dst[n++] = suff[s];
dst[n] = '\0';
}

int dap_varnum(char *vname)
{
int v;

if (vname)
	{
	for (v = 0; v < dap_obs[dap_ono].do_nvar; v++)
		{
		if (!strcmp(vname, dap_obs[dap_ono].do_nam[v]))
			return v;
		}
	}
return -1;
}

int dap_arrnum(char vname[], int *dim)
{
int v;
int n;
int d;

for (v = 0; v < dap_obs[dap_ono].do_nvar; v++)
	{
	for (n = 0; vname[n] && vname[n] == dap_obs[dap_ono].do_nam[v][n]; n++)
		;
	if (!vname[n] && dap_obs[dap_ono].do_nam[v][n] == '[')
		{
		for (d = 1; v + d < dap_obs[dap_ono].do_nvar; d++)
			{
			for (n = 0; vname[n] &&
					vname[n] == dap_obs[dap_ono].do_nam[v + d][n]; n++)
				;
			if (vname[n] || dap_obs[dap_ono].do_nam[v + d][n] != '[')
				break;
			}
		*dim = d;
		return v;
		}
	}
return -1;
}

static int getline(DFILE *fp, char *line)
{
int l;
int c;
int cc;

c = EOF;
switch (intype)
	{
case TEXT:
case DSET:
	for (l = 0; (c = dgetc(fp)) != EOF; )
		{
		if (c == '\n')
			break;
		else if (c == '\r')
			{
			if ((cc = dgetc(fp)) != '\n')
				undgetc(cc, fp);
			break;
			}
		if (l < dap_linelen)
			line[l++] = c;
		else
			{
			line[l] = '\0';
			fprintf(dap_err, "(getline) line too long:\n%s\n", line);
			exit(1);
			}
		}
	line[l] = '\0';
	break;
case DBASE:
	dgetc(fp);      /* dump space */
	for (l = 0; l < inlen; l++)
		{
		if ((c = dgetc(fp)) == EOF)
			break;
		else
			line[l] = c;
		}
	if (l < inlen)
		l = 0;
	line[l] = '\0';
	break;
default:
	fprintf(dap_err, "(getline) bad infile type: %d\n", intype);
	exit(1);
	}
if (!l && c == EOF)
	return -1;
return l;
}

void dap_swap()
{
int iv;		/* index of input variable */
double dbltmp;	/* temp for swapping doubles */
int inttmp;	/* temp for swapping int */
char strtmp;	/* temp char for swapping strings */
int s;		/* index to strings */
char *so, *sp;	/* pointers to strings for obs, prev */

for (iv = 0; iv < dap_obs[dap_ono].do_ivar; iv++)
	{
	switch (dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[iv]])
		{
	case DBL:
		dbltmp = dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]];
		dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]] =
			dap_prev[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]];
		dap_prev[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]] = dbltmp;
		if (dap_obs[dap_ono].do_dl[dap_obs[dap_ono].do_in[iv]])
			*dap_obs[dap_ono].do_dl[dap_obs[dap_ono].do_in[iv]] =
			dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]];
		break;
	case INT:
		inttmp = dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]];
		dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]] =
			dap_prev[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]];
		dap_prev[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]] = inttmp;
		if (dap_obs[dap_ono].do_il[dap_obs[dap_ono].do_in[iv]])
			*dap_obs[dap_ono].do_il[dap_obs[dap_ono].do_in[iv]] =
			dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]];
		break;
	default:
		so = dap_obs[dap_ono].do_str[dap_obs[dap_ono].do_in[iv]];
		sp = dap_prev[dap_ono].do_str[dap_obs[dap_ono].do_in[iv]];
		for (s = 0; s < dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[iv]]; s++)
			{
			strtmp = so[s];
			so[s] = sp[s];
			sp[s] = strtmp;
			}
		break;
		}
	}
}

void dap_save()
{
int iv;

for (iv = 0; iv < dap_obs[dap_ono].do_ivar; iv++)
	{
	switch (dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[iv]])
		{
	case DBL:
		dosave.do_dbl[dap_obs[dap_ono].do_in[iv]] =
			dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]];
		break;
	case INT:
		dosave.do_int[dap_obs[dap_ono].do_in[iv]] =
			dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]];
		break;
	default:
		if (dosave.do_str[dap_obs[dap_ono].do_in[iv]])
			dap_free(dosave.do_str[dap_obs[dap_ono].do_in[iv]]);
		dosave.do_str[dap_obs[dap_ono].do_in[iv]] =
				dap_malloc(dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[iv]] + 1,
						dap_obs[dap_ono].do_nam[dap_obs[dap_ono].do_in[iv]]);
		strncpy(dosave.do_str[dap_obs[dap_ono].do_in[iv]],
			dap_obs[dap_ono].do_str[dap_obs[dap_ono].do_in[iv]],
			dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[iv]] + 1);
		break;
		}
	}
}

void dap_rest()
{
int iv;

for (iv = 0; iv < dap_obs[dap_ono].do_ivar; iv++)
	{
	switch (dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[iv]])
		{
	case DBL:
		dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]] =
			dosave.do_dbl[dap_obs[dap_ono].do_in[iv]];
		if (dap_obs[dap_ono].do_dl[dap_obs[dap_ono].do_in[iv]])
			*dap_obs[dap_ono].do_dl[dap_obs[dap_ono].do_in[iv]] =
			dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]];
		break;
	case INT:
		dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]] =
			dosave.do_int[dap_obs[dap_ono].do_in[iv]];
		if (dap_obs[dap_ono].do_il[dap_obs[dap_ono].do_in[iv]])
			*dap_obs[dap_ono].do_il[dap_obs[dap_ono].do_in[iv]] =
			dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]];
		break;
	default:
		strncpy(dap_obs[dap_ono].do_str[dap_obs[dap_ono].do_in[iv]],
			dosave.do_str[dap_obs[dap_ono].do_in[iv]],
			dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[iv]] + 1);
		break;
		}
	}
}

long dap_ftell(DFILE *fp)
{
if (fp->dfile_disk)
	return ftell(fp->dfile_disk);
return fp->dfile_ram->rfile_pos - fp->dfile_ram->rfile_str;
}

void dap_mark()
{
filepos[dap_ono] = dap_ftell(dap_in[dap_ono]);
}

static void dfseek(DFILE *fp, long pos, int mode)
{
if (fp->dfile_disk)
	fseek(fp->dfile_disk, pos, mode);
else
	{
	if (fp->dfile_ram->rfile_str + pos < fp->dfile_ram->rfile_end)
		fp->dfile_ram->rfile_pos = fp->dfile_ram->rfile_str + pos;
	else
		{
		fprintf(dap_err, "(dfseek) seek past end of ramfile %s\n",
			fp->dfile_name);
		exit(1);
		}
	}
}

void dap_rewind()
{
if (dap_in[dap_ono])
	{
	if (filepos[dap_ono] < dap_ftell(dap_in[dap_ono]))
		eof[dap_ono] = 0;
	dfseek(dap_in[dap_ono], filepos[dap_ono], SEEK_SET);
	dap_obs[dap_ono].do_valid = 0;
	}
else
	{
	fprintf(dap_err, "(dap_rewind) file (%d) is closed.\n", dap_ono);
	exit(1);
	}
}

int dap_blank(char str[])
{
int b;

for (b = 0; str[b] == ' '; b++)
	;
if (str[b])
	return 0;
return 1;
}

void skip(int nlines)
{
char *line;

line = dap_malloc(dap_linelen + 1, "dap_linelen");
while (--nlines >= 0)
	{
	if (!dap_in[dap_ono] || eof[dap_ono])
		{
		fprintf(dap_err, "(skip) tried to read past end of file (%d).\n",
						dap_ono);
		exit(1);
		}
	if (getline(dap_in[dap_ono], line) < 0)
		eof[dap_ono] = 1;
	lineno[dap_ono]++;
	}
dap_free(line);
}

int step()
{
static int stepinit = 0;
static char *line = NULL;
static char *value = NULL;
int v;				/* index to variables */
int l;
int i;
int iv;
int nread;	/* number of chars read by getline */

if (!stepinit)
	{
	stepinit = 1;
	line = dap_malloc(dap_linelen + 1, "dap_linelen");
	value = dap_malloc(dap_linelen + 1, "dap_linelen");
	}
if (!dap_in[dap_ono] || eof[dap_ono])
	{
	fprintf(dap_err, "(step) ERROR: tried to read past end of file (%s).\n",
		(dap_in[dap_ono] ? dap_in[dap_ono]->dfile_name : "?"));
	exit(1);
	}
for (iv = 0; iv < dap_obs[dap_ono].do_ivar; iv++)
	{
	switch (dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[iv]])
		{
	case DBL:
		dap_prev[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]] =
				dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_in[iv]];
		break;
	case INT:
		dap_prev[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]] =
				dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_in[iv]];
		break;
	default:
		strncpy(dap_prev[dap_ono].do_str[dap_obs[dap_ono].do_in[iv]],
				dap_obs[dap_ono].do_str[dap_obs[dap_ono].do_in[iv]],
				dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[iv]] + 1);
		}
	}
dap_prev[dap_ono].do_valid = dap_obs[dap_ono].do_valid;
if ((nread = getline(dap_in[dap_ono], line)) <= 0)
	{
	eof[dap_ono] = 1;
	dap_obs[dap_ono].do_valid = 0;
	fprintf(stderr, "(step) %d lines read from %s\n",
		lineno[dap_ono] - 1, dap_in[dap_ono]->dfile_name);
	if (!nread)
		fputs("(step) WARNING: terminated on null line\n", stderr);
	fflush(stderr);
	fprintf(dap_log, "(step) %d lines read from  %s\n",
		lineno[dap_ono] - 1, dap_in[dap_ono]->dfile_name);
	if (!nread)
		fputs("(step) WARNING: terminated on null line\n", dap_log);
	return 0;
	}
if (nfields && dap_obs[dap_ono].do_ivar != nfields)
	{
	fprintf(dap_err,
"(step) ERROR: number of input variables %d different from number of fields specified %d for %s.\n",
		dap_obs[dap_ono].do_ivar, nfields, dap_in[dap_ono]->dfile_name);
	exit(1);
	}
for (v = 0, l = 0; v < dap_obs[dap_ono].do_ivar; v++)
	{
	if (nfields)
		{
		for (i = 0; line[l + i] && i < fieldwd[v]; i++)
			value[i] = line[l + i];
		value[i] = '\0';
		if (i < fieldwd[v])
			{
			fprintf(dap_log,
		"(step (%s:%d)) ERROR: got %d of %d characters for fixed length field for %s: %s\n",
				dap_in[dap_ono]->dfile_name,
				lineno[dap_ono], i, fieldwd[v],
				dap_obs[dap_ono].do_nam[dap_obs[dap_ono].do_in[v]],
				value);
			fprintf(dap_err,
		"(step (%s:%d)) ERROR: got %d of %d characters for fixed length field for %s: %s\n",
				dap_in[dap_ono]->dfile_name,
				lineno[dap_ono], i, fieldwd[v],
				dap_obs[dap_ono].do_nam[dap_obs[dap_ono].do_in[v]],
				value);
			exit(1);
			}
		}
	else
		{
		for (i = 0; line[l + i] && line[l + i] != dap_delim; i++)
			value[i] = line[l + i];
		value[i] = '\0';
		}
	if (dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[v]] == INT)
		{
		if (intype == DSET)
			dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_in[v]] = dap_getint(value);
		else if (sscanf(value, " %d",
				dap_obs[dap_ono].do_int + dap_obs[dap_ono].do_in[v]) != 1)
			{
			if (dap_blank(value))
				dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_in[v]] = 0;
			else
				{
				fprintf(dap_err,
					"(step (%s:%d)) ERROR: invalid integer data for %s: %s\n",
					dap_in[dap_ono]->dfile_name,
					lineno[dap_ono],
					dap_obs[dap_ono].do_nam[dap_obs[dap_ono].do_in[v]],
					value);
				exit(1);
				}
			}
		if (!dap_ono && dap_obs[dap_ono].do_il[dap_obs[0].do_in[v]])
			*dap_obs[dap_ono].do_il[dap_obs[0].do_in[v]] =
					dap_obs[0].do_int[dap_obs[0].do_in[v]];
		}
	else if (dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[v]] == DBL)
		{
		if (intype == DSET)
			dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_in[v]] = dap_getdouble(value);
		else
			{
			if (!i || (nfields && dap_blank(value)) || !strcmp(value, "."))
				dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_in[v]] = 0.0 / 0.0;
			else if (sscanf(value, " %lf",
				dap_obs[dap_ono].do_dbl + dap_obs[dap_ono].do_in[v]) != 1)
				{
				fprintf(dap_err,
					"(step (%s:%d)) ERROR: invalid double data for %s: %s\n",
					dap_in[dap_ono]->dfile_name,
					lineno[dap_ono],
					dap_obs[dap_ono].do_nam[dap_obs[dap_ono].do_in[v]],
					value);
				exit(1);
				}
			}
		if (!dap_ono && dap_obs[dap_ono].do_dl[dap_obs[0].do_in[v]])
			*dap_obs[dap_ono].do_dl[dap_obs[0].do_in[v]] =
					dap_obs[0].do_dbl[dap_obs[0].do_in[v]];
		}
	else
		{
		if (i <= dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[v]])
			strcpy(dap_obs[dap_ono].do_str[dap_obs[dap_ono].do_in[v]], value);
		else
			{
			strncpy(dap_obs[dap_ono].do_str[dap_obs[dap_ono].do_in[v]],
				value, dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[v]]);
			dap_obs[dap_ono].do_str[dap_obs[dap_ono].do_in[v]]
				[dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[v]]] = '\0';
			if (!toolong)
				{
				fprintf(dap_log,
				"(step (%s:%d)) WARNING: string data too long (%d) for %s (%d): %s\n",
				dap_in[dap_ono]->dfile_name,
				lineno[dap_ono], i,
				dap_obs[dap_ono].do_nam[dap_obs[dap_ono].do_in[v]],
				dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_in[v]],
				value);
				toolong = 1;
				}
			}
		}
	l += i;
	if (!nfields && line[l] == dap_delim)
		l++;
	}
dap_obs[dap_ono].do_valid = 1;
lineno[dap_ono]++;
return 1;
}

int dap_vd(char *varspec, int invar)
{
int i;
int s;
char *varnam;
int sign;
int v;
int vlen;
int redeclare;

v = -1;
varnam = dap_malloc(strlen(varspec), varspec);
for (s = 0; varspec[s] == ' '; )
	s++;
while (varspec[s])
	{
	if (dap_obs[dap_ono].do_nvar < dap_maxvar)
		{
		for (i = 0; varspec[s + i] && varspec[s + i] != ' '; i++)
			varnam[i] = varspec[s + i];
		varnam[i] = '\0';
		if ((v = dap_varnum(varnam)) < 0)
			{
			redeclare = 0;
			v = dap_obs[dap_ono].do_nvar;
			if (dap_obs[dap_ono].do_nam[v])
				dap_free(dap_obs[dap_ono].do_nam[v]);
			dap_obs[dap_ono].do_nam[v] = dap_malloc(i + 1, varnam);
			strcpy(dap_obs[dap_ono].do_nam[v], varnam);
			if (invar)
				dap_obs[dap_ono].do_in[dap_obs[dap_ono].do_ivar++] = v;
			}
		else
			redeclare = 1;
		while (varspec[s + i] && varspec[s + i] == ' ')
			i++;
		if (!varspec[s + i])
			{
			fprintf(dap_err, "(dap_vd) missing variable length: %s\n",
									varspec + s);
			exit(1);
			}
		s += i;
		if (varspec[s] == '-')
			{
			sign = -1;
			for (s++; varspec[s] == ' '; s++)
				;
			}
		else
			sign = 1;
		for (i = 0, vlen = 0; '0' <= varspec[s + i] && varspec[s + i] <= '9'; i++)
			vlen = 10 * vlen + varspec[s + i] - '0';
		vlen *= sign;
		if (redeclare)
			{
			if (dap_obs[dap_ono].do_len[v] != vlen)
				{
				if (vlen > 0 && dap_obs[dap_ono].do_len[v] > 0)
					fprintf(dap_err,
				"(dap_vd) respecification of length of %s from %d to %d\n",
						varnam, dap_obs[dap_ono].do_len[v] + 1, vlen + 1);
				else
					fprintf(dap_err,
					"(dap_vd) respecification of type of %s\n",
						varnam);
				exit(1);
				}
			}
		else
			dap_obs[dap_ono].do_len[v] = vlen;
		if (varspec[s + i] && varspec[s + i] != ' ')
			{
			fprintf(dap_err,
			"(dap_vd) invalid variable length for %s: %s\n",
				varnam, varspec + s);
			exit(1);
			}
		if (!redeclare && vlen > 0)
			{
			if (!dap_obs[dap_ono].do_sl[v] && dap_obs[dap_ono].do_str[v])
				dap_free(dap_obs[dap_ono].do_str[v]);
			if (dap_ono < 2 && dap_prev[dap_ono].do_str[v])
				dap_free(dap_prev[dap_ono].do_str[v]);
			dap_obs[dap_ono].do_str[v] = dap_malloc(vlen + 1, varnam);
			dap_obs[dap_ono].do_sl[v] = 0;
			if (dap_ono < 2)
				dap_prev[dap_ono].do_str[v] = dap_malloc(vlen + 1, varnam);
			}
		if (vlen == DBL)
			dap_obs[dap_ono].do_dbl[v] = 0.0 / 0.0;
		else if (vlen == INT)
			dap_obs[dap_ono].do_int[v] = 0;
		else
			dap_obs[dap_ono].do_str[v][0] = '\0';
		s += i;
		while (varspec[s] == ' ')
			s++;
		}
	else
		{
		fprintf(dap_err, "(dap_vd) too many variables: %s\n", varspec + s);
		exit(1);
		}
	if (!redeclare)
		dap_obs[dap_ono].do_nvar++;
	}
dap_free(varnam);
return v;
}

void dap_dl(char varname[], double *dbl)
{
int v;
int dim;
int d;

if ((v = dap_varnum(varname)) >= 0)
	{
	dap_obs[dap_ono].do_dl[v] = dbl;
	*dbl = 0.0 / 0.0;
	}
else
	{
	if ((v = dap_arrnum(varname, &dim)) > 0)
		{
		for (d = 0; d < dim; d++)
			{
			dap_obs[dap_ono].do_dl[v + d] = dbl + d;
			dbl[d] = 0.0 / 0.0;
			}
		}
	else
		{
		fprintf(dap_err, "(dap_dl) unknown variable %s\n", varname);
		exit(1);
		}
	}
}

void dap_il(char varname[], int *i)
{
int v;
int dim;
int d;

if ((v = dap_varnum(varname)) > 0)
	{
	dap_obs[dap_ono].do_il[v] = i;
	*i = 0;
	}
else
	{
	if ((v = dap_arrnum(varname, &dim)) > 0)
		{
		for (d = 0; d < dim; d++)
			{
			dap_obs[dap_ono].do_il[v + d] = i + d;
			i[d] = 0;
			}
		}
	else
		{
		fprintf(dap_err, "(dap_il) unknown variable %s\n", varname);
		exit(1);
		}
	}
}

void dap_sl(char varname[], char *s)
{
int v;

if ((v = dap_varnum(varname)) >= 0)
	{
	dap_free(dap_obs[dap_ono].do_str[v]);
	dap_obs[dap_ono].do_str[v] = s;
	dap_obs[dap_ono].do_sl[v] = 1;
	s[0] = '\0';
	}
else
	{
	fprintf(dap_err, "(dap_sl) unknown variable %s\n", varname);
	exit(1);
	}
}

void dap_name(char dname[], char *fname)
{
struct stat statbuf;

if (!fname || !fname[0])
	{
	strcpy(dname, "/dev/null");
	return;
	}
strcpy(dname, dap_setdir);
if (stat(dname, &statbuf) < 0)
	{
	if (mkdir(dname, (mode_t) 0700) < 0)
		{
		perror(dap_dapname);
		exit(1);
		}
	}
else if (!(statbuf.st_mode & S_IFDIR))
	{
	fprintf(dap_err, "%s: non-directory file exists: %s\n", dap_dapname, dname);
	exit(1);
	}
strcat(dname, "/");
strcat(dname, fname);
}

void dataset(char oldname[], char newname[], char action[])
{
static int datsetinit = 0;
char *dold;
char *dnew;
int fold, fnew;
DFILE *doldf, *dnewf;
int c;
int v;
int onum;
int maxnamlen;			/* max length of variable name in dataset */
static dataobs baseobs;		/* dataobs for append */
char *varspec;
char *oldvmem;
char *newvmem;
char **oldvar;
char **newvar;
int nvar;
int vn, nv;
int dim, ndim;
char *outlist;
static char dimstr[7];	/* depends on dap_maxvar <= 10,000 */
int *clearvar;		/* indices of variables to clear in APPEND */
int nclear;		/* number of variables to clear */

if (!datsetinit)	/* need to allocate and initiate baseobs? */
	{
	datsetinit = 1;
	initdo(&baseobs);
	}
if (dap_in[0])		/* if input file left hanging... */
	{		/* ... finish it up */
	dfclose(dap_in[0]);
	dap_in[0] = (DFILE *) NULL;
	}
if (dap_out)		/* if output file left hanging... */
	{		/* ... finish it up */
	dfclose(dap_out);
	dap_out = (DFILE *) NULL;
	}
outlist = dap_malloc(dap_listlen, "dap_listlen");
oldvmem = dap_malloc(dap_maxvar * (dap_namelen + 1), "dap_maxvar, dap_namelen");
oldvar = (char **) dap_malloc(sizeof(char *) * dap_maxvar, "dap_maxvar");
for (v = 0; v < dap_maxvar; v++)
	oldvar[v] = oldvmem + v * (dap_namelen + 1);
newvmem = dap_malloc(dap_maxvar * (dap_namelen + 1), "dap_maxvar, dap_namelen");
newvar = (char **) dap_malloc(sizeof(char *) * dap_maxvar, "dap_maxvar");
for (v = 0; v < dap_maxvar; v++)
	newvar[v] = newvmem + v * (dap_namelen + 1);
dold = dap_malloc(strlen(oldname) + strlen(dap_setdir) + 2, oldname);
dnew = dap_malloc(strlen(newname) + strlen(dap_setdir) + 2, newname);
dap_name(dold, oldname);	/* set up dataset names */
dap_name(dnew, newname);
clearvar = NULL;
if (!strcmp(action, "RENAME"))
	{
	if (oldname[0] == '<')
		{
		if (newname[0] == '<')
			{
			for (fold = 0; fold < dap_nrfiles; fold++)
				{
				if (!strcmp(dfile[3 + fold].dfile_name, oldname + 1))
					break;
				}
			if (fold < dap_nrfiles)
				{
				for (fnew = 0; fnew < dap_nrfiles; fnew++)
					{
					if (!strcmp(dfile[3 + fnew].dfile_name, newname + 1))
						break;
					}
				if (fnew < dap_nrfiles)
					dfile[3 + fnew].dfile_name[0] = '\0';
				strcpy(dfile[3 + fold].dfile_name, newname + 1);
				}
			else
				{
				fprintf(dap_err,
					"(dataset) can't find ramfile %s\n", oldname);
				exit(1);
				}
			}
		else
			{
			fprintf(dap_err,
				"(dataset) can't rename ramfile %s to disk file %s\n",
						oldname, newname);
			exit(1);
			}
		}
	else if (newname[0] == '<')
		{
		fprintf(dap_err, "(dataset) can't rename disk file %s to ramfile %s\n",
						oldname, newname);
			exit(1);
		}
	else
		rename(dold, dnew);
	}
else if (!strncmp(action, "COPY", 4))
	{
	for (v = 4; action[v] == ' '; v++)	/* get past COPY */
		;
	for (nvar = 0; action[v]; nvar++)
		{
		for (c = 0; action[v] && action[v] != ' ' && action[v] != '>'; )
			{
			if (c < dap_namelen)
				oldvar[nvar][c++] = action[v++];
			else
				{
				fprintf(dap_err, "(dataset) variable name too long: %s\n",
									action);
				exit(1);
				}
			}
		oldvar[nvar][c] = '\0';		/* now oldvar[nvar] is named variable */
		while (action[v] == ' ')
			v++;
		if (action[v] == '>')		/* if changing name */
			{
			for (v++; action[v] == ' '; v++)
				;
			for (c = 0; action[v] && action[v] != ' '; )
				{
				if (c < dap_namelen)
					newvar[nvar][c++] = action[v++];
				else
					{
					fprintf(dap_err,
					"(dataset) new variable name too long: %s\n",
									action);
					exit(1);
					}
				}
			newvar[nvar][c] = '\0';	/* now newvar[nvar] is new name */
			while (action[v] == ' ')
				v++;
			}
		else
			strcpy(newvar[nvar], oldvar[nvar]);
		}
	if (nvar)		/* if there are variables to copy */
		{
		inset(oldname);
		for (v = 0, outlist[0] = '\0'; v < nvar; v++)
			{
			if (v)
				strcat(outlist, " ");
			strcat(outlist, newvar[v]);	/* output list has newvars */
			if (strcmp(oldvar[v], newvar[v]))	/* if changing name */
				{
				if ((vn = dap_varnum(oldvar[v])) >= 0)
					{
				/* if newvar exists in oldname, zap its first char to '0' */
					if ((nv = dap_varnum(newvar[v])) >= 0)
						dap_obs[0].do_nam[nv][0] = '0';
				/* and change the name of oldvar to newvar */
					dap_free(dap_obs[0].do_nam[vn]);
					dap_obs[0].do_nam[vn] =
						dap_malloc(strlen(newvar[v]) + 1, "");
					strcpy(dap_obs[0].do_nam[vn], newvar[v]);
					}
				else if ((vn = dap_arrnum(oldvar[v], &dim)) >= 0)
					{
				/* do the same for arrays */
					if ((nv = dap_arrnum(newvar[v], &ndim)) >= 0)
						{
						for (c = 0; c < ndim; c++)
							{
							dap_obs[0].do_nam[nv + c][0] = '0';
							sprintf(dimstr, "[%d]", c);
							strcat(dap_obs[0].do_nam[vn + c], dimstr);
							}
						}
					for (c = 0; c < dim; c++)
						{
						sprintf(dimstr, "[%d]", c);
						dap_free(dap_obs[0].do_nam[vn + c]);
						dap_obs[0].do_nam[vn + c] =
							dap_malloc(strlen(newvar[v]) +
									strlen(dimstr) + 1, "");
						strcpy(dap_obs[0].do_nam[vn + c], newvar[v]);
						strcat(dap_obs[0].do_nam[vn + c], dimstr);
						}
					}
				else
					{
					fprintf(dap_err, "(dataset) unknown variable %s\n",
								oldvar[v]);
					exit(1);
					}
				}
			}
		outset(newname, outlist);
		while (step())
			output();
		}
	else
		{
		if (!(doldf = dfopen(oldname, "r")))
			{
			fprintf(dap_err, "(dataset) can't read %s for copy.\n",
								oldname);
			exit(1);
			}
		if (!(dnewf = dfopen(newname, "w")))
			{
			fprintf(dap_err, "(dataset) can't write %s for copy.\n",
								newname);
			exit(1);
			}
		while ((c = dgetc(doldf)) != EOF)
			dap_putc(c, dnewf);
		dfclose(dnewf);
		dfclose(doldf);
		}
	}
else if (!strcmp(action, "REMOVE"))
	{
	if (oldname[0] == '<')
		{
		for (fold = 0; fold < dap_nrfiles; fold++)
			{
			if (!strcmp(dfile[3 + fold].dfile_name, oldname + 1))
				{
				dap_free(dfile[3 + fold].dfile_name);
				dfile[3 + fold].dfile_name = NULL;
				dap_free(rfile[fold].rfile_str);
				break;
				}
			}
		}
	else
		unlink(dold);
	}
else if (!strcmp(action, "APPEND"))
	{
	if (!(dap_out = dfopen(newname, "r")))	/* no dataset to append to... */
		{				/* ...so just copy. */
		if (!(dap_out = dfopen(newname, "w")))
			{
			fprintf(dap_err,
				"(dataset) Can't create new data set for append: %s\n",
				newname);
			exit(1);
			}
		if (!(dap_in[0] = dfopen(oldname, "r")))
			{
			fprintf(dap_err,
				"(dataset) can't read old data set for append: %s\n",
				oldname);
			exit(1);
			}
		while ((c = dgetc(dap_in[0])) != EOF)
			dap_putc(c, dap_out);
		dfclose(dap_in[0]);
		dap_in[0] = (DFILE *) NULL;
		dfclose(dap_out);
		dap_out = (DFILE *) NULL;
		return;
		}
	else				/* really append */
		{
		inset(newname);			/* to get variable list */
		outset("dap_null", "");		/* so don't output anywhere */
		baseobs.do_nvar = dap_obs[0].do_nvar;	/* save newname's nvar */
		for (v = 0; v < baseobs.do_nvar; v++)
			{			/* copy names, lens from newname */
			baseobs.do_len[v] = dap_obs[0].do_len[v];
			baseobs.do_nam[v] = dap_malloc(strlen(dap_obs[0].do_nam[v]) + 1,
                                                     dap_obs[0].do_nam[v]);
			strcpy(baseobs.do_nam[v], dap_obs[0].do_nam[v]);
			}
		dfclose(dap_out);
		dap_out = (DFILE *) NULL;
		inset(oldname);		/* open "old" dataset to append to newname */
		dap_obs[0].do_ovar = 0;
		for (v = 0, maxnamlen = 0; v < baseobs.do_nvar; v++)
			{		/* find longest name length in newname */
			if (maxnamlen < strlen(baseobs.do_nam[v]))
				maxnamlen = strlen(baseobs.do_nam[v]);
			}
		/* set up for specification for declaration of those variables in
                 * newname that are not in oldname
                 */
		varspec = dap_malloc(maxnamlen + dap_intlen + 2, "");
		clearvar = (int *) dap_malloc(sizeof(int) * baseobs.do_nvar, "");
		for (v = 0, nclear = 0; v < baseobs.do_nvar; v++)
			{
			if ((onum = dap_varnum(baseobs.do_nam[v])) < 0)
				{	/* get index in oldname */
				/* if none give it newname's specs for it */
				sprintf(varspec, "%s %d", baseobs.do_nam[v],
							baseobs.do_len[v]);
				onum = dap_vd(varspec, 0);
				/* mark for setting to "", 0, and 0.0 */
				clearvar[nclear++] = onum;
				}
			else		/* if it's there, check type, len */
				{
				if (dap_obs[0].do_len[onum] != baseobs.do_len[v])
					{
					fprintf(stderr,
		"dataset: variable %s has different lengths (%d appended to %d) in datasets\n",
					baseobs.do_nam[v], dap_obs[0].do_len[onum],
					baseobs.do_len[v]);
					exit(1);
					}
				}
			dap_obs[0].do_out[dap_obs[0].do_ovar++] = onum;
			}
		if (!(dap_out = dfopen(newname, "a")))
			{
			fprintf(dap_err, "(dataset) can't append to new data set: %s\n",
				newname);
			exit(1);
			}
		while (step())
			{
			/* and set to "", 0, and 0.0 */
			for (v = 0; v < nclear; v++)
				{
				if (dap_obs[0].do_str[clearvar[v]])
					dap_obs[0].do_str[clearvar[v]][0] = '\0';
				dap_obs[0].do_sl[clearvar[v]] = 0;
				dap_obs[0].do_int[clearvar[v]] = 0;
				dap_obs[0].do_dbl[clearvar[v]] = 0.0;
				}
			output();
			}
		for (v = 0; v < baseobs.do_nvar; v++)
			{
			dap_free(baseobs.do_nam[v]);
			baseobs.do_nam[v] = NULL;
			}
		dap_free(varspec);
		}
	}
else
	{
	fprintf(dap_err, "(dataset) unknown action: %s\n", action);
	exit(1);
	}
dap_free(dold);
dap_free(dnew);
dap_free(oldvmem);
dap_free(oldvar);
dap_free(newvmem);
dap_free(newvar);
dap_free(outlist);
if (clearvar)
	dap_free(clearvar);
}

#define BLOCKLEN 32

static int getblock(DFILE *fp)
{
char block[BLOCKLEN];
int b;

if ((block[0] = dgetc(fp)) == '\r')
        return 0;
for (b = 1; b < BLOCKLEN; b++)
        block[b] = dgetc(fp);
return (block[16] & 0xff);
}

void infile(char *ifname, char *idelim)
{
static int infinit = 0;
int v;
int d;
int infldlen;
char *fname;
static char *delim;
static int delimlen;

if (!infinit)
	{
	infinit = 1;
	delimlen = (dap_linelen + 1) / 8 - 1;
	delim = dap_malloc(delimlen + 1, "dap_linelen");
	fieldwd = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
	}
if (dap_in[dap_ono])
	{
	dfclose(dap_in[dap_ono]);
	dap_in[dap_ono] = (DFILE *) NULL;
	}
if (dap_out)
	{
	dfclose(dap_out);
	dap_out = (DFILE *) NULL;
	}
if (!ifname || !ifname[0])
	{
	fname = "/dev/null";
	strcpy(delim, "|");
	}
else
	{
	fname = ifname;
	if (!idelim || !idelim[0])
		{
		fprintf(dap_err,
			"(infile) Delimiter string must be at least one character: %s\n",
					delim);
		exit(1);
		}
	else if (strlen(idelim) <= delimlen)
		strcpy(delim, idelim);
	else
		{
		fprintf(dap_err,
			"(infile) Delimiter string too long: %s\n", idelim);
		exit(1);
		}
	}
if (!(dap_in[dap_ono] = dfopen(fname, "rf")))
	{
	fprintf(dap_err, "(infile) can't read data file: %s\n", fname);
	exit(1);
	}
intype = TEXT;
if (!strcmp(fname + strlen(fname) - 4, ".dbf"))
	{
	intype = DBASE;
	getblock(dap_in[dap_ono]);     /* dump header */
	for (inlen = 0; (infldlen = getblock(dap_in[dap_ono])); inlen += infldlen)
		;
	}
dap_delim = delim[0];
for (nfields = 0, d = 1; delim[d]; nfields++)
	{
	if (nfields < dap_maxvar)
		{
		for (fieldwd[nfields] = 0; delim[d] && delim[d] != dap_delim; )
			fieldwd[nfields] = 10 * fieldwd[nfields] + delim[d++] - '0';
		if (delim[d])
			d++;
		}
	else
		{
		fputs("(infile) too many field width specifiers.\n", dap_err);
		exit(1);
		}
	}
dap_prev[dap_ono].do_ivar = 0;
dap_obs[dap_ono].do_ivar = 0;
dap_obs[dap_ono].do_ovar = 0;
dap_obs[dap_ono].do_nvar = 0;
dap_obs[dap_ono].do_valid = 0;
eof[dap_ono] = 0;
for (v = 0; v < dap_maxvar; v++)
	{
	if (dap_obs[dap_ono].do_nam[v])
		dap_free(dap_obs[dap_ono].do_nam[v]);
	dap_obs[dap_ono].do_nam[v] = NULL;
	dap_obs[dap_ono].do_dl[v] = NULL;
	dap_obs[dap_ono].do_il[v] = NULL;
	if (!dap_obs[dap_ono].do_sl[v] && dap_obs[dap_ono].do_str[v])
		dap_free(dap_obs[dap_ono].do_str[v]);
	dap_obs[dap_ono].do_str[v] = NULL;
	dap_obs[dap_ono].do_sl[v] = 0;
	if (dap_prev[dap_ono].do_str[v])
		dap_free(dap_prev[dap_ono].do_str[v]);
	dap_prev[dap_ono].do_str[v] = NULL;
	}
dap_vd("_type_ 8", 0);
if ((v = dap_varnum("_type_")) < 0)
	{
	fprintf(dap_err, "(infile(%s)) missing _type_ variable\n", fname);
	exit(1);
	}
dap_obs[dap_ono].do_out[dap_obs[dap_ono].do_ovar++] = v;
strcpy(dap_obs[dap_ono].do_str[v], "OBS");
lineno[dap_ono] = 1;
toolong = 0;
}

void input(char varlist[])
{
int v;
int l;
int i;
static char *vname = NULL;
int dim;
int d;

if (!vname)
	vname = dap_malloc(dap_namelen + 1, "dap_namelen");
for (l = 0; varlist[l] == ' '; l++)
	;
while (varlist[l])
	{
	for (i = 0; varlist[l + i] && varlist[l + i] != ' '; i++)
		{
		if (i < dap_namelen)
			vname[i] = varlist[l + i];
		else
			{
			vname[i] = '\0';
			fprintf(dap_err, "(input) variable name too long: %s\n", vname);
			exit(1);
			}
		}
	vname[i] = '\0';
	if ((v = dap_varnum(vname)) >= 0)
		dap_obs[dap_ono].do_in[dap_obs[dap_ono].do_ivar++] = v;
	else if ((v = dap_arrnum(vname, &dim)) >= 0)
		{
		for (d = 0; d < dim; d++)
			dap_obs[dap_ono].do_in[dap_obs[dap_ono].do_ivar++] = v + d;
		}
	else
		{
		fprintf(dap_err, "(input) unknown variable: %s\n", vname);
		exit(1);
		}
	l += i;
	while (varlist[l] == ' ')
		l++;
	}
}

void inset(char *fname)
{
int v;
static char *varspec = NULL;
double testd;

if (!varspec)
	varspec = dap_malloc(dap_linelen + 1, "dap_linelen");
if (dap_in[dap_ono])
	{
	dfclose(dap_in[dap_ono]);
	dap_in[dap_ono] = (DFILE *) NULL;
	}
if (dap_out)
	{
	dfclose(dap_out);
	dap_out = (DFILE *) NULL;
	}
if (!fname)
	{
	fputs("(inset) missing input dataset name.\n", dap_err);
	exit(1);
	}
if (!(dap_in[dap_ono] = dfopen(fname, "r")))
	{
	fprintf(dap_err, "(inset) can't read data set: %s\n", fname);
	exit(1);
	}
intype = DSET;
dap_delim = '\0';
if (getline(dap_in[dap_ono], varspec) < 0)
	{
	fprintf(dap_err, "(inset) data set empty: %s\n", fname);
	exit(1);
	}
nfields = 0;
dap_prev[dap_ono].do_ivar = 0;
dap_obs[dap_ono].do_ivar = 0;
dap_obs[dap_ono].do_ovar = 0;
dap_obs[dap_ono].do_nvar = 0;
dap_obs[dap_ono].do_valid = 0;
eof[dap_ono] = 0;
for (v = 0; v < dap_maxvar; v++)
	{
	if (dap_obs[dap_ono].do_nam[v])
		dap_free(dap_obs[dap_ono].do_nam[v]);
	dap_obs[dap_ono].do_nam[v] = NULL;
	dap_obs[dap_ono].do_dl[v] = NULL;
	dap_obs[dap_ono].do_il[v] = NULL;
	if (!dap_obs[dap_ono].do_sl[v] && dap_obs[dap_ono].do_str[v])
		dap_free(dap_obs[dap_ono].do_str[v]);
	dap_obs[dap_ono].do_str[v] = NULL;
	dap_obs[dap_ono].do_sl[v] = 0;
	if (dap_prev[dap_ono].do_str[v])
		dap_free(dap_prev[dap_ono].do_str[v]);
	dap_prev[dap_ono].do_str[v] = NULL;
	}
dap_delim = SETDELIM;
dap_vd(varspec, 1);
if ((v = dap_varnum("_type_")) < 0)
	{
	fprintf(dap_err, "(inset(%s)) missing _type_ variable\n", fname);
	exit(1);
	}
dap_obs[dap_ono].do_out[dap_obs[dap_ono].do_ovar++] = v;
lineno[dap_ono] = 1;
}

/* Remove bracketed indices from explicit entry references
 * but do not create duplications of variable names
 */
static void fixlist(char *varl, char *varlist)
{
static int fixinit = 0;
int l;			/* index to var lists */
int f;
int inbrack;
static char *vname;
int dim;
static int *outv;
int l0;
int l1;
int f1;
int nv;
int v;
int vn;

if (!fixinit)
	{
	fixinit = 1;
	vname = dap_malloc(dap_namelen + 1, "dap_namelen");
	outv = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
	}
if (!varl)
	{
	fputs("(fixlist) missing variable list.\n", dap_err);
	exit(1);
	}
if (!varlist)
	{
	fputs("(fixlist) missing string for fixed variable list.\n", dap_err);
	exit(1);
	}
/* First copy varl to varlist, eliminating bracketed indices */
for (inbrack = 0, l = 0, f = 0; varl[l]; l++)
	{
	if (inbrack)
		{
		if (varl[l] == ']')
			inbrack = 0;
		}
	else if (varl[l] == '[')
		inbrack = 1;
	else
		varlist[f++] = varl[l];
	}
varlist[f] = '\0';
for (l = 0; varlist[l] == ' '; l++)
	;
l0 = 0;
if (varlist[l] == '!')
	{
	for (l++; varlist[l] == ' '; l++)
		;
	varlist[l0++] = '!';
	}
for (nv = 0, l1 = l0; varlist[l]; )
	{
	for (f = 0; varlist[l + f] && varlist[l + f] != ' '; f++)
		{
		if (f < dap_namelen)
			vname[f] = varlist[l + f];
		else
			{
			vname[f] = '\0';
			fprintf(dap_err, "(fixlist) variable name too long: %s\n", vname);
			exit(1);
			}
		}
	vname[f] = '\0';
	if (((v = dap_varnum(vname)) < 0) && ((v = dap_arrnum(vname, &dim)) < 0))
		{
		fprintf(dap_err, "(fixlist) unknown variable: %s\n", vname);
		exit(1);
		}
	for (vn = 0; vn < nv; vn++)	/* see if variable already appeared */
		{
		if (outv[vn] == v)
			break;
		}
	if (vn == nv)				/* if not */
		{
		outv[nv++] = v;			/* enter it in list of indices */
		if (l1 > l0)
			varlist[l1++] = ' ';
		for (f1 = 0; f1 < f; f1++)	/* copy name from varlist to varlist */
			varlist[l1++] = varlist[l++];
		}
	else
		l += f;
	while (varlist[l] == ' ')
		l++;
	}
varlist[l1] = '\0';
}

void outset(char *fname, char *varl)
{
static int outinit = 0;
int l;
int i;
static char *varlist;
static char *vname;
int v;
int w;
int first;
int dim;
int d;
double testd;

if (dap_out)
	{
	dfclose(dap_out);
	dap_out = (DFILE *) NULL;
	}
if (!fname)
	{
	fputs("(outset) no dataset name.\n", dap_err);
	exit(1);
	}
if (!outinit)
	{
	outinit = 1;
	varlist = dap_malloc(dap_listlen, "dap_listlen");
	vname = dap_malloc(dap_listlen, "dap_listlen");
	}
if (!(dap_out = dfopen(fname, "w")))
	{
	fprintf(dap_err, "(outset) can't write data set: %s\n", fname);
	exit(1);
	}
if (!varl)
	{
	fputs("(outset) missing variable list.\n", dap_err);
	exit(1);
	}
fixlist(varl, varlist);
if (varlist[0])
	{
	for (l = 0; varlist[l] == ' '; l++)
		;
	if (varlist[l] == '!')	/* listing variables NOT to include */
		{		/* first put them all in */
		for (l++; varlist[l] == ' '; l++)
			;
		dap_obs[dap_ono].do_ovar = dap_obs[dap_ono].do_nvar;
		for (v = 0; v < dap_obs[dap_ono].do_nvar; v++)
			dap_obs[dap_ono].do_out[v] = v;
		while (varlist[l])	/* then take them out */
			{
			for (i = 0; varlist[l + i] && varlist[l + i] != ' '; i++)
				vname[i] = varlist[l + i];
			vname[i] = '\0';
			if ((w = dap_varnum(vname)) >= 0)
				{
				for (v = 0; v < dap_obs[dap_ono].do_ovar; v++)
					{
					if (dap_obs[dap_ono].do_out[v] == w)
						break;
					}
				if (v == dap_obs[dap_ono].do_ovar)
					{
					fprintf(dap_err,
			"(outset) variable not in list of variables to exclude: %s\n",
						vname);
					exit(1);
					}
				while (v < dap_obs[dap_ono].do_ovar - 1)
					{
					dap_obs[dap_ono].do_out[v] =
						dap_obs[dap_ono].do_out[v + 1];
					v++;
					}
				dap_obs[dap_ono].do_ovar--;
				}
			else if ((w = dap_arrnum(vname, &dim)) >= 0)
				{
				for (v = 0; v < dap_obs[dap_ono].do_ovar; v++)
					{
					if (dap_obs[dap_ono].do_out[v] == w)
						break;
					}
				while (v < dap_obs[dap_ono].do_ovar - dim)
					{
					dap_obs[dap_ono].do_out[v] =
						dap_obs[dap_ono].do_out[v + dim];
					v++;
					}
				dap_obs[dap_ono].do_ovar -= dim;
				}
			else
				{
				fprintf(dap_err, "(outset(%s)) unknown variable: %s\n",
					fname, vname);
				exit(1);
				}
			l += i;
			while (varlist[l] == ' ')
				l++;
			}
		}
	else
		{
		while (varlist[l])
			{
			for (i = 0; varlist[l + i] && varlist[l + i] != ' '; i++)
				vname[i] = varlist[l + i];
			vname[i] = '\0';
			if ((v = dap_varnum(vname)) >= 0)
				dap_obs[dap_ono].do_out[dap_obs[dap_ono].do_ovar++] = v;
			else if ((v = dap_arrnum(vname, &dim)) >= 0)
				{
				for (d = 0; d < dim; d++)
					dap_obs[dap_ono].do_out[dap_obs[dap_ono].do_ovar++] = v + d;
				}
			else
				{
				fprintf(dap_err, "(outset(%s)) unknown variable: %s\n",
					fname, vname);
				exit(1);
				}
			l += i;
			while (varlist[l] == ' ')
				l++;
			}
		}
	}
else
	{
	dap_obs[dap_ono].do_ovar = dap_obs[dap_ono].do_nvar;
	for (v = 0; v < dap_obs[dap_ono].do_nvar; v++)
		dap_obs[dap_ono].do_out[v] = v;
	}
if (dap_varnum("_type_") < 0)
	{
	fputs("(outset) missing _type_ variable\n", dap_err);
	exit(1);
	}
for (v = 0; v < dap_obs[dap_ono].do_ovar; v++)
	{
	for (w = v + 1; w < dap_obs[dap_ono].do_ovar; w++)
		{
		if (dap_obs[dap_ono].do_out[v] == dap_obs[dap_ono].do_out[w])
			{
			fprintf(dap_err,
			"(outset) duplicate variable in output list: %s\n",
				dap_obs[dap_ono].do_nam[dap_obs[dap_ono].do_out[w]]);
			exit(1);
			}
		}
	}
for (v = 0, first = 1; v < dap_obs[dap_ono].do_ovar; v++)
	{
	if (!first)
		dap_putc(' ', dap_out);
	dputs(dap_obs[dap_ono].do_nam[dap_obs[dap_ono].do_out[v]], " ", dap_out);
	dputi(dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_out[v]], dap_out);
	first = 0;
	}
dap_putc('\n', dap_out);
dflush(dap_out);
outline = 0;
}

void output()
{
int v;
int first;

if ((v = dap_varnum("_type_")) < 0)
	{
	fprintf(dap_err, "(output) missing _type_ variable\n");
	exit(1);
	}
for (v = 0, first = 1; v < dap_obs[dap_ono].do_ovar; v++)
	{
	if (!first)
		dap_putc(SETDELIM, dap_out);
	if (dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_out[v]] == INT)
		{
		if (!dap_ono && dap_obs[dap_ono].do_il[dap_obs[0].do_out[v]])
			dap_obs[0].do_int[dap_obs[0].do_out[v]] =
				*dap_obs[dap_ono].do_il[dap_obs[0].do_out[v]];
		dap_putint(dap_obs[dap_ono].do_int[dap_obs[dap_ono].do_out[v]], dap_out);
		}
	else if (dap_obs[dap_ono].do_len[dap_obs[dap_ono].do_out[v]] == DBL)
		{
		if (!dap_ono && dap_obs[dap_ono].do_dl[dap_obs[0].do_out[v]])
			dap_obs[0].do_dbl[dap_obs[0].do_out[v]] =
				*dap_obs[dap_ono].do_dl[dap_obs[0].do_out[v]];
		dap_putdouble(dap_obs[dap_ono].do_dbl[dap_obs[dap_ono].do_out[v]],
				dap_out);
		}
	else
		{
		dputs(dap_obs[dap_ono].do_str[dap_obs[dap_ono].do_out[v]], "",
				dap_out);
		}
	first = 0;
	}
dap_putc('\n', dap_out);
dflush(dap_out);
if (dap_outreport && !(++outline % dap_outreport))
	{
	fprintf(stderr, "(output) %d lines written to %s...\n",
		outline, dap_out->dfile_name);
	fflush(stderr);
	}
}

static int expand(char *varlist, int *varv, int maxvars)
{
int nvars;
int m;
int i;
char *mname;
char *newname;
int arrn;		/* var index of array's first element */
int dim;		/* array dimension */
int d;			/* index to array */

if (!varlist)
        return 0;
if (!varv)
        {
        fputs("(expand) missing variable index list.\n", dap_err);
        exit(1);
        }
mname = dap_malloc(strlen(varlist) + 1, varlist);
newname = dap_malloc(strlen(varlist) + 1, varlist);
for (m = 0; varlist[m] == ' '; m++)
        ;
for (nvars = 0; varlist[m]; )
        {
        for (i = 0; varlist[m + i] && varlist[m + i] != ' ' && varlist[m + i] != '>'; i++)
		mname[i] = varlist[m + i];
        mname[i] = '\0';
        m += i;
        while (varlist[m] == ' ')
                m++;
	newname[0] = '\0';
	if (varlist[m] == '>')
		{
		for (m++; varlist[m] == ' '; m++)
			;
		for (i = 0; varlist[m + i] && varlist[m + i] != ' ' && varlist[m + i] != '>';
											i++)
			newname[i] = varlist[m + i];
		newname[i] = '\0';
		m += i;
		while (varlist[m] == ' ')
			m++;
		}
        if (nvars >= maxvars)
                {
                fprintf(dap_err, "(expand) more than %d variables: %s\n", maxvars, varlist);
                exit(1);
                }
	if ((arrn = dap_arrnum(mname, &dim)) >= 0)
		{
		for (d = 0; d < dim; d++)
			{
			if (newname[0])
				{
				dap_free(dap_obs[dap_ono].do_nam[arrn]);
				dap_obs[dap_ono].do_nam[arrn] =
						dap_malloc(strlen(newname) + 6, newname);
				sprintf(dap_obs[dap_ono].do_nam[arrn],
						"%s[%d]", newname, d);
				}
			varv[nvars++] = arrn++;
			}
		}
        else if ((varv[nvars] = dap_varnum(mname)) > 0)
		{
		if (newname[0])
			{
			dap_free(dap_obs[dap_ono].do_nam[varv[nvars]]);
			dap_obs[dap_ono].do_nam[varv[nvars]] =
					dap_malloc(strlen(newname) + 1, newname);
			strcpy(dap_obs[dap_ono].do_nam[varv[nvars]], newname);
			}
		nvars++;
		}
	else
                {
                fprintf(dap_err, "(expand) variable unknown: %s\n", mname);
                exit(1);
                }
        }
dap_free(mname);
dap_free(newname);
return nvars;
}

static void varcat(char *to, char *from)
{
int t, f, ff;

for (f = 0; from[f] == ' '; f++)
	;
for (t = 0; to[t]; t++)
	;
while (from[f])
	{
	for (ff = f; from[ff] && from[ff] != ' ' && from[ff] != '>'; ff++)
		;
	while (from[ff] == ' ')
		ff++;
	if (from[ff] == '>')
		{
		for (f = ff + 1; from[f] == ' '; f++)
			;
		}
	while (from[f] && from[f] != ' ')
		to[t++] = from[f++];
	to[t++] = ' ';
	while (from[f] == ' ')
		f++;
	}
to[t] = '\0';
}

void merge(char *fname1, char *vars1, char *fname2, char *vars2,
		char *marks, char *outname)
{
int *varv1, *varv2;
int *ovarv1, *ovarv2;
int nvar1, nvar2;
int *markv1;
int *markv2;
int nmark;
int v1, v2;
char *outlist;
char *outlist1;
char *outlist2;
int goon1;
double ddiff;
int isdiff;
int vars1null, vars2null;

if (!fname1 || !fname2 || !outname)
	{
	fputs("(merge) missing dataset name.\n", dap_err);
	exit(1);
	}
if (index(vars1, '['))
	{
	fprintf(dap_err, "(merge) variable lists may not contain individual array elements: %s\n",
			vars1);
	exit(1);
	}
if (index(vars2, '['))
	{
	fprintf(dap_err, "(merge) variable lists may not contain individual array elements: %s\n",
			vars2);
	exit(1);
	}
vars1null = 0;
vars2null = 0;
outlist = dap_malloc(dap_listlen, "dap_listlen");
outlist1 = dap_malloc(dap_listlen, "dap_listlen");
outlist2 = dap_malloc(dap_listlen, "dap_listlen");
varv1 = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
varv2 = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
ovarv1 = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
ovarv2 = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
markv1 = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
markv2 = (int *) dap_malloc(sizeof(int) * dap_maxvar, "dap_maxvar");
inset(fname1);	/* set up input from dataset 1 */
if (vars1 && !vars1[0])		/* vars1 null string: use all variables */
	{
	vars1null = 1;
	vars1 = dap_malloc(dap_listlen, "dap_listlen");
	vars1[0] = '\0';
	for (v1 = 0; v1 < dap_obs[dap_ono].do_nvar; v1++)
		{		/* copy all vars except _type_ */
		if (strcmp(dap_obs[dap_ono].do_nam[v1], "_type_"))
			{
			strcat(vars1, " ");
			strcat(vars1, dap_obs[dap_ono].do_nam[v1]);
			}
		}
	}
nvar1 = expand(vars1, varv1, dap_maxvar);
nmark = dap_list(marks, markv1, dap_maxvar);
dap_ono = 1;
inset(fname2);
if (vars2 && !vars2[0])		/* vars2 null string: use all variables */
	{
	vars2null = 1;
	vars2 = dap_malloc(dap_listlen, "dap_listlen");
	vars2[0] = '\0';
	for (v2 = 0; v2 < dap_obs[dap_ono].do_nvar; v2++)
		{
		if (strcmp(dap_obs[dap_ono].do_nam[v2], "_type_"))
			{	/* copy all vars except _type_ */
			strcat(vars2, " ");
			strcat(vars2, dap_obs[dap_ono].do_nam[v2]);
			}
		}
	}
nvar2 = expand(vars2, varv2, dap_maxvar);
dap_list(marks, markv2, dap_maxvar);
for (v1 = 0; v1 < nmark; v1++)	/* check types of part vars across datasets */
	{
	if (dap_obs[0].do_len[markv1[v1]] != dap_obs[1].do_len[markv2[v1]])
		{
		fprintf(dap_err,
		"(merge) Part variables of different types: %s (%d) and %s (%d)\n",
			dap_obs[0].do_nam[markv1[v1]], dap_obs[0].do_len[markv1[v1]],
			dap_obs[1].do_nam[markv2[v1]], dap_obs[1].do_len[markv2[v1]]);
		exit(1);
		}
	}
outlist[0] = '\0';
if (vars2)			/* start assembling list of output variables */
	varcat(outlist, vars2);
outlist2[0] = '\0';
if (vars2)
	varcat(outlist2, vars2);
dap_ono = 2;
dap_obs[2].do_nvar = 0;
dap_obs[2].do_ovar = 0;
dap_obs[2].do_out[dap_obs[2].do_ovar++] = dap_vd("_type_ 8", 0);
for (v2 = 0; v2 < nvar2; v2++)
	{
	if (dap_obs[2].do_nam[dap_obs[2].do_nvar])
		dap_free(dap_obs[2].do_nam[dap_obs[2].do_nvar]);
	dap_obs[2].do_nam[dap_obs[2].do_nvar] =
			dap_malloc(strlen(dap_obs[1].do_nam[varv2[v2]]) + 1,
				dap_obs[1].do_nam[varv2[v2]]);
	strcpy(dap_obs[2].do_nam[dap_obs[2].do_nvar],
					dap_obs[1].do_nam[varv2[v2]]);
	dap_obs[2].do_len[dap_obs[2].do_nvar] = dap_obs[1].do_len[varv2[v2]];
	dap_obs[2].do_nvar++;
	}
outlist1[0] = '\0';
if (vars1)
	{
	strcat(outlist, " ");
	varcat(outlist, vars1);
	varcat(outlist1, vars1);
	}
for (v1 = 0; v1 < nvar1; v1++)
	{
	for (v2 = 0; v2 < nvar2; v2++)
		{
		if (!strcmp(dap_obs[0].do_nam[varv1[v1]], dap_obs[1].do_nam[varv2[v2]]))
			break;
		}
	if (v2 < nvar2)
		{
		fprintf(dap_err, "(merge) variable appears in both lists: %s\n",
				dap_obs[0].do_nam[varv1[v1]]);
		exit(1);
		}
	else
		{
		if (dap_obs[2].do_nam[dap_obs[2].do_nvar])
			dap_free(dap_obs[2].do_nam[dap_obs[2].do_nvar]);
		dap_obs[2].do_nam[dap_obs[2].do_nvar] =
				dap_malloc(strlen(dap_obs[0].do_nam[varv1[v1]]) + 1,
						dap_obs[0].do_nam[varv1[v1]]);
		strcpy(dap_obs[2].do_nam[dap_obs[2].do_nvar],
						dap_obs[0].do_nam[varv1[v1]]);
		dap_obs[2].do_len[dap_obs[2].do_nvar] = dap_obs[0].do_len[varv1[v1]];
		dap_obs[2].do_nvar++;
		}
	}
outset(outname, outlist);
strcpy(dap_obs[2].do_str[dap_varnum("_type_")], "OBS");
expand(outlist1, ovarv1, dap_maxvar);
expand(outlist2, ovarv2, dap_maxvar);
dap_ono = 0;
goon1 = step();
for ( ; ; )
	{
	dap_ono = 1;
	if (step())
		{
		while (goon1)
			{
			isdiff = 0;
			for (v1 = 0; v1 < nmark; v1++)
				{
				if (dap_obs[0].do_len[markv1[v1]] == DBL)
					{
					ddiff = dap_obs[0].do_dbl[markv1[v1]] -
							dap_obs[1].do_dbl[markv2[v1]];
					if (ddiff < 0.0)
						isdiff = -1;
					else if (ddiff > 0.0)
						isdiff = 1;
					else
						isdiff = 0;
					if (isdiff)
						break;
					}
				else if (dap_obs[0].do_len[markv1[v1]] == INT)
					{
					isdiff = dap_obs[0].do_int[markv1[v1]] -
							dap_obs[1].do_int[markv2[v1]];
					if (isdiff)
						break;
					}
				else if ((isdiff = strcmp(dap_obs[0].do_str[markv1[v1]],
						dap_obs[1].do_str[markv2[v1]])))
					break;
				}
			if (isdiff < 0)
				{
				dap_ono = 0;
				goon1 = step();
				}
			else if (isdiff > 0)
				break;
			else
				{
				for (v1 = 0; v1 < nvar1; v1++)
					{
					if (dap_obs[0].do_len[varv1[v1]] == DBL)
						dap_obs[2].do_dbl[ovarv1[v1]] =
							dap_obs[0].do_dbl[varv1[v1]];
					else if (dap_obs[0].do_len[varv1[v1]] == INT)
						dap_obs[2].do_int[ovarv1[v1]] =
							dap_obs[0].do_int[varv1[v1]];
					else
						{
						if (dap_obs[2].do_str[ovarv1[v1]])
							dap_free(dap_obs[2].do_str[ovarv1[v1]]);
						dap_obs[2].do_str[ovarv1[v1]] =
							dap_malloc(strlen(dap_obs[0].do_str[varv1[v1]]) + 1,
									dap_obs[0].do_str[varv1[v1]]);
						strcpy(dap_obs[2].do_str[ovarv1[v1]],
								dap_obs[0].do_str[varv1[v1]]);
						}
					}
				for (v2 = 0; v2 < nvar2; v2++)
					{
					if (dap_obs[1].do_len[varv2[v2]] == DBL)
						dap_obs[2].do_dbl[ovarv2[v2]] =
							dap_obs[1].do_dbl[varv2[v2]];
					else if (dap_obs[1].do_len[varv2[v2]] == INT)
						dap_obs[2].do_int[ovarv2[v2]] =
							dap_obs[1].do_int[varv2[v2]];
					else
						{
						if (dap_obs[2].do_str[ovarv2[v2]])
							dap_free(dap_obs[2].do_str[ovarv2[v2]]);
						dap_obs[2].do_str[ovarv2[v2]] =
							dap_malloc(strlen(dap_obs[1].do_str[varv2[v2]]) + 1,
									dap_obs[1].do_str[varv2[v2]]);
						strcpy(dap_obs[2].do_str[ovarv2[v2]],
								dap_obs[1].do_str[varv2[v2]]);
						}
					}
				dap_ono = 2;
				output();
				dap_ono = 0;
				goon1 = step();
				}
			}
		}
	else
		break;
	}
dap_ono = 0;
if (dap_in[0])
	{
	dfclose(dap_in[0]);
	dap_in[0] = (DFILE *) NULL;
	}
if (dap_in[1])
	{
	dfclose(dap_in[1]);
	dap_in[1] = (DFILE *) NULL;
	}
if (dap_out)
	{
	dfclose(dap_out);
	dap_out = (DFILE *) NULL;
	}
dap_free(outlist);
dap_free(outlist1);
dap_free(outlist2);
dap_free(varv1);
dap_free(varv2);
dap_free(ovarv1);
dap_free(ovarv2);
dap_free(markv1);
dap_free(markv2);
if (vars1null)
	dap_free(vars1);
if (vars2null)
	dap_free(vars2);
}

void title(char *text)
{
dap_title = text;
}

void dap_head(int markv[], int nmark)
{
time_t t;
int m;

fputs("\n=================================", dap_lst);
fprintf(dap_lst, "\nDap %3d. ", pageno++);
time(&t);
fputs(ctime(&t), dap_lst);
putc('\n', dap_lst);
if (dap_title)
	{
        fputs(dap_title, dap_lst);
	putc('\n', dap_lst);
	}
if (nmark)
	{
	fprintf(dap_lst, "\nFor: ");
	for (m = 0; m < nmark; m++)
		{
		switch(dap_obs[0].do_len[markv[m]])
			{
		case DBL:
			fprintf(dap_lst, "%s = %g",
				dap_obs[0].do_nam[markv[m]],
				dap_obs[0].do_dbl[markv[m]]);
			break;
		case INT:
			fprintf(dap_lst, "%s = %d",
				dap_obs[0].do_nam[markv[m]],
				dap_obs[0].do_int[markv[m]]);
			break;
		default:
			fprintf(dap_lst, "%s = %s",
				dap_obs[0].do_nam[markv[m]],
				dap_obs[0].do_str[markv[m]]);
			break;
			}
		if (m < nmark - 1)
			fputs(", ", dap_lst);
		}
	putc('\n', dap_lst);
	}
putc('\n', dap_lst);
}
