/*
     This file is part of GNUnet.
     Copyright (C) 2012 GNUnet e.V.

     GNUnet is free software: you can redistribute it and/or modify it
     under the terms of the GNU Affero General Public License as published
     by the Free Software Foundation, either version 3 of the License,
     or (at your option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     Affero General Public License for more details.

     You should have received a copy of the GNU Affero General Public License
     along with this program.  If not, see <http://www.gnu.org/licenses/>.

     SPDX-License-Identifier: AGPL3.0-or-later
 */
/**
 * @file namestore/test_namestore_api_store_locking.c
 * @brief testcase for namestore_api.c: store a record, locking
 */
#include "platform.h"
#include "gnunet_namestore_service.h"
#include "gnunet_testing_lib.h"
#include "gnunet_dnsparser_lib.h"

#define TEST_RECORD_TYPE GNUNET_DNSPARSER_TYPE_TXT

#define TEST_RECORD_DATALEN 123

#define TEST_RECORD_DATA 'a'

#define TIMEOUT GNUNET_TIME_relative_multiply (GNUNET_TIME_UNIT_SECONDS, 100)


static struct GNUNET_NAMESTORE_Handle *nsh;

static struct GNUNET_NAMESTORE_Handle *nsh_alt;

static struct GNUNET_SCHEDULER_Task *endbadly_task;

static struct GNUNET_IDENTITY_PrivateKey privkey;

static struct GNUNET_IDENTITY_PublicKey pubkey;

static int res;

static struct GNUNET_NAMESTORE_QueueEntry *nsqe;

static struct GNUNET_NAMESTORE_QueueEntry *nsqe_alt;


static void
cleanup ()
{
  if (NULL != nsh)
  {
    GNUNET_NAMESTORE_disconnect (nsh);
    nsh = NULL;
  }
  if (NULL != nsh_alt)
  {
    GNUNET_NAMESTORE_disconnect (nsh_alt);
    nsh_alt = NULL;
  }
  GNUNET_SCHEDULER_shutdown ();
}


/**
 * Re-establish the connection to the service.
 *
 * @param cls handle to use to re-connect.
 */
static void
endbadly (void *cls)
{
  if (NULL != nsqe)
  {
    GNUNET_NAMESTORE_cancel (nsqe);
    nsqe = NULL;
  }
  cleanup ();
  res = 1;
}


static void
end (void *cls)
{
  cleanup ();
  res = 0;
}


static void
open_alt_second_failed (void *cls)
{
  GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
              "Got did not get lock when I should...\n");
  nsqe_alt = NULL;
  GNUNET_SCHEDULER_cancel (endbadly_task);
  endbadly_task = NULL;
  GNUNET_SCHEDULER_add_now (&endbadly, NULL);
}


void
open_alt_second_cont (void *cls,
               const struct
               GNUNET_IDENTITY_PrivateKey *zone,
               const char *label,
               unsigned int rd_count,
               const struct GNUNET_GNSRECORD_Data *rd)
{
  nsqe_alt = NULL;
  GNUNET_SCHEDULER_cancel (endbadly_task);
  endbadly_task = NULL;
  GNUNET_SCHEDULER_add_now (&end, NULL);
}


static void
put_cont (void *cls, int32_t success, const char *emsg)
{
  const char *name = cls;

  nsqe = NULL;
  GNUNET_assert (NULL != cls);
  GNUNET_log (GNUNET_ERROR_TYPE_DEBUG,
              "Name store added record for `%s': %s\n",
              name,
              (success == GNUNET_OK) ? "SUCCESS" : "FAIL");
  /* This should not work */
  nsqe_alt = GNUNET_NAMESTORE_records_open (nsh_alt,
                                            &privkey,
                                            name,
                                            &open_alt_second_failed,
                                            NULL,
                                            &open_alt_second_cont,
                                            NULL);
}

static void
open_alt_failed (void *cls)
{
  struct GNUNET_GNSRECORD_Data rd;
  const char *name = "dummy";

  nsqe_alt = NULL;
  rd.expiration_time = GNUNET_TIME_absolute_get ().abs_value_us;
  rd.record_type = TEST_RECORD_TYPE;
  rd.data_size = TEST_RECORD_DATALEN;
  rd.data = GNUNET_malloc (TEST_RECORD_DATALEN);
  rd.flags = 0;
  memset ((char *) rd.data, 'a', TEST_RECORD_DATALEN);

  nsqe = GNUNET_NAMESTORE_records_commit (nsh,
                                          &privkey,
                                          name,
                                          1,
                                          &rd,
                                          &put_cont,
                                          (void *) name);

  GNUNET_free_nz ((void *) rd.data);

}


void
open_alt_cont (void *cls,
               const struct
               GNUNET_IDENTITY_PrivateKey *zone,
               const char *label,
               unsigned int rd_count,
               const struct GNUNET_GNSRECORD_Data *rd)
{
  GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
              "Got lock when I should not...\n");
  nsqe_alt = NULL;
  GNUNET_SCHEDULER_cancel (endbadly_task);
  endbadly_task = NULL;
  GNUNET_SCHEDULER_add_now (&endbadly, NULL);
}

void
open_cont (void *cls,
           const struct
           GNUNET_IDENTITY_PrivateKey *zone,
           const char *label,
           unsigned int rd_count,
           const struct GNUNET_GNSRECORD_Data *rd)
{
  const char *name = "dummy";
  /* Record set does not exist */
  GNUNET_assert (NULL == rd);
  nsqe = NULL;
  endbadly_task = GNUNET_SCHEDULER_add_delayed (TIMEOUT,
                                                &endbadly, NULL);

  /* This should not work */
  nsqe_alt = GNUNET_NAMESTORE_records_open (nsh_alt,
                                            &privkey,
                                            name,
                                            &open_alt_failed,
                                            NULL,
                                            &open_alt_cont,
                                            NULL);

}


static void
open_cont_failed (void *cls)
{
  nsqe = NULL;
  GNUNET_SCHEDULER_cancel (endbadly_task);
  endbadly_task = NULL;
  GNUNET_SCHEDULER_add_now (&endbadly, NULL);
}


static void
run (void *cls,
     const struct GNUNET_CONFIGURATION_Handle *cfg,
     struct GNUNET_TESTING_Peer *peer)
{
  const char *name = "dummy";
  privkey.type = htonl (GNUNET_GNSRECORD_TYPE_PKEY);
  GNUNET_CRYPTO_ecdsa_key_create (&privkey.ecdsa_key);
  GNUNET_IDENTITY_key_get_public (&privkey, &pubkey);


  nsh = GNUNET_NAMESTORE_connect (cfg);
  GNUNET_break (NULL != nsh);
  nsh_alt = GNUNET_NAMESTORE_connect (cfg);
  GNUNET_break (NULL != nsh_alt);

  nsqe = GNUNET_NAMESTORE_records_open (nsh,
                                        &privkey,
                                        name,
                                        &open_cont_failed,
                                        NULL,
                                        &open_cont,
                                        NULL);
  if (NULL == nsqe)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                _ ("Namestore cannot store no block\n"));
  }
}


#include "test_common.c"


int
main (int argc, char *argv[])
{
  const char *plugin_name;
  char *cfg_name;

  SETUP_CFG (plugin_name, cfg_name);
  res = 1;
  if (0 !=
      GNUNET_TESTING_peer_run ("test-namestore-api",
                               cfg_name,
                               &run,
                               NULL))
  {
    res = 1;
  }
  GNUNET_DISK_purge_cfg_dir (cfg_name,
                             "GNUNET_TEST_HOME");
  GNUNET_free (cfg_name);
  return res;
}


/* end of test_namestore_api_store.c */
