/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009, 2010, 2011  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gui.h"

/**
 * lw6gui_button_register_down
 *
 * @button: the button to update
 * @timestamp: the current ticks (milliseconds)
 *
 * Registers a "down" (press) event on a button.
 *
 * Return value: none.
 */
void
lw6gui_button_register_down (lw6gui_button_t * button, int64_t timestamp)
{
  button->is_pressed = 1;
  button->press_queue++;
  button->last_press = timestamp;
  button->last_repeat = 0;
}

/**
 * lw6gui_button_register_up
 *
 * @button: the button to update
 *
 * Registers a "up" (release) event on a button.
 *
 * Return value: none.
 */
void
lw6gui_button_register_up (lw6gui_button_t * button)
{
  button->is_pressed = 0;
  button->last_press = 0;
  button->last_repeat = 0;
}

/**
 * lw6gui_button_is_pressed
 *
 * @button: the button to query
 *
 * Tells wether a button is pressed or not.
 *
 * Return value: 1 if pressed, 0 if not.
 */
int
lw6gui_button_is_pressed (lw6gui_button_t * button)
{
  int ret = 0;

  ret = button->is_pressed;

  return ret;
}

/**
 * lw6gui_button_pop_press
 *
 * @button: the button to query
 *
 * Tells how many times the button has been pressed. Typical usage: the
 * button is pressed, released, pressed, released several times. Then,
 * after all this, you want to know how many times it has been pressed.
 * Querying its state with @lw6gui_button_is_pressed won't tell you much
 * but this @pop_press function will return 1 for each press there's been.
 *
 * Return value: 1 if there's a press event in the queue, 0 if empty.
 */
int
lw6gui_button_pop_press (lw6gui_button_t * button)
{
  int ret = 0;

  if (button->press_queue > 0)
    {
      ret = 1;
      button->press_queue--;
    }

  return ret;
}

/**
 * lw6gui_button_update_repeat
 *
 * @button: the button to update
 * @repeat_settings: the repeat settings (delay + interval)
 * @timestamp: the current ticks (milliseconds)
 *
 * Updates the repeat informations for a button, must be called
 * regularly, as often as possible.
 *
 * Return value: none.
 */
void
lw6gui_button_update_repeat (lw6gui_button_t * button,
			     lw6gui_repeat_settings_t * repeat_settings,
			     int64_t timestamp)
{
  if (button->is_pressed)
    {
      if ((repeat_settings->delay > 0 && button->last_repeat == 0
	   && button->last_press <= timestamp - repeat_settings->delay)
	  || (repeat_settings->interval > 0 && button->last_repeat != 0
	      && button->last_repeat <=
	      timestamp - repeat_settings->interval))
	{
	  (button->press_queue)++;
	  button->last_repeat = timestamp;
	}
    }
}

/**
 * lw6gui_button_sync
 *
 * @dst: the target button object
 * @src: the source button object
 *
 * Synchronizes two button objects. This is typically used to pass data from
 * one thread to another. This is not a simple copy, it will handle data
 * such as "when was it pressed last" it an intelligent manner, popping src
 * data to put it in dst, and clearing src.
 *
 * Return value: 1 if success, O if failure.
 */
int
lw6gui_button_sync (lw6gui_button_t * dst, lw6gui_button_t * src)
{
  int ret = 1;

  dst->press_queue += src->press_queue;
  src->press_queue = 0;		// clear src queue, it's been transfered to dst
  dst->is_pressed = src->is_pressed;
  dst->last_press = src->last_press;
  dst->last_repeat = src->last_repeat;

  return ret;
}
