/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  assertUnreachable,
  Duration,
  HttpStatusCode,
  LoginTokenRequest,
  LoginTokenScope,
  TalerError,
  TranslatedString,
} from "@gnu-taler/taler-util";
import {
  useChallengeHandler,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { AsyncButton } from "../../components/exception/AsyncButton.js";
import { NotificationCard } from "../../components/menu/index.js";
import { SolveMFAChallenges } from "../../components/SolveMFA.js";
import { useSessionContext } from "../../context/session.js";
import { Notification } from "../../utils/types.js";

interface Props {
  showCreateAccount?: boolean;
}

export const TEMP_TEST_TOKEN = (description: TranslatedString) =>
  ({
    scope: LoginTokenScope.All,
    duration: Duration.toTalerProtocolDuration(Duration.fromMilliseconds(100)),
    description,
  }) as LoginTokenRequest;

export const FOREVER_REFRESHABLE_TOKEN = (description: TranslatedString) =>
  ({
    scope: LoginTokenScope.All_Refreshable,
    duration: Duration.toTalerProtocolDuration(Duration.getForever()),
    description,
  }) as LoginTokenRequest;

const VERSION = typeof __VERSION__ !== "undefined" ? __VERSION__ : undefined;
export function LoginPage({ showCreateAccount }: Props): VNode {
  const [password, setPassword] = useState("");
  const [notif, setNotif] = useState<Notification | undefined>(undefined);
  const { state, logIn, getInstanceForUsername, config } = useSessionContext();
  const [username, setUsername] = useState(showCreateAccount? "" : state.instance);

  const { i18n } = useTranslationContext();

  const mfa = useChallengeHandler();
  const [doLogin, repeatLogin] = mfa.withMfaHandler(
    ({ challengeIds, onChallengeRequired }) =>
      async () => {
        const api = getInstanceForUsername(username);

        try {
          const result = await api.createAccessToken(
            username,
            password,
            FOREVER_REFRESHABLE_TOKEN(i18n.str`Logged in`),
            {
              challengeIds,
            },
          );
          if (result.type === "ok") {
            const { access_token: token } = result.body;
            logIn(username, token);
            return;
          } else {
            switch (result.case) {
              case HttpStatusCode.Unauthorized: {
                setNotif({
                  message: i18n.str`Your password is incorrect`,
                  type: "ERROR",
                });
                return;
              }
              case HttpStatusCode.NotFound: {
                setNotif({
                  message: i18n.str`Your instance cannot be found`,
                  type: "ERROR",
                });
                return;
              }
              case HttpStatusCode.Accepted: {
                onChallengeRequired(result.body);
                return;
              }
              default: {
                assertUnreachable(result);
              }
            }
          }
        } catch (error) {
          const details =
            error instanceof TalerError
              ? JSON.stringify(error.errorDetail)
              : undefined;
          setNotif({
            message: i18n.str`Failed to login.`,
            type: "ERROR",
            description: error instanceof Error ? error.message : undefined,
            details,
          });
        }
      },
  );

  if (mfa.pendingChallenge) {
    return (
      <SolveMFAChallenges
        currentChallenge={mfa.pendingChallenge}
        onCompleted={repeatLogin}
        onCancel={mfa.doCancelChallenge}
      />
    );
  }

  return (
    <Fragment>
      <NotificationCard notification={notif} />
      <div class="columns is-centered" style={{ margin: "auto" }}>
        <div class="column is-two-thirds ">
          <div class="modal-card" style={{ width: "100%", margin: 0 }}>
            <header
              class="modal-card-head"
              style={{ border: "1px solid", borderBottom: 0 }}
            >
              <p class="modal-card-title">
                <i18n.Translate>Login required</i18n.Translate>
              </p>
            </header>
            <section
              class="modal-card-body"
              style={{ border: "1px solid", borderTop: 0, borderBottom: 0 }}
            >
              <i18n.Translate>
                Please enter your password for <b>"{state.instance}"</b>.
              </i18n.Translate>

              <div class="field is-horizontal">
                <div class="field-label is-normal">
                  <label class="label">
                    <i18n.Translate>Username</i18n.Translate>
                  </label>
                </div>
                <div class="field-body">
                  <div class="field">
                    <p class="control is-expanded">
                      <input
                        class="input"
                        type="text"
                        placeholder={"instance name"}
                        name="username"
                        onKeyPress={(e) =>
                          e.keyCode === 13 ? doLogin() : null
                        }
                        value={username}
                        onInput={(e): void =>
                          setUsername(e?.currentTarget.value)
                        }
                      />
                    </p>
                  </div>
                </div>
              </div>
              <div class="field is-horizontal">
                <div class="field-label is-normal">
                  <label class="label">
                    <i18n.Translate>Password</i18n.Translate>
                  </label>
                </div>
                <div class="field-body">
                  <div class="field">
                    <p class="control is-expanded">
                      <input
                        class="input"
                        type="password"
                        placeholder={"current password"}
                        name="token"
                        onKeyPress={(e) =>
                          e.keyCode === 13 ? doLogin() : null
                        }
                        value={password}
                        onInput={(e): void =>
                          setPassword(e?.currentTarget.value)
                        }
                      />
                    </p>
                  </div>
                </div>
              </div>
            </section>
            <footer
              class="modal-card-foot "
              style={{
                justifyContent: "space-between",
                border: "1px solid",
                borderTop: 0,
              }}
            >
              {!config.have_self_provisioning ? (
                <div />
              ) : (
                <a
                  href={
                    !username || username === "admin"
                      ? undefined
                      : `#/account/reset/${username}`
                  }
                  class="button "
                  disabled={!username || username === "admin"}
                >
                  <i18n.Translate>Forgot password</i18n.Translate>
                </a>
              )}
              <AsyncButton disabled={!username || !password} onClick={doLogin}>
                <i18n.Translate>Confirm</i18n.Translate>
              </AsyncButton>
            </footer>
          </div>
          {!showCreateAccount ? undefined : (
            <div style={{ marginTop: 8 }}>
              <a href={"#/account/new"} class="has-icon">
                <span class="icon">
                  <i class="mdi mdi-account-plus" />
                </span>
                <span class="menu-item-label">
                  <i18n.Translate>Create new account</i18n.Translate>
                </span>
              </a>
            </div>
          )}
        </div>
      </div>
      <div
        style={{
          position: "absolute",
          bottom: 0,
          height: "2rem",
        }}
      >
        Version {VERSION}
      </div>
    </Fragment>
  );
}
