/*
 This file is part of GNU Taler
 (C) 2020-2025 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  PreparePayResultType,
  succeedOrThrow,
  TalerErrorCode,
  TalerMerchantInstanceHttpClient,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  createSimpleTestkudosEnvironmentV3,
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

/**
 * Run test where a wallet tries to claim an already claimed order.
 */
export async function runPaymentClaimTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, merchant, merchantAdminAccessToken } =
    await createSimpleTestkudosEnvironmentV3(t);

  const merchantClient = new TalerMerchantInstanceHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  const w2 = await createWalletDaemonWithClient(t, { name: "w2" });

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  await wres.withdrawalFinishedCond;

  // Set up order.

  const orderResp = succeedOrThrow(
    await merchantClient.createOrder(merchantAdminAccessToken, {
      order: {
        summary: "Buy me!",
        amount: "TESTKUDOS:5",
        fulfillment_url: "taler://fulfillment-success/thx",
      },
    }),
  );

  let orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id),
  );

  t.assertTrue(orderStatus.order_status === "unpaid");

  const talerPayUri = orderStatus.taler_pay_uri;

  // Make wallet pay for the order

  const preparePayResult = await walletClient.call(
    WalletApiOperation.PreparePayForUri,
    {
      talerPayUri,
    },
  );

  t.assertTrue(
    preparePayResult.status === PreparePayResultType.PaymentPossible,
  );

  const errOne = t.assertThrowsTalerErrorAsyncLegacy(
    w2.walletClient.call(WalletApiOperation.PreparePayForUri, {
      talerPayUri,
    })
  );

  console.log(errOne);

  await walletClient.call(WalletApiOperation.ConfirmPay, {
    transactionId: preparePayResult.transactionId,
  });

  // Check if payment was successful.

  orderStatus = succeedOrThrow(
    await merchantClient.getOrderDetails(merchantAdminAccessToken, orderResp.order_id),
  );

  t.assertTrue(orderStatus.order_status === "paid");

  await w2.walletClient.call(WalletApiOperation.ClearDb, {});

  const err = await t.assertThrowsTalerErrorAsyncLegacy(
    w2.walletClient.call(WalletApiOperation.PreparePayForUri, {
      talerPayUri,
    })
  );

  t.assertTrue(err.hasErrorCode(TalerErrorCode.WALLET_ORDER_ALREADY_CLAIMED));

  await t.shutdown();
}

runPaymentClaimTest.suites = ["wallet"];
