/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008, 2009, 2010, 2011, 2012, 2013, 2014  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mat.h"

#include <math.h>

/**
 * lw6mat_dvec4_zero
 *
 * @dvec4: the vector to initialize.
 *
 * Fills the vector with zeros, regardless of what was there before.
 * Internally, does a memset the only advantage is that this function
 * should use the right sizeof and therefore avoids typo errors.
 *
 * Return value: none.
 */
void
lw6mat_dvec4_zero (lw6mat_dvec4_t * dvec4)
{
  memset (dvec4, 0, sizeof (lw6mat_dvec4_t));
}

/**
 * lw6mat_dvec4_is_same
 *
 * @dvec4_a: 1st vector to compare
 * @dvec4_b: 2nd vector to compare
 *
 * Compares two vectors, returns true if they are equal.
 *
 * Return value: 1 if equal, 0 if different.
 */
int
lw6mat_dvec4_is_same (const lw6mat_dvec4_t * dvec4_a,
		      const lw6mat_dvec4_t * dvec4_b)
{
  return (!memcmp
	  ((void *) dvec4_a, (void *) dvec4_b, sizeof (lw6mat_dvec4_t)));
}

/**
 * lw6mat_dvec4_len_sq
 *
 * @dvec4: the vector to query.
 *
 * Returns the square of a vector length. To get the real length
 * one should then apply a square root but at this stage one has
 * at least an idea about vector length, and this information is
 * enough to compare them.
 *
 * Return value: sigma(coord*coord)
 */
double
lw6mat_dvec4_len_sq (const lw6mat_dvec4_t * dvec4)
{
  return dvec4->p.x * dvec4->p.x + dvec4->p.y * dvec4->p.y +
    dvec4->p.z * dvec4->p.z + dvec4->p.w * dvec4->p.w;
}

/**
 * lw6mat_dvec4_len_sq
 *
 * @dvec4: the vector to query.
 *
 * Returns the size/length of a vector, this is the distance
 * of the point from origin, not the number of elements.
 *
 * Return value: the length of the vector.
 */
double
lw6mat_dvec4_len (const lw6mat_dvec4_t * dvec4)
{
  return sqrt (lw6mat_dvec4_len_sq (dvec4));
}

/**
 * Normalizes a vector
 *
 * @dvec4: the vector to normalize.
 *
 * Normalizes a vector, that is, make its length be 1.
 *
 * Return value: 1 if OK, 0 if error, such as trying to normalize vector zero.
 */
int
lw6mat_dvec4_norm (lw6mat_dvec4_t * dvec4)
{
  double len = lw6mat_dvec4_len (dvec4);

  if (len)
    {
      dvec4->p.x /= len;
      dvec4->p.y /= len;
      dvec4->p.z /= len;
      dvec4->p.w /= len;

      return 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _x_ ("trying to normalize vector zero"));

      return 0;
    }
}

/**
 * lw6mat_dvec4_neg
 *
 * @dvec4: vector to modify
 *
 * Calcs the opposite vector, by making a negation on all its members
 *
 * Return value: none
 */
void
lw6mat_dvec4_neg (lw6mat_dvec4_t * dvec4)
{
  dvec4->p.x = -dvec4->p.x;
  dvec4->p.y = -dvec4->p.y;
  dvec4->p.z = -dvec4->p.z;
  dvec4->p.w = -dvec4->p.w;
}

/**
 * lw6mat_dvec4_add
 *
 * @dvec4: result vector
 * @dvec4_a: 1st vector to add
 * @dvec4_b: 2nd vector to add
 *
 * Adds two vectors.
 *
 * Return value: none
 */
void
lw6mat_dvec4_add (lw6mat_dvec4_t * dvec4, const lw6mat_dvec4_t * dvec4_a,
		  const lw6mat_dvec4_t * dvec4_b)
{
  dvec4->p.x = dvec4_a->p.x + dvec4_b->p.x;
  dvec4->p.y = dvec4_a->p.y + dvec4_b->p.y;
  dvec4->p.z = dvec4_a->p.z + dvec4_b->p.z;
  dvec4->p.w = dvec4_a->p.w + dvec4_b->p.w;
}

/**
 * lw6mat_dvec4_sub
 *
 * @dvec4: result vector
 * @dvec4_a: 1st vector
 * @dvec4_b: 2nd vector, will be substracted to 1st vector
 *
 * Substracts vector b from vector a.
 *
 * Return value: none
 */
void
lw6mat_dvec4_sub (lw6mat_dvec4_t * dvec4, const lw6mat_dvec4_t * dvec4_a,
		  const lw6mat_dvec4_t * dvec4_b)
{
  dvec4->p.x = dvec4_a->p.x - dvec4_b->p.x;
  dvec4->p.y = dvec4_a->p.y - dvec4_b->p.y;
  dvec4->p.z = dvec4_a->p.z - dvec4_b->p.z;
  dvec4->p.w = dvec4_a->p.w - dvec4_b->p.w;
}

/**
 * lw6mat_dvec4_dot
 *
 * @dvec4_a: 1st vector
 * @dvec4_b: 2nd vector
 *
 * Calculates the dot AKA scalar product of the two vectors.
 *
 * Return value: none
 */
double
lw6mat_dvec4_dot (const lw6mat_dvec4_t * dvec4_a,
		  const lw6mat_dvec4_t * dvec4_b)
{
  return dvec4_a->p.x * dvec4_b->p.x + dvec4_a->p.y * dvec4_b->p.y +
    dvec4_a->p.z * dvec4_b->p.z + dvec4_a->p.w * dvec4_b->p.w;
}

/**
 * lw6mat_dvec4_cross
 *
 * @dvec3: result vector
 * @dvec4_a: 1st vector
 * @dvec4_b: 2nd vector
 *
 * Calculates the cross AKA vectorial product of the two vectors.
 * Since cross product only really makes sense in 3D, this function
 * will interpret the 4D vectors as 3D vectors only, ignoring
 * the last value.
 *
 * Return value: none
 */
void
lw6mat_dvec4_cross (lw6mat_dvec3_t * dvec3, const lw6mat_dvec4_t * dvec4_a,
		    const lw6mat_dvec4_t * dvec4_b)
{
  dvec3->p.x = dvec4_a->p.y * dvec4_b->p.z - dvec4_a->p.z * dvec4_b->p.y;
  dvec3->p.y = dvec4_a->p.z * dvec4_b->p.x - dvec4_a->p.x * dvec4_b->p.z;
  dvec3->p.z = dvec4_a->p.x * dvec4_b->p.y - dvec4_a->p.y * dvec4_b->p.x;
}

/**
 * lw6mat_dvec4_scale
 *
 * @dvec4: vector to modify
 * @f: scale factor
 *
 * Scales the vector by multiplying all its members by a scalar value.
 *
 * Return value: none
 */
void
lw6mat_dvec4_scale (lw6mat_dvec4_t * dvec4, double f)
{
  dvec4->p.x *= f;
  dvec4->p.y *= f;
  dvec4->p.z *= f;
  dvec4->p.w *= f;
}
