/* msqlfuncs.c: -*- C -*-  Functions for manipulating MSQL databases. */

/*  Copyright (c) 1996 Brian J. Fox, Henry Minsky
    Author: Brian J. Fox (bfox@ai.mit.edu) Wed Jun 26 07:27:10 1996.
            Henry Minsky (hqm@ua.com)

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

#define LANGUAGE_DEFINITIONS_FILE 1

#if defined (HAVE_CONFIG_H)
#  include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <regex.h>
#include <setjmp.h>
#include <sys/types.h>
#include <time.h>
#if defined (Solaris)
#  include <ucbinclude/sys/fcntl.h>
#  include <ucbinclude/sys/file.h>
#else
#  include <sys/file.h>
#endif /* !Solaris */

#include <msql.h>
#include <locking.h>
#include <bprintf/bprintf.h>
#include <xmalloc/xmalloc.h>
#include <wisper/wisp.h>
#include "forms.h"
#include "pages.h"
#include "parser.h"

extern jmp_buf page_jmp_buffer;

#if defined (macintosh)
extern char *strdup (const char *string);
#  include <sys/fcntl.h>
#  define os_open(name, flags, mode) open (name, flags)
#else
#  define os_open(name, flags, mode) open (name, flags, mode)
#endif

static void pf_with_open_database (PFunArgs);
static void pf_database_query (PFunArgs);
static void pf_database_exec_query (PFunArgs);
static void pf_database_next_record (PFunArgs);
static void pf_host_databases (PFunArgs);
static void pf_database_tables (PFunArgs);
static void pf_database_table_fields (PFunArgs);
static void pf_database_num_rows (PFunArgs);
static void pf_database_set_pos (PFunArgs);
static void pf_database_save_record (PFunArgs);
static void pf_database_load_record (PFunArgs);
static void pf_database_delete_record (PFunArgs);
static void pf_database_save_package (PFunArgs);

PACKAGE_INITIALIZER (initialize_msql_functions)

PFunDesc msqlfunc_table[] =
{
  { "MSQL::WITH-OPEN-DATABASE",		1, 0, pf_with_open_database },
  { "MSQL::DATABASE-EXEC-QUERY",	0, 0, pf_database_exec_query },
  { "MSQL::DATABASE-NEXT-RECORD",	0, 0, pf_database_next_record },
  { "MSQL::DATABASE-SAVE-RECORD",	0, 0, pf_database_save_record },
  { "MSQL::DATABASE-LOAD-RECORD",	0, 0, pf_database_load_record },
  { "MSQL::DATABASE-DELETE-RECORD",	0, 0, pf_database_delete_record },
  { "MSQL::DATABASE-SAVE-PACKAGE",	0, 0, pf_database_save_package },
  { "MSQL::NUMBER-OF-ROWS",   	        0, 0, pf_database_num_rows},
  { "MSQL::SET-ROW-POSITION",  	        0, 0, pf_database_set_pos},
  { "MSQL::DATABASE-QUERY",		0, 0, pf_database_query },
  { "MSQL::HOST-DATABASES",	        0, 0, pf_host_databases },
  { "MSQL::DATABASE-TABLES",     	0, 0, pf_database_tables },
  { "MSQL::DATABASE-TABLE-FIELDS",      0, 0, pf_database_table_fields },
  { (char *)NULL,			0, 0, (PFunHandler *)NULL }
};

static void initialize_database_stack (void);

void
initialize_msql_functions (Package *package)
{
  register int i;
  Symbol *sym;

  for (i = 0; msqlfunc_table[i].tag != (char *)NULL; i++)
    {
      sym = symbol_intern_in_package (package, msqlfunc_table[i].tag);
      sym->type = symtype_FUNCTION;
      sym->values = (char **)(&msqlfunc_table[i]);
    }

  initialize_database_stack ();
}

/* A function which returns void and takes a pointer to void. */
typedef void FREEFUN (void *);

/* A stack which is implemented with a growable array. */
typedef struct
{
  void **data;
  int index;
  int slots;
} IStack;

static IStack *
make_istack (void)
{
  register int i;

  IStack *stack = (IStack *) xmalloc (sizeof (IStack));
  stack->slots = 10;
  stack->index = 0;
  stack->data = (void **) xmalloc (sizeof (void *) * stack->slots);

  for (i = 0; i < stack->slots; i++)
    stack->data[i] = (void *)NULL;

  return (stack);
}

#if 0
static int istack_emptyp (IStack *stack)
{
  return (stack->index == 0);
}
#endif

/* Returns the index of the pushed item. */
static int
istack_push (IStack *stack, void *data)
{
  if (stack->index + 2 > stack->slots)
    stack->data = (void **)xrealloc
      (stack->data, ((stack->slots += 5) * sizeof (void *)));

  stack->data[stack->index++] = data;
  stack->data[stack->index] = (void *)NULL;
  return (stack->index - 1);
}

void *
istack_aref (IStack *stack, int i)
{
  if ((i > -1) && (i < stack->index))
    return (stack->data[i]);
  else
    return ((void *)NULL);
}

void *
istack_pop (IStack *stack)
{
  void *data = (void *)NULL;

  if (stack->index > 0)
    {
      data = stack->data[--(stack->index)];
      stack->data[stack->index] = (void *)NULL;
    }

  return (data);
}

/* This iterates over the elements of stack, calling
   the free_fun on each one. Then it frees the stack
   itself. */
static void
istack_free (IStack *stack, FREEFUN *free_fun)
{
  register int i;

  for (i = 0; i < stack->index; i++)
    (*free_fun) (stack->data[i]);

  free (stack);
}

/****************************************************************
 * Cursors and Databases
 ****************************************************************/

/****************************************************************
 * The Database object:
 *
 * contains a stack of cursors, and information about the open
 * connection.
 ****************************************************************/

typedef struct
{
  char *dbname;
  char *hostname;
  int sock;
  IStack *cursors;
} Database;

/* For MSQL, a cursor points to a m_result object. */
typedef struct
{
  m_result *result;		/* The results of <database-exec-query..> */
  Database *db;			/* Associated database connection. */
} DBCursor;

static void free_cursor (DBCursor *cursor);

static Database *
make_database (void)
{
  Database *db = (Database *) xmalloc (sizeof (Database));
  db->cursors = make_istack ();
  return (db);
}

static IStack *open_databases = (IStack *)NULL;

static void
initialize_database_stack (void)
{
  open_databases = make_istack ();
}

/* Install this database, and return the index which is used to identify it. */
static int
add_open_database (int sock, char *dbname, char *hostname)
{
  Database *dbhandle = make_database ();
  int result;

  dbhandle->dbname = strdup (dbname);
  dbhandle->hostname = strdup (hostname);
  dbhandle->sock = sock;

  result = istack_push (open_databases, (void *) dbhandle);

  return (result);
}

static Database *
get_database (int i)
{
  Database *db = (Database *) istack_aref (open_databases, i);
  return (db);
}

/* Pop each cursor off the cursor's istack, and free it.
   Then free the istack itself. */
static void
free_database_cursors (Database *db)
{
  if (db->cursors)
    istack_free (db->cursors, (FREEFUN *) free_cursor);
}

/* De-install this current db which is at the top of the stack.
   Free up any cursor resources, then free the Database object.

   Note: This does not close the database connection. That is
   only done by unlock_database. */
static void
pop_database (void)
{
  Database *database = (Database *) istack_pop (open_databases);

  if (database != (Database *) NULL)
    {
      if (database->dbname) free (database->dbname);
      if (database->hostname) free (database->hostname);

      free_database_cursors (database);
      free (database);
    }
}

/* For a given open database, a list of cursors is kept.
   A cursor is encoded as a 16 bit integer whose high
   byte indexes into the database list, and the low byte
   indexes into the list of cursors for that database. */
static DBCursor *
make_cursor (Database *db)
{
  DBCursor *cursor = (DBCursor *) xmalloc (sizeof (DBCursor));
  cursor->db = db;
  return (cursor);
}

/* Returns the (MSQL-specific) handle to a database connection. */
static int
cursor_database_connection (DBCursor *cursor)
{
  return (cursor->db->sock);
}

static void
free_cursor (DBCursor *cursor)
{
  if (cursor && cursor->result)
    msqlFreeResult (cursor->result);
}

/* Dereferences a Meta-HTML variable which indexes a cursor object.
   Takes the high byte, and computes an index into the open databases,
   and then uses the low byte to index into that database's cursor list. */
static DBCursor *
get_cursor_ref (Package *vars)
{
  char *cursor_ref = page_evaluate_string (get_positional_arg (vars, 0));
  Database * db;
  int cursor_token = -1;		/* The combined db::cursor ids */
  int cursor_id;
  int db_id;

  /* Extract the token from the variable string value. */
  if (!empty_string_p (cursor_ref))
    {
      char *rep = pagefunc_get_variable (cursor_ref);
      char *end;

      if (!empty_string_p (rep))
	{
	  /* Get high byte of the token */
	  cursor_token = (int)strtol (rep, &end, 16);
	  if (*end != '\0')
	    cursor_token = -1;
	}
    }

  if (cursor_ref != (char *)NULL) free (cursor_ref);

  if (cursor_token == -1)
    return ((DBCursor *) NULL);

  db_id = (cursor_token & 0xFF00) >> 8;
  cursor_id = (cursor_token & 0xFF);

  db = get_database (db_id);

  if (db != (Database *) NULL)
    {
      /* OK, we found an open database.  Now find the cursor in
	 that database's cursor list. */
      return ((DBCursor *)istack_aref (db->cursors, cursor_id));
    }
  else
    return ((DBCursor *) NULL);
}

/************************************************************/
/*							    */
/*		Database Manipulation Functions		    */
/*							    */
/************************************************************/

static int database_environment_level = 0;

static char *
db_lockname (char *dbname, char *hostname)
{
  static char lockname[1024];

  if ((dbname == (char *)NULL) || (strlen (dbname) > 1018))
    return ((char *)NULL);

  strcpy (lockname, dbname);
  strcat (lockname, "-");
  strcat (lockname, hostname);
  strcat (lockname, ".LCK");
  return (lockname);
}

static void
lock_database (char *dbname, char *host, int *lock, int *db)
{
  char *lockname = db_lockname (dbname, host);
  int fd;

  *lock = -1;
  *db = -1;

  fd = os_open (lockname, O_CREAT | O_WRONLY | O_APPEND, 0666);

  if ((fd < 0) || (LOCKFILE (fd) == -1))
    {
      if (fd >= 0)
	{
	  char pid_name[100];
	  sprintf (pid_name, "%ld\n", (long)getpid ());
	  write (fd, (void *)pid_name, (size_t) strlen (pid_name));
	  close (fd);
	}
      return;
    }
  else
    {
      int sock = msqlConnect (host);

      if ((sock < 0) || (msqlSelectDB (sock, dbname) < 0))
	{
	  unlink (lockname);
	  UNLOCKFILE (fd);
	  close (fd);
	}
      else
	{
	  *lock = fd;
	  *db = sock;
	}
    }
}

static void
unlock_database (char *dbname, char *hostname, int *lock, int *db)
{
  if (*db > -1)
    msqlClose (*db);

  if (*lock > -1)
    {
      char *lockname = db_lockname (dbname, hostname);
      unlink (lockname);
      UNLOCKFILE (*lock);
      close (*lock);
    }
}

/* Translates a Meta-HTML db token (a small int) into an index into
   the table of open databases, which returns a Database object.

   The Database object contains an MSQL db handle (which happens to be
   an int which represents a socket).  */
static int
get_dbref_internal (Package *vars, int *dbindex)
{
  char *dbref = page_evaluate_string (get_positional_arg (vars, 0));
  Database *db = (Database *)NULL;

  *dbindex = -1;

  if (!empty_string_p (dbref))
    {
      char *rep = pagefunc_get_variable (dbref);
      char *end;

      if (!empty_string_p (rep))
	{
	  *dbindex = strtol (rep, &end, 16);
	  if (*end != '\0')
	    *dbindex = -1;
	}
    }

  if (dbref != (char *)NULL) free (dbref);

  if (*dbindex != -1)
    db = get_database (*dbindex);

  if (db != (Database *)NULL)
    return (db->sock);
  else
    return (-1);
}

static int
get_dbref (Package *vars)
{
  int dbindex;
  return (get_dbref_internal (vars, &dbindex));
}

/* <with-open-database variable DATABASE=dbname HOST=hostname>
   [code using the open database]
   </with-open-database>
   Opens the database specified by DBNAME, and stores a referent
   to it in VARIABLE.  If the db-connect fails, value of VARIABLE is
   the empty string.*/
static void
pf_with_open_database (PFunArgs)
{
  char *varname  = page_evaluate_string (get_positional_arg (vars, 0));
  char *dbname   = page_evaluate_string (get_value (vars, "DATABASE"));
  char *hostname = page_evaluate_string (get_value (vars, "HOST"));
  int jump_again = 0;

  /* No errors yet! */
  pagefunc_set_variable ("msql::msql-error-message[]", "");

  if (empty_string_p (hostname))
    {
      if (hostname != (char *)NULL) free (hostname);
      hostname = strdup ("localhost");
    }

  if ((!empty_string_p (varname)) && (!empty_string_p (dbname)))
    {
      int dbsock;
      int lock;

      lock_database (dbname, hostname,  &lock, &dbsock);

      if (dbsock != -1)
	{
	  char dbvalue[40];
	  PAGE *body_code = page_copy_page (body);
	  int dbindex;

	  /* Encode pointer to this database as an index into the table
	     of open databases. */
	  dbindex = add_open_database (dbsock, dbname, hostname);
	  sprintf (dbvalue, "%x", dbindex);

	  pagefunc_set_variable (varname, dbvalue);

	  {
	    PageEnv *page_environ;

	    page_environ = pagefunc_save_environment ();
	    database_environment_level++;

	    if ((jump_again = setjmp (page_jmp_buffer)) == 0)
	      page_process_page_internal (body_code);

	    database_environment_level--;
	    pagefunc_restore_environment (page_environ);

	    if (jump_again && (body_code != (PAGE *)NULL))
	      {
		page_free_page (body_code);
		body_code = (PAGE *)NULL;
	      }
	  }

	  if (body_code != (PAGE *)NULL)
	    {
	      if (body_code->buffer != (char *)NULL)
		{
		  bprintf_insert (page, start, "%s", body_code->buffer);
		  *newstart = start + (body_code->bindex);
		}

	      page_free_page (body_code);
	    }
	}
      else
	{
	  pagefunc_set_variable ("msql::msql-error-message[]", msqlErrMsg);
	}

      unlock_database (dbname, hostname, &lock, &dbsock);
      pop_database ();
    }

  if (dbname != (char *)NULL)   free (dbname);
  if (varname != (char *)NULL)  free (varname);
  if (hostname != (char *)NULL) free (hostname);

  if (jump_again)
    longjmp (page_jmp_buffer, 1);
}

/* <database-exec-query dbvar query-string cursor=cursor>
   Executes an MSQL query.
   Returns a CURSOR object. */
static void
pf_database_exec_query (PFunArgs)
{
  m_result *query_result;
  int dbindex;

  if (database_environment_level != 0)
    {
      int db = get_dbref_internal (vars, &dbindex);

      if (db != -1)
	{
	  char *dbquery  = page_evaluate_string (get_positional_arg (vars, 1));

	  if (dbquery != (char *)NULL)
	    {
	      if (msqlQuery (db, dbquery) < 0)
		{
		  pagefunc_set_variable ("msql::msql-error-message[]",
					 msqlErrMsg);
		}
	      else
		{
		  DBCursor *cursor;
		  int cursor_index;
		  Database *dbobj;
		  int cursor_token;
		  char cursor_string[16];

		  query_result = msqlStoreResult ();

		  /* Get the Database object  associated with the input arg */
		  dbobj = get_database (dbindex);

		  /* Make a cursor for this result */
		  cursor = make_cursor (dbobj);
		  cursor->result = query_result;

		  /* Push the cursor on the databases's cursor stack. */
		  cursor_index = istack_push (dbobj->cursors, (void *) cursor);

		  /* Make a token of db::cursor id and print it. */
		  cursor_token = ((dbindex & 0xff) << 8) | (cursor_index & 0xff);

		  sprintf (cursor_string, "%x", cursor_token);
		  bprintf_insert (page, start, "%s", cursor_string);
		  *newstart += strlen (cursor_string);
		}

	      free (dbquery);
	    }
	}
    }
}

/* Returns the number of rows which were returned from the last query.
   This operates on a CURSOR object.
   <msql::number-of-rows CURSOR> */
static void
pf_database_num_rows (PFunArgs)
{
  DBCursor *cursor = get_cursor_ref (vars);

  cursor = get_cursor_ref (vars);

  if ((cursor != (DBCursor *)NULL) && (cursor->result != (m_result *)NULL))
    {
      int nrows = msqlNumRows (cursor->result);

      bprintf_insert (page, start, "%d", nrows);
    }
}

/* Set the current row number position in the result set from the last query.
   This operates on a CURSOR object.

   <msql::set-row-position CURSOR n> */
static void
pf_database_set_pos (PFunArgs)
{
  DBCursor *cursor = get_cursor_ref (vars);
  long position = -1;

  if (cursor != (DBCursor *)NULL)
    {
      char *index_ref =  page_evaluate_string (get_positional_arg (vars, 1));

      /* Get the position argument. */
      if (!empty_string_p (index_ref))
	{
	  char *rep = index_ref;
	  char *ends;

	  if (!empty_string_p (rep))
	    {
	      /* Get high byte of the token */
	      position = strtol (rep, &ends, 10);
	      if (*ends != '\0')
		{
		  if (debug_level)
		    page_debug ("Invalid position in set-row-position");
		  position = -1;
		}
	    }
	}

      if (index_ref != (char *)NULL) free (index_ref);

      if ((position > -1) && (cursor->result != (m_result *)NULL))
	msqlDataSeek (cursor->result, position);
    }
}

/* <database-next-record cursor [package=PKGNAME]>
   This should be called after a call to database-exec-query.
   It fetches the next result row, and binds all the fields of the
   result row to variables in the current package, or in PKGNAME
   if specified.
   Return "true" if successful, or "" if not. */
static void
pf_database_next_record (PFunArgs)
{
  char *pkgname;
  m_result *query_result;
  DBCursor *cursor;
  int db;

  pkgname = page_evaluate_string
    (get_one_of (vars, "PACKAGE", "PREFIX", (char *)NULL));

  if (database_environment_level != 0)
    {
      cursor = get_cursor_ref (vars);
      if (cursor == (DBCursor *) NULL)
	return;

      query_result = cursor->result;
      db = cursor_database_connection (cursor);

      if ((db != -1) && (query_result != (m_result *)NULL))
	{
	  m_row msql_row = msqlFetchRow (query_result);

	  if (msql_row == (m_row)NULL)
	    {
	      /* End of data has been reached. */
	    }
	  else
	    {
	      /* Loop over fields of the row, binding variables.
		 In the case of MSQL, everything is a string, so
		 we don't need to worry about converting INT or
		 REAL to string datatypes */
	      register int i;

	      /* Reset the field descriptor cursor to start of record */
	      msqlFieldSeek (query_result, 0);

	      for (i = 0; i < msqlNumFields (query_result); i++)
		{
		  BPRINTF_BUFFER *varname = bprintf_create_buffer ();
		  m_field *field = msqlFetchField (query_result);

		  if (empty_string_p (pkgname))
		    bprintf (varname, "%s", field->name);
		  else
		    bprintf (varname, "%s::%s", pkgname, field->name);

		  pagefunc_set_variable (varname->buffer, msql_row[i]);

		  bprintf_free_buffer (varname);
		}

	      bprintf_insert (page, start, "true");
	      *newstart += 4;
	    }
	}
    }

  if (pkgname != (char *)NULL) free (pkgname);
}

static void
array_to_symbol (char **array, char *symbol_name)
{
  Symbol *sym;

  if ((sym = symbol_remove (symbol_name)) != (Symbol *)NULL)
    symbol_free (sym);

  /* Only assign when there really is anything to assign. */
  if (array != (char **)NULL)
    {
      register int i;

      for (i = 0; array[i] != (char *)NULL; i++);

      sym = symbol_intern (symbol_name);
      sym->values = array;
      sym->values_index = i;
      sym->values_slots = i;
    }
}

/* List all the table names in this database. This must be done
   with a database already open, i.e., within the scope of a
   <with-open-database>.

   Args: <msql::database-tables dbvar [result=VAR]>
   Returns resulting array, or places it into VAR if supplied. */
static void
pf_database_tables (PFunArgs)
{
  if (database_environment_level != 0)
    {
      int db = get_dbref (vars);

      if (db != -1)
	{
	  char *resultvar = page_evaluate_string (get_value (vars, "RESULT"));
	  m_result *result = msqlListTables (db);

	  if (result != (m_result *)NULL)
	    {
	      int rows = msqlNumRows (result);
	      char **tablenames;
	      m_row row;
	      int count = 0;

	      tablenames = (char **)xmalloc ((rows + 1) * sizeof (char *));

	      /* Loop over rows returned; the table name will be passed in
		 the first field of each 'row'.  Add names to the result
		 array.  */
	      while ((row = msqlFetchRow (result)) != (m_row)NULL)
		tablenames[count++] = strdup (row[0]);

	      tablenames[count] = (char *)NULL;

	      if (!empty_string_p (resultvar))
		{
		  array_to_symbol (tablenames, resultvar);
		}
	      else
		{
		  register int i;

		  for (i = 0; tablenames[i] != (char *)NULL; i++)
		    {
		      bprintf_insert (page, start, "%s\n", tablenames[i]);
		      start += 1 + strlen (tablenames[i]);
		      free (tablenames[i]);
		    }
		  free (tablenames);

		  *newstart = start;
		}
	      if (result != (m_result *)NULL) msqlFreeResult (result);
	    }
	}
    }
}

/* List all the fieldnames for a table. This must be done
   with a database already open, i.e., within the scope of a
   <msql::with-open-database>.

   Args: <msql::database-table-fields dbvar tablename [result=VAR]>
   Returns an array of column names, or sets the value of VAR
   if supplied. */
static void
pf_database_table_fields (PFunArgs)
{
  /* No errors yet! */
  pagefunc_set_variable ("msql::msql-error-message[]", "");

  if (database_environment_level != 0)
    {
      int db = get_dbref (vars);

      if (db != -1)
	{
	  char *tablename;

	  /* The name of the table we are enquiring about. */
	  tablename = page_evaluate_string (get_positional_arg (vars, 1));

	  if (!empty_string_p (tablename))
	    {
	      m_result *result = msqlListFields (db, tablename);

	      if (result != (m_result *)NULL)
		{
		  register int i;
		  /* m_row msql_row = msqlFetchRow (result); */
		  int cols = msqlNumFields (result);
		  char *varname;
		  char **colnames;

		  /* The name of the variable to stuff the results into. */
		  varname = page_evaluate_string (get_value (vars, "RESULT"));
		  colnames = (char **)xmalloc ((cols + 1) * sizeof (char *));

		  /* Reset the field descriptor cursor to start of record */
		  msqlFieldSeek (result, 0);

		  for (i = 0; i < cols; i++)
		    {
		      m_field *field = msqlFetchField (result);

		      colnames[i] = strdup (field->name);
		    }
		  colnames[i] = (char *) NULL;

		  if (!empty_string_p (varname))
		    {
		      array_to_symbol (colnames, varname);
		    }
		  else
		    {
		      for (i = 0; i < cols; i++)
			{
			  bprintf_insert (page, start, "%s\n", colnames[i]);
			  start += 1 + strlen (colnames[i]);
			  free (colnames[i]);
			}
		      free (colnames);

		      *newstart = start;
		    }

		  if (varname != (char *)NULL) free (varname);
		  msqlFreeResult (result);
		}
	    }
	  if (tablename != (char *)NULL) free (tablename);
	}
    }
}

/* <msql::host-databases [hostname] [result=varname]>
   Returns an array of the databases available on HOST.
   If VARNAME is supplied, the array is placed into that variable instead. */
static void
pf_host_databases (PFunArgs)
{
  char *host = page_evaluate_string (get_positional_arg (vars, 0));
  char *resultvar = page_evaluate_string (get_value (vars, "result"));
  int sock;

  /* No errors yet! */
  pagefunc_set_variable ("msql::msql-error-message[]", "");

  if (empty_string_p (host))
    {
      if (host != (char *)NULL) free (host);
      host = strdup ("localhost");
    }

  if ((sock = msqlConnect (host)) > -1)
    {
      m_result *result = msqlListDBs (sock);
      int nrows = (result ? msqlNumRows (result) : 0);

      if (nrows != 0)
	{
	  int count = 0;
	  char **dbnames = (char **) xmalloc ((nrows + 1) * sizeof (char *));
	  m_row msql_row;

	  /* Loop over rows returned; the db name will be passed in the first
	     field of each 'row'.  Add names to the result array.  */
	  while ((msql_row = msqlFetchRow (result)) != (m_row)NULL)
	    dbnames[count++] = strdup (msql_row[0]);

	  dbnames[count] = (char *) NULL;

	  if (!empty_string_p (resultvar))
	    {
	      array_to_symbol (dbnames, resultvar);
	    }
	  else
	    {
	      register int i;

	      for (i = 0; dbnames[i] != (char *)NULL; i++)
		{
		  bprintf_insert (page, start, "%s\n", dbnames[i]);
		  start += 1 + strlen (dbnames[i]);
		  free (dbnames[i]);
		}
	      free (dbnames);
	      *newstart = start;
	    }
	}

      if (result != (m_result *)NULL) msqlFreeResult (result);
      msqlClose (sock);
    }
  else
    {
      pagefunc_set_variable ("msql::msql-error-message[]", msqlErrMsg);
    }

  if (host != (char *)NULL) free (host);
  if (resultvar != (char *)NULL) free (resultvar);
}

/* Read the record in DB at KEY, and return a package containing the
   fields and values. */
static Package *
package_from_row (m_result *query_result, m_row row)
{
  register int i;
  int num_fields = 0;
  Package *package = (Package *)NULL;

  package = symbol_get_package ((char *)NULL);

  /* Loop over fields of the row, binding variables. I
     think everything is a string, so we don't need to
     worry about converting INT or REAL to string datatypes. */

  /* Reset the field descriptor cursor to start of record */
  msqlFieldSeek (query_result, 0);
  num_fields = msqlNumFields (query_result);

  for (i = 0; i < num_fields; i++)
    {
      m_field *field;

      field = msqlFetchField (query_result);
      forms_set_tag_value_in_package (package, field->name, row[i]);
    }

  return (package);
}

/* <database-query db expr query=query-string [format=<expr>] [keys=varname]>
   Select and optionally format records in the database according
   to the SQL query QUERY-STRING.   If the result of the query is
   not empty, and if FORMAT is present, it is an expression to evaluate
   in the context of the database fields.

   We might add an argument to specify which column to use as the "key"
   column, for compatibility with the keys=... option in the GDBM database
   interface. */
typedef struct
{
  char *key;
  Package *contents;
} DBRecord;

static void
pf_database_query (PFunArgs)
{
  int db = get_dbref (vars);

  if ((database_environment_level == 0) || (db == -1))
    return;
  else
    {
      register int i;
      char *query = page_evaluate_string (get_value (vars, "QUERY"));
      static m_result *query_result = (m_result *)NULL;

      /* Execute the query. */
      if (!empty_string_p (query))
	{
	  msqlQuery (db, query);
	  query_result = msqlStoreResult ();
	}

      /* If anything was found, deal with it now. */
      if (query_result != (m_result *)NULL)
	{
	  char *expr = get_positional_arg (vars, 1);
	  Package *save = CurrentPackage;
	  DBRecord **records = (DBRecord **)NULL;
	  int rec_slots = 0;
	  int rec_index = 0;
	  int search_limit = -1;

	  {
	    char *search_limit_string =
	      page_evaluate_string (get_value (vars, "search-limit"));

	    if (!empty_string_p (search_limit_string))
	      {
		search_limit = atoi (search_limit_string);
		if (search_limit == 0) search_limit = -1;
	      }

	    if (search_limit_string)
	      free (search_limit_string);
	  }

	  while ((search_limit < 0) || (rec_index < search_limit))
	    {
	      m_row row = msqlFetchRow (query_result);
	      Package *db_fields;

	      /* If at end of data, we're done with this loop. */
	      if (row == (m_row)NULL)
		break;

	      /* Otherwise, get the field data. */
	      db_fields = package_from_row (query_result, row);

	      if (db_fields)
		{
		  char *expr_result = (char *)NULL;

		  /* If the user supplied an expression, evaluate it now, in
		     the context of this record. */
		  if (expr != (char *)NULL)
		    {
		      symbol_set_default_package (db_fields);
		      expr_result = page_evaluate_string (expr);
		      symbol_set_default_package (save);
		    }

		  /* If satisfied, save this record. */
		  if ((expr == (char *)NULL) ||
		      (!empty_string_p (expr_result)))
		    {
		      DBRecord *rec = (DBRecord *)xmalloc (sizeof (DBRecord));
		      rec->key = (char *) NULL;
		      rec->contents = db_fields;

		      if (rec_index + 2 > rec_slots)
			records = (DBRecord **)xrealloc
			  (records, (rec_slots += 30) * sizeof (DBRecord *));

		      records[rec_index++] = rec;
		      records[rec_index] = (DBRecord *)NULL;
		    }
		  else
		    symbol_destroy_package (db_fields);

		  if (expr_result) free (expr_result);
		}
	    }

	  msqlFreeResult (query_result);

	  /* If there are any matching records, then sort, format,
	     and/or return the keys. */
	  if (rec_index != 0)
	    {
	      char *format_expr = get_value (vars, "format");

	      /* If there is a format operator, evaluate it now. */
	      if (format_expr != (char *)NULL)
		{
		  int format_limit = -1;
		  char *temp;
		  char *fl;

		  fl = page_evaluate_string (get_value (vars, "format-limit"));
		  if (!empty_string_p (fl))
		    {
		      format_limit = atoi (fl);
		      if (format_limit == 0) format_limit = -1;
		    }

		  if (fl != (char *)NULL) free (fl);

		  for (i = 0; ((i < rec_index) &&
			       ((format_limit < 0) || (i < format_limit)));
		       i++)
		    {
		      symbol_set_default_package (records[i]->contents);
		      temp = page_evaluate_string (format_expr);
		      symbol_set_default_package (save);

		      if (temp)
			{
			  bprintf_insert (page, start, "%s", temp);
			  start += strlen (temp);
			  free (temp);
			}
		    }
		}

	      /* Finally, free the memory that we have used. */
	      for (i = 0; i < rec_index; i++)
		{
		  symbol_destroy_package (records[i]->contents);
		  free (records[i]->key);
		  free (records[i]);
		}
	      free (records);
	    }
	}
      if (query != (char *)NULL) free (query);
    }
}

/* Quote single quotes for SQL string.
   Takes a field specifier, used to decide where to truncate the output
   to CHAR fields.
   If field == NULL, don't actually truncate anything. */
static void
bprintf_msql_escape (BPRINTF_BUFFER *out, char *src, m_field *field)
{
  char *p = src;
  char c;
  int i = 0;
  int maxlength;

  /*  We only truncate character type */
  if (field == (m_field *)NULL)
    maxlength = 0;
  else if (field->type == CHAR_TYPE)
    maxlength = field->length;
  else maxlength = 0;

  if (src == (char *) NULL)
    return;

  while ((c = *p++) != 0)
    {
      i++;
      if ((maxlength != 0) && (i > maxlength)) return;

      if (c == '\'')
	bprintf (out, "\\");

      bprintf (out, "%c",c);


    }
}

/* Do a case insensitive lookup of name in fields.
   Return a matching field, or NULL.  */
static m_field *
lookup_fieldname (char *name, m_result *fields)
{
  int i, num_fields;

  msqlFieldSeek (fields, 0);
  num_fields = msqlNumFields (fields);

  for (i = 0; i < num_fields; i++)
    {
      m_field *field;

      field = msqlFetchField (fields);
      if (!strcasecmp (name, field->name))
	return field;
    }

  return (m_field *)NULL;
}

/* <database-save-record db key var1 var2 ... varN
   table=table keyname=fieldname>

   Does an INSERT and UPDATE to try to store the data into table
   using single key keyname, with value key.

   NOTE: If the key field for table you are using is not
   configured as the PRIMARY KEY, then you can get duplicate
   entries in the database.

   We use msqlListFields() to get a list of field names, so we
   can make sure to only try to set valid field names in the
   table.

   We also need the field type to decide whether to use single
   quotes around the data values (for char type). */

#define QUOTE_KEYVAL_IF_NEEDED if (keyfield->type == CHAR_TYPE) \
                                     bprintf (query, "'")
static void
pf_database_save_record (PFunArgs)
{
  int db = get_dbref (vars);
  char *key     = page_evaluate_string (get_positional_arg (vars, 1));
  char *keyname = page_evaluate_string (get_value (vars, "keyname"));
  char *table   = page_evaluate_string (get_value (vars, "table"));
  BPRINTF_BUFFER *query = bprintf_create_buffer ();
  char *result = (char *)NULL;
  m_result *table_fields;
  int items_printed = 0;
  int errors_found = 0;

  pagefunc_set_variable ("msql::msql-error-message[]", "");

  if ((db >= 0) && (database_environment_level != 0) &&
      !(empty_string_p (key)) &&
      !(empty_string_p (table)) &&
      !(empty_string_p (keyname)))
    {

      char *name;
      int position;
      int status;
      m_field *field;
      m_field *keyfield;

      table_fields = msqlListFields (db, table);
      if (table_fields == (m_result *) NULL)
	{
	  pagefunc_set_variable ("msql::msql-error-message[]",
				 "table does not exist or has no columns");
	  errors_found++;
	}

      if (!errors_found)
	{
	  keyfield = lookup_fieldname (keyname, table_fields);
	  if (keyfield == (m_field *) NULL)
	    {
	      pagefunc_set_variable ("msql::msql-error-message[]",
				     "primary key column name not found");
	      errors_found++;
	    }
	}

      if (!errors_found)
	{
	  /* First we'll do an SQL INSERT, and if that fails, we'll do an
	     UPDATE.

	     INSERT INTO tablename (var1, var2, ...) VALUES ('val1', 'val2'...)

	     We need to escape single quotes in the char values. */
	  bprintf (query, "INSERT INTO %s (", table);

	  position = 2;

	  /* Comma-separated list of all the variable names, including key. */
	  bprintf (query, "%s", keyfield->name);

	  while ((name = get_positional_arg (vars, position)) != (char *)NULL)
	    {
	      Symbol *sym;
	      position++;

	      sym = symbol_lookup (name);
	      if (sym == (Symbol *)NULL) continue;

	      field = lookup_fieldname (sym->name, table_fields);

	      /* If the field exists in the table, print its name */
	      if (field != (m_field *) NULL)
		{
		  bprintf (query, ",");
		  bprintf (query, "%s", field->name);
		}
	    }

	  bprintf (query, ") VALUES (");

	  QUOTE_KEYVAL_IF_NEEDED;
	  bprintf_msql_escape (query, key, keyfield);
	  QUOTE_KEYVAL_IF_NEEDED;

	  /* A comma-separated list of values, including the key value.
	     We check the field data type and use single-quotes if it
	     is a CHAR field. */
	  position = 2;
	  while ((name = get_positional_arg (vars, position)) != (char *)NULL)
	    {
	      char *value = pagefunc_get_variable (name);
	      Symbol *sym;

	      position++;
	      sym = symbol_lookup (name);
	      if (sym == (Symbol *)NULL) continue;

	      field = lookup_fieldname (sym->name, table_fields);

	      /* If the field exists in the table, print its name.
		 Exception: Don't print this field if it is the
		 keyfield. */
	      if ((field != (m_field *) NULL) &&
		  (strcasecmp (field->name, keyfield->name) != 0))
		{
		  bprintf (query, ",");

		  /* We only use single quotes around CHAR type data. */
		  if (field->type == CHAR_TYPE)
		    bprintf (query, "'");

		  bprintf_msql_escape (query, value, field);

		  if (field->type == CHAR_TYPE)
		    bprintf (query, "'");
		}
	    }

	  bprintf (query, ")");

	  /* Execute the query. */
	  status = msqlQuery (db, query->buffer);
	  bprintf_free_buffer (query);

	  if (status != -1)
	    {
	      result = "true";
	    }
	  else
	    {
	      /* If the INSERT failed, try an UPDATE. */
	      query = bprintf_create_buffer ();

	      /* UPDATE emp_details SET salary=30000, age=20
		 WHERE emp_id = 1234 */
	      bprintf (query, "UPDATE %s SET ", table);

	      position = 2;

	      /* This is used to tell if we need to print a comma */
	      items_printed = 0;

	      while ((name = get_positional_arg (vars, position))
		     != (char *)NULL)
		{
		  char *value = pagefunc_get_variable (name);
		  Symbol *sym;

		  position++;

		  sym = symbol_lookup (name);
		  if (sym == (Symbol *)NULL) continue;

		  field = lookup_fieldname (sym->name, table_fields);

		  if (field != (m_field *) NULL)
		    {
		      if (items_printed > 0) bprintf (query, ",");
		      items_printed++;

		      bprintf (query, "%s = ", field->name);
		      if (field->type == CHAR_TYPE)
			bprintf (query, "'");
		      bprintf_msql_escape (query, value, field);
		      if (field->type == CHAR_TYPE)
			bprintf (query, "'");

		    }
		}

	      bprintf (query, " WHERE ");
	      bprintf (query, "%s", keyfield->name);

	      bprintf (query, " = ");
	      QUOTE_KEYVAL_IF_NEEDED;
	      bprintf_msql_escape (query, key, keyfield);
	      QUOTE_KEYVAL_IF_NEEDED;

	      /* Execute the query. */
	      status = msqlQuery (db, query->buffer);
	      bprintf_free_buffer (query);

	      if (status != -1)
		result = "true";
	      else
		pagefunc_set_variable ("msql::msql-error-message[]",
				       msqlErrMsg);
	    }
	}
    }

  if (table_fields != (m_result *) NULL) msqlFreeResult (table_fields);

  if (key != (char *)NULL) free (key);
  if (keyname != (char *)NULL) free (keyname);
  if (table != (char *)NULL) free (table);

  if (result)
    bprintf_insert (page, start, "%s", result);
}

/* <database-load-record db key table=table keyname=fieldname [package=name]>

   Does a SELECT to recover data from the table, using
   WHERE keyname='key' as the query. */
static void
pf_database_load_record (PFunArgs)
{
  int db = get_dbref (vars);
  char *key     = page_evaluate_string (get_positional_arg (vars, 1));
  char *keyname = page_evaluate_string (get_value (vars, "keyname"));
  char *table   = page_evaluate_string (get_value (vars, "table"));
  BPRINTF_BUFFER *query = bprintf_create_buffer ();
  char *pkgname = (char *) NULL;
  char *result = (char *)NULL;
  m_result *table_fields = (m_result *)NULL;
  int errors_found = 0;

  pagefunc_set_variable ("msql::msql-error-message[]", "");

  if ((db >= 0) && (database_environment_level != 0) &&
      !(empty_string_p (key)) &&
      !(empty_string_p (table)) &&
      !(empty_string_p (keyname)))
    {

      int status = -1;
      m_field *keyfield = (m_field *) NULL;
      m_result *query_result = (m_result *) NULL;

      pkgname = page_evaluate_string
	(get_one_of (vars, "PACKAGE", "PREFIX", (char *)NULL));

      table_fields = msqlListFields (db, table);
      if (table_fields == (m_result *) NULL)
	{
	  pagefunc_set_variable ("msql::msql-error-message[]",
				 "table does not exist or has no columns");
	  errors_found++;
	}

      if (!errors_found)
	{
	  keyfield = lookup_fieldname (keyname, table_fields);

	  if (keyfield == (m_field *) NULL)
	    {
	      pagefunc_set_variable ("msql::msql-error-message[]",
				     "primary key column name not found");
	      errors_found++;
	    }
	}

      if (!errors_found)
	{
	  bprintf (query, "SELECT * FROM %s where ", table);
	  bprintf (query, "%s", keyfield->name);
	  bprintf (query, " = ");

	  QUOTE_KEYVAL_IF_NEEDED;
	  bprintf_msql_escape (query, key, keyfield);
	  QUOTE_KEYVAL_IF_NEEDED;

	  status = msqlQuery (db, query->buffer);
	  bprintf_free_buffer (query);
	}

      if (status != -1)		/* Query had no errors */
	{
	  query_result = msqlStoreResult ();

	  if (query_result != (m_result *)NULL)
	    {
	      m_row msql_row = msqlFetchRow (query_result);

	      if (msql_row == (m_row)NULL)
		{
		  /* End of data has been reached. */
		}
	      else
		{
		  /* Loop over fields of the row, binding variables.
		     In the case of MSQL, everything is a string, so
		     we don't need to worry about converting INT or
		     REAL to string datatypes */
		  register int i;

		  /* Reset the field descriptor cursor to start of record */
		  msqlFieldSeek (query_result, 0);

		  for (i = 0; i < msqlNumFields (query_result); i++)
		    {
		      BPRINTF_BUFFER *varname = bprintf_create_buffer ();
		      m_field *field = msqlFetchField (query_result);

		      if (empty_string_p (pkgname))
			bprintf (varname, "%s", field->name);
		      else
			bprintf (varname, "%s::%s", pkgname, field->name);

		      pagefunc_set_variable (varname->buffer, msql_row[i]);

		      bprintf_free_buffer (varname);
		    }
		  result = "true";
		}
	    }
	}
      else if (!errors_found)
	{
	  pagefunc_set_variable ("msql::msql-error-message[]", msqlErrMsg);
	}
    }

  if (table_fields != (m_result *) NULL) msqlFreeResult (table_fields);
  if (key != (char *)NULL) free (key);
  if (keyname != (char *)NULL) free (keyname);
  if (table != (char *)NULL) free (table);
  if (pkgname != (char *)NULL) free (pkgname);

  if (result != (char *)NULL)
    bprintf_insert (page, start, "%s", result);
}

/* <database-delete-record DB KEY table=tablename keyname=fieldname>
   table and keyname must be supplied, where keyname is the name
   of the primary key field.  */
static void
pf_database_delete_record (PFunArgs)
{
  int db = get_dbref (vars);
  char *key     = page_evaluate_string (get_positional_arg (vars, 1));
  char *keyname = page_evaluate_string (get_value (vars, "keyname"));
  char *table   = page_evaluate_string (get_value (vars, "table"));
  BPRINTF_BUFFER *query = bprintf_create_buffer ();
  char *result = (char *)NULL;
  m_result *table_fields = (m_result *) NULL;
  int errors_found = 0;

  pagefunc_set_variable ("msql::msql-error-message[]", "");

  if ((db >= 0) && (database_environment_level != 0) &&
      !(empty_string_p (key)) &&
      !(empty_string_p (table)) &&
      !(empty_string_p (keyname)))
    {
      m_field *keyfield;
      int status = -1;

      table_fields = msqlListFields (db, table);
      if (table_fields == (m_result *) NULL)
	{
	  pagefunc_set_variable ("msql::msql-error-message[]",
				 "table does not exist or has no columns");
	  errors_found++;
	}

      if (!errors_found)
	{
	  keyfield = lookup_fieldname (keyname, table_fields);

	  if (keyfield == (m_field *) NULL)
	    {
	      pagefunc_set_variable ("msql::msql-error-message[]",
				     "primary key column name not found");
	      errors_found++;
	    }
	}

      if (!errors_found)
	{
	  /* DELETE FROM table where keyname='keyval' */
	  bprintf (query, "DELETE FROM %s where ", table);
	  bprintf (query, "%s", keyfield->name);
	  bprintf (query, " = ");

	  QUOTE_KEYVAL_IF_NEEDED;
	  bprintf_msql_escape (query, key, keyfield);
	  QUOTE_KEYVAL_IF_NEEDED;

	  status = msqlQuery (db, query->buffer);
	  bprintf_free_buffer (query);
	}

      if (status != -1)
	result = "true";
      else if (!errors_found)
	pagefunc_set_variable ("msql::msql-error-message[]", msqlErrMsg);
    }

  if (table_fields != (m_result *) NULL) msqlFreeResult (table_fields);
  if (key != (char *)NULL) free (key);
  if (keyname != (char *)NULL) free (keyname);
  if (table != (char *)NULL) free (table);

  if (result != (char *)NULL)
    bprintf_insert (page, start, "%s", result);
}

/* <database-save-package db key package keyname=fieldname table=table>

   This gets the fieldnames and datatypes using msqlListFields, and
   only saves variables who have names which match (case insensitive)
   with table fields. */
static void
pf_database_save_package (PFunArgs)
{
  int db = get_dbref (vars);
  char *key     = page_evaluate_string (get_positional_arg (vars, 1));
  char *keyname = page_evaluate_string (get_value (vars, "keyname"));
  char *table   = page_evaluate_string (get_value (vars, "table"));
  BPRINTF_BUFFER *query = bprintf_create_buffer ();
  char *result = (char *)NULL;
  int status;
  m_result *table_fields = (m_result *) NULL;
  m_field *field;
  m_field *keyfield;
  Symbol **symbols = (Symbol **)NULL;
  int items_printed = 0;
  int errors_found = 0;

  pagefunc_set_variable ("msql::msql-error-message[]", "");

  if ((db >= 0) && (database_environment_level != 0) &&
      !(empty_string_p (key)) &&
      !(empty_string_p (table)) &&
      !(empty_string_p (keyname)))
    {
      char *package_name = page_evaluate_string (get_positional_arg (vars, 2));

      table_fields = msqlListFields (db, table);
      if (table_fields == (m_result *) NULL)
	{
	  pagefunc_set_variable ("msql::msql-error-message[]",
				 "table does not exist or has no columns");
	  errors_found++;
	}

      if (!errors_found)
	{
	  keyfield = lookup_fieldname (keyname, table_fields);
	  if (keyfield == (m_field *) NULL)
	    {
	      pagefunc_set_variable ("msql::msql-error-message[]",
				     "primary key column name not found");
	      errors_found++;
	    }
	}

      if (!errors_found)
	{
	  if (!empty_string_p (package_name))
	    symbols = symbol_package_symbols (package_name);

	  if (package_name != (char *)NULL)
	    free (package_name);

	  if (symbols != (Symbol **)NULL)
	    {
	      register int i;
	      Symbol *sym;

	      /* Let's try an INSERT on this key and keyvalue:
		 "INSERT INTO table (key, name, name..) VALUES
		 ('keyval', 'val', 'val', ...)" */
	      bprintf (query, "INSERT INTO %s (", table);

	      /* Comma-separated list of all the variable names,
		 including key. */
	      bprintf (query, "%s", keyfield->name);

	      for (i = 0; (sym = symbols[i]) != (Symbol *)NULL; i++)
		{
		  if (sym->type != symtype_STRING)
		    continue;

		  field = lookup_fieldname (sym->name, table_fields);

		  /* If the field exists in the table, print its name.
		     Exception: if this is the keyfield, we have already
		     printed it. */
		  if ((field != (m_field *) NULL) &&
		      (strcasecmp (field->name, keyfield->name) != 0))
		    {
		      bprintf (query, ",");
		      bprintf (query, "%s", field->name);
		    }
		}

	      bprintf (query, ") VALUES (");
	      QUOTE_KEYVAL_IF_NEEDED;
	      bprintf_msql_escape (query, key, keyfield);
	      QUOTE_KEYVAL_IF_NEEDED;

	      /* Print symbol values */
	      for (i = 0; (sym = symbols[i]) != (Symbol *)NULL; i++)
		{
		  if (sym->type != symtype_STRING)
		    continue;

		  field = lookup_fieldname (sym->name, table_fields);

		  /* If the field exists in the table, print its name */
		  if (field != (m_field *) NULL)
		    {
		      bprintf (query, ",");

		      /* We only use single quotes around CHAR type data. */
		      if (field->type == CHAR_TYPE)
			bprintf (query, "'");

		      /* Print out the various values.  If there are none,
			 print the empty string. */
		      if (sym->values_index == 0)
			{
			  /* do nothing */
			}
		      else
			{
			  if (sym->values_index == 1)
			    bprintf_msql_escape (query, sym->values[0], field);
			  else
			    {
			      register int j;
			      BPRINTF_BUFFER *tmpbuf;

			      tmpbuf = bprintf_create_buffer ();

			      for (j = 0; j < sym->values_index; j++)
				bprintf (tmpbuf, sym->values[j]);

			      bprintf_msql_escape
				(query, tmpbuf->buffer, field);

			      bprintf_free_buffer (tmpbuf);
			    }
			}

		      if (field->type == CHAR_TYPE)
			bprintf (query, "'");
		    }
		}

	      bprintf (query, ")");

	      /* Execute the query. */
	      status = msqlQuery (db, query->buffer);
	      bprintf_free_buffer (query);

	      if (status != -1)
		result = "true";
	      else
		{
		  /* The INSERT failed for some reason, so let's try
		     an UPDATE. */
		  query = bprintf_create_buffer ();

		  /* Let's try an UPDATE on this key and keyvalue:
		     UPDATE emp_details SET salary=30000 WHERE emp_id = 1234 */

		  bprintf (query, "UPDATE %s SET ", table);

		  /* Comma-separated list of all the variable names
		     and values. */
		  items_printed = 0;

		  for (i = 0; (sym = symbols[i]) != (Symbol *)NULL; i++)
		    {
		      if (sym->type != symtype_STRING)
			continue;

		      field = lookup_fieldname (sym->name, table_fields);

		      if (field != (m_field *) NULL)
			{
			  if (items_printed > 0) bprintf (query, ",");
			  items_printed++;

			  bprintf (query, "%s = ", field->name);

			  if (field->type == CHAR_TYPE)
			    bprintf (query, "'");

			  if (sym->values_index == 0)
			    {
			      /* do nothing */
			    }
			  else
			    {
			      if (sym->values_index == 1)
				bprintf_msql_escape
				  (query, sym->values[0], field);
			      else
				{
				  /* Collect up the array values
				     into a string. */
				  register int j;
				  BPRINTF_BUFFER *tmpbuf;

				  tmpbuf = bprintf_create_buffer ();
				  for (j = 0; j < sym->values_index; j++)
				    bprintf (tmpbuf, sym->values[j]);

				  bprintf_msql_escape
				    (query, tmpbuf->buffer, field);

				  bprintf_free_buffer (tmpbuf);
				}
			    }

			  if (field->type == CHAR_TYPE)
			    bprintf (query, "'");
			}
		    }

		  bprintf (query, " WHERE ");
		  bprintf (query, "%s", keyfield->name);
		  bprintf (query, " = ");
		  QUOTE_KEYVAL_IF_NEEDED;
		  bprintf_msql_escape (query, key, keyfield);
		  QUOTE_KEYVAL_IF_NEEDED;

		  /* Execute the query. */
		  status = msqlQuery (db, query->buffer);
		  bprintf_free_buffer (query);

		  if (status != -1)
		    result = "true";
		  else
		    pagefunc_set_variable ("msql::msql-error-message[]",
					   msqlErrMsg);
		}
	    }
	}
      if (table_fields != (m_result *) NULL) msqlFreeResult (table_fields);
      if (key != (char *)NULL) free (key);
      if (keyname != (char *)NULL) free (keyname);
      if (table != (char *)NULL) free (table);

      if (result)
	bprintf_insert (page, start, "%s", result);
    }
}
