/* mhc.c: -*- C -*-  Run Meta-HTML on a single file. */

/* Author: Brian J. Fox (bfox@ai.mit.edu) Thu Nov 30 13:31:36 1995.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

#include "language.h"
#if !defined (MHTML_SYSTEM_TYPE)
#  define MHTML_SYSTEM_TYPE "Incorrectly Compiled"
#endif

#if !defined (MHTML_VERSION_STRING)
static char *mhtml_version_string = "";
#else
static char *mhtml_version_string = MHTML_VERSION_STRING;
#endif

#if !defined (macintosh)
extern char **environ;
#endif /* !macintosh */

static void parse_program_args (int argc, char *argv[]);
static void prep_for_page_process (PAGE *page, int argc, char *argv[]);
static void usage (void);

static char *progname = (char *)NULL;
static PAGE *input_contents = (PAGE *)NULL;

#if defined (CHECK_TIMESTAMP)
static void check_timestamp (char *filename);
#endif

extern char *metahtml_copyright_string;

/* This program takes a single filename as an argument, and runs the
   page processor on it.  You can set variables on the command line
   with the `--set' flag.  The output is written to standard out.
   If no filename is passed, the file used is argv[0]. */
int
main (int argc, char *argv[])
{
#if defined (NOTDEF)
  register int i;

  for (i = 0; i < argc; i++)
    fprintf (stdout, "argv[%d] = `%s'<br>\n", i, argv[i]);
#endif

  parse_program_args (argc, argv);

#if defined (CHECK_TIMESTAMP)
  check_timestamp (argv[0]);
#endif

  if (!input_contents)
    usage ();

  prep_for_page_process (input_contents, argc, argv);

  page_process_page (input_contents);
  if (input_contents && input_contents->buffer)
    {
      if (pagefunc_get_variable ("mhc::explicit-output-only") == (char *)NULL)
	fprintf (stdout, "%s", input_contents->buffer);
    }

  return (0);
}

static void
usage (void)
{
  if (isatty (fileno (stdin)))
    {
      fprintf (stderr, "mhc: %s%s\n\n", MHTML_SYSTEM_TYPE,
	       metahtml_copyright_string);
      fprintf (stderr, "Usage: %s [--set NAME VALUE]... FILENAME\n", progname);
    }
  exit (1);
}

static void
prep_for_page_process (PAGE *page, int argc, char *argv[])
{
  register int i;

  if (page && page->buffer)
    {
      if (page->buffer[0] == '#' && page->buffer[1] == '!')
	{
	  for (i = 2; i < page->bindex; i++)
	    {
	      if (page->buffer[i] == '\n')
		{
		  i++;
		  break;
		}
	    }

	  bprintf_delete_range (page, 0, i);
	}

      pagefunc_set_variable ("mhtml::version", mhtml_version_string);
      pagefunc_set_variable ("mhtml::system-type", MHTML_SYSTEM_TYPE);
      pagefunc_set_variable ("mhtml::exec-path", (char *)getenv ("PATH"));

#if !defined (macintosh)
      /* Now set the environment variables. */
      if (environ != (char **)NULL)
	{
	  static char *buffer = (char *)NULL;
	  static int size = 0;

	  for (i = 0; environ[i] != (char *)NULL; i++)
	    {
	      char *name = environ[i];
	      char *value;
	      int len;

	      len = 6 + strlen (name);

	      if (len > size)
		buffer = (char *)xrealloc (buffer, (size += (len * 2)));

	      strcpy (buffer, "env::");
	      strcat (buffer, name);
	      value = strchr (buffer, '=');
	      if (value)
		{
		  *value = '\0';
		  value++;
		}
	      pagefunc_set_variable (buffer, value);
	    }
	  if (buffer != (char *)NULL)
	    free (buffer);
	}
#endif /* !macintosh */
      /* Now set the meta-html variables. */
      forms_input_data (argc, argv);
    }
}

static void
parse_program_args (int argc, char *argv[])
{
  int arg_index = 1;
  char *filename = (char *)NULL;

  progname = argv[0];
  pagefunc_set_variable ("mhc::script-name", progname);

  while (arg_index < argc)
    {
      char *arg = argv[arg_index++];

      if ((strcasecmp (arg, "--set") == 0) && (arg_index + 1 < argc))
	{
	  char *name = argv[arg_index++];
	  char *value = argv[arg_index++];

	  pagefunc_set_variable (name, value);
	}
      else if (strncmp (arg, "--", 2) != 0)
	{
	  if (filename)
	    {
	      static Symbol *sym = (Symbol *)NULL;

	      if (!sym) sym = symbol_intern ("mhtml::program-arguments");

	      symbol_add_value (sym, arg);
	    }
	  else
	    filename = arg;
	}
      else
	usage ();
    }

  if (!filename && !input_contents)
    usage ();
  else if (filename)
    input_contents = page_read_template (filename);

  {
    char *temp = pagefunc_get_variable ("mhtml::include-prefix");

    if (empty_string_p (temp))
      {
	temp = strrchr (filename, '/');

	if (temp != (char *)NULL)
	  {
	    *temp = '\0';
	    pagefunc_set_variable ("mhtml::include-prefix", filename);
	  }
	else
	  {
	    char dir[1024];
	    temp = getcwd (dir, sizeof (dir));
	    temp = strrchr (dir, '/');
	    if (temp != (char *)NULL)
	      {
		*temp = '\0';
		pagefunc_set_variable ("mhtml::include-prefix", dir);
	      }
	  }
      }

    /* Define default functions here. */
    {
      PAGE *p = page_create_page ();

      bprintf (p, "<defsubst meta-html>");
      bprintf (p, "<font color=\"#00309B\">Meta-</font>HTML");
      bprintf (p, "<if <not <get-var mhtml::printed-tm>>");
      bprintf (p, "<prog <sup><font size=-2>TM</font></sup>");
      bprintf (p, "<increment mhtml::printed-tm>>>");
      bprintf (p, "</defsubst>");
      bprintf (p, "<defmacro ptext>");
      bprintf (p, "<plain-text first-char=<font size=+2> nobr=true>");
      bprintf (p, "%s", "%body</plain-text></defmacro>");
      page_process_page (p);
      page_free_page (p);
    }
  }
}

#if defined (CHECK_TIMESTAMP)
#include "../libtimestamp/timestamp.h"

static void
check_timestamp (char *filename)
{
  if (check_stamp_expired (filename))
    {
      fprintf (stderr, "Sorry, but your 45 day free trial has ended.\n");
      fprintf (stderr, "You may obtain another copy of MHC from\n");
      fprintf (stderr, "\n\thttp://www.metahtml.com\n\n");
      fprintf (stderr, "We hope that you have enjoyed the use of our\n");
      fprintf (stderr, "Meta-HTML Standalone Processor, and will continue\n");
      fprintf (stderr, "to watch our site for new additions to our product\n");
      fprintf (stderr, "suite.\n");
      exit (2);
    }
}
#endif /* CHECK_TIMESTAMP */

