// Copyright (C) 1999-2001 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
// 
// As a special exception to the GNU General Public License, permission is 
// granted for additional uses of the text contained in its release 
// of ccaudio.
//
// The exception is that, if you link the ccaudio library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the ccaudio library code into it.
//
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
// 
// This exception applies only to the code released under the 
// name ccaudio.  If you copy code from other releases into a copy of
// ccaudio, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
// 
// If you write modifications of your own for ccaudio, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.  

#ifndef	CCXX_AUDIO_H_
#define	CCXX_AUDIO_H_

#ifndef	CCXX_CONFIG_H_
#include <cc++/config.h>
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
#endif

/**
 * Generic audio class to hold master data types and various useful
 * class encapsulated friend functions as per GNU Common C++ 2 coding
 * standard.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short Master audio class.
 */
class Audio
{
public:
	enum	Rate
	{
		rateUnknown,
		rate6khz = 6000,
		rate8khz = 8000,
		rate44khz = 44100
	};
	typedef enum Rate Rate;

	enum	Encoding
	{
		unknownEncoding = 0,
		g721ADPCM,
		g722Audio,
		g722_7bit,
		g722_6bit,
		g723_3bit,
		g723_5bit,
		gsmVoice,
		mulawAudio,
		alawAudio,
		okiADPCM,
		voxADPCM,
		cdaStereo,
		cdaMono,
		pcm8Stereo,
		pcm8Mono,
		pcm16Stereo,
		pcm16Mono,
		pcm32Stereo,
		pcm32Mono
	};
	typedef enum Encoding Encoding;

	enum Format
	{
		raw,
		sun,
		riff,
		wave
	};
	typedef enum Format Format;

	enum Error
	{
		errSuccess = 0,
		errReadLast,
		errNotOpened,
		errEndOfFile,
		errStartOfFile,
		errRateInvalid,
		errEncodingInvalid,
		errReadInterrupt,
		errWriteInterrupt,
		errReadFailure,
		errWriteFailure,
		errReadIncomplete,
		errWriteIncomplete,
		errRequestInvalid,
		errTOCFailed,
		errStatFailed,
		errInvalidTrack,
		errPlaybackFailed,
		errNotPlaying
	};
	typedef enum Error Error;

	class Info
	{
	public:
		Audio::Format format;
		Audio::Encoding encoding;
		unsigned rate;
		unsigned order;
		char *annotation;
	};

	static CCXX_EXPORT(bool) isMono(Encoding encoding);
	static CCXX_EXPORT(bool) isStereo(Encoding encoding);
	static CCXX_EXPORT(Rate) getRate(Encoding encoding);
	static CCXX_EXPORT(int) getFrame(Encoding encoding, int samples = 0);
	static CCXX_EXPORT(int) getCount(Encoding encoding);
	static CCXX_EXPORT(unsigned long) toSamples(Encoding encoding, size_t bytes);
	static CCXX_EXPORT(unsigned long) toBytes(Encoding encoding, unsigned long samples);
	static CCXX_EXPORT(void) fill(unsigned char *addr, int samples, Encoding encoding);
};

/**
 * A class used to manipulate audio data.  This class provides file
 * level access to audio data stored in different formats.  This class
 * also provides the ability to write audio data into a disk file.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short audio file access.
 */
class CCXX_CLASS_EXPORT AudioFile: public Audio
{
private:
	char *pathname;
	Error error;
	Info info;
	unsigned long header;		// offset to start of audio
	unsigned long minimum;		// minimum sample size required

	void initialize(void);
	void getWaveFormat(int size);

protected:
	union
	{
		int fd;
		void *handle;
	} file;

	unsigned long limit;

	virtual bool afCreate(const char *path);
	virtual bool afOpen(const char *path);
	virtual bool afPeek(unsigned char *data, unsigned size);
	virtual int afRead(unsigned char *data, unsigned size);
	virtual int afWrite(unsigned char *data, unsigned size);
	virtual bool afSeek(unsigned long pos);
	virtual void afClose(void);

	virtual char *getContinuation(void)
		{return NULL;};

	Error setError(Error err);

	unsigned short getShort(unsigned char *data);
	void setShort(unsigned char *data, unsigned short value);
	unsigned long getLong(unsigned char *data);
	void setLong(unsigned char *data, unsigned long value);

public:
	AudioFile(const char *fname, unsigned long samples = 0);
	AudioFile(const char *fname, Info *info, unsigned long min = 0);

	AudioFile()
		{initialize();};

	~AudioFile()
		{clear();};

	void open(const char *fname);
	void create(const char *fname, Info *info);
	void close(void);
	void clear(void);
	int getBuffer(void *addr, unsigned len);
	int putBuffer(void *attr, unsigned len);
	Error getSamples(void *addr, unsigned samples);
	Error putSamples(void *addr, unsigned samples);
	Error skip(long samples);
	Error setPosition(unsigned long samples = ~0l);
	Error setLimit(unsigned long samples = 0l);
	Error getInfo(Info *info);
	Error setMinimum(unsigned long samples);
	unsigned long getPosition(void);
	virtual bool isOpen(void);
	virtual bool hasPositioning(void)
		{return true;};

	inline Encoding getEncoding(void)
		{return info.encoding;};

	inline Format getFormat(void)
		{return info.format;};

	inline unsigned getSampleRate(void)
		{return info.rate;};

	inline char *getAnnotation(void)
		{return info.annotation;};

	inline Error getError(void)
		{return error;};

	inline bool operator!(void)
		{return (bool)!isOpen();};
};

/**
 * This class allows one to control audio playback from the CD player
 * on the target platform.  Audio playback can be used to play selective
 * tracks, to eject, etc.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short control cd audio player.
 */
class CCXX_CLASS_EXPORT CDAudio : public Audio
{
private:
	union
	{
		int fd;
		void *handle;
	} file;
	unsigned char v0, v1;
#ifdef	__WIN32__
	CRITICAL_SECTION crit;
	bool paused;
	bool opened;
	char position[20];
	char endmark[24];
	char ret[256];
	DWORD command(char *fmt, ...);
#endif
	Error err;

public:
	CDAudio(int devnbr = 0);
	~CDAudio();

	Error play(int start, int end = 0);
	Error stop(void);
	Error pause(void);
	Error resume(void);
	Error eject(void);
	Error reload(void);
	int getFirst(void);
	int getLast(void);
	bool isPaused(void);
	bool isAudio(int track);
	bool isOpen(void);

	unsigned char getVolume(int speaker);
	void setVolume(unsigned char left, unsigned char right);

	inline unsigned char getVolumeLeft(void)
		{return getVolume(0);};

	inline unsigned char getVolumeRight(void)
		{return getVolume(1);};

	inline Error getError(void)
		{return err;};

	inline bool operator!(void)
		{return (bool)!isOpen();};
};

/**
 * This class is use to represent and process audio data held in memory.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short audio data in memory.
 */
class CCXX_CLASS_EXPORT AudioSample : public Audio
{
protected:
	friend class AudioCopy;

	Encoding encoding;
	unsigned rate;
	unsigned count;
	unsigned char *samples;

public:
	AudioSample(unsigned frame, Encoding coding = pcm16Mono, unsigned rate = 8000);
	~AudioSample();

	inline unsigned getCount(void)
		{return count;};

	inline unsigned getRate(void)
		{return rate;};

	inline Encoding getEncoding(void)
		{return encoding;};

	inline unsigned char *getSamples(void)
		{return samples;};
};

/**
 * The codec class is a virtual used for transcoding audio samples between
 * linear frames (or other known format) and an encoded "sample" buffer.
 * This class is only abstract and describes the core interface for
 * loadable codec modules.  This class is normally merged with AudioSample.
 * A derived AudioCodecXXX will typically include a AudioRegisterXXX static
 * class to automatically initialize and register the codec with the codec
 * registry.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short process codec interface.
 */
class CCXX_CLASS_EXPORT AudioCodec : public AudioSample
{
protected:
	AudioCodec(unsigned frames, Encoding encoding, unsigned rate);
	unsigned length;

public:
	/**
	 * Encode a linear sample frame into the codec sample buffer.
	 *
	 * @param linear sample buffer to encode.
	 * @param number of samples, or use default count.
	 * @param channel number or 0 for all channels.
	 */
	virtual void encode(unsigned short *buffer, unsigned samples = 0, int channel = 0) = 0;

	/**
	 * Encode an "AudioSamples" object into the new encoding scheme if
	 * conversion is supported.
	 *
	 * @return true if successful.
	 * @param object source.
	 */
	virtual bool encode(AudioSample *sample) = 0;

	/**
	 * Decode the sample frame into linear samples.
	 *
	 * @return number of samples extracted.
	 * @param sample buffer to save linear samples into.
	 * @param channel number or 0 for all channels.
	 */
	virtual unsigned decode(unsigned short *buffer, int channel = 0) = 0;

	/**
	 * Decode into an :AudioSample" if the conversion is supported.
	 *
 	 * @return true if successful.
	 * @param object to save into.
	 */
	virtual bool decode(AudioSample *sample) = 0;

	/**
	 * Load a codec from a byte stream data source.  This is used for
	 * streaming of compressed codec's where frames are of irregular
	 * sizes.
	 *
	 * @return number of bytes actually read into codec workspace.
	 * @param pointer to byte data stream.
	 * @param number of data bytes.
	 */
	virtual unsigned load(unsigned char *data, unsigned length);

	/**
	 * Save a codec to a byte stream data source.  This is used for
	 * streaming of compressed codec's where frames are of irregular
	 * sizes.
	 *
	 * @return number of bytes actually written to workspace.
	 * @param pointer to byte data stream.
	 * @param size of byte data stream.
	 */
	virtual unsigned save(unsigned char *data, unsigned length);


	/**
	 * Get current number of samples of data, based on load or encode.
	 *
	 * @return length.
	 */
	inline unsigned getLength(void)
		{return length;};

	/**
	 * Get size of the audio frame.
	 */
	unsigned getFrameSize(void);

	/**
	 * Check if stereo/multi-channel codec.
	 */
	virtual unsigned getChannels(void);
};

/**
 * This class is used by loadable modules and application specific code to
 * register new codec modules.  A derived xxxRegisterCodec class will
 * typically exist as a static member of an xxxCodec class to
 * automatically register the codec at system load time.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short a base class for codec registration.
 */
class CCXX_CLASS_EXPORT AudioRegisterCodec : public Audio
{
private:
	static AudioRegisterCodec *first;
	AudioRegisterCodec *next;

protected:
	unsigned minframe, framecnt;
	AudioRegisterCodec(unsigned min, unsigned def = 1);

public:
	/**
	 * Get the encoding format of this codec registry.
	 *
	 * @return audio encoding format.
	 */
	virtual Encoding getEncoding(void) = 0;

	/**
	 * Get the default sample rate for this codec type.
	 *
	 * @return samples per second.
	 */
	virtual unsigned getRate(void)
		{Audio::getRate(getEncoding());};

	/**
	 * Get a logical name for this codec.
	 *
	 * @return null terminated name string.
	 */
	virtual char *getName(void) = 0;

	/**
	 * Creates a new codec from the registry class.  This creates
	 * an object of "frame" size, as a multiple of the min. frame
	 * size.  This is usually just a new xxxCodec(frames * minframe).
	 *
	 * @param number of "minframe" sample frames in transcoder.
	 */
	virtual AudioCodec *getCodec(unsigned frames) = 0;

	/**
	 * Create a new Tone object from the registry class.  This
	 * creates a tone buffer (AudioTone-like sample) encoded in
	 * the format of the given codec.
	 *
	 * @param first tone frequency.
	 * @param second tone frequency.
	 * @param max frames of sample.
         * @param max frames of tone.
	 * @return audio sample in encoded format.
	 */
	virtual AudioSample *getTone(unsigned f1, unsigned f2, unsigned maxsize, unsigned maxtone = 0) = 0;

	/**
	 * Return minimum codec frame size in samples.
	 *
	 * @return codec transcoding sample size resolution.
	 */
	inline unsigned getFrameSize(void)
		{return minframe;};

	/**
	 * Get the default or preferred number of frames for optimal
	 * performance.
	 *
	 * @return codec transcoding preferred sample resolution.
	 */
	inline unsigned getFrameCount(void)
		{return framecnt;};

	/**
	 * Locate a codec registry object.
	 *
	 * @param encoding of codec registry to locate.
	 * @param optional sample rate of codec registry to locate.
	 */
	static AudioRegisterCodec *findRegistration(Encoding encoding, unsigned rate = 0);

	/**
	 * Locate a codec registry object by name.
	 *
	 * @param name of codec registry to locate.
	 */
	static AudioRegisterCodec *findRegistration(const char *name);

	/**
	 * Create a new codec object from the registry to use for
	 * transcoding.
	 *
	 * @param encoding of codec to create.
	 * @param optional sample rate of codec being created.
	 * @param sample frames (multiples of minframe) or 0 for defframe.
	 */
	static AudioCodec *findCodec(Encoding encoding, unsigned rate = 0, unsigned frames = 0);

	/**
	 * Create a new codec object from a named registry to use for
	 * transcoding.
	 *
	 * @param name of codec to create.
	 * @param frame count to process or 0 for default frame size.
	 */
	static AudioCodec *findCodec(const char *name, unsigned frame = 0);
};

/**
 * The tone class is used to construct or generate a tone sample in memory.
 * These tone samples can be either single or dual tones, and are created
 * as linear samples of a specified sample rate.
 *
 * @author David Sugar <dyfet@ostel.com>
 * @short generate audio sample in memory.
 */
class CCXX_CLASS_EXPORT AudioTone : public AudioSample
{
protected:
	double p1, p2, v1, v2, fa1, fa2;

public:
	AudioTone(unsigned size, unsigned f1 = 0, unsigned f2 = 0, unsigned rate = 8000);

	/**
	 * Fill the current sample frame with more tone data.
	 */
	void fill(unsigned max = 0);

	/*
	 * Set frequency.
	 *
	 * @param f1 first frequency.
	 * @param f2 second frequency.
	 */
	void setFreq(unsigned f1, unsigned f2 = 0);
};

/**
 * AudioCopy is used for certain kinds of stream join operations.
 * Essentially it fills the current sample buffer from another AudioSample
 * object and requests a new object thru a virtual every time it empties.
 * This is used in dtmf dialing, for example, to fill in the next digit
 * tone.
 *
 * @short fill an audio sample frame from another.
 * @author David Sugar <dyfet@ostel.com>.
 */
class CCXX_CLASS_EXPORT AudioCopy : public AudioSample
{
protected:
	unsigned char *next;
	unsigned left;

	virtual AudioSample *fill(void) = 0;

public:
	AudioCopy(unsigned frame, Encoding encoding = pcm16Mono, unsigned rate = 8000);
	bool copy(void);

	inline bool isEmpty(void)
		{return next == NULL;};
};

#ifdef	CCXX_NAMESPACES
};
#endif

#endif

