/* <gcc.c>
 *
 *	A DCL driver for GNU "CC".  It accepts a CLD syntax compatable
 *	with Digital's "VAX C".
 *
 *	Since GCC consists of more than 1 pass we have to do the parsing
 *	here and have a DCL Command (.COM) file do the real work -- so that
 *	multiple images can be invoked WITHOUT having to resort to LIB$SPAWN.
 */
#define GCC_DRIVER_VERSION "2.1.0"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#undef EXIT_FAILURE
#define EXIT_FAILURE 0x1000002C		/* SS$_ABORT|STS$M_INHIB_MSG */

#define LOCAL  0
#define GLOBAL 1
#define VERB_STAT 1	/* verbose */
#define VERB_ECHO 2	/* verify */
#define VERB_SILENT 256 /* suppress GCC-I-foo messages */
#define PROF_FUNC 1	/* function profiling */
#define PROF_BLOC 2	/* block profiling */

static char *Input_File =0;		/* Input file specifications */
static char *Debug =0;			/* /DEBUG specification      */
static char *Object =0;			/* Object file name	     */
static char *List_File =0;		/* List file name	     */
static char *Target_Machine =0;		/* For cross-compiling	     */
static char *Assembly_File =0;		/* Assembly file name	     */
static int GCC_Version=0;		/* Version of gcc we are running */
static int Optimize=0;			/* Do optimized compile      */
static int G_Float=0;			/* Use G double precision floating */
static int Machine_Code=0;		/* Output assembly lang      */
static int Verbose=0;			/* Make noise while compiling*/
static int Plus=0;			/* Use C++ compiler	     */
static int Show=0;			/* Show options		     */
static int Profile=0;			/* Use compiler profiler     */
static int Warn=0;			/* Generate warning messages */
static int Case_Hack=1;			/* Hack symbols with upper case*/
static int List=0;			/* Save preprocessor output  */
static int Version=0;			/* Show compiler version number */
static int Generate_Object=0;		/* Should we generate obj?   */
static int Standard=0;			/* traditional vs normal vs pedantic */
/*
 * DCL has a limit of 256 characters for on a single command line.
 */
static char cpp_Options[256];		/* Options for the CPP pass  */
static char cc1_Options[256];		/* Options for the CC1 pass  */
static char gas_Options[256];		/* Options for the GAS pass  */
static char xxx_Overflow[256];	/* kludge; inadequate bounds checking */

extern int getpid();
static int cli_present(), cli_get_value(), cli_negated();
extern int cli$present(), cli$get_value();
static int lib_do_command(), lib_set_symbol();
extern int lib$do_command(), lib$set_symbol();

		/* - assorted utility routines - */

/*
 *	Save a string in dynamic memory
 */
static char *savestr(const char *String)
{
    return strcpy((char *)malloc(strlen(String)+1), String);
}

/*
 *	Append one string to another, doubling quote chars as encountered
 */
static void mystrcat(char *string, const char *append)
{
    register char *dst = string + strlen(string);	/* point to nul */
    register const char *src = append;
    while (*src) {
	if (*src == '"') *dst++ = '"';			/* double quotes */
	*dst++ = *src++;
    }
    *dst = '\0';    /* and terminate the string */
}

/*
 *	Search one string for another
 */
static char *locate(const char *source, const char *target)	/* strstr() */
{
    register const char *pnt = source;
    while (*pnt) {
	if (*pnt == *target && !strncmp(pnt,target,strlen(target)))
	    return (char *)pnt;
	pnt++;
    }
    return (char *)0;
}

		/* - command parsing code - */

/*
 *	Do the parsing
 */
static void checkswitches(int flag)
{
    char Temp[256];

    /*
     *	/INCLUDE_DIRECTORY=(dir1[,dir2,...])
     */
    if (cli_present("INCLUDE_DIRECTORY",0,flag)) {
	while (cli_get_value("INCLUDE_DIRECTORY",Temp,sizeof Temp)) {
	    strcat(cpp_Options,"\"-I");
	    mystrcat(cpp_Options,Temp);
	    strcat(cpp_Options,"\" ");
	}
    }
    /*
     *	/DEFINE=(macro1[=value1][,macro2[=value2],...])
     */
    if (cli_present("DEFINE",0,flag)) {
	while (cli_get_value("DEFINE",Temp,sizeof Temp)) {
	    strcat(cpp_Options,"\"-D");
	    mystrcat(cpp_Options,Temp);
	    strcat(cpp_Options,"\" ");
	}
    }
    /*
     *	/UNDEFINE=(macro1[,macro2,...])
     */
    if (cli_present("UNDEFINE",0,flag)) {
	while (cli_get_value("UNDEFINE",Temp,sizeof Temp)) {
	    strcat(cpp_Options,"\"-U");
	    mystrcat(cpp_Options,Temp);
	    strcat(cpp_Options,"\" ");
	}
    }
    /*
     *	/SCAN=(file1[,file2,...])
     */
    if (cli_present("SCAN",0,flag)) {
	while (cli_get_value("SCAN",Temp,sizeof Temp)) {
	    strcat(cpp_Options,GCC_Version==1 ? "\"-i" : "\"-imacros\" ");
	    mystrcat(cpp_Options,Temp);
	    strcat(cpp_Options,GCC_Version==1 ? "\" " : " ");
	}
    }
    /*
     *	/[NO]OBJECT[=file]
     */
    if (Generate_Object = cli_present("OBJECT",Generate_Object,flag)) {
	if (Object) free(Object);
	if (cli_get_value("OBJECT",Temp,sizeof Temp))
	    Object = savestr(Temp);
	else
	    Object = 0;
    }
    /*
     *	/[NO]MACHINE_CODE[=file]
     *	    used to save the ".s" assembler intermediate output
     */
    if (Machine_Code = cli_present("MACHINE_CODE",Machine_Code,flag)) {
	if (Assembly_File) free(Assembly_File);
	if (cli_get_value("MACHINE_CODE",Temp,sizeof Temp))
	    Assembly_File = savestr(Temp);
	else
	    Assembly_File = 0;
    }
    /*
     *	/TARGET=[host]
     *	    used for cross-compilation.
     */
    if (cli_present("TARGET",0,flag)) {
	if (Target_Machine) free(Target_Machine);
	if (cli_get_value("TARGET",Temp,sizeof Temp))
	    Target_Machine = savestr(Temp);
	else
	    Target_Machine = 0;
    }
    /*
     *	/[NO]LIST[=file], /PROCESS_ONLY[=file]
     */
    if ((List = cli_present("LIST",List,flag)) != 0
     || (List = cli_present("PREPROCESS_ONLY",List,flag)) != 0) {
	if (List_File) free(List_File),  List_File = 0;
	if (cli_get_value("LIST",Temp,sizeof Temp))
	    List_File = savestr(Temp);
	else if (cli_get_value("PREPROCESS_ONLY",Temp,sizeof Temp))
	    List_File = savestr(Temp);
    }
    if (cli_present("PREPROCESS_ONLY",0,flag))
	Generate_Object = 0;
    /*
     *	/DEBUG[=ALL|NONE|([NO]TRACEBACK,[NO]SYMBOLS)
     *	    note: only /NODEBUG (the default) and /DEBUG=ALL are supported.
     */
    if (cli_present("DEBUG",0,flag)) {
	/* Get the value (Default = ALL), and save it */
	if (!cli_get_value("DEBUG",Temp,sizeof Temp))
	    strcpy(Temp,"ALL");
	Debug = savestr(Temp);
    }
    /*
     *	/[NO]OPTIMIZE[=anything...]
     */
    if (cli_present("OPTIMIZE",Optimize,flag)) {
	if (!cli_get_value("OPTIMIZE",Temp,sizeof Temp))
	    Optimize = 1;
	else {
	    register char *p;
	    for (p = Temp; *p; p++)
		if (*p < '0' || *p > '9') break;
	    if (!*p)  Optimize = atoi(Temp);
	}
    } else if (cli_negated())
	Optimize = 0;
    /*
     *	/CC1_OPTIONS=arbitrary_string
     */
    if (cli_present("CC1_OPTIONS",0,flag)) {
	cli_get_value("CC1_OPTIONS",cc1_Options,sizeof cc1_Options);
	strcat(cc1_Options," ");
    }
    /*
     *	/[NO]WARNINGS[=ALL|NONE|([NO]INFORMATIONALS,[NO]WARNINGS)]
     *	    note:  values are ignored
     */
    Warn = cli_present("WARNINGS",Warn,flag);
    /*
     *	/VERSION
     */
    Version = cli_present("VERSION",Version,flag);
    /*
     *	/[NO]PLUS
     */
    Plus = cli_present("PLUS_PLUS",Plus,flag);
    /*
     *	/[NO]G_FLOAT
     */
    G_Float = cli_present("G_FLOAT",G_Float,flag);
    /*
     *	/[NO]VERBOSE
     */
    if (cli_present("VERBOSE",0,flag)) {
	if ((!cli_get_value("VERBOSE",Temp,sizeof Temp))
	 || cli_present("VERBOSE.ALL",0,flag))
	    Verbose = VERB_STAT|VERB_ECHO;	/* all */
	else {
	    if (cli_present("VERBOSE.STATISTICS",0,flag)) Verbose |= VERB_STAT;
	    if (cli_present("VERBOSE.ECHO",0,flag)
	     || cli_present("VERBOSE.VERIFY",0,flag)) Verbose |= VERB_ECHO;
	}
    } else if (cli_negated())
	Verbose = VERB_SILENT;
    /*
     *	/PROFILE[=ALL|BLOCK|FUNCTION]
     */
    if (cli_present("PROFILE",0,flag)) {
	if (!cli_get_value("PROFILE",Temp,sizeof Temp))
	    Profile = PROF_FUNC;	/* function */
	else {
	    if (cli_present("PROFILE.ALL",0,flag))
		Profile = PROF_FUNC|PROF_BLOC;
	    if (cli_present("PROFILE.BLOCK",0,flag)) Profile |= PROF_BLOC;
	    if (cli_present("PROFILE.FUNCTION",0,flag)) Profile |= PROF_FUNC;
	}
    }
    /*
     *	/SHOW[=ALL|RULES|DEFINITIONS]
     */
    if (cli_present("SHOW",0,flag)) {
	if (!cli_get_value("SHOW",Temp,sizeof Temp))
	    Show = 3;		/* all */
	else {
	    if	    (cli_present("SHOW.ALL",0,flag)) Show = 3;
	    else if (cli_present("SHOW.RULES",0,flag)) Show = 2;
	    else if (cli_present("SHOW.DEFINITIONS",0,flag)) Show = 1;
	    else if (cli_present("SHOW.NONE",0,flag)) Show = 0;
	}
    }
    /*
     *	/NAMES={UPPER|LOWER|MIXED|HEX_SUFFIX}
     */
    if (cli_present("NAMES",0,flag)) {
	if (!cli_get_value("NAMES",Temp,sizeof Temp))
	    Case_Hack = 1;	/* HEX */
	else {
	    if	    (cli_present("NAMES.UPPER",0,flag)) Case_Hack = 0;
	    else if (cli_present("NAMES.HEX_SUFFIX",0,flag)) Case_Hack = 1;
	    else if (cli_present("NAMES.LOWER",0,flag)) Case_Hack = 2;
	    else if (cli_present("NAMES.MIXED",0,flag)
		  || cli_present("NAMES.AS_IS",0,flag)) Case_Hack = 3;
	}
    } else {
	/*
	 *  /[NO]CASE_HACK
	 */
	Case_Hack = cli_present("CASE_HACK",Case_Hack,flag);
    }
    /*
     *	/[NO]STANDARD[=[NO]PORTABLE]
     *		/standard=portable	=> -ansi -pedantic
     *		/standard=noportable	=> -ansi
     *		/nostandard		=> -traditional
     *		(default is "none of the above")
     */
    if (cli_present("STANDARD",0,flag)) {
	Standard = cli_present("STANDARD.PORTABLE",0,flag) ? 2 : 1;
	/* explicit /standard=noansi resets to the default behavior */
	if (!cli_present("STANDARD.ANSI",0,flag) && cli_negated()) Standard = 0;
    } else if (cli_negated())
	Standard = -1;
}

/*
 *	Construct GNU syntax gcc-cpp/gcc-cc1/gcc-as commands
 *	from DCL syntax GCC command, then invoke gcc.com to
 *	execute them.
 */
int main(void)
{
    register char *cp,*cp1;
    char *gcc_scratch;
    char Temp[256];
    int pid = getpid ();

    cpp_Options[0] = cc1_Options[0] = gas_Options[0] = xxx_Overflow[0] = '\0';
    /*
     *	    First we need to figure out which version of GCC we are running.
     */
    cp = getenv("GNU_CC_VERSION");
    if (!cp || (*cp < '1' || *cp > '2')) {
	fprintf(stderr, "\n%s -\n- %s -\n- %s -\n- %s\n",
		"%GCC-E-UNIDENTIFIED_VERSION,",
		"Cannot determine which version of GCC is being run",
		cp ? "because GNU_CC_VERSION has an invalid value."
		   : "because GNU_CC_VERSION is not defined.",
		"Please check the installation procedure.");
	exit(EXIT_FAILURE);
    }
    GCC_Version = *cp - '0';	/* major version number, 1 or 2 */
    /*
     * create the default name for the scratch files
     */
    gcc_scratch = getenv ("GNU_CC_SCRATCH");
    gcc_scratch = gcc_scratch ? gcc_scratch : "SYS$SCRATCH:";
    sprintf (Temp, "%sGCC_%08x.CPP", gcc_scratch, pid);
    List_File = savestr(Temp);
    sprintf (Temp, "%sGCC_%08x.S", gcc_scratch, pid);
    Assembly_File = savestr(Temp);
    /*
     *	    Get the Input file
     */
    checkswitches(GLOBAL);
    if (cli_get_value("GCC_INPUT",Temp,sizeof Temp)) {
	Input_File = savestr(Temp);
	checkswitches(LOCAL);
    }
    /*
     *	    Handle any version identification request early on.
     */
    if (Version) {
	if (!(Verbose & VERB_SILENT)) printf(
		"%%GCC-I-VERSION, this is version %s of the GCC-VMS driver.\n",
					     GCC_DRIVER_VERSION);
	strcat(cpp_Options,"-v ");
	strcat(cc1_Options,"-version ");
#if 0
	strcat(gas_Options,"\"-V\" ");
#endif
    }
    /*
     *	    Find the base name of the source file
     */
    cp = Input_File;
    cp1 = strrchr(cp,']');  if (cp1) cp = cp1+1;
    cp1 = strrchr(cp,'>');  if (cp1) cp = cp1+1;
    cp1 = strrchr(cp,':');  if (cp1) cp = cp1+1;
    cp1 = Temp;
    while (*cp != '.' && *cp != ';' && *cp != '\0')
	*cp1++ = *cp++;
    /*
     *	    Next check to see if we need to change the name of the list file
     */
    if (Show) {
	    Generate_Object = 0;
	    Machine_Code = 0;
	    switch (Show) {
	      case 1:	strcat(cpp_Options,GCC_Version==1 ? "-d " : "\"-dM\" ");
			cp = ".DEF";
			break;
	      case 2:	strcat(cpp_Options,"\"-MM\" ");
			cp = ".";	/* ".MAK" perhaps? */
			break;
	      case 3:	strcat(cpp_Options,"\"-M\" ");
			cp = ".";
			break;
	      default:	cp = 0;
			break;
	    }
	    if (cp && (!List_File || !List)) {
		if (List_File) free(List_File);
		strcpy(cp1,cp);
		List_File = savestr(Temp);
	    }
	    List = 1;
    }
    /* Next check and see if we need to supply default file names */
    if (!List_File) {
	    strcpy(cp1,".CPP");
	    List_File = savestr(Temp);
    }
    if (!Assembly_File) {
	    strcpy(cp1,".S");
	    Assembly_File = savestr(Temp);
    }
    if (!Object) {
	strcpy(cp1,".OBJ");
	Object = savestr(Temp);
    }
    /* check whether this compilation is just performing syntax checking */
    if (!Generate_Object && !Machine_Code && !List) {
	if (!locate(cc1_Options,"-fsyntax-only")) {
	    if (!(Verbose & VERB_SILENT))
		printf("%%GCC-I-NOOUTPUT, no output was requested.\n");
#if 0
	    /* -fsyntax-only is available but not reliable under gcc 1.40 */
	    if (GCC_Version > 1) strcat(cc1_Options,"-fsyntax-only ");
#endif
	}
	/* preprocess and compile [to the null device] but don't assemble */
	Machine_Code++;
	if (Assembly_File) free(Assembly_File);
	Assembly_File = savestr("_NLA0:");	/* output to /dev/null */
    }

    /*
     *	    Generate gcc and gas switches from assorted DCL qualifiers.
     */
    if (Standard) {
	if (Standard < 0) {
	    strcat(cpp_Options,"-traditional ");
	    strcat(cc1_Options,"-traditional ");
	} else {
	    strcat(cc1_Options,"-ansi ");
	    if (Standard > 1) {
		strcat(cpp_Options,"-pedantic ");
		strcat(cc1_Options,"-pedantic ");
	    }
	}
    }
#if 0		/* not used */
    if (List) strcat(cpp_Options,"\"-C\" ");
#endif
#if 0		/* redundant */
    strcat(cpp_Options,"\"-D__GNUC__\" ");
#endif
    if (G_Float) {
	strcat(cpp_Options,"\"-DCC$gfloat\" ");
	strcat(cpp_Options,"\"-D__GFLOAT__\" ");
	strcat(cc1_Options,"-mg ");
    }
    if (Plus) {
	strcat(gas_Options,"-+ ");
	switch (GCC_Version) {
	  case 1: strcat(cpp_Options,"-+ \"-D__GNUG__\" \"-D__cplusplus\" ");
	    break;
	  case 2: strcat(cpp_Options,"-+ \"-D__GNUG__=2\" \"-D__cplusplus\" ");
	    break;
	  };
      }
    /*
     *	    Some version-dependent handling is needed.
     */
    switch (GCC_Version) {
      case 1:	if (Debug) strcat(cc1_Options, Plus ? "-g0 " : "-g ");
		if (Optimize) strcat(cc1_Options,"\"-O\" ");
		if (Plus) strcat(cc1_Options,"-fforce-addr ");
		strcat(gas_Options,"-1 ");
		break;
      case 2:	if (Debug) strcat(cc1_Options,"-g ");
		if (Optimize) {
		    sprintf(Temp,"\"-O%d\" ",Optimize);
		    strcat(cc1_Options,Temp);
		}
		break;
    }
    if (Optimize && Debug && !(Verbose & VERB_SILENT)) printf(
      "%%GCC-I-DBGOPT, caution: /debug specified with optimization enabled.\n");
    if (Warn) {
	strcat(cc1_Options,"\"-Wall\" ");
	strcat(cpp_Options,"\"-Wall\" ");
    }
    if (!(Verbose & VERB_STAT)) strcat(cc1_Options,"-quiet ");
    else if (Plus) strcat(gas_Options,"\"-H\" ");

    if (Profile & PROF_FUNC) strcat(cc1_Options,"-p ");
    if (Profile & PROF_BLOC) strcat(cc1_Options,"-a ");

    if	    (Case_Hack == 0) strcat(gas_Options,"-h ");
    else if (Case_Hack == 2) strcat(gas_Options,"-h2 ");
    else if (Case_Hack == 3) strcat(gas_Options,"-h3 ");

    if (cp = locate(cc1_Options,"-mdont-save-r2-r5")) {
	cp1 = cp + strlen("-mdont-save-r2-r5");
	while (*cp1) *cp++ = *cp1++;
	*cp = '\0';
	strcat(cc1_Options,
	       " -fcall-used-r2 -fcall-used-r3 -fcall-used-r4 -fcall-used-r5 ");
    }
    if (cp = locate(cc1_Options,"-mpcc-alignment")) {
	cp1 = cp + strlen("-mpcc-alignment");
	while (*cp1) *cp++ = *cp1++;
	*cp = '\0';
	strcat(cpp_Options,"\"-DPCC_ALIGNMENT\" ");
    } else if (!locate(cc1_Options,"-mvaxc-alignment")) {
	strcat(cc1_Options,"-mvaxc-alignment ");
    }

    /*
     *	    Generate the command string.
     */
    cp = Temp;

    /* Invoke the .COM file */
    cp1 = "@GNU_CC:[000000]GCC";
    while (*cp1) *cp++ = *cp1++;

    /* P1 = File to compile */
    *cp++ = ' ';
    cp1 = Input_File;
    while (*cp1) *cp++ = *cp1++;

    /* P2 = Options */
    *cp++ = ' ';
    *cp++ = '"';
    if (Plus == 1) *cp++ = 'P';
    if (List == 1) *cp++ = 'L';
    if (Generate_Object == 1) *cp++ = '*';
    if (Machine_Code) *cp++ = 'M';
    if (Verbose & VERB_ECHO) *cp++ = 'V';
    *cp++ = '"';

    /* P3 = Name of object file */
    *cp++ = ' ';
    if (Object) {
	cp1 = Object;
	while (*cp1) *cp++ = *cp1++;
    } else {
	*cp++ = '"';
	*cp++ = '"';
    }

    /* P4 = Name of listing file */
    *cp++ = ' ';
    if (List_File) {
	cp1 = List_File;
	while (*cp1) *cp++ = *cp1++;
    } else {
	*cp++ = '"';
	*cp++ = '"';
    }

    /* P5 = Name of assembly file */
    *cp++ = ' ';
    if (Assembly_File) {
	cp1 = Assembly_File;
	while (*cp1) *cp++ = *cp1++;
    } else {
	*cp++ = '"';
	*cp++ = '"';
    }

    /* better late than never...  No bounds checking was performed above. */
    if (strlen(cpp_Options) >= sizeof cpp_Options) {
	fprintf(stderr,"%% gcc-cpp command too long.  Unable to compile.\n");
	exit(EXIT_FAILURE);
    }
    if (strlen(cc1_Options) >= sizeof cc1_Options) {
	fprintf(stderr,"%% gcc-cc1 command too long.  Unable to compile.\n");
	exit(EXIT_FAILURE);
    }
    if (strlen(gas_Options) >= sizeof gas_Options) {
	fprintf(stderr,"%% gcc-as command too long.  Unable to compile.\n");
	exit(EXIT_FAILURE);
    }

/*
 * The symbols are assigned in this way, since we do not have to worry about
 * DCL parsing them.  This reduces the complexity, since we do not have to
 * double the single quotes, treble the double quotes, etc.
 *
 * Any single quotes within an include, define, undefine, or scan will be
 * doubled up, since we do require these to be passed along.
 */

    if(Target_Machine) lib_set_symbol("GCC_BinDir",Target_Machine);
	else lib_set_symbol("GCC_BinDir","000000");

    lib_set_symbol("cc1_Options",cc1_Options);
    lib_set_symbol("cpp_Options",cpp_Options);
    lib_set_symbol("gas_Options",gas_Options);

    /*
     *	    Do it
     */
    return lib_do_command(Temp, cp - Temp);
}


		/* - DCL interface routines - */

/*
 *	Execute the given DCL command
 */
static int lib_do_command(const char *Text, int Size)
{
    struct {int Size; const char *Ptr;} Descr;

    Descr.Ptr = Text;
    Descr.Size = Size;
    return lib$do_command(&Descr);
}

/*
 *	Define a local DCL symbol
 */
static int lib_set_symbol(const char *Symbol, const char *Value)
{
    struct {int Size; const char *Ptr;} Symbol_Descr, Value_Descr;
    int sym_typ = 1;		/* LIB$K_CLI_LOCAL_SYM */

    Symbol_Descr.Ptr = Symbol;
    Symbol_Descr.Size = strlen(Symbol);

    if(Value == 0)
       return lib$delete_symbol(&Symbol_Descr, &sym_typ);

    Value_Descr.Ptr = Value;
    Value_Descr.Size = strlen(Value);
    return lib$set_symbol(&Symbol_Descr, &Value_Descr, &sym_typ);
}


/************		DCL PARSING ROUTINES		**********/

#define CLI$_ABSENT	0x381f0
#define CLI$_NEGATED	0x381f8
#define CLI$_LOCNEG	0x38230
#define CLI$_PRESENT	0x3fd19
#define CLI$_DEFAULTED	0x3fd21
#define CLI$_LOCPRES	0x3fd31
#define CLI$_COMMA	0x3fd39

static int cli_status;

/*
 *	See if "NAME" is present, absent or negated.
 */
static int cli_present(const char *Name, int oldflag, int gblflag)
{
    struct {int Size; const char *Ptr;} Key;

    Key.Ptr = Name;
    Key.Size = strlen(Name);
    cli_status = cli$present(&Key);
    switch (cli_status) {
	case CLI$_ABSENT:
		return oldflag;
	case CLI$_NEGATED:
		if (gblflag == LOCAL) return oldflag;
	case CLI$_LOCNEG:
		return 0;
	case CLI$_PRESENT:
	case CLI$_DEFAULTED:
		if (gblflag == LOCAL) return oldflag;
	case CLI$_LOCPRES:
		return 1;
	default:	/*(shouldn't be possible)*/
		return 0;
    }
}

/*
 *	Return additional information about the last CLI operation
 */
static int cli_negated(void)
{
    return cli_status == CLI$_NEGATED || cli_status == CLI$_LOCNEG;
}

/*
 *	Get value of "NAME"
 */
static int cli_get_value(const char *Name, char *Buffer, int Size)
{
    struct {int Size; const char *Ptr;} Key;
    struct {int Size; char *Ptr;} Value;

    Key.Ptr = Name;
    Key.Size = strlen(Name);
    Value.Ptr = Buffer;
    Value.Size = Size - 1;
    cli_status = cli$get_value(&Key, &Value, &Value.Size);
    if (cli_status & 1) {
	Buffer[Value.Size] = '\0';
	return 1;
    } else {
	Buffer[0] = '\0';
	return 0;
    }
}
