/*
 * Decompiled with CFR 0.152.
 */
package org.gnunet.util;

import com.google.common.collect.Lists;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOError;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.FileChannel;
import java.nio.channels.Pipe;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.spi.SelectorProvider;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Queue;
import org.gnunet.util.AbsoluteTime;
import org.gnunet.util.Cancelable;
import org.gnunet.util.RelativeTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Scheduler {
    private static final Logger logger = LoggerFactory.getLogger(Scheduler.class);
    static ThreadLocal<SchedulerInstance> threadScheduler = new ThreadLocal<SchedulerInstance>(){

        @Override
        protected SchedulerInstance initialValue() {
            return new SchedulerInstance();
        }
    };

    public static void addContinuation(Task task, EnumSet<Reason> reasons) {
        threadScheduler.get().addContinuation(task, reasons);
    }

    public static Cancelable add(Task task) {
        return Scheduler.addDelayed(RelativeTime.ZERO, task);
    }

    public static TaskIdentifier addDelayed(RelativeTime delay, Task task) {
        TaskConfiguration tid = new TaskConfiguration(delay, task);
        return tid.schedule(threadScheduler.get());
    }

    public static TaskIdentifier addRead(RelativeTime timeout, SelectableChannel chan, Task task) {
        TaskConfiguration tid = new TaskConfiguration(timeout, task);
        tid.addSelectEvent(chan, 1);
        return tid.schedule(threadScheduler.get());
    }

    public static void run() {
        threadScheduler.get().run();
    }

    public static void run(Task initialTask) {
        threadScheduler.get().run(initialTask);
    }

    public static boolean hasTasks() {
        return threadScheduler.get().hasTasks();
    }

    public static TaskIdentifier addWrite(RelativeTime timeout, SelectableChannel chan, Task task) {
        TaskConfiguration tid = new TaskConfiguration(timeout, task);
        tid.addSelectEvent(chan, 4);
        return tid.schedule(threadScheduler.get());
    }

    private static void addReasonsFromInterestOp(EnumSet<Reason> reasons, int interestOps) {
        if ((interestOps & 1) != 0) {
            reasons.add(Reason.READ_READY);
        }
        if ((interestOps & 4) != 0) {
            reasons.add(Reason.WRITE_READY);
        }
        if ((interestOps & 8) != 0) {
            reasons.add(Reason.CONNECT_READY);
        }
        if ((interestOps & 0x10) != 0) {
            reasons.add(Reason.ACCEPT_READY);
        }
    }

    public static void forceReset() {
        threadScheduler.get().forceReset();
    }

    public static FilePipe openFilePipe(File file) {
        FilePipeThread fpt = new FilePipeThread(file);
        fpt.setDaemon(true);
        fpt.start();
        return new FilePipe(fpt);
    }

    public static void debugPrintPendingTasks() {
        System.err.println("pending tasks:");
        for (TaskIdentifier i : Scheduler.threadScheduler.get().pending) {
            System.err.println(i.task.getClass());
        }
    }

    private static class FilePipeThread
    extends Thread {
        public File file;
        public Pipe pipe;

        FilePipeThread(File file) {
            this.file = file;
            try {
                this.pipe = SelectorProvider.provider().openPipe();
                this.pipe.source().configureBlocking(false);
                this.pipe.sink().configureBlocking(false);
            }
            catch (IOException e) {
                throw new RuntimeException("selector provider has no pipes");
            }
        }

        @Override
        public void run() {
            FileChannel fileChannel;
            try {
                FileInputStream stream = new FileInputStream(this.file);
                fileChannel = stream.getChannel();
            }
            catch (FileNotFoundException e) {
                throw new IOError(e);
            }
            ByteBuffer buffer = ByteBuffer.allocate(1);
            boolean quit = false;
            while (!quit) {
                try {
                    buffer.clear();
                    fileChannel.read(buffer);
                    buffer.flip();
                    this.pipe.sink().write(buffer);
                }
                catch (IOException e) {
                    quit = true;
                    try {
                        fileChannel.close();
                    }
                    catch (IOException ex) {
                        // empty catch block
                    }
                    try {
                        this.pipe.sink().close();
                    }
                    catch (IOException ex) {
                        // empty catch block
                    }
                    try {
                        this.pipe.source().close();
                    }
                    catch (IOException ex) {}
                }
            }
        }
    }

    public static class FilePipe {
        private FilePipeThread filePipeThread;

        private FilePipe(FilePipeThread filePipeThread) {
            this.filePipeThread = filePipeThread;
        }

        public Pipe.SourceChannel getSource() {
            return this.filePipeThread.pipe.source();
        }
    }

    public static class TaskConfiguration {
        private final Task task;
        private boolean lifeness = true;
        private Priority priority;
        private final AbsoluteTime deadline;
        private Subscriptions subscriptions;
        private SchedulerInstance scheduler;

        TaskConfiguration(SchedulerInstance scheduler, RelativeTime delay, Task task) {
            if (delay == null) {
                throw new AssertionError((Object)"task delay may not be 'null'");
            }
            this.scheduler = scheduler;
            this.task = task;
            this.deadline = delay.toAbsolute();
        }

        public TaskConfiguration(RelativeTime delay, Task task) {
            this(threadScheduler.get(), delay, task);
        }

        public TaskIdentifier schedule() {
            return this.schedule(threadScheduler.get());
        }

        public TaskIdentifier schedule(SchedulerInstance scheduler) {
            if (this.priority == null) {
                this.priority = scheduler.activeTask != null ? scheduler.activeTask.priority : Priority.DEFAULT;
            }
            TaskIdentifier tid = new TaskIdentifier(scheduler, this);
            if (this.subscriptions != null) {
                this.subscriptions.apply(tid);
            }
            scheduler.pending.add(tid);
            return tid;
        }

        public void addSelectEvent(SelectableChannel channel, int event) {
            if (channel == null) {
                throw new AssertionError((Object)"channel may not be null");
            }
            if (this.subscriptions == null) {
                this.subscriptions = new Subscriptions(this.scheduler.selector);
            }
            this.subscriptions.add(channel, event);
        }

        public void setLifeness(boolean b) {
            this.lifeness = b;
        }
    }

    public static class TaskIdentifier
    implements Cancelable {
        private boolean hasRun = false;
        private boolean isCanceled = false;
        private final Task task;
        private final RunContext ctx = new RunContext();
        private final boolean lifeness;
        private final Priority priority;
        private final AbsoluteTime deadline;
        private final Subscriptions subscriptions;
        private SchedulerInstance scheduler;

        public TaskIdentifier(SchedulerInstance scheduler, Task task, EnumSet<Reason> reasons) {
            this.scheduler = scheduler;
            this.ctx.reasons = reasons;
            this.task = task;
            this.lifeness = true;
            this.priority = Priority.DEFAULT;
            this.deadline = null;
            this.subscriptions = null;
        }

        public TaskIdentifier(SchedulerInstance scheduler, TaskConfiguration tc) {
            this.scheduler = scheduler;
            this.task = tc.task;
            this.subscriptions = tc.subscriptions;
            this.deadline = tc.deadline;
            this.priority = tc.priority;
            this.lifeness = tc.lifeness;
        }

        private void run() {
            if (this.hasRun) {
                throw new AssertionError((Object)"same task ran twice");
            }
            if (this.isCanceled) {
                return;
            }
            TaskIdentifier old = this.scheduler.activeTask;
            this.scheduler.activeTask = this;
            this.task.run(this.ctx);
            this.hasRun = true;
            this.scheduler.activeTask = old;
        }

        @Override
        public void cancel() {
            if (this.hasRun) {
                throw new AssertionError((Object)"can't onCancel task that already ran");
            }
            if (this.isCanceled) {
                throw new AssertionError((Object)"task canceled twice");
            }
            this.isCanceled = true;
            this.scheduler.pending.remove(this);
        }

        private void deregister() {
            if (this.subscriptions != null) {
                this.subscriptions.stop(this);
            }
        }
    }

    public static interface Task {
        public void run(RunContext var1);
    }

    private static class Subscriptions {
        private final Selector selector;
        List<ChannelInterest> channelInterests = Lists.newLinkedList();

        void add(SelectableChannel channel, int interestOps) {
            boolean found = false;
            for (ChannelInterest ci : this.channelInterests) {
                if (ci.channel != channel) continue;
                ci.interestOps |= interestOps;
                if ((ci.interestOps | 8 | 1) != 0) {
                    throw new AssertionError((Object)"OP_CONNECT and OP_READ are incompatible in java");
                }
                found = true;
                break;
            }
            if (!found) {
                ChannelInterest ci = new ChannelInterest();
                ci.channel = channel;
                ci.interestOps = interestOps;
                this.channelInterests.add(ci);
            }
        }

        void apply(TaskIdentifier tid) {
            for (ChannelInterest ci : this.channelInterests) {
                SelectionKey key = ci.channel.keyFor(this.selector);
                if (key == null || !key.isValid()) {
                    try {
                        key = ci.channel.register(this.selector, ci.interestOps);
                        key.attach(new LinkedList());
                    }
                    catch (ClosedChannelException e) {
                        throw new IOError(e);
                    }
                } else {
                    key.interestOps(key.interestOps() | ci.interestOps);
                }
                LinkedList opl = (LinkedList)key.attachment();
                TaskInterestOps tio = new TaskInterestOps();
                tio.tid = tid;
                tio.interestOps = ci.interestOps;
                opl.add(tio);
            }
        }

        void stop(TaskIdentifier tid) {
            for (ChannelInterest ci : this.channelInterests) {
                SelectionKey key = ci.channel.keyFor(this.selector);
                if (key == null || !key.isValid()) {
                    logger.warn("missing selection key");
                    return;
                }
                LinkedList interestList = (LinkedList)key.attachment();
                Iterator it = interestList.iterator();
                int remainingInterestOps = 0;
                while (it.hasNext()) {
                    TaskInterestOps ops = (TaskInterestOps)it.next();
                    if (ops.tid == tid) {
                        it.remove();
                        continue;
                    }
                    remainingInterestOps |= ops.interestOps;
                }
                key.interestOps(remainingInterestOps);
            }
        }

        public Subscriptions(Selector selector) {
            this.selector = selector;
        }

        private static class ChannelInterest {
            SelectableChannel channel;
            int interestOps;

            private ChannelInterest() {
            }
        }
    }

    private static class TaskInterestOps {
        TaskIdentifier tid;
        int interestOps;

        private TaskInterestOps() {
        }
    }

    public static class RunContext {
        public EnumSet<Reason> reasons = EnumSet.noneOf(Reason.class);
    }

    public static enum Reason {
        STARTUP,
        SHUTDOWN,
        TIMEOUT,
        READ_READY,
        WRITE_READY,
        ACCEPT_READY,
        CONNECT_READY;

    }

    public static enum Priority {
        IDLE,
        BACKGROUND,
        DEFAULT,
        HIGH,
        UI,
        URGENT,
        SHUTDOWN;

        private static final int numberOfPriorities;

        static {
            numberOfPriorities = Priority.values().length;
        }
    }

    private static class SchedulerInstance {
        TaskIdentifier activeTask = null;
        int readyCount = 0;
        Selector selector = null;
        final LinkedList<TaskIdentifier>[] readyLists = new LinkedList[Priority.access$000()];
        boolean schedulerRunning = false;
        final Queue<TaskIdentifier> pending = new PriorityQueue<TaskIdentifier>(5, new Comparator<TaskIdentifier>(){

            @Override
            public int compare(TaskIdentifier a, TaskIdentifier b) {
                return a.deadline.compareTo(b.deadline);
            }
        });

        private boolean checkLiveness() {
            if (this.readyCount > 0) {
                return true;
            }
            for (TaskIdentifier t : this.pending) {
                if (!t.lifeness) continue;
                return true;
            }
            if (!this.pending.isEmpty()) {
                logger.debug("tasks pending but not alive -- disconnect");
                this.shutdown();
                return true;
            }
            return false;
        }

        private void queueReady(TaskIdentifier tid) {
            int idx = tid.priority.ordinal();
            this.readyLists[idx].add(tid);
            ++this.readyCount;
            this.pending.remove(tid);
        }

        void addContinuation(Task task, EnumSet<Reason> reasons) {
            this.readyLists[Priority.DEFAULT.ordinal()].add(new TaskIdentifier(this, task, reasons));
            ++this.readyCount;
        }

        private RelativeTime handleTimeouts() {
            TaskIdentifier t;
            RelativeTime timeout = RelativeTime.FOREVER;
            while ((t = this.pending.peek()) != null) {
                RelativeTime remaining = t.deadline.getRemaining();
                if (remaining.getMicroseconds() <= 0L) {
                    t.deregister();
                    ((TaskIdentifier)t).ctx.reasons = EnumSet.of(Reason.TIMEOUT);
                    this.queueReady(t);
                    continue;
                }
                timeout = remaining;
                break;
            }
            return timeout;
        }

        public SchedulerInstance() {
            for (int i = 0; i < Priority.numberOfPriorities; ++i) {
                this.readyLists[i] = new LinkedList();
            }
            try {
                this.selector = SelectorProvider.provider().openSelector();
            }
            catch (IOException e) {
                logger.error("fatal: cannot create selector");
                System.exit(-1);
            }
        }

        private void handleSelect(RelativeTime timeout) {
            long timeout_ms = timeout.getMicroseconds() / 1000L;
            try {
                if (timeout_ms == 0L) {
                    this.selector.selectNow();
                } else if (timeout.isForever()) {
                    logger.debug("selecting, timeout=forever");
                    this.selector.select(0L);
                } else {
                    this.selector.select(timeout_ms);
                }
            }
            catch (IOException e) {
                throw new IOError(e);
            }
            logger.debug("select over");
            HashSet<TaskIdentifier> executableTasks = new HashSet<TaskIdentifier>();
            for (SelectionKey sk : this.selector.selectedKeys()) {
                LinkedList subscribers = (LinkedList)sk.attachment();
                for (TaskInterestOps ops : subscribers) {
                    if ((sk.readyOps() & ops.interestOps) == 0) continue;
                    executableTasks.add(ops.tid);
                    Scheduler.addReasonsFromInterestOp(((TaskIdentifier)ops.tid).ctx.reasons, sk.readyOps() & ops.interestOps);
                }
            }
            for (TaskIdentifier tt : executableTasks) {
                tt.deregister();
                this.queueReady(tt);
            }
        }

        public void run() {
            this.run(null);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void run(Task initialTask) {
            logger.debug("running scheduler");
            if (this.schedulerRunning) {
                throw new AssertionError((Object)"Scheduler already running");
            }
            this.schedulerRunning = true;
            try {
                this.runUnchecked(initialTask);
            }
            finally {
                logger.debug("cleaning up after scheduler ran");
                this.forceReset();
            }
        }

        public void shutdown() {
            for (TaskIdentifier tid : new ArrayList<TaskIdentifier>(this.pending)) {
                ((TaskIdentifier)tid).ctx.reasons.add(Reason.SHUTDOWN);
                this.queueReady(tid);
            }
            this.pending.clear();
        }

        public void forceReset() {
            this.schedulerRunning = false;
            this.readyCount = 0;
            this.activeTask = null;
            for (int i = 0; i < Priority.numberOfPriorities; ++i) {
                this.readyLists[i] = Lists.newLinkedList();
            }
            this.pending.clear();
        }

        public boolean hasTasks() {
            return this.readyCount != 0 || !this.pending.isEmpty();
        }

        private void runReady() {
            do {
                if (this.readyCount == 0) {
                    return;
                }
                for (int p = Priority.numberOfPriorities - 1; p >= 0; --p) {
                    LinkedList<TaskIdentifier> queue = this.readyLists[p];
                    while (!queue.isEmpty()) {
                        TaskIdentifier tid = queue.removeFirst();
                        --this.readyCount;
                        tid.run();
                    }
                }
            } while (this.pending.size() == 0);
        }

        private void runUnchecked(Task initialTask) {
            if (initialTask != null) {
                this.addContinuation(initialTask, EnumSet.of(Reason.STARTUP));
            }
            while (this.checkLiveness()) {
                RelativeTime nextTimeout = this.handleTimeouts();
                if (nextTimeout.getMicroseconds() < 0L) {
                    logger.warn("negative timeout for select");
                }
                if (this.readyCount == 0 && this.pending.isEmpty()) {
                    return;
                }
                if (this.readyCount > 0) {
                    this.handleSelect(RelativeTime.ZERO);
                } else {
                    this.handleSelect(nextTimeout);
                }
                this.runReady();
            }
            if (this.readyCount != 0) {
                throw new AssertionError((Object)"tasks ready after scheduler ran (count)");
            }
            for (LinkedList<TaskIdentifier> readyList : this.readyLists) {
                if (!readyList.isEmpty()) {
                    throw new AssertionError((Object)"tasks ready after scheduler ran (list)");
                }
            }
            if (this.pending.size() != 0) {
                throw new AssertionError((Object)"pending tasks after scheduler ran");
            }
            if (this.activeTask != null) {
                throw new AssertionError((Object)"active task after scheduler ran");
            }
        }
    }
}

