/*
**  GNU Pth - The GNU Portable Threads
**  Copyright (c) 1999-2000 Ralf S. Engelschall <rse@engelschall.com>
**
**  This file is part of GNU Pth, a non-preemptive thread scheduling
**  library which can be found at http://www.gnu.org/software/pth/.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
**  USA, or contact Ralf S. Engelschall <rse@engelschall.com>.
**
**  pth_ring.c: Pth ring data structure
*/
                             /* ``Unix was not designed to stop people
                                  from doing stupid things, because that
                                  would also stop them from doing clever
                                  things.''         --Doug Gwyn          */
#include "pth_p.h"

intern void pth_ring_init(pth_ring_t *r, pth_ringnode_t *rn)
{
    if (r != NULL)
        r->rn_hook = NULL;
    if (rn != NULL) {
        rn->rn_next = rn;
        rn->rn_prev = rn;
    }
    return;
}

intern int pth_ring_empty(pth_ring_t *r)
{
    if (r == NULL)
        return FALSE;
    return (r->rn_hook == NULL ? TRUE : FALSE);
}

intern int pth_ring_elements(pth_ring_t *r)
{
    int n;
    pth_ringnode_t *rn;

    if (r == NULL)
        return -1;
    n = 0;
    rn = r->rn_hook;
    if (rn != NULL) {
        do {
            n++;
            rn = rn->rn_next;
        } while (rn != r->rn_hook);
    }
    return n;
}

#if cpp
#define pth_ring_next(r, rn) \
    (((r) == NULL || (rn) == NULL) ? NULL : (rn)->rn_next)
#endif

#if cpp
#define pth_ring_prev(r, rn) \
    (((r) == NULL || (rn) == NULL) ? NULL : (rn)->rn_prev)
#endif

#if cpp
#define pth_ring_first(r) \
    ((r) == NULL ? NULL : (r)->rn_hook)
#endif

intern pth_ringnode_t *pth_ring_last(pth_ring_t *r)
{
    if (r == NULL)
        return NULL;
    if (r->rn_hook == NULL)
        return NULL;
    return r->rn_hook->rn_prev;
}

intern void pth_ring_insert_after(pth_ring_t *r, pth_ringnode_t *rn1, pth_ringnode_t *rn2)
{
    if (r == NULL || rn1 == NULL || rn2 == NULL)
        return;
    rn2->rn_prev = rn1;
    rn2->rn_next = rn1->rn_next;
    rn2->rn_prev->rn_next = rn2;
    rn2->rn_next->rn_prev = rn2;
    return;
}

intern void pth_ring_insert_before(pth_ring_t *r, pth_ringnode_t *rn1, pth_ringnode_t *rn2)
{
    if (r == NULL || rn1 == NULL || rn2 == NULL)
        return;
    rn2->rn_next = rn1;
    rn2->rn_prev = rn1->rn_prev;
    rn2->rn_prev->rn_next = rn2;
    rn2->rn_next->rn_prev = rn2;
    return;
}

intern void pth_ring_remove(pth_ring_t *r, pth_ringnode_t *rn)
{
    if (r == NULL || rn == NULL)
        return;
    if (r->rn_hook == rn && rn->rn_prev == rn && rn->rn_next == rn)
        r->rn_hook = NULL;
    else {
        if (r->rn_hook == rn)
            r->rn_hook = rn->rn_next;
        rn->rn_prev->rn_next = rn->rn_next;
        rn->rn_next->rn_prev = rn->rn_prev;
    }
    rn->rn_next = rn;
    rn->rn_prev = rn;
    return;
}

intern void pth_ring_prepend(pth_ring_t *r, pth_ringnode_t *rn)
{
    if (r == NULL || rn == NULL)
        return;
    if (r->rn_hook == NULL) {
        r->rn_hook = rn;
        rn->rn_next = rn;
        rn->rn_prev = rn;
    }
    else {
        rn->rn_next = r->rn_hook;
        rn->rn_prev = r->rn_hook->rn_prev;
        rn->rn_next->rn_prev = rn;
        rn->rn_prev->rn_next = rn;
        r->rn_hook = rn;
    }
    return;
}

intern void pth_ring_append(pth_ring_t *r, pth_ringnode_t *rn)
{
    if (r == NULL || rn == NULL)
        return;
    if (r->rn_hook == NULL) {
        r->rn_hook = rn;
        rn->rn_next = rn;
        rn->rn_prev = rn;
    }
    else {
        rn->rn_next = r->rn_hook;
        rn->rn_prev = r->rn_hook->rn_prev;
        rn->rn_next->rn_prev = rn;
        rn->rn_prev->rn_next = rn;
    }
    return;
}

#if cpp
#define pth_ring_push(r, rn) \
    pth_ring_prepend((r), (rn))
#endif

#if cpp
#define pth_ring_unshift(r, rn) \
    pth_ring_append((r), (rn))
#endif

intern pth_ringnode_t *pth_ring_pop(pth_ring_t *r)
{
    pth_ringnode_t *rn;

    rn = pth_ring_first(r);
    if (rn != NULL)
        pth_ring_remove(r, rn);
    return rn;
}

intern pth_ringnode_t *pth_ring_shift(pth_ring_t *r)
{
    pth_ringnode_t *rn;

    rn = pth_ring_last(r);
    if (rn != NULL)
        pth_ring_remove(r, rn);
    return rn;
}

