/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { useTranslationContext } from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { Amount } from "../../components/Amount.js";
import { AmountField } from "../../components/AmountField.js";
import { EnabledBySettings } from "../../components/EnabledBySettings.js";
import { Part } from "../../components/Part.js";
import { QR } from "../../components/QR.js";
import { SelectList } from "../../components/SelectList.js";
import { TermsOfService } from "../../components/TermsOfService/index.js";
import {
  Input,
  LinkSuccess,
  SvgIcon,
  WarningBox,
} from "../../components/styled/index.js";
import { Button } from "../../mui/Button.js";
import { Grid } from "../../mui/Grid.js";
import editIcon from "../../svg/edit_24px.inline.svg";
import {
  ExchangeDetails,
  WithdrawDetails,
  getAmountWithFee,
} from "../../wallet/Transaction.js";
import { State } from "./index.js";
import { Amounts } from "@gnu-taler/taler-util";

export function FinalStateOperation(state: State.AlreadyCompleted): VNode {
  const { i18n } = useTranslationContext();
  // document.location.href = res.confirmTransferUrl
  if (state.thisWallet) {
    switch (state.operationState) {
      case "confirmed": {
        state.redirectToTx();
        return (
          <WarningBox>
            <div style={{ justifyContent: "center", lineHeight: "25px" }}>
              <i18n.Translate>
                This operation has already been completed.
              </i18n.Translate>
            </div>
          </WarningBox>
        );
      }
      case "aborted": {
        state.redirectToTx();
        return (
          <WarningBox>
            <div style={{ justifyContent: "center", lineHeight: "25px" }}>
              <i18n.Translate>
                This operation has already been aborted
              </i18n.Translate>
            </div>
          </WarningBox>
        );
      }
      case "selected": {
        if (state.confirmTransferUrl) {
          document.location.href = state.confirmTransferUrl;
        }
        return (
          <WarningBox>
            <div style={{ justifyContent: "center", lineHeight: "25px" }}>
              <i18n.Translate>
                This operation has started and should be completed in the bank.
              </i18n.Translate>
            </div>
            {state.confirmTransferUrl && (
              <div style={{ justifyContent: "center", lineHeight: "25px" }}>
                <i18n.Translate>
                  You can confirm the operation in
                </i18n.Translate>
                &nbsp;
                <a
                  target="_bank"
                  rel="noreferrer"
                  href={state.confirmTransferUrl}
                >
                  <i18n.Translate>this page</i18n.Translate>
                </a>
              </div>
            )}
          </WarningBox>
        );
      }
    }
  }

  switch (state.operationState) {
    case "confirmed":
      return (
        <WarningBox>
          <div style={{ justifyContent: "center", lineHeight: "25px" }}>
            <i18n.Translate>
              This operation has already been completed by another wallet.
            </i18n.Translate>
          </div>
        </WarningBox>
      );
    case "aborted":
      return (
        <WarningBox>
          <div style={{ justifyContent: "center", lineHeight: "25px" }}>
            <i18n.Translate>
              This operation has already been aborted
            </i18n.Translate>
          </div>
        </WarningBox>
      );
    case "selected":
      return (
        <WarningBox>
          <div style={{ justifyContent: "center", lineHeight: "25px" }}>
            <i18n.Translate>
              This operation has already been used by another wallet.
            </i18n.Translate>
          </div>
          <div style={{ justifyContent: "center", lineHeight: "25px" }}>
            <i18n.Translate>It can be confirmed in</i18n.Translate>&nbsp;
            <a target="_bank" rel="noreferrer" href={state.confirmTransferUrl}>
              <i18n.Translate>this page</i18n.Translate>
            </a>
          </div>
        </WarningBox>
      );
  }
}

export function SuccessView(state: State.Success): VNode {
  const { i18n } = useTranslationContext();
  return (
    <Fragment>
      <section style={{ textAlign: "left" }}>
        <Part
          title={
            <div
              style={{
                display: "flex",
                alignItems: "center",
              }}
            >
              <i18n.Translate>Exchange</i18n.Translate>
              <EnabledBySettings name="showExchangeManagement">
                <Button onClick={state.doSelectExchange.onClick} variant="text">
                  <SvgIcon
                    title="Edit"
                    dangerouslySetInnerHTML={{ __html: editIcon }}
                    color="black"
                  />
                </Button>
              </EnabledBySettings>
            </div>
          }
          text={
            <ExchangeDetails exchange={state.currentExchange.exchangeBaseUrl} />
          }
          kind="neutral"
          big
        />
        {state.editableAmount ? (
          <Fragment>
            <AmountField handler={state.amount} label={i18n.str`Amount`} />
          </Fragment>
        ) : undefined}
        {state.chooseCurrencies.length > 0 ? (
          <Fragment>
            <p>
              {state.chooseCurrencies.map((currency) => {
                return (
                  <Button
                    key={currency}
                    variant={
                      currency === state.selectedCurrency
                        ? "contained"
                        : "outlined"
                    }
                    onClick={async () => {
                      state.changeCurrency(currency);
                    }}
                  >
                    {currency}
                  </Button>
                );
              })}
            </p>
          </Fragment>
        ) : (
          <Fragment />
        )}

        <Part
          title={i18n.str`Details`}
          text={
            <WithdrawDetails
              conversion={state.conversionInfo?.amount}
              amount={getAmountWithFee(
                state.toBeReceived,
                state.toBeSent,
                "credit",
              )}
              bankFee={state.bankFee}
            />
          }
        />
        {state.ageRestriction && (
          <Input>
            <SelectList
              label={i18n.str`Age restriction`}
              list={state.ageRestriction.list}
              name="age"
              value={state.ageRestriction.value}
              onChange={state.ageRestriction.onChange}
            />
          </Input>
        )}
      </section>

      <section>
        <TermsOfService exchangeUrl={state.currentExchange.exchangeBaseUrl}>
          <Button
            variant="contained"
            color="success"
            disabled={!state.doWithdrawal.onClick}
            onClick={state.doWithdrawal.onClick}
          >
            <i18n.Translate>
              Withdraw &nbsp; <Amount value={state.toBeReceived} />
            </i18n.Translate>
          </Button>
        </TermsOfService>
      </section>
      {state.talerWithdrawUri ? (
        <WithdrawWithMobile talerWithdrawUri={state.talerWithdrawUri} />
      ) : undefined}
    </Fragment>
  );
}

function WithdrawWithMobile({
  talerWithdrawUri,
}: {
  talerWithdrawUri: string;
}): VNode {
  const { i18n } = useTranslationContext();
  const [showQR, setShowQR] = useState<boolean>(false);

  return (
    <section>
      <LinkSuccess upperCased onClick={() => setShowQR((qr) => !qr)}>
        {!showQR ? i18n.str`Withdraw to a mobile phone` : i18n.str`Hide QR`}
      </LinkSuccess>
      {showQR && (
        <div>
          <QR text={talerWithdrawUri} />
          <i18n.Translate>
            Scan the QR code or &nbsp;
            <a href={talerWithdrawUri}>
              <i18n.Translate>click here</i18n.Translate>
            </a>
          </i18n.Translate>
        </div>
      )}
    </section>
  );
}

export function SelectAmountView({
  amount,
  exchangeBaseUrl,
  confirm,
}: State.SelectAmount): VNode {
  const { i18n } = useTranslationContext();
  return (
    <Fragment>
      <section style={{ textAlign: "left" }}>
        <Part
          title={
            <div
              style={{
                display: "flex",
                alignItems: "center",
              }}
            >
              <i18n.Translate>Exchange</i18n.Translate>
            </div>
          }
          text={<ExchangeDetails exchange={exchangeBaseUrl} />}
          kind="neutral"
          big
        />
        <Grid container columns={2} justifyContent="space-between">
          <AmountField label={i18n.str`Amount`} required handler={amount} />
        </Grid>
      </section>
      <section>
        <Button
          variant="contained"
          color="info"
          disabled={!confirm.onClick}
          onClick={confirm.onClick}
        >
          <i18n.Translate>See details</i18n.Translate>
        </Button>
      </section>
    </Fragment>
  );
}
