;;; turbo-man.el --- browse UNIX manual pages (quickly)
;; Keywords: help

;; $Log: turbo-man.el $
; Revision 1.6  1995/08/26  12:58:57  rdw
; Fixed problem with finding sections ending in .Z .gz .z
;
; Revision 1.5  1995/08/26  12:34:57  rdw
; Speeded up formatting slightly when there are many xrefs....
;
; Revision 1.4  1995/08/26  12:33:56  rdw
; Fixed MANPATH problem. (I hope)
;
; Now searches for entries in all sections, but still formats them in
; seperate buffers/frames.
;
; Revision 1.3  1995/08/10  14:01:42  rdw
; Fixed display table stuff to use specifiers and added clicking on xrefs.
;
; Revision 1.2  1995/05/04  17:13:55  rdw
; Added popup menu of sections and xrefs. Changed terminfo to use 
; single characters and switched to using display tables not invisible 
; text to hide text attributes.
;
; Revision 1.1  1995/05/01  23:58:46  rdw
; Initial revision
;
;;
;; Created by Rich Williams <rdw@hplb.hpl.hp.com> 28-apr-95 
;;

(defvar Turbo-man-mode-hook nil
  "*List of hooks to run when entering Turbo man mode.")

(defvar Turbo-man-program "man"
  "*Name of the program to invoke to find and format manual pages.")

(defvar Turbo-man-terminfo-dir nil 
  "*Directory containing the 'emancs' terminfo file.
This is only used if the system supports the terminfo capability 
database. If nil, defaults to 'data-directory'.")

(defvar Turbo-man-chop-headers-and-footers t
  "*t if headers and footers should be chopped, nil otherwise.

See also the variables Turbo-man-keep-first-header and 
Turbo-man-keep-last-footer.")

(defvar Turbo-man-keep-first-header nil
  "*t if the the first header should not be removed, nil otherwise.")

(defvar Turbo-man-keep-last-footer nil
  "*t if the the first header should not be removed, nil otherwise.")

(defvar Turbo-man-directory-list nil "\
A list of directories used to search for manual pages. \

This is initialised by Turbo-man-init-directory-list if not set.")

(make-face 'man-italic)
(or (face-differs-from-default-p 'man-italic)
    (copy-face 'italic 'man-italic))

(make-face 'man-bold)
(or (face-differs-from-default-p 'man-bold)
    (copy-face 'bold 'man-bold))

(make-face 'man-heading)
(or (face-differs-from-default-p 'man-heading)
    (copy-face 'man-bold 'man-heading))

(make-face 'man-xref)
(or (face-differs-from-default-p 'man-xref)
    (set-face-underline-p 'man-xref t))

(defvar Turbo-man-mode-map
  (let ((km (make-sparse-keymap)))
    (set-keymap-name km 'Turbo-man-mode-map)
    (define-key km 'button2 'Turbo-man-mouse-xref)
    (suppress-keymap km t)
    km))

(defvar Turbo-man-display-table
  (let ((dt (make-display-table))
	(ve [nil]))
    (aset dt 1 ve)
    (aset dt 2 ve)
    (aset dt 3 ve)
    (aset dt 4 ve)
    (aset dt 5 ve)
    dt)
  "Display table used in Turbo-man-mode.")

(defun Turbo-man-init-directory-list ()
  "Initialise Turbo-man-directory-list."
  (setq Turbo-man-directory-list nil)
  (let ((manpath (or (getenv "MANPATH") "/usr/local/man:/usr/man"))
	(dirlist nil) dir)
    (setenv "MANPATH" manpath)
    (while (string-match "\\`:*\\([^:]+\\)" manpath)
      (setq dir (substring manpath (match-beginning 1) (match-end 1)))
      (and (not (member dir dirlist))
	   (setq dirlist (cons dir dirlist)))
      (setq manpath (substring manpath (match-end 0))))
    (setq Turbo-man-directory-list (nreverse dirlist))))

(defun Turbo-man-find-section (section &optional vague)
  (let ((reg (if vague 
		 (concat section ".*\\([^.]*\\)$")
	       (concat "^" section "\\.\\([^.]*\\)$")))
	(result nil))
    (if (null Turbo-man-directory-list)
	(Turbo-man-init-directory-list))
    (mapcar 
     #'(lambda (path)
	 (mapcar 
	  #'(lambda (subpath)
	      (mapcar #'(lambda (match)
			  (if (string-match 
			       "^\\(.+\\)\\(\\.\\(gz\\|Z\\|z\\)\\)$" match)
			      (setq match (string-match 1 match)))
			  (if (not (member match result))
			      (setq result (cons match result)))
			  nil)
		      (directory-files subpath nil 
				       reg nil t)))
	  (directory-files path t "man\\|cat" nil 0))
	 nil)
     Turbo-man-directory-list)
    result))

(defun Turbo-man-mode ()
  "Major mode for viewing manual pages."
  (use-local-map Turbo-man-mode-map)
  (setq buffer-read-only t
	truncate-lines t
	major-mode 'Turbo-man-mode
	mode-name "Turbo man")
  (add-spec-to-specifier current-display-table Turbo-man-display-table
			 (current-buffer))
  (run-hooks 'Turbo-man-mode-hook))

(defun Turbo-man-get-buffer (entry)
  "Gets some random buffer ready for Turbo-man to spew into."
  (let ((buf (get-buffer-create (format " * turbo-man - %s *" entry))))
    (set-buffer buf)
    (setq buffer-read-only nil)
    (erase-buffer)
    (kill-all-local-variables)
    (make-local-variable 'Turbo-man-entry-section)
    (make-local-variable 'Turbo-man-xref-list)
    (make-local-variable 'Turbo-man-section-list)
    (setq Turbo-man-entry-section entry)
    buf))

(defun Turbo-man-invoke-man (buf entry &optional section)
  "Invokes the man program, and makes it spew into a buffer."
  (let ((args (if section
		  (concat section " " entry)
		entry)))
    (save-excursion
      (set-buffer buf)
      (let* ((process-environment 
	      (list "TERM=emancs" 
		    (concat "MANPATH="
			    (getenv "MANPATH"))
		    (concat "TERMINFO=" 
			    (if Turbo-man-terminfo-dir
				Turbo-man-terminfo-dir
			      data-directory))
		    (concat "TERMCAP=emancs:co#80:it#1:" ;; cols = 80 tabs = 1
			    "me=\\003:ue=\\003:" ;; all off / underline off = 3
			    "mh=\\001:us=\\002:"))) ;; bold = 1 underline = 2
	     (proc (start-process-shell-command "turbo-man" buf 
						Turbo-man-program 
						args "| ul")))
	(set-process-sentinel proc 'Turbo-man-sentinel)
	(message "Invoking %s %s in the background..." Turbo-man-program args)
	(setq Turbo-man-in-progress (cons (cons entry proc) Turbo-man-in-progress))))))

(defun Turbo-man-sentinel (proc str) 
  (let ((buf (process-buffer proc)) 
	s f n d )
    (delete-process proc)
    (setq s (+ (* 1000000 (nth 1 (current-time))) (nth 2 (current-time))))
    (message "Frobnicating... ")
    (Turbo-man-frobnicate buf)
    (setq f (+ (* 1000000 (nth 1 (current-time))) (nth 2 (current-time)))
	  d (/ (float (- f s)) 1000000)
	  n (/ (float (- (point-max buf) (point-min buf))) d))
    (Turbo-man-make-menu buf)
    (message "Frobnicating... done (%.3f seconds - %.3f chars/second)" d n)
    (save-excursion
      (set-buffer buf)
      (Turbo-man-mode))
    (Turbo-man-display-buffer buf)))

(defun Turbo-man-display-buffer (buf)
  (save-excursion
    (switch-to-buffer-other-frame buf)
    (set-window-start (get-buffer-window buf) 0)))

(defvar Turbo-man-in-progress nil
  "List of (entry . process) pairs which are in progress.")

(defun Turbo-man-xref (section)
  "Follow an xref."
  (Turbo-man-invoke-man (Turbo-man-get-buffer section) section))

(defun Turbo-man-mouse-xref (ev)
  "Follow an xref with the mouse."
  (interactive "e")
  (let* ((pt (event-point ev))
	 (ex (and pt 
		  (extent-at pt (event-buffer ev) 'xref))))
    (if ex
	(Turbo-man-xref (extent-property ex 'xref))
      (error "No xref there."))))

(defun Turbo-man-make-menu (&optional buf)
  "Make a popup menu from the headings / xrefs."
  (save-excursion
    (if buf (set-buffer buf))
    (setq mode-popup-menu
	  (list "Turbo man" 
		  (cons "Sections:"
			(reverse 
			 (mapcar #'(lambda (sp)
				     (vector (car sp) ;; heading
					     `(goto-char ,(cdr sp)) 
					     t))
				 Turbo-man-section-list)))
		  (cons "References:"
			(mapcar #'(lambda (xr)
				    (vector xr
					    `(Turbo-man-xref ',xr) 
					    t))
				(sort Turbo-man-xref-list
				      #'(lambda (a b)
					    (string< a b)))))))))

(defsubst Turbo-man-clean-string (str)
  "Clean STRING by removing any non-printable cruft."
  (let ((st t))
    (mapconcat #'(lambda (x) 
		   ;; This probably should do something better than
		   ;; this. ascii < 32 is probably not a good definition
		   ;; of 'cruft', but its pretty close.
		   (if (< x 32) nil 
		     (if (and st (eq x 32))
			 nil
		       (setq st nil)
		       (char-to-string x))))
	       str nil)))

(defun Turbo-man-frobnicate (buf)
  "Does all sorts of wonderful things."
  (save-excursion
    (set-buffer buf)
    (let (s e x y  m f)
      ;; Highlight bold and underline text.
      ;; Look for ^Abold^C or ^Bunderline^C and create an extent
      (goto-char (point-max))
      (while (re-search-backward;; is backward bad?
	      "\\([\001\002\003]\\).*\\([\001\002\003]\\)" (point-min) t)
	(setq x (make-extent (match-end 1) (match-beginning 2))
	      f (char-after (match-beginning 1)))
	(set-extent-face x (cond ((eq f 1) 'man-bold)
				 ((eq f 2) 'man-italic))))
      ;; Highlight section headings.
      ;; Look for a line with just caps, whitespace and codes on it
      ;; This might be a bit too general - but its quick!
      (setq Turbo-man-section-list nil)
      (goto-char (point-min))
      (while (re-search-forward 
	      "^[ \t\001\002\003A-Z]+$" nil t)
	(setq s (match-beginning 0)
	      x (make-extent s (match-end 0))
	      m (cons (Turbo-man-clean-string (match-string 0)) s)
	      Turbo-man-section-list (cons m Turbo-man-section-list))
	(set-extent-face x 'man-heading)
	(set-extent-priority x 1))
      ;; Highlight and mouseify xrefs."
      ;; Look for reference(section), by looking for (section) 
      ;; and backtracking. Someone once said this would be faster 
      ;; than looking for the whole thing. I guess its probably the 
      ;; brackets and the numbers which make this a winner.
      (goto-char (point-min))
      (setq Turbo-man-xref-list nil) 
      (while (re-search-forward 
	      "([\001\002\003]*\\([0-9][a-zA-Z0-9]*\\)[\001\002\003]*)"
	      nil t)
	(setq e (point)
	      f (match-beginning 0)
	      y (match-string 1))
	(skip-chars-backward "a-zA-Z0-9()-_\001\002\003")
	(setq s (point)
	      x (make-extent s e)
	      m (concat (Turbo-man-clean-string (buffer-substring s f)) "." y))
	(if (not (member m Turbo-man-xref-list))
	    (setq Turbo-man-xref-list (cons m Turbo-man-xref-list)))
	(set-extent-property x 'highlight t)
	(set-extent-property x 'xref m)
	(set-extent-priority x 2)
	(set-extent-face x 'man-xref)
	(goto-char e))
      ;;
      ;; Bold assumptions about page length made here!
      ;; chop first 7 then skip forward 52 chop 14
      ;;
      ;; This is done with 'eyes shut', and will probably 
      ;; have to be changed, as it almost certainly will 
      ;; chop too much / little on anything but hpux.
      ;;
      ;; We (optionally) keep the first and last ones
      ;;
      (if Turbo-man-chop-headers-and-footers
	  (progn
	    (goto-char (point-min))
	    (setq x (make-extent (point) (and (forward-line 7) (point))))
	    (set-extent-property x 'invisible 
				 (not Turbo-man-keep-first-header))
	    (while (not (eobp))
	      (forward-line 52)
	      (setq x (make-extent (point) (and (forward-line 14) (point))))
	      (set-extent-property x 'invisible t))
	    (if Turbo-man-keep-last-footer
		(set-extent-property x 'invisible nil)))))))

(defun Turbo-man-wipe-out ()
  (interactive)
  (map-extents (lambda (e m)
		 (delete-extent e))))

(defun turbo-manual-entry (entry)
  "*Display the Unix manual entry ENTRY."
  (interactive
   (list (let* ((fmh "-A-Za-z0-9_.")
              (default (save-excursion
                         (buffer-substring
                          (progn
                            (re-search-backward "\\sw" nil t)
                            (skip-chars-backward fmh) (point))
                          (progn (skip-chars-forward fmh) (point)))))
              (thing (read-string
                      (if (equal default "") "Manual entry: "
                        (concat "Manual entry: (default " default ") ")))))
         (if (equal thing "") default thing))))
  (mapcar #'(lambda (section)
	      (Turbo-man-invoke-man 
	       (Turbo-man-get-buffer section) section)
	      nil)
	  (Turbo-man-find-section entry)))
