;;;			Command History for Shell Mode
;;;
;;;			       Harlan B. Sexton
;;;
;;;				 hbs@lucid.com
;;;			      Lucid, Incorporated
;;;				  9 June 1988
;;;
;;;
;;; This file provides a command history facility for GNU Emacs Shell Mode.  In
;;; Shell Mode and Inferior-Lisp Mode, a history of commands typed by the user
;;; is automatically maintained, and can be cycled through by M-P (backward) or
;;; M-N (forward).  Type a prefix of a previously typed command and M-P and the
;;; remainder of the command will be inserted.  Giving an argument to M-P will
;;; cause it to prompt for a regular expression, which will be used for
;;; searching the command history.  Only commands longer than two characters
;;; and less than *saved-commands-length-limit* will be kept.


;;; make sure that the shell-stuff is in place
(require 'shell)

(defvar *null-string* "")
(defvar *saved-commands-length-limit* 325)
(setq *buffer-histories-list* nil)
(setq *default-history-length* 100)

;;; entries in a commands node
(setq *previous-commands-vector* nil)
(setq *current-command-index* nil)
(setq *active-command-index* nil)
(setq *previous-command-prefix* nil)
(setq *using-command-pattern* nil)
(setq *kill-backward-mark* nil)

;;; a few functions to maintain our "commands-nodes"
(defun make-commands-node ()
  (vector
    (current-buffer) 
    (make-vector *default-history-length* *null-string*)
    ;; current-command-index
    0
    ;; active-command-index
    0
    ;; previous-command-prefix
    nil
    ;; using-command-pattern
    nil))

(defun add-commands-node ()
  (if (memq (key-binding "
")
            '(meta-p-shell-send-input meta-p-telnet-send-input))
      (let ((node (make-commands-node)))
	(setq *buffer-histories-list*
	      (cons node *buffer-histories-list*))
        node)))

(defun get-commands-node ()
  (let ((buffer (current-buffer))
        (histories-list *buffer-histories-list*)
	possible-node node)
    (while (and histories-list 
                (consp histories-list))
      (setq possible-node (car histories-list))
      (setq histories-list (cdr histories-list))
      (if (eq buffer (aref possible-node 0))
          (progn
            (setq node possible-node)
            (setq histories-list nil))))
    (or node (add-commands-node))))

(defun update-commands-node ()
  (let ((node (get-commands-node)))
    (if (vectorp *previous-commands-vector*)
      (aset node 1 *previous-commands-vector*))
    (if (integerp *current-command-index*)
      (aset node 2 *current-command-index*))
    (if (integerp *active-command-index*)
      (aset node 3 *active-command-index*))
    (if (or (null *previous-command-prefix*)
	      (stringp *previous-command-prefix*))
     (aset node 4 *previous-command-prefix*))
    (aset node 5 *using-command-pattern*)
    node))

(defun retrieve-commands-node ()
  (let ((node (get-commands-node)))
    (setq *previous-commands-vector* (aref node 1))
    (setq *current-command-index* (aref node 2))
    (setq *active-command-index* (aref node 3))
    (setq *previous-command-prefix* (aref node 4))
    (setq *using-command-pattern* (aref node 5))
    (setq *kill-backward-mark* 
	  (process-mark (get-buffer-process (current-buffer))))
    node))

;;; these are the "user" functions for maintaining
;;; commands nodes:

;;; this makes sure we are "in the right buffer"
(defun setup-current-commands ()
  (retrieve-commands-node))

;;; this makes sure the "global" state and
;;; the node agree
(defun save-current-commands ()
  (update-commands-node)
  (retrieve-commands-node))

;;; this makes sure the commands node is up-to-date
(defun update-current-commands ()
  (setq *kill-backward-mark* 
	(process-mark (get-buffer-process (current-buffer))))
  (update-commands-node)
  (retrieve-commands-node))

;;; this resets things for meta-p-shell-send-input
(defun reset-current-commands ()
  (setq *previous-command-prefix* nil)
  (setq *using-command-pattern* nil)
  (update-current-commands))


;;; we redefine this thing so Meta-P will work
(defun meta-p-shell-send-input ()
  "Send input to subshell, saving input > 2 and < *saved-commands-length-limit*
chars in length. Previous commands may be retrieved via Meta-P and Meta-N. At
end of buffer, sends all text after last output as input to the subshell,
including a newline inserted at the end. Not at end, copies current line to the
end of the buffer and sends it, after first attempting to discard any prompt at
the beginning of the line by matching, if possible, the regexp
shell-prompt-pattern (which should start with \"^\")."
  (interactive)
  (shell-completion-cleanup)
  (end-of-line)
  (if (eobp)
      (progn
	(move-marker last-input-start
		     (process-mark (get-buffer-process (current-buffer))))
	(insert ?\n)
	(move-marker last-input-end (point)))
      (beginning-of-line)
      (re-search-forward shell-prompt-pattern nil t)
      (let ((copy (buffer-substring (point)
				    (progn (forward-line 1) (point)))))
	(goto-char (point-max))
	(move-marker last-input-start (point))
	(insert copy)
	(move-marker last-input-end (point))))
  ;; Even if we get an error trying to hack the working directory,
  ;; still send the input to the subshell.
  (condition-case ()
      (save-excursion
	(goto-char last-input-start)
	(cond ((and (looking-at shell-popd-regexp)
		    (memq (char-after (match-end 0)) '(?\; ?\n)))
	       (if shell-directory-stack
		   (progn
		     (cd (car shell-directory-stack))
		     (setq shell-directory-stack 
			   (cdr shell-directory-stack)))))
	      ((looking-at shell-pushd-regexp)
	       (cond ((memq (char-after (match-end 0)) '(?\; ?\n))
		      (if shell-directory-stack
			  (let ((old default-directory))
			    (cd (car shell-directory-stack))
			    (setq shell-directory-stack
				  (cons old 
					(cdr shell-directory-stack))))))
		     ((memq (char-after (match-end 0)) '(?\  ?\t))
		      (let (dir)
			(skip-chars-forward "^ ")
			(skip-chars-forward " \t")
			(if (file-directory-p
			      (setq dir
				    (expand-file-name
				      (substitute-in-file-name
					(buffer-substring
					  (point)
					  (progn
					    (skip-chars-forward
					      "^\n \t;")
					    (point)))))))
			    (progn
			      (setq shell-directory-stack
				    (cons default-directory 
					  shell-directory-stack))
			      (cd dir))
			    (setq changing-working-directory 'pushd))))))
	      ((looking-at shell-cd-regexp)
	       (cond ((memq (char-after (match-end 0)) '(?\; ?\n))
		      (cd (getenv "HOME")))
		     ((memq (char-after (match-end 0)) '(?\  ?\t))
		      (let (dir)
			(forward-char 3)
			(skip-chars-forward " \t")
			(if (file-directory-p
			      (setq dir 
				    (expand-file-name
				      (substitute-in-file-name
					(buffer-substring
					  (point)
					  (progn
					    (skip-chars-forward 
					      "^\n \t;")
					    (point)))))))
			    (cd dir)
			    (setq changing-working-directory 'cd))))))))
    (error nil))
  (let ((process (get-buffer-process (current-buffer)))
	(last-command (buffer-substring last-input-start 
					(1- last-input-end))))
    (setup-current-commands)
    (add-new-command last-command)
    (send-region process last-input-start last-input-end)
    (reset-current-commands)
    (set-marker (process-mark process) (point))))

;;; stick a new command in the "list"; if it is a duplicate, just move it
;;; to the end
(defun add-new-command (new-command)
  (if (and (stringp new-command)
           (> (length new-command) 2)
           (< (length new-command) *saved-commands-length-limit*)	     
           (not (string-equal 
                 new-command
                 (aref *previous-commands-vector*
                       *current-command-index*))))
      (let ((index (incf-index *current-command-index* -1))
            (stop-index *current-command-index*)
            (found-it nil)
            (command nil)
            (vector *previous-commands-vector*))
        (while (and (/= index stop-index)
                    (not found-it))
          (setq command (aref vector index))
          (if (string-equal new-command command)
              (setq found-it t))
          (if (not found-it)
              (setq index (incf-index index -1))))
        (if found-it
            (blt-shift-backward vector index stop-index)
            (progn
              (setq *current-command-index*
                    (incf-index *current-command-index* 1))
              (aset vector *current-command-index* new-command)
              ))))
  (setq *active-command-index* 
	(incf-index *current-command-index* 1))
  (save-current-commands))

;;; we want our *previous-commands-vector* to be
;;; like a doubly-linked list
(defun incf-index (index increment)
  (let ((length (length *previous-commands-vector*)))
    (mod (+ index (+ length increment)) length)))

;;; This function cycles contents of a "subvector" "backward":
;;;
;;;          ____-->______
;;;         /             \
;;;  [    start ...<--... stop    ]
;;;
;;; and
;;;
;;;   _______________-->_______________
;;;  /                                /
;;; [...<--...stop      start...<--...]
;;;
;;; leaving "old" vector[start] in "new" vector[stop]
(defun blt-shift-backward (vector start stop)
  (let ((length (length vector)))
    (if (>= stop start)
	(blt-shift vector start stop -1)
	(let ((end-index (- length 1))
	      (start-elt (aref vector start)))
	  (blt-shift vector start end-index -1)
	  (aset vector end-index (aref vector 0))
	  (blt-shift vector 0 stop -1)
	  (aset vector stop start-elt)))))

;;; the windows-hackers will be green...
(defun blt-shift (vector start stop increment)
  (if (and (< start stop) 
           (<= 0 start)
           (< stop (length vector)))
      (let ((length (length vector)))
        (if (>= increment 0)
            (let ((stop-elt (aref vector stop))
                  (index stop))
              (while (> index start)
                (aset vector index (aref vector (1- index)))
                (setq index (1- index)))
              (aset vector start stop-elt))
            (let ((start-elt (aref vector start))
                  (index start))
              (while (> stop index)
                (aset vector index (aref vector (1+ index)))
                (setq index (1+ index)))
              (aset vector stop start-elt))))))

;;; things for cycling through previous shell input
(defun previous-shell-input-command (arg)
  "Return the most recent input to the current shell buffer which is
more than 2 characters long, and less than the value of the variable
*saved-commands-length-limit* (initially 325). When a \"prefix\" has
been typed to the shell buffer, only input matching this prefix is
returned. Repeated invocations of the function without new input
search farther back. The search space is a cycle of length the
value of '*default-history-length*, initially 100. When called with
a numeric argument other than 1, prompts for a string to use as a
\"regular expression\" for matching previous commands. The command
doesn't do anything unless the point is past the process input mark.
The function is bound to \\[previous-shell-input-command]. 

See also NEXT-SHELL-INPUT-COMMAND."
  (interactive "p")
  (if (memq (key-binding "
")
            '(meta-p-shell-send-input meta-p-telnet-send-input))
      (progn
        (if (not (<= (process-mark (get-buffer-process (current-buffer)))
                     (point)))
            (error "Point is in wrong place for shell input."))
        (if (not (= arg 1))
            (call-interactively 'get-command-pattern))
        (find-command-internal -1))))

;;; cycle the other direction
(defun next-shell-input-command (arg)
  "Similar to PREVIOUS-SHELL-INPUT-COMMAND, except that the search is in the 
\"opposite direction\". Bound to \\[next-shell-input-command]."
  (interactive "p")
  (if (memq (key-binding "
")
            '(meta-p-shell-send-input meta-p-telnet-send-input))
      (progn
        (if (not (<= (process-mark (get-buffer-process (current-buffer)))
                     (point)))
            (error "Point is in wrong place for shell input."))
        (if (not (= arg 1))
            (call-interactively 'get-command-pattern))
        (find-command-internal 1))))

(defun find-command-internal (increment)
  (setup-current-commands)
  (let* ((prefix (previous-command-prefix))
	 (pattern *using-command-pattern*)
	 (command (find-matching-command prefix increment)))
    (if command
	(progn
	  (update-current-commands)      
	  (command-from-string command))
	(if pattern
	    (error "Can't match command-pattern: %s" pattern)
	    (error "Can't match command-prefix: %s" prefix)))))

;;; query the user for a string to use as a regexp for
;;; matching commands
(defun get-command-pattern (pattern)
  (interactive "sPattern (regexp): ")
  (setq *previous-command-prefix* *null-string*)
  (setq *using-command-pattern* pattern)
  (save-current-commands))

;;; find a prefix to use for the search in the previous commands
(defun previous-command-prefix ()
  (interactive)
  (if (and (<= *kill-backward-mark* (point))
           (not
            (buffer-substring-search
             (aref *previous-commands-vector* *active-command-index*)
             *kill-backward-mark* (point))))
      (reset-current-commands))
  (if (stringp *previous-command-prefix*)
      *previous-command-prefix*
      (if (< *kill-backward-mark* (point))
	  (setq *previous-command-prefix*
		(buffer-substring *kill-backward-mark* (point)))
	  (setq *previous-command-prefix* *null-string*))))

;;; search for a match of the prefix in the commands list
(defun find-matching-command (prefix increment)
  (if (not (stringp prefix))
      (setq prefix *null-string*))
  (let ((inc (if (> increment 0) 1 -1))
	(command nil)
	(count 0)
	(found-it nil)
	(vector *previous-commands-vector*)
	(count-limit (length *previous-commands-vector*)))
    (while (and (> count-limit count)
		(not found-it))
      (setq *active-command-index*
	    (incf-index *active-command-index* inc))
      (setq command (aref vector *active-command-index*))
      (setq count (1+ count))
      (if (and (> (length command) 0)
               (if *using-command-pattern*
                   (string-match *using-command-pattern*
                                 command)
                   (substrp prefix command)))
          (setq found-it t)))
    (if found-it
	command
	(progn
	  (reset-current-commands)
	  nil))))

;;; a very slightly less stupid substring checker
(defun substrp (string1 string2)
  (if (and (stringp string1) (stringp string2))
      (let ((len1 (length string1))
            (len2 (length string2)))
        (and (<= len1 len2)
             (or (= len1 0)
                 (string-search string1 string2 0))))))

;;; now that we have the string, put it in the buffer
(defun command-from-string (string)
  (interactive)
  (if (<= *kill-backward-mark* (point))
      (delete-region *kill-backward-mark* (point)))
  (insert-string string))

;;; set the key-map to use these pups
(define-key shell-mode-map "\ep" 'previous-shell-input-command)
(define-key shell-mode-map "\en" 'next-shell-input-command)
(define-key shell-mode-map "\C-m" 'meta-p-shell-send-input)
(define-key inferior-lisp-mode-map "\ep" 'previous-shell-input-command)
(define-key inferior-lisp-mode-map "\en" 'next-shell-input-command)
(define-key inferior-lisp-mode-map "\C-m" 'meta-p-shell-send-input)
