/*	NGT -- Network Graphics Terminal for Apollo Emacs using GPR	     */
/*									     */
/*	Copyright (C) 1987 by Leonard N. Zubkoff, All Rights Reserved	     */
/*									     */
/*	This software is provided free and without any warranty.	     */
/*	Permission to copy for any purpose is hereby granted so		     */
/*	long as this copyright notice remains intact.			     */
/*									     */
/*	Revision:	17-Feb-91 23:37:51				     */


#define begin	    {
#define end	    }
#define then
#define do
#define hidden	    static
#define visible
#define procedure   void


typedef unsigned long	    word;
typedef unsigned short	    halfword;
typedef unsigned char	    byte;


#include <fcntl.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <netinet/tcp.h>


#include <apollo/base.h>
#include <apollo/ec2.h>
#include <apollo/fontn.h>
#include <apollo/gpr.h>
#include <apollo/ios.h>
#include <apollo/kbd.h>
#include <apollo/ms.h>
#include <apollo/pad.h>
#include <apollo/pfm.h>
#include <apollo/tone.h>
#include "ngt.h"
#include "tokstr.h"


#define NIL	    0L


#define Ctrl(Character)	    ((Character) & 0x1F)
#define Meta(Character)	    ((Character) | 0x80)

#define MouseCursorHeight   16
#define MouseCursorWidth    8
#define MouseCursorXOrigin  0
#define MouseCursorYOrigin  0
#define VisibleListSize	    64


hidden int
	Keyboard,
	AcquireCount,
	CursorLine,
	CursorColumn,
	ScreenLines,
	ScreenColumns,
	WindowLines;


hidden short
	FontHeight,
	FontWidth,
	FontOffset,
	FontVerticalSpacing,
	FontHorizontalSpacing,
	FontNameLength,
	ScreenVisibleCount,
	FixedMappings[256],
	MouseCursorPattern[MouseCursorHeight] =
	    { 0x8000, 0xC000, 0xE000, 0xF000,
	      0xF800, 0xFC00, 0xFE00, 0xF000,
	      0xD800, 0x9800, 0x0C00, 0x0C00,
	      0x0600, 0x0600, 0x0300, 0x0300 };


hidden boolean
	ColorDisplay,
	LineClear[256],
	ScreenUnobscured,
	LastScreenUnobscured = true,
	MouseCursorInWindow = true;


hidden pad_$string_t
	FontName;


hidden gpr_$bitmap_desc_t
	ScreenBitmap,
	MouseCursorBitmap,
	MouseCursorTempBitmap;

hidden gpr_$position_t
	MouseCursorPosition;


hidden gpr_$pixel_value_t
	Background,
	Characters;


hidden gpr_$raster_op_t
	ColorCursor1[4] =
	    { gpr_$rop_zeros, gpr_$rop_src, gpr_$rop_zeros, gpr_$rop_src },
	ColorCursor2[4] =
	    { gpr_$rop_zeros, gpr_$rop_zeros, gpr_$rop_src, gpr_$rop_src };


hidden gpr_$mask_32_t
	NormalPlaneMask,
	FullPlaneMask,
	PlaneMask123;


hidden gpr_$window_t
	ScreenVisibleList[VisibleListSize];


hidden ios_$id_t
	NetworkStream;


#define GraphicsEventCount  0
#define NetworkEventCount   1


hidden ec2_$ptr_t
	EventCountPointers[2];


hidden long
	EventCountValues[2];


hidden short
	MetaKey =	    KBD3_$L9;


hidden gpr_$keyset_t
	CodedKeysSet,
	FunctionKeysSet,
	PhysicalKeysSet,
	MouseButtonsSet;

hidden byte
	InputBuffer[16384+1],
	OutputBuffer[4096],
	DisplayBuffer[256],
	*InputLimit,
	*InputPointer = InputBuffer,
	*OutputPointer = OutputBuffer,
	*DisplayPointer = DisplayBuffer;


#define InputByte()	    (*InputPointer++)


#define OutputByte(Character)						    \
    begin								    \
	*OutputPointer++ = (Character);					    \
    end


#define DisplayCharacter(Character)					    \
    begin								    \
	*DisplayPointer++ = (Character);				    \
    end


hidden procedure RemoveAcceptedInput()
    begin
	if (InputPointer < InputLimit) then
	    begin
		byte *Pointer = InputPointer;
		InputPointer = InputBuffer;
		while (Pointer < InputLimit) do
		    *InputPointer++ = *Pointer++;
	    end
	else InputPointer = InputBuffer;
    end


hidden procedure FatalError(char *Message,
			    char *Argument)
    begin
	extern int errno;
	extern char *sys_errlist[];
	if (Argument == NIL) then
	    printf(Message,sys_errlist[errno]);
	else printf(Message,Argument);
	exit(1);
    end

hidden boolean FillInputBuffer()
    begin
	status_$t Status;
	long ByteCount, BytesRead;
	boolean AnyInput = false;
	while (true) do
	    begin
		ByteCount = sizeof(InputBuffer)-1-(InputPointer-InputBuffer);
		BytesRead = ios_$get(NetworkStream,ios_$cond_opt
						   +ios_$no_rec_bndry_opt,
				     (char *)InputPointer,
				     ByteCount,&Status);
		if (Status.all == ios_$end_of_file) then exit(0);
		if (Status.all == ios_$get_conditional_failed) then break;
		if (Status.all != status_$ok) then pfm_$error_trap(Status);
		InputPointer += BytesRead;
		if (BytesRead > 0) then AnyInput = true;
	    end;
	InputLimit = InputPointer;
	return AnyInput;
    end

hidden procedure FlushOutputBuffer()
    begin
	status_$t Status;
	long BytesRemaining = OutputPointer-OutputBuffer, ByteCount;
	OutputPointer = OutputBuffer;
	while (BytesRemaining > 0) do
	    begin
		ByteCount = ios_$putp(NetworkStream,ios_$partial_ok_opt,
				      (char *)OutputPointer,
				      BytesRemaining,&Status);
		if (Status.all != status_$ok) then pfm_$error_trap(Status);
		BytesRemaining -= ByteCount;
		OutputPointer += ByteCount;
	    end;
	OutputPointer = OutputBuffer;
    end


hidden procedure FlushDisplayBuffer()
    begin
	extern procedure ClearRectangle();
	gpr_$coordinate_t CursorX, CursorY;
	status_$t Status;
	int DisplayCount = DisplayPointer-DisplayBuffer, i;
	if (DisplayCount == 0) then return;
	if (!LineClear[CursorLine]) then
	    ClearRectangle(CursorLine,CursorColumn,1,DisplayCount);
	for (i=0; i<ScreenVisibleCount; i++) do
	    begin
		if (!ScreenUnobscured) then
		    begin
			gpr_$set_clip_window(ScreenVisibleList[i],&Status);
			CursorX = CursorColumn*FontWidth;
			CursorY = FontOffset+CursorLine*FontHeight;
			gpr_$move(CursorX,CursorY,&Status);
		    end;
		gpr_$text((char *)DisplayBuffer,(short)DisplayCount,&Status);
	    end;
	CursorColumn += DisplayCount;
	LineClear[CursorLine] = false;
	DisplayPointer = DisplayBuffer;
    end

hidden procedure ClearRectangle(int Line,
				int Column,
				int LineCount,
				int ColumnCount)
    begin
	gpr_$window_t Rectangle;
	status_$t Status;
	int i;
	Rectangle.window_base.x_coord = Column*FontWidth;
	Rectangle.window_base.y_coord = Line*FontHeight;
	Rectangle.window_size.x_size = ColumnCount*FontWidth;
	Rectangle.window_size.y_size = LineCount*FontHeight;
	for (i=0; i<ScreenVisibleCount; i++) do
	    begin
		if (!ScreenUnobscured) then
		    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
		gpr_$rectangle(Rectangle,&Status);
	    end;
	if (ColumnCount == ScreenColumns) then
	    for (i=Line; i<Line+LineCount; i++) do
		LineClear[i] = true;
    end


hidden procedure CopyRectangle(int SourceLine,
			       int SourceColumn,
			       int DestinationLine,
			       int DestinationColumn,
			       int LineCount,
			       int ColumnCount)
    begin
	gpr_$window_t SourceWindow;
	gpr_$position_t DestinationOrigin;
	status_$t Status;
	int i;
	SourceWindow.window_base.x_coord = SourceColumn*FontWidth;
	SourceWindow.window_base.y_coord = SourceLine*FontHeight;
	SourceWindow.window_size.x_size = ColumnCount*FontWidth;
	SourceWindow.window_size.y_size = LineCount*FontHeight;
	DestinationOrigin.x_coord = DestinationColumn*FontWidth;
	DestinationOrigin.y_coord = DestinationLine*FontHeight;
	if (ScreenUnobscured) then
	    gpr_$pixel_blt(ScreenBitmap,SourceWindow,
			   DestinationOrigin,&Status);
	if (ColumnCount == ScreenColumns) then
	    if (SourceLine > DestinationLine) then
		for (i=0; i<LineCount; i++) do
		    LineClear[DestinationLine+i] = LineClear[SourceLine+i];
		else for (i=LineCount-1; i>=0; --i) do
		    LineClear[DestinationLine+i] = LineClear[SourceLine+i];
    end

hidden procedure DisplayCharacterCursor(boolean Activate)
    begin
	gpr_$window_t SourceWindow;
	status_$t Status;
	int i;
	SourceWindow.window_base.x_coord = CursorColumn*FontWidth-1;
	SourceWindow.window_base.y_coord = CursorLine*FontHeight;
	SourceWindow.window_size.x_size = FontWidth-FontHorizontalSpacing+2;
	SourceWindow.window_size.y_size = FontHeight-FontVerticalSpacing+2;
	if (!ColorDisplay) then
	    begin
		gpr_$set_raster_op(0,gpr_$rop_not_dst,&Status);
		for (i=0; i<ScreenVisibleCount; i++) do
		    begin
			if (!ScreenUnobscured) then
			    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
			gpr_$pixel_blt(ScreenBitmap,SourceWindow,
				       SourceWindow.window_base,&Status);
		    end;
		gpr_$set_raster_op(0,gpr_$rop_src,&Status);
	    end
	else if (Activate) then
	    begin
		gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
		gpr_$set_raster_op_mask(PlaneMask123,gpr_$rop_zeros,&Status);
		for (i=0; i<ScreenVisibleCount; i++) do
		    begin
			if (!ScreenUnobscured) then
			    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
			gpr_$additive_blt(ScreenBitmap,SourceWindow,3,
					  SourceWindow.window_base,&Status);
		    end;
		gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
	    end
	else
	    begin
		int Option = (Background & 7) >> 1;
		for (i=0; i<ScreenVisibleCount; i++) do
		    begin
			if (!ScreenUnobscured) then
			    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
			gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
			gpr_$bit_blt(ScreenBitmap,SourceWindow,3,
				     SourceWindow.window_base,0,&Status);
			gpr_$set_raster_op(0,gpr_$rop_zeros,&Status);
			gpr_$set_raster_op(1,ColorCursor1[Option],&Status);
			gpr_$set_raster_op(2,ColorCursor2[Option],&Status);
			gpr_$additive_blt(ScreenBitmap,SourceWindow,0,
					  SourceWindow.window_base,&Status);
		    end;
		gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
	    end;
    end

hidden procedure DisplayMouseCursor(boolean Activate)
    begin
	static gpr_$window_t SourceWindow =
	    { 0, 0, MouseCursorWidth, MouseCursorHeight };
	static gpr_$window_t SourceWindow2 =
	    { 0, 0, MouseCursorWidth, MouseCursorHeight };
	status_$t Status;
	int i;
	if (!MouseCursorInWindow) then return;
	SourceWindow2.window_base = MouseCursorPosition;
	if (!ColorDisplay) then
	    begin
		gpr_$set_raster_op(0,gpr_$rop_src_xor_dst,&Status);
		for (i=0; i<ScreenVisibleCount; i++) do
		    begin
			if (!ScreenUnobscured) then
			    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
			gpr_$pixel_blt(MouseCursorBitmap,SourceWindow,
				       MouseCursorPosition,&Status);
		    end;
		gpr_$set_raster_op(0,gpr_$rop_src,&Status);
	    end
	else if (Activate) then
	    begin
		for (i=0; i<ScreenVisibleCount; i++) do
		    begin
			if (!ScreenUnobscured) then
			    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
			gpr_$set_bitmap(MouseCursorTempBitmap,&Status);
			gpr_$set_raster_op(0,gpr_$rop_src,&Status);
			gpr_$bit_blt(ScreenBitmap,SourceWindow2,0,
				     SourceWindow.window_base,0,&Status);
			gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
			gpr_$bit_blt(ScreenBitmap,SourceWindow2,3,
				     SourceWindow.window_base,0,&Status);
			gpr_$set_raster_op(0,gpr_$rop_src_and_dst,&Status);
			gpr_$bit_blt(MouseCursorBitmap,SourceWindow,0,
				     SourceWindow.window_base,0,&Status);
			gpr_$set_bitmap(ScreenBitmap,&Status);
			gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);
			gpr_$set_raster_op_mask(PlaneMask123,
						gpr_$rop_not_src_and_dst,
						&Status);
			gpr_$additive_blt(MouseCursorTempBitmap,SourceWindow,0,
					  MouseCursorPosition,&Status);
		    end;
		gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
	    end
	else
	    begin
		int Option = (Background & 7) >> 1;
		for (i=0; i<ScreenVisibleCount; i++) do
		    begin
			if (!ScreenUnobscured) then
			    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
			gpr_$set_raster_op(0,gpr_$rop_src_or_dst,&Status);

			gpr_$bit_blt(ScreenBitmap,SourceWindow2,3,
				     MouseCursorPosition,0,&Status);
			gpr_$set_raster_op(0,gpr_$rop_zeros,&Status);
			gpr_$set_raster_op(1,ColorCursor1[Option],&Status);
			gpr_$set_raster_op(2,ColorCursor2[Option],&Status);
			gpr_$additive_blt(ScreenBitmap,SourceWindow2,0,
					  MouseCursorPosition,&Status);
		    end;
		gpr_$set_raster_op_mask(NormalPlaneMask,gpr_$rop_src,&Status);
	    end;
    end


hidden procedure AcquireDisplay(boolean RemoveCharacterCursor)
    begin
	status_$t Status;
	if (AcquireCount++ > 0) then return;
	ScreenUnobscured = gpr_$acquire_display(&Status);
	if (!ScreenUnobscured) then
	    gpr_$inq_vis_list(VisibleListSize,&ScreenVisibleCount,
			      ScreenVisibleList,&Status);
	else ScreenVisibleCount = 1;
	if (RemoveCharacterCursor) then
	    DisplayCharacterCursor(false);
	DisplayMouseCursor(false);
	if (ScreenVisibleCount == 0) then return;
	if (ScreenUnobscured == LastScreenUnobscured) then return;
	OutputByte(N_SetObscured);
	OutputByte(ScreenUnobscured);
	FlushOutputBuffer();
	LastScreenUnobscured = ScreenUnobscured;
    end


hidden procedure ReleaseDisplay(boolean RestoreCharacterCursor)
    begin
	static gpr_$window_t FullClipWindow =
	    { 0, 0, gpr_$max_x_size, gpr_$max_y_size };
	status_$t Status;
	if (--AcquireCount > 0) then return;
	DisplayMouseCursor(true);
	if (RestoreCharacterCursor) then
	    DisplayCharacterCursor(true);
	if (!ScreenUnobscured) then
	    gpr_$set_clip_window(FullClipWindow,&Status);
	gpr_$release_display(&Status);
    end

hidden procedure RefreshProcedure(boolean &Unobscured,
				  boolean &PositionChanged)
    begin
	pad_$window_desc_t CurrentWindow;
	gpr_$position_t WindowOrigin;
	status_$t Status;
	short WindowCount, AcquireCount;
	if (PositionChanged) then
	    begin
		gpr_$force_release(&AcquireCount,&Status);
		pad_$inq_windows(ios_$stdout,&CurrentWindow,
				 1,&WindowCount,&Status);
		while (--AcquireCount >= 0) do
		    Unobscured = gpr_$acquire_display(&Status);
		ScreenLines = (CurrentWindow.height-2)/FontHeight;
		ScreenColumns = (CurrentWindow.width-2)/FontWidth;
		WindowOrigin.x_coord =
		    (CurrentWindow.width-ScreenColumns*FontWidth)/2;
		WindowOrigin.y_coord =
		    (CurrentWindow.height-ScreenLines*FontHeight)/2;
		gpr_$set_coordinate_origin(WindowOrigin,&Status);
	    end;
	ScreenUnobscured = Unobscured;
	if (!ScreenUnobscured) then
	    gpr_$inq_vis_list(VisibleListSize,&ScreenVisibleCount,
			      ScreenVisibleList,&Status);
	else ScreenVisibleCount = 1;
	OutputByte(N_Refresh);
	OutputByte(ScreenLines);
	OutputByte(ScreenColumns);
	OutputByte(ScreenUnobscured);
	FlushOutputBuffer();
	LastScreenUnobscured = ScreenUnobscured;
    end

hidden procedure InitializeTerminal()
    begin
	extern procedure InitDecompressor();
	extern procedure HighlightModeOff();
	extern procedure ClearScreen();
	CursorLine = 0;
	CursorColumn = 0;
	WindowLines = ScreenLines;
	InitDecompressor();
	HighlightModeOff();
	ClearScreen();
    end


hidden procedure ResetTerminal()
    begin
    end


hidden procedure IconizeTerminal()
    begin
	pad_$position_t IconPosition;
	status_$t Status;
	char IconFont[256];
	boolean IconMoved;
	strcpy(IconFont,"/gnuemacs/etc/apollo.icons");
	pad_$set_icon_font(ios_$stdout,1,IconFont,strlen(IconFont),&Status);
	if (Status.all == status_$ok) then
	    pad_$make_icon(ios_$stdout,1,(char)'J',&Status);
	else pad_$make_icon(ios_$stdout,1,(char)'E',&Status);
      	while (true) do
	    begin
		pad_$icon_wait(ios_$stdout,1,&IconMoved,&IconPosition,&Status);
		if (!IconMoved) then break;
	    end;
    end


hidden procedure RingBell()
    begin
	static time_$clock_t BellTime = { 0, 32768 };
	tone_$time(BellTime);
    end

hidden procedure DefineWindow(int NewWindowLines)
    begin
	WindowLines = NewWindowLines;
    end


hidden procedure ClearScreen()
    begin
	extern procedure WriteCursorPosition();
	status_$t Status;
	int i;
	if (ColorDisplay) then
	    gpr_$set_plane_mask_32(FullPlaneMask,&Status);
	for (i=0; i<ScreenVisibleCount; i++) do
	    begin
		if (!ScreenUnobscured) then
		    gpr_$set_clip_window(ScreenVisibleList[i],&Status);
		gpr_$clear(Background,&Status);
	    end;
	if (ColorDisplay) then
	    gpr_$set_plane_mask_32(NormalPlaneMask,&Status);
	WriteCursorPosition(0,0);
    end


hidden procedure ClearToEndOfLine()
    begin
	ClearRectangle(CursorLine,CursorColumn,1,ScreenColumns-CursorColumn);
    end


hidden procedure ClearToEndOfWindow()
    begin
	if (CursorColumn > 0) then
	    begin
		ClearToEndOfLine(ScreenColumns);
		if (CursorLine < WindowLines-1) then
		    ClearRectangle(CursorLine+1,0,
				   WindowLines-CursorLine-1,ScreenColumns);
	    end
	else ClearRectangle(CursorLine,0,WindowLines-CursorLine,ScreenColumns);
    end


hidden procedure WriteCursorPosition(int Line,
				     int Column)
    begin
	gpr_$coordinate_t CursorX, CursorY;
	status_$t Status;
	CursorLine = Line;
	CursorColumn = Column;
	CursorX = CursorColumn*FontWidth;
	CursorY = FontOffset+CursorLine*FontHeight;
	gpr_$move(CursorX,CursorY,&Status);
    end

hidden procedure DeleteCharacters(int DeleteCount)
    begin
	CopyRectangle(CursorLine,CursorColumn+DeleteCount,
		      CursorLine,CursorColumn,
		      1,ScreenColumns-(CursorColumn+DeleteCount));
	ClearRectangle(CursorLine,ScreenColumns-DeleteCount,1,DeleteCount);
    end


hidden procedure OpenSpaces(int OpenCount)
    begin
	CopyRectangle(CursorLine,CursorColumn,
		      CursorLine,CursorColumn+OpenCount,
		      1,ScreenColumns-(CursorColumn+OpenCount));
	ClearRectangle(CursorLine,CursorColumn,1,OpenCount);
    end


hidden procedure InsertSpaces(int InsertCount)
    begin
	CopyRectangle(CursorLine,CursorColumn,
		      CursorLine,CursorColumn+InsertCount,
		      1,ScreenColumns-(CursorColumn+InsertCount));
	ClearRectangle(CursorLine,CursorColumn,1,InsertCount);
	WriteCursorPosition(CursorLine,CursorColumn+InsertCount);
    end


hidden procedure OverwriteSpaces(int OverwriteCount)
    begin
	ClearRectangle(CursorLine,CursorColumn,1,OverwriteCount);
	WriteCursorPosition(CursorLine,CursorColumn+OverwriteCount);
    end


hidden procedure OverwriteCharacters(int OverwriteCount,
				     int OverwriteCharacter)
    begin
	while (--OverwriteCount >= 0) do
	    DisplayCharacter(OverwriteCharacter);
    end


hidden procedure DeleteLines(int Line,
			     int DeleteCount)
    begin
	CopyRectangle(Line+DeleteCount,0,Line,0,
		      WindowLines-(Line+DeleteCount),ScreenColumns);
	ClearRectangle(WindowLines-DeleteCount,0,DeleteCount,ScreenColumns);
    end


hidden procedure InsertLines(int Line,
			     int InsertCount)
    begin
	CopyRectangle(Line,0,Line+InsertCount,0,
		      WindowLines-(Line+InsertCount),ScreenColumns);
	ClearRectangle(Line,0,InsertCount,ScreenColumns);
    end

hidden procedure HighlightModeOff()
    begin
	status_$t Status;
	gpr_$set_text_background_value(Background,&Status);
	gpr_$set_text_value(Characters,&Status);
	gpr_$set_fill_value(Background,&Status);
    end


hidden procedure HighlightModeOn()
    begin
	status_$t Status;
	gpr_$set_text_background_value(Characters,&Status);
	gpr_$set_text_value(Background,&Status);
	gpr_$set_fill_value(Characters,&Status);
    end


hidden procedure EnableApolloFunctionKey(int KeyCode)
    begin
	status_$t Status;
	FixedMappings[KeyCode] = -abs(FixedMappings[KeyCode]);
	if (FixedMappings[KeyCode] < 0) then return;
	lib_$add_to_set(FunctionKeysSet,256,(short)KeyCode);
	gpr_$enable_input(gpr_$function_keys,FunctionKeysSet,&Status);
    end


hidden procedure DisableApolloFunctionKey(int KeyCode)
    begin
	status_$t Status;
	FixedMappings[KeyCode] = abs(FixedMappings[KeyCode]);
	if (FixedMappings[KeyCode] > 0) then return;
	lib_$clr_from_set(FunctionKeysSet,256,(short)KeyCode);
	gpr_$enable_input(gpr_$function_keys,FunctionKeysSet,&Status);
    end


hidden procedure EnableApolloMouseButton(int KeyCode)
    begin
	status_$t Status;
	lib_$add_to_set(MouseButtonsSet,256,(short)KeyCode);
	gpr_$enable_input(gpr_$buttons,MouseButtonsSet,&Status);
    end


hidden procedure DisableApolloMouseButton(int KeyCode)
    begin
	status_$t Status;
	lib_$clr_from_set(MouseButtonsSet,256,(short)KeyCode);
	gpr_$enable_input(gpr_$buttons,MouseButtonsSet,&Status);
    end

hidden procedure SetApolloMetaKey(int MetaKeyCode)
    begin
	status_$t Status;
	MetaKey = MetaKeyCode;
	lib_$init_set(PhysicalKeysSet,256);
	lib_$add_to_set(PhysicalKeysSet,256,MetaKey);
	lib_$add_to_set(PhysicalKeysSet,256,MetaKey+KBD3_$KEY_UP);
	gpr_$enable_input(gpr_$physical_keys,PhysicalKeysSet,&Status);
    end


hidden procedure WritePasteBuffer(int PasteBufferLength)
    begin
	int FileDescriptor =
	    open("`node_data/paste_buffers/default.txt",
		 O_RDWR|O_CREAT|O_TRUNC,0777);
	int ByteCount;
	if (FileDescriptor < 0) then return;
	while (PasteBufferLength > 0) do
	    begin
		ByteCount = InputLimit-InputPointer;
		if (ByteCount == 0) then
		    begin
			RemoveAcceptedInput();
			FillInputBuffer();
			InputPointer = InputBuffer;
			ByteCount = InputLimit-InputPointer;
			if (ByteCount == 0) then sleep(1);
		    end;
		if (ByteCount > PasteBufferLength) then
		    ByteCount = PasteBufferLength;
		write(FileDescriptor,InputPointer,ByteCount);
		InputPointer += ByteCount;
		PasteBufferLength -= ByteCount;
	    end;
	close(FileDescriptor);
    end

hidden procedure ReadPasteBuffer()
    begin
	int FileDescriptor =
	    open("`node_data/paste_buffers/default.txt",O_RDONLY);
	struct stat FileStatBuffer;
	if (FileDescriptor < 0) then return;
	if (fstat(FileDescriptor,&FileStatBuffer) == 0) then
	    begin
		int FileLength = FileStatBuffer.st_size;
		OutputByte(N_PasteBuffer);
		OutputByte((FileLength>>16));
		OutputByte((FileLength>>8));
		OutputByte((FileLength));
		FlushOutputBuffer();
		while (FileLength > 0) do
		    begin
			int ByteCount = FileLength;
			if (ByteCount > 255) then ByteCount = 255;
			OutputByte(0);
			ByteCount =
			    read(FileDescriptor,OutputPointer,ByteCount);
			if (ByteCount > 0) then
			    begin
				OutputBuffer[0] = ByteCount;
				OutputPointer += ByteCount;
			    end;
			FlushOutputBuffer();
			FileLength -= ByteCount;
			if (ByteCount == 0) then break;
		    end;
	    end;
	close(FileDescriptor);
    end


hidden procedure ExecuteDMCommand(byte *DMString,
				  int DMStringLength)
    begin
	status_$t Status;
	pad_$dm_cmd(ios_$stdout,(char *)DMString,
		    (short)DMStringLength,&Status);
    end

hidden int
	NextDynamicTokenNumber;


hidden boolean
	TrailingSpace[8] =
	    { false, false, false, true, false, true, false, true };


hidden char
	CaseConvert1[8] =
	    { 0, 0, ' ', ' ', 0, 0, ' ', ' ' },
	CaseConvert2[8] =
	    { 0, 0, ' ', ' ', 0, 0, 0, 0 },
	Letters1[16] =
	    { 0, 'e', 't', 'i', 'o', 'n', 'a', 's',
	      'r', 'l', 'h', 'c', 'd', 'm', 'u', 'p' },
	Letters2[12] =
	    { 0, 'f', 'g', 'b', 'y', 'w', 'v', 'k', 'x', 'j', 'q', 'z' },
	Shifts[16] =
	    { 24, 28, 16, 20, 8, 12, 0, 4, 24, 28, 16, 20, 8, 12, 0, 4 };


hidden struct
    begin
	word Word0, Word1;
    end
	TokenWords[256];


hidden procedure InitDecompressor()
    begin
	NextDynamicTokenNumber = 0;
    end

hidden procedure DisplayDynamicToken(int TokenNumber,
				     int TokenType)
    begin
	word TokenWord = TokenWords[TokenNumber].Word0;
	int Letter, Nibble = 0;
	Letter = Letters1[(TokenWord>>Shifts[Nibble++])&15];
	if (Letter == 0) then
	    Letter = Letters2[(TokenWord>>Shifts[Nibble++])&15];
	DisplayCharacter(Letter-CaseConvert1[TokenType]);
	while (true) do
	    begin
		Letter = Letters1[(TokenWord>>Shifts[Nibble++])&15];
		if (Letter == 0) then
		    begin
			if (Nibble == 8) then
			    TokenWord = TokenWords[TokenNumber].Word1;
			else if (Nibble == 16) then break;
			Letter = Letters2[(TokenWord>>Shifts[Nibble++])&15];
		    end;
		if (Letter == 0) then break;
		DisplayCharacter(Letter-CaseConvert2[TokenType]);
		if (Nibble == 8) then
		    TokenWord = TokenWords[TokenNumber].Word1;
		else if (Nibble == 16) then break;
	    end;
	if (TrailingSpace[TokenType]) then
	    DisplayCharacter(' ');
    end

hidden procedure ProcessInputBuffer()
    begin
	DefineCommandArgumentCounts;
	InputPointer = InputBuffer;
	*InputLimit = 0xFF;
	while (InputPointer < InputLimit) do
	    begin
		int Character = InputByte();
		int ArgumentCount = CommandArgumentCounts[Character];
		int Argument;
		if (ArgumentCount == 8) then
		    ArgumentCount = 1+(Character & 7);
		else if (ArgumentCount == 9) then
		    ArgumentCount = InputPointer[0]+1;
		if (InputLimit-InputPointer < ArgumentCount) then
		    begin
			--InputPointer;
			break;
		    end;
		if (Character >= L_DefineAndDisplayDynamicToken &&
			Character <= U_DefineAndDisplayDynamicToken) then
		    begin
			int TokenType = (Character>>3) & 7;
			int TokenByteCount = 1+(Character & 7);
			byte *Pointer =
			    (byte *) &TokenWords[NextDynamicTokenNumber];
			TokenWords[NextDynamicTokenNumber].Word0 = 0;
			TokenWords[NextDynamicTokenNumber].Word1 = 0;
			while (--TokenByteCount >= 0) do
			    *Pointer++ = InputByte();
			DisplayDynamicToken(NextDynamicTokenNumber,TokenType);
			NextDynamicTokenNumber++;
			NextDynamicTokenNumber &= 255;
		    end
		else if (Character >= L_DisplayDynamicToken &&
			 Character <= U_DisplayDynamicToken) then
		    DisplayDynamicToken(InputByte(),Character & 7);
		else if (Character >= L_DisplayStaticToken &&
			 Character <= U_DisplayStaticToken) then
		    begin
			char *Pointer =
			    StaticTokens[((Character & 3)<<8)+InputByte()];
			int TokenType = (Character>>2) & 7;
			Character = *Pointer++;
			DisplayCharacter(Character-CaseConvert1[TokenType]);
			while ((Character = *Pointer++) != '\0') do
			    DisplayCharacter(Character
					     -CaseConvert2[TokenType]);
			if (TrailingSpace[TokenType]) then
			    DisplayCharacter(' ');
		    end

		else if (Character >= L_DisplayQuickToken &&
			 Character <= U_DisplayQuickToken) then
		    begin
			char *Pointer = QuickTokens[Character & 0x1F];
			while ((Character = *Pointer++) != '\0') do
			    DisplayCharacter(Character);
		    end
		else
		    begin
			if (Character < ' ' || Character > '~') then
			    FlushDisplayBuffer();
			switch (Character)
				begin
			    case C_InitializeTerminal:
				InitializeTerminal();
				break;
			    case C_ResetTerminal:
				ResetTerminal();
				break;
			    case C_IconizeTerminal:
				ReleaseDisplay(false);
				IconizeTerminal();
				AcquireDisplay(false);
				break;
			    case C_RingBell:
				RingBell();
				break;
			    case C_DefineWindow:
				DefineWindow(InputByte());
				break;
			    case C_ClearScreen:
				ClearScreen();
				break;
			    case C_ClearToEndOfLine:
				ClearToEndOfLine();
				break;
			    case C_ClearToEndOfWindow:
				ClearToEndOfWindow();
				break;
			    case C_HomeCursor:
				WriteCursorPosition(0,0);
				break;
			    case C_CursorDownAndReturn:
				WriteCursorPosition(CursorLine+1,0);
				break;

			    case C_CursorDownAndWriteColumn:
				WriteCursorPosition(CursorLine+1,InputByte());
				break;
			    case C_WriteCursorPosition:
				Argument = InputByte();
				WriteCursorPosition(Argument,InputByte());
				break;
			    case C_DeleteCharacters:
				DeleteCharacters(InputByte());
				break;
			    case C_OpenSpaces:
				OpenSpaces(InputByte());
				break;
			    case C_InsertSpaces:
				InsertSpaces(InputByte());
				break;
			    case C_OverwriteSpaces:
				OverwriteSpaces(InputByte());
				break;
			    case C_OverwriteCharacters:
				Argument = InputByte();
				OverwriteCharacters(Argument,InputByte());
				break;
			    case C_DeleteLines:
				Argument = InputByte();
				DeleteLines(Argument,InputByte());
				break;
			    case C_InsertLines:
				Argument = InputByte();
				InsertLines(Argument,InputByte());
				break;
			    case C_HighlightModeOff:
				HighlightModeOff();
				break;
			    case C_HighlightModeOn:
				HighlightModeOn();
				break;
			    case C_EnableApolloFunctionKey:
				EnableApolloFunctionKey(InputByte());
				break;
			    case C_DisableApolloFunctionKey:
				DisableApolloFunctionKey(InputByte());
				break;

			    case C_EnableApolloMouseButton:
				EnableApolloMouseButton(InputByte());
				break;
			    case C_DisableApolloMouseButton:
				DisableApolloMouseButton(InputByte());
				break;
			    case C_SetApolloMetaKey:
				SetApolloMetaKey(InputByte());
				break;
			    case C_WritePasteBuffer:
				ReleaseDisplay(false);
				Argument = InputByte();
				Argument = (Argument<<8)+InputByte();
				Argument = (Argument<<8)+InputByte();
				WritePasteBuffer(Argument);
				AcquireDisplay(false);
				break;
			    case C_ReadPasteBuffer:
				ReleaseDisplay(false);
				ReadPasteBuffer();
				AcquireDisplay(false);
				break;
			    case C_ExecuteDMCommand:
				ReleaseDisplay(false);
				Argument = InputByte();
				ExecuteDMCommand(InputPointer,Argument);
				InputPointer += Argument;
				AcquireDisplay(false);
				break;
			    default:
				DisplayCharacter(Character);
				break;
			    end;
		    end;
	    end;
	RemoveAcceptedInput();
	FlushDisplayBuffer();
    end

hidden procedure ProcessNetworkInput()
    begin
	while (true) do
	    begin
		if (!FillInputBuffer()) then break;
		AcquireDisplay(true);
		ProcessInputBuffer();
		ReleaseDisplay(true);
	    end;
    end


hidden procedure InitializeNetwork(char *NetworkDisplay)
    begin
	status_$t Status;
	char HostName[256], *HostAndPort = NetworkDisplay, *Pointer = HostName;
	struct hostent *HostEntry;
	struct sockaddr_in InternetSocketAddress;
	struct sockaddr *SocketAddress;
	int SocketAddressLength, One = 1;
	word HostAddress;
	short Port = 6900;
	while (*HostAndPort != ':' &&
	       *HostAndPort != '#' &&
	       *HostAndPort != '\0') do
	    *Pointer++ = *HostAndPort++;
	*Pointer = '\0';
	if (*HostAndPort == ':') then Port += atoi(++HostAndPort);
	else if (*HostAndPort == '#') then Port = atoi(++HostAndPort);
	if ((HostAddress = inet_addr(HostName)) == -1) then
	    begin
		HostEntry = gethostbyname(HostName);
		if (HostEntry == NIL || HostEntry->h_addrtype != AF_INET) then
		    FatalError("Unknown Host '%s'\n",HostName);
		bcopy(HostEntry->h_addr,&InternetSocketAddress.sin_addr,
		      sizeof(InternetSocketAddress.sin_addr));
	    end
	else InternetSocketAddress.sin_addr.s_addr = HostAddress;
	InternetSocketAddress.sin_family = AF_INET;
	InternetSocketAddress.sin_port = htons(Port);
	SocketAddress = (struct sockaddr *) &InternetSocketAddress;
	SocketAddressLength = sizeof(InternetSocketAddress);
	while (true) do
	    begin
		NetworkStream = socket(SocketAddress->sa_family,SOCK_STREAM,0);
		if (NetworkStream < 0) then
		    FatalError("Unable to create socket - %s\n",NIL);
		setsockopt(NetworkStream,IPPROTO_TCP,TCP_NODELAY,
			   &One,sizeof(int));
		if (connect(NetworkStream,SocketAddress,
			    SocketAddressLength) == 0) then break;
		close(NetworkStream);
		sleep(1);
	    end;
	ios_$get_ec(NetworkStream,ios_$get_ec_key,
		    &EventCountPointers[NetworkEventCount],&Status);
	if (Status.all != status_$ok) then pfm_$error_trap(Status);
    end

hidden procedure ProcessGraphicsInput()
    begin
	static boolean MetaDepressed = false;
	gpr_$event_t EventType;
	status_$t Status;
	byte EventData[1];
	gpr_$position_t EventPosition;
	short AcquireCount;
	while (true) do
	    begin
		gpr_$cond_event_wait(&EventType,&EventData[0],
				     &EventPosition,&Status);
		if (EventType == gpr_$no_event) then break;
		if (Status.all != status_$ok) then pfm_$error_trap(Status);
		if (EventType == gpr_$coded_keys) then
		    begin
			int Keystroke = EventData[0];
			OutputByte(N_Keystroke);
			if (MetaDepressed) then
			    OutputByte(Meta(Keystroke))
			else OutputByte(Keystroke);
		    end
		else if (EventType == gpr_$function_keys) then
		    begin
			int Keystroke = EventData[0];
			if (FixedMappings[Keystroke] > 0) then
			    begin
				OutputByte(N_Keystroke);
				if (MetaDepressed) then
				    OutputByte(Meta(FixedMappings[Keystroke]))
				else OutputByte(FixedMappings[Keystroke])
			    end
			else
			    begin
				OutputByte(N_FunctionKey);
				OutputByte(Ctrl('^'));
				OutputByte((MetaDepressed ? 0x4C : 0x48)
					   | (Keystroke >> 6));
				OutputByte(0x40 | (Keystroke & 0x3F));
			    end;
		    end

		else if (EventType == gpr_$physical_keys) then
		    MetaDepressed = (EventData[0] == MetaKey);
		else if (EventType == gpr_$locator_update) then
		    begin
			AcquireDisplay(false);
			MouseCursorPosition = EventPosition;
			ReleaseDisplay(ColorDisplay);
		    end
		else if (EventType == gpr_$buttons) then
		    begin
			AcquireDisplay(false);
			MouseCursorPosition = EventPosition;
			ReleaseDisplay(ColorDisplay);
			OutputByte(N_Buttons);
			OutputByte(Ctrl('^'));
			OutputByte(EventData[0]+(MetaDepressed ? 16 : 0));
			OutputByte(8+(MouseCursorPosition.x_coord
				      +MouseCursorXOrigin)/FontWidth);
			OutputByte(8+(MouseCursorPosition.y_coord
				      +MouseCursorYOrigin)/FontHeight);
		    end
		else if (EventType == gpr_$entered_window) then
		    begin
			AcquireDisplay(false);
			MouseCursorInWindow = true;
			MouseCursorPosition = EventPosition;
			ReleaseDisplay(ColorDisplay);
		    end
		else if (EventType == gpr_$left_window) then
		    begin
			AcquireDisplay(false);
			MouseCursorInWindow = false;
			ReleaseDisplay(ColorDisplay);
		    end;
	    end;
	if (OutputPointer > OutputBuffer) then
	    begin
		gpr_$force_release(&AcquireCount,&Status);
		pad_$pop_push_window(ios_$stdout,1,true,&Status);
		while (--AcquireCount >= 0) do
		    ScreenUnobscured = gpr_$acquire_display(&Status);
	    end;
	FlushOutputBuffer();
    end

hidden procedure InitializeGraphics()
    begin
	pad_$string_t KeyboardType;
	fontn_$table_tver1 *FontTable;
	pad_$window_desc_t CurrentWindow;
	pad_$position_t IconPosition;
	gpr_$offset_t ScreenBitmapSize, MouseCursorBitmapSize;
	gpr_$rgb_plane_t HiPlane;
	gpr_$position_t WindowOrigin;
	gpr_$attribute_desc_t CursorAttributeBlock;
	gpr_$keyset_t KeySet;
	status_$t Status;
	word LengthMapped;
	short KeyboardLength, WindowCount, FontID, Key;
	short *MouseCursorBitmapPointer, CursorLineWidth, i;
	boolean IconMoved;
	pad_$inq_kbd(ios_$stdin,sizeof(KeyboardType),KeyboardType,
		     &KeyboardLength,&Status);
	Keyboard = KeyboardType[0]-'0';
	AcquireCount = 0;
      	while (true) do
	    begin
		pad_$icon_wait(ios_$stdout,1,&IconMoved,&IconPosition,&Status);
		if (!IconMoved) then break;
	    end;
	pad_$inq_font(ios_$stdout,&FontWidth,&FontHeight,
		      FontName,sizeof(FontName),&FontNameLength,&Status);
	if (Status.all != status_$ok) then
	    begin
		printf("Pad_$Inq_Font failed, using f7x13.b\n");
		strcpy(FontName,"/sys/dm/fonts/f7x13.b");
		FontNameLength = strlen(FontName);
	    end;
	pad_$raw(ios_$stdin,&Status);
	FontTable =
	    (fontn_$table_tver1 *) ms_$mapl(FontName,FontNameLength,
					    0,sizeof(fontn_$table_tver1),
					    ms_$nr_xor_1w,ms_$r,
					    false,&LengthMapped,&Status);
	FontHeight = (FontTable->max_height)+(FontTable->v_spacing);
	FontWidth = (FontTable->space_size)+(FontTable->inter_space_dflt);
	FontOffset = FontTable->baseline_offset+FontTable->v_spacing/2-1
		     -(FontTable->font_height+1-FontTable->max_height)/2;
	FontVerticalSpacing = FontTable->v_spacing;
	FontHorizontalSpacing = FontTable->inter_space_dflt;
	ms_$unmap(FontTable,LengthMapped,&Status);

	pad_$set_scale(ios_$stdout,1,1,&Status);
	pad_$inq_windows(ios_$stdout,&CurrentWindow,1,&WindowCount,&Status);
	ScreenLines = (CurrentWindow.height-2)/FontHeight;
	ScreenColumns = (CurrentWindow.width-2)/FontWidth;
	ScreenBitmapSize.x_size = gpr_$max_x_size;
	ScreenBitmapSize.y_size = gpr_$max_y_size;
	gpr_$init(gpr_$direct,ios_$stdout,ScreenBitmapSize,
		  gpr_$highest_plane,&ScreenBitmap,&Status);
	gpr_$inq_bitmap_dimensions(ScreenBitmap,&ScreenBitmapSize,
				   &HiPlane,&Status);
	ColorDisplay = (HiPlane > 0);
	Background = (ColorDisplay ? gpr_$inq_background(&Status) : 0);
	Characters = (ColorDisplay ? 0 : 1);
	if (ColorDisplay) then
	    begin
		NormalPlaneMask = (1<<4)-1;
		FullPlaneMask = (1<<(HiPlane+1))-1;
		PlaneMask123 = (1<<1)|(1<<2)|(1<<3);
		gpr_$set_plane_mask_32(NormalPlaneMask,&Status);
	    end; 
	WindowOrigin.x_coord = (CurrentWindow.width-ScreenColumns*FontWidth)/2;
	WindowOrigin.y_coord = (CurrentWindow.height-ScreenLines*FontHeight)/2;
	gpr_$set_coordinate_origin(WindowOrigin,&Status);
	gpr_$set_obscured_opt(gpr_$input_ok_if_obs,&Status);
	gpr_$set_refresh_entry(RefreshProcedure,(gpr_$rhdm_pr_t)NIL,&Status);
	gpr_$load_font_file(FontName,FontNameLength,&FontID,&Status);
	gpr_$set_text_font(FontID,&Status);
	MouseCursorBitmapSize.x_size = MouseCursorWidth;
	MouseCursorBitmapSize.y_size = MouseCursorHeight;
	gpr_$allocate_attribute_block(&CursorAttributeBlock,&Status);
	gpr_$allocate_bitmap(MouseCursorBitmapSize,0,CursorAttributeBlock,
			     &MouseCursorBitmap,&Status);
	gpr_$acquire_display(&Status);
	gpr_$allocate_hdm_bitmap(MouseCursorBitmapSize,0,CursorAttributeBlock,
				 &MouseCursorTempBitmap,&Status);
	if (Status.all != status_$ok) then
	    gpr_$allocate_bitmap(MouseCursorBitmapSize,0,CursorAttributeBlock,
				 &MouseCursorTempBitmap,&Status);
	gpr_$release_display(&Status);

	gpr_$inq_bitmap_pointer(MouseCursorBitmap,
				(char **)&MouseCursorBitmapPointer,
				&CursorLineWidth,&Status);
	for (i=0; i<MouseCursorHeight; i++) do
	    begin
		*MouseCursorBitmapPointer = MouseCursorPattern[i];
		MouseCursorBitmapPointer += CursorLineWidth;
	    end;
	bzero(FixedMappings,sizeof(FixedMappings));
	FixedMappings[KBD3_$ESC] = Ctrl('[');
	FixedMappings[KBD3_$TAB] = Ctrl('I');
	FixedMappings[KBD3_$RET] = Ctrl('M');
	FixedMappings[KBD3_$BS] = 0x7F;
	FixedMappings[KBD3_$DEL] = 0x7F;
	for (i=0; i<=9; i++) do
	    FixedMappings[KBD3_$NP0+i] = '0'+i;
	FixedMappings[KBD3_$NPE] = Ctrl('M');
	FixedMappings[KBD3_$NPF] = '-';
	FixedMappings[KBD3_$NPG] = '+';
	FixedMappings[KBD3_$NPP] = '.';
	lib_$init_set(CodedKeysSet,256);
	lib_$init_set(FunctionKeysSet,256);
	lib_$init_set(PhysicalKeysSet,256);
	lib_$init_set(MouseButtonsSet,256);
	for (Key=0; Key<0x80; Key++) do
	    lib_$add_to_set(CodedKeysSet,256,Key);
	for (Key=0; Key<256; Key++) do
	    if (FixedMappings[Key] > 0) then
		lib_$add_to_set(FunctionKeysSet,256,Key);
	lib_$add_to_set(PhysicalKeysSet,256,MetaKey);
	lib_$add_to_set(PhysicalKeysSet,256,MetaKey+KBD3_$KEY_UP);
	gpr_$enable_input(gpr_$coded_keys,CodedKeysSet,&Status);
	gpr_$enable_input(gpr_$function_keys,FunctionKeysSet,&Status);
	gpr_$enable_input(gpr_$physical_keys,PhysicalKeysSet,&Status);
	gpr_$enable_input(gpr_$locator_update,MouseButtonsSet,&Status);
	gpr_$enable_input(gpr_$entered_window,MouseButtonsSet,&Status);
	gpr_$enable_input(gpr_$left_window,MouseButtonsSet,&Status);
	gpr_$set_quit_event(gpr_$keystroke,'\377',&Status);
	gpr_$get_ec(gpr_$input_ec,
		    &EventCountPointers[GraphicsEventCount],&Status);
	EventCountValues[GraphicsEventCount] = 0;
    end


hidden procedure SendDisplayConfiguration()
    begin
	status_$t Status;
	byte Buffer[3];
	Buffer[0] = Keyboard;
	Buffer[1] = ScreenLines;
	Buffer[2] = ScreenColumns;
	ios_$put(NetworkStream,ios_$no_put_get_opts,
		 (char *)Buffer,sizeof(Buffer),&Status);
	if (Status.all != status_$ok) then
	    FatalError("Client from wrong Host\n","");
    end

visible int main(ArgCount,ArgVector,Environment)
	int ArgCount;
	char *ArgVector[], *Environment[];
    begin
	status_$t Status;
	if (ArgCount != 2) then
	    FatalError("Usage: ngt <network-display>\n","");
	InitializeNetwork(ArgVector[1]);
	InitializeGraphics();
	SendDisplayConfiguration();
	while (true) do
	    begin
		switch (ec2_$wait(EventCountPointers,
				  EventCountValues,2,&Status)-1)
		    begin
		case GraphicsEventCount:
		    EventCountValues[GraphicsEventCount] =
			1+ec2_$read(*EventCountPointers[GraphicsEventCount]);
		    ProcessGraphicsInput();
		    break;
		case NetworkEventCount:
		    EventCountValues[NetworkEventCount] =
			1+ec2_$read(*EventCountPointers[NetworkEventCount]);
		    ProcessNetworkInput();
		    break;
		    end;
	    end;
    end
