/*	STRINGOPS -- String Search Operations for GNU ELISP		     */
/*									     */
/*	Copyright (C) 1988 by Harlan B. Sexton, All Rights Reserved	     */
/*									     */
/*	This software is provided free and without any warranty.	     */
/*	Permission to copy for any purpose is hereby granted so		     */
/*	long as this copyright notice remains intact.			     */


#include "config.h"
#include "lisp.h"
#include "buffer.h"


DEFUN ("string-search", Fstring_search, Sstring_search, 3, 5, 0,
       "NIL if contents of first string are not identical to any of the \n\
specified substrings of the second string. (The substrings are those \n\
beginning at offset given by the third argument, and number the value \n\
of the (optional) fourth argument - this count argument defaults to 1. \n\
If the search is successful, function returns the smallest (legal) offset \n\
into the second string at which a copy of the first string resides. \n\
If the fifth arg is non-NIL, the search is run \"backward\".\n\
Symbols are also  allowed; their print names are used instead.")
(s1, s2, o, c, go_backward)
Lisp_Object s1, s2, o, c, go_backward;
{
  register int offset, count, size, direction;
  register char *str1, *str2;
  Lisp_Object value;

  /* verify the string arguments */
  if (XTYPE (s1) == Lisp_Symbol)
    XSETSTRING (s1, XSYMBOL (s1)->name), XSETTYPE (s1, Lisp_String);
  if (XTYPE (s2) == Lisp_Symbol)
    XSETSTRING (s2, XSYMBOL (s2)->name), XSETTYPE (s2, Lisp_String);
  CHECK_STRING (s1, 0);
  CHECK_STRING (s2, 1);

  /* verify the offset argument */
  CHECK_NUMBER (o, 2);
  offset = XINT(o);
  if (offset < 0) return Qnil;
	
  /* verify the count argument */
  if (NULL (c))
    count = 1;
  else
    { 
      CHECK_NUMBER (c, 3);
      count = XINT(c);
      if (count <= 0) return Qnil;
    }
  
  /* make sure that the second string argument is long enough */
  if (XSTRING (s1)->size > ((XSTRING (s2)->size) - offset))
    return Qnil;


  /* make sure that the count is not too big */
  if (XSTRING (s1)->size > ((XSTRING (s2)->size) - (offset + count-1)))
    count = 1 + (XSTRING (s2)->size) - (offset + (XSTRING (s1)->size));

  /* set up some locals */
  size = XSTRING(s1)->size;
  str1 = (char *) XSTRING(s1)->data;
  /* note that the "second string" really starts at (s2->data)+offset */
  str2 = (char *) XSTRING(s2)->data;
  str2 += offset;
     
  /* set up the "direction" */
  if (NULL (go_backward))
    direction = 1;
  else
    {
      direction = -1;
      /* if we are going in reverse, we increment our string and the offset
	 variable to point at the last substring */
      str2 += count-1;
      offset += count-1;
    }

  /* look for a copy of the first string inside the second */
  while (count > 0)
    {
      if (bcmp (str1,str2,size) == 0)
	{
	  XFASTINT (value) = offset;
	  return (value);
	}
      count--;
      offset += direction;
      str2 += direction;
    }
  
  /* if we got here, we must have failed to find a match */
  return Qnil;
}


DEFUN ("buffer-substring-search", Fbuffer_substring_search, 
Sbuffer_substring_search, 3, 5, 0,
       "Same as STRING-SEARCH of first (string) argument, buffer-substring \n\
specified by the second and third args, and the fourth through \n\
sixth args, except that if successful the function returns the buffer \n\
offset at which the substring match occurs.")
(s, b, e, c, go_backward)
Lisp_Object s, b, e, c, go_backward;
{
  Lisp_Object value;
  register char *str;
  register int beg, end_or_size, count, direction;

  /* verify the beginning and end marks */
  validate_region (&b, &e);
  beg = XINT (b);
  end_or_size = XINT (e);

  /* verify the count argument */
  if (NULL (c))
    count = 1;
  else
    { 
      CHECK_NUMBER (c, 0);
      count = XINT(c);
      if (count <= 0) return Qnil;
    }

  /* verify the string we are trying to match */
  if (XTYPE (s) == Lisp_Symbol)
    XSETSTRING (s, XSYMBOL (s)->name), XSETTYPE (s, Lisp_String);
  CHECK_STRING (s, 1);

  /* make sure that the specified region is big enough for the string */
  if ( (XSTRING (s)->size) > (end_or_size - beg) )
    return Qnil;

  /* make sure that the count is not too big */
  if ( (XSTRING (s)->size) > (end_or_size - (beg + count-1)) )
    count = 1 + (end_or_size - ((XSTRING (s)->size) + beg));

  /* if the region straddles the gap, move the gap */
  if (beg <= GPT && end_or_size > GPT)
    move_gap (beg);

  /* use the end_or_size var to hold the size now - it just might be
     in a register... */
  end_or_size = XSTRING (s)->size;
  str = (char *) XSTRING (s)->data;

  /* set up the "direction" */
  if (NULL (go_backward))
    direction = 1;
  else
    {
      direction = -1;
      /* if we are going in reverse, we "begin" at beg+count-1 */
      beg += count-1;
    }

  /* look for a copy of the string argument in the specified region */
  while (count > 0)
    {
      if (bcmp (str, (char *) &FETCH_CHAR(beg), end_or_size) == 0)
	{ 
	  XFASTINT (value) = beg;
	  return (value);
	}
      count --;
      beg += direction;
    }

  /* if we got here, we must have failed to find a match */
  return Qnil;
}


syms_of_string_ops ()
{
  defsubr (&Sstring_search);
  defsubr (&Sbuffer_substring_search);
}
