;;; -*- Mode: Emacs-Lisp -*-
;; Copyright (c) 1988,1989, Apollo Computer, Inc. 
;; 
;; Mark Weissman
;; 
;; The software and information herein are the property
;; of Apollo Computer, Inc.
;;
;; This file may be distributed without further permission 
;; from Apollo Computer, Inc. as long as:
;; 
;;    * all copyright notices and this notice are preserved
;;    * it is not part of a product for resale
;;    * a detailed description of all modifications to this
;;      file is included in the modified file.
;;  
;; Version 1.1
;; Fri Sep  8 10:11:26 1989
;;
;; This software is normally unsupported and without warranty.
;; Support will be provided only for customers of 
;; Domain/CommonLISP from APOLLO Computers Inc.
;;
;; This file provides a gnuemacs interface to Domain/CommonLISP
;; designed for use on APOLLO workstations.  This code should
;; work with LUCID Common Lisp with X windows.  Other Common
;; Lisp's may be used with some appreciated porting effort.
;;
;; Questions, comments and enhancements are welcome from
;; all sources.
;;
;; Send questions, comments and enhancements to:
;;  lisp-tools:@apollo.com
;;  mit-eddie!ulowell!apollo!lisp-tools:
;; or 
;;  Domain/CommonLISP Group
;;  Apollo Computer Inc.
;;  330 Billerica Road
;;  Chelmsford MA 01824
;;


;;;
;; Known Deficiencies
;;  Symbol-Count is ignored from attribute list
;;  Some of the description commands only work for common lisp not emacs lisp
;;  No Source compare merge facility
;;  No Patch file facility exists
;;  No Edit Warnings facility
;;  If user changes keys, may give incorrect help messages
;;  May lose by renaming inferior lisp buffer.
;;  Inferior lisp communication loses if lisp ignores errors or in GC
;;  Changed definitions only knows files from load-file or load-compile-file.
;;  Changed definitions won't know about recursively lisp-loaded files


;;; Begin Instructions
;    Documentation strings for Emacs lisp extensions
;    Mark Weissman
;    Apollo Computer Inc.
;
;;;; To Use Domain/CommonLISP Programming Environment:
;
;Add the following line into the file: ~/user_data/common_lisp/startup.lisp
;       (load "/sys/common_lisp/gnuemacs_support/apollo-lisp-support")
;Add the following lines into the file ~/.emacs
;       (load "/sys/common_lisp/gnuemacs_support/clisp.elc")
;        ;;; Note that the lisp called will be the same as when
;        ;;; typing the value of inferior-lisp-program to the shell.
;       (setq inferior-lisp-program "lisp")
;       ;; (setq apollo:gnuemacs-client-p t) ;; use newer interface
;       (save-excursion (lisp))
;
;There is newer, faster, more robust code to handle the communication
;interface between emacs and Domain/CommonLISP 3.0.  This code has not been
;tested as fully as the current interface.  If you wish to use this
;newer interface, uncomment the line: 
;        ;; (setq apollo:gnuemacs-client-p t) ;; use newer interface
;in the above code.
;
;To use RMAIL on an Apollo ring running Apollo DPSS/MAIL (not UNIX Mail), add the
;following lines into the file ~/.emacs:
;       (setq rmail-primary-inbox-list '("~/user_data/mail/inbox/"))
;       (load "/sys/common_lisp/gnuemacs_support/dpss-rmail-support.elc")
;  and if you are running a mail server add to speed up mail access:
;       (setq apollo-rmail-fetch nil)
;Note: dpss-rmail-support is for DPSS mail and not for UNIX sendmail.  Loading
;      dpss-rmail-support will cause UNIX sendmail to stop working!
;
;The last line in your ~/.emacs file should be:
;       (switch-to-buffer "*scratch*")
;
;
;The following is a description of the functionality made available by doing the above.
;
;;; End Instructions
;
;;; Concepts
;
; 1. Hooks
; 
; Hooks are lists of functions which are automatically executed before or after an
; Emacs Lisp function. They come in 3 varieties:
;       1. before-hooks: a list of functions which are funcalled (executed) before
;          the original function is executed. These functions are called with the
;          same arguments as the original function.
;       2. unwind-hooks:  a list of functions which are funcalled after the original
;          function is executed. These functions are called with the same arguments as
;          the original function.  These functions are unwind protected so that they
;          will run in the presence of throws, errors, aborts etc.
;       3. after-hooks: a list of functions which are funcalled after the original
;          function is executed.  These functions will be called with the result of
;          executing the original function followed by the original arguments.
; When a function has hooks added to it (via the macro apollo:add-hooks), the following
; local-variables are created:
;          <function-name> bound to the symbol function of the original function
;          <function-name>-documentation which will be the doc string for the original function.
;          <function-name>-before-hooks, the list of before-hooks.

;          <function-name>-unwind-hooks, the list of unwind-hooks.
;          <function-name>-after-hooks, the list of after-hooks.
;
; An example of where you might see hooks mentioned is when you type meta-x 
; describe-function message. The resultant output is:
;
;        message:
;        Print a one-line message at the bottom of the screen.
;        The first argument is a control string.
;        It may contain %s or %d or %c to print successive following arguments.
;        %s means print an argument as a string, %d means print as number in decimal,
;        %c means print a number as a single character.
;        The argument used by %s must be a string or a symbol;
;        the argument used by %d or %c must be a number.
;
;        Added hooks: apollo:message-before-hooks,
;                     apollo:message-unwind-hooks &
;                     apollo:message-after-hooks.
;
; Notice that the Emacs Lisp function message has before, unwind, and after
; hooks listed. If you evaluated the symbols, you would see that
; apollo:message-before-hooks was bound to the list: (apollo:notification),
; and that apollo:message-unwind-hooks and apollo:message-after-hooks were
; both bound to NIL. This means that the function message only has a
; before-hook, apollo:notification, which is called with the same arguments
; as the original function, message, prior to its execution.
;
;
; 2. Sections
;
; A section is a compilation unit.  Currently they are only supported in Lisp and
; Emacs-Lisp Modes.  The current definition used for incremental compilation by 
; isapollo:lisp-compile-defun the section in which the cursor is found. To see the
; sections of a buffer use the command meta-x list-sections.
;
;
; 3. S-Expression (abbreviated sexp)
;
; A symbol, string, bracketed expression, or list of s-expressions.
; A list is a parenthesized expression. In non-Lisp languages, where
; s-expressions are used in commands (e.g. mouse-insert-thing), the 
; behavior may occasionally be unexpected (e.g. a Lisp symbol is a 
; superset of C identifiers).
;
;
; 4. Region
;
; The area of text between mark and point.  Mark may be explicitly set
; using control-space (some commands set mark automatically). Point is 
; the location of the cursor. Many commands operate on a region (e.g.
; control-y).
;
;
; 5. Modes
;
; Modes may affect the applicability and/or behavior of commands.
; Commands prefixed by Control-c are either present or absent based on the
; mode of a buffer (e.g. Control-c . is only available in Lisp modes).
; Meta-x commands may do nothing in some modes (e.g. meta-x 
; evaluate-into-buffer in non-Lisp buffers) or behave differently
; (e.g. meta-x update-attribute-list will insert a language dependent
; leading comment in a buffer).

;
;
; 6. Attribute List
;
; Each file or Gnuemacs buffer has a set of attributes which can be displayed
; textually. It must be the first line in the source and should be a comment in the
; language (if there is one) of the source code. The attribute list parser expects
; a list of keyword-value pairs between -*- delimiters. The attribute list appears
; as follows:
;         CommentStart -*- Attribute1: Value1; AttributeN: ValueN -*- CommentEnd
;       where:
;         CommentStart is language dependent (e.g. ; in Lisp, and /* in C)
;         CommentEnd would be the null string in Lisp and */ in C.
; A sample attribute list might look like:
;       ;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: Apollo -*-
;
; This is parsed when the file is read in and the appropriate variables are then
; set in the buffer.  These attributes can affect what commands are applicable,
; as the Mode does, or cause the compiler to exhibit different behavior as the
; Syntax and Package attributes do.
;
;
; 7. Numeric Arguments to Emacs Commands
;
; Numeric arguments are used to modify the behavior of a command.
; When numeric arguments are available for a specific command, then
; the command description will mention "ARG N" where N is an integer.
; The way to explicitly pass an argument to a meta-x, control-x, or 
; control-c command is via typing meta-N, followed by the command
; (e.g. you would type "meta-2 meta-x evaluate-common-lisp" to
; have the results displayed in a buffer named *LISP-OUTPUT* ).
; A synonym for the use of meta-N above is control-u N, thus the example
; could read "control-u 2 meta-x evaluate-common-lisp" to achieve the
; same behavior.
;
;;; Further Notes
;
; 1. The meta and Escape keys are similar, but have some important 
; differences. The Escape key is a command prefix, i.e. it is
; an independent keystroke.  The meta key is a shift key. Thus
; "meta-." executes the same command as "<escape> .", but the former
; is one keystroke and the latter is two (i.e. the meta key must be
; held down while typing the ".", but the Escape key is typed prior
; to typing the ".").
;
;
;;; End Concepts


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Things to think about
;;; Source-Compare-Merge
;;; Edit-Warnings
;;; Flavor stuff
;;; lisp trace to buffer?
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
(defun apollo:get-os-version ()
  (let ((bldt (cond 
                ((file-exists-p (concat apollo:aegis-directory "/bldt"))
                 (concat apollo:aegis-directory "/bldt"))
                ((file-exists-p "/com/bldt") "/com/bldt")
                ((file-exists-p "/usr/apollo/bin/bldt") "/usr/apollo/bin/bldt"))))
    (if bldt
        (progn
          (set-buffer (get-buffer-create " *HACK*"))
          (delete-region (point-min) (point-max))
          (call-process bldt nil t nil)
          (goto-char (point-min))
          (re-search-forward "revision[ \t\f\r\n]*\\([^ \t\f\r\n]*\\)[ \t\f\r\n]*," nil t)
          (let ((r (buffer-substring (match-beginning 1) (match-end 1))))
            (kill-buffer (current-buffer))
            r))
        "")))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Configuration variables

(defvar apollo:aegis-directory "/com"
  "Location of Apollo Domain Aegis Files")

(defvar apollo:os-version (apollo:get-os-version)
  "String indicating version of Domain OS.
        Does bldt command to find string to use as default.")

(defvar apollo:*lowercase-file-directories* (eql 0 (string-match "9" apollo:os-version))
  "If true (that is, the variable has a non-nil value), Emacs changes
all uppercase letters in directory pathnames to the pre-SR10 convention of using
lowercase letters preceded by colons.  For example, DOC becomes :d:o:c and Doc
becomes :doc.

        If false (that is, the variable has a nil value), uppercase letters in
directory pathnames remain unchanged.")

(defvar apollo:lisp-interrupt 
  (if (eql 0 (string-match "9" apollo:os-version)) 'quit-process 'interrupt-process)
  "This must be reset according to Apollo Domain OS!
        This is the argument to /bin/kill to provide a keyboard
        interrupt to lisp.
        For SR10  use interrupt-process
        For SR9.7 use quit-process")

(defvar apollo:eval-on-closure t
  "If this variable contains a value other than nil and all nested
parentheses in a Lisp listener buffer become closed, Emacs evaluates the
s-expression in that buffer.")

(defvar apollo:check-parenthesis-before-saving-p t
  "When this Boolean's value is t and unbalanced parentheses exist,
Emacs queries before saving files.")

(defvar apollo:look-for-in-package-p t
  "If true (that is, the variable has a non-nil value), Emacs searches
backwards in the common lisp buffer for a line starting with the in-package
function and uses the specified package when evaluating Common Lisp forms.

        If false (that is, the variable has a nil value) or if Emacs cannot
find the in-package function, then Emacs uses the apollo:
look-for-attribute-line-package-p configuration variable.  If this second
variable has a value other than nil, Emacs checks the buffer's attribute line;
otherwise, Emacs checks the value of the apollo: package variable for an
uppercase string representing the package.  If this third variable has the value
nil, Emacs uses the value of the *package* variable in the current inferior lisp
process.")

(defvar apollo:look-for-attribute-line-package-p t
  "If true (that is, the variable has a non-nil value), Emacs examines
the attribute lists of lisp buffers while evaluating common lisp expressions
within those buffers.  Consult the description of the apollo:
look-for-in-package-p configuration variable for more information.")

(defvar apollo:page-size 62
  "The value of this variable determines the largest allowable page
size for the pagify-lisp-buffer command (also described in this section).")

(defvar apollo:tmp-dir "/tmp"
  "The value of this variable is the pathname of the temporary
directory.")


(defvar apollo:default-printer ""
  "The value of this variable is the entry directory name (for example,
\"//flasher\") of the print queue used by the print-buffer-apollo and
print-region-apollo commands.

        If the value of the variable is nil or a pair of empty quotation marks,
Emacs prompts for the entry directory name.")

(defvar apollo:gnuemacs-client-p nil 
"Use client/server protocol to talk to inferior lisp.  Normally,
communication between lisp and emacs is handled by interrupting the
lisp process, eating break messages, sending input to lisp &
continuing from interrupt.  There is also a mechanism for use with
Domain/CommonLISP 3.0 which uses named pipes and a lisp task to do
this communication.  The latter scheme has not been fully tested and
is therefore turned off by default.  The latter scheme though is
potentially faster and more robust.  Another difference with the
latter scheme is that lisp output from all commands not originating
in the lisp listener buffer will go into special history buffers.")

(defvar apollo:printer-args ""
  "This string variable contains the /com/prf command arguments (for
example, \"-pr cx -s //flasher\") used by the print-buffer-apollo and the
print-region-apollo commands.")

(setq ; for evaluating lisp forms on closure in inferior lisp
  blink-paren-hook 'apollo:blink-matching-open)

(defvar apollo:inferior-lisp-switches ""
  "List of strings representing starup switches to lisp program.")

(setq ;; prompt for inferior-lisp process
  inferior-lisp-prompt "^\\(\\(\\(->\\)+\\)\\|>\\) ")

(defvar apollo:update-attribute-list-when-creating-file-p t
  "If true (that is, the variable has a non-nil value), Emacs updates a
file's attribute list from local variables when creating the file's buffer.")

(defvar apollo:modes-not-to-update-attribute-list-when-creating-file nil
  "List of strings representing mode names for modes that don't update
attribute lists when creating file buffers of that mode.  These mode
names are the values returned by executing (apollo:mode) from within
the new buffer.")

(defvar apollo:*pagify-lisp-buffer-p* 't
  "When printing a Lisp buffer, pagify it first.")


;;; End Configuration variables


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Generate local variables
(make-variable-buffer-local 'apollo:buffer-process)

(make-variable-buffer-local 'apollo:original)

(make-variable-buffer-local 'apollo:symbols)

(set-default 'apollo:symbols '(apollo:mode)) ; All buffers have a mode

(make-variable-buffer-local 'apollo:buffer-sections)

(make-variable-buffer-local 'apollo:dynamic-mouse-binding)

(make-variable-buffer-local 'apollo:base)

(make-variable-buffer-local 'apollo:superpackage)

(make-variable-buffer-local 'apollo:package)

(make-variable-buffer-local 'apollo:used-packages)

(make-variable-buffer-local 'apollo:ibase)

(make-variable-buffer-local 'apollo:mode)

(make-variable-buffer-local 'apollo:local-info)

(make-variable-buffer-local 'apollo:readtable)

(make-variable-buffer-local 'apollo:command-kill-ring)

(make-variable-buffer-local 'apollo:command-kill-ring-yank-pointer)

(make-variable-buffer-local 'apollo:kill-ring)

(make-variable-buffer-local 'apollo:kill-ring-yank-pointer)

(make-variable-buffer-local 'apollo:yank-type)

(set-default 'apollo:yank-type   'apollo:yank-command)

(set-default 'apollo:dynamic-mouse-binding 'apollo:mouse-find-file)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; All the global variables...

(defvar apollo:terminals nil)
(defvar apollo:key-bindings-forms nil)

(defvar apollo:locate-symbol nil)

(defvar apollo:last-ask-common-lisp nil)

(defvar apollo:last-canonical-file-name nil)

(defvar apollo:lcd-last-result nil)

(defvar apollo:lcd-last-arg nil)

(defvar apollo:ask-common-lisp-queue nil)

(defvar apollo:ask-common-lisp-display-p nil)

(defvar apollo:ask-common-lisp-result nil)

(defvar apollo:ask-common-lisp-temp    "")

(defvar apollo:ask-common-lisp-last-output-marker nil)

(defvar apollo:string nil)

(defvar apollo:arg-file (make-temp-name (format "%s/emacs" apollo:tmp-dir)))

(defvar apollo:buffer-definitions nil)

(defvar apollo:buffer-definitions-type nil)

(defvar apollo:callers nil)

(defvar apollo:clisp-bin-pattern  "\\.lbin$")

(defvar apollo:clisp-file-pattern "\\.lisp$")

(defvar apollo:compile nil)

(defvar apollo:current-definition nil)

(defvar apollo:elisp-bin-pattern  "\\.elc$")

(defvar apollo:elisp-file-pattern "\\.\\(\\(el\\)\\|\\(emacs\\)\\)$")

(defvar apollo:alpha-numeric "[a-zA-z01-9]")

(defvar apollo:extended-mouse-insert-alpha-numeric "[a-zA-z01-9*]")

(defvar apollo:file-chars "[\~$\.a-zA-Z/_01-9-]")

(defvar apollo:find-source-code-info nil)

(defvar apollo:force-common-lisp-p nil)

(defvar apollo:grabbing-dispatched-names-p nil)

(defvar apollo:in-dynamic-mouse-call-p nil)

(defvar apollo:initial-lisp-process-filter nil)


(defvar apollo:initial-lisp-packages
  '(USER LISP TRANSLATE FLAVORS KEYWORD LUCID SYSTEM FLAVOR-INTERNALS WINDOWS EDITOR))

(defvar apollo:command-kill-ring nil)

(defvar apollo:command-kill-ring-yank-pointer nil)

(defvar apollo:last-arg-file nil)

(defvar apollo:last-notification nil)

(defvar apollo:lisp-name nil)

(defvar apollo:lisp-processes nil)

(defvar apollo:lisp-prompt-pointer nil)

(defvar apollo:locals-to-preserve
  '(apollo:original apollo:readtable apollo:buffer-sections apollo:base apollo:ibase
    apollo:kill-ring apollo:yank-type apollo:command-kill-ring
    apollo:package apollo:superpackage apollo:symbols apollo:dynamic-mouse-binding apollo:syntax))

(defvar apollo:loop-count nil)

(defvar apollo:mark nil)

(defvar apollo:marker (make-marker))

(defvar apollo:max-loop-count 10)

(defvar apollo:modifications-original nil)

(defvar apollo:find-source-code-symbol nil)

(defvar apollo:temp-package nil)

(defvar apollo:temp-base nil)

(defvar apollo:temp-ibase nil)

(defvar apollo:pattern nil)

(defvar apollo:point nil)

(defvar apollo:prev-command-ptr nil)

(defvar apollo:prompt-flag 'APOLLO:PROMPT-FLAG)

(defvar apollo:prompt-for-cl-use-default-p nil)

(defvar apollo:evaluate-common-lisp-into-minibuffer  1)

(defvar apollo:evaluate-common-lisp-into-temp-buffer 2)

(defvar apollo:evaluate-common-lisp-as-typin    3)

(defvar apollo:evaluate-common-lisp-insert           4)

(defvar apollo:evaluate-common-lisp-delete-and-insert 5)


(defvar apollo:sectionize-buffer t
  "Non-nil to sectionize buffer at file read time.")

(defvar apollo:since-read            1)

(defvar apollo:since-read-or-save    2)

(defvar apollo:since-eval-or-compile 3)

(defvar apollo:symbol nil)

(defvar apollo:symbol-chars "[~$.:%a-z<>^A-Z/_*|01-9-]")

(defvar apollo:symbol-chars-no-directory "[~$.:%a-z^A-Z_*|01-9-]")

(defvar apollo:symbol-chars-no-package "[~$.%a-z<>^A-Z/_*|0-9-]")

(defvar apollo:whitespace-chars  '(?\t ?\f ?\r ?\n ?\ ))

(defvar apollo:non-whitespace "[^ \t\f\r\n]")

(defvar apollo:whitespace     "[ \t\f\r\n]")

(defvar apollo:chars-in-sharp<> "[() \t\f\r\n]")

(defvar apollo:verbose t)

(defvar apollo:non-whitespace-or-lpar "[^( \t\f\r\n]")

(defvar apollo:all-whitespace (concat "^" apollo:whitespace "*$"))

(defvar apollo:common-lisp-minibuffer-local-must-match-map
  '(keymap
    (7 . abort-recursive-edit)
    (9 . apollo:lisp-complete-symbol)
    (10 . exit-minibuffer)
    (13 . exit-minibuffer)))

(defconst apollo:any-def-package
    (format "%s*:?:?" apollo:symbol-chars-no-package))

(defconst apollo:any-def-def
    (format "\\(%s*def%s*\\)" apollo:symbol-chars-no-package apollo:non-whitespace))

(defconst apollo:any-def-object
    (format "\\(%s%s*\\)" apollo:non-whitespace-or-lpar apollo:non-whitespace))

(defconst apollo:any-def-method
    (format "\\((%s+%s+%s+\\(%s+%s+\\)?)\\)"
            apollo:non-whitespace apollo:whitespace apollo:non-whitespace
            apollo:whitespace apollo:non-whitespace-or-lpar))

(defconst apollo:flavor-method-info ;; like above but different match info
    (format "(\\(%s+\\)%s+\\(%s+\\)%s*\\(%s%s+\\)?)"
            ;; (<flavor-class> <method-name>)
            apollo:non-whitespace apollo:whitespace apollo:non-whitespace
            apollo:whitespace
            apollo:whitespace apollo:non-whitespace-or-lpar))


(defconst apollo:any-def ;; Handles flavor methods ... Sort of
    (format
     "^(%s%s%s*\\(%s\\|%s\\)%s"
     ;; <package>::<def-prefix>def<def-suffix>
     "" ;apollo:any-def-package
     apollo:any-def-def
     apollo:whitespace
     apollo:any-def-object
     apollo:any-def-method
     apollo:whitespace))

(defmacro apollo:match (n &optional where)
  (if where
      (list 'substring where (list 'match-beginning n) (list 'match-end n))
      (list 'buffer-substring (list 'match-beginning n) (list 'match-end n))))

(defmacro apollo:match-def-type ()
  (list 'downcase (macroexpand '(apollo:match 1))))

(defun apollo:definition-name (string)
  (setq string (apollo:strip-whitespace string))
  (if (> (length string) 0)
      (if (eq ?\( (aref string 0))
          (save-excursion
            (if (string-match (concat "(:METHOD" apollo:whitespace "+") string)
                (setq string (concat "(" (substring string (match-end 0)))))
            (string-match apollo:flavor-method-info string)
            (let ((md (match-data)))
              (let ((flavor (upcase (substring string (match-beginning 1) (match-end 1))))
                    (daemon (upcase (substring string (match-beginning 2) (match-end 2))))
                    (method (if (<= 8 (length md))
                                (upcase (substring string (1+ (match-beginning 3)) (match-end 3))))))
                (if (not method) (setq method daemon daemon ":PRIMARY"))
                (if (string-match (concat ":METHOD" apollo:whitespace "+") flavor)
                    (setq flavor (substring flavor (match-end 0))))
                (list flavor daemon method))))
          string)))

(defmacro apollo:match-def-name (&optional type)
  '(if (eq ?\( (char-after (match-beginning 2)))
    (save-excursion
      (goto-char (match-beginning 2))
      (looking-at apollo:flavor-method-info)
      (let ((md (match-data)))
        (let ((flavor (upcase (buffer-substring (match-beginning 1) (match-end 1))))
              (daemon (upcase (buffer-substring (match-beginning 2) (match-end 2))))
              (method (if (<= 8 (length md))
                          (upcase (buffer-substring (1+ (match-beginning 3)) (match-end 3))))))
          (if (not method) 
              (setq method daemon 
                    type (upcase type)
                    daemon (cond ((equal type "DEFWHOPPER") ":WHOPPER")
                                 ((equal type "DEFWRAPPER") ":WRAPPER")
                                 (t ":PRIMARY"))))
          (list flavor daemon method))))
    (apollo:match 2)))

(defmacro apollo:match-def-start () '(match-beginning 0))


(defconst apollo:attribute-line-prefix
    (concat apollo:whitespace "*%s*" apollo:whitespace "*-\\*-" apollo:whitespace "*"))

(defconst apollo:attribute-line-end
    (concat apollo:whitespace "*;?" apollo:whitespace "*-\\*-" apollo:whitespace "*"))

(defconst apollo:attribute-end
    (concat apollo:whitespace "*\\("
            "\\(;" "\\(" apollo:attribute-line-end "\\)?" apollo:whitespace "*\\)\\|"
            "\\(" apollo:attribute-line-end "\\)"
            "\\)"))

(defconst apollo:attribute-value-separator
    (concat apollo:whitespace "*\\(:\\|\\(" apollo:attribute-line-end "\\)\\|" apollo:whitespace "\\)"
            apollo:whitespace "*"))

(defconst apollo:unexported-package
    (concat "\\(" apollo:symbol-chars-no-package "\\)\\(:\\)\\("
            apollo:symbol-chars-no-package "\\)"))

(defconst apollo:package&name
    (concat "\\(" apollo:symbol-chars-no-package "*\\)\\(::?\\)"
            apollo:symbol-chars-no-package "*"))

(defconst apollo:find-files-shell-command
    (concat apollo:aegis-directory "/ld -C -NWARN")
  "Used for shell completions.  Finds a list of commands")

(defconst apollo:find-files-shell-separator ": "
  "String of separators for shell command")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Buffer names

(defconst apollo:changed-lines-buffer              "*CHANGED-DEFINITIONS_APOLLO*")

(defconst apollo:command-history-buffer            "*COMMAND HISTORY_APOLLO*")

(defconst apollo:completions-buffer                "*COMPLETIONS_APOLLO*")

(defconst apollo:disassemble-buffer                "*DISASSEMBLE_APOLLO*")

(defconst apollo:emacs-clisp-buffer                " *CLISP_APOLLO*")

(defconst apollo:emacs-elisp-buffer                " *ELISP_APOLLO*")

(defconst apollo:grep-buffer                       "*GREP_APOLLO*")

(defconst apollo:sections-buffer                   "*BUFFER-SECTIONS_APOLLO*")

(defconst apollo:kill-history-buffer               "*KILL-HISTORY_APOLLO*")

(defconst apollo:lisp-output-buffer                "*LISP-OUTPUT_APOLLO*")

(defconst apollo:macroexpand-buffer                "*MACRO-EXPAND_APOLLO*")

(defconst apollo:modifications-buffer              "*MODIFICATIONS_APOLLO*")

(defconst apollo:notifications-buffer              "*NOTIFICATIONS_APOLLO*")

(defconst apollo:emacs-eval-buffer                 "*EVALUATED_APOLLO*")

(defconst apollo:callers-buffer                    "*CALLERS_APOLLO*")

(defconst apollo:hack-buffer                       " *SOME-RANDOM-HACK-BUFFER_APOLLO*")

(defconst apollo:emacs-original-buffer             " *EMACS-EVALUATED_APOLLO*")

(defconst apollo:print-buffer                      " *PRINT-BUFFER_APOLLO*")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Kludge to allow this file to byte compile, expanding macros appropriately

(require ;; This stuff is evaluated at load and comile time!!
  (progn (provide 'apollo:byte-compile-macro-expand-hack)
         (defvar apollo:gensym-name "CLISP-GENSYM")
         (defvar apollo:gensym-number 0)
         (defvar apollo:old-max-lisp-eval-depth max-lisp-eval-depth)
         (setq max-lisp-eval-depth ;; make something reasonable
               (* 2 apollo:old-max-lisp-eval-depth))
         (defun apollo:gensym () ; compiler wants this to be a defun
           "Generate a new unused symbol.  This could conflict with
                other gensyms.  Care should be taken to insure
                that apollo:gensym-name is unique to a given file."
           (let (s)
             (while (or (boundp
                         (setq s
                               (intern
                                (format "apollo:%s-%s"
                                        apollo:gensym-name
                                        (setq apollo:gensym-number
                                              (1+ apollo:gensym-number))))))
                        (fboundp s)))
             s))
         (defconst apollo:read-from-list nil)
         (defvar apollo:old-eobp (symbol-function 'eobp))
         (defvar apollo:old-read (symbol-function 'read))
         (defun eobp () (or apollo:read-from-list (funcall apollo:old-eobp)))
         ;; Redefine read!!
         ;; See definition byte-compile-file in "bytecomp.el" for ramifications
         ;; of this stuff.
         ;; Macroexpands read forms and treats toplevel progn forms as if they
         ;; were read forms.
         ;; When called with an argument other then the compiler input buffer,
         ;; this will revert to the old definition of read!!
         ;; Kludge... Kludge... Kludge...
         (defun read (&optional stream)
           (if (and stream (eq stream (get-buffer " *Compiler Input*")))
               (let ((apollo:gensym-name "COMPILER-GENSYM"))
                 (if apollo:read-from-list ; Ignore stream if read list
                     (let ((r (car apollo:read-from-list)))
                       (setq apollo:read-from-list (cdr apollo:read-from-list))
                       (macroexpand r byte-compile-macro-environment))
                     (let ((r (macroexpand (funcall apollo:old-read stream)
                                           byte-compile-macro-environment)))
                       (setq r 
                             (if (and (consp r) (eq (car r) 'progn))
                                 (progn
                                   (setq apollo:read-from-list (cdr (cdr r)))
                                   (car (cdr r)))
                                 r))
                       (if (and (consp r) (eq (car r) 'defmacro)) (eval r))
                       r
                       )))
               (progn
                 (setq apollo:read-from-list nil)
                 (fset 'read apollo:old-read)
                 (fset 'eobp apollo:old-eobp)
                 (funcall apollo:old-read stream))))
         'apollo:byte-compile-macro-expand-hack)) ; don't load any files

(fset 'read apollo:old-read) ;; don't use read hack when loading


(fset 'eobp apollo:old-eobp)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Some useful macros
(defvar apollo:match-string " ")
(defmacro apollo:char-in-regexp (char regexp)
  (list 'progn
        (list 'aset 'apollo:match-string 0 char)
        (list 'string-match regexp 'apollo:match-string)))
  
(defmacro apollo:extended-mouse-insert-alphanumericp (char)
  (list 'apollo:char-in-regexp char 'apollo:extended-mouse-insert-alpha-numeric))
                            
(defmacro apollo:gensym ()
  "Generate a new unused symbol"
  (let (s)
    (while (or (boundp (setq s (intern (format "apollo:%s-%s"
                                               apollo:gensym-name
                                               (setq apollo:gensym-number
                                                     (1+ apollo:gensym-number))))))
               (fboundp s)))
    (list 'quote s)))

(defmacro apollo:arg-val (arg &optional default)
  "if cons return car else value"
  (list 'or (list 'if (list 'consp arg) (list 'car arg) arg) default))

(defmacro abs (a)        (list 'if (list '> a 0) a (list '- a)))

(defmacro eol  ()        '(save-excursion (end-of-line) (point)))

(defmacro cadr (a)       (list 'car (list 'cdr a)))

(defmacro caar (a)       (list 'car (list 'car a)))

(defmacro cdar (a)       (list 'cdr (list 'car a)))

(defmacro cddr (a)       (list 'cdr (list 'cdr a)))

(defmacro caddr (a)      (list 'car (list 'cdr (list 'cdr a))))

(defmacro push (b a)     (list 'setq a (list 'cons b a)))

(defmacro push-new (b a) (list 'if (list 'memq b a)
                               a
                               (list 'setq a (list 'cons b a))))

(defmacro pop  (a)       (let ((v (apollo:gensym)))
                           (list 'let (list (list v (list 'car a)))
                                 (list 'setq a (list 'cdr a)) v)))

(defmacro apollo:incf (a &optional inc) (list 'setq a (list '+ a (or inc 1))))

(defmacro apollo:decf (a &optional dec) (list 'setq a (list '- a (or dec 1))))

(defmacro with-buffer (b &rest r)
  "Perform some action in another buffer.
          Uses save-excursion."
  (append (list 'save-excursion (list 'set-buffer b)) r))


(defmacro with-buffer-set (b &rest r)
  "Perform some action in another buffer.
          No save-excursion."
  (let ((c (apollo:gensym)))
    (list 'let
          (list (list c '(current-buffer)))
          (list 'unwind-protect
                (append (list 'progn (list 'set-buffer b)) r)
                (list 'set-buffer c)))))

(defmacro with-buffer-name-preserved (b &rest r)
  "Perform some action keeping name of buffer."
  (let ((name (apollo:gensym))
        (buffer (apollo:gensym)))
    (list 'let (list (list buffer b)
                     (list name (list 'buffer-name b)))
          (list 'unwind-protect 
                (cons 'progn
                      r)
                (list 'if (list 'not (list 'equal name (list 'buffer-name buffer)))
                      (list 'with-buffer-set buffer
                            (list 'rename-buffer name)))))))
  
(defmacro with-buffer-bury (buffer &rest body)
  "Perform some action in another buffer.
          Uses save-excursion.
          Erases buffer prior to use.
          Burys buffer when done"
  (let ((b (apollo:gensym)))
    (list 'let (list (list b buffer))
          (list 'unwind-protect
                (append (list 'with-buffer b '(erase-buffer)) body)
                (list 'if (list 'get-buffer b) (list 'bury-buffer b))))))

(defmacro with-process-filter (process&filter &rest body)
  (let ((old-filter (apollo:gensym))
        (process (car  process&filter))
        (filter  (car (cdr process&filter))))
    (list 'let
          (list (list old-filter (list 'process-filter process)))
          (list 'unwind-protect
                (cons  'progn
                       (cons (list 'set-process-filter process filter)
                             body))
                (list 'set-process-filter process old-filter)))))

(defmacro apollo:return (&optional value)
  "Used to return a value from a apollo:dolist or apollo:dotimes"
  (list 'throw (list 'quote 'apollo:return) value))


(defmacro apollo:dolist (a &rest b)
  (let ((l (apollo:gensym))
        (v (car a))
        (d (car (cdr (cdr a)))))
    (list 'let
          (list (list l (list 'eval (list 'nth 1 (list 'quote a)))))
          (list 'catch (list 'quote 'apollo:return)
                (list 'while l
                      (list 'let
                            (list (list v (list 'car l)))
                            (list 'setq l (list 'cdr l))
                            (cons 'progn b)))
                d))))

(defmacro apollo:dotimes (a &rest b)
  (let ((l (apollo:gensym))
        (v (car a))
        (d (car (cdr (cdr a)))))
    (list 'let
          (list (list l (list 'eval (list 'nth 1 (list 'quote a))))
                (list v 0))
          (list 'catch (list 'quote 'apollo:return)
                (list 'while (list '< v l)
                      (cons 'progn b)
                      (list 'setq v (list '+ v 1)))
                d))))

(defmacro apollo:apollo-p () 
  '(or 
    (fboundp 'select-APOLLO-meta-key)
    (fboundp 'select-apollo-meta-key)
    (fboundp 'apollo-keyboard-type)))

(defun apollo:mouse-string (c &rest meta-p)
  (cond ( ;; Emacs pre-18.51
         (fboundp 'select-APOLLO-meta-key) 
         (concat  "\C-c" (char-to-string c)))
        ;; Emacs 18.51
        ((fboundp 'select-apollo-meta-key) 
         (concat  (if meta-p "\M-+" "\M-*") (char-to-string c)))
        ;; Emacs 18.52
        ((fboundp 'apollo-keyboard-type)
         (concat  (if meta-p "\M-+" "\M-*") (char-to-string c)))
        (error "Invalid apollo emacs version")))
  

(defmacro defun-APOLLO (&rest body)
  (if (apollo:apollo-p)
      (cons 'defun body)))


(defmacro apollo:redefun (&rest function)
  "This macro is called just like defun.  A defvar is created
          called apollo:<function-name> bound to the old symbol-function
          of function-name.  A defvar is also created called
          apollo:<function-name>-documentation.  This is typically
          used to redefine a function allowing the old
          function to be called via (apply <apollo:function-name> ...)"
  (let* ((apollo-name   (concat "apollo:" (symbol-name (car function))))
         (apollo-doc    (concat apollo-name "-documentation")))
    (if (fboundp (car function))
        (list 'progn
              (list 'defvar
                    (intern (concat "apollo:" (symbol-name (car function))))
                    (list 'symbol-function (list 'quote (car function))))
              (list 'defvar (intern apollo-doc)
                    (list 'documentation (list 'quote (car function))))
              ;; (list 'byte-compile (cons 'defun function))
              (cons 'defun function))
        (cons 'defun function))))


(defmacro apollo:add-hooks (function)
  "This macro is called with a symbol representing an emacs lisp
          FUNCTION.  This FUNCTION is redefined to funcall a list
          of hooks before and after function execution.
          This will create local-variables as follows:
          apollo:<function-name> bound to the symbol function of FUNCTION
          apollo:<function-name>-documentation which will be the doc
                string for the FUNCTION.
          apollo:<function-name>-before-hooks, a list of functions to funcall
                before execution. These functions will be called with the
                same arguments as the original FUNCTION.
          apollo:<function-name>-unwind-hooks, a list of functions to funcall
                after  execution. These functions will be called with the
                same arguments as the original FUNCTION.  These functions 
                are unwind protected so that they will run on throws, errors etc.
          apollo:<function-name>-after-hooks, a list of functions to funcall
                after execution.  These functions will be called with the
                result of executing the original FUNCTION followed by the
                original arguments.
          This will take same arguments and return the same value as 
                the original FUNCTION.
          "
  (let* ((apollo-name   (concat "apollo:" (symbol-name function)))
         (apollo-before (concat apollo-name "-before-hooks"))
         (apollo-unwind (concat apollo-name "-unwind-hooks"))
         (apollo-after  (concat apollo-name "-after-hooks"))
         (apollo-doc    (concat apollo-name "-documentation"))
         (apollo-catch  (concat apollo-name "-catch"))
         (args          (concat apollo-name "-args"))
         (hook          (concat apollo-name "-hook"))
         (result        (concat apollo-name "-result")))
    (if (fboundp function)
        (list 'progn
              (list 'defvar (intern apollo-name)
                    (list 'symbol-function (list 'quote function)))
              (list 'defvar (intern apollo-doc)
                    (list 'documentation   (list 'quote function)))
              (list 'defvar (intern apollo-before) nil)
              (list 'defvar (intern apollo-unwind) nil)
              (list 'defvar (intern apollo-after)  nil)
              (list 'defun function (list '&rest (intern args))
                    (concat
                      (if (boundp (intern apollo-doc))
                          (eval (intern apollo-doc))
                          (documentation function))
                      "\n\nAdded hooks: " apollo-before ",\n"
                      "             " apollo-unwind " &\n"
                      "             " apollo-after ".")
                    '(interactive)
                    (list 'catch (list 'quote (intern apollo-catch))
                          (list 'mapcar
                                (list 'function
                                      (list 'lambda (list (intern hook))
                                            (list 'apply (intern hook) (intern args))))
                                (intern apollo-before))
                          (list 'let (list (list (intern result) nil))
                                (list 'unwind-protect
                                      (list 'setq (intern result)
                                            (list 'apply (intern apollo-name) (intern args)))
                                      (list 'mapcar
                                            (list 'function 
                                                  (list 'lambda (list (intern hook))

                                                        (list 'apply (intern hook) (intern args))))
                                            (intern apollo-unwind)))
                                (list 'mapcar
                                      (list 'function 
                                            (list 'lambda (list (intern hook))
                                                  (list 'apply (intern hook) (intern result) (intern args))))
                                      (intern apollo-after))
                                (intern result))))))))

(defmacro with-read-only (read-only &rest body)
  (let ((old-read-only (apollo:gensym)))
    (list 'let (list (list old-read-only 'buffer-read-only))
          (list 'unwind-protect
                (append 
                  (list 'progn
                        (list 'if (list 'not (list 'eq 'buffer-read-only read-only)) '(toggle-read-only)))
                  body))
          (list 'if (list 'not (list 'eq 'buffer-read-only old-read-only)) '(toggle-read-only)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Initialize hooks and GNUEMACS variables

(push-new 'buffer-enable-undo   find-file-hooks)

(push-new 'sectionize-buffer    find-file-hooks)

(push-new 'apollo:set-original     find-file-hooks)

(push-new 'parse-attribute-list find-file-hooks)

(push-new 'apollo:unbal-check      write-file-hooks)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Fix broken system functions

(apollo:redefun eval-expression (expression)
  "Evaluate EXPRESSION and print value in minibuffer.
       Value is also consed on to front of variable values 's value.
       This version pushes the right thing onto command-histroy and lets
       dynamic-mouse grab symbols from a buffer."
  (interactive "P")
  (let ((apollo:dynamic-mouse-binding 'apollo:grab-thing-dynamic-mouse)
        (apollo:yank-type             'apollo:yank-command))
    (push (or expression 
              (car (read-from-string (read-string "Eval: " ""))))
          command-history)
    (setq apollo:command-kill-ring-yank-pointer command-history)
    (push (eval (car command-history)) values)
    (princ (car values))))

(apollo:redefun expand-file-name (filename &optional default)
  "Convert FILENAME to absolute, and canonicalize it.
       Second arg DEFAULT is directory to start with if FILENAME is relative
       (does not start with slash); if DEFAULT is nil or missing,
       the current buffer's value of default-directory is used.
       Filenames containing . or .. as components are simplified;
       initial ~ is expanded.  See also the function  substitute-in-file-name.
       Actually, this is a version with a apollo:canonical-file-name wrapper
       around it."
  (apollo:canonical-file-name
    (apply apollo:expand-file-name (list filename default))))

(defvar apollo:enable-APOLLO-mouse-buttons
  (if (fboundp 'enable-APOLLO-mouse-buttons)
      (symbol-function 'enable-APOLLO-mouse-buttons)
      (if (fboundp 'enable-apollo-mouse-button)
          '(lambda (buttons)
            (let ((n (length buttons)))
              (while (> n 0)
                (enable-apollo-mouse-button (aref buttons (1- n)))
                (setq n (1- n))))))))

(defvar apollo:enable-apollo-mouse-button
  (if (fboundp 'enable-apollo-mouse-button)
        (symbol-function 'enable-apollo-mouse-button)))

(defun  apollo:enable-APOLLO-mouse-buttons (s)
  "In /gnuemacs/lisp/term/apollo.el, the buttons are
        clobbered depending on the function binding of
        enable-APOLLO-mouse-buttons.  This moves the
        function to apollo:enable-APOLLO-mouse-buttons
        so that the buttons initialized here wont be
        clobbered."
  (apply apollo:enable-APOLLO-mouse-buttons (list s)))

(if (apollo:apollo-p) (apollo:enable-APOLLO-mouse-buttons "abcA"))

(if (fboundp 'enable-APOLLO-mouse-buttons)
        (fmakunbound 'enable-APOLLO-mouse-buttons))


(apollo:redefun process-send-string (process string)
  "Send PROCESS the contents of STRING as input.
       PROCESS may be a process name.
       This is actually a special version that calls the system
       version with 100 or fewer characters at a time.  This is
       necessary due to GNUEMACS bug which caused failure if the
       string sent was too long.  There is also a bug which smashes
       this if the input line has > 250 characters in a row without
       a newline!  Oy!!"
  (let ((p 0) (l (length string)) (s string))
    (while (< 0 (length (setq s (substring s p))))
      (setq p (if (string-match "\n" s) (match-end 0) (length s)))
      (if (> p 250)
          (error "Input String Too long, insert returns!")
          (progn
            (apply apollo:process-send-string
                   (list process (substring s 0 (min 100 p))))
            (if (> p 100)
                (progn
                  (apply apollo:process-send-string
                         (list process (substring s 100 (min 200 p))))
                  (if (> p 200)
                      (apply apollo:process-send-string
                             (list process (substring s 200 p)))))))))))

(apollo:redefun process-send-region (process start end)
   "Send current contents of region as input to PROCESS.
       PROCESS may be a process name.  This also adds the command to the 
       buffers apollo:kill-ring list if the string is not all whitespace,
       the string does not begin with a ':' and the string is not the
       same as the first on apollo:kill-ring.
       Called from program, takes three arguments, PROCESS, START and END.
       This is actually a special version that calls the system
       version with 100 or fewer characters at a time.  This is
       necessary due to GNUEMACS bug. which caused failure if the
       region sent was too large"
   (let ((s (buffer-substring start end))
         (m (mark)))
     (set-mark start);; make it easy to wipe if error
     (process-send-string process s)
     (set-mark m)
     (setq s (apollo:strip-whitespace s))
     (if (not (or (equal "" s) (eq (aref s 0) ?:) (equal s (car apollo:kill-ring))))
         (push s apollo:kill-ring))))

(apollo:redefun kill-all-local-variables ()
  "Eliminate all the buffer-local variable values of the current buffer.
       This buffer will then see the default values of all variables.
       This is a rework of the system function that will not kill local
       variables in the list apollo:locals-to-preserve"
  (let ((l nil))
    (apollo:dolist (v apollo:locals-to-preserve)
      (if (boundp v) (push (cons v (eval v)) l)))
    (apollo:dolist (v apollo:symbols)
      (if (and (not (memq v apollo:locals-to-preserve)) (boundp v))
          (push (cons v (eval v)) l)))
    (apply apollo:kill-all-local-variables nil)
    (apollo:dolist (v l)
      (set (make-variable-buffer-local (car v)) (cdr v)))))


(apollo:redefun kill-buffer (buffer)
  "One arg, a string or a buffer.  Get rid of the specified buffer.
        This version also kills associated apollo:original buffers
        if necessary.  If buffer being deleted is the current
        common lisp buffer designated for common lisp commands,
        this will attempt to pop back to previously designated
        inferior common lisp buffer."
  (interactive "P")
  (let ((buffer (get-buffer 
                  (or buffer
                      (read-buffer "Kill Buffer: " (current-buffer)))))
        (new-buffer nil)
        (old-sections nil)
        (old-original nil)
        (old-original-sections nil)
        (inferior-lisp-buffer-p nil))
    (with-buffer buffer 
      (setq old-sections apollo:buffer-sections 
            apollo:buffer-sections nil
            old-original apollo:original
            apollo:original nil))
    (if old-original 
        (with-buffer old-original
          (setq old-original-sections apollo:buffer-sections 
                apollo:buffer-sections nil
                apollo:original nil)))
    (let ((p apollo:lisp-processes))
      (while (and p (not inferior-lisp-buffer-p))
        (let ((apollo:lisp-name (car (car p))))
          (if (eq (apollo:lisp-buffer 'NO-ERROR) buffer)
              (progn
                (reset-lisp-filter t)
                (setq apollo:lisp-processes (delq (car p) apollo:lisp-processes)
                      inferior-lisp-buffer-p t))
              (setq p (cdr p))))))
    (if inferior-lisp-buffer-p 
        (setq apollo:lisp-name (car (car apollo:lisp-processes))
              new-buffer (if (eq buffer (current-buffer)) (apollo:lisp-buffer 'NO_CREATE))))
    ;; (message "Killing Buffer: %s" buffer)
    (funcall apollo:kill-buffer buffer)
    (if (and t (not (get-buffer buffer)))
        (progn
          (if old-original 
              (progn 
                ;; (message "Killing Buffer: %s" old-original)
                (funcall apollo:kill-buffer old-original)))
          (setq apollo:section-reuse-list
                (nconc apollo:section-reuse-list old-original-sections old-sections))
          (if new-buffer 
              (progn
                ;; (message "Selecting Buffer: %s" (buffer-name new-buffer))
                (select-window (display-buffer new-buffer))))))))
  
(defvar apollo:call-interactively-recursive nil)

(apollo:redefun call-interactively (function &optional record-flag)
  "Call FUNCTION, reading args according to its interactive calling specs.
     The function contains a specification of how to do the argument reading.
     In the case of user-defined functions, this is specified by placing a call
     to the function `interactive' at the top level of the function body.
     See `interactive'.

     Optional second arg RECORD-FLAG non-nil
     means unconditionally put this command in the command-history.
     Otherwise, this is done only if an arg is read using the minibuffer.

    This version will automatically place M-x commands on command history."
  (let* ((recurse apollo:call-interactively-recursive)
         (apollo:call-interactively-recursive t))
    (funcall apollo:call-interactively function (or record-flag (not recurse)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; hooks for system functions

(defun apollo:setup-dynamic-mouse (&rest ignore)
  "This function takes a GNUEMACS created completions buffer
       and makes it mousable via click right.  The buffer will
       also display a message stating what clicking right will
       do."
  (let ((b (cond ((eq standard-output t) (get-buffer " *Completions*"))
                 (t standard-output))))
    ;; This is hacked for unix filenames, may work funny for some 
    ;; symbols with backslash in name.
    (if (eq (char-after (1- (point))) ?/)
        (setq apollo:point (point) apollo:mark apollo:point)
        (apollo:current-thing apollo:symbol-chars-no-directory nil))
    (if b
        (with-buffer b
          (goto-char (point-min))
          (end-of-line)
          (set (make-variable-buffer-local 'apollo:dynamic-mouse-binding)
               'apollo:grab-thing-dynamic-mouse)
          (let* ((d (documentation apollo:dynamic-mouse-binding))
                (title (concat "\n[Mouse Right]: "
                               (substring d 0 (string-match "\n\\|$" d)) "\n")))
            (if (not (looking-at (regexp-quote title))) 
                (progn
                  (insert title)
                  (if (and (boundp temp-buffer-show-hook) temp-buffer-show-hook)
                      (funcall temp-buffer-show-hook (current-buffer))))))))))

(apollo:add-hooks find-file-noselect)

(defun update-attribute-list-when-creating-file (file-buffer filename &optional nowarn)
  ;; (declare (Ignore result nowarn))
  (if (and apollo:update-attribute-list-when-creating-file-p
           (not (file-exists-p filename)))
      (with-buffer-set file-buffer
        (let ((m  (apollo:mode))
              (ml apollo:modes-not-to-update-attribute-list-when-creating-file))
          (catch 'done (while ml (if (equal m (car ml)) (throw 'done ml) (pop ml))))
          (if (not ml) (update-attribute-list))))))

(push-new 'update-attribute-list-when-creating-file apollo:find-file-noselect-after-hooks)

(apollo:add-hooks display-completion-list)

(push-new 'apollo:setup-dynamic-mouse apollo:display-completion-list-after-hooks)

(apollo:add-hooks lisp-complete-symbol)

(push-new 'apollo:call-current-thing apollo:lisp-complete-symbol-after-hooks)

(apollo:add-hooks minibuffer-complete)

(push-new 'apollo:current-thing-no-directory apollo:minibuffer-complete-after-hooks)

(push-new 'apollo:setup-dynamic-mouse apollo:minibuffer-complete-after-hooks)

(apollo:add-hooks minibuffer-complete-and-exit)

(push-new 'apollo:current-thing-no-directory
          apollo:minibuffer-complete-and-exit-after-hooks)

(push-new 'apollo:setup-dynamic-mouse apollo:minibuffer-complete-and-exit-after-hooks)

(apollo:add-hooks minibuffer-complete-word)


(push-new 'apollo:current-thing-no-directory
          apollo:minibuffer-complete-word-after-hooks)

(push-new 'apollo:setup-dynamic-mouse apollo:minibuffer-complete-word-after-hooks)

(apollo:add-hooks minibuffer-completion-help)

(push-new 'apollo:setup-dynamic-mouse apollo:minibuffer-completion-help-after-hooks)

(defvar apollo:notification-hacks-p t)

(defvar apollo:notifications-kept 200)
(defun apollo:notification (&rest args)
  "Place a copy of each message in buffer *NOTIFICATIONS*.
       The newest message will be placed at the top of the
       buffer.  This is here because messages go by too
       quick for me!
       There is a flag called apollo:notifications-hacks-p used
       by this function when if non-nil, does some processing
       on what to display, This is slightly slower but makes
       this buffer easier to look at."
  (with-buffer-set (get-buffer-create apollo:notifications-buffer)
    (buffer-flush-undo (current-buffer))
    (let ((s (apply (function format) args)))
      (goto-char (point-min))
      (if (and apollo:notification-hacks-p
               (or (string= "Mark set" s)
                   (eq 0 (string-match apollo:all-whitespace s))))
          nil
          (if (and apollo:notification-hacks-p
                   (string-match "^I-search:" s)
                   (string-match
                    (regexp-quote (buffer-substring 1 (eol))) s))
              (delete-region  1 (min (point-max) (1+ (eol))))
              (progn
                (setq apollo:last-notification s)
                (insert s "\n")
                (if (zerop (forward-line apollo:notifications-kept))
                    (delete-region (point) (point-max)))))))))

(apollo:add-hooks error)

(push-new 'apollo:notification apollo:error-before-hooks)

(apollo:add-hooks message)

(push-new 'apollo:notification apollo:message-before-hooks)

(defvar apollo:printed-startup-message nil)

(defvar apollo:copyright-startup-notice
  "
  Copyright \(c\) 1988,1989, Apollo Computer, Inc. 
  The software and information herein are the property
  of Apollo Computer, Inc.
  This package may be distributed without further permission 
  from Apollo Computer, Inc. as long as:
    * all copyright notices and this notice are preserved
    * it is not part of a product for resale
    * a detailed description of all modifications is included
      in source files.
  This software is normally unsupported and without warranty.
  Support will be provided only for customers of 
  Domain/CommonLISP from APOLLO Computers Inc.
    lisp-tools:@apollo.com
    mit-eddie!ulowell!apollo!lisp-tools:
  or 
    Domain/CommonLISP Group
    Apollo Computer Inc.
    330 Billerica Road
    Chelmsford MA 01824
  Questions, comments and enhancements are welcome from all sources.")

(defun apollo:print-startup-message (&rest arg)
  (with-buffer-set 
      (if (equal arg '(apollo:load))
          (let ((b (get-buffer-create " *APOLLO-NOTICE-BUFFER*"))) (display-buffer b) b)
          "*scratch*")
    (setq apollo:printed-startup-message t)
    (if (not (equal arg '(apollo:load))) (insert "\n\n"))
    (insert   "*** APOLLO GNU Emacs Common Lisp Support Loaded ***"
              apollo:copyright-startup-notice)
    (goto-char 1)))

(apollo:add-hooks command-line-1)
(push-new 'apollo:print-startup-message apollo:command-line-1-before-hooks)

(if (not apollo:printed-startup-message) (apollo:print-startup-message 'apollo:load))

(defun apollo:error-if-mouse-trapped (&optional window)
  (interactive "p")
  (if (null window) (setq window 1))
  (if (numberp window)
      (let ((i window))
        (setq window (selected-window))
        (if (> i 0)
          (while (> i 0) (setq window (next-window     window) i (1- i)))
          (while (< i 0) (setq window (previous-window window) i (1+ i))))))
  (if (and (not apollo:in-dynamic-mouse-call-p)
           (eq (minibuffer-window) (selected-window)) (not (eq window (selected-window))))
      (error "Cannot Change Windows!")))

(apollo:add-hooks select-window)
(push-new 'apollo:error-if-mouse-trapped apollo:select-window-before-hooks)

(defun apollo:fix-other-window-args (&optional arg)
  (setq apollo:other-window-args (list (or arg 1))))


(apollo:add-hooks other-window)
(push-new 'apollo:fix-other-window-args apollo:other-window-before-hooks)
(push-new 'apollo:error-if-mouse-trapped apollo:other-window-before-hooks)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Apollo Specific AEGIS or UNIX functions
 
(defun apollo:real-directory (d &optional proc)
  "Sets a global variable apollo:last-canonical-file-name to (<dir> . <true-dir>).
       This actually goes to the directory and asks where it is. Ugh!"
  (if (apollo:apollo-p)
      (apollo:APOLLO-real-directory d proc)
      (apollo:UNIX-real-directory d proc)))

(defvar apollo:true-path-command nil)
(defun apollo:UNIX-real-directory (d &optional proc)
  "Sets a global variable apollo:last-canonical-file-name to (<dir> . <true-dir>).
       This actually goes to the directory and asks where it is. Ugh!
       This version wont work in APOLLO /com/sh!! and requires /bin/sh."
  (with-buffer-bury (get-buffer-create apollo:hack-buffer)
    (let ((command 
	   (if apollo:true-path-command
	       (concat "cd " d "; /bin/pwd\n")
	       (concat apollo:true-path-command " " d "\n"))))
      (if proc
          (with-process-filter (proc (function apollo:quiet-filter))
            (let ((apollo:string ""))
              (process-send-string proc command)
              (while (or (equal apollo:string "")
                         (eq (aref apollo:string (1- (length apollo:string))) ?\n))
                (accept-process-output proc))
              (if (string-match "\n" apollo:string)
                  (setq s (substring apollo:string 0 (match-beginning 0))))))
          (progn
	    (insert command)
            (call-process-region
             (point-min) (point-max) "/bin/sh" (not 'delete) t (not 'display))
            (goto-char (- (point-max) 1))
            (beginning-of-line)
            (if (looking-at "/bin/sh: .*: bad directory")
                (error "Bad Directory Error: %s" (buffer-substring (match-end 0) (eol))))
            (setq s (buffer-substring (point) (- (point-max) 1)))))
      (setq apollo:last-canonical-file-name (cons d s)))))



(defun-APOLLO apollo:APOLLO-real-directory (d &optional proc)
  "Sets a global variable apollo:last-canonical-file-name to (<dir> . <true-dir>).
       This actually goes to the directory and asks where it is. Ugh!
       This version works in all 3 APOLLO shells!"
  (with-buffer-bury (get-buffer-create apollo:hack-buffer)
    (let ((aegis-dir "") s)
      (apollo:dotimes (c (length d))
        (setq aegis-dir (concat
                         aegis-dir
                         (let ((char (substring d c (1+ c))))
                           (if (and apollo:*lowercase-file-directories*
                               (>= (aref char 0) ?A) (<= (aref char 0) ?Z))
                               (concat ":" (downcase char))
                               char)))))
      (insert (concat apollo:aegis-directory "/wd " aegis-dir "; "
                      apollo:aegis-directory "/wd\n"))
      (if proc
          (with-process-filter (proc (function apollo:quiet-filter))
            (let ((apollo:string ""))
              (process-send-string proc (concat apollo:aegis-directory "/wd\n"))
              (while (or (equal apollo:string "")
                         (eq (aref apollo:string (1- (length apollo:string))) ?\n))
                (accept-process-output proc))
              (if (string-match "\n" apollo:string)
                  (setq s (substring apollo:string 0 (match-beginning 0))))))
          (progn
            (call-process-region
             (point-min) (point-max) (concat apollo:aegis-directory "/sh")
             (not 'delete) t (not 'display) "-b")
            (goto-char (- (point-max) 1))
            (beginning-of-line)
            (if (looking-at "\?(wd)  ")
                (error "Bad Director Error: %s" (buffer-substring (match-end 0) (eol))))
            (setq s (buffer-substring (point) (- (point-max) 1)))))
      (setq apollo:last-canonical-file-name (cons d s)))))


(defun-APOLLO apollo:APOLLO-print-buffer (arg)
  "Send current buffer to printer.
       If ARG is 2, this will try to print in 2 column format.
       ARG 2 will only work on IMAGEN printers."
  (cond
    ((eq arg 2)
     (save-excursion
       (goto-char (point-min))
       (let ((buffer-read-only nil)
             (mod (buffer-modified-p)))
         (unwind-protect
              (progn
                (insert
                 "@document(language printer, jobheader on, "
                 "\"" "For:"        "\"" " " "\"" (getenv "USER")
                 "\"" ", "
                 "\"" "Document:"   "\"" " " "\"" (buffer-name (current-buffer))
                 "\"" ", "
                 "\"" "Printed on:" "\"" " " "\"" (current-time-string)
                 "\"" ", "
                 "formlength 66, formsperpage 2)")
                (call-process-region
                 (point-min) (point-max)
                 (concat apollo:aegis-directory "/prf")
                 (not 'delete) (not 'buffer) (not 'display)
                 (concat "-pr " apollo:default-printer
                         " -transparent -banner off " apollo:printer-args)))
           (progn (delete-region (point-min) (point))
                  (if (not mod) (not-modified)))))))
    (t (call-process-region
        (point-min) (point-max) (concat apollo:aegis-directory "/prf")
        (not 'delete) (not 'buffer) (not 'display)
        (concat "-pr " apollo:default-printer " " apollo:printer-args)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; buffer mode tests
(defun apollo:special-buffer-p (b)
  (with-buffer b
    (string-match "[ *]" (substring (buffer-name (current-buffer)) 0 1))))

(defun apollo:elisp-buffer-p ()
  "Is the current buffer an emacs-lisp buffer?"
  (or (eq major-mode 'emacs-lisp-mode)
      (and buffer-file-name
           (string-match apollo:elisp-file-pattern buffer-file-name))))

(defun apollo:ilisp-buffer-p ()
  "Is the current buffer an inferior-lisp buffer?"
  (eq major-mode 'inferior-lisp-mode))

(defun apollo:clisp-buffer-p ()
  "Is the current buffer a common-lisp buffer?"
  (or (eq major-mode 'lisp-mode)
      (and buffer-file-name
           (string-match apollo:clisp-file-pattern buffer-file-name))))

(defun apollo:lisp-buffer-p ()
  "Is the current buffer an lisp buffer?"
  (or (apollo:ilisp-buffer-p) (apollo:clisp-buffer-p) (apollo:elisp-buffer-p)
      (memq major-mode '(debugger-mode lisp-interaction-mode))))

(defun apollo:common-lisp-buffer-p ()
  (or apollo:force-common-lisp-p (apollo:clisp-buffer-p) (apollo:ilisp-buffer-p)))

(defun apollo:emacs-lisp-buffer-p () 
  (and (not apollo:force-common-lisp-p)
       (or (apollo:elisp-buffer-p) (eq major-mode 'lisp-interaction-mode))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Attribute stuff

(defvar apollo:update-attribute-list-force-p nil)

(defun apollo:set-variable
    (var in-val interpret-value-function &optional completions-func)
  "This is used to set a buffer local VAR.
       If IN-VAL is nil, you will be prompted to enter a value.
       Otherwise in val must be a the value this variable will be set to.  
       To set a variable to nil, supply 'nil.
       If non-nil, INTERPRET-VALUE-FUNCTION
       will be applied to IN-VAL to give the actual value the
       local variable will have.  The local variable will be
       named the concatenation of apollo: and the symbol name of VAR.
       If there is no attribute list for the current buffer, or
       the new value disagrees with that in the attribute list,
       you will be prompted to update the buffers attribute list."
  (let (sn val local (prompt-p nil))
    (setq sn (capitalize (symbol-name var)))
    (setq local (intern (concat "apollo:" (downcase sn))))
    (if (null in-val)
        (setq prompt-p t
              in-val
              (car (read-from-string
                    (if completions-func
                        (completing-read
                         (concat "Set " sn ": ")
                         (funcall completions-func) nil 'require-match)
                        (apollo:prompt-for  (concat "Set " sn) ""))))))
    (setq val (if (equal in-val '(quote nil)) nil in-val))
    (if interpret-value-function
        (setq val (funcall interpret-value-function val)))
    (if (or val (equal in-val '(quote nil)))
        (progn
          (set (make-variable-buffer-local local) val)
          (push-new local apollo:symbols)
          (if (or apollo:update-attribute-list-force-p prompt-p)
              (apollo:update-attribute-list sn (or val "'nil")))))))

(defun apollo:update-attribute-list (sn val)
  "Do update of attribute list in buffer
        SN     symbol name of variable thing being set
        VAL    a string to display as value on attribute list"
  (save-excursion
    (goto-char (point-min))
    (let ((a-regexp (concat (regexp-quote sn) apollo:attribute-value-separator))
          (val (capitalize (format "%s" val)))
          (mode-p (equal sn "Mode"))
          (a-start
           (if (re-search-forward 
                (format apollo:attribute-line-prefix (regexp-quote (or comment-start "")))
                (eol) t)
               (point))))
      (if a-start
          (progn
            (set-marker 
             apollo:marker
             (if (re-search-forward apollo:attribute-line-end nil t) 
                 (match-beginning 0)
                 (eol)))
            (goto-char a-start)
            (if (re-search-forward a-regexp apollo:marker 'end)
                (if (and (not (looking-at (regexp-quote val)))
                         (y-or-n-p 
                          (format "Update %s: %s in attribute list? " sn val)))
                    (let ((m (point)))
                      (re-search-forward apollo:attribute-end nil t)
                      (goto-char (match-beginning 0))
                      (delete-region m (point))
                      (insert val)
                      (if apollo:verbose (message "%s" (buffer-substring a-start apollo:marker)))))
                (if (y-or-n-p 
                     (format "Add %s: %s to attribute list also? " sn val))
                    (let ((s (if mode-p 
                                 (concat sn ": " val "; ") 
                                 (concat "; " sn ": " val)))) 
                      (goto-char (if mode-p a-start apollo:marker))
                      (insert s)
                      (if apollo:verbose
                          (message "%s" 
                                   (buffer-substring
                                     a-start 
                                     (+ (marker-position apollo:marker) (length s)))))))))
          (if (y-or-n-p (format "Create attribute list with %s: %s? "  sn val))
              (let ((s (concat comment-start " -*- " sn ": " val " -*-" comment-end "\n")))
                (insert s)
                (if apollo:verbose (message "%s" s))))))))


(defun apollo:new-set-variable (local &optional value)
  "Sets a local variable LOCAL to VALUE.
       If none exists this will also
       define a M-x set-<variable-name>
       command for LOCAL.  If VALUE is nil or
       Not specified, only the function will be defined."
  (let ((f (intern (concat "set-" (symbol-name local)))))
    (if (or nil (not (fboundp f)))
        (let ((variable (apollo:gensym)))
          (eval
            (car
              (read-from-string
                (format
                  "(defun set-%s (%s) 
                      (interactive \"P\") 
                      (apollo:set-variable '%s %s nil))" 
                  local variable local variable))))))
    (if value (apply f (list value)))))

(apollo:new-set-variable 'readtable)
(apollo:new-set-variable 'base)
(apollo:new-set-variable 'ibase)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Original buffer stuff for changed definition commands

(defun apollo:set-original ()
  "Create a buffer called *<buffer-file-name>.~original~* for current-buffer.
       This will hold a copy of the file when it was read in.
       This will be used by changed-definition commands to compare
       with the current buffer."
  (if (and buffer-file-name 
           (not (and apollo:original (get-buffer apollo:original)))
           (or (string-match apollo:clisp-file-pattern buffer-file-name)
               (string-match apollo:elisp-file-pattern buffer-file-name)))
      (let ((curr (current-buffer)))
        (setq apollo:original (format " *%s.~original~*" buffer-file-name))
        (with-buffer-set (get-buffer-create apollo:original)
          (erase-buffer)
          (insert-buffer curr)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Boring useful functions

(defun apollo:strip-whitespace (string)
  (let* ((l (length string))
         (min (apollo:dotimes (n l l)
                (if (not (memq (aref string n) apollo:whitespace-chars))
                    (apollo:return n))))
         (max (apollo:dotimes (n (- l min) min)
                (if (not (memq (aref string (- l n 1)) apollo:whitespace-chars))
                    (apollo:return (- l n))))))
    (if (and (= min 0) (= max l)) string (substring string min max))))

(defun apollo:minibuffer-p ()
  (string-match " \\*Minibuf-[0-9]?\\*" (buffer-name (current-buffer))))

(defun apollo:strip-leading-whitespace (string)
  (if (string-match apollo:non-whitespace string)
      (if (> (match-beginning 0) 0) 
          (substring string (match-beginning 0))
          string)
      ""))

(defun arglist (obj)
  (let ((name 'nil))
    (while (symbolp obj) (setq name obj obj (symbol-function obj)))
    (if (subrp obj) (error "Can't find arglist for #<subr %s>" name))
    (if (eq (car obj) 'macro) (setq  obj (cdr obj)))
    (if (not (eq (car obj) 'lambda)) (error "not a function"))
    (autoload 'byte-compile-lambda "bytecomp")
    (if (not (assq 'byte-code obj)) 
        (setq obj (byte-compile-lambda obj)))
    (cadr obj)))
  
(defun apollo:page-string (date buffer page)
  ""
  (interactive)
  (let* ((buffer (buffer-name buffer))
         (page   (format "%s" page))
         (space-after-date 
          (make-string (max 1 (- 40 (length date) (/ (length buffer) 2))) ?\ ))
         (page-string (concat date space-after-date buffer))
         (space-after-buffer
          (make-string 
           (max 1 (- 80 (length page-string) (length page))) ?\ )))
    (concat page-string space-after-buffer page "\n")))
        
(defun apollo:mouse-move-point (&optional arg)
  "Mark region from where mouse is pressed to where it is release.
        Mark is left at beginning and point at end."
  (interactive)
  (let (term)
    (cond ((memq window-system '(x x11)) (x-mouse-set-point arg))
	  ((and (setq term (getenv "TERM"))
		(setq term (assq (intern term) apollo:terminals)))
	   (autoload (car term) (cdr term))
	   (apollo:terminal-move-point))
	  ((apollo:apollo-p) 
	   (autoload 'apollo-mouse-move-point "term/apollo") 
	   (apollo-mouse-move-point))
	  (t (error "Unknown mouse definition for apollo:mouse-move-point")))))


(defun apollo:mode ()
  (let ((m (symbol-name major-mode)))
    (cond ((and (eq major-mode 'lisp-mode) 
                (string-match "^Common-Lisp" mode-name))
           "common-lisp")
          ((string-match "-mode$" m) (substring m 0 (match-beginning 0)))
          (t m))))

(defun apollo:macro-expand-expression-all (sexp)
  "Recursive macroexpand SEXP.  Returns expanded s-expression."
  (if (consp sexp) 
      (mapcar (function apollo:macro-expand-expression-all) (macroexpand sexp))
      sexp))

(defun apollo:macro-expand-expression (sexp)
  "Macroexpand SEXP.  Returns expanded s-expression."
  (macroexpand sexp))

(defun apollo:pprint (s &optional stream)
  "A rude & ugly quick try at pretty print."
  (with-buffer-set (or stream (current-buffer))
    (let (sep)
      (lisp-indent-line)
      (cond ((consp s)
             (insert "(")
             (apollo:pprint (car s) stream)
             (setq sep
                   (if (or (consp (car s))
                           (memq (car s) 
                                 '(save-excursion let let* while catch 
                                   progn if cond apollo:dolist 
                                   apollo:dotimes lambda unwind-protect))
                           (and (not (memq (car s) '()))
                                (> (length s) 3)))
                       ?\n
                       " "))
             (apollo:dolist (i (cdr s)) (insert sep) (apollo:pprint i stream))
             (insert ")"))
            ((stringp s)
             (insert "\"" (apollo:escape-string s "\"") "\""))
            ((symbolp s) (insert (symbol-name s)))
            (t (insert (format "%s" s)))))))

(defun apollo:canonical-file-name (filename)
  "Canonical form of file name."
  (let ((d (file-name-directory filename))
        (f (file-name-nondirectory filename)))
    (concat (cdr (apollo:real-directory d)) "/" f)))


(defun apollo:get-completion (arg o name c &optional mode package dynamic-mouse)
  "This is the grunt of completion stuff in this file.
       If ARG is non-nil, all strings with given pattern are selected.
       If ARG is nil, only strings with the given pattern at the beginning
           are selected.
       ALIST is an association list of completion choices such that the first
           element of each sublist is a string to be selected from completions.
       NAME is the name of the operation completion is for.
       STRING is the string to use for pattern matching on ALIST.
       Optional third argument MODE is the mode for the completions buffer.
       Optional fourth argument PACKAGE is package for the completions buffer.
       Optional fifth argument DYNAMIC-MOUSE is binding for apollo:dynamic-mouse-binding
       which is a function to do if the mouse is clicked right on this buffer."
  (if (get-buffer apollo:completions-buffer) (bury-buffer apollo:completions-buffer))
  (if (not o)
      (error (format "No %s completions for \"%s\"." name (or c "")))
      (let ((completion (if arg c (try-completion c o))))
        (if (eq completion t)
            (message (format "%s is already complete." c))
            (let ((name (concat name " completions for \"" c "\":")))
              (if (> (length o) 1)
                  (apollo:lines-to-buffer 
                   apollo:completions-buffer name 'refresh o
                   (function (lambda (s) (if (consp s) (car s) s)))
                   mode package dynamic-mouse))
              (progn
                (delete-region apollo:mark apollo:point)
                (goto-char apollo:mark)
                (insert completion)
                (setq apollo:point (point))
                ))))))


(defun apollo:key-printable (k)
  "Given a string with control characters representing a key binding,
       this retruns a printable version of that string.  The returned
       string is not suitable for binding keys, only for display."
  (if k
      (let ((result ""))
        (apollo:dotimes (n (length k))
          (if (string-match "[\1-\32]" (substring k n (+ n 1)))
              (setq result 
                    (concat 
                     result "Control-" 
                     (char-to-string (+ ?a -1 (aref k n))) (if (= n 0) " ")))
              (if (string-match "[\33]" (substring k n (+ n 1)))
                  (setq result (concat result (if (= n 0) "Meta-" "<Esc>")))
                  (setq result (concat result (substring k n (+ n 1)))))))
        (let ((r (and (or (string-match "Meta-Control-" result)
                          (string-match "Meta-" result))
                      (substring result (match-end 0)))))
          (if (and r (>= (length r) 2))
              (let* ((r1 (aref r 1))
                     (mouse-result (cond ((eq r1 ?a) "<Mouse Left Down>")
                                         ((eq r1 ?b) "<Mouse Middle Down>")
                                         ((eq r1 ?c) "<Mouse Right Down>")
                                         ((eq r1 ?A) "<Mouse Left Up>")
                                         ((eq r1 ?B) "<Mouse Middle Up>")
                                         ((eq r1 ?C) "<Mouse Right Up>")
                                         (t nil))))
                (if mouse-result
                    (setq result
                          (cond ((eq (aref r 0) ?*) mouse-result)
                                ((eq (aref r 0) ?+) (concat (substring result 0 (match-end 0)) mouse-result))
                                (t 
                                   result)))))))
        (or result k))))

(defvar apollo:section-separator 
  (concat "^;;;;*" apollo:whitespace "*\n;;;" apollo:whitespace "*"))
(defvar apollo:next-defun (concat "^(defun" apollo:whitespace "*"))
(defvar apollo:next-defvar (concat "^(defvar" apollo:whitespace "*"))


(defun generate-readme ()
  "This is just to turn this buffer into a README buffer for this stuff"
  (interactive)
  (save-excursion
    (let ((r (save-excursion (get-buffer-create "*READ-ME*")))
          (instructions nil))
      (goto-char (point-min))
      (re-search-forward "^;;; Begin Instructions\n" nil t)
      (set-mark (point))
      (re-search-forward "^;;; End Concepts\n" nil t)
      (previous-line 1)
      (setq instructions (buffer-substring (point) (mark)))
      (with-buffer-set r 
        (erase-buffer)
        (insert "\n" instructions)
        (goto-char (point-min))
        (if (re-search-forward "^;;; End Instructions\n" nil t)
            (progn (forward-line -1)
                   (delete-region (progn (beginning-of-line) (point)) (eol))))
        (goto-char (point-min))
        (mark-whole-buffer)
        (let ((comment-start ";")
              (comment-end ""))
          (uncomment-out-region 1)) 
        (goto-char (point-min))
        (insert-date))
      (goto-char (point-min))
      ;; Document the configuration variables
      (if (re-search-forward "^;;; Configuration variables" nil t)
          (let ((next-var nil)
                (end (or (save-excursion 
                           (re-search-forward "^;;; End Configuration variables" nil t) 
                           (match-beginning 0)) 
                         (point-max))))
            (save-restriction
              (narrow-to-region (point) end)
              (with-buffer-set r
                (goto-char (point-max))
                (insert 12 ?\n "CONFIGURATION VARIABLES:\n\n"))
              (while (save-excursion (and (re-search-forward apollo:next-defvar nil t) (setq next-var (point))))
                (let* ((str (progn (goto-char next-var) (mark-sexp 1) 
                                   (buffer-substring (mark) (point))))
                       (sym (intern str))
                       (default (progn (forward-sexp 1)
                                       (mark-sexp 1)
                                       (buffer-substring (mark) (point))))
                       ;; These doc strings should be recorded just as function doc strings are.
                       (documentation (progn (forward-sexp 1)
                                             (mark-sexp 1)
                                             (buffer-substring (mark) (point)))))
                  (with-buffer-set r
                    (goto-char (point-max))
                    (insert str "\n   Default: " default "   " documentation "\n\n"))
                  (message "Defvar %s" str))))))
      (goto-char (point-min))
      ;; Document the commands
      (if (re-search-forward "^;;; Interactive Stuff" nil t)
          (let ((end (or (save-excursion (re-search-forward "^;;; End of Interactive Stuff" nil t) 
                                         (match-beginning 0)) 
                         (point-max))))
            (save-restriction
              (narrow-to-region (point) end)

              (while (save-excursion (and (re-search-forward apollo:next-defun nil t) (setq next-def (point))))
                (let ()
                  (if (re-search-forward apollo:section-separator next-def t)
                      (let ((section-name (buffer-substring (point) (eol))))
                        (beginning-of-line)
                        (while (looking-at "^;") (forward-line 1))
                        (with-buffer-set r
                          (goto-char (point-max))
                          (insert 12 ?\n (upcase section-name) ":\n\n")))
                      (let*
                          ((str (progn (goto-char next-def) (mark-sexp 1) 
                                       (buffer-substring (mark) (point))))
                           (sym (intern str))
                           (modes '(lisp-mode emacs-lisp-mode 
                                    inferior-lisp-mode shell-mode global)))
                        (message "%s" str)
                        (with-buffer-set r 
                          (goto-char (point-max)) 
                          (insert str "\n")
                          (let ((result-string nil)
                                (last-key-seen nil)
                                (all-modes-the-same-p t)
                                (global-seen-p nil))
                            (apollo:dolist (m modes)
                              (let* ((mode-map (intern (format "%s-map" m)))
                                     (mode-name (capitalize (symbol-name m)))
                                     (key (car (where-is-internal sym (eval mode-map)))))
                                (if key 
                                    (progn 
                                      (if (string= mode-name "Global")
                                          (setq global-seen-p 
                                                (concat
                                                  "    Key Binding:    "
                                                  mode-name " [" (apollo:key-printable key) "]\n")))
                                      (setq all-modes-the-same-p
                                            (and all-modes-the-same-p
                                                 (or (null last-key-seen) (string= key last-key-seen))))
                                      (setq last-key-seen key)
                                      (setq result-string
                                            (concat result-string
                                                    "    Key Binding:    "
                                                    mode-name " [" (apollo:key-printable key) "]\n"))))))
                            (goto-char (point-max))
                            (if result-string 
                                (insert (if (and all-modes-the-same-p global-seen-p)
                                            global-seen-p
                                            result-string))))
                          (insert "    " (or (documentation sym) "") "\n\n")))))))
            (with-buffer-set r (write-file "/emacs-files/README")))
          (error "No interactive marker")))))


(defun apollo:prompt-for-cl (prompt-string &optional default)
  "Prompt MESSAGE with DEFAULT using common lisp mode keymap"
  (let ((map apollo:common-lisp-minibuffer-local-must-match-map)
        (apollo:dynamic-mouse-binding 'apollo:grab-thing-dynamic-mouse)
        (apollo:mark 1)
        (apollo:point 1)
        (apollo:force-common-lisp-p t)
        (apollo:temp-package (apollo:find-package))
        (apollo:temp-base    apollo:base)
        (apollo:temp-ibase   apollo:ibase)
        (prompt (if default 
                    (format "%s:[%s] " prompt-string default)
                    (concat prompt-string ": "))))
    (let ((input (if apollo:prompt-for-cl-use-default-p
                     ""   
                     (read-from-minibuffer prompt nil map))))
      (if (or (null input) (equal "" input)) default input))))

(defun apollo:prompt-for (mess default)
  "Prompt MESSAGE with DEFAULT"
  (let* ((apollo:dynamic-mouse-binding 'apollo:grab-thing-dynamic-mouse)
         (apollo:mark 1)
         (apollo:point 1)
        (s (read-string (format "%s:[%s] " mess default))))
    (if (equal s "")
        (if (equal default "")
            (progn (message (format "%s Cancelled" mess)) (sit-for 5) nil)
            default)
        s)))

(defun apollo:read-buffer (string switch-buff &optional yank-default)
  (if (not yank-default) (setq yank-default switch-buff))
  (let ((kill-ring (cons yank-default kill-ring)))
    (read-buffer string switch-buff)))

(defun apollo:stringify-caller (line)
  "Display LINE, a list of (changed-definition-name definition-type buffer-name)
       to the current buffer.  
       References external symbol lengths set up in apollo:lines-to-buffer.
       This is a list of maximum lengths of each column."
  (let* ((name  (symbol-name (apollo:section-name line)))
         (lname (length name))
         (def-col (+ 2 (nth 0 column-widths)))
         (name-space (make-string (max 1 (- def-col lname)) ?\ ))
         (def   (caddr line))
         (ldef  (length  def))
         (buf-col (+ def-col (+ 2 (nth 2 column-widths))))
         (def-space (make-string (max 1 (- buf-col (+ def-col ldef))) ?\ ))
         (buf   (cadr line)))
    (concat name name-space def def-space buf "\n")))


(defun apollo:lisp-callers (arg)
  "Returns a list of a Common Lisp functions callers.
       With no ARG, this will look in current package.
       ARG of 1 or  4 (c-u) will look in all packages.
       ARG of 2 or 16 (c-u c-u) will prompt for a package to use.
       This function will prompt for a symbol to look for."
  (setq arg
        (cond ((eq arg nil) 
               (concat "'" (apollo:find-package)))
              ((memq arg '(2 16))
               (concat "'" 
                       (apollo:prompt-for 
                        "What package do you want to search?" 
                        (apollo:find-package))))
              ((memq arg '(1 4)) (setq arg nil))
              (t (error 
                  (format "Illegal argument [%s] to apollo:lisp-callers" arg)))))
  (setq apollo:symbol 
        (apollo:packagify 
         (apollo:prompt-for-cl
          (format "%sFind Callers for" 
                  (concat "(Package: " 
                          (if arg (upcase arg) "<All PACKAGES>") ") "))
          (or (apollo:current-thing nil (not 'sexps-ok)) ""))))
  (if (string-match ":$" apollo:symbol)
      (error "No symbol specified")
      (progn
        (message "Finding callers of %s" apollo:symbol)
        (let ((callers 
               (evaluate-common-lisp 
                 nil (format "(APOLLO::WHO-CALLS '%s %s)" apollo:symbol arg))))
          (setq apollo:callers (if (eq callers 'NIL) nil callers))))))


(defun apollo:evaluate-current-thing 
    (string form &optional value click-right without-package-p buffer-prefix)
  "Wrap a common lisp form around symbol and evaluate.
       STRING is the name of the form being applied.  STRING is used for
       prompting and error message by this function.
       FORM is a format string representing a common lisp form.  It should have
       a single %s field in the FORM to be filled in with the current thing.
       This will prompt for the symbol to use in the form with the default being
       the variable at point.
       Optional third argument VALUE may be a string representing a symbol.
       If supplied, function will not prompt, but use VALUE when evaluating.
       Optional CLICK-RIGHT is an action to perform in the display buffer 
           when clicking right
       Optional WITHOUT-PACKAGE-P if non-nil strips the package from the symbol,
       Optional WITHOUT-PACKAGE-P if nil will glue the package to the symbol.
       Optional BUFFER-PREFIX non-nil if name of thing to preceed buffer name.
       "
  (if (not (stringp value)) (setq value nil))
  (if (apollo:common-lisp-buffer-p)
      (let* 
          ((thing-in  (or value  (apollo:current-thing nil (not 'sexps-ok))))
           (thing     (if thing-in  
                          (if without-package-p 
                              (apollo:strip-package thing-in)  
                              (apollo:packagify thing-in  t))))
           (symbol-in (apollo:prompt-for-cl string thing))
           (symbol    (if symbol-in 
                          (if without-package-p 
                              (apollo:strip-package symbol-in) 
                              (apollo:packagify symbol-in t))))
           (result    (if symbol 
                          (evaluate-common-lisp nil (format form symbol))))
           (buffer    (upcase 
                       (concat "*" 
                               (if buffer-prefix 
                                   (concat (upcase (apollo:strip-package symbol))
                                           ":_")) string "*")))
           (title     (if symbol (format "%s \"%s\"" string (upcase symbol)))))
        (if (get-buffer buffer) (bury-buffer buffer))
        (if symbol
            (if result
                (apollo:lines-to-buffer 
                 buffer title 'refresh result nil 
                 "common-lisp" "USER" click-right))
            (error (format "No symbol for %s." string))))
      (error (format "%s only works for common lisp buffers" string))))

(defun apollo:all-buffers (test)
  "Sort of a apollo:dolist for buffers
       Calls a TEST function on all buffers and returns
       a list of those that satisfied test."
  (let ((all-buffers  (buffer-list))
        (buffers nil))
    (apollo:dolist (b all-buffers)
      (if (funcall test (buffer-name b))
          (push b  buffers)))
    buffers))


(defun apollo:escape-string (string pattern &optional escape-prefix match-number)
  "Escape sequences in STRING matching PATTERN with ESCAPE-PREFIX.
       Optional ESCAPE-PREFIX will default to backslash if nil.
       Optional MATCH-NUMBER is the number of the pattern in the regular
       expression PATTERN to escape."
  (if escape-prefix nil (setq escape-prefix "\\"))
  (if match-number nil  (setq match-number 0))
  (let ((e ""))
    (while (string-match pattern string)
      (setq e 
            (concat 
             e
             (substring string 0 (match-beginning match-number))
             escape-prefix
             (substring string (match-beginning match-number) 
                        (match-end match-number)))
            string (substring string (match-end match-number))))
    (concat e string)))

(defun apollo:mark-defun ()
  "Marks defun with point at '(' and mark at after ')'."
  (if (not (eobp)) (forward-char))
  (beginning-of-defun)
  (set-mark (point))
  (setq apollo:mark (point))
  (end-of-defun)
  (search-backward ")")
  (forward-char)
  (setq apollo:point (point))
  (exchange-point-and-mark))


(defun apollo:lines-to-buffer  
    (b title refresh lines print &optional mode package dynamic-mouse)
  "Displays a list of lines to BUFFER.  TITLE will be printed
       at the top of the display if non-nil.
       The buffer is cleared before doing any display if
       REFRESH is non-nil.  A list of LINES to be printed
       is supplied.  The entries in this list may be of
       any form so long as funcall'ing PRINT on that
       list will print a single line.  Optional arguments
       MODE is a string representing the new mode and PACKAGE
       is a package to set in the new package if the mode is
       lisp or common-lisp.  MODE should not contain the ending
       -mode."
  (if (stringp lines) (setq lines (list lines)))
  (if print
      (let ((column-widths (list 0)))
        (if (consp (car lines))
            (condition-case foo
                (let ((max 0))
                  (apollo:dolist (l lines) (let ((ll (length l))) (if (> ll max) (setq max ll))))
                  (setq column-widths (make-list max 0))
                  (apollo:dolist (l lines)
                    (if (consp l)
                        (let ((cwp column-widths))
                          (apollo::dolist (c l)
                            (let ((cw (length (if (symbolp s) (symbol-name c) c))))
                              (if (> cw (car cwp)) (rplaca cwp cw)))
                            (pop cwp))))))
              (error (setq column-widths nil)))
            (apollo:dolist (l lines)
              (let ((ll (length (if (symbolp l) (symbol-name l) l))))
                (if (> ll (car column-widths)) (rplaca column-widths ll)))))
	(let ((n 0))
	  (setq lines (mapcar (function (lambda (p) (apollo:incf n) (funcall print p))) lines)))))
  (with-buffer-set (get-buffer-create b)
    (if buffer-read-only (toggle-read-only))
    (if refresh (erase-buffer) (goto-char (point-max)))
    (insert
      (if title (concat title "\n") "")
      (if dynamic-mouse
        (let ((d (documentation dynamic-mouse)))
          (set (make-variable-buffer-local 'apollo:dynamic-mouse-binding) dynamic-mouse)
          (concat "[Mouse Right]: " 
                  (substring d 0 (string-match "\n\\|$" d))
                  "\n\n"))
        ""))
    (if (consp lines)
        (let ((w-width (window-width (selected-window)))
              (i-width 0)
              (add-newlines (not (string-match "\n" (car lines)))))
          (setq i-width 
                (if add-newlines 
                    (+ 4 
                       (apply (function max) 
                              (mapcar (function length) lines))) w-width))
          (let* ((columns 
                  (max 1 (/ w-width i-width))) ; leave 4 spaces between items
                 (c -1))
            (apollo:dolist (l lines)
              (if (>= (apollo:incf c) columns)
                  (progn (if add-newlines (insert "\n")) (setq c 0))
                  (indent-to-column (* c i-width)))
              (insert l)))))

    (if mode
        (progn
          (if (and package (or (equal mode "lisp") (equal mode "common-lisp")))
              (setq apollo:package package))
          (funcall (intern (concat mode "-mode")))))
    (if refresh 
        (let ((window (display-buffer b)))
          ;;(enlarge-window ;; This form makes the new window the size of what it displays
          ;;  (- (window-height window) 1 
          ;;     (- 1000000 (save-excursion (goto-char (point-min)) (forward-line 1000000)))))
          (set-window-start window 1)))
    (if (not buffer-read-only) (toggle-read-only))
    (set-buffer-modified-p nil))
  lines)

(defun apollo:matching-lines-to-buffer (str b)
  "Writes lines REGEXP to a BUFFER"
  (let ((done nil))
    (while (not done)
      (beginning-of-line)
      (set-mark (point))
      (end-of-line)
      (append-to-buffer b (mark) (1+ (point)))
      (set-mark (point-max))
      (if (not (re-search-forward str nil t))
          (setq done t)))
    (message "Done")))

(defun apollo:arg-file (arg)
  "Returns buffer corresponding to ARG.
       ARG of 1 uses changes since file was read in.
       ARG of 2 will use changes since file read in or saved.
       ARG of 3 uses changes since definition was compiled or evaluated."
  (cond ((or (eq arg apollo:since-read) (eq arg apollo:since-eval-or-compile))
         (if (not apollo:original) (apollo:reset-original nil))
         (let ((b (get-buffer apollo:original)))
           (if (not (eq b apollo:last-arg-file))
               (with-buffer apollo:original
                 (setq apollo:last-arg-file b)
                 (write-region 
                  (point-min) (point-max) apollo:arg-file nil 'no-write-message)))
           apollo:arg-file))
        ((eq arg apollo:since-read-or-save) buffer-file-name)
        (t (error (format "Bad arg: %s" arg)))))


(defun apollo:find-unbalanced-parentheses (&optional verbose)
  "Search current buffer for unmatched parentheses.
       If VERBOSE is non-nil, this will also display a
       message when parenthesis are OK."
  (let ((prev 0) (fl 0) (err nil))
    (save-excursion
      (goto-char (point-min))
      (while (< (setq prev (point)) 
                (setq fl 
                      (condition-case foo 
                          (forward-list) 
                        (error (setq err foo) prev))))))
    (if err
        (progn (goto-char prev)
               (if (looking-at apollo:whitespace)
                   (progn
                     (re-search-forward "[()]" nil t)
                     (backward-char)))
               (cond ((looking-at "(") (error "Unmatched left paren"))
                     ((looking-at ")") (error "Unmatched right paren"))
                     (t (error "Parenthesis appear Unbalanced after point")))
               (point))
        (progn (if verbose (message "Parentheses appear balanced")) nil))))

(defun apollo:unbal-check ()
  "This is used to make sure that parenthesis balance befor saving
       a lisp file.  This happens on a lisp machine."
  (if apollo:check-parenthesis-before-saving-p
      (if (or (apollo:elisp-buffer-p) (apollo:clisp-buffer-p))
          (condition-case foo 
              (funcall (function apollo:find-unbalanced-parentheses))
            (error
              (if (y-or-n-p (concat (car (cdr foo)) "! Save anyway? "))
                  nil
                  (progn 
                    (message "Buffer not saved: %s" 
                             (buffer-name (current-buffer))) 
                    t)))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Stuff for figuring out and manipulating package

(defun apollo:current-inferior-lisp-package (&optional used-p)
  (if used-p
      (setq apollo:*used-packages*
            (apollo:ask-common-lisp 
              "(MAPCAR 'PACKAGE-NAME (PACKAGE-USE-LIST *PACKAGE*))")))
  (apollo:ask-common-lisp "(APOLLO::LISP-PACKAGE)\n"))

(defun apollo:find-buffer-package (&optional used-p)
  "Look in attribute list for buffer package.
        If apollo:look-for-in-package-p is
        Non-nil will search backwards in common lisp buffer
        for line starting with (In-Package <package>)
        and use this <package> when evaluating common lisp forms.
        If nil or (In-Package <package>) is not found,
        Non-nil apollo:look-for-attribute-line-package-p
        will cause the buffers attribute line to be
        checked.  Otherwise the variable apollo:package
        will be examined for an uppercase string representing
        the package.  apollo:package should be set with
        M-x set-package  If this variable is nil, the
        current inferior lisp package will be used."
  (if (and apollo:look-for-attribute-line-package-p
           (or (apollo:ilisp-buffer-p) (apollo:clisp-buffer-p)))
      (save-excursion
        (goto-char (point-min))
        (let ((eol (eol)) (result nil))
          (if (and (re-search-forward
                    (format apollo:attribute-line-prefix (regexp-quote (or comment-start "")))
                    eol t)
                   (re-search-forward 
                    (concat "Package" apollo:attribute-value-separator) eol t))
              (let ((m (point)))
                (if (re-search-forward apollo:attribute-end eol t)
                    (let (value-string apollo:package apollo:superpackage 
                                       apollo:symbol-count (mode-name mode-name))
                      (set-package (buffer-substring m (match-beginning 0)))
                      (if used-p (setq apollo:used-packages apollo:superpackage))
                      apollo:package))))))))


(defun apollo:find-in-package (&optional used-p)
  "Search backwards for current package.
        If apollo:look-for-in-package-p is
        Non-nil will search backwards in common lisp buffer
        for line starting with (In-Package <package>)
        and use this <package> when evaluating common lisp forms.
        If nil or (In-Package <package>) is not found,
        Non-nil apollo:look-for-attribute-line-package-p
        will cause the buffers attribute line to be
        checked.  Otherwise the variable apollo:package
        will be examined for an uppercase string representing
        the package.  apollo:package should be set with
        M-x set-package.  If this variable is nil, the
        current inferior lisp package will be used."
  (if (and apollo:look-for-in-package-p
           (or (apollo:ilisp-buffer-p) (apollo:clisp-buffer-p)))
      (save-excursion
        (if (re-search-backward "^(In-Package" nil t)
            (let ((result nil))
              (apollo:mark-defun)
              (forward-char)
              (forward-sexp)
              (skip-chars-forward apollo:whitespace)
              (forward-char)            ; skip quotes
              (set-mark (point))
              (forward-sexp)
              (setq result (upcase (buffer-substring (mark) (point))))
              (if (looking-at "\"") (forward-char 1))
              (if used-p
                  (progn
                    (setq apollo:used-packages nil)
                    (if (and (not (looking-at (concat apollo:whitespace "*)")))
                             (re-search-forward 
                              (concat ":use" apollo:whitespace "*") 
                              (max apollo:mark apollo:point) 'no-error))
                        (setq apollo:used-packages
                              (let ((r (read (current-buffer))))
                                (if (and (consp r) (eq (car r) 'quote)) (setq r (cadr r)))
                                (mapcar '(lambda (x) 
                                          (upcase (if (symbolp x) (symbol-name x) x)))
                                        (if (consp r) r (list r))))))))
              result)))))


(defun apollo:find-package (&optional used-p)
  "Find package for evaluation at point.
        If apollo:look-for-in-package-p is
        Non-nil will search backwards in common lisp buffer
        for line starting with (In-Package <package>)
        and use this <package> when evaluating common lisp forms.
        If nil or (In-Package <package>) is not found,
        Non-nil apollo:look-for-attribute-line-package-p
        will cause the buffers attribute line to be
        checked.  Otherwise the variable apollo:package
        will be examined for an uppercase string representing
        the package.  apollo:package should be set with
        M-x set-package.  If this variable is nil, the
        current inferior lisp package will be used."
  (if (or (apollo:clisp-buffer-p) (apollo:ilisp-buffer-p))
      (or
        (and (apollo:ilisp-buffer-p) (apollo:current-inferior-lisp-package used-p))
        (apollo:find-in-package     used-p)
        (apollo:find-buffer-package used-p)
        (if (boundp 'apollo:package)
            (progn (if used-p (setq apollo:used-packages apollo:superpackage)) 
                   apollo:package)))))                                

(defun apollo:strip-package (symbol) ;; return symbol sans foo:: or foo:
  "A STRING representing a common lisp function name is
       searched for a prepended package as in package::string.
       The string is returned sans the package and colans"
  (if (and (string-match "::?" symbol) 
           (> (match-beginning 0) 0)) 
      (substring symbol (match-end 0)) symbol))

(defun apollo:extract-package (symbol)             ; return package name of symbol
  "A STRING representing a common lisp function name is
       searched for a prepended package as in package::string.
       The package returned sans the string and colans"
  (if (string-match "::?" symbol) (substring symbol 0 (match-beginning 0)) nil))


(defun apollo:packagify (sym &optional clisp-p p)
  "A string representing a common lisp SYMBOL is
       examined.  If a package is a part of the name as
       in package::name this is returned.  Otherwise
       an attempt is made to find the package using (apollo:find-package)
       and if one is found it is prepended to to this name
       and returned.  If no package is found, the
       name is returned."
  (if clisp-p
      (if (eq clisp-p t) 
          (if p nil (setq p (apollo:find-package))) (setq clisp-p nil))
      (setq clisp-p (or (apollo:clisp-buffer-p) (apollo:ilisp-buffer-p))))
  (if clisp-p
      (let* ((osym sym)
             (sym  (if (consp sym) (nth 0 sym) sym))
             (result
              (if (memq (aref sym 0) '(?\())
                  sym
                  (upcase
                    (if (string-match "::?" sym)
                        (if (> (match-beginning 0) 0)
                            (concat (substring sym 0 (match-beginning 0)) 
                                    "::" 
                                    (substring sym (match-end 0)))
                            sym);; Keyword
                        (if (apollo:ilisp-buffer-p)
                            (evaluate-common-lisp nil
                                                  (concat "(CONCATENATE 'STRING 
                                           (PACKAGE-NAME (SYMBOL-PACKAGE '" sym "))
                                           \"::\" 
                                           (SYMBOL-NAME '" sym "))"))
                            (if (or p (setq p (apollo:find-package)))
                                (concat p "::" sym)
                                sym)))))))
        (if (consp osym) (concat "(" result " " (nth 1 osym) " " (nth 2 osym) ")") result))
      sym))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Section Stuff

(defvar apollo:section-reuse-list nil)

(defmacro apollo:section-marker (section) (list 'car (list 'cdr section)))

(defmacro apollo:section-start  (section) 
  (list 'marker-position (list 'apollo:section-marker section)))

(defmacro apollo:section-buffer (section) 
  (list 'marker-buffer   (list 'apollo:section-marker section)))

(defmacro apollo:section-name   (section) (list 'car section))

(defmacro apollo:section-type   (section) (list 'cdr (list 'cdr section)))

(defmacro apollo:point-section (point)
  (let ((s (apollo:gensym)))
    (list 'apollo:dolist (list s 'apollo:buffer-sections) 
          (list 'if (list '<= (list 'apollo:section-start s) point)
                (list 'apollo:return s)))))

(defmacro apollo:get-section    (name sections) (list 'assq name sections))

(defmacro apollo:sections-at    (name sections) 
  (list 'memq (list 'assq name sections) sections))

(defmacro apollo:sections-after (section sections) 
  (list 'cdr (list 'memq section sections)))

(defmacro apollo:new-section (sym b p type) 
  (list 'cons sym (list 'cons (list 'set-marker '(make-marker) p b) type)))

(defmacro apollo:reset-buffer-sections ()
  '(setq apollo:section-reuse-list 
    (nconc apollo:section-reuse-list apollo:buffer-sections)
    apollo:buffer-sections nil))

(defmacro apollo:reuse-section (reuse sym b p type)
  (let ((old (apollo:gensym)))
    (list 'let
          (list (list old reuse))
          (list 'rplaca old sym)
          (list 'set-marker (macroexpand (list 'apollo:section-marker old)) p b)
          (list 'rplacd (list 'cdr old) type))))

(defmacro apollo:add-section (sym b p type)
  (let ((l (apollo:gensym))
        (r (apollo:gensym)))
    (list 'if 'apollo:section-reuse-list
          (list 'let
                (list (list r 'apollo:section-reuse-list))
                (list 'setq 'apollo:section-reuse-list 
                      (list 'cdr 'apollo:section-reuse-list))
                (list 'rplacd r 'apollo:buffer-sections)
                (list 'setq 'apollo:buffer-sections r)
                (macroexpand 
                 (list 'apollo:reuse-section (list 'car r) sym b p type)))
          (macroexpand (list 'push 
                             (macroexpand 
                              (list 'apollo:new-section sym b p type)) 
                             'apollo:buffer-sections)))))


(defun apollo:stringify-section (section)
  "Returns string for display of a SECTION."
  (let ((status nil))
    (with-buffer-set (if (apollo:common-lisp-buffer-p) 
                         (with-buffer-set (apollo:lisp-buffer) apollo:original) 
                         apollo:emacs-original-buffer)
      (let ((sections apollo:buffer-sections))
        (while (and (setq sections 
                          (apollo:sections-at 
                           (apollo:section-name section) 
                           apollo:buffer-sections))
                    (not (equal (apollo:section-type section) 
                                (apollo:section-type (car sections))))))
        (if sections 
            (setq status 
                  (if (memq (car sections) apollo:local-info)
                      "    COMPILED" "     EVALUATED")))))
    (let* ((name  (symbol-name (apollo:section-name section)))
           (lname (length name))
           (def-col (max (+ 2 lname) 40))
           (name-space (make-string (max 1 (- def-col lname)) ?\ ))
           (def   (apollo:section-type section))
           (ldef  (length  def))
           (buf-col (+ def-col (max (+ 2 ldef) 16)))
           (def-space 
               (make-string 
                (max 1 (- buf-col (+ lname (length name-space) ldef))) 
                ?\ ))
           (buf   (buffer-name (apollo:section-buffer section))))
      (concat name name-space def def-space buf status"\n"))))


(defun apollo:sectionize-buffer (b clisp-p)
  "Create a list of section markers for find-source-code for current-buffer.
       BUFFER is buffer object for current buffer.
       CLISP-P is non-nil if this is a common lisp buffer.
       It creates an assq list for a buffer of definition
       name and a marker for that definition.  This will be
       used by many routines to find definitions in the buffer."
  (let ((done nil))
    (unwind-protect
         (progn
           (apollo:reset-buffer-sections)
           (goto-char (point-min))
           (let ((p-ptr 
                  (save-excursion 
                    (if (and clisp-p (re-search-forward "^(In-Package" nil t))
                        (point) (point-max))))
                 (p (if clisp-p (apollo:find-package)))
                 start sym type)
             (if clisp-p nil (setq clisp-p 'ELISP))
             (while (re-search-forward apollo:any-def nil t)
               (setq start (apollo:match-def-start) 
                     type (apollo:match-def-type) 
                     sym (apollo:match-def-name type))
               (if (> (point) p-ptr)
                   (setq p (apollo:find-package)
                         p-ptr (save-excursion 
                                 (if (re-search-forward "^(In-Package" nil t)
                                     (point) (point-max)))))
               (setq sym (intern (apollo:packagify sym clisp-p p)))
               (apollo:add-section sym b start type)))
           (setq done t))
      (if (not done) (error "Failed to sectionize %s" b (ding))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Find Source code stuff

(defun apollo:find-def (symbol &optional point)
  "Find definition of in current buffer of string SYMBOL.
       This will find the last-most definition in the buffer
       and return its point.
       If the buffer is not sectionized, this will run
       sectionize-buffer.
       If this fails, it will search backwards from
       the end of the file.  The former method is much faster
       since no search is required.  If point is supplied, search
       will be backwards from POINT instead of the end of buffer."
  (if (symbolp symbol) (setq symbol (symbol-name symbol)))
  (let ((clisp (apollo:common-lisp-buffer-p))
        (result nil))
    (if (not apollo:buffer-sections) (sectionize-buffer))
    (if apollo:buffer-sections
        (let ((s apollo:buffer-sections)
              (symbol (intern (if clisp (upcase symbol) symbol))))
          (while s
            (let ((d (apollo:get-section symbol s)))
              (if (and d (or (not point) (<= (apollo:section-marker d) point)))
                  (setq result d s nil)
                  (setq s (apollo:sections-after d s)))))))
    result))


(defun apollo:find-source-code (symbol select-file-p &optional no-query)
  "Find source code for symbol known to Lisp.
       SYMBOL is symbol to find source to.
       SELECT-FILE-P is non-nil if found definition is to be displayed."
  (let ((result nil)
        (b (buffer-name (current-buffer))))
    (setq apollo:find-source-code-symbol (apollo:packagify symbol))
    (if (not no-query)
        (if (apollo:common-lisp-buffer-p)  ; if this common lisp file
            ;; ask lisp where the definition is
            (let ((file 
                   (condition-case foo
                       (evaluate-common-lisp 
                         nil 
                         (concat 
                           "(LET ((FILE (APOLLO::GET-SOURCE-FILE '" 
                           (if (eq (aref apollo:find-source-code-symbol 0) ?\()
                               (concat "\(:METHOD " (substring apollo:find-source-code-symbol 1))
                               apollo:find-source-code-symbol)
                           ")))\n"
                           "  (AND FILE (NAMESTRING FILE)))\n"))
                       (error nil)
                       )))
              (if (and file (not (eq file 'NIL)))
                  (let* ((file 
                          (concat 
                           (substring file 0 (- (length file) 5))
                           ".lisp"))
                         (buff (find-file-noselect file)))
                    (if (not buff)
                        (message (format "File Not Found: %s" file (ding)))
                        (setq result buff)))))
            (if (fboundp (intern symbol));; Try to win on autoload functions
                (let ((f (symbol-function (intern symbol))))
                  (if (and (consp f) (eq (car f) 'autoload))
                      (apollo:dolist (p load-path)
                        (setq p (concat p (cadr f) ".el"))
                        (if (file-exists-p p) 
                            (apollo:return (find-file (setq result p))))))))))
    (message "Searching through Buffers for %s" apollo:find-source-code-symbol)
    (let ((buffers (buffer-list)))
      (if result (setq buffers (cons result (delq result buffers))))
      (apollo:buffer-definitions 
        'ignore 'find-source-code buffers '(edit) (not 'query)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Changed lines stuff

(defvar apollo:change-pattern 
  "^\\([0-9]+\\),?\\([0-9]*\\)[acd][0-9]\\([0-9]+\\),?\\([0-9]*\\)")
(defun apollo:cln (arg);; returns list of ranges of changed line numbers
  "Returns a list of changed line numbers in the current buffer.
       The entries in this list may be either single line numbers
       or dotted pairs of line numbers representing a range of changed
       line numbers where the car is the first and the cdr the last
       changed line.
       THIS FUNCTION REQUIRES THE FORMAT OF UNIX 'DIFF' OUTPUT!!
       ARG of 1 uses changes since file was read in.
       ARG of 2 will use changes since file read in or saved.
       ARG of 3 uses changes since definition was compiled or evaluated."
  (message "Finding Changed Line Numbers")
  (let* ((l nil)
         (o (apollo:arg-file arg))
         (c (current-buffer))
         (b (get-buffer-create apollo:modifications-buffer)))
    (list-modifications arg 'NO-DISPLAY)
    (with-buffer b
      (if buffer-read-only (toggle-read-only))
      (goto-char (point-min))
      (while (re-search-forward apollo:change-pattern nil t)
	(let* ((start 
		(car 
		  (read-from-string 
		    (apollo:match 1))))
	       (end-string (apollo:match 2))
	       (end (if (equal end-string "") 
			start 
			(car (read-from-string end-string)))))
	  (push (cons start end) l))))
    (message "Done Finding Changed Line Numbers")
    l))

(defmacro apollo:region-start (region) (list 'car region))

(defmacro apollo:region-end   (region) (list 'cdr region))

(defmacro apollo:make-region  (start end) (list 'cons start end))

(defun apollo:changed-lines-to-regions (lines)
  "Converts a list of lines too a list of regions.
       Takes a list LINES which is list of dotted pairs of
       line numbers and returns a list, in the same order
       of dotted pairs of points delimitting the lines."
  (message "Finding Changed Regions")
  (let ((regions nil))
    (apollo:dolist (l lines);; convert changed lines to changed regions
      (push
        (apollo:make-region
          (progn (goto-char (point-min)) (forward-line (1- (car l))) (point))
          (progn (goto-char (point-min)) (forward-line (1- (cdr l))) (eol)))
        regions))
    (message "Done Finding Changed Regions")
    (nreverse regions)))


(defun apollo:changed-regions-to-sections (regions)
  "Takes a list of REGIONS and returns a list of sections.
       The input list is a list of dotted pairs of points
       delimitting definitions.  This returns a list of
       sections whose definitions cross the region boundries
       in REGIONS."
  (let ((s apollo:buffer-sections)
        (result nil))
    (while (and s regions)
      (let ((r (pop regions)))
        ;; Find first section from end in region
        (let ((r-start (apollo:region-start r))
              (r-end   (apollo:region-end r)))
          (while (and s (> (apollo:section-start (car s)) r-end)) (pop s))
          (if (and s 
                   (> r-start 
                      (progn (goto-char (apollo:section-start (car s))) 
                             (apollo:mark-defun) 
                             apollo:point)))
              nil
              (progn
                (if s (push (pop s) result))
                (while (and s (>= (apollo:section-start (car s)) r-start))
                  (push (pop s) result))
                (if    (and s (< r-start 
                                 (progn 
                                   (goto-char (apollo:section-start (car s)))
                                   (apollo:mark-defun)
                                   apollo:point))) 
                       (push (pop s) result))
                (while (and 
                        s 
                        regions 
                        result 
                        (>= (apollo:region-start (car regions))
                            (apollo:section-start (car result)))) 
                  (pop regions)))))))
    result))


(defun apollo:current-definitions (actions buffer sections clisp-p)
  "Remove sections matching cache of currently evaluated definitions.
       ACTIONS are a list of things to be performed on the definitions
       found.  The elements of actions may be 'list 'eval 'compile 'edit.
       This looks in the currently evaluated definition cache in
       BUFFER for definitions matching those in the list of SECTIONS.
       CLISP-P is non-nil if this is a for common lisp and nil
       if for emacs lisp."
  (message "Seeing if definitions need to be compiled...")
  (with-buffer-set buffer
    (let ((b-sections (if (memq 'compile actions) 
                          apollo:local-info
                          apollo:buffer-sections))
          (changed-sections-p nil))
      (apollo:dolist (s sections)
        (let ((bs (apollo:sections-at (apollo:section-name s) b-sections))
              (b nil))
          (while bs ;; See if definitions has been evaluated already
            (if (equal (apollo:section-type s) (apollo:section-type (car bs)))
                (setq b (car bs) bs nil)
                (setq bs (apollo:sections-at (apollo:section-name s) (cdr bs)))))
          (if b ;; b will have the same name and definition type as s
              (let ((new (with-buffer (apollo:section-buffer s)
                           (goto-char (apollo:section-start s))
                           (apollo:mark-defun)
                           (buffer-substring apollo:mark apollo:point))))
                (goto-char (apollo:section-marker b))
                (apollo:mark-defun)
                (if (string= (buffer-substring apollo:mark apollo:point) new)
                    (setq sections
                          (delq s 
                                (if changed-sections-p 
                                    sections
                                    (copy-sequence sections)))
                          changed-sections-p t))))))))
  (message "Seeing if definitions need to be compiled, Done.")
  sections)


(defun apollo:lcd (arg actions)                    ; List Changed Definition
  "List changed definitions.
       ARG of 0 will use definitions from last changed-definition command.
       ARG of 1 or no arg uses changes since file was read in.
       ARG of 2 will use changes since file read in or saved.
       ARG of 3 uses changes since definition was compiled or evaluated.
       ACTIONS are a list of things to be performed on the definitions
       found.  The elements of actions may be 'list 'eval 'compile 'edit."
  (if (eq arg 0)
      (if apollo:lcd-last-arg apollo:lcd-last-result 
          (error "No previous changed definition command!"))
      (save-excursion
        (message "Looking For Changed Definitions")
        (let ((lines (apollo:cln arg)))
          (if lines
              (let ((clisp-p (apollo:common-lisp-buffer-p)))
                (apollo:sectionize-buffer 
                 (current-buffer) 
                 (apollo:common-lisp-buffer-p))
                (setq
                  apollo:lcd-last-arg    arg
                  apollo:lcd-last-result 
                  (apollo:changed-regions-to-sections 
                   (apollo:changed-lines-to-regions lines)))
                (let ((b (if clisp-p 
                             (with-buffer-set (apollo:lisp-buffer) apollo:original)
                             (get-buffer-create apollo:emacs-original-buffer))))
                  (if (eq arg apollo:since-eval-or-compile) 
                      ;; remove unchanged definitions
                      (setq apollo:lcd-last-result 
                            (apollo:current-definitions 
                             actions b apollo:lcd-last-result clisp-p))))
                (if (and (null apollo:lcd-last-result) 
                         (get-buffer apollo:changed-lines-buffer)) 
                    (kill-buffer apollo:changed-lines-buffer))
                (message "Done Looking For Changed Definitions")
                apollo:lcd-last-result))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Work horses

(defun apollo:buffer-definitions (arg operation buffers actions query)
  "This function is used to perform various actions on changed definitions.
       ARG of 0 will use definitions from last changed-definition command.
       ARG of 1 or no arg uses changes since file was read in.
       ARG of 2 will use changes since file read in or saved.
       ARG of 3 uses changes since definition was compiled or evaluated.
       OPERATION is what this is 'changed-definitions or 'find-source-code.
       BUFFERS is a list of buffers to act on.
       ACTIONS are a list of things to be performed on the definitions
       found.  The elements of actions may be 'list 'eval 'compile 'edit.
       QUERY is 0 if it is desired to query before performing ACTIONs
        or else corresponds to the arg for evaluate-common-lisp."
  (setq apollo:buffer-definitions-type operation)
  (if (and (eq arg 0) (> (length buffers) 1))
      (error "Zero option only valid for single last buffer!")
      (let* ((done nil)
             (clisp-p (apollo:common-lisp-buffer-p))
             (elisp-p (apollo:emacs-lisp-buffer-p)))
        (setq apollo:buffer-definitions nil)
        (while (and buffers (not done))
          (let* ((b (car buffers))
                 (name (buffer-name b)))
            (setq buffers (cdr buffers))
            (if (and name (not (memq (aref name 0) '(?\  ?\*))))
                (with-buffer-set b
                  (cond 
                    ((and (not clisp-p) (not elisp-p)) nil)
                    ((and clisp-p (not (apollo:clisp-buffer-p))) nil)
                    ((and elisp-p (not (apollo:elisp-buffer-p))) nil)
                    ((eq operation 'changed-definition)
                     (message (format "Checking buffer %s" name))
                     (get-buffer-create apollo:changed-lines-buffer)
                     (setq apollo:buffer-definitions 
                           (append (apollo:lcd arg actions) 
                                   apollo:buffer-definitions)))
                    ((eq operation 'find-source-code)
                     (if apollo:verbose (message (format "Searching buffer %s" name)))
                     (let ((def (apollo:find-def apollo:find-source-code-symbol)))
                       (if def
                           (setq 
                            done t
                            apollo:find-source-code-info 
                            (cons (- (apollo:section-start def) 1) 
                                  (cons b buffers))
                            apollo:buffer-definitions 
                            (cons def 'apollo:find-source-code-info)))))
                    (t (error 
                        (format 
                         "Illegal operation to apollo:buffer-definitions: %s"
                         operation))))))))
        (if apollo:buffer-definitions
            (apollo:buffer-definitions-do-actions arg actions query)
            (if (eq operation 'find-source-code)
                (save-excursion
                  (find-tag (downcase (apollo:strip-package apollo:find-source-code-symbol)))
                  (find-file-other-window (buffer-file-name (current-buffer))))
                (progn (ding) (message (format "No definition found")) nil))))))


(defun apollo:buffer-definitions-do-actions (arg actions query)
  "Perform a list of actions on apollo:buffer-definitions.
       ARG of 1 or no arg uses changes since file was read in.
       ARG of 2 will use changes since file read in or saved.
       ARG of 3 uses changes since definition was compiled or evaluated.
       ACTIONS are a list of things to be performed on the definitions
       found.  The elements of actions may be 'list 'eval 'compile 'edit.
       QUERY is 0 if it is desired to query before performing ACTIONs
        or else corresponds to the arg for evaluate-common-lisp."
  (let ((clisp-p (apollo:common-lisp-buffer-p)))
    (if (memq 'list    actions)
        (let 
            ((title 
              (concat 
               (if (eq arg 0) (progn (setq arg apollo:lcd-last-arg) "Reusing "))
               "Changed Definitions since "
               (cond 
                 ((eq arg apollo:since-read)            "read in: ")
                 ((eq arg apollo:since-read-or-save)    "last saved: ")
                 ((eq arg apollo:since-eval-or-compile) "evaluated or compiled: "))
               "\n")))
          (apollo:lines-to-buffer apollo:changed-lines-buffer  
                               title 
                               'refresh
                               apollo:buffer-definitions 
                               (function apollo:stringify-section)
                               (apollo:mode) (apollo:find-package))))
    (if (memq 'compile actions) 
        (save-excursion 
          (apollo:evaluate-sections arg apollo:buffer-definitions 'compile query)))
    (if (memq 'eval    actions) 
        (save-excursion 
          (apollo:evaluate-sections arg apollo:buffer-definitions 'eval    query)))
    (if (memq 'edit    actions) 
        (progn (edit-next-definition) 
               (message (format "Use <C-c .> to find definitions.")))))
  apollo:buffer-definitions)

(defun apollo:fix-sharp-char (string)
  (let* ((sharp-match (string-match "#" string))
         (left-match  (and sharp-match 
                           (eq (aref string (1+ sharp-match)) ?<)
                           (1+ sharp-match)))
         (right-match (and left-match  
                           (let ((r (string-match ">" (substring string left-match))))
                             (and r (+ left-match r))))))
    (if sharp-match
        (if right-match
            (concat (substring string 0 sharp-match)
                    "\\"
                    (substring string sharp-match left-match)
                    "\\"
                    (apollo:escape-string (substring string left-match right-match) apollo:chars-in-sharp<>)
                    "\\"
                    (apollo:fix-sharp-char (substring string right-match)))
            (concat (substring string 0 sharp-match)
                    "\\#\\"
                    (apollo:fix-sharp-char (substring string (1+ sharp-match)))))
        string)))


(defun apollo:defvars-to-setqs ()
  "Replace all defvars in current buffer with setqs.
       Defvar doc strings will be removed also since
       they dont map onto setq."
  (goto-char (point-min))
  (while (re-search-forward "^(\\(defvar\\)" nil t)
    (delete-region (match-beginning 1) (match-end 1))
    (goto-char (match-beginning 1))
    (insert "setq")
    (goto-char (match-beginning 0))
    (mark-sexp 1)
    (let* ((string (buffer-substring (mark) (point)))
           (form (car (read-from-string (apollo:fix-sharp-char string))))
           (l (length form)))
      (if (zerop (logand l 1))
          (progn
            (rplacd (nthcdr (- l 2) form) nil)
            (replace-string string (format "%s" form)))))))

(defun apollo:query-sections (action)
  "Ask to perform ACTION on sections.
       ACTION is 'compile or 'eval or 'quiet-eval.
       Delete from current buffer and apollo:buffer-sections if
       answer is 'n'"
  (if (or (equal (buffer-name (current-buffer)) apollo:emacs-elisp-buffer)
          (equal (buffer-name (current-buffer)) apollo:emacs-clisp-buffer))
      (apollo:dolist (section (reverse apollo:buffer-sections))
        (let ((query (format "%s %s %s? " 
                             (if (eq action 'compile) "Compile" "Evaluate")
                             (apollo:section-type section)
                             (apollo:section-name section))))
        (if (not (y-or-n-p query))
            (progn
              (goto-char (apollo:section-start section))
              (apollo:mark-defun)
              (unwind-protect
                   (setq apollo:buffer-sections (delq section apollo:buffer-sections))
                (delete-region (point) (mark)))))))
      (error "apollo:query-sections not called from correct buffer %s" 
             (current-buffer))))


(defun apollo:evaluate-sections (arg sections action query)
  "Perform an action on all elements of a list of sections.
       ARG of 1 or no arg uses changes since file was read in.
       ARG of 2 will use changes since file read in or saved.
       ARG of 3 uses changes since definition was compiled or evaluated.
       SECTIONS is a list of sections to perform an action on.
       ACTION is to be performed on the definitions found.
       ACTION may be 'eval 'compile or 'quiet-eval.
       QUERY is 0 if it is desired to query before performing ACTIONs
        or else corresponds to the arg for evaluate-common-lisp."
  (let ((c (current-buffer))
        (refresh 'refresh))
    (apollo:dolist (s sections)
      (if (or (not query)
              (y-or-n-p (format "%s %s %s in buffer %s? "
                                (if (eq action 'compile) "Compile" "Evaluate")
                                (apollo:section-type s)
                                (apollo:section-name s)
                                (buffer-name (apollo:section-buffer s)))))
          (with-buffer-set (apollo:section-buffer s)
            (goto-char (apollo:section-marker s))
            (set-mark (point))
            (apollo:evaluate-region 
             'no-hack action refresh (not 'send) (not 'query) arg)
            (setq refresh nil))))
    (if refresh
        (message "No forms to evaluate." (ding))
        (save-excursion
          (apollo:evaluate-region 
           'no-hack action (not 'refresh) 'no-eval (not 'query) arg)))))


(defun apollo:evaluate-region (no-hack action refresh send query since-when)
  "Evaluate or compile current region.
       This function copies marked region to a storage buffer.  If no region
       is marked,  the current definition pointed to is taken.
       NO-HACK is non-nil when it is not desired to use setq instead
       of defvar before evaluating.
       ACTION may be one of 'compile or 'eval 'quiet-eval.
       REFRESH is non-nil if storage buffer is to be cleared before executing.
       SEND is non-nil if storage buffer is to be evaluated or compiled now.
       SEND is 'no-eval if no forms are being supplied.
       QUERY is 0 if it is desired to query before performing ACTIONs
        or else corresponds to the arg for evaluate-common-lisp
       SINCE-WHEN of 1 or no arg uses changes since file was read in.
                     2 will use changes since file read in or saved.
                     3 uses changes since definition was compiled or evaluated."
  (if (apollo:lisp-buffer-p)
      (let* ((source-file-name buffer-file-name)
             (clisp-p (apollo:common-lisp-buffer-p))
             (buffer 
              (if clisp-p apollo:emacs-clisp-buffer apollo:emacs-elisp-buffer)))
        (if refresh
            (progn
              (with-buffer-set (get-buffer-create apollo:emacs-clisp-buffer) 
                (lisp-mode)
                (erase-buffer))
              (with-buffer-set (get-buffer-create apollo:emacs-elisp-buffer)
                (emacs-lisp-mode)
                (erase-buffer))))
        (if (not (eq send 'no-eval))
            (save-excursion
              (if (= (mark) (point)) (apollo:mark-defun))
              (if (< (mark) (point)) (exchange-point-and-mark))
              ;; used-p sets apollo:used-packages as side effect!
              (let* ((p (and clisp-p  (apollo:find-package 'used-p))) 
                     (u apollo:used-packages)
                     (i (and clisp-p  (boundp 'apollo:ibase)     apollo:ibase))
                     (b (and clisp-p  (boundp 'apollo:base)      apollo:base))
                     (r (and clisp-p  (boundp 'apollo:readtable) apollo:readtable)))
                (with-buffer-set buffer
                  (setq apollo:ibase i 
                        apollo:base  b 
                        apollo:readtable r 
                        apollo:superpackage u)
                  (goto-char (point-max))
                  (if p 
                      (insert 
                       (format "\n(In-Package '%s %s)"
                               p (if u (concat ":USE '" (format "%s" u)) ""))))
                  (insert "\n")
                  (if clisp-p
                      (insert 
                        "#+LUCID (setq user::*source-pathname* "
                        "\"" (or source-file-name "nil")  "\")\n")))
                ;; Following loop is because append-to-buffer
                ;; breaks for large files
                (let ((s (min (mark) (point))) 
                      (e (max (mark) (point))))
                  (while (<= s e) 
                    (append-to-buffer buffer s (min e (apollo:incf s 10000))))))))
        (if send (apollo:eval-code-buffers query no-hack action since-when)))))


(defun apollo:eval-code-buffers (query no-hack action since-when)
  "This is used to set up the cache buffers of things.
       to compile or evaluate and do so accordingly.
       NO-HACK when nil means change defvar's to setq's
       before evaluating.
       ACTION may be one of 'compile or 'eval or 'quiet-eval.
       SINCE-WHEN of 1 or no arg uses changes since file was read in.
                     2 will use changes since file read in or saved.
                     3 uses changes since definition was compiled or evaluated."
  (if (not (memq action '(eval compile quiet-eval)))
      (error "Action must be 'compile, 'eval or 'quiet-eval"))
  (let* ((clisp-p (apollo:common-lisp-buffer-p))
         (b (if clisp-p apollo:emacs-clisp-buffer apollo:emacs-elisp-buffer))
         (compiled-defs-buffer 
          (get-buffer-create 
           (if clisp-p 
               (with-buffer-set (apollo:lisp-buffer) apollo:original) 
               apollo:emacs-original-buffer)))
         (apollo:force-common-lisp-p (if clisp-p 'force)))
    (with-buffer-set b
      (apollo:find-unbalanced-parentheses)
      (if (> (point-max) (point-min))
          (progn
            (if (not no-hack) (apollo:defvars-to-setqs))
            (goto-char (point-min))
            (apollo:sectionize-buffer (current-buffer) clisp-p)
            (if (eq query 0) (apollo:query-sections action))
            (if (or (eq query 0) (eq query nil))
                (setq query (if (eq action 'compile) 
                                apollo:evaluate-common-lisp-as-typin
                                apollo:evaluate-common-lisp-into-minibuffer)))
            (apollo:update-current-definitions 
             action apollo:buffer-sections compiled-defs-buffer (if clisp-p t nil))
            (if (eq (point-min) (point-max))
                (message "No changed definitions to compile.")
                (if clisp-p
                    (apollo:lisp-send-buffer  apollo:emacs-clisp-buffer action query)
                    (apollo:emacs-send-buffer apollo:emacs-elisp-buffer action query)
                    )))))))


(defun apollo:emacs-send-buffer (b action &optional query)
  "Send an indicated region to emacs lisp.
       BUFFER is a buffer whose contents are sent to lisp.
       ACTION may be one of 'compile or 'eval or 'quiet-eval."
  (with-buffer-set b
    (if (eq action 'compile)
        (let ((version-control 'never)
              (make-backup-files nil)
              (filename 
                (format "%s/%s.el" apollo:tmp-dir (make-temp-name "emacs")))
              (filenamec nil))
          (setq filenamec (concat filename "c"))
          (with-buffer-name-preserved (current-buffer) (write-file filename))
          (byte-compile-file filename)
          (load filenamec)
          (if (file-exists-p filename) (delete-file filename))
          (if (file-exists-p filenamec) (delete-file filenamec)))
        (if (> (length apollo:buffer-sections) 1)
            (with-output-to-temp-buffer apollo:emacs-eval-buffer
              (eval-region (point-min) (point-max) standard-output))
            (if (and (eq action 'quiet-eval) (null apollo:buffer-sections))
                (message "No forms to evaluate" (ding))
                (eval-region (point-min) (point-max) standard-output))))))


(defun apollo:lisp-send-buffer (b action &optional arg)
  "Send an indicated region to common lisp.
       BUFFER is a buffer whose contents are sent to lisp.
       ACTION may be one of 'compile or 'eval or 'quiet-eval.
       ARG is an argument for evaluate-common-lisp.
       Lisp will be displayed in the other buffer.
       It will also print its messages in the lisp window.
       This will interrupt lisp to do the compile or eval
       and then tell it to continue."
  (with-buffer-set b
    (if (eq action 'quiet-eval)
        (if (null apollo:buffer-sections)
            (message "No forms to evaluate" (ding))
            (progn
              (goto-char (apollo:section-start (car apollo:buffer-sections)))
              (apollo:mark-defun)
              (evaluate-common-lisp arg (buffer-substring (mark) (point)))))
        (let ((version-control 'never)
              (make-backup-files nil)
              (cmd 
               (concat 
                "(UNWIND-PROTECT\n"
                (format
                 "       (LOAD %s :PRINT T)\n"
                 (if (eq action 'compile) "(COMPILE-FILE \"%s\")" "\"%s\""))
                "    (DELETE-FILE \"%s\")"
                (if (eq action 'compile) "\n    (DELETE-FILE \"%s\")")
                ")"))
              (apollo:compile (eq action 'compile))
              (filename-lisp nil)
              (filename-lbin nil))
          (apollo:dotimes (n 100)
            (let ((filename
                   (format "%s/emlisp%d%d" apollo:tmp-dir (process-id (apollo:lisp-process)) n)))
              (if (not (file-exists-p (setq filename-lisp (concat filename ".lisp"))))
                  (apollo:return (setq filename-lbin (concat filename ".lbin"))))))
          (if (file-exists-p filename-lbin) (delete-file filename-lbin))
          (with-buffer-name-preserved (current-buffer) (write-file filename-lisp))
          (evaluate-common-lisp arg (format cmd filename-lisp filename-lisp filename-lbin))))))

(defun apollo:replace-current-definition (buffer old-section new-definition)
  "Replace the old definition in cache of current-definitions with a newer version.
       BUFFER is the cache buffer of current definitions.
       OLD-SECTION is the section of the old definition
       NEW-DEFINITION is the new definition to put at this section."
  (with-buffer buffer
    (goto-char (apollo:section-marker old-section))
    (if (eq action 'compile)            ; apollo:local-info is list of compiled sections
        (push-new old-section apollo:local-info)
        (setq apollo:local-info (delq old-section apollo:local-info)))
    (apollo:mark-defun)
    (unwind-protect 
         (delete-region (mark) (point))
      (insert new-definition))))


(defun apollo:new-current-definition (buffer new-definition package)
  "Replace the old definition in cache of current-definitions with a newer version.
       BUFFER is the cache buffer of current definitions.
       NEW-DEFINITION is the new definition to put at this section.
       PACKAGE of new definition."
  (with-buffer buffer                   ; if adding new definition
    (goto-char (point-max))
    (if (and package 
             (not (equal (apollo:find-package 'used-p) package)))
        (insert 
          (format 
            "(In-Package '%s%s)\n"
            package 
            (if apollo:used-packages 
                (format " :USE '%s" apollo:used-packages) 
                ""))))
    (insert "\n")
    (let ((m (point)))
      (insert new-definition)
      (goto-char m)
      (if (re-search-forward apollo:any-def nil t)
          (let* ((type (apollo:match-def-type))
                 (pname (apollo:packagify (apollo:match-def-name type) (if clisp-p t 'ELISP) package)))
            (apollo:add-section (intern pname) buffer (match-beginning 0) type)
            (if (eq action 'compile) 
                (push (car apollo:buffer-sections) apollo:local-info)))))))

(defun apollo:update-current-definitions (action sections buffer clisp-p)
  "Update the list of current definitions in cache.
       ACTION may be one of 'compile or 'eval or 'quiet-eval.
       SECTIONS is a list of sections being evaluated or compiled.
       BUFFER is the cache buffer of current definitions.
       CLISP-P is t if this is for common-lisp and nil for emacs lisp
       "
  (let ((b-sections (with-buffer-set buffer apollo:buffer-sections)))
    (apollo:dolist (s sections)
      (let ((b nil)
            (package nil))
        (let ((bs (apollo:sections-at (apollo:section-name s) b-sections)))
          (while bs
            (setq b (car bs))
            (if (equal (apollo:section-type s) (apollo:section-type b))
                (setq bs nil)
                (setq b nil 
                      bs (apollo:sections-at (apollo:section-name s) (cdr bs))))))
        (let ((new (with-buffer (apollo:section-buffer s)
                     (goto-char (apollo:section-start  s))
                     (if clisp-p (setq package (apollo:find-package 'used-p)))
                     (apollo:mark-defun)
                     (buffer-substring (mark) (point)))))
          (if b                         ; if already evaluated
              (apollo:replace-current-definition buffer b new)
              (apollo:new-current-definition buffer new package)))))))


(defun apollo:file-op (operation file)
  "This is used to load or compile lisp files.
       It keeps track of binaries and will try to load the best version of
       the file.  It also offers to edit files being loaded if necessary.
       OPERATION may be 'load, 'compile or 'load-compile.
       File is the name of the FILE to load.  If FILE is not a string,
       you are prompted for the name of the FILE.  If file is a number,
       this will force compilation without querying."
  (let* ((force-p (numberp file))
         (file
          (expand-file-name
            (if (stringp file) 
                file
                (read-file-name 
                  (format "%s File: " operation) nil buffer-file-name 'MUSTMATCH))))
          (f-name (file-name-nondirectory file))
         (load (memq operation '(load load-compile)))
         (compile (memq operation '(compile load-compile)))
         src bin buff type)
    (cond 
      ((string-match apollo:elisp-file-pattern file)
       (setq src file bin (concat file "c") type 'emacs))
      ((string-match apollo:clisp-file-pattern file)
       (setq src file 
             bin (concat (substring file 0 (match-beginning 0)) ".lbin")
             type 'lisp))
      ((string-match apollo:elisp-bin-pattern file)
       (setq src (substring file 0 (- (length file) 1)) bin file type 'emacs))
      ((string-match apollo:clisp-bin-pattern file)
       (setq src (concat (substring file 0 (match-beginning 0)) ".lisp")
             bin file 
             type 'lisp))
      (t (error "Unknown file type: %s" file)))
    (setq f-name (file-name-nondirectory src))
    (if (and (file-exists-p src)
             (not (get-file-buffer src))
             (y-or-n-p (format "No Buffer for %s, Read into buffer before %s? " 
                               f-name operation)))
        (find-file src))
    (if (and (setq buff (get-file-buffer src))
             (with-buffer-set buff (buffer-modified-p))
             (y-or-n-p (format "Buffer %s is modified.  Save before %s? "
                               (buffer-name buff) operation)))
        (with-buffer-set buff (save-buffer)))
    (if (and (eq operation 'load) (file-exists-p src) 
             (or (not (file-exists-p bin)) (file-newer-than-file-p src bin)))
        (setq compile
              (or force-p
                  (y-or-n-p 
                    (format 
                      "Source is newer than object for %s. Compile before loading? "
                      f-name)))))
    (if compile
        (if (file-exists-p src)
            (progn
              (setq file bin)
              (if (and (not force-p) (file-exists-p bin) (file-newer-than-file-p bin src))
                  (progn
                    (message "%s already is up to date" (file-name-nondirectory file))
                    (setq compile nil))
                  (if (eq type 'emacs)
                      (progn (message "Compiling %s..." src)

                             (byte-compile-file src)
                             (message "Compiling %s Done" src))
                      ;; Send only 1 command to lucid,load-compile comes later.
                      (progn
                        (if (not load) 
                            (evaluate-common-lisp apollo:evaluate-common-lisp-as-typin
                              (format "(lisp::compile-file \"%s\")" src))
                            (setq file src))))))
            (error "Cannot find file to compile: %s." src)))
    (if load
        (if (file-exists-p file)
            (progn
              (if (eq type 'emacs)
                  (let ((f (if (and (not (equal file bin)) 
                                    (file-newer-than-file-p src bin)) 
                               src 
                               bin)))
                    (message "Loading %s..." f) 
                    (load f) 
                    (message "Loading %s Done" f))
                  (if compile
                      (evaluate-common-lisp
                        apollo:evaluate-common-lisp-as-typin
                       (format "(lisp::load (lisp::compile-file \"%s\"))" src))
                      (let ((f (if (and (not (equal file bin))
                                        (file-newer-than-file-p src bin)) 
                                   src 
                                   bin)))
                        (evaluate-common-lisp
                          apollo:evaluate-common-lisp-as-typin
                          (format "(lisp::load \"%s\")" f)))))
              (let ((buff (get-file-buffer file)))
                (if buff (with-buffer buff (apollo:reset-original nil)))))
            (error "Cannot find file to load %s." file)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Inferior lisp control functions
(defun apollo:ask-LUCID-fix-input-command (str)
  (setq str (apollo:escape-string str apollo:unexported-package ":" 2))
  (if apollo:ask-common-lisp-display-p
      (concat "(apollo:recoverable-eval " str ")\n")
      (concat str "\n")
      )) ; Always a valid string

(defun apollo:ask-LUCID-recover-from-error (proc)
  (if apollo:ask-common-lisp-display-p
      (with-process-filter (proc (function apollo:quiet-filter))
        (let ((apollo:string ""))
          (process-send-string proc "\n(apollo::recover-from-eval)\n")
          (while (not (string-match inferior-lisp-prompt apollo:string))
            (accept-process-output proc))))))

(defun apollo:ask-LUCID-resume-command (proc)
  (process-send-string proc "\n:C\n"))

(defvar apollo:Interrupt-Message 
  (concat "^" inferior-lisp-prompt "*Resume Interrupted Instruction\n"))

(defun apollo:ask-LUCID-wait-for-resume (proc)
  (while (not (or 
                  (string-match "^Resume Interrupted Instruction\n" apollo:string)
                  (string-match apollo:Interrupt-Message apollo:string)
                  (string-match "^Will resume normal return\n"      apollo:string)
                  (string-match "^Back to Lisp Top Level\n"         apollo:string)
                  (string-match "^:GNUEMACS-ERROR-RECOVERED\n"      apollo:string)
                  (string-match "^:C\n"                             apollo:string)
                  (string-match "^:DONE\n"                          apollo:string)
                  (string-match "^Return from Break\n"              apollo:string)
                  ))
    (accept-process-output proc)))

(defun apollo:fix-LUCID-output (string)
  (if (string-match inferior-lisp-prompt string)
      (let ((last (if (eq (aref string (1- (length string))) ?\n) "" "\n")) ;; make sure line has return
            ;; extract the prompt
            (prompt (substring string (match-beginning 0) (match-end 0)))) 
        ;; strip the prompt form the string
        (setq string (substring string 0 (match-beginning 0)) 
              ; figure out one prompt up
              apollo:next-lisp-prompt (if (string-match "^-?> " prompt) "> " (substring prompt 2)))
        ;; glue the return and proper prompt onto string
        (concat string last apollo:next-lisp-prompt))
      string))

(defun apollo:lisp-process (&optional nil-if-not-found)
  "This returns the current lisp process or starts a lisp if none exists.
       If this is called by an emacs routine that needs common lisp, the
       routine will need to be recalled!  This does not wait for the lisp
       to be ready to query yet!
       Optional NIL-IF-NOT-FOUND when non-Nil will cause an error to be
       signalled if no lisp process exists."
  (if (and apollo:lisp-name (get-process apollo:lisp-name))
      (get-process apollo:lisp-name)
      (if nil-if-not-found
          nil
          (progn (lisp 'start-process)
                 (error "Re-execute command after lisp is started!!")))))


(defun apollo:lisp-server-name (&optional direction)
  (let* ((p (assq apollo:lisp-name apollo:lisp-processes))
         (lsn (format "%s-%s" (car p) (cdr p))))
    (cond ((eq direction 'in)  (concat lsn "-in"))
          ((eq direction 'out) (concat lsn "-out"))
          (t lsn))))

(defun apollo:lisp-buffer (&optional nil-if-not-found)
  "This returns the current lisp buffer or starts a lisp if none exists.
       If this is called by an emacs routine that needs common lisp, the
       routine will need to be recalled!  This does not wait for the lisp
       to be ready to query yet!
       Optional NIL-IF-NOT-FOUND when non-Nil will cause an error to be
       signalled if no lisp process exists."
  (let ((p (apollo:lisp-process nil-if-not-found))) (and p (process-buffer p))))

(defvar apollo:server-display-buffer "*Lisp Transaction Display: APOLLO*")
(defvar apollo:server-journal-buffer "*Lisp Transaction Journal: APOLLO*")
(defvar apollo:server-exit-message   "\\*\\*\\* SERVER EXIT \\*\\*\\*")
(defvar apollo:server-error-message   "\\*\\*\\* SERVER ERROR \\*\\*\\*")
(defvar apollo:server-filter-buffer  apollo:server-display-buffer)


(defun apollo:lisp-server-interrupt ()
  "Interrupt the server for the inferior lisp process.
This is useful when you wish to stop a compile or other long process
in the lisp server window."
  (interactive)
  (if apollo:gnuemacs-client-p
      (apollo:client-send-string 
	"(apollo::interrupt-process-named \"LISP-SERVER\")\n"
	(apollo:lisp-process)
	(not 'display))))

(defun apollo:client-send-string (string proc display)
  (if display (setq apollo:ask-common-lisp-display-p t))
  (let ((string (concat string
                        ;; (apollo:escape-string string apollo:unexported-package ":" 2)
                        "\n"))
        (pname  (apollo:lisp-server-name))
        (b (if display apollo:server-display-buffer apollo:server-journal-buffer))
        input-start input-end output-start)
    (let ((new-buffer-p (not (get-buffer b))))
      (with-buffer-set (get-buffer-create b)
        (if new-buffer-p (progn (toggle-read-only) (common-lisp-mode)))
        (if display (set-window-start (display-buffer (current-buffer) 'other-window) (point-max)))
        (with-read-only nil
          (goto-char (point-max))
          (insert "*** Sending to server: " (process-name proc) " ***\n> ")
          (setq input-start (point))
          (insert string)
          (backward-delete-char-untabify 1 nil)
          (setq input-end (point))
          (insert "*** Response From Server ***\n")
          (setq output-start (point)))))
    (if (or (and apollo:ask-common-lisp-display-p (not display))
            (eq apollo:server-filter-buffer apollo:server-journal-buffer))
        ;; client is busy displaying, so fork off temporary client
        (let ((program (file-name-nondirectory (nth 5 (process-command proc))))
              (server-arg "-client")
              (result nil))
          (message "Forking off Lisp Client.")
          (with-buffer-set b
            (goto-char (point-max))
            (with-read-only nil
              (call-process-region input-start input-end program nil b nil server-arg (concat pname "+"))
              (setq result (buffer-substring output-start (point-max)))
              (insert apollo:server-exit-message "\n")))
          (message "Lisp Done.")
          result)
        (let* ((apollo:server-filter-buffer b)
               (pname-in  (apollo:lisp-server-name 'in))
               (pname-out (apollo:lisp-server-name 'out))
               (pexe      (file-name-nondirectory
                            (nth 5 (process-command (apollo:lisp-process)))))
               (pfile     (concat "/tmp/" pexe ".o." pname))   
               (process-connection-type nil)
               (p-in      (or (get-process pname-in)
                              (start-process pname-in nil pexe "-client" pname)))
               (p-out     (or (get-process pname-out)
                              (while (not (file-exists-p pfile)))
                              (start-process pname-out  nil "cat" "-u" pfile)))
               (filter    (if display 'apollo:gnuemacs-client-filter 'apollo:gnuemacs-quiet-filter)))
          (set-process-filter p-out filter)
          (message (if display "Dispatching To Lisp..." "Waiting For Lisp..."))
          (setq apollo:string "")
          (with-buffer-set b 
            (process-send-region p-in input-start input-end)
            (process-send-string p-in "\n:GNUEMACS_DONE_TOKEN\n")
            (if (not display) 
                (let ((server-done nil)) 
                  (while (not server-done) (accept-process-output p-out))
                  (message "Lisp Done.")
                  apollo:string)))))))


(defun apollo:gnuemacs-quiet-filter (process string)
  (setq apollo:string (concat apollo:string string))
  (if (or (string-match ":GNUEMACS_DONE_TOKEN" apollo:string)
          (string-match apollo:server-exit-message apollo:string)
          (string-match apollo:server-error-message apollo:string))
      (with-buffer-set apollo:server-filter-buffer
        (setq apollo:string (substring apollo:string 0 (match-beginning 0))
              server-done t)
        (with-read-only nil
          (insert apollo:string "\n*** SERVER EXIT ***\n")))))

(defun apollo:gnuemacs-client-filter (process string)
  (let ((ostring apollo:string))
    (setq apollo:string (concat apollo:string string))
    (with-buffer-set (get-buffer-create apollo:server-filter-buffer)
      (goto-char (point-max))
      (if (or (string-match ":GNUEMACS_DONE_TOKEN" apollo:string)
              (string-match apollo:server-exit-message apollo:string)
              (string-match apollo:server-error-message apollo:string))
          (let* ((match-in-string (- (match-beginning 0) (length ostring)))
                 (string (if (> match-in-string 0) (substring string 0 match-in-string) "")))
            (if (< match-in-string 0) 
                (with-read-only nil
                  (delete-region (+ (point-max) match-in-string) (point-max))))
            (setq apollo:string
                  (if (>= (match-end 0) (length apollo:string))
                      ""
                      (substring apollo:string (match-end 0))))
            (with-read-only nil (insert string "\n*** SERVER EXIT ***\n"))
            (if apollo:ask-common-lisp-display-p
                (progn
                  (if apollo:ask-common-lisp-queue
                      (apollo:client-send-string (pop apollo:ask-common-lisp-queue) process t)
                      (setq apollo:ask-common-lisp-display-p nil)))
                (setq server-done t)))
          (with-read-only nil (insert string)))
      (let ((w (get-buffer-window (current-buffer))))
        (if w (set-window-point w (point)))))))
  
(defun apollo:create-gnuemacs-client ()
  (sit-for 1) ;; wait for process to start
  (with-buffer-set (apollo:lisp-buffer)
    (let ((server-name (apollo:lisp-server-name))
	  (server-format-string "(apollo::create-lisp-servers \"%s\" \"%s\")\n"))
      (process-send-string 
        (apollo:lisp-process)
	(format server-format-string server-name (concat server-name "+")))
      (goto-char (point-max))
      (insert ";;; Starting Lisp Server: " server-name "\n")
      (set-marker (process-mark (apollo:lisp-process)) (point))))
  (reset-lisp-filter nil))

(defun apollo:ask-common-lisp-interrupt (proc)
  (funcall apollo:lisp-interrupt proc))

(defun apollo:ask-common-lisp-wait-for-prompt (proc)
  (while (not (string-match inferior-lisp-prompt apollo:string))
    (accept-process-output proc)))


(defun apollo:ask-common-lisp-make-sure-packages-exist (process packages)
  (let ((known-packages (with-buffer-set (apollo:lisp-buffer) apollo:lisp-packages)))
    (apollo:dolist (p packages)
      (if (memq p known-packages)
          (setq packages (delq p packages))))
    (if packages
        (with-process-filter (process (function apollo:quiet-filter))
          (let ((apollo:string ""))
            (process-send-string
              process
              (format "(MAPCAR #'(LAMBDA (P)
                                   (LET ((*PACKAGE* *PACKAGE*))
                                      (In-Package P)))
                               '%s)\n"
                      packages))
            (apollo:ask-common-lisp-wait-for-prompt process))))
    (with-buffer-set (apollo:lisp-buffer)
      (setq apollo:lisp-packages (append packages apollo:lisp-packages)))))

(defun apollo:quiet-filter (proc string)
  (setq apollo:string (concat apollo:string string)))

(defun apollo:lisp-interrupt (process)
  (with-process-filter (process (function apollo:quiet-filter))
    (let ((apollo:string ""))
      (apollo:ask-common-lisp-interrupt process)
      (apollo:ask-common-lisp-wait-for-prompt process)
      (if apollo:ask-common-lisp-display-p
          (if (or (string-match "^>>Break: Keyboard Interrupt" apollo:string)
                  (string-match "^>>.*: interrupt fault (UNIX/signal)" 
                                apollo:string))
              (let ((string (substring apollo:string 0 (match-beginning 0))))
                (apollo:ask-common-lisp-display-results process string)
                (setq apollo:ask-common-lisp-temp 
                      (concat apollo:ask-common-lisp-temp string))))))))

(defun apollo:ask-common-lisp-check-command (string)
  "Check that parenthesis are balanced in string."
  (with-buffer-bury (get-buffer-create apollo:hack-buffer)
    (insert (concat "(" string ")"))
    (lisp-mode)
    (apollo:find-unbalanced-parentheses))
  (let ((packages nil))
    (while (string-match apollo:package&name string)
      (let ((package (apollo:match 1 string)))
        (if (> (length package) 0) 
            (push-new (intern (upcase package)) packages))
        (setq string (substring string (match-end 0)))))
    packages))


(defun apollo:ask-common-lisp-setup-display (proc str)
  (with-buffer-set (process-buffer proc)
    (let ((w (display-buffer (current-buffer)))
          (mb nil) (me nil) (p nil))
      (save-excursion 
        (goto-char (point-max)) 
        (re-search-backward inferior-lisp-prompt nil t)
        (setq mb (match-beginning 0) me (match-end 0))
        (goto-char me)
        (insert str)
        (setq p (point)))
      (set-window-start w mb)
      (move-marker last-input-start me)
      (move-marker last-input-end p)
      (move-marker (process-mark proc) p)
      (set-window-point w p))))

(defun apollo:ask-common-lisp-display-results (proc string)
  (if (> (length string) 0)
      (with-buffer-set (process-buffer proc)
        (let ((p nil))
          (save-excursion
            (goto-char (process-mark proc))
            (insert string)
            (setq p (point)))
          (move-marker (process-mark proc) p)
          (let ((w (get-buffer-window (process-buffer proc))))
            (if w (set-window-point w p)))))))

(defun apollo:ask-common-lisp-resume-old-lisp-process (proc)
  (let ((apollo:string ""))
    (with-process-filter (proc (function apollo:quiet-filter))
      (apollo:ask-LUCID-resume-command proc)
      (apollo:ask-LUCID-wait-for-resume proc))))


(defun apollo:noisy-filter (proc string)
  (let ((prompt-p (string-match inferior-lisp-prompt string))
        (ok nil))
    (unwind-protect
         (progn
           (if prompt-p
               (setq string (apollo:fix-LUCID-output string)
                     apollo:ask-common-lisp-result
                     (concat apollo:ask-common-lisp-temp string)
                     apollo:ask-common-lisp-temp "")
               (setq apollo:ask-common-lisp-temp 
                     (concat apollo:ask-common-lisp-temp string)))
           (if apollo:ask-common-lisp-display-p 
               (apollo:ask-common-lisp-display-results proc string)))
      (if (and apollo:ask-common-lisp-display-p prompt-p)
          (let ((cmd nil))
            (apollo:ask-LUCID-recover-from-error proc)
            (if (setq cmd (pop apollo:ask-common-lisp-queue))
                (unwind-protect
                     (progn
                       (message "Unqueing Command")
                       (apollo:ask-common-lisp-do-command proc cmd)
                       (accept-process-output proc)
                       (setq ok t))
                  (if (not ok)
                      (progn
                        (apollo:ask-LUCID-recover-from-error proc)
                        (apollo:ask-common-lisp-resume-old-lisp-process proc)
                        (setq apollo:ask-common-lisp-display-p nil)
                        (set-process-filter proc apollo:initial-lisp-process-filter)
                        (message "Lisp Command Aborted"))))
                (unwind-protect
                     (apollo:ask-common-lisp-resume-old-lisp-process proc)
                  (progn
                    (setq apollo:ask-common-lisp-display-p nil)
                    (set-process-filter proc apollo:initial-lisp-process-filter)
                    (message "Lisp Command Done")
                    ))))))))

(defun apollo:ask-common-lisp-wait-for-old-to-finish (str)
   (message "Lisp process is still executing.  Queuing command.")
   (setq apollo:ask-common-lisp-queue 
         (nconc apollo:ask-common-lisp-queue (list str))))

(defun apollo:ask-common-lisp-do-command (process str)
  (apollo:ask-common-lisp-make-sure-packages-exist 
   process (apollo:ask-common-lisp-check-command str))
  (if apollo:ask-common-lisp-display-p 
      (apollo:ask-common-lisp-setup-display process str))
  (set-process-filter process 'apollo:noisy-filter) ; set the filter
  (setq str (apollo:ask-LUCID-fix-input-command str))
  (process-send-string process str)     ; go for it
  (setq apollo:last-ask-common-lisp str)
  ) ; save string for inspection


(defun apollo:ask-common-lisp-wait-for-result ()
  (let (error result)
    (while (not apollo:ask-common-lisp-result) (accept-process-output process))
    (setq apollo:ask-common-lisp-result
          (apollo:fix-sharp-char apollo:ask-common-lisp-result))
    (setq result (condition-case foo
                     (read-from-string apollo:ask-common-lisp-result)
                   (error (message "apollo:ask-common-lisp-wait-for-result: %s" foo (ding)) nil)))
    (car result)))

(defun apollo:ask-common-lisp-gnuemacs-client (str &optional display allow-error)
  (setq apollo:ask-common-lisp-result
        (apollo:client-send-string str (apollo:lisp-process) display))
  (if (not display)
      (let (error result)
        (setq apollo:ask-common-lisp-result
              (apollo:fix-sharp-char apollo:ask-common-lisp-result))
        (setq result (condition-case foo
                         (read-from-string apollo:ask-common-lisp-result)
                       (error (message "apollo:ask-common-lisp-gnuemacs-client: %s" foo (ding)) nil)))
        (car result))))


(defun apollo:ask-common-lisp (str &optional display allow-error)
  "STR is a common lisp command to send to lisp.
       This will interrupt lisp to get this done but won't
       display anything in the lisp buffer unless DISPLAY is non-nil.
       DISPLAY is non-nil if to display in lisp process window.
       ALLOW-ERROR is non-nil if its ok to call error.
       WARNING: This functions and its called are LUCID specific and should be
       checked for other other common lisps."
  (if (and display apollo:ask-common-lisp-display-p)
      (apollo:ask-common-lisp-wait-for-old-to-finish str)
      (if (eq apollo:gnuemacs-client-p t)
          (apollo:ask-common-lisp-gnuemacs-client str display allow-error)
          (let* ((process (apollo:lisp-process))
                 (old-filter (process-filter process))
                 old-display-p old-result old-temp result)
            (unwind-protect
                 (progn
                   (message "Waiting for lisp")
                   (apollo:lisp-interrupt process)
                   (setq old-display-p apollo:ask-common-lisp-display-p 
                         old-result apollo:ask-common-lisp-result 
                         old-temp apollo:ask-common-lisp-temp)
                   (setq apollo:ask-common-lisp-result nil 
                         apollo:ask-common-lisp-temp "" 
                         apollo:ask-common-lisp-display-p display)
                   (apollo:ask-common-lisp-do-command process str))
              (if (not apollo:ask-common-lisp-display-p)
                  (unwind-protect 
                       (setq result (apollo:ask-common-lisp-wait-for-result))
                    (progn
                      (setq apollo:ask-common-lisp-display-p old-display-p 
                            apollo:ask-common-lisp-result old-result 
                            apollo:ask-common-lisp-temp old-temp)
                      (set-process-filter process old-filter)
                      (apollo:ask-common-lisp-resume-old-lisp-process process)
                      (message "Lisp Done")))
                  (message "Lisp Command Executing...")))
            result))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Generic shell control

(defun apollo:shell (name file args)
  "Creates a shell in buffer NAME running FILE with arguments ARGS"
  (setq explicit-shell-file-name file)
  (setq argvar 
        (intern (concat "explicit-" (file-name-nondirectory file) "-args")))
  (set argvar (list args))
  (shell)
  (setq apollo:yank-type 'yank-prev-command)
  (define-key shell-mode-map            "\e\t"       'apollo:shell-complete-symbol)
  (define-key shell-mode-map            "\C-a"       'apollo:beginning-of-line)
  (define-key shell-mode-map            "\C-c\C-c"   'apollo:break)
  (define-key shell-mode-map            "\M-p"       'apollo:yank-pop)
  (define-key shell-mode-map            "\M-n"       'apollo:yank-pop-negative)
  (define-key shell-mode-map            "\C-m"       'shell-send-input)
  (rename-buffer name)
  (goto-char (point-max)))


(defun apollo:blink-matching-open ()
  "Move cursor momentarily to the beginning of the sexp before point.
       This version allows apollo:eval-on-closure for lisp expressions.
       It looks bad but its just edited from GNU source"
  (and (> (point) (1+ (point-min)))
       (/= (char-syntax (char-after (- (point) 2))) ?\\ )
       blink-matching-paren
       (let* ((oldpos (point))
              (blinkpos)
              (mismatch)
              (evaled-this-baby nil)
              (sit-for-time 1))
         (save-excursion
           (save-restriction
             (if blink-matching-paren-distance
                 (narrow-to-region 
                  (max (point-min)
                       (- (point) blink-matching-paren-distance))
                  oldpos))
             (condition-case ()
                 (setq blinkpos (scan-sexps oldpos -1))
               (error nil)))
           (and blinkpos (/= (char-syntax (char-after blinkpos))
                             ?\$)
                (setq mismatch
                      (/= last-input-char
                          (logand (lsh (aref (syntax-table)
                                             (char-after blinkpos))
                                       -8)
                                  ?\177))))
           (if mismatch (setq blinkpos nil))
           (if blinkpos
               (progn
                 (setq end-input (point))
                 (goto-char blinkpos)
                 (if (and (eq major-mode 'inferior-lisp-mode)
                          apollo:eval-on-closure)
                     (progn
                       (save-excursion
                         (set-mark 1)
                         (forward-char)
                         (re-search-backward inferior-lisp-prompt nil t)
                         (re-search-forward  apollo:whitespace nil t)
                         (re-search-forward apollo:non-whitespace)
                         (setq start-input (1- (point)))
                         (goto-char blinkpos))
                       (save-excursion
                         (forward-sexp)
                         (backward-sexp)
                         (if (eq (point) start-input)
                             (progn
                               (setq sit-for-time 0)
                               (end-of-buffer)
                               (set-mark (point))
                               (setq evaled-this-baby t)
                               (shell-send-input))))))
                 (if (pos-visible-in-window-p)
                     (sit-for sit-for-time)
                     (goto-char blinkpos)
                     (message
                      "Matches %s"
                      (if (save-excursion

                            (skip-chars-backward " \t")
                            (not (bolp)))
                          (buffer-substring (progn (beginning-of-line) (point))
                                            (1+ blinkpos))
                          (buffer-substring blinkpos
                                            (progn
                                              (forward-char 1)
                                              (skip-chars-forward "\n \t")
                                              (end-of-line)
                                              (point)))))))
               (cond (mismatch
                      (message "Mismatched parentheses"))
                     ((not blink-matching-paren-distance)
                      (message "Unmatched parenthesis")))))
         (if evaled-this-baby (end-of-buffer))
         )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Current thing functions

(defun apollo:key-symbol (mess)
  "Prints message MESSAGE to prompt to use pointed to symbol"
  (apollo:prompt-for mess (apollo:current-thing nil nil)))

(defun apollo:mouse-thing (&optional args n-c sexps-ok)
  "Grab thing near point"
  (interactive)
  (apollo:mouse-move-point args)
  (apollo:current-thing n-c sexps-ok))

(defun apollo:call-current-thing (&rest ignore)
  (apollo:current-thing nil nil))

(defun apollo:current-thing-no-directory (&rest ignore)
  (let ((s (apollo:current-thing nil nil)))
    (if (and s (string-match ".*/\\([^/]*\\)" s))
        (apollo:incf apollo:mark (match-beginning 1))))
  (buffer-substring apollo:mark apollo:point))

(defun apollo:grab-dispatched-object-name ()
  "If point is within #<...> or following whitespace return string \"#<...>\" else nil."
  (interactive)
  (save-excursion
    (if (not (looking-at apollo:non-whitespace))
        (re-search-backward apollo:non-whitespace nil t))
    (let* ((bol (save-excursion (beginning-of-line) (point)))
           (begin-dispatched-object-name
            (save-excursion
              (if (looking-at "#") (forward-char 1))
              (if (looking-at "<") (forward-char 1))
              (and (search-backward "#<" bol t) (point))))
           (end-dispatched-object-name 
            (save-excursion (and (search-backward ">" bol t) (point)))))
      (if (and begin-dispatched-object-name
               (or (not end-dispatched-object-name)
                   (> begin-dispatched-object-name end-dispatched-object-name)))
          (let* ((eol (save-excursion (end-of-line) (point)))
                 (end-dispatched-object-name (save-excursion (and (search-forward ">" eol t) (point)))))
            (if end-dispatched-object-name
                (concat "(APOLLO:OBJECT-NAMED \""
                        (apollo:escape-string
                          (buffer-substring begin-dispatched-object-name end-dispatched-object-name)
                          "\"" "\\")
                        "\")")))))))


(defun apollo:current-thing (&optional n-c sexps-ok)
  "Grab thing near point"
  (interactive)
  (or (and apollo:grabbing-dispatched-names-p (apollo:grab-dispatched-object-name))
      (let* ((name-chars (or n-c apollo:symbol-chars))
             (non-name-chars (concat "[^" (substring name-chars 1))))
        (save-excursion
          (condition-case foo
              (let ((bol (save-excursion (beginning-of-line) (point)))
                    (eol (eol)))
                (if (not sexps-ok)
                    (if (not (looking-at name-chars))
                        (if (re-search-backward name-chars bol t) nil
                            (re-search-forward name-chars eol t)))
                    (let ((eobp (eobp)))
                      (if (not (and (looking-at "[\"('#]")
                                    (or eobp
                                        ;; A non-atomic s-expression to the left of the
                                        ;; cursor should always prevail over one to the right
                                        ;; of the cursor.
                                        (not (memq (char-after (- (point) 1))
                                                   '(?\" ?\) ?\# ?\' ))))))
                          (progn
                            (if (and (looking-at name-chars) (not eobp))
                                (forward-char 1))
                            (backward-sexp 1)))))
                (if (and sexps-ok (looking-at "[\"('#]"))
                    (mark-sexp 1)
                    (progn
                      (if (re-search-backward non-name-chars bol t)
                          (if (not (eobp)) (forward-char))
                          (goto-char bol))
                      (set-mark (point))
                      (if (not (re-search-forward non-name-chars eol t))
                          (goto-char eol)
                          (if (not (bobp)) (backward-char))))))
            (error (set-mark (point))))
          (let* ((s (buffer-substring (mark) (point)))
                 (l (length s)))
            (if (and (> l 0) (string-match "[.,:]$" s))
                (setq s (substring s 0 (match-beginning 0))))
            (if (eq 0 (length s))
                (progn (setq s nil)))
            (setq apollo:mark (mark) apollo:point (point))
            s)))))


;;; apollo:dynamic-mouse-call 
;;; is interactive, but is not included in the section immediately
;;; following, entitled Interactive Stuff, because we do not want
;;; generate-readme to document this function; since it is actually
;;; an internal function and not part of the user interface.
(defun apollo:dynamic-mouse-call (&optional args)
  "This calls the function bound to apollo:dynamic-mouse-binding.
       It is used for context-sensitive mouse bindings (e.g. for
       completion and mouse-insert-thing."
  (interactive)
  (let ((apollo:old-window (selected-window))
        (apollo:old-point  (point))
        (apollo:old-buffer (current-buffer))
        (apollo:in-dynamic-mouse-call-p t))
    (apollo:mouse-move-point args)
    (funcall apollo:dynamic-mouse-binding)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Interactive Stuff


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Attribute list commands

(defun parse-attribute-list
    (&optional arg)
  "Use the buffer's attribute list to set local variables.  (For a
description of attribute lists, see the beginning of this chapter.)

        Emacs parses the attribute list when reading the file into the buffer
and sets the appropriate variables.  If the optional argument UPDATE is non-nil,
Emacs checks the corresponding local variables and, if they vary from those in
the attribute list, queries to update the attribute list."
  (interactive "P")
  (condition-case foo
      (if arg
          (update-attribute-list)
          (save-excursion
            (goto-char (point-min))
            (let ((attribute-list nil)
                  (done nil))
              (if (re-search-forward 
                    (format apollo:attribute-line-prefix (regexp-quote (or comment-start "")))
                    (+ 1 (eol)) t)
                  (while (not done)
                    (set-mark (point))
                    (if (re-search-forward apollo:attribute-value-separator nil t)
                        (let (attribute e)
                          (setq attribute 
                                (intern 
                                  (downcase 
                                    (buffer-substring (mark) (match-beginning 0)))))
                          (setq e (match-end 0))
                          (if (save-excursion 
                                (goto-char (match-beginning 0))
                                (looking-at apollo:attribute-line-end))
                              (progn (goto-char (mark)) (setq attribute 'mode))
                              (set-mark e))
                          (if (re-search-forward apollo:attribute-end nil t)
                              (let ((value-string (buffer-substring (mark) (match-beginning 0))))
                                (setq done (string-match apollo:attribute-line-end (apollo:match 1)))
                                (if (eq attribute 'mode)
                                    (set-mode value-string)
                                    (push (cons attribute value-string)
                                          attribute-list)))
                              (setq done t)))
                        (setq done t))))
              (apollo:dolist (a attribute-list)
                (let ((f (intern (concat "set-" (symbol-name (car a)))))
                      (r (car (read-from-string (downcase (cdr a))))))
                  (rplacd a r)
                  (if (fboundp f)
                      (funcall f r)
                      (apollo:new-set-variable (car a) r))))
              attribute-list)))
    (error (message "Attribute list for %s is incorrect" 
                    
                    (buffer-name (current-buffer)) (ding)))))

(if (fboundp 'parse-attribute-list)
    (fset 'reparse-attribute-list (symbol-function 'parse-attribute-list)))


(defun update-attribute-list (&optional called-interactively)
  "Compare the buffer's local variables with its attribute list.  If
they disagree, query to update the attribute list."
  (interactive "p")
  (if (y-or-n-p "Update attribute list? ")
      (let ((apollo:update-attribute-list-force-p t)
            (apollo:mode nil))
        (apollo:dolist (s apollo:symbols)
          (if (eq s 'apollo:mode) (setq apollo:mode (apollo:mode)))
          (if (and (boundp s) (eval s) (string-match "^apollo:" (symbol-name s)))
              (let ((f (intern 
                        (concat "set-" 
                                (substring (symbol-name s) (match-end 0))))))
                (if (fboundp f) (funcall f (eval s))))))
        (parse-attribute-list nil))))

(defun common-lisp-mode ()
  "Display the package name on the current buffer's mode line."
  (interactive)
  (lisp-mode)
  (setq apollo:yank-type 'yank-prev-command)
  (setq mode-name
        (if (and (boundp 'apollo:package) apollo:package)
            (format "Common-Lisp, Package: %s" apollo:package)
            "Common-Lisp")))

(defun set-syntax (s)
  "Set the buffer's mode to the specified SYNTAX.

        At present, the command only recognizes the syntax \"common-lisp\" for
the Lisp mode."
  (interactive "P")
  (apollo:set-variable 'syntax s
                    (function (lambda (s)
                      (if (and (eq major-mode 'lisp-mode)
                               (eq s 'common-lisp)) 
                          (common-lisp-mode))
                      s))))

(defun set-mode (m)
  "Prompt for the MODE of the buffer and set it as specified."
  (interactive "P")
  (if (apollo:ilisp-buffer-p)
      (error "Cannot change mode of inferior lisp buffer")
      (apollo:set-variable 
       'mode 
       m
       (function (lambda (mode)
         (if (symbolp mode) (setq m (symbol-name mode)))
         (setq m (downcase (if (string-match "-mode$" m) m (concat m "-mode"))))
         (string-match "-mode$" m)
         (let ((name (substring m 0 (match-beginning 0))))
           (let ((f (intern m)))
             (if (fboundp f)
                 (progn
                   (if (and (eq f 'lisp-mode) 
                            (boundp 'apollo:syntax)
                            (eq apollo:syntax 'common-lisp))
                       (common-lisp-mode)
                       (funcall f))
                   (intern name))
                 (progn (ding) (message (format "No Mode %s." m))))))))
       (if (null m)
           (function (lambda () 
             (mapcar (function (lambda (s) (list (symbol-name s))))
                     (apropos "-mode$" 'commandp 'noprint))))))))


(defun set-package (p)
  "Prompt for the default PACKAGE of the buffer and set it as
specified."
  (interactive "P")
  (if (apollo:ilisp-buffer-p)
      (error "Type (In-Package '<PACKAGE-NAME>) to inferior lisp interpretter.")
      (apollo:set-variable 
       'package 
       p
       (function (lambda (p)
         (if (stringp p) 
             (progn (setq p (car (read-from-string p)))
                    (if (equal p '(quote nil)) (setq p nil))))
         (let (pack)
           (setq pack (and p (upcase (symbol-name (if (consp p) (car p) p))))
                 p    (if (consp p) (cdr p) nil))
           (if p
               (let (super)
                 (setq super (car p)
                       p (cdr p))
                 (setq super
                       (cond 
                         ((null super) nil)
                         ((symbolp super) super)
                         (t
                          (list (upcase (symbol-name super)))
                          (mapcar '(lambda (x) (upcase (symbol-name x))) 
                                  super))))
                 (set (make-variable-buffer-local 'apollo:superpackage) super)
                 (if p
                     (let (symbols)
                       (setq symbols (car p))
                       (set (make-variable-buffer-local 'apollo:symbol-count)
                            symbols)))))
           (if (string-match "^Common-Lisp" mode-name)
               (setq mode-name (concat "Common-Lisp, Package: " pack)))
           pack))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Parenthesis and comment commands

(defun close-definition ()
  "Add the correct number of right parentheses to close the current
definition."
  (interactive)
  (let ((done nil)
        (n 0))
    (save-excursion
      (while (and (not done) (not (looking-at "^(")))
        (condition-case foo (or (backward-up-list 1) (apollo:incf n))
          (error (setq done t)))))
    (insert-char (aref ")" 0) n)
    n))

(defun find-unbalanced-parentheses ()
  "Search the current buffer for unmatched parentheses and move point to
the first occurrence."
  (interactive)
  (apollo:find-unbalanced-parentheses 'VERBOSE))

(defun comment-out-region (arg)
  "In the marked region, begin each line with the string in
comment-start and end it with the string in comment-end; in Lisp, these are
\";\" and \"\", respectively.  If the command has a negative argument, remove
these comment characters from the beginning and end of lines in the marked
region."
  (interactive "p")
  (save-restriction
    (save-excursion
      (narrow-to-region (point) (mark))
      (goto-char (point-min))
      (let ((comment-start (or comment-start ""))
            (comment-end   (or comment-end   "")))
        (if (>= arg 0)
            (progn
              (replace-regexp "^" comment-start)
              (goto-char (point-min))
              (replace-regexp "$" comment-end))
            (progn
              (replace-regexp (concat "^" comment-start) "")
              (goto-char (point-min))
              (replace-regexp (concat comment-end "$") "")))))))

(defun uncomment-out-region (arg)
  "Remove the comment characters defined in comment-start and
comment-end from the beginning and end of lines in the marked region."
  (interactive "p")
  (setq arg (apollo:arg-val arg))
  (if (= arg 0) (setq arg 1))
  (comment-out-region (- arg)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Section and find-source-code commands (Lisp and Emacs Lisp only)

(defun sectionize-buffer (&optional arg)
  "Create a list of section markers for the current buffer for use by
the apollo:find-source-code commands; if ARG exists, create lists for all
buffers.

        The command creates an association list in the form of a name and
marker for each definition.  Many routines (and Emacs commands) use this list to
locate definitions in the buffer."
  (interactive "P")
  (if apollo:sectionize-buffer
      (let ((buffers (if arg (buffer-list) (list (current-buffer)))))
        (apollo:dolist (b buffers)
          (let ((clisp-p nil))
            (with-buffer b
              (if (and 
                   (or 
                    (setq clisp-p 
                          (or apollo:force-common-lisp-p (apollo:clisp-buffer-p)))
                    (apollo:elisp-buffer-p))
                   (not (apollo:special-buffer-p b)))
                  (progn
                    (if apollo:verbose (message "Sectionizing %s..." (buffer-name b)))
                    (apollo:sectionize-buffer b clisp-p)
                    (if apollo:verbose (message "Sectionizing %s Done" (buffer-name b)))))))))))

(defun apollo:mouse-find-file ()
  "When the user presses M3 (the right mouse button), search for a file
with pathname given by the string under the mouse cursor, and load the file into
a buffer."
  (interactive)
  (let ((proc (get-buffer-process (current-buffer))))
    (if (and proc (string-match "shell" (process-name proc)))
        (setq default-directory (cdr (apollo:real-directory "./" proc)))))
  (let* ((file (apollo:current-thing apollo:file-chars (not 'sexps-ok)))
         (l (length file)))
    (if (and (>= l 2) 
             (eq (aref "~" 0) (aref file 0))
             (not (eq (aref "/" 0) (aref file 1))))
        (setq file (format "~/%s" (substring file 1))))
    (message file)
    (if (file-exists-p file)
        (find-file-other-window file)
        (error (concat "File does not exist: " file)))))

(defun find-file-no-sectionize (&optional filename)
  "Move point to the buffer visiting the file FILENAME; create the
buffer if none exists.

        This command inhibits \"sectionizing\" of the buffer for
find-source-code commands (see the sectionize-buffer command description for
details).  However, Emacs can sectionize the buffer's contents if needed."
  (let ((apollo:sectionize-buffer nil))
    (find-file filename)))

(defun list-sections (arg)
  "Section the contents of the current buffer and list all sections; if
the command has a non-nil argument, simply list the sections.

        The command searches for lines beginning with (def or


(<package_name>::def and lists the associated sections.  Clicking M2 on a
section name in the list loads the section for editing."
  (interactive "P")
  (if (not arg) (sectionize-buffer))
  (if apollo:buffer-sections
      (let ((title (format "Sections for %s\n" 
                           (buffer-name (current-buffer)))))
        (apollo:lines-to-buffer
          apollo:sections-buffer
          title
          'refresh
          (reverse apollo:buffer-sections)
          (function (lambda (s) (symbol-name (car s))))
          (apollo:mode)
          (apollo:find-package)))
      (message "No Buffer Sections")))

(defun apollo:mouse-find-source-code (&optional args)
  "When the user presses M2 (the middle mouse button), search for a
function whose name is given by the string under the mouse cursor, load the
source code for the identified function into a buffer.

        The command is defined for Lisp and Emacs-Lisp modes only."
  (interactive)
  (let ((thing (apollo:mouse-thing
                 args
                 apollo:symbol-chars
                 t)))
    (if thing
        (if (apollo:lisp-buffer-p)
            (apollo:find-source-code (apollo:definition-name thing) 'display)
            (progn (message "%s" thing) thing))
        (error "Mouse is not pointing at symbol"))))

(defun apollo:key-find-source-code (&optional no-query)
  "Prompt for a function NAME and load the source code for the specified
function in a buffer."
  (interactive "P")
  (if (apollo:lisp-buffer-p)
      (let ((string (apollo:key-symbol "Find Symbol's Source Code")))
        (if string
            (apollo:find-source-code (apollo:definition-name string) 'display no-query)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; File Evaluation and compilation commands

(defun compile-file (file)
  "Prompt for the name of a Lisp FILE and compile it."
  (interactive "P")
  (apollo:file-op 'compile file))

(defun load-file (file)
  "Prompt for the name of a Lisp FILE and load it in the current
buffer."
  (interactive "P")
  (apollo:file-op 'load file))

(defun load-compile-file (file)
  "Prompt for the name of a Lisp FILE, compile it, and load it in the
current buffer.  A numeric argument will compile the file even if the compiled
version is up-to-date."
  (interactive "P")
  (apollo:file-op 'load-compile file))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Evaluation and compilation commands (Lisp and Emacs Lisp only)

(defun evaluate-buffer (arg)
  "Evaluate the current buffer by parsing all text in the buffer as one
or more Lisp expressions and evaluating them one at a time.

        An ARG of 0 causes the command to query before evaluating the buffer.
All other values of ARG prevent querying and pass the value to the
evaluate-common-lisp function.

        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (save-excursion
    (set-mark (point-min))
    (goto-char (point-max))
    (let ((query (apollo:arg-val arg)))
      (apollo:evaluate-region 
       'no-hack 'eval 'refresh 'send query (not 'changed)))))

(defun compile-buffer (arg)
  "Compile the contents of the current buffer.

        An ARG of 0 causes the command to query before compiling the contents
of the buffer.  All other values of ARG prevent querying and pass the value to
the evaluate-common-lisp function.

        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (save-excursion
    (set-mark (point-min))
    (goto-char (point-max))
    (let ((query (apollo:arg-val arg)))
      (apollo:evaluate-region 
       'no-hack 'compile 'refresh 'send query (not 'changed)))))

(defun apollo:evaluate-last-sexp (arg)
  "If Common Lisp, send the last s-expression (that is, the one to the
left of point) to the Lisp process invoked by the M-x lisp command for
evaluation.

        If Emacs-Lisp mode, evaluate the last s-expression in Emacs Lisp.

        An ARG of 0 causes the command to query before evaluating the
s-expression.  All other values of ARG prevent querying and pass the value to
the evaluate-common-lisp function.

        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (if (looking-at apollo:any-def)
      (apollo:lisp-send-defun arg)
      (let ((apollo:mark (save-excursion (forward-sexp -1) (point)))
            (apollo:point (point)))
        (funcall 
          (if (apollo:common-lisp-buffer-p)
              'evaluate-common-lisp
              'evaluate-emacs-lisp)
          (or arg 1) 
          (apollo:current-thing nil t)))))


(defun apollo:lisp-send-defun (arg)
  "If Common Lisp mode, send the current definition (that is, the top-level
form containing point) to the Lisp process invoked by the M-x lisp command for
evaluation.

        If Emacs-Lisp mode, evaluate the current definition in Emacs Lisp.

        An ARG of 0 causes the command to query before evaluating the
definition.  All other values of ARG prevent querying and pass the value to the
evaluate-common-lisp function.

        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (save-excursion
    (apollo:mark-defun)
    (if (looking-at apollo:any-def)
        (progn
          (let ((query (apollo:arg-val arg)))
            (set-mark (point))
            (apollo:evaluate-region 
              'no-hack 'quiet-eval 'refresh 'send query (not 'changed))))
        (if (apollo:common-lisp-buffer-p)
            (evaluate-common-lisp (or arg 1) (buffer-substring apollo:mark apollo:point))
            (evaluate-emacs-lisp  (or arg 1) (buffer-substring apollo:mark apollo:point))))))

(defun apollo:lisp-compile-defun (arg)
  "Send the current defun macro to the Lisp process invoked by the M-x
lisp command for compilation.

        An ARG of 0 causes the command to query before compiling the macro.
All other values of ARG prevent querying and pass the value to the
evaluate-common-lisp function.

        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (save-excursion
    (set-mark (point))
    (let ((query (apollo:arg-val arg)))
      (apollo:evaluate-region 
       'no-hack 'compile 'refresh 'send query (not 'changed)))))

(defun evaluate-region (arg)
  "Evaluate the marked region in the current buffer.  The command parses
all text in the region as one or more Lisp expressions and evaluates them one at
a time.

        An ARG of 0 causes the command to query before evaluating the region.
All other values of ARG prevent querying and pass the value to to the
evaluate-common-lisp function.

        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (let ((query (apollo:arg-val arg)))
    (apollo:evaluate-region 'no-hack 'eval 'refresh 'send query (not 'changed))))


(defun evaluate-region-hack (arg)
  "Evaluate the marked region in the current buffer using setq rather than
defvar.

        An ARG of 0 causes the command to query before evaluating the region.
All other values of ARG prevent querying and pass the value to the
evaluate-common-lisp function.

        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (let ((query (apollo:arg-val arg)))
    (apollo:evaluate-region 
     (not 'no-hack) 'eval 'refresh 'send query (not 'changed))))

(defun compile-region (arg)
  "Compile the marked region in the current buffer.

        An ARG of 0 causes the command to query before compiling the region.
All other values of ARG prevent querying and pass the value to the
evaluate-common-lisp function.

        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (let ((query (apollo:arg-val arg)))
    (apollo:evaluate-region 
     'no-hack 'compile 'refresh 'send query (not 'changed))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; common lisp evaluation
(defun evaluate-common-lisp (arg &optional form short-form)
  "Evaluate a Common Lisp form.


        If ARG is

        nil     Evaluate the form

        1       (Default) Print the results of evaluation in the minibuffer

        2       Display results of evaluation in a buffer named *lisp-output*

        3       Bring up an inferior Lisp process and evaluate the form
there   

        4       Insert the result at point in the current buffer

        5       Replace the current form with the result of its evaluation

        If FORM is specified, evaluate it with an inferior Lisp process
initiated with M-x lisp.  Otherwise, prompt for the form."
  (interactive "p")
  (if (apollo:common-lisp-buffer-p)
      (progn
        (if (not form) (setq form (apollo:prompt-for-cl "Common Lisp Form")))
        (if (null form)
            (message "No Common Lisp form to evaluate")
            (let* ((omark apollo:mark)
                   (opoint apollo:point)
                   (p (apollo:find-package 'used-p))
                   (b (or apollo:base apollo:ibase))
                   (i apollo:ibase)
                   (r apollo:readtable)
                   (u apollo:used-packages)
                   (display (eq arg apollo:evaluate-common-lisp-as-typin))
                   (q (not display))
                   (c apollo:compile)
                   (eval-form 
                    (concat 
                      "(multiple-value-list (apollo::eval-string\n"
                      "  \"" (apollo:escape-string form "[\"\\]") "\""
                      (if p (concat "\n  :PACKAGE       '" p))
                      (if b (concat "\n  :BASE          "  b))
                      (if i (concat "\n  :IBASE         "  i))
                      (if r (concat "\n  :READTABLE     "  r))
                      (if u (concat "\n  :USED-PACKAGES '" (prin1-to-string u)))
                      (if c (concat "\n  :COMPILE       "  "T"))
                      (if q (concat "\n  :QUIET         "  "T"))
                      "))\n"))
                   (results (apollo:ask-common-lisp eval-form display t))
                   (lvalues  (car results))
                   (error   (cadr results))
                   (output  (caddr results)))
              (if (consp lvalues) 
                  (setq lvalues (mapcar '(lambda (v) 
                                         (car (read-from-string 
                                               (apollo:fix-sharp-char v)
                                               )))
                                       lvalues)))
              (cond
                ((eq arg apollo:evaluate-common-lisp-as-typin)
                 nil)

                ((and error (not (eq error 'NIL))) 
                 (error "Lisp Error: %s" lvalues))
                ((null arg) 
                 (car lvalues))
                ((eq arg apollo:evaluate-common-lisp-into-minibuffer)
                 (if (or lvalues (and output (not (equal output ""))))
                     (message "%s%s"
                              (if (> (length lvalues) 1) 
                                  (format "Values: %s" lvalues)
                                  (format "Result: %s" (car lvalues)))
                              (if (equal output "") "" (format ", Output: %s" output))))
                 (car lvalues))
                ((eq arg apollo:evaluate-common-lisp-into-temp-buffer)
                 (let* ((title-header "Evaluated Common Lisp Form:\n  ")
                        (l (- (window-width) (length title-header) 1))
                        (tail " ... )")
                        (title 
                         (concat
                           title-header
                           (if (< (length (or short-form form)) l) 
                               (or short-form form)
                               (concat (substring (or short-form form) 0 (- l (length tail))) tail))
                           (if (> (length lvalues) 1) "\n\nValues:" "\n\nResult:"))))
                   (apollo:lines-to-buffer 
                     apollo:lisp-output-buffer title 'refresh lvalues
                     (if (> (length lvalues) 1)
                         (function (lambda (s) (format "[%s]: %s" n (apollo:strip-leading-whitespace s))))
                         (function (lambda (s) (format "%s"         (apollo:strip-leading-whitespace s)))))
                     "common-lisp" p)
                   (if (and output (not (equal output "")))
                       (apollo:lines-to-buffer 
                         apollo:lisp-output-buffer 
                         "\nOutput:"
                         (not 'refresh) output nil))
                   (car lvalues)))
                ((eq arg apollo:evaluate-common-lisp-insert)
                 (if lvalues (insert (prin1-to-string (car lvalues)))) (car lvalues))
                ((eq arg apollo:evaluate-common-lisp-delete-and-insert)
                 (let ((min (min omark opoint)) 
                       (max (max omark opoint)))
                   (kill-region min max)
                   (goto-char min)
                   (insert (format "%s" (car lvalues)))
                   (car lvalues)))
                (error "Unknown argument to evaluate-common-lisp: %s" arg)))))))


(defun evaluate-emacs-lisp (arg &optional form-string short-form-which-is-ignored)
  "Evaluate an Emacs Lisp form.


        If ARG is

        nil     Evaluate the form

        1       (Default) Print the results of evaluation in the minibuffer

        2       Display results of evaluation in a buffer named *lisp-output*

        3       Bring up an inferior Lisp process and evaluate the form there

        4       Insert the result at point in the current buffer

        5       Replace the current form with the result of its evaluation

        If FORM is specified, evaluate it.  Otherwise, prompt for the form."
  (interactive "p")
  (if (not form-string) (setq form-string (apollo:prompt-for-cl "Emacs Lisp Form")))
  (let* ((omark apollo:mark)
         (opoint apollo:point)
         (form (and form-string (car (read-from-string form-string))))
         (result (eval form)))
    (cond 
      ((null arg) result)
      ((eq arg apollo:evaluate-common-lisp-into-minibuffer) (prin1 result t) result)
      ((memq arg (list apollo:evaluate-common-lisp-into-temp-buffer
                       apollo:evaluate-common-lisp-as-typin))
       (let* ((title-header "Evaluated Emacs Lisp Form; ")
              (l (- (window-width) (length title-header) 1))
              (tail " ... )")
              (title 
               (concat
                 "Evaluated Emacs Lisp Form: "
                 (if (< (length form-string) l) 
                     form-string 
                     (concat (substring form-string 0 (- l (length tail)))
                             tail))
                 "\nResult:\n")))
         (with-buffer-set (get-buffer-create "*EMACS OUTPUT*" )
           (delete-region (point-min) (point-max))
           (emacs-lisp-mode)
           (apollo:pprint result (current-buffer))
           (goto-char (point-min))
           (insert title))
         (display-buffer "*EMACS OUTPUT*" 'not-this-window))
       result)
      ((eq arg apollo:evaluate-common-lisp-insert)
       (apollo:pprint result (current-buffer)))
      ((eq arg apollo:evaluate-common-lisp-delete-and-insert)
       (let ((min (min omark opoint)) 
             (max (max omark opoint)))
         (kill-region min max)
         (goto-char min)
         (apollo:pprint result)
         result))
      ((error "Unknown argument to evaluate-common-lisp: %s" arg)))))


(defun evaluate-and-replace-into-buffer (arg)
  "Evaluate the current s-expression and replace it with the result."
  (interactive "P")
  (apollo:evaluate-last-sexp apollo:evaluate-common-lisp-delete-and-insert))

(defun evaluate-into-buffer (form)
  "Prompt for a FORM.  Evaluate the specified form and place the result
at point."
  (interactive "P")
  (cond ((apollo:emacs-lisp-buffer-p)
         (evaluate-emacs-lisp apollo:evaluate-common-lisp-insert form))
        ((apollo:common-lisp-buffer-p)
         (evaluate-common-lisp apollo:evaluate-common-lisp-insert form))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Changed definition commands (Language independent)

(defun list-modifications (arg &optional no-display)
  "Use the UNIX diff command to list all modifications made to the
contents of the current buffer.


        If ARG is

        0       Display the results of the last call to this command

        1       (Default) List the modifications made since Emacs read the

                file into the buffer

        2       List the modifications made since Emacs last read the file

                into the buffer or saved the file

        All changed-definitions commands call this command."
  (interactive "p")
  (if (eq arg 0)
      (set-window-start 
       (display-buffer (get-buffer-create apollo:modifications-buffer)) 1)
      (let ((b (get-buffer-create apollo:modifications-buffer))
            (s (format 
                "Modifications for %s since %s:"
                (buffer-name (current-buffer)) 
                (if (eq arg 2) "last saved" "read in")))
	    (f (apollo:arg-file arg))
            (c (current-buffer))
            (mod-p t))
        (with-buffer-set b (with-read-only nil (erase-buffer)))
        (if (not no-display) (set-window-start (display-buffer b) 1))
        (message "Finding Modifications...")
        (apollo:lines-to-buffer 
         b s nil nil nil (apollo:mode) apollo:package 
         'apollo:list-modifications-dynamic-mouse)
	(with-buffer-set b
	  (with-read-only nil
	    (call-process-region 
	      (point-min) (point-max) "diff" nil
	      (get-buffer b) nil "-" f)))
        (setq apollo:modifications-original (buffer-name c))
        (message "Finding Modifications Done."))))


(defun apollo:list-modifications-dynamic-mouse ()
  "Find and display source of modifications at point.
       This is designed to be used by list modification commands
       to let clicking right on the mouse, find the modification."
  (interactive)
  (if (equal (buffer-name (current-buffer)) apollo:modifications-buffer)
      (progn
        (if (re-search-backward "^[0-9]" nil t)
            (let ((m (point)))
              (recenter 0)
              (if (re-search-forward "[^0-9]" nil t)
                    (let* ((w (display-buffer apollo:modifications-original 'NOTTHISWINDOW))
                           (l (car 
                               (read-from-string 
                                (buffer-substring m (1- (point))))))
                           (c (with-buffer apollo:modifications-original 
                                (goto-char (point-min)) 
                                (forward-line (1- l)) (point))))
                      (set-window-start w c))))))))



(defun edit-next-definition ()
  "Load the source code for the next definition on a list of definitions
into an edit buffer.

        To operate, this command requires lists generated by other commands
such as M-. and M-x edit-changed-definitions."
  (interactive)
  (if (apollo:lisp-buffer-p)
      (let ((b nil) (def nil) 
            (find-source-code-p
             (let ((apollo:buffer-definitions-type nil))
               (eq apollo:buffer-definitions 'apollo:find-source-code-info))))
        (if find-source-code-p
            (let* ((point        (car  apollo:find-source-code-info))
                   (buffer       (cadr apollo:find-source-code-info))
                   (rest-buffers (cddr apollo:find-source-code-info)))
              (with-buffer-set buffer
                (setq def (apollo:find-def apollo:find-source-code-symbol point))
                (if def
                    (progn
                      (rplaca apollo:find-source-code-info 
                              (- (apollo:section-start def) 1))
                      (push def apollo:buffer-definitions))
                    (if rest-buffers
                        (apollo:buffer-definitions 
                         'ignore 'find-source-code rest-buffers
                         (not 'ACTIONS) (not 'query))
                        (setq apollo:buffer-definitions nil))))))
        (setq apollo:current-definition (car apollo:buffer-definitions))
        (setq apollo:buffer-definitions (cdr apollo:buffer-definitions))
        (if apollo:current-definition
            (let ((b (get-buffer (apollo:section-buffer apollo:current-definition)))
                  (symbol (apollo:section-name apollo:current-definition)))
              (if b
                  (with-buffer-set b
                    (if (not def) (setq def (apollo:find-def symbol)))
                    (if def
                        (progn
                          (with-buffer b
                            (goto-char (apollo:section-start def))
                            (let ((w (display-buffer b 'NOT-THIS-WINDOW)))
                              (set-window-start w (apollo:section-start def))
                              (if apollo:locate-symbol
                                  (with-buffer-set b
                                    (goto-char (apollo:section-start def))
                                    (if (re-search-forward 
                                         (regexp-quote 
                                          (apollo:strip-package apollo:symbol)) nil t)
                                        (set-window-point w (match-beginning 0))))
                                  (if apollo:buffer-definitions
                                      (message "Use <C-c .> to find next definition.")
                                      (message "No More Definitions"))))))
                        (error "Definition not found for %s in buffer %s" symbol b)))
                  (progn
                    (message (format "Buffer Not Found: %s" b))
                    (while (eq (apollo:section-buffer (car apollo:buffer-definitions))
                               (apollo:section-buffer apollo:current-definition))
                      (pop apollo:buffer-definitions)))))
            (if (eq apollo:buffer-definitions-type 'apollo:find-source-code)
                (save-excursion
                  (find-tag (downcase (apollo:strip-package apollo:find-source-code-symbol)))
                  (find-file-other-window (buffer-file-name (current-buffer))))

                (message "No More Definitions" (ding)))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Changed definition commands  (Lisp and Emacs Lisp only)

(defun apollo:reset-original (arg)
  "Resets what the editor treats as the original source of a file to the
       current source in the buffer. This impacts further use of any 
       changed definitions command.  Use this command with care!
       With an ARG specified, all buffers of of the lisp type of the
       current-buffer will be reset."
  (interactive "P")
  (setq apollo:last-arg-file nil)
  (let ((clisp-p (apollo:common-lisp-buffer-p)))
    (apollo:dolist (b (if arg (buffer-list) (list (current-buffer))))
      (if (not (apollo:special-buffer-p b))
          (with-buffer-set b
            (if (if clisp-p (apollo:clisp-buffer-p) (apollo:elisp-buffer-p))
                (progn
                  (message (format "Reseting %s" (buffer-name b)))
                  (setq apollo:original nil)
                  (apollo:set-original))))))))

(defun list-buffer-changed-definitions (arg) ;
  "In a separate window, list the names of the changed definitions
located in the current buffer.  If the M2 mouse button clicks on a definition in
the list, load the source code of that definition into an edit buffer.

        If ARG is

        0       Use the definitions from the last changed-definition

                command

        1       Use the changes made since Emacs read the file into the

                current buffer

        2       Use the changes made since Emacs read the file into the

                buffer or saved the buffer's contents

        3       (Default) Use the changes made since compiling or evaluat-

                ing the definitions or, if never compiled or evaluated, since

                reading the file into the current buffer


        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (apollo:buffer-definitions 
   (apollo:arg-val arg apollo:since-eval-or-compile) 'changed-definition 
   (list (current-buffer)) '(list) (not 'query)))


(defun evaluate-buffer-changed-definitions (arg)
    "Evaluate all changed definitions in the current buffer.

        If ARG is

        0       Use the definitions from the last change-definition command

        1       Use the changes made since Emacs read the file into the

                current buffer

        2       Use the changes made since Emacs read the file into the

                buffer or saved the buffer's contents

        3       (Default) Use the changes made since evaluating the defini-

                tions or, if never evaluated, since reading the file into the

                current buffer


        The command is defined in Lisp and Emacs-Lisp modes only."
    (interactive "P")
    (apollo:buffer-definitions 
     (apollo:arg-val arg apollo:since-eval-or-compile) 
     'changed-definition (list (current-buffer)) '(eval) (not 'query)))

(defun compile-buffer-changed-definitions (arg) ;
    "Compile all changed definitions in the current buffer.

        If ARG is

        0       Use the definitions from the last change-definition command

        1       Use the changes made since Emacs read the file into the

                current buffer

        2       Use the changes made since Emacs read the file into the

                buffer or saved the buffer's contents

        3       (Default) Use the changes made since compiling the defini

                tions or, if never compiled, since reading the file into the

                current buffer


        The command is defined in Lisp and Emacs-Lisp modes only."
    (interactive "P")
    (apollo:buffer-definitions 
     (apollo:arg-val arg apollo:since-eval-or-compile) 'changed-definition
     (list (current-buffer)) '(compile) (not 'query)))


(defun edit-buffer-changed-definitions (arg)
  "Edit all changed definitions in the current buffer.

        If ARG is

        0       Use the definitions from the last change-definition command

        1       Use the changes made since Emacs read the file into the

                current buffer

        2       Use the changes made since Emacs read the file into the

                buffer or saved the buffer's contents

        3       (Default) Use the changes made since compiling or evaluat-

                ing the definitions or, if never compiled or evaluated, since

                reading the file into the current buffer


        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (apollo:buffer-definitions 
   (apollo:arg-val arg apollo:since-eval-or-compile) 'changed-definition
   (list (current-buffer)) '(edit) (not 'query)))

(defun list-changed-definitions (arg)
  "List all changed definitions in all buffers of the current Lisp type
(Common Lisp or Emacs Lisp).

        If ARG is

        1       Use the changes made since Emacs read the file(s) into the

                buffer(s)

        2       Use the changes made since Emacs read the file(s) into the

                buffer(s) or saved the buffers' contents

        3       (Default) Use the changes made since compiling or evaluat-

                ing the definitions or, if never compiled or evaluated, since

                reading the file(s) into the buffer(s)


        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (apollo:buffer-definitions 
   (apollo:arg-val arg apollo:since-eval-or-compile) 'changed-definition
   (buffer-list) '(list) (not 'query)))


(defun evaluate-changed-definitions (arg)
    "Evaluate all changed definitions in all buffers of the current Lisp
type (Common Lisp or Emacs Lisp).

        If ARG is

        1       Use the changes made since Emacs read the file(s) into the

                buffer(s)

        2       Use the changes made since Emacs read the file(s) into the

                buffer(s) or saved the buffers' contents

        3       (Default) Use the changes made since evaluating the defini-

                tions or, if never evaluated, since reading the file(s) into the

                buffer(s)


        The command is defined in Lisp and Emacs-Lisp modes only."
    (interactive "P")
    (apollo:buffer-definitions 
     (apollo:arg-val arg apollo:since-eval-or-compile) 'changed-definition
     (buffer-list) '(eval) (not 'query)))

(defun compile-changed-definitions (arg)
  "Compile all changed definitions in all buffers of the current Lisp
type.

        If ARG is

        1       Use the changes made since Emacs read the file(s) into the

                buffer(s)

        2       Use the changes made since Emacs read the file(s) into the

                buffer(s) or saved the buffers' contents

        3       (Default) Use the changes made since compiling the defini-

                tions or, if never compiled, since reading the file(s) into the

                buffer(s)


        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (apollo:buffer-definitions 
   (apollo:arg-val arg apollo:since-eval-or-compile) 'changed-definition
   (buffer-list) '(compile) (not 'query)))

(defun edit-changed-definitions (arg)
  "Edit all changed definitions in all buffers of the current Lisp type

(Common Lisp or Emacs Lisp).

        If ARG is

        1       Use the changes made since Emacs read the file(s) into the

                buffer(s)

        2       Use the changes made since Emacs read the file(s) into the

                buffer(s) or saved the buffers' contents

        3       (Default) Use the changes made since compiling or evaluat-

                ing the definitions or, if never compiled or evaluated, since

                reading the file(s) into the buffer(s)


        The command is defined in Lisp and Emacs-Lisp modes only."
  (interactive "P")
  (apollo:buffer-definitions 
   (apollo:arg-val arg apollo:since-eval-or-compile) 'changed-definition
   (buffer-list) '(edit) (not 'query)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Callers commands

(defun edit-callers (arg)
  "Prompt for a Common Lisp symbol name and edit its callers.  If no
argument is specified, look in the current package.


        If ARG is

        4 or C-u        Look in all packages

        16 or C-u C-u   Prompt user for package name"
  (interactive "P")
  (if (apollo:common-lisp-buffer-p)
      (progn
        (apollo:lisp-callers (apollo:arg-val arg))
        (next-caller))
      (error "List callers may only be called from common lisp buffer")))

(defun next-caller ()
  "Load the source code for the next caller on a list of definitions
into an edit buffer.


        Commands such as M-x edit-callers sometimes generate lists of
definitions."
  (interactive)
  (if (apollo:common-lisp-buffer-p)
      (if apollo:callers
          (let ((c (pop apollo:callers)))
            (message "Finding caller %s of %s" (car c) apollo:symbol)
            (if (not (equal (cadr c) "Unknown Source"))
                (find-file-noselect (cadr c)))
            (let ((apollo:locate-symbol apollo:symbol))
              (if (apollo:find-source-code  (symbol-name (car c)) 'display)
                  (message 
                   "<C-c n> for next caller, <C-c .> for other definitions.")
                  (error "%s not found.  <C-c n> for next caller." (car c)))))
          (error (format "No More callers for %s" apollo:symbol)))
      (error "Next caller may only be called from common lisp buffer")))


(defun list-callers (arg)
  "Prompt for a Common Lisp symbol name and list its callers.  If no
argument is specified, look in the current package.


        If ARG is

        4 or C-u        Look in all packages

        16 or C-u C-u   Prompt user for package name"
  (interactive "P")
  (if (apollo:common-lisp-buffer-p)
      (if (apollo:lisp-callers (apollo:arg-val arg))
          (let* ((package (apollo:current-inferior-lisp-package))
                 (title (format "%s is used by the functions below:" 
                                (upcase apollo:symbol) package)))
            (setq title (concat title "\n[C-c n] Edit next caller.\n"))
            (apollo:lines-to-buffer
             apollo:callers-buffer title 'refresh apollo:callers
             (function apollo:stringify-caller) "common-lisp" package))
          (progn 
            (setq apollo:callers nil)
            (message "No Callers for %s" apollo:symbol (ding))))
      (error "List callers may only be called from common lisp buffer")))

;; Define alternate name for list callers
(defun who-calls (arg)
  "(This is another name for the list-callers command.)"
  (interactive "P")
  (list-callers arg))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Description commands

(defun disassemble-lisp-code (object &optional stream indent interactive-p)
  "Print disassembled code for OBJECT on optional STREAM.  If necessary,
compile OBJECT if necessary but do not redefine.  For Common Lisp functions,
take the output of (disassemble #'object) and display it in a buffer.

        The OBJECT can be a function name, lambda expression, or any function
object returned by the symbol-function function."
  (interactive 
   (list
    (if (apollo:common-lisp-buffer-p)
        (let* ((c (apollo:current-thing nil (not 'sexps-ok)))
               (s (read-string (format "Disassemble function[%s]: " c))))
          (if (equal s "") c s))
        (intern (completing-read "Disassemble function: " obarray 'fboundp t)))
    nil 0 t))
  (or indent (setq indent 0))           ;Default indent to zero
  (if (not (apollo:common-lisp-buffer-p))
      (autoload 'disassemble-internal "disass"))
  (if (or interactive-p (apollo:common-lisp-buffer-p))
      (with-output-to-temp-buffer apollo:disassemble-buffer
        (if (apollo:common-lisp-buffer-p)
            (progn
              (setq object (apollo:packagify object))
              (princ (format "Disassemble %s\n" object))
              (princ 
                (evaluate-common-lisp 
                  nil
                  (concat
                    "(WITH-OUTPUT-TO-STRING (*STANDARD-OUTPUT*) (DISASSEMBLE #'"
                    object 
                    "))")
                  (concat "(DISASSEMBLE #'" object ")")
                  )))
            (disassemble-internal object standard-output indent t)))
      (disassemble-internal object (or stream standard-output) indent nil))
  nil)


(defun quick-arglist (arg &optional function-name)
  "Query Common Lisp and display the argument list of the function at
point.  If ARG is not specified, direct the output into a temporary buffer."
  (interactive "P")
  (let ((arglist nil))
    (if (not function-name)
        (save-excursion
          (if (< (point) (point-max)) (forward-char))
          (search-backward "(" nil t)
          (forward-char)
          (mark-sexp 1)
          (setq function-name (buffer-substring (mark) (point)))))
    (let ((s function-name))
      (if (not function-name) (message "Finding Arglist for [%s]" s))
      (if (apollo:common-lisp-buffer-p)
          (let* ((name (apollo:packagify s))
                 (short-form (concat "(USER::ARGLIST '" name ")")))
            (setq arglist (evaluate-common-lisp 
                            (or arg apollo:evaluate-common-lisp-into-temp-buffer)
                            (concat "(with-output-to-string (*TERMINAL-IO*)"
                                    " (" 
                                    (if arg "prin1" "pprint")
                                    " " short-form "))")
                            short-form)))
          (setq arglist 
                (evaluate-emacs-lisp (or arg apollo:evaluate-common-lisp-into-temp-buffer)
                                     (format "(arglist '%s)" (intern s))))))
    arglist))

(defun macro-expand-last-sexp (arg)
  "Expand the last s-expression.  If there is no argument, or the
argument is nil, expand all levels of the definitions in the s-expression, and
place the expansion in the temp buffer.  If the argument is 1, expand only the
uppermost level of the last s-expression, and place the expansion in the temp
buffer.

        If the argument is 4, or C-u, the expansion is the same as with a nil
argument, except that the expansion is done in place, in the current buffer
instead of in a temp buffer.  With an argument of 16, or C-u C-u, only the
uppermost level of the macro expansion is substituted in place.

        If Common Lisp, send the s-expression to the Lisp process invoked with
the M-x lisp command.

        If Emacs Lisp, expand the s-expression without sending it to another
Lisp process."
  (interactive "P")
  (let ((apollo:point (point))
        (apollo:mark (save-excursion (forward-sexp -1) (point))))
    (macro-expand-expression arg (buffer-substring apollo:mark apollo:point))))


(defun macro-expand-expression (arg &optional string)
  "Expand the current Lisp definition into a temporary buffer. If there
is no argument, or the argument is nil, expand all levels of the definitions in
the s-expression, and place the expansion in the temp buffer.  If the argument
is 1, expand only the uppermost level of the last s-expression, and place the
expansion in the temp buffer.

        If the argument is 4, or C-u, the expansion is the same as with a nil
argument, except that the expansion is done in place, in the current buffer
instead of in a temp buffer.  With an argument of 16, or C-u C-u, only the
uppermost level of the macro expansion is substituted in place.

        This command works with Emacs Lisp and Common Lisp."
  (interactive "P")
  (message "Expanding Macro")
  (setq arg (apollo:arg-val arg))
  (prog1
      (funcall (if (apollo:common-lisp-buffer-p) 'evaluate-common-lisp 'evaluate-emacs-lisp)
               (if (memq arg '(4 16)) 
                   apollo:evaluate-common-lisp-delete-and-insert 
                   apollo:evaluate-common-lisp-into-temp-buffer)
               (format "(apollo:macro-expand-expression%s '%s)" 
                       (if (memq arg '(1 16)) "" "-all")
                       (or string     
                           (save-excursion
                             (apollo:mark-defun)
                             (buffer-substring apollo:mark apollo:point)))))
    (if (and (memq arg '(4 16)) (apollo:common-lisp-buffer-p))
        (save-excursion
          (goto-char (min apollo:mark apollo:point))
          (if (looking-at (concat apollo:whitespace "*"))
              (delete-region (match-beginning 0) (match-end 0)))
          (indent-sexp)))
    (message "Done Expanding Macro")))

(defun where-is-symbol (&optional symbol)
  "Optional SYMBOL may be a Common Lisp symbol name.  If SYMBOL is nil,
prompt for a Common Lisp symbol name.  List all packages containing the symbol."
  (interactive)
  (apollo:evaluate-current-thing 
   "Where_Is_Symbol_APOLLO" "(apollo:where-is-symbol \"%s\")" symbol
   'apollo:grab-thing-dynamic-mouse 'without-package-p))

(defun apollo:describe-variable-at-point-dynamic-mouse ()
  "Describe Variable Under Mouse."
  (interactive)
  (let ((apollo:prompt-for-cl-use-default-p t))
    (describe-variable-at-point)))
  
(defvar apollo:describe-function "lisp:describe")

(defun describe-variable-at-point (&optional variable)
  "Use the Common Lisp describe function to exhibit the value,
definition, and properties of the specified symbol and direct the output to a
buffer.  The optional VARIABLE specifies a Common Lisp symbol.  If VARIABLE is
nil, prompt for the symbol."
  (interactive)
  (apollo:evaluate-current-thing 
   "Describe_Variable_APOLLO" 
   (concat "(with-output-to-string (*standard-output*) (" apollo:describe-function " %s))")
   variable 
   'apollo:describe-variable-at-point-dynamic-mouse
   nil t))

(defun describe-function-at-point (&optional function)
  "Use the Common Lisp describe function to exhibit the value,
definition, and properties of FUNCTION and direct the output to the current
buffer.

        If FUNCTION is a numeric argument, take the function at the beginning
of the current s-expression as the default and prompt the user.

        If FUNCTION is neither numeric nor nil, then use it as the function
name for describe.  (There's no prompt with this argument.)

        If FUNCTION is nil or not specified, take the symbol at point as the
default and prompt the user."
  (interactive "P")
  (if (numberp function) (setq function (apollo:current-function)))
  (apollo:evaluate-current-thing 
   "Describe_Function_APOLLO"
   (concat "(with-output-to-string (*standard-output*) (" apollo:describe-function " #'%s))")
   function 
   'apollo:describe-variable-at-point-dynamic-mouse
   nil t))

(defun show-lisp-documentation (&optional object)
  "Query Common Lisp and display documentation for OBJECT at point in a
temporary buffer."
  (interactive "P")
  (apollo:evaluate-current-thing
    "Documentation"
    (concat 
     "(MAPCAN #'(LAMBDA (D)\n"
     "            (LET ((DOC (DOCUMENTATION '%s D)))\n"
     "              (IF DOC (LIST (FORMAT NIL \"~S: ~S\n\" D DOC)))))\n"
     "        '(FUNCTION VARIABLE STRUCTURE TYPE SETF))")
    object))


(defun apropos-symbol-at-point (&optional variable)
  "Use the Common Lisp apropos function to search the package of the
specified variable.  Locate all variables that contain the specified name in
their print names and direct the output to a buffer.

        If VARIABLE is a numeric argument, take the variable at the beginning
of the current s-expression as the default and prompt the user.

        If VARIABLE is neither numeric nor nil, then use it as the name of the
variable for apropos.  (There's no prompt with this argument.)

        If VARIABLE is nil or not specified, take the variable at point as the
default and prompt the user."
  (interactive "P")
  (if (numberp variable) (setq variable (apollo:current-variable)))
  (apollo:lisp-complete-symbol 1 variable))

(defun what-package ()
  "Print the name of the current package to the minibuffer."
  (interactive)
  (let ((p (apollo:find-package)))
    (message "Current Packgage: %s" (or p "<Undefined>"))
    p))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Shell commands

(defun csh (&optional arg shell-name shell-program)  
  "Start an inferior shell process (default csh).
With numeric ARG
  Create a new shell buffer running shell.
Without numeric ARG
  if shell buffer exists, switch to that buffer
  Otherwise create new shell buffer.
SHELL-NAME is string to use as root name of shell buffer. (default \"csh\")
SHELL-PROGRAM is string for path to shell program (default \"/bin/csh\")"
  (interactive "P")
  (let ((shell-name (or shell-name "csh"))
        (shell-program (or shell-program "/bin/csh"))
        (shell-number 0))
    (if (or arg 
            (not (get-buffer shell-name))
            (not (equal 
                   shell-program
                   (with-buffer-set shell-name
                     (nth 5 (process-command (get-buffer-process (current-buffer))))))))
        (progn
          (while (get-buffer shell-name)
            (setq shell-name (format "%s<%d>" shell-name (apollo:incf shell-number))))
          (apollo:shell shell-name shell-program "-i"))
        (switch-to-buffer shell-name))))

(defun sh (&optional arg)   
  "Start an inferior sh shell.
With numeric ARG
  Create a new shell buffer running shell.
Without numeric ARG
  if shell buffer exists, switch to that buffer
  Otherwise create new shell buffer.
On apollo this will start an aegis /com/sh.
On other unix platforms, this will start a /bin/sh"
  (interactive) 
  (csh arg "sh" (if (apollo:apollo-p) (concat apollo:aegis-directory "/sh") "/bin/csh")))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Inferior lisp commands

(defun run-lisp ()
  "Run an inferior Lisp process in the current buffer."
  (interactive)
  (if (or (not (boundp 'inferior-lisp-program)) inferior-lisp-program)
      (setq inferior-lisp-program "lisp"))
  (if (y-or-n-p 
       "Use this LISP process with DOMAIN Programming Environment extensions?")
      (lisp)
      (progn
        (autoload 'make-shell "shell")
        (switch-to-buffer (make-shell "lisp" inferior-lisp-program))
        (inferior-lisp-mode))))

(defun lisp (&optional force-create-p)
  "Run the Lisp program specified by the variable inferior-lisp-program
in an inferior Common Lisp process.  If an inferior Lisp process already exists,
and no ARG is given, simply switch to that buffer.  If an ARG is given, create a
new process without regard to whether one exists or not."
  (interactive "P")
  (if (and (apollo:lisp-process 'NO-ERROR)
           (eq (process-status (apollo:lisp-process 'NO-ERROR)) 'run)
           (null force-create-p))
        (select-window (display-buffer (apollo:lisp-buffer)))
        (progn
          (if (or (not (boundp 'inferior-lisp-program)) (not inferior-lisp-program)) 
              (setq inferior-lisp-program "lisp"))
          (autoload 'make-shell "shell")
          (switch-to-buffer 
	    (make-shell (file-name-nondirectory inferior-lisp-program)
			inferior-lisp-program nil apollo:inferior-lisp-switches))
          (rename-buffer (setq apollo:lisp-name (process-name (get-buffer-process (current-buffer)))))
          (inferior-lisp-mode)
          (define-inferior-lisp-mode-keys)
          (setq apollo:yank-type 'yank-prev-command)
          (setq apollo:initial-lisp-process-filter (process-filter (apollo:lisp-process)))
          (set (make-variable-buffer-local 'apollo:buffer-process)
               (cons apollo:lisp-name (process-id (get-buffer-process (current-buffer)))))
          (setq apollo:lisp-processes (cons apollo:buffer-process apollo:lisp-processes))
          (set  (make-variable-buffer-local 'apollo:lisp-packages) apollo:initial-lisp-packages)
          (if apollo:gnuemacs-client-p (apollo:create-gnuemacs-client))
          (setq apollo:original 
                (format " *COMPILED-DEFINITIONS-FOR-%s*" apollo:lisp-name))
          (with-buffer-set (get-buffer-create apollo:original) 
            (lisp-mode) 
            (erase-buffer))
          (goto-char (point-max))
          (set-process-sentinel ;;; kill the lisp buffer after exiting lisp
            (apollo:lisp-process)
            (function (lambda (p s) 
              (let ((b (process-buffer p)))
                (if (and (string-match "finished" s) b (buffer-name b))
                    (progn
                      (with-buffer-set b
                        (message "Buff: %s, Proc: %s" b apollo:buffer-process)
                        (let ((apollo:lisp-name (car apollo:buffer-process)))
                          (reset-lisp-filter t))
                        (setq apollo:lisp-processes (delq apollo:buffer-process apollo:lisp-processes))
                        (if (eq apollo:lisp-name (car apollo:buffer-process))
                            (setq apollo:lisp-name (car (car apollo:lisp-processes)))))
                      (kill-buffer b)))))))
          (apollo:lisp-process))))


(defun apollo:break ()
  "Interrupt the process owned by the current buffer."
  (interactive)
  (funcall apollo:lisp-interrupt (get-buffer-process (current-buffer))))

(defun reset-lisp-filter (arg)
  "Reset an inferior Lisp process but do not return to Lisp's top-level
loop.  (This command regains control when the process fails to respond and
begins to lose Lisp input/output.)"
  (interactive "P")
  (setq apollo:ask-common-lisp-display-p nil apollo:ask-common-lisp-queue nil)
  (if apollo:gnuemacs-client-p
      (progn
        (if (get-process (apollo:lisp-server-name 'in))
            (progn 
              (kill-process (get-process (apollo:lisp-server-name 'in)))
              (while (get-process (apollo:lisp-server-name 'in)) (sit-for 1))
              (sit-for 1) ;; wait for process to really and truly die
              ))
        (if (get-process (apollo:lisp-server-name 'out))
            (progn 
              (kill-process (get-process (apollo:lisp-server-name 'out)))
              (while (get-process (apollo:lisp-server-name 'out)) (sit-for 1))
              (sit-for 1))))
      (if (not arg) (with-buffer-set (apollo:lisp-buffer) (apollo:break))))
  (let ((p (apollo:lisp-process 'nil-if-not-found)))
    (if p
        (set-process-filter p apollo:initial-lisp-process-filter))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Yank history commands

(defun apollo:yank (arg &optional yank-type pop pattern)
  "Reinsert the text of the most recent kill (or yank) with mark at the
front and point at the end of the text.


        If ARG is

                0               Display the kill history in a buffer

                n               With numeric argument n, reinsert the text of

                                the nth most recent kill

                C-u     Put point at front and mark at end of rein-

                                serted text

        The following are optional arguments:

        YANK-TYPE       This is a symbol whose name is the yank com

                                mand being performed

        POP                     This is a symbol with a non-nil value if re-

                                peating the yank command for previous items

                                in the kill history

        PATTERN         This is a string specifying the prefix criteria for

                                a match (see yank-prefix-command)"
  (interactive "p")
  (if (not yank-type) (setq yank-type 'yank))
  (setq arg (or arg 1))
  (catch 'DONE
    (let* ((k-ring 
            (cond ((eq yank-type 'yank)                'kill-ring)
                  ((eq yank-type 'yank-prev-command)
                   (if (equal (apollo:mode) "common-lisp")
                     (let (kr kryp)
                       (with-buffer-set (apollo:lisp-buffer)
                         (setq kr apollo:kill-ring kryp apollo:kill-ring))
                       (setq apollo:kill-ring kr apollo:kill-ring kryp)))
                   'apollo:kill-ring)
                  ((eq yank-type 'apollo:yank-command) 
                   (if (not pop)
                     (setq apollo:command-kill-ring (mapcar 'prin1-to-string command-history)))
                   'apollo:command-kill-ring)))
           (k-ring-yank-pointer 
            (cond ((eq yank-type 'yank)                'kill-ring-yank-pointer)
                  ((eq yank-type 'yank-prev-command)   'apollo:kill-ring-yank-pointer)
                  ((eq yank-type 'apollo:yank-command) 'apollo:command-kill-ring-yank-pointer)))
           (mode   (if (eq yank-type 'apollo:yank-command) "emacs-lisp" (apollo:mode)))
           (kill-ring   (eval k-ring))
           (kryp (eval k-ring-yank-pointer))
           (result nil))
      (if (eq arg 0)
          (apollo:lines-to-buffer 
           apollo:kill-history-buffer 
           (concat (capitalize (symbol-name yank-type)) " Kill History:")
           'refresh kill-ring

           ;;Below References external symbol N set up in apollo:lines-to-buffer.
           (function (lambda (s) (concat "[" (format "%s" (1+ n)) "]: " s "\n")))
           mode)
          (unwind-protect
               (let* ((kill-ring-yank-pointer (if pop kryp kill-ring))
                      (lkr   (length kill-ring))
                      (inc   (if (> arg 0) 1 (- lkr 1)))
                      (okryp kill-ring-yank-pointer))
                 (apollo:dotimes (n (abs arg))
                   (if pattern
                       (or 
                        (apollo:dotimes (m lkr)
                          (rotate-yank-pointer inc)
                          (if (and kill-ring-yank-pointer
                                   (string-match pattern (car kill-ring-yank-pointer)))
                              (apollo:return 'pattern-found)))
                        (throw 'DONE nil))
                       (rotate-yank-pointer inc)))
                 (if pop 
                     (let ((last-command 'yank)) (yank-pop 0)) 
                     (yank (if pattern 1 (if (< arg 0) 1 0))))
                 (setq result t kryp kill-ring-yank-pointer))
            (progn (setq this-command yank-type) (set k-ring-yank-pointer kryp))))
      result)))

(defun yank-prev-command (arg &optional pop pattern)
  "Yank the previous command.


        If ARG is

                0               Display the command history in a buffer

                n               With numeric argument n, retrieve the nth

                                most recent shell command (default = 1).

        Optional arguments are:

                POP     If POP has a non-nil value, replace the previ-

                                ously yanked command with the next most

                                recent one (see apollo:yank-pop)

                PATTERN This is a string specifying the prefix criteria for

                                a match (see yank-prefix-command)


        If the current buffer is in Common-Lisp mode and an inferior lisp
is running, the command yanks strings from the input command history of the
inferior lisp buffer."
  (interactive "p")
  (apollo:yank arg apollo:yank-type pop pattern))


(defun yank-prefix-command (arg &optional pop)
  "Invoke yank-prev-command and use the current line as a pattern; in
other words, yank commands that match the current command.


        If ARG is n, retrieve the nth matching command.

        If POP has a non-nil value, replace the previously yanked command with
the next most recent one (see apollo:yank-pop)."
  (interactive "p")
  (if (not pop)
      (if (apollo:minibuffer-p)
          (let ((minibuffer (window-buffer (minibuffer-window)))) 
            (with-buffer-set minibuffer 
              (setq apollo:pattern (buffer-string))
              (delete-region (point-min) (point-max))))
          (let ((prompt (cond ((eq major-mode 'inferior-lisp-mode) inferior-lisp-prompt)
                              ((eq major-mode 'shell-mode)         shell-prompt-pattern)
                              (t "^"))))
            (save-excursion
              (set-mark (point))
              (if (re-search-backward prompt nil t)
                  (progn
                    (re-search-forward prompt nil t)
                    (setq apollo:pattern (buffer-substring (mark) (point)))
                    (delete-region (mark) (point)))
                  (setq apollo:pattern nil))))))
  (if (not (yank-prev-command 
            arg pop 
            (if apollo:pattern 
                (concat "^" apollo:whitespace "*" (regexp-quote apollo:pattern)))))
      (progn (insert apollo:pattern) (message "Pattern Not Found: %s" apollo:pattern))
      (setq this-command 'yank-prefix-command)))


(defun apollo:yank-pop (arg)
  "Replace the text of the most recently yanked kill with the specified
text.  If nothing is specified, reinsert the text displaced by the yanked text.

        This command must execute immediately after a yank, apollo:yank,
yank-prev-command, yank-prefix-command, or apollo:yank-pop command; otherwise,
the command performs a yank-prev-command.  Immediately after one of these yank
commands, a marked region exists containing the text of a reinserted kill; the
apollo:yank-pop command deletes the text in the marked region and reinserts an
earlier kill.  Note that the newest kill comes after the oldest kill in the kill
ring.


        If ARG is

        n               With the positive numeric argument n, insert the text of

                        the nth previous kill

        -n              With the negative numeric argument n, insert the nth more

                        recent kill"
  (interactive "*p")
  (cond ((memq last-command '(yank yank-prev-command apollo:yank-command))
         (apollo:yank arg last-command 'pop))
        ((eq last-command 'yank-prefix-command)
         (yank-prefix-command arg 'pop))
        (t (yank-prefix-command arg))))

(defun apollo:yank-pop-negative (arg) 
  "Call apollo:yank-pop with ARG of -n."
  (interactive "p") 
  (apollo:yank-pop (- arg)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Completion commands

(defconst apollo:next-path
    (format "^\\([^%s]*\\)[%s]?" 
            apollo:find-files-shell-separator apollo:find-files-shell-separator))

(defun apollo:shell-complete-symbol (arg)
  "Use the command string in apollo:find-files-shell-command to complete
the shell command located in the buffer whose process name begins with
\"shell.\"

        In a UNIX shell, complete the partial command by comparing it with the
command names in the directories given by the shell variable path.  In an Aegis
shell, complete the partial command by comparing it with the command names in
the directories given by the csr command.

        With an ARG, GNU Emacs searches for a command that contains the string
to be completed anywhere.  Without an ARG, GNU Emacs searches only for commands
that begin with the typed string."
  (interactive "P")
  (if (string-match "shell" 
                    (process-name (get-buffer-process (current-buffer))))
      (let ((c (apollo:current-thing nil (not 'sexps-ok)))
            o)
        (if c
            (let ((p (get-buffer-process (current-buffer))) 
                  (s nil)
                  (uc (regexp-quote c))
                  (l nil)
                  (aegis-path nil))
              (message "Waiting For Shell")
              (with-process-filter 
                  (p (function (lambda (pp ps) (setq s (concat s ps)))))
                (process-send-string p "echo environment PATH = $PATH\n")
                (accept-process-output p)
                (if (eq 0 (string-match "\?(sh) " s))
                    (progn
                      (setq s nil aegis-path t)
                      (process-send-string p "csr\n")
                      (accept-process-output p)
                      (string-match "^\\(.*\\)$" s)
                      ))
                (if (or (and aegis-path (not s))
                        (and (not aegis-path) 
                             (not (string-match "environment PATH = \\(.*\\)$" s))))
                    (error "Could not find PATH variable in shell")
                    (let ((dirs "")
                          end dir)
                      (setq l (apollo:match 1 s))
                      (while (> (length l) 0)
                        (string-match apollo:next-path l)
                        (setq dir (apollo:match 1 l))
                        (setq l (substring l (match-end 0)))
                        (if (file-exists-p (concat dir "/"))
                            (setq dirs 
                                  (concat " " dir "/" 
                                          (if (and arg aegis-path) "?")
                                          (if arg "*")
                                          c 
                                          (if aegis-path "?") "*" dirs))))
                      (setq l (concat apollo:find-files-shell-command dirs "\n"))
                      (setq s "")
                      (process-send-string p l)
                      (accept-process-output p)
                      (while (eq (aref s (1- (length s))) ?\n) (accept-process-output p))
                      (setq l nil)
                      (while (string-match "\\(.*\\)\n" s)
                        (let ((f (file-name-nondirectory (apollo:match 1 s))))

                          (if (and (not (assoc f l)) 
                                   (string-match uc f)
                                   (or arg (eq 0 (match-beginning 0))))
                              (push (list f) l)))
                        (setq s (substring s (match-end 0)))))))
              (setq o (sort l (function (lambda (a b) 
                                (string< (car a) (car b))))))))
        (apollo:get-completion 
         arg o "Shell" c (null 'mode) (null 'package)
         'apollo:grab-thing-dynamic-mouse)
        (message "Shell Done"))
      (lisp-complete-symbol)))


(defun apollo:lisp-complete-symbol (arg &optional symbol)
  "Complete the partially typed Lisp symbol preceding point.

        If the incomplete symbol immediately follows an open parenthesis,
consider only existing symbols with function definitions.  Otherwise, consider
all existing symbols regardless of function definitions, values, or properties.

        If the buffer is in Lisp or Inferior-Lisp mode, query the Common Lisp
process for completion using the apropos command.

        Use a package name preceding the incomplete symbol name as the package
name for Common Lisp symbols.  If no preceding package name exists, search
backwards in the current buffer for \"in-package\" at the beginning of a line.
If the search fails, use the package name from the buffer's mode line.

        With ARG, a match can occur anywhere within the symbol's name.  Without
ARG, a match can occur only at the beginning of the name of an existing symbol.

        When calling from Emacs Lisp, you can substitute an optional SYMBOL for
the one at point."
  (interactive "P")
  (if (apollo:common-lisp-buffer-p)
      (let ((curr (or symbol (apollo:current-thing nil (not 'sexps-ok)))))
        (if (not curr)
            (error "No lisp symbol to complete")
            (let* ((cr (upcase curr))
                   (c  (apollo:strip-package cr))
                   (f  (save-excursion 
                         (goto-char (- apollo:mark 1))
                         (looking-at "(")))
                   (e  (or (apollo:extract-package cr) apollo:temp-package))
                   (s  (format 
                        "(APOLLO::EMACS-LISP-COMPLETE-SYMBOL %s \"%s\" %s %s)"
                        f c (or (if e (concat "'" e)) "*PACKAGE*") arg))
                   (o  (evaluate-common-lisp nil s)))
              (save-excursion 
                (setq apollo:point (point))
                (if (re-search-backward (concat "[:('#]\\|" apollo:whitespace) 
                                        nil t)
                    (setq apollo:mark (+ (point) 1))
                    (progn (re-search-backward "^" nil t) 
                           (setq apollo:mark (point)))))
              (apollo:get-completion
                arg (if (eq o 'NIL) nil o) 
                "Common Lisp Apropos" c "common-lisp" (or e (apollo:find-package)) 
                'apollo:grab-thing-dynamic-mouse))))
      (lisp-complete-symbol)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Document Processing and Print commands

(defun delete-all-font-info ()
  ;; this function courtesy of TMC.  I havent tried it yet.
  "Remove all font information (such as lisp-machine font control
characters) from the current buffer."
  (interactive)
  (save-excursion
    (goto-char (point-min))
    (while (if (search-forward "\^F" nil t)
               (progn
                 (cond ((looking-at "\^E")
                        (delete-region (1- (point))
                                       (progn (search-forward "\^F" nil t)
                                              (goto-char (1- (point))))))
                       ((looking-at "(")
                        (delete-region (1- (point))
                                       (progn (forward-sexp 1)
                                              (point))))
                       ((looking-at "[0123456789]")
                        (delete-region (1- (point)) (1+ (point))))
                       (t t))
                 t)))))

(defun fix-man-output ()
  "Adjust the output from the UNIX man command to improve readability.
The man command must be the last command executed."
  (interactive)
  (if (string-match "\\(^\\|/\\)man" 
                    (buffer-substring last-input-start last-input-end))
      (save-excursion
        (goto-char last-input-end)
        (replace-regexp "\10." ""))
      (error "Man must have been the last shell command")))

(defun insert-date ()
  "Insert the current date and time at point in UNIX format (for
example, \"Thu Dec 8 00:29:46 EST 1988\")."
  (interactive)
  (insert (current-time-string)))

(defvar apollo:page-headers-p t)

(defun pagify-lisp-buffer (arg)
  "Insert a formfeed into the current buffer at line intervals
specified by the configuration variable apollo:page-size.  If ARG is specified
as the numeric argument n, place the formfeeds at line intervals of n.  Remove
lines consisting only of formfeeds.

        Create a new page wherever the characters \";;;\" appear at the
beginning of a line.  However, when the characters \";;;\" appear at the
beginning of subsequent and consecutive lines, then paginate according to the
prevailing page size."
  (interactive "P")
  (untabify (point-min) (point-max))
  (if (not arg) (setq arg apollo:page-size))
  (save-excursion
    (goto-char (point-min))
    (replace-regexp "^\f\n" "")
    (goto-char (point-min))
    (let (f c)
      (while (not (eobp))
        (goto-char 
         (let* 
             ((s (save-excursion (beginning-of-line) (+ 1 (point))))
              (p (save-excursion 
                   (goto-char s) 
                   (if (< 0 (forward-line arg))
                       (point-max)
                       (progn (beginning-of-line) (point)))))
              (b (save-excursion 
                   (goto-char p)
                   (if (eobp) 
                       (point-max)
                       (progn (beginning-of-defun) (point)))))
              (e (save-excursion 
                   (goto-char b)
                   (end-of-defun)
                   (beginning-of-line) (point)))
              (c (save-excursion 
                   (goto-char s)
                   (if (re-search-forward "^;;;" e t)
                       (progn (beginning-of-line) (point))
                       (point-max)))))
             (min (if (<= b s) e b) c p)))
        (if (not (eobp)) (insert "\f\n"))
        (let ((n 0))
          (while (and (looking-at "^;") (= 0 (forward-line 1)))
            (beginning-of-line)
            (if (>= (setq n (+ n 1)) arg)
                (progn (setq n 0) (insert "\f\n"))))))))
  (sectionize-buffer))


(defun print-buffer-apollo (arg)
  "Send the contents of the current buffer to the last printer used or
as specified by ARG.


        If ARG is

        1       Prompt for the printer's name and change the default printer

                accordingly

        2       Print the document in 2-column format on the default

                printer

        Pass the arguments, if any, in the variable apollo:printer-args to the
printer.  Use the default printer specified in the variable
apollo:default-printer."
  (interactive "P")
  (let ((curr (current-buffer))
        (lisp-buff-p (and apollo:*pagify-lisp-buffer-p* (apollo:lisp-buffer-p))))
    (with-buffer-set (get-buffer-create apollo:print-buffer)
      (erase-buffer)
      (insert-buffer curr)
      (if lisp-buff-p;; this stuff should be made to work for all modes
          (progn
            (pagify-lisp-buffer (- apollo:page-size 2))
            ))
      (while (or (eq arg 1) (zerop (length apollo:default-printer)))
        (setq apollo:default-printer
              (apollo:prompt-for 
               "What printer do you want to send this to?"
               apollo:default-printer)
              arg nil))
      (apollo:APOLLO-print-buffer arg)
      (message "Buffer Sent to printer %s" apollo:default-printer))))


(defun print-region-apollo (arg)
   "If ARG is not specified or is nil, send the text in the marked
region to the last printer used.

        If ARG is

        1       Prompt for the printer's name and change the default

                printer accordingly

        2       Print the document in 2-column format on the default

                printer

        Pass the arguments, if any, in the variable apollo:printer-args to the
printer.  Use the default printer specified in the variable
apollo:default-printer."
   (interactive "P")
   (save-restriction
     (narrow-to-region (point) (mark))
     (print-buffer-apollo arg)
     (message "Region Sent to printer %s" apollo:default-printer)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Miscellaneous commands

(defun list-notifications (arg)
  "Display a buffer containing all messages from the error and message
functions.  If ARG is the numeric argument n, redisplay the first line of the
nth message and set apollo:last-notification to that string."
  (interactive "P")
  (if (and (integerp arg) (> arg 0))
      (with-buffer-set apollo:notifications-buffer
        (goto-char (point-min)) 
        (forward-line (1- arg))
        (princ (setq apollo:last-notification (buffer-substring (point) (eol)))))
      (let* ((b (get-buffer-create apollo:notifications-buffer))
             (w (display-buffer b)))
        (if w
            (with-buffer-set b
              (set-window-start w 1)
              (set-window-point w 1)
              (set-mark 1)
              (goto-char 1))))))

(defun apollo:redisplay-function ()
  "Move the beginning of the current definition to the top of the
window."
  (interactive)
  (forward-char)
  (if (apollo:lisp-buffer-p) (beginning-of-defun) (beginning-of-line))
  (recenter 0))


(defun apollo:switch-to-buffer (&optional b norecord)
  "Select BUFFER (or its name) for display in the current window.  If
the value of the optional argument NORECORD is non-nil, do not put this buffer
at the front of the list of recently selected buffers.


NOTE:   Use set-buffer to temporarily work on another buffer within a
Lisp program.  Using the apollo:switch-to-buffer command can disrupt
window/buffer correspondence.


        If no BUFFER is nil, the default action is to switch to the other
buffer designated by other-buffer.

        If ARG is

        0       Display a buffer list

        1       Put the name of the current buffer at the bottom of the

                buffer list

        n       With other numeric argument n, switch to the nth buffer in

                the history

        Because the argument 0 does not complete the buffer name upon return,
it's useful for creating new buffers with names that partially match the names
of existing buffers.

        If no argument, or an argument other than 0, is specified, try to
switch to existing buffers.  Prompt for the name of the buffer to select.  If a
partial buffer name is specified and it matches only one buffer, then select
that buffer.  If a partial buffer name is specified and it matches multiple
buffer names, then display a completion menu of the buffer names, or create a
buffer with the partial name when <RETURN> is pressed.

        If creating a Common-Lisp or Emacs-Lisp buffer (.lisp or .el
extension), prompt for an attribute list for the file.

        With C-y command, bring the default buffer name temporarily inserted
into the kill ring onto the prompt line."
  (interactive "P")
  (let* ((b-list (buffer-list))
	 (original-b (current-buffer))
	 new-window
	 o
	 (bnames-func (function (lambda (b) (list (buffer-name b)))))
	 (l (length b-list)))
    (if (integerp b)
        (cond
          ((> b 1)
           (switch-to-buffer 
            (apollo:dolist (c b-list)
              (if (eq (aref (buffer-name c) 0) ?\ ) nil
                  (if (zerop (apollo:decf b)) (apollo:return c))))))
          ((= b 1) (bury-buffer (current-buffer)))
          ((< b 1)
           (apollo:lines-to-buffer 
            apollo:completions-buffer "Buffers:" 'refresh
            (all-completions 
	      "" 
	      (or o (setq o (mapcar bnames-func b-list))))
	    nil nil nil 'apollo:grab-thing-dynamic-mouse)
           (setq apollo:mark 1 apollo:point 1)
           (switch-to-buffer 
            (apollo:read-buffer "Switch to Buffer: " (buffer-name (other-buffer))))))
        (let ((old-b nil)

              (w (current-window-configuration)))
          (while (and (not b) (not new-window))
            (setq b
                  (if (and old-b (not (equal old-b "")))
                      (let ((in-buff 
                             (apollo:read-buffer 
                               (concat (ding) "Switch to Buffer:(Create [" old-b "]) ")
                               nil
                               old-b)))
                        (if (and in-buff (not (equal in-buff old-b)))
                            in-buff
                            (let ((buff-name 
                                   (buffer-name (get-buffer-create old-b))))
                              (cond ((not (get-buffer buff-name)) nil)
                                    ((string-match apollo:elisp-file-pattern 
                                                   buff-name)
                                     (with-buffer-set buff-name 
                                       (set-mode "emacs-lisp")))
                                    ((string-match apollo:clisp-file-pattern 
                                                   buff-name)
                                     (with-buffer-set buff-name 
                                       (set-mode "common-lisp-mode"))))
                              buff-name)))
                      (apollo:read-buffer "Switch to Buffer: " (buffer-name (other-buffer)))))
            (setq apollo:mark 1 apollo:point 1)
            (if (and b (not (get-buffer b)))
                (if (and old-b (equal b "") (not (equal old-b "")))
                    (setq b old-b)
                    (let ((completions (all-completions b (or o (setq o (mapcar bnames-func b-list))))))
		      (apollo:dolist (b (buffer-list))
			(if (memq (buffer-name b) completions)
			    (progn
			      (if (and (not new-window) (not (eq b original-b)) (get-buffer-window b))
				  (setq new-window b completions (delq (buffer-name b) completions)))
			      (if (or (eq b original-b) (get-buffer-window b))
				  (setq completions (delq (buffer-name b) completions))))))
                      (cond (completions (setq b (get-buffer (car completions)) new-window nil))
			    (new-window t)
			    (t (setq old-b b b nil)))))))
          (set-window-configuration w)
          (if (get-buffer-window apollo:completions-buffer) 
              (bury-buffer apollo:completions-buffer))
          (if new-window (select-window (get-buffer-window new-window)) (switch-to-buffer b))))))

(defun grep-to-temp-buffer ()
  "Prompt for the Emacs regular expression regexp and copy any matching
lines to a temporary buffer."
  (interactive)
  (let ((str (read-string "Regular Expr: "))
        (b apollo:grep-buffer))
    (save-excursion
      (set-mark (point-max))
      (if (not (re-search-forward str nil t))
          (message (concat "String Not Found: " str (ding)))
          (with-output-to-temp-buffer b
            (apollo:matching-lines-to-buffer str b))))))


(defun apollo:beginning-of-line ()
  "Go to the beginning-of-line.  If used in an inferior Lisp or shell
process, this command goes to the prompt if the line starts with a prompt
instead of the beginning-of-line."
  (interactive "*")
  (beginning-of-line)
  (let ((prompt (if (eq major-mode 'inferior-lisp-mode)
                    inferior-lisp-prompt
                    (if (eq major-mode 'shell-mode)
                        shell-prompt-pattern))))
    (if (and prompt (looking-at prompt))
      (goto-char (match-end 0)))))

(defun apollo:mouse-insert-thing (&optional arg regexp no-sexps no-spacing-p)
  "Take the s-expression under the mouse cursor and insert it at point.
If the mouse is between two s-expressions, the one on the left of point always
prevails over one to the right."
  (interactive)
  (apollo:mouse-no-op)
  (save-excursion
    (save-window-excursion
      (let* ((apollo:dynamic-mouse-binding 
	      (function (lambda (&optional arg)
		(save-excursion (apollo:mouse-thing args regexp (not no-sexps)))))))
	(insert 
	  (concat 
	    (if (and (not no-spacing-p)
		     (apollo:extended-mouse-insert-alphanumericp 
		       (char-after (- (point) 1))))
		" ")
	    (save-excursion (save-window-excursion (apollo:dynamic-mouse-call arg)))
	    (if (and (not no-spacing-p) 
		     (apollo:extended-mouse-insert-alphanumericp (char-after (point))))
		" ")))))))

(defun apollo:mouse-move-mark (&optional args)
  "Pressing M-<M1> (holding the Meta key down while pressing the left
mouse button) sets a mark; releasing the button sets point.  In this way, you
can use the mouse to mark the beginning of a region and set point at the
region's end."
  (interactive)
  (let (m)
    (save-excursion
      (apollo:mouse-move-point args)
      (setq m (point)))
    (set-mark (point))
    (goto-char m)))


(defun apollo:grab-thing-dynamic-mouse ()
  "Insert completion at point.
       This is designed to be used by some completion commands
       to let clicking right on the mouse, grab the completion.
       If pointing is at end of thing to be completed, the partial
       string will be replaced with the full completion clicked on."
  (interactive)
  (let* ((b (current-buffer))
         (s (or (let (apollo:mark apollo:point) (apollo:current-thing nil nil)) ""))
         (ok (save-excursion 
               (let ((p (point)))
                 (goto-char (point-min))
                 (forward-line 2)
                 (>= p (point))))))
    (if (eq apollo:old-buffer b)
        (error "Move Point to where completion is to be inserted"))
    (select-window apollo:old-window)
    (with-buffer-set apollo:old-buffer
      (goto-char apollo:old-point)
      (if (equal (buffer-name (current-buffer)) apollo:completions-buffer)
          (error "Move Point to other buffer to grab completion")
          (if ok
              (if (eq (point) apollo:point)
                  (progn
                    (delete-region apollo:mark apollo:point)
                    (insert s)
                    (setq apollo:point (point)))
                  (insert s))
              (progn
                (message "Click on item to select completion")
                (ding)))))))

;;; End of Interactive Stuff
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Mode Specific Key Bindings
;; Where possible, key bindings are on C-c 
;; unless compatabibility with LISPM is an issue

(defun define-inferior-lisp-mode-keys ()
  (define-key inferior-lisp-mode-map   "\M-p"       'apollo:yank-pop)
  (define-key inferior-lisp-mode-map   "\M-n"       'apollo:yank-pop-negative)
  (define-key inferior-lisp-mode-map   "\C-m"       'shell-send-input)
  (define-key inferior-lisp-mode-map   "\C-c\e"     'evaluate-common-lisp)
  (define-key inferior-lisp-mode-map   "\C-a"       'apollo:beginning-of-line)
  (define-key inferior-lisp-mode-map   "\C-c\C-e"   'edit-callers)
  (define-key inferior-lisp-mode-map   "\C-c\C-x"   'evaluate-region-hack)
  (define-key inferior-lisp-mode-map   "\C-c."      'edit-next-definition)
  (define-key inferior-lisp-mode-map   "\C-c\C-c"   'apollo:break)
  (define-key inferior-lisp-mode-map   "\C-c\C-a"   'quick-arglist)
  (define-key inferior-lisp-mode-map   "\C-cd"      'show-lisp-documentation)
  (define-key inferior-lisp-mode-map   "\C-cf"      'describe-function-at-point)
  (define-key inferior-lisp-mode-map   "\C-cm"      'macro-expand-expression)
  (define-key inferior-lisp-mode-map   "\C-c\C-m"   'macro-expand-last-sexp)
  (define-key inferior-lisp-mode-map   "\C-cn"      'next-caller)
  (define-key inferior-lisp-mode-map   "\C-cv"      'describe-variable-at-point)
  (define-key inferior-lisp-mode-map   "\C-cw"      'where-is-symbol)
  (define-key inferior-lisp-mode-map   "\C-cx"      'evaluate-region)
  (define-key inferior-lisp-mode-map   "\e."        'apollo:key-find-source-code)
  (define-key inferior-lisp-mode-map   "\e\C-x"     'apollo:lisp-send-defun)
  (define-key inferior-lisp-mode-map   "\C-x\C-e"   'apollo:evaluate-last-sexp)
  (define-key inferior-lisp-mode-map   "\e\t"       'apollo:lisp-complete-symbol)
  (define-key inferior-lisp-mode-map   "\e\C-c"     'apollo:lisp-compile-defun)
  (define-key inferior-lisp-mode-map   "\C-c\C-g"   'apollo:lisp-server-interrupt)
  (apollo:inferior-lisp-bind-mouse-keys))

( define-key lisp-mode-map             "\e\C-x"     'apollo:lisp-send-defun)
( define-key emacs-lisp-mode-map       "\e\C-x"     'apollo:lisp-send-defun)
( define-key lisp-interaction-mode-map "\e\C-x"     'apollo:lisp-send-defun)
( define-key lisp-mode-map             "\C-x\C-e"   'apollo:evaluate-last-sexp)
( define-key emacs-lisp-mode-map       "\C-x\C-e"   'apollo:evaluate-last-sexp)
( define-key lisp-interaction-mode-map "\C-x\C-e"   'apollo:evaluate-last-sexp)
( define-key lisp-mode-map             "\e\C-c"     'apollo:lisp-compile-defun)
( define-key emacs-lisp-mode-map       "\e\C-c"     'apollo:lisp-compile-defun)
( define-key lisp-interaction-mode-map "\e\C-c"     'apollo:lisp-compile-defun)
( define-key lisp-mode-map             "\e\C-r"     'apollo:redisplay-function)
( define-key emacs-lisp-mode-map       "\e\C-r"     'apollo:redisplay-function)
( define-key lisp-mode-map             "\e."        'apollo:key-find-source-code)
( define-key emacs-lisp-mode-map       "\e."        'apollo:key-find-source-code)
( define-key lisp-interaction-mode-map "\e."        'apollo:key-find-source-code)
( define-key lisp-mode-map             "\C-c\C-a"   'quick-arglist)
( define-key emacs-lisp-mode-map       "\C-c\C-a"   'quick-arglist)
( define-key lisp-interaction-mode-map "\C-c\C-a"   'quick-arglist)
( define-key lisp-mode-map             "\C-cv"      'describe-variable-at-point)
( define-key lisp-mode-map             "\C-cf"      'describe-function-at-point)
( define-key lisp-mode-map             "\C-cw"      'where-is-symbol)
( define-key lisp-mode-map             "\C-c."      'edit-next-definition)
( define-key emacs-lisp-mode-map       "\C-c."      'edit-next-definition)
( define-key lisp-interaction-mode-map "\C-c."      'edit-next-definition)
( define-key lisp-mode-map             "\C-cd"      'show-lisp-documentation)
( define-key lisp-mode-map             "\C-c\C-e"   'edit-callers)
( define-key lisp-mode-map             "\C-c\C-x"   'evaluate-region-hack)
( define-key emacs-lisp-mode-map       "\C-c\C-x"   'evaluate-region-hack)
( define-key lisp-interaction-mode-map "\C-c\C-x"   'evaluate-region-hack)
( define-key emacs-lisp-mode-map       "\C-cm"      'macro-expand-expression)
( define-key emacs-lisp-mode-map       "\C-c\C-m"   'macro-expand-last-sexp)
( define-key lisp-interaction-mode-map "\C-cm"      'macro-expand-expression)
( define-key lisp-interaction-mode-map "\C-c\C-m"   'macro-expand-last-sexp)
( define-key lisp-mode-map             "\C-cm"      'macro-expand-expression)
( define-key lisp-mode-map             "\C-c\C-m"   'macro-expand-last-sexp)
( define-key lisp-mode-map             "\C-cn"      'next-caller)

( define-key emacs-lisp-mode-map       "\C-cn"      'next-caller)
( define-key lisp-interaction-mode-map "\C-cn"      'next-caller)
( define-key lisp-mode-map             "\C-cx"      'evaluate-region)
( define-key emacs-lisp-mode-map       "\C-cx"      'evaluate-region)
( define-key lisp-interaction-mode-map "\C-cx"      'evaluate-region)
( define-key lisp-mode-map             "\e\t"       'apollo:lisp-complete-symbol)
( define-key lisp-mode-map             "\C-c\e"     'evaluate-common-lisp)
( define-key emacs-lisp-mode-map       "\C-c\e"     'evaluate-emacs-lisp)
( define-key lisp-interaction-mode-map "\C-c\e"     'evaluate-emacs-lisp)

( define-key lisp-mode-map             "\C-c\C-g"   'apollo:lisp-server-interrupt)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Global Key bindings
;; Where possible, key bindings are mode specific.  In this case,
;; yank commands are more lisp machine like, mouse commands are
;; pervasive, and file name completion is so universally useful
;; that these are made global.
(global-set-key                       "\C-c\t"      'shell-file-name-completion)

;; Yanking keys

(global-set-key                       "\C-y"        'apollo:yank)
(global-set-key                       "\e\C-y"      'yank-prev-command)
(global-set-key                       "\M-y"        'apollo:yank-pop)
(global-set-key                       "\e\C-p"      'yank-prefix-command)
(global-set-key                       "\M-p"        'apollo:yank-pop)
(global-set-key                       "\M-n"        'apollo:yank-pop-negative)

;; The following calls a fancier version of switch-to-buffer
;; It automatically completes buffer names and allows
;; you to select buffers by number.  I probably shouldn't
;; redefine this, as it is just a matter of taste, but I
;; like it enough more, so that I'll redefine this key anyway...

(global-set-key  "\C-xb" 'apollo:switch-to-buffer)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; Mouse key bindings
(defun apollo:mouse-no-op (&rest ignore) 
  (interactive) 
  "No Op"
  (let (term)
    (cond ((memq window-system '(x x11)) nil)
	  ((and (setq term (getenv "TERM"))
		(setq term (assq (intern term) apollo:terminals)))
	   (autoload (car term) (cdr term))
	   (apollo:terminal-no-op))
	  ((apollo:apollo-p) (read-char) (read-char)))))

(defvar apollo:mouse-buttons
  (cond ((memq window-system '(x x11))
         (require 'x-mouse)
         (list
           x-button-left x-button-middle x-button-right
           x-button-left-up x-button-middle-up x-button-right-up
           x-button-m-left x-button-m-middle x-button-m-right
           x-button-m-left-up x-button-m-middle-up x-button-m-right-up))
        ((apollo:apollo-p)
         (list
           (apollo:mouse-string ?a) (apollo:mouse-string ?b) (apollo:mouse-string ?c)
           (apollo:mouse-string ?A) (apollo:mouse-string ?B) (apollo:mouse-string ?C)
           (apollo:mouse-string ?a 'meta-p) (apollo:mouse-string ?b 'meta-p) (apollo:mouse-string ?c 'meta-p)
           (apollo:mouse-string ?A 'meta-p) (apollo:mouse-string ?B 'meta-p) (apollo:mouse-string ?C 'meta-p)))
        (t (error "Unknown window system for key bindings %s" window-system))))

(defun apollo:mouse (button direction &optional modifier)
  (let ((position nil))
    (setq position (cdr (assoc button '((left . 0) (middle . 1) (right . 2)))))
    (if (eq direction 'up) (apollo:incf position 3))
    (if (eq modifier 'meta) (apollo:incf position 6))
    (nth position apollo:mouse-buttons)))

(defun apollo:bind-mouse (map key binding)
  (if (memq window-system '(x x11)) (setq map mouse-map))
  (define-key map key binding))

(defun apollo:inferior-lisp-bind-mouse-keys ()
  (let ((inferior-lisp-mouse-bindings
         '(lambda ()
           (apollo:bind-mouse inferior-lisp-mode-map (apollo:mouse 'middle 'up) 'apollo:mouse-find-source-code)
           (apollo:bind-mouse inferior-lisp-mode-map (apollo:mouse 'middle 'down) 'apollo:mouse-no-op))))
    (funcall inferior-lisp-mouse-bindings)
    (if (not (eq apollo:key-bindings-forms ':DONE))
        (push inferior-lisp-mouse-bindings apollo:key-bindings-forms))))

(if (not (eq apollo:key-bindings-forms ':DONE))
    (push 
      '(lambda ()
        (apollo:bind-mouse global-map (apollo:mouse 'left   'up)         'apollo:mouse-move-mark)
        (apollo:bind-mouse global-map (apollo:mouse 'left   'down)       'apollo:mouse-move-point)
        (apollo:bind-mouse global-map (apollo:mouse 'right  'up)         'apollo:mouse-no-op)
        (apollo:bind-mouse global-map (apollo:mouse 'right  'down)       'apollo:dynamic-mouse-call)
        (apollo:bind-mouse global-map (apollo:mouse 'middle 'down 'meta) 'apollo:mouse-insert-thing)
        (apollo:bind-mouse global-map (apollo:mouse 'middle 'up   'meta) 'apollo:mouse-no-op)
        (apollo:bind-mouse global-map (apollo:mouse 'middle 'down)       'apollo:mouse-no-op)
        (apollo:bind-mouse global-map (apollo:mouse 'middle 'up)         'apollo:mouse-no-op)
        (apollo:bind-mouse lisp-mode-map             (apollo:mouse 'middle 'up) 'apollo:mouse-find-source-code)
        (apollo:bind-mouse lisp-mode-map             (apollo:mouse 'middle 'down) 'apollo:mouse-no-op)
        (apollo:bind-mouse emacs-lisp-mode-map       (apollo:mouse 'middle 'up) 'apollo:mouse-find-source-code)
        (apollo:bind-mouse emacs-lisp-mode-map       (apollo:mouse 'middle 'down) 'apollo:mouse-no-op)
        (apollo:bind-mouse lisp-interaction-mode-map (apollo:mouse 'middle 'up) 'apollo:mouse-find-source-code)
        (apollo:bind-mouse lisp-interaction-mode-map (apollo:mouse 'middle 'down) 'apollo:mouse-no-op))
      apollo:key-bindings-forms))

(defvar *OLD-apollo-key-bindings-hook* term-setup-hook)


(if (consp apollo:key-bindings-forms)
    (setq term-setup-hook
          (list 'lambda '() 
                (let ((f (list 'quote 
                               (append apollo:key-bindings-forms 
                                       (if *OLD-apollo-key-bindings-hook*
                                           (list *OLD-apollo-key-bindings-hook*)))))) 
                  (setq apollo:key-bindings-forms ':DONE)
                  (list 'mapcar (list 'quote 'funcall) f)))))

(if term-setup-hook (funcall term-setup-hook))


