;; Calculator for GNU Emacs version 1.04, part II
;; Copyright (C) 1990 Dave Gillespie

;; This file is part of GNU Emacs.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.


;;;; [calc-ext.el]

(provide 'calc-ext)

(setq calc-extensions-loaded t)

;;; This function is the autoload "hook" to cause this file to be loaded.
(defun calc-extensions ()
  t
)

;;; Auto-load part I, in case this part was loaded first.
(if (fboundp 'calc)
    (and (eq (car-safe (symbol-function 'calc)) 'autoload)
	 (load (nth 1 (symbol-function 'calc))))
  (error "Main part of Calc must be present in order to load this file."))

;;; If the following fails with "Cannot open load file: calc"
;;; do "M-x load-file calc.elc" before compiling calc-ext.el.
(require 'calc)  ;;; This should only occur in the byte compiler.



;;; The following was made a function so that it could be byte-compiled.
(defun calc-init-extensions ()

  (define-key calc-mode-map ":" 'calc-fdiv)
  (define-key calc-mode-map "\\" 'calc-idiv)
  (define-key calc-mode-map "|" 'calc-concat)
  (define-key calc-mode-map "!" 'calc-factorial)
  (define-key calc-mode-map "A" 'calc-abs)
  (define-key calc-mode-map "B" 'calc-log)
  (define-key calc-mode-map "C" 'calc-cos)
  (define-key calc-mode-map "D" 'calc-redo)
  (define-key calc-mode-map "E" 'calc-exp)
  (define-key calc-mode-map "F" 'calc-floor)
  (define-key calc-mode-map "G" 'calc-argument)
  (define-key calc-mode-map "H" 'calc-hyperbolic)
  (define-key calc-mode-map "I" 'calc-inverse)
  (define-key calc-mode-map "J" 'calc-conj)
  (define-key calc-mode-map "K" 'calc-call-last-kbd-macro)
  (define-key calc-mode-map "L" 'calc-ln)
  (define-key calc-mode-map "M" 'calc-more-recursion-depth)
  (define-key calc-mode-map "N" 'calc-eval-num)
  (define-key calc-mode-map "P" 'calc-pi)
  (define-key calc-mode-map "Q" 'calc-sqrt)
  (define-key calc-mode-map "R" 'calc-round)
  (define-key calc-mode-map "S" 'calc-sin)
  (define-key calc-mode-map "T" 'calc-tan)
  (define-key calc-mode-map "U" 'calc-undo)
  (define-key calc-mode-map "X" 'calc-last-x)
  (define-key calc-mode-map "l" 'calc-let)
  (define-key calc-mode-map "r" 'calc-recall)
  (define-key calc-mode-map "s" 'calc-store)
  (define-key calc-mode-map "x" 'calc-execute-extended-command)

  (define-key calc-mode-map "(" 'calc-begin-complex)
  (define-key calc-mode-map ")" 'calc-end-complex)
  (define-key calc-mode-map "[" 'calc-begin-vector)
  (define-key calc-mode-map "]" 'calc-end-vector)
  (define-key calc-mode-map "," 'calc-comma)
  (define-key calc-mode-map ";" 'calc-semi)
  (define-key calc-mode-map "`" 'calc-edit)
  (define-key calc-mode-map "=" 'calc-evaluate)
  (define-key calc-mode-map "~" 'calc-num-prefix)
  (define-key calc-mode-map "y" 'calc-copy-to-buffer)
  (define-key calc-mode-map "\C-k" 'calc-kill)
  (define-key calc-mode-map "\M-k" 'calc-copy-as-kill)
  (define-key calc-mode-map "\C-w" 'calc-kill-region)
  (define-key calc-mode-map "\M-w" 'calc-copy-region-as-kill)
  (define-key calc-mode-map "\C-y" 'calc-yank)
  (define-key calc-mode-map "\C-_" 'calc-undo)
  (define-key calc-mode-map "\C-xu" 'calc-undo)

  (define-key calc-mode-map "a" nil)
  (define-key calc-mode-map "a?" 'calc-a-prefix-help)
  (define-key calc-mode-map "ab" 'calc-substitute)
  (define-key calc-mode-map "ac" 'calc-collect)
  (define-key calc-mode-map "ad" 'calc-derivative)
  (define-key calc-mode-map "ae" 'calc-simplify-extended)
  (define-key calc-mode-map "ai" 'calc-integral)
  (define-key calc-mode-map "ar" 'calc-rewrite)
  (define-key calc-mode-map "as" 'calc-simplify)
  (define-key calc-mode-map "at" 'calc-taylor)
  (define-key calc-mode-map "ax" 'calc-expand)
  (define-key calc-mode-map "aI" 'calc-integral-limit)
  (define-key calc-mode-map "aS" 'calc-solve-for)
  (define-key calc-mode-map "a=" 'calc-equal-to)
  (define-key calc-mode-map "a#" 'calc-not-equal-to)
  (define-key calc-mode-map "a<" 'calc-less-than)
  (define-key calc-mode-map "a>" 'calc-greater-than)
  (define-key calc-mode-map "a[" 'calc-less-equal)
  (define-key calc-mode-map "a]" 'calc-greater-equal)
  (define-key calc-mode-map "a{" 'calc-in-set)
  (define-key calc-mode-map "a&" 'calc-logical-and)
  (define-key calc-mode-map "a|" 'calc-logical-or)
  (define-key calc-mode-map "a!" 'calc-logical-not)

  (define-key calc-mode-map "b" nil)
  (define-key calc-mode-map "b?" 'calc-b-prefix-help)
  (define-key calc-mode-map "ba" 'calc-and)
  (define-key calc-mode-map "bc" 'calc-clip)
  (define-key calc-mode-map "bd" 'calc-diff)
  (define-key calc-mode-map "bl" 'calc-lshift-binary)
  (define-key calc-mode-map "bL" 'calc-lshift-arith)
  (define-key calc-mode-map "bn" 'calc-not)
  (define-key calc-mode-map "bo" 'calc-or)
  (define-key calc-mode-map "br" 'calc-rshift-binary)
  (define-key calc-mode-map "bR" 'calc-rshift-arith)
  (define-key calc-mode-map "bt" 'calc-rotate-binary)
  (define-key calc-mode-map "bw" 'calc-word-size)
  (define-key calc-mode-map "bx" 'calc-xor)

  (define-key calc-mode-map "c" nil)
  (define-key calc-mode-map "c?" 'calc-c-prefix-help)
  (define-key calc-mode-map "c1" 'calc-clean-1)
  (define-key calc-mode-map "c2" 'calc-clean-2)
  (define-key calc-mode-map "c3" 'calc-clean-3)
  (define-key calc-mode-map "cc" 'calc-clean)
  (define-key calc-mode-map "cd" 'calc-to-degrees)
  (define-key calc-mode-map "cf" 'calc-float)
  (define-key calc-mode-map "ch" 'calc-to-hms)
  (define-key calc-mode-map "cp" 'calc-polar)
  (define-key calc-mode-map "cr" 'calc-to-radians)
  (define-key calc-mode-map "cF" 'calc-fraction)

  (define-key calc-mode-map "d" nil)
  (define-key calc-mode-map "d?" 'calc-d-prefix-help)
  (define-key calc-mode-map "d0" 'calc-decimal-radix)
  (define-key calc-mode-map "d2" 'calc-binary-radix)
  (define-key calc-mode-map "d6" 'calc-hex-radix)
  (define-key calc-mode-map "d8" 'calc-octal-radix)
  (define-key calc-mode-map "db" 'calc-line-breaking)
  (define-key calc-mode-map "dc" 'calc-complex-notation)
  (define-key calc-mode-map "de" 'calc-eng-notation)
  (define-key calc-mode-map "df" 'calc-fix-notation)
  (define-key calc-mode-map "dg" 'calc-group-digits)
  (define-key calc-mode-map "dh" 'calc-hms-notation)
  (define-key calc-mode-map "di" 'calc-i-notation)
  (define-key calc-mode-map "dj" 'calc-j-notation)
  (define-key calc-mode-map "dl" 'calc-line-numbering)
  (define-key calc-mode-map "dn" 'calc-normal-notation)
  (define-key calc-mode-map "do" 'calc-over-notation)
  (define-key calc-mode-map "dr" 'calc-radix)
  (define-key calc-mode-map "ds" 'calc-sci-notation)
  (define-key calc-mode-map "dt" 'calc-truncate-stack)
  (define-key calc-mode-map "dw" 'calc-auto-why)
  (define-key calc-mode-map "dz" 'calc-leading-zeros)
  (define-key calc-mode-map "dB" 'calc-big-language)
  (define-key calc-mode-map "dC" 'calc-c-language)
  (define-key calc-mode-map "dF" 'calc-fortran-language)
  (define-key calc-mode-map "dM" 'calc-mathematica-language)
  (define-key calc-mode-map "dN" 'calc-normal-language)
  (define-key calc-mode-map "dO" 'calc-flat-language)
  (define-key calc-mode-map "dP" 'calc-pascal-language)
  (define-key calc-mode-map "dT" 'calc-tex-language)
  (define-key calc-mode-map "dU" 'calc-unformatted-language)
  (define-key calc-mode-map "d[" 'calc-truncate-up)
  (define-key calc-mode-map "d]" 'calc-truncate-down)
  (define-key calc-mode-map "d." 'calc-point-char)
  (define-key calc-mode-map "d," 'calc-group-char)
  (define-key calc-mode-map "d\"" 'calc-display-strings)
  (define-key calc-mode-map "d<" 'calc-left-justify)
  (define-key calc-mode-map "d=" 'calc-center-justify)
  (define-key calc-mode-map "d>" 'calc-right-justify)
  (define-key calc-mode-map "d'" 'calc-display-raw)
  (define-key calc-mode-map "d`" 'calc-realign)
  (define-key calc-mode-map "d~" 'calc-refresh)

  (define-key calc-mode-map "k" nil)
  (define-key calc-mode-map "k?" 'calc-k-prefix-help)
  (define-key calc-mode-map "ka" 'calc-random-again)
  (define-key calc-mode-map "kb" 'calc-choose)
  (define-key calc-mode-map "kd" 'calc-double-factorial)
  (define-key calc-mode-map "kf" 'calc-prime-factors)
  (define-key calc-mode-map "kg" 'calc-gcd)
  (define-key calc-mode-map "kl" 'calc-lcm)
  (define-key calc-mode-map "km" 'calc-moebius)
  (define-key calc-mode-map "kn" 'calc-next-prime)
  (define-key calc-mode-map "kp" 'calc-prime-test)
  (define-key calc-mode-map "kr" 'calc-random)
  (define-key calc-mode-map "kt" 'calc-totient)
  (define-key calc-mode-map "kG" 'calc-extended-gcd)

  (define-key calc-mode-map "m" nil)
  (define-key calc-mode-map "m?" 'calc-m-prefix-help)
  (define-key calc-mode-map "ma" 'calc-algebraic-mode)
  (define-key calc-mode-map "md" 'calc-degrees-mode)
  (define-key calc-mode-map "mf" 'calc-frac-mode)
  (define-key calc-mode-map "mh" 'calc-hms-mode)
  (define-key calc-mode-map "mm" 'calc-save-modes)
  (define-key calc-mode-map "mp" 'calc-polar-mode)
  (define-key calc-mode-map "mr" 'calc-radians-mode)
  (define-key calc-mode-map "ms" 'calc-symbolic-mode)
  (define-key calc-mode-map "mw" 'calc-working)
  (define-key calc-mode-map "mx" 'calc-always-load-extensions)
  (define-key calc-mode-map "mA" 'calc-alg-simplify-mode)
  (define-key calc-mode-map "mB" 'calc-bin-simplify-mode)
  (define-key calc-mode-map "mD" 'calc-default-simplify-mode)
  (define-key calc-mode-map "mE" 'calc-ext-simplify-mode)
  (define-key calc-mode-map "mN" 'calc-num-simplify-mode)
  (define-key calc-mode-map "mO" 'calc-no-simplify-mode)
  (define-key calc-mode-map "mU" 'calc-units-simplify-mode)

  (define-key calc-mode-map "t" nil)
  (define-key calc-mode-map "t?" 'calc-t-prefix-help)
  (define-key calc-mode-map "tb" 'calc-trail-backward)
  (define-key calc-mode-map "td" 'calc-trail-display)
  (define-key calc-mode-map "tf" 'calc-trail-forward)
  (define-key calc-mode-map "th" 'calc-trail-here)
  (define-key calc-mode-map "ti" 'calc-trail-in)
  (define-key calc-mode-map "tk" 'calc-trail-kill)
  (define-key calc-mode-map "tm" 'calc-trail-marker)
  (define-key calc-mode-map "tn" 'calc-trail-next)
  (define-key calc-mode-map "to" 'calc-trail-out)
  (define-key calc-mode-map "tp" 'calc-trail-previous)
  (define-key calc-mode-map "tr" 'calc-trail-isearch-backward)
  (define-key calc-mode-map "ts" 'calc-trail-isearch-forward)
  (define-key calc-mode-map "ty" 'calc-trail-yank)
  (define-key calc-mode-map "t[" 'calc-trail-first)
  (define-key calc-mode-map "t]" 'calc-trail-last)
  (define-key calc-mode-map "t<" 'calc-trail-scroll-left)
  (define-key calc-mode-map "t>" 'calc-trail-scroll-right)

  (define-key calc-mode-map "u" 'nil)
  (define-key calc-mode-map "u?" 'calc-u-prefix-help)
  (define-key calc-mode-map "ub" 'calc-base-units)
  (define-key calc-mode-map "uc" 'calc-convert-units)
  (define-key calc-mode-map "ud" 'calc-define-unit)
  (define-key calc-mode-map "ue" 'calc-explain-units)
  (define-key calc-mode-map "ug" 'calc-get-unit-definition)
  (define-key calc-mode-map "up" 'calc-permanent-units)
  (define-key calc-mode-map "ur" 'calc-remove-units)
  (define-key calc-mode-map "us" 'calc-simplify-units)
  (define-key calc-mode-map "ut" 'calc-convert-temperature)
  (define-key calc-mode-map "uu" 'calc-undefine-unit)
  (define-key calc-mode-map "uv" 'calc-enter-units-table)
  (define-key calc-mode-map "ux" 'calc-extract-units)
  (define-key calc-mode-map "uV" 'calc-view-units-table)

  (define-key calc-mode-map "v" 'nil)
  (define-key calc-mode-map "v?" 'calc-v-prefix-help)
  (define-key calc-mode-map "va" 'calc-arrange-vector)
  (define-key calc-mode-map "vb" 'calc-build-vector)
  (define-key calc-mode-map "vc" 'calc-mcol)
  (define-key calc-mode-map "vd" 'calc-diag)
  (define-key calc-mode-map "vh" 'calc-histogram)
  (define-key calc-mode-map "vi" 'calc-ident)
  (define-key calc-mode-map "vl" 'calc-vlength)
  (define-key calc-mode-map "vn" 'calc-rnorm)
  (define-key calc-mode-map "vp" 'calc-pack)
  (define-key calc-mode-map "vr" 'calc-mrow)
  (define-key calc-mode-map "vs" 'calc-sort)
  (define-key calc-mode-map "vt" 'calc-transpose)
  (define-key calc-mode-map "vu" 'calc-unpack)
  (define-key calc-mode-map "vx" 'calc-index)
  (define-key calc-mode-map "vA" 'calc-apply)
  (define-key calc-mode-map "vC" 'calc-cross)
  (define-key calc-mode-map "vD" 'calc-mdet)
  (define-key calc-mode-map "vI" 'calc-inv)
  (define-key calc-mode-map "vJ" 'calc-conj-transpose)
  (define-key calc-mode-map "vL" 'calc-mlud)
  (define-key calc-mode-map "vM" 'calc-map)
  (define-key calc-mode-map "vN" 'calc-cnorm)
  (define-key calc-mode-map "vR" 'calc-reduce)
  (define-key calc-mode-map "vT" 'calc-mtrace)
  (define-key calc-mode-map "v<" 'calc-matrix-left-justify)
  (define-key calc-mode-map "v=" 'calc-matrix-center-justify)
  (define-key calc-mode-map "v>" 'calc-matrix-right-justify)
  (define-key calc-mode-map "v," 'calc-vector-commas)
  (define-key calc-mode-map "v[" 'calc-vector-brackets)
  (define-key calc-mode-map "v{" 'calc-vector-braces)
  (define-key calc-mode-map "v(" 'calc-vector-parens)
  (aset calc-mode-map ?V (aref calc-mode-map ?v))

  (define-key calc-mode-map "z" 'nil)
  (define-key calc-mode-map "z?" 'calc-z-prefix-help)

  (define-key calc-mode-map "Z" 'nil)
  (define-key calc-mode-map "Z?" 'calc-shift-Z-prefix-help)
  (define-key calc-mode-map "Zd" 'calc-user-define)
  (define-key calc-mode-map "Ze" 'calc-user-define-edit)
  (define-key calc-mode-map "Zf" 'calc-user-define-formula)
  (define-key calc-mode-map "Zg" 'calc-get-user-defn)
  (define-key calc-mode-map "Zi" 'calc-insert-variables)
  (define-key calc-mode-map "Zk" 'calc-user-define-kbd-macro)
  (define-key calc-mode-map "Zp" 'calc-user-define-permanent)
  (define-key calc-mode-map "Zu" 'calc-user-undefine)
  (define-key calc-mode-map "Zv" 'calc-permanent-variable)
  (define-key calc-mode-map "Z[" 'calc-kbd-if)
  (define-key calc-mode-map "Z:" 'calc-kbd-else)
  (define-key calc-mode-map "Z|" 'calc-kbd-else-if)
  (define-key calc-mode-map "Z]" 'calc-kbd-end-if)
  (define-key calc-mode-map "Z<" 'calc-kbd-repeat)
  (define-key calc-mode-map "Z>" 'calc-kbd-end-repeat)
  (define-key calc-mode-map "Z(" 'calc-kbd-for)
  (define-key calc-mode-map "Z)" 'calc-kbd-end-for)
  (define-key calc-mode-map "Z{" 'calc-kbd-loop)
  (define-key calc-mode-map "Z}" 'calc-kbd-end-loop)
  (define-key calc-mode-map "Z/" 'calc-kbd-break)
  (define-key calc-mode-map "Z`" 'calc-kbd-push)
  (define-key calc-mode-map "Z'" 'calc-kbd-pop)
  (define-key calc-mode-map "Z=" 'calc-kbd-report)
  (define-key calc-mode-map "Z#" 'calc-kbd-query)

;;;; (Autoloads here)

)

(calc-init-extensions)




;;;; Miscellaneous.

(defun calc-record-message (tag &rest args)
  (let ((msg (apply 'format args)))
    (message "%s" msg)
    (calc-record msg tag))
  (calc-clear-command-flag 'clear-message)
)


(defun calc-do-prefix-help (msgs group key)
  (if (cdr msgs)
      (progn
	(setq calc-prefix-help-phase
	      (if (eq this-command last-command)
		  (% (1+ calc-prefix-help-phase) (1+ (length msgs)))
		0))
	(let ((msg (nth calc-prefix-help-phase msgs)))
	  (message "%s" (if msg
			    (concat group ": " msg ":"
				    (make-string
				     (- (apply 'max (mapcar 'length msgs))
					(length msg)) 32)
				    "  [MORE]"
				    (if key
					(concat "  " (char-to-string key) "-")
				      ""))
			  (format "%c-" key)))))
    (setq calc-prefix-help-phase 0)
    (if key
	(if msgs
	    (message (concat group ": " (car msgs) ":  "
			     (char-to-string key) "-"))
	  (message (concat group ": (none)  " (char-to-string key) "-")))
      (message (concat group ": " (car msgs)))))
  (and key
       (setq unread-command-char key))
)
(defvar calc-prefix-help-phase 0)




;;;; Commands.


;;; General.

(defun calc-inverse (&optional n)
  "Next Calculator operation is inverse."
  (interactive "P")
  (calc-wrapper
   (calc-set-command-flag 'keep-flags)
   (setq calc-inverse-flag (not calc-inverse-flag)
	 prefix-arg n)
   (message (if calc-inverse-flag "Inverse..." "")))
)

(defun calc-invert-func ()
  (setq calc-inverse-flag (not (calc-is-inverse))
	calc-hyperbolic-flag (calc-is-hyperbolic)
	current-prefix-arg nil)
)

(defun calc-is-inverse ()
  calc-inverse-flag
)

(defun calc-hyperbolic (&optional n)
  "Next Calculator operation is hyperbolic."
  (interactive "P")
  (calc-wrapper
   (calc-set-command-flag 'keep-flags)
   (setq calc-hyperbolic-flag (not calc-hyperbolic-flag)
	 prefix-arg n)
   (message (if calc-hyperbolic-flag "Hyperbolic..." "")))
)

(defun calc-hyperbolic-func ()
  (setq calc-inverse-flag (calc-is-inverse)
	calc-hyperbolic-flag (not (calc-is-hyperbolic))
	current-prefix-arg nil)
)

(defun calc-is-hyperbolic ()
  calc-hyperbolic-flag
)


(defun calc-evaluate (n)
  "Evaluate all variables in the expression on the top of the stack.
With a numeric prefix argument, evaluate each of the top N stack elements."
  (interactive "p")
  (calc-slow-wrapper
   (if (= n 0)
       (setq n (calc-stack-size)))
   (if (< n 0)
       (error "Argument must be positive"))
   (calc-with-default-simplification
    (calc-pop-push-record-list n "eval" (mapcar 'math-evaluate-expr
						(calc-top-list-n n))))
   (calc-handle-whys))
)


(defun calc-eval-num (n)
  "Evaluate numerically the expression on the top of the stack.
This is only necessary when the calculator is in Symbolic mode."
  (interactive "P")
  (calc-slow-wrapper
   (let* ((nn (prefix-numeric-value n))
	  (calc-internal-prec (cond ((>= nn 3) nn)
				    ((< nn 0) (max (+ calc-internal-prec nn)
						   3))
				    (t calc-internal-prec)))
	  (calc-symbolic-mode nil))
     (calc-with-default-simplification
      (calc-pop-push-record 1 "num" (math-evaluate-expr (calc-top-n 1)))))
   (calc-handle-whys))
)


(defun calc-execute-extended-command (n)
  "Just like M-x, but inserts \"calc-\" prefix automatically."
  (interactive "P")
  (let* ((prompt (concat (calc-num-prefix-name n) "M-x "))
	 (cmd (intern (completing-read prompt obarray 'commandp t "calc-"))))
    (setq prefix-arg n)
    (command-execute cmd))
)


(defun calc-num-prefix (n)
  "Use the number at the top of stack as the numeric prefix for the next command.
With a prefix, push that prefix as a number onto the stack."
  (interactive "P")
  (calc-wrapper
   (if n
       (calc-enter-result 0 "" (prefix-numeric-value n))
     (let ((num (calc-top 1)))
       (if (math-messy-integerp num)
	   (setq num (math-trunc num)))
       (or (integerp num)
	   (error "Argument must be a small integer"))
       (calc-pop 1)
       (setq prefix-arg num)
       (message "%d-" num))))    ; a (lame) simulation of the real thing...
)


(defun calc-more-recursion-depth (n)
  "Double the max-lisp-eval-depth value, in case this limit is wrongly exceeded.
This also doubles max-specpdl-size."
  (interactive "P")
  (let ((n (if n (prefix-numeric-value n) 2)))
    (if (> n 1)
	(setq max-specpdl-size (* max-specpdl-size n)
	      max-lisp-eval-depth (* max-lisp-eval-depth n))))
  (message "max-lisp-eval-depth is now %d" max-lisp-eval-depth)
)

(defun calc-less-recursion-depth (n)
  "Halve the max-lisp-eval-depth value, in case this limit is too high.
This also halves max-specpdl-size.
Lower limits are 200 and 600, respectively."
  (interactive "P")
  (let ((n (if n (prefix-numeric-value n) 2)))
    (if (> n 1)
	(setq max-specpdl-size
	      (max (/ max-specpdl-size n) 600)
	      max-lisp-eval-depth
	      (max (/ max-lisp-eval-depth n) 200))))
  (message "max-lisp-eval-depth is now %d" max-lisp-eval-depth)
)



;;;; [calc-forms.el]

(defun calc-time ()
  "Push the current time of day on the stack as an HMS form.
\(Why?  Why not!)"
  (interactive)
  (calc-wrapper
   (let ((time (current-time-string)))
     (calc-enter-result 0 "time"
			(list 'mod
			      (list 'hms
				    (string-to-int (substring time 11 13))
				    (string-to-int (substring time 14 16))
				    (string-to-int (substring time 17 19)))
			      (list 'hms 24 0 0)))))
)



;;;; [calc-incom.el]

;;; Incomplete forms.

(defun calc-begin-complex ()
  "Begin entering a complex number in the Calculator."
  (interactive)
  (calc-wrapper
   (if calc-algebraic-mode
       (calc-alg-entry "(")
     (calc-push (list 'incomplete calc-complex-mode))))
)

(defun calc-end-complex ()
  "Complete a complex number being entered in the Calculator."
  (interactive)
  (calc-comma t)
  (calc-wrapper
   (let ((top (calc-top 1)))
     (if (and (eq (car-safe top) 'incomplete)
	      (eq (nth 1 top) 'intv))
	 (progn
	   (while (< (length top) 5)
	     (setq top (append top '(0))))
	   (calc-enter-result 1 "..)" (cdr top)))
       (if (not (and (eq (car-safe top) 'incomplete)
		     (memq (nth 1 top) '(cplx polar))))
	   (error "Not entering a complex number"))
       (while (< (length top) 4)
	 (setq top (append top '(0))))
       (if (not (and (math-realp (nth 2 top))
		     (math-anglep (nth 3 top))))
	   (error "Components must be real"))
       (calc-enter-result 1 "()" (cdr top)))))
)

(defun calc-begin-vector ()
  "Begin entering a vector in the Calculator."
  (interactive)
  (calc-wrapper
   (if calc-algebraic-mode
       (calc-alg-entry "[")
     (calc-push '(incomplete vec))))
)

(defun calc-end-vector ()
  "Complete a vector being entered in the Calculator."
  (interactive)
  (calc-comma t)
  (calc-wrapper
   (let ((top (calc-top 1)))
     (if (and (eq (car-safe top) 'incomplete)
	      (eq (nth 1 top) 'intv))
	 (progn
	   (while (< (length top) 5)
	     (setq top (append top '(0))))
	   (setcar (cdr (cdr top)) (1+ (nth 2 top)))
	   (calc-enter-result 1 "..]" (cdr top)))
       (if (not (and (eq (car-safe top) 'incomplete)
		     (eq (nth 1 top) 'vec)))
	   (error "Not entering a vector"))
       (calc-pop-push-record 1 "[]" (cdr top)))))
)

(defun calc-comma (&optional allow-polar)
  "Separate components of a complex number or vector during entry."
  (interactive)
  (calc-wrapper
   (let ((num (calc-find-first-incomplete
	       (nthcdr calc-stack-top calc-stack) 1)))
     (if (= num 0)
	 (error "Not entering a vector or complex number"))
     (let* ((inc (calc-top num))
	    (stuff (calc-top-list (1- num)))
	    (new (append inc stuff)))
       (if (and (null stuff)
		(not allow-polar)
		(or (eq (nth 1 inc) 'vec)
		    (< (length new) 4)))
	   (setq new (append new
			     (if (= (length new) 2)
				 '(0)
			       (nthcdr (1- (length new)) new)))))
       (or allow-polar
	   (if (eq (nth 1 inc) 'polar)
	       (setq inc (append '(incomplete cplx) (cdr (cdr inc))))
	     (if (eq (nth 1 inc) 'intv)
		 (setq inc (append '(incomplete cplx)
				   (cdr (cdr (cdr inc))))))))
       (if (and (memq (nth 1 new) '(cplx polar))
		(> (length new) 4))
	   (error "Too many components in complex number"))
       (calc-pop-push num new))))
)

(defun calc-semi ()
  "Separate parts of a polar complex number or rows of a matrix during entry."
  (interactive)
  (calc-wrapper
   (let ((num (calc-find-first-incomplete
	       (nthcdr calc-stack-top calc-stack) 1)))
     (if (= num 0)
	 (error "Not entering a vector or complex number"))
     (let ((inc (calc-top num))
	   (stuff (calc-top-list (1- num))))
       (if (eq (nth 1 inc) 'cplx)
	   (setq inc (append '(incomplete polar) (cdr (cdr inc))))
	 (if (eq (nth 1 inc) 'intv)
	     (setq inc (append '(incomplete polar) (cdr (cdr (cdr inc)))))))
       (cond ((eq (nth 1 inc) 'polar)
	      (let ((new (append inc stuff)))
		(if (> (length new) 4)
		    (error "Too many components in complex number")
		  (if (= (length new) 2)
		      (setq new (append new '(1)))))
		(calc-pop-push num new)))
	     ((null stuff)
	      (if (> (length inc) 2)
		  (if (math-vectorp (nth 2 inc))
		      (calc-comma)
		    (calc-pop-push 1
				   (list 'incomplete 'vec (cdr (cdr inc)))
				   (list 'incomplete 'vec)))))
	     ((math-vectorp (car stuff))
	      (calc-comma))
	     ((eq (car-safe (car-safe (nth (+ num calc-stack-top)
					   calc-stack))) 'incomplete)
	      (calc-end-vector)
	      (calc-comma)
	      (let ((calc-algebraic-mode nil))
		(calc-begin-vector)))
	     ((or (= (length inc) 2)
		  (math-vectorp (nth 2 inc)))
	      (calc-pop-push num
			     (append inc (list (cons 'vec stuff)))
			     (list 'incomplete 'vec)))
	     (t
	      (calc-pop-push num
			     (list 'incomplete 'vec
				   (cons 'vec (append (cdr (cdr inc)) stuff)))
			     (list 'incomplete 'vec)))))))
)

(defun calc-dots ()
  "Separate parts of an interval form during entry with a \"..\" symbol."
  (interactive)
  (calc-wrapper
   (let ((num (calc-find-first-incomplete
	       (nthcdr calc-stack-top calc-stack) 1)))
     (if (= num 0)
	 (error "Not entering an interval form"))
     (let* ((inc (calc-top num))
	    (stuff (calc-top-list (1- num)))
	    (new (append inc stuff)))
       (if (not (eq (nth 1 new) 'intv))
	   (setq new (append '(incomplete intv)
			     (if (eq (nth 1 new) 'vec) '(2) '(0))
			     (cdr (cdr new)))))
       (if (and (null stuff)
		(or (eq (nth 1 inc) 'vec)
		    (< (length new) 5)))
	   (setq new (append new
			     (if (= (length new) 2)
				 '(0)
			       (nthcdr (1- (length new)) new)))))
       (if (> (length new) 5)
	   (error "Too many components in interval form"))
       (calc-pop-push num new))))
)

(defun calc-find-first-incomplete (stack n)
  (cond ((null stack)
	 0)
	((eq (car-safe (car-safe (car stack))) 'incomplete)
	 n)
	(t
	 (calc-find-first-incomplete (cdr stack) (1+ n))))
)




;;;; [calc-undo.el]

;;; Undo.

(defun calc-undo (n)
  "Undo the most recent operation in the Calculator.
With a numeric prefix argument, undo the last N operations.
With a negative argument, same as calc-redo.
With a zero argument, same as calc-last-x."
  (interactive "p")
  (and calc-executing-macro
       (error "Use C-x e, not K, to run a keyboard macro that uses Undo."))
  (if (<= n 0)
      (if (< n 0)
	  (calc-redo (- n))
	(calc-last-x 1))
    (calc-wrapper
     (if (null (nthcdr (1- n) calc-undo-list))
	 (error "No further undo information available"))
     (setq calc-undo-list
	   (prog1
	       (nthcdr n calc-undo-list)
	     (let ((saved-stack-top calc-stack-top))
	       (let ((calc-stack-top 0))
		 (calc-handle-undos calc-undo-list n))
	       (setq calc-stack-top saved-stack-top))))
     (message "Undo!")))
)

(defun calc-handle-undos (cl n)
  (if (> n 0)
      (progn
	(let ((old-redo calc-redo-list))
	  (setq calc-undo-list nil)
	  (calc-handle-undo (car cl))
	  (setq calc-redo-list (append calc-undo-list old-redo)))
	(calc-handle-undos (cdr cl) (1- n))))
)

(defun calc-handle-undo (list)
  (and list
       (let ((action (car list)))
	 (cond
	  ((eq (car action) 'push)
	   (calc-pop-stack 1 (nth 1 action)))
	  ((eq (car action) 'pop)
	   (calc-push-list (nth 2 action) (nth 1 action)))
	  ((eq (car action) 'set)
	   (calc-record-undo (list 'set (nth 1 action)
				   (symbol-value (nth 1 action))))
	   (set (nth 1 action) (nth 2 action)))
	  ((eq (car action) 'store)
	   (let ((v (intern (nth 1 action))))
	     (calc-record-undo (list 'store (nth 1 action)
				     (and (boundp v) (symbol-value v))))
	     (if (y-or-n-p (format "Un-store variable %s? " (nth 1 action)))
		 (if (nth 2 action)
		     (set v (nth 2 action))
		   (makunbound v)))))
	  ((eq (car action) 'eval)
	   (calc-record-undo (append (list 'eval (nth 2 action) (nth 1 action))
				     (cdr (cdr (cdr action)))))
	   (apply (nth 1 action) (cdr (cdr (cdr action))))))
	 (calc-handle-undo (cdr list))))
)

(defun calc-redo (n)
  "Redo a command which was just inadvertently undone."
  (interactive "p")
  (and calc-executing-macro
       (error "Use C-x e, not K, to run a keyboard macro that uses Redo."))
  (if (< n 0)
      (calc-undo (- n))
    (calc-wrapper
     (if (null (nthcdr (1- n) calc-redo-list))
	 (error "Unable to redo"))
     (setq calc-redo-list
	   (prog1
	       (nthcdr n calc-redo-list)
	     (let ((saved-stack-top calc-stack-top))
	       (let ((calc-stack-top 0))
		 (calc-handle-redos calc-redo-list n))
	       (setq calc-stack-top saved-stack-top))))
     (message "Redo!")))
)

(defun calc-handle-redos (cl n)
  (if (> n 0)
      (progn
	(let ((old-undo calc-undo-list))
	  (setq calc-undo-list nil)
	  (calc-handle-undo (car cl))
	  (setq calc-undo-list (append calc-undo-list old-undo)))
	(calc-handle-redos (cdr cl) (1- n))))
)

(defun calc-last-x (n)
  "Restore the arguments to the last command, without removing its result.
With a numeric prefix argument, restore the arguments of the Nth last
command which popped things from the stack."
  (interactive "p")
  (and calc-executing-macro
       (error "Use C-x e, not K, to run a keyboard macro that uses Last X."))
  (calc-wrapper
   (let ((urec (calc-find-last-x calc-undo-list n)))
     (if urec
	 (calc-handle-last-x urec)
       (error "Not enough undo information available"))))
)

(defun calc-handle-last-x (list)
  (and list
       (let ((action (car list)))
	 (if (eq (car action) 'pop)
	     (calc-pop-push-record-list 0 "lstx"
					(delq 'top-of-stack (nth 2 action))))
	 (calc-handle-last-x (cdr list))))
)

(defun calc-find-last-x (ul n)
  (and ul
       (if (calc-undo-does-pushes (car ul))
	   (if (<= n 1)
	       (car ul)
	     (calc-find-last-x (cdr ul) (1- n)))
	 (calc-find-last-x (cdr ul) n)))
)

(defun calc-undo-does-pushes (list)
  (and list
       (or (eq (car (car list)) 'pop)
	   (calc-undo-does-pushes (cdr list))))
)



;;;; [calc-arith.el]

;;; Arithmetic.

(defun calc-min (arg)
  "Compute the minimum of the top two elements of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "min" 'calcFunc-min arg))
)

(defun calc-max (arg)
  "Compute the maximum of the top two elements of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "max" 'calcFunc-max arg))
)

(defun calc-abs (arg)
  "Compute the absolute value of the top element of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "abs" 'calcFunc-abs arg))
)

;;;; [calc-math.el]

(defun calc-sqrt (arg)
  "Take the square root of the top element of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-inverse)
       (calc-unary-op "^2" 'calcFunc-sqr arg)
     (calc-unary-op "sqrt" 'calcFunc-sqrt arg)))
)

;;;; [calc-arith.el]

(defun calc-idiv (arg)
  "Compute the integer quotient of the top two elements of the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "\\" 'calcFunc-idiv arg 1))
)

;;;; [calc-frac.el]

(defun calc-fdiv (arg)
  "Compute the quotient (in fraction form) of the top two elements of the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op ":" 'calcFunc-fdiv arg 1))
)

;;;; [calc-arith.el]

(defun calc-floor (arg)
  "Truncate to an integer (toward minus infinity) the top element of the stack.
With Inverse flag, truncates toward plus infinity.
With Hyperbolic flag, represent result in floating-point."
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-inverse)
       (if (calc-is-hyperbolic)
	   (calc-unary-op "ceil" 'calcFunc-fceil arg)
	 (calc-unary-op "ceil" 'calcFunc-ceil arg))
     (if (calc-is-hyperbolic)
	 (calc-unary-op "flor" 'calcFunc-ffloor arg)
       (calc-unary-op "flor" 'calcFunc-floor arg))))
)

(defun calc-ceiling (arg)
  "Truncate to an integer (toward plus infinity) the top element of the stack."
  (interactive "P")
  (calc-invert-func)
  (calc-floor arg)
)

(defun calc-round (arg)
  "Round to the nearest integer the top element of the Calculator stack.
With Inverse flag, truncate (toward zero) to an integer.
With Hyperbolic flag, represent result in floating-point."
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-inverse)
       (if (calc-is-hyperbolic)
	   (calc-unary-op "trnc" 'calcFunc-ftrunc arg)
	 (calc-unary-op "trnc" 'calcFunc-trunc arg))
     (if (calc-is-hyperbolic)
	 (calc-unary-op "rond" 'calcFunc-fround arg)
       (calc-unary-op "rond" 'calcFunc-round arg))))
)

(defun calc-trunc (arg)
  "Truncate to an integer (toward zero) the top element of the Calculator stack."
  (interactive "P")
  (calc-invert-func)
  (calc-round arg)
)

(defun calc-mant-part (arg)
  "Extract the mantissa part of a floating-point number."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "mant" 'calcFunc-mant arg))
)

(defun calc-xpon-part (arg)
  "Extract the exponent part of a floating-point number."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "xpon" 'calcFunc-xpon arg))
)

(defun calc-scale-float (arg)
  "Scale a floating-point number by an integer power of ten."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "scal" 'calcFunc-scf arg))
)

(defun calc-abssqr (arg)
  "Compute the absolute value squared of the top element of the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "absq" 'calcFunc-abssqr arg))
)

;;;; [calc-cplx.el]

(defun calc-argument (arg)
  "Compute the complex argument of the top element of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "arg" 'calcFunc-arg arg))
)

(defun calc-re (arg)
  "Replace the top element of the Calculator stack with its real part."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "re" 'calcFunc-re arg))
)

(defun calc-im (arg)
  "Replace the top element of the Calculator stack with its imaginary part."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "im" 'calcFunc-im arg))
)

;;;; [calc-math.el]

(defun calc-hypot (arg)
  "Take the square root of sum of squares of the top two elements of the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "hypt" 'calcFunc-hypot arg))
)

(defun calc-ln (arg)
  "Take the natural logarithm of the top element of the Calculator stack.
With Inverse flag or negative prefix arg, computes e^x.
With Hyperbolic flag or even prefix arg, computes log_10 or 10^x."
  (interactive "P")
  (calc-invert-func)
  (calc-exp arg)
)

(defun calc-log10 (arg)
  "Take the logarithm (base 10) of the top element of the Calculator stack.
With Inverse flag or negative prefix arg, computes 10^x."
  (interactive "P")
  (calc-hyperbolic-func)
  (calc-ln arg)
)

(defun calc-log (arg)
  "Take the logarithm base B of X.  B is top-of-stack, X is second-to-top.
With Inverse flag, computes B^X.  (Note that \"^\" would compute X^B.)"
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-inverse)
       (calc-binary-op "Ilog" 'calcFunc-ilog arg)
     (calc-binary-op "log" 'calcFunc-log arg)))
)

(defun calc-lnp1 (arg)
  "Take the logarithm (ln(x+1)) of one plus the top element of the stack."
  (interactive "P")
  (calc-invert-func)
  (calc-expm1 arg)
)

(defun calc-exp (arg)
  "Take the exponential (e^x) of the top element of the Calculator stack.
With Inverse flag or negative prefix arg, takes the natural logarithm.
With Hyperbolic flag or even prefix arg, computes 10^x or log_10."
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-hyperbolic)
       (if (calc-is-inverse)
	   (calc-unary-op "lg10" 'calcFunc-log10 arg)
	 (calc-unary-op "10^" 'calcFunc-pow10 arg))
     (if (calc-is-inverse)
	 (calc-unary-op "ln" 'calcFunc-ln arg)
       (calc-unary-op "exp" 'calcFunc-exp arg))))
)

(defun calc-expm1 (arg)
  "Take the exponential minus one (e^x - 1) of the top element of the stack."
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-inverse)
       (calc-unary-op "ln+1" 'calcFunc-lnp1 arg)
     (calc-unary-op "ex-1" 'calcFunc-expm1 arg)))
)

(defun calc-pi ()
  "Push Pi (at the current precision) on the Calculator stack.
With Hyperbolic flag, pushes `e' (the base of natural logarithms)."
  (interactive)
  (calc-slow-wrapper
   (if (calc-is-hyperbolic)
       (if calc-symbolic-mode
	   (calc-pop-push-record 0 "e" '(var e var-e))
	 (calc-pop-push-record 0 "e" (math-e)))
     (if calc-symbolic-mode
	 (calc-pop-push-record 0 "pi" '(var pi var-pi))
       (calc-pop-push-record 0 "pi" (math-pi)))))
)

(defun calc-sin (arg)
  "Take the sine of the top element of the Calculator stack.
With Inverse flag or negative prefix arg, takes the inverse sine.
With Hyperbolic flag or even prefix arg, computes sinh or arcsinh."
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-hyperbolic)
       (if (calc-is-inverse)
	   (calc-unary-op "asnh" 'calcFunc-arcsinh arg)
	 (calc-unary-op "sinh" 'calcFunc-sinh arg))
     (if (calc-is-inverse)
	 (calc-unary-op "asin" 'calcFunc-arcsin arg)
       (calc-unary-op "sin" 'calcFunc-sin arg))))
)

(defun calc-arcsin (arg)
  "Take the inverse sine of the top element of the Calculator stack."
  (interactive "P")
  (calc-invert-func)
  (calc-sin arg)
)

(defun calc-sinh (arg)
  "Take the hyperbolic sine of the top element of the Calculator stack."
  (interactive "P")
  (calc-hyperbolic-func)
  (calc-sin arg)
)

(defun calc-arcsinh (arg)
  "Take the inverse hyperbolic sine of the top element of the Calculator stack."
  (interactive "P")
  (calc-invert-func)
  (calc-hyperbolic-func)
  (calc-sin arg)
)

(defun calc-cos (arg)
  "Take the cosine of the top element of the Calculator stack.
With Inverse flag or negative prefix arg, takes the inverse cosine.
With Hyperbolic flag or even prefix arg, computes cosh or arccosh."
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-hyperbolic)
       (if (calc-is-inverse)
	   (calc-unary-op "acsh" 'calcFunc-arccosh arg)
	 (calc-unary-op "cosh" 'calcFunc-cosh arg))
     (if (calc-is-inverse)
	 (calc-unary-op "acos" 'calcFunc-arccos arg)
       (calc-unary-op "cos" 'calcFunc-cos arg))))
)

(defun calc-arccos (arg)
  "Take the inverse cosine of the top element of the Calculator stack."
  (interactive "P")
  (calc-invert-func)
  (calc-cos arg)
)

(defun calc-cosh (arg)
  "Take the hyperbolic cosine of the top element of the Calculator stack."
  (interactive "P")
  (calc-hyperbolic-func)
  (calc-cos arg)
)

(defun calc-arccosh (arg)
  "Take the inverse hyperbolic cosine of the top element of the Calculator stack."
  (interactive "P")
  (calc-invert-func)
  (calc-hyperbolic-func)
  (calc-cos arg)
)

(defun calc-sincos ()
  "Compute the sine and cosine of the top element of the Calculator stack.
Result is a vector [cos(x), sin(x)].
Inverse and Hyperbolic flags are not recognized."
  (interactive)
  (calc-slow-wrapper
   (if (calc-is-inverse)
       (calc-enter-result 1 "asnc" (list 'calcFunc-arcsincos (calc-top-n 1)))
     (calc-enter-result 1 "sncs" (list 'calcFunc-sincos (calc-top-n 1)))))
)

(defun calc-tan (arg)
  "Take the tangent of the top element of the Calculator stack.
With Inverse flag or negative prefix arg, takes the inverse tangent.
With Hyperbolic flag or even prefix arg, computes tanh or arctanh."
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-hyperbolic)
       (if (calc-is-inverse)
	   (calc-unary-op "atnh" 'calcFunc-arctanh arg)
	 (calc-unary-op "tanh" 'calcFunc-tanh arg))
     (if (calc-is-inverse)
	 (calc-unary-op "atan" 'calcFunc-arctan arg)
       (calc-unary-op "tan" 'calcFunc-tan arg))))
)

(defun calc-arctan (arg)
  "Take the inverse tangent of the top element of the Calculator stack."
  (interactive "P")
  (calc-invert-func)
  (calc-tan arg)
)

(defun calc-tanh (arg)
  "Take the hyperbolic tangent of the top element of the Calculator stack."
  (interactive "P")
  (calc-hyperbolic-func)
  (calc-tan arg)
)

(defun calc-arctanh (arg)
  "Take the inverse hyperbolic tangent of the top element of the stack."
  (interactive "P")
  (calc-invert-func)
  (calc-hyperbolic-func)
  (calc-tan arg)
)

(defun calc-arctan2 ()
  "Compute the full-circle arc tangent of the ratio of two numbers."
  (interactive)
  (calc-slow-wrapper
   (calc-enter-result 2 "atn2" (cons 'calcFunc-arctan2 (calc-top-list-n 2))))
)

(defun calc-conj (arg)
  "Compute the complex conjugate of the top element of the Calculator stack."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op "conj" 'calcFunc-conj arg))
)

(defun calc-imaginary ()
  "Multiply the top element of the Calculator stack by complex \"i\"."
  (interactive)
  (calc-slow-wrapper
   (calc-pop-push-record 1 "i*" (math-imaginary (calc-top-n 1))))
)



;;;; [calc-store.el]

;;; Memory commands.

(defun calc-store (n &optional var oper)
  "Store the value at the top of the Calculator stack in variable VAR.
If VAR is of the form +V, -V, *V, /V, ^V, or |V, top of stack is combined
into V with the appropriate operation.
With any numeric prefix argument, unsets the specified variable."
  (interactive "P")
  (calc-wrapper
   (if n
       (progn
	 (or var
	     (setq var (let ((minibuffer-completion-table obarray)
			     (minibuffer-completion-predicate 'boundp)
			     (minibuffer-completion-confirm t)
			     (oper "r"))
			 (read-from-minibuffer
			  "Unstore: " "var-" calc-store-var-map nil))))
	 (if (equal var "")
	     ()
	   (makunbound (intern var))))
     (while (or (null var) (equal var "")
		(string-match "\\`[-+*/^|].*" var))
       (if (and var (> (length var) 0))
	   (setq oper (substring var 0 1)
		 var (substring var 1))
	 (setq var (let ((minibuffer-completion-table obarray)
			 (minibuffer-completion-predicate 'boundp)
			 (minibuffer-completion-confirm t))
		     (read-from-minibuffer
		      (if oper (format "Store %s: " oper) "Store: ")
		      "var-" calc-store-var-map nil)))))
     (if (equal var "")
	 ()
       (let* ((ivar (intern var))
	      (ival (if (boundp ivar) (symbol-value ivar) nil)))
	 (if (null oper)
	     (set ivar (calc-top 1))
	   (if (null ival)
	       (error "No such variable"))
	   (if (stringp ival)
	       (setq ival (math-read-expr ival)))
	   (setq ival (calc-normalize ival))
	   (cond ((equal oper "+")
		  (set ivar (calc-normalize
			     (list '+ ival (calc-top-n 1)))))
		 ((equal oper "-")
		  (set ivar (calc-normalize
			     (list '- ival (calc-top-n 1)))))
		 ((equal oper "*")
		  (set ivar (calc-normalize
			     (list '* ival (calc-top-n 1)))))
		 ((equal oper "/")
		  (set ivar (calc-normalize
			     (list '/ ival (calc-top-n 1)))))
		 ((equal oper "^")
		  (set ivar (calc-normalize
			     (list '^ ival (calc-top-n 1)))))
		 ((equal oper "|")
		  (set ivar (calc-normalize
			     (list '| ival (calc-top-n 1)))))))
	 (calc-record-undo (list 'store var ival))
	 (calc-record (symbol-value ivar)
		      (concat ">" (or oper "")
			      (if (string-match "\\`var-.+\\'" var)
				  (substring var 4) var)))))))
)

(defun calc-unstore (&optional var oper)
  (interactive)
  (calc-store -1 var oper)
)

(defvar calc-store-var-map nil "Keymap for use by the calc-store command.")
(if calc-store-var-map
    ()
  (setq calc-store-var-map (copy-keymap minibuffer-local-completion-map))
  (mapcar (function
	   (lambda (x)
	     (define-key calc-store-var-map (char-to-string x)
	       'calcVar-digit)))
	  "0123456789")
  (mapcar (function
	   (lambda (x)
	     (define-key calc-store-var-map (char-to-string x)
	       'calcVar-oper)))
	  "+-*/^|")
)

(defun calcVar-digit ()
  (interactive)
  (if (calc-minibuffer-contains "var-\\'")
      (self-insert-and-exit)
    (self-insert-command 1))
)

(defun calcVar-oper ()
  (interactive)
  (if (calc-minibuffer-contains "var-\\'")
      (if (null oper)
	  (progn
	    (erase-buffer)
	    (self-insert-and-exit))
	(beep))
    (self-insert-command 1))
)

(defun calc-recall (&optional var)
  "Recall the value of variable VAR into the Calculator stack."
  (interactive)
  (calc-wrapper
   (or var
       (setq var (let ((minibuffer-completion-table obarray)
		       (minibuffer-completion-predicate 'boundp)
		       (minibuffer-completion-confirm t)
		       (oper "r"))
		   (read-from-minibuffer
		    "Recall: " "var-" calc-store-var-map nil))))
   (if (equal var "")
       ()
     (setq ivar (intern var))
     (if (not (and (boundp ivar) ivar))
	 (error "No such variable"))
     (let ((ival (symbol-value ivar)))
       (if (stringp ival)
	   (setq ival (math-read-expr ival)))
       (if (eq (car-safe ival) 'error)
	   (error "Bad format in variable contents: %s" (nth 2 ival)))
       (setq ival (calc-normalize ival))
       (calc-record ival (concat "<"
				 (if (string-match "\\`var-.+\\'" var)
				     (substring var 4) var)))
       (calc-push ival))))
)

(defun calc-let (&optional var)
  "Evaluate second-in-stack where variable VAR equals top of stack."
  (interactive)
  (calc-wrapper
   (or var
       (setq var (let ((minibuffer-completion-table obarray)
		       (minibuffer-completion-predicate 'boundp)
		       (minibuffer-completion-confirm t)
		       (oper "r"))
		   (read-from-minibuffer
		    "Let variable: " "var-" calc-store-var-map nil))))
   (if (equal var "")
       ()
     (setq ivar (intern var))
     (calc-pop-push-record
      2 (concat "="
		(if (string-match "\\`var-.+\\'" var)
		    (substring var 4) var))
      (let ((saved-val (and (boundp ivar) (symbol-value ivar))))
	(unwind-protect
	    (progn
	      (set ivar (calc-top-n 1))
	      (math-evaluate-expr (calc-top-n 2)))
	  (if saved-val
	      (set ivar saved-val)
	    (makunbound ivar)))))))
)




;;;; [calc-yank.el]

;;; Kill ring commands.

(defun calc-kill (nn &optional no-delete)
  "Kill the Calculator stack element containing the cursor.
With numeric prefix argument N, kill the N stack elements at+below cursor."
  (interactive "P")
  (calc-wrapper
   (calc-force-refresh)
   (calc-set-command-flag 'no-align)
   (let ((num (max (calc-locate-cursor-element (point)) 1))
	 (n (prefix-numeric-value nn)))
     (if (< n 0)
	 (progn
	   (if (eobp)
	       (setq num (1- num)))
	   (setq num (- num n)
		 n (- n))))
     (let ((stuff (calc-top-list n (- num n -1))))
       (calc-cursor-stack-index num)
       (let ((first (point)))
	 (calc-cursor-stack-index (- num n))
	 (if (null nn)
	     (backward-char 1))   ; don't include newline for raw C-k
	 (copy-region-as-kill first (point))
	 (if (not no-delete)
	     (calc-pop-stack n (- num n -1))))
       (setq calc-last-kill (cons (car kill-ring) stuff)))))
)

(defun calc-force-refresh ()
  (if calc-executing-macro
      (let ((calc-executing-macro nil))
	(calc-refresh)))
)

(defun calc-locate-cursor-element (pt)
  (save-excursion
    (goto-char (point-max))
    (calc-locate-cursor-scan (- calc-stack-top) calc-stack pt))
)

(defun calc-locate-cursor-scan (n stack pt)
  (if (or (<= (point) pt)
	  (null stack))
      n
    (forward-line (- (nth 1 (car stack))))
    (calc-locate-cursor-scan (1+ n) (cdr stack) pt))
)

(defun calc-kill-region (top bot &optional no-delete)
  "Kill the Calculator stack elements between Point and Mark."
  (interactive "r")
  (calc-wrapper
   (calc-force-refresh)
   (calc-set-command-flag 'no-align)
   (let* ((top-num (calc-locate-cursor-element top))
	  (bot-num (calc-locate-cursor-element (1- bot)))
	  (num (- top-num bot-num -1)))
     (copy-region-as-kill top bot)
     (setq calc-last-kill (cons (car kill-ring) (calc-top-list num bot-num)))
     (if (not no-delete)
	 (calc-pop-stack num bot-num))))
)

(defun calc-copy-as-kill (n)
  "Copy the Calculator stack element containing the cursor into the Kill Ring.
The stack element is not deleted.  With numeric prefix argument N, copy the
N stack elements at+below cursor."
  (interactive "P")
  (calc-kill n t)
)

(defun calc-copy-region-as-kill (top bot)
  "Copy the Calculator stack elements between Point and Mark into the Kill Ring.
The stack elements are not deleted."
  (interactive "r")
  (calc-kill-region top bot t)
)

;;; This function uses calc-last-kill if possible to get an exact result,
;;; otherwise it just parses the yanked string.
(defun calc-yank ()
  "Enter the contents of the last Killed text into the Calculator stack.
This text must be formatted as a number or list of numbers."
  (interactive)
  (calc-wrapper
   (calc-pop-push-record-list
    0 "yank"
    (if (eq (car-safe calc-last-kill) (car kill-ring-yank-pointer))
	(cdr calc-last-kill)
      (if (stringp (car kill-ring-yank-pointer))
	  (let ((val (math-read-exprs
		      (calc-clean-newlines (car kill-ring-yank-pointer)))))
	    (if (eq (car-safe val) 'error)
		(error "Bad format in yanked data")
	      val))))))
)

(defun calc-clean-newlines (s)
  (cond
   
   ;; Omit leading/trailing whitespace
   ((or (string-match "\\`[ \n\r]+\\([^\001]*\\)\\'" s)
	(string-match "\\`\\([^\001]*\\)[ \n\r]+\\'" s))
    (calc-clean-newlines (math-match-substring s 1)))

   ;; Convert newlines to commas
   ((string-match "\\`\\(.*\\)[\n\r]+\\([^\001]*\\)\\'" s)
    (calc-clean-newlines (concat (math-match-substring s 1) ","
				 (math-match-substring s 2))))
   
   (t s))
)

(defun calc-grab-region (top bot arg)
  "Parse the region as a matrix of numbers and push it on the Calculator stack.
This is intended to be used in a non-Calculator buffer!
If the start and the end of the region are in column zero, the contained lines
are parsed into rows of the matrix.  Otherwise, point and mark define a
rectangle which is parsed into a matrix.

With a positive numeric prefix N, each line is divided into columns of
width N which become the elements of that row of the matrix.  With a
prefix of zero, each line is interpreted in its entirety as a formula
which becomes a row of the (one-column) matrix.

Otherwise, if a line contains a portion delimited by square brackets
or curly braces (possibly surrounded by other ignored text), that portion
is interpreted as a vector which becomes a row of the matrix.  This can be
suppressed by giving a negative numeric prefix argument.

Otherwise, the entire contents of the line are parsed as if surrounded by
brackets.  If a stack-style line number (as in \"23: \") is present it is
first removed."
  (interactive "r\nP")
  (and (memq major-mode '(calc-mode calc-trail-mode))
       (error "This command works only in a regular text buffer."))
  (let* ((col1 (save-excursion (goto-char top) (current-column)))
	 (col2 (save-excursion (goto-char bot) (current-column)))
	 (from-buffer (current-buffer))
	 data mat vals lnum pt pos)
    (if (= col1 col2)
	(save-excursion
	  (or (= col1 0)
	      (error "Point and mark must be at beginning of line, or define a rectangle"))
	  (goto-char top)
	  (while (< (point) bot)
	    (setq pt (point))
	    (forward-line 1)
	    (setq data (cons (buffer-substring pt (1- (point))) data)))
	  (setq data (nreverse data)))
      (setq data (extract-rectangle top bot)))
    (calc)
    (setq mat (list 'vec)
	  lnum 0)
    (and arg
	 (setq arg (prefix-numeric-value arg)))
    (while data
      (if (natnump arg)
	  (progn
	    (if (= arg 0)
		(setq arg 1000000))
	    (setq pos 0
		  vals (list 'vec))
	    (let ((w (length (car data)))
		  j v)
	      (while (< pos w)
		(setq j (+ pos arg)
		      v (if (>= j w)
			    (math-read-expr (substring (car data) pos))
			  (math-read-expr (substring (car data) pos j))))
		(if (eq (car-safe v) 'error)
		    (setq vals v w 0)
		  (setq vals (nconc vals (list v))
			pos j)))))
	(if (and (null arg)
		 (string-match "[[{][^][{}]*[]}]" (car data)))
	    (setq pos (match-beginning 0)
		  vals (math-read-expr (math-match-substring (car data) 0)))
	  (if (string-match "\\`\\([0-9]+:[ \t]\\)?\\(.*[^, \t]\\)[, \t]*\\'"
			    (car data))
	      (setq pos -1
		    vals (math-read-expr (concat "["
						 (math-match-substring
						  (car data) 2)
						 "]")))
	    (setq pos -1
		  vals (math-read-expr (concat "[" (car data) "]"))))))
      (if (eq (car-safe vals) 'error)
	  (progn
	    (calc-quit)
	    (switch-to-buffer from-buffer)
	    (goto-char top)
	    (forward-line lnum)
	    (forward-char (+ (nth 1 vals) (min col1 col2) pos))
	    (error (nth 2 vals))))
      (setq mat (cons vals mat)
	    data (cdr data)
	    lnum (1+ lnum)))
    (calc-wrapper
     (calc-enter-result 0 "grab" (nreverse mat))))
)

(defun calc-copy-to-buffer (nn)
  "Copy the top of stack into the most recently used editing buffer.
With a positive numeric prefix argument, copy the top N lines.
With a negative argument, copy the Nth line.
With an argument of zero, copy the entire stack.
With plain \"C-u\" as an argument, replaces region in other buffer."
  (interactive "P")
  (let (oldbuf newbuf)
    (calc-wrapper
     (save-excursion
       (calc-force-refresh)
       (let ((n (prefix-numeric-value nn))
	     top bot)
	 (setq oldbuf (current-buffer)
	       newbuf (or (calc-find-writable-buffer (buffer-list) 0)
			  (calc-find-writable-buffer (buffer-list) 1)
			  (error "No other buffer")))
	 (cond ((and (or (null nn)
			 (consp nn))
		     (= (calc-substack-height 0)
			(1- (calc-substack-height 1))))
		(calc-cursor-stack-index 1)
		(if (looking-at
		     (if calc-line-numbering "[0-9]+: *[^ \n]" " *[^ \n]"))
		    (goto-char (1- (match-end 0))))
		(setq top (point))
		(calc-cursor-stack-index 0)
		(setq bot (1- (point))))
	       ((> n 0)
		(calc-cursor-stack-index n)
		(setq top (point))
		(calc-cursor-stack-index 0)
		(setq bot (point)))
	       ((< n 0)
		(calc-cursor-stack-index (- n))
		(setq top (point))
		(calc-cursor-stack-index (1- (- n)))
		(setq bot (point)))
	       (t
		(goto-char (point-min))
		(forward-line 1)
		(setq top (point))
		(calc-cursor-stack-index 0)
		(setq bot (point))))
	 (save-excursion
	   (set-buffer newbuf)
	   (if (consp nn)
	       (kill-region (region-beginning) (region-end)))
	   (push-mark (point) t)
	   (insert-buffer-substring oldbuf top bot)
	   (if (get-buffer-window (current-buffer))
	       (set-window-point (get-buffer-window (current-buffer))
				 (point)))))))
    (if (consp nn)
	(progn
	  (calc-quit)
	  (switch-to-buffer newbuf))))
)

;;; First, require that buffer is visible and does not begin with "*"
;;; Second, require only that it not begin with "*Calc"
(defun calc-find-writable-buffer (buf mode)
  (and buf
       (if (or (string-match "\\`\\( .*\\|\\*Calc.*\\)"
			     (buffer-name (car buf)))
	       (and (= mode 0)
		    (or (string-match "\\`\\*.*" (buffer-name (car buf)))
			(not (get-buffer-window (car buf))))))
	   (calc-find-writable-buffer (cdr buf) mode)
	 (car buf)))
)

(defun calc-edit (n)
  "Edit the top entry on the stack using normal Emacs editing commands.
With a positive numeric prefix, edit the top N elements of the stack.
With a zero prefix, edit all stack elements.
Type RET or LFD or C-c C-c to finish editing."
  (interactive "p")
  (calc-wrapper
   (if (= n 0)
       (setq n (calc-stack-size)))
   (if (< n 0)
       (error "Argument must be positive or zero"))
   (let ((list (mapcar (function (lambda (x) (math-format-flat-expr x 0)))
		       (calc-top-list n))))
     (calc-edit-mode (list 'calc-finish-stack-edit n))
     (while list
       (insert (car list) "\n")
       (setq list (cdr list)))))
  (calc-show-edit-buffer)
)

(defvar calc-edit-mode-map nil "Keymap for use by the calc-edit command.")
(if calc-edit-mode-map
    ()
  (setq calc-edit-mode-map (make-sparse-keymap))
  (define-key calc-edit-mode-map "\n" 'calc-edit-finish)
  (define-key calc-edit-mode-map "\r" 'calc-edit-finish)
  (define-key calc-edit-mode-map "\C-c\C-c" 'calc-edit-finish)
)

(defun calc-edit-mode (&optional handler)
  "Calculator editing mode.  Press RET, LFD, or C-c C-c to finish.
To cancel the edit, simply kill the *Calc Edit* buffer."
  (interactive)
  (or handler
      (error "This command can be used only indirectly through calc-edit."))
  (let ((oldbuf (current-buffer))
	(buf (get-buffer-create "*Calc Edit*")))
    (set-buffer buf)
    (kill-all-local-variables)
    (use-local-map calc-edit-mode-map)
    (setq buffer-read-only nil)
    (setq truncate-lines nil)
    (setq major-mode 'calc-edit-mode)
    (setq mode-name "Calc Edit")
    (run-hooks 'calc-edit-mode-hook)
    (make-local-variable 'calc-original-buffer)
    (setq calc-original-buffer oldbuf)
    (make-local-variable 'calc-edit-handler)
    (setq calc-edit-handler handler)
    (make-local-variable 'calc-restore-trail)
    (setq calc-restore-trail calc-display-trail)
    (erase-buffer)
    (insert "Calc Edit Mode.  Press RET to finish.  Press C-x k RET to cancel.\n"))
)
(put 'calc-edit-mode 'mode-class 'special)

(defun calc-show-edit-buffer ()
  (switch-to-buffer (get-buffer-create "*Calc Edit*"))
  (if (and (< (window-width) (screen-width))
	   calc-display-trail)
      (let ((win (get-buffer-window (calc-trail-buffer))))
	(if win
	    (delete-window win))))
  (set-buffer-modified-p nil)
  (goto-char (point-min))
  (forward-line 1)
)

(defun calc-edit-finish ()
  "Finish calc-edit mode.  Parse buffer contents and push them on the stack."
  (interactive)
  (or (and (boundp 'calc-original-buffer)
	   (boundp 'calc-edit-handler)
	   (boundp 'calc-restore-trail)
	   (eq major-mode 'calc-edit-mode))
      (error "This command is valid only in buffers created by calc-edit."))
  (let ((buf (current-buffer))
	(original calc-original-buffer)
	(disp-trail calc-restore-trail))
    (save-excursion
      (set-buffer original)
      (if (not (eq major-mode 'calc-mode))
	  (error "Original calculator buffer has been corrupted.")))
    (goto-char (point-min))
    (if (looking-at "Calc Edit")
	(forward-line 1))
    (if (buffer-modified-p)
	(eval calc-edit-handler))
    (switch-to-buffer original)
    (kill-buffer buf)
    (calc-wrapper
     (if disp-trail
	 (calc-trail-display 1 t))))
)

(defun calc-finish-stack-edit (num)
  (let ((buf (current-buffer))
	(str (buffer-substring (point) (point-max)))
	(start (point))
	pos)
    (while (setq pos (string-match "\n." str))
      (aset str pos ?\,))
    (set-buffer calc-original-buffer)
    (let ((vals (math-read-exprs str)))
      (if (eq (car-safe vals) 'error)
	  (progn
	    (set-buffer buf)
	    (goto-char (+ start (nth 1 vals)))
	    (error (nth 2 vals))))
      (calc-wrapper
       (calc-enter-result num "edit" vals))))
)




;;;; [calc-ext.el]

;;; Algebra commands.

(defun calc-a-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("Simplify, Extended-simplify; eXpand, Collect"
     "Derivative, Integral, Taylor; suBstitute; Rewrite"
     "SHIFT + Solve; Integral-limit"
     "relations: =, # (not =), <, >, [ (< or =), ] (> or =)"
     "logical: & (and), | (or), ! (not); misc: { (in-set)")
   "algebra" ?a)
)

;;;; [calc-alg.el]

(defun calc-simplify ()
  "Simplify the formula on top of the stack."
  (interactive)
  (calc-slow-wrapper
   (calc-with-default-simplification
    (calc-enter-result 1 "simp" (math-simplify (calc-top-n 1)))))
)

(defun calc-simplify-extended ()
  "Simplify the formula on top of the stack.
This allows some \"dangerous\" simplifications, such as \"(a^b)^c -> a^(b c)\"
even if c is a non-integer, and \"arcsin(sin(x)) -> x\"."
  (interactive)
  (calc-slow-wrapper
   (calc-with-default-simplification
    (calc-enter-result 1 "esmp" (math-simplify-extended (calc-top-n 1)))))
)

(defun calc-expand (n)
  "Expand the formula on top of the stack using the distributive law.
With a numeric prefix argument, expand only that many times, then stop.
With a negative prefix, expand only that many nesting-levels down."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 1 "expa" (math-expand-tree
				(calc-top-n 1)
				(and n (prefix-numeric-value n)))))
)

(defun calc-collect (var)
  "Collect terms involving a given variable (or sub-expression).
The result will be expressed like a polynomial.
If you enter a blank line, top of stack is the variable, next-to-top is expr."
  (interactive "sCollect terms involving: ")
  (calc-slow-wrapper
   (if (equal var "")
       (calc-enter-result 2 "clct" (math-collect-terms (calc-top-n 2)
						       (calc-top-n 1)))
     (let ((var (math-read-expr var)))
       (if (eq (car-safe var) 'error)
	   (error "Bad format in expression: %s" (nth 1 var)))
       (calc-enter-result 1 "clct" (math-collect-terms (calc-top-n 1)
						       var)))))
)

(defun calc-substitute (&optional oldname newname)
  "Substitute all occurrences of a given sub-expression with another.
If you enter a blank line for \"old\", top of stack is the new expr,
next-to-top is the old expr, and third is the target expr.
If you enter a blank line for \"new\" only, top of stack is the new
expr and next-to-top is the target expr."
  (interactive "sSubstitute old: ")
  (calc-slow-wrapper
   (let (old new (num 1) expr)
     (if (or (equal oldname "") (null oldname))
	 (setq new (calc-top-n 1)
	       old (calc-top-n 2)
	       expr (calc-top-n 3)
	       num 3)
       (or newname
	   (setq newname (read-string (concat "Substitute old: "
					      oldname
					      ", new: ")
				      oldname)))
       (if (or (equal newname "") (null newname))
	   (setq new (calc-top-n 1)
		 expr (calc-top-n 2)
		 num 2)
	 (setq new (if (stringp newname) (math-read-expr newname) newname))
	 (if (eq (car-safe new) 'error)
	     (error "Bad format in expression: %s" (nth 1 new)))
	 (setq expr (calc-top-n 1)))
       (setq old (if (stringp oldname) (math-read-expr oldname) oldname))
       (if (eq (car-safe old) 'error)
	   (error "Bad format in expression: %s" (nth 1 old)))
       (or (math-expr-contains expr old)
	   (error "No occurrences found.")))
     (calc-enter-result num "sbst" (math-expr-subst expr old new))))
)

(defun calc-rewrite (rules many)
  "Perform substitutions in an expression using pattern-based rewrite rules.
This command prompts for the rule(s) to use, which should be either a
vector of the form [LHS, RHS] or [LHS, RHS, COND], or a vector of such
vectors, or a variable which contains a rules vector.  If you enter a
blank line, the rules are taken from top-of-stack, expr from next-to-top.
In each rule, LHS is a formula in which each unique variable name stands
for any sub-expression, RHS is a formula typically also containing these
variables, and COND is an optional formula which specifies a condition.
A rule applies to an expression if the LHS is the same as the expression
where each variable in LHS corresponds to some sub-expression, and if COND
evaluates to a non-zero real number (under those assignments of the
variables).  If so, the expression is replaced by RHS with any variables
that occur in LHS expanded.
By default, the rules are applied once to the any part of the expression
which matches (but preferably to the whole expression).  With a positive
numeric prefix argument, the rules are applied up to that many times, or
until no further changes can be made.  With a negative prefix argument,
the rules are applied that many times but only at the top level of the
expression."
  (interactive "sRewrite rule(s): \np")
  (calc-slow-wrapper
   (let (n expr)
     (if (or (null rules) (equal rules ""))
	 (setq expr (calc-top-n 2)
	       rules (calc-top-n 1)
	       n 2)
       (setq rules (if (stringp rules) (math-read-expr rules) rules))
       (if (eq (car-safe rules) 'error)
	   (error "Bad format in expression: %s" (nth 1 rules)))
       (setq expr (calc-top-n 1)
	     n 1))
     (and (eq many 0) (setq many 25))
     (calc-enter-result n "rwrt" (math-rewrite expr rules many))))
)

;;;; [calc-alg-2.el]

(defun calc-derivative (var)
  "Differentiate the formula on top of the stack with respect to a variable.
If you enter a blank line, top of stack is the variable, next-to-top is expr.
With Hyperbolic flag, performs a total derivative: all variables are
considered to be inter-dependent.  Otherwise, all variables except VAR
are treated as constant."
  (interactive "sDifferentiate with respect to: ")
  (calc-slow-wrapper
   (let ((func (if (calc-is-hyperbolic) 'calcFunc-tderiv 'calcFunc-deriv)))
     (if (equal var "")
	 (calc-enter-result 2 "derv" (list func
					   (calc-top-n 2)
					   (calc-top-n 1)))
       (let ((var (math-read-expr var)))
	 (if (eq (car-safe var) 'error)
	     (error "Bad format in expression: %s" (nth 1 var)))
	 (calc-enter-result 1 "derv" (list func
					   (calc-top-n 1)
					   var))))))
)

(defun calc-integral (var)
  "Integrate the formula on top of the stack with respect to a variable.
This computes an indefinite integral.
If you enter a blank line, top of stack is the variable, next-to-top is expr."
  (interactive "sIntegration variable: ")
  (calc-slow-wrapper
   (if (equal var "")
       (calc-enter-result 2 "intg" (list 'calcFunc-integ
					 (calc-top-n 2)
					 (calc-top-n 1)))
     (let ((var (math-read-expr var)))
       (if (eq (car-safe var) 'error)
	   (error "Bad format in expression: %s" (nth 1 var)))
       (calc-enter-result 1 "intg" (list 'calcFunc-integ
					 (calc-top-n 1)
					 var)))))
)

(defun calc-integral-limit (n)
  "Display current integral limit, or set the limit to N levels."
  (interactive "P")
  (calc-wrapper
   (if (consp n)
       (calc-pop-push-record 0 "prec" calc-integral-limit)
     (if (and (integerp n) (> n 0))
	 (progn
	   (setq calc-integral-limit (prefix-numeric-value n))
	   (calc-record calc-integral-limit "ilim")))
     (message "Integration nesting limit is %d levels." calc-integral-limit)))
)

(defun calc-solve-for (var)
  "Solve an equation for a given variable.
If the top-of-stack is not of the form A = B, it is treated as A = 0.
If you enter a blank line, top of stack is the variable, next-to-top is eqn.
With Hyperbolic flag, finds a fully general solution in which n1, n2, ...
represent independent arbitrary integers and s1, s2, ... are independent
arbitrary signs.
With Inverse flag, computes the inverse of the expression, written in terms
of the original variable."
  (interactive "sVariable to solve for: ")
  (calc-slow-wrapper
   (let ((func (if (calc-is-inverse)
		   (if (calc-is-hyperbolic) 'calcFunc-ffinv 'calcFunc-finv)
		 (if (calc-is-hyperbolic) 'calcFunc-fsolve 'calcFunc-solve))))
     (if (equal var "")
	 (calc-enter-result 2 "solv" (list func
					   (calc-top-n 2)
					   (calc-top-n 1)))
       (let ((var (math-read-expr var)))
	 (if (eq (car-safe var) 'error)
	     (error "Bad format in expression: %s" (nth 1 var)))
	 (calc-enter-result 1 "solv" (list func
					   (calc-top-n 1)
					   var))))))
)

(defun calc-taylor (var nterms)
  "Compute the Taylor expansion of a formula."
  (interactive "sTaylor expansion variable: \nNNumber of terms: ")
  (calc-slow-wrapper
   (let ((var (math-read-expr var)))
     (if (eq (car-safe var) 'error)
	 (error "Bad format in expression: %s" (nth 1 var)))
     (calc-enter-result 1 "tylr" (list 'calcFunc-taylor
				       (calc-top-n 1)
				       var
				       nterms))))
)


;;;; [calc-prog.el]

(defun calc-equal-to (arg)
  "Return 1 if numbers are equal, 0 if they are unequal."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "eq" 'calcFunc-eq arg))
)

(defun calc-not-equal-to (arg)
  "Return 1 if numbers are unequal, 0 if they are equal."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "neq" 'calcFunc-neq arg))
)

(defun calc-less-than (arg)
  "Return 1 if numbers are less, 0 if they are not less."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "lt" 'calcFunc-lt arg))
)

(defun calc-greater-than (arg)
  "Return 1 if numbers are greater, 0 if they are not greater."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "gt" 'calcFunc-gt arg))
)

(defun calc-less-equal (arg)
  "Return 1 if numbers are less than or equal to, 0 if they are not leq."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "leq" 'calcFunc-leq arg))
)

(defun calc-greater-equal (arg)
  "Return 1 if numbers are greater than or equal to, 0 if they are not geq."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "geq" 'calcFunc-geq arg))
)

(defun calc-in-set (arg)
  "Return 1 if a number is in the set specified by a vector or interval.
Return 0 if it is not in the set."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "in" 'calcFunc-in arg))
)

(defun calc-logical-and (arg)
  "Return 1 if both numbers are non-zero, 0 if either is zero."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "land" 'calcFunc-land arg 1))
)

(defun calc-logical-or (arg)
  "Return 1 if either number is non-zero, 0 if both are zero."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "lor" 'calcFunc-lor arg 0))
)

(defun calc-logical-not (arg)
  "Return 1 if a number is zero, 0 if it is non-zero."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op "lnot" 'calcFunc-lnot arg))
)




;;;; [calc-ext.el]

;;; b-prefix binary commands.

(defun calc-b-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("And, Or, Xor, Diff, Not; Wordsize, Clip"
     "Lshift, Rshift, roTate; SHIFT + signed Lshift, Rshift")
   "binary" ?b)
)

;;;; [calc-bin.el]

(defun calc-and (n)
  "Compute the bitwise binary AND of the top two elements on the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 2 "and"
		      (append '(calcFunc-and)
			      (calc-top-list-n 2)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-or (n)
  "Compute the bitwise binary OR of the top two elements on the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 2 "or"
		      (append '(calcFunc-or)
			      (calc-top-list-n 2)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-xor (n)
  "Compute the bitwise binary XOR of the top two elements on the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 2 "xor"
		      (append '(calcFunc-xor)
			      (calc-top-list-n 2)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-diff (n)
  "Compute the bitwise binary AND-NOT of the top two elements on the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 2 "diff"
		      (append '(calcFunc-diff)
			      (calc-top-list-n 2)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-not (n)
  "Compute the bitwise binary NOT of the top element on the stack.
A prefix argument specifies word size to use for this operation (instead of
the default).  The result is clipped to fit in the word size."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 1 "not"
		      (append '(calcFunc-not)
			      (calc-top-list-n 1)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-lshift-binary (n)
  "Shift the top element on the stack one bit left in binary.
With a numeric prefix argument, shift N bits left.
With a negative prefix argument, logically shift -N bits right.
The result is clipped to the current word size."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 1 "lsh"
		      (append '(calcFunc-lsh)
			      (calc-top-list-n 1)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-rshift-binary (n)
  "Shift the top element on the stack one bit right in binary (logically).
With a numeric prefix argument, logically shift N bits right.
With a negative prefix argument, shift -N bits left.
The result is clipped to the current word size."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 1 "rsh"
		      (append '(calcFunc-rsh)
			      (calc-top-list-n 1)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-lshift-arith (n)
  "Shift the top element on the stack one bit left in binary.
With a numeric prefix argument, shift N bits left.
With a negative prefix argument, arithmetically shift -N bits right.
The result is clipped to the current word size."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 1 "ash"
		      (append '(calcFunc-ash)
			      (calc-top-list-n 1)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-rshift-arith (n)
  "Shift the top element on the stack one bit right in binary (arithmetically).
With a numeric prefix argument, arithmetically shift N bits right.
With a negative prefix argument, shift -N bits left.
The result is clipped to the current word size."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 1 "rash"
		      (append '(calcFunc-rash)
			      (calc-top-list-n 1)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-rotate-binary (n)
  "Rotate the top element on the Calculator stack one bit left in binary.
With a numeric prefix argument, rotate N bits left.
With a negative prefix argument, rotate -N bits right.
The result is clipped to the current word size."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 1 "rot"
		      (append '(calcFunc-rot)
			      (calc-top-list-n 1)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-clip (n)
  "Clip the integer at the top of the stack to the current binary word size.
A prefix argument specifies an alternate word size to use."
  (interactive "P")
  (calc-slow-wrapper
   (calc-enter-result 1 "clip"
		      (append '(calcFunc-clip)
			      (calc-top-list-n 1)
			      (and n (list (prefix-numeric-value n))))))
)

(defun calc-word-size (n)
  "Display current word size for Calculator binary operations, or set to N bits.
\(All other bitwise operations accept a prefix argument to override this
default size.)
If N is negative, use |N|-bit, 2's complement arithmetic."
  (interactive "P")
  (calc-wrapper
   (if n
       (progn
	 (setq calc-word-size (prefix-numeric-value n)
	       calc-previous-modulo (math-power-of-2
				     (math-abs calc-word-size)))
	 (if calc-leading-zeros
	     (calc-refresh))))
   (if (< calc-word-size 0)
       (message "Binary word size is %d bits (2's complement)."
		(- calc-word-size))
     (message "Binary word size is %d bits." calc-word-size)))
)




;;;; [calc-ext.el]

;;; Conversions.

(defun calc-c-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("Deg, Rad, HMS; Float; Polar; Clean, 1, 2, 3"
     "SHIFT + Fraction")
   "convert" ?c)
)

(defun calc-clean (n)
  "Clean up the number at the top of the Calculator stack.
Re-round to current precision, or to that specified by a prefix argument.
This temporarily cancels no-simplify mode, if necessary."
  (interactive "P")
  (calc-slow-wrapper
   (calc-with-default-simplification
    (calc-enter-result 1 "cln"
		       (if n
			   (let ((n (prefix-numeric-value n)))
			     (list 'calcFunc-clean
				   (calc-top-n 1)
				   (if (< n 0)
				       (+ n calc-internal-prec)
				     n)))
			 (list 'calcFunc-clean (calc-top-n 1))))))
)

(defun calc-clean-1 ()
  "Clean up the number on the top of the stack by rounding off one digit."
  (interactive)
  (calc-clean -1)
)

(defun calc-clean-2 ()
  "Clean up the number on the top of the stack by rounding off two digits."
  (interactive)
  (calc-clean -2)
)

(defun calc-clean-3 ()
  "Clean up the number on the top of the stack by rounding off three digits."
  (interactive)
  (calc-clean -3)
)

(defun calc-float (arg)
  "Convert the top element of the Calculator stack to floating-point form."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "flt" 'calcFunc-float arg))
)

;;;; [calc-frac.el]

(defun calc-fraction (arg)
  "Convert the top element of the Calculator stack to fractional form.
For floating-point arguments, the fraction is exactly equivalent within
the limits of the current precision.
If a numeric prefix N is supplied, it is used as a tolerance value.
If N is zero, top-of-stack contains a tolerance value.
If the tolerance is a positive integer, the fraction will be accurate to
within that many significant figures.
If the tolerance is a non-positive integer, the fraction will be accurate to
within that many figures less than the current precision.
If the tolerance is a floating-point number, the fraction will be accurate
to within that absolute value."
  (interactive "P")
  (calc-slow-wrapper
   (if (eq arg 0)
       (calc-enter-result 2 "frac" (list 'calcFunc-frac
					 (calc-top-n 2)
					 (calc-top-n 1)))
     (calc-enter-result 1 "frac" (list 'calcFunc-frac
				       (calc-top-n 1)
				       (prefix-numeric-value (or arg 0))))))
)

;;;; [calc-forms.el]

(defun calc-to-hms (arg)
  "Convert the top element of the stack to hours-minutes-seconds form.
Number is interpreted as degrees or radians according to current mode."
  (interactive "P")
  (calc-wrapper
   (if (calc-is-inverse)
       (if (eq calc-angle-mode 'rad)
	   (calc-unary-op ">rad" 'calcFunc-rad arg)
	 (calc-unary-op ">deg" 'calcFunc-deg arg))
     (calc-unary-op ">hms" 'calcFunc-hms arg)))
)

(defun calc-from-hms (arg)
  "Convert the top element of the stack from hours-minutes-seconds form."
  (interactive "P")
  (calc-invert-func)
  (calc-to-hms arg)
)

;;;; [calc-math.el]

(defun calc-to-degrees (arg)
  "Convert the top element of the stack from radians or HMS to degrees."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op ">deg" 'calcFunc-deg arg))
)

(defun calc-to-radians (arg)
  "Convert the top element of the stack from degrees or HMS to radians."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op ">rad" 'calcFunc-rad arg))
)

;;;; [calc-cplx.el]

(defun calc-polar ()
  "Convert the top element of the stack to polar complex form."
  (interactive)
  (calc-slow-wrapper
   (let ((arg (calc-top-n 1)))
     (if (or (calc-is-inverse)
	     (eq (car-safe arg) 'polar))
	 (calc-enter-result 1 "p-r" (list 'calcFunc-rect arg))
       (calc-enter-result 1 "r-p" (list 'calcFunc-polar arg)))))
)



;;;; [calc-ext.el]

;;; d-prefix mode commands.

(defun calc-d-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("Group, \",\"; Normal, Fix, Sci, Eng, \".\""
     "Radix, Zeros, 2, 8, 0, 6; Over; Hms; Complex, I, J"
     "Why; Line-nums, line-Breaks; <, =, > (justify)"
     "Truncate, [, ]; ` (align); ~ (refresh)"
     "SHIFT + language: Normal, One-line, Big, Unformatted"
     "SHIFT + language: C, Pascal, Fortran, TeX, Mathematica")
   "display" ?d)
)

;;;; [calc-bin.el]

(defun calc-radix (n)
  "Set the display radix for integers and rationals to N, from 2 to 36."
  (interactive "NDisplay radix (2-36): ")
  (calc-wrapper
   (if (and (>= n 2) (<= n 36))
       (progn
	 (setq calc-number-radix n)
	 (setq-default calc-number-radix n)))  ; so minibuffer sees it
   (calc-refresh)
   (message "Number radix is %d." calc-number-radix))
)

(defun calc-decimal-radix ()
  "Set the display radix for integers and rationals to decimal."
  (interactive)
  (calc-radix 10)
)

(defun calc-binary-radix ()
  "Set the display radix for integers and rationals to binary."
  (interactive)
  (calc-radix 2)
)

(defun calc-octal-radix ()
  "Set the display radix for integers and rationals to octal."
  (interactive)
  (calc-radix 8)
)

(defun calc-hex-radix ()
  "Set the display radix for integers and rationals to hex."
  (interactive)
  (calc-radix 16)
)

(defun calc-leading-zeros (n)
  "Toggle display of leading zeros in integers."
  (interactive "P")
  (calc-wrapper
   (setq calc-leading-zeros (if n
				(> (prefix-numeric-value n) 0)
			      (not calc-leading-zeros)))
   (calc-refresh))
)

;;;; [calc-mode.el]

(defun calc-line-numbering (n)
  "Toggle display of line numbers in the Calculator stack.
With positive numeric prefix, turn mode on.
With 0 or negative prefix, turn mode off."
  (interactive "P")
  (calc-wrapper
   (setq calc-line-numbering (if n
				 (> (prefix-numeric-value n) 0)
			       (not calc-line-numbering)))
   (calc-refresh))
)

(defun calc-line-breaking (n)
  "Toggle breaking of long values across multiple lines in Calculator stack.
With positive numeric prefix, turn mode on.
With 0 or negative prefix, turn mode off."
  (interactive "P")
  (calc-wrapper
   (setq calc-line-breaking (if n
				(> (prefix-numeric-value n) 0)
			      (not calc-line-breaking)))
   (calc-refresh))
)

(defun calc-display-strings (n)
  "Toggle display of vectors of byte-sized integers as strings.
With positive numeric prefix, turn mode on.
With 0 or negative prefix, turn mode off."
  (interactive "P")
  (calc-wrapper
   (setq calc-display-strings (if n
				  (> (prefix-numeric-value n) 0)
				(not calc-display-strings)))
   (calc-refresh))
)

(defun calc-left-justify ()
  "Display stack entries left-justified in the window."
  (interactive)
  (calc-wrapper
   (setq calc-display-just nil)
   (calc-refresh))
)

(defun calc-center-justify ()
  "Display stack entries centered in the window."
  (interactive)
  (calc-wrapper
   (setq calc-display-just 'center)
   (calc-refresh))
)

(defun calc-right-justify ()
  "Display stack entries right-justified in the window."
  (interactive)
  (calc-wrapper
   (setq calc-display-just 'right)
   (calc-refresh))
)

(defun calc-auto-why (n)
  "Toggle automatic explanations of why results were left in symbolic form.
This can always be requested explicitly with the calc-why command.
With positive numeric prefix, turn mode on.
With 0 or negative prefix, turn mode off."
  (interactive "P")
  (calc-wrapper
   (setq calc-auto-why (if n
			   (> (prefix-numeric-value n) 0)
			 (not calc-auto-why)))
   (if calc-auto-why
       (message "Automatically executing a \"why\" command when appropriate.")
     (message "User must execute a \"why\" command to explain unsimplified results.")))
)

(defun calc-group-digits (n)
  "Toggle grouping of digits, or set group size to N digits.
With numeric prefix 0, display current setting.
With numeric prefix -1, disable grouping.
With other negative prefix, group after decimal point as well as before."
  (interactive "P")
  (calc-wrapper
   (if (consp n)
       (calc-pop-push-record 0 "grp" (cond ((null calc-group-digits) -1)
					   ((eq calc-group-digits t)
					    (if (memq calc-number-radix
						      '(2 16)) 4 3))
					   (t calc-group-digits)))
     (if n
	 (let ((n (prefix-numeric-value n)))
	   (cond ((or (> n 0) (< n -1))
		  (setq calc-group-digits n))
		 ((= n -1)
		  (setq calc-group-digits nil))))
       (setq calc-group-digits (not calc-group-digits)))
     (calc-refresh)
     (cond ((null calc-group-digits)
	    (message "Grouping is off."))
	   ((integerp calc-group-digits)
	    (message "Grouping every %d digits." (math-abs calc-group-digits)))
	   (t
	    (message "Grouping is on.")))))
)

(defun calc-group-char (ch)
  "Set the character to be used for grouping digits in calc-group-digits mode."
  (interactive "cGrouping character: ")
  (calc-wrapper
   (or (>= ch 32)
       (error "Control characters not allowed for grouping"))
   (setq calc-group-char (char-to-string ch))
   (if calc-group-digits
       (calc-refresh)))
)

(defun calc-point-char (ch)
  "Set the character to be used as the decimal point."
  (interactive "cCharacter to use as decimal point: ")
  (calc-wrapper
   (or (>= ch 32)
       (error "Control characters not allowed as decimal point"))
   (setq calc-point-char (char-to-string ch))
   (calc-refresh))
)

(defun calc-normal-notation (n)
  "Set normal (floating) notation for floating-point numbers.
With argument N > 0, round to N significant digits.
With argument -N < 0, round to current precision - N significant digits."
  (interactive "P")
  (calc-wrapper
   (setq calc-float-format (list 'float
				 (if n (prefix-numeric-value n) 0)))
   (setq calc-full-float-format (list 'float 0))
   (calc-refresh))
)

(defun calc-fix-notation (n)
  "Set fixed-point notation for floating-point numbers."
  (interactive "NDigits after decimal point: ")
  (calc-wrapper
   (let ((n (prefix-numeric-value n)))
     (setq calc-float-format (list 'fix n)))
   (setq calc-full-float-format (list 'float 0))
   (calc-refresh))
)

(defun calc-sci-notation (n)
  "Set scientific notation for floating-point numbers.
With argument N > 0, round to N significant digits.
With argument -N < 0, round to current precision - N significant digits."
  (interactive "P")
  (calc-wrapper
   (let ((n (if n (prefix-numeric-value n) 0)))
     (setq calc-float-format (list 'sci n)))   ; (if (> n 0) (1+ n) n)
   (setq calc-full-float-format (list 'sci 0))
   (calc-refresh))
)

(defun calc-eng-notation (n)
  "Set engineering notation for floating-point numbers.
With argument N > 0, round to N significant digits.
With argument -N < 0, round to current precision - N significant digits."
  (interactive "P")
  (calc-wrapper
   (let ((n (if n (prefix-numeric-value n) 0)))
     (setq calc-float-format (list 'eng n)))
   (setq calc-full-float-format (list 'eng 0))
   (calc-refresh))
)

;;;; [calc-cplx.el]

(defun calc-complex-notation ()
  "Set (x,y) notation for display of complex numbers."
  (interactive)
  (calc-wrapper
   (setq calc-complex-format nil)
   (calc-refresh))
)

(defun calc-i-notation ()
  "Set x+yi notation for display of complex numbers."
  (interactive)
  (calc-wrapper
   (setq calc-complex-format 'i)
   (calc-refresh))
)

(defun calc-j-notation ()
  "Set x+yj notation for display of complex numbers."
  (interactive)
  (calc-wrapper
   (setq calc-complex-format 'j)
   (calc-refresh))
)

;;;; [calc-frac.el]

(defun calc-over-notation (fmt)
  "Set notation used for fractions.  Argument should be one of :, ::, /, //, :/.
\(During numeric entry, the : key is always used.)"
  (interactive "sFraction separator (:, ::, /, //, :/): ")
  (calc-wrapper
   (if (string-match "\\`[^ ][^ ]?\\'" fmt)
       (setq calc-frac-format fmt)
     (error "Bad fraction separator format."))
   (calc-refresh))
)

(defun calc-slash-notation (n)
  "Set \"a/b\" notation for fractions.
With a prefix argument, set \"a/b/c\" notation."
  (interactive "P")
  (calc-wrapper
   (setq calc-frac-format (if n "//" "/")))
)

;;;; [calc-forms.el]

(defun calc-hms-notation (fmt)
  "Set notation used for hours-minutes-seconds values.
Argument should be something like: hms, deg m s, o'\".
\(During numeric entry, @ ' \", o ' \", or h ' \" format must be used.)"
  (interactive "sHours-minutes-seconds format (hms, @ ' \", etc.): ")
  (calc-wrapper
   (if (string-match "\\`\\([^,; ]+\\)\\([,; ]*\\)\\([^,; ]\\)\\([,; ]*\\)\\([^,; ]\\)\\'" fmt)
       (progn
	 (setq calc-hms-format (concat "%s" (math-match-substring fmt 1)
				       (math-match-substring fmt 2)
				       "%s" (math-match-substring fmt 3)
				       (math-match-substring fmt 4)
				       "%s" (math-match-substring fmt 5)))
	 (setq-default calc-hms-format calc-hms-format))  ; for minibuffer
     (error "Bad hours-minutes-seconds format."))
   (calc-refresh))
)

;;;; [calc-mode.el]

(defun calc-truncate-stack (n &optional rel)
  "Treat cursor line as \"top of stack\" for all further operations.
Objects below this line are frozen, but still displayed."
  (interactive "P")
  (calc-wrapper
   (let ((oldtop calc-stack-top)
	 (newtop calc-stack-top))
     (calc-record-undo (list 'set 'saved-stack-top calc-stack-top))
     (let ((calc-stack-top 0)
	   (nn (prefix-numeric-value n)))
       (setq newtop
	     (if n
		 (progn
		   (if rel
		       (setq nn (+ oldtop nn))
		     (if (< nn 0)
			 (setq nn (+ nn (calc-stack-size)))
		       (setq nn (1+ nn))))
		   (if (< nn 1)
		       1
		     (if (> nn (calc-stack-size))
			 (calc-stack-size)
		       nn)))
	       (max 1 (calc-locate-cursor-element (point)))))
       (if (= newtop oldtop)
	   ()
	 (calc-pop-stack 1 oldtop)
	 (calc-push-list '(top-of-stack) newtop)
	 (if calc-line-numbering
	     (calc-refresh))))
     (calc-record-undo (list 'set 'saved-stack-top 0))
     (setq calc-stack-top newtop)))
)

(defun calc-truncate-up (n)
  (interactive "p")
  (calc-truncate-stack n t)
)

(defun calc-truncate-down (n)
  (interactive "p")
  (calc-truncate-stack (- n) t)
)

(defun calc-display-raw ()
  (interactive)
  (calc-wrapper
   (setq calc-display-raw (not (eq calc-display-raw t)))
   (calc-refresh)
   (if calc-display-raw
       (message "Press d ' again to cancel \"raw\" display mode.")))
)

(defun calc-display-unformatted ()
  (interactive)
  (calc-wrapper
   (setq calc-display-raw (if (eq calc-display-raw 0) nil 0))
   (calc-refresh)
   (if calc-display-raw
       (message "Press d \" again to cancel \"unformatted\" display mode.")))
)



;;;; [calc-lang.el]

;;; Alternate entry/display languages.

(defun calc-set-language (lang &optional option no-refresh)
  (setq calc-language lang
	calc-language-option (and option (prefix-numeric-value option))
	math-expr-opers (or (get lang 'math-oper-table) math-standard-opers)
	math-expr-function-mapping (get lang 'math-function-table)
	math-expr-variable-mapping (get lang 'math-variable-table)
	calc-language-input-filter (get lang 'math-input-filter)
	calc-language-output-filter (get lang 'math-output-filter)
	calc-vector-brackets (or (get lang 'math-vector-brackets) "[]")
	calc-complex-format (get lang 'math-complex-format)
	calc-radix-formatter (get lang 'math-radix-formatter)
	calc-function-open (or (get lang 'math-function-open) "(")
	calc-function-close (or (get lang 'math-function-close) ")"))
  (or no-refresh
      (calc-refresh))
)

(defun calc-normal-language ()
  "Set normal entry and display notation."
  (interactive)
  (calc-wrapper
   (calc-set-language nil))
)

(defun calc-flat-language ()
  "Set normal entry and display notation, with one-line display of matrices."
  (interactive)
  (calc-wrapper
   (calc-set-language 'flat))
)

(defun calc-big-language ()
  "Set big-format display notation."
  (interactive)
  (calc-wrapper
   (calc-set-language 'big))
)

(defun calc-unformatted-language ()
  "Set normal entry and display notation with no operators: add(a, mul(b,c))."
  (interactive)
  (calc-wrapper
   (calc-set-language 'unform))
)


(defun calc-c-language ()
  "Set C-language entry and display notation."
  (interactive)
  (calc-wrapper
   (calc-set-language 'c))
)

(put 'c 'math-oper-table
  '( ( "u+"    ident	     -1 1000 )
     ( "u-"    neg	     -1 1000 )
     ( "u!"    calcFunc-lnot -1 1000 )
     ( "~"     calcFunc-not  -1 1000 )
     ( "*"     *	     190 191 )
     ( "/"     /	     190 191 )
     ( "%"     %	     190 191 )
     ( "+"     +	     180 181 )
     ( "-"     -	     180 181 )
     ( "<<"    calcFunc-lsh  170 171 )
     ( ">>"    calcFunc-rsh  170 171 )
     ( "<"     calcFunc-lt   160 161 )
     ( ">"     calcFunc-gt   160 161 )
     ( "<="    calcFunc-leq  160 161 )
     ( ">="    calcFunc-geq  160 161 )
     ( "=="    calcFunc-eq   150 151 )
     ( "!="    calcFunc-neq  150 151 )
     ( "&"     calcFunc-and  140 141 )
     ( "^"     calcFunc-xor  131 130 )
     ( "|"     calcFunc-or   120 121 )
     ( "&&"    calcFunc-land 110 111 )
     ( "||"    calcFunc-lor  100 101 )
     ( "?"     calcFunc-if    91  90 )
     ( "="     calcFunc-assign 81 80 )
)) ; should support full assignments

(put 'c 'math-function-table
  '( ( acos	   . calcFunc-arccos )
     ( acosh	   . calcFunc-arccosh )
     ( asin	   . calcFunc-arcsin )
     ( asinh	   . calcFunc-arcsinh )
     ( atan	   . calcFunc-arctan )
     ( atan2	   . calcFunc-arctan2 )
     ( atanh	   . calcFunc-arctanh )
))

(put 'c 'math-variable-table
  '( ( M_PI	   . var-pi )
     ( M_E	   . var-e )
))

(put 'c 'math-vector-brackets "{}")

(put 'c 'math-radix-formatter
     (function (lambda (r s)
		 (if (= r 16) (format "0x%s" s)
		   (if (= r 8) (format "0%s" s)
		     (format "%d#%s" r s))))))


(defun calc-pascal-language (n)
  "Set Pascal-language entry and display notation."
  (interactive "P")
  (calc-wrapper
   (calc-set-language 'pascal n))
)

(put 'pascal 'math-oper-table
  '( ( "not"   calcFunc-lnot -1 1000 )
     ( "*"     *	     190 191 )
     ( "/"     /	     190 191 )
     ( "and"   calcFunc-and  190 191 )
     ( "div"   calcFunc-idiv 190 191 )
     ( "mod"   %	     190 191 )
     ( "u+"    ident	     -1  185 )
     ( "u-"    neg	     -1  185 )
     ( "+"     +	     180 181 )
     ( "-"     -	     180 181 )
     ( "or"    calcFunc-or   180 181 )
     ( "xor"   calcFunc-xor  180 181 )
     ( "shl"   calcFunc-lsh  180 181 )
     ( "shr"   calcFunc-rsh  180 181 )
     ( "in"    calcFunc-in   160 161 )
     ( "<"     calcFunc-lt   160 161 )
     ( ">"     calcFunc-gt   160 161 )
     ( "<="    calcFunc-leq  160 161 )
     ( ">="    calcFunc-geq  160 161 )
     ( "="     calcFunc-eq   160 161 )
     ( "<>"    calcFunc-neq  160 161 )
     ( ":="    calcFunc-assign 81 80 )
))

(put 'pascal 'math-input-filter 'calc-input-case-filter)
(put 'pascal 'math-output-filter 'calc-output-case-filter)

(defun calc-input-case-filter (str)
  (cond ((or (null calc-language-option) (= calc-language-option 0))
	 str)
	(t
	 (downcase str)))
)

(defun calc-output-case-filter (str)
  (cond ((or (null calc-language-option) (= calc-language-option 0))
	 str)
	((> calc-language-option 0)
	 (upcase str))
	(t
	 (downcase str)))
)


(defun calc-fortran-language (n)
  "Set Fortran-language entry and display notation."
  (interactive "P")
  (calc-wrapper
   (calc-set-language 'fortran n))
)

(put 'fortran 'math-oper-table
  '( ( "**"    ^             201 200 )
     ( "u+"    ident	     -1  191 )
     ( "u-"    neg	     -1  191 )
     ( "*"     *	     190 191 )
     ( "/"     /	     190 191 )
     ( "+"     +	     180 181 )
     ( "-"     -	     180 181 )
))

(put 'fortran 'math-vector-brackets "//")

(put 'fortran 'math-function-table
  '( ( acos	   . calcFunc-arccos )
     ( acosh	   . calcFunc-arccosh )
     ( aimag	   . calcFunc-im )
     ( aint	   . calcFunc-ftrunc )
     ( asin	   . calcFunc-arcsin )
     ( asinh	   . calcFunc-arcsinh )
     ( atan	   . calcFunc-arctan )
     ( atan2	   . calcFunc-arctan2 )
     ( atanh	   . calcFunc-arctanh )
     ( conjg	   . calcFunc-conj )
     ( log	   . calcFunc-ln )
     ( nint	   . calcFunc-round )
     ( real	   . calcFunc-re )
))

(put 'fortran 'math-input-filter 'calc-input-case-filter)
(put 'fortran 'math-output-filter 'calc-output-case-filter)


(defun calc-tex-language (n)
  "Set TeX entry and display notation."
  (interactive "P")
  (calc-wrapper
   (calc-set-language 'tex n))
)

(put 'tex 'math-oper-table
  '( ( "u+"       ident		   -1 1000 )
     ( "u-"       neg		   -1 1000 )
     ( "u|"       calcFunc-abs	   -1    0 )
     ( "|"        ident		    0   -1 )
     ( "\\lfloor" calcFunc-floor   -1    0 )
     ( "\\rfloor" ident             0   -1 )
     ( "\\lceil"  calcFunc-ceil    -1    0 )
     ( "\\rceil"  ident             0   -1 )
     ( "\\pm"	  sdev		   300 300 )
     ( "!"        calcFunc-fact	   210  -1 )
     ( "^"	  ^		   201 200 )
     ( "_"	  calcFunc-subscr  201 200 )
     ( "\\times"  *		   191 190 )
     ( "2x"	  *		   191 190 )
     ( "+"	  +		   180 181 )
     ( "-"	  -		   180 181 )
     ( "\\over"	  /		   170 171 )
     ( "/"	  /		   170 171 )
     ( "\\choose" calcFunc-choose  170 171 )
     ( "\\mod"	  %		   170 171 )
))

(put 'tex 'math-function-table
  '( ( \\arccos	   . calcFunc-arccos )
     ( \\arcsin	   . calcFunc-arcsin )
     ( \\arctan	   . calcFunc-arctan )
     ( \\arg	   . calcFunc-arg )
     ( \\cos	   . calcFunc-cos )
     ( \\cosh	   . calcFunc-cosh )
     ( \\det	   . calcFunc-det )
     ( \\exp	   . calcFunc-exp )
     ( \\gcd	   . calcFunc-gcd )
     ( \\ln	   . calcFunc-ln )
     ( \\log	   . calcFunc-log10 )
     ( \\max	   . calcFunc-max )
     ( \\min	   . calcFunc-min )
     ( \\tan	   . calcFunc-tan )
     ( \\sin	   . calcFunc-sin )
     ( \\sinh	   . calcFunc-sinh )
     ( \\sqrt	   . calcFunc-sqrt )
     ( \\tanh	   . calcFunc-tanh )
     ( \\phi	   . calcFunc-totient )
     ( \\mu	   . calcFunc-moebius )
))

(put 'tex 'math-variable-table
  '( ( \\pi	   . var-pi )
))

(put 'tex 'math-complex-format 'i)


(defun calc-mathematica-language ()
  "Set Mathematica(tm) entry and display notation."
  (interactive)
  (calc-wrapper
   (calc-set-language 'math))
)

(put 'math 'math-oper-table
  '( ( "!"     calcFunc-fact  210 -1 )
     ( "!!"    calcFunc-dfact 210 -1 )
     ( "^"     ^	     201 200 )
     ( "u+"    ident	     -1  197 )
     ( "u-"    neg	     -1  197 )
     ( "/"     /	     195 196 )
     ( "*"     *	     190 191 )
     ( "2x"    *	     190 191 )
     ( "+"     +	     180 181 )
     ( "-"     -	     180 181 )
     ( "<"     calcFunc-lt   160 161 )
     ( ">"     calcFunc-gt   160 161 )
     ( "<="    calcFunc-leq  160 161 )
     ( ">="    calcFunc-geq  160 161 )
     ( "=="    calcFunc-eq   150 151 )
     ( "!="    calcFunc-neq  150 151 )
     ( "&&"    calcFunc-land 110 111 )
     ( "||"    calcFunc-lor  100 101 )
))

(put 'math 'math-function-table
  '( ( Abs	   . calcFunc-abs )
     ( ArcCos	   . calcFunc-arccos )
     ( ArcCosh	   . calcFunc-arccosh )
     ( ArcSin	   . calcFunc-arcsin )
     ( ArcSinh	   . calcFunc-arcsinh )
     ( ArcTan	   . calcFunc-arctan )
     ( ArcTanh	   . calcFunc-arctanh )
     ( Arg	   . calcFunc-arg )
     ( Binomial	   . calcFunc-choose )
     ( Ceiling	   . calcFunc-ceil )
     ( Conjugate   . calcFunc-conj )
     ( Cos	   . calcFunc-cos )
     ( Cosh	   . calcFunc-cosh )
     ( D	   . calcFunc-deriv )
     ( Dt	   . calcFunc-tderiv )
     ( Det	   . calcFunc-det )
     ( Exp	   . calcFunc-exp )
     ( EulerPhi	   . calcFunc-totient )
     ( Floor	   . calcFunc-floor )
     ( Gamma	   . calcFunc-gamma )
     ( GCD	   . calcFunc-gcd )
     ( If	   . calcFunc-if )
     ( Im	   . calcFunc-im )
     ( Inverse	   . calcFunc-inv )
     ( Integrate   . calcFunc-integ )
     ( Join	   . calcFunc-vconcat )
     ( LCM	   . calcFunc-lcm )
     ( Log	   . calcFunc-ln )
     ( Max	   . calcFunc-max )
     ( Min	   . calcFunc-min )
     ( Mod	   . calcFunc-mod )
     ( MoebiusMu   . calcFunc-moebius )
     ( Random	   . calcFunc-random )
     ( Round	   . calcFunc-round )
     ( Re	   . calcFunc-re )
     ( Sign	   . calcFunc-sign )
     ( Sin	   . calcFunc-sin )
     ( Sinh	   . calcFunc-sinh )
     ( Sqrt	   . calcFunc-sqrt )
     ( Tan	   . calcFunc-tan )
     ( Tanh	   . calcFunc-tanh )
     ( Transpose   . calcFunc-trn )
     ( Length	   . calcFunc-vlen )
))

(put 'math 'math-variable-table
  '( ( I	   . var-i )
     ( Pi	   . var-pi )
     ( E	   . var-e )
))

(put 'math 'math-vector-brackets "{}")
(put 'math 'math-complex-format 'I)
(put 'math 'math-function-open "[")
(put 'math 'math-function-close "]")

(put 'math 'math-radix-formatter
     (function (lambda (r s) (format "%d^^%s" r s))))




;;;; [calc-ext.el]

;;; Combinatorics

(defun calc-k-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("GCD, LCM; Binomial, Dbl-fact; Random, random-Again"
     "Factors, Prime-test, Next-prime, Totient, Moebius"
     "SHIFT + extended-GCD")
   "combinatorics" ?k)
)

;;;; [calc-comb.el]

(defun calc-gcd (arg)
  "Compute the GCD of the top two elements of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "gcd" 'calcFunc-gcd arg))
)

(defun calc-lcm (arg)
  "Compute the LCM of the top two elements of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "lcm" 'calcFunc-lcm arg))
)

(defun calc-extended-gcd ()
  "Compute the extended GCD of the top two elements of the Calculator stack.
This is a list [g,a,b] where g = gcd(x,y) = ax + by, and x and y are the
second-to-top and top values on the stack, respectively."
  (interactive)
  (calc-slow-wrapper
   (calc-enter-result 2 "egcd" (cons 'calcFunc-egcd (calc-top-list-n 2))))
)

(defun calc-factorial (arg)
  "Compute the factorial of the number on the top of the Calculator stack.
If the number is an integer, computes an exact result.
If the number is floating-point, computes a floating-point approximate result."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "fact" 'calcFunc-fact arg))
)

(defun calc-gamma (arg)
  "Compute the Euler Gamma function of the number on the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "gmma" 'calcFunc-gamma arg))
)

(defun calc-double-factorial (arg)
  "Compute the double factorial of the number on the Calculator stack.
For even numbers, this is the product of even integers up to N.
For odd numbers, this is the product of odd integers up to N.
If the number is an integer, computes an exact result.
If the number is floating-point, computes a floating-point approximate result."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "dfac" 'calcFunc-dfact arg))
)

(defun calc-choose (arg)
  "Compute the binomial coefficient C(N,M) of the numbers on the stack.
If the numbers are integers, computes an exact result.
If either number is floating-point, computes an approximate result.
With Hyperbolic flag, computes number-of-permutations instead."
  (interactive "P")
  (calc-slow-wrapper
   (if (calc-is-hyperbolic)
       (calc-binary-op "perm" 'calcFunc-perm arg)
     (calc-binary-op "chos" 'calcFunc-choose arg)))
)

(defun calc-perm (arg)
  "Compute the number-of-permutations P(N,M) of the numbers on the stack.
If the numbers are integers, computes an exact result.
If either number is floating-point, computes an approximate result.
With Hyperbolic flag, computes binomial coefficient instead."
  (interactive "P")
  (calc-hyperbolic-func)
  (calc-choose arg)
)

(defvar calc-last-random-limit '(float 1 0))
(defun calc-random (n)
  "Produce a random integer between 0 (inclusive) and N (exclusive).
N is the numeric prefix argument, if any, otherwise it is taken from the stack.
If N is real, produce a random real number in the specified range.
If N is zero, produce a Gaussian-distributed value with mean 0, variance 1."
  (interactive "P")
  (calc-slow-wrapper
   (if n
       (calc-enter-result 0 "rand" (list 'calcFunc-random
					 (setq calc-last-random-limit
					       (prefix-numeric-value n))))
     (calc-enter-result 1 "rand" (list 'calcFunc-random
				       (setq calc-last-random-limit
					     (calc-top-n 1))))))
)

(defun calc-rrandom ()
  "Produce a random real between 0 and 1."
  (interactive)
  (calc-slow-wrapper
   (setq calc-last-random-limit '(float 1 0))
   (calc-enter-result 0 "rand" (list 'calcFunc-random '(float 1 0))))
)

(defun calc-random-again ()
  "Produce another random number in the same range as the last one generated."
  (interactive)
  (calc-slow-wrapper
   (calc-enter-result 0 "rand" (list 'calcFunc-random calc-last-random-limit)))
)

(defun calc-report-prime-test (res)
  (cond ((eq (car res) t)
	 (calc-record-message "prim" "Prime (guaranteed)"))
	((eq (car res) nil)
	 (if (cdr res)
	     (if (eq (nth 1 res) 'unknown)
		 (calc-record-message
		  "prim" "Non-prime (factors unknown)")
	       (calc-record-message
		"prim" "Non-prime (%s is a factor)"
		(math-format-number (nth 1 res))))
	   (calc-record-message "prim" "Non-prime")))
	(t
	 (calc-record-message
	  "prim" "Probably prime (%d iters; %s%% chance of error)"
	  (nth 1 res)
	  (let ((calc-float-format '(fix 2)))
	    (math-format-number (nth 2 res))))))
)

(defun calc-prime-test (iters)
  "Determine whether the number on the top of the stack is prime.
For large numbers (> 8 million), this test is probabilistic.
Execute this command repeatedly to improve certainty of result.
With a numeric prefix argument, execute (up to) N iterations at once."
  (interactive "p")
  (calc-slow-wrapper
   (let* ((n (calc-top-n 1))
	  (res (math-prime-test n iters)))
     (calc-report-prime-test res)))
)

(defun calc-next-prime (iters)
  "Determine the next prime greater than the number on the top of the stack.
The top-of-stack is replaced by this number.
For numbers above 8 million, this finds the next number that passes one
iteration of calc-prime-test.  With a prefix argument, the number must
pass the specified number of calc-prime-test iterations.
With Inverse flag, find the previous prime instead."
  (interactive "p")
  (calc-slow-wrapper
   (let ((calc-verbose-nextprime t))
     (if (calc-is-inverse)
	 (calc-enter-result 1 "prvp" (list 'calcFunc-prevprime
					   (calc-top-n 1) (math-abs iters)))
       (calc-enter-result 1 "nxtp" (list 'calcFunc-nextprime
					 (calc-top-n 1) (math-abs iters))))))
)

(defun calc-prev-prime (iters)
  "Determine the next prime less than the number on the top of the stack.
With Inverse flag, find the next greater prime instead."
  (interactive "p")
  (calc-invert-func)
  (calc-next-prime iters)
)

(defun calc-prime-factors (iters)
  "Attempt to reduce the integer at top of stack to a list of its prime factors.
This algorithm is guaranteed for N up to 25 million.  For larger N, it may
not find all of the prime factors."
  (interactive "p")
  (calc-slow-wrapper
   (let ((res (math-prime-factors (calc-top-n 1))))
     (if (not math-prime-factors-finished)
	 (calc-record-message "pfac" "Warning:  May not be fully factored"))
     (calc-enter-result 1 "pfac" res)))
)

(defun calc-totient (arg)
  "Compute the Euler Totient function phi(n).
This is the number of integers less than n which are relatively prime to n."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "phi" 'calcFunc-totient arg))
)

(defun calc-moebius (arg)
  "Compute the Moebius Mu function mu(n).
This is (-1)^k if n has k distinct prime factors, or 0 if n has some
duplicate factors."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "mu" 'calcFunc-moebius arg))
)




;;;; [calc-ext.el]

;;; Mode commands.

(defun calc-m-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("Deg, Rad, HMS; Frac; Polar; Algebraic; Symbolic"
     "Working; Xtensions; M=save"
     "SHIFT + simplify: Off, Num, Default, Bin-clip, Alg, Units")
   "mode" ?m)
)

;;;; [calc-mode.el]

(defun calc-save-modes ()
  "Save all mode variables' values in your .emacs file."
  (interactive)
  (calc-wrapper
   (let (pos
	 (vals (mapcar (function (lambda (v) (symbol-value (car v))))
		       calc-mode-var-list)))
     (set-buffer (find-file-noselect (substitute-in-file-name
				      calc-settings-file)))
     (goto-char (point-min))
     (if (and (search-forward ";;; Mode settings stored by Calc" nil t)
	      (progn
		(beginning-of-line)
		(setq pos (point))
		(search-forward "\n;;; End of mode settings" nil t)))
	 (progn
	   (beginning-of-line)
	   (forward-line 1)
	   (delete-region pos (point)))
       (goto-char (point-max))
       (insert "\n\n")
       (forward-char -1))
     (insert ";;; Mode settings stored by Calc on " (current-time-string) "\n")
     (let ((list calc-mode-var-list))
       (while list
	 (let* ((v (car (car list)))
		(def (nth 1 (car list)))
		(val (car vals)))
	   (or (equal val def)
	       (progn
		 (insert "(setq " (symbol-name v) " ")
		 (if (and (or (listp val)
			      (symbolp val))
			  (not (memq val '(nil t))))
		     (insert "'"))
		 (insert (prin1-to-string val) ")\n"))))
	 (setq list (cdr list)
	       vals (cdr vals))))
     (run-hooks 'calc-mode-save-hook)
     (insert ";;; End of mode settings\n")
     (save-buffer)))
)

(defun calc-algebraic-mode ()
  "Turn Algebraic mode on or off.
In algebraic mode, numeric entry accepts whole expressions without needing \"'\"."
  (interactive)
  (calc-wrapper
   (setq calc-algebraic-mode (not calc-algebraic-mode)))
)

(defun calc-symbolic-mode ()
  "Turn Symbolic mode on or off.
In symbolic mode, inexact numeric computations like sqrt(2) are postponed."
  (interactive)
  (calc-wrapper
   (setq calc-symbolic-mode (not calc-symbolic-mode)))
)

(defun calc-set-simplify-mode (mode arg)
  (setq calc-simplify-mode (if arg
			       (and (> (prefix-numeric-value arg) 0)
				    mode)
			     (and (not (eq calc-simplify-mode mode))
				  mode)))
)

(defun calc-no-simplify-mode (arg)
  "Turn off automatic simplification of algebraic expressions."
  (interactive "P")
  (calc-wrapper
   (calc-set-simplify-mode 'none arg))
)

(defun calc-num-simplify-mode (arg)
  "Enable automatic simplification of expressions with constant argments only."
  (interactive "P")
  (calc-wrapper
   (calc-set-simplify-mode 'num arg))
)

(defun calc-default-simplify-mode ()
  "Turn on default automatic simplification of algebraic expressions."
  (interactive)
  (calc-wrapper
   (setq calc-simplify-mode nil))
)

(defun calc-bin-simplify-mode (arg)
  "Turn on automatic simplification with math-clip."
  (interactive "P")
  (calc-wrapper
   (calc-set-simplify-mode 'binary arg))
)

(defun calc-alg-simplify-mode (arg)
  "Turn on automatic algebraic simplification of expressions."
  (interactive "P")
  (calc-wrapper
   (calc-set-simplify-mode 'alg arg))
)

(defun calc-ext-simplify-mode (arg)
  "Turn on automatic \"extended\" algebraic simplification of expressions."
  (interactive "P")
  (calc-wrapper
   (calc-set-simplify-mode 'ext arg))
)

(defun calc-units-simplify-mode (arg)
  "Turn on automatic units-simplification of expressions."
  (interactive "P")
  (calc-wrapper
   (calc-set-simplify-mode 'units arg))
)

(defun calc-working (n)
  "Display level of \"Working...\" messages, or set level to N.
With numeric prefix argument 0, disables messages.
With argument 1, enables messages.
With argument 2, enables more detailed messages."
  (interactive "P")
  (calc-wrapper
   (cond ((consp n)
	  (calc-pop-push-record 0 "work"
				(cond ((eq calc-display-working-message t) 1)
				      (calc-display-working-message 2)
				      (t 0))))
	 ((eq n 2) (setq calc-display-working-message 'lots))
	 ((eq n 0) (setq calc-display-working-message nil))
	 ((eq n 1) (setq calc-display-working-message t)))
   (cond ((eq calc-display-working-message t)
	  (message "\"Working...\" messages enabled."))
	 (calc-display-working-message
	  (message "Detailed \"Working...\" messages enabled."))
	 (t
	  (message "\"Working...\" messages disabled."))))
)

(defun calc-always-load-extensions ()
  "Toggle mode in which calc-ext extensions are loaded automatically with calc."
  (interactive)
  (calc-wrapper
   (if (setq calc-always-load-extensions (not calc-always-load-extensions))
       (message "Always loading extensions package.")
     (message "Loading extensions package on demand only.")))
)

;;;; [calc-math.el]

(defun calc-degrees-mode ()
  "Set Calculator to use degrees for all angles."
  (interactive)
  (calc-wrapper
   (setq calc-angle-mode 'deg)
   (message "Angles measured in degrees."))
)

(defun calc-radians-mode ()
  "Set Calculator to use degrees for all angles."
  (interactive)
  (calc-wrapper
   (setq calc-angle-mode 'rad)
   (message "Angles measured in radians."))
)

;;;; [calc-forms.el]

(defun calc-hms-mode ()
  "Set Calculator to use degrees-minutes-seconds for all angles."
  (interactive)
  (calc-wrapper
   (setq calc-angle-mode 'hms)
   (message "Angles measured in degrees-minutes-seconds."))
)

;;;; [calc-cplx.el]

(defun calc-polar-mode (n)
  "Toggle mode complex number preference between rectangular and polar forms."
  (interactive "P")
  (calc-wrapper
   (if (if n
	   (> (prefix-numeric-value n) 0)
	 (eq calc-complex-mode 'cplx))
       (progn
	 (setq calc-complex-mode 'polar)
	 (message "Preferred complex form is polar."))
     (setq calc-complex-mode 'cplx)
     (message "Preferred complex form is rectangular.")))
)

;;;; [calc-frac.el]

(defun calc-frac-mode (n)
  "Toggle mode in which Calculator prefers fractions over floats.
With positive prefix argument, sets mode on (fractions).
With negative or zero prefix argument, sets mode off (floats)."
  (interactive "P")
  (calc-wrapper
   (if (if n
	   (> (prefix-numeric-value n) 0)
	 (not calc-prefer-frac))
       (progn
	 (setq calc-prefer-frac t)
	 (message "Integer division will now generate fractions."))
     (setq calc-prefer-frac nil)
     (message "Integer division will now generate floating-point results.")))
)




;;;; [calc-ext.el]

;;; Trail commands.

(defun calc-t-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("Display; Fwd, Back; Next, Prev, Here, [, ]; Yank"
     "Search, Reverse; In, Out; <, >; Kill; Marker")
   "trail" ?t)
)

;;;; [calc-trail.el]

(defun calc-trail-in ()
  "Switch to the Calc Trail window."
  (interactive)
  (let ((win (get-buffer-window (calc-trail-display t))))
    (and win (select-window win)))
)

(defun calc-trail-out ()
  "Switch back to the main Calculator window."
  (interactive)
  (calc-select-buffer)
  (let ((win (get-buffer-window (current-buffer))))
    (if win
	(select-window win)
      (calc)))
)

(defmacro calc-with-trail-buffer (&rest body)
  (` (let ((save-buf (current-buffer))
	   (calc-command-flags nil))
       (unwind-protect
	   (, (append '(progn
			 (set-buffer (calc-trail-display t))
			 (goto-char calc-trail-pointer))
		      body))
	 (set-buffer save-buf))))
)

(defun calc-trail-next (n)
  "Move the trail pointer down one line."
  (interactive "p")
  (calc-with-trail-buffer
   (forward-line n)
   (calc-trail-here))
)

(defun calc-trail-previous (n)
  "Move the trail pointer up one line."
  (interactive "p")
  (calc-with-trail-buffer
   (forward-line (- n))
   (calc-trail-here))
)

(defun calc-trail-first (n)
  "Move the trail pointer to the beginning of the trail."
  (interactive "p")
  (calc-with-trail-buffer
   (goto-char (point-min))
   (forward-line n)
   (calc-trail-here))
)

(defun calc-trail-last (n)
  "Move the trail pointer to the end of the trail."
  (interactive "p")
  (calc-with-trail-buffer
   (goto-char (point-max))
   (forward-line (- n))
   (calc-trail-here))
)

(defun calc-trail-scroll-left (n)
  "Scroll the trail window horizontally to the left."
  (interactive "P")
  (let ((curwin (selected-window)))
    (calc-with-trail-buffer
     (unwind-protect
	 (progn
	   (select-window (get-buffer-window (current-buffer)))
	   (calc-scroll-left n))
       (select-window curwin))))
)

(defun calc-trail-scroll-right (n)
  "Scroll the trail window horizontally to the right."
  (interactive "P")
  (let ((curwin (selected-window)))
    (calc-with-trail-buffer
     (unwind-protect
	 (progn
	   (select-window (get-buffer-window (current-buffer)))
	   (calc-scroll-right n))
       (select-window curwin))))
)

(defun calc-trail-forward (n)
  "Move the trail pointer forward one page."
  (interactive "p")
  (calc-with-trail-buffer
   (forward-line (* n (1- (window-height))))
   (calc-trail-here))
)

(defun calc-trail-backward (n)
  "Move the trail pointer backward one page."
  (interactive "p")
  (calc-with-trail-buffer
   (forward-line (- (* n (1- (window-height)))))
   (calc-trail-here))
)

(defun calc-trail-isearch-forward ()
  "Search incrementally forward in the trail buffer."
  (interactive)
  (calc-with-trail-buffer
   (save-window-excursion
     (select-window (get-buffer-window (current-buffer)))
     (isearch t nil))
   (calc-trail-here))
)

(defun calc-trail-isearch-backward ()
  "Search incrementally backward in the trail buffer."
  (interactive)
  (calc-with-trail-buffer
   (save-window-excursion
     (select-window (get-buffer-window (current-buffer)))
     (isearch nil nil))
   (calc-trail-here))
)

(defun calc-trail-yank ()
  "Yank the value indicated by the trail pointer onto the Calculator stack."
  (interactive)
  (calc-wrapper
   (calc-set-command-flag 'hold-trail)
   (calc-enter-result 0 "yank"
		      (calc-with-trail-buffer
		       (if (or (looking-at "Emacs Calc")
			       (looking-at "----")
			       (looking-at " ? ? ?[^ \n]* *$")
			       (looking-at "..?.?$"))
			   (error "Can't yank that line"))
		       (forward-char 4)
		       (search-forward " ")
		       (let* ((next (save-excursion (forward-line 1) (point)))
			      (str (buffer-substring (point) (1- next)))
			      (calc-language nil)
			      (math-expr-opers math-standard-opers)
			      (val (math-read-expr str)))
			 (if (eq (car-safe val) 'error)
			     (error "Can't yank that line: " (nth 2 val))
			   val)))))
)

(defun calc-trail-marker (str)
  "Put a textual marker into the Calculator trail."
  (interactive "sText to insert in trail: ")
  (calc-with-trail-buffer
   (forward-line 1)
   (let ((buffer-read-only nil))
     (insert "---- " str "\n"))
   (forward-line -1)
   (calc-trail-here))
)

(defun calc-trail-kill (n)
  "Kill one line from the Calculator trail.
This line can be yanked into text buffers, but cannot be yanked back into
the trail."
  (interactive "p")
  (calc-with-trail-buffer
   (let ((buffer-read-only nil))
     (save-restriction
       (narrow-to-region   ; don't delete "Emacs Trail" header
	(save-excursion
	  (goto-char (point-min))
	  (forward-line 1)
	  (point))
	(point-max))
       (kill-line n)))
   (calc-trail-here))
)



;;;; [calc-ext.el]

;;; Units commands.

(defun calc-u-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("Simplify, Convert, Temperature-convert, Base-units"
     "Remove, eXtract; Explain; View-table"
     "Define, Undefine, Get-defn, Permanent")
   "units" ?u)
)

;;;; [calc-units.el]

(defun calc-base-units ()
  "Convert the value on the stack into \"base\" units, like m, g, and s."
  (interactive)
  (calc-slow-wrapper
   (calc-enter-result 1 "bsun" (math-simplify-units
				(math-to-standard-units (calc-top-n 1) nil))))
)

(defun calc-convert-units (&optional old-units new-units)
  "Convert the value on the stack to the specified new units.
Unit name may also be \"si\", \"mks\", or \"cgs\" to convert to that system.
Temperature units are converted as relative temperatures."
  (interactive)
  (calc-slow-wrapper
   (let ((expr (calc-top-n 1))
	 (uoldname nil)
	 unew)
     (or (math-units-in-expr-p expr t)
	 (let ((uold (or old-units
			 (progn
			   (setq uoldname (read-string "Old units: "))
			   (if (equal uoldname "")
			       (progn
				 (setq uoldname "1")
				 1)
			     (math-read-expr uoldname))))))
	   (if (eq (car-safe uold) 'error)
	       (error "Bad format in units expression: %s" (nth 1 uold)))
	   (setq expr (math-mul expr uold))))
     (or new-units
	 (setq new-units (read-string (if uoldname
					  (concat "Old units: "
						  uoldname
						  ", new units: ")
					"New units: "))))
     (setq units (math-read-expr new-units))
     (if (eq (car-safe units) 'error)
	 (error "Bad format in units expression: %s" (nth 2 units)))
     (let ((unew (math-units-in-expr-p units t))
	   (std (and (eq (car-safe units) 'var)
		     (assq (nth 1 units) math-standard-units-systems))))
       (if std
	   (calc-enter-result 1 "cvun" (math-simplify-units
					(math-to-standard-units expr
								(nth 1 std))))
	 (or unew
	     (error "No units specified"))
	 (calc-enter-result 1 "cvun" (math-simplify-units
				      (math-convert-units expr units)))))))
)

(defun calc-convert-temperature (&optional old-units new-units)
  "Convert the value on the stack to the specified new temperature units.
This converts absolute temperature, i.e., \"0 degC\" converts to \"32 degF\"."
  (interactive)
  (calc-slow-wrapper
   (let ((expr (calc-top-n 1))
	 (uold nil)
	 (uoldname nil)
	 unew)
     (setq uold (or old-units
		    (let ((units (math-single-units-in-expr-p expr)))
		      (if units
			  (if (consp units)
			      (list 'var (car units)
				    (intern (concat "var-"
						    (symbol-name
						     (car units)))))
			    (error "Not a pure temperature expression"))
			(math-read-expr
			 (setq uoldname (read-string
					 "Old temperature units: ")))))))
     (if (eq (car-safe uold) 'error)
	 (error "Bad format in units expression: %s" (nth 2 uold)))
     (or (math-units-in-expr-p expr nil)
	 (setq expr (math-mul expr uold)))
     (setq unew (or new-units
		    (math-read-expr
		     (read-string (if uoldname
				      (concat "Old temperature units: "
					      uoldname
					      ", new units: ")
				    "New temperature units: ")))))
     (if (eq (car-safe unew) 'error)
	 (error "Bad format in units expression: %s" (nth 2 unew)))
     (calc-enter-result 1 "cvtm" (math-simplify-units
				  (math-convert-temperature expr uold unew)))))
)

(defun calc-remove-units ()
  "Remove all unit names from the value on the top of the stack."
  (interactive)
  (calc-slow-wrapper
   (calc-enter-result 1 "rmun" (math-simplify-units
				(math-remove-units (calc-top-n 1)))))
)

(defun calc-extract-units ()
  "Extract the units from the unit expression on the top of the stack."
  (interactive)
  (calc-slow-wrapper
   (calc-enter-result 1 "rmun" (math-simplify-units
				(math-extract-units (calc-top-n 1)))))
)

(defun calc-explain-units ()
  "Produce an English explanation of the units of the expression on the stack."
  (interactive)
  (calc-wrapper
   (let ((num-units nil)
	 (den-units nil))
     (calc-explain-units-rec (calc-top-n 1) 1)
     (and den-units (string-match "^[^(].* .*[^)]$" den-units)
	  (setq den-units (concat "(" den-units ")")))
     (if num-units
	 (if den-units
	     (message "%s per %s" num-units den-units)
	   (message "%s" num-units))
       (if den-units
	   (message "1 per %s" den-units)
	 (message "No units in expression")))))
)

(defun calc-explain-units-rec (expr pow)
  (let ((u (math-check-unit-name expr))
	pos)
    (if (and u (not (math-zerop pow)))
	(let ((name (or (nth 2 u) (symbol-name (car u)))))
	  (if (eq (aref name 0) ?\*)
	      (setq name (substring name 1)))
	  (if (string-match "[^a-zA-Z0-9']" name)
	      (if (string-match "^[a-zA-Z0-9' ()]*$" name)
		  (while (setq pos (string-match "[ ()]" name))
		    (setq name (concat (substring name 0 pos)
				       (if (eq (aref name pos) 32) "-" "")
				       (substring name (1+ pos)))))
		(setq name (concat "(" name ")"))))
	  (or (eq (nth 1 expr) (car u))
	      (setq name (concat (nth 2 (assq (aref (symbol-name
						     (nth 1 expr)) 0)
					      math-unit-prefixes))
				 (if (and (string-match "[^a-zA-Z0-9']" name)
					  (not (memq (car u) '(mHg gf))))
				     (concat "-" name)
				   (downcase name)))))
	  (cond ((or (math-equal-int pow 1)
		     (math-equal-int pow -1)))
		((or (math-equal-int pow 2)
		     (math-equal-int pow -2))
		 (if (equal (nth 4 u) '((m . 1)))
		     (setq name (concat "Square-" name))
		   (setq name (concat name "-squared"))))
		((or (math-equal-int pow 3)
		     (math-equal-int pow -3))
		 (if (equal (nth 4 u) '((m . 1)))
		     (setq name (concat "Cubic-" name))
		   (setq name (concat name "-cubed"))))
		(t
		 (setq name (concat name "^"
				    (math-format-number (math-abs pow))))))
	  (if (math-posp pow)
	      (setq num-units (if num-units
				  (concat num-units " " name)
				name))
	    (setq den-units (if den-units
				(concat den-units " " name)
			      name))))
      (cond ((eq (car-safe expr) '*)
	     (calc-explain-units-rec (nth 1 expr) pow)
	     (calc-explain-units-rec (nth 2 expr) pow))
	    ((eq (car-safe expr) '/)
	     (calc-explain-units-rec (nth 1 expr) pow)
	     (calc-explain-units-rec (nth 2 expr) (- pow)))
	    ((memq (car-safe expr) '(neg + -))
	     (calc-explain-units-rec (nth 1 expr) pow))
	    ((and (eq (car-safe expr) '^)
		  (math-realp (nth 2 expr)))
	     (calc-explain-units-rec (nth 1 expr)
				     (math-mul pow (nth 2 expr)))))))
)

(defun calc-simplify-units ()
  "Simplify the units expression on top of the stack."
  (interactive)
  (calc-slow-wrapper
   (calc-with-default-simplification
    (calc-enter-result 1 "smun" (math-simplify-units (calc-top-n 1)))))
)

(defun calc-view-units-table (n)
  "Display a temporary buffer for displaying the Units Table."
  (interactive "P")
  (and n (setq math-units-table-buffer-valid nil))
  (math-build-units-table-buffer nil)
)

(defun calc-enter-units-table (n)
  "Switch to a temporary buffer for displaying the Units Table."
  (interactive "P")
  (and n (setq math-units-table-buffer-valid nil))
  (math-build-units-table-buffer t)
  (message (substitute-command-keys "Type \\[calc] to return to the Calculator."))
)

(defun calc-define-unit (uname desc)
  "Define a new type of unit using the formula on the top of the stack."
  (interactive "SDefine unit name: \nsDescription: ")
  (calc-wrapper
   (let ((form (calc-top-n 1))
	 (unit (assq uname math-additional-units)))
     (or unit
	 (setq math-additional-units
	       (cons (setq unit (list uname nil nil))
		     math-additional-units)
	       math-units-table nil))
     (setcar (cdr unit) (and (not (and (eq (car-safe form) 'var)
				       (eq (nth 1 form) uname)))
			     (not (math-equal-int form 1))
			     (math-format-flat-expr form 0)))
     (setcar (cdr (cdr unit)) (and (not (equal desc ""))
				   desc))))
  (calc-invalidate-units-table)
)

(defun calc-undefine-unit (uname)
  "Remove the definition of a user-defined unit."
  (interactive "SUndefine unit name: ")
  (calc-wrapper
   (let ((unit (assq uname math-additional-units)))
     (or unit
	 (if (assq uname math-standard-units)
	     (error "\"%s\" is a predefined unit name" uname)
	   (error "Unit name \"%s\" not found" uname)))
     (setq math-additional-units (delq unit math-additional-units)
	   math-units-table nil)))
  (calc-invalidate-units-table)
)

(defun calc-invalidate-units-table ()
  (setq math-units-table nil)
  (let ((buf (get-buffer "*Units Table*")))
    (save-excursion
      (set-buffer buf)
      (save-excursion
	(goto-char (point-min))
	(if (looking-at "Calculator Units Table")
	    (let ((buffer-read-only nil))
	      (insert "(Obsolete) "))))))
)

(defun calc-get-unit-definition (uname)
  "Push the definition of a unit as a formula on the Calculator stack."
  (interactive "SGet definition for unit: ")
  (calc-wrapper
   (math-build-units-table)
   (let ((unit (assq uname math-units-table)))
     (or unit
	 (error "Unit name \"%s\" not found" uname))
     (let ((msg (nth 2 unit)))
       (if (stringp msg)
	   (if (string-match "^\\*" msg)
	       (setq msg (substring msg 1)))
	 (setq msg (symbol-name uname)))
       (if (nth 1 unit)
	   (progn
	     (calc-enter-result 0 "ugdf" (nth 1 unit))
	     (message "Derived unit: %s" msg))
	 (calc-enter-result 0 "ugdf" (list 'var uname
					   (intern
					    (concat "var-"
						    (symbol-name uname)))))
	 (message "Base unit: %s" msg)))))
)

(defun calc-permanent-units ()
  "Save all user-defined units in your .emacs file."
  (interactive)
  (calc-wrapper
   (let (pos)
     (set-buffer (find-file-noselect (substitute-in-file-name
				      calc-settings-file)))
     (goto-char (point-min))
     (if (and (search-forward ";;; Custom units stored by Calc" nil t)
	      (progn
		(beginning-of-line)
		(setq pos (point))
		(search-forward "\n;;; End of custom units" nil t)))
	 (progn
	   (beginning-of-line)
	   (forward-line 1)
	   (delete-region pos (point)))
       (goto-char (point-max))
       (insert "\n\n")
       (forward-char -1))
     (insert ";;; Custom units stored by Calc on " (current-time-string) "\n")
     (if math-additional-units
	 (progn
	   (insert "(setq math-additional-units '(\n")
	   (let ((list math-additional-units))
	     (while list
	       (insert "  (" (symbol-name (car (car list))) " "
		       (if (nth 1 (car list))
			   (if (stringp (nth 1 (car list)))
			       (prin1-to-string (nth 1 (car list)))
			     (prin1-to-string (math-format-flat-expr
					       (nth 1 (car list)) 0)))
			 "nil")
		       " "
		       (prin1-to-string (nth 2 (car list)))
		       ")\n")
	       (setq list (cdr list))))
	   (insert "))\n"))
       (insert ";;; (no custom units defined)\n"))
     (insert ";;; End of custom units\n")
     (save-buffer)))
)




;;;; [calc-ext.el]

;;; Vector commands.

(defun calc-v-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("Pack, Unpack, Identity, Diagonal, indeX, Build"
     "Row, Col, Length; rNorm"
     "Tranpose, Arrange; Sort, Histogram"
     "SHIFT + Det, Inv, LUD, Trace, conJtrn, Cross, cNorm"
     "SHIFT + Reduce, Map, Apply"
     "<, =, > (justification); , (commas); [, {, ( (brackets)")
   "vec/mat" ?v)
)

(defun calc-concat (arg)
  "Concatenate the two vectors at the top of the stack.
Or concatenate a scalar value and a vector."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "|" 'calcFunc-vconcat arg '(vec)))
)

;;;; [calc-mode.el]

(defun calc-matrix-left-justify ()
  "Left-justify elements of matrices."
  (interactive)
  (calc-wrapper
   (setq calc-matrix-just nil)
   (calc-refresh))
)

(defun calc-matrix-center-justify ()
  "Center elements of matrices."
  (interactive)
  (calc-wrapper
   (setq calc-matrix-just 'center)
   (calc-refresh))
)

(defun calc-matrix-right-justify ()
  "Right-justify elements of matrices."
  (interactive)
  (calc-wrapper
   (setq calc-matrix-just 'right)
   (calc-refresh))
)

(defun calc-vector-commas ()
  "Turn separating commas in vectors on and off."
  (interactive)
  (calc-wrapper
   (setq calc-vector-commas (if calc-vector-commas nil ","))
   (calc-refresh))
)

(defun calc-vector-brackets ()
  "Surround vectors and matrices with square brackets.
If already using brackets, turn the brackets off."
  (interactive)
  (calc-wrapper
   (setq calc-vector-brackets (if (equal calc-vector-brackets "[]") nil "[]"))
   (calc-refresh))
)

(defun calc-vector-braces ()
  "Surround vectors and matrices with curly braces.
If already using braces, turn the braces off."
  (interactive)
  (calc-wrapper
   (setq calc-vector-brackets (if (equal calc-vector-brackets "{}") nil "{}"))
   (calc-refresh))
)

(defun calc-vector-parens ()
  "Surround vectors and matrices with parentheses.
If already using parens, turn the parens off."
  (interactive)
  (calc-wrapper
   (setq calc-vector-brackets (if (equal calc-vector-brackets "()") nil "()"))
   (calc-refresh))
)

;;;; [calc-vec.el]

(defun calc-pack (n)
  "Pack the top two numbers on the Calculator stack into a complex number.
Given a numeric prefix, pack the top N numbers into a vector.
Given a -1 prefix, pack the top 2 numbers into a rectangular complex number.
Given a -2 prefix, pack the top 2 numbers into a polar complex number.
Given a -3 prefix, pack the top 3 numbers into an HMS form.
Given a -4 prefix, pack the top 2 numbers into an error form.
Given a -5 prefix, pack the top 2 numbers into a modulo form.
Given a -6 prefix, pack the top 2 numbers into a [ .. ] interval form.
Given a -7 prefix, pack the top 2 numbers into a [ .. ) interval form.
Given a -8 prefix, pack the top 2 numbers into a ( .. ] interval form.
Given a -9 prefix, pack the top 2 numbers into a ( .. ) interval form."
  (interactive "P")
  (calc-wrapper
   (let ((num (prefix-numeric-value n)))
     (cond ((and n (>= num 0))
	    (calc-enter-result num nil (cons 'vec (calc-top-list num))))
	   ((= num -3)
	    (let ((h (calc-top 3))
		  (m (calc-top 2))
		  (s (calc-top 1)))
	      (if (and (math-num-integerp h)
		       (math-num-integerp m))
		  (calc-enter-result 3 nil (list 'hms h m s))
		(error "Hours and minutes must be integers"))))
	   ((= num -4)
	    (let ((x (calc-top-n 2))
		  (sigma (calc-top-n 1)))
	      (if (and (or (math-anglep x) (not (math-objvecp x)))
		       (or (math-anglep sigma) (not (math-objvecp sigma))))
		  (calc-enter-result 2 nil (math-make-sdev x sigma))
		(error "Components must be real"))))
	   ((= num -5)
	    (let ((a (calc-top-n 2))
		  (m (calc-top-n 1)))
	      (if (and (math-anglep a) (math-anglep m))
		  (if (math-posp m)
		      (calc-enter-result 2 nil (math-make-mod a m))
		    (error "Modulus must be positive"))
		(error "Components must be real"))))
	   ((memq num '(-6 -7 -8 -9))
	    (let ((lo (calc-top-n 2))
		  (hi (calc-top-n 1)))
	      (if (and (or (math-anglep lo) (not (math-objvecp lo)))
		       (or (math-anglep hi) (not (math-objvecp hi))))
		  (calc-enter-result 2 nil (math-make-intv (+ num 6) lo hi))
		(error "Components must be real"))))
	   ((or (= num -2)
		(and (eq calc-complex-mode 'polar)
		     (= num 0)))
	    (let ((r (calc-top 2))
		  (theta (calc-top 1)))
	      (if (and (math-realp r) (math-anglep theta))
		  (calc-enter-result 2 nil (list 'polar r theta))
		(error "Components must be real"))))
	   (t
	    (let ((real (calc-top 2))
		  (imag (calc-top 1)))
	      (if (and (math-realp real) (math-realp imag))
		  (calc-enter-result 2 nil (list 'cplx real imag))
		(error "Components must be real")))))))
)

(defun calc-unpack ()
  "Unpack complex number, vector, HMS form, error form, etc. at top of stack."
  (interactive)
  (calc-wrapper
   (let ((num (calc-top)))
     (if (or (and (not (memq (car-safe num) '(frac float cplx polar vec hms
						   sdev mod)))
		  (math-objvecp num))
	     (eq (car-safe num) 'var))
	 (error "Argument must be a vector, complex number, or HMS, error, or modulo form"))
     (calc-pop-push-list 1 (cdr num))))
)

(defun calc-diag (n)
  "Build an NxN element diagonal matrix out of top-of-stack.
If top-of-stack is a vector, numeric prefix N must match or be omitted.
If top-of-stack is a scalar, numeric prefix N is required."
  (interactive "P")
  (calc-wrapper
   (calc-enter-result 1 "diag" (if n
				   (list 'calcFunc-diag (calc-top-n 1)
					 (prefix-numeric-value n))
				 (list 'calcFunc-diag (calc-top-n 1)))))
)

(defun calc-ident (n)
  "Push an NxN element identity matrix on the stack."
  (interactive "NDimension of identity matrix = ")
  (calc-wrapper
   (calc-enter-result 0 "idn" (list 'calcFunc-diag 1
				    (prefix-numeric-value n))))
)

(defun calc-index (n)
  "Generate a vector of the form [1, 2, ..., N]."
  (interactive "NSize of vector = ")
  (calc-wrapper
   (calc-enter-result 0 "indx" (list 'calcFunc-index
				     (prefix-numeric-value n))))
)

(defun calc-build-vector (n)
  "Generate a vector of N copies of top-of-stack."
  (interactive "NSize of vector = ")
  (calc-wrapper
   (calc-enter-result 1 "bldv" (list 'calcFunc-cvec
				     (calc-top-n 1)
				     (prefix-numeric-value n))))
)

(defun calc-vlength (arg)
  "Replace a vector with its length, in the form of an integer."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op "len" 'calcFunc-vlen arg))
)

(defun calc-arrange-vector (n)
  "Rearrange a matrix to have a specific number of columns."
  (interactive "NNumber of columns = ")
  (calc-wrapper
   (calc-enter-result 1 "arng" (list 'calcFunc-arrange (calc-top-n 1)
				     (prefix-numeric-value n))))
)

(defun calc-sort ()
  "Sort the matrix at top of stack into increasing order.
With Inverse flag or a negative numeric prefix, sort into decreasing order."
  (interactive)
  (calc-slow-wrapper
   (if (calc-is-inverse)
       (calc-enter-result 1 "rsrt" (list 'calcFunc-rsort (calc-top-n 1)))
     (calc-enter-result 1 "sort" (list 'calcFunc-sort (calc-top-n 1)))))
)

(defun calc-histogram (n)
  "Compile a histogram of a vector of integers in the range [0..N).
N is the numeric prefix argument.
With Hyperbolic flag, top-of-stack is a vector of weights to associate
with elements of next-to-top."
  (interactive "NNumber of bins: ")
  (calc-slow-wrapper
   (if calc-hyperbolic-flag
       (calc-enter-result 2 "hist" (list 'calcFunc-histogram
					 (calc-top-n 2)
					 (calc-top-n 1)
					 (prefix-numeric-value n)))
     (calc-enter-result 1 "hist" (list 'calcFunc-histogram
				       (calc-top-n 1)
				       1
				       (prefix-numeric-value n)))))
)

(defun calc-transpose (arg)
  "Replace the matrix at top of stack with its transpose."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op "trn" 'calcFunc-trn arg))
)

(defun calc-conj-transpose (arg)
  "Replace the matrix at top of stack with its conjugate transpose."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op "ctrn" 'calcFunc-ctrn arg))
)

(defun calc-cross (arg)
  "Compute the right-handed cross product of two 3-vectors."
  (interactive "P")
  (calc-wrapper
   (calc-binary-op "cros" 'calcFunc-cross arg))
)

;;;; [calc-mat.el]

(defun calc-mdet (arg)
  "Compute the determinant of the square matrix on the top of the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "mdet" 'calcFunc-det arg))
)

(defun calc-mtrace (arg)
  "Compute the trace of the square matrix on the top of the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "mtr" 'calcFunc-tr arg))
)

(defun calc-mlud (arg)
  "Perform an L-U decomposition of the matrix on the top of the stack.
Result is a vector of two matrices, L and U."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "mlud" 'calcFunc-lud arg))
)

;;;; [calc-vec.el]

(defun calc-rnorm (arg)
  "Compute the row norm of the vector or matrix on the top of the stack.
This is the maximum row-absolute-value-sum of the matrix.
For a vector, this is the maximum of the absolute values of the elements."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op "rnrm" 'calcFunc-rnorm arg))
)

(defun calc-cnorm (arg)
  "Compute the column norm of the vector or matrix on the top of the stack.
This is the maximum column-absolute-value-sum of the matrix.
For a vector, this is the sum of the absolute values of the elements."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op "cnrm" 'calcFunc-cnorm arg))
)

(defun calc-mrow (n)
  "Replace matrix at top of stack with its Nth row.
Numeric prefix N must be between 1 and the height of the matrix.
If top of stack is a non-matrix vector, extract its Nth element.
If N is negative, remove the Nth row (or element)."
  (interactive "NRow number: ")
  (calc-wrapper
   (setq n (prefix-numeric-value n))
   (if (= n 0)
       (calc-enter-result 1 "getd" (list 'calcFunc-getdiag (calc-top-n 1)))
     (if (< n 0)
	 (calc-enter-result 1 "rrow" (list 'calcFunc-mrrow
					   (calc-top-n 1) (- n)))
       (calc-enter-result 1 "mrow" (list 'calcFunc-mrow (calc-top-n 1) n)))))
)

(defun calc-mcol (n)
  "Replace matrix at top of stack with its Nth column.
Numeric prefix N must be between 1 and the width of the matrix.
If top of stack is a non-matrix vector, extract its Nth element.
If N is negative, remove the Nth column (or element)."
  (interactive "NColumn number: ")
  (calc-wrapper
   (setq n (prefix-numeric-value n))
   (if (= n 0)
       (calc-enter-result 1 "getd" (list 'calcFunc-getdiag (calc-top-n 1)))
     (if (< n 0)
	 (calc-enter-result 1 "rcol" (list 'calcFunc-mrcol
					   (calc-top-n 1) (- n)))
       (calc-enter-result 1 "mcol" (list 'calcFunc-mcol (calc-top-n 1) n)))))
)

;;;; [calc-map.el]

(defun calc-apply (&optional oper)
  "Apply an operator to the elements of a vector.
For example, applying f to [1, 2, 3] produces f(1, 2, 3)."
  (interactive)
  (calc-wrapper
   (let* ((calc-dollar-values (mapcar 'car-safe
				      (nthcdr calc-stack-top calc-stack)))
	  (calc-dollar-used 0)
	  (oper (or oper (calc-get-operator "Apply"
					    (and (math-vectorp (calc-top 1))
						 (1- (length (calc-top 1)))))))
	  (expr (calc-top-n (1+ calc-dollar-used))))
     (message "Working...")
     (calc-set-command-flag 'clear-message)
     (calc-enter-result (1+ calc-dollar-used)
			(concat (substring "apl" 0 (- 4 (length (nth 2 oper))))
				(nth 2 oper))
			(list 'calcFunc-apply
			      (math-calcFunc-to-var (nth 1 oper))
			      expr))))
)

(defun calc-reduce (&optional oper)
  "Apply a binary operator across all elements of a vector.
For example, applying + computes the sum of vector elements."
  (interactive)
  (calc-wrapper
   (let* ((calc-dollar-values (mapcar 'car-safe
				      (nthcdr calc-stack-top calc-stack)))
	  (calc-dollar-used 0)
	  (oper (or oper (calc-get-operator "Reduce" 2))))
     (message "Working...")
     (calc-set-command-flag 'clear-message)
     (calc-enter-result (1+ calc-dollar-used)
			(concat (substring "red" 0 (- 4 (length (nth 2 oper))))
				(nth 2 oper))
			(list (intern (concat "calcFunc-reduce"
					      (or calc-mapping-dir "")))
			      (math-calcFunc-to-var (nth 1 oper))
			      (calc-top-n (1+ calc-dollar-used))))))
)

(defun calc-map (&optional oper)
  "Apply an operator elementwise to one or two vectors.
For example, applying * computes a vector of products."
  (interactive)
  (calc-wrapper
   (let* ((calc-dollar-values (mapcar 'car-safe
				      (nthcdr calc-stack-top calc-stack)))
	  (calc-dollar-used 0)
	  (oper (or oper (calc-get-operator "Map")))
	  (nargs (if (or (equal calc-mapping-dir "a")
			 (equal calc-mapping-dir "d"))
		     1
		   (car oper))))
     (message "Working...")
     (calc-set-command-flag 'clear-message)
     (calc-enter-result (+ nargs calc-dollar-used)
			(concat (substring "map" 0 (- 4 (length (nth 2 oper))))
				(nth 2 oper))
			(cons (intern (concat "calcFunc-map"
					      (or calc-mapping-dir "")))
			      (cons (math-calcFunc-to-var (nth 1 oper))
				    (calc-top-list-n
				     nargs
				     (1+ calc-dollar-used)))))))
)

;;; Return a list of the form (nargs func name)
(defun calc-get-operator (msg &optional nargs)
  (let ((inv nil) (hyp nil) (prefix nil)
	done key oper (which 0)
	(msgs '( "(Press ? for help)"
		 "+, -, *, /, ^, %, \\, :, !, |, Neg"
		 "SHIFT + Abs, conJ, arG; maX, miN; Floor, Round; sQrt"
		 "SHIFT + Inv, Hyp; Sin, Cos, Tan; Exp, Ln, logB"
		 "Binary + And, Or, Xor, Diff; Not, Clip"
		 "Conversions + Deg, Rad, HMS; Float; SHIFT + Fraction"
		 "Kombinatorics + Dfact, Lcm, Gcd, Binomial, Perms; Random"
		 "Matrix-dir + Elements, Rows, Cols, Across, Down"
		 "X or Z = any function by name; ' = alg entry; $ = stack")))
    (while (not done)
      (message "%s%s: %s: %s%s%s"
	       msg
	       (cond ((equal calc-mapping-dir "r") " rows")
		     ((equal calc-mapping-dir "c") " columns")
		     ((equal calc-mapping-dir "a") " across")
		     ((equal calc-mapping-dir "d") " down")
		     (t ""))
	       (nth which msgs)
	       (if inv "Inv " "") (if hyp "Hyp " "")
	       (if prefix (concat (char-to-string prefix) "-") ""))
      (setq key (read-char))
      (cond ((= key ?\C-g)
	     (keyboard-quit))
	    ((= key ??)
	     (setq which (% (1+ which) (length msgs))))
	    ((= key ?I)
	     (setq inv (not inv)
		   prefix nil))
	    ((= key ?H)
	     (setq hyp (not hyp)
		   prefix nil))
	    ((eq key prefix)
	     (setq prefix nil))
	    ((and (memq key '(?b ?c ?k ?m)) (null prefix))
	     (setq inv nil hyp nil
		   prefix key))
	    ((eq prefix ?m)
	     (setq prefix nil)
	     (if (eq key ?e)
		 (setq calc-mapping-dir nil)
	       (if (memq key '(?r ?c ?a ?d))
		   (setq calc-mapping-dir (char-to-string key))
		 (beep))))
	    ((memq key '(?\$ ?\'))
	     (let ((expr (if (eq key ?\$)
			     (progn
			       (setq calc-dollar-used 1)
			       (if calc-dollar-values
				   (list (car calc-dollar-values))
				 (error "Stack underflow")))
			   (calc-do-alg-entry "" "Function: ")))
		   (arglist nil))
	       (if (/= (length expr) 1)
		   (error "Bad format"))
	       (if (eq (car-safe (car expr)) 'calcFunc-lambda)
		   (setq oper (list "$" (- (length (car expr)) 2) (car expr))
			 done t)
		 (calc-default-formula-arglist (car expr))
		 (setq arglist (sort arglist 'string-lessp)
		       arglist (read-from-minibuffer
				"Function argument list: "
				(if arglist
				    (prin1-to-string arglist)
				  "()")
				minibuffer-local-map
				t))
		 (setq oper (list "$"
				  (length arglist)
				  (append '(calcFunc-lambda)
					  (mapcar
					   (function
					    (lambda (x)
					      (list 'var
						    x
						    (intern
						     (concat
						      "var-"
						      (symbol-name x))))))
					   arglist)
					  expr))
		       done t))))
	    ((setq oper (assq key (cond ((eq prefix ?b) calc-b-oper-keys)
					((eq prefix ?c) calc-c-oper-keys)
					((eq prefix ?k) calc-k-oper-keys)
					(inv (if hyp
						 calc-inv-hyp-oper-keys
					       calc-inv-oper-keys))
					(t (if hyp
					       calc-hyp-oper-keys
					     calc-oper-keys)))))
	     (if (eq (nth 1 oper) 'user)
		 (let ((func (intern
			      (completing-read "Function name: "
					       obarray 'fboundp
					       nil "calcFunc-"))))
		   (if nargs
		       (setq oper (list "z" nargs func)
			     done t)
		     (if (and (fboundp func)
			      (consp (symbol-function func)))
			 (let* ((defn (symbol-function func))
				(args (nth 1 defn)))
			   (if (and (eq (car defn) 'lambda)
				    args
				    (not (memq (car args)
					       '(&optional &rest)))
				    (or (memq (nth 2 args)
					      '(&optional &rest nil))
					(memq (nth 1 args)
					      '(&optional &rest))))
			       (setq oper (list "z"
						(if (memq (nth 1 args)
							  '(&optional
							    &rest nil))
						    1 2)
						func)
				     done t)
			     (error "Function is not suitable for this operation")))
		       (message "Number of arguments: ")
		       (let ((nargs (read-char)))
			 (if (and (>= nargs ?0) (<= nargs ?9))
			     (setq oper (list "z" (- nargs ?0) func)
				   done t)
			   (beep))))))
	       (setq done t)))
	    (t (beep))))
    (and nargs
	 (/= nargs (nth 1 oper))
	 (error "Must be a %d-argument operator" nargs))
    (append (cdr oper)
	    (list
	     (concat (if prefix (char-to-string prefix) "")
		     (if inv "I" "") (if hyp "H" "")
		     (char-to-string key)))))
)

(defconst calc-oper-keys '( ( ?+ 2 calcFunc-add )
			    ( ?- 2 calcFunc-sub )
			    ( ?* 2 calcFunc-mul )
			    ( ?/ 2 calcFunc-div )
			    ( ?^ 2 calcFunc-pow )
			    ( ?| 2 calcFunc-vconcat )
			    ( ?% 2 calcFunc-mod )
			    ( ?\\ 2 calcFunc-idiv )
			    ( ?: 2 calcFunc-fdiv )
			    ( ?! 1 calcFunc-fact )
			    ( ?n 1 calcFunc-neg )
			    ( ?x user )
			    ( ?z user )
			    ( ?A 1 calcFunc-abs )
			    ( ?J 1 calcFunc-conj )
			    ( ?G 1 calcFunc-arg )
			    ( ?Q 1 calcFunc-sqrt )
			    ( ?N 2 calcFunc-min )
			    ( ?X 2 calcFunc-max )
			    ( ?F 1 calcFunc-floor )
			    ( ?R 1 calcFunc-round )
			    ( ?S 1 calcFunc-sin )
			    ( ?C 1 calcFunc-cos )
			    ( ?T 1 calcFunc-tan )
			    ( ?L 1 calcFunc-ln )
			    ( ?E 1 calcFunc-exp )
			    ( ?B 2 calcFunc-log )
))
(defconst calc-b-oper-keys '( ( ?a 2 calcFunc-and )
			      ( ?o 2 calcFunc-or )
			      ( ?x 2 calcFunc-xor )
			      ( ?d 2 calcFunc-diff )
			      ( ?n 1 calcFunc-not )
			      ( ?c 1 calcFunc-clip )
			      ( ?l 2 calcFunc-lsh )
			      ( ?r 2 calcFunc-rsh )
			      ( ?L 2 calcFunc-ash )
			      ( ?R 2 calcFunc-rash )
			      ( ?t 2 calcFunc-rot )
))
(defconst calc-c-oper-keys '( ( ?d 1 calcFunc-deg )
			      ( ?r 1 calcFunc-rad )
			      ( ?h 1 calcFunc-hms )
			      ( ?f 1 calcFunc-float )
			      ( ?F 1 calcFunc-frac )
))
(defconst calc-k-oper-keys '( ( ?g 2 calcFunc-gcd )
			      ( ?l 2 calcFunc-lcm )
			      ( ?b 2 calcFunc-choose )
			      ( ?d 1 calcFunc-dfact )
			      ( ?m 1 calcFunc-moebius )
			      ( ?p 2 calcFunc-perm )
			      ( ?r 1 calcFunc-random )
			      ( ?t 1 calcFunc-totient )
))
(defconst calc-inv-oper-keys '( ( ?F 1 calcFunc-ceil )
				( ?R 1 calcFunc-trunc )
				( ?Q 1 calcFunc-sqr )
				( ?S 1 calcFunc-arcsin )
				( ?C 1 calcFunc-arccos )
				( ?T 1 calcFunc-arctan )
				( ?L 1 calcFunc-exp )
				( ?E 1 calcFunc-ln )
))
(defconst calc-hyp-oper-keys '( ( ?F 1 calcFunc-ffloor )
				( ?R 1 calcFunc-fround )
				( ?S 1 calcFunc-sinh )
				( ?C 1 calcFunc-cosh )
				( ?T 1 calcFunc-tanh )
				( ?L 1 calcFunc-log10 )
				( ?E 1 calcFunc-exp10 )
))
(defconst calc-inv-hyp-oper-keys '( ( ?F 1 calcFunc-fceil )
				    ( ?R 1 calcFunc-ftrunc )
				    ( ?S 1 calcFunc-arcsinh )
				    ( ?C 1 calcFunc-arccosh )
				    ( ?T 1 calcFunc-arctanh )
				    ( ?L 1 calcFunc-exp10 )
				    ( ?E 1 calcFunc-log10 )
))




;;;; [calc-ext.el]

;;; User menu.

(defun calc-user-key-map ()
  (cdr (elt calc-mode-map ?z))
)

(defun calc-z-prefix-help ()
  (interactive)
  (let* ((msgs nil)
	 (buf "")
	 (kmap (sort (copy-sequence (calc-user-key-map))
		     (function (lambda (x y) (< (car x) (car y))))))
	 (flags (apply 'logior
		       (mapcar (function
				(lambda (k)
				  (calc-user-function-classify (car k))))
			       kmap))))
    (if (= (logand flags 8) 0)
	(calc-user-function-list kmap 7)
      (calc-user-function-list kmap 1)
      (setq msgs (cons buf msgs)
	    buf "")
      (calc-user-function-list kmap 6))
    (if (/= flags 0)
	(setq msgs (cons buf msgs)))
    (calc-do-prefix-help (nreverse msgs) "user" ?z))
)

(defun calc-user-function-classify (key)
  (cond ((/= key (downcase key))    ; upper-case
	 (if (assq (downcase key) (calc-user-key-map)) 9 1))
	((/= key (upcase key)) 2)   ; lower-case
	((= key ??) 0)
	(t 4))   ; other
)

(defun calc-user-function-list (map flags)
  (and map
       (let* ((key (car (car map)))
	      (kind (calc-user-function-classify key))
	      (func (cdr (car map))))
	 (if (= (logand kind flags) 0)
	     ()
	   (let* ((name (symbol-name func))
		  (name (if (string-match "\\`calc-" name)
			    (substring name 5) name))
		  (pos (string-match (char-to-string key) name))
		  (desc
		   (if (symbolp func)
		       (if (= (logand kind 3) 0)
			   (format "`%c' = %s" key name)
			 (if pos
			     (format "%s%c%s"
				     (downcase (substring name 0 pos))
				     (upcase key)
				     (downcase (substring name (1+ pos))))
			   (format "%c = %s"
				   (upcase key)
				   (downcase name))))
		     (char-to-string (upcase key)))))
	     (if (= (length buf) 0)
		 (setq buf (concat (if (= flags 1) "SHIFT + " "")
				   desc))
	       (if (> (+ (length buf) (length desc)) 58)
		   (setq msgs (cons buf msgs)
			 buf (concat (if (= flags 1) "SHIFT + " "")
				     desc))
		 (setq buf (concat buf ", " desc))))))
	 (calc-user-function-list (cdr map) flags)))
)



(defun calc-shift-Z-prefix-help ()
  (interactive)
  (calc-do-prefix-help
   '("Define, Undefine, Formula, Kbd-macro, Edit, Get-defn"
     "Permanent; Var-perm"
     "kbd-macros: [ (if), : (else), | (else-if), ] (end-if)"
     "kbd-macros: < > (repeat), ( ) (for), { } (loop)"
     "kbd-macros: / (break)"
     "kbd-macros: ` (save), ' (restore)")
   "user" ?Z)
)

;;;; [calc-prog.el]

(defun calc-user-define ()
  "Bind a Calculator command to a key sequence using the z prefix."
  (interactive)
  (message "Define user key: z-")
  (let ((key (read-char)))
    (if (= (calc-user-function-classify key) 0)
	(error "Can't redefine \"?\" key"))
    (let ((func (intern (completing-read (concat "Set key z "
						 (char-to-string key)
						 " to command: ")
					 obarray
					 'commandp
					 t
					 "calc-"))))
      (let* ((kmap (calc-user-key-map))
	     (old (assq key kmap)))
	(if old
	    (setcdr old func)
	  (setcdr kmap (cons (cons key func) (cdr kmap)))))))
)

(defun calc-user-undefine ()
  "Remove the definition on a Calculator z prefix key."
  (interactive)
  (message "Undefine user key: z-")
  (let ((key (read-char)))
    (if (= (calc-user-function-classify key) 0)
	(error "Can't undefine \"?\" key"))
    (let* ((kmap (calc-user-key-map)))
      (delq (or (assq key kmap)
		(assq (upcase key) kmap)
		(assq (downcase key) kmap)
		(error "No such user key is defined"))
	    kmap)))
)

(defun calc-user-define-formula ()
  "Define a new Calculator z-prefix command using formula at top of stack."
  (interactive)
  (calc-wrapper
   (let* ((form (calc-top 1))
	  (arglist nil)
	  odef key keyname cmd cmd-base func alist is-symb)
     (calc-default-formula-arglist form)
     (setq arglist (sort arglist 'string-lessp))
     (message "Define user key: z-")
     (setq key (read-char))
     (if (= (calc-user-function-classify key) 0)
	 (error "Can't redefine \"?\" key"))
     (setq key (and (not (memq key '(13 32))) key)
	   keyname (and key
			(if (or (and (<= ?0 key) (<= key ?9))
				(and (<= ?a key) (<= key ?z))
				(and (<= ?A key) (<= key ?Z)))
			    (char-to-string key)
			  (format "%03d" key)))
	   odef (assq key (calc-user-key-map)))
     (while
	 (progn
	   (setq cmd (completing-read "Define M-x command name: "
				      obarray 'commandp nil
				      (if (and odef (symbolp (cdr odef)))
					  (symbol-name (cdr odef))
					"calc-"))
		 cmd-base (and (string-match "\\`calc-\\(.+\\)\\'" cmd)
			       (math-match-substring cmd 1))
		 cmd (and (not (or (string-equal cmd "")
				   (string-equal cmd "calc-")))
			  (intern cmd)))
	   (and cmd
		(fboundp cmd)
		odef
		(not
		 (y-or-n-p
		  (if (get cmd 'calc-user-defn)
		      (concat "Replace previous definition for "
			      (symbol-name cmd) "? ")
		    "That name conflicts with a built-in Emacs function.  Replace this function? "))))))
     (if (and key (not cmd))
	 (setq cmd (intern (concat "calc-User-" keyname))))
     (while
	 (progn
	   (setq func (completing-read "Define algebraic function name: "
				       obarray 'fboundp nil
				       (concat "calcFunc-"
					       (if cmd-base
						   (if (string-match
							"\\`User-.+" cmd-base)
						       (concat
							"User"
							(substring cmd-base 5))
						     cmd-base)
						 "")))
		 func (and (not (or (string-equal func "")
				    (string-equal func "calcFunc-")))
			   (intern func)))
	   (and func
		(fboundp func)
		(not (fboundp cmd))
		odef
		(not
		 (y-or-n-p
		  (if (get func 'calc-user-defn)
		      (concat "Replace previous definition for "
			      (symbol-name func) "? ")
		    "That name conflicts with a built-in Emacs function.  Replace this function? "))))))
     (if (not func)
	 (setq func (intern (concat "calcFunc-User"
				    (or keyname
					(and cmd (symbol-name cmd))
					(format "%05d" (% (random) 10000)))))))
     (while
	 (progn
	   (setq alist (read-from-minibuffer "Function argument list: "
					     (if arglist
						 (prin1-to-string arglist)
					       "()")
					     minibuffer-local-map
					     t))
	   (and (not (calc-subsetp alist arglist))
		(y-or-n-p
		 "Okay for arguments that don't appear in formula to be ignored? "))))
     (setq is-symb (and alist
			func
			(y-or-n-p
			 "Leave it symbolic for non-constant arguments? ")))
     (if cmd
	 (progn
	   (fset cmd
		 (list 'lambda
		       '()
		       "User-defined Calculator function."
		       '(interactive)
		       (list 'calc-wrapper
			     (list 'calc-enter-result
				   (length alist)
				   (let ((name (symbol-name (or func cmd))))
				     (and (string-match
					   "\\([^-][^-]?[^-]?[^-]?\\)[^-]*\\'"
					   name)
					  (math-match-substring name 1)))
				   (list 'cons
					 (list 'quote func)
					 (list 'calc-top-list-n
					       (length alist)))))))
	   (put cmd 'calc-user-defn t)))
     (let ((body (list 'math-normalize (calc-fix-user-formula form))))
       (fset func
	     (append
	      (list 'lambda alist)
	      (and is-symb
		   (mapcar (function (lambda (v)
				       (list 'math-check-const v)))
			   alist))
	      (list body))))
     (put func 'calc-user-defn form)
     (if key
	 (let* ((kmap (calc-user-key-map))
		(old (assq key kmap)))
	   (if old
	       (setcdr old cmd)
	     (setcdr kmap (cons (cons key cmd) (cdr kmap)))))))
   (message ""))
)

(defun calc-default-formula-arglist (form)
  (if (consp form)
      (if (eq (car form) 'var)
	  (if (or (memq (nth 1 form) arglist)
		  (boundp (nth 2 form)))
	      ()
	    (setq arglist (cons (nth 1 form) arglist)))
	(calc-default-formula-arglist-step (cdr form))))
)

(defun calc-default-formula-arglist-step (l)
  (and l
       (progn
	 (calc-default-formula-arglist (car l))
	 (calc-default-formula-arglist-step (cdr l))))
)

(defun calc-subsetp (a b)
  (or (null a)
      (and (memq (car a) b)
	   (calc-subsetp (cdr a) b)))
)

(defun calc-fix-user-formula (f)
  (if (consp f)
      (cond ((and (eq (car f) 'var)
		  (memq (nth 1 f) alist))
	     (nth 1 f))
	    ((math-constp f)
	     (list 'quote f))
	    (t
	     (cons 'list
		   (cons (list 'quote (car f))
			 (mapcar 'calc-fix-user-formula (cdr f))))))
    f)
)


(defun calc-user-define-kbd-macro (arg)
  "Bind the most recent keyboard macro to a key sequence using the z prefix."
  (interactive "P")
  (or last-kbd-macro
      (error "No keyboard macro defined"))
  (message "Define last kbd macro on user key: z-")
  (let ((key (read-char)))
    (if (= (calc-user-function-classify key) 0)
	(error "Can't redefine \"?\" key"))
    (let ((cmd (intern (completing-read "Full name for new command: "
					obarray
					'commandp
					nil
					(concat "calc-User-"
						(if (or (and (>= key ?a)
							     (<= key ?z))
							(and (>= key ?A)
							     (<= key ?Z))
							(and (>= key ?0)
							     (<= key ?9)))
						    (char-to-string key)
						  (format "%03d" key)))))))
      (and (fboundp cmd)
	   (not (let ((f (symbol-function cmd)))
		  (or (stringp f)
		      (and (consp f)
			   (eq (car-safe (nth 3 f))
			       'calc-execute-kbd-macro)))))
	   (error "Function %s is already defined and not a keyboard macro"
		  cmd))
      (put cmd 'calc-user-defn t)
      (fset cmd (if (< (prefix-numeric-value arg) 0)
		    last-kbd-macro
		  (list 'lambda
			'(arg)
			'(interactive "P")
			(list 'calc-execute-kbd-macro
			      last-kbd-macro
			      'arg))))
      (let* ((kmap (calc-user-key-map))
	     (old (assq key kmap)))
	(if old
	    (setcdr old cmd)
	  (setcdr kmap (cons (cons key cmd) (cdr kmap)))))))
)


(defun calc-user-define-edit (prefix)
  "Edit the definition of a z-prefix command."
  (interactive "P")  ; but no calc-wrapper!
  (message "Edit definition of command: z-")
  (let* ((key (read-char))
	 (def (or (assq key (calc-user-key-map))
		  (assq (upcase key) (calc-user-key-map))
		  (assq (downcase key) (calc-user-key-map))
		  (error "No command defined for that key")))
	 (cmd (cdr def)))
    (if (symbolp cmd)
	(setq cmd (symbol-function cmd)))
    (cond ((or (stringp cmd)
	       (and (consp cmd)
		    (eq (car-safe (nth 3 cmd)) 'calc-execute-kbd-macro)))
	   (if (and (>= (prefix-numeric-value prefix) 0)
		    (fboundp 'edit-kbd-macro)
		    (symbolp (cdr def))
		    (eq major-mode 'calc-mode))
	       (progn
		 (if (and (< (window-width) (screen-width))
			  calc-display-trail)
		     (let ((win (get-buffer-window (calc-trail-buffer))))
		       (if win
			   (delete-window win))))
		 (edit-kbd-macro (cdr def) prefix nil
				 (function
				  (lambda (x)
				    (and calc-display-trail
					 (calc-wrapper
					  (calc-trail-display 1 t)))))
				 (function
				  (lambda (cmd)
				    (if (stringp (symbol-function cmd))
					(symbol-function cmd)
				      (nth 1 (nth 3 (symbol-function cmd))))))
				 (function
				  (lambda (new cmd)
				    (if (stringp (symbol-function cmd))
					(fset cmd new)
				      (setcar (cdr (nth 3 (symbol-function
							   cmd)))
					      new))))))
	     (calc-wrapper
	      (calc-edit-mode (list 'calc-finish-macro-edit
				    (list 'quote def)))
	      (insert (if (stringp cmd)
			  cmd
			(nth 1 (nth 3 cmd)))))
	     (calc-show-edit-buffer)))
	  (t (let* ((func (calc-stack-command-p cmd))
		    (defn (and func
			       (symbolp func)
			       (get func 'calc-user-defn))))
	       (if (and defn (calc-valid-formula-func func))
		   (progn
		     (calc-wrapper
		      (calc-edit-mode (list 'calc-finish-formula-edit
					    (list 'quote func)))
		      (insert (math-format-flat-expr defn 0) "\n"))
		     (calc-show-edit-buffer))
		 (error "That command's definition cannot be edited"))))))
)

(defun calc-finish-macro-edit (def)
  (let ((str (buffer-substring (point) (point-max))))
    (if (symbolp (cdr def))
	(if (stringp (symbol-function (cdr def)))
	    (fset (cdr def) str)
	  (setcar (cdr (nth 3 (symbol-function (cdr def)))) str))
      (setcdr def str)))
)

;;; The following are hooks into the MacEdit package from macedit.el.
(put 'calc-execute-extended-command 'MacEdit-print
     (function (lambda ()
		 (setq macro-str (concat "\excalc-" macro-str))))
)

(put 'calcDigit-start 'MacEdit-print
     (function (lambda ()
		 (if calc-algebraic-mode
		     (calc-macro-edit-algebraic)
		   (MacEdit-unread-chars key-last)
		   (let ((str "")
			 (min-bsp 0)
			 ch last)
		     (while (and (setq ch (MacEdit-read-char))
				 (or (and (>= ch ?0) (<= ch ?9))
				     (memq ch '(?\. ?e ?\_ ?n ?\: ?\# ?M
						    ?o ?h ?\@ ?\"))
				     (and (memq ch '(?\' ?m ?s))
					  (string-match "[@oh]" str))
				     (and (or (and (>= ch ?a) (<= ch ?z))
					      (and (>= ch ?A) (<= ch ?Z)))
					  (string-match
					   "^[-+]?\\(1[1-9]\\|[2-9][0-9]\\)#"
					   str))
				     (and (memq ch '(?\177 ?\C-h))
					  (> (length str) 0))
				     (and (memq ch '(?+ ?-))
					  (> (length str) 0)
					  (eq (aref str (1- (length str)))
					      ?e))))
		       (if (or (and (>= ch ?0) (<= ch ?9))
			       (and (or (not (memq ch '(?\177 ?\C-h)))
					(<= (length str) min-bsp))
				    (setq min-bsp (1+ (length str)))))
			   (setq str (concat str (char-to-string ch)))
			 (setq str (substring str 0 -1))))
		     (if (memq ch '(32 10 13))
			 (setq str (concat str (char-to-string ch)))
		       (MacEdit-unread-chars ch))
		     (insert "type \"")
		     (MacEdit-insert-string str)
		     (insert "\"\n")))))
)

(defun calc-macro-edit-algebraic ()
  (MacEdit-unread-chars key-last)
  (let ((str "")
	(min-bsp 0))
    (while (progn
	     (MacEdit-lookup-key calc-alg-ent-map)
	     (or (and (memq key-symbol '(self-insert-command
					 calcAlg-previous))
		      (< (length str) 60))
		 (memq key-symbol
			    '(backward-delete-char
			      delete-backward-char
			      backward-delete-char-untabify))
		 (eq key-last 9)))
      (setq macro-str (substring macro-str (length key-str)))
      (if (or (eq key-symbol 'self-insert-command)
	      (and (or (not (memq key-symbol '(backward-delete-char
					       delete-backward-char
					       backward-delete-char-untabify)))
		       (<= (length str) min-bsp))
		   (setq min-bsp (+ (length str) (length key-str)))))
	  (setq str (concat str key-str))
	(setq str (substring str 0 -1))))
    (if (memq key-last '(10 13))
	(setq str (concat str key-str)
	      macro-str (substring macro-str (length key-str))))
    (if (> (length str) 0)
	(progn
	  (insert "type \"")
	  (MacEdit-insert-string str)
	  (insert "\"\n"))))
)
(put 'calc-algebraic-entry 'MacEdit-print 'calc-macro-edit-algebraic)
(put 'calc-dollar-sign 'MacEdit-print 'calc-macro-edit-algebraic)

(defun calc-macro-edit-variable ()
  (let ((str "") ch)
    (insert (symbol-name key-symbol) "\n")
    (if (memq (MacEdit-peek-char) '(?\+ ?\- ?\* ?\/ ?\^ ?\|))
	(setq str (char-to-string (MacEdit-read-char))))
    (if (and (setq ch (MacEdit-peek-char))
	     (>= ch ?0) (<= ch ?9))
	(insert "type \"" str
		(char-to-string (MacEdit-read-char)) "\"\n")
      (if (> (length str) 0)
	  (insert "type \"" str "\"\n"))
      (MacEdit-read-argument)))
)
(put 'calc-store 'MacEdit-print 'calc-macro-edit-variable)
(put 'calc-recall 'MacEdit-print 'calc-macro-edit-variable)
(put 'calc-let 'MacEdit-print 'calc-macro-edit-variable)


(defun calc-finish-formula-edit (func)
  (let ((buf (current-buffer))
	(str (buffer-substring (point) (point-max)))
	(start (point))
	(body (calc-valid-formula-func func)))
    (set-buffer calc-original-buffer)
    (let ((val (math-read-expr str)))
      (if (eq (car-safe val) 'error)
	  (progn
	    (set-buffer buf)
	    (goto-char (+ start (nth 1 val)))
	    (error (nth 2 val))))
      (setcar (cdr body)
	      (let ((alist (nth 1 (symbol-function func))))
		(calc-fix-user-formula val)))
      (put func 'calc-user-defn val)))
)

(defun calc-valid-formula-func (func)
  (let ((def (symbol-function func)))
    (and (consp def)
	 (eq (car def) 'lambda)
	 (progn
	   (setq def (cdr (cdr def)))
	   (while (and def
		       (not (eq (car (car def)) 'math-normalize)))
	     (setq def (cdr def)))
	   (car def))))
)


(defun calc-get-user-defn ()
  "Extract the definition from a z-prefix command as a formula."
  (interactive)
  (calc-wrapper
   (message "Get definition of command: z-")
   (let* ((key (read-char))
	  (def (or (assq key (calc-user-key-map))
		   (assq (upcase key) (calc-user-key-map))
		   (assq (downcase key) (calc-user-key-map))
		   (error "No command defined for that key")))
	  (cmd (cdr def)))
     (if (symbolp cmd)
	 (setq cmd (symbol-function cmd)))
     (cond ((stringp cmd)
	    (message "Keyboard macro: %s" cmd))
	   (t (let* ((func (calc-stack-command-p cmd))
		     (defn (and func
				(symbolp func)
				(get func 'calc-user-defn))))
		(if defn
		    (calc-enter-result 0 "gdef" defn)
		  (error "That command is not defined by a formula")))))))
)


(defun calc-user-define-permanent ()
  "Make a user definition permanent by storing it in your .emacs file."
  (interactive)
  (calc-wrapper
   (message "Record in %s the command: z-" calc-settings-file)
   (let* ((key (read-char))
	  (def (or (assq key (calc-user-key-map))
		   (assq (upcase key) (calc-user-key-map))
		   (assq (downcase key) (calc-user-key-map))
		   (error "No command defined for that key"))))
     (set-buffer (find-file-noselect (substitute-in-file-name
				      calc-settings-file)))
     (goto-char (point-max))
     (insert "\n;;; Definition stored by Calc on " (current-time-string)
	     "\n(setq calc-defs (append '(\n")
     (let* ((cmd (cdr def))
	    (fcmd (and cmd (symbolp cmd) (symbol-function cmd)))
	    (pt (point))
	    (fill-column 70))
       (if (and fcmd
		(eq (car-safe fcmd) 'lambda)
		(get cmd 'calc-user-defn))
	   (progn
	     (insert (prin1-to-string
		      (cons 'defun (cons cmd (cdr fcmd))))
		     "\n")
	     (fill-region pt (point))
	     (indent-rigidly pt (point) 3)
	     (delete-region pt (1+ pt))
	     (let* ((func (calc-stack-command-p cmd))
		    (ffunc (and func (symbolp func) (symbol-function func)))
		    (pt (point)))
	       (and ffunc
		    (eq (car-safe ffunc) 'lambda)
		    (get func 'calc-user-defn)
		    (progn
		      (insert (prin1-to-string
			       (cons 'defun (cons func (cdr ffunc))))
			      "\n")
		      (fill-region pt (point))
		      (indent-rigidly pt (point) 3)
		      (delete-region pt (1+ pt))))))
	 (and (stringp fcmd)
	      (insert "  (fset '" (prin1-to-string cmd)
		      " " (prin1-to-string fcmd) ")\n")))
       (insert "  (define-key calc-mode-map "
	       (prin1-to-string (concat "z" (char-to-string key)))
	       " '"
	       (prin1-to-string cmd)
	       "))\n"))
     (insert " (and (boundp 'calc-defs) calc-defs)))\n")
     (save-buffer)))
)

(defun calc-stack-command-p (cmd)
  (if (and cmd (symbolp cmd))
      (and (fboundp cmd)
	   (calc-stack-command-p (symbol-function cmd)))
    (and (consp cmd)
	 (eq (car cmd) 'lambda)
	 (setq cmd (or (assq 'calc-wrapper cmd)
		       (assq 'calc-slow-wrapper cmd)))
	 (setq cmd (assq 'calc-enter-result cmd))
	 (memq (car (nth 3 cmd)) '(cons list))
	 (eq (car (nth 1 (nth 3 cmd))) 'quote)
	 (nth 1 (nth 1 (nth 3 cmd)))))
)

(defun calc-permanent-variable ()
  "Save a variable's value in your .emacs file."
  (interactive)
  (calc-wrapper
   (let ((var (let ((minibuffer-completion-table obarray)
		    (minibuffer-completion-predicate 'boundp)
		    (minibuffer-completion-confirm t)
		    (oper "r"))
		(read-from-minibuffer
		 "Save variable (default=all): " "var-"
		 calc-store-var-map nil)))
	 pos)
     (or (equal var "") (equal var "var-")
	 (and (boundp (intern var)) (intern var))
	 (error "No such variable"))
     (set-buffer (find-file-noselect (substitute-in-file-name
				      calc-settings-file)))
     (if (or (equal var "") (equal var "var-"))
	 (mapatoms (function
		    (lambda (x)
		      (and (string-match "\\`var-" (symbol-name x))
			   (boundp x)
			   (symbol-value x)
			   (not (eq (car-safe (symbol-value x))
				    'special-const))
			   (calc-insert-permanent-variable x)))))
       (calc-insert-permanent-variable (intern var)))
     (save-buffer)))
)

(defun calc-insert-permanent-variable (var)
  (goto-char (point-min))
  (if (search-forward (concat "(setq " (symbol-name var) " '") nil t)
      (progn
	(setq pos (point-marker))
	(forward-line -1)
	(if (looking-at ";;; Variable .* stored by Calc on ")
	    (progn
	      (delete-region (match-end 0) (progn (end-of-line) (point)))
	      (insert (current-time-string))))
	(goto-char (- pos 8 (length (symbol-name var))))
	(forward-sexp 1)
	(backward-char 1)
	(delete-region pos (point)))
    (goto-char (point-max))
    (insert "\n;;; Variable \""
	    (symbol-name var)
	    "\" stored by Calc on "
	    (current-time-string)
	    "\n(setq "
	    (symbol-name var)
	    " ')\n")
    (backward-char 2))
  (insert (prin1-to-string (symbol-value var)))
  (forward-line 1)
)


(defun calc-insert-variables (buf)
  "Insert all variables beginning with \"var-\" in the specified buffer."
  (interactive "bBuffer in which to save variable values: ")
  (save-excursion
    (set-buffer buf)
    (mapatoms (function
	       (lambda (x)
		 (and (string-match "\\`var-" (symbol-name x))
		      (boundp x)
		      (symbol-value x)
		      (not (eq (car-safe (symbol-value x)) 'special-const))
		      (insert "(setq "
			      (symbol-name x)
			      " "
			      (prin1-to-string
			       (if (stringp (symbol-value x))
				   (symbol-value x)
				 (math-format-value (symbol-value x) 1000)))
			      ")\n"))))))
)


(defun calc-call-last-kbd-macro (arg)
  "Execute the most recent keyboard macro."
  (interactive "P")
  (and defining-kbd-macro
       (error "Can't execute anonymous macro while defining one"))
  (or last-kbd-macro
      (error "No kbd macro has been defined"))
  (calc-execute-kbd-macro last-kbd-macro arg)
)

(defun calc-execute-kbd-macro (mac arg)
  (if (< (prefix-numeric-value arg) 0)
      (execute-kbd-macro mac (- (prefix-numeric-value arg)))
    (if calc-executing-macro
	(execute-kbd-macro mac arg)
      (calc-slow-wrapper
       (let ((old-stack-whole (copy-sequence calc-stack))
	     (old-stack-top calc-stack-top)
	     (old-buffer-size (buffer-size))
	     (old-refresh-count calc-refresh-count))
	 (unwind-protect
	     (let ((calc-executing-macro mac))
	       (execute-kbd-macro mac arg))
	   (calc-select-buffer)
	   (let ((new-stack (reverse calc-stack))
		 (old-stack (reverse old-stack-whole)))
	     (while (and new-stack old-stack
			 (equal (car new-stack) (car old-stack)))
	       (setq new-stack (cdr new-stack)
		     old-stack (cdr old-stack)))
	     (calc-record-list (mapcar 'car new-stack) "kmac")
	     (calc-record-undo (list 'set 'saved-stack-top old-stack-top))
	     (and old-stack
		  (calc-record-undo (list 'pop 1 (mapcar 'car old-stack))))
	     (let ((calc-stack old-stack-whole)
		   (calc-stack-top 0))
	       (calc-cursor-stack-index (length old-stack)))
	     (if (and (= old-buffer-size (buffer-size))
		      (= old-refresh-count calc-refresh-count))
		 (let ((buffer-read-only nil))
		   (delete-region (point) (point-max))
		   (while new-stack
		     (calc-record-undo (list 'push 1))
		     (let ((fmt (math-format-stack-value
				 (car (car new-stack)))))
		       (setcar (cdr (car new-stack)) (calc-count-lines fmt))
		       (insert fmt "\n"))
		     (setq new-stack (cdr new-stack)))
		   (calc-renumber-stack))
	       (while new-stack
		 (calc-record-undo (list 'push 1))
		 (setq new-stack (cdr new-stack)))
	       (calc-refresh))
	     (calc-record-undo (list 'set 'saved-stack-top 0))))))))
)


(defun calc-kbd-if ()
  "An \"if\" statement in a Calc keyboard macro.
Usage:  cond  Z[  then-part  Z:  cond  Z|  else-if-part ...  Z:  else-part  Z]"
  (interactive)
  (calc-wrapper
   (let ((cond (calc-top-n 1)))
     (calc-pop-stack 1)
     (if (math-is-true cond)
	 (if defining-kbd-macro
	     (message "If true..."))
       (if defining-kbd-macro
	   (message "Condition is false; skipping to Z: or Z] ..."))
       (calc-kbd-skip-to-else-if t))))
)

(defun calc-kbd-else-if ()
  (interactive)
  (calc-kbd-if)
)

(defun calc-kbd-skip-to-else-if (else-okay)
  (let ((count 0)
	ch)
    (while (>= count 0)
      (setq ch (read-char))
      (if (= ch -1)
	  (error "Unterminated Z[ in keyboard macro"))
      (if (= ch ?Z)
	  (progn
	    (setq ch (read-char))
	    (cond ((= ch ?\[)
		   (setq count (1+ count)))
		  ((= ch ?\])
		   (setq count (1- count)))
		  ((= ch ?\:)
		   (and (= count 0)
			else-okay
			(setq count -1)))
		  ((eq ch 7)
		   (keyboard-quit))))))
    (and defining-kbd-macro
	 (if (= ch ?\:)
	     (message "Else...")
	   (message "End-if..."))))
)

(defun calc-kbd-end-if ()
  (interactive)
  (if defining-kbd-macro
      (message "End-if..."))
)

(defun calc-kbd-else ()
  (interactive)
  (if defining-kbd-macro
      (message "Else; skipping to Z] ..."))
  (calc-kbd-skip-to-else-if nil)
)


(defun calc-kbd-repeat ()
  "A counted loop in a Calc keyboard macro.
Usage:  count  Z<  body  Z>

Any number of break-commands may be embedded in the body:
   cond  Z/  stops the loop prematurely if cond is true."
  (interactive)
  (let (count)
    (calc-wrapper
     (setq count (math-trunc (calc-top-n 1)))
     (or (Math-integerp count)
	 (error "Count must be an integer"))
     (if (Math-integer-negp count)
	 (setq count 0))
     (or (integerp count)
	 (setq count 1000000))
     (calc-pop-stack 1))
    (calc-kbd-loop count))
)

(defun calc-kbd-for (dir)
  "A counted loop in a Calc keyboard macro.
Usage:  initial  final  Z(  body  step  Z)

During the loop, an internal counter is incremented from INITIAL to FINAL
in steps of STEP.  The Z( command pops INITIAL and FINAL, and pushes the
current counter value each time through the loop.  The Z) command pops
STEP.  If INITIAL < FINAL, the loop terminates as soon as the counter
exceeds FINAL.  If INITIAL > FINAL, the loop terminates as soon as the
counter becomes less than FINAL.  If INITIAL = FINAL, the loop executes
once.  If INITIAL and FINAL cannot be compared (say because at least one
is a symbolic formula), the loop continues until it is halted with Z/.
No matter what the relationship between INITIAL and FINAL, the body
always executes at least once.

A numeric prefix argument specifies a forced direction:  If 1, the loop
terminates when the counter exceeds FINAL, and will execute zero times
if INITIAL > FINAL.  Likewise, -1 forces a downward-counting loop.

Any number of break-commands may be embedded in the body:
   cond  Z/  stops the loop prematurely if cond is true."
  (interactive "P")
  (let (init final)
    (calc-wrapper
     (setq init (calc-top-n 2)
	   final (calc-top-n 1))
     (or (and (math-anglep init) (math-anglep final))
	 (error "Initial and final values must be real numbers"))
     (calc-pop-stack 2))
    (calc-kbd-loop nil init final (and dir (prefix-numeric-value dir))))
)

(defun calc-kbd-loop (rpt-count &optional initial final dir)
  "A conditional loop in a Calc keyboard macro.
Usage:  Z{  body  Z}

At least one break-command is normally present in the body:
   cond  Z/  stops the loop if cond is true.

With a numeric prefix argument, loops at most that many times."
  (interactive "P")
  (setq rpt-count (if rpt-count (prefix-numeric-value rpt-count) 1000000))
  (let* ((count 0)
	 (parts nil)
	 (body "")
	 (open last-command-char)
	 (counter initial)
	 ch)
    (or executing-macro
	(message "Reading loop body..."))
    (while (>= count 0)
      (setq ch (read-char))
      (if (= ch -1)
	  (error "Unterminated Z%c in keyboard macro" open))
      (if (= ch ?Z)
	  (progn
	    (setq ch (read-char)
		  body (concat body "Z" (char-to-string ch)))
	    (cond ((memq ch '(?\< ?\( ?\{))
		   (setq count (1+ count)))
		  ((memq ch '(?\> ?\) ?\}))
		   (setq count (1- count)))
		  ((and (= ch ?/)
			(= count 0))
		   (setq parts (nconc parts (list (substring body 0 -2)))
			 body ""))
		  ((eq ch 7)
		   (keyboard-quit))))
	(setq body (concat body (char-to-string ch)))))
    (if (/= ch (cdr (assq open '( (?\< . ?\>) (?\( . ?\)) (?\{ . ?\}) ))))
	(error "Mismatched Z%c and Z%c in keyboard macro" open ch))
    (or executing-macro
	(message "Looping..."))
    (setq body (substring body 0 -2))
    (and (not executing-macro)
	 (= rpt-count 1000000)
	 (null parts)
	 (null counter)
	 (progn
	   (message "Warning: Infinite loop!  Not executing.")
	   (setq rpt-count 0)))
    (or (not initial) dir
	(setq dir (math-compare final initial)))
    (calc-wrapper
     (while (> rpt-count 0)
       (let ((part parts))
	 (if counter
	     (if (cond ((eq dir 0) (math-equal final counter))
		       ((eq dir 1) (math-lessp final counter))
		       ((eq dir -1) (math-lessp counter final)))
		 (setq rpt-count 0)
	       (calc-push counter)))
	 (while (and part (> rpt-count 0))
	   (execute-kbd-macro (car part))
	   (if (math-is-true (calc-top-n 1))
	       (setq rpt-count 0)
	     (setq part (cdr part)))
	   (calc-pop-stack 1))
	 (if (> rpt-count 0)
	     (progn
	       (execute-kbd-macro body)
	       (if counter
		   (let ((step (calc-top-n 1)))
		     (calc-pop-stack 1)
		     (setq counter (calcFunc-add counter step)))
		 (setq rpt-count (1- rpt-count))))))))
    (or executing-macro
	(message "Looping...done")))
)

(defun calc-kbd-end-repeat ()
  (interactive)
  (error "Unbalanced Z> in keyboard macro")
)

(defun calc-kbd-end-for ()
  (interactive)
  (error "Unbalanced Z) in keyboard macro")
)

(defun calc-kbd-end-loop ()
  (interactive)
  (error "Unbalanced Z} in keyboard macro")
)

(defun calc-kbd-break ()
  "Break out of a keyboard macro, or out of a Z< Z> or Z{ Z} loop in a macro.
Usage:  cond  Z/    breaks only if cond is true.  Use \"1 Z/\" to break always."
  (interactive)
  (calc-wrapper
   (let ((cond (calc-top-n 1)))
     (calc-pop-stack 1)
     (if (math-is-true cond)
	 (error "Keyboard macro aborted."))))
)


(defun calc-kbd-push ()
  "Save modes and quick variables around a section of a keyboard macro.

Saved:  var-0 thru var-9, precision, word size, angular mode,
simplification mode, vector mapping direction, Alg, Sym, Frac, Polar modes.

Values are restored on exit, even if the macro halts with an error."
  (interactive)
  (calc-wrapper
   (let* ((var-0 (and (boundp 'var-0) var-0))
	  (var-1 (and (boundp 'var-1) var-1))
	  (var-2 (and (boundp 'var-2) var-2))
	  (var-3 (and (boundp 'var-3) var-3))
	  (var-4 (and (boundp 'var-4) var-4))
	  (var-5 (and (boundp 'var-5) var-5))
	  (var-6 (and (boundp 'var-6) var-6))
	  (var-7 (and (boundp 'var-7) var-7))
	  (var-8 (and (boundp 'var-8) var-8))
	  (var-9 (and (boundp 'var-9) var-9))
	  (calc-internal-prec calc-internal-prec)
	  (calc-word-size calc-word-size)
	  (calc-angle-mode calc-angle-mode)
	  (calc-simplify-mode calc-simplify-mode)
	  (calc-mapping-dir calc-mapping-dir)
	  (calc-algebraic-mode calc-algebraic-mode)
	  (calc-symbolic-mode calc-symbolic-mode)
	  (calc-prefer-frac calc-prefer-frac)
	  (calc-complex-mode calc-complex-mode)
	  (count 0)
	  (body "")
	  ch)
     (if (or executing-macro defining-kbd-macro)
	 (progn
	   (if defining-kbd-macro
	       (message "Reading body..."))
	   (while (>= count 0)
	     (setq ch (read-char))
	     (if (= ch -1)
		 (error "Unterminated Z` in keyboard macro"))
	     (if (= ch ?Z)
		 (progn
		   (setq ch (read-char)
			 body (concat body "Z" (char-to-string ch)))
		   (cond ((eq ch ?\`)
			  (setq count (1+ count)))
			 ((eq ch ?\')
			  (setq count (1- count)))
			 ((eq ch 7)
			  (keyboard-quit))))
	       (setq body (concat body (char-to-string ch)))))
	   (if defining-kbd-macro
	       (message "Reading body...done"))
	   (let ((calc-kbd-push-level 0))
	     (execute-kbd-macro (substring body 0 -2))))
       (let ((calc-kbd-push-level (1+ calc-kbd-push-level)))
	 (message "Saving modes; type Z' to restore")
	 (recursive-edit)))))
)
(setq calc-kbd-push-level 0)

(defun calc-kbd-pop ()
  (interactive)
  (if (> calc-kbd-push-level 0)
      (progn
	(message "Mode settings restored")
	(exit-recursive-edit))
    (error "Unbalanced Z' in keyboard macro"))
)


(defun calc-kbd-report (msg)
  "Display the number on the top of the stack in the echo area.
This will normally be used to report progress in a keyboard macro."
  (interactive "sMessage: ")
  (calc-wrapper
   (let ((executing-macro nil)
	 (defining-kbd-macro nil))
     (math-working msg (calc-top-n 1))))
)

(defun calc-kbd-query (msg)
  "Pause during keyboard macro execution to do an algebraic entry."
  (interactive "sPrompt: ")
  (calc-wrapper
   (let ((executing-macro nil)
	 (defining-kbd-macro nil))
     (calc-alg-entry nil (and (not (equal msg "")) msg))))
)






;;;; [calc-ext.el]

;;;; Caches.

(defmacro math-defcache (name init form)
  (let ((cache-prec (intern (concat (symbol-name name) "-cache-prec")))
	(cache-val (intern (concat (symbol-name name) "-cache")))
	(last-prec (intern (concat (symbol-name name) "-last-prec")))
	(last-val (intern (concat (symbol-name name) "-last"))))
    (list 'progn
	  (list 'setq cache-prec (if init (math-numdigs (nth 1 init)) -100))
	  (list 'setq cache-val (list 'quote init))
	  (list 'setq last-prec -100)
	  (list 'setq last-val nil)
	  (list 'setq 'math-cache-list
		(list 'cons
		      (list 'quote cache-prec)
		      (list 'cons
			    (list 'quote last-prec)
			    'math-cache-list)))
	  (list 'defun
		name ()
		(list 'or
		      (list '= last-prec 'calc-internal-prec)
		      (list 'setq
			    last-val
			    (list 'math-normalize
				  (list 'progn
					(list 'or
					      (list '>= cache-prec
						    'calc-internal-prec)
					      (list 'setq
						    cache-val
						    (list 'let
							  '((calc-internal-prec
							     (+ calc-internal-prec
								4)))
							  form)
						    cache-prec
						    '(+ calc-internal-prec 2)))
					cache-val))
			    last-prec 'calc-internal-prec))
		last-val)))
)
(put 'math-defcache 'lisp-indent-hook 2)

;;; Betcha didn't know that pi = 16 atan(1/5) - 4 atan(1/239).   [F] [Public]
(math-defcache math-pi (float (bigpos 463 238 793 589 653 592 141 3) -21)
  (math-add-float (math-mul-float '(float 16 0)
				  (math-arctan-raw '(float 2 -1)))
		  (math-mul-float '(float -4 0)
				  (math-arctan-raw
				   (math-float '(frac 1 239))))))

(math-defcache math-two-pi nil
  (math-mul-float (math-pi) '(float 2 0)))

(math-defcache math-pi-over-2 nil
  (math-mul-float (math-pi) '(float 5 -1)))

(math-defcache math-pi-over-4 nil
  (math-mul-float (math-pi) '(float 25 -2)))

(math-defcache math-pi-over-180 nil
  (math-div-float (math-pi) '(float 18 1)))

(math-defcache math-sqrt-e (float (bigpos 849 146 128 700 270 721 648 1) -21)
  (math-add-float '(float 1 0) (math-exp-minus-1-raw '(float 5 -1))))

(math-defcache math-e nil
  (math-sqr (math-sqrt-e)))


(defun math-half-circle (symb)
  (if (eq calc-angle-mode 'rad)
      (if symb
	  '(var pi var-pi)
	(math-pi))
    180)
)

(defun math-full-circle (symb)
  (math-mul 2 (math-half-circle symb))
)

(defun math-quarter-circle (symb)
  (math-div (math-half-circle symb) 2)
)




;;;; Miscellaneous math routines.

;;; True if A is an odd integer.  [P R R] [Public]
(defun math-oddp (a)
  (if (consp a)
      (and (memq (car a) '(bigpos bigneg))
	   (= (% (nth 1 a) 2) 1))
    (/= (% a 2) 0))
)

;;; True if A is numerically an integer.  [P x] [Public]
(defun math-num-integerp (a)
  (or (Math-integerp a)
      (Math-messy-integerp a))
)
(defmacro Math-num-integerp (a)
  (` (or (not (consp (, a)))
	 (memq (car (, a)) '(bigpos bigneg))
	 (and (eq (car (, a)) 'float)
	      (>= (nth 2 (, a)) 0))))
)

;;; True if A is (numerically) a non-negative integer.  [P N] [Public]
(defun math-num-natnump (a)
  (or (natnump a)
      (eq (car-safe a) 'bigpos)
      (and (eq (car-safe a) 'float)
	   (Math-natnump (nth 1 a))
	   (>= (nth 2 a) 0)))
)

;;; True if A is an integer or will evaluate to an integer.  [P x] [Public]
(defun math-provably-integerp (a)
  (or (Math-integerp a)
      (memq (car-safe a) '(calcFunc-trunc
			   calcFunc-round
			   calcFunc-floor
			   calcFunc-ceil)))
)

;;; True if A is a real or will evaluate to a real.  [P x] [Public]
(defun math-provably-realp (a)
  (or (Math-realp a)
      (math-provably-integer a)
      (memq (car-safe a) '(abs arg)))
)

;;; True if A is a non-real, complex number.  [P x] [Public]
(defun math-complexp (a)
  (memq (car-safe a) '(cplx polar))
)

;;; True if A is a non-real, rectangular complex number.  [P x] [Public]
(defun math-rect-complexp (a)
  (eq (car-safe a) 'cplx)
)

;;; True if A is a non-real, polar complex number.  [P x] [Public]
(defun math-polar-complexp (a)
  (eq (car-safe a) 'polar)
)

;;; True if A is a matrix.  [P x] [Public]
(defun math-matrixp (a)
  (and (Math-vectorp a)
       (Math-vectorp (nth 1 a))
       (cdr (nth 1 a))
       (math-matrixp-step (cdr (cdr a)) (length (nth 1 a))))
)

(defun math-matrixp-step (a len)   ; [P L]
  (or (null a)
      (and (Math-vectorp (car a))
	   (= (length (car a)) len)
	   (math-matrixp-step (cdr a) len)))
)

;;; True if A is a square matrix.  [P V] [Public]
(defun math-square-matrixp (a)
  (let ((dims (math-mat-dimens a)))
    (and (cdr dims)
	 (= (car dims) (nth 1 dims))))
)

;;; True if A is any real scalar data object.  [P x]
(defun math-real-objectp (a)    ;  [Public]
  (or (integerp a)
      (memq (car-safe a) '(bigpos bigneg frac float hms sdev intv mod)))
)

;;; True if A is an object not composed of sub-formulas .  [P x] [Public]
(defun math-primp (a)
  (or (integerp a)
      (memq (car-safe a) '(bigpos bigneg frac float cplx polar
				  hms mod var)))
)
(defmacro Math-primp (a)
  (` (or (not (consp (, a)))
	 (memq (car (, a)) '(bigpos bigneg frac float cplx polar
				    hms mod var))))
)

;;; True if A is a constant or vector of constants.  [P x] [Public]
(defun math-constp (a)
  (or (math-scalarp a)
      (and (memq (car-safe a) '(sdev intv vec))
	   (progn
	     (while (and (setq a (cdr a))
			 (math-constp (car a))))
	     (null a))))
)

(defmacro Math-lessp (a b)
  (` (= (math-compare (, a) (, b)) -1))
)


;;; Verify that A is an integer and return A in integer form.  [I N; - x]
(defun math-check-integer (a)   ;  [Public]
  (cond ((integerp a) a)  ; for speed
	((math-integerp a) a)
	((math-messy-integerp a)
	 (math-trunc a))
	(t (math-reject-arg a 'integerp)))
)

;;; Verify that A is a small integer and return A in integer form.  [S N; - x]
(defun math-check-fixnum (a)   ;  [Public]
  (cond ((integerp a) a)  ; for speed
	((Math-num-integerp a)
	 (let ((a (math-trunc a)))
	   (if (integerp a)
	       a
	     (if (or (Math-lessp (lsh -1 -1) a)
		     (Math-lessp a (- (lsh -1 -1))))
		 (math-reject-arg a 'fixnump)
	       (math-fixnum a)))))
	(t (math-reject-arg a 'fixnump)))
)

;;; Verify that A is an integer >= 0 and return A in integer form.  [I N; - x]
(defun math-check-natnum (a)    ;  [Public]
  (cond ((natnump a) a)
	((and (not (math-negp a))
	      (Math-num-integerp a))
	 (math-trunc a))
	(t (math-reject-arg a 'natnump)))
)

;;; Verify that A is in floating-point form, or force it to be a float.  [F N]
(defun math-check-float (a)    ; [Public]
  (cond ((eq (car-safe a) 'float) a)
	((Math-vectorp a) (math-map-vec 'math-check-float a))
	((Math-objectp a) (math-float a))
	(t a))
)

;;; Verify that A is a constant.
(defun math-check-const (a)
  (if (math-constp a)
      a
    (math-reject-arg a 'constp))
)


;;; Coerce integer A to be a small integer.  [S I]
(defun math-fixnum (a)
  (if (consp a)
      (if (cdr a)
	  (if (eq (car a) 'bigneg)
	      (- (math-fixnum-big (cdr a)))
	    (math-fixnum-big (cdr a)))
	0)
    a)
)

(defun math-fixnum-big (a)
  (if (cdr a)
      (+ (car a) (* (math-fixnum-big (cdr a)) 1000))
    (car a))
)


(defun math-bignum-test (a)   ; [B N; B s; b b]
  (if (consp a)
      a
    (math-bignum a))
)
(defmacro Math-bignum-test (a)   ; [B N; B s; b b]
  (` (if (consp (, a))
	 (, a)
       (math-bignum (, a))))
)


;;; Return 0 for zero, -1 for negative, 1 for positive.  [S n] [Public]
(defun math-sign (a)
  (cond ((math-posp a) 1)
	((math-negp a) -1)
	((math-zerop a) 0)
	(t (calc-record-why 'realp a)
	   (list 'calcFunc-sign a)))
)
(fset 'calcFunc-sign (symbol-function 'math-sign))

;;; Return 0 if A is numerically equal to B, <0 if less, >0 if more.
;;; Arguments must be normalized!  [S N N]
(defun math-compare (a b)
  (cond ((equal a b) 0)
	((and (integerp a) (Math-integerp b))
	 (if (consp b)
	     (if (eq (car b) 'bigpos) -1 1)
	   (if (< a b) -1 1)))
	((and (eq (car-safe a) 'bigpos) (Math-integerp b))
	 (if (eq (car-safe b) 'bigpos)
	     (math-compare-bignum (cdr a) (cdr b))
	   1))
	((and (eq (car-safe a) 'bigneg) (Math-integerp b))
	 (if (eq (car-safe b) 'bigneg)
	     (math-compare-bignum (cdr b) (cdr a))
	   -1))
	((eq (car-safe a) 'frac)
	 (if (eq (car-safe b) 'frac)
	     (math-compare (math-mul (nth 1 a) (nth 2 b))
			   (math-mul (nth 1 b) (nth 2 a)))
	   (math-compare (nth 1 a) (math-mul b (nth 2 a)))))
	((eq (car-safe b) 'frac)
	 (math-compare (math-mul a (nth 2 b)) (nth 1 b)))
	((and (eq (car-safe a) 'float) (eq (car-safe b) 'float))
	 (if (math-lessp-float a b) -1 1))
	((and (Math-anglep a) (Math-anglep b))
	 (math-sign (math-add a (math-neg b))))
	((eq (car-safe a) 'var)
	 2)
	(t
	 (if (and (consp a) (consp b)
		  (eq (car a) (car b))
		  (math-compare-lists (cdr a) (cdr b)))
	     0
	   2)))
)

;;; Compare two bignum digit lists, return -1 for A<B, 0 for A=B, 1 for A>B.
(defun math-compare-bignum (a b)   ; [S l l]
  (let ((res 0))
    (while (and a b)
      (if (< (car a) (car b))
	  (setq res -1)
	(if (> (car a) (car b))
	    (setq res 1)))
      (setq a (cdr a)
	    b (cdr b)))
    (if a
	(progn
	  (while (eq (car a) 0) (setq a (cdr a)))
	  (if a 1 res))
      (while (eq (car b) 0) (setq b (cdr b)))
      (if b -1 res)))
)

(defun math-compare-lists (a b)
  (cond ((null a) (null b))
	((null b) nil)
	(t (and (math-equal (car a) (car b))
		(math-compare-lists (cdr a) (cdr b)))))
)

(defun math-lessp-float (a b)   ; [P F F]
  (let ((ediff (- (nth 2 a) (nth 2 b))))
    (if (>= ediff 0)
	(if (>= ediff (+ calc-internal-prec calc-internal-prec))
	    (Math-integer-negp (nth 1 a))
	  (Math-lessp (math-scale-int (nth 1 a) ediff)
		      (nth 1 b)))
      (if (>= (setq ediff (- ediff))
	      (+ calc-internal-prec calc-internal-prec))
	  (Math-integer-posp (nth 1 b))
	(Math-lessp (nth 1 a)
		    (math-scale-int (nth 1 b) ediff)))))
)

;;; True if A is numerically equal to B.  [P N N] [Public]
(defun math-equal (a b)
  (= (math-compare a b) 0)
)

;;; True if A is numerically less than B.  [P R R] [Public]
(defun math-lessp (a b)
  (= (math-compare a b) -1)
)

;;; True if A is numerically equal to the integer B.  [P N S] [Public]
;;; B must not be a multiple of 10.
(defun math-equal-int (a b)
  (or (eq a b)
      (and (eq (car-safe a) 'float)
	   (eq (nth 1 a) b)
	   (= (nth 2 a) 0)))
)
(defmacro Math-equal-int (a b)
  (` (or (eq (, a) (, b))
	 (and (consp (, a))
	      (eq (car (, a)) 'float)
	      (eq (nth 1 (, a)) (, b))
	      (= (nth 2 (, a)) 0))))
)


;;;; [calc-map.el]

;;; Convert a variable name (as a formula) into a like-looking function name.
(defun math-var-to-calcFunc (f)
  (if (eq (car-safe f) 'var)
      (if (fboundp (nth 2 f))
	  (nth 2 f)
	(intern (concat "calcFunc-" (symbol-name (nth 1 f)))))
    (if (memq (car-safe f) '(lambda calcFunc-lambda))
	f
      (math-reject-arg f "Expected a function name")))
)

;;; Convert a function name into a like-looking variable name formula.
(defun math-calcFunc-to-var (f)
  (if (symbolp f)
      (let ((base (if (string-match "\\`calcFunc-\\(.+\\)\\'" (symbol-name f))
		      (math-match-substring (symbol-name f) 1)
		    (symbol-name f))))
	(list 'var
	      (intern base)
	      (intern (concat "var-" base))))
    f)
)

;;; Expand a function call using "lambda" notation.
(defun math-build-call (f args)
  (if (eq (car-safe f) 'calcFunc-lambda)
      (if (= (length args) (- (length f) 2))
	  (let ((argnames (cdr f))
		(argvals args)
		(res (nth (1- (length f)) f)))
	    (while argvals 
	      (setq res (math-expr-subst res (car argnames) (car argvals))
		    argnames (cdr argnames)
		    argvals (cdr argvals)))
	    res)
	(cons 'calcFunc-call (cons (math-calcFunc-to-var f) args)))
    (cons f args))
)

(defun calcFunc-call (f &rest args)
  (setq args (math-build-call (math-var-to-calcFunc f) args))
  (if (eq (car-safe args) 'calcFunc-call)
      args
    (math-normalize args))
)

(defun calcFunc-apply (f args)
  (or (Math-vectorp args)
      (math-reject-arg args 'vectorp))
  (apply 'calcFunc-call (cons f (cdr args)))
)



;;;; [calc-vec.el]

;;;; Vectors.

;;; Return the dimensions of a matrix as a list.  [l x] [Public]
(defun math-mat-dimens (m)
  (if (math-vectorp m)
      (if (math-matrixp m)
	  (cons (1- (length m))
		(math-mat-dimens (nth 1 m)))
	(list (1- (length m))))
    nil)
)


;;; Apply a function elementwise to vector A.  [V X V; N X N] [Public]
(defun math-map-vec (f a)
  (if (math-vectorp a)
      (cons 'vec (mapcar f (cdr a)))
    (funcall f a))
)

(defun math-dimension-error ()
  (calc-record-why "Dimension error")
  (signal 'wrong-type-argument nil)
)


;;; Build a vector out of a list of objects.  [Public]
(defun math-build-vector (&rest objs)
  (cons 'vec objs)
)
(fset 'calcFunc-vec (symbol-function 'math-build-vector))


;;; Build a constant vector or matrix.  [Public]
(defun math-make-vec (obj &rest dims)
  (math-make-vec-dimen obj dims)
)
(fset 'calcFunc-cvec (symbol-function 'math-make-vec))

(defun math-make-vec-dimen (obj dims)
  (if dims
      (if (natnump (car dims))
	  (if (or (cdr dims)
		  (not (math-numberp obj)))
	      (cons 'vec (copy-sequence
			  (make-list (car dims)
				     (math-make-vec-dimen obj (cdr dims)))))
	    (cons 'vec (make-list (car dims) obj)))
	(math-reject-arg (car dims) 'natnump))
    obj)
)


;;;; [calc-mat.el]

;;; Coerce row vector A to be a matrix.  [V V]
(defun math-row-matrix (a)
  (if (and (Math-vectorp a)
	   (not (math-matrixp a)))
      (list 'vec a)
    a)
)

;;; Coerce column vector A to be a matrix.  [V V]
(defun math-col-matrix (a)
  (if (and (Math-vectorp a)
	   (not (math-matrixp a)))
      (cons 'vec (mapcar (function (lambda (x) (list 'vec x))) (cdr a)))
    a)
)


;;;; [calc-ext.el]

(defun calc-binary-op-fancy (name func arg ident unary)
  (let ((n (prefix-numeric-value arg)))
    (cond ((> n 1)
	   (calc-enter-result n
			      name
			      (list 'calcFunc-reduce
				    (math-calcFunc-to-var func)
				    (cons 'vec (calc-top-list-n n)))))
	  ((= n 1)
	   (if unary
	       (calc-enter-result 1 name (list unary (calc-top-n 1)))))
	  ((= n 0)
	   (if ident
	       (calc-enter-result 0 name ident)
	     (error "Argument must be nonzero")))
	  (t
	   (let ((rhs (calc-top-n 1)))
	     (calc-enter-result (- 1 n)
				name
				(mapcar (function
					 (lambda (x)
					   (list func x rhs)))
					(calc-top-list-n (- n) 2)))))))
)

(defun calc-unary-op-fancy (name func arg)
  (let ((n (prefix-numeric-value arg)))
    (cond ((> n 0)
	   (calc-enter-result n
			      name
			      (mapcar (function
				       (lambda (x)
					 (list func x)))
				      (calc-top-list-n n))))
	  ((= n 0))
	  (t
	   (error "Argument must be positive"))))
)


;;;; [calc-vec.el]

;;; Apply a function elementwise to vectors A and B.  [O X O O] [Public]
(defun math-map-vec-2 (f a b)
  (if (math-vectorp a)
      (if (math-vectorp b)
	  (cons 'vec (math-map-vec-2-step f (cdr a) (cdr b)))
	(cons 'vec (math-map-vec-2-left f (cdr a) b)))
    (if (math-vectorp b)
	(cons 'vec (math-map-vec-2-right f a (cdr b)))
      (funcall f a b)))
)

(defun math-map-vec-2-step (f a b)   ; [L X L L]
  (cond
   ((null a) (if b (math-dimension-error)))
   ((null b) (math-dimension-error))
   (t (cons (funcall f (car a) (car b))
	    (math-map-vec-2-step f (cdr a) (cdr b)))))
)

(defun math-map-vec-2-left (f a b)   ; [L X L N]
  (and a
       (cons (funcall f (car a) b)
	     (math-map-vec-2-left f (cdr a) b)))
)

(defun math-map-vec-2-right (f a b)   ; [L X N L]
  (and b
       (cons (funcall f a (car b))
	     (math-map-vec-2-right f a (cdr b))))
)


;;;; [calc-map.el]

;;; Map a function over a vector symbolically. [Public]
(defun math-symb-map (f mode args)
  (let* ((func (math-var-to-calcFunc f))
	 (nargs (length args))
	 (ptrs (vconcat args))
	 (vflags (make-vector nargs nil))
	 (vec nil)
	 (i -1)
	 len cols obj expr)
    (if (eq mode 'rows)
	()
      (while (and (< (setq i (1+ i)) nargs)
		  (not (math-matrixp (aref ptrs i)))))
      (if (< i nargs)
	  (if (eq mode 'elems)
	      (setq func (list 'lambda '(&rest x)
			       (list 'math-symb-map
				     (list 'quote f) '(quote elems) 'x))
		    mode 'rows)
	    (while (< i nargs)
	      (if (math-matrixp (aref ptrs i))
		  (aset ptrs i (math-transpose (aref ptrs i))))
	      (setq i (1+ i))))
	(setq mode 'elems))
      (setq i -1))
    (while (< (setq i (1+ i)) nargs)
      (setq obj (aref ptrs i))
      (if (and (eq (car-safe obj) 'vec)
	       (or (eq mode 'elems)
		   (math-matrixp obj)))
	  (progn
	    (aset vflags i t)
	    (if len
		(or (= (length obj) len)
		    (math-dimension-error))
	      (setq len (length obj))))))
    (or len
	(if (= nargs 1)
	    (math-reject-arg (aref ptrs 0) 'vectorp)
	  (math-reject-arg "At least one argument must be a vector")))
    (while (> (setq len (1- len)) 0)
      (setq expr nil
	    i -1)
      (while (< (setq i (1+ i)) nargs)
	(if (aref vflags i)
	    (progn
	      (aset ptrs i (cdr (aref ptrs i)))
	      (setq expr (nconc expr (list (car (aref ptrs i))))))
	  (setq expr (nconc expr (list (aref ptrs i))))))
      (setq vec (cons (math-build-call func expr) vec)))
    (if (eq mode 'cols)
	(math-transpose (math-normalize (cons 'vec (nreverse vec))))
      (math-normalize (cons 'vec (nreverse vec)))))
)

(defun calcFunc-map (func &rest args)
  (math-symb-map func 'elems args)
)

(defun calcFunc-mapr (func &rest args)
  (math-symb-map func 'rows args)
)

(defun calcFunc-mapc (func &rest args)
  (math-symb-map func 'cols args)
)

(defun calcFunc-mapa (func arg)
  (if (math-matrixp arg)
      (math-symb-map func 'elems (cdr (math-transpose arg)))
    (math-symb-map func 'elems arg))
)

(defun calcFunc-mapd (func arg)
  (if (math-matrixp arg)
      (math-symb-map func 'elems (cdr arg))
    (math-symb-map func 'elems arg))
)


;;;; [calc-vec.el]

;;; "Reduce" a function over a vector (left-associatively).  [O X V] [Public]
(defun math-reduce-vec (f a)
  (if (math-vectorp a)
      (if (cdr a)
	  (math-reduce-vec-step f (car (cdr a)) (cdr (cdr a)))
	0)
    a)
)

(defun math-reduce-vec-step (f tot a)   ; [O X O L]
  (if a
      (math-reduce-vec-step f
			    (funcall f tot (car a))
			    (cdr a))
    tot)
)

;;; Reduce a function over the columns of matrix A.  [V X V] [Public]
(defun math-reduce-cols (f a)
  (if (math-matrixp a)
      (cons 'vec (math-reduce-cols-col-step f (cdr a) 1 (length (nth 1 a))))
    a)
)

(defun math-reduce-cols-col-step (f a col cols)
  (and (< col cols)
       (cons (math-reduce-cols-row-step f (nth col (car a)) col (cdr a))
	     (math-reduce-cols-col-step f a (1+ col) cols)))
)

(defun math-reduce-cols-row-step (f tot col a)
  (if a
      (math-reduce-cols-row-step f
				 (funcall f tot (nth col (car a)))
				 col
				 (cdr a))
    tot)
)


;;;; [calc-map.el]

;;; Reduce a function over a vector symbolically. [Public]
(defun calcFunc-reduce (func vec)
  (if (math-matrixp vec)
      (let (expr row)
	(setq func (math-var-to-calcFunc func))
	(or (math-vectorp vec)
	    (math-reject-arg vec 'vectorp))
	(while (setq vec (cdr vec))
	  (setq row (car vec))
	  (while (setq row (cdr row))
	    (setq expr (if expr
			   (math-build-call func (list expr (car row)))
			 (car row)))))
	(math-normalize expr))
    (calcFunc-reducer func vec))
)

(defun calcFunc-reducer (func vec)
  (setq func (math-var-to-calcFunc func))
  (or (math-vectorp vec)
      (math-reject-arg vec 'vectorp))
  (let ((expr (car (setq vec (cdr vec)))))
    (or expr
	(math-reject-arg vec "Vector is empty"))
    (while (setq vec (cdr vec))
      (setq expr (math-build-call func (list expr (car vec)))))
    (math-normalize expr))
)

(defun calcFunc-reducec (func vec)
  (if (math-matrixp vec)
      (calcFunc-reducer func (math-transpose vec))
    (calcFunc-reducer func vec))
)

(defun calcFunc-reducea (func vec)
  (if (math-matrixp vec)
      (cons 'vec
	    (mapcar (function (lambda (x) (calcFunc-reducer func x)))
		    (cdr vec)))
    (calcFunc-reducer func vec))
)

(defun calcFunc-reduced (func vec)
  (if (math-matrixp vec)
      (cons 'vec
	    (mapcar (function (lambda (x) (calcFunc-reducer func x)))
		    (cdr (math-transpose vec))))
    (calcFunc-reducer func vec))
)


;;;; [calc-mat.el]

;;; Multiply matrix vector element lists A and B.  [L L L]
(defun math-mul-mats (a b)
  (and a
       (cons (cons 'vec (math-mul-mat-row (car a) b))
	     (math-mul-mats (cdr a) b)))
)

(defun math-mul-mat-row (a b)   ; [L L L]
  (if (math-no-empty-rows b)
      (cons
       (math-reduce-vec 'math-add
			(math-map-vec-2 'math-mul
					a
					(cons 'vec (mapcar 'car b))))
       (math-mul-mat-row a (mapcar 'cdr b)))
    (if (math-list-all-nil b)
	nil
      (math-dimension-error)))
)

(defun math-no-empty-rows (a)   ; [P L]
  (or (null a)
      (and (consp (car a))
	   (math-no-empty-rows (cdr a))))
)

(defun math-list-all-nil (a)   ; [P L]
  (or (null a)
      (and (null (car a))
	   (math-list-all-nil (cdr a))))
)


;;;; [calc-vec.el]

;;; Return the number of elements in vector V.  [Public]
(defun math-vec-length (v)
  (if (math-vectorp v)
      (1- (length v))
    0)
)
(fset 'calcFunc-vlen (symbol-function 'math-vec-length))

;;; Get the Nth row of a matrix.
(defun math-mat-row (mat n)
  (elt mat n)
)

(defun calcFunc-mrow (mat n)   ; [Public]
  (and (integerp (setq n (math-check-integer n)))
       (> n 0)
       (math-vectorp mat)
       (nth n mat))
)

;;; Get the Nth column of a matrix.
(defun math-mat-col (mat n)
  (cons 'vec (mapcar (function (lambda (x) (elt x n))) (cdr mat)))
)

(defun calcFunc-mcol (mat n)   ; [Public]
  (and (integerp (setq n (math-check-integer n)))
       (> n 0)
       (math-vectorp mat)
       (if (math-matrixp mat)
	   (and (< n (length (nth 1 mat)))
		(math-mat-col mat n))
	 (nth n mat)))
)

;;; Remove the Nth row from a matrix.
(defun math-mat-less-row (mat n)
  (if (<= n 0)
      (cdr mat)
    (cons (car mat)
	  (math-mat-less-row (cdr mat) (1- n))))
)

(defun calcFunc-mrrow (mat n)   ; [Public]
  (and (integerp (setq n (math-check-integer n)))
       (> n 0)
       (< n (length mat))
       (math-mat-less-row mat n))
)

;;; Remove the Nth column from a matrix.
(defun math-mat-less-col (mat n)
  (cons 'vec (mapcar (function (lambda (x) (math-mat-less-row x n)))
		     (cdr mat)))
)

(defun calcFunc-mrcol (mat n)   ; [Public]
  (and (integerp (setq n (math-check-integer n)))
       (> n 0)
       (if (math-matrixp mat)
	   (and (< n (length (nth 1 mat)))
		(math-mat-less-col mat n))
	 (math-mat-less-row mat n)))
)

(defun math-get-diag (mat)   ; [Public]
  (if (math-square-matrixp mat)
      (cons 'vec (math-get-diag-step (cdr mat) 1))
    (calc-record-why 'math-square-matrixp mat)
    (list 'calcFunc-getdiag mat))
)
(fset 'calcFunc-getdiag (symbol-function 'math-get-diag))

(defun math-get-diag-step (row n)
  (and row
       (cons (nth n (car row))
	     (math-get-diag-step (cdr row) (1+ n))))
)

(defun math-transpose (mat)   ; [Public]
  (if (math-vectorp mat)
      (if (math-matrixp mat)
	  (cons 'vec
		(math-trn-step mat 1 (length (nth 1 mat))))
	(math-col-matrix mat))
    (and (math-numberp mat)
	 mat))
)
(fset 'calcFunc-trn (symbol-function 'math-transpose))

(defun calcFunc-ctrn (mat)
  (let ((trn (math-transpose mat)))
    (and trn
	 (math-conj trn)))
)

(defun math-trn-step (mat col cols)
  (and (< col cols)
       (cons (math-mat-col mat col)
	     (math-trn-step mat (1+ col) cols)))
)

(defun math-arrange-vector (vec cols)   ; [Public]
  (if (and (math-vectorp vec) (integerp cols))
      (let* ((flat (math-flatten-vector vec))
	     (mat (list 'vec))
	     next)
	(if (<= cols 0)
	    (nconc mat flat)
	  (while (>= (length flat) cols)
	    (setq next (nthcdr cols flat))
	    (setcdr (nthcdr (1- cols) flat) nil)
	    (setq mat (nconc mat (list (cons 'vec flat)))
		  flat next))
	  (if flat
	      (setq mat (nconc mat (list (cons 'vec flat)))))
	  mat)))
)
(fset 'calcFunc-arrange (symbol-function 'math-arrange-vector))

(defun math-flatten-vector (vec)   ; [L V]
  (if (math-vectorp vec)
      (apply 'append (mapcar 'math-flatten-vector (cdr vec)))
    (list vec))
)


;;; Copy a matrix.  [Public]
(defun math-copy-matrix (m)
  (if (math-vectorp (nth 1 m))
      (cons 'vec (mapcar 'copy-sequence (cdr m)))
    (copy-sequence m))
)

;;; Convert a scalar or vector into an NxN diagonal matrix.  [Public]
(defun math-diag-matrix (a &optional n)
  (and n (not (integerp n))
       (setq n (math-check-fixnum n)))
  (if (math-vectorp a)
      (if (and n (/= (length a) (1+ n)))
	  (list 'calcFunc-diag a n)
	(if (math-matrixp a)
	    (if (and n (/= (length (elt a 1)) (1+ n)))
		(list 'calcFunc-diag a n)
	      a)
	  (cons 'vec (math-diag-step (cdr a) 0 (1- (length a))))))
    (if n
	(cons 'vec (math-diag-step (make-list n a) 0 n))
      (list 'calcFunc-diag a)))
)
(fset 'calcFunc-diag (symbol-function 'math-diag-matrix))

(defun math-diag-step (a n m)
  (if (< n m)
      (cons (cons 'vec
		  (nconc (make-list n 0)
			 (cons (car a)
			       (make-list (1- (- m n)) 0))))
	    (math-diag-step (cdr a) (1+ n) m))
    nil)
)

;;; Create a vector of consecutive integers. [Public]
(defun math-vec-index (n)
  (and (not (integerp n))
       (setq n (math-check-fixnum n)))
  (or (natnump n) (math-reject-arg n 'natnump))
  (let ((vec nil))
    (while (> n 0)
      (setq vec (cons n vec)
	    n (1- n)))
    (cons 'vec vec))
)
(fset 'calcFunc-index (symbol-function 'math-vec-index))


;;; Compute the row and column norms of a vector or matrix.  [Public]
(defun math-rnorm (a)
  (if (and (Math-vectorp a)
	   (math-constp a))
      (if (math-matrixp a)
	  (math-reduce-vec 'math-max (math-map-vec 'math-cnorm a))
	(math-reduce-vec 'math-max (math-map-vec 'math-abs a)))
    (calc-record-why 'vectorp a)
    (list 'calcFunc-rnorm a))
)
(fset 'calcFunc-rnorm (symbol-function 'math-rnorm))

(defun math-cnorm (a)
  (if (and (Math-vectorp a)
	   (math-constp a))
      (if (math-matrixp a)
	  (math-reduce-vec 'math-max
			   (math-reduce-cols 'math-add-abs a))
	(math-reduce-vec 'math-add-abs a))
    (calc-record-why 'vectorp a)
    (list 'calcFunc-cnorm a))
)
(fset 'calcFunc-cnorm (symbol-function 'math-cnorm))

(defun math-add-abs (a b)
  (math-add (math-abs a) (math-abs b))
)


;;; Sort the elements of a vector into increasing order.
(defun math-sort-vector (vec)   ; [Public]
  (if (math-vectorp vec)
      (cons 'vec (sort (copy-sequence (cdr vec)) 'math-beforep))
    (math-reject-arg vec 'vectorp))
)
(fset 'calcFunc-sort (symbol-function 'math-sort-vector))

(defun math-rsort-vector (vec)   ; [Public]
  (if (math-vectorp vec)
      (cons 'vec (nreverse (sort (copy-sequence (cdr vec)) 'math-beforep)))
    (math-reject-arg vec 'vectorp))
)
(fset 'calcFunc-rsort (symbol-function 'math-rsort-vector))


;;; Compile a histogram of data from a vector.
(defun math-histogram (vec wts n)
  (or (Math-vectorp vec)
      (math-reject-arg vec 'vectorp))
  (if (Math-vectorp wts)
      (or (= (length vec) (length wts))
	  (math-dimension-error)))
  (or (natnump n)
      (math-reject-arg n 'natnump))
  (let ((res (make-vector n 0))
	(vp vec)
	(wvec (Math-vectorp wts))
	(wp wts)
	bin)
    (while (setq vp (cdr vp))
      (setq bin (car vp))
      (or (natnump bin)
	  (setq bin (math-floor bin)))
      (and (natnump bin)
	   (< bin n)
	   (aset res bin (math-add (aref res bin)
				   (if wvec (car (setq wp (cdr wp))) wts)))))
    (cons 'vec (append res nil)))
)
(fset 'calcFunc-histogram (symbol-function 'math-histogram))


;;;; [calc-mat.el]

(defun math-matrix-trace (mat)   ; [Public]
  (if (math-square-matrixp mat)
      (math-matrix-trace-step 2 (1- (length mat)) mat (nth 1 (nth 1 mat)))
    (math-reject-arg mat 'square-matrixp))
)
(fset 'calcFunc-tr (symbol-function 'math-matrix-trace))

(defun math-matrix-trace-step (n size mat sum)
  (if (<= n size)
      (math-matrix-trace-step (1+ n) size mat
			      (math-add sum (nth n (nth n mat))))
    sum)
)


;;; Matrix inverse and determinant.
(defun math-matrix-inv-raw (m)
  (let ((n (1- (length m))))
    (if (<= n 3)
	(let ((det (math-det-raw m)))
	  (and (not (math-zerop det))
	       (math-div
		(cond ((= n 1) 1)
		      ((= n 2)
		       (list 'vec
			     (list 'vec
				   (nth 2 (nth 2 m))
				   (math-neg (nth 2 (nth 1 m))))
			     (list 'vec
				   (math-neg (nth 1 (nth 2 m)))
				   (nth 1 (nth 1 m)))))
		      ((= n 3)
		       (list 'vec
			     (list 'vec
				   (math-sub (math-mul (nth 3 (nth 3 m))
						       (nth 2 (nth 2 m)))
					     (math-mul (nth 3 (nth 2 m))
						       (nth 2 (nth 3 m))))
				   (math-sub (math-mul (nth 3 (nth 1 m))
						       (nth 2 (nth 3 m)))
					     (math-mul (nth 3 (nth 3 m))
						       (nth 2 (nth 1 m))))
				   (math-sub (math-mul (nth 3 (nth 2 m))
						       (nth 2 (nth 1 m)))
					     (math-mul (nth 3 (nth 1 m))
						       (nth 2 (nth 2 m)))))
			     (list 'vec
				   (math-sub (math-mul (nth 3 (nth 2 m))
						       (nth 1 (nth 3 m)))
					     (math-mul (nth 3 (nth 3 m))
						       (nth 1 (nth 2 m))))
				   (math-sub (math-mul (nth 3 (nth 3 m))
						       (nth 1 (nth 1 m)))
					     (math-mul (nth 3 (nth 1 m))
						       (nth 1 (nth 3 m))))
				   (math-sub (math-mul (nth 3 (nth 1 m))
						       (nth 1 (nth 2 m)))
					     (math-mul (nth 3 (nth 2 m))
						       (nth 1 (nth 1 m)))))
			     (list 'vec
				   (math-sub (math-mul (nth 2 (nth 3 m))
						       (nth 1 (nth 2 m)))
					     (math-mul (nth 2 (nth 2 m))
						       (nth 1 (nth 3 m))))
				   (math-sub (math-mul (nth 2 (nth 1 m))
						       (nth 1 (nth 3 m)))
					     (math-mul (nth 2 (nth 3 m))
						       (nth 1 (nth 1 m))))
				   (math-sub (math-mul (nth 2 (nth 2 m))
						       (nth 1 (nth 1 m)))
					     (math-mul (nth 2 (nth 1 m))
						       (nth 1 (nth 2 m))))))))
		det)))
      (let ((lud (math-matrix-lud m)))
	(and lud
	     (math-lud-solve lud (math-diag-matrix 1 n))))))
)

(defun math-matrix-det (m)
  (if (math-square-matrixp m)
      (math-with-extra-prec 2 (math-det-raw m))
    (math-reject-arg m 'square-matrixp))
)
(fset 'calcFunc-det (symbol-function 'math-matrix-det))

(defun math-det-raw (m)
  (let ((n (1- (length m))))
    (cond ((= n 1)
	   (nth 1 (nth 1 m)))
	  ((= n 2)
	   (math-sub (math-mul (nth 1 (nth 1 m))
			       (nth 2 (nth 2 m)))
		     (math-mul (nth 2 (nth 1 m))
			       (nth 1 (nth 2 m)))))
	  ((= n 3)
	   (math-sub
	    (math-sub
	     (math-sub
	      (math-add
	       (math-add
		(math-mul (nth 1 (nth 1 m))
			  (math-mul (nth 2 (nth 2 m))
				    (nth 3 (nth 3 m))))
		(math-mul (nth 2 (nth 1 m))
			  (math-mul (nth 3 (nth 2 m))
				    (nth 1 (nth 3 m)))))
	       (math-mul (nth 3 (nth 1 m))
			 (math-mul (nth 1 (nth 2 m))
				   (nth 2 (nth 3 m)))))
	      (math-mul (nth 3 (nth 1 m))
			(math-mul (nth 2 (nth 2 m))
				  (nth 1 (nth 3 m)))))
	     (math-mul (nth 1 (nth 1 m))
		       (math-mul (nth 3 (nth 2 m))
				 (nth 2 (nth 3 m)))))
	    (math-mul (nth 2 (nth 1 m))
		      (math-mul (nth 1 (nth 2 m))
				(nth 3 (nth 3 m))))))
	  (t (let ((lud (math-matrix-lud m)))
	       (if lud
		   (let ((lu (car lud)))
		     (math-det-step n (nth 2 lud)))
		 0)))))
)

(defun math-det-step (n prod)
  (if (> n 0)
      (math-det-step (1- n) (math-mul prod (nth n (nth n lu))))
    prod)
)

;;; This returns a list (LU index d), or NIL if not possible.
;;; Argument M must be a square matrix.
(defun math-matrix-lud (m)
  (let ((old (assoc m math-lud-cache))
	(context (list calc-internal-prec calc-prefer-frac)))
    (if (and old (equal (nth 1 old) context))
	(cdr (cdr old))
      (let* ((lud (catch 'singular (math-do-matrix-lud m)))
	     (entry (cons context lud)))
	(if old
	    (setcdr old entry)
	  (setq math-lud-cache (cons (cons m entry) math-lud-cache)))
	lud)))
)
(defvar math-lud-cache nil)

;;; Numerical Recipes section 2.3; implicit pivoting omitted.
(defun math-do-matrix-lud (m)
  (let* ((lu (math-copy-matrix m))
	 (n (1- (length lu)))
	 i (j 1) k imax sum big
	 (d 1) (index nil))
    (while (<= j n)
      (setq i 1
	    big 0
	    imax j)
      (while (< i j)
	(math-working "LUD step" (format "%d/%d" j i))
	(setq sum (nth j (nth i lu))
	      k 1)
	(while (< k i)
	  (setq sum (math-sub sum (math-mul (nth k (nth i lu))
					    (nth j (nth k lu))))
		k (1+ k)))
	(setcar (nthcdr j (nth i lu)) sum)
	(setq i (1+ i)))
      (while (<= i n)
	(math-working "LUD step" (format "%d/%d" j i))
	(setq sum (nth j (nth i lu))
	      k 1)
	(while (< k j)
	  (setq sum (math-sub sum (math-mul (nth k (nth i lu))
					    (nth j (nth k lu))))
		k (1+ k)))
	(setcar (nthcdr j (nth i lu)) sum)
	(let ((dum (math-abs-approx sum)))
	  (if (Math-lessp big dum)
	      (setq big dum
		    imax i)))
	(setq i (1+ i)))
      (if (> imax j)
	  (setq lu (math-swap-rows lu j imax)
		d (- d)))
      (setq index (cons imax index))
      (let ((pivot (nth j (nth j lu))))
	(if (math-zerop pivot)
	    (throw 'singular nil)
	  (setq i j)
	  (while (<= (setq i (1+ i)) n)
	    (setcar (nthcdr j (nth i lu))
		    (math-div (nth j (nth i lu)) pivot)))))
      (setq j (1+ j)))
    (list lu (nreverse index) d))
)

(defun math-swap-rows (m r1 r2)
  (or (= r1 r2)
      (let* ((r1prev (nthcdr (1- r1) m))
	     (row1 (cdr r1prev))
	     (r2prev (nthcdr (1- r2) m))
	     (row2 (cdr r2prev))
	     (r2next (cdr row2)))
	(setcdr r2prev row1)
	(setcdr r1prev row2)
	(setcdr row2 (cdr row1))
	(setcdr row1 r2next)))
  m
)

(defun math-abs-approx (a)
  (cond ((Math-negp a)
	 (math-neg a))
	((Math-anglep a)
	 a)
	((eq (car a) 'cplx)
	 (math-add (math-abs (nth 1 a)) (math-abs (nth 2 a))))
	((eq (car a) 'polar)
	 (nth 1 a))
	((eq (car a) 'sdev)
	 (math-abs (nth 1 a)))
	((eq (car a) 'intv)
	 (math-max (math-abs (nth 2 a)) (math-abs (nth 3 a))))
	((eq (car a) 'vec)
	 (math-cnorm a))
	((eq (car a) 'calcFunc-abs)
	 (car a))
	(t a))
)

(defun math-lud-solve (lud b)
  (and lud
	 (let* ((x (math-copy-matrix b))
		(n (1- (length x)))
		(m (1- (length (nth 1 x))))
		(lu (car lud))
		(col 1)
		i j ip ii index sum)
	   (while (<= col m)
	     (math-working "LUD solver step" col)
	     (setq i 1
		   ii nil
		   index (nth 1 lud))
	     (while (<= i n)
	       (setq ip (car index)
		     index (cdr index)
		     sum (nth col (nth ip x)))
	       (setcar (nthcdr col (nth ip x)) (nth col (nth i x)))
	       (if (null ii)
		   (or (math-zerop sum)
		       (setq ii i))
		 (setq j ii)
		 (while (< j i)
		   (setq sum (math-sub sum (math-mul (nth j (nth i lu))
						     (nth col (nth j x))))
			 j (1+ j))))
	       (setcar (nthcdr col (nth i x)) sum)
	       (setq i (1+ i)))
	     (while (>= (setq i (1- i)) 1)
	       (setq sum (nth col (nth i x))
		     j i)
	       (while (<= (setq j (1+ j)) n)
		 (setq sum (math-sub sum (math-mul (nth j (nth i lu))
						   (nth col (nth j x))))))
	       (setcar (nthcdr col (nth i x))
		       (math-div sum (nth i (nth i lu)))))
	     (setq col (1+ col)))
	   x))
)

(defun calcFunc-lud (m)
  (if (math-square-matrixp m)
      (or (math-with-extra-prec 2
	    (let ((lud (math-matrix-lud m)))
	      (and lud
		   (let* ((lmat (math-copy-matrix (car lud)))
			  (umat (math-copy-matrix (car lud)))
			  (n (1- (length (car lud))))
			  (perm (math-diag-matrix 1 n))
			  i (j 1))
		     (while (<= j n)
		       (setq i 1)
		       (while (< i j)
			 (setcar (nthcdr j (nth i lmat)) 0)
			 (setq i (1+ i)))
		       (setcar (nthcdr j (nth j lmat)) 1)
		       (while (<= (setq i (1+ i)) n)
			 (setcar (nthcdr j (nth i umat)) 0))
		       (setq j (1+ j)))
		     (while (>= (setq j (1- j)) 1)
		       (let ((pos (nth (1- j) (nth 1 lud))))
			 (or (= pos j)
			     (setq perm (math-swap-rows perm j pos)))))
		     (list 'vec perm lmat umat)))))
	  (math-reject-arg m "Singular matrix"))
    (math-reject-arg m 'square-matrixp))
)

;;;; [calc-vec.el]

;;; Compute a right-handed vector cross product.  [O O O] [Public]
(defun math-cross (a b)
  (if (and (eq (car-safe a) 'vec)
	   (= (length a) 4))
      (if (and (eq (car-safe b) 'vec)
	       (= (length b) 4))
	  (list 'vec
		(math-sub (math-mul (nth 2 a) (nth 3 b))
			  (math-mul (nth 3 a) (nth 2 b)))
		(math-sub (math-mul (nth 3 a) (nth 1 b))
			  (math-mul (nth 1 a) (nth 3 b)))
		(math-sub (math-mul (nth 1 a) (nth 2 b))
			  (math-mul (nth 2 a) (nth 1 b))))
	(math-reject-arg b "Three-vector expected"))
    (math-reject-arg a "Three-vector expected"))
)
(fset 'calcFunc-cross (symbol-function 'math-cross))




;;;; [calc-forms.el]

;;;; Hours-minutes-seconds forms.

(defun math-normalize-hms (a)
  (let ((h (math-normalize (nth 1 a)))
	(m (math-normalize (nth 2 a)))
	(s (let ((calc-internal-prec (max (- calc-internal-prec 4) 3)))
	     (math-normalize (nth 3 a)))))
    (if (math-negp h)
	(progn
	  (if (math-posp s)
	      (setq s (math-add s -60)
		    m (math-add m 1)))
	  (if (math-posp m)
	      (setq m (math-add m -60)
		    h (math-add h 1)))
	  (if (not (math-lessp -60 s))
	      (setq s (math-add s 60)
		    m (math-add m -1)))
	  (if (not (math-lessp -60 m))
	      (setq m (math-add m 60)
		    h (math-add h -1))))
      (if (math-negp s)
	  (setq s (math-add s 60)
		m (math-add m -1)))
      (if (math-negp m)
	  (setq m (math-add m 60)
		h (math-add h -1)))
      (if (not (math-lessp s 60))
	  (setq s (math-add s -60)
		m (math-add m 1)))
      (if (not (math-lessp m 60))
	  (setq m (math-add m -60)
		h (math-add h 1))))
    (if (and (eq (car-safe s) 'float)
	     (<= (+ (math-numdigs (nth 1 s)) (nth 2 s))
		 (- 2 calc-internal-prec)))
	(setq s 0))
    (list 'hms h m s))
)

;;; Convert A from ANG or current angular mode to HMS format.
(defun math-to-hms (a &optional ang)   ; [X R] [Public]
  (cond ((eq (car-safe a) 'hms) a)
	((eq (car-safe a) 'sdev)
	 (math-make-sdev (math-to-hms (nth 1 a))
			 (math-to-hms (nth 2 a))))
	((not (Math-numberp a))
	 (list 'calcFunc-hms a))
	((math-negp a)
	 (math-neg (math-to-hms (math-neg a) ang)))
	((eq (or ang calc-angle-mode) 'rad)
	 (math-to-hms (math-div a (math-pi-over-180)) 'deg))
	((memq (car-safe a) '(cplx polar)) a)
	(t
	 ;(setq a (let ((calc-internal-prec (max (1- calc-internal-prec) 3)))
	 ;	    (math-normalize a)))
	 (math-normalize
	  (let* ((b (math-mul a 3600))
		 (hm (math-trunc (math-div b 60)))
		 (hmd (math-idivmod hm 60)))
	    (list 'hms
		  (car hmd)
		  (cdr hmd)
		  (math-sub b (math-mul hm 60)))))))
)
(fset 'calcFunc-hms (symbol-function 'math-to-hms))

;;; Convert A from HMS format to ANG or current angular mode.
(defun math-from-hms (a &optional ang)   ; [R X] [Public]
  (cond ((not (eq (car-safe a) 'hms))
	 (if (Math-numberp a)
	     a
	   (if (eq (car-safe a) 'sdev)
	     (math-make-sdev (math-from-hms (nth 1 a))
			     (math-from-hms (nth 2 a)))
	     (if (eq (or ang calc-angle-mode) 'rad)
		 (list '>rad a)
	       (list '>deg a)))))
	((math-negp a)
	 (math-neg (math-from-hms (math-neg a) ang)))
	((eq (or ang calc-angle-mode) 'rad)
	 (math-mul (math-from-hms a 'deg) (math-pi-over-180)))
	((memq (car-safe a) '(cplx polar)) a)
	(t
	 (math-add (math-div (math-add (math-div (nth 3 a)
						 '(float 6 1))
				       (nth 2 a))
			     60)
		   (nth 1 a))))
)



;;;; [calc-cplx.el]

;;;; Complex numbers.

(defun math-normalize-polar (a)
  (let ((r (math-normalize (nth 1 a)))
	(th (math-normalize (nth 2 a))))
    (cond ((math-zerop r)
	   '(polar 0 0))
	  ((or (math-zerop th))
	   r)
	  ((and (not (eq calc-angle-mode 'rad))
		(or (equal th '(float 18 1))
		    (equal th 180)))
	   (math-neg r))
	  ((math-negp r)
	   (math-neg (list 'polar (math-neg r) th)))
	  (t
	   (list 'polar r th))))
)


;;; Coerce A to be complex (rectangular form).  [c N]
(defun math-complex (a)
  (cond ((eq (car-safe a) 'cplx) a)
	((eq (car-safe a) 'polar)
	 (if (math-zerop (nth 1 a))
	     (nth 1 a)
	   (let ((sc (math-sin-cos (nth 2 a))))
	     (list 'cplx
		   (math-mul (nth 1 a) (nth 1 sc))
		   (math-mul (nth 1 a) (nth 2 sc))))))
	(t (list 'cplx a 0)))
)

;;; Coerce A to be complex (polar form).  [c N]
(defun math-polar (a)
  (cond ((eq (car-safe a) 'polar) a)
	((math-zerop a) '(polar 0 0))
	(t
	 (list 'polar
	       (math-abs a)
	       (math-cplx-arg a))))
)

;;; Multiply A by the imaginary constant i.  [N N] [Public]
(defun math-imaginary (a)
  (if (and (Math-objvecp a)
	   (not calc-symbolic-mode))
      (math-mul a
		(if (or (eq (car-safe a) 'polar)
			(and (not (eq (car-safe a) 'cplx))
			     (eq calc-complex-mode 'polar)))
		    (list 'polar 1 (math-quarter-circle nil))
		  '(cplx 0 1)))
    (math-mul a '(var i var-i)))
)



;;;; [calc-forms.el]

;;;; Error forms.

;;; Build a standard deviation form.  [X X X]
(defun math-make-sdev (x sigma)
  (if (memq (car-safe x) '(cplx polar mod sdev intv vec))
      (math-reject-arg x 'realp))
  (if (memq (car-safe sigma) '(cplx polar mod sdev intv vec))
      (math-reject-arg sigma 'realp))
  (if (math-negp sigma)
      (list 'sdev x (math-neg sigma))
    (if (and (math-zerop sigma) (Math-scalarp x))
	x
      (list 'sdev x sigma)))
)
(defun calcFunc-sdev (x sigma)
  (math-make-sdev x sigma)
)



;;;; Modulo forms.

(defun math-normalize-mod (a)
  (let ((n (math-normalize (nth 1 a)))
	(m (math-normalize (nth 2 a))))
    (if (and (math-anglep n) (math-anglep m) (math-posp m))
	(math-make-mod n m)
      (if (math-anglep n)
	  (if (math-anglep m)
	      (calc-record-why "Modulus must be positive" m)
	    (calc-record-why "Modulus must be real" m))
	(calc-record-why "Value must be real" n))
      (list 'calcFunc-makemod n m)))
)

;;; Build a modulo form.  [N R R]
(defun math-make-mod (n m)
  (setq calc-previous-modulo m)
  (and n
       (if (not (and (Math-anglep n) (Math-anglep m)))
	   (math-reject-arg n 'anglep)
	 (if (or (Math-negp n)
		 (not (Math-lessp n m)))
	     (list 'mod (math-mod n m) m)
	   (list 'mod n m))))
)
(defun calcFunc-makemod (n m)
  (math-make-mod n m)
)



;;;; Interval forms.

;;; Build an interval form.  [X I X X]
(defun math-make-intv (mask lo hi)
  (if (memq (car-safe lo) '(cplx polar mod sdev intv vec))
      (math-reject-arg lo 'realp))
  (if (memq (car-safe hi) '(cplx polar mod sdev intv vec))
      (math-reject-arg hi 'realp))
  (if (and (Math-realp lo) (Math-realp hi))
      (let ((cmp (math-compare lo hi)))
	(if (= cmp 0)
	    (if (= mask 3)
		lo
	      (list 'intv mask lo hi))
	  (if (> cmp 0)
	      (if (= mask 3)
		  (list 'intv 2 lo lo)
		(list 'intv mask lo lo))
	    (list 'intv mask lo hi))))
    (list 'intv mask lo hi))
)

(defun math-sort-intv (mask lo hi)
  (if (Math-lessp hi lo)
      (math-make-intv (aref [0 2 1 3] mask) hi lo)
    (math-make-intv mask lo hi))
)



;;;; [calc-arith.el]

;;;; Arithmetic.

(defun math-neg-fancy (a)
  (cond ((eq (car a) 'polar)
	 (list 'polar
	       (nth 1 a)
	       (if (math-posp (nth 2 a))
		   (math-sub (nth 2 a) (math-half-circle nil))
		 (math-add (nth 2 a) (math-half-circle nil)))))
	((eq (car a) 'mod)
	 (if (math-zerop (nth 1 a))
	     a
	   (list 'mod (math-sub (nth 2 a) (nth 1 a)) (nth 2 a))))
	((eq (car a) 'sdev)
	 (list 'sdev (math-neg (nth 1 a)) (nth 2 a)))
	((eq (car a) 'intv)
	 (math-make-intv (aref [0 2 1 3] (nth 1 a))
			 (math-neg (nth 3 a))
			 (math-neg (nth 2 a))))
	((eq (car a) '-)
	 (math-sub (nth 2 a) (nth 1 a)))
	((and (memq (car a) '(* /))
	      (math-looks-negp (nth 1 a)))
	 (list (car a) (math-neg (nth 1 a)) (nth 2 a)))
	((and (memq (car a) '(* /))
	      (math-looks-negp (nth 2 a)))
	 (list (car a) (nth 1 a) (math-neg (nth 2 a))))
	((and (memq (car a) '(* /))
	      (or (math-numberp (nth 1 a))
		  (and (eq (car (nth 1 a)) '*)
		       (math-numberp (nth 1 (nth 1 a))))))
	 (list (car a) (math-neg (nth 1 a)) (nth 2 a)))
	((and (eq (car a) '/)
	      (or (math-numberp (nth 2 a))
		  (and (eq (car (nth 2 a)) '*)
		       (math-numberp (nth 1 (nth 2 a))))))
	 (list (car a) (nth 1 a) (math-neg (nth 2 a))))
	((eq (car a) 'neg)
	 (nth 1 a))
	(t (list 'neg a)))
)

(defun math-neg-float (a)
  (list 'float (Math-integer-neg (nth 1 a)) (nth 2 a))
)

(defun math-add-objects-fancy (a b)
  (cond ((and (Math-numberp a) (Math-numberp b))
	 (setq aa (math-complex a)
	       bb (math-complex b))
	 (math-normalize
	  (let ((res (list 'cplx
			   (math-add (nth 1 aa) (nth 1 bb))
			   (math-add (nth 2 aa) (nth 2 bb)))))
	    (if (math-want-polar a b)
		(math-polar res)
	      res))))
	((or (Math-vectorp a) (Math-vectorp b))
	 (math-map-vec-2 'math-add a b))
	((eq (car-safe a) 'sdev)
	 (if (eq (car-safe b) 'sdev)
	     (math-make-sdev (math-add (nth 1 a) (nth 1 b))
			     (math-hypot (nth 2 a) (nth 2 b)))
	   (and (or (Math-anglep b)
		    (not (Math-objvecp b)))
		(math-make-sdev (math-add (nth 1 a) b)
				(nth 2 a)))))
	((and (eq (car-safe b) 'sdev)
	      (or (Math-anglep a)
		  (not (Math-objvecp a))))
	 (math-make-sdev (math-add a (nth 1 b))
			 (nth 2 b)))
	((eq (car-safe a) 'intv)
	 (if (eq (car-safe b) 'intv)
	     (math-make-intv (logand (nth 1 a) (nth 1 b))
			     (math-add (nth 2 a) (nth 2 b))
			     (math-add (nth 3 a) (nth 3 b)))
	   (and (or (Math-anglep b)
		    (not (Math-objvecp b)))
		(math-make-intv (nth 1 a)
				(math-add (nth 2 a) b)
				(math-add (nth 3 a) b)))))
	((and (eq (car-safe b) 'intv)
	      (or (Math-anglep a)
		  (not (Math-objvecp a))))
	 (math-make-intv (nth 1 b)
			 (math-add a (nth 2 b))
			 (math-add a (nth 3 b))))
	((and (eq (car-safe a) 'mod)
	      (eq (car-safe b) 'mod)
	      (equal (nth 2 a) (nth 2 b)))
	 (math-make-mod (math-add (nth 1 a) (nth 1 b)) (nth 2 a)))
	((and (eq (car-safe a) 'mod)
	      (Math-anglep b))
	 (math-make-mod (math-add (nth 1 a) b) (nth 2 a)))
	((and (eq (car-safe b) 'mod)
	      (Math-anglep a))
	 (math-make-mod (math-add a (nth 1 b)) (nth 2 b)))
	((and (or (eq (car-safe a) 'hms) (eq (car-safe b) 'hms))
	      (and (Math-anglep a) (Math-anglep b)))
	 (or (eq (car-safe a) 'hms) (setq a (math-to-hms a)))
	 (or (eq (car-safe b) 'hms) (setq b (math-to-hms b)))
	 (math-normalize
	  (if (math-negp a)
	      (math-neg (math-add (math-neg a) (math-neg b)))
	    (if (math-negp b)
		(let* ((s (math-add (nth 3 a) (nth 3 b)))
		       (m (math-add (nth 2 a) (nth 2 b)))
		       (h (math-add (nth 1 a) (nth 1 b))))
		  (if (math-negp s)
		      (setq s (math-add s 60)
			    m (math-add m -1)))
		  (if (math-negp m)
		      (setq m (math-add m 60)
			    h (math-add h -1)))
		  (if (math-negp h)
		      (math-add b a)
		    (list 'hms h m s)))
	      (let* ((s (math-add (nth 3 a) (nth 3 b)))
		     (m (math-add (nth 2 a) (nth 2 b)))
		     (h (math-add (nth 1 a) (nth 1 b))))
		(list 'hms h m s))))))
	(t (calc-record-why "Incompatible arguments" a b)))
)

(defun math-add-symb-fancy (a b)
  (or (and (eq (car-safe b) '+)
	   (math-add (math-add a (nth 1 b))
		     (nth 2 b)))
      (and (eq (car-safe b) '-)
	   (math-sub (math-add a (nth 1 b))
		     (nth 2 b)))
      (and (eq (car-safe b) 'neg)
	   (eq (car-safe (nth 1 b)) '+)
	   (math-sub (math-sub a (nth 1 (nth 1 b)))
		     (nth 2 (nth 1 b))))
      (cond
       ((eq (car-safe a) '+)
	(let ((temp (math-combine-sum (nth 2 a) b nil nil t)))
	  (and temp
	       (math-add (nth 1 a) temp))))
       ((eq (car-safe a) '-)
	(let ((temp (math-combine-sum (nth 2 a) b t nil t)))
	  (and temp
	       (math-add (nth 1 a) temp))))
       ((and (Math-objectp a) (Math-objectp b))
	nil)
       (t
	(math-combine-sum a b nil nil nil)))
      (and (Math-looks-negp b)
	   (list '- a (math-neg b)))
      (and (Math-looks-negp a)
	   (list '- b (math-neg a)))
      (list '+ a b))
)


(defun math-mul-objects-fancy (a b)
  (cond ((and (Math-numberp a) (Math-numberp b))
	 (math-normalize
	  (if (math-want-polar a b)
	      (let ((a (math-polar a))
		    (b (math-polar b)))
		(list 'polar
		      (math-mul (nth 1 a) (nth 1 b))
		      (math-fix-circular (math-add (nth 2 a) (nth 2 b)))))
	    (setq a (math-complex a)
		  b (math-complex b))
	    (list 'cplx
		  (math-sub (math-mul (nth 1 a) (nth 1 b))
			    (math-mul (nth 2 a) (nth 2 b)))
		  (math-add (math-mul (nth 1 a) (nth 2 b))
			    (math-mul (nth 2 a) (nth 1 b)))))))
	((Math-vectorp a)
	 (if (Math-vectorp b)
	     (if (math-matrixp a)
		 (if (math-matrixp b)
		     (cons 'vec (math-mul-mats (cdr a)
					       (mapcar 'cdr
						       (cdr b))))
		   (math-mat-col
		    (cons 'vec
			  (if (= (length (nth 1 a)) 2)
			      (math-mul-mats (cdr a)
					     (mapcar 'cdr
						     (cdr (math-row-matrix
							   b))))
			    (math-mul-mats (cdr a)
					   (mapcar 'cdr
						   (cdr (math-col-matrix
							 b))))))
		    1))
	       (if (math-matrixp b)
		   (cons 'vec (math-mul-mat-row a (mapcar 'cdr (cdr b))))
		 (car (math-mul-mat-row a
					(mapcar 'cdr
						(cdr (math-col-matrix
						      b)))))))
	   (math-map-vec-2 'math-mul a b)))
	((Math-vectorp b)
	 (math-map-vec-2 'math-mul a b))
	((eq (car-safe a) 'sdev)
	 (if (eq (car-safe b) 'sdev)
	     (math-make-sdev (math-mul (nth 1 a) (nth 1 b))
			     (math-hypot (math-mul (nth 2 a) (nth 1 b))
					 (math-mul (nth 2 b) (nth 1 a))))
	   (and (or (Math-anglep b)
		    (not (Math-objvecp b)))
		(math-make-sdev (math-mul (nth 1 a) b)
				(math-abs (math-mul (nth 2 a) b))))))
	((and (eq (car-safe b) 'sdev)
	      (or (Math-anglep a)
		  (not (Math-objvecp a))))
	 (math-make-sdev (math-mul a (nth 1 b))
			 (math-abs (math-mul a (nth 2 b)))))
	((and (eq (car-safe a) 'intv) (Math-anglep b))
	 (if (Math-negp b)
	     (math-neg (math-mul a (math-neg b)))
	   (math-make-intv (nth 1 a)
			   (math-mul (nth 2 a) b)
			   (math-mul (nth 3 a) b))))
	((and (eq (car-safe b) 'intv) (Math-anglep a))
	 (math-mul b a))
	((and (eq (car-safe a) 'intv) (math-constp a)
	      (eq (car-safe b) 'intv) (math-constp b))
	 (let ((lo (math-mul a (nth 2 b)))
	       (hi (math-mul a (nth 3 b))))
	   (and (Math-anglep lo)
		(setq lo (list 'intv (if (memq (nth 1 b) '(2 3)) 3 0) lo lo)))
	   (and (Math-anglep hi)
		(setq hi (list 'intv (if (memq (nth 1 b) '(1 3)) 3 0) hi hi)))
	   (math-combine-intervals (nth 2 lo) (and (memq (nth 1 b) '(2 3))
						   (memq (nth 1 lo) '(2 3)))
				   (nth 3 lo) (and (memq (nth 1 b) '(2 3))
						   (memq (nth 1 lo) '(1 3)))
				   (nth 2 hi) (and (memq (nth 1 b) '(1 3))
						   (memq (nth 1 hi) '(2 3)))
				   (nth 3 hi) (and (memq (nth 1 b) '(1 3))
						   (memq (nth 1 hi) '(1 3))))))
	((and (eq (car-safe a) 'mod)
	      (eq (car-safe b) 'mod)
	      (equal (nth 2 a) (nth 2 b)))
	 (math-make-mod (math-mul (nth 1 a) (nth 1 b)) (nth 2 a)))
	((and (eq (car-safe a) 'mod)
	      (Math-anglep b))
	 (math-make-mod (math-mul (nth 1 a) b) (nth 2 a)))
	((and (eq (car-safe b) 'mod)
	      (Math-anglep a))
	 (math-make-mod (math-mul a (nth 1 b)) (nth 2 b)))
	((and (eq (car-safe a) 'hms) (Math-realp b))
	 (math-with-extra-prec 2
	   (math-to-hms (math-mul (math-from-hms a 'deg) b) 'deg)))
	((and (eq (car-safe b) 'hms) (Math-realp a))
	 (math-mul b a))
	(t (calc-record-why "Incompatible arguments" a b)))
)

;;; Fast function to multiply floating-point numbers.
(defun math-mul-float (a b)   ; [F F F]
  (math-make-float (math-mul (nth 1 a) (nth 1 b))
		   (+ (nth 2 a) (nth 2 b)))
)

(defun math-sqr-float (a)   ; [F F]
  (math-make-float (math-mul (nth 1 a) (nth 1 a))
		   (+ (nth 2 a) (nth 2 a)))
)

;;;; [calc-forms.el]

(defun math-combine-intervals (a am b bm c cm d dm)
  (let (res)
    (if (= (setq res (math-compare a c)) 1)
	(setq a c am cm)
      (if (= res 0)
	  (setq am (or am cm))))
    (if (= (setq res (math-compare b d)) -1)
	(setq b d bm dm)
      (if (= res 0)
	  (setq bm (or bm dm))))
    (math-make-intv (+ (if am 2 0) (if bm 1 0)) a b))
)

;;;; [calc-arith.el]

(defun math-mul-symb-fancy (a b)
  (or (and (Math-equal-int a 1)
	   b)
      (and (Math-equal-int a -1)
	   (math-neg b))
      (and (Math-numberp b)
	   (math-mul b a))
      (and (eq (car-safe a) 'neg)
	   (math-neg (math-mul (nth 1 a) b)))
      (and (eq (car-safe b) 'neg)
	   (math-neg (math-mul a (nth 1 b))))
      (and (eq (car-safe a) '*)
	   (math-mul (nth 1 a)
		     (math-mul (nth 2 a) b)))
      (and (eq (car-safe a) '^)
	   (Math-looks-negp (nth 2 a))
	   (not (and (eq (car-safe b) '^) (Math-looks-negp (nth 2 b))))
	   (math-div b (math-normalize
			(list '^ (nth 1 a) (math-neg (nth 2 a))))))
      (and (eq (car-safe b) '^)
	   (Math-looks-negp (nth 2 b))
	   (not (and (eq (car-safe a) '^) (Math-looks-negp (nth 2 a))))
	   (math-div a (math-normalize
			(list '^ (nth 1 b) (math-neg (nth 2 b))))))
      (and (eq (car-safe a) '/)
	   (math-div (math-mul (nth 1 a) b) (nth 2 a)))
      (and (eq (car-safe b) '/)
	   (math-div (math-mul a (nth 1 b)) (nth 2 b)))
      (and (eq (car-safe b) '+)
	   (Math-numberp a)
	   (or (Math-numberp (nth 1 b))
	       (Math-numberp (nth 2 b)))
	   (math-add (math-mul a (nth 1 b))
		     (math-mul a (nth 2 b))))
      (and (eq (car-safe b) '-)
	   (Math-numberp a)
	   (or (Math-numberp (nth 1 b))
	       (Math-numberp (nth 2 b)))
	   (math-sub (math-mul a (nth 1 b))
		     (math-mul a (nth 2 b))))
      (and (eq (car-safe b) '*)
	   (Math-numberp (nth 1 b))
	   (not (Math-numberp a))
	   (math-mul (nth 1 b) (math-mul a (nth 2 b))))
      (and (or t  ; this seems more reasonable...
	       (eq (car-safe a) '-)
	       (math-looks-negp a))
	   (math-looks-negp b)
	   (math-mul (math-neg a) (math-neg b)))
      (and (eq (car-safe b) '-)
	   (math-looks-negp a)
	   (math-mul (math-neg a) (math-neg b)))
      (cond
       ((eq (car-safe b) '*)
	(let ((temp (math-combine-prod a (nth 1 b) nil nil t)))
	  (and temp
	       (math-mul temp (nth 2 b)))))
       (t
	(math-combine-prod a b nil nil nil)))
      (list '* a b))
)

;;;; [calc-cplx.el]

(defun math-want-polar (a b)
  (cond ((eq (car-safe a) 'polar)
	 (if (eq (car-safe b) 'cplx)
	     (eq car-complex-mode 'polar)
	   t))
	((eq (car-safe a) 'cplx)
	 (if (eq (car-safe b) 'polar)
	     (eq car-complex-mode 'polar)
	   nil))
	((eq (car-safe b) 'polar)
	 t)
	((eq (car-safe b) 'cplx)
	 nil)
	(t (eq (car-complex-mode 'polar))))
)

;;; Force A to be in the (-pi,pi] or (-180,180] range.
(defun math-fix-circular (a &optional dir)   ; [R R]
  (cond ((eq calc-angle-mode 'deg)
	 (cond ((and (Math-lessp '(float 18 1) a) (not (eq dir 1)))
		(math-fix-circular (math-add a '(float -36 1)) -1))
	       ((or (Math-lessp '(float -18 1) a) (eq dir -1))
		a)
	       (t
		(math-fix-circular (math-add a '(float 36 1)) 1))))
	((eq calc-angle-mode 'hms)
	 (cond ((and (Math-lessp 180 (nth 1 a)) (not (eq dir 1)))
		(math-fix-circular (math-add a '(float -36 1)) -1))
	       ((or (Math-lessp -180 (nth 1 a)) (eq dir -1))
		a)
	       (t
		(math-fix-circular (math-add a '(float 36 1)) 1))))
	(t
	 (cond ((and (Math-lessp (math-pi) a) (not (eq dir 1)))
		(math-fix-circular (math-sub a (math-two-pi)) -1))
	       ((or (Math-lessp (math-neg (math-pi)) a) (eq dir -1))
		a)
	       (t
		(math-fix-circular (math-add a (math-two-pi)) 1)))))
)

;;;; [calc-arith.el]

(defun math-div-objects-fancy (a b)
  (cond ((and (Math-numberp a) (Math-numberp b))
	 (math-normalize
	  (cond ((math-want-polar a b)
		 (let ((a (math-polar a))
		       (b (math-polar b)))
		   (list 'polar
			 (math-div (nth 1 a) (nth 1 b))
			 (math-fix-circular (math-sub (nth 2 a)
						      (nth 2 b))))))
		((Math-realp b)
		 (setq a (math-complex a))
		 (list 'cplx (math-div (nth 1 a) b)
		       (math-div (nth 2 a) b)))
		(t
		 (setq a (math-complex a)
		       b (math-complex b))
		 (math-div
		  (list 'cplx
			(math-add (math-mul (nth 1 a) (nth 1 b))
				  (math-mul (nth 2 a) (nth 2 b)))
			(math-sub (math-mul (nth 2 a) (nth 1 b))
				  (math-mul (nth 1 a) (nth 2 b))))
		  (math-add (math-sqr (nth 1 b))
			    (math-sqr (nth 2 b))))))))
	((math-matrixp b)
	 (if (math-square-matrixp b)
	     (let ((n1 (length b)))
	       (if (Math-vectorp a)
		   (if (math-matrixp a)
		       (if (= (length a) n1)
			   (math-lud-solve (math-matrix-lud b) a)
			 (if (= (length (nth 1 a)) n1)
			     (math-transpose
			      (math-lud-solve (math-matrix-lud
					       (math-transpose b))
					      (math-transpose a)))
			   (math-dimension-error)))
		     (if (= (length a) n1)
			 (math-mat-col (math-lud-solve (math-matrix-lud b)
						       (math-col-matrix a))
				       1)
		       (math-dimension-error)))
		 (if (Math-equal-int a 1)
		     (math-inv b)
		   (math-mul a (math-inv b)))))
	   (math-reject-arg b 'square-matrixp)))
	((Math-vectorp a)
	 (math-map-vec-2 'math-div a b))
	((eq (car-safe a) 'sdev)
	 (if (eq (car-safe b) 'sdev)
	     (let ((x (math-div (nth 1 a) (nth 1 b))))
	       (math-make-sdev
		x
		(math-div
		 (math-hypot (nth 2 a) (math-mul (nth 2 b) x))
		 (math-abs (nth 1 b)))))
	   (and (or (Math-anglep b)
		    (not (Math-objvecp b)))
		(math-make-sdev (math-div (nth 1 a) b)
				(math-abs (math-div (nth 2 a) b))))))
	((and (eq (car-safe b) 'sdev)
	      (or (Math-anglep a)
		  (not (Math-objvecp a))))
	 (let ((x (math-div a (nth 1 b))))
	   (math-make-sdev x
			   (math-abs (math-div (math-mul (nth 2 b) x)
					       (nth 1 b))))))
	((and (eq (car-safe a) 'intv) (Math-anglep b))
	 (if (Math-negp b)
	     (math-neg (math-div a (math-neg b)))
	   (math-make-intv (nth 1 a)
			   (math-div (nth 2 a) b)
			   (math-div (nth 3 a) b))))
	((and (eq (car-safe b) 'intv) (Math-anglep a))
	 (if (Math-posp (nth 2 b))
	     (if (Math-negp a)
		 (math-neg (math-div (math-neg a) b))
	       (math-make-intv (aref [0 2 1 3] (nth 1 b))
			       (math-div a (nth 3 b))
			       (math-div a (nth 2 b))))
	   (if (Math-negp (nth 3 b))
	       (math-neg (math-div a (math-neg b)))
	     (calc-record-why "Division by zero" b)
	     nil)))
	((and (eq (car-safe a) 'intv) (math-constp a)
	      (eq (car-safe b) 'intv) (math-constp b))
	 (if (or (Math-posp (nth 2 b)) (Math-negp (nth 3 b)))
	     (let ((lo (math-div a (nth 2 b)))
		   (hi (math-div a (nth 3 b))))
	       (and (Math-anglep lo)
		    (setq lo (list 'intv (if (memq (nth 1 b) '(2 3)) 3 0)
				   lo lo)))
	       (and (Math-anglep hi)
		    (setq hi (list 'intv (if (memq (nth 1 b) '(1 3)) 3 0)
				   hi hi)))
	       (math-combine-intervals
		(nth 2 lo) (and (memq (nth 1 b) '(2 3))
				(memq (nth 1 lo) '(2 3)))
		(nth 3 lo) (and (memq (nth 1 b) '(2 3))
				(memq (nth 1 lo) '(1 3)))
		(nth 2 hi) (and (memq (nth 1 b) '(1 3))
				(memq (nth 1 hi) '(2 3)))
		(nth 3 hi) (and (memq (nth 1 b) '(1 3))
				(memq (nth 1 hi) '(1 3)))))
	   (calc-record-why "Division by zero" b)
	   nil))
	((and (eq (car-safe a) 'mod)
	      (eq (car-safe b) 'mod)
	      (equal (nth 2 a) (nth 2 b)))
	 (math-make-mod (math-div-mod (nth 1 a) (nth 1 b) (nth 2 a))
			(nth 2 a)))
	((and (eq (car-safe a) 'mod)
	      (Math-anglep b))
	 (math-make-mod (math-div-mod (nth 1 a) b (nth 2 a)) (nth 2 a)))
	((and (eq (car-safe b) 'mod)
	      (Math-anglep a))
	 (math-make-mod (math-div-mod a (nth 1 b) (nth 2 b)) (nth 2 b)))
	((eq (car-safe a) 'hms)
	 (if (eq (car-safe b) 'hms)
	     (math-with-extra-prec 1
	       (math-div (math-from-hms a 'deg)
			 (math-from-hms b 'deg)))
	   (math-with-extra-prec 2
	     (math-to-hms (math-div (math-from-hms a 'deg) b) 'deg))))
	(t (calc-record-why "Incompatible arguments" a b)))
)

(defun math-div-symb-fancy (a b)
  (or (and (Math-equal-int b 1) a)
      (and (Math-equal-int b -1) (math-neg a))
      (and (eq (car-safe b) '^)
	   (or (Math-looks-negp (nth 2 b)) (Math-equal-int a 1))
	   (math-mul a (math-normalize
			(list '^ (nth 1 b) (math-neg (nth 2 b))))))
      (and (eq (car-safe a) 'neg)
	   (math-neg (math-div (nth 1 a) b)))
      (and (eq (car-safe b) 'neg)
	   (math-neg (math-div a (nth 1 b))))
      (and (eq (car-safe a) '/)
	   (math-div (nth 1 a) (math-mul (nth 2 a) b)))
      (and (eq (car-safe b) '/)
	   (math-div (math-mul a (nth 2 b)) (nth 1 b)))
      (and (eq (car-safe b) 'frac)
	   (math-mul a (math-make-frac (nth 2 b) (nth 1 b))))
      (and (eq (car-safe a) '+)
	   (or (Math-numberp (nth 1 a))
	       (Math-numberp (nth 2 a)))
	   (Math-numberp b)
	   (math-add (math-div (nth 1 a) b)
		     (math-div (nth 2 a) b)))
      (and (eq (car-safe a) '-)
	   (or (Math-numberp (nth 1 a))
	       (Math-numberp (nth 2 a)))
	   (Math-numberp b)
	   (math-sub (math-div (nth 1 a) b)
		     (math-div (nth 2 a) b)))
      (and (or (eq (car-safe a) '-)
	       (math-looks-negp a))
	   (math-looks-negp b)
	   (math-div (math-neg a) (math-neg b)))
      (and (eq (car-safe b) '-)
	   (math-looks-negp a)
	   (math-div (math-neg a) (math-neg b)))
      (if (eq (car-safe a) '*)
	  (if (eq (car-safe b) '*)
	      (let ((c (math-combine-prod (nth 1 a) (nth 1 b) nil t t)))
		(and c
		     (math-div (math-mul c (nth 2 a)) (nth 2 b))))
	    (let ((c (math-combine-prod (nth 1 a) b nil t t)))
	      (and c
		   (math-mul c (nth 2 a)))))
	(if (eq (car-safe b) '*)
	    (let ((c (math-combine-prod a (nth 1 b) nil t t)))
	      (and c
		   (math-div c (nth 2 b))))
	  (math-combine-prod a b nil t nil)))
      (list '/ a b))
)

;;;; [calc-forms.el]

(defun math-div-mod (a b m)   ; [R R R R]  (Returns nil if no solution)
  (and (Math-integerp a) (Math-integerp b) (Math-integerp m)
       (let ((u1 1) (u3 b) (v1 0) (v3 m))
	 (while (not (eq v3 0))   ; See Knuth sec 4.5.2, exercise 15
	   (let* ((q (math-idivmod u3 v3))
		  (t1 (math-sub u1 (math-mul v1 (car q)))))
	     (setq u1 v1  u3 v3  v1 t1  v3 (cdr q))))
	 (let ((q (math-idivmod a u3)))
	   (and (eq (cdr q) 0)
		(math-mod (math-mul (car q) u1) m)))))
)

(defun math-mod-intv (a b)
  (let* ((q1 (math-floor (math-div (nth 2 a) b)))
	 (q2 (math-floor (math-div (nth 3 a) b)))
	 (m1 (math-sub (nth 2 a) (math-mul q1 b)))
	 (m2 (math-sub (nth 3 a) (math-mul q2 b))))
    (cond ((equal q1 q2)
	   (math-sort-intv (nth 1 a) m1 m2))
	  ((and (math-equal-int (math-sub q2 q1) 1)
		(math-zerop m2)
		(memq (nth 1 a) '(0 2)))
	   (math-make-intv (nth 1 a) m1 b))
	  (t
	   (math-make-intv 2 0 b))))
)

;;;; [calc-arith.el]

(defun math-pow-fancy (a b)
  (cond ((and (Math-numberp a) (Math-numberp b))
	 (cond ((and (eq (car-safe b) 'frac)
		     (equal (nth 2 b) 2))
		(math-ipow (math-sqrt-raw (math-float a)) (nth 1 b)))
	       ((equal b '(float 5 -1))
		(math-sqrt-raw (math-float a)))
	       (t
		(math-with-extra-prec 2
		  (math-exp-raw
		   (math-float (math-mul b (math-ln-raw (math-float a)))))))))
	((or (not (Math-objvecp a))
	     (not (Math-objectp b)))
	 (cond ((and (eq (car-safe a) 'calcFunc-sqrt)
		     (math-evenp b))
		(math-pow (nth 1 a) (math-div2 b)))
	       ((eq (car-safe a) '*)
		(math-mul (math-pow (nth 1 a) b)
			  (math-pow (nth 2 a) b)))
	       ((eq (car-safe a) '/)
		(math-div (math-pow (nth 1 a) b)
			  (math-pow (nth 2 a) b)))
	       ((and (eq (car-safe a) '^)
		     (Math-integerp b))
		(math-pow (nth 1 a) (math-mul (nth 2 a) b)))
	       ((and (math-looks-negp a)
		     (Math-integerp b))
		(if (math-evenp b)
		    (math-pow (math-neg a) b)
		  (math-neg (math-pow (math-neg a) b))))
	       (t (if (Math-objectp a)
		      (calc-record-why 'objectp b)
		    (calc-record-why 'objectp a))
		  (list '^ a b))))
	((and (eq (car-safe a) 'sdev) (eq (car-safe b) 'sdev))
	 (if (and (math-constp a) (math-constp b))
	     (math-with-extra-prec 2
	       (let* ((ln (math-ln-raw (math-float (nth 1 a))))
		      (pow (math-exp-raw
			    (math-float (math-mul (nth 1 b) ln)))))
		 (list 'sdev
		       pow
		       (math-mul
			pow
			(math-hypot (math-mul (nth 2 a)
					      (math-div (nth 1 b)
							(nth 1 a)))
				    (math-mul (nth 2 b) ln))))))
	   (let ((pow (math-pow (nth 1 a) (nth 1 b))))
	     (list 'sdev
		   pow
		   (math-mul pow
			     (math-hypot (math-mul (nth 2 a)
						   (math-div (nth 1 b)
							     (nth 1 a)))
					 (math-mul (nth 2 b)
						   (math-ln
						    (nth 1 a)))))))))
	((and (eq (car-safe a) 'sdev) (Math-realp b))
	 (if (math-constp a)
	     (math-with-extra-prec 2
	       (let ((pow (math-pow (nth 1 a) (math-sub b 1))))
		 (list 'sdev
		       (math-mul pow (nth 1 a))
		       (math-mul pow (math-mul (nth 2 a) b)))))
	   (list 'sdev
		 (math-mul (math-pow (nth 1 a) b))
		 (math-mul (math-pow (nth 1 a) (math-add b -1))
			   (math-mul (nth 2 a) b)))))
	((and (eq (car-safe b) 'sdev) (Math-realp a))
	 (math-with-extra-prec 2
	   (let* ((ln (math-ln-raw (math-float a)))
		  (pow (math-exp (math-mul (nth 1 b) ln))))
	     (list 'sdev
		   pow
		   (math-mul pow (math-mul (nth 2 b) ln))))))
	((and (eq (car-safe a) 'intv) (math-constp a)
	      (Math-realp b)
	      (or (Math-posp (nth 2 a))
		  (Math-natnump b)
		  (and (math-zerop (nth 2 a))
		       (Math-posp b))))
	 (if (math-evenp b)
	     (setq a (math-abs a)))
	 (math-sort-intv (nth 1 a)
			 (math-pow (nth 2 a) b)
			 (math-pow (nth 3 a) b)))
	((and (eq (car-safe b) 'intv) (math-constp b)
	      (Math-posp a))
	 (math-sort-intv (nth 1 b)
			 (math-pow a (nth 2 b))
			 (math-pow a (nth 3 b))))
	((and (eq (car-safe a) 'intv) (math-constp a)
	      (eq (car-safe b) 'intv) (math-constp b)
	      (or (and (not (Math-negp (nth 2 a)))
		       (not (Math-negp (nth 2 b))))
		  (and (Math-posp (nth 2 a))
		       (not (Math-posp (nth 3 b))))))
	 (let ((lo (math-pow a (nth 2 a)))
	       (hi (math-pow a (nth 3 a))))
	   (math-combine-intervals (nth 2 lo) (and (memq (nth 1 a) '(2 3))
						   (memq (nth 1 lo) '(2 3)))
				   (nth 3 lo) (and (memq (nth 1 a) '(2 3))
						   (memq (nth 1 lo) '(1 3)))
				   (nth 2 hi) (and (memq (nth 1 a) '(1 3))
						   (memq (nth 1 hi) '(2 3)))
				   (nth 3 hi) (and (memq (nth 1 a) '(1 3))
						   (memq (nth 1 hi) '(1 3))))))
	((and (eq (car-safe a) 'mod) (eq (car-safe b) 'mod)
	      (equal (nth 2 a) (nth 2 b)))
	 (math-make-mod (math-pow-mod (nth 1 a) (nth 1 b) (nth 2 a))
			(nth 2 a)))
	((and (eq (car-safe a) 'mod) (Math-anglep b))
	 (math-make-mod (math-pow-mod (nth 1 a) b (nth 2 a)) (nth 2 a)))
	((and (eq (car-safe b) 'mod) (Math-anglep a))
	 (math-make-mod (math-pow-mod a (nth 1 b) (nth 2 b)) (nth 2 b)))
	((not (Math-numberp a))
	 (math-reject-arg a 'numberp))
	(t
	 (math-reject-arg b 'numberp)))
)

;;; This assumes A < M and M > 0.
(defun math-pow-mod (a b m)   ; [R R R R]
  (if (and (Math-integerp a) (Math-integerp b) (Math-integerp m))
      (if (Math-negp b)
	  (math-div-mod 1 (math-pow-mod a (Math-integer-neg b) m) m)
	(if (eq m 1)
	    0
	  (math-pow-mod-step a b m)))
    (math-mod (math-pow a b) m))
)

(defun math-pow-mod-step (a n m)   ; [I I I I]
  (math-working "pow" a)
  (let ((val (cond
	      ((eq n 0) 1)
	      ((eq n 1) a)
	      (t
	       (let ((rest (math-pow-mod-step
			    (math-imod (math-mul a a) m)
			    (math-div2 n)
			    m)))
		 (if (math-evenp n)
		     rest
		   (math-mod (math-mul a rest) m)))))))
    (math-working "pow" val)
    val)
)

;;;; [calc-bin.el]

(defvar math-power-of-2-cache (list 1 2 4 8 16 32 64 128 256 512 1024))
(defvar math-big-power-of-2-cache nil)
(defun math-power-of-2 (n)    ;  [I I] [Public]
  (if (and (natnump n) (<= n 100))
      (or (nth n math-power-of-2-cache)
	  (let* ((i (length math-power-of-2-cache))
		 (val (nth (1- i) math-power-of-2-cache)))
	    (while (<= i n)
	      (setq val (math-mul val 2)
		    math-power-of-2-cache (nconc math-power-of-2-cache
						 (list val))
		    i (1+ i)))
	    val))
    (let ((found (assq n math-big-power-of-2-cache)))
      (if found
	  (cdr found)
	(let ((po2 (math-ipow 2 n)))
	  (setq math-big-power-of-2-cache
		(cons (cons n po2) math-big-power-of-2-cache))
	  po2))))
)

(defun math-integer-log2 (n)    ; [I I] [Public]
  (let ((i 0)
	(p math-power-of-2-cache)
	val)
    (while (and p (Math-natnum-lessp (setq val (car p)) n))
      (setq p (cdr p)
	    i (1+ i)))
    (if p
	(and (equal val n)
	     i)
      (while (Math-natnum-lessp
	      (prog1
		  (setq val (math-mul val 2))
		(setq math-power-of-2-cache (nconc math-power-of-2-cache
						   (list val))))
	      n)
	(setq i (1+ i)))
      (and (equal val n)
	   i)))
)



;;;; [calc-math.el]

;;; Compute the integer square-root floor(sqrt(A)).  A > 0.  [I I] [Public]
;;; This method takes advantage of the fact that Newton's method starting
;;; with an overestimate always works, even using truncating integer division!
(defun math-isqrt (a)
  (cond ((Math-zerop a) a)
	((Math-negp a)
	 (math-imaginary (math-isqrt (math-neg a))))
	((integerp a)
	 (math-isqrt-small a))
	((eq (car a) 'bigpos)
	 (math-normalize (cons 'bigpos (cdr (math-isqrt-bignum (cdr a))))))
	(t
	 (math-floor (math-sqrt a))))
)

;;; This returns (flag . result) where the flag is T if A is a perfect square.
(defun math-isqrt-bignum (a)   ; [P.l L]
  (let ((len (length a)))
    (if (= (% len 2) 0)
	(let* ((top (nthcdr (- len 2) a)))
	  (math-isqrt-bignum-iter
	   a
	   (math-scale-bignum-3
	    (math-bignum-big
	     (1+ (math-isqrt-small
		  (+ (* (nth 1 top) 1000) (car top)))))
	    (1- (/ len 2)))))
      (let* ((top (nth (1- len) a)))
	(math-isqrt-bignum-iter
	 a
	 (math-scale-bignum-3
	  (list (1+ (math-isqrt-small top)))
	  (/ len 2))))))
)

(defun math-isqrt-bignum-iter (a guess)   ; [l L l]
  (math-working "isqrt" (cons 'bigpos guess))
  (let* ((q (math-div-bignum a guess))
	 (s (math-add-bignum (car q) guess))
	 (g2 (math-div2-bignum s))
	 (comp (math-compare-bignum g2 guess)))
    (if (< comp 0)
	(math-isqrt-bignum-iter a g2)
      (cons (and (= comp 0)
		 (math-zerop-bignum (cdr q))
		 (= (% (car s) 2) 0))
	    guess)))
)

(defun math-scale-bignum-3 (a n)   ; [L L S]
  (while (> n 0)
    (setq a (cons 0 a)
	  n (1- n)))
  a
)

(defun math-isqrt-small (a)   ; A > 0.  [S S]
  (let ((g (cond ((>= a 10000) 1000)
		 ((>= a 100) 100)
		 (t 10)))
	g2)
    (while (< (setq g2 (/ (+ g (/ a g)) 2)) g)
      (setq g g2))
    g)
)


;;;; [calc-ext.el]

(defun math-inexact-result ()
  (and calc-symbolic-mode
       (signal 'inexact-result nil))
)


;;;; [calc-math.el]

;;; Compute the square root of a number.
;;; [T N] if possible, else [F N] if possible, else [C N].  [Public]
(defun math-sqrt (a)
  (or
   (and (Math-zerop a) a)
   (and (Math-negp a)
	(math-imaginary (math-sqrt (math-neg a))))
   (and (integerp a)
	(let ((sqrt (math-isqrt-small a)))
	  (if (= (* sqrt sqrt) a)
	      sqrt
	    (math-sqrt-float (math-float a) (math-float sqrt)))))
   (and (eq (car-safe a) 'bigpos)
	(let* ((res (math-isqrt-bignum (cdr a)))
	       (sqrt (math-normalize (cons 'bigpos (cdr res)))))
	  (if (car res)
	      sqrt
	    (math-sqrt-float (math-float a) (math-float sqrt)))))
   (and (eq (car-safe a) 'frac)
	(let* ((num-res (math-isqrt-bignum (cdr (Math-bignum-test (nth 1 a)))))
	       (num-sqrt (math-normalize (cons 'bigpos (cdr num-res))))
	       (den-res (math-isqrt-bignum (cdr (Math-bignum-test (nth 2 a)))))
	       (den-sqrt (math-normalize (cons 'bigpos (cdr den-res)))))
	  (if (and (car num-res) (car den-res))
	      (list 'frac num-sqrt den-sqrt)
	    (math-sqrt-float (math-float a)
			     (math-div (math-float num-sqrt) den-sqrt)))))
   (and (eq (car-safe a) 'float)
	(if calc-symbolic-mode
	    (if (= (% (nth 2 a) 2) 0)
		(let ((res (math-isqrt-bignum
			    (cdr (Math-bignum-test (nth 1 a))))))
		  (if (car res)
		      (math-make-float (math-normalize
					(cons 'bigpos (cdr res)))
				       (/ (nth 2 a) 2))
		    (signal 'inexact-result nil)))
	      (signal 'inexact-result nil))
	  (math-sqrt-float a)))
   (and (eq (car-safe a) 'cplx)
	(math-with-extra-prec 2
	  (let* ((d (math-abs a))
		 (imag (math-sqrt (math-mul (math-sub d (nth 1 a))
					    '(float 5 -1)))))
	    (list 'cplx
		  (math-sqrt (math-mul (math-add d (nth 1 a)) '(float 5 -1)))
		  (if (math-negp (nth 2 a)) (math-neg imag) imag)))))
   (and (eq (car-safe a) 'polar)
	(list 'polar
	      (math-sqrt (nth 1 a))
	      (math-mul (nth 2 a) '(float 5 -1))))
   (and (eq (car-safe a) 'sdev) (not (math-negp (nth 1 a)))
	(let ((sqrt (math-sqrt (nth 1 a))))
	  (math-make-sdev sqrt
			  (math-div (nth 2 a) (math-mul sqrt 2)))))
   (and (eq (car-safe a) 'intv)
	(math-make-intv (nth 1 a) (math-sqrt (nth 2 a)) (math-sqrt (nth 3 a))))
   (and (memq (car-safe a) '(* /))
	(let ((s1 (math-sqrt (nth 1 a)))
	      (s2 (math-sqrt (nth 2 a))))
	  (and (not (and (eq (car-safe s1) 'calcFunc-sqrt)
			 (eq (car-safe s2) 'calcFunc-sqrt)))
	       (if (eq (car a) '*)
		   (math-mul s1 s2)
		 (math-div s1 s2)))))
   (progn
     (calc-record-why 'numberp a)
     (list 'calcFunc-sqrt a)))
)
(fset 'calcFunc-sqrt (symbol-function 'math-sqrt))

(defun math-sqrt-float (a &optional guess)   ; [F F F]
  (if calc-symbolic-mode
      (signal 'inexact-result nil)
    (math-with-extra-prec 1 (math-sqrt-raw a guess)))
)

(defun math-sqrt-raw (a &optional guess)   ; [F F F]
  (if (not (Math-posp a))
      (math-sqrt a)
    (if (null guess)
	(let ((ldiff (- (math-numdigs (nth 1 a)) 6)))
	  (or (= (% (+ (nth 2 a) ldiff) 2) 0) (setq ldiff (1+ ldiff)))
	  (setq guess (math-make-float (math-isqrt-small
					(math-scale-int (nth 1 a) (- ldiff)))
				       (/ (+ (nth 2 a) ldiff) 2)))))
    (math-sqrt-float-iter a guess))
)

(defun math-sqrt-float-iter (a guess)   ; [F F F]
  (math-working "sqrt" guess)
  (let ((g2 (math-mul-float (math-add-float guess (math-div-float a guess))
			    '(float 5 -1))))
     (if (math-nearly-equal-float g2 guess)
	 g2
       (math-sqrt-float-iter a g2)))
)

;;; True if A and B differ only in the last digit of precision.  [P F F]
(defun math-nearly-equal-float (a b)
  (let ((diff (nth 1 (math-sub-float a b))))
    (or (eq diff 0)
	(and (not (consp diff))
	     (< diff 10)
	     (> diff -10)
	     (= diff (if (< (nth 2 a) (nth 2 b))
			 (nth 2 a) (nth 2 b)))
	     (or (= (math-numdigs (nth 1 a)) calc-internal-prec)
		 (= (math-numdigs (nth 1 b)) calc-internal-prec)))))
)

(defun math-nearly-equal (a b)   ;  [P R R] [Public]
  (math-nearly-equal-float (math-float a) (math-float b))
)

;;; True if A is nearly zero compared to B.  [P F F]
(defun math-nearly-zerop-float (a b)
  (or (eq (nth 1 a) 0)
      (<= (+ (math-numdigs (nth 1 a)) (nth 2 a))
	  (1+ (- (+ (math-numdigs (nth 1 b)) (nth 2 b)) calc-internal-prec))))
)

(defun math-nearly-zerop (a b)
  (math-nearly-zerop-float (math-float a) (math-float b))
)

;;; This implementation could be improved, accuracy-wise.
(defun math-hypot (a b)
  (cond ((Math-zerop a) (math-abs b))
	((Math-zerop b) (math-abs a))
	((not (Math-scalarp a))
	 (calc-record-why 'scalarp a)
	 (list 'calcFunc-hypot a b))
	((not (Math-scalarp b))
	 (calc-record-why 'scalarp b)
	 (list 'calcFunc-hypot a b))
	((and (Math-realp a) (Math-realp b))
	 (math-with-extra-prec 1
	   (math-sqrt (math-add (math-sqr a) (math-sqr b)))))
	((eq (car-safe a) 'hms)
	 (if (eq (car-safe b) 'hms)   ; this helps sdev's of hms forms
	     (math-to-hms (math-hypot (math-from-hms a 'deg)
				      (math-from-hms b 'deg)))
	   (math-to-hms (math-hypot (math-from-hms a 'deg) b))))
	((eq (car-safe b) 'hms)
	 (math-to-hms (math-hypot a (math-from-hms b 'deg))))
	(t nil))
)
(fset 'calcFunc-hypot (symbol-function 'math-hypot))

(defun calcFunc-sqr (x)
  (math-pow x 2)
)



;;;; [calc-arith.el]

;;; Compute the minimum of two real numbers.  [R R R] [Public]
(defun math-min (a b)
  (if (and (consp a) (eq (car a) 'intv))
      (if (and (consp b) (eq (car b) 'intv))
	  (let ((lo (nth 2 a))
		(lom (memq (nth 1 a) '(2 3)))
		(hi (nth 3 a))
		(him (memq (nth 1 a) '(1 3)))
		res)
	    (if (= (setq res (math-compare (nth 2 b) lo)) -1)
		(setq lo (nth 2 b) lom (memq (nth 1 b) '(2 3)))
	      (if (= res 0)
		  (setq lom (or lom (memq (nth 1 b) '(2 3))))))
	    (if (= (setq res (math-compare (nth 3 b) hi)) -1)
		(setq hi (nth 3 b) him (memq (nth 1 b) '(1 3)))
	      (if (= res 0)
		  (setq him (or him (memq (nth 1 b) '(1 3))))))
	    (math-make-intv (+ (if lom 2 0) (if him 1 0)) lo hi))
	(math-min a (list 'intv 3 b b)))
    (if (and (consp b) (eq (car b) 'intv))
	(math-min (list 'intv 3 a a) b)
      (if (Math-lessp a b)
	  a
	b)))
)

(defun calcFunc-min (a &rest b)
  (if (not (or (Math-anglep a)
	       (and (eq (car a) 'intv) (math-constp a))))
      (math-reject-arg a 'anglep))
  (math-min-list a b)
)

(defun math-min-list (a b)
  (if b
      (if (or (Math-anglep (car b))
	      (and (eq (car (car b)) 'intv) (math-constp (car b))))
	  (math-min-list (math-min a (car b)) (cdr b))
	(math-reject-arg (car b) 'anglep))
    a)
)

;;; Compute the maximum of two real numbers.  [R R R] [Public]
(defun math-max (a b)
  (if (or (and (consp a) (eq (car a) 'intv))
	  (and (consp b) (eq (car b) 'intv)))
      (math-neg (math-min (math-neg a) (math-neg b)))
    (if (Math-lessp a b)
	b
      a))
)

(defun calcFunc-max (a &rest b)
  (if (not (or (Math-anglep a)
	       (and (eq (car a) 'intv) (math-constp a))))
      (math-reject-arg a 'anglep))
  (math-max-list a b)
)

(defun math-max-list (a b)
  (if b
      (if (or (Math-anglep (car b))
	      (and (eq (car (car b)) 'intv) (math-constp (car b))))
	  (math-max-list (math-max a (car b)) (cdr b))
	(math-reject-arg (car b) 'anglep))
    a)
)


;;; Compute the absolute value of A.  [O O; r r] [Public]
(defun math-abs (a)
  (cond ((Math-negp a)
	 (math-neg a))
	((Math-anglep a)
	 a)
	((eq (car a) 'cplx)
	 (math-hypot (nth 1 a) (nth 2 a)))
	((eq (car a) 'polar)
	 (nth 1 a))
	((eq (car a) 'vec)
	 (if (cdr (cdr (cdr a)))
	     (math-sqrt (math-abssqr a))
	   (if (cdr (cdr a))
	       (math-hypot (nth 1 a) (nth 2 a))
	     (if (cdr a)
		 (math-abs (nth 1 a))
	       a))))
	((eq (car a) 'sdev)
	 (list 'sdev (math-abs (nth 1 a)) (nth 2 a)))
	((and (eq (car a) 'intv) (math-constp a))
	 (if (Math-posp a)
	     a
	   (let* ((nlo (math-neg (nth 2 a)))
		  (res (math-compare nlo (nth 3 a))))
	     (cond ((= res 1)
		    (math-make-intv (if (memq (nth 1 a) '(0 1)) 2 3) 0 nlo))
		   ((= res 0)
		    (math-make-intv (if (eq (nth 1 a) 0) 2 3) 0 nlo))
		   (t
		    (math-make-intv (if (memq (nth 1 a) '(0 2)) 2 3)
				    0 (nth 3 a)))))))
	((eq (car a) 'calcFunc-abs)
	 (car a))
	((math-looks-negp a)
	 (list 'calcFunc-abs (math-neg a)))
	(t (calc-record-why 'numvecp a)
	   (list 'calcFunc-abs a)))
)
(fset 'calcFunc-abs (symbol-function 'math-abs))


(defun math-trunc-fancy (a)
  (cond ((eq (car a) 'cplx) (math-trunc (nth 1 a)))
	((eq (car a) 'polar) (math-trunc (math-complex a)))
	((eq (car a) 'hms) (list 'hms (nth 1 a) 0 0))
	((eq (car a) 'mod)
	 (if (math-messy-integerp (nth 2 a))
	     (math-trunc (math-make-mod (nth 1 a) (math-trunc (nth 2 a))))
	   (math-make-mod (math-trunc (nth 1 a)) (nth 2 a))))
	((eq (car a) 'intv)
	 (math-make-intv 3
			 (if (and (Math-negp (nth 2 a))
				  (Math-num-integerp (nth 2 a))
				  (memq (nth 1 a) '(0 1)))
			     (math-add (math-trunc (nth 2 a)) 1)
			   (math-trunc (nth 2 a)))
			 (if (and (Math-posp (nth 3 a))
				  (Math-num-integerp (nth 3 a))
				  (memq (nth 1 a) '(0 2)))
			     (math-add (math-trunc (nth 3 a)) -1)
			   (math-trunc (nth 3 a)))))
	((math-provably-integerp a) a)
	(t (math-reject-arg a 'numberp)))
)
(defun calcFunc-ftrunc (a)
  (if (Math-messy-integerp a)
      a
    (math-float (math-trunc a)))
)

(defun math-floor-fancy (a)
  (cond ((math-provably-integerp a) a)
	((eq (car a) 'hms)
	 (if (or (math-posp a)
		 (and (math-zerop (nth 2 a))
		      (math-zerop (nth 3 a))))
	     (math-trunc a)
	   (math-add (math-trunc a) -1)))
	((eq (car a) 'intv)
	 (math-make-intv 3
			 (math-floor (nth 2 a))
			 (if (and (Math-num-integerp (nth 3 a))
				  (memq (nth 1 a) '(0 2)))
			     (math-add (math-floor (nth 3 a)) -1)
			   (math-floor (nth 3 a)))))
	(t (math-reject-arg a 'anglep)))
)
(defun calcFunc-ffloor (a)
  (if (Math-messy-integerp a)
      a
    (math-float (math-floor a)))
)

;;; Coerce A to be an integer (by truncation toward plus infinity).  [I N]
(defun math-ceiling (a)   ;  [Public]
  (cond ((Math-integerp a) a)
	((Math-messy-integerp a) (math-trunc a))
	((Math-realp a)
	 (if (Math-posp a)
	     (math-add (math-trunc a) 1)
	   (math-trunc a)))
	((math-provably-integerp a) a)
	((eq (car a) 'hms)
	 (if (or (math-negp a)
		 (and (math-zerop (nth 2 a))
		      (math-zerop (nth 3 a))))
	     (math-trunc a)
	   (math-add (math-trunc a) 1)))
	((eq (car a) 'intv)
	 (math-make-intv 3
			 (if (and (Math-num-integerp (nth 2 a))
				  (memq (nth 1 a) '(0 1)))
			     (math-add (math-floor (nth 2 a)) 1)
			   (math-ceiling (nth 2 a)))
			 (math-ceiling (nth 3 a))))
	(t (math-reject-arg a 'anglep)))
)
(fset 'calcFunc-ceil (symbol-function 'math-ceiling))
(defun calcFunc-fceil (a)
  (if (Math-messy-integerp a)
      a
    (math-float (math-ceiling a)))
)

;;; Coerce A to be an integer (by rounding to nearest integer).  [I N] [Public]
(defun math-round (a)
  (cond ((Math-anglep a)
	 (if (Math-num-integerp a)
	     (math-trunc a)
	   (if (Math-negp a)
	       (math-neg (math-round (math-neg a)))
	     (math-floor
	      (let ((calc-angle-mode 'deg))   ; in case of HMS forms
		(math-add a (if (Math-ratp a)
				'(frac 1 2)
			      '(float 5 -1))))))))
	((math-provably-integerp a) a)
	((eq (car a) 'intv)
	 (math-floor (math-add a '(frac 1 2))))
	(t (math-reject-arg a 'anglep)))
)
(fset 'calcFunc-round (symbol-function 'math-round))
(defun calcFunc-fround (a)
  (if (Math-messy-integerp a)
      a
    (math-float (math-round a)))
)


;;; Pull floating-point values apart into mantissa and exponent.
(defun math-mant-part (x)
  (if (Math-realp x)
      (if (or (Math-ratp x)
	      (eq (nth 1 x) 0))
	  x
	(list 'float (nth 1 x) (- 1 (math-numdigs (nth 1 x)))))
    (calc-record-why 'realp x)
    (list 'calcFunc-mant x))
)
(fset 'calcFunc-mant (symbol-function 'math-mant-part))

(defun math-xpon-part (x)
  (if (Math-realp x)
      (if (or (Math-ratp x)
	      (eq (nth 1 x) 0))
	  0
	(math-add (nth 2 x) (1- (math-numdigs (nth 1 x)))))
    (calc-record-why 'realp x)
    (list 'calcFunc-xpon x))
)
(fset 'calcFunc-xpon (symbol-function 'math-xpon-part))

(defun math-scale-float (x n)
  (if (integerp n)
      (cond ((= n 0)
	     x)
	    ((Math-integerp x)
	     (if (> n 0)
		 (math-scale-int x n)
	       (math-div x (math-scale-int 1 (- n)))))
	    ((eq (car x) 'frac)
	     (if (> n 0)
		 (math-make-frac (math-scale-int (nth 1 x) n) (nth 2 x))
	       (math-make-frac (nth 1 x) (math-scale-int (nth 2 x) (- n)))))
	    ((eq (car x) 'float)
	     (math-make-float (nth 1 x) (+ (nth 2 x) n)))
	    ((memq (car x) '(cplx sdev))
	     (math-normalize
	      (list (car x)
		    (math-scale-float (nth 1 x) n)
		    (math-scale-float (nth 2 x) n))))
	    ((memq (car x) '(polar mod))
	     (math-normalize
	      (list (car x)
		    (math-scale-float (nth 1 x) n)
		    (nth 2 x))))
	    ((eq (car x) 'intv)
	     (math-normalize
	      (list (car x)
		    (nth 1 x)
		    (math-scale-float (nth 2 x) n)
		    (math-scale-float (nth 3 x) n))))
	    (t
	     (calc-record-why 'realp x)
	     (list 'calcFunc-scf x n)))
    (if (math-messy-integerp n)
	(math-scale-float x (math-trunc n))
      (calc-record-why 'integerp n)
      (list 'calcFunc-scf x n)))
)
(fset 'calcFunc-scf (symbol-function 'math-scale-float))


;;;; [calc-frac.el]

;;; Convert a real value to fractional form.  [T R I; T R F] [Public]
(defun math-to-fraction (a &optional tol)
  (or tol (setq tol 0))
  (cond ((Math-ratp a)
	 a)
	((memq (car a) '(cplx polar vec hms sdev intv mod))
	 (cons (car a) (mapcar (function
				(lambda (x)
				  (math-to-fraction x tol)))
			       (cdr a))))
	((Math-negp a)
	 (math-neg (math-to-fraction (math-neg a) tol)))
	((not (eq (car a) 'float))
	 (math-reject-arg a 'numberp))
	((integerp tol)
	 (if (<= tol 0)
	     (setq tol (+ tol calc-internal-prec)))
	 (math-to-fraction a (list 'float 5
				   (- (+ (math-numdigs (nth 1 a))
					 (nth 2 a))
				      (1+ tol)))))
	((not (eq (car tol) 'float))
	 (if (Math-realp tol)
	     (math-to-fraction a (math-float tol))
	   (math-reject-arg tol 'realp)))
	((Math-negp tol)
	 (math-to-fraction a (math-neg tol)))
	((Math-zerop tol)
	 (math-to-fraction a 0))
	((not (math-lessp-float tol '(float 1 0)))
	 (math-trunc a))
	((Math-zerop a)
	 0)
	(t
	 (let ((cfrac (math-continued-fraction a tol))
	       (calc-prefer-frac t))
	   (math-eval-continued-fraction cfrac))))
)
(fset 'calcFunc-frac (symbol-function 'math-to-fraction))

(defun math-continued-fraction (a tol)
  (let ((calc-internal-prec (+ calc-internal-prec 2)))
    (let ((cfrac nil)
	  (aa a)
	  (calc-prefer-frac nil)
	  int)
      (while (or (null cfrac)
		 (and (not (Math-zerop aa))
		      (not (math-lessp-float
			    (math-abs
			     (math-sub a
				       (let ((f (math-eval-continued-fraction
						 cfrac)))
					 (math-working "Fractionalize" f)
					 f)))
			    tol))))
	(setq int (math-trunc aa)
	      aa (math-sub aa int)
	      cfrac (cons int cfrac))
	(or (Math-zerop aa)
	    (setq aa (math-div 1 aa))))
      cfrac))
)

(defun math-eval-continued-fraction (cf)
  (let ((n (car cf))
	(d 1)
	temp)
    (while (setq cf (cdr cf))
      (setq temp (math-add (math-mul (car cf) n) d)
	    d n
	    n temp))
    (math-div n d))
)


;;;; [calc-ext.el]

(defun math-clean-number (a &optional prec)   ; [X X S] [Public]
  (if prec
      (cond ((Math-messy-integerp prec)
	     (math-clean-number a (math-trunc prec)))
	    ((or (not (integerp prec))
		 (< prec 3))
	     (calc-record-why "Precision must be an integer 3 or above")
	     (list 'calcFunc-clean a prec))
	    ((not (Math-objvecp a))
	     (list 'calcFunc-clean a prec))
	    (t (let ((calc-internal-prec prec))
		 (math-clean-number (math-normalize a)))))
    (cond ((eq (car-safe a) 'polar)
	   (let ((theta (math-mod (nth 2 a)
				  (if (eq calc-angle-mode 'rad)
				      (math-two-pi)
				    360))))
	     (math-neg
	      (math-neg
	       (math-normalize
		(list 'polar (nth 1 a) theta))))))
	  ((Math-vectorp a)
	   (math-map-vec 'math-clean-number a))
	  ((Math-objectp a) a)
	  (t (list 'calcFunc-clean a))))
)
(fset 'calcFunc-clean (symbol-function 'math-clean-number))




;;;; [calc-prog.el]

;;;; Logical operations.

(defun calcFunc-eq (a b)
  (let ((res (math-compare a b)))
    (if (= res 0)
	1
      (if (= res 2)
	  (list 'calcFunc-eq a b)
	0)))
)

(defun calcFunc-neq (a b)
  (let ((res (math-compare a b)))
    (if (= res 0)
	0
      (if (= res 2)
	  (list 'calcFunc-neq a b)
	1)))
)

(defun calcFunc-lt (a b)
  (let ((res (math-compare a b)))
    (if (= res -1)
	1
      (if (= res 2)
	  (list 'calcFunc-lt a b)
	0)))
)

(defun calcFunc-gt (a b)
  (let ((res (math-compare a b)))
    (if (= res 1)
	1
      (if (= res 2)
	  (list 'calcFunc-gt a b)
	0)))
)

(defun calcFunc-leq (a b)
  (let ((res (math-compare a b)))
    (if (= res 1)
	0
      (if (= res 2)
	  (list 'calcFunc-leq a b)
	1)))
)

(defun calcFunc-geq (a b)
  (let ((res (math-compare a b)))
    (if (= res -1)
	0
      (if (= res 2)
	  (list 'calcFunc-geq a b)
	1)))
)

(defun calcFunc-land (a b)
  (cond ((Math-zerop a)
	 a)
	((Math-zerop b)
	 b)
	((Math-numberp a)
	 b)
	((Math-numberp b)
	 a)
	(t (list 'calcFunc-land a b)))
)

(defun calcFunc-lor (a b)
  (cond ((Math-zerop a)
	 b)
	((Math-zerop b)
	 a)
	((Math-numberp a)
	 a)
	((Math-numberp b)
	 b)
	(t (list 'calcFunc-lor a b)))
)

(defun calcFunc-lnot (a)
  (if (Math-zerop a)
      1
    (if (Math-numberp a)
	0
      (list 'calcFunc-lnot a)))
)

(defun calcFunc-if (c e1 e2)
  (if (Math-zerop c)
      e2
    (if (Math-numberp c)
	e1
      (list 'calcFunc-if c e1 e2)))
)

(defun math-normalize-logical-op (a)
  (or (and (eq (car a) 'calcFunc-if)
	   (= (length a) 4)
	   (let ((a1 (math-normalize (nth 1 a))))
	     (if (Math-zerop a1)
		 (math-normalize (nth 3 a))
	       (if (Math-numberp a1)
		   (math-normalize (nth 2 a))
		 (list 'calcFunc-if a1 (nth 2 a) (nth 3 a))))))
      a)
)

(defun calcFunc-in (a b)
  (or (and (eq (car-safe b) 'vec)
	   (let ((bb b))
	     (while (and (setq bb (cdr bb))
			 (not (if (memq (car-safe (car bb)) '(vec intv))
				  (eq (calcFunc-in a (car bb)) 1)
				(math-equal a (car bb))))))
	     (if bb 1 (and (math-constp a) (math-constp bb) 0))))
      (and (eq (car-safe b) 'intv)
	   (let ((res (math-compare a (nth 2 b))))
	     (cond ((= res -1)
		    0)
		   ((= res 0)
		    (if (memq (nth 1 b) '(2 3)) 1 0))
		   ((/= res 1)
		    nil)
		   ((= (setq res (math-compare a (nth 3 b))) 1)
		    0)
		   ((= res 0)
		    (if (memq (nth 1 b) '(1 3)) 1 0))
		   ((/= res -1)
		    nil)
		   (t 1))))
      (and (math-equal a b)
	   1)
      (and (math-constp a) (math-constp b)
	   0)
      (list 'calcFunc-in a b))
)

(defun calcFunc-typeof (a)
  (cond ((Math-integerp a) 1)
	((eq (car a) 'frac) 2)
	((eq (car a) 'float) 3)
	((eq (car a) 'hms) 4)
	((eq (car a) 'cplx) 5)
	((eq (car a) 'polar) 6)
	((eq (car a) 'sdev) 7)
	((eq (car a) 'intv) 8)
	((eq (car a) 'mod) 9)
	((eq (car a) 'var) 100)
	((eq (car a) 'vec) (if (math-matrixp a) 102 101))
	(t (let ((func (assq (car a) '( ( + . calcFunc-add )
					( - . calcFunc-sub )
					( * . calcFunc-mul )
					( / . calcFunc-div )
					( ^ . calcFunc-pow )
					( % . calcFunc-mod )
					( neg . calcFunc-neg )
					( | . calcFunc-vconcat ) ))))
	     (setq func (if func (cdr func) (car a)))
	     (math-calcFunc-to-var func))))
)

(defun calcFunc-integer (a)
  (if (Math-integerp a)
      1
    (if (Math-objvecp a)
	0
      (list 'calcFunc-integer a)))
)

(defun calcFunc-real (a)
  (if (Math-realp a)
      1
    (if (Math-objvecp a)
	0
      (list 'calcFunc-real a)))
)

(defun calcFunc-constant (a)
  (if (math-constp a)
      1
    (if (Math-objvecp a)
	0
      (list 'calcFunc-constant a)))
)

(defun calcFunc-refers (a b)
  (if (math-expr-contains a b)
      1
    (if (eq (car-safe a) 'var)
	(list 'calcFunc-refers a b)
      0))
)




;;;; [calc-cplx.el]

;;;; Complex numbers.

(defun math-to-polar (a)   ; [C N] [Public]
  (cond ((Math-vectorp a)
	 (math-map-vec 'math-to-polar a))
	((Math-realp a) a)
	((Math-numberp a)
	 (math-normalize (math-polar a)))
	(t (list 'calcFunc-polar)))
)
(fset 'calcFunc-polar (symbol-function 'math-to-polar))

(defun math-to-rectangular (a)   ; [N N] [Public]
  (cond ((Math-vectorp a)
	 (math-map-vec 'math-to-rectangular a))
	((Math-realp a) a)
	((Math-numberp a)
	 (math-normalize (math-complex a)))
	(t (list 'calcFunc-rect)))
)
(fset 'calcFunc-rect (symbol-function 'math-to-rectangular))

;;; Compute the complex conjugate of A.  [O O] [Public]
(defun math-conj (a)
  (cond ((math-real-objectp a)
	 a)
	((eq (car a) 'cplx)
	 (list 'cplx (nth 1 a) (math-neg (nth 2 a))))
	((eq (car a) 'polar)
	 (list 'polar (nth 1 a) (math-neg (nth 2 a))))
	((eq (car a) 'vec)
	 (math-map-vec 'math-conj a))
	((eq (car a) 'calcFunc-conj)
	 (nth 1 a))
	(t (calc-record-why 'numberp a)
	   (list 'calcFunc-conj a)))
)
(fset 'calcFunc-conj (symbol-function 'math-conj))

;;;; [calc-arith.el]

;;; Compute the absolute value squared of A.  [F N] [Public]
(defun math-abssqr (a)
  (cond ((Math-realp a)
	 (math-sqr a))
	((eq (car a) 'cplx)
	 (math-add (math-sqr (nth 1 a)) (math-sqr (nth 2 a))))
	((eq (car a) 'polar)
	 (math-sqr (nth 1 a)))
	((and (memq (car a) '(sdev intv)) (math-constp a))
	 (math-sqr (math-abs a)))
	((eq (car a) 'vec)
	 (math-reduce-vec 'math-add (math-map-vec 'math-abssqr a)))
	(t (calc-record-why 'numvecp a)
	   (list 'calcFunc-abssqr a)))
)
(fset 'calcFunc-abssqr (symbol-function 'math-abssqr))

;;;; [calc-cplx.el]

;;; Compute the complex argument of A.  [F N] [Public]
(defun math-cplx-arg (a)
  (cond ((Math-anglep a)
	 (if (math-negp a) (math-half-circle nil) 0))
	((eq (car-safe a) 'cplx)
	 (math-arctan2 (nth 2 a) (nth 1 a)))
	((eq (car-safe a) 'polar)
	 (nth 2 a))
	((eq (car a) 'vec)
	 (math-map-vec 'math-cplx-arg a))
	(t (calc-record-why 'numvecp a)
	   (list 'calcFunc-arg a)))
)
(fset 'calcFunc-arg (symbol-function 'math-cplx-arg))

;;; Extract the real or complex part of a complex number.  [R N] [Public]
;;; Also extracts the real part of a modulo form.
(defun math-real-part (a)
  (cond ((memq (car-safe a) '(mod sdev))
	 (nth 1 a))
	((math-real-objectp a) a)
	((eq (car a) 'cplx)
	 (nth 1 a))
	((eq (car a) 'polar)
	 (math-mul (nth 1 a) (math-cos (nth 2 a))))
	((eq (car a) 'vec)
	 (math-map-vec 'math-real-part a))
	(t (calc-record-why 'numberp a)
	   (list 'calcFunc-re a)))
)
(fset 'calcFunc-re (symbol-function 'math-real-part))

(defun math-imag-part (a)
  (cond ((math-real-objectp a)
	 (if (math-floatp a) '(float 0 0) 0))
	((eq (car a) 'cplx)
	 (nth 2 a))
	((eq (car a) 'polar)
	 (math-mul (nth 1 a) (math-sin (nth 2 a))))
	((eq (car a) 'vec)
	 (math-map-vec 'math-imag-part a))
	(t (calc-record-why 'numberp a)
	   (list 'calcFunc-im a)))
)
(fset 'calcFunc-im (symbol-function 'math-imag-part))



;;;; [calc-math.el]

;;;; Transcendental functions.

;;; All of these functions are defined on the complex plane.
;;; (Branch cuts, etc. follow Steele's Common Lisp book.)

;;; Most functions increase calc-internal-prec by 2 digits, then round
;;; down afterward.  "-raw" functions use the current precision, require
;;; their arguments to be in float (or complex float) format, and always
;;; work in radians (where applicable).

(defun math-to-radians (a)   ; [N N]
  (cond ((eq (car-safe a) 'hms)
	 (math-from-hms a 'rad))
	((memq calc-angle-mode '(deg hms))
	 (math-mul a (math-pi-over-180)))
	(t a))
)

(defun math-from-radians (a)   ; [N N]
  (cond ((eq calc-angle-mode 'deg)
	 (if (math-constp a)
	     (math-div a (math-pi-over-180))
	   (list 'calcFunc-deg a)))
	((eq calc-angle-mode 'hms)
	 (math-to-hms a 'rad))
	(t a))
)

(defun math-to-radians-2 (a)   ; [N N]
  (cond ((eq (car-safe a) 'hms)
	 (math-from-hms a 'rad))
	((memq calc-angle-mode '(deg hms))
	 (if calc-symbolic-mode
	     (math-div (math-mul a '(var pi var-pi)) 180)
	   (math-mul a (math-pi-over-180))))
	(t a))
)

(defun math-from-radians-2 (a)   ; [N N]
  (cond ((memq calc-angle-mode '(deg hms))
	 (if calc-symbolic-mode
	     (math-div (math-mul 180 a) '(var pi var-pi))
	   (math-div a (math-pi-over-180))))
	(t a))
)



;;; Sine, cosine, and tangent.

(defun math-sin (x)   ; [N N] [Public]
  (cond ((Math-scalarp x)
	 (math-with-extra-prec 2
	   (math-sin-raw (math-to-radians (math-float x)))))
	((eq (car x) 'sdev)
	 (if (math-constp x)
	     (math-with-extra-prec 2
	       (let* ((xx (math-to-radians (math-float (nth 1 x))))
		      (xs (math-to-radians (math-float (nth 2 x))))
		      (sc (math-sin-cos-raw xx)))
		 (math-make-sdev (car sc)
				 (math-mul xs (math-abs (cdr sc))))))
	   (math-make-sdev (math-sin (nth 1 x))
			   (math-mul (nth 2 x)
				     (math-abs (math-cos (nth 1 x)))))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-cos (math-sub x (math-quarter-circle nil))))
	(t (calc-record-why 'scalarp x)
	   (list 'calcFunc-sin x)))
)
(fset 'calcFunc-sin (symbol-function 'math-sin))

(defun math-cos (x)   ; [N N] [Public]
  (cond ((Math-scalarp x)
	 (math-with-extra-prec 2
	   (math-cos-raw (math-to-radians (math-float x)))))
	((eq (car x) 'sdev)
	 (if (math-constp x)
	     (math-with-extra-prec 2
	       (let* ((xx (math-to-radians (math-float (nth 1 x))))
		      (xs (math-to-radians (math-float (nth 2 x))))
		      (sc (math-sin-cos-raw xx)))
		 (math-make-sdev (cdr sc)
				 (math-mul xs (math-abs (car sc))))))
	   (math-make-sdev (math-cos (nth 1 x))
			   (math-mul (nth 2 x)
				     (math-abs (math-sin (nth 1 x)))))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-with-extra-prec 2
	   (let* ((xx (math-to-radians (math-float x)))
		  (na (math-floor (math-div (nth 2 xx) (math-pi))))
		  (nb (math-floor (math-div (nth 3 xx) (math-pi))))
		  (span (math-sub nb na)))
	     (if (memq span '(0 1))
		 (let ((int (math-sort-intv (nth 1 x)
					    (math-cos-raw (nth 2 xx))
					    (math-cos-raw (nth 3 xx)))))
		   (if (eq span 1)
		       (if (math-evenp na)
			   (math-make-intv (logior (nth 1 x) 2)
					   -1
					   (nth 3 int))
			 (math-make-intv (logior (nth 1 x) 1)
					 (nth 2 int)
					 1))
		     int))
	       (list 'intv 3 -1 1)))))
	(t (calc-record-why 'scalarp x)
	   (list 'calcFunc-cos x)))
)
(fset 'calcFunc-cos (symbol-function 'math-cos))

(defun math-sin-cos (x)   ; [V N] [Public]
  (if (Math-scalarp x)
      (math-with-extra-prec 2
	(let ((sc (math-sin-cos-raw (math-to-radians (math-float x)))))
	  (list 'vec (cdr sc) (car sc))))    ; the vector [cos, sin]
    (list 'vec (math-sin x) (math-cos x)))
)
(fset 'calcFunc-sincos (symbol-function 'math-sin-cos))

(defun math-tan (x)   ; [N N] [Public]
  (cond ((Math-scalarp x)
	 (math-with-extra-prec 2
	   (math-tan-raw (math-to-radians (math-float x)))))
	((eq (car x) 'sdev)
	 (if (math-constp x)
	     (math-with-extra-prec 2
	       (let* ((xx (math-to-radians (math-float (nth 1 x))))
		      (xs (math-to-radians (math-float (nth 2 x))))
		      (sc (math-sin-cos-raw xx)))
		 (if (math-zerop (cdr sc))
		     (progn
		       (calc-record-why "Division by zero")
		       (list 'calcFunc-tan x))
		   (math-make-sdev (math-div-float (car sc) (cdr sc))
				   (math-div-float xs (math-sqr (cdr sc)))))))
	   (math-make-sdev (math-tan (nth 1 x))
			   (math-div (nth 2 x)
				     (math-sqr (math-cos (nth 1 x)))))))
	((and (eq (car x) 'intv) (math-constp x))
	 (or (math-with-extra-prec 2
	       (let* ((xx (math-to-radians (math-float x)))
		      (na (math-floor (math-div (math-sub (nth 2 xx)
							  (math-pi-over-2))
						(math-pi))))
		      (nb (math-floor (math-div (math-sub (nth 3 xx)
							  (math-pi-over-2))
						(math-pi)))))
		 (and (equal na nb)
		      (math-sort-intv (nth 1 x)
				      (math-tan-raw (nth 2 xx))
				      (math-tan-raw (nth 3 xx))))))
	     (progn
	       (calc-record-why "Infinite interval" x)
	       (list 'calcFunc-tan x))))
	(t (calc-record-why 'scalarp x)
	   (list 'calcFunc-tan x)))
)
(fset 'calcFunc-tan (symbol-function 'math-tan))

(defun math-sin-raw (x)   ; [N N]
  (cond ((eq (car-safe x) 'cplx)
	 (let* ((expx (math-exp-raw (nth 2 x)))
		(expmx (math-div-float '(float 1 0) expx))
		(sc (math-sin-cos-raw (nth 1 x))))
	   (list 'cplx
		 (math-mul-float (car sc)
				 (math-mul-float (math-sub expx expmx)
						 '(float 5 -1)))
		 (math-mul-float (cdr sc)
				 (math-mul-float (math-add-float expx expmx)
						 '(float 5 -1))))))
	((eq (car-safe x) 'polar)
	 (math-polar (math-sin-raw (math-complex x))))
	((Math-integer-negp (nth 1 x))
	 (math-neg-float (math-sin-raw (math-neg-float x))))
	((math-lessp-float '(float 7 0) x)  ; avoid inf loops due to roundoff
	 (math-sin-raw (math-mod x (math-two-pi))))
	(t (math-sin-raw-2 x x)))
)

(defun math-cos-raw (x)   ; [N N]
  (if (eq (car-safe x) 'polar)
      (math-polar (math-cos-raw (math-complex x)))
    (math-sin-raw (math-sub-float (math-pi-over-2) x)))
)

;;; This could use a smarter method:  Reduce x as in math-sin-raw, then
;;;   compute either sin(x) or cos(x), whichever is smaller, and compute
;;;   the other using the identity sin(x)^2 + cos(x)^2 = 1.
(defun math-sin-cos-raw (x)   ; [F.F F]  (result is (sin x . cos x))
  (cons (math-sin-raw x) (math-cos-raw x))
)

(defun math-tan-raw (x)   ; [N N]
  (cond ((eq (car-safe x) 'cplx)
	 (let* ((x (math-mul-float x '(float 2 0)))
		(expx (math-exp-raw (nth 2 x)))
		(expmx (math-div-float '(float 1 0) expx))
		(sc (math-sin-cos-raw (nth 1 x)))
		(d (math-add-float (cdr sc)
				   (math-mul-float (math-add-float expx expmx)
						   '(float 5 -1)))))
	   (and (not (eq (nth 1 d) 0))
		(list 'cplx
		      (math-div-float (car sc) d)
		      (math-div-float (math-mul-float (math-add expx expmx)
						      '(float 5 -1)) d)))))
	((eq (car-safe x) 'polar)
	 (math-polar (math-tan-raw (math-complex x))))
	(t
	 (let ((sc (math-sin-cos-raw x)))
	   (if (eq (nth 1 (cdr sc)) 0)
	       (math-reject-arg x "Division by zero")
	     (math-div-float (car sc) (cdr sc))))))
)

(defun math-sin-raw-2 (x orgx)   ; This avoids poss of inf recursion.  [F F]
  (let ((xmpo2 (math-sub-float (math-pi-over-2) x)))
    (cond ((Math-integer-negp (nth 1 xmpo2))
	   (math-neg-float (math-sin-raw-2 (math-sub-float x (math-pi))
					   orgx)))
	  ((math-lessp-float (math-pi-over-4) x)
	   (math-cos-raw-2 xmpo2 orgx))
	  ((math-lessp-float x (math-neg (math-pi-over-4)))
	   (math-neg (math-cos-raw-2 (math-add (math-pi-over-2) x) orgx)))
	  ((math-nearly-zerop-float x orgx) '(float 0 0))
	  (calc-symbolic-mode (signal 'inexact-result nil))
	  (t (math-sin-series x 6 4 x (math-neg-float (math-sqr-float x))))))
)

(defun math-cos-raw-2 (x orgx)   ; [F F]
  (cond ((math-nearly-zerop-float x orgx) '(float 1 0))
	(calc-symbolic-mode (signal 'inexact-result nil))
	(t (let ((xnegsqr (math-neg-float (math-sqr-float x))))
	     (math-sin-series
	      (math-add-float '(float 1 0)
			      (math-mul-float xnegsqr '(float 5 -1)))
	      24 5 xnegsqr xnegsqr))))
)

(defun math-sin-series (sum nfac n x xnegsqr)
  (math-working "sin" sum)
  (let* ((nextx (math-mul-float x xnegsqr))
	 (nextsum (math-add-float sum (math-div-float nextx
						      (math-float nfac)))))
    (if (math-nearly-equal-float sum nextsum)
	sum
      (math-sin-series nextsum (math-mul nfac (* n (1+ n)))
		       (+ n 2) nextx xnegsqr)))
)


;;; Inverse sine, cosine, tangent.

(defun math-arcsin (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (math-from-radians (math-arcsin-raw (math-float x)))))
	((and (eq (car x) 'sdev)
	      (or (not (math-constp (nth 1 x)))
		  (not (Math-lessp 1 (math-abs (nth 1 x))))))
	 (math-make-sdev (math-arcsin (nth 1 x))
			 (math-from-radians
			  (math-div (nth 2 x)
				    (math-sqrt
				     (math-sub 1 (math-sqr (nth 1 x))))))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-sort-intv (nth 1 x)
			 (math-arcsin (nth 2 x))
			 (math-arcsin (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-arcsin x)))
)
(fset 'calcFunc-arcsin (symbol-function 'math-arcsin))

(defun math-arccos (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (math-from-radians (math-arccos-raw (math-float x)))))
	((and (eq (car x) 'sdev)
	      (or (not (math-constp (nth 1 x)))
		  (not (Math-lessp 1 (math-abs (nth 1 x))))))
	 (math-make-sdev (math-arccos (nth 1 x))
			 (math-from-radians
			  (math-div (nth 2 x)
				    (math-sqrt
				     (math-sub 1 (math-sqr (nth 1 x))))))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-sort-intv (nth 1 x)
			 (math-arccos (nth 2 x))
			 (math-arccos (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-arccos x)))
)
(fset 'calcFunc-arccos (symbol-function 'math-arccos))

(defun math-arctan (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (math-from-radians (math-arctan-raw (math-float x)))))
	((eq (car x) 'sdev)
	 (math-make-sdev (math-arctan (nth 1 x))
			 (math-from-radians
			  (math-div (nth 2 x)
				    (math-add 1 (math-sqr (nth 1 x)))))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-sort-intv (nth 1 x)
			 (math-arctan (nth 2 x))
			 (math-arctan (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-arctan x)))
)
(fset 'calcFunc-arctan (symbol-function 'math-arctan))

(defun math-arcsin-raw (x)   ; [N N]
  (let ((a (math-sqrt-raw (math-sub '(float 1 0) (math-sqr x)))))
    (if (or (memq (car-safe x) '(cplx polar))
	    (memq (car-safe a) '(cplx polar)))
	(math-with-extra-prec 2   ; use extra precision for difficult case
	  (math-mul '(cplx 0 -1)
		    (math-ln-raw (math-add (math-mul '(cplx 0 1) x) a))))
      (math-arctan2-raw x a)))
)

(defun math-arccos-raw (x)   ; [N N]
  (math-sub (math-pi-over-2) (math-arcsin-raw x))
)

(defun math-arctan-raw (x)   ; [N N]
  (cond ((memq (car-safe x) '(cplx polar))
	 (math-with-extra-prec 2   ; extra-extra
	   (math-mul '(cplx 0 -1)
		     (math-ln-raw (math-mul
				   (math-add 1 (math-mul '(cplx 0 1) x))
				   (math-sqrt-raw
				    (math-div 1 (math-add
						 1 (math-sqr x)))))))))
	((Math-integer-negp (nth 1 x))
	 (math-neg-float (math-arctan-raw (math-neg-float x))))
	((math-zerop x) x)
	((math-equal-int x 1) (math-pi-over-4))
	((math-equal-int x -1) (math-neg (math-pi-over-4)))
	(calc-symbolic-mode (signal 'inexact-result nil))
	((math-lessp-float '(float 414214 -6) x)  ; if x > sqrt(2) - 1, reduce
	 (if (math-lessp-float '(float 1 0) x)
	     (math-sub-float (math-mul-float (math-pi) '(float 5 -1))
			     (math-arctan-raw (math-div-float '(float 1 0) x)))
	   (math-sub-float (math-mul-float (math-pi) '(float 25 -2))
			   (math-arctan-raw (math-div-float
					     (math-sub-float '(float 1 0) x)
					     (math-add-float '(float 1 0)
							     x))))))
	(t (math-arctan-series x 3 x (math-neg-float (math-sqr-float x)))))
)

(defun math-arctan-series (sum n x xnegsqr)
  (math-working "arctan" sum)
  (let* ((nextx (math-mul-float x xnegsqr))
	 (nextsum (math-add-float sum (math-div-float nextx (math-float n)))))
    (if (math-nearly-equal-float sum nextsum)
	sum
      (math-arctan-series nextsum (+ n 2) nextx xnegsqr)))
)

(defun math-arctan2 (y x)   ; [F R R] [Public]
  (if (Math-anglep y)
      (if (Math-anglep x)
	  (math-with-extra-prec 2
	    (math-from-radians (math-arctan2-raw (math-float y)
						 (math-float x))))
	(calc-record-why 'anglep x)
	(list 'calcFunc-arctan2 y x))
    (calc-record-why 'anglep y)
    (list 'calcFunc-arctan2 y x))
)
(fset 'calcFunc-arctan2 (symbol-function 'math-arctan2))

(defun math-arctan2-raw (y x)   ; [F R R]
  (cond ((math-zerop y)
	 (if (math-negp x) (math-pi) 0))
	((math-zerop x)
	 (if (math-posp y)
	     (math-pi-over-2)
	   (math-neg (math-pi-over-2))))
	((math-posp x)
	 (math-arctan-raw (math-div-float y x)))
	((math-posp y)
	 (math-add-float (math-arctan-raw (math-div-float y x))
			 (math-pi)))
	(t
	 (math-sub-float (math-arctan-raw (math-div-float y x))
			 (math-pi))))
)

(defun math-arc-sin-cos (x)   ; [V N] [Public]
  (if (and (Math-vectorp x)
	   (= (length x) 3))
      (math-arctan2 (nth 2 x) (nth 1 x))
    (math-reject-arg x "Two-element vector expected"))
)
(fset 'calcFunc-arcsincos (symbol-function 'math-arc-sin-cos))



;;; Exponential function.

(defun math-exp (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2 (math-exp-raw (math-float x))))
	((eq (car-safe x) 'sdev)
	 (let ((ex (math-exp (nth 1 x))))
	   (math-make-sdev ex (math-mul (nth 2 x) ex))))
	((eq (car-safe x) 'intv)
	 (math-make-intv (nth 1 x) (math-exp (nth 2 x)) (math-exp (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-exp x)))
)
(fset 'calcFunc-exp (symbol-function 'math-exp))

(defun math-exp-minus-1 (x)   ; [N N] [Public]
  (cond ((math-zerop x) '(float 0 0))
	(calc-symbolic-mode (signal 'inexact-result nil))
	((Math-numberp x)
	 (math-with-extra-prec 2
	   (let ((x (math-float x)))
	     (if (and (eq (car x) 'float)
		      (math-lessp-float x '(float 1 0))
		      (math-lessp-float '(float -1 0) x))
		 (math-exp-minus-1-raw x)
	       (math-add (math-exp-raw x) -1)))))
	((eq (car-safe x) 'sdev)
	 (if (math-constp x)
	     (let ((ex (math-exp-minus-1 (nth 1 x))))
	       (math-make-sdev ex (math-mul (nth 2 x) (math-add ex 1))))
	   (math-make-sdev (math-exp-minus-1 (nth 1 x))
			   (math-mul (nth 2 x) (math-exp (nth 1 x))))))
	((eq (car-safe x) 'intv)
	 (math-make-intv (nth 1 x)
			 (math-exp-minus-1 (nth 2 x))
			 (math-exp-minus-1 (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-expm1 x)))
)
(fset 'calcFunc-expm1 (symbol-function 'math-exp-minus-1))

(defun math-exp10 (x)   ; [N N] [Public]
  (math-pow '(float 1 1) x)
)
(fset 'calcFunc-exp10 (symbol-function 'math-exp10))

(defun math-exp-raw (x)   ; [N N]
  (cond ((math-zerop x) '(float 1 0))
	(calc-symbolic-mode (signal 'inexact-result nil))
	((eq (car x) 'cplx)
	 (let ((expx (math-exp-raw (nth 1 x)))
	       (sc (math-sin-cos-raw (nth 2 x))))
	   (list 'cplx
		 (math-mul-float expx (cdr sc))
		 (math-mul-float expx (car sc)))))
	((eq (car x) 'polar)
	 (let ((xc (math-complex x)))
	   (list 'polar
		 (math-exp-raw (nth 1 x))
		 (nth 2 x))))
	((or (math-lessp-float '(float 5 -1) x)
	     (math-lessp-float x '(float -5 -1)))
	 (let* ((two-x (math-mul-float x '(float 2 0)))
		(hint (math-scale-int (nth 1 two-x) (nth 2 two-x)))
		(hfrac (math-sub-float x (math-mul-float (math-float hint)
							 '(float 5 -1)))))
	   (math-mul-float (math-ipow (math-sqrt-e) hint)
			   (math-add-float '(float 1 0)
					   (math-exp-minus-1-raw hfrac)))))
	(t (math-add-float '(float 1 0) (math-exp-minus-1-raw x))))
)

(defun math-exp-minus-1-raw (x)   ; [F F]
  (math-exp-series x 2 3 x x)
)

(defun math-exp-series (sum nfac n xpow x)
  (math-working "exp" sum)
  (let* ((nextx (math-mul-float xpow x))
	  (nextsum (math-add-float sum (math-div-float nextx
						       (math-float nfac)))))
     (if (math-nearly-equal-float sum nextsum)
	 sum
       (math-exp-series nextsum (math-mul nfac n) (1+ n) nextx x)))
)



;;; Logarithms.

(defun math-ln (x)   ; [N N] [Public]
  (cond ((math-zerop x)
	 (math-reject-arg x "Logarithm of zero"))
	((Math-numberp x)
	 (math-with-extra-prec 2 (math-ln-raw (math-float x))))
	((and (eq (car-safe x) 'sdev)
	      (or (not (math-constp (nth 1 x)))
		  (math-posp (nth 1 x))))
	 (math-make-sdev (math-ln (nth 1 x))
			 (math-div (nth 2 x) (nth 1 x))))
	((and (eq (car-safe x) 'intv) (Math-posp (nth 2 x)))
	 (math-make-intv (nth 1 x) (math-ln (nth 2 x)) (math-ln (nth 3 x))))
	((equal x '(var e var-e))
	 1)
	((and (eq (car-safe x) '^)
	      (equal (nth 1 x) '(var e var-e)))
	 (nth 2 x))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-ln x)))
)
(fset 'calcFunc-ln (symbol-function 'math-ln))

(defun math-log10 (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (let ((xf (math-float x)))
	     (if (eq (nth 1 xf) 0)
		 (math-reject-arg x "Logarithm of zero"))
	     (if (Math-integer-posp (nth 1 xf))
		 (if (eq (nth 1 xf) 1)    ; log10(1*10^n) = n
		     (math-float (nth 2 xf))
		   (let ((xdigs (1- (math-numdigs (nth 1 xf)))))
		     (math-add-float
		      (math-div-float (math-ln-raw-2
				       (list 'float (nth 1 xf) (- xdigs)))
				      (math-ln-10))
		      (math-float (+ (nth 2 xf) xdigs)))))
	       (math-div (math-ln xf) (math-ln-10))))))
	((and (eq (car-safe x) 'sdev)
	      (or (not (math-constp (nth 1 x)))
		  (math-posp (nth 1 x))))
	 (math-make-sdev (math-log10 (nth 1 x))
			 (math-div (nth 2 x)
				   (math-mul (nth 1 x) (math-ln 10)))))
	((and (eq (car-safe x) 'intv) (Math-posp (nth 2 x)))
	 (math-make-intv (nth 1 x)
			 (math-log10 (nth 2 x))
			 (math-log10 (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-log10 x)))
)
(fset 'calcFunc-log10 (symbol-function 'math-log10))

(defun calcFunc-pow10 (x)
  (math-pow '(float 1 1) x)
)

(defun math-log (x b)   ; [N N N] [Public]
  (cond ((or (eq b 10) (equal b '(float 1 1)))
	 (math-log10 x))
	((math-zerop x)
	 (math-reject-arg x "Logarithm of zero"))
	((math-zerop b)
	 (math-reject-arg b "Logarithm of zero"))
	((and (Math-numberp x) (Math-numberp b))
	 (math-with-extra-prec 2
	   (math-div (math-ln-raw (math-float x))
		     (math-log-base-raw b))))
	((and (eq (car-safe x) 'sdev)
	      (or (not (math-constp (nth 1 x)))
		  (math-posp (nth 1 x)))
	      (Math-numberp b))
	 (math-make-sdev (math-log (nth 1 x) b)
			 (math-div (nth 2 x)
				   (math-mul (nth 1 x)
					     (math-log-base-raw b)))))
	((and (eq (car-safe x) 'intv) (Math-posp (nth 2 x)))
	 (math-make-intv (nth 1 x)
			 (math-log (nth 2 x) b)
			 (math-log (nth 3 x) b)))
	(t (if (Math-numberp b)
	       (calc-record-why 'numberp x)
	     (calc-record-why 'numberp b))
	   (list 'calcFunc-log x b)))
)

(defun calcFunc-log (x &optional b)
  (if b
      (if (or (eq b 10) (equal b '(float 1 1)))
	  (math-normalize (list 'calcFunc-log10 x))
	(if (equal b '(var e var-e))
	    (math-normalize (list 'calcFunc-ln x))
	  (math-log x b)))
    (math-normalize (list 'calcFunc-ln x)))
)
(defun calcFunc-ilog (x &optional b)
  (if b
      (if (equal b '(var e var-e))
	  (math-normalize (list 'calcFunc-exp x))
	(math-pow b x))
    (math-normalize (list 'calcFunc-exp x)))
)


(defun math-log-base-raw (b)   ; [N N]
  (if (not (and (equal (car math-log-base-cache) b)
		(eq (nth 1 math-log-base-cache) calc-internal-prec)))
      (setq math-log-base-cache (list b calc-internal-prec
				      (math-ln-raw (math-float b)))))
  (nth 2 math-log-base-cache)
)
(setq math-log-base-cache nil)

(defun math-ln-plus-1 (x)   ; [N N] [Public]
  (cond ((Math-equal-int x -1) (math-reject-arg x "Logarithm of zero"))
	((math-zerop x) '(float 0 0))
	(calc-symbolic-mode (signal 'inexact-result nil))
	((Math-numberp x)
	 (math-with-extra-prec 2
	   (let ((x (math-float x)))
	     (if (and (eq (car x) 'float)
		      (math-lessp-float x '(float 5 -1))
		      (math-lessp-float '(float -5 -1) x))
		 (math-ln-plus-1-raw x)
	       (math-ln-raw (math-add-float x '(float 1 0)))))))
	((and (eq (car-safe x) 'sdev)
	      (or (not (math-constp (nth 1 x)))
		  (Math-lessp -1 (nth 1 x))))
	 (math-make-sdev (math-ln-plus-1 (nth 1 x))
			 (math-div (nth 2 x) (math-add (nth 1 x) 1))))
	((and (eq (car-safe x) 'intv) (Math-posp (nth 2 x)))
	 (math-make-intv (nth 1 x)
			 (math-ln-plus-1 (nth 2 x))
			 (math-ln-plus-1 (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-lnp1 x)))
)
(fset 'calcFunc-lnp1 (symbol-function 'math-ln-plus-1))

(defun math-ln-raw (x)    ; [N N] --- must be float format!
  (cond ((eq (car-safe x) 'cplx)
	 (list 'cplx
	       (math-mul-float (math-ln-raw
				(math-add-float (math-sqr-float (nth 1 x))
						(math-sqr-float (nth 2 x))))
			       '(float 5 -1))
	       (math-arctan2-raw (nth 2 x) (nth 1 x))))
	((eq (car x) 'polar)
	 (math-polar (list 'cplx
			   (math-ln-raw (nth 1 x))
			   (nth 2 x))))
	((Math-equal-int x 1)
	 '(float 0 0))
	(calc-symbolic-mode (signal 'inexact-result nil))
	((math-posp (nth 1 x))    ; positive and real
	 (let ((xdigs (1- (math-numdigs (nth 1 x)))))
	   (math-add-float (math-ln-raw-2 (list 'float (nth 1 x) (- xdigs)))
			   (math-mul-float (math-float (+ (nth 2 x) xdigs))
					   (math-ln-10)))))
	((math-zerop x)
	 (error "Logarithm of zero"))
	((eq calc-complex-mode 'polar)    ; negative and real
	 (math-polar
	  (list 'cplx   ; negative and real
		(math-ln-raw (math-neg-float x))
		(math-pi))))
	(t (list 'cplx   ; negative and real
		 (math-ln-raw (math-neg-float x))
		 (math-pi))))
)

(defun math-ln-raw-2 (x)    ; [F F]
  (cond ((math-lessp-float '(float 14 -1) x)
	 (math-add-float (math-ln-raw-2 (math-mul-float x '(float 5 -1)))
			 (math-ln-2)))
	(t    ; now .7 < x <= 1.4
	 (math-ln-raw-3 (math-div-float (math-sub-float x '(float 1 0))
					(math-add-float x '(float 1 0))))))
)

(defun math-ln-raw-3 (x)   ; [F F]
  (math-mul-float (math-ln-raw-series x 3 x (math-sqr-float x))
		  '(float 2 0))
)

;;; Compute ln((1+x)/(1-x))
(defun math-ln-raw-series (sum n x xsqr)
  (math-working "log" sum)
  (let* ((nextx (math-mul-float x xsqr))
	 (nextsum (math-add-float sum (math-div-float nextx (math-float n)))))
    (if (math-nearly-equal-float sum nextsum)
	sum
      (math-ln-raw-series nextsum (+ n 2) nextx xsqr)))
)

(defun math-ln-plus-1-raw (x)
  (math-lnp1-series x 2 x (math-neg x))
)

(defun math-lnp1-series (sum n xpow x)
  (math-working "lnp1" sum)
  (let* ((nextx (math-mul-float xpow x))
	 (nextsum (math-add-float sum (math-div-float nextx (math-float n)))))
    (if (math-nearly-equal-float sum nextsum)
	sum
      (math-lnp1-series nextsum (1+ n) nextx x)))
)

(math-defcache math-ln-10 (float (bigpos 018 684 045 994 092 585 302 2) -21)
  (math-ln-raw-2 '(float 1 1)))

(math-defcache math-ln-2 (float (bigpos 417 309 945 559 180 147 693) -21)
  (math-ln-raw-3 (math-float '(frac 1 3))))



;;; Hyperbolic functions.

(defun math-sinh (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (let ((expx (math-exp-raw (math-float x))))
	     (math-mul (math-add expx (math-div -1 expx)) '(float 5 -1)))))
	((eq (car-safe x) 'sdev)
	 (math-make-sdev (math-sinh (nth 1 x))
			 (math-mul (nth 2 x) (math-cosh (nth 1 x)))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-sort-intv (nth 1 x)
			 (math-sinh (nth 2 x))
			 (math-sinh (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-sinh x)))
)
(fset 'calcFunc-sinh (symbol-function 'math-sinh))

(defun math-cosh (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (let ((expx (math-exp-raw (math-float x))))
	     (math-mul (math-add expx (math-div 1 expx)) '(float 5 -1)))))
	((eq (car-safe x) 'sdev)
	 (math-make-sdev (math-cosh (nth 1 x))
			 (math-mul (nth 2 x)
				   (math-abs (math-sinh (nth 1 x))))))
	((and (eq (car x) 'intv) (math-constp x))
	 (setq x (math-abs x))
	 (math-sort-intv (nth 1 x)
			 (math-cosh (nth 2 x))
			 (math-cosh (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-cosh x)))
)
(fset 'calcFunc-cosh (symbol-function 'math-cosh))

(defun math-tanh (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (let* ((expx (math-exp (math-float x)))
		  (expmx (math-div 1 expx)))
	     (math-div (math-sub expx expmx)
		       (math-add expx expmx)))))
	((eq (car-safe x) 'sdev)
	 (math-make-sdev (math-tanh (nth 1 x))
			 (math-div (nth 2 x)
				   (math-sqr (math-cosh (nth 1 x))))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-sort-intv (nth 1 x)
			 (math-tanh (nth 2 x))
			 (math-tanh (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-tanh x)))
)
(fset 'calcFunc-tanh (symbol-function 'math-tanh))

(defun math-arcsinh (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (math-ln-raw (math-add x (math-sqrt-raw (math-add (math-sqr x)
							     '(float 1 0)))))))
	((eq (car-safe x) 'sdev)
	 (math-make-sdev (math-arcsinh (nth 1 x))
			 (math-div (nth 2 x)
				   (math-sqrt
				    (math-add (math-sqr (nth 1 x)) 1)))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-sort-intv (nth 1 x)
			 (math-arcsinh (nth 2 x))
			 (math-arcsinh (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-arcsinh x)))
)
(fset 'calcFunc-arcsinh (symbol-function 'math-arcsinh))

(defun math-arccosh (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (if (or t    ; need to do this even in the real case!
		   (memq (car-safe x) '(cplx polar)))
	       (let ((xp1 (math-add 1 x)))    ; this gets the branch cuts right
		 (math-ln-raw
		  (math-add x (math-mul xp1
					(math-sqrt-raw (math-div (math-sub
								  x
								  '(float 1 0))
								 xp1))))))
	     (math-ln-raw
	      (math-add x (math-sqrt-raw (math-add (math-sqr x)
						   '(float -1 0))))))))
	((and (eq (car-safe x) 'sdev)
	      (or (not (math-constp (nth 1 x)))
		  (not (Math-lessp (nth 1 x) 1))))
	 (math-make-sdev (math-arccosh (nth 1 x))
			 (math-div (nth 2 x)
				   (math-sqrt
				    (math-add (math-sqr (nth 1 x)) -1)))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-sort-intv (nth 1 x)
			 (math-arccosh (nth 2 x))
			 (math-arccosh (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-arccosh x)))
)
(fset 'calcFunc-arccosh (symbol-function 'math-arccosh))

(defun math-arctanh (x)   ; [N N] [Public]
  (cond ((Math-numberp x)
	 (math-with-extra-prec 2
	   (if (memq (car-safe x) '(cplx polar))
	       (math-ln-raw
		(math-mul (math-add 1 x)
			  (math-sqrt-raw
			   (math-div '(float 1 0) (math-sub 1 (math-sqr x))))))
	     (math-mul (math-ln-raw (math-div (math-add '(float 1 0) x)
					      (math-sub 1 x)))
		       '(float 5 -1)))))
	((and (eq (car-safe x) 'sdev)
	      (or (not (math-constp (nth 1 x)))
		  (Math-lessp (math-abs (nth 1 x)) 1)))
	 (math-make-sdev (math-arctanh (nth 1 x))
			 (math-div (nth 2 x)
				   (math-sub 1 (math-sqr (nth 1 x))))))
	((and (eq (car x) 'intv) (math-constp x))
	 (math-sort-intv (nth 1 x)
			 (math-arctanh (nth 2 x))
			 (math-arctanh (nth 3 x))))
	(t (calc-record-why 'numberp x)
	   (list 'calcFunc-arctanh x)))
)
(fset 'calcFunc-arctanh (symbol-function 'math-arctanh))


;;; Convert A from HMS or degrees to radians.
(defun math-deg-to-rad (a)   ; [R R] [Public]
  (cond ((or (Math-numberp a)
	     (eq (car a) 'intv))
	 (math-mul a (math-pi-over-180)))
	((eq (car a) 'hms)
	 (math-from-hms a 'rad))
	((eq (car a) 'sdev)
	 (math-make-sdev (math-deg-to-rad (nth 1 a))
			 (math-deg-to-rad (nth 2 a))))
	(t (list 'calcFunc-rad a)))
)
(fset 'calcFunc-rad (symbol-function 'math-deg-to-rad))

;;; Convert A from HMS or radians to degrees.
(defun math-rad-to-deg (a)   ; [R R] [Public]
  (cond ((or (Math-numberp a)
	     (eq (car a) 'intv))
	 (math-div a (math-pi-over-180)))
	((eq (car a) 'hms)
	 (math-from-hms a 'deg))
	((eq (car a) 'sdev)
	 (math-make-sdev (math-rad-to-deg (nth 1 a))
			 (math-rad-to-deg (nth 2 a))))
	(t (list 'calcFunc-deg a)))
)
(fset 'calcFunc-deg (symbol-function 'math-rad-to-deg))




;;;; [calc-arith.el]

;;;; Number theory.

(defun calcFunc-idiv (a b)   ; [I I I] [Public]
  (cond ((and (Math-natnump a) (Math-natnump b) (not (eq b 0)))
	 (math-quotient a b))
	((Math-realp a)
	 (if (Math-realp b)
	     (let ((calc-prefer-frac t))
	       (math-floor (math-div a b)))
	   (math-reject-arg b 'realp)))
	((eq (car-safe a) 'hms)
	 (if (eq (car-safe b) 'hms)
	     (let ((calc-prefer-frac t))
	       (math-floor (math-div a b)))
	   (math-reject-arg b 'hmsp)))
	((and (or (eq (car-safe a) 'intv) (Math-realp a))
	      (or (eq (car-safe b) 'intv) (Math-realp b)))
	 (math-floor (math-div a b)))
	(t (math-reject-arg a 'anglep)))
)

;;;; [calc-frac.el]

(defun calcFunc-fdiv (a b)   ; [R I I] [Public]
  (if (Math-num-integerp a)
      (if (Math-num-integerp b)
	  (if (Math-zerop b)
	      (math-reject-arg a "Division by zero")
	    (math-make-frac (math-trunc a) (math-trunc b)))
	(math-reject-arg b 'integerp))
    (math-reject-arg a 'integerp))
)

;;;; [calc-comb.el]

(defun math-lcm (a b)
  (let ((g (math-gcd a b)))
    (if (Math-numberp g)
	(math-div (math-mul a b) g)
      (list 'calcFunc-lcm a b)))
)
(fset 'calcFunc-lcm (symbol-function 'math-lcm))

(defun math-extended-gcd (a b)   ; Knuth section 4.5.2
  (cond
   ((not (Math-integerp a))
    (if (Math-messy-integerp a)
	(math-extended-gcd (math-trunc a) b)
      (calc-record-why 'integerp a)
      (list 'calcFunc-egcd a b)))
   ((not (Math-integerp b))
    (if (Math-messy-integerp b)
	(math-extended-gcd a (math-trunc b))
      (calc-record-why 'integerp b)
      (list 'calcFunc-egcd a b)))
   (t
    (let ((u1 1) (u2 0) (u3 a)
	  (v1 0) (v2 1) (v3 b)
	  t1 t2 q)
      (while (not (eq v3 0))
	(setq q (math-idivmod u3 v3)
	      t1 (math-sub u1 (math-mul v1 (car q)))
	      t2 (math-sub u2 (math-mul v2 (car q)))
	      u1 v1  u2 v2  u3 v3
	      v1 t1  v2 t2  v3 (cdr q)))
      (list 'vec u3 u1 u2))))
)
(fset 'calcFunc-egcd (symbol-function 'math-extended-gcd))


;;; Factorial and related functions.

(defun math-factorial (n)   ; [I I] [F F] [Public]
  (let (temp)
    (cond ((Math-integer-negp n) (list 'calcFunc-fact n))
	  ((Math-zerop n) 1)
	  ((integerp n) (math-factorial-iter (1- n) 2 1))
	  ((and (math-messy-integerp n)
		(Math-lessp (setq temp (math-trunc n)) 100))
	   (if (natnump temp)
	       (math-with-extra-prec 1
		 (math-factorial-iter (1- temp) 2 '(float 1 0)))
	     (list 'calcFunc-fact max)))
	  ((math-realp n)
	   (math-with-extra-prec 3
	     (math-gammap1-raw (math-float n))))
	  (t (calc-record-why 'realp n)
	     (list 'calcFunc-fact n))))
)
(fset 'calcFunc-fact (symbol-function 'math-factorial))

(defun math-factorial-iter (count n f)
  (if (= (% n 5) 1)
      (math-working (format "factorial(%d)" (1- n)) f))
  (if (> count 0)
      (math-factorial-iter (1- count) (1+ n) (math-mul n f))
    f)
)

(math-defcache math-sqrt-two-pi nil
  (math-sqrt (math-two-pi)))

(defun math-gammap1-raw (x)   ; compute gamma(1 + x)
  (cond ((math-lessp-float x '(float 1 1))
	 (if (math-lessp-float x '(float -10 0))
	     (setq x (math-neg-float
		      (math-div-float
		       (math-pi)
		       (math-mul-float (math-gammap1-raw
					(math-add-float (math-neg-float x)
							'(float -1 0)))
				       (math-sin-raw
					(math-mul (math-pi) x)))))))
	 (let ((xplus1 (math-add-float x '(float 1 0))))
	   (math-div-float (math-gammap1-raw xplus1) xplus1)))
	(t   ; x now >= 10.0
	 (let ((xinv (math-div 1 x))
	       (lnx (math-ln-raw x)))
	   (math-mul (math-sqrt-two-pi)
		     (math-exp-raw
		      (math-gamma-series
		       (math-sub (math-mul (math-add x '(float 5 -1))
					   lnx)
				 x)
		       xinv
		       (math-sqr xinv)
		       2))))))
)

(defun calcFunc-gamma (x)
  (calcFunc-fact (math-add x -1))
)

(defun math-gamma-series (sum x xinvsqr n)
  (math-working "gamma" sum)
  (let* ((bn (math-bernoulli-number n))   ; this will always be a "frac" form.
	 (next (math-add-float
		sum
		(math-mul-float (math-div-float (math-float (nth 1 bn))
						(math-float (* (nth 2 bn)
							       (* n (1- n)))))
				x))))
    (if (math-nearly-equal-float sum next)
	next
      (if (= n 24)
	  (progn
	    (calc-record-why "Gamma computation stopped early, not all digits may be valid")
	    next)
	(math-gamma-series next (math-mul-float x xinvsqr) xinvsqr (+ n 2)))))
)

(defun math-bernoulli-number (n)
  (if (= n 1)
      '(frac -1 2)
    (if (= (% n 2) 1)
	0
      (aref '[ 1 (frac 1 6) (frac -1 30) (frac 1 42) (frac -1 30)
	       (frac 5 66) (frac -691 2730) (frac 7 6) (frac -3617 510)
	       (frac 43867 798) (frac -174611 330) (frac 854513 138)
	       (frac (bigneg 91 364 236) 2730) ]
	    (/ n 2))))
)
;;; To come up with more, we could use this rule:
;;;   Bn = n! bn
;;;   bn = - sum_k=0^n-1 bk / (n-k+1)!

(defun math-double-factorial (n)   ; [I I] [F F] [Public]
  (cond ((Math-integer-negp n) (list 'calcFunc-dfact n))
	((Math-zerop n) 1)
	((integerp n) (math-double-factorial-iter n (+ 2 (% n 2)) 1))
	((math-messy-integerp n)
	 (let ((temp (math-trunc n)))
	   (if (natnump temp)
	       (math-with-extra-prec 1
		 (math-double-factorial-iter temp (+ 2 (% temp 2))
					     '(float 1 0)))
	     (list 'calcFunc-dfact max))))
	(t (calc-record-why 'natnump n)
	   (list 'calcFunc-dfact n)))
)
(fset 'calcFunc-dfact (symbol-function 'math-double-factorial))

(defun math-double-factorial-iter (max n f)
  (if (< (% n 10) 2)
      (math-working (format "dfact(%d)" (- n 2)) f))
  (if (<= n max)
      (math-double-factorial-iter max (+ n 2) (math-mul n f))
    f)
)

(defun math-permutations (n m)   ; [I I I] [F F F] [Public]
  (cond ((and (integerp n) (integerp m) (<= m n) (>= m 0))
	 (math-factorial-iter n (1+ (- n m)) 1))
	((or (not (math-num-integerp n))
	     (not (math-num-integerp m)))
	 (or (math-realp n) (math-reject-arg n 'realp))
	 (or (math-realp m) (math-reject-arg m 'realp))
	 (and (math-num-integerp n) (math-negp n) (math-reject-arg n 'range))
	 (and (math-num-integerp m) (math-negp m) (math-reject-arg m 'range))
	 (math-div (math-factorial n) (math-factorial m)))
	(t
	 (let ((tn (math-trunc n))
	       (tm (math-trunc m)))
	   (or (integerp tn) (math-reject-arg tn 'fixnump))
	   (or (integerp tm) (math-reject-arg tm 'fixnump))
	   (or (and (<= tm tn) (>= tm 0)) (math-reject-arg tm 'range))
	   (math-with-extra-prec 1
	     (math-factorial-iter tm (1+ (- tn tm)) '(float 1 0))))))
)
(fset 'calcFunc-perm (symbol-function 'math-permutations))

(defun math-choose (n m)   ; [I I I] [F F F] [Public]
  (cond ((and (integerp n) (integerp m) (<= m n) (>= m 0))
	 (if (> m (/ n 2))
	     (math-choose-iter (- n m) n 1 1)
	   (math-choose-iter m n 1 1)))
	((not (math-realp n))
	 (math-reject-arg n 'realp))
	((not (math-realp m))
	 (math-reject-arg m 'realp))
	((not (math-num-integerp m))
	 (if (and (math-num-integerp n) (math-negp n))
	     (list 'calcFunc-choose n m)
	   (math-div (math-factorial (math-float n))
		     (math-mul (math-factorial m)
			       (math-factorial (math-sub n m))))))
	((math-negp m) 0)
	((math-negp n)
	 (let ((val (math-choose (math-add (math-add n m) -1) m)))
	   (if (math-evenp (math-trunc m))
	       val
	     (math-neg val))))
	((and (math-num-integerp n)
	      (Math-lessp n m))
	 0)
	(t
	 (let ((tm (math-trunc m)))
	   (or (integerp tm) (math-reject-arg tm 'fixnump))
	   (if (> tm 100)
	       (math-div (math-factorial (math-float n))
			 (math-mul (math-factorial (math-float m))
				   (math-factorial (math-float
						    (math-sub n m)))))
	     (math-with-extra-prec 1
	       (math-choose-float-iter tm n 1 '(float 1 0)))))))
)
(fset 'calcFunc-choose (symbol-function 'math-choose))

(defun math-choose-iter (m n i c)
  (if (= (% i 5) 1)
      (math-working (format "choose(%d)" (1- i)) c))
  (if (<= i m)
      (math-choose-iter m (1- n) (1+ i)
			(math-quotient (math-mul c n) i))
    c)
)

(defun math-choose-float-iter (count n i c)
  (if (= (% i 5) 1)
      (math-working (format "choose(%d)" (1- i)) c))
  (if (> count 0)
      (math-choose-float-iter (1- count) (math-sub n 1) (1+ i)
			      (math-div (math-mul c n) i))
    c)
)


;;; Produce a random digit in the range 0..999.
;;; Avoid various pitfalls that may lurk in the built-in (random) function!
(defun math-random-digit ()
  (prog1
      (% (lsh (random math-first-random-flag) -4) 1000)
    (setq math-first-random-flag nil))
)
(setq math-first-random-flag t)

;;; Produce an N-digit random integer.
(defun math-random-digits (n)
  (cond ((<= n 6)
	 (math-scale-right (+ (* (math-random-digit) 1000) (math-random-digit))
			   (- 6 n)))
	(t (let* ((slop (% (- 900003 n) 3))
		  (i (/ (+ n slop) 3))
		  (digs nil))
	     (while (> i 0)
	       (setq digs (cons (math-random-digit) digs)
		     i (1- i)))
	     (math-normalize (math-scale-right (cons 'bigpos digs)
					       slop)))))
)

;;; Produce a uniformly-distributed random float 0 <= N < 1.
(defun math-random-float ()
  (math-make-float (math-random-digits calc-internal-prec)
		   (- calc-internal-prec))
)

;;; Produce a Gaussian-distributed random float with mean=0, sigma=1.
(defun math-gaussian-float ()
  (math-with-extra-prec 2
    (if (and math-gaussian-cache
	     (= (car math-gaussian-cache) calc-internal-prec))
	(prog1
	    (cdr math-gaussian-cache)
	  (setq math-gaussian-cache nil))
      (let* ((v1 (math-add (math-mul (math-random-float) 2) -1))
	     (v2 (math-add (math-mul (math-random-float) 2) -1))
	     (r (math-add (math-sqr v1) (math-sqr v2))))
	(while (or (not (Math-lessp r 1)) (math-zerop r))
	  (setq v1 (math-add (math-mul (math-random-float) 2) -1)
		v2 (math-add (math-mul (math-random-float) 2) -1)
		r (math-add (math-sqr v1) (math-sqr v2))))
	(let ((fac (math-sqrt (math-mul (math-div (math-ln r) r) -2))))
	  (setq math-gaussian-cache (cons calc-internal-prec
					  (math-mul v1 fac)))
	  (math-mul v2 fac)))))
)
(setq math-gaussian-cache nil)

;;; Produce a random integer or real 0 <= N < MAX.
(defun math-random (max)
  (cond ((Math-zerop max)
	 (math-gaussian-float))
	((Math-integerp max)
	 (let* ((digs (math-numdigs max))
		(r (math-random-digits (+ digs 3))))
	   (math-mod r max)))
	((Math-realp max)
	 (math-mul (math-random-float) max))
	((and (eq (car max) 'intv) (math-constp max)
	      (Math-lessp (nth 2 max) (nth 3 max)))
	 (if (math-floatp max)
	     (let ((val (math-add (math-mul (math-random-float)
					    (math-sub (nth 3 max) (nth 2 max)))
				  (nth 2 max))))
	       (if (or (and (memq (nth 1 max) '(0 1))      ; almost not worth
			    (math-equal val (nth 2 max)))  ;   checking!
		       (and (memq (nth 1 max) '(0 2))
			    (math-equal val (nth 3 max))))
		   (math-random max)
		 val))
	   (let ((lo (if (memq (nth 1 max) '(0 1))
			 (math-add (nth 2 max) 1) (nth 2 max)))
		 (hi (if (memq (nth 1 max) '(1 3))
			 (math-add (nth 3 max) 1) (nth 3 max))))
	     (if (Math-lessp lo hi)
		 (math-add (math-random (math-sub hi lo)) lo)
	       (math-reject-arg max "Empty interval")))))
	(t (math-reject-arg max 'realp)))
)
(fset 'calcFunc-random (symbol-function 'math-random))


;;; Check if the integer N is prime.  [X I]
;;; Return (nil) if non-prime,
;;;        (nil N) if non-prime with known factor N,
;;;        (nil unknown) if non-prime with no known factors,
;;;        (t) if prime,
;;;        (maybe N P) if probably prime (after N iters with probability P%)
(defun math-prime-test (n iters)
  (if (and (Math-vectorp n) (cdr n))
      (setq n (nth (1- (length n)) n)))
  (if (Math-messy-integerp n)
      (setq n (math-trunc n)))
  (let ((res))
    (while (> iters 0)
      (setq res
	    (cond ((and (integerp n) (<= n 5003))
		   (list (= (math-next-small-prime n) n)))
		  ((not (Math-integerp n))
		   (error "Argument must be an integer"))
		  ((Math-integer-negp n)
		   '(nil))
		  ((Math-natnum-lessp n '(bigpos 0 0 8))
		   (setq n (math-fixnum n))
		   (let ((i -1) v)
		     (while (and (> (% n (setq v (aref math-primes-table
						       (setq i (1+ i)))))
				    0)
				 (< (* v v) n)))
		     (if (= (% n v) 0)
			 (list nil v)
		       '(t))))
		  ((not (equal n (car math-prime-test-cache)))
		   (cond ((= (% (nth 1 n) 2) 0) '(nil 2))
			 ((= (% (nth 1 n) 5) 0) '(nil 5))
			 (t (let ((dig (cdr n)) (sum 0))
			      (while dig
				(if (cdr dig)
				    (setq sum (% (+ (+ sum (car dig))
						    (* (nth 1 dig) 1000))
						 111111)
					  dig (cdr (cdr dig)))
				  (setq sum (% (+ sum (car dig)) 111111)
					dig nil)))
			      (cond ((= (% sum 3) 0) '(nil 3))
				    ((= (% sum 7) 0) '(nil 7))
				    ((= (% sum 11) 0) '(nil 11))
				    ((= (% sum 13) 0) '(nil 13))
				    ((= (% sum 37) 0) '(nil 37))
				    (t
				     (setq math-prime-test-cache-k 1
					   math-prime-test-cache-q
					   (math-div2 n)
					   math-prime-test-cache-nm1
					   (math-add n -1))
				     (while (math-evenp
					     math-prime-test-cache-q)
				       (setq math-prime-test-cache-k
					     (1+ math-prime-test-cache-k)
					     math-prime-test-cache-q
					     (math-div2
					      math-prime-test-cache-q)))
				     (setq iters (1+ iters))
				     (list 'maybe
					   0
					   (math-sub
					    100
					    (math-div
					     '(float 232 0)
					     (math-numdigs n))))))))))
		  ((not (eq (car (nth 1 math-prime-test-cache)) 'maybe))
		   (nth 1 math-prime-test-cache))
		  (t   ; Fermat step
		   (let* ((x (math-add (math-random (math-add n -2)) 2))
			  (y (math-pow-mod x math-prime-test-cache-q n))
			  (j 0))
		     (while (and (not (eq y 1))
				 (not (equal y math-prime-test-cache-nm1))
				 (< (setq j (1+ j)) math-prime-test-cache-k))
		       (setq y (math-mod (math-mul y y) n)))
		     (if (or (equal y math-prime-test-cache-nm1)
			     (and (eq y 1) (eq j 0)))
			 (list 'maybe
			       (1+ (nth 1 (nth 1 math-prime-test-cache)))
			       (math-mul (nth 2 (nth 1 math-prime-test-cache))
					 '(float 25 -2)))
		       '(nil unknown))))))
      (setq math-prime-test-cache (list n res)
	    iters (if (eq (car res) 'maybe)
		      (1- iters)
		    0)))
    res)
)
(defvar math-prime-test-cache '(-1))

;;; Theory: summing base-10^6 digits modulo 111111 is "casting out 999999s".
;;; Initial probability that N is prime is 1/ln(N) = log10(e)/log10(N).
;;; After culling [2,3,5,7,11,13,37], probability of primality is 5.36 x more.
;;; Initial reported probability of non-primality is thus 100% - this.
;;; Each Fermat step multiplies this probability by 25%.
;;; The Fermat step is algorithm P from Knuth section 4.5.4.


(defun math-prime-factors (n)
  (setq math-prime-factors-finished t)
  (if (Math-messy-integerp n)
      (setq n (math-trunc n)))
  (if (and (Math-natnump n) (Math-natnum-lessp 2 n))
      (let (factors res p (i 0))
	(while (and (not (eq n 1))
		    (< i (length math-primes-table)))
	  (setq p (aref math-primes-table i))
	  (while (eq (cdr (setq res (cond ((eq n p) (cons 1 0))
					  ((eq n 1) (cons 0 1))
					  ((consp n) (math-idivmod n p))
					  (t (cons (/ n p) (% n p))))))
		     0)
	    (math-working "factor" p)
	    (setq factors (nconc factors (list p))
		  n (car res)))
	  (or (eq n 1)
	      (Math-natnum-lessp p (car res))
	      (setq factors (nconc factors (list n))
		    n 1))
	  (setq i (1+ i)))
	(or (setq math-prime-factors-finished (eq n 1))
	    (setq factors (nconc factors (list n))))
	(cons 'vec factors))
    (calc-record-why 'integerp n)
    (list 'calcFunc-prfac n))
)
(fset 'calcFunc-prfac (symbol-function 'math-prime-factors))

(defun math-totient (n)
  (if (Math-messy-integerp n)
      (setq n (math-trunc n)))
  (if (Math-natnump n)
      (if (Math-natnum-lessp n 2)
	  (if (Math-negp n)
	      (math-totient (math-abs n))
	    n)
	(let ((factors (cdr (math-prime-factors n)))
	      p)
	  (if math-prime-factors-finished
	      (progn
		(while factors
		  (setq p (car factors)
			n (math-mul (math-div n p) (math-add p -1)))
		  (while (equal p (car factors))
		    (setq factors (cdr factors))))
		n)
	    (calc-record-why "Number too big to factor" n)
	    (list 'calcFunc-totient n))))
    (calc-record-why 'natnump n)
    (list 'calcFunc-totient n))
)
(fset 'calcFunc-totient (symbol-function 'math-totient))

(defun math-moebius (n)
  (if (Math-messy-integerp n)
      (setq n (math-trunc n)))
  (if (and (Math-natnump n) (not (eq n 0)))
      (if (Math-natnum-lessp n 2)
	  (if (Math-negp n)
	      (math-moebius (math-abs n))
	    1)
	(let ((factors (cdr (math-prime-factors n)))
	      (mu 1))
	  (if math-prime-factors-finished
	      (progn
		(while factors
		  (setq mu (if (equal (car factors) (nth 1 factors))
			       0 (math-neg mu))
			factors (cdr factors)))
		mu)
	    (calc-record-why "Number too big to factor" n)
	    (list 'calcFunc-moebius n))))
    (calc-record-why 'natnump n)
    (list 'calcFunc-moebius n))
)
(fset 'calcFunc-moebius (symbol-function 'math-moebius))


(defun math-next-prime (n iters)
  (if (Math-integerp n)
      (if (Math-integer-negp n)
	  2
	(if (and (integerp n) (< n 5003))
	    (math-next-small-prime (1+ n))
	  (if (math-evenp n)
	      (setq n (math-add n -1)))
	  (let (res)
	    (while (not (car (setq res (math-prime-test
					(setq n (math-add n 2)) iters)))))
	    (if (and calc-verbose-nextprime
		     (eq (car res) 'maybe))
		(calc-report-prime-test res)))
	  n))
    (if (Math-realp n)
	(math-next-prime (math-trunc n) iters)
      (math-reject-arg n 'integerp)))
)
(fset 'calcFunc-nextprime (symbol-function 'math-next-prime))
(setq calc-verbose-nextprime nil)

(defun math-prev-prime (n iters)
  (if (Math-integerp n)
      (if (Math-lessp n 4)
	  2
	(if (math-evenp n)
	    (setq n (math-add n 1)))
	(let (res)
	  (while (not (car (setq res (math-prime-test
				      (setq n (math-add n -2)) iters)))))
	  (if (and calc-verbose-nextprime
		   (eq (car res) 'maybe))
	      (calc-report-prime-test res)))
	n)
    (if (Math-realp n)
	(math-prev-prime (math-ceiling n) iters)
      (math-reject-arg n 'integerp)))
)
(fset 'calcFunc-prevprime (symbol-function 'math-prev-prime))

(defun math-next-small-prime (n)
  (if (and (integerp n) (> n 2))
      (let ((lo -1)
	    (hi (length math-primes-table))
	    mid)
	(while (> (- hi lo) 1)
	  (if (> n (aref math-primes-table
			 (setq mid (ash (+ lo hi) -1))))
	      (setq lo mid)
	    (setq hi mid)))
	(aref math-primes-table hi))
    2)
)

(defconst math-primes-table
  [2 3 5 7 11 13 17 19 23 29 31 37 41 43 47 53 59 61 67 71 73 79 83 89
     97 101 103 107 109 113 127 131 137 139 149 151 157 163 167 173 179 181
     191 193 197 199 211 223 227 229 233 239 241 251 257 263 269 271 277
     281 283 293 307 311 313 317 331 337 347 349 353 359 367 373 379 383
     389 397 401 409 419 421 431 433 439 443 449 457 461 463 467 479 487
     491 499 503 509 521 523 541 547 557 563 569 571 577 587 593 599 601
     607 613 617 619 631 641 643 647 653 659 661 673 677 683 691 701 709
     719 727 733 739 743 751 757 761 769 773 787 797 809 811 821 823 827
     829 839 853 857 859 863 877 881 883 887 907 911 919 929 937 941 947
     953 967 971 977 983 991 997 1009 1013 1019 1021 1031 1033 1039 1049
     1051 1061 1063 1069 1087 1091 1093 1097 1103 1109 1117 1123 1129 1151
     1153 1163 1171 1181 1187 1193 1201 1213 1217 1223 1229 1231 1237 1249
     1259 1277 1279 1283 1289 1291 1297 1301 1303 1307 1319 1321 1327 1361
     1367 1373 1381 1399 1409 1423 1427 1429 1433 1439 1447 1451 1453 1459
     1471 1481 1483 1487 1489 1493 1499 1511 1523 1531 1543 1549 1553 1559
     1567 1571 1579 1583 1597 1601 1607 1609 1613 1619 1621 1627 1637 1657
     1663 1667 1669 1693 1697 1699 1709 1721 1723 1733 1741 1747 1753 1759
     1777 1783 1787 1789 1801 1811 1823 1831 1847 1861 1867 1871 1873 1877
     1879 1889 1901 1907 1913 1931 1933 1949 1951 1973 1979 1987 1993 1997
     1999 2003 2011 2017 2027 2029 2039 2053 2063 2069 2081 2083 2087 2089
     2099 2111 2113 2129 2131 2137 2141 2143 2153 2161 2179 2203 2207 2213
     2221 2237 2239 2243 2251 2267 2269 2273 2281 2287 2293 2297 2309 2311
     2333 2339 2341 2347 2351 2357 2371 2377 2381 2383 2389 2393 2399 2411
     2417 2423 2437 2441 2447 2459 2467 2473 2477 2503 2521 2531 2539 2543
     2549 2551 2557 2579 2591 2593 2609 2617 2621 2633 2647 2657 2659 2663
     2671 2677 2683 2687 2689 2693 2699 2707 2711 2713 2719 2729 2731 2741
     2749 2753 2767 2777 2789 2791 2797 2801 2803 2819 2833 2837 2843 2851
     2857 2861 2879 2887 2897 2903 2909 2917 2927 2939 2953 2957 2963 2969
     2971 2999 3001 3011 3019 3023 3037 3041 3049 3061 3067 3079 3083 3089
     3109 3119 3121 3137 3163 3167 3169 3181 3187 3191 3203 3209 3217 3221
     3229 3251 3253 3257 3259 3271 3299 3301 3307 3313 3319 3323 3329 3331
     3343 3347 3359 3361 3371 3373 3389 3391 3407 3413 3433 3449 3457 3461
     3463 3467 3469 3491 3499 3511 3517 3527 3529 3533 3539 3541 3547 3557
     3559 3571 3581 3583 3593 3607 3613 3617 3623 3631 3637 3643 3659 3671
     3673 3677 3691 3697 3701 3709 3719 3727 3733 3739 3761 3767 3769 3779
     3793 3797 3803 3821 3823 3833 3847 3851 3853 3863 3877 3881 3889 3907
     3911 3917 3919 3923 3929 3931 3943 3947 3967 3989 4001 4003 4007 4013
     4019 4021 4027 4049 4051 4057 4073 4079 4091 4093 4099 4111 4127 4129
     4133 4139 4153 4157 4159 4177 4201 4211 4217 4219 4229 4231 4241 4243
     4253 4259 4261 4271 4273 4283 4289 4297 4327 4337 4339 4349 4357 4363
     4373 4391 4397 4409 4421 4423 4441 4447 4451 4457 4463 4481 4483 4493
     4507 4513 4517 4519 4523 4547 4549 4561 4567 4583 4591 4597 4603 4621
     4637 4639 4643 4649 4651 4657 4663 4673 4679 4691 4703 4721 4723 4729
     4733 4751 4759 4783 4787 4789 4793 4799 4801 4813 4817 4831 4861 4871
     4877 4889 4903 4909 4919 4931 4933 4937 4943 4951 4957 4967 4969 4973
     4987 4993 4999 5003])




;;;; [calc-bin.el]

;;; Bitwise operations.

(defun math-and (a b &optional w)   ; [I I I] [Public]
  (cond ((Math-messy-integerp w)
	 (math-and a b (math-trunc w)))
	((and w (not (integerp w)))
	 (math-reject-arg w 'integerp))
	((and (integerp a) (integerp b))
	 (math-clip (logand a b) w))
	((or (eq (car-safe a) 'mod) (eq (car-safe b) 'mod))
	 (math-binary-modulo-args 'math-and a b w))
	((not (Math-num-integerp a))
	 (math-reject-arg a 'integerp))
	((not (Math-num-integerp b))
	 (math-reject-arg b 'integerp))
	(t (math-clip (cons 'bigpos
			    (math-and-bignum (math-binary-arg a w)
					     (math-binary-arg b w)))
		      w)))
)
(fset 'calcFunc-and (symbol-function 'math-and))

(defun math-binary-arg (a w)
  (if (not (Math-integerp a))
      (setq a (math-trunc a)))
  (if (Math-integer-negp a)
      (math-not-bignum (cdr (math-bignum-test (math-sub -1 a)))
		       (math-abs (if w (math-trunc w) calc-word-size)))
    (cdr (Math-bignum-test a)))
)

(defun math-binary-modulo-args (f a b w)
  (let (mod)
    (if (eq (car-safe a) 'mod)
	(progn
	  (setq mod (nth 2 a)
		a (nth 1 a))
	  (if (eq (car-safe b) 'mod)
	      (if (equal mod (nth 2 b))
		  (setq b (nth 1 b))
		(math-reject-arg b "Inconsistent modulos"))))
      (setq mod (nth 2 b)
	    b (nth 1 b)))
    (if (Math-messy-integerp mod)
	(setq mod (math-trunc mod))
      (or (Math-integerp mod)
	  (math-reject-arg mod 'integerp)))
    (let ((bits (math-integer-log2 mod)))
      (if bits
	  (if w
	      (if (/= w bits)
		  (calc-record-why
		   "Warning: Modulo inconsistent with word size"))
	    (setq w bits))
	(calc-record-why "Warning: Modulo is not a power of 2"))
      (math-make-mod (if b
			 (funcall f a b w)
		       (funcall f a w))
		     mod)))
)

(defun math-and-bignum (a b)   ; [l l l]
  (and a b
       (let ((qa (math-div-bignum-digit a 512))
	     (qb (math-div-bignum-digit b 512)))
	 (math-mul-bignum-digit (math-and-bignum (math-norm-bignum (car qa))
						  (math-norm-bignum (car qb)))
				 512
				 (logand (cdr qa) (cdr qb)))))
)

(defun math-or (a b &optional w)   ; [I I I] [Public]
  (cond ((Math-messy-integerp w)
	 (math-or a b (math-trunc w)))
	((and w (not (integerp w)))
	 (math-reject-arg w 'integerp))
	((and (integerp a) (integerp b))
	 (math-clip (logior a b) w))
	((or (eq (car-safe a) 'mod) (eq (car-safe b) 'mod))
	 (math-binary-modulo-args 'math-or a b w))
	((not (Math-num-integerp a))
	 (math-reject-arg a 'integerp))
	((not (Math-num-integerp b))
	 (math-reject-arg b 'integerp))
	(t (math-clip (cons 'bigpos
			    (math-or-bignum (math-binary-arg a w)
					    (math-binary-arg b w)))
		      w)))
)
(fset 'calcFunc-or (symbol-function 'math-or))

(defun math-or-bignum (a b)   ; [l l l]
  (and (or a b)
       (let ((qa (math-div-bignum-digit a 512))
	     (qb (math-div-bignum-digit b 512)))
	 (math-mul-bignum-digit (math-or-bignum (math-norm-bignum (car qa))
						 (math-norm-bignum (car qb)))
				 512
				 (logior (cdr qa) (cdr qb)))))
)

(defun math-xor (a b &optional w)   ; [I I I] [Public]
  (cond ((Math-messy-integerp w)
	 (math-xor a b (math-trunc w)))
	((and w (not (integerp w)))
	 (math-reject-arg w 'integerp))
	((and (integerp a) (integerp b))
	 (math-clip (logxor a b) w))
	((or (eq (car-safe a) 'mod) (eq (car-safe b) 'mod))
	 (math-binary-modulo-args 'math-xor a b w))
	((not (Math-num-integerp a))
	 (math-reject-arg a 'integerp))
	((not (Math-num-integerp b))
	 (math-reject-arg b 'integerp))
	(t (math-clip (cons 'bigpos
			    (math-xor-bignum (math-binary-arg a w)
					     (math-binary-arg b w)))
		      w)))
)
(fset 'calcFunc-xor (symbol-function 'math-xor))

(defun math-xor-bignum (a b)   ; [l l l]
  (and (or a b)
       (let ((qa (math-div-bignum-digit a 512))
	     (qb (math-div-bignum-digit b 512)))
	 (math-mul-bignum-digit (math-xor-bignum (math-norm-bignum (car qa))
						  (math-norm-bignum (car qb)))
				 512
				 (logxor (cdr qa) (cdr qb)))))
)

(defun math-diff (a b &optional w)   ; [I I I] [Public]
  (cond ((Math-messy-integerp w)
	 (math-diff a b (math-trunc w)))
	((and w (not (integerp w)))
	 (math-reject-arg w 'integerp))
	((and (integerp a) (integerp b))
	 (math-clip (logand a (lognot b)) w))
	((or (eq (car-safe a) 'mod) (eq (car-safe b) 'mod))
	 (math-binary-modulo-args 'math-diff a b w))
	((not (Math-num-integerp a))
	 (math-reject-arg a 'integerp))
	((not (Math-num-integerp b))
	 (math-reject-arg b 'integerp))
	(t (math-clip (cons 'bigpos
			    (math-diff-bignum (math-binary-arg a w)
					      (math-binary-arg b w)))
		      w)))
)
(fset 'calcFunc-diff (symbol-function 'math-diff))

(defun math-diff-bignum (a b)   ; [l l l]
  (and a
       (let ((qa (math-div-bignum-digit a 512))
	     (qb (math-div-bignum-digit b 512)))
	 (math-mul-bignum-digit (math-diff-bignum (math-norm-bignum (car qa))
						   (math-norm-bignum (car qb)))
				 512
				 (logand (cdr qa) (lognot (cdr qb))))))
)

(defun math-not (a &optional w)   ; [I I] [Public]
  (cond ((Math-messy-integerp w)
	 (math-not a (math-trunc w)))
	((eq (car-safe a) 'mod)
	 (math-binary-modulo-args 'math-not a nil w))
	((and w (not (integerp w)))
	 (math-reject-arg w 'integerp))
	((not (Math-num-integerp a))
	 (math-reject-arg a 'integerp))
	((< (or w (setq w calc-word-size)) 0)
	 (math-clip (math-not a (- w)) w))
	(t (math-normalize
	    (cons 'bigpos
		  (math-not-bignum (math-binary-arg a w)
				   w)))))
)
(fset 'calcFunc-not (symbol-function 'math-not))

(defun math-not-bignum (a w)   ; [l l]
  (let ((q (math-div-bignum-digit a 512)))
    (if (<= w 9)
	(list (logand (lognot (cdr q))
		      (1- (lsh 1 w))))
      (math-mul-bignum-digit (math-not-bignum (math-norm-bignum (car q))
					       (- w 9))
			      512
			      (logxor (cdr q) 511))))
)

(defun math-lshift-binary (a &optional n w)   ; [I I] [Public]
  (setq a (math-trunc a)
	n (if n (math-trunc n) 1))
  (if (eq (car-safe a) 'mod)
      (math-binary-modulo-args 'math-lshift-binary a n w)
    (setq w (if w (math-trunc w) calc-word-size))
    (or (integerp w)
	(math-reject-arg w 'integerp))
    (or (Math-integerp a)
	(math-reject-arg a 'integerp))
    (or (Math-integerp n)
	(math-reject-arg n 'integerp))
    (if (< w 0)
	(math-clip (math-lshift-binary a n (- w)) w)
      (if (Math-integer-negp a)
	  (setq a (math-clip a w)))
      (cond ((or (Math-lessp n (- w))
		 (Math-lessp w n))
	     0)
	    ((< n 0)
	     (math-quotient (math-clip a w) (math-power-of-2 (- n))))
	    (t
	     (math-clip (math-mul a (math-power-of-2 n)) w)))))
)
(fset 'calcFunc-lsh (symbol-function 'math-lshift-binary))

(defun math-rshift-binary (a &optional n w)   ; [I I] [Public]
  (math-lshift-binary a (math-neg (or n 1)) w)
)
(fset 'calcFunc-rsh (symbol-function 'math-rshift-binary))

(defun math-lshift-arith (a &optional n w)   ; [I I] [Public]
  (if (or (null n)
	  (not (Math-negp n)))
      (math-lshift-binary a n w)
    (setq a (math-trunc a)
	  n (if n (math-trunc n) 1))
    (if (eq (car-safe a) 'mod)
	(math-binary-modulo-args 'math-lshift-arith a n w)
      (setq w (if w (math-trunc w) calc-word-size))
      (or (integerp w)
	  (math-reject-arg w 'integerp))
      (or (Math-integerp a)
	  (math-reject-arg a 'integerp))
      (or (Math-integerp n)
	  (math-reject-arg n 'integerp))
      (if (< w 0)
	  (math-clip (math-lshift-arith a n (- w)) w)
	(if (Math-integer-negp a)
	    (setq a (math-clip a w)))
	(let ((two-to-sizem1 (math-power-of-2 (1- w)))
	      (sh (math-lshift-binary a n w)))
	  (cond ((Math-natnum-lessp a two-to-sizem1)
		 sh)
		((Math-lessp n (- 1 w))
		 (math-add (math-mul two-to-sizem1 2) -1))
		(t (let ((two-to-n (math-power-of-2 (- n))))
		     (math-add (math-lshift-binary (math-add two-to-n -1)
						   (+ w n) w)
			       sh))))))))
)
(fset 'calcFunc-ash (symbol-function 'math-lshift-arith))

(defun math-rshift-arith (a &optional n w)   ; [I I] [Public]
  (math-lshift-arith a (math-neg (or n 1)) w)
)
(fset 'calcFunc-rash (symbol-function 'math-rshift-arith))

(defun math-rotate-binary (a &optional n w)   ; [I I] [Public]
  (setq a (math-trunc a)
	n (if n (math-trunc n) 1))
  (if (eq (car-safe a) 'mod)
      (math-binary-modulo-args 'math-rotate-binary a n w)
    (setq w (if w (math-trunc w) calc-word-size))
    (or (integerp w)
	(math-reject-arg w 'integerp))
    (or (Math-integerp a)
	(math-reject-arg a 'integerp))
    (or (Math-integerp n)
	(math-reject-arg n 'integerp))
    (if (< w 0)
	(math-clip (math-rotate-binary a n (- w)) w)
      (if (Math-integer-negp a)
	  (setq a (math-clip a w)))
      (cond ((or (Math-integer-negp n)
		 (not (Math-natnum-lessp n w)))
	     (math-rotate-binary a (math-mod n w) w))
	    (t
	     (math-add (math-lshift-binary a (- n w) w)
		       (math-lshift-binary a n w))))))
)
(fset 'calcFunc-rot (symbol-function 'math-rotate-binary))

(defun math-clip (a &optional w)   ; [I I] [Public]
  (cond ((Math-messy-integerp w)
	 (math-clip a (math-trunc w)))
	((eq (car-safe a) 'mod)
	 (math-binary-modulo-args 'math-clip a nil w))
	((and w (not (integerp w)))
	 (math-reject-arg w 'integerp))
	((not (Math-num-integerp a))
	 (math-reject-arg a 'integerp))
	((< (or w (setq w calc-word-size)) 0)
	 (setq a (math-clip a (- w)))
	 (if (Math-natnum-lessp a (math-power-of-2 (- -1 w)))
	     a
	   (math-sub a (math-power-of-2 (- w)))))
	((Math-negp a)
	 (math-normalize (cons 'bigpos (math-binary-arg a w))))
	((and (integerp a) (< a 1000000))
	 (if (>= w 20)
	     a
	   (logand a (1- (lsh 1 w)))))
	(t
	 (math-normalize
	  (cons 'bigpos
		(math-clip-bignum (cdr (math-bignum-test (math-trunc a)))
				  w)))))
)
(fset 'calcFunc-clip (symbol-function 'math-clip))

(defun math-clip-bignum (a w)   ; [l l]
  (let ((q (math-div-bignum-digit a 512)))
    (if (<= w 9)
	(list (logand (cdr q)
		      (1- (lsh 1 w))))
      (math-mul-bignum-digit (math-clip-bignum (math-norm-bignum (car q))
						(- w 9))
			      512
			      (cdr q))))
)



;;;; [calc-ext.el]

;;;; Algebra.

;;; Evaluate variables in an expression.
(defun math-evaluate-expr (x)  ; [Public]
  (math-normalize (math-evaluate-expr-rec x))
)

(defun math-evaluate-expr-rec (x)
  (if (consp x)
      (setq x (cons (car x)
		    (mapcar 'math-evaluate-expr-rec (cdr x)))))
  (if (eq (car-safe x) 'var)
      (if (and (boundp (nth 2 x))
	       (symbol-value (nth 2 x))
	       (not (eq (car-safe (symbol-value (nth 2 x)))
			'incomplete)))
	  (let ((val (symbol-value (nth 2 x))))
	    (if (stringp val)
		(setq val (math-read-expr val)))
	    (if (eq (car-safe val) 'error)
		x
	      (if (eq (car-safe val) 'special-const)
		  (if calc-symbolic-mode
		      x
		    val)
		val)))
	x)
    x)
)


;;; Combine two terms being added, if possible.
(defun math-combine-sum (a b nega negb scalar-okay)
  (if (and scalar-okay (Math-objvecp a) (Math-objvecp b))
      (math-add-or-sub a b nega negb)
    (let ((amult 1) (bmult 1))
      (and (consp a)
	   (cond ((and (eq (car a) '*)
		       (Math-numberp (nth 1 a)))
		  (setq amult (nth 1 a)
			a (nth 2 a)))
		 ((and (eq (car a) '/)
		       (Math-numberp (nth 2 a)))
		  (setq amult (if (Math-integerp (nth 2 a))
				  (list 'frac 1 (nth 2 a))
				(math-div 1 (nth 2 a)))
			a (nth 1 a)))
		 ((eq (car a) 'neg)
		  (setq amult -1
			a (nth 1 a)))))
      (and (consp b)
	   (cond ((and (eq (car b) '*)
		       (Math-numberp (nth 1 b)))
		  (setq bmult (nth 1 b)
			b (nth 2 b)))
		 ((and (eq (car b) '/)
		       (Math-numberp (nth 2 b)))
		  (setq bmult (if (Math-integerp (nth 2 b))
				  (list 'frac 1 (nth 2 b))
				(math-div 1 (nth 2 b)))
			b (nth 1 b)))
		 ((eq (car b) 'neg)
		  (setq bmult -1
			b (nth 1 b)))))
      (and (equal a b)
	   (progn
	     (if nega (setq amult (math-neg amult)))
	     (if negb (setq bmult (math-neg bmult)))
	     (setq amult (math-add amult bmult))
	     (math-mul amult a)))))
)

(defun math-add-or-sub (a b aneg bneg)
  (if aneg (setq a (math-neg a)))
  (if bneg (setq b (math-neg b)))
  (math-add a b)
)

;;; The following is expanded out four ways for speed.
(defun math-combine-prod (a b inva invb scalar-okay)
  (cond
   ((or (and inva (Math-zerop a))
	(and invb (Math-zerop b)))
    nil)
   ((and scalar-okay (Math-objvecp a) (Math-objvecp b))
    (math-mul-or-div a b inva invb))
   ((and (eq (car-safe a) '^)
	 inva
	 (math-looks-negp (nth 2 a)))
    (math-mul (math-pow (nth 1 a) (math-neg (nth 2 a))) b))
   ((and (eq (car-safe b) '^)
	 invb
	 (math-looks-negp (nth 2 b)))
    (math-mul a (math-pow (nth 1 b) (math-neg (nth 2 b)))))
   (t (let ((apow 1) (bpow 1))
	(and (consp a)
	     (cond ((and (eq (car a) '^)
			 (or math-simplify-symbolic-powers
			     (Math-numberp (nth 2 a))))
		    (setq apow (nth 2 a)
			  a (nth 1 a)))
		   ((and (eq (car a) 'calcFunc-sqrt))
		    (setq apow '(frac 1 2)
			  a (nth 1 a)))))
	(and (consp b)
	     (cond ((and (eq (car b) '^)
			 (or math-simplify-symbolic-powers
			     (Math-numberp (nth 2 b))))
		    (setq bpow (nth 2 b)
			  b (nth 1 b)))
		   ((and (eq (car b) 'calcFunc-sqrt))
		    (setq bpow '(frac 1 2)
			  b (nth 1 b)))))
	(and (equal a b)
	     (progn
	       (if inva (setq apow (math-neg apow)))
	       (if invb (setq bpow (math-neg bpow)))
	       (setq apow (math-add apow bpow))
	       (cond ((equal apow '(frac 1 2))
		      (list 'calcFunc-sqrt a))
		     ((equal apow '(frac -1 2))
		      (math-div 1 (list 'calcFunc-sqrt a)))
		     (t (math-pow a apow))))))))
)
(setq math-simplify-symbolic-powers nil)

(defun math-mul-or-div (a b ainv binv)
  (if ainv
      (if binv
	  (math-div (math-div 1 a) b)
	(math-div b a))
    (if binv
	(math-div a b)
      (math-mul a b)))
)



;;; True if A comes before B in a canonical ordering of expressions.  [P X X]
(defun math-beforep (a b)   ; [Public]
  (cond ((and (Math-realp a) (Math-realp b))
	 (let ((comp (math-compare a b)))
	   (or (eq comp -1)
	       (and (eq comp 0)
		    (not (equal a b))
		    (> (length (memq (car-safe a)
				     '(bigneg nil bigpos frac float)))
		       (length (memq (car-safe b)
				     '(bigneg nil bigpos frac float))))))))
	((Math-realp a) t)
	((Math-realp b) nil)
	((eq (car a) 'var)
	 (if (eq (car b) 'var)
	     (string-lessp (symbol-name (nth 1 a)) (symbol-name (nth 1 b)))
	   (not (Math-numberp b))))
	((eq (car b) 'var) (Math-numberp a))
	((eq (car a) (car b))
	 (while (and (setq a (cdr a) b (cdr b)) a
		     (equal (car a) (car b))))
	 (and b
	      (or (null a)
		  (math-beforep (car a) (car b)))))
	(t (string-lessp (symbol-name (car a)) (symbol-name (car b)))))
)



;;;; [calc-alg.el]

(setq math-living-dangerously nil)   ; true if unsafe simplifications are okay.

(defun math-simplify-extended (a)
  (let ((math-living-dangerously t))
    (math-simplify a))
)

(defun math-simplify (top-expr)
  (calc-with-default-simplification
   (let ((math-simplify-symbolic-powers t)
	 res)
     (while (not (equal top-expr (setq res (math-simplify-step
					    (math-normalize top-expr)))))
       (setq top-expr res))))
  top-expr
)

;;; The following has a "bug" in that if any recursive simplifications
;;; occur only the first handler will be tried; this doesn't really
;;; matter, since math-simplify-step is iterated to a fixed point anyway.
(defun math-simplify-step (a)
  (if (Math-primp a)
      a
    (let ((aa (cons (car a) (mapcar 'math-simplify-step (cdr a)))))
      (and (symbolp (car aa))
	   (let ((handler (get (car aa) 'math-simplify)))
	     (and handler
		  (progn
		    (while (and handler
				(equal (setq aa (or (funcall (car handler) aa)
						    aa))
				       a))
		      (setq handler (cdr handler)))
		    res))))
      aa))
)

;;;; [calc-ext.el]

(defmacro math-defsimplify (funcs &rest code)
  "Define a simplification rule for the specified function.
If FUNCS is a list of functions, the same rule is applied for each function.
CODE is a body of Lisp code that returns a simpler form of EXPR.
More than one definition may be made per function.  All definitions are tried
in the order they were encountered; the first non-NIL value which is different
from the original expression returned is used.  The argument EXPR may be
destructively modified."
  (append '(progn)
	  (mapcar (function
		   (lambda (func)
		     (list 'put (list 'quote func) ''math-simplify
			   (list 'nconc
				 (list 'get (list 'quote func) ''math-simplify)
				 (list 'list
				       (list 'function
					     (append '(lambda (expr))
						     code)))))))
		  (if (symbolp funcs) (list funcs) funcs)))
)
(put 'math-defsimplify 'lisp-indent-hook 1)

;;;; [calc-alg.el]

(math-defsimplify (+ -)
  (math-simplify-plus))

(defun math-simplify-plus ()
  (cond ((and (memq (car-safe (nth 1 expr)) '(+ -))
	      (Math-numberp (nth 2 (nth 1 expr)))
	      (not (Math-numberp (nth 2 expr))))
	 (let ((x (nth 2 expr))
	       (op (car expr)))
	   (setcar (cdr (cdr expr)) (nth 2 (nth 1 expr)))
	   (setcar expr (car (nth 1 expr)))
	   (setcar (cdr (cdr (nth 1 expr))) x)
	   (setcar (nth 1 expr) op)))
	((and (eq (car expr) '+)
	      (Math-numberp (nth 1 expr))
	      (not (Math-numberp (nth 2 expr))))
	 (let ((x (nth 2 expr)))
	   (setcar (cdr (cdr expr)) (nth 1 expr))
	   (setcar (cdr expr) x))))
  (let ((aa expr)
	aaa temp)
    (while (memq (car-safe (setq aaa (nth 1 aa))) '(+ -))
      (if (setq temp (math-combine-sum (nth 2 aaa) (nth 2 expr)
				       (eq (car aaa) '-) (eq (car expr) '-) t))
	  (progn
	    (setcar (cdr (cdr expr)) temp)
	    (setcar expr '+)
	    (setcar (cdr (cdr aaa)) 0)))
      (setq aa (nth 1 aa)))
    (if (setq temp (math-combine-sum aaa (nth 2 expr)
				     nil (eq (car expr) '-) t))
	(progn
	  (setcar (cdr (cdr expr)) temp)
	  (setcar expr '+)
	  (setcar (cdr aa) 0)))
    expr)
)

(math-defsimplify *
  (math-simplify-times))

(defun math-simplify-times ()
  (if (eq (car-safe (nth 2 expr)) '*)
      (and (math-beforep (nth 1 (nth 2 expr)) (nth 1 expr))
	   (let ((x (nth 1 expr)))
	     (setcar (cdr expr) (nth 1 (nth 2 expr)))
	     (setcar (cdr (nth 2 expr)) x)))
    (and (math-beforep (nth 2 expr) (nth 1 expr))
	 (let ((x (nth 2 expr)))
	   (setcar (cdr (cdr expr)) (nth 1 expr))
	   (setcar (cdr expr) x))))
  (let ((aa expr)
	aaa temp)
    (while (eq (car-safe (setq aaa (nth 2 aa))) '*)
      (if (setq temp (math-combine-prod (nth 1 expr) (nth 1 aaa) nil nil t))
	  (progn
	    (setcar (cdr expr) temp)
	    (setcar (cdr aaa) 1)))
      (setq aa (nth 2 aa)))
    (if (setq temp (math-combine-prod aaa (nth 1 expr) nil nil t))
	(progn
	  (setcar (cdr expr) temp)
	  (setcar (cdr (cdr aa)) 1)))
    expr)
)

(math-defsimplify /
  (math-simplify-divide))

(defun math-simplify-divide ()
  (let ((np (cdr expr))
	n nn)
    (setq nn (math-common-constant-factor (nth 2 expr)))
    (if nn
	(progn
	  (setq n (math-common-constant-factor (nth 1 expr)))
	  (if (and (consp nn) (eq (nth 1 nn) 1) (not n))
	      (progn
		(setcar (cdr expr) (math-mul (nth 1 expr) nn))
		(setcar (cdr (cdr expr))
			(math-cancel-common-factor (nth 2 expr) nn)))
	    (if (and n (not (eq (setq n (math-frac-gcd n nn)) 1)))
		(progn
		  (setcar (cdr expr)
			  (math-cancel-common-factor (nth 1 expr) n))
		  (setcar (cdr (cdr expr))
			  (math-cancel-common-factor (nth 2 expr) n)))))))
    (while (eq (car-safe (setq n (car np))) '*)
      (math-simplify-divisor (cdr n) (cdr (cdr expr)))
      (setq np (cdr (cdr n))))
    (math-simplify-divisor np (cdr (cdr expr)))
    expr)
)

(defun math-simplify-divisor (np dp)
  (let ((n (car np))
	d dd temp)
    (while (eq (car-safe (setq d (car dp))) '*)
      (if (setq temp (math-combine-prod n (nth 1 d) nil t t))
	  (progn
	    (setcar np (setq n temp))
	    (setcar (cdr d) 1)))
      (setq dp (cdr (cdr d))))
    (if (setq temp (math-combine-prod n d nil t t))
	(progn
	  (setcar np (setq n temp))
	  (setcar dp 1))))
)

(defun math-common-constant-factor (expr)
  (if (Math-primp expr)
      (if (Math-ratp expr)
	  (and (not (memq expr '(0 1)))
	       (math-abs expr))
	(if (Math-ratp (setq expr (math-to-simple-fraction expr)))
	    (math-common-constant-factor expr)))
    (if (memq (car expr) '(+ -))
	(let ((f1 (math-common-constant-factor (nth 1 expr)))
	      (f2 (math-common-constant-factor (nth 2 expr))))
	  (and f1 f2
	       (not (eq (setq f1 (math-frac-gcd f1 f2)) 1))
	       f1))
      (if (memq (car expr) '(* /))
	  (math-common-constant-factor (nth 1 expr)))))
)

(defun math-cancel-common-factor (expr val)
  (if (memq (car-safe expr) '(+ -))
      (progn
	(setcar (cdr expr) (math-cancel-common-factor (nth 1 expr) val))
	(setcar (cdr (cdr expr)) (math-cancel-common-factor (nth 2 expr) val))
	expr)
    (math-div expr val))
)

(defun math-frac-gcd (a b)
  (if (and (Math-integerp a)
	   (Math-integerp b))
      (math-gcd a b)
    (or (Math-integerp a) (setq a (list 'frac a 1)))
    (or (Math-integerp b) (setq b (list 'frac b 1)))
    (math-make-frac (math-gcd (nth 1 a) (nth 1 b))
		    (math-gcd (nth 2 a) (nth 2 b))))
)

(math-defsimplify calcFunc-sin
  (or (and (eq (car-safe (nth 1 expr)) 'calcFunc-arcsin)
	   (nth 1 (nth 1 expr)))
      (and (math-looks-negp (nth 1 expr))
	   (math-neg (list 'calcFunc-sin (math-neg (nth 1 expr)))))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-arccos)
	   (list 'calcFunc-sqrt (math-sub 1 (math-sqr (nth 1 (nth 1 expr))))))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-arctan)
	   (math-div (nth 1 (nth 1 expr))
		     (list 'calcFunc-sqrt
			   (math-add 1 (math-sqr (nth 1 (nth 1 expr))))))))
)

(math-defsimplify calcFunc-cos
  (or (and (eq (car-safe (nth 1 expr)) 'calcFunc-arccos)
	   (nth 1 (nth 1 expr)))
      (and (math-looks-negp (nth 1 expr))
	   (list 'calcFunc-cos (math-neg (nth 1 expr))))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-arcsin)
	   (list 'calcFunc-sqrt (math-sub 1 (math-sqr (nth 1 (nth 1 expr))))))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-arctan)
	   (math-div 1
		     (list 'calcFunc-sqrt
			   (math-add 1 (math-sqr (nth 1 (nth 1 expr))))))))
)

(math-defsimplify calcFunc-tan
  (or (and (eq (car-safe (nth 1 expr)) 'calcFunc-arctan)
	   (nth 1 (nth 1 expr)))
      (and (math-looks-negp (nth 1 expr))
	   (math-neg (list 'calcFunc-tan (math-neg (nth 1 expr)))))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-arcsin)
	   (math-div (nth 1 (nth 1 expr))
		     (list 'calcFunc-sqrt
			   (math-sub 1 (math-sqr (nth 1 (nth 1 expr)))))))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-arccos)
	   (math-div (list 'calcFunc-sqrt
			   (math-sub 1 (math-sqr (nth 1 (nth 1 expr)))))
		     (nth 1 (nth 1 expr)))))
)

(math-defsimplify calcFunc-sinh
  (and (eq (car-safe (nth 1 expr)) 'calcFunc-arcsinh)
       (nth 1 (nth 1 expr)))
)

(math-defsimplify calcFunc-cosh
  (and (eq (car-safe (nth 1 expr)) 'calcFunc-arccosh)
       (nth 1 (nth 1 expr)))
)

(math-defsimplify calcFunc-tanh
  (and (eq (car-safe (nth 1 expr)) 'calcFunc-arctanh)
       (nth 1 (nth 1 expr)))
)

(math-defsimplify calcFunc-arcsin
  (or (and (math-looks-negp (nth 1 expr))
	   (math-neg (list 'calcFunc-arcsin (math-neg (nth 1 expr)))))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-sin)
	   (nth 1 (nth 1 expr)))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-cos)
	   (math-sub (math-div '(var pi var-pi) 2)
		     (nth 1 (nth 1 expr)))))
)

(math-defsimplify calcFunc-arccos
  (or (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-cos)
	   (nth 1 (nth 1 expr)))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-sin)
	   (math-sub (math-div '(var pi var-pi) 2)
		     (nth 1 (nth 1 expr)))))
)

(math-defsimplify calcFunc-arctan
  (or (and (math-looks-negp (nth 1 expr))
	   (math-neg (list 'calcFunc-arctan (math-neg (nth 1 expr)))))
      (and math-living-dangerously
	   (eq (car-safe (nth 1 expr)) 'calcFunc-tan)
	   (nth 1 (nth 1 expr))))
)

(math-defsimplify calcFunc-arcsinh
  (and math-living-dangerously
       (eq (car-safe (nth 1 expr)) 'calcFunc-sinh)
       (nth 1 (nth 1 expr)))
)

(math-defsimplify calcFunc-arccosh
  (and math-living-dangerously
       (eq (car-safe (nth 1 expr)) 'calcFunc-cosh)
       (nth 1 (nth 1 expr)))
)

(math-defsimplify calcFunc-arctanh
  (and math-living-dangerously
       (eq (car-safe (nth 1 expr)) 'calcFunc-tanh)
       (nth 1 (nth 1 expr)))
)

(math-defsimplify calcFunc-sqrt
  (or (let ((fac (math-common-constant-factor (nth 1 expr))))
	(and fac
	     (math-mul (list 'calcFunc-sqrt fac)
		       (list 'calcFunc-sqrt
			     (math-cancel-common-factor (nth 1 expr) fac)))))
      (and (eq (car-safe (nth 1 expr)) '-)
	   (math-equal-int (nth 1 (nth 1 expr)) 1)
	   (eq (car-safe (nth 2 (nth 1 expr))) '^)
	   (math-equal-int (nth 2 (nth 2 (nth 1 expr))) 2)
	   (or (and (eq (car-safe (nth 1 (nth 2 (nth 1 expr)))) 'calcFunc-sin)
		    (list 'calcFunc-cos
			  (nth 1 (nth 1 (nth 2 (nth 1 expr))))))
	       (and (eq (car-safe (nth 1 (nth 2 (nth 1 expr)))) 'calcFunc-cos)
		    (list 'calcFunc-sin
			  (nth 1 (nth 1 (nth 2 (nth 1 expr))))))))
      (and math-living-dangerously
	   (or (and (eq (car-safe (nth 1 expr)) '^)
		    (list '^
			  (nth 1 (nth 1 expr))
			  (math-div (nth 2 (nth 1 expr)) 2)))
	       (and (eq (car-safe (nth 1 expr)) 'calcFunc-sqrt)
		    (list '^ (nth 1 (nth 1 expr)) (math-div 1 4))))))
)

(math-defsimplify 'calcFunc-exp
  (and (eq (car-safe (nth 1 expr)) 'calcFunc-ln)
       (nth 1 (nth 1 expr)))
)

(math-defsimplify 'calcFunc-ln
  (and math-living-dangerously
       (eq (car-safe (nth 1 expr)) 'calcFunc-exp)
       (nth 1 (nth 1 expr)))
)

(math-defsimplify '^
  (math-simplify-pow))

(defun math-simplify-pow ()
  (or (and math-living-dangerously
	   (or (and (eq (car-safe (nth 1 expr)) '^)
		    (list '^
			  (nth 1 (nth 1 expr))
			  (math-mul (nth 2 expr) (nth 2 (nth 1 expr)))))
	       (and (eq (car-safe (nth 1 expr)) 'calcFunc-sqrt)
		    (list '^
			  (nth 1 (nth 1 expr))
			  (math-div (nth 2 expr) 2)))))
      (and (math-equal-int (nth 1 expr) 10)
	   (eq (car-safe (nth 2 expr)) 'calcFunc-log10)
	   (nth 1 (nth 2 expr)))
      (and (equal (nth 1 expr) '(var e var-e))
	   (eq (car-safe (nth 2 expr)) 'calcFunc-ln)
	   (nth 1 (nth 2 expr))))
)

(math-defsimplify 'calcFunc-log10
  (and math-living-dangerously
       (eq (car-safe (nth 1 expr)) '^)
       (math-equal-int (nth 1 (nth 1 expr)) 10)
       (nth 2 (nth 1 expr)))
)




(defun math-expand-term (expr)
  (cond ((and (eq (car-safe expr) '*)
	      (memq (car-safe (nth 1 expr)) '(+ -)))
	 (math-add-or-sub (math-mul (nth 1 (nth 1 expr)) (nth 2 expr))
			  (math-mul (nth 2 (nth 1 expr)) (nth 2 expr))
			  nil (eq (car (nth 1 expr)) '-)))
	((and (eq (car-safe expr) '*)
	      (memq (car-safe (nth 2 expr)) '(+ -)))
	 (math-add-or-sub (math-mul (nth 1 expr) (nth 1 (nth 2 expr)))
			  (math-mul (nth 1 expr) (nth 2 (nth 2 expr)))
			  nil (eq (car (nth 2 expr)) '-)))
	((and (eq (car-safe expr) '/)
	      (memq (car-safe (nth 1 expr)) '(+ -)))
	 (math-add-or-sub (math-div (nth 1 (nth 1 expr)) (nth 2 expr))
			  (math-div (nth 2 (nth 1 expr)) (nth 2 expr))
			  nil (eq (car (nth 1 expr)) '-)))
	((and (eq (car-safe expr) '^)
	      (memq (car-safe (nth 1 expr)) '(+ -))
	      (integerp (nth 2 expr))
	      (if (> (nth 2 expr) 0)
		  (list '*
			(nth 1 expr)
			(math-pow (nth 1 expr) (1- (nth 2 expr))))
		(if (< (nth 2 expr) 0)
		    (math-div 1 (math-pow (nth 1 expr)
					  (- (nth 2 expr))))))))
	(t expr))
)

(defun math-expand-tree (expr &optional many)
  (math-map-tree 'math-expand-term expr many)
)

(defun math-map-tree (mmt-func mmt-expr &optional mmt-many)
  (or mmt-many (setq mmt-many 1000000))
  (math-map-tree-rec mmt-expr)
)

(defun math-map-tree-rec (mmt-expr)
  (or (= mmt-many 0)
      (let ((mmt-done nil)
	    mmt-nextval)
	(while (not mmt-done)
	  (while (and (/= mmt-many 0)
		      (setq mmt-nextval (funcall mmt-func mmt-expr))
		      (not (equal mmt-expr mmt-nextval)))
	    (setq mmt-expr mmt-nextval
		  mmt-many (if (> mmt-many 0) (1- mmt-many) (1+ mmt-many))))
	  (if (or (Math-primp mmt-expr)
		  (<= mmt-many 0))
	      (setq mmt-done t)
	    (setq mmt-nextval (cons (car mmt-expr)
				(mapcar 'math-map-tree-rec (cdr mmt-expr))))
	    (if (equal mmt-nextval mmt-expr)
		(setq mmt-done t)
	      (setq mmt-expr mmt-nextval))))))
  mmt-expr
)




(defun math-apply-rewrite (expr lhs rhs &optional cond)
  (let ((matches-found nil))
    (and (math-match-pattern expr lhs)
	 (or (null cond)
	     (math-is-true (math-simplify (math-replace-variables cond))))
	 (math-replace-variables rhs)))
)

(defun math-apply-rewrite-rules (expr rules)
  (let ((r rules)
	next)
    (while (and r
		(or (not (setq next (math-apply-rewrite expr
							(nth 1 (car r))
							(nth 2 (car r))
							(nth 3 (car r)))))
		    (equal expr (setq next (math-normalize next)))))
      (setq r (cdr r)))
    (and r
	 next))
)

(defun math-rewrite (expr rules &optional many)
  (setq rules (math-check-rewrite-rules rules))
  (math-map-tree (function (lambda (x) (math-apply-rewrite-rules x rules)))
		 expr many)
)

(defun math-check-rewrite-rules (rules)
  (if (and (eq (car-safe rules) 'var)
	   (boundp (nth 2 rules))
	   (symbol-value (nth 2 rules)))
      (setq rules (symbol-value (nth 2 rules))))
  (or (Math-vectorp rules)
      (error "Rules must be a vector"))
  (setq rules (if (Math-vectorp (nth 1 rules))
		  (cdr rules)
		(list rules)))
  (let ((r rules))
    (while r
      (or (and (Math-vectorp (car r))
	       (cdr (cdr (car r)))
	       (not (nthcdr 4 (car r))))
	  (error "Malformed rules vector"))
      (setq r (cdr r))))
  rules
)

(defun math-match-pattern (expr pat)
  (cond ((Math-primp pat)
	 (or (math-equal expr pat)
	     (and (eq (car-safe pat) 'var)
		  (let ((match (assq (nth 1 pat) matches-found)))
		    (if match
			(equal expr (nth 1 match))
		      (setq matches-found (cons (list (nth 1 pat)
						      expr)
						matches-found)))))))
	((eq (car pat) 'calcFunc-quote)
	 (equal expr (nth 1 pat)))
	(t
	 (and (eq (car pat) (car-safe expr))
	      (progn
		(while (and (setq expr (cdr expr) pat (cdr pat))
			    expr
			    (math-match-pattern (car expr) (car pat))))
		(and (null expr) (null pat))))))
)

(defun math-replace-variables (expr)
  (if (Math-primp expr)
      (if (eq (car-safe expr) 'var)
	  (let ((match (assq (nth 1 expr) matches-found)))
	    (if match
		(nth 1 match)
	      expr))
	expr)
    (cons (car expr) (mapcar 'math-replace-variables (cdr expr))))
)

;;;; [calc-ext.el]

(defun math-is-true (expr)
  (and (Math-realp expr)
       (not (Math-zerop expr)))
)




;;;; [calc-alg-2.el]

(defun math-derivative (expr)   ; uses global values: deriv-var, deriv-total.
  (cond ((equal expr deriv-var)
	 1)
	((or (Math-scalarp expr)
	     (eq (car expr) 'sdev)
	     (and (eq (car expr) 'var)
		  (not deriv-total)))
	 0)
	((eq (car expr) '+)
	 (math-add (math-derivative (nth 1 expr))
		   (math-derivative (nth 2 expr))))
	((eq (car expr) '-)
	 (math-sub (math-derivative (nth 1 expr))
		   (math-derivative (nth 2 expr))))
	((eq (car expr) 'neg)
	 (math-neg (math-derivative (nth 1 expr))))
	((eq (car expr) '*)
	 (math-add (math-mul (nth 2 expr)
			     (math-derivative (nth 1 expr)))
		   (math-mul (nth 1 expr)
			     (math-derivative (nth 2 expr)))))
	((eq (car expr) '/)
	 (math-sub (math-div (math-derivative (nth 1 expr))
			     (nth 2 expr))
		   (math-div (math-mul (nth 1 expr)
				       (math-derivative (nth 2 expr)))
			     (math-sqr (nth 2 expr)))))
	((eq (car expr) '^)
	 (let ((du (math-derivative (nth 1 expr)))
	       (dv (math-derivative (nth 2 expr))))
	   (or (Math-zerop du)
	       (setq du (math-mul (nth 2 expr)
				  (math-mul (math-normalize
					     (list '^
						   (nth 1 expr)
						   (math-add (nth 2 expr) -1)))
					    du))))
	   (or (Math-zerop dv)
	       (setq dv (math-mul (math-normalize
				   (list 'calcFunc-ln (nth 1 expr)))
				  (math-mul expr dv))))
	   (math-add du dv)))
	((eq (car expr) '%)
	 (math-derivative (nth 1 expr)))   ; a reasonable definition
	((eq (car expr) 'vec)
	 (math-map-vec 'math-derivative expr))
	((and (eq (car expr) 'calcFunc-log)
	      (= (length expr) 3)
	      (not (Math-zerop (nth 2 expr))))
	 (let ((lnv (math-normalize (list 'calcFunc-ln (nth 2 expr)))))
	   (math-sub (math-div (math-derivative (nth 1 expr))
			       (math-mul lnv (nth 1 expr)))
		     (math-div (math-derivative (nth 2 expr))
			       (math-mul (math-sqr lnv)
					 (nth 2 expr))))))
	(t (or (and (= (length expr) 2)
		    (symbolp (car expr))
		    (let ((handler (get (car expr) 'math-derivative)))
		      (and handler
			   (let ((deriv (math-derivative (nth 1 expr))))
			     (if (Math-zerop deriv)
				 deriv
			       (math-mul (funcall handler (nth 1 expr))
					 deriv))))))
	       (if deriv-symb
		   (throw 'math-deriv nil)
		 (if (or (Math-objvecp expr)
			 (not (symbolp (car expr))))
		     (list (if deriv-total 'calcFunc-tderiv 'calcFunc-deriv)
			   expr
			   deriv-var)
		   (let ((accum 0)
			 (arg expr)
			 (n 1)
			 derv)
		     (while (setq arg (cdr arg))
		       (or (Math-zerop (setq derv (math-derivative (car arg))))
			   (let ((func (intern (concat (symbol-name (car expr))
						       "'"
						       (if (> n 1)
							   (int-to-string n)
							 "")))))
			     (setq accum (math-add
					  accum
					  (math-mul derv
						    (cons func
							  (cdr expr)))))))
		       (setq n (1+ n)))
		     accum))))))
)

(defun calcFunc-deriv (expr deriv-var &optional deriv-value deriv-symb)
  (let* ((deriv-total nil)
	 (res (catch 'math-deriv (math-derivative expr))))
    (or (eq (car-safe res) 'calcFunc-deriv)
	(null res)
	(setq res (math-normalize res)))
    (and res
	 (if deriv-value
	     (math-expr-subst res deriv-var deriv-value)
	   res)))
)

(defun calcFunc-tderiv (expr deriv-var &optional deriv-value deriv-symb)
  (let* ((deriv-total t)
	 (res (catch 'math-deriv (math-derivative expr))))
    (or (eq (car-safe res) 'calcFunc-tderiv)
	(null res)
	(setq res (math-normalize res)))
    (and res
	 (if deriv-value
	     (math-expr-subst res deriv-var deriv-value)
	   res)))
)

(put 'calcFunc-inv 'math-derivative
     (function (lambda (u) (math-neg (math-div 1 (math-sqr u))))))

(put 'calcFunc-sqrt 'math-derivative
     (function (lambda (u) (math-div 1 (math-mul 2 (list 'calcFunc-sqrt u))))))

(put 'calcFunc-conj 'math-derivative
     (function (lambda (u) (math-normalize (list 'calcFunc-conj u)))))

(put 'calcFunc-deg 'math-derivative
     (function (lambda (u) (math-div (math-pi-over-180) u))))

(put 'calcFunc-rad 'math-derivative
     (function (lambda (u) (math-mul (math-pi-over-180) u))))

(put 'calcFunc-ln 'math-derivative
     (function (lambda (u) (math-div 1 u))))

(put 'calcFunc-log10 'math-derivative
     (function (lambda (u)
		 (math-div (math-div 1 (math-normalize '(calcFunc-ln 10)))
			   u))))

(put 'calcFunc-lnp1 'math-derivative
     (function (lambda (u) (math-div 1 (math-add u 1)))))

(put 'calcFunc-exp 'math-derivative
     (function (lambda (u) (math-normalize (list 'calcFunc-exp u)))))

(put 'calcFunc-expm1 'math-derivative
     (function (lambda (u) (math-normalize (list 'calcFunc-expm1 u)))))

(put 'calcFunc-sin 'math-derivative
     (function (lambda (u) (math-to-radians-2 (math-normalize
					       (list 'calcFunc-cos u))))))

(put 'calcFunc-cos 'math-derivative
     (function (lambda (u) (math-neg (math-to-radians-2
				      (math-normalize
				       (list 'calcFunc-sin u)))))))

(put 'calcFunc-tan 'math-derivative
     (function (lambda (u) (math-to-radians-2
			    (math-div 1 (math-sqr
					 (math-normalize
					  (list 'calcFunc-cos u))))))))

(put 'calcFunc-arcsin 'math-derivative
     (function (lambda (u)
		 (math-from-radians-2
		  (math-div 1 (math-normalize
			       (list 'calcFunc-sqrt
				     (math-sub 1 (math-sqr u)))))))))

(put 'calcFunc-arccos 'math-derivative
     (function (lambda (u)
		 (math-from-radians-2
		  (math-div -1 (math-normalize
				(list 'calcFunc-sqrt
				      (math-sub 1 (math-sqr u)))))))))

(put 'calcFunc-arctan 'math-derivative
     (function (lambda (u) (math-from-radians-2
			    (math-div 1 (math-add 1 (math-sqr u)))))))

(put 'calcFunc-sinh 'math-derivative
     (function (lambda (u) (math-normalize (list 'calcFunc-cosh u)))))

(put 'calcFunc-cosh 'math-derivative
     (function (lambda (u) (math-normalize (list 'calcFunc-sinh u)))))

(put 'calcFunc-tanh 'math-derivative
     (function (lambda (u) (math-div 1 (math-sqr
					(math-normalize
					 (list 'calcFunc-cosh u)))))))

(put 'calcFunc-arcsinh 'math-derivative
     (function (lambda (u)
		 (math-div 1 (math-normalize
			      (list 'calcFunc-sqrt
				    (math-add (math-sqr u) 1)))))))

(put 'calcFunc-arccosh 'math-derivative
     (function (lambda (u)
		  (math-div 1 (math-normalize
			       (list 'calcFunc-sqrt
				     (math-add (math-sqr u) -1)))))))

(put 'calcFunc-arctanh 'math-derivative
     (function (lambda (u) (math-div 1 (math-sub 1 (math-sqr u))))))



(setq math-integ-var '(var X ---))
(setq math-integ-var-2 '(var Y ---))
(setq math-integ-vars (list 'f math-integ-var math-integ-var-2))

(defmacro math-tracing-integral (&rest parts)
  (list 'and
	'trace-buffer
	(list 'save-excursion
	      '(set-buffer trace-buffer)
	      '(goto-char (point-max))
	      (list 'and
		    '(bolp)
		    '(insert (make-string (- calc-integral-limit
					     math-integ-level) 32)
			     (format "%2d " math-integ-depth)
			     (make-string math-integ-level 32)))
	      (cons 'insert parts)
	      '(sit-for 0)))
)

;;; The following wrapper caches results and avoids infinite recursion.
;;; Each cache entry is: ( A B )          Integral of A is B;
;;;			 ( A N )          Integral of A failed at level N;
;;;			 ( A busy )	  Currently working on integral of A;
;;;			 ( A parts )	  Currently working, integ-by-parts;
;;;			 ( A parts2 )	  Currently working, integ-by-parts;
;;;			 ( A cancelled )  Ignore this cache entry;
;;;			 ( A [B] )        Same result as for cur-record = B.
(defun math-integral (expr &optional simplify same-as-above)
  (let* ((simp cur-record)
	 (cur-record (assoc expr math-integral-cache))
	 (math-integ-depth (1+ math-integ-depth))
	 (val 'cancelled))
    (math-tracing-integral "Integrating "
			   (math-format-value expr 1000)
			   "...\n")
    (and cur-record
	 (progn
	   (math-tracing-integral "Found "
				  (math-format-value (nth 1 cur-record) 1000))
	   (and (consp (nth 1 cur-record))
		(math-replace-integral-parts cur-record))
	   (math-tracing-integral " => "
				  (math-format-value (nth 1 cur-record) 1000)
				  "\n")))
    (or (and cur-record
	     (not (eq (nth 1 cur-record) 'cancelled))
	     (or (not (integerp (nth 1 cur-record)))
		 (>= (nth 1 cur-record) math-integ-level)))
	(and (consp expr)
	     (eq (car expr) 'var)
	     (eq (nth 1 expr) 'PARTS)
	     (listp (nth 2 expr))
	     (progn
	       (setq val nil)
	       t))
	(unwind-protect
	    (progn
	      (let (math-integ-msg)
		(if (eq calc-display-working-message 'lots)
		    (progn
		      (calc-set-command-flag 'clear-message)
		      (setq math-integ-msg (format
					    "Working... Integrating %s"
					    (math-format-flat-expr expr 0)))
		      (message math-integ-msg)))
		(if cur-record
		    (setcar (cdr cur-record)
			    (if same-as-above (vector simp) 'busy))
		  (setq cur-record
			(list expr (if same-as-above (vector simp) 'busy))
			math-integral-cache (cons cur-record
						  math-integral-cache)))
		(if (eq simplify 'yes)
		    (progn
		      (math-tracing-integral "Simplifying...")
		      (setq simp (math-simplify expr))
		      (setq val (if (equal simp expr)
				    (progn
				      (math-tracing-integral " no change\n")
				      (math-do-integral expr))
				  (math-tracing-integral " simplified\n")
				  (math-integral simp 'no t))))
		  (or (setq val (math-do-integral expr))
		      (eq simplify 'no)
		      (let ((simp (math-simplify expr)))
			(or (equal simp expr)
			    (progn
			      (math-tracing-integral "Trying again after "
						     "simplification...\n")
			      (setq val (math-integral simp 'no t))))))))
	      (if (eq calc-display-working-message 'lots)
		  (message math-integ-msg)))
	  (setcar (cdr cur-record) (or val math-integ-level))))
    (setq val cur-record)
    (while (vectorp (nth 1 val))
      (setq val (aref (nth 1 val) 0)))
    (setq val (if (memq (nth 1 val) '(parts parts2))
		  (progn
		    (setcar (cdr val) 'parts2)
		    (list 'var 'PARTS val))
		(and (not (eq (nth 1 val) 'busy))
		     (not (integerp (nth 1 val)))
		     (nth 1 val))))
    (math-tracing-integral "Integral of "
			   (math-format-value expr 1000)
			   "  is  "
			   (math-format-value val 1000)
			   "\n")
    val)
)
(defvar math-integral-cache nil)
(defvar math-integral-cache-state nil)

(defun math-replace-integral-parts (expr)
  (or (Math-primp expr)
      (while (setq expr (cdr expr))
	(and (consp (car expr))
	     (if (eq (car (car expr)) 'var)
		 (and (eq (nth 1 (car expr)) 'PARTS)
		      (consp (nth 2 (car expr)))
		      (if (listp (nth 1 (nth 2 (car expr))))
			  (progn
			    (setcar expr (nth 1 (nth 2 (car expr))))
			    (math-replace-integral-parts (cons 'foo expr)))
			(setcar (cdr cur-record) 'cancelled)))
	       (math-replace-integral-parts (car expr))))))
)

(defun math-do-integral (expr)
  (let (t1 t2)
    (or (cond ((not (math-expr-contains expr math-integ-var))
	       (math-mul expr math-integ-var))
	      ((equal expr math-integ-var)
	       (math-div (math-sqr expr) 2))
	      ((eq (car expr) '+)
	       (and (setq t1 (math-integral (nth 1 expr)))
		    (setq t2 (math-integral (nth 2 expr)))
		    (math-add t1 t2)))
	      ((eq (car expr) '-)
	       (and (setq t1 (math-integral (nth 1 expr)))
		    (setq t2 (math-integral (nth 2 expr)))
		    (math-sub t1 t2)))
	      ((eq (car expr) 'neg)
	       (and (setq t1 (math-integral (nth 1 expr)))
		    (math-neg t1)))
	      ((eq (car expr) '*)
	       (cond ((not (math-expr-contains (nth 1 expr) math-integ-var))
		      (and (setq t1 (math-integral (nth 2 expr)))
			   (math-mul (nth 1 expr) t1)))
		     ((not (math-expr-contains (nth 2 expr) math-integ-var))
		      (and (setq t1 (math-integral (nth 1 expr)))
			   (math-mul t1 (nth 2 expr))))
		     ((memq (car-safe (nth 1 expr)) '(+ -))
		      (math-integral (list (car (nth 1 expr))
					   (math-mul (nth 1 (nth 1 expr))
						     (nth 2 expr))
					   (math-mul (nth 2 (nth 1 expr))
						     (nth 2 expr)))
				     'yes t))
		     ((memq (car-safe (nth 2 expr)) '(+ -))
		      (math-integral (list (car (nth 2 expr))
					   (math-mul (nth 1 (nth 2 expr))
						     (nth 1 expr))
					   (math-mul (nth 2 (nth 2 expr))
						     (nth 1 expr)))
				     'yes t))))
	      ((eq (car expr) '/)
	       (cond ((not (math-expr-contains (nth 2 expr) math-integ-var))
		      (and (setq t1 (math-integral (nth 1 expr)))
			   (math-div t1 (nth 2 expr))))
		     ((and (eq (car-safe (nth 1 expr)) '*)
			   (not (math-expr-contains (nth 1 (nth 1 expr))
						    math-integ-var)))
		      (and (setq t1 (math-integral
				     (math-div (nth 2 (nth 1 expr))
					       (nth 2 expr))))
			   (math-mul t1 (nth 1 (nth 1 expr)))))
		     ((and (eq (car-safe (nth 2 expr)) '*)
			   (not (math-expr-contains (nth 1 (nth 2 expr))
						    math-integ-var)))
		      (and (setq t1 (math-integral
				     (math-div (nth 1 expr)
					       (nth 2 (nth 2 expr)))))
			   (math-div t1 (nth 1 (nth 2 expr)))))
		     ((memq (car-safe (nth 1 expr)) '(+ -))
		      (math-integral (list (car (nth 1 expr))
					   (math-div (nth 1 (nth 1 expr))
						     (nth 2 expr))
					   (math-div (nth 2 (nth 1 expr))
						     (nth 2 expr)))
				     'yes t))))
	      ((eq (car expr) '^)
	       (cond ((not (math-expr-contains (nth 1 expr) math-integ-var))
		      (or (and (setq t1 (math-is-polynomial (nth 2 expr)
							    math-integ-var 1))
			       (math-div expr
					 (math-mul (nth 1 t1)
						   (math-normalize
						    (list 'calcFunc-ln
							  (nth 1 expr))))))
			  (math-integral
			   (list 'calcFunc-exp
				 (math-mul (nth 2 expr)
					   (math-normalize
					    (list 'calcFunc-ln
						  (nth 1 expr)))))
			   'yes t)))
		     ((not (math-expr-contains (nth 2 expr) math-integ-var))
		      (if (Math-equal-int (nth 2 expr) -1)
			  (math-integral (math-div 1 (nth 1 expr)) nil t)
			(or (and (setq t1 (math-is-polynomial (nth 1 expr)
							      math-integ-var
							      1))
				 (setq t2 (math-add (nth 2 expr) 1))
				 (math-div (math-pow (nth 1 expr) t2)
					   (math-mul t2 (nth 1 t1))))
			    (and (Math-negp (nth 2 expr))
				 (math-integral
				  (math-div 1
					    (math-pow (nth 1 expr)
						      (math-neg
						       (nth 2 expr))))
				  nil t))
			    nil))))))

	;; Integral of a polynomial.
	(and (setq t1 (math-is-polynomial expr math-integ-var 20))
	     (let ((accum 0)
		   (n 1))
	       (while t1
		 (if (setq accum (math-add accum
					   (math-div (math-mul (car t1)
							       (math-pow
								math-integ-var
								n))
						     n))
			   t1 (cdr t1))
		     (setq n (1+ n))))
	       accum))

	;; Try looking it up!
	(cond ((= (length expr) 2)
	       (and (symbolp (car expr))
		    (setq t1 (get (car expr) 'math-integral))
		    (progn
		      (while (and t1
				  (not (setq t2 (funcall (car t1)
							 (nth 1 expr)))))
			(setq t1 (cdr t1)))
		      (and t2 (math-normalize t2)))))
	      ((= (length expr) 3)
	       (and (symbolp (car expr))
		    (setq t1 (get (car expr) 'math-integral-2))
		    (progn
		      (while (and t1
				  (not (setq t2 (funcall (car t1)
							 (nth 1 expr)
							 (nth 2 expr)))))
			(setq t1 (cdr t1)))
		      (and t2 (math-normalize t2))))))

	;; Integration by substitution, for various likely sub-expressions.
	;; (We should also try some of the classic non-obvious substitutions.)
	(let ((so-far nil))
	  (math-integ-try-substitutions expr))

	;; Integration by parts:
	;;   integ(f(x) g(x),x) = f(x) h(x) - integ(h(x) f'(x),x)
	;;     where h(x) = integ(g(x),x).
	(and (eq (car expr) '*)
	     (not (math-polynomial-p (nth 2 expr) math-integ-var))
	     (math-integrate-by-parts (nth 1 expr) (nth 2 expr)))
	(and (eq (car expr) '/)
	     (math-expr-contains (nth 1 expr) math-integ-var)
	     (let ((recip (math-div 1 (nth 2 expr))))
	       (or (math-integrate-by-parts (nth 1 expr) recip)
		   (math-integrate-by-parts recip (nth 1 expr)))))
	(and (eq (car expr) '^)
	     (math-integrate-by-parts (nth 1 expr)
				      (math-pow (nth 1 expr)
						(math-sub (nth 2 expr) 1))))

	;; Symmetries.
	(and (eq (car expr) '*)
	     (math-integral (list '* (nth 2 expr) (nth 1 expr)) 'no t))

	;; Give up.
	nil))
)

(defun math-integrate-by-parts (u vprime)
  (and (> math-integ-level 0)
       (not (boundp 'math-disable-parts))
       (let ((math-integ-level (1- math-integ-level))
	     v temp)
	 (unwind-protect
	     (progn
	       (setcar (cdr cur-record) 'parts)
	       (math-tracing-integral "Integrating by parts, u = "
				      (math-format-value u 1000)
				      ", v' = "
				      (math-format-value vprime 1000)
				      "\n")
	       (and (setq v (math-integral vprime))
		    (setq temp (calcFunc-deriv u
					       math-integ-var
					       nil t))
		    (setq temp (math-integral (math-mul v temp) 'yes))
		    (setq temp (math-sub (math-mul u v) temp))
		    (if (eq (nth 1 cur-record) 'parts)
			temp
		      (setq v (list 'var 'PARTS cur-record)
			    temp (math-solve-for (math-sub v temp) 0 v nil))
		      (and temp (math-simplify-extended temp)))))
	   (setcar (cdr cur-record) 'busy))))
)

;;; This tries two different formulations, hoping the algebraic simplifier
;;; will be strong enough to handle at least one.
(defun math-integrate-by-substitution (expr u)
  (and (> math-integ-level 0)
       (let ((math-integ-level (1- math-integ-level))
	     (math-living-dangerously t)
	     uinv deriv temp)
	 (and (setq uinv (math-solve-for u
					 math-integ-var-2
					 math-integ-var nil))
	      (progn
		(math-tracing-integral "Integrating by substitution, u = "
				       (math-format-value u 1000)
				       "\n")
		(or (and (not (boundp 'math-disable-subst1))
			 (setq deriv (calcFunc-deriv u
						     math-integ-var nil t))
			 (setq temp (math-integral (math-expr-subst
						    (math-expr-subst
						     (math-expr-subst
						      (math-div expr deriv)
						      u
						      math-integ-var-2)
						     math-integ-var
						     uinv)
						    math-integ-var-2
						    math-integ-var)
						   'yes)))
		    (and (not (boundp 'math-disable-subst2))
			 (setq deriv (calcFunc-deriv uinv
						     math-integ-var-2
						     math-integ-var t))
			 (setq temp (math-integral (math-mul
						    (math-expr-subst
						     (math-expr-subst
						      (math-expr-subst
						       expr
						       u
						       math-integ-var-2)
						      math-integ-var
						      uinv)
						     math-integ-var-2
						     math-integ-var)
						    deriv)
						   'yes)))))
	      (math-simplify-extended
	       (math-expr-subst temp math-integ-var u)))))
)

;;; Recursively try different substitutions based on various sub-expressions.
(defun math-integ-try-substitutions (sub-expr)
  (and (not (Math-primp sub-expr))
       (math-expr-contains sub-expr math-integ-var)
       (not (equal sub-expr math-integ-var))
       (not (assoc sub-expr so-far))
       (or (and (not (eq sub-expr expr))
		(math-integrate-by-substitution expr sub-expr))
	   (let ((res nil))
	     (setq so-far (cons (list sub-expr) so-far))
	     (while (and (setq sub-expr (cdr sub-expr))
			 (not (setq res (math-integ-try-substitutions
					 (car sub-expr))))))
	     res)))
)

(defun math-fix-const-terms (expr except-vars)
  (cond ((not (math-expr-depends expr except-vars)) 0)
	((Math-primp expr) expr)
	((eq (car expr) '+)
	 (math-add (math-fix-const-terms (nth 1 expr) except-vars)
		   (math-fix-const-terms (nth 2 expr) except-vars)))
	((eq (car expr) '-)
	 (math-sub (math-fix-const-terms (nth 1 expr) except-vars)
		   (math-fix-const-terms (nth 2 expr) except-vars)))
	(t expr))
)

(defun calc-dump-integral-cache (&optional arg)
  "Command for debugging the Calculator's symbolic integrator."
  (interactive "P")
  (let ((buf (current-buffer)))
    (unwind-protect
	(let ((p math-integral-cache)
	      cur-record)
	  (display-buffer (get-buffer-create "*Integral Cache*")) 
	  (set-buffer (get-buffer "*Integral Cache*"))
	  (erase-buffer)
	  (while p
	    (setq cur-record (car p))
	    (or arg (math-replace-integral-parts cur-record))
	    (insert (math-format-flat-expr (car cur-record) 0)
		    " --> "
		    (if (symbolp (nth 1 cur-record))
			(concat "(" (symbol-name (nth 1 cur-record)) ")")
		      (math-format-flat-expr (nth 1 cur-record) 0))
		    "\n")
	    (setq p (cdr p)))
	  (goto-char (point-min)))
      (set-buffer buf)))
)

(defun calcFunc-integ (expr var &optional low high)
  (let ((state (list calc-angle-mode
		     calc-symbolic-mode
		     calc-prefer-frac
		     calc-internal-prec)))
    (or (equal state math-integral-cache-state)
	(setq math-integral-cache-state state
	      math-integral-cache nil)))
  (let* ((math-integ-level calc-integral-limit)
	 (math-integ-depth 0)
	 (math-integ-msg "Working...done")
	 (cur-record nil)   ; a technicality
	 (sexpr (math-expr-subst expr var math-integ-var))
	 (trace-buffer (get-buffer "*Trace*"))
	 (calc-language (if (eq calc-language 'big) nil calc-language))
	 (res (if trace-buffer
		  (let ((calcbuf (current-buffer))
			(calcwin (selected-window)))
		    (unwind-protect
			(progn
			  (if (get-buffer-window trace-buffer)
			      (select-window (get-buffer-window trace-buffer)))
			  (set-buffer trace-buffer)
			  (goto-char (point-max))
			  (or (assq 'scroll-stop (buffer-local-variables))
			      (progn
				(make-local-variable 'scroll-step)
				(setq scroll-step 3)))
			  (insert "\n\n\n")
			  (set-buffer calcbuf)
			  (math-integral sexpr 'yes))
		      (select-window calcwin)
		      (set-buffer calcbuf)))
		(math-integral sexpr 'yes))))
    (if res
	(math-normalize
	 (if (and low high)
	     (math-sub (math-expr-subst res math-integ-var high)
		       (math-expr-subst res math-integ-var low))
	   (setq res (math-fix-const-terms res math-integ-vars))
	   (if low
	       (math-expr-subst res math-integ-var low)
	     (math-expr-subst res math-integ-var var))))
      (append (list 'calcFunc-integ expr var)
	      (and low (list low))
	      (and high (list high)))))
)

;;;; [calc-ext.el]

(defmacro math-defintegral (funcs &rest code)
  "Define an integration rule for the specified function.
If FUNCS is a list of functions, the same rule is applied for each function.
CODE is a body of Lisp code that returns the integral of FUNCS(U).
More than one definition may be made per function.  All definitions are tried
in the order they were encountered; the first non-NIL value returned is used."
  (setq math-integral-cache nil)
  (append '(progn)
	  (mapcar (function
		   (lambda (func)
		     (list 'put (list 'quote func) ''math-integral
			   (list 'nconc
				 (list 'get (list 'quote func) ''math-integral)
				 (list 'list
				       (list 'function
					     (append '(lambda (u))
						     code)))))))
		  (if (symbolp funcs) (list funcs) funcs)))
)
(put 'math-defintegral 'lisp-indent-hook 1)

(defmacro math-defintegral-2 (funcs &rest code)
  "Define an integration rule for the specified function.
If FUNCS is a list of functions, the same rule is applied for each function.
CODE is a body of Lisp code that returns the integral of FUNCS(U,V).
More than one definition may be made per function.  All definitions are tried
in the order they were encountered; the first non-NIL value returned is used."
  (setq math-integral-cache nil)
  (append '(progn)
	  (mapcar (function
		   (lambda (func)
		     (list 'put (list 'quote func) ''math-integral-2
			   (list 'nconc
				 (list 'get (list 'quote func)
				       ''math-integral-2)
				 (list 'list
				       (list 'function
					     (append '(lambda (u v))
						     code)))))))
		  (if (symbolp funcs) (list funcs) funcs)))
)
(put 'math-defintegral-2 'lisp-indent-hook 1)

;;;; [calc-alg-2.el]

(math-defintegral calcFunc-inv
  (math-integral (math-div 1 u)))

(math-defintegral calcFunc-conj
  (let ((int (math-integral u)))
    (and int
	 (list 'calcFunc-conj int))))

(math-defintegral calcFunc-deg
  (let ((int (math-integral u)))
    (and int
	 (list 'calcFunc-deg int))))

(math-defintegral calcFunc-rad
  (let ((int (math-integral u)))
    (and int
	 (list 'calcFunc-rad int))))

(math-defintegral calcFunc-re
  (let ((int (math-integral u)))
    (and int
	 (list 'calcFunc-re int))))

(math-defintegral calcFunc-im
  (let ((int (math-integral u)))
    (and int
	 (list 'calcFunc-im int))))

(math-defintegral calcFunc-sqrt
  (and (equal u math-integ-var)
       (math-mul '(frac 2 3)
		 (list 'calcFunc-sqrt (math-pow u 3)))))

(math-defintegral calcFunc-exp
  (and (equal u math-integ-var)
       (list 'calcFunc-exp u)))

(math-defintegral calcFunc-ln
  (or (and (equal u math-integ-var)
	   (math-sub (math-mul u (list 'calcFunc-ln u)) u))
      (and (eq (car u) '*)
	   (math-integral (math-add (list 'calcFunc-ln (nth 1 u))
				    (list 'calcFunc-ln (nth 2 u)))))
      (and (eq (car u) '/)
	   (math-integral (math-sub (list 'calcFunc-ln (nth 1 u))
				    (list 'calcFunc-ln (nth 2 u)))))
      (and (eq (car u) '^)
	   (math-integral (math-mul (nth 2 u)
				    (list 'calcFunc-ln (nth 1 u)))))))

(math-defintegral calcFunc-log10
  (and (equal u math-integ-var)
       (math-sub (math-mul u (list 'calcFunc-ln u))
		 (math-div u (list 'calcFunc-ln 10)))))

(math-defintegral-2 calcFunc-log
  (math-integral (math-div (list 'calcFunc-ln u)
			   (list 'calcFunc-ln v))))

(math-defintegral calcFunc-sin
  (and (equal u math-integ-var)
       (math-neg (math-from-radians-2 (list 'calcFunc-cos u)))))

(math-defintegral calcFunc-cos
  (and (equal u math-integ-var)
       (math-from-radians-2 (list 'calcFunc-sin u))))

(math-defintegral calcFunc-tan
  (and (equal u math-integ-var)
       (math-neg (math-from-radians-2
		  (list 'calcFunc-ln (list 'calcFunc-cos u))))))

(math-defintegral calcFunc-arcsin
  (and (equal u math-integ-var)
       (math-add (math-mul u (list 'calcFunc-arcsin u))
		 (math-from-radians-2
		  (list 'calcFunc-sqrt (math-sub 1 (math-sqr u)))))))

(math-defintegral calcFunc-arccos
  (and (equal u math-integ-var)
       (math-sub (math-mul u (list 'calcFunc-arccos u))
		 (math-from-radians-2
		  (list 'calcFunc-sqrt (math-sub 1 (math-sqr u)))))))

(math-defintegral calcFunc-arctan
  (and (equal u math-integ-var)
       (math-sub (math-mul u (list 'calcFunc-arctan u))
		 (math-from-radians-2
		  (math-div (list 'calcFunc-ln (math-add 1 (math-sqr u)))
			    2)))))

(math-defintegral calcFunc-sinh
  (and (equal u math-integ-var)
       (list 'calcFunc-cosh u)))

(math-defintegral calcFunc-cosh
  (and (equal u math-integ-var)
       (list 'calcFunc-sinh u)))

(math-defintegral calcFunc-tanh
  (and (equal u math-integ-var)
       (list 'calcFunc-ln (list 'calcFunc-cosh u))))

(math-defintegral calcFunc-arcsinh
  (and (equal u math-integ-var)
       (math-sub (math-mul u (list 'calcFunc-arcsinh u))
		 (list 'calcFunc-sqrt (math-add (math-sqr u) 1)))))

(math-defintegral calcFunc-arccosh
  (and (equal u math-integ-var)
       (math-sub (math-mul u (list 'calcFunc-arccosh u))
		 (list 'calcFunc-sqrt (math-sub 1 (math-sqr u))))))

(math-defintegral calcFunc-arctanh
  (and (equal u math-integ-var)
       (math-sub (math-mul u (list 'calcFunc-arctan u))
		 (math-div (list 'calcFunc-ln
				 (math-add 1 (math-sqr u)))
			   2))))

;;; 1 / (ax^2 + bx + c) forms.
(math-defintegral-2 /
  (and (not (math-expr-contains u math-integ-var))
       (let ((p1 (math-is-polynomial v math-integ-var 2))
	     q rq part)
	 (cond ((null p1) nil)
	       ((null (cdr (cdr p1)))
		(math-mul u (math-div (list 'calcFunc-ln v) (nth 1 p1))))
	       ((math-zerop
		 (setq part (math-add (math-mul 2
						(math-mul (nth 2 p1)
							  math-integ-var))
				      (nth 1 p1))
		       q (math-sub (math-mul 4
					     (math-mul (nth 0 p1)
						       (nth 2 p1)))
				   (math-sqr (nth 1 p1)))))
		(math-div (math-mul -2 u) part))
	       ((math-negp q)
		(setq rq (list 'calcFunc-sqrt (math-neg q)))
		(math-div (math-mul u
				    (list 'calcFunc-ln
					  (math-div (math-add part rq)
						    (math-sub part rq))))
			  rq))
	       (t
		(setq rq (list 'calcFunc-sqrt q))
		(math-div (math-mul 2
				    (math-mul u
					      (list 'calcFunc-arctan
						    (math-div part rq))))
			  rq))))))



;;; Attempt to reduce lhs = rhs to solve-var = rhs', where solve-var appears
;;; in lhs but not in rhs or rhs'; return rhs'.
(defun math-try-solve-for (lhs rhs)    ; uses global values: solve-*.
  (let (t1 t2 t3)
    (cond ((equal lhs solve-var)
	   rhs)
	  ((Math-primp lhs)
	   nil)
	  ((setq t2 (math-polynomial-base
		     lhs
		     (function (lambda (b)
				 (and (setq t1 (math-is-polynomial lhs b 2))
				      (math-expr-depends b solve-var)
				      (not (equal b lhs)))))))
	   (if (cdr (cdr t1))
	       (math-try-solve-for
		t2
		(if (math-looks-evenp (nth 1 t1))
		    (let ((halfb (math-div (nth 1 t1) 2)))
		      (math-div
		       (math-add
			(math-neg halfb)
			(math-solve-get-sign
			 (math-normalize
			  (list 'calcFunc-sqrt
				(math-add (math-sqr halfb)
					  (math-mul (math-sub rhs (car t1))
						    (nth 2 t1)))))))
		       (nth 2 t1)))
		  (math-div
		   (math-add
		    (math-neg (nth 1 t1))
		    (math-solve-get-sign
		     (math-normalize
		      (list 'calcFunc-sqrt
			    (math-add (math-sqr (nth 1 t1))
				      (math-mul 4
						(math-mul (math-sub rhs
								    (car t1))
							  (nth 2 t1))))))))
		   (math-mul 2 (nth 2 t1)))))
	     (and (cdr t1)
		  (math-try-solve-for t2
				      (math-div (math-sub rhs (car t1))
						(nth 1 t1))))))
	  ((eq (car lhs) '+)
	   (cond ((not (math-expr-depends (nth 1 lhs) solve-var))
		  (math-try-solve-for (nth 2 lhs)
				      (math-sub rhs (nth 1 lhs))))
		 ((not (math-expr-depends (nth 2 lhs) solve-var))
		  (math-try-solve-for (nth 1 lhs)
				      (math-sub rhs (nth 2 lhs))))))
	  ((memq (car lhs) '(- calcFunc-eq))
	   (cond ((not (math-expr-depends (nth 1 lhs) solve-var))
		  (math-try-solve-for (nth 2 lhs)
				      (math-sub (nth 1 lhs) rhs)))
		 ((not (math-expr-depends (nth 2 lhs) solve-var))
		  (math-try-solve-for (nth 1 lhs)
				      (math-add rhs (nth 2 lhs))))))
	  ((eq (car lhs) 'neg)
	   (math-try-solve-for (nth 1 lhs) (math-neg rhs)))
	  ((eq (car lhs) '*)
	   (cond ((not (math-expr-depends (nth 1 lhs) solve-var))
		  (math-try-solve-for (nth 2 lhs)
				      (math-div rhs (nth 1 lhs))))
		 ((not (math-expr-depends (nth 2 lhs) solve-var))
		  (math-try-solve-for (nth 1 lhs)
				      (math-div rhs (nth 2 lhs))))))
	  ((eq (car lhs) '/)
	   (cond ((not (math-expr-depends (nth 1 lhs) solve-var))
		  (math-try-solve-for (nth 2 lhs)
				      (math-div (nth 1 lhs) rhs)))
		 ((not (math-expr-depends (nth 2 lhs) solve-var))
		  (math-try-solve-for (nth 1 lhs)
				      (math-mul rhs (nth 2 lhs))))
		 ((and (setq t1 (math-is-polynomial (nth 1 lhs) solve-var 2))
		       (setq t2 (math-is-polynomial (nth 2 lhs) solve-var 2)))
		  (math-try-solve-for (math-build-polynomial-expr
				       (math-poly-mix t2 rhs t1 -1)
				       solve-var)
				      0))
		 ((setq t3 (math-polynomial-base
			    (nth 1 lhs)
			    (function (lambda (b)
					(and (math-expr-depends b solve-var)
					     (setq t1 (math-is-polynomial
						       (nth 1 lhs) b 2))
					     (setq t2 (math-is-polynomial
						       (nth 2 lhs) b 2)))))))
		  (math-try-solve-for (math-build-polynomial-expr
				       (math-poly-mix t2 rhs t1 -1)
				       t3)
				      0))))
	  ((eq (car lhs) '^)
	   (cond ((not (math-expr-depends (nth 1 lhs) solve-var))
		  (math-try-solve-for
		   (nth 2 lhs)
		   (math-add (math-normalize
			      (list 'calcFunc-log rhs (nth 1 lhs)))
			     (math-div
			      (math-mul 2
					(math-mul '(var pi var-pi)
						  (math-solve-get-int
						   '(var i var-i))))
			      (math-normalize
			       (list 'calcFunc-ln (nth 1 lhs)))))))
		 ((not (math-expr-depends (nth 2 lhs) solve-var))
		  (cond ((math-equal-int (nth 2 lhs) 2)
			 (math-try-solve-for
			  (nth 1 lhs)
			  (math-solve-get-sign
			   (math-normalize (list 'calcFunc-sqrt rhs)))))
			(t (math-try-solve-for
			    (nth 1 lhs)
			    (math-mul
			     (math-normalize
			      (list 'calcFunc-exp
				    (if (Math-realp (nth 2 lhs))
					(math-div (math-mul
						   '(var pi var-pi)
						   (math-solve-get-int
						    '(var i var-i)))
						  (math-div (nth 2 lhs) 2))
				      (math-div (math-mul
						 2
						 (math-mul
						  '(var pi var-pi)
						  (math-solve-get-int
						   '(var i var-i))))
						(nth 2 lhs)))))
			     (math-normalize
			      (list '^
				    rhs
				    (math-div 1 (nth 2 lhs)))))))))))
	  ((and (eq (car lhs) '%)
		(not (math-expr-depends (nth 2 lhs) solve-var)))
	   (math-try-solve-for (nth 1 lhs) (math-add rhs
						     (math-solve-get-int
						      (nth 2 lhs)))))
	  ((and (= (length lhs) 2)
		(symbolp (car lhs))
		(setq t1 (get (car lhs) 'math-inverse))
		(setq t2 (funcall t1 rhs)))
	   (math-try-solve-for (nth 1 lhs) (math-normalize t2)))
	  (t
	   (calc-record-why "No inverse known" lhs)
	   nil)))
)

(defun math-get-from-counter (name)
  (let ((ctr (assq name calc-command-flags)))
    (if ctr
	(setcdr ctr (1+ (cdr ctr)))
      (setq ctr (cons name 1)
	    calc-command-flags (cons ctr calc-command-flags)))
    (cdr ctr))
)

(defun math-solve-get-sign (val)
  (if solve-full
      (let ((var (concat "s" (math-get-from-counter 'solve-sign))))
	(math-mul (list 'var (intern var) (intern (concat "var-" var)))
		  val))
    (calc-record-why "Choosing positive solution")
    val)
)

(defun math-solve-get-int (val)
  (if solve-full
      (let ((var (concat "n" (math-get-from-counter 'solve-int))))
	(math-mul val
		  (list 'var (intern var) (intern (concat "var-" var)))))
    (calc-record-why "Choosing 0 for arbitrary integer in solution")
    0)
)

(defun math-looks-evenp (expr)
  (if (Math-integerp expr)
      (math-evenp expr)
    (if (memq (car expr) '(* /))
	(math-looks-evenp (nth 1 expr))))
)

(defun math-solve-for (lhs rhs solve-var solve-full)
  (if (math-expr-contains rhs solve-var)
      (math-solve-for (math-sub lhs rhs) 0 solve-var solve-full)
    (and (math-expr-contains lhs solve-var)
	 (math-try-solve-for lhs rhs)))
)

(defun calcFunc-solve (expr var)
  (let ((res (math-solve-for expr 0 var nil)))
    (if res
	(list 'calcFunc-eq var res)
      (list 'calcFunc-solve expr var)))
)

(defun calcFunc-fsolve (expr var)
  (let ((res (math-solve-for expr 0 var t)))
    (if res
	(list 'calcFunc-eq var res)
      (list 'calcFunc-fsolve expr var)))
)

(defun calcFunc-finv (expr var)
  (let ((res (math-solve-for expr math-integ-var var nil)))
    (if res
	(math-normalize (math-expr-subst res math-integ-var var))
      (list 'calcFunc-finv expr var)))
)

(defun calcFunc-ffinv (expr var)
  (let ((res (math-solve-for expr math-integ-var var t)))
    (if res
	(math-normalize (math-expr-subst res math-integ-var var))
      (list 'calcFunc-finv expr var)))
)


(put 'calcFunc-inv 'math-inverse
     (function (lambda (x) (math-div 1 x))))

(put 'calcFunc-sqrt 'math-inverse
     (function (lambda (x) (math-sqr x))))

(put 'calcFunc-conj 'math-inverse
     (function (lambda (x) (list 'calcFunc-conj x))))

(put 'calcFunc-abs 'math-inverse
     (function (lambda (x) (math-solve-get-sign x))))

(put 'calcFunc-deg 'math-inverse
     (function (lambda (x) (list 'calcFunc-rad x))))

(put 'calcFunc-rad 'math-inverse
     (function (lambda (x) (list 'calcFunc-deg x))))

(put 'calcFunc-ln 'math-inverse
     (function (lambda (x) (list 'calcFunc-exp x))))

(put 'calcFunc-log10 'math-inverse
     (function (lambda (x) (list 'calcFunc-exp10 x))))

(put 'calcFunc-lnp1 'math-inverse
     (function (lambda (x) (list 'calcFunc-expm1 x))))

(put 'calcFunc-exp 'math-inverse
     (function (lambda (x) (math-add (math-normalize (list 'calcFunc-ln x))
				     (math-mul 2
					       (math-mul '(var pi var-pi)
							 (math-solve-get-int
							  '(var i var-i))))))))

(put 'calcFunc-expm1 'math-inverse
     (function (lambda (x) (math-add (math-normalize (list 'calcFunc-lnp1 x))
				     (math-mul 2
					       (math-mul '(var pi var-pi)
							 (math-solve-get-int
							  '(var i var-i))))))))

(put 'calcFunc-sin 'math-inverse
     (function (lambda (x) (let ((n (math-solve-get-int 1)))
			     (math-add (math-mul (math-normalize
						  (list 'calcFunc-arcsin x))
						 (math-pow -1 n))
				       (math-mul (math-half-circle t)
						 n))))))

(put 'calcFunc-cos 'math-inverse
     (function (lambda (x) (math-add (math-solve-get-sign
				      (math-normalize
				       (list 'calcFunc-arccos x)))
				     (math-solve-get-int
				      (math-full-circle t))))))

(put 'calcFunc-tan 'math-inverse
     (function (lambda (x) (math-add (math-normalize (list 'calcFunc-arctan x))
				     (math-solve-get-int
				      (math-half-circle t))))))

(put 'calcFunc-arcsin 'math-inverse
     (function (lambda (x) (math-normalize (list 'calcFunc-sin x)))))

(put 'calcFunc-arccos 'math-inverse
     (function (lambda (x) (math-normalize (list 'calcFunc-cos x)))))

(put 'calcFunc-arctan 'math-inverse
     (function (lambda (x) (math-normalize (list 'calcFunc-tan x)))))

(put 'calcFunc-sinh 'math-inverse
     (function (lambda (x) (let ((n (math-solve-get-int 1)))
			     (math-add (math-mul (math-normalize
						  (list 'calcFunc-arctanh x))
						 (math-pow -1 n))
				       (math-mul (math-half-circle t)
						 (math-mul
						  '(var i var-i)
						  n)))))))

(put 'calcFunc-cosh 'math-inverse
     (function (lambda (x) (math-add (math-solve-get-sign
				      (math-normalize
				       (list 'calcFunc-arctanh x)))
				     (math-mul (math-full-circle t)
					       (math-solve-get-int
						'(var i var-i)))))))

(put 'calcFunc-tanh 'math-inverse
     (function (lambda (x) (math-add (math-normalize
				      (list 'calcFunc-arctanh x))
				     (math-mul (math-half-circle t)
					       (math-solve-get-int
						'(var i var-i)))))))

(put 'calcFunc-arcsinh 'math-inverse
     (function (lambda (x) (math-normalize (list 'calcFunc-sinh x)))))

(put 'calcFunc-arccosh 'math-inverse
     (function (lambda (x) (math-normalize (list 'calcFunc-cosh x)))))

(put 'calcFunc-arctanh 'math-inverse
     (function (lambda (x) (math-normalize (list 'calcFunc-tanh x)))))



(defun calcFunc-taylor (expr var num)
  (let ((x0 0) (v var))
    (if (memq (car-safe var) '(+ - calcFunc-eq))
	(setq x0 (if (eq (car var) '+) (math-neg (nth 2 var)) (nth 2 var))
	      v (nth 1 var)))
    (or (and (eq (car-safe v) 'var)
	     (math-expr-contains expr v)
	     (natnump num)
	     (let ((accum (math-expr-subst expr v x0))
		   (var2 (if (eq (car var) 'calcFunc-eq)
			     (cons '- (cdr var))
			   var))
		   (n 0)
		   (nfac 1)
		   (fprime expr))
	       (while (and (<= (setq n (1+ n)) num)
			   (setq fprime (calcFunc-deriv fprime v nil t)))
		 (setq fprime (math-simplify fprime)
		       nfac (math-mul nfac n)
		       accum (math-add accum
				       (math-div (math-mul (math-pow var2 n)
							   (math-expr-subst
							    fprime v x0))
						 nfac))))
	       (and fprime
		    (math-normalize accum))))
	(list 'calcFunc-taylor expr var num)))
)




;;;; [calc-alg.el]

;;; Simple operations on expressions.

;;; Return number of ocurrences of thing in expr, or nil if none.
(defun math-expr-contains (expr thing)
  (cond ((equal expr thing) 1)
	((Math-primp expr) nil)
	(t
	 (let ((num 0))
	   (while (setq expr (cdr expr))
	     (setq num (+ num (or (math-expr-contains (car expr) thing) 0))))
	   (and (> num 0)
		num))))
)

;;; Return non-nil if any variable of thing occurs in expr.
(defun math-expr-depends (expr thing)
  (if (Math-primp thing)
      (and (eq (car-safe thing) 'var)
	   (math-expr-contains expr thing))
    (while (and (setq thing (cdr thing))
		(not (math-expr-depends expr (car thing)))))
    thing)
)

;;; Substitute all occurrences of old for new in expr (non-destructive).
(defun math-expr-subst (expr old new)
  (math-expr-subst-rec expr)
)

(defun math-expr-subst-rec (expr)
  (cond ((equal expr old) new)
	((Math-primp expr) expr)
	((memq (car expr) '(calcFunc-deriv
			    calcFunc-tderiv))
	 (if (= (length expr) 2)
	     (if (equal (nth 1 expr) old)
		 (append expr (list new))
	       expr)
	   (list (car expr) (nth 1 expr)
		 (math-expr-subst-rec (nth 2 expr)))))
	(t
	 (cons (car expr)
	       (mapcar 'math-expr-subst-rec (cdr expr)))))
)

;;; Various measures of the size of an expression.
(defun math-expr-weight (expr)
  (if (Math-primp expr)
      1
    (let ((w 1))
      (while (setq expr (cdr expr))
	(setq w (+ w (math-expr-weight (car expr)))))
      w))
)

(defun math-expr-height (expr)
  (if (Math-primp expr)
      0
    (let ((h 0))
      (while (setq expr (cdr expr))
	(setq h (max h (math-expr-height (car expr)))))
      (1+ h)))
)




;;; Polynomial operations (to support the integrator and solve-for).

(defun math-collect-terms (expr base)
  (let ((p (math-is-polynomial expr base 20 t)))
    (if (cdr p)
	(math-build-polynomial-expr p base)
      expr))
)

;;; If expr is of the form "a + bx + cx^2 + ...", return the list (a b c ...),
;;; else return nil if not in polynomial form.  If "loose", coefficients
;;; may contain x, e.g., sin(x) + cos(x) x^2 is a loose polynomial in x.
(defun math-is-polynomial (expr var &optional degree loose)
  (let ((poly (math-is-poly-rec expr)))
    (and (or (null degree)
	     (<= (length poly) (1+ degree)))
	 poly))
)

(defun math-is-poly-rec (expr)
  (math-poly-simplify
   (or (cond ((equal expr var)
	      (list 0 1))
	     ((Math-objectp expr)
	      (list expr))
	     ((memq (car expr) '(+ -))
	      (let ((p1 (math-is-poly-rec (nth 1 expr))))
		(and p1
		     (let ((p2 (math-is-poly-rec (nth 2 expr))))
		       (and p2
			    (math-poly-mix p1 1 p2
					   (if (eq (car expr) '+) 1 -1)))))))
	     ((eq (car expr) 'neg)
	      (mapcar 'math-neg (math-is-poly-rec (nth 1 expr))))
	     ((eq (car expr) '*)
	      (let ((p1 (math-is-poly-rec (nth 1 expr))))
		(and p1
		     (let ((p2 (math-is-poly-rec (nth 2 expr))))
		       (and p2
			    (or (null degree)
				(<= (- (+ (length p1) (length p2)) 2) degree))
			    (math-poly-mul p1 p2))))))
	     ((eq (car expr) '/)
	      (and (not (math-expr-depends (nth 2 expr) var))
		   (not (Math-zerop (nth 2 expr)))
		   (let ((p1 (math-is-poly-rec (nth 1 expr))))
		     (mapcar (function (lambda (x) (math-div x (nth 2 expr))))
			     p1))))
	     ((eq (car expr) '^)
	      (and (natnump (nth 2 expr))
		   (let ((p1 (math-is-poly-rec (nth 1 expr)))
			 (n (nth 2 expr))
			 (accum (list 1)))
		     (and p1
			  (or (null degree)
			      (<= (* (1- (length p1)) n) degree))
			  (progn
			    (while (>= n 1)
			      (setq accum (math-poly-mul accum p1)
				    n (1- n)))
			    accum)))))
	     (t nil))
       (and (or (not (math-expr-depends expr var))
		loose)
	    (not (memq (car expr) '(vec)))
	    (list expr))))
)

;;; Check if expr is a polynomial in var; if so, return its degree.
(defun math-polynomial-p (expr var)
  (cond ((equal expr var) 1)
	((Math-primp expr) 0)
	((memq (car expr) '(+ -))
	 (let ((p1 (math-polynomial-p (nth 1 expr) var))
	       (p2 (math-polynomial-p (nth 2 expr) var)))
	   (and p1 p2 (max p1 p2))))
	((eq (car expr) '*)
	 (let ((p1 (math-polynomial-p (nth 1 expr) var))
	       (p2 (math-polynomial-p (nth 2 expr) var)))
	   (and p1 p2 (+ p1 p2))))
	((eq (car expr) 'neg)
	 (math-polynomial-p (nth 1 expr) var))
	((and (eq (car expr) '/)
	      (not (math-expr-depends (nth 1 expr) var)))
	 (math-polynomial-p (nth 1 expr) var))
	((and (eq (car expr) '^)
	      (natnump (nth 2 expr)))
	 (let ((p1 (math-polynomial-p (nth 1 expr) var)))
	   (and p1 (* p1 (nth 2 expr)))))
	((math-expr-depends expr var) nil)
	(t 0))
)

;;; Find the variable (or sub-expression) which is the base of polynomial expr.
(defun math-polynomial-base (mpb-top-expr &optional mpb-pred)
  (or mpb-pred
      (setq mpb-pred (function (lambda (base) (math-polynomial-p
					       mpb-top-expr base)))))
  (or (let ((const-ok nil))
	(math-polynomial-base-rec mpb-top-expr))
      (let ((const-ok t))
	(math-polynomial-base-rec mpb-top-expr)))
)

(defun math-polynomial-base-rec (mpb-expr)
  (and (not (Math-objvecp mpb-expr))
       (or (and (memq (car mpb-expr) '(+ - *))
		(or (math-polynomial-base-rec (nth 1 mpb-expr))
		    (math-polynomial-base-rec (nth 2 mpb-expr))))
	   (and (memq (car mpb-expr) '(/ neg))
		(math-polynomial-base-rec (nth 1 mpb-expr)))
	   (and (eq (car mpb-expr) '^)
		(natnump (nth 2 mpb-expr))
		(math-polynomial-base-rec (nth 1 mpb-expr)))
	   (and (or const-ok (math-expr-contains-vars mpb-expr))
		(funcall mpb-pred mpb-expr)
		mpb-expr)))
)

;;; Return non-nil if expr refers to any variables.
(defun math-expr-contains-vars (expr)
  (or (eq (car-safe expr) 'var)
      (and (not (Math-primp expr))
	   (progn
	     (while (and (setq expr (cdr expr))
			 (not (math-expr-contains-vars (car expr)))))
	     expr)))
)

;;; Simplify a polynomial in list form by stripping off high-end zeros.
;;; This always leaves the constant part, i.e., nil->nil and nonnil->nonnil.
(defun math-poly-simplify (p)
  (and p
       (if (Math-zerop (nth (1- (length p)) p))
	   (let ((pp (copy-sequence p)))
	     (while (and (cdr pp)
			 (Math-zerop (nth (1- (length pp)) pp)))
	       (setcdr (nthcdr (- (length pp) 2) pp) nil))
	     pp)
	 p))
)

;;; Compute ac*a + bc*b for polynomials in list form a, b and
;;; coefficients ac, bc.  Result may be unsimplified.
(defun math-poly-mix (a ac b bc)
  (and (or a b)
       (cons (math-add (math-mul (or (car a) 0) ac)
		       (math-mul (or (car b) 0) bc))
	     (math-poly-mix (cdr a) ac (cdr b) bc)))
)

;;; Multiply two polynomials in list form.
(defun math-poly-mul (a b)
  (and a b
       (math-poly-mix b (car a)
		      (math-poly-mul (cdr a) (cons 0 b)) 1))
)

;;; Build an expression from a polynomial list.
(defun math-build-polynomial-expr (p var)
  (if p
      (let ((accum (car p))
	    (n 0))
	(while (setq p (cdr p))
	  (setq n (1+ n)
		accum (math-add (math-mul (car p) (math-pow var n)) accum)))
	accum))
)




;;;; [calc-units.el]

;;; Units operations.

(defvar math-standard-units
  '( ;; Length
     ( m       nil		     "*Meter" )
     ( in      "2.54 cm"             "Inch" )
     ( ft      "12 in"		     "Foot" )
     ( yd      "3 ft"		     "Yard" )
     ( mi      "5280 ft"	     "Mile" )
     ( au      "1.495979e11 m"       "Astronomical Unit" )
     ( lyr     "9460536207068016 m"  "Light Year" )
     ( pc      "206264.80625 au"     "Parsec" )
     ( nmi     "1852 m"		     "Nautical Mile" )
     ( fath    "6 ft"		     "Fathom" )
     ( u       "1 um"		     "Micron" )
     ( mil     "in/1000"	     "Mil" )
     ( point   "in/72"		     "Point" )
     ( Ang     "1e-10 m"	     "Angstrom" )
     
     ;; Area
     ( hect    "10000 m^2"	     "*Hectare" )
     ( acre    "mi^2 / 640"	     "Acre" )
     ( b       "1e-28 m^2"	     "Barn" )
     
     ;; Volume
     ( l       "1e-3 m^3"	     "*Liter" )
     ( L       "1e-3 m^3"	     "Liter" )
     ( gal     "4 qt"		     "US Gallon" )
     ( qt      "2 pt"		     "Quart" )
     ( pt      "2 cup"		     "Pint" )
     ( cup     "8 ozfl"		     "Cup" )
     ( ozfl    "2 tbsp"		     "Fluid Ounce" )
     ( tbsp    "3 tsp"		     "Tablespoon" )
     ( tsp     "4.92892159375 ml"    "Teaspoon" )
     ( galC    "4.54609 l"	     "Canadian Gallon" )
     ( galUK   "4.546092 l"	     "UK Gallon" )
     
     ;; Time
     ( s       nil		     "*Second" )
     ( min     "60 s"		     "Minute" )
     ( hr      "60 min"		     "Hour" )
     ( day     "24 hr"		     "Day" )
     ( wk      "7 day"		     "Week" )
     ( yr      "365.25 day"	     "Year" )
     ( Hz      "1/s"		     "Hertz" )

     ;; Speed
     ( mph     "mi/hr"		     "*Miles per hour" )
     ( kph     "km/hr"		     "Kilometers per hour" )
     ( knot    "nmi/hr"		     "Knot" )
     ( c       "2.99792458e8 m/s"    "Speed of light" )     
     
     ;; Acceleration
     ( ga      "9.80665 m/s^2"	     "*\"g\" acceleration" )

     ;; Mass
     ( g       nil                   "*Gram" )
     ( lb      "16 oz"		     "Pound (mass)" )
     ( oz      "28.349523125 g"	     "Ounce (mass)" )
     ( ton     "2000 lb"	     "Ton" )
     ( t       "1000 kg"	     "Metric ton" )
     ( tonUK   "1016.0469088 kg"     "UK ton" )
     ( lbt     "12 ozt"		     "Troy pound" )
     ( ozt     "31.103475 g"	     "Troy ounce" )
     ( ct      ".2 g"		     "Carat" )
     ( amu     "1.6605655e-24 g"     "Unified atomic mass" )

     ;; Force
     ( N       "m kg/s^2"	     "*Newton" )
     ( dyn     "1e-5 N"		     "Dyne" )
     ( gf      "9.60665e-3 N"	     "Gram (force)" )
     ( lbf     "4.44822161526 N"     "Pound (force)" )
     ( kip     "1000 lbf"	     "Kilopound (force)" )
     ( pdl     "0.138255 N"	     "Poundal" )

     ;; Energy
     ( J       "N m"		     "*Joule" )
     ( erg     "1e-7 J"		     "Erg" )
     ( cal     "4.1868 J"	     "International Table Calorie" )
     ( Btu     "1055.05585262 J"     "International Table Btu" )
     ( eV      "1.6021892e-19 J"     "Electron volt" )
     ( therm   "105506000 J"	     "EEC therm" )

     ;; Power
     ( W       "J/s"		     "*Watt" )
     ( hp      "745.7 W"	     "Horsepower" )

     ;; Temperature
     ( K       nil                   "*Degree Kelvin"     K )
     ( dK      "K"		     "Degree Kelvin"	  K )
     ( degK    "K"		     "Degree Kelvin"	  K )
     ( dC      "K"		     "Degree Celsius"	  C )
     ( degC    "K"      	     "Degree Celsius"	  C )
     ( dF      "(5/9) K"	     "Degree Fahrenheit"  F )
     ( degF    "(5/9) K"	     "Degree Fahrenheit"  F )

     ;; Pressure
     ( Pa      "N/m^2"		     "*Pascal" )
     ( bar     "1e5 Pa"		     "Bar" )
     ( atm     "101325 Pa"	     "Standard atmosphere" )
     ( torr    "atm/760"	     "Torr" )
     ( mHg     "1000 torr"	     "Meter of mercury" )
     ( inHg    "25.4 mmHg"	     "Inch of mercury" )
     ( inH2O   "248.84 Pa"	     "Inch of water" )
     ( psi     "6894.75729317 Pa"    "Pound per square inch" )

     ;; Viscosity
     ( P       "0.1 Pa s"	     "*Poise" )
     ( St      "1e-4 m^2/s"	     "Stokes" )

     ;; Electromagnetism
     ( A       nil                   "*Ampere" )
     ( C       "A s"		     "Coulomb" )
     ( Fdy     "96487 C"	     "Faraday" )
     ( e       "1.6021892e-19 C"     "Elementary charge" )
     ( V       "W/A"		     "Volt" )
     ( ohm     "V/A"		     "Ohm" )
     ( mho     "A/V"		     "Mho" )
     ( S       "A/V"		     "Siemens" )
     ( F       "C/V"		     "Farad" )
     ( H       "Wb/A"		     "Henry" )
     ( T       "Wb/m^2"		     "Tesla" )
     ( G       "1e-4 T"		     "Gauss" )
     ( Wb      "V s"		     "Weber" )

     ;; Luminous intensity
     ( cd      nil                   "*Candela" )
     ( sb      "1e4 cd/m^2"	     "Stilb" )
     ( lm      "cd sr"		     "Lumen" )
     ( lx      "lm/m^2"		     "Lux" )
     ( ph      "1e4 lx"		     "Phot" )
     ( fc      "10.76 lx"	     "Footcandle" )
     ( lam     "1e4 lm/m^2"	     "Lambert" )
     ( flam    "1.07639104e-3 lam"   "Footlambert" )

     ;; Radioactivity
     ( Bq      "1/s"  		     "*Becquerel" )
     ( Ci      "3.7e8 Bq"	     "Curie" )
     ( Gy      "J/kg"		     "Gray" )
     ( Sv      "Gy"		     "Sievert" )
     ( R       "2.58e-4 C/kg"	     "Roentgen" )
     ( rd      ".01 Sv"		     "Rad" )
     ( rem     "rd"		     "Rem" )

     ;; Amount of substance
     ( mol     nil                   "*Mole" )

     ;; Plane angle
     ( rad     nil                   "*Radian" )
     ( circ    "2 pi rad"	     "Full circle" )
     ( deg     "circ/360"            "Degree" )
     ( arcmin  "deg/60"		     "Arc minute" )
     ( arcsec  "arcmin/60"	     "Arc second" )
     ( grad    "circ/400"            "Grade" )

     ;; Solid angle
     ( sr      nil		     "*Steradian" )

     ;; Other physical quantities (CRC chem & phys, 67th ed)
     ( h       "6.626176e-34 J s"    "*Planck's constant" )
     ( hbar    "h / 2 pi"	     "Planck's constant" )
     ( mu0     "4 pi 1e-7 H/m"       "Permeability of vacuum" )
     ( Grav    "6.6720e-11 N m^2/kg^2"  "Gravitational constant" )
     ( Nav     "6.0222e23 / mol"     "Avagadro's constant" )
     ( me      "9.109534e-31 kg"     "Electron rest mass" )
     ( mp      "1.6726485e-27 kg"    "Proton rest mass" )
     ( mn      "1.6749543e-27 kg"    "Neutron rest mass" )
     ( mu      "1.883566e-28 kg"     "Muon rest mass" )
     ( Ryd     "1.097373177e7 / m"   "Rydberg's constant" )
     ( k       "Ryd / Nav"	     "Boltzmann's constant" )
     ( fsc     "7.2973506e-3"	     "Fine structure constant" )
     ( mue     "9.284832e-24 J/T"    "Electron magnetic moment" )
     ( mup     "1.4106171e-26 J/T"   "Proton magnetic moment" )
     ( R0      "8.31441 J/mol K"     "Molar gas constant" )
     ( V0      "22.4136 L/mol"	     "Standard volume of ideal gas" )
))


(defvar math-additional-units nil
  "*Additional units table for user-defined units.
Must be formatted like math-standard-units.
If this is changed, be sure to set math-units-table to nil to ensure
that the combined units table will be rebuilt.")

(defvar math-unit-prefixes
  '( ( ?E  (float 1 18)  "Exa"    )
     ( ?P  (float 1 15)  "Peta"   )
     ( ?T  (float 1 12)  "Tera"	  )
     ( ?G  (float 1 9)   "Giga"	  )
     ( ?M  (float 1 6)   "Mega"	  )
     ( ?k  (float 1 3)   "Kilo"	  )
     ( ?K  (float 1 3)   "Kilo"	  )
     ( ?h  (float 1 2)   "Hecto"  )
     ( ?H  (float 1 2)   "Hecto"  )
     ( ?D  (float 1 1)   "Deka"	  )
     ( ?d  (float 1 -1)  "Deci"	  )
     ( ?c  (float 1 -2)  "Centi"  )
     ( ?m  (float 1 -3)  "Milli"  )
     ( ?u  (float 1 -6)  "Micro"  )
     ( ?n  (float 1 -9)  "Nano"	  )
     ( ?p  (float 1 -12) "Pico"	  )
     ( ?f  (float 1 -15) "Femto"  )
     ( ?a  (float 1 -18) "Atto"   )
))

(defvar math-standard-units-systems
  '( ( base  nil )
     ( si    ( ( g   '(* (var kg var-kg) (float 1 -3)) ) ) )
     ( mks   ( ( g   '(* (var kg var-kg) (float 1 -3)) ) ) )
     ( cgs   ( ( m   '(* (var cm var-cm) 100         ) ) ) )
))

(defvar math-units-table nil
  "Internal units table derived from math-defined-units.
Entries are (SYMBOL EXPR DOC-STRING TEMP-TYPE BASE-UNITS).")

(defvar math-units-table-buffer-valid nil)


(defun math-build-units-table ()
  (or math-units-table
      (let* ((combined-units (append math-additional-units
				     math-standard-units))
	     (unit-list (mapcar 'car combined-units))
	     (calc-language nil)
	     (math-expr-opers math-standard-opers)
	     tab)
	(message "Building units table...")
	(setq math-units-table-buffer-valid nil)
	(setq tab (mapcar (function
			   (lambda (x)
			     (list (car x)
				   (and (nth 1 x)
					(if (stringp (nth 1 x))
					    (let ((exp (math-read-expr
							(nth 1 x))))
					      (if (eq (car-safe exp) 'error)
						  (error "Format error in definition of %s in units table: %s"
							 (car x) (nth 2 exp))
						exp))
					  (nth 1 x)))
				   (nth 2 x)
				   (nth 3 x)
				   (and (not (nth 1 x))
					(list (cons (car x) 1))))))
			  combined-units))
	(let ((math-units-table tab))
	  (mapcar 'math-find-base-units tab))
	(message "Building units table...done")
	(setq math-units-table tab)))
)

(defun math-find-base-units (entry)
  (if (eq (nth 4 entry) 'boom)
      (error "Circular definition involving unit %s" (car entry)))
  (or (nth 4 entry)
      (let (base)
	(setcar (nthcdr 4 entry) 'boom)
	(math-find-base-units-rec (nth 1 entry) 1)
	'(or base
	    (error "Dimensionless definition for unit %s" (car entry)))
	(while (eq (cdr (car base)) 0)
	  (setq base (cdr base)))
	(let ((b base))
	  (while (cdr b)
	    (if (eq (cdr (car (cdr b))) 0)
		(setcdr b (cdr (cdr b)))
	      (setq b (cdr b)))))
	(setq base (sort base 'math-compare-unit-names))
	(setcar (nthcdr 4 entry) base)
	base))
)

(defun math-compare-unit-names (a b)
  (memq (car b) (cdr (memq (car a) unit-list)))
)

(defun math-find-base-units-rec (expr pow)
  (let ((u (math-check-unit-name expr)))
    (cond (u
	   (let ((ulist (math-find-base-units u)))
	     (while ulist
	       (let ((p (* (cdr (car ulist)) pow))
		     (old (assq (car (car ulist)) base)))
		 (if old
		     (setcdr old (+ (cdr old) p))
		   (setq base (cons (cons (car (car ulist)) p) base))))
	       (setq ulist (cdr ulist)))))
	  ((math-scalarp expr))
	  ((and (eq (car expr) '^)
		(integerp (nth 2 expr)))
	   (math-find-base-units-rec (nth 1 expr) (* pow (nth 2 expr))))
	  ((eq (car expr) '*)
	   (math-find-base-units-rec (nth 1 expr) pow)
	   (math-find-base-units-rec (nth 2 expr) pow))
	  ((eq (car expr) '/)
	   (math-find-base-units-rec (nth 1 expr) pow)
	   (math-find-base-units-rec (nth 2 expr) (- pow)))
	  ((eq (car expr) 'neg)
	   (math-find-base-units-rec (nth 1 expr) pow))
	  ((eq (car expr) 'var)
	   (or (eq (nth 1 expr) 'pi)
	       (error "Unknown name %s in defining expression for unit %s"
		      (nth 1 expr) (car entry))))
	  (t (error "Malformed defining expression for unit %s" (car entry)))))
)


(defun math-units-in-expr-p (expr sub-exprs)
  (and (consp expr)
       (if (eq (car expr) 'var)
	   (math-check-unit-name expr)
	 (and (or sub-exprs
		  (memq (car expr) '(* / ^)))
	      (or (math-units-in-expr-p (nth 1 expr) sub-exprs)
		  (math-units-in-expr-p (nth 2 expr) sub-exprs)))))
)

(defun math-only-units-in-expr-p (expr)
  (and (consp expr)
       (if (eq (car expr) 'var)
	   (math-check-unit-name expr)
	 (if (memq (car expr) '(* /))
	     (and (math-only-units-in-expr-p (nth 1 expr))
		  (math-only-units-in-expr-p (nth 2 expr)))
	   (and (eq (car expr) '^)
		(and (math-only-units-in-expr-p (nth 1 expr))
		     (math-realp (nth 2 expr)))))))
)

(defun math-single-units-in-expr-p (expr)
  (cond ((math-scalarp expr) nil)
	((eq (car expr) 'var)
	 (math-check-unit-name expr))
	((eq (car expr) '*)
	 (let ((u1 (math-single-units-in-expr-p (nth 1 expr)))
	       (u2 (math-single-units-in-expr-p (nth 2 expr))))
	   (or (and u1 u2 'wrong)
	       u1
	       u2)))
	((eq (car expr) '/)
	 (if (math-units-in-expr-p (nth 2 expr))
	     'wrong
	   (math-single-units-in-expr-p (nth 1 expr))))
	(t 'wrong))
)

(defun math-check-unit-name (v)
  (and (eq (car-safe v) 'var)
       (or (assq (nth 1 v) (or math-units-table (math-build-units-table)))
	   (let ((name (symbol-name (nth 1 v))))
	     (and (> (length name) 1)
		  (assq (aref name 0) math-unit-prefixes)
		  (or (assq (intern (substring name 1)) math-units-table)
		      (and (eq (aref name 0) ?M)
			   (> (length name) 3)
			   (eq (aref name 1) ?e)
			   (eq (aref name 2) ?g)
			   (assq (intern (substring name 3))
				 math-units-table)))))))
)


(defun math-to-standard-units (expr which-standard)
  (math-to-standard-rec expr)
)

(defun math-to-standard-rec (expr)
  (if (eq (car-safe expr) 'var)
      (let ((u (math-check-unit-name expr))
	    (base (nth 1 expr)))
	(if u
	    (progn
	      (if (nth 1 u)
		  (setq expr (math-to-standard-rec (nth 1 u)))
		(let ((st (assq (car u) which-standard)))
		  (if st
		      (setq expr (nth 1 st))
		    (setq expr (list 'var (car u)
				     (intern (concat "var-"
						     (symbol-name
						      (car u)))))))))
	      (or (null u)
		  (eq base (car u))
		  (setq expr (list '*
				   (nth 1 (assq (aref (symbol-name base) 0)
						math-unit-prefixes))
				   expr)))
	      expr)
	  (if (eq base 'pi)
	      (math-pi)
	    expr)))
    (if (Math-primp expr)
	expr
      (cons (car expr)
	    (mapcar 'math-to-standard-rec (cdr expr)))))
)

(defun math-convert-units (expr new-units)
  (if (math-units-in-expr-p expr t)
      (math-convert-units-rec expr)
    (list '*
	  (math-to-standard-units (list '/ expr new-units) nil)
	  new-units))
)

(defun math-convert-units-rec (expr)
  (if (math-units-in-expr-p expr nil)
      (list '*
	    (math-to-standard-units (list '/ expr new-units) nil)
	    new-units)
    (if (Math-primp expr)
	expr
      (cons (car expr)
	    (mapcar 'math-convert-units-rec (cdr expr)))))
)

(defun math-convert-temperature (expr old new)
  (let* ((units (math-single-units-in-expr-p expr))
	 (uold (if old
		   (if (or (null units)
			   (equal (nth 1 old) (car units)))
		       (math-check-unit-name old)
		     (error "Inconsistent temperature units"))
		 units))
	 (unew (math-check-unit-name new)))
    (or (and (consp unew) (nth 3 unew))
	(error "Not a valid temperature unit"))
    (or (and (consp uold) (nth 3 uold))
	(error "Not a pure temperature expression"))
    (let ((v (car uold)))
      (setq expr (list '/ expr (list 'var v
				     (intern (concat "var-"
						     (symbol-name v)))))))
    (or (eq (nth 3 uold) (nth 3 unew))
	(cond ((eq (nth 3 uold) 'K)
	       (setq expr (list '- expr '(float 27315 -2)))
	       (if (eq (nth 3 unew) 'F)
		   (setq expr (list '+ (list '* expr '(frac 9 5)) 32))))
	      ((eq (nth 3 uold) 'C)
	       (if (eq (nth 3 unew) 'F)
		   (setq expr (list '+ (list '* expr '(frac 9 5)) 32))
		 (setq expr (list '+ expr '(float 27315 -2)))))
	      (t
	       (setq expr (list '* (list '- expr 32) '(frac 5 9)))
	       (if (eq (nth 3 unew) 'K)
		   (setq expr (list '+ expr '(float 27315 -2)))))))
    (list '* expr new))
)


(setq math-simplifying-units nil)

(defun math-simplify-units (a)
  (let ((math-simplifying-units t))
    (math-simplify a))
)

(math-defsimplify (+ -)
  (and math-simplifying-units
       (math-units-in-expr-p (nth 1 expr) nil)
       (let* ((units (math-extract-units (nth 1 expr)))
	      (ratio (math-simplify (math-to-standard-units
				     (list '/ (nth 2 expr) units) nil))))
	 (if (math-units-in-expr-p ratio nil)
	     (progn
	       (calc-record-why "Inconsistent units" expr)
	       expr)
	   (list '* (math-add (math-remove-units (nth 1 expr)) ratio)
		 units))))
)

(math-defsimplify /
  (and math-simplifying-units
       (let ((np (cdr expr))
	     n nn)
	 (while (eq (car-safe (setq n (car np))) '*)
	   (math-simplify-units-divisor (cdr n) (cdr (cdr expr)))
	   (setq np (cdr (cdr n))))
	 (math-simplify-units-divisor np (cdr (cdr expr)))
	 expr))
)

(defun math-simplify-units-divisor (np dp)
  (let ((n (car np))
	d dd temp)
    (while (eq (car-safe (setq d (car dp))) '*)
      (if (setq temp (math-simplify-units-quotient n (nth 1 d)))
	  (progn
	    (setcar np (setq n temp))
	    (setcar (cdr d) 1)))
      (setq dp (cdr (cdr d))))
    (if (setq temp (math-simplify-units-quotient n d))
	(progn
	  (setcar np (setq n temp))
	  (setcar dp 1))))
)

;; Simplify, e.g., "in / cm" to "2.54" in a units expression.
(defun math-simplify-units-quotient (n d)
  (let ((un (math-check-unit-name n))
	(ud (math-check-unit-name d)))
    (and un ud
	 (equal (nth 4 un) (nth 4 ud))
	 (math-to-standard-units (list '/ n d) nil)))
)

(math-defsimplify ^
  (and math-simplifying-units
       (math-realp (nth 2 expr))
       (math-simplify-units-pow (nth 1 expr) (nth 2 expr)))
)

(math-defsimplify calcFunc-sqrt
  (and math-simplifying-units
       (if (memq (car-safe (nth 1 expr)) '(* /))
	   (list (car (nth 1 expr))
		 (list 'calcFunc-sqrt (nth 1 (nth 1 expr)))
		 (list 'calcFunc-sqrt (nth 2 (nth 1 expr))))
	 (math-simplify-units-pow (nth 1 expr) '(frac 1 2))))
)

(math-defsimplify (calcFunc-floor
		   calcFunc-ceil
		   calcFunc-round
		   calcFunc-trunc
		   calcFunc-float
		   calcFunc-frac
		   calcFunc-abs
		   calcFunc-clean)
  (and math-simplifying-units
       (if (math-only-units-in-expr-p (nth 1 expr))
	   (nth 1 expr)
	 (if (and (memq (car-safe (nth 1 expr)) '(* /))
		  (or (math-only-units-in-expr-p
		       (nth 1 (nth 1 expr)))
		      (math-only-units-in-expr-p
		       (nth 2 (nth 1 expr)))))
	     (list (car (nth 1 expr))
		   (cons (car expr)
			 (cons (nth 1 (nth 1 expr))
			       (cdr (cdr expr))))
		   (cons (car expr)
			 (cons (nth 2 (nth 1 expr))
			       (cdr (cdr expr)))))))))

(defun math-simplify-units-pow (a pow)
  (if (and (eq (car-safe a) '^)
	   (math-check-unit-name (nth 1 a))
	   (math-realp (nth 2 a)))
      (list '^ (nth 1 a) (math-mul pow (nth 2 a)))
    (let* ((u (math-check-unit-name a))
	   (pf (math-to-simple-fraction pow))
	   (d (and (eq (car-safe pf) 'frac) (nth 2 pf))))
      (and u
	   (eq (car-safe pow) 'frac)
	   (math-units-are-multiple u d)
	   (list '^ (math-to-standard-units a nil) pow))))
)

;;;; [calc-alg.el]

(defun math-to-simple-fraction (f)
  (or (and (eq (car-safe f) 'float)
	   (or (and (>= (nth 2 f) 0)
		    (math-scale-int (nth 1 f) (nth 2 f)))
	       (and (integerp (nth 1 f))
		    (> (nth 1 f) -1000)
		    (< (nth 1 f) 1000)
		    (math-make-frac (nth 1 f)
				    (math-scale-int 1 (- (nth 2 f)))))))
      f)
)

;;;; [calc-units.el]

(defun math-units-are-multiple (u n)
  (setq u (nth 4 u))
  (while (and u (= (% (cdr (car u)) n) 0))
    (setq u (cdr u)))
  (null u)
)

(math-defsimplify calcFunc-sin
  (and math-simplifying-units
       (math-units-in-expr-p (nth 1 expr) nil)
       (let ((rad (math-simplify-units
		   (math-evaluate-expr
		    (math-to-standard-units (nth 1 expr) nil))))
	     (calc-angle-mode 'rad))
	 (and (eq (car-safe rad) '*)
	      (Math-realp (nth 1 rad))
	      (eq (car-safe (nth 2 rad)) 'var)
	      (eq (nth 1 (nth 2 rad)) 'rad)
	      (list 'calcFunc-sin (nth 1 rad)))))
)

(math-defsimplify calcFunc-cos
  (and math-simplifying-units
       (math-units-in-expr-p (nth 1 expr) nil)
       (let ((rad (math-simplify-units
		   (math-evaluate-expr
		    (math-to-standard-units (nth 1 expr) nil))))
	     (calc-angle-mode 'rad))
	 (and (eq (car-safe rad) '*)
	      (Math-realp (nth 1 rad))
	      (eq (car-safe (nth 2 rad)) 'var)
	      (eq (nth 1 (nth 2 rad)) 'rad)
	      (list 'calcFunc-cos (nth 1 rad)))))
)

(math-defsimplify calcFunc-tan
  (and math-simplifying-units
       (math-units-in-expr-p (nth 1 expr) nil)
       (let ((rad (math-simplify-units
		   (math-evaluate-expr
		    (math-to-standard-units (nth 1 expr) nil))))
	     (calc-angle-mode 'rad))
	 (and (eq (car-safe rad) '*)
	      (Math-realp (nth 1 rad))
	      (eq (car-safe (nth 2 rad)) 'var)
	      (eq (nth 1 (nth 2 rad)) 'rad)
	      (list 'calcFunc-tan (nth 1 rad)))))
)


(defun math-remove-units (expr)
  (if (math-check-unit-name expr)
      1
    (if (Math-primp expr)
	expr
      (cons (car expr)
	    (mapcar 'math-remove-units (cdr expr)))))
)

(defun math-extract-units (expr)
  (if (memq (car-safe expr) '(* /))
      (cons (car expr)
	    (mapcar 'math-extract-units (cdr expr)))
    (if (math-check-unit-name expr) expr 1))
)

(defun math-build-units-table-buffer (enter-buffer)
  (if (not (and math-units-table math-units-table-buffer-valid
		(get-buffer "*Units Table*")))
      (let ((buf (get-buffer-create "*Units Table*"))
	    (uptr (math-build-units-table))
	    (calc-language (if (eq calc-language 'big) nil calc-language))
	    (calc-float-format '(float 0))
	    (calc-group-digits nil)
	    (calc-number-radix 10)
	    (calc-point-char ".")
	    (std nil)
	    u name shadowed)
	(save-excursion
	  (message "Formatting units table...")
	  (set-buffer buf)
	  (setq buffer-read-only nil)
	  (erase-buffer)
	  (insert "Calculator Units Table:\n\n")
	  (insert "Unit    Type  Definition                  Description\n\n")
	  (while uptr
	    (setq u (car uptr)
		  name (nth 2 u))
	    (if (eq (car u) 'm)
		(setq std t))
	    (setq shadowed (and std (assq (car u) math-additional-units)))
	    (if (and name
		     (> (length name) 1)
		     (eq (aref name 0) ?\*))
		(progn
		  (or (eq uptr math-units-table)
		      (insert "\n"))
		  (setq name (substring name 1))))
	    (insert " ")
	    (and shadowed (insert "("))
	    (insert (symbol-name (car u)))
	    (and shadowed (insert ")"))
	    (if (nth 3 u)
		(progn
		  (indent-to 10)
		  (insert (symbol-name (nth 3 u))))
	      (or std
		  (progn
		    (indent-to 10)
		    (insert "U"))))
	    (indent-to 14)
	    (and shadowed (insert "("))
	    (if (nth 1 u)
		(insert (math-format-value (nth 1 u) 80))
	      (insert (symbol-name (car u))))
	    (and shadowed (insert ")"))
	    (indent-to 42)
	    (if name
		(insert name))
	    (if shadowed
		(insert " (redefined above)")
	      (or (nth 1 u)
		  (insert " (base unit)")))
	    (insert "\n")
	    (setq uptr (cdr uptr)))
	  (insert "\n\nUnit Prefix Table:\n\n")
	  (setq uptr math-unit-prefixes)
	  (while uptr
	    (setq u (car uptr))
	    (insert " " (char-to-string (car u)))
	    (if (equal (nth 1 u) (nth 1 (nth 1 uptr)))
		(insert " " (char-to-string (car (car (setq uptr (cdr uptr)))))
			"   ")
	      (insert "     "))
	    (insert "10^" (int-to-string (nth 2 (nth 1 u))))
	    (indent-to 15)
	    (insert "   " (nth 2 u) "\n")
	    (setq uptr (cdr uptr)))
	  (insert "\n")
	  (setq buffer-read-only t)
	  (message "Formatting units table...done"))
	(setq math-units-table-buffer-valid t)
	(let ((oldbuf (current-buffer)))
	  (set-buffer buf)
	  (goto-char (point-min))
	  (set-buffer oldbuf))
	(if enter-buffer
	    (pop-to-buffer buf)
	  (display-buffer buf)))
    (if enter-buffer
	(pop-to-buffer (get-buffer "*Units Table*"))
      (display-buffer (get-buffer "*Units Table*"))))
)




;;;; [calc-prog.el]

;;;; User-programmability.

;;; Compiling Lisp-like forms to use the math library.

(defun math-do-defmath (func args body)
  (let* ((fname (intern (concat "calcFunc-" (symbol-name func))))
	 (doc (if (stringp (car body)) (list (car body))))
	 (clargs (mapcar 'math-clean-arg args))
	 (body (math-define-function-body
		(if (stringp (car body)) (cdr body) body)
		clargs)))
    (list 'progn
	  (if (and (consp (car body))
		   (eq (car (car body)) 'interactive))
	      (let ((inter (car body)))
		(setq body (cdr body))
		(if (or (> (length inter) 2)
			(integerp (nth 1 inter)))
		    (let ((hasprefix nil) (hasmulti nil))
		      (if (stringp (nth 1 inter))
			  (progn
			    (cond ((equal (nth 1 inter) "p")
				   (setq hasprefix t))
				  ((equal (nth 1 inter) "m")
				   (setq hasmulti t))
				  (t (error
				      "Can't handle interactive code string \"%s\""
				      (nth 1 inter))))
			    (setq inter (cdr inter))))
		      (if (not (integerp (nth 1 inter)))
			  (error
			   "Expected an integer in interactive specification"))
		      (append (list 'defun
				    (intern (concat "calc-"
						    (symbol-name func)))
				    (if (or hasprefix hasmulti)
					'(&optional n)
				      ()))
			      doc
			      (if (or hasprefix hasmulti)
				  '((interactive "P"))
				'((interactive)))
			      (list
			       (append
				'(calc-slow-wrapper)
				(and hasmulti
				     (list
				      (list 'setq
					    'n
					    (list 'if
						  'n
						  (list 'prefix-numeric-value
							'n)
						  (nth 1 inter)))))
				(list
				 (list 'calc-enter-result
				       (if hasmulti 'n (nth 1 inter))
				       (nth 2 inter)
				       (if hasprefix
					   (list 'append
						 (list 'quote (list fname))
						 (list 'calc-top-list-n
						       (nth 1 inter))
						 (list 'and
						       'n
						       (list
							'list
							(list
							 'math-normalize
							 (list
							  'prefix-numeric-value
							  'n)))))
					 (list 'cons
					       (list 'quote fname)
					       (list 'calc-top-list-n
						     (if hasmulti
							 'n
						       (nth 1 inter)))))))))))
		  (append (list 'defun
				(intern (concat "calc-" (symbol-name func)))
				args)
			  doc
			  (list
			   inter
			   (cons 'calc-wrapper body))))))
	  (append (list 'defun fname clargs)
		  doc
		  (math-do-arg-list-check args nil nil)
		  body)))
)

(defun math-clean-arg (arg)
  (if (consp arg)
      (math-clean-arg (nth 1 arg))
    arg)
)

(defun math-do-arg-check (arg var is-opt is-rest)
  (if is-opt
      (let ((chk (math-do-arg-check arg var nil nil)))
	(list (cons 'and
		    (cons var
			  (if (cdr chk)
			      (setq chk (list (cons 'progn chk)))
			    chk)))))
    (and (consp arg)
	 (let* ((rest (math-do-arg-check (nth 1 arg) var is-opt is-rest))
		(qual (car arg))
		(qqual (list 'quote qual))
		(qual-name (symbol-name qual))
		(chk (intern (concat "math-check-" qual-name))))
	   (if (fboundp chk)
	       (append rest
		       (list
			(if is-rest
			    (list 'setq var
				  (list 'mapcar (list 'quote chk) var))
			  (list 'setq var (list chk var)))))
	     (if (fboundp (setq chk (intern (concat "math-" qual-name))))
		 (append rest
			 (list
			  (if is-rest
			      (list 'mapcar
				    (list 'function
					  (list 'lambda '(x)
						(list 'or
						      (list chk 'x)
						      (list 'math-reject-arg
							    'x qqual))))
				    var)
			    (list 'or
				  (list chk var)
				  (list 'math-reject-arg var qqual)))))
	       (if (and (string-match "\\`not-\\(.*\\)\\'" qual-name)
			(fboundp (setq chk (intern
					    (concat "math-"
						    (math-match-substring
						     qual-name 1))))))
		   (append rest
			   (list
			    (if is-rest
				(list 'mapcar
				      (list 'function
					    (list 'lambda '(x)
						  (list 'and
							(list chk 'x)
							(list 'math-reject-arg
							      'x qqual))))
				      var)
			      (list 'and
				    (list chk var)
				    (list 'math-reject-arg var qqual)))))
		 (error "Unknown qualifier `%s'" qual-name)))))))
)

(defun math-do-arg-list-check (args is-opt is-rest)
  (cond ((null args) nil)
	((consp (car args))
	 (append (math-do-arg-check (car args)
				    (math-clean-arg (car args))
				    is-opt is-rest)
		 (math-do-arg-list-check (cdr args) is-opt is-rest)))
	((eq (car args) '&optional)
	 (math-do-arg-list-check (cdr args) t nil))
	((eq (car args) '&rest)
	 (math-do-arg-list-check (cdr args) nil t))
	(t (math-do-arg-list-check (cdr args) is-opt is-rest)))
)

(defconst math-prim-funcs
  '( (~= . math-nearly-equal)
     (% . math-mod)
     (lsh . math-lshift-binary)
     (ash . math-lshift-arith)
     (logand . math-and)
     (logandc2 . math-diff)
     (logior . math-or)
     (logxor . math-xor)
     (lognot . math-not)
     (equal . equal)   ; need to leave these ones alone!
     (eq . eq)
     (and . and)
     (or . or)
     (if . if)
     (^ . math-pow)
     (expt . math-pow)
   )
)

(defconst math-prim-vars
  '( (nil . nil)
     (t . t)
     (&optional . &optional)
     (&rest . &rest)
   )
)

(defun math-define-function-body (body env)
  (let ((body (math-define-body body env)))
    (if (math-body-refers-to body 'math-return)
	(list (cons 'catch (cons '(quote math-return) body)))
      body))
)

(defun math-define-body (body exp-env)
  (math-define-list body)
)

(defun math-define-list (body &optional quote)
  (cond ((null body)
	 nil)
	((and (eq (car body) ':)
	      (stringp (nth 1 body)))
	 (cons (let* ((math-read-expr-quotes t)
		      (calc-language nil)
		      (math-expr-opers math-standard-opers)
		      (exp (math-read-expr (nth 1 body))))
		 (if (eq (car exp) 'error)
		     (error "Bad format: %s" (nth 1 body))
		   (math-define-exp exp)))
	       (math-define-list (cdr (cdr body)))))
	(quote
	 (cons (cond ((consp (car body))
		      (math-define-list (cdr body) t))
		     (t
		      (car body)))
	       (math-define-list (cdr body))))
	(t
	 (cons (math-define-exp (car body))
	       (math-define-list (cdr body)))))
)

(defun math-define-exp (exp)
  (cond ((consp exp)
	 (let ((func (car exp)))
	   (cond ((memq func '(quote function))
		  (if (and (consp (nth 1 exp))
			   (eq (car (nth 1 exp)) 'lambda))
		      (cons 'quote
			    (math-define-lambda (nth 1 exp) exp-env))
		    exp))
		 ((memq func '(let let* for foreach))
		  (let ((head (nth 1 exp))
			(body (cdr (cdr exp))))
		    (if (memq func '(let let*))
			()
		      (setq func (cdr (assq func '((for . math-for)
						   (foreach . math-foreach)))))
		      (if (not (listp (car head)))
			  (setq head (list head))))
		    (macroexpand
		     (cons func
			   (cons (math-define-let head)
				 (math-define-body body
						   (nconc
						    (math-define-let-env head)
						    exp-env)))))))
		 ((and (memq func '(setq setf))
		       (math-complicated-lhs (cdr exp)))
		  (if (> (length exp) 3)
		      (cons 'progn (math-define-setf-list (cdr exp)))
		    (math-define-setf (nth 1 exp) (nth 2 exp))))
		 ((eq func 'condition-case)
		  (cons func
			(cons (nth 1 exp)
			      (math-define-body (cdr (cdr exp))
						(cons (nth 1 exp)
						      exp-env)))))
		 ((eq func 'cond)
		  (cons func
			(math-define-cond (cdr exp))))
		 ((and (consp func)   ; ('spam a b) == force use of plain spam
		       (eq (car func) 'quote))
		  (cons func (math-define-list (cdr exp))))
		 ((symbolp func)
		  (let ((args (math-define-list (cdr exp)))
			(prim (assq func math-prim-funcs)))
		    (cond (prim
			   (cons (cdr prim) args))
			  ((eq func 'floatp)
			   (list 'eq (car args) '(quote float)))
			  ((eq func '+)
			   (math-define-binop 'math-add 0
					      (car args) (cdr args)))
			  ((eq func '-)
			   (if (= (length args) 1)
			       (cons 'math-neg args)
			     (math-define-binop 'math-sub 0
						(car args) (cdr args))))
			  ((eq func '*)
			   (math-define-binop 'math-mul 1
					      (car args) (cdr args)))
			  ((eq func '/)
			   (math-define-binop 'math-div 1
					      (car args) (cdr args)))
			  ((eq func 'min)
			   (math-define-binop 'math-min 0
					      (car args) (cdr args)))
			  ((eq func 'max)
			   (math-define-binop 'math-max 0
					      (car args) (cdr args)))
			  ((eq func '<)
			   (if (and (math-numberp (nth 1 args))
				    (math-zerop (nth 1 args)))
			       (list 'math-posp (car args))
			     (cons 'math-lessp args)))
			  ((eq func '>)
			   (if (and (math-numberp (nth 1 args))
				    (math-zerop (nth 1 args)))
			       (list 'math-posp (car args))
			     (list 'math-lessp (nth 1 args) (nth 0 args))))
			  ((eq func '<=)
			   (list 'not
				 (if (and (math-numberp (nth 1 args))
					  (math-zerop (nth 1 args)))
				     (list 'math-posp (car args))
				   (cons 'math-lessp args))))
			  ((eq func '>=)
			   (list 'not
				 (if (and (math-numberp (nth 1 args))
					  (math-zerop (nth 1 args)))
				     (list 'math-negp (car args))
				   (list 'math-lessp
					 (nth 1 args) (nth 0 args)))))
			  ((eq func '=)
			   (if (and (math-numberp (nth 1 args))
				    (math-zerop (nth 1 args)))
			       (list 'math-zerop (nth 0 args))
			     (if (and (integerp (nth 1 args))
				      (/= (% (nth 1 args) 10) 0))
				 (cons 'math-equal-int args)
			       (cons 'math-equal args))))
			  ((eq func '/=)
			   (list 'not
				 (if (and (math-numberp (nth 1 args))
					  (math-zerop (nth 1 args)))
				     (list 'math-zerop (nth 0 args))
				   (if (and (integerp (nth 1 args))
					    (/= (% (nth 1 args) 10) 0))
				       (cons 'math-equal-int args)
				     (cons 'math-equal args)))))
			  ((eq func '1+)
			   (list 'math-add (car args) 1))
			  ((eq func '1-)
			   (list 'math-add (car args) -1))
			  ((eq func 'not)   ; optimize (not (not x)) => x
			   (if (eq (car-safe args) func)
			       (car (nth 1 args))
			     (cons func args)))
			  ((and (eq func 'elt) (cdr (cdr args)))
			   (math-define-elt (car args) (cdr args)))
			  (t
			   (macroexpand
			    (let* ((name (symbol-name func))
				   (cfunc (intern (concat "calcFunc-" name)))
				   (mfunc (intern (concat "math-" name))))
			      (cond ((fboundp cfunc)
				     (cons cfunc args))
				    ((fboundp mfunc)
				     (cons mfunc args))
				    ((or (fboundp func)
					 (string-match "\\`calcFunc-.*" name))
				     (cons func args))
				    (t
				     (cons cfunc args)))))))))
		 (t (cons func args)))))
	((symbolp exp)
	 (let ((prim (assq exp math-prim-vars))
	       (name (symbol-name exp)))
	   (cond (prim
		  (cdr prim))
		 ((memq exp exp-env)
		  exp)
		 ((string-match "-" name)
		  exp)
		 (t
		  (intern (concat "var-" name))))))
	((integerp exp)
	 (if (or (<= exp -1000000) (>= exp 1000000))
	     (list 'quote (math-normalize exp))
	   exp))
	(t exp))
)

(defun math-define-cond (forms)
  (and forms
       (cons (math-define-list (car forms))
	     (math-define-cond (cdr forms))))
)

(defun math-complicated-lhs (body)
  (and body
       (or (not (symbolp (car body)))
	   (math-complicated-lhs (cdr (cdr body)))))
)

(defun math-define-setf-list (body)
  (and body
       (cons (math-define-setf (nth 0 body) (nth 1 body))
	     (math-define-setf-list (cdr (cdr body)))))
)

(defun math-define-setf (place value)
  (setq place (math-define-exp place)
	value (math-define-exp value))
  (cond ((symbolp place)
	 (list 'setq place value))
	((eq (car-safe place) 'nth)
	 (list 'setcar (list 'nthcdr (nth 1 place) (nth 2 place)) value))
	((eq (car-safe place) 'elt)
	 (list 'setcar (list 'nthcdr (nth 2 place) (nth 1 place)) value))
	((eq (car-safe place) 'car)
	 (list 'setcar (nth 1 place) value))
	((eq (car-safe place) 'cdr)
	 (list 'setcdr (nth 1 place) value))
	(t
	 (error "Bad place form for setf: %s" place)))
)

(defun math-define-binop (op ident arg1 rest)
  (if rest
      (math-define-binop op ident
			 (list op arg1 (car rest))
			 (cdr rest))
    (or arg1 ident))
)

(defun math-define-let (vlist)
  (and vlist
       (cons (if (consp (car vlist))
		 (cons (car (car vlist))
		       (math-define-list (cdr (car vlist))))
	       (car vlist))
	     (math-define-let (cdr vlist))))
)

(defun math-define-let-env (vlist)
  (and vlist
       (cons (if (consp (car vlist))
		 (car (car vlist))
	       (car vlist))
	     (math-define-let-env (cdr vlist))))
)

(defun math-define-lambda (exp exp-env)
  (nconc (list (nth 0 exp)   ; 'lambda
	       (nth 1 exp))  ; arg list
	 (math-define-function-body (cdr (cdr exp))
				    (append (nth 1 exp) exp-env)))
)

(defun math-define-elt (seq idx)
  (if idx
      (math-define-elt (list 'elt seq (car idx)) (cdr idx))
    seq)
)



;;; Useful programming macros.

(defmacro math-while (head &rest body)
  (let ((body (cons 'while (cons head body))))
    (if (math-body-refers-to body 'math-break)
	(cons 'catch (cons '(quote math-break) (list body)))
      body))
)
(put 'math-while 'lisp-indent-hook 1)


(defmacro math-for (head &rest body)
  (let ((body (if head
		  (math-handle-for head body)
		(cons 'while (cons t body)))))
    (if (math-body-refers-to body 'math-break)
	(cons 'catch (cons '(quote math-break) (list body)))
      body))
)
(put 'math-for 'lisp-indent-hook 1)

(defun math-handle-for (head body)
  (let* ((var (nth 0 (car head)))
	 (init (nth 1 (car head)))
	 (limit (nth 2 (car head)))
	 (step (or (nth 3 (car head)) 1))
	 (body (if (cdr head)
		   (list (math-handle-for (cdr head) body))
		 body))
	 (all-ints (and (integerp init) (integerp limit) (integerp step)))
	 (const-limit (or (integerp limit)
			  (and (eq (car-safe limit) 'quote)
			       (math-realp (nth 1 limit)))))
	 (const-step (or (integerp step)
			 (and (eq (car-safe step) 'quote)
			      (math-realp (nth 1 step)))))
	 (save-limit (if const-limit limit (make-symbol "<limit>")))
	 (save-step (if const-step step (make-symbol "<step>"))))
    (cons 'let
	  (cons (append (if const-limit nil (list (list save-limit limit)))
			(if const-step nil (list (list save-step step)))
			(list (list var init)))
		(list
		 (cons 'while
		       (cons (if all-ints
				 (if (> step 0)
				     (list '<= var save-limit)
				   (list '>= var save-limit))
			       (list 'not
				     (if const-step
					 (if (or (math-posp step)
						 (math-posp
						  (cdr-safe step)))
					     (list 'math-lessp
						   save-limit
						   var)
					   (list 'math-lessp
						 var
						 save-limit))
				       (list 'if
					     (list 'math-posp
						   save-step)
					     (list 'math-lessp
						   save-limit
						   var)
					     (list 'math-lessp
						   var
						   save-limit)))))
			     (append body
				     (list (list 'setq
						 var
						 (list (if all-ints
							   '+
							 'math-add)
						       var
						       save-step))))))))))
)


(defmacro math-foreach (head &rest body)
  (let ((body (math-handle-foreach head body)))
    (if (math-body-refers-to body 'math-break)
	(cons 'catch (cons '(quote math-break) (list body)))
      body))
)
(put 'math-foreach 'lisp-indent-hook 1)

(defun math-handle-foreach (head body)
  (let ((var (nth 0 (car head)))
	(data (nth 1 (car head)))
	(body (if (cdr head)
		  (list (math-handle-foreach (cdr head) body))
		body)))
    (cons 'let
	  (cons (list (list var data))
		(list
		 (cons 'while
		       (cons var
			     (append body
				     (list (list 'setq
						 var
						 (list 'cdr var))))))))))
)


(defun math-body-refers-to (body thing)
  (or (equal body thing)
      (and (consp body)
	   (or (math-body-refers-to (car body) thing)
	       (math-body-refers-to (cdr body) thing))))
)

(defun math-break (&optional value)
  (throw 'math-break value)
)

(defun math-return (&optional value)
  (throw 'math-return value)
)




;;;; [calc-ext.el]

;;; Nontrivial number parsing.

(defun math-read-number-fancy (s)

  (cond

   ;; Modulo forms
   ((string-match "^\\(.*\\) *mod *\\(.*\\)$" s)
    (let* ((n (math-match-substring s 1))
	   (m (math-match-substring s 2))
	   (n (math-read-number n))
	   (m (math-read-number m)))
      (and n m (math-anglep n) (math-anglep m)
	   (list 'mod n m))))

   ;; Error forms
   ((string-match "^\\(.*\\) *\\+/- *\\(.*\\)$" s)
    (let* ((x (math-match-substring s 1))
	   (sigma (math-match-substring s 2))
	   (x (math-read-number x))
	   (sigma (math-read-number sigma)))
      (and x sigma (math-anglep x) (math-anglep sigma)
	   (list 'sdev x sigma))))

   ;; Hours (or degrees)
   ((or (string-match "^\\([^#^]+\\)[@oOhH]\\(.*\\)$" s)
	(string-match "^\\([^#^]+\\)[dD][eE]?[gG]?\\(.*\\)$" s))
    (let* ((hours (math-match-substring s 1))
	   (minsec (math-match-substring s 2))
	   (hours (math-read-number hours))
	   (minsec (if (> (length minsec) 0) (math-read-number minsec) 0)))
      (and hours minsec
	   (math-num-integerp hours)
	   (not (math-negp hours)) (not (math-negp minsec))
	   (cond ((math-num-integerp minsec)
		  (and (Math-lessp minsec 60)
		       (list 'hms hours minsec 0)))
		 ((and (eq (car-safe minsec) 'hms)
		       (math-zerop (nth 1 minsec)))
		  (math-add (list 'hms hours 0 0) minsec))
		 (t nil)))))
   
   ;; Minutes
   ((string-match "^\\([^'#^]+\\)[mM']\\(.*\\)$" s)
    (let* ((minutes (math-match-substring s 1))
	   (seconds (math-match-substring s 2))
	   (minutes (math-read-number minutes))
	   (seconds (if (> (length seconds) 0) (math-read-number seconds) 0)))
      (and minutes seconds
	   (math-num-integerp minutes)
	   (not (math-negp minutes)) (not (math-negp seconds))
	   (cond ((math-realp seconds)
		  (and (Math-lessp minutes 60)
		       (list 'hms 0 minutes seconds)))
		 ((and (eq (car-safe seconds) 'hms)
		       (math-zerop (nth 1 seconds))
		       (math-zerop (nth 2 seconds)))
		  (math-add (list 'hms 0 minutes 0) seconds))
		 (t nil)))))
   
   ;; Seconds
   ((string-match "^\\([^\"#^]+\\)[sS\"]$" s)
    (let ((seconds (math-read-number (math-match-substring s 1))))
      (and seconds (math-realp seconds)
	   (not (math-negp seconds))
	   (Math-lessp seconds 60)
	   (list 'hms 0 0 seconds))))
   
   ;; Integer+fraction with explicit radix
   ((string-match "^\\([0-9]+\\)\\(#\\|\\^\\^\\)\\([0-9a-zA-Z]*\\)[:/]\\([0-9a-zA-Z]*\\)[:/]\\([0-9a-zA-Z]\\)$" s)
    (let ((radix (string-to-int (math-match-substring s 1)))
	  (int (math-match-substring s 3))
	  (num (math-match-substring s 4))
	  (den (math-match-substring s 5)))
      (let ((int (if (> (length int) 0) (math-read-radix int radix) 0))
	    (num (if (> (length num) 0) (math-read-radix num radix) 1))
	    (den (if (> (length num) 0) (math-read-radix den radix) 1)))
	(and int num den (not (math-zerop den))
	     (list 'frac
		   (math-add num (math-mul int den))
		   den)))))
   
   ;; Fraction with explicit radix
   ((string-match "^\\([0-9]+\\)\\(#\\|\\^\\^\\)\\([0-9a-zA-Z]*\\)[:/]\\([0-9a-zA-Z]*\\)$" s)
    (let ((radix (string-to-int (math-match-substring s 1)))
	  (num (math-match-substring s 3))
	  (den (math-match-substring s 4)))
      (let ((num (if (> (length num) 0) (math-read-radix num radix) 1))
	    (den (if (> (length num) 0) (math-read-radix den radix) 1)))
	(and num den (not (math-zerop den)) (list 'frac num den)))))
   
   ;; Integer with explicit radix
   ((string-match "^\\([0-9]+\\)\\(#\\|\\^\\^\\)\\([0-9a-zA-Z]+\\)$" s)
    (math-read-radix (math-match-substring s 3)
		     (string-to-int (math-match-substring s 1))))
   
   ;; C language hexadecimal notation
   ((and (eq calc-language 'c)
	 (string-match "^0[xX]\\([0-9a-fA-F]+\\)$" s))
    (let ((digs (math-match-substring s 1)))
      (math-read-radix digs 16)))
   
   ;; Fraction using "/" instead of ":"
   ((string-match "^\\([0-9]+\\)/\\([0-9/]+\\)$" s)
    (math-read-number (concat (math-match-substring s 1) ":"
			      (math-match-substring s 2))))

   ;; Syntax error!
   (t nil))
)

(defun math-read-radix (s r)   ; [I X D]
  (catch 'gonzo
    (math-read-radix-loop (upcase s) (1- (length s)) r))
)

(defun math-read-radix-loop (s i r)   ; [I X S D]
  (if (< i 0)
      0
    (let ((dig (math-read-radix-digit (elt s i))))
      (if (and dig (< dig r))
	  (math-add (math-mul (math-read-radix-loop s (1- i) r)
			      r)
		    dig)
	(throw 'gonzo nil))))
)



;;; Expression parsing.

(defun math-read-expr (exp-str)
  (let ((exp-pos 0)
	(exp-old-pos 0)
	(exp-keep-spaces nil)
	exp-token exp-data)
    (while (setq exp-token (string-match "\\.\\." exp-str))
      (setq exp-str (concat (substring exp-str exp-token) "\\dots"
			    (substring exp-str (+ exp-token 2)))))
    (math-read-token)
    (let ((val (catch 'syntax (math-read-expr-level 0))))
      (if (stringp val)
	  (list 'error exp-old-pos val)
	(if (equal exp-token 'end)
	    val
	  (list 'error exp-old-pos "Syntax error")))))
)

;;;; [calc-vec.el]

(defun math-read-brackets (space-sep close)
  (and space-sep (setq space-sep (not (math-check-for-commas))))
  (math-read-token)
  (while (eq exp-token 'space)
    (math-read-token))
  (if (or (equal exp-data close)
	  (eq exp-token 'end))
      (progn
	(math-read-token)
	'(vec))
    (let ((vals (let ((exp-keep-spaces space-sep))
		  (math-read-vector))))
      (if (equal exp-data "\\dots")
	  (progn
	    (math-read-token)
	    (setq vals (if (> (length vals) 2)
			   (cons 'calcFunc-mul (cdr vals)) (nth 1 vals)))
	    (let ((exp2 (math-read-expr-level 0)))
	      (setq vals
		    (list 'intv
			  (if (equal exp-data ")") 2 3)
			  vals
			  exp2)))
	    (if (not (or (equal exp-data close)
			 (equal exp-data ")")
			 (eq exp-token 'end)))
		(throw 'syntax "Expected `]'")))
	(if (equal exp-data ";")
	    (let ((exp-keep-spaces space-sep))
	      (setq vals (cons 'vec (math-read-matrix (list vals))))))
	(if (not (or (equal exp-data close)
		     (eq exp-token 'end)))
	    (throw 'syntax "Expected `]'")))
      (math-read-token)
      vals))
)

(defun math-check-for-commas ()
  (let ((count 0)
	(pos (1- exp-pos)))
    (while (and (>= count 0)
		(setq pos (string-match "[],[{}()]" exp-str (1+ pos)))
		(or (/= (aref exp-str pos) ?,) (> count 0)))
      (cond ((memq (aref exp-str pos) '(?\[ ?\{ ?\())
	     (setq count (1+ count)))
	    ((memq (aref exp-str pos) '(?\] ?\} ?\)))
	     (setq count (1- count)))))
    (and pos (= (aref exp-str pos) ?,)))
)

(defun math-read-vector ()
  (let* ((val (list (math-read-expr-level 0)))
	 (last val))
    (while (progn
	     (while (eq exp-token 'space)
	       (math-read-token))
	     (and (not (eq exp-token 'end))
		  (not (equal exp-data ";"))
		  (not (equal exp-data close))
		  (not (equal exp-data "\\dots"))))
      (if (equal exp-data ",")
	  (math-read-token))
      (while (eq exp-token 'space)
	(math-read-token))
      (let ((rest (list (math-read-expr-level 0))))
	(setcdr last rest)
	(setq last rest)))
    (cons 'vec val))
)

(defun math-read-matrix (mat)
  (while (equal exp-data ";")
    (math-read-token)
    (while (eq exp-token 'space)
      (math-read-token))
    (setq mat (nconc mat (list (math-read-vector)))))
  mat
)

;;;; [calc-ext.el]

(defun math-read-string ()
  (let ((str (read-from-string (concat exp-data "\""))))
    (or (and (= (cdr str) (1+ (length exp-data)))
	     (stringp (car str)))
	(throw 'syntax "Error in string constant"))
    (math-read-token)
    (append '(vec) (car str) nil))
)





;;; Nontrivial "flat" formatting.

(defun math-format-flat-expr-fancy (a prec)
  (cond
   ((eq (car a) 'incomplete)
    (concat "'" (prin1-to-string a)))
   ((eq (car a) 'vec)
    (concat "[" (math-format-flat-vector (cdr a) ", "
					 (if (cdr (cdr a)) 0 1000)) "]"))
   ((eq (car a) 'intv)
    (concat (if (memq (nth 1 a) '(0 1)) "(" "[")
	    (math-format-flat-expr (nth 2 a) 1000)
	    " .. "
	    (math-format-flat-expr (nth 3 a) 1000)
	    (if (memq (nth 1 a) '(0 2)) ")" "]")))
   ((eq (car a) 'var)
    (symbol-name (nth 1 a)))
   (t
    (let ((op (math-assq2 (car a) math-standard-opers)))
      (cond ((and op (= (length a) 3))
	     (if (> prec (min (nth 2 op) (nth 3 op)))
		 (concat "(" (math-format-flat-expr a 0) ")")
	       (let ((lhs (math-format-flat-expr (nth 1 a) (nth 2 op)))
		     (rhs (math-format-flat-expr (nth 2 a) (nth 3 op))))
		 (setq op (car op))
		 (if (equal op "^")
		     (if (= (aref lhs 0) ?-)
			 (setq lhs (concat "(" lhs ")")))
		   (setq op (concat " " op " ")))
		 (concat lhs op rhs))))
	    ((eq (car a) 'neg)
	     (concat "-" (math-format-flat-expr (nth 1 a) 1000)))
	    (t
	     (concat (math-remove-dashes
		      (if (string-match "\\`calcFunc-\\([a-zA-Z0-9']+\\)\\'"
					(symbol-name (car a)))
			  (math-match-substring (symbol-name (car a)) 1)
			(symbol-name (car a))))
		     "("
		     (math-format-flat-vector (cdr a) ", " 0)
		     ")"))))))
)

(defun math-format-flat-vector (vec sep prec)
  (if vec
      (let ((buf (math-format-flat-expr (car vec) prec)))
	(while (setq vec (cdr vec))
	  (setq buf (concat buf sep (math-format-flat-expr (car vec) prec))))
	buf)
    "")
)

(defun math-assq2 (v a)
  (cond ((null a) nil)
	((eq v (nth 1 (car a))) (car a))
	(t (math-assq2 v (cdr a))))
)


(defun math-format-number-fancy (a)
  (cond
   ((eq (car a) 'cplx)
    (if (null calc-complex-format)
	(concat "(" (math-format-number (nth 1 a))
		", " (math-format-number (nth 2 a)) ")")
      (if (math-zerop (nth 1 a))
	  (concat (math-format-number (nth 2 a))
		  (symbol-name calc-complex-format))
	(concat (math-format-number (nth 1 a))
		(if (math-negp (nth 2 a)) " - " " + ")
		(math-format-number (math-abs (nth 2 a)))
		(symbol-name calc-complex-format)))))
   ((eq (car a) 'polar)
    (concat "(" (math-format-number (nth 1 a))
	    "; " (math-format-number (nth 2 a)) ")"))
   ((eq (car a) 'hms)
    (if (math-negp a)
	(concat "-" (math-format-number (math-neg a)))
      (let ((calc-number-radix 10)
	    (calc-leading-zeros nil)
	    (calc-group-digits nil))
	(format calc-hms-format
		(math-format-number (nth 1 a))
		(math-format-number (nth 2 a))
		(math-format-number (nth 3 a))))))
   (t (format "%s" a)))
)

(defun math-format-bignum-fancy (a)   ; [X L]
  (let ((str (cond ((= calc-number-radix 10)
		    (math-format-bignum-decimal a))
		   ((= calc-number-radix 2)
		    (math-format-bignum-binary a))
		   ((= calc-number-radix 8)
		    (math-format-bignum-octal a))
		   ((= calc-number-radix 16)
		    (math-format-bignum-hex a))
		   (t (math-format-bignum-radix a)))))
    (if calc-leading-zeros
	(let* ((calc-internal-prec 6)
	       (digs (math-compute-max-digits (math-abs calc-word-size)
					      calc-number-radix))
	       (len (length str)))
	  (if (< len digs)
	      (setq str (concat (make-string (- digs len) ?0) str)))))
    (if calc-group-digits
	(let ((i (length str))
	      (g (if (integerp calc-group-digits)
		     (math-abs calc-group-digits)
		   (if (memq calc-number-radix '(2 16)) 4 3))))
	  (while (> i g)
	    (setq i (- i g)
		  str (concat (substring str 0 i)
			      calc-group-char
			      (substring str i))))
	  str))
    (if (and (/= calc-number-radix 10)
	     math-radix-explicit-format)
	(if calc-radix-formatter
	    (funcall calc-radix-formatter calc-number-radix str)
	  (format "%d#%s" calc-number-radix str))
      str))
)

;;;; [calc-bin.el]

(defvar math-max-digits-cache nil)
(defun math-compute-max-digits (w r)
  (let* ((pair (+ (* r 100000) w))
	 (res (assq pair math-max-digits-cache)))
    (if res
	(cdr res)
      (let* ((calc-command-flags nil)
	     (digs (math-ceiling (math-div w (math-real-log2 r)))))
	(setq math-max-digits-cache (cons (cons pair digs)
					  math-max-digits-cache))
	digs)))
)

(defvar math-log2-cache (list '(2 . 1)
			      '(4 . 2)
			      '(8 . 3)
			      '(10 . (float 332193 -5))
			      '(16 . 4)
			      '(32 . 5)))
(defun math-real-log2 (x)   ;;; calc-internal-prec must be 6
  (let ((res (assq x math-log2-cache)))
    (if res
	(cdr res)
      (let* ((calc-symbolic-mode nil)
	     (log (math-log x 2)))
	(setq math-log2-cache (cons (cons x log) math-log2-cache))
	log)))
)

(defconst math-radix-digits ["0" "1" "2" "3" "4" "5" "6" "7" "8" "9"
			     "A" "B" "C" "D" "E" "F" "G" "H" "I" "J"
			     "K" "L" "M" "N" "O" "P" "Q" "R" "S" "T"
			     "U" "V" "W" "X" "Y" "Z"])
(defmacro math-format-radix-digit (a)   ; [X D]
  (` (aref math-radix-digits (, a)))
)

(defun math-format-radix (a)   ; [X S]
  (if (< a calc-number-radix)
      (if (< a 0)
	  (concat "-" (math-format-radix (- a)))
	(math-format-radix-digit a))
    (let ((s ""))
      (while (> a 0)
	(setq s (concat (math-format-radix-digit (% a calc-number-radix)) s)
	      a (/ a calc-number-radix)))
      s))
)

(defconst math-binary-digits ["000" "001" "010" "011"
			      "100" "101" "110" "111"])
(defun math-format-binary (a)   ; [X S]
  (if (< a 8)
      (if (< a 0)
	  (concat "-" (math-format-binary (- a)))
	(math-format-radix a))
    (let ((s ""))
      (while (> a 7)
	(setq s (concat (aref math-binary-digits (% a 8)) s)
	      a (/ a 8)))
      (concat (math-format-radix a) s)))
)

(defun math-format-bignum-radix (a)   ; [X L]
  (cond ((null a) "0")
	((and (null (cdr a))
	      (< (car a) calc-number-radix))
	 (math-format-radix-digit (car a)))
	(t
	 (let ((q (math-div-bignum-digit a calc-number-radix)))
	   (concat (math-format-bignum-radix (math-norm-bignum (car q)))
		   (math-format-radix-digit (cdr q))))))
)

(defun math-format-bignum-binary (a)   ; [X L]
  (cond ((null a) "0")
	((null (cdr a))
	 (math-format-binary (car a)))
	(t
	 (let ((q (math-div-bignum-digit a 512)))
	   (concat (math-format-bignum-binary (math-norm-bignum (car q)))
		   (aref math-binary-digits (/ (cdr q) 64))
		   (aref math-binary-digits (% (/ (cdr q) 8) 8))
		   (aref math-binary-digits (% (cdr q) 8))))))
)

(defun math-format-bignum-octal (a)   ; [X L]
  (cond ((null a) "0")
	((null (cdr a))
	 (math-format-radix (car a)))
	(t
	 (let ((q (math-div-bignum-digit a 512)))
	   (concat (math-format-bignum-octal (math-norm-bignum (car q)))
		   (math-format-radix-digit (/ (cdr q) 64))
		   (math-format-radix-digit (% (/ (cdr q) 8) 8))
		   (math-format-radix-digit (% (cdr q) 8))))))
)

(defun math-format-bignum-hex (a)   ; [X L]
  (cond ((null a) "0")
	((null (cdr a))
	 (math-format-radix (car a)))
	(t
	 (let ((q (math-div-bignum-digit a 256)))
	   (concat (math-format-bignum-hex (math-norm-bignum (car q)))
		   (math-format-radix-digit (/ (cdr q) 16))
		   (math-format-radix-digit (% (cdr q) 16))))))
)

;;;; [calc-ext.el]

(defun math-group-float (str)   ; [X X]
  (let* ((pt (or (string-match "[^0-9]" str) (length str)))
	 (g (if (integerp calc-group-digits) (math-abs calc-group-digits) 3))
	 (i pt))
    (if (and (integerp calc-group-digits) (< calc-group-digits 0))
	(while (< (setq i (+ (1+ i) g)) (length str))
	  (setq str (concat (substring str 0 i)
			    calc-group-char
			    (substring str i)))))
    (setq i pt)
    (while (> i g)
      (setq i (- i g)
	    str (concat (substring str 0 i)
			calc-group-char
			(substring str i))))
    str)
)







;;;; [calc-comp.el]

;;; A "composition" has one of the following forms:
;;;
;;;    "string"              A literal string
;;;
;;;    (horiz C1 C2 ...)     Horizontally abutted sub-compositions
;;;
;;;    (break LEVEL)         A potential line-break point
;;;
;;;    (vleft N C1 C2 ...)   Vertically stacked, left-justified sub-comps
;;;    (vcent N C1 C2 ...)   Vertically stacked, centered sub-comps
;;;    (vright N C1 C2 ...)  Vertically stacked, right-justified sub-comps
;;;                          N specifies baseline of the stack, 0=top line.
;;;
;;;    (supscr C1 C2)        Composition C1 with superscript C2
;;;    (subscr C1 C2)        Composition C1 with subscript C2
;;;    (rule)                Horizontal line, full width of enclosing comp

(defun math-compose-expr (a prec)
  (let ((math-compose-level (1+ math-compose-level)))
    (cond
     ((math-scalarp a)
      (if (and (eq (car-safe a) 'frac)
	       (memq calc-language '(tex math)))
	  (math-compose-expr (list '/ (nth 1 a) (nth 2 a)) prec)
	(math-format-number a)))
     ((not (consp a)) (concat "'" (prin1-to-string a)))
     ((eq (car a) 'vec)
      (let ((left-bracket (if calc-vector-brackets
			      (substring calc-vector-brackets 0 1) ""))
	    (right-bracket (if calc-vector-brackets
			       (substring calc-vector-brackets 1 2) ""))
	    (comma (or calc-vector-commas " "))
	    (just (cond ((eq calc-matrix-just 'right) 'vright)
			((eq calc-matrix-just 'center) 'vcent)
			(t 'vleft))))
	(if (and (math-matrixp a) (not (math-matrixp (nth 1 a)))
		 (memq calc-language '(nil big)))
	    (if (= (length a) 2)
		(list 'horiz
		      (concat left-bracket left-bracket " ")
		      (math-compose-vector (cdr (nth 1 a))
					   (concat comma " "))
		      (concat " " right-bracket right-bracket))
	      (let* ((rows (1- (length a)))
		     (cols (1- (length (nth 1 a))))
		     (base (/ (1- rows) 2))
		     (calc-language 'flat))
		(append '(horiz)
			(list (append '(vleft)
				      (list base)
				      (list (concat left-bracket
						    " "
						    left-bracket
						    " "))
				      (make-list (1- rows)
						 (concat "  "
							 left-bracket
							 " "))))
			(math-compose-matrix (cdr a) 1 cols base)
			(list (append '(vleft)
				      (list base)
				      (make-list (1- rows)
						 (concat " "
							 right-bracket
							 comma))
				      (list (concat " "
						    right-bracket
						    " "
						    right-bracket)))))))
	  (if (and calc-display-strings
		   (math-vector-is-string a))
	      (prin1-to-string (concat (cdr a)))
	    (list 'horiz
		  left-bracket
		  (math-compose-vector (cdr a)
				       (concat (or calc-vector-commas "") " "))
		  right-bracket)))))
     ((eq (car a) 'incomplete)
      (if (cdr (cdr a))
	  (cond ((eq (nth 1 a) 'vec)
		 (list 'horiz "["
		       (math-compose-vector (cdr (cdr a)) ", ")
		       " ..."))
		((eq (nth 1 a) 'cplx)
		 (list 'horiz "("
		       (math-compose-vector (cdr (cdr a)) ", ")
		       ", ..."))
		((eq (nth 1 a) 'polar)
		 (list 'horiz "("
		       (math-compose-vector (cdr (cdr a)) "; ")
		       "; ..."))
		((eq (nth 1 a) 'intv)
		 (list 'horiz
		       (if (memq (nth 2 a) '(0 1)) "(" "[")
		       (math-compose-vector (cdr (cdr (cdr a))) " .. ")
		       " .. ..."))
		(t (format "%s" a)))
	(cond ((eq (nth 1 a) 'vec) "[ ...")
	      ((eq (nth 1 a) 'intv)
	       (if (memq (nth 2 a) '(0 1)) "( ..." "[ ..."))
	      (t "( ..."))))
     ((eq (car a) 'var)
      (let ((v (rassq (nth 2 a) math-expr-variable-mapping)))
	(if v
	    (symbol-name (car v))
	  (if (and (eq calc-language 'tex)
		   calc-language-option
		   (string-match "\\`[a-zA-Z][a-zA-Z0-9]+\\'"
				 (symbol-name (nth 1 a))))
	      (format "\\hbox{%s}" (symbol-name (nth 1 a)))
	    (symbol-name (nth 1 a))))))
     ((eq (car a) 'intv)
      (list 'horiz
	    (if (memq (nth 1 a) '(0 1)) "(" "[")
	    (math-compose-expr (nth 2 a) 0)
	    (if (eq calc-language 'tex) " \\dots " " .. ")
	    (math-compose-expr (nth 3 a) 0)
	    (if (memq (nth 1 a) '(0 2)) ")" "]")))
     ((and (eq (car a) 'calcFunc-subscr)
	   (memq calc-language '(c pascal fortran)))
      (list 'horiz
	    (math-compose-expr (nth 1 a) 1000)
	    (if (eq calc-language 'fortran) "(" "[")
	    (math-compose-vector (cdr (cdr a)) ", ")
	    (if (eq calc-language 'fortran) ")" "]")))
     ((and (eq (car a) 'calcFunc-subscr)
	   (eq calc-language 'big))
      (let ((a1 (math-compose-expr (nth 1 a) 1000))
	    (a2 (math-compose-expr (nth 2 a) 0)))
	(if (eq (car-safe a1) 'subscr)
	    (list 'subscr
		  (nth 1 a1)
		  (list 'horiz
			(nth 2 a1)
			", "
			a2))
	  (list 'subscr a1 a2))))
     ((and (eq (car a) 'calcFunc-sqrt)
	   (eq calc-language 'tex))
      (list 'horiz
	    "\\sqrt{"
	    (math-compose-expr (nth 1 a) 0)
	    "}"))
     ((and (eq (car a) '^)
	   (eq calc-language 'big))
      (list 'supscr
	    (if (math-looks-negp (nth 1 a))
		(list 'horiz "(" (math-compose-expr (nth 1 a) 0) ")")
	      (math-compose-expr (nth 1 a) 201))
	    (let ((calc-language 'flat))
	      (math-compose-expr (nth 2 a) 0))))
     ((and (eq (car a) '/)
	   (eq calc-language 'big))
      (let ((a1 (math-compose-expr (nth 1 a) 0))
	    (a2 (math-compose-expr (nth 2 a) 0)))
	(list 'vcent
	      (math-comp-height a1)
	      a1 '(rule) a2)))
     (t
      (let ((op (and (not (eq calc-language 'unform))
		     (math-assq2 (car a) math-expr-opers))))
	(cond ((and op (= (length a) 3)
		    (/= (nth 3 op) -1)
		    (not (eq (car a) 'calcFunc-if)))
	       (cond
		((> prec (min (nth 2 op) (nth 3 op)))
		 (if (and (eq calc-language 'tex)
			  (not (math-tex-expr-is-flat a)))
		     (if (eq (car-safe a) '/)
			 (list 'horiz "{" (math-compose-expr a -1) "}")
		       (list 'horiz "\\left( "
			     (math-compose-expr a -1)
			     " \\right)"))
		   (list 'horiz "(" (math-compose-expr a 0) ")")))
		((and (eq calc-language 'tex)
		      (memq (car a) '(/ calcFunc-choose))
		      (>= prec 0))
		 (list 'horiz "{" (math-compose-expr a -1) "}"))
		(t
		 (let ((lhs (math-compose-expr (nth 1 a) (nth 2 op)))
		       (rhs (math-compose-expr (nth 2 a) (nth 3 op))))
		   (and (equal (car op) "^")
			(= (math-comp-first-char lhs) ?-)
			(setq lhs (list 'horiz "(" lhs ")")))
		   (and (eq calc-language 'tex)
			(or (equal (car op) "^") (equal (car op) "_"))
			(not (and (stringp rhs) (= (length rhs) 1)))
			(setq rhs (list 'horiz "{" rhs "}")))
		   (or (and (eq (car a) '*)
			    (or (null calc-language)
				(assoc "2x" math-expr-opers))
			    (let ((prevt (math-prod-last-term (nth 1 a)))
				  (nextt (math-prod-first-term (nth 2 a)))
				  (prevc (math-comp-last-char lhs))
				  (nextc (math-comp-first-char rhs)))
			      (and prevc nextc
				   (or (and (>= nextc ?a) (<= nextc ?z))
				       (and (>= nextc ?A) (<= nextc ?Z))
				       (and (>= nextc ?0) (<= nextc ?9))
				       (memq nextc '(?. ?_ ?\( ?\[ ?\{ ?\\)))
				   (not (and (eq (car-safe prevt) 'var)
					     (equal nextc ?\()))
				   (list 'horiz
					 lhs
					 (list 'break math-compose-level)
					 " "
					 rhs))))
		       (list 'horiz
			     lhs
			     (list 'break math-compose-level)
			     (if (or (equal (car op) "^")
				     (equal (car op) "_")
				     (equal (car op) "*"))
				 (car op)
			       (concat " " (car op) " "))
			     rhs))))))
	      ((and op (= (length a) 2) (= (nth 3 op) -1))
	       (cond
		((> prec (nth 2 op))
		 (if (and (eq calc-language 'tex)
			  (not (math-tex-expr-is-flat a)))
		     (list 'horiz "\\left( "
			   (math-compose-expr a -1)
			   " \\right)")
		   (list 'horiz "(" (math-compose-expr a 0) ")")))
		(t
		 (let ((lhs (math-compose-expr (nth 1 a) (nth 2 op))))
		 (list 'horiz
		       lhs
		       (if (or (> (length (car op)) 1)
			       (not (math-comp-is-flat lhs)))
			   (concat " " (car op))
			 (car op)))))))
	      ((and op (= (length a) 2) (= (nth 2 op) -1))
	       (cond
		((eq (nth 3 op) 0)
		 (let ((lr (and (eq calc-language 'tex)
				(not (math-tex-expr-is-flat (nth 1 a))))))
		   (list 'horiz
			 (if lr "\\left" "")
			 (if (string-match "\\`u\\([^a-zA-Z]\\)\\'" (car op))
			     (substring (car op) 1)
			   (car op))
			 (if (or lr (> (length (car op)) 2)) " " "")
			 (math-compose-expr (nth 1 a) -1)
			 (if (or lr (> (length (car op)) 2)) " " "")
			 (if lr "\\right" "")
			 (car (nth 1 (memq op math-expr-opers))))))
		((> prec (nth 3 op))
		 (if (and (eq calc-language 'tex)
			  (not (math-tex-expr-is-flat a)))
		     (list 'horiz "\\left( "
			   (math-compose-expr a -1)
			   " \\right)")
		   (list 'horiz "(" (math-compose-expr a 0) ")")))
		(t
		 (let ((rhs (math-compose-expr (nth 1 a) (nth 3 op))))
		   (list 'horiz
			 (let ((ops (if (string-match "\\`u\\([^a-zA-Z]\\)\\'"
						      (car op))
					(substring (car op) 1)
				      (car op))))
			   (if (or (> (length ops) 1)
				   (not (math-comp-is-flat rhs)))
			       (concat ops " ")
			     ops))
			 rhs)))))
	      ((and op (= (length a) 4) (eq (car a) 'calcFunc-if))
	       (list 'horiz
		     (math-compose-expr (nth 1 a) (nth 2 op))
		     " ? "
		     (math-compose-expr (nth 2 a) 0)
		     " : "
		     (math-compose-expr (nth 3 a) (nth 3 op))))
	      ((and (eq calc-language 'big)
		    (setq op (get (car a) 'math-compose-big)))
	       (funcall op a prec))
	      (t
	       (let* ((func (car a))
		      (func2 (assq func '(( mod . calcFunc-makemod )
					  ( sdev . calcFunc-sdev )
					  ( + . calcFunc-add )
					  ( - . calcFunc-sub )
					  ( * . calcFunc-mul )
					  ( / . calcFunc-div )
					  ( % . calcFunc-mod )
					  ( ^ . calcFunc-pow )
					  ( neg . calcFunc-neg )
					  ( | . calcFunc-vconcat ))))
		      left right args)
		 (if func2
		     (setq func (cdr func2)))
		 (if (setq func2 (rassq func math-expr-function-mapping))
		     (setq func (car func2)))
		 (setq func (math-remove-dashes
			     (if (string-match
				  "\\`calcFunc-\\([a-zA-Z0-9']+\\)\\'"
				  (symbol-name func))
				 (math-match-substring (symbol-name func) 1)
			       (symbol-name func))))
		 (if (and (eq calc-language 'tex)
			  calc-language-option
			  (string-match "\\`[a-zA-Z][a-zA-Z0-9]+\\'" func))
		     (setq func (format "\\hbox{%s}" func)))
		 (cond ((and (eq calc-language 'tex)
			     (or (> (length a) 2)
				 (not (math-tex-expr-is-flat (nth 1 a)))))
			(setq left "\\left( "
			      right " \\right)"))
		       ((and (eq calc-language 'tex)
			     (eq (aref func 0) ?\\)
			     (= (length a) 2)
			     (or (Math-realp (nth 1 a))
				 (memq (car (nth 1 a)) '(var *))))
			(setq left "{"
			      right "}"))
		       (t (setq left calc-function-open
				right calc-function-close)))
		 (list 'horiz func left
		       (math-compose-vector (cdr a) ", ")
		       right))))))))
)
(setq math-compose-level 0)

(defun math-prod-first-term (x)
  (if (eq (car-safe x) '*)
      (math-prod-first-term (nth 1 x))
    x)
)

(defun math-prod-last-term (x)
  (if (eq (car-safe x) '*)
      (math-prod-last-term (nth (1- (length x)) x))
    x)
)

(defun math-compose-vector (a sep)
  (if a
      (cons 'horiz
	    (cons (math-compose-expr (car a) 0)
		  (math-compose-vector-step (cdr a))))
    "")
)

(defun math-compose-vector-step (a)
  (and a
       (cons sep
	     (cons (list 'break math-compose-level)
		   (cons (math-compose-expr (car a) 0)
			 (math-compose-vector-step (cdr a))))))
)

(defun math-compose-matrix (a col cols base)
  (math-compose-matrix-step a col)
)

(defun math-compose-matrix-step (a col)
  (if (= col cols)
      (list (cons just
		  (cons base
			(mapcar (function (lambda (r)
					    (math-compose-expr (nth col r) 0)))
				a))))
    (cons (cons just
		(cons base
		      (mapcar (function
			       (lambda (r) (list 'horiz
						 (math-compose-expr (nth col r)
								    0)
						 (concat comma " "))))
			      a)))
	  (math-compose-matrix-step a (1+ col))))
)

(defun math-vector-is-string (a)
  (and (cdr a)
       (progn
	 (while (and (setq a (cdr a))
		     (natnump (car a))
		     (<= (car a) 255)))
	 (null a)))
)

(defun math-tex-expr-is-flat (a)
  (or (Math-integerp a)
      (memq (car a) '(float var))
      (and (memq (car a) '(+ - *))
	   (progn
	     (while (and (setq a (cdr a))
			 (math-tex-expr-is-flat (car a))))
	     (null a))))
)



;;; Convert a composition to string form, with embedded \n's if necessary.

(defun math-composition-to-string (c &optional width)
  (or width (setq width (calc-window-width)))
  (if calc-display-raw
      (math-comp-to-string-raw c 0)
    (if (math-comp-is-flat c)
	(math-comp-to-string-flat c width)
      (math-vert-comp-to-string
       (math-comp-simplify c width))))
)

(defun math-comp-is-flat (c)     ; check if c's height is 1.
  (cond ((not (consp c)) t)
	((eq (car c) 'break) t)
	((eq (car c) 'horiz)
	 (while (and (setq c (cdr c))
		     (math-comp-is-flat (car c))))
	 (null c))
	((memq (car c) '(vleft vcent vright))
	 (and (= (length c) 3)
	      (= (nth 1 c) 0)
	      (math-comp-is-flat (nth 2 c))))
	(t nil))
)


;;; Convert a one-line composition to a string.

(defun math-comp-to-string-flat (c full-width)
  (let ((comp-buf "")
	(comp-word "")
	(comp-pos 0)
	(comp-wlen 0))
    (math-comp-to-string-flat-term c)
    (math-comp-to-string-flat-term '(break -1))
    comp-buf)
)

(defun math-comp-to-string-flat-term (c)
  (cond ((not (consp c))
	 (setq comp-word (concat comp-word c)
	       comp-wlen (+ comp-wlen (length c))))
	((eq (car c) 'horiz)
	 (while (setq c (cdr c))
	   (math-comp-to-string-flat-term (car c))))
	((eq (car c) 'break)
	 (if (or (<= (+ comp-pos comp-wlen) full-width)
		 (= (length comp-buf) 0)
		 (not calc-line-breaking))
	     (setq comp-buf (concat comp-buf comp-word)
		   comp-pos (+ comp-pos comp-wlen))
	   (if calc-line-numbering
	       (setq comp-buf (concat comp-buf "\n     " comp-word)
		     comp-pos (+ comp-wlen 5))
	     (setq comp-buf (concat comp-buf "\n " comp-word)
		   comp-pos (1+ comp-wlen))))
	 (setq comp-word ""
	       comp-wlen 0))
	(t (math-comp-to-string-flat-term (nth 2 c))))
)


;;; Simplify a composition to a canonical form consisting of
;;;   (vleft n "string" "string" "string" ...)
;;; where 0 <= n < number-of-strings.

(defun math-comp-simplify (c full-width)
  (let ((comp-buf (list ""))
	(comp-base 0)
	(comp-height 1)
	(comp-hpos 0)
	(comp-vpos 0))
    (math-comp-simplify-term c)
    (cons 'vleft (cons comp-base comp-buf)))
)

(defun math-comp-add-string (s h v)
  (and (> (length s) 0)
       (let ((vv (+ v comp-base)))
	 (if (< vv 0)
	     (setq comp-buf (nconc (make-list (- vv) "") comp-buf)
		   comp-base (- v)
		   comp-height (- comp-height vv)
		   vv 0)
	   (if (>= vv comp-height)
	       (setq comp-buf (nconc comp-buf
				     (make-list (1+ (- vv comp-height)) ""))
		     comp-height (1+ vv))))
	 (let ((str (nthcdr vv comp-buf)))
	   (setcar str (concat (car str)
			       (make-string (- h (length (car str))) 32)
			       s)))))
)

(defun math-comp-simplify-term (c)
  (cond ((stringp c)
	 (math-comp-add-string c comp-hpos comp-vpos)
	 (setq comp-hpos (+ comp-hpos (length c))))
	((eq (car c) 'break)
	 nil)
	((eq (car c) 'horiz)
	 (while (setq c (cdr c))
	   (math-comp-simplify-term (car c))))
	((memq (car c) '(vleft vcent vright))
	 (let* ((comp-vpos (+ (- comp-vpos (nth 1 c))
			      (1- (math-comp-ascent (nth 2 c)))))
		(widths (mapcar 'math-comp-width (cdr (cdr c))))
		(maxwid (apply 'max widths))
		(bias (cond ((eq (car c) 'vleft) 0)
			    ((eq (car c) 'vcent) 1)
			    (t 2))))
	   (setq c (cdr c))
	   (while (setq c (cdr c))
	     (if (eq (car-safe (car c)) 'rule)
		 (math-comp-add-string (make-string maxwid ?-)
				       comp-hpos comp-vpos)
	       (let ((comp-hpos (+ comp-hpos (/ (* bias (- maxwid
							   (car widths)))
						2))))
		 (math-comp-simplify-term (car c))))
	     (and (cdr c)
		  (setq comp-vpos (+ comp-vpos
				     (+ (math-comp-descent (car c))
					(math-comp-ascent (nth 1 c))))
			widths (cdr widths))))
	   (setq comp-hpos (+ comp-hpos maxwid))))
	((eq (car c) 'supscr)
	 (math-comp-simplify-term (nth 1 c))
	 (let* ((asc (math-comp-ascent (nth 1 c)))
		(desc (math-comp-descent (nth 2 c)))
		(comp-vpos (- comp-vpos (+ asc desc))))
	   (math-comp-simplify-term (nth 2 c))))
	((eq (car c) 'subscr)
	 (math-comp-simplify-term (nth 1 c))
	 (let* ((asc (math-comp-ascent (nth 2 c)))
		(desc (math-comp-descent (nth 1 c)))
		(comp-vpos (+ comp-vpos (+ asc desc))))
	   (math-comp-simplify-term (nth 2 c)))))
)


;;; Measuring a composition.

(defun math-comp-first-char (c)
  (cond ((stringp c)
	 (and (> (length c) 0)
	      (elt c 0)))
	((memq (car c) '(horiz subscr supscr))
	 (let (ch)
	   (while (and (setq c (cdr c))
		       (not (setq ch (math-comp-first-char (car c))))))
	   ch)))
)

(defun math-comp-last-char (c)
  (cond ((stringp c)
	 (and (> (length c) 0)
	      (elt c (1- (length c)))))
	((eq (car c) 'horiz)
	 (let ((c (reverse (cdr c))) ch)
	   (while (and c
		       (not (setq ch (math-comp-last-char (car c)))))
	     (setq c (cdr c)))
	   ch)))
)

(defun math-comp-width (c)
  (cond ((not (consp c)) (length c))
	((memq (car c) '(horiz subscr supscr))
	 (let ((accum 0))
	   (while (setq c (cdr c))
	     (setq accum (+ accum (math-comp-width (car c)))))
	   accum))
	((memq (car c) '(vcent vleft vright))
	 (setq c (cdr c))
	 (let ((accum 0))
	   (while (setq c (cdr c))
	     (setq accum (max accum (math-comp-width (car c)))))
	   accum))
	(t 0))
)

(defun math-comp-height (c)
  (if (stringp c)
      1
    (+ (math-comp-ascent c) (math-comp-descent c)))
)

(defun math-comp-ascent (c)
  (cond ((not (consp c)) 1)
	((eq (car c) 'horiz)
	 (let ((accum 0))
	   (while (setq c (cdr c))
	     (setq accum (max accum (math-comp-ascent (car c)))))
	   accum))
	((memq (car c) '(vcent vleft vright))
	 (if (> (nth 1 c) 0) (1+ (nth 1 c)) 1))
	((eq (car c) 'supscr)
	 (+ (math-comp-ascent (nth 1 c)) (math-comp-height (nth 2 c))))
	((eq (car c) 'subscr)
	 (math-comp-ascent (nth 1 c)))
	(t 1))
)

(defun math-comp-descent (c)
  (cond ((not (consp c)) 0)
	((eq (car c) 'horiz)
	 (let ((accum 0))
	   (while (setq c (cdr c))
	     (setq accum (max accum (math-comp-descent (car c)))))
	   accum))
	((memq (car c) '(vcent vleft vright))
	 (let ((accum (- (nth 1 c))))
	   (setq c (cdr c))
	   (while (setq c (cdr c))
	     (setq accum (+ accum (math-comp-height (car c)))))
	   (max (1- accum) 0)))
	((eq (car c) 'supscr)
	 (math-comp-descent (nth 1 c)))
	((eq (car c) 'subscr)
	 (+ (math-comp-descent (nth 1 c)) (math-comp-height (nth 2 c))))
	(t 0))
)


;;; Convert a simplified composition into string form.

(defun math-vert-comp-to-string (c)
  (if (stringp c)
      c
    (math-vert-comp-to-string-step (cdr (cdr c))))
)

(defun math-vert-comp-to-string-step (c)
  (if (cdr c)
      (concat (car c) "\n" (math-vert-comp-to-string-step (cdr c)))
    (car c))
)


;;; Convert a composition to a string in "raw" form (for debugging).

(defun math-comp-to-string-raw (c indent)
  (cond ((not (consp c))
	 (prin1-to-string c))
	(t
	 (let ((next-indent (+ indent 2 (length (symbol-name (car c))))))
	   (if (null (cdr c))
	       (concat "(" (symbol-name (car c)) ")")
	     (concat "("
		     (symbol-name (car c))
		     " "
		     (math-comp-to-string-raw (nth 1 c) next-indent)
		     (math-comp-to-string-raw-step (cdr (cdr c))
						   next-indent)
		     ")")))))
)

(defun math-comp-to-string-raw-step (cl indent)
  (if cl
      (concat "\n"
	      (make-string indent 32)
	      (math-comp-to-string-raw (car cl) indent)
	      (math-comp-to-string-raw-step (cdr cl) indent))
    "")
)






;;;; [end]


;;;; Splitting calc-ext.el into smaller parts.  [Suggested by Juha Sarlin.]

(defun calc-split (directory no-save)
  "Split the file \"calc-ext.el\" into smaller parts for faster loading.
This should be done during installation of Calc only."
  (interactive "DDirectory for resulting files: \nP")
  (or (string-match "calc-ext.el" (buffer-file-name))
      (error "This command is for Calc installers only.  (Refer to the documentation.)"))
  (or (equal directory "")
      (setq directory (file-name-as-directory (expand-file-name directory))))
  (and (or (get-buffer "calc-incom.el")
	   (file-exists-p (concat directory "calc-incom.el")))
       (error "calc-split has already been used!"))
  (let (copyright-point
	autoload-point
	(start (point-marker))
	filename
	(dest-buffer nil)
	(done nil)
	(func-list nil)
	(cmd-list nil)
	(file-list nil))
    (goto-char (point-min))
    (search-forward ";;;; (Autoloads here)\n")
    (setq autoload-point (point-marker))
    (goto-char (point-min))
    (search-forward ";;;;")
    (forward-char -4)
    (setq copyright-point (point))
    (copy-file (buffer-file-name) "calc-old.el" t)
    (while (not done)
      (re-search-forward "^;;;; \\[\\(.*\\)\\]\n\\|^(defun \\|^(fset '")
      (if (equal (buffer-substring (match-beginning 0)
				   (1+ (match-beginning 0)))
		 ";")
	  (progn
	    (setq filename (buffer-substring (match-beginning 1)
					     (match-end 1)))
	    (and dest-buffer
		 (progn
		   (append-to-buffer dest-buffer
				     start (match-beginning 0))
		   (delete-region start (match-beginning 0))))
	    (if (equal filename "end")
		(progn
		  (delete-region (point) (point-max))
		  (setq done t))
	      (set-marker start (point))
	      (setq dest-buffer (and (not (equal filename "calc-ext.el"))
				     (find-file-noselect
				      (concat directory filename))))
	      (message "Splitting to %s..." filename)
	      (and dest-buffer
		   (save-excursion
		     (set-buffer dest-buffer)
		     (= (buffer-size) 0))
		   (save-excursion
		     (append-to-buffer dest-buffer
				       (point-min) copyright-point)
		     (set-buffer dest-buffer)
		     (goto-char (point-min))
		     (end-of-line)
		     (insert " [" filename "]")
		     (goto-char (point-max))
		     (insert "\n"
			     ";; This file is autoloaded from calc-ext, which in turn is loaded from calc.\n"
			     "(require 'calc-ext)\n\n")))))
	(and dest-buffer
	     (let* ((name (progn
			    (looking-at "[^ \n]*")
			    (buffer-substring (match-beginning 0)
					      (match-end 0))))
		    (interactive (and (not (string-match
					    "calcFunc-\\|math-" name))
				      (save-excursion
					(re-search-forward "^ *(")
					(looking-at "interactive"))))
		    (which (if interactive 'cmd-list 'func-list))
		    (small-filename (substring filename 0 -3))
		    (found (or (assoc small-filename (symbol-value which))
			       (car (set which
					 (cons (list small-filename)
					       (symbol-value which)))))))
	       (or (assoc filename file-list)
		   (setq file-list (cons (list filename) file-list)))
	       (setcdr found (cons (intern name) (cdr found)))))))
    (goto-char autoload-point)
    (insert "  (let ((dir \"" directory "\"))\n"
	    "    (mapcar (function (lambda (x)\n"
	    "      (let ((file (concat dir (car x))))\n"
	    "        (mapcar (function (lambda (func)\n"
	    "          (autoload func file))) (cdr x)))))\n"
	    "      '" (prin1-to-string func-list) ")\n"
	    "    (mapcar (function (lambda (x)\n"
	    "      (let ((file (concat dir (car x))))\n"
	    "        (mapcar (function (lambda (cmd)\n"
	    "          (autoload cmd file nil t))) (cdr x)))))\n"
	    "      '" (prin1-to-string cmd-list) "))\n")
    (fill-region autoload-point (point))
    (goto-char (point-min))
    (or no-save
	(progn
	  (save-some-buffers t)
	  (if (y-or-n-p "Byte-compile all files? ")
	      (progn
		(require 'calc)
		(byte-compile-file "calc-ext.el")
		(load-file "calc-ext.elc")
		(mapcar (function
			 (lambda (x)
			   (byte-compile-file
			    (concat directory (car x)))))
			file-list)))))
    (message "Done."))
)

;;; Type C-x C-e at the beginning of this line before running calc-split.




;;;; End.

