;; Extra DIRED commands for Emacs. $Revision: 1.17 $
;; Copyright (C) 1990 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;; Installation: In your ~/.emacs, say
;;   (autoload 'dired-extra-startup "dired-extra")
;;
;; and either
;;   (setq dired-mode-hook 'dired-extra-startup)
;; or
;;   (setq dired-mode-hook '(lambda () ... (dired-extra-startup)))
;; if you already have other things in your dired-mode-hook.
;;
;; Add dired-omit-expunge to dired-readin-hook:  either
;;   (setq dired-readin-hook 'dired-omit-expung)
;; or
;; (setq dired-readin-hook '(lambda () ... (dired-omit-expunge)))
;; if you already have other things in your dired-readin-hook.

(require 'dired)			; we will redefine some functions

(defconst dired-extra-version
  "$Id: dired-extra.el,v 1.17 1991/01/18 16:10:09 sk Exp $")

(defvar dired-mark-keys '("1" "2")
  "*List of keys that insert themselves as file markers.")

(defvar dired-mark-use-gmhist t
  "*Set it to t if you want shell command history and have gmhist.el")

(defvar dired-dangerous-shell-command "rm" ; non-anchored match, e.g. "rmdir"
  "*Regexp for dangerous shell commands that should never be the default.")

(defvar dired-default-marker dired-marker-char
  "*The value of dired-marker-char in effect before dired-extra was
loaded and the value which is restored if the marker stack underflows.
This is usually the asterisk.")

(defvar dired-extra-startup nil
  "t if dired-extra-startup has been called already.")

(defun dired-extra-startup ()
  "Put this on your dired-mode-hook to get extra dired features:

  V    -- VM on folder
  (, ) -- change and display dired-marker-char dynamically.
  M-o  -- toggle omitting of files

For more features, see variables dired-omit-files, dired-omit-extenstions
dired-mark-use-gmhist, dired-dangerous-shell-command, dired-mark-keys.
See also function dired-sort-on-size."
  (interactive)
  (if dired-extra-startup
      nil
    (setq dired-extra-startup t)
    (define-key dired-mode-map "V" 'dired-vm)
    (define-key dired-mode-map "\(" 'dired-set-marker-char)
    (define-key dired-mode-map "\)" 'dired-restore-marker-char)
    (define-key dired-mode-map "I" 'dired-mark-insert-subdir)
    (define-key dired-mode-map "\M-\C-@" 'dired-push-mark-subdir)
    ;;(define-key dired-mode-map "\M-f" 'dired-flag-extension)
    (define-key dired-mode-map "\M-M" 'dired-mark-unmark)
    (define-key dired-mode-map "\M-o" 'dired-omit-toggle)
    (mapcar (function
	     (lambda (x)
	       (define-key dired-mode-map x 'dired-mark-with-this-char)))
	    dired-mark-keys))
  ;; This must be done in each new dired buffer:
  (make-local-variable 'dired-omit-files-p)
  ;;- protect with?: or (assoc 'dired-omit-files-p minor-mode-alist)
  (setq minor-mode-alist (cons '(dired-omit-files-p " Omit")
			       minor-mode-alist))
  (dired-omit-expunge)
  (make-local-variable 'dired-marker-char)
  (make-local-variable 'dired-del-marker)
  (make-local-variable 'dired-marker-stack)
  (setq minor-mode-alist
	(cons '(dired-marker-stack dired-marker-string)
	      minor-mode-alist))
  )

;;; Handle customization

(if (not dired-mark-use-gmhist)
    nil
  ;; Else use generic minibuffer history
  (autoload 'read-with-history-in "gmhist")
  (put 'dired-shell-command-history 'dangerous dired-dangerous-shell-command)
  ;; Redefinition - when this is loaded, dired.el has alreay been loaded.

  (defun dired-read-shell-command (prompt arg)
    "Read a dired shell command using generic minibuffer history."
    (dired-mark-pop-up
     (function
      (lambda ()
	(read-with-history-in 'dired-shell-command-history
			      (format prompt (dired-mark-prompt arg)))))
     nil 'shell))

  (defun dired-read-regexp (prompt)
    (setq dired-flagging-regexp
	  (read-with-history-in 'regexp-history prompt)))
)


;;; Dynamic Markers

(defun dired-mark-with-this-char (arg)
  "Mark the current file with the key you pressed to invoke this command."
  (interactive "p")
  (let ((dired-marker-char (aref (this-command-keys) 0)))
    (dired-mark-file arg)))

(defvar dired-marker-stack nil
  "List of previously used marker characters.")

(defvar dired-marker-string ""
  "String version of dired-marker-stack.")

(defun dired-marker-string ()
  "Computes and returns dired-marker-string."
  (setq dired-marker-string
	(concat " "
		(mapconcat (function char-to-string)
			   (reverse dired-marker-stack)
			   ""))))

(defun dired-set-marker-char (c)
  "Set the marker character to something else.
Use \\[dired-restore-marker-char] to restore the previous value."
  (interactive "cNew marker character: ")
  (setq dired-marker-stack (cons c dired-marker-stack))
  (dired-marker-string)
  (setq dired-marker-char c)
  (if (and (boundp 'dired-traditional) (not dired-traditional))
      ;; Must keep del-marker in sync
      (setq dired-del-marker c))
  (set-buffer-modified-p (buffer-modified-p)) ; update mode line
  (message "New marker is %c" dired-marker-char))

(defun dired-restore-marker-char ()
  "Restore the marker character to its previous value.
Uses dired-default-marker if the marker stack is empty."
  (interactive)
  (setq dired-marker-stack (cdr dired-marker-stack)
	dired-marker-char (car dired-marker-stack))
  (dired-marker-string)
  (set-buffer-modified-p (buffer-modified-p)) ; update mode line
  (or dired-marker-char (setq dired-marker-char dired-default-marker))
  (if (and (boundp 'dired-traditional) (not dired-traditional))
      ;; Must keep del-marker in sync
      (setq dired-del-marker dired-marker-char))
  (message "Marker is %c" dired-marker-char))

;;; Sort on Size kludge if your ls can't do it

(defun dired-sort-on-size ()
  "Sorts a dired listing on file size.
If your ls cannot sort on size, this is useful as dired-readin-hook:
    (setq dired-readin-hook 'dired-sort-on-size)"
  (require 'sort)
  (goto-char (point-min))
  (dired-goto-next-file)		; skip `total' line
  (beginning-of-line)
  (sort-subr t				; biggest file first
	     'forward-line 'end-of-line 'dired-get-file-size))

(defun dired-get-file-size ()
  (re-search-forward "\\(Jan\\|Feb\\|Mar\\|Apr\\|May\\|Jun\\|Jul\\|Aug\\|Sep\\|Oct\\|Nov\\|Dec\\)")
  (goto-char (match-beginning 1))
  (forward-char -1)
  (string-to-int (buffer-substring (save-excursion
				     (backward-word 1)
				     (point))
				   (point))))

;;; Misc. (mostly featurismic) commands

(defun dired-vm ()
  "Run VM on this file."
  (interactive)
  (vm-visit-folder (dired-get-filename))
  (set (make-local-variable 'vm-folder-directory) default-directory))

(defun dired-rmail ()
  "Run `rmail' on this file."
  (interactive)
  (let ((buffer-read-only nil)
	(buffer (current-buffer))
	(file (dired-get-filename)))
    (rmail file)
    (recursive-edit)
    (switch-to-buffer buffer)
    (dired-redisplay file)))

(defun dired-mark-insert-subdir ()
  "Insert all marked subdirectories in situ.
Non-directories are silently ignored."
  (interactive)
  (let ((files (or (dired-mark-get-files)
		   (error "No files marked."))))
    (while files
      (if (file-directory-p (car files))
	  (dired-insert-subdir (car files)))
      (setq files (cdr files)))))

(defun dired-push-mark-subdir ()
  "Put mark at end of subdir, point at beginning."
  (interactive)
  (push-mark (dired-subdir-max))
  ;; This must come last, as (dired-subdir-min) actually moves outside
  ;; the current subdir to put point before the header line.
  (goto-char (dired-subdir-min)))

(defun dired-flag-extension (extension &optional arg)
  "In dired, flag all files with a certain extension.
Directories are not flagged unless a prefix argument is given."
  (interactive "sFlagging extension: \nP")
  ;; match only when basename is non-empty:
  (dired-flag-regexp-files (concat ".\\." (regexp-quote extension) "$" arg)))

(defun dired-mark-unmark (unmarker)
  "Unmark marked files by replacing the marker with another string.
The string defaults to \" \" (a space), effectively unmarking them."
  (interactive "sChange marker to: ")
  (if (string= unmarker "")
      (setq unmarker " "))
  (setq unmarker (substring unmarker 0 1))
  (let ((regexp (dired-marker-regexp))
	(buffer-read-only nil))
    (save-excursion
      (goto-char (point-min))
      (while (re-search-forward regexp nil t)
	(replace-match unmarker)))))

;;; Omitting

;;; Enhanced omitting of lines from directory listings.
;;; Marked files are never omitted.
;;; Adapted from code submitted by:
;;; Michael D. Ernst, mernst@theory.lcs.mit.edu, 1/11/91

(defvar dired-omit-files-p nil
  "*If non-nil, \"uninteresting\" files are not listed.
Use \\[dired-omit-toggle] to toggle its value.
Uninteresting files are those whose filenames match regexp dired-omit-files, 
plus those ending with extensions in dired-omit-extensions.")

(defvar dired-omit-files "\\.$\\|#"
  "*Filenames matching this regexp will not be displayed.
This only has effect when dired-omit-files-p is t.
See also dired-omit-extensions.")

(defvar dired-omit-extensions completion-ignored-extensions
  "*If non-nil, a list of extensions (strings) to omit from Dired listings.
Defaults to completion-ignored-extensions.")

;; \017=^O for Omit - other packages can chose other control characters.
(defconst dired-omit-marker-char ?\017
  "Temporary marker used by by dired-omit.
Should never be used as marker by the user or other packages.")

(defun dired-omit-toggle ()
  "Toggle between displaying and omitting files matching dired-omit-files."
  (interactive)
  (setq dired-omit-files-p (not dired-omit-files-p))
  (if (not dired-omit-files-p)
      (revert-buffer)
    (message "Omitting `%s'..." (dired-omit-regexp))
    (dired-omit-expunge)
    (message "Omitted `%s'." (dired-omit-regexp))))

(defun dired-omit-expunge (&optional regexp)
  "Erases all unmarked files matching REGEXP.
Does nothing if global variable dired-omit-files-p is nil.
If REGEXP is nil or not specified, uses dired-omit-files, and also omits
  filenames ending in dired-omit-extensions.
If REGEXP is the empty string, this function is a no-op.

This functions works by temporarily binding dired-marker-char to
dired-omit-marker-char's value and calling dired-mark-kill."
  (interactive "sOmit files (regexp): ")
  (if dired-omit-files-p
     (let ((omit-re (or regexp (dired-omit-regexp))))
       (or (string= omit-re "")
	   (let ((dired-marker-char dired-omit-marker-char))
	     (dired-mark-unmarked-files omit-re nil)
	     (dired-mark-kill))))))

(defun dired-omit-regexp ()
  (concat (if dired-omit-files (concat "\\(" dired-omit-files "\\)") "")
	  (if (and dired-omit-files dired-omit-extensions) "\\|" "")
	  (if dired-omit-extensions
	      (concat ".+\\("
		      (mapconcat 'regexp-quote dired-omit-extensions "\\|")
		      "\\)$")
	    "")))

(defun dired-mark-unmarked-files (regexp msg &optional unflag-p)
  "Marks unmarked files matching REGEXP, displaying MSG.
Does not re-mark files which already have a mark.
With prefix argument, unflag all those files."
  (interactive "P")
  (let ((dired-marker-char (if unflag-p ?\  dired-marker-char)))
    (dired-mark-if 
     (and
      ;; not already marked
      (looking-at " ")
      ;; uninteresting
      (save-excursion
	(dired-move-to-filename)
	(looking-at regexp)))
     msg)))

