;; dired-vms.el - VMS support for dired. $Revision: 1.5 $
;; Copyright (C) 1990 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 1, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

;; $Id: dired-vms.el,v 1.5 90/12/21 12:28:38 sk Exp $

(setq dired-subdir-regexp "^ *Directory \\([][:.A-Z-0-9_$;<>]+\\)\\(\\)[\n\r]")

(defconst dired-vms-filename-regexp "[][:.A-Z-0-9_$;<>]+"
  "Regexp matching a VMS filename.")

(defvar dired-directory-command
  "DIRECTORY/SIZE/DATE/PROT"
  "Directory command for dired under VMS.")

;; requires vmsproc.el to work
(defun dired-vms-read-directory (dirname switches buffer)
  (subprocess-command-to-buffer
   (concat dired-directory-command " " dirname)
   buffer)
  (goto-char (point-min))
  (replace-regexp " *$" ""))

(defun dired-ls (file &optional switches wildcard full-directory-p)
  "Insert ls output of FILE, optionally formatted with SWITCHES.
Optional third arg WILDCARD means treat FILE as shell wildcard.
Optional fourth arg FULL-DIRECTORY-P means file is a directory and
switches do not contain `d'.

SWITCHES default to dired-listing-switches.

This is the VMS version of this UNIX command.
The SWITCHES and WILDCARD arguments are ignored.
Uses dired-directory-command."
  (save-restriction;; Must drag point along:
    (narrow-to-region (point) (point))
    (subprocess-command-to-buffer
     (concat dired-directory-command " " file)
     (current-buffer))
    (if full-directory-p
	(goto-char (point-max))
      ;; Just the file line if no full directory required:
      (goto-char (point-min))  
      (let ((case-fold-search nil))
	(re-search-forward dired-subdir-regexp)
	(re-search-forward (concat "^" dired-vms-filename-regexp)))
      (beginning-of-line)
      (delete-region (point-min) (point))
      (forward-line 1)
      (delete-region (point) (point-max)))))

(defun dired-insert-headerline (dir)	; redefinition
  ;; VMS dired-ls makes its own headerline, but we must position the
  ;; cursor where dired-insert-subdir expects it.
  ;; This does not check whether the headerline matches DIR.
  (re-search-forward dired-subdir-regexp)
  (goto-char (match-end 1)))

(defun dired-make-absolute (file dir)
  ;;"Convert FILE (a pathname relative to DIR) to an absolute pathname."
  ;; This should be good enough for ange-ftp, but might easily be
  ;; redefined (for VMS?).
  ;; It should be reasonably fast, though, as it is called in
  ;; dired-get-filename.
  (expand-file-name file dir))

(defun dired-make-relative (file dir)
  ;;"Convert FILE (an absolute pathname) to a pathname relative to DIR.
  ;;Else error."
  ;; DIR must be file-name-as-directory, as with all directory args in
  ;; elisp code. 
  (or (dired-in-this-tree file dir)
      (error  "%s: not in directory tree growing at %s." file dir))
  (let* ((dir-split (dired-vms-split-filename dir))
	 (dir-dev (nth 0 dir-split))
	 (dir-dir (nth 1 dir-split) )
	 (file-split (dired-vms-split-filename file))
	 (file-dev (nth 0 file-split))
	 (file-dir (nth 1 file-split))
	 (file-fil (nth 2 file-split))
	 )
    (or (string= dir-dev file-dev)	; paranoid
	(error "Device mismatch: %s: not in directory tree growing at %s."))
    (if (string-match (regexp-quote dir-dir) file-dir)
	(concat "[" (substring file-dir (match-end 0)) "]" file-fil))))

(defun dired-in-this-tree (file dir)
  ;;"Is FILE part of the directory tree starting at DIR?"
  ;; Under VMS, file="DEV:[foo.bar]zod", dir="DEV:[foo]"
  (or (string= (substring dir -1) "\]")
      (error ""))
  (string-match (concat "^" (regexp-quote (substring dir 0 -1)))
		 file))

(defun dired-vms-split-filename (file)
  (if (string-match;; "DEV:[DIR]FIL" \1=DEV \2=DIR \3=FIL
       "^\\([.A-Z-0-9_$;]*\\):?[[<]\\([.A-Z-0-9_$;]*\\)[]>]\\([.A-Z-0-9_$;]*\\)$"
       file)
      (mapcar '(lambda (x)
		 (substring file (match-beginning x) (match-end x)))
	      '(1 2 3))))

;; Must use this in dired-noselect instead of expand-file-name and
;; file-name-as-directory
;; Taken from the VMS dired version by
;;Roland Roberts                      BITNET: roberts@uornsrl
;;  Nuclear Structure Research Lab  INTERNET: rbr4@uhura.cc.rochester.edu
;;  271 East River Road                 UUCP: rochester!ur-cc!uhura!rbr4
;;  Rochester, NY  14267                AT&T: (716) 275-8962

(defun dired-fix-directory (dirname)
  "Fix up dirname to be a valid directory name and return it"
  ;; Under VMS, "[000000]" is the root dir and "[-]" is "../".
  (or dirname (setq dirname default-directory))
  (if (eq system-type 'vax-vms)
      (progn
	(setq dirname (expand-file-name dirname))
	(let ((end (1- (length dirname)))
	      (beg 0)
	      (fn nil))
	(if (and (file-directory-p dirname)
		 (not (char-equal ?\] (elt dirname end))))
	    (progn
	      (while (= 0 beg)
		(if (char-equal ?\. (elt dirname end))
		    (setq beg (1- end))
		  (setq end (1- end))))
	      (while (null fn)
		(if (char-equal ?\] (elt dirname beg))
		    (setq fn (substring dirname (1+ beg) end))
		  (setq beg (1- beg))))
	      (message "prevdir = %s" (substring dirname (- beg 6) beg))
	      (if (string= "000000" (substring dirname (- beg 6) beg))
		  (concat (substring dirname 0 (- beg 6)) fn "]")
		(concat (substring dirname 0 beg) "." fn "]")))
	  (if (char-equal ?- (elt dirname (1- end)))
	      (concat (substring dirname 0 (1- end)) "000000]")
	    dirname))))
    ;; else UNIX style
    (if (string-match "./$" dirname)
	(setq dirname (substring dirname 0 -1)))
    (setq dirname (expand-file-name dirname))
    (and (not (string-match "/$" dirname))
	 (file-directory-p dirname)
	 (setq dirname (concat dirname "/")))
    dirname))

(defun dired-noselect (dirname &optional arg)
  "Like M-x dired but returns the dired buffer as value, does not select it."
  (setq dirname (dired-fix-directory dirname))
  (dired-internal-noselect dirname arg))

;; Versions are not yet supported in dired.el (as of version 4.53):
(setq dired-file-version-regexp "[.;][0-9]+$")
