;;!emacs
;;
;; FILE:         hibtypes.el
;; SUMMARY:      Hyperbole System Implicit Button Types.
;; USAGE:        GNU Emacs Lisp Library
;;
;; AUTHOR:       Bob Weiner
;; ORG:          Brown U.
;;
;; ORIG-DATE:    19-Sep-91 at 20:45:31
;; LAST-MOD:     13-Dec-91 at 14:47:28 by Bob Weiner
;;
;; This file is part of Hyperbole.
;; 
;; Copyright (C) 1991, Brown University, Providence, RI
;; Developed with support from Motorola Inc.
;; 
;; Permission to use, modify and redistribute this software and its
;; documentation for any purpose other than its incorporation into a
;; commercial product is hereby granted without fee.  A distribution fee
;; may be charged with any redistribution.  Any distribution requires
;; that the above copyright notice appear in all copies, that both that
;; copyright notice and this permission notice appear in supporting
;; documentation, and that neither the name of Brown University nor the
;; author's name be used in advertising or publicity pertaining to
;; distribution of the software without specific, written prior permission.
;; 
;; Brown University makes no representations about the suitability of this
;; software for any purpose.  It is provided "as is" without express or
;; implied warranty.
;;
;;
;; DESCRIPTION:  
;;
;;   See doc for 'ibtype:create' for details on implicit button type creation.
;;
;;   Define implicit button types in REVERSE order that you want their
;;   predicates tested by the Hyperbole evaluator, i.e. most important last.
;;
;;   Each at-p form of each implicit button type must return either
;;   a 3 element list: 
;;      (button-label-key label-start-position label-end-position)
;;   or a 1 element list:
;;      (button-label-key)
;;
;; DESCRIP-END.

;;; ************************************************************************
;;; Other required Elisp libraries
;;; ************************************************************************

(require 'hbut)
(require 'hactypes)

;;; ************************************************************************
;;; Public implicit button types
;;; ************************************************************************
  
;;; ========================================================================
;;; Handles internal references within an annotated bibliography, delimiters=[]
;;; ========================================================================

(defib annot-bib ()
  "Displays annotated bibliography entries referenced internally.
References must be delimited by square brackets."
  (and (not (bolp)) (hbut:label-p nil "[" "]" t)))

;;; ========================================================================
;;; Executes or documents command bindings of brace delimited key sequences.
;;; ========================================================================

(require 'hib-kbd)

;;; ========================================================================
;;; Makes directory summaries into file list menus.
;;; ========================================================================

(defib dir-summary ()
  "Detects filename buttons in files named \"MANIFEST\" or \"DIR\".
Displays selected files."
  (if buffer-file-name
      (let ((file (file-name-nondirectory buffer-file-name)))
	(if (or (string= file "DIR") (string= file "MANIFEST"))
	    (save-excursion
	      (beginning-of-line)
	      (if (looking-at "\\([^(){}* \t\n]+\\)[ \t]+[^(){}* \t\n]")
		  (list (buffer-substring (match-beginning 1) (match-end 1))
			(match-beginning 1) (match-end 1)))))))
  link-to-file)

;;; ========================================================================
;;; Makes source entries in Hyperbole reports selectable.
;;; ========================================================================

(defib hyp-source ()
  "Makes source entries in Hyperbole reports into buttons that jump to source."
  (save-excursion
    (beginning-of-line)
    (if (looking-at hbut:source-prefix)
	(let ((src (hbut:source)))
	  (if src
	      (list (if (stringp src) src (prin1-to-string src))
		    (point) (progn (end-of-line) (point))))))))

;;; ========================================================================
;;; Shows man page associated with a man apropos entry.
;;; ========================================================================

(defib man-apropos ()
  "Makes man apropos entries display associated man pages when selected."
  (save-excursion
    (beginning-of-line)
    (let ((nm "[^ \t\n!@,][^ \t\n,]*"))
      (if (looking-at
	   (concat
	    "^\\(\\*[ \t]+[!@]\\)?\\(" nm "[ \t]*,[ \t]*\\)*\\(" nm "\\)[ \t]*"
	    "\\(([-0-9a-zA-z]+)\\)\\(::\\)?[ \t]+-[ \t]+[^ \t\n]"))
	  (list 
	   (concat
	    (buffer-substring (match-beginning 3) (match-end 3))
	    (buffer-substring (match-beginning 4) (match-end 4)))
	   (match-beginning 3) (match-end 4)
	   ))))
  man-show
  )

;;; ========================================================================
;;; Displays files and directories when double quoted pathname is activated.
;;; ========================================================================

(defib pathname ()
  "Makes a delimited, valid pathname display the path entry.
See 'hpath:at-p' for possible delimiters."
     (let ((path (hpath:at-p)))
       (if path (cons path nil)))
     link-to-file)

(provide 'hibtypes)

