;;!emacs
;;
;; FILE:         hsys-www.el
;; SUMMARY:      Hyperbole support for WorldWide Web (WWW) document browsing.
;; USAGE:        GNU Emacs Lisp Library
;;
;; AUTHOR:       Bob Weiner
;; ORG:          Brown U.
;;
;; ORIG-DATE:    12-Oct-91 at 03:48:23
;; LAST-MOD:     13-Dec-91 at 14:54:16 by Bob Weiner
;;
;; This file is part of Hyperbole.
;; 
;; Copyright (C) 1991, Brown University, Providence, RI
;; Developed with support from Motorola Inc.
;; 
;; Permission to use, modify and redistribute this software and its
;; documentation for any purpose other than its incorporation into a
;; commercial product is hereby granted without fee.  A distribution fee
;; may be charged with any redistribution.  Any distribution requires
;; that the above copyright notice appear in all copies, that both that
;; copyright notice and this permission notice appear in supporting
;; documentation, and that neither the name of Brown University nor the
;; author's name be used in advertising or publicity pertaining to
;; distribution of the software without specific, written prior permission.
;; 
;; Brown University makes no representations about the suitability of this
;; software for any purpose.  It is provided "as is" without express or
;; implied warranty.
;;
;;
;; DESCRIPTION:  
;;
;;   A Hyperbole button should be created that has 'hwww:start' as its
;;   action type.  It may optionally contain a file name argument as
;;   the initial file to display.  When selected, it starts a 'www'
;;   process and displays the initial file.
;;
;;   The 'hwww:link-follow' implicit button type is then used when the
;;   user clicks inside the buffer containing the 'www' output.  It
;;   passes commands to the 'hwww:link-follow' action type.
;;
;; DESCRIP-END.

;;; ************************************************************************
;;; Other required Elisp libraries
;;; ************************************************************************

;;; Requires external 'www' executable available via anonymous ftp
;;; from info.cern.ch.

(require 'hinit)

;;; ************************************************************************
;;; Public variables
;;; ************************************************************************

(defib hwww:link-follow ()
  "When in a www buffer, returns a link follow or history recall command."
  (let* ((www (get-buffer-process (current-buffer)))
	 (www-proc-nm (and www (process-name www)))
	 selection)
    (if (and www-proc-nm (equal (string-match "www" www-proc-nm) 0))
	(cond (;; Hyper ref
	       (save-excursion
		 (skip-chars-backward "^ \t\n")
		 (looking-at "[^][ \t\n]+\\[\\([0-9]+\\)\\]"))
	       (list (setq selection
			   (buffer-substring (match-beginning 1)
					     (match-end 1)))
		     (match-beginning 1) (match-end 1)))
	      (;; History list entry
	       (save-excursion
		 (beginning-of-line)
		 (looking-at "[ \t]*\\([0-9]+\\)\)[ \t]+[^ \t\n]"))
	       (list
		(concat "recall "
			(setq selection
			      (buffer-substring (match-beginning 1)
						(match-end 1))))
		(match-beginning 1) (match-end 1)))
	      (;; Hyper ref list
	       (save-excursion
		 (beginning-of-line)
		 (looking-at "[ \t]*\\[\\([0-9]+\\)\\][ \t]+[^ \t\n]"))
	       (list (setq selection
			   (buffer-substring (match-beginning 1)
					     (match-end 1)))
		     (match-beginning 1) (match-end 1)))))))

(defact hwww:link-follow (link-num-str)
  "Follows a link given by LINK-NUM-STR or displays a www history list."
  (interactive "sNumber of WWW link to follow: ")
  (or (stringp link-num-str)
      (error "(hwww:link-follow): Link number must be given as a string."))
  (let ((www (get-buffer-process (current-buffer))))
    (if www
	(progn
	  (setq buffer-read-only nil)
	  (erase-buffer)
	  (process-send-string www (concat link-num-str "\n"))
	  )
      (error "(hwww:link-follow): No current WWW process.  Use 'hwww:start'."))))

(defun hwww:link-follow:help (&optional but)
  "Displays history list of www nodes previously visited."
  (interactive)
  (funcall (actype:action "hwww:link-follow")
	   "recall"))

(defact hwww:start (&optional file)
  "Starts a www process and displays optional FILE.
Without FILE (an empty string), displays default initial www file."
  (interactive "FWWW file to start with: ")
  (or (stringp file)
      (error "(hwww:start): FILE argument is not a string."))
  (let ((www-buf (get-buffer-create "WWW"))
	(www-proc (get-process "www")))
    (save-excursion
      (set-buffer www-buf)
      (setq buffer-read-only nil)
      (erase-buffer)
      )
    (if www-proc
	(pop-to-buffer www-buf)
      (if (setq www-proc
		(if (or (equal file "") (equal file "\"\""))
		    (start-process "www" www-buf "www" "-p")
		  (start-process "www" www-buf "www" "-p" file)))
	  (progn (set-process-sentinel www-proc 'hwww:sentinel)
		 (set-process-filter www-proc 'hwww:filter)
		 (process-kill-without-query www-proc)
		 (pop-to-buffer www-buf)
		 (shell-mode)
		 (make-local-variable 'explicit-shell-file-name)
		 (setq explicit-shell-file-name "www")
		 (use-local-map hwww:mode-map)
		 (if hwww:mode-map
		     nil
		   (setq hwww:mode-map (copy-keymap shell-mode-map))
		   (define-key hwww:mode-map "\C-m" 'hwww:send-input)
		   (define-key hwww:mode-map " " 'hwww:scroll-up)
		   (define-key hwww:mode-map "\177" 'hwww:scroll-down)
		   )
		 (goto-char (point-min))
		 )))))

;;; ************************************************************************
;;; Private functions
;;; ************************************************************************

(defun hwww:filter (process str)
  (if (and (> (length str) 3)
	   (equal "==> " (substring str -4)))
      (progn
	(insert str)
	(goto-char (point-min))
	(ep:but-create (concat "\\([^ \t\n]*\\[[0-9]+\\]\\|"
			       "^[ \t]*\\[\\([0-9]+\\)\\][ \t]+[^ \t\n]+\\|"
			       "^[ ]+[0-9]+\).*\\)")
		       'regexp))
    (insert str)
    ))

(defun hwww:scroll-up (&optional arg)
  "If on last line of buffer, insert space, else scroll up a page."
  (interactive "P")
  (if (last-line-p) (insert " ") (scroll-up arg)))

(defun hwww:scroll-down (&optional arg)
  "If on last line of buffer, delete char backwards, else scroll down a page."
  (interactive "P")
  (if (last-line-p) (backward-delete-char-untabify (or arg 1))
    (scroll-down arg)))

(defun hwww:send-input ()
  (interactive)
  (cond ((eobp)
	 (let ((www (get-buffer-process (current-buffer))))
	   (if www
	       (progn
		 (beginning-of-line)
		 ;; Exclude the shell prompt, if any.
		 (re-search-forward shell-prompt-pattern
				    (save-excursion (end-of-line) (point))
				    t)
		 (let ((cmd (concat (buffer-substring (point)
						      (progn (forward-line 1)
							     (point)))
				    "\n")))
		   (erase-buffer)
		   (process-send-string www cmd)
		   ))
	     (error "(hwww:link-follow): No current WWW process.  Use 'hwww:start'."))))
	((ibut:at-p) (hui:hbut-act))
	(t (end-of-buffer))
	))

(defun hwww:sentinel (process signal)
  (princ
   (format "Process: %s received the msg: %s" process signal))
  (or (string-match "killed" signal)
      (pop-to-buffer (process-buffer process))))

;;; ************************************************************************
;;; Private variables
;;; ************************************************************************

(defvar hwww:mode-map nil)

(provide 'hsys-www)
