# RadioGrp.w -- RadioGroup widget
# Author: Bert Bos <bert@let.rug.nl>
# Version: 1.0 for FWF 4.0

@class XfwfRadioGroup (XfwfGroup)  @file=RadioGrp

@ The RadioGroup widget is a simple specialization of the Group
widget. It has an extra resource, |labels|, which holds the labels of
the toggle buttons that are created automatically. For the common case
that a Group widget contains only radio buttons, the RadioGroup widget
is much more convenient. It is also much less flexible. E.g., it is
very difficult to change resources of the radio buttons, when the
defaults are not satisfactory.

In particular, the Toggle widgets are created with the following
hardcoded resources: |shrinkToFit = True|, |border_width = 0|,
|frameWidth = 0|. The names of the Toggle widgets are equal to their
labels.

@public

@ The |labels| resource is an array of strings. Each string will
become the label of a toggle button. The last member of the array must
be a |NULL| pointer.

There is a converter that allows the list of strings to be entered as
a single string, with any character as separator.

	@var StringArray labels = NULL


@exports

@ The type |StringArray| represents an array of |String|s, with the
proviso that by convention the last member of a |StringArray| is
always a |NULL| pointer. There is a converter that can construct a
|StringArray| from a single string.

	@type StringArray = String *


@methods

@ |class_initialize| registers the type converter.

@proc class_initialize
{
    XtSetTypeConverter(XtRString, "StringArray", cvtStringToStringArray,
		       NULL, 0, XtCacheNone, NULL);
}

@ In the |initialize| method, the utility function |create_toggles| is
called, which creates toggle widgets for each label in the |labels|
resource.

@proc initialize
{
    $labels = newStringArray($labels);
    create_toggles($);
}

@ The |set_values| method makes a private copy of the |labels|
resource and calls |create_labels|. Since the RadioGroup doesn't draw
anything itself, the return value is always |False|.

@proc set_values
{
    if ($old$labels != $labels) {
	freeStringArray($old$labels);
	$labels = newStringArray($labels);
	create_toggles($);
    }
    return False;
}

@utilities

@ The function |create_toggles| first destroys all existing children
and then creates new ones. For each string in the |labels| resource
there will be an XfwfToggle button.

@proc create_toggles($)
{
    Cardinal i;
    StringArray s;

    for (i = 0; i < $num_children; i++) 
	XtDestroyWidget($children[i]);

    if ($labels == NULL) return;

    for (s = $labels, i = 0; s[i] != NULL; i++) {
	(void) XtVaCreateManagedWidget
	    (s[i], xfwfToggleWidgetClass, $, XtNlabel, s[i], XtNshrinkToFit,
	     True, XtNborderWidth, 0, XtNframeWidth, 0, NULL);
    }
}

@ The converter from |String| to |StringArray| makes a copy of the
passed string and then replaces all occurences of the delimiter with
a nul byte. The |StringArray| a is filled with pointers to the parts
of the string.

The delimiter character is the first character in the string.

@def done(type, value) =
    do {
	if (to->addr != NULL) {
	    if (to->size < sizeof(type)) {
	        to->size = sizeof(type);
	        return False;
	    }
	    *(type*)(to->addr) = (value);
        } else {
	    static type static_val;
	    static_val = (value);
	    to->addr = (XtPointer)&static_val;
        }
        to->size = sizeof(type);
        return True;
    } while (0)

@proc Boolean cvtStringToStringArray(Display *display, XrmValuePtr args, Cardinal *num_args, XrmValuePtr from, XrmValuePtr to, XtPointer *converter_data)
{
    String t, s;
    StringArray a = NULL;
    Cardinal i;
    char delim;

    if (*num_args != 0)
	XtAppErrorMsg(XtDisplayToApplicationContext(display),
		      "cvtStringToStringArray", "wrongParameters",
		      "XtToolkitError",
		      "String to StringArray conversion needs no arguments",
		      (String*) NULL, (Cardinal*) NULL);

    delim = ((String) from->addr)[0];
    s = XtNewString((String) from->addr + 1);
    i = 0;
    while (s && *s) {
	t = strchr(s, delim);
        if (t) *t = '\0';
	a = (StringArray) XtRealloc((String) a, (i + 1) * sizeof(*a));
	a[i] = s;
	i++;
        s = t ? t + 1 : NULL;
    }
    a = (StringArray) XtRealloc((String) a, (i + 1) * sizeof(*a));
    a[i] = NULL;
    done(StringArray, a);
}

@ The function |newStringArray| makes a copy of a |StringArray|. It
allocates new space for the array itself and for the strings that it
contains.

@proc StringArray newStringArray(StringArray a)
{
    Cardinal n, i;
    StringArray s;

    if (!a) return NULL;
    for (s = a, n = 0; s[n]; n++) ;
    s = (StringArray) XtMalloc((n + 1) * sizeof(*s));
    for (i = 0; i < n; i++) s[i] = XtNewString(a[i]);
    s[n] = NULL;
    return s;
}

@ |freeStringArray| deallocates the array and all strings it contains.

@proc freeStringArray(StringArray a)
{
    Cardinal i;

    if (!a) return;
    for (i = 0; a[i]; i++) XtFree(a[i]);
    XtFree((String) a);
}



@imports

@incl <Xfwf/Toggle.h>
