/*
FUNCTION
       <<modf>>, <<modff>>---split fractional and integer parts

INDEX
	modf
INDEX
	modff

ANSI_SYNOPSIS
	#include <math.h>
	double modf(double <[val]>, double *<[ipart]>);
        float modff(float <[val]>, float *<[ipart]>);

TRAD_SYNOPSIS
	#include <math.h>
	double modf(<[val]>, <[ipart]>)
        double <[val]>;
        double *<[ipart]>;

	float modff(<[val]>, <[ipart]>)
	float <[val]>;
        float *<[ipart]>;

DESCRIPTION
	<<modf>> splits the double <[val]> apart into an integer part
	and a fractional part, returning the fractional part and
	storing the integer part in <<*<[ipart]>>>.  No rounding
	whatsoever is done; the sum of the integer and fractional
	parts is guaranteed to be exactly  equal to <[val]>.   That
	is, if . <[realpart]> = modf(<[val]>, &<[intpart]>); then
	`<<<[realpart]>+<[intpart]>>>' is the same as <[val]>.
	<<modff>> is identical, save that it takes and returns
	<<float>> rather than <<double>> values. 

RETURNS
	The fractional part is returned.  Each result has the same
	sign as the supplied argument <[val]>.

PORTABILITY
	<<modf>> is ANSI C. <<modff>> is an extension.

QUICKREF
	modf  ansi pure 
	modff - pure

*/


#include "mathimpl.h"



TYPE_RET
_DEFUN(modf, (aval, ipart_ptr), 
       TYPE_ARG aval _AND 
       TYPE *ipart_ptr) 
{ 
  register TYPE absvalue; 
  register TYPE ipart; 
  TYPE val = aval;
  

  absvalue = val > 0 ? val : -val;

  if (absvalue >= maxpowtwo())
  { 
    /* This number is too big to have a fraction */
    *ipart_ptr = val;
    return 0.0;
  } 
  else 
  { 
    /* Add a large number, so that the fraction gets lots, then
       Then remove the large number, which will yield the number
       without the fraction.
       Make sure that we didn't get any nasty rounding 
       */
    ipart = absvalue + maxpowtwo();
    ipart -= maxpowtwo();
    if (ipart > absvalue)
     ipart -= 1.0;		
    ipart = ipart > 0 ? ipart : -ipart;
  } 

  if ((val - ipart) == 1.0)
  {
    ipart+= 1.0; 
  } 

  if (val >= 0) 
  { 
    *ipart_ptr = ipart; 
    return (val - ipart);
  } 
  else 
  { 
    *ipart_ptr = -ipart; 
    return (val + ipart); 
  } 
} 

