/* Perform queries on the contents of a GNATS database.
   Copyright (C) 1993 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <sys/types.h>
#include <getopt.h>
#include <time.h>

#include "gnats.h"
#include "pathmax.h"

/* The name this program was run with.  */
char *program_name;

/* Where the results of query-pr should go.  */
FILE *outfile = stdout;

/* Where to keep the static index if necessary.  */
Index *index_chain;

/* If specified, search for the originator of this PR.  */
char *originator = NULL;

/* Whether or not we're searching more than one (or every) PR.  */
int searching = 0;

/* If 1, print the whole PR out, not just the choice bits.  */
int full_report = 0;

/* If 1, emit info for a SQL database.  */
int sql_format = 0;

/* If 1, don't talk about closed PRs.  */
int skip_closed = 0; /* FIXME: This doesn't make sense when we have
			configurable state names.  */

/* If 1, print a string like "/gnats/GNATS/g++/145:0:" for the
   emacs next-error function.  */
int print_path = 0;

/* If 1, emit a summary of the PR.  */
int summary = 0;

/* If 1, we found a match and should output a newline.  */
int found_match = 0;

/* Defined in version.c.  */
extern char *version_string;

extern PR_entry pr[NUM_PR_ITEMS];

struct option long_options[] =
{
  {"directory", 1, NULL, 'd'},
  {"category", 1, NULL, 'c'},
  {"submitter", 1, NULL, 'S'},
  {"responsible", 1, NULL, 'r'},
  {"originator", 1, NULL, 'O'},
  {"state", 1, NULL, 's'},
  {"confidential", 1, NULL, 'C'},
  {"severity", 1, NULL, 'e'},
  {"priority", 1, NULL, 'p'},
  {"print-path", 1, NULL, 'P'},
  {"output", 1, NULL, 'o'},
  {"version", 0, NULL, 'V'},
  {"full", 0, NULL, 'F'},
  {"sql", 0, NULL, 'i'},
  {"skip-closed", 0, NULL, 'x'},
  {"help", 0, NULL, 'h'},
  {"summary", 0, NULL, 'q'},
  {NULL, 0, NULL, 0}
};

void usage (), version ();

typedef enum { Severity, Priority, State, Class } Sql_Types;
static int
sql_types (p, type)
     char *p;
     Sql_Types type;
{
  switch (type)
    {
    case Severity:
      if (*p == 'c')
	return 1;
      else if (*p == 's')
	return 2;
      else if (*p == 'n')
	return 3;
      break;
    case Priority:
      if (*p == 'h')
	return 1;
      else if (*p == 'm')
	return 2;
      else if (*p == 'l')
	return 3;
      break;
    case State:
      if (*p == 'o')
	return 1;
      else if (*p == 'a')
	return 2;
      else if (*p == 's')
	return 3;
      else if (*p == 'f')
	return 4;
      else if (*p == 'c')
	return 5;
      break;
    case Class:
      if (p[1] == 'w') /* sw-bug */
	return 1;
      else if (p[1] == 'o') /* doc-bug */
	return 2;
      else if (*p == 's') /* support */
	return 3;
      else if (*p == 'c') /* change-request */
	return 4;
      else if (*p == 'm')
	return 5;
      else if (*p == 'd') /* duplicate */
	return 6;
      break;
    }

  return 0;
}

static char *
sql_time (s)
     char *s;
{
  extern time_t get_date ();
  time_t t;
  struct tm *ar_time;
  char *buf = (char *) xmalloc (16);

  t = get_date (s, NULL);
  ar_time = localtime (&t);
  strftime (buf, 16, "%y-%m-%d %H:%M", ar_time);

  return buf;
}

char *
get_category (p)
     char *p;
{
  FILE *fp;
  char *path, *category;

  fp = open_index ();
  if (fp == (FILE *)NULL)
    return NULL;

  category = find_pr_category (fp, p);

  if (category == NULL)
    return NULL;

  close_index (fp);

  path = (char *) xmalloc (PATH_MAX);
  if (category)
    sprintf (path, "%s/%s/%s", gnats_root, category, p);
  else
    sprintf (path, "%s/%s", gnats_root, p);

  return path;
}

int
get_pr (path, pr)
     char *path;
     char *pr;
{
  FILE *fp = fopen (path, "r");
  if (fp == (FILE *)NULL)
    {
      fprintf (stderr, "%s: couldn't read PR %s\n", program_name, pr);
      return 0;
    }

  read_header (fp);
  read_pr (fp, ! full_report);

  fclose (fp);

  return 1;
}

void
print_pr (path, p, opened)
     char *path;
     char *p;
     int opened;
{
  /* Note: don't use P in any real way, cuz it could be "1234" or
     "category/1234".  */
  if (! opened && ! get_pr (path, p))
    return;

  if (full_report)
    {
      write_header (outfile, NUM_HEADER_ITEMS);
      fprintf (outfile, "\n");
      write_pr (outfile, NUM_PR_ITEMS);
    }
  else if (sql_format)
    {
      /* Trim `foo (Full Foo)' to just `foo'.  */
      char *q = (char *) strchr (pr[RESPONSIBLE].value, ' ');
      if (q != NULL)
	*q = '\0';

      fprintf (outfile, "%-8.8s|%-16.16s|%-128.128s|%-3.3s", pr[NUMBER].value,
	       pr[CATEGORY].value, pr[SYNOPSIS].value, pr[CONFIDENTIAL].value);
      fprintf (outfile, "|%1.1d|%1.1d|%-16.16s|%1.1d|%1.1d|%-16.16s",
	       sql_types (pr[SEVERITY].value, Severity),
	       sql_types (pr[PRIORITY].value, Priority), pr[RESPONSIBLE].value,
	       sql_types (pr[STATE].value, State),
	       sql_types (pr[CLASS].value, Class), pr[SUBMITTER].value);
      fprintf (outfile, "|%-16.16s|%-64.64s|%-64.64s|",
	       sql_time (pr[ARRIVAL_DATE].value),
	       pr[ORIGINATOR].value, pr[RELEASE].value);
    }
  else if (summary)
    {
      char *s;

      s = (char *) strchr (pr[RESPONSIBLE].value, ' ');
      if (s)
	*s = '\0';
      fprintf (outfile,
	       "%8s %-8.8s %-8.8s %-9.9s %-9.9s %-8.8s %-10.10s %s",
	       pr[NUMBER].value, pr[RESPONSIBLE].value,
	       pr[CATEGORY].value, pr[STATE].value, pr[SEVERITY].value,
	       pr[PRIORITY].value, pr[SUBMITTER].value,
	       pr[SYNOPSIS].value);
    }
  else
    {
      /* Print this out for emacs when people do `M-x query-pr' and want
	 to be able to use the next-error function on the buffer.  */
      if (print_path)
	fprintf (outfile, "%s:0:\n", path);

      write_pr (outfile, NUMBER);
      write_pr (outfile, CATEGORY);
      write_pr (outfile, SYNOPSIS);
      write_pr (outfile, CONFIDENTIAL);
      write_pr (outfile, SEVERITY);
      write_pr (outfile, PRIORITY);
      write_pr (outfile, RESPONSIBLE);
      write_pr (outfile, STATE);
      write_pr (outfile, CLASS);
      write_pr (outfile, SUBMITTER);
      write_pr (outfile, ORIGINATOR);
      write_pr (outfile, RELEASE);
      write_pr (outfile, ARRIVAL_DATE);
    }

  found_match = 1;
}

char *
make_path (c, n)
     char *c, *n;
{
  char *path = (char *) xmalloc (PATH_MAX);

  sprintf (path, "%s/%s/%s", gnats_root, c, n);
  return path;
}

static int
pr_matches (s, i)
     Index *s;
     Index *i;
{
  return (!s->category || (strcmp (s->category, i->category) == 0))
    && (!s->submitter || (strcmp (s->submitter, i->submitter) == 0))
    && (!s->responsible || (strcmp (s->responsible, i->responsible) == 0))
    && (!s->state || (strcmp (s->state, i->state) == 0))
    && (!s->confidential || (strcmp (s->confidential, i->confidential) == 0))
    && (!s->severity || (strcmp (s->severity, i->severity) == 0))
    && (!s->priority || (strcmp (s->priority, i->priority) == 0));
}

/* Possibly print out a PR, checking if we want to verify the originator
   first.  */
int
do_pr (i)
     Index *i;
{
  char *path;
  int pr_loaded = 0;

  path = make_path (i->category, i->number);

  if (originator)
    {
      if (! get_pr (path, i->number))
	return 0;

      if (pr[ORIGINATOR].value == NULL
	  || strncasecmp (pr[ORIGINATOR].value,
			  originator,
			  strlen (originator)) != 0)
	return 0;
      pr_loaded = 1;
    }
		    
  print_pr (path, i->number, pr_loaded);
  free (path);

  return 1;
}

int
query_pr (p, s)
     char *p;
     Index *s;
{
  char *path;
  Index *i;

  if (p == NULL)
    {
      /* We weren't given a list of PRs to check, so we do the
	 whole shooting match.  */
      for (i = index_chain; i ; i = i->next)
	if ((!searching && !s) || pr_matches (s, i))
	  {
	    /* Only skip closed PRs if we aren't searching
	       the state of a PR.  */
	    if (skip_closed && (!s || !s->state)
		&& strcmp (i->state, "closed") == 0)
	      continue;

	    if (do_pr (i))
	      fprintf (outfile, "\n");
	  }
    }
  else
    {
      if (searching)
	{
	  /* Search in the problem report P for a satisfying condition. :) */
	  for (i = index_chain; i ; i = i->next)
	    if ((strcmp (i->number, p) == 0)
		&& pr_matches (s, i))
	      {
		if (skip_closed && (!s || !s->state)
		    && strcmp (i->state, "closed") == 0)
		  continue;

		do_pr (i);
		break;
	      }
	}
      else
	{
	  /* They did a plain query-pr with a list of PRs, just print
	     this one out.  */
	  path = p;
	  if (((char *) strchr (p, '/')) == NULL)
	    path = get_category (p);
	  else
	    {
	      path = (char *) xmalloc (PATH_MAX);
	      sprintf (path, "%s/%s", gnats_root, p);
	    }

	  if (path)
	    {
	      if (get_pr (path, p)
		  && ! (skip_closed
			&& strcmp (pr[STATE].value, "closed") == 0))
		print_pr (path, p, 1);

	      free (path);
	    }
	  else
	    {
	      fprintf (stderr, "%s: couldn't find PR %s\n",
		       program_name, p);
	      return 1;
	    }
	}
    }

  return 0;
}

void
main (argc, argv)
     int argc;
     char **argv;
{
  int optc;
  Index *s = (Index *) xmalloc (sizeof (Index));
  int errors = 0;

  program_name = (char *) basename (argv[0]);

  memset (s, 0, sizeof (Index));

  while ((optc = getopt_long (argc, argv, "c:C:d:e:o:O:p:Ps:S:r:VFixhq",
			      long_options, (int *) 0)) != EOF)
    {
      switch (optc)
	{
	case 'd':
	  gnats_root = optarg;
	  break;

	case 'o':
	  outfile = fopen (optarg, "w+");
	  if (outfile == (FILE *) NULL)
	    fprintf (stderr, "can not read file %s", optarg);
	  exit (1);
	  break;

	case 'r':
	  s->responsible = optarg;
	  searching = 1;
	  break;

	case 'c':
	  s->category = optarg;
	  searching = 1;
	  break;

	case 'C':
	  s->confidential = optarg;
	  searching = 1;
	  break;

	case 'e':
	  s->severity = optarg;
	  searching = 1;
	  break;

	case 'O':
	  originator = optarg;
	  searching = 1;
	  break;

	case 'p':
	  s->priority = optarg;
	  searching = 1;
	  break;

	case 'P':
	  print_path = 1;
	  break;

	case 's':
	  s->state = optarg;
	  searching = 1;
	  break;

	case 'S':
	  s->submitter = optarg;
	  searching = 1;
	  break;

	case 'V':
	  version ();
	  break;

	case 'F':
	  full_report = 1;
	  break;

	case 'i':
	  sql_format = 1;
	  break;

	case 'x':
	  skip_closed = 1;
	  break;

	case 'q':
	  summary = 1;
	  break;

	case 'h':
	  /* Fall through.  */

	default:
	  usage ();
	}
    }

  if ((sql_format + full_report + summary) > 1)
    {
      fprintf (stderr, "%s: only one of -i, -F, or -q may be specified\n",
	       program_name);
      exit (1);
    }

  init_gnats ();

  if (searching || (optind == argc))
    index_chain = get_index ();

  if (optind == argc)
    {
      if (! searching)
	errors += query_pr ((char *) NULL, (Index *) NULL);
      else
	errors += query_pr ((char *) NULL, s);
    }
  else
    {
      while (optind < argc)
	{
	  found_match = 0;
	  errors += query_pr (argv[optind], searching ? s : (Index *) NULL);
	  if ((++optind < argc) && found_match)
	    fprintf (outfile, "\n");
	}

      if (found_match && (summary || sql_format))
	fprintf (outfile, "\n");
    }

  /* Exit non-zero if there were any troubles.  */
  exit (errors != 0);
}

void
usage ()
{
  fprintf (stderr, "\
Usage: %s [-iqxFPV] [-d directory] [-c category] [-S submitter]\n\
       [-r responsible] [-s state] [-C confidential] [-e severity]\n\
       [-p priority] [-o outfile] [-O originator] [--sql] [--full]\n\
       [--version] [--help] [--category=category] [--confidential=[yes|no]]\n\
       [--output=outfile] [--originator=name] [--print-path]\n\
       [--priority=level] [--responsible=person] [--severity=severity]\n\
       [--state=state] [--skip-closed] [--submitter=submitter]\n\
       [--summary] [PR]\n",
	   program_name);
  exit (1);
}

void
version ()
{
  printf ("query-pr %s\n", version_string);
}
