/*      if UNIX:        cc -O cpr.c
 *      if MSDOS:       cc -O -DMSDOS cpr.c
 *      if VMS:         define sys sys$library; cc cpr.c
 *
 *      This program prints the files named in its argument list, preceding
 *      the output with a table of contents. Each file is assumed to be C
 *      source code (but doesn't have to be) in that the program searches
 *      for the beginning and end of functions. Function names are added to
 *      the table of contents, provided the name starts at the beginning of
 *      a line. The function name in the output is bolded.
 *
 *      By default blank space is inserted after every closing '}'
 *      character. Thus functions and structure declarations are nicely
 *      isolated in the output. The only drawback to this is that structure
 *      initialization tables sometimes produce lots of white space.
 *      The "-r" option removes this space, or changes it to the indicated
 *      length.
 *
 *      The option "-l" indicates that the following argument is to be
 *      the page length used for output (changing the page length hasn't been
 *      tested much).
 *
 *      The option "-s" indicates that the table of contents should be sorted
 *      by function name within each file.
 *
 *      The option "-n" indicates that output lines should be numbered with
 *      the corresponding line number from the input file.
 *
 *      The option "-p" indicates what proportion of the page in steps of 16
 *      should be used for deciding if a new function needs a new page.
 *      That is -p12 (the default) indicates that if a function starts
 *      within the top 12/16 (3/4) of the page then do it, otherwise put it
 *      on a new page.  Thus the higher the number (upto 16) the closer to
 *      the bottom of the page will functions be started. -p0 says put each
 *      func on a new page.
 *
 *      Try it! You'll like it. (I call it cpr.c)
 *
 *      Written by:
 *              Paul Breslin
 *              Human Computing Resources Corp.
 *              10 St. Mary St.
 *              Toronto, Ontario
 *              Canada, M4Y 1P9
 *
 *              -- ...!decvax!utcsrgv!hcr!phb
 *
 *      Sorting and standard input reading from:
 *              Rick Wise, CALCULON Corp., Rockville, MD.
 *              -- ...!decvax!harpo!seismo!rlgvax!cvl!umcp-cs!cal-unix!wise
 *
 *      File modified time,
 *      numbered output,
 *      optional white space,
 *      improved function start tests from:
 *              David Wasley, U.C.Berkeley
 *              -- ...!ucbvax!topaz.dlw
 *      Modified the -r to leave variable amounts of space
 *              Patrick Powell, U. Waterloo
 *
 *      Changed handling of form feeds to start a new page AND print heading:
 *              Terry Doner, U of Waterloo
 *
 *      Fixed up to locate more functions, and added -p option
 *              Dennis Vadura, U of Waterloo
 *              dvadura@watdragon.waterloo.edu (Dennis Vadura)
 *
 *              It will find things like  struct foo *f()...
 *              but not things like     int
 *                                      f
 *                                      ()...
 *              ie. the constraint is that the () must appear on the same line
 *              as the function name.
 *
 *  Clean up a bit for 80286 machines (lints a bit cleaner, too)
 *      Dan Frank, Prairie Computing
 *
 *  Fixed a whole bunch of stuff and added lots of new flags.
 *      -S       sort and be case insensitive.
 *      -N       start numbering pages at 1 for each new file
 *      -T title cat the file title before the table of contents.
 *      -C       print only the table of contents
 *      -c       only try to look for function names in files whose suffix ends
 *               in .c
 *      -f file  to handle file containing list of files to print. (for MSDOS)
 *      Dennis Vadura
 *
 *  Added VMS and Language support, -h, -o options, reorganized bolding, and
 *  shortened long names (so page numbers aren't lost).  Also put in getopt().
 *  Oh yeah, expanded tabs (which wasn't what -t originally meant).
 *      -h str   String to put at the top of each page instead of file name.
 *      -o off   Number of spaces to put in front of each line of code.
 *      -a lang  Assume the following language.  Default is AUTO (use file
 *               name suffix to guess language).  NONE is allowed to mean
 *               don't look for function names at all.
 *      -cC      Dennis Vadura's -c option is thus generalized and removed.
 *               -C becomes -c (VMS requires quotes around uppercase options,
 *               so where possible use lower case...)  New -C added to override
 *               supression of table of contents for small output jobs.  (NONE
 *               means no table of content entries, hence none printed.)
 *
 *               Adding a language involves (at least):
 *               1) Add to enum langs.
 *               2) Add recognition of language keyword (FORTRAN) to getopt().
 *               3) Add reference in man page and in Usage() function.
 *               4) Add recognition to Scan() for end of functions.
 *               5) Build a LooksLikeXXXX() to call inside LooksLikeFunction().
 *               6) Add suffix recognition to WhichLanguage().
 *               Then search everywhere for Language and see if you've missed
 *               anything; if so edit this comment (smile).
 *
 *      John Campbell (...!arizona!naucse!jdc  or  CAMPBELL@NAUVAX)
 *
 *  Dennis Vadura:
 *	Added a few options, -i, to ignore form-feed chars in original
 *	source, -O to force output for two-up printing (this option
 *	affects both table of contents and the file listing output and
 *	handles the -T option correctly), added -F to print only file
 *	listings.
 *
 *	renamed John Campbell's -h str to -H str so that -h can be used as
 *	help (ie. the -? gets you in trouble in most shells and you have to
 *	escape it etc.)
 *
 *	Cleaned up printing of help message.
 *
 *	Oh, what the hell, took the a2ps.c postscript converter and added the
 *	capability to produce a postscript output file.  Seems Really nice,
 *	might need a bit of tweeking but hey, I don't have hours to spend.
 *	It's good enough for me :-)
 *
 *	To do this I hacked the code quite a bit, re-organized some functions
 *	put in new ones, and made all output go through a common set of
 *	routines.  I also changed cpr's output format to match that of the
 *	postscript code.
 *
 *	I took the a2ps postscript code and embeded it into cpr, I prefered
 *	this over keeping it as an included separate file, in an effort to
 *	keep cpr self contained.
 *
 *	Bumped the version # to 2.5.
 */
char *version = "2.5"; /* Just a guess--never had one before. */

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <ctype.h>
#include <signal.h>
#include <string.h>
#ifdef __TURBOC__
#define MSDOS 1
#endif
#ifndef VMS
extern int errno; /* system error number */
extern char *sys_errlist[]; /* error message */
#define CANT_OPEN(p1) \
   fprintf(stderr,"%s: Can't open file '%s': %s\n", \
           ProgName, p1, sys_errlist[errno] )
#else
#include <perror.h>
#define CANT_OPEN(p1) \
   if (errno == EVMSERR) {\
      fprintf (stderr, "Can't open %s\n", p1);\
      LIB$STOP (vaxc$errno);\
   }\
   fprintf(stderr,"%s: Can't open file '%s': %s\n",\
           ProgName, p1, sys_errlist[errno] )
#endif
#if MSDOS || VMS
#include "getopt.c"
#endif

extern char *malloc() ; /* important for 8086-like systems */

#define FALSE		0
#define TRUE		1

#define TOC_SIZE        4096
#define MAXLINE          256

#define NEWFILE         1
#define NEWFUNCTION     2

#define SMALLC          8   /* Too few contents for a table of contents. */
#define SMALLP         10   /* Too few pages for a table of contents. */

FILE *File, *FList = NULL;

int Braces; /* Keeps track of brace depth       */
int LineNumber; /* Count output lines           */
int PageNumber = 1; /* You figure this one out  */
int ActualPageCount = 0; /* Actual number of pages printed */
int PageLength = 66; /* -l <len> Normal paper length    */
int PagePart = 12; /* Decision on paging for new fn*/
int PageEnd; /* Accounts for space at bottom    */
int SawFunction;
int InComment;
int InString;
int ResetPage=0;
int ContentsOnly=0;
int FilesOnly=0;
int AlwaysContents=0;
int CaseInsensitive=0;
int StartOdd=0;
int IgnoreFF=0;

long FileLineNumber; /* Input file line number  */

char *TitleFile = NULL;
char *ProgName;
char Today[30];
char *Name; /* Current file name                */
char *Header = NULL;  /* User's header (-h "header") */
char *Title  = NULL;

char *FileDate; /* Last modified time of file        */
char FunctionName[MAXLINE+1];

char SortFlag; /* -s == sort table of contents  */
char NumberFlag; /* -n == output line numbers   */
int OffsetValue = 0; /* -o <number> Offset each line (of code) N spaces */
int Space_to_leave = 5; /* -r<number> space to leave    */
int TabWidth = 8; /* -t <number> width of tabs  */

/* Language types */
enum langs {NONE, AUTO, C, FORTRAN, ICON, LISP}
   Language = AUTO;
/*
   Printer types (Would you believe I have to support a printer that can't
   backspace?  Geesh!)
*/
#define DUMB       0
#define BACKSPACE  1       /* Probably most reasonable default. */
#define ANSI       2       /* Just uses ansi escape sequences to bold */
#define LN03       3       /* For now same as ANSI--later smaller print! */
#define NECP5200   4       /* NEC 24 pin printer--in HS mode. */
#define POSTSCRIPT 5	   /* print on a postscript printer */
int Printer = BACKSPACE;   /* Choose your default. */

extern char *optarg;
extern int optind, opterr;

static char *Toc[TOC_SIZE];
static int TocPages[TOC_SIZE];
static int TocCount;


#ifdef VMS
#include <errno.h>  /* Watch out for EVMSERR (special 65535 errno) */
#define unlink delete
#define toupper _toupper  /* Faster to use macro version. */
/*
   Local (NAU) support for redirection, other VMS sites can leave this out,
   but then they can't use wild cards (*.c), or redirection (>cpr.out).
   If you are a VMS site and want this contact CAMPBELL@NAUVAX.bitnet.
*/
#include "nau_utils:redexp.vms"
#endif
main(argc, argv)
char **argv;
{
   register int i;
   char *ctime();
   char *pname=NULL;
   char *s;
   time_t thetime, time();
   int c;
   enum langs start_lang, WhichLanguage();

   FileDate = (char *)malloc(100);
   ProgName = argv[0];
   thetime = time((time_t *)0);
   strcpy(Today,ctime(&thetime));
   if( (s = strchr(Today,'\n')) != NULL ) *s = '\0';

/* Parse options. */
   while ((c = getopt (argc, argv, "cCFhisSnNOa:f:H:t:T:l:o:r:p:P:")) != EOF) {
      switch (c) {
      case 'a':                    /* Assume a language ('C', FORTRAN,...) */
         c = *optarg;
         switch (c) {
         case 'C':
         case 'c':
            Language = C;
         break;
         case 'F':
         case 'f':
            Language = FORTRAN;
         break;
         case 'I':
         case 'i':
            Language = ICON;
         break;
         case 'L':
         case 'l':
            Language = LISP;
         break;
         case 'N':
         case 'n':
            Language = NONE;
         break;
         case 'A':
         case 'a':
            Language = AUTO;
         break;
         default:
            fprintf (stderr, "Unknown or unsupported language\n");
	    exit(1);
         }
         break;

      case 'F':
	 ++FilesOnly;
	 break;

      case 'O':
	 ++StartOdd;
	 break;

      case 'i':
	 ++IgnoreFF;
	 break;

      case 'f':
         if (*optarg == '-') {
            FList = stdin;
         }
         else if ((FList = fopen (optarg, "r")) == NULL) {
            fprintf (stderr, "Can't open file names list %s\n", optarg);
	    exit(1);
         }
         break;
      case 'H':       /* User's header */
         Header = optarg;
         break;

      case 't':
         TabWidth = atoi(optarg);
         if( TabWidth < 0 )
            TabWidth = 0;
         break;

      case 'T':
         TitleFile = optarg;
         break;

      case 'l':
         PageLength = atoi(optarg);
         if( PageLength < 10) PageLength = 10;
         break;

      case 'S':
         ++CaseInsensitive;
      case 's':
         ++SortFlag;
         break;

      case 'C':
         ++AlwaysContents;
         break;

      case 'c':
         ++ContentsOnly;
         break;

      case 'n':
         ++NumberFlag;
         break;

      case 'N':
         ++ResetPage;
         break;

      case 'o':            /* Offset code by <number> */
         OffsetValue = atoi(optarg);
         if (OffsetValue <= 0) {
            fprintf (stderr, "Offset must be a positive integer only\n");
	    exit(1);
         }
         if (OffsetValue > 32) {
            fprintf (stderr, "Offset must be less than 32\n");
	    exit(1);
         }
         break;

      case 'r':
      /* It's ok to have a '0' from the "?" here... */
         Space_to_leave = atoi(optarg);
         break;

      case 'P':
         pname = optarg;  /* Override printer default or CPRINTER environ */
         break;

      case 'p':
         PagePart = atoi(optarg);
         PagePart = PagePart <= 16 ? PagePart : 16;
         break;

      case 'h':
      default:
         Usage();
         break;
      }
   }

   start_lang = Language;

   Init (pname);
   StartTempFile();

   i = optind;

   if( FList == NULL && i == argc )
   { /* no file names */
      File = stdin;
      Name = "Standard Input";
      if (start_lang == AUTO) Language = C;
      List();
   }

   if( FList != NULL)
   {
      char b[1024];

      while( fgets(b, 1024, FList) != NULL )
      {
         if( strlen(b) ) b[strlen(b)-1]=0;

         if( strcmp(b, "-") != 0 )
         {
            if( (File = fopen( Name = b, "r" )) == NULL )
            {
               CANT_OPEN (Name);
               continue;
            }
            if (start_lang == AUTO) {
               Language = WhichLanguage (Name);
            }
         }
         else {
            Name = "Standard Input";
            if (start_lang == AUTO) Language = C;
            File = stdin;
         }

         List();
         if( File != stdin ) fclose(File);
      }
   }
   for(; i < argc; ++i )
   {
      if( strcmp(argv[i], "-") == 0 )
      {
         File = stdin;
         Name = "Standard Input";
         if (start_lang == AUTO) Language = C;
         List();
      }
      else {
         if( (File = fopen( Name = argv[i], "r" )) == NULL )
         {
            CANT_OPEN (Name);
            continue;
         }
         if (start_lang == AUTO) {
            Language = WhichLanguage (Name);
         }
         List();
         if( File != stdin ) fclose(File);
      }
   }

   if( PageNumber > 1 || LineNumber > 0 ) BlankPage();
   if( StartOdd && ((ActualPageCount % 2) != 0) ) BlankPage();

   Fini();
   EndTempFile();

   if( Printer == POSTSCRIPT ) DumpPostscriptHeader();
   DumpTableOfContents();
   DumpTempFiles();
   Done();
}

Usage()
{
   char buf[132];
   char *p;
   sprintf( buf, "Usage: %s ", ProgName );

   printf("%s[-CcFhiNnOsS] [-a language] [-H header] [-l pagelen]\n", buf);
   for(p=buf; *p; *p++ = ' ' );
   printf("%s[-o offset] [-P printer] [-p[num]] [-r[num]] [-T title]\n", buf);
   printf("%s[-t tabwidth] [[-f flist] | file...]\n\n", buf );

   puts("OPTIONS: (first group can be combined)");
   puts("   -C   - Force output of table of contents" );
   puts("   -c   - Print only table of contents (implies -C, overrides -F)" );
   puts("   -F   - Print only file contents" );
   puts("   -h   - Provide help message (you're reading it)" );
   puts("   -i   - Ignore form-feeds in original source");
   puts("   -N   - Number pages of each file starting at page #1" );
   puts("   -n   - Number output lines of each file" );
   puts("   -O   - Force new files to start at an odd numbered actual page" );
   puts("   -s   - Sort TOC by function name within each file" );
   puts("   -S   - Same sort as -s, but ignore case\n" );

   puts("   -a language - select language class" );
   puts("   -H header   - specify personal heading" );
   puts("   -l pagelen  - define new pagelength" );
   puts("   -o offset   - set offset from left of page" );
   puts("   -P print    - select printer class" );
   puts("   -p[num]     - control function placement on page" );
   puts("   -r[num]     - control function spacing on page" );
   puts("   -T title    - print a title from file 'title'" );
   puts("   -t tabwidth - set tabwidth" );
   puts("   -f flist    - read 'flist' for list of files to print\n" );

   puts("Language choices are:  C, FORTRAN, ICON, and LISP" );
   puts("Supported printers:    DUMB, BACKSPACE, ANSI, LN03, NECP5200, and POSTSCRIPT");
   exit(1);
}

int SaveOut;
char *TempName;
char *Temp2Name;

StartTempFile()
{
   int Done();
   extern char *mktemp();

   CatchSignalsPlease(Done);

   SaveOut = dup(1);
#if MSDOS | VMS
   TempName = "cpr0001.tmp";
#else
   TempName = mktemp("/tmp/cprXXXXXX");
#endif
   if( freopen(TempName, "w", stdout) == NULL )
   {
      CANT_OPEN (TempName);
      exit(1);
   }
}

EndTempFile()
{
#if MSDOS | VMS
   Temp2Name = "cpr0002.tmp";
#else
   Temp2Name = mktemp("/tmp/cprXXXXXX");
#endif
   if( freopen(Temp2Name, "w", stdout) == NULL )
   {
      CANT_OPEN (Temp2Name);
      exit(1);
   }
}

DumpTempFiles()
{
#if (MSDOS | VMS)
   FILE *f;
   char b[256];
#endif
   register int pid, w;

   fclose(stdout);

#if !(MSDOS || VMS)
   dup(SaveOut);
   while( (pid = fork()) < 0 ) sleep(1);
   if( pid )
      while ((w = wait((int *)0)) != pid && w != -1);
   else
      {
      CatchSignalsPlease(SIG_DFL);

      if( ContentsOnly )
         execl( "/bin/cat", "cat", Temp2Name, (char *)0 );
      else {
      /*
         Ok, use a heuristic to see if it is worth putting out the
         table of contents.  They seem useless and annoying when cpr
         is just printing a small job.  Heuristic: more than 10 entries
         in the table of contents AND more than 8 pages of output.
      */
         if( !FilesOnly &&  (AlwaysContents ||
                      (TocCount > SMALLC && TocPages[TocCount-1] > SMALLP)))
	       execl( "/bin/cat", "cat", Temp2Name, TempName, (char *)0 );
         else
            execl( "/bin/cat", "cat", TempName, (char *)0 );
      }
      fprintf(stderr, "%s: exec of /bin/cat failed: %s\n", ProgName,
      sys_errlist[errno]);
      exit(1);
   }
#else
   CatchSignalsPlease(SIG_DFL);
   /*
      Use a heuristic to see if it is worth putting out the table of contents.
      They seem useless and annoying when cpr is printing a small job.
      Heuristic: > 10 entries in the table of contents or > 8 pages of output.
   */
   if( !FilesOnly && (AlwaysContents || (TocCount > SMALLC &&
 TocPages[TocCount-1] > SMALLP))) {
      if( (f=fopen(Temp2Name,"r")) == NULL ) {
         CANT_OPEN (Temp2Name);
      }
      else
      {
         while( fgets(b, MAXLINE, f) != NULL )
            write(SaveOut,b,strlen(b));

         fclose(f);
      }
   }
   if( !ContentsOnly )
      if( (f=fopen(TempName,"r")) == NULL ) {
      CANT_OPEN (TempName);
      }
      else
      {
         while( fgets(b, MAXLINE, f) != NULL )
            write(SaveOut,b,strlen(b));

         fclose(f);
      }
#endif
}

Done()
{
   CatchSignalsPlease(SIG_DFL);

   fclose( stdout );
   if( TempName ) unlink( TempName );
   if( Temp2Name ) unlink( Temp2Name );

   exit(0);
}

CatchSignalsPlease(action)
#ifdef __TURBOC__
void (*action)();
#else
int (*action)();
#endif
{
   if( signal(SIGINT, SIG_IGN) != SIG_IGN ) signal(SIGINT, action);
#ifndef MSDOS
   if( signal(SIGQUIT, SIG_IGN) != SIG_IGN ) signal(SIGQUIT, action);
   if( signal(SIGHUP, SIG_IGN) != SIG_IGN ) signal(SIGHUP, action);
#endif
}

List()
{
   register int bp;
   register char *bufp;
   char buffer[MAXLINE];

   NewFile(0);
   bp = Braces = 0;
   InString = InComment = 0; /* reset for new file -DV */

   if (Language == C)
      SawFunction = 0;
   else if (Language == FORTRAN)
      SawFunction = 1;  /* Put space after main program END card */
   else if (Language == NONE)
      Braces = 1;       /* Don't even call LooksLikeFunction */

   bufp = buffer;
   while( fgets(bufp, MAXLINE, File) != NULL )
   {
      ++FileLineNumber;
      if( bp ) NewFunction();

      if( ++LineNumber >= PageEnd ) NewPage();

      if( bufp[0] == '\f'
         && bufp[1] == '\n'
         && bufp[2] == '\0' )
      {
	 if( !IgnoreFF )
	    NewPage(); /* was strcpy(bufp, "^L\n");*/
         continue;
      }

      if( (Braces == 0) && LooksLikeFunction(bufp) )
         AddToTableOfContents(NEWFUNCTION);
      else
         PutString(buffer, -1, NumberFlag, 1);

      if (Language != NONE)
         bp = Scan(buffer);
   }
}

Scan(l)
register char *l;
{
   extern char *EndComment();
   extern char *EndString();
   register char c;
   int bp, offset;
   char *save;

   bp = 0;
   switch (Language) {
   case C:
      for( save = l; c = *l; ++l )
         if( InComment )
            l = EndComment(l);
         else if( InString )
            l = EndString(l);
         else
            switch(c)
            {
            case '{':
               ++Braces;
               break;

            case '}':
               if( --Braces == 0 )
                     bp = 1;
               break;

            case '\'':
               for( ++l; *l && *l != '\''; ++l )
                  if( *l == '\\' && *(l+1) ) ++l;
               break;

            case '"':
               InString = 1;
               break;

            case '/':
               if( *(l+1) == '*' )
               {
                  InComment = 1;
                  ++l;
               }
               break;
            }
   break;
   case FORTRAN:
   /* Just check for an END card to indicate the routine is over... */
      save = l;
      while (isdigit(*l)) ++l;  /* Skip over any line statements. */
      while (*l && (*l == ' ' || *l == '\t')) ++l;  /* Get to non-blank */
   /* Icon really should only compare against lower case here. */
      if (save != l && (offset = Compare (l, "END"))) {
      /*
         Check to make sure it isn't an ENDIF or some other variable name
         (Unfortunately, build in a VMS ! comment checker here...)
      */
         l += offset;
         if (*l == '\n' || *l == '!') {
            Braces = 0;
            bp = 1;
         }
      }
   break;
   case ICON:
   /* Just check for an end keyword to indicate the routine is over... */
      save = l;
      while (*l && (*l == ' ' || *l == '\t')) ++l;  /* Get to non-blank */
      if (l[0] == 'e' && l[1] == 'n' && l[2] == 'd') {
      /*
         Check to make sure it isn't an ENDIF or some other variable name
      */
         l += 3;
         if (*l == '\n' || *l == '#') {
            Braces = 0;
            bp = 1;
         }
      }
    break;
    case LISP:
    /*
       Lisp requires the context to be preserved, hence Braces stays 0 (a
       lie) and LooksLikeLisp() handles everything.
    */
    break;
   }
   return(bp);
}

PutString (str, len, line, newline)
char *str;
int len;
int line;
int newline;
{
/*
   All lines of program text that are written through here are checked for
   tabs.  This should include all parts of each line of code (to keep the
   internal column counter straight).
*/
   static char *lbuf = (char *)0;
   static int   lbuflen = 0;
   static char spaces[] = {"                                 "};
   static int col = 0;
   static int front=1;
   int i, need;
   register char *d, *s, c;

   if (len == -1) len = strlen (str);

   need =(len+OffsetValue+20)<<1;  /* mult by 2, so that \ work for postcript
 */
   if( need > lbuflen ) {
      if( lbuf ) free(lbuf);
      lbuflen = need;
      lbuf = (char *)malloc(lbuflen);
      if( ! lbuf ) {
	 fprintf( stderr, "MALLOC error, insufficient memory\n" );
	 exit(1);
      }
   }
   *lbuf = '\0';

   if( OffsetValue && front ) sprintf (lbuf, "%.*s", OffsetValue, spaces);
   if( line  && front ) {
      sprintf (lbuf, "%6d  ", FileLineNumber);
   }
   front = 0;
   col = strlen(lbuf);
   d = lbuf+col;

   s = str;
   for (i=0; i < len; ++i, ++s) {
      c = *s;
      switch (c) {
      case '\t':
         do {
	    *d++ = ' ';
         } while (col++ % TabWidth != (TabWidth - 1));
      break;
      case '\b':
	 *d++ = c;
         if (col > 0) --col;
      break;
      case '\n':
         col = 0;
	 if( s[1] ) *d++ = '\n';
	 break;
      case '\f':
         col = 0;
	 *d++ = c;
      break;

      case '(':
      case ')':
	 if( Printer == POSTSCRIPT ) {
	    *d++ = '\\';
	    *d++ = c;
	    break;
	 }
	 /*FALLTHROUGH*/

      default:
	 *d++ = c;
         ++col;
      }
   }
   *d++ = '\0';

   switch(Printer) {
      case POSTSCRIPT:
         printf( "(%s) %s\n", lbuf, newline ? "s":"sn" );
	 break;

      default:
	 printf( "%s", lbuf );
	 if(newline) putchar('\n');
	 break;
   }
   if( newline ) front = 1;
}

PutBold (bstr, len, flag)
char *bstr;
int len, flag;
/*-
   Put ``str'' bolded on stdout (possibly using ^H's).  (This test a flag
   to support other types of bolding options.)  Assume ``str'' is ``len''
   characters long, unless ``len'' == -1, then use strlen for the length
   of ``str''.  If ``flag'' is 0 then print here without calling PutString
   (in other words, don't pass it on as part of a code line for tab
   column counting/expansion).

   When working with escape sequences, don't call PutString to print out the
   escape sequences themselves.  Otherwise the tab expansion will be off.
   (PutString adjusts column counter for \b's, but it doesn't need to know
   about escape sequences.)
*/
{
   int str_flag = 0;
   char *str;

   if( len == -1 ) {
      len = strlen(bstr);
      str = bstr;
   }
   else {
      str_flag = 1;
      str = malloc(len+2);
      strncpy(str, bstr, len);
   }

   switch (Printer) {
   case DUMB:
      PutString (str, len, NumberFlag, 0);  /* Just give up--printer is too
 dumb */
   break;
   case ANSI:
   case LN03:
      printf ("\033[1m");
      PutString (str, len, NumberFlag, 0);
      printf ("\033[22m");
   break;
   case NECP5200:
      printf ("\033E");
      PutString (str, len, NumberFlag, 0);
      printf ("\033F");
   break;
   case POSTSCRIPT:
      printf( "(%s) sb\n", str );
      break;

   case BACKSPACE:
   {
      int j, k;
      char *buf, *s;

      s = buf = (char *)malloc(len*3+10);
      for( k=0; *str; ) {
	 *s++ = *str;
	 *s++ = '\b';
	 *s++ = *str++;
	 k += 3;
      }
      *s++ = '\0';
      /*
      strcpy( s, str );
      s += len;
      for(k=0; k<len; k++ ) *s++='\b';
      strcpy( s, str );
      */

      if (flag)
         PutString (buf, k, NumberFlag, 0);
      else
         printf(buf);
      free(buf);
   }
   break;

   default:
      fprintf (stderr, "Unknown printer type in PutBold\n");
      exit (1);
   }
   if( str_flag ) free(str);
}

char *
EndComment(p)
register char *p;
{
   register char c;

   /*
         * Always return pointer to last non-null char looked at.
         */
   while( c = *p++ )
      if( c == '*' && *p == '/' )
      {
         InComment = 0;
         return(p);
      }
   return(p-2);
}

char *
EndString(p)
register char *p;
{
   register char c;

   /*
         * Always return pointer to last non-null char looked at.
         */
   while( c = *p++ )
      if( c == '\\' && *p )
      {
         ++p;
         continue;
      }
      else if( c == '"' )
      {
         InString = 0;
         return(p-1);
      }
   return(p-2);
}

NewFunction()
{
   register int i;

   if( Space_to_leave <= 0 || !SawFunction ) return;
   if( LineNumber + Space_to_leave > (PageLength * PagePart /16) )
      NewPage();
   else {
      for( i=0; i < (Space_to_leave); ++i ) PutString("", -1, 0, 1);
      LineNumber += Space_to_leave;
   }

   SawFunction = 0;
}

#define HEADER_SIZE 3

NewPage()
{
   ActualPageCount++;
   ++PageNumber;

   switch( Printer ) {
      case POSTSCRIPT:
	 puts( "endpage" );
	 PutHeader();
	 puts( "startpage" );
	 break;

      default:
	 putchar('\f');
	 PutHeader();
	 break;
   }
   LineNumber = 0;
}


BlankPage()
{
   ActualPageCount++;
   switch( Printer ) {
      case POSTSCRIPT:
	 break;

      default:
	 if( LineNumber == 0 ) printf( "   " );
	 putchar('\f');
	 break;
   }
   LineNumber = 0;
}


PutHeader()
{
   register int l, j;
   char *lname=Title;

   switch( Printer ) {
      case POSTSCRIPT:
	 printf( "/pagenum %d def\n", PageNumber );
	 break;

      default:
	 if( lname == NULL ) break;
	 putchar('\n');
	 ++LineNumber;

	 l = strlen(lname);
	 printf( "%s", FileDate );
	 j = strlen(FileDate);
	 j = GoToColumn(j, 40-(l/2) );

	 PutBold(lname, -1, 0);
	 GoToColumn(j+l, 70);
	 printf("Page:%4d\n\n", PageNumber);
	 ++LineNumber;
	 ++LineNumber;
	 break;
   }
}

GoToColumn(from, to)
register int from, to;
{
   if( from < to)
   {
      for( ; from < to; from++ )
         putchar(' ');
   }
   return( to );
}

#define isidchr(c)      (isalnum(c) || (c == '_'))

/* This used to incorrectly identify a declaration such as
 *     int (*name[])() = { initializers ... }
 * as a function.  It also picked up this in an assembler file:
 *     #define MACRO(x) stuff
 *     MACRO(x):
 * Fixed both of these.   -IAN!
 */
LooksLikeFunction(s)
register char *s;
{
   register char *p;
   register int i;
   char *save;

/* By default this routine does 'C', other languages, other routines. */
   switch (Language) {
      case FORTRAN:    return LooksLikeFortran(s);
      case ICON:       return LooksLikeIcon(s);
      case LISP:       return LooksLikeLisp(s);
   }

   if( InComment || InString ) return(0);

   save = s;

   i = 0;
   do
      {
      p = FunctionName;

      while( *s && (*s == ' ') || (*s == '\t') ) ++s;
      if( *s == '*' ) ++s;
      if( *s && (*s == ' ') || (*s == '\t') ) continue;
      if( !*s || ((*s != '_') && !isalpha(*s)) ) return(0);

   /* Store this guess into FunctionName. */
      while( isidchr(*s) )
         *p++ = *s++;
      *p = '\0';

      while( *s && (*s == ' ') || (*s == '\t') ) ++s;
      i++;
   }
   while ( *s && *s != '(' && i < 4 );

   if( *s != '(' || *(s+1) == '*' ) return(0);

   for (i = 0; *s; s++)
   {
      switch( *s )
      {
      case '(':
         ++i;
         continue;

      case ')':
         --i;
         break;

      default:
         break;
      }
      if( i == 0 ) break;
   }
   if( !*s ) return(0);

   while( *s )
   {
      if( *s == '{') break;
      if( *s == ';' || *s == ':' ) return(0);
      ++s;
   }
   /* Don't match macro definitions (terminated with \) */
   if (s > save + 3 && (*(s-2) == '\\' ||
                 (*s == '{' && save[strlen(save)-2] == '\\')))
       return 0;
   /*
      This finds the function name (again) and causes it to be bolded.
      Note that this assumes the name and opening parentheses are on the
      same line...  (Note also that FunctionName and what we find here are
      done by two different mechanisms, why?  JDC)
   */
   if (p = strchr( save, '(' ) ) {
      p--;
      while (p != save && (*p == ' ' || *p == '\t')) --p;
      for (i=1; p != save && isidchr(*p); i++) p--;

   /* p now points to a string that is 'i' long that needs to be bolded. */
      if (p > save)
         PutString (save, p-save, NumberFlag, 0);
      PutBold (p, i, 1);
      p += i;
      PutString (p, -1, 0, 1);
   }
   else
      PutString (save, -1, NumberFlag, 1);

   SawFunction = 1;
   return(1);
}


int LooksLikeFortran(str)
char *str;
{
   register char *p, *s=str;
   register int i;
   char *save;
   int  offset = 0;

/* Ignore comment or labeled lines (and be stupid about strings) */
   if (*s != ' ' && *s != '\t') return(0);

   save = s;

   i = 0;
/*
      Fortran is easier than 'C', just check if the first word is
      SUBROUTINE, FUNCTION or PROGRAM (for those who use this on some systems)
*/
   while (*s == ' ' || *s == '\t') ++s;  /* Skip white space */

/* SUBROUTINE, FUNCTION, or PROGRAM lines must start with S, F, or P */

   switch (*s) {
   case 's':
   case 'S':
      offset = Compare (s, "SUBROUTINE");
   break;
   case 'F':
   case 'f':
      offset = Compare (s, "FUNCTION");
   break;
   case 'P':
   case 'p':
      offset = Compare (s, "PROGRAM");
   break;
   }
   if (offset == 0) {
   /*
      Still a chance that this line is INTEGER FUNCTION or some such.
      (Second Word is "FUNCTION", No legal keyword or type contains
      an "F", thank goodness.)
   */
      while (*s && *s != 'F' && *s != 'f' && *s != '"') ++s;
      if (!*s)
         return 0;
      offset = Compare (s, "FUNCTION");
      if (!offset)
         return 0;
   }
   p = s + offset;

   /* p now points to a string that is the subroutine or function name. */
   for (i=0; isidchr(p[i]); ++i)
      FunctionName[i] = p[i];
   FunctionName[i] = '\0';

   /* i is the length of the function name that needs to be bolded. */
   if (p > save)
      PutString (save, p-save, NumberFlag, 0);
   PutBold (p, i, 1);
   PutString (p+i, -1, NumberFlag, 1);

   SawFunction = 1;
   Braces = 1;  /* Indicate we are in the scope of a routine (inside). */
   return(1);
}


int LooksLikeIcon(str)
char *str;
{
   register char *p, *s=str;
   register int i;
   char *save;

   save = s;
   i    = 0;

/* For Icon, just check if the first word is ``procedure'' */
   while (*s == ' ' || *s == '\t') ++s;  /* Skip white space */

   if (s[0] != 'p' || strncmp (s, "procedure", 9) != 0)
      return 0;

   p = s + 9;  /* Point to character after ``procedure'' */
   while (*p == ' ' || *p == '\t') ++p;  /* Skip white space */

/* p now points to a string that is the procedure name. */
   for (i=0; isidchr(p[i]); ++i)
      FunctionName[i] = p[i];
   FunctionName[i] = '\0';

/* i is the length of the procedure name that needs to be bolded. */
   if (p > save)
      PutString (save, p-save, NumberFlag, 0);
   PutBold (p, i, 1);
   PutString (p+i, -1, NumberFlag, 1);

   SawFunction = 1;
   Braces = 1;  /* Indicate we are in the scope of a routine (inside). */
   return(1);
}

int LooksLikeLisp(str)
char *str;
{
#define MAXLDEPTH 32
   register char *p, *s = str;
   register int i;
   char *fend;
   static int plevel = 0, fcount = -1, flevel[MAXLDEPTH], endseen = 0;
   int  offset = 0, fseen = 0;
/*
   Lisp is a little harder than some others (perhaps indicating we should
   rework the whole program...)  In order to handle nested definitions and
   definitions appearing on the same line we need to restrict scanning of
   input (in order to preserve state information) to one routine.  Hence,
   Scan() is not used for Lisp and this routine needs to remember that
   return (1) implies line was printed and AddToTableOfContents() will
   be called while return (0) implies line will be printed.

   We also have to call NewFunction ourselves (and after the line closing
   the function definition has been printed).
*/

   if (endseen) {
      NewFunction ();
      endseen = 0;
   }

   for (fend = s; *s != '\n'; ++s) {
      switch (*s) {
      case ';':    /* Rest of line is a comment. */
         goto comment_seen;  /* So I used a goto, shoot me. */
      case '/':    /* Maybe... */
         if (s[1] == '/') goto comment_seen;
      case '(':
      case '[':
         ++plevel;
      break;
      case ')':
         if (fcount >= 0 && plevel == flevel[fcount]) {
            if (--fcount < 0) fcount = -1;
            endseen = 1;
         }
         --plevel;
      break;
      case ']':
         if (fcount >= 0) {
            if (--fcount < 0)  {
               fcount = -1;
               plevel = 0;
            }
            else
               plevel = flevel[fcount];
         }
         endseen = 1;
      break;
      case 'D':
      case 'd':
         if (offset = Compare (s, "DEFUN")) {
            if (fseen) {
            /* Take care of the other function we found on this line. */
               AddToTableOfContents(NEWFUNCTION);
            }
            p = s + offset;
         /*
            Be pretty lax about lisp identifiers (as opposed to 'C').
            This doesn't look safe, but \n is a space character.
         */
            for (i = 0; !isspace(p[i]) && p[i] != '('; ++i)
               FunctionName[i] = p[i];
            FunctionName[i] = '\0';
            if (p > fend)
               PutString (fend, p-fend, NumberFlag, 0);  /* Front part. */
            PutBold (p, i, 1);
            fend = p+i;
            if (++fcount > MAXLDEPTH) {
               fprintf (stderr, "Function defuns nested too deep\n");
               --fcount;
            }
            flevel[fcount] = plevel;
            fseen = 1;
         }
      break;
      }  /* switch */
   }  /* while */

comment_seen:
   if (fseen) {
      PutString (fend, -1, NumberFlag, 1);  /* End of string part. */
      SawFunction = 1;
      return 1;
   }
   return 0;
}


static int Compare (str, key)
char *str, *key;
/* Return the start (in s) of the next word (0 relative offest) if the
   key matches.  ``key'' must be in upper case. Otherwise return 0.
*/
{
   register char *s=str, *k=key;
   int c;

   do {
      ++s, ++k;
      c = *s;
      if (islower(c)) c = toupper (c);
   } while (c == *k);

   if (*k != '\0') return 0;
/*
   Now skip to the start of the next word;
*/
   while (*s && (*s == ' ' || *s == '\t')) ++s;

   if (*s == '\0') return 0;

   return s - str;  /* Offset of next word (to be bolded) */
}

AddToTableOfContents(type)
{
   if( TocCount > TOC_SIZE )
      return;
   if( TocCount == TOC_SIZE )
   {
      fprintf(stderr, "%s: More than %d Table of contents entries; others ignored.\n",
      ProgName, TOC_SIZE);
      ++TocCount;
      return;
   }

   if( type == NEWFILE )
      AddFile();
   else
      AddFunction();
}

AddFunction()
{
   register int l;
   register char *p;

   /* This heuristic stops multiple occurrences of a function,
         * selected by #ifdefs, to all end up many times over in the
         * Table of Contents.  One only needs to see it once.  -IAN!
         */
   if( TocCount > 0 && TocPages[TocCount-1] == PageNumber
      && strcmp(Toc[TocCount-1],FunctionName) == 0 )
      return;
   l = strlen(FunctionName);
   if( l >40 ) l = 40;
   p = Toc[TocCount] = (char *)malloc(l+1);
   strncpy(p, FunctionName, l);
   p[l] = '\0';
   TocPages[TocCount] = PageNumber;
   ++TocCount;
}

AddFile()
{
   register int l;
   register int len;
   char temp[20];
#define MAXFNLEN 59+17  /* Room for 3 dots (...) */

   len = strlen(Name) + 20;
   Toc[TocCount] = (char *)malloc(130);
   if (len > MAXFNLEN)
      sprintf(Toc[TocCount], "\n    File: ...%s ", &Name[len-MAXFNLEN-16]);
   else
      sprintf(Toc[TocCount], "\n    File: %s ", Name);
   l = strlen(Toc[TocCount]);
   if( l < 64 )
   {
      while( l < 64 )
         Toc[TocCount][l++] = ' ';

      Toc[TocCount][l++] = '\0';
   }
   sprintf(temp, "  Page %4d\n", PageNumber);
   strcat(Toc[TocCount], temp);
   ++TocCount;
}

#define MAXNAME	30
NewFile( reset )
int reset;
{
   static first = 1;
   static char buf[45];

   if( reset )
      first = 1;
   else
      GetFileTime();

   if( strlen(Name) > MAXNAME ) {
      strcpy( buf, "... " );
      strcat( buf, Name+strlen(Name)-MAXNAME-4 );
      Title = buf;
   }
   else
      Title = Name;

   if( ResetPage ) PageNumber=1;

   switch( Printer ) {
      case POSTSCRIPT:
	 if( !first && !ResetPage ) PageNumber++;
	 if( !first ) puts( "endpage" );
	 printf( "(%s) setdate\n", FileDate );
	 printf( "(%s) newfile\n", Title );
	 puts( "/sheet 1 def" );
	 printf( "/pagenum %d def\n", PageNumber );
	 puts( "startpage" );
	 break;

      default:
	 if( first )
	    PutHeader();
	 else {
	    BlankPage();
	    if( StartOdd && (ActualPageCount % 2) != 0 ) BlankPage();
	    if( !ResetPage ) PageNumber++;
	    PutHeader();
	 }

	 break;
   }

   first = 0;
   AddToTableOfContents(NEWFILE);
   FileLineNumber = 0;
}


GetFileTime()
{
   struct stat st;
   extern char *ctime();

   if( File == NULL ) return;

   if( File == stdin )
      strncpy(FileDate, &Today[4], 20);
   else
      {
      fstat(fileno(File), &st);
      strncpy(FileDate, ctime((time_t *)&st.st_mtime) + 4, 20);
   }
   strncpy(&FileDate[12], &FileDate[15], 5);
   FileDate[17] = '\0';
}

DumpTableOfContents()
{
   register int i, j;
   int index[TOC_SIZE];
   char buf[200];

   if( TocCount == 0 ) return;

   for (i = 0; i < TocCount; i++) index[i] = i;
   if( SortFlag )
      SortTable(index);

   File = NULL;
   Name = "Table of Contents";
   FileDate = Today;
   ActualPageCount = 0;
   PageNumber = 1;
   LineNumber = 0;
   NewFile(1);

   if( TitleFile != NULL )
   {
      FILE *f;
      char b[MAXLINE];

      if( (f=fopen(TitleFile,"r")) == NULL ) {
         CANT_OPEN (TitleFile);
      }
      else
      {
         while( fgets(b, MAXLINE, f) != NULL )
         {
            if( strlen(b) ) b[strlen(b)-1]=0;
            PutString(b, -1, 0, 1);
            LineNumber++;
            if( ++LineNumber >= PageEnd ) NewPage();
         }

         fclose(f);
      }
      NewPage();
   }

   TocCount--; /* Bumped by NewFile for TOC */
   for( i=0; i < TocCount; ++i ) {
      char *s;

      if( Toc[index[i]][0] == '\n' ) {
         if( (LineNumber + 5) >= PageEnd ) NewPage();

	 PutString("", -1, 0, 1);
	 PutString(Toc[index[i]]+1, -1, 0, 1);
         LineNumber += 2;
         continue;
      }
      ++LineNumber;
      if( LineNumber >= PageEnd && (i+1) != TocCount ) NewPage();

      sprintf(buf, "        %s ", Toc[index[i]]);
      s = buf+strlen(buf);
      for( j=strlen(Toc[index[i]]); j < 48; ++j ) *s++ = '.';
      *s++ = '\0';
      sprintf( buf, "%s %4d", buf, TocPages[index[i]]);
      PutString(buf, -1, 0, 1);
   }

   if( Printer == POSTSCRIPT )
      puts( "endpage" );
   else {
      BlankPage();
      if( StartOdd && ((ActualPageCount % 2) != 0) ) BlankPage();
   }
}

SortTable(index)
register int *index;
{
   register int i, temp, flag;
   char name1[MAXLINE];
   char name2[MAXLINE];

   do {
      flag = 0;
      for (i = 0; i < TocCount - 1; i++)
      {
         if( Toc[index[i]][0] == '\n' || Toc[index[i+1]][0] == '\n' )
            continue; /* don't sort across file names */
         strcpy( name1, Toc[index[i]] );
         strcpy( name2, Toc[index[i+1]] );

         if( CaseInsensitive )
         {
            char *p;
            char c;
            for(p = name1; c = *p; p++ )
               if( islower(c) ) *p=toupper(c);
            for(p = name2; c = *p; p++ )
               if( islower(c) ) *p=toupper(c);
         }

         if( strcmp(name1, name2) > 0)
         {
            temp = index[i];
            index[i] = index[i+1];
            index[i+1] = temp;
            flag = 1;
         }
      }
   }
   while( flag );
}

enum langs WhichLanguage (file_name)
char *file_name;
/*
   Routine to return one of the enum languages based on the suffix of
   the file name.
*/
{
   char *end;

   if (end = strrchr (file_name, '.')) {
      ++end;  /* Move past the '.' */
      switch (*end) {
      case 'C':    /* Allow .c, .C as suffixes (and ;231 for VMS, etc.) */
      case 'c':
      case 'Y':    /* And treat YACC as 'C' */
      case 'y':
         if (!isalpha(end[1]))
            return C;
      break;
      case 'F':     /* Allow .f, .F, .FOR, .for as suffixes */
      case 'f':
         if (!isalpha(end[1]))
            return FORTRAN;

         if ((end[1] == 'o' || end[1] == 'O') &&
             (end[2] == 'r' || end[2] == 'R') && !isalpha(end[3]))
            return FORTRAN;
      break;
      case 'I':   /* Allow .icn or .ICN (or even .iCn) as suffixes */
      case 'i':
         if ((end[1] == 'c' || end[1] == 'C') &&
             (end[2] == 'n' || end[2] == 'N') && !isalpha(end[3]))
            return ICON;
      break;
      case 'L':   /* Allow both .lsp and .l */
      case 'l':
         if ((end[1] == 's' || end[1] == 'S') &&
             (end[2] == 'p' || end[2] == 'P') && !isalpha(end[3]))
            return LISP;
         if (!isalpha(end[1])) {
         /*
            Ok, we have .l for lisp or .l for lex (sigh).  To determine
            which, read in some lines and look for Lex '%'s or a "/*" or
            a bunch of lisp comments ';' or starting parens '('.
         */
            char buf[256], *ptr;
            int lcount=0, lisp = 0, lex = 0;
            while (fgets(buf, MAXLINE, File) != NULL) {
               ptr = buf;
               while (*ptr == ' ' || *ptr == '\t') ++ptr;
               if (*ptr == '\/') {
                  if (ptr[1] == '\*') {
                  /* Decisive, a line starts with a 'C' comment. */
                     lex = lisp + 1;
                     break;
                  }
                  else if (ptr[1] == '\/') {
                     ++lisp;  /* Lisp type comment seen (//) */
                  }
               }
               else if (*ptr == '%') {
               /* Decisive, lex requires '%'s to separate sections. */
                  lex = lisp + 1;
                  break;
               }
               else if (*ptr == '(' || *ptr == ';' || *ptr == '[') {
               /* Should be tons of these starting lines in lisp programs. */
                  if (++lisp > 4) {
                     lisp = lex + 1;
                     break;
                  }
               }
               if (++lcount > 50) {
               /* Give up. */
                  break;
               }
            }
         /* Rewind the file. */
            fseek (File, 0, 0);
            if (lisp > lex) return LISP;
            else return C;  /* Default to treat .l as lex code. */
         }
      /* Fall through */
      }
   }
   return NONE;
}


Init (pname)
char *pname;
/*
   Routine to select which printer is to be assumed.  Either the default
   (compiled in), or the environment variable CPRINTER, or the argument
   line choice.  Possible argument line or CPRINTER variables: "DUMB",
   "BACKSPACE", "ANSI", etc.  The passed in pointer ``printer'' is from
   the command line.

   After printer has been selected, anything else that needs to be done
   to this printer (change pitch etc.) is done here.
*/
{
   char *cprinter, *getenv();
   char *psheader;

   if ((cprinter = pname) != NULL || (cprinter = getenv("CPRINTER")) != NULL) {
   /* Figure out what this person wants to use as a cpr printer. */
      if (strcmp (cprinter, "DUMB") == 0)           Printer = DUMB;
      else if (strcmp (cprinter, "BACKSPACE") == 0) Printer = BACKSPACE;
      else if (strcmp (cprinter, "ANSI") == 0)      Printer = ANSI;
      else if (strcmp (cprinter, "LN03") == 0)      Printer = LN03;
      else if (strcmp (cprinter, "NECP5200") == 0)  Printer = NECP5200;
      else if (strcmp (cprinter, "POSTSCRIPT") == 0)Printer = POSTSCRIPT;
      else {
         fprintf (stderr, "Unknown printer %s\n", cprinter);
         exit(1);
      }
   }
   switch (Printer) {
      case POSTSCRIPT:
	 PageEnd = PageLength;
	 break;

      case NECP5200:
      /* Setup the NEC P5200 (draft mode, 12 cpi, left margin) */
	 printf ("\033x%c\033!%c\033l%c", 0, 1, 5);
	 /*FALLTHROUGH*/

      default:
	 PageEnd = PageLength - ((PageLength > 30) ? 7 : 1);
   }
}


Fini()
/*
   Undo whatever Init() did to the printer, etc.
*/
{
   switch (Printer) {
      case NECP5200:
	 /* Reset to LQ mode, 10 cpi, left margin to 0 */
	 printf ("\033x%c\033!%c\033l%c", 1, 0, 0);
	 break;

      case POSTSCRIPT:
	 puts("endpage\n\n%%Trailer");
	 puts("cleanup\ndocsave restore end\n");
	 break;
   }
}


DumpPostscriptHeader()
{
   puts("%! a2ps 3.0");
   puts("");
   puts("/$a2psdict 100 dict def");
   puts("$a2psdict begin");
   puts("% Initialize page description variables.");
   puts("/inch {72 mul} bind def");
   puts("/landscape true def");
   puts("/twinpage true def");
   puts("/sheetheight 11.64 inch def");
   puts("/sheetwidth 8.27 inch def");
   puts("/margin 1.2 inch def");
   puts("/noborder false def");
   puts("/noheader false def");
   puts("/headersize 0.22 inch def");
   puts("/bodyfontsize 7.5 def");
   puts("/lines 66 def");
   puts("/columns 80 def");
   puts("/datewidth 0 def");
   puts("%!  PostScript Source Code");
   puts("%");
   puts("%  File: imag:/users/local/a2ps/header.ps");
   puts("%  Created: Tue Nov 29 12:14:02 1988 by miguel@imag (Miguel Santana)");
   puts("%  Version: 2.0");
   puts("%  Description: PostScript prolog for a2ps ascii to PostScript program.");
   puts("% ");
   puts("%  Edit History:");
   puts("%  - Original version by evan@csli (Evan Kirshenbaum).");
   puts("%  - Modified by miguel@imag to:");
   puts("%    1) Correct an overflow bug when printing page number 10 (operator");
   puts("%	cvs).");
   puts("%    2) Define two other variables (sheetwidth, sheetheight) describing");
   puts("%	the physical page (by default A4 format).");
   puts("%    3) Minor changes (reorganization, comments, etc).");
   puts("%  - Modified by tullemans@apolloway.prl.philips.nl");
   puts("%    1) Correct stack overflows with regard to operators cvs and copy.");
   puts("%       The resulting substrings where in some cases not popped off");
   puts("%       the stack, what can result in a stack overflow.");
   puts("%    2) Replaced copypage and erasepage by showpage. Page througput");
   puts("%       degrades severely (see red book page 140) on our ps-printer");
   puts("%       after printing sheet 16 (i.e. page 8) of a file which was ");
   puts("%       actually bigger. For this purpose the definitions of startdoc");
   puts("%       and startpage are changed.");
   puts("%  - Modified by Tim Clark <T.Clark@uk.ac.warwick> to:");
   puts("%    1) Print one page per sheet (portrait) as an option.");
   puts("%    2) Reduce size of file name heading, if it's too big.");
   puts("%    3) Save and restore PostScript state at begining/end. It now uses");
   puts("%	conventional %%Page %%Trailer markers.");
   puts("%    4) Print one wide page per sheet in landscape mode as an option.");
   puts("%  - Modified by miguel@imag.fr to");
   puts("%    1) Add new option to print n copies of a file.");
   puts("%    2) Add new option to suppress heading printing.");
   puts("%    3) Add new option to suppress page surrounding border printing.");
   puts("%    4) Add new option to change font size. Number of lines and columns");
   puts("%	are now automatically adjusted, depending on font size and");
   puts("%	printing mode used.");
   puts("%    5) Minor changes (best layout, usage message, etc).");
   puts("%");
   puts("");
   puts("% Copyright (c) 1988, Miguel Santana, miguel@imag.imag.fr");
   puts("%");
   puts("% Permission is granted to copy and distribute this file in modified");
   puts("% or unmodified form, for noncommercial use, provided (a) this copyright");
   puts("% notice is preserved, (b) no attempt is made to restrict redistribution");
   puts("% of this file, and (c) this file is not distributed as part of any");
   puts("% collection whose redistribution is restricted by a compilation copyright.");
   puts("%");
   puts("");
   puts("");
   puts("% General macros.");
   puts("/xdef {exch def} bind def");
   puts("/getfont {exch findfont exch scalefont} bind def");
   puts("");
   puts("% Page description variables and inch function are defined by a2ps program.");
   puts("");
   puts("% Character size for differents fonts.");
   puts("   landscape");
   puts("   { /filenamefontsize bodyfontsize 1.5 mul def }");
   puts("   { /filenamefontsize bodyfontsize 1.5 mul 16 def }");
   puts("ifelse");
   puts("/datefontsize filenamefontsize 0.6 mul def");
   puts("/headermargin filenamefontsize 0.25 mul def");
   puts("/bodymargin bodyfontsize 0.7 mul def");
   puts("");
   puts("% Font assignment to differents kinds of \"objects\"");
   puts("/filenamefontname /Helvetica-Bold def");
   puts("/stdfilenamefont filenamefontname filenamefontsize getfont def");
   puts("/datefont /Helvetica-Oblique datefontsize getfont def");
   puts("/bodyfont /Courier bodyfontsize getfont def");
   puts("/boldbodyfont /Courier-Bold bodyfontsize getfont def");
   puts("");
   puts("% Logical page attributes (a half of a real page or sheet).");
   puts("/pagewidth");
   puts("   bodyfont setfont (0) stringwidth pop columns mul bodymargin dup add add");
   puts("   def");
   puts("/pageheight");
   puts("   bodyfontsize lines mul bodymargin dup add add headersize add");
   puts("   def");
   puts("");
   puts("% Coordinates for upper corner of a logical page and for sheet number.");
   puts("% Coordinates depend on format mode used.");
   puts("% In twinpage mode, coordinate x of upper corner is not the same for left");
   puts("% and right pages: upperx is an array of two elements, indexed by sheetside.");
   puts("/rightmargin margin 4 div def");
   puts("/leftmargin margin 3 mul 4 div def");
   puts("/topmargin margin twinpage {3.6} {2} ifelse div def");
   puts("landscape");
   puts("{  % Landscape format");
   puts("   /uppery rightmargin pageheight add bodymargin add def");
   puts("   /sheetnumbery sheetwidth leftmargin pageheight add datefontsize add sub def");
   puts("   twinpage");
   puts("   {  % Two logical pages");
   puts("      /upperx [ topmargin			% upperx for left page");
   puts("		dup 2 mul pagewidth add		% upperx for right page");
   puts("	      ] def");
   puts("      /sheetnumberx sheetheight topmargin sub def");
   puts("   }");
   puts("   {  /upperx [ topmargin dup ] def");
   puts("      /sheetnumberx sheetheight topmargin sub datefontsize sub def");
   puts("   }");
   puts("   ifelse");
   puts("}");
   puts("{  % Portrait format");
   puts("   /uppery topmargin pageheight add def");
   puts("   /upperx [ leftmargin dup ] def");
   puts("   /sheetnumberx sheetwidth rightmargin sub datefontsize sub def");
   puts("   /sheetnumbery");
   puts("	 sheetheight ");
   puts("	 topmargin pageheight add datefontsize add headermargin add");
   puts("      sub");
   puts("      def");
   puts("");
   puts("}");
   puts("ifelse");
   puts("");
   puts("% Strings used to make easy printing numbers");
   puts("/pnum 12 string def");
   puts("/empty 12 string def");
   puts("");
   puts("% Other initializations.");
   puts("/setdate");
   puts("{" );
   puts("/date exch def" );
   puts("/datewidth date stringwidth pop def");
   puts("filenameroom");
   puts("} def\n");
   puts("/filenameroom {");
   puts("         pagewidth");
   puts("	 filenamefontsize 4 mul datewidth add (Page 9999) stringwidth pop add");
   puts("      sub");
   puts("   } def");
   puts("");
   puts("");
   puts("% Function startdoc: initializes printer and global variables.");
   puts("/startdoc");
   puts("    { /sheetside 0 def			% sheet side that contains current page");
   puts("      /sheet 1 def			% sheet number");
   puts("   } bind def");
   puts("");
   puts("% Function newfile: init file name and reset page number for each new file.");
   puts("/newfile");
   puts("    { cleanup");
   puts("      /filename xdef");
   puts("      /filenamewidth filename stringwidth pop def");
   puts("      /filenamefont");
   puts("	 filenamewidth filenameroom gt");
   puts("	 {");
   puts("	       filenamefontname");
   puts("	       filenamefontsize filenameroom mul filenamewidth div");
   puts("	    getfont");
   puts("	 }");
   puts("	 {  stdfilenamefont }");
   puts("	 ifelse");
   puts("	 def");
   puts("    } bind def");
   puts("");
   puts("% Function printpage: Print a physical page.");
   puts("/printpage");
   puts("    { /sheetside 0 def");
   puts("      twinpage");
   puts("      {  noborder not");
   puts("	    { sheetnumber }");
   puts("	 if");
   puts("      }");
   puts("      {  noheader noborder not and");
   puts("	    { sheetnumber }");
   puts("	 if");
   puts("      }");
   puts("      ifelse");
   puts("      showpage ");
   puts("%      pagesave restore");
   puts("      /sheet sheet 1 add def");
   puts("    } bind def");
   puts("");
   puts("% Function cleanup: terminates printing, flushing last page if necessary.");
   puts("/cleanup");
   puts("    { twinpage sheetside 1 eq and");
   puts("         { printpage }");
   puts("      if");
   puts("    } bind def");
   puts("");
   puts("% Function startpage: prints page header and page border and initializes");
   puts("% printing of the file lines.");
   puts("/startpage");
   puts("    { sheetside 0 eq");
   puts("	{ % /pagesave save def");
   puts("	  landscape");
   puts("	    { sheetwidth 0 inch translate	% new coordinates system origin");
   puts("	      90 rotate				% landscape format");
   puts("	    } if");
   puts("	} if");
   puts("      noborder not { printborder } if");
   puts("      noheader not { printheader } if");
   puts("	 upperx sheetside get  bodymargin  add");
   puts("	    uppery");
   puts("	    bodymargin bodyfontsize add  noheader {0} {headersize} ifelse add");
   puts("	 sub");
   puts("      moveto");
   puts("      bodyfont setfont");
   puts("      gsave");
   puts("    } bind def");
   puts("");
   puts("% Function printheader: prints page header.");
   puts("/printheader");
   puts("    { upperx sheetside get  uppery headersize sub 1 add  moveto");
   puts("      datefont setfont");
   puts("      gsave");
   puts("        datefontsize headermargin rmoveto");
   puts("	date show					% date/hour");
   puts("      grestore");
   puts("      gsave");
   puts("	pagenum pnum cvs pop");
   puts("	   pagewidth (Page 999) stringwidth pop sub");
   puts("	   headermargin");
   puts("	rmoveto");
   puts("        (Page ) show pnum show				% page number");
   puts("      grestore");
   puts("      empty pnum copy pop");
   puts("      gsave");
   puts("        filenamefont setfont");
   puts("	      filenameroom filename stringwidth pop sub 2 div datewidth add");
   puts("	      bodymargin 2 mul ");
   puts("	   add ");
   puts("	   headermargin");
   puts("	rmoveto");
   puts("        filename show						% file name");
   puts("      grestore");
   puts("    } bind def");
   puts("");
   puts("% Function printborder: prints border page.");
   puts("/printborder ");
   puts("    { upperx sheetside get uppery moveto");
   puts("      gsave					% print the four sides");
   puts("        pagewidth 0 rlineto			% of the square");
   puts("        0 pageheight neg rlineto");
   puts("        pagewidth neg 0 rlineto");
   puts("        closepath stroke");
   puts("      grestore");
   puts("      noheader not");
   puts("         { 0 headersize neg rmoveto pagewidth 0 rlineto stroke }");
   puts("      if");
   puts("    } bind def");
   puts("");
   puts("% Function endpage: adds a sheet number to the page (footnote) and prints");
   puts("% the formatted page (physical impression). Activated at the end of each");
   puts("% source page (lines reached or FF character).");
   puts("/endpage {");
   puts("     grestore");
   puts("     /pagenum pagenum 1 add def");
   puts("     twinpage  sheetside 0 eq  and");
   puts("        { /sheetside 1 def }");
   puts("        { printpage }");
   puts("     ifelse");
   puts("   } bind def");
   puts("");
   puts("% Function sheetnumber: prints the sheet number.");
   puts("/sheetnumber");
   puts("    { sheetnumberx sheetnumbery moveto");
   puts("      datefont setfont");
   puts("      sheet pnum cvs");
   puts("	 dup stringwidth pop (0) stringwidth pop sub neg 0 rmoveto show");
   puts("      empty pnum copy pop");
   puts("    } bind def");
   puts("");
   puts("% Function s: print a source line");
   puts("/newline {");
   puts("    grestore");
   puts("    0 bodyfontsize neg rmoveto");
   puts("    gsave");
   puts("} bind def");
   puts("/s  { show");
   puts("      newline");
   puts("    } bind def");
   puts("/sb  {");
   puts("        boldbodyfont setfont");
   puts("        show");
   puts("	 bodyfont setfont");
   puts("    } bind def");
   puts("/sn  {");
   puts("        show");
   puts("    } bind def");
   puts("%%EndProlog");
   puts("\n\n/docsave save def\nstartdoc\n");
}
