/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */

#include "agg.h"

// **************************************************************************
// Module List              20/10/92                  Jochen Alt (ja)
// **************************************************************************
// implements methods of classes: List
// **************************************************************************

// Die Knoten werden direkt ueber psm abgespeichert, um bei einheitlicher
// Typinformation bzw. einheitlichem Container viiiiel Platz zu sparen. 
// Da eventuell der Container und der Typ nicht abgespeichert werden muss,
// kann dadurch pro Knoten 12 Byte gespart werden. 
// 
// Es wird fuer den Eintrag ein default Typ und ein default
// Container abgespeichert, beide werden beim Eintragen des ersten Elementes 
// gesetzt. Entspricht ein folgender Eintrag einem solchen Defaultwert,
// wird nur das unbedingt notwendige gespeichert. In der Standardanwendung,
// in der der Eintrag immer denselben Typ hat und im selben Container liegt,
// braucht ein Knoten dann 16 Bytes 
// 
// 
// Ein Knoten wird in folgendem Format abgespeichert:
//   sos_Offset : Verweis auf Vorgaengerknoten
//   sos_Offset : Verweis auf Nachfolgerknoten
//   sos_Int    : Typinformation des Knotens und Groesse des Knotens
//      evt. sos_Typed_id Typinformation des Eintrages
//      evt. sos_Container Container des Eintrages
//   sos_Offset : Offset des Eintrages
// 
//
// Der Typ eines Knotens wird durch die folgenden Konstanten identifiziert,
// die die Wertigkeit des Bits angeben, welches bestimmt, ob ein jeweiliges
// Datum abgespeichert wurde. Beispiel: Ist in einem Knoten
// der Typ abgespeichert, weil er dem Standard-typ nicht entspricht, 
// so ist das ENTRY_TYPE_DIFF.te Bit gesetzt.
// Die Groesse des Knotens ist im 2.Byte abgespeichert (Koennte man sich
// aus dem Type bequem berechnen, den Platz im sos_Int kriegt man jedoch 
// geschenkt).

const sos_Offset NIL=0;
#define ENTRY_CNT_DIFF 1
#define ENTRY_TYPE_DIFF 2
// Die Groesse befindet sich im 2.ten Byte der Typinformation eines Knotens
#define SIZE_POS 256  

// Erzeuge den Typ Node_type mit NODE_TYPE_SIZE und den entsprechenden bcopies
#define Node_type sos_Int
#define NODE_TYPE_SIZE SOS_INT_SIZE
#define bcopy_from_Node_type bcopy_from_sos_Int
#define bcopy_to_Node_type bcopy_to_sos_Int


LOCAL const int MAX_NODE_SIZE = 2*SOS_OFFSET_SIZE + NODE_TYPE_SIZE + 
			  (SOS_CONTAINER_SIZE + 
			   SOS_OFFSET_SIZE +
			   SOS_ID_SIZE);

// **************************************************************************
void sos_Object_List::change_list_cursor (sos_Bool new_list_cursor)
// **************************************************************************
{  self.set_list_cursor (new_list_cursor);
} // ** list::change_list_cursor ***

// **************************************************************************
void sos_Object_List::set_next (sos_Offset offset, sos_Offset next)
// **************************************************************************
// Set the field with the successor to 'next'
{  if (offset == NIL)
   { self.set_first (next);
     return;
   }
 
   err_assert ((offset != next),"List:set_next:recursion");
   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;
 
   bcopy_from_sos_Offset (&next, &u.mem);
 
   self.container().write (offset+SOS_OFFSET_SIZE, SOS_OFFSET_SIZE, &u.mem);
}  // *** List::set_next ***

// **************************************************************************
void sos_Object_List::set_prev (sos_Offset offset, sos_Offset prev)
// **************************************************************************
{  if (offset == NIL)
   { self.set_last (prev);
     return;
   }
 
   err_assert (offset != prev,"List:set_prev:recursion");
   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;
 
   bcopy_from_sos_Offset (&prev, &u.mem);
 
   self.container().write (offset, SOS_OFFSET_SIZE, &u.mem);
}  // *** sos_Object_List::set_prev ***
 
 

// ************************************************************************** 
sos_Offset sos_Object_List::create_node (sos_Object entry, 
					 sos_Offset prev, sos_Offset next)
// ************************************************************************** 
// create a node with contents (entry, prev, next) and return the offset
{  
      // garuantee correct Byte position
   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;

   sos_Id default_type = self.get_default_type();
   sos_Container default_container = self.get_default_container();

   sos_Container cnt = self.container();
   if (self.get_cardinality() == 0)
   {  self.set_default_type (default_type = entry._type_id());
      self.set_default_container(default_container=entry.container());
   }
   
   char* ptr = u.mem;
   ptr += 2*SOS_OFFSET_SIZE;
   ptr += NODE_TYPE_SIZE;
   Node_type node_type = 0;

   sos_Id entry_type = entry._type_id();
   if (entry_type != default_type)
   {  bcopy_from_sos_Id (&entry_type, ptr);
      ptr += SOS_ID_SIZE;
      node_type += ENTRY_TYPE_DIFF;
   }
   sos_Container entry_cnt = entry.container();
   if (entry_cnt != default_container)
   {  bcopy_from_sos_Container (&entry_cnt, ptr);
      ptr += SOS_CONTAINER_SIZE;
      node_type += ENTRY_CNT_DIFF;
   }
   sos_Offset entry_offset = entry.offset();
   bcopy_from_sos_Offset (&entry_offset, ptr);
   ptr += SOS_OFFSET_SIZE;

   sos_Int size = ptr - u.mem;
   node_type += size*SIZE_POS;

   ptr = u.mem;
   bcopy_from_sos_Offset (&prev, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_from_sos_Offset (&next, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_from_Node_type (&node_type, ptr);

   sos_Offset offset = cnt.allocate (size);
   err_assert (size<=MAX_NODE_SIZE,"create_node");
   cnt.write (offset, size, &u);

   self.set_size( self.get_size() + size);
   self.set_cardinality( self.get_cardinality() + 1);
   
   self.set_prev (next, offset);
   self.set_next (prev, offset);
   return offset;
}  // *** List::create_node ***


// ************************************************************************** 
void sos_Object_List::remove_node(sos_Offset offset, sos_Offset pred,
                                  sos_Offset next, sos_Int size)
// ************************************************************************** 
// remove a node at position with the size of size
{  err_assert ((size>=16) AND (size<=MAX_NODE_SIZE),"List:remove_node:internal error1");
   err_assert (offset != 0,"List:remove_node:internal Error2");
   self.container().deallocate (offset,size);
   self.set_size (self.get_size() - size);
   self.set_cardinality( self.get_cardinality() - 1);

   self.set_next (pred, next);
   self.set_prev (next, pred);
} // ** List::remove_node ***

// **************************************************************************
void sos_Object_List::read_node (sos_Offset offset,
                                 sos_Object &entry, 
                                 sos_Offset &prev,
                                 sos_Offset &next,
                                 sos_Int    &size)
// **************************************************************************
// read a node at position offset, and set the parameters
{  sos_Id entry_type = self.get_default_type();
   sos_Container entry_container = self.get_default_container();

   union {sos_Offset dummy;
          char mem[MAX_NODE_SIZE]; } u;

   err_assert (offset != 0,"List:read_node:internal error1");
   sos_Container cnt = self.container();
   cnt.read (offset, NODE_TYPE_SIZE +2*SOS_OFFSET_SIZE, &u);

   char* ptr = u.mem;
   Node_type node_type;
   bcopy_to_sos_Offset (&prev, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_to_sos_Offset (&next, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_to_Node_type(&node_type, ptr);
   ptr += NODE_TYPE_SIZE;
   size = node_type/SIZE_POS;

   err_assert (size<=MAX_NODE_SIZE,"List:read_node:internal error2");
   cnt.read (offset, size, &u);
   if (node_type BITAND ENTRY_TYPE_DIFF)
   {  bcopy_to_sos_Id (&entry_type, ptr);  
      ptr += SOS_ID_SIZE;
   }
   if (node_type BITAND ENTRY_CNT_DIFF)
   {  bcopy_to_sos_Container (&entry_container, ptr);  
      ptr += SOS_CONTAINER_SIZE;
   }
   sos_Offset entry_offset;
   bcopy_to_sos_Offset (&entry_offset, ptr);
   ptr += SOS_OFFSET_SIZE;

 
   sos_Typed_id entry_tpid = 
      sos_Typed_id::make(sos_Id::make(entry_container,entry_offset),entry_type);
   entry = sos_Object::make (entry_tpid);
} // *** List::read_node ***

// ************************************************************************** 
void sos_Object_List::local_initialize (sos_Object_List l)
// ************************************************************************** 
{  T_PROC("List::local_initialize");
   TT(agg_H, T_ENTER);

   l.set_cardinality (0);
   l.set_size (0);
   l.set_first (NIL);
   l.set_last (NIL);

   TT(agg_H,T_LEAVE);
}; // ** List::local_initialize **


// ************************************************************************** 
void sos_Object_List::local_finalize (sos_Object_List l)
// ************************************************************************** 
{  T_PROC("List::local_finalize");
   TT (agg_H, T_ENTER);

   l.clear();

   TT (agg_H, T_LEAVE);
} // ** List::local_finalize **


// ************************************************************************** 
void sos_Object_List::append (sos_Object o)
// ************************************************************************** 
{  T_PROC("List::append");
   TT(agg_H, T_ENTER);

   self.create_node (o, self.get_last(), NIL);

   TT(agg_H,T_LEAVE);   
} // ** List::append **

// ************************************************************************** 
void sos_Object_List::insert (sos_Int pos, sos_Object o)
// ************************************************************************** 
{  T_PROC("List::insert");
   TT(agg_H, T_ENTER);

   sos_Int crd = self.get_cardinality();

   err_assert ((0 <= pos) AND (pos <= crd+1), "List:insert:pos invalid");

      // pos == 0 forces the list to append the entity
   if (pos == 0)
      pos = crd + 1;

   if (pos <=crd/2)
   {
      sos_Offset prev = NIL;
      sos_Offset list = NIL;
      sos_Offset next = self.get_first();
      for (sos_Int i = 1;(i<pos) AND (next != NIL); i++)
      {  sos_Object tmp_entry;
         sos_Int size;
         list = next;
         self.read_node (list, tmp_entry, prev, next, size);
      }
      self.create_node (o, list, next);
   }
   else
   {  sos_Offset prev = self.get_last();
      sos_Offset list = NIL;
      sos_Offset next = NIL;
      for (sos_Int i = crd;(i>=pos) AND (prev != NIL); i--)
      {  sos_Object tmp_entry;
         sos_Int size;
         list = prev;
         self.read_node (list, tmp_entry, prev, next, size);
      }
      self.create_node (o, prev, list);
   }

   TT(agg_H,T_LEAVE);   
} // *** List::insert ***

// ************************************************************************** 
void sos_Object_List::remove (sos_Int pos)
// ************************************************************************** 
{  T_PROC("List::remove");
   TT(agg_H, T_ENTER);

   sos_Int crd = self.get_cardinality();

   err_assert ((0<pos) AND (pos<=crd),"List:remove:pos invalid");

   if (pos <=crd/2)
   {
      sos_Offset prev = NIL;
      sos_Offset next = self.get_first();
      sos_Offset list = NIL;
      sos_Int size;
 
      for (sos_Int i = 0;(i<pos); i++)
      {  sos_Object key;
         list = next;
         self.read_node (list, key, prev, next, size);
      }
  
      self.remove_node (list,prev, next,size);
   }
   else
   {  sos_Offset prev = self.get_last();
      sos_Offset next = NIL;
      sos_Offset list = NIL;
      sos_Int size;
  
      for (sos_Int i = crd;(i>=pos); i--)
      {  sos_Object key;
         list = prev;
         self.read_node (list, key, prev, next, size);
      }
  
      self.remove_node (list,prev, next,size);
   }
 
   TT(agg_H,T_LEAVE)
}// ** List::remove **

// ************************************************************************** 
sos_Object sos_Object_List::get_nth(sos_Int pos)
// ************************************************************************** 
{  T_PROC("List::get_nth(sos_Int)");
   TT(agg_H, T_ENTER);

   err_assert ((0 < pos) AND (pos <= self.get_cardinality()),
	       "List:get_nth:position invalid");

   sos_Offset prev = NIL;
   sos_Offset next = self.get_first();;
   sos_Offset list = NIL;
   sos_Object entry = NO_OBJECT;
   for (sos_Int i = 0;(i<pos) AND (next != NIL); i++)
   {  sos_Int size;
      list = next;
      self.read_node (list, entry, prev, next, size);
   }

   TT(agg_H,T_LEAVE);
   return entry;
} // ** List::get_nth **

// ************************************************************************** 
void sos_Object_List::set_nth(Index pos, sos_Object o)
// ************************************************************************** 
{  T_PROC("List::set_nth(sos_Int)");
   TT(agg_H, T_ENTER);

   err_assert ((0 < pos) AND (pos <= self.get_cardinality()),
	       "List:set_nth:position invalid");

   sos_Cursor c = self.open_cursor();
   self.move_cursor (c, pos);
   self.set (c,o);
   self.close_cursor(c);

   TT(agg_H,T_LEAVE);
} // *** List:set_nth ***

// ************************************************************************** 
Index sos_Object_List::find(sos_Object o)
// ************************************************************************** 
{  T_PROC("List::find(sos_Object)");
   TT(agg_H, T_ENTER);

   Index result = 0; // assume not found
   sos_Bool boe = self.get_based_on_equal();
 
   sos_Cursor c = self.open_cursor();
   for (sos_Bool valid = self.is_valid(c); valid; valid = self.to_succ(c))
   {
      sos_Object x = self.get(c);
      if (agg_same_entity (o, x, boe, EQ_WEAK))
      {  result = self.current_pos(c);
         break;
      }
   }
   self.close_cursor(c);

   TT(agg_H,T_LEAVE);

   return result;
} // *** List::find ***


// ************************************************************************** 
sos_Object sos_Object_List::find_or_append (sos_Object o)
// ************************************************************************** 
{  T_PROC ("List::find_or_append");
   TT( agg_H, T_ENTER);
   
   Index      i = self.find(o);
   sos_Object result;
   if (i == 0) // not found, then append
   {  self.append(o);
      result = o;
   }
   else
      result = self.get_nth(i);
  

   TT(agg_H,T_LEAVE);
   return result;
} // *** List::find_or_append ***

// **************************************************************************
void sos_Object_List::operator+= (sos_Object_List alist)
// **************************************************************************
{
   T_PROC ("sos_Object_List::operator+=");
   TT (agg_H, T_ENTER);
 
   agg_iterate (alist, sos_Object o)
      self.append (o);
   agg_iterate_end(alist,o)
 
   TT (agg_H, T_LEAVE);
} // ** operator+= **
 
// **************************************************************************
Index sos_Object_List::current_pos (sos_Cursor c)
// **************************************************************************
{
   T_PROC ("sos_Object_List::current_pos");
   TT (agg_H, T_ENTER);
 
   Index result = 0;
 
   err_assert (self.is_valid (c), "sos_Object_List::current_pos");

   sos_List_node node = sos_List_node::make(c.get_current());
   sos_Offset list = node.get_offset();
   for (;list != NIL;result++)
   {  sos_Object entry;
      sos_Offset prev, next;
      sos_Int size;
      self.read_node (list,entry, prev, next, size);
      list = prev;
   }

   TT (agg_H, T_LEAVE; TI (result));
   return result;
} // ** current_pos **

// **************************************************************************
sos_Bool sos_Object_List::move_cursor (sos_Cursor c, Index pos)
// **************************************************************************
{
   T_PROC ("sos_Object_List::move_cursor");
   TT (agg_H, T_ENTER; TI (pos));

   sos_Int crd = self.get_cardinality();

   err_assert ((0 < pos) AND (pos <= crd),
	       "List:move_cursor:position invalid");

   if (pos <= crd/2)
   {  self.to_first (c);
      if (pos > 1)
         self.to_succ (c,pos-1);
   }
   else
   {  self.to_last(c);
      if (pos < crd)
	 self.to_pred (c, crd-pos);
   }

   TT (agg_H, T_LEAVE);
   return self.is_valid(c);
} // ** move_cursor **
 
 
// ************************************************************************** 
void sos_Object_List::insert_before (sos_Cursor c, sos_Object o)
// ************************************************************************** 
{  T_PROC("List::insert_before");
   TT(agg_H, T_ENTER);

   err_assert (self.is_valid(c), "List:insert_before:cursor invalid");
   err_assert (c.defined_for (self), "List:insert_after:cursor invalid");

   sos_List_node node = sos_List_node::make (c.get_current());
   sos_Object entry;
   sos_Int size;
   sos_Offset offset = node.get_offset();
   sos_Offset prev, next;
   self.read_node (offset, entry, prev, next, size);
   node.set_prev (self.create_node (o, prev, offset));
   node.set_entry (o);

   TT(agg_H, T_LEAVE);
} // ** List::insert_before **

// ************************************************************************** 
void sos_Object_List::insert_after (sos_Cursor c, sos_Object o)
// ************************************************************************** 
{  T_PROC("List::insert_after");
   TT(agg_H, T_ENTER);
 
   err_assert (self.is_valid(c), "List:insert_after:cursor invalid");
   err_assert (c.defined_for (self), "List:insert_after:cursor invalid");

   sos_List_node node = sos_List_node::make (c.get_current());
   sos_Object entry;
   sos_Int size;
   sos_Offset offset = node.get_offset();
   sos_Offset prev, next;
   self.read_node (offset, entry, prev, next, size);
   node.set_next (self.create_node (o, offset, next));
   node.set_entry (o);

   TT(agg_H, T_LEAVE);
} // ** List::insert_after **

// ************************************************************************** 
void sos_Object_List::local_assign (sos_Object_List x,
                                    sos_Object      o)
// ************************************************************************** 
{  T_PROC("List::local_assign");

   sos_Object_List y = sos_Object_List::make (o);
   x.clear();
   x.operator+= (y);

   TT(agg_H,T_LEAVE); 
} // ** List::local_assign **

// ************************************************************************** 
sos_Bool sos_Object_List::local_equal (sos_Object_List x,
                                       sos_Object      o,
                                       sos_Eq_kind     eq_kind) 
// ************************************************************************** 
{  sos_Bool result;
 
   sos_Object_List y = sos_Object_List::make (o);
   if (x.get_cardinality() != y.get_cardinality())
      result = FALSE;
   else
   {  result = TRUE;
      sos_Bool based_on_equal = x.get_based_on_equal();
      int  comp = 0;
      agg_iterate_double (x, sos_Object x_e, y, sos_Object y_e, comp)
      {  result = agg_same_entity (x_e, y_e, based_on_equal, eq_kind);
	 if (NOT result) break;
      }
      agg_iterate_double_end (x, x_e, y, y_e, comp);
      result = (sos_Bool)(result AND comp == 0);
   }
   
   return result;
} // ** List::local_equal **

// ************************************************************************** 
sos_Int sos_Object_List::local_hash_value (sos_Object_List x)
// ************************************************************************** 
{
   T_PROC ("sos_Object_List::local_hash_value");
   TT (agg_H, T_ENTER );
 
   sos_Int result;
 
   if (x.is_empty())
      result = 0;
   else
      result = x.get_nth (1).hash_value();
 
   TT (agg_H, T_LEAVE; TB(result));
 
   return result;
} // ** List::local_hash_value **

// **************************************************************************
sos_Object sos_Object_List::get (sos_Cursor c)
// **************************************************************************
{
   T_PROC ("sos_Object_List::get (sos_Cursor)");
   TT (agg_H, T_ENTER);
 
   err_assert (self.is_valid (c), "sos_Object_List::get (sos_Cursor):cursor invalid");

   sos_List_node node = sos_List_node::make (c.get_current());
   sos_Object entry;
   if (NOT node.get_list_cursor())
      entry = node.get_entry();
   else
   {  sos_Offset offset = node.get_offset();
      sos_Int size;
      sos_Offset prev, next;
      self.read_node (offset, entry, prev, next, size);
   }

   TT (agg_H, T_LEAVE);
   return entry;
} // ** get **

// **************************************************************************
void sos_Object_List::set (sos_Cursor c, sos_Object o)
// **************************************************************************
{
   T_PROC ("sos_Object_List::set");
   TT (agg_H, T_ENTER);
 
   err_assert (self.is_valid (c), "sos_Object_List::set:cursor invalid");

   sos_List_node node = sos_List_node::make (c.get_current());
   sos_Object entry;
   sos_Int size;
   sos_Offset offset = node.get_offset();
   sos_Offset prev, next;

   self.read_node (offset, entry, prev, next, size);
   self.remove_node (offset, prev, next, size);
   offset = self.create_node (o, prev, next);

   node.set_offset (offset);
   node.set_prev (prev);
   node.set_next (next);
   node.set_entry (o);
 
   TT (agg_H, T_LEAVE);
} // ** set **

// ************************************************************************** 
void sos_Object_List::remove_at(sos_Cursor c)
// ************************************************************************** 
{  T_PROC("List::remove_at");
   TT(agg_H, T_ENTER);

   err_assert (self.is_valid(c), "List:remove_at:cursor invalid");
   err_assert (c.defined_for (self), "List:remove_at:cursor invalid");

   sos_List_node node = sos_List_node::make (c.get_current());
   sos_Offset list = node.get_offset();
   sos_Offset next,prev;
   sos_Object entry;
   sos_Int size;

   self.read_node (list, entry, prev, next,size);
   self.remove_node (list,prev, next,size);

   if (next == NIL)
      node.set_offset (NIL);
   else
   {  list = next;
      self.read_node (list, entry, prev, next,size);
      node.set_offset (list);
      node.set_prev (prev);
      node.set_next (next);
      node.set_entry (entry);
   }

   TT(agg_H, T_LEAVE);
} // ** List::remove_at **
 

// ************************************************************************** 
sos_Int sos_Object_List::size()
// ************************************************************************** 
{  return self.get_size() +self.sos_Object::size();
} // ** List::size **

// **************************************************************************
void sos_Object_List::clear()
// **************************************************************************
{  T_PROC("List::clear");
   TT(agg_H, T_ENTER);
 
   for (;self.get_cardinality() > 0;)
      self.remove (1);
 
   TT(agg_H, T_LEAVE);
} // ** List::clear **

// **************************************************************************
sos_Int sos_Object_List::card ()
// **************************************************************************
{  T_PROC ("List::card");
   TT (agg_H, T_ENTER);

   sos_Int crd = self.get_cardinality();

   TT (agg_H, T_LEAVE; TI(crd));
   return crd;
} // ** List::card **

// ************************************************************************** 
sos_Cursor sos_Object_List::open_cursor(sos_Container cnt)
// ************************************************************************** 
{  T_PROC ("List::open_cursor");
   TT( agg_H, T_ENTER);

   sos_Cursor c = sos_Cursor::create (cnt, self);
   sos_List_node node = sos_List_node::create (cnt, self.get_list_cursor());
   c.set_current (node);
   self.to_first (c);

   TT(agg_H, T_LEAVE);
   return c;
} // ** List::open_cursor **

// ************************************************************************** 
void sos_Object_List::close_cursor(sos_Cursor c)
// ************************************************************************** 
{     
   err_assert (c != NO_OBJECT, "List:close_cursor:cursor invalid");
   err_assert ((c.defined_for (self)), "List:close_cursor:cursor invalid");
   c.get_current().destroy();
   c.destroy();
} // ** List::close_cursor **

// ************************************************************************** 
sos_Cursor sos_Object_List::duplicate(sos_Cursor c, 
				      sos_Container cnt /*= TEMP_CONTAINER */)
// ************************************************************************** 
{
   err_assert ((c.defined_for (self)), "List:duplicate:cursor invalid");

   sos_Cursor new_c = self.open_cursor (cnt);
   new_c.get_current().assign (c.get_current());
   return new_c;
} // ** List::duplicate **

// **************************************************************************
sos_Bool sos_Object_List::is_valid (sos_Cursor c)
// **************************************************************************
{
   sos_List_node node = sos_List_node::make (c.get_current());
   return (sos_Bool)(node.get_offset() != NIL);
} // ** is_valid **

// ************************************************************************** 
sos_Bool sos_Object_List::to_first(sos_Cursor c)
// ************************************************************************** 
{  err_assert (c != NO_OBJECT, "List::to_first:cursor invalid");
   // err_assert ((c.defined_for (self)), "List:to_first");
  
   sos_List_node node = sos_List_node::make (c.get_current());
   sos_Int size;
   sos_Object entry;
   sos_Offset list = self.get_first();
   sos_Offset prev, next;
   if (list != NIL)
      self.read_node (list, entry, prev, next,size);

   node.set_offset (list);
   node.set_prev (prev);
   node.set_next (next);
   node.set_entry (entry);

   return self.is_valid (c);
} // ** List::to_first **

// ************************************************************************** 
sos_Bool sos_Object_List::to_last(sos_Cursor c)
// ************************************************************************** 
{  err_assert (c != NO_OBJECT, "List::to_last:cursor invalid");
   err_assert ((c.defined_for (self)), "List:to_last:cursor invalid");

   sos_List_node node = sos_List_node::make (c.get_current());
   sos_Int size;
   sos_Object entry;
   sos_Offset list = self.get_last();
   sos_Offset prev, next;
   if (list != NIL)
      self.read_node (list, entry, prev, next,size);

   node.set_offset (list);
   node.set_prev (prev);
   node.set_next (next);
   node.set_entry (entry);

   return self.is_valid (c);
} // ** List::to_last **

// ************************************************************************** 
sos_Bool sos_Object_List::to_succ(sos_Cursor c, Index i) 
// ************************************************************************** 
{  err_assert (c != NO_OBJECT, "List::to_succ:cursor invalid");
   err_assert (self.is_valid(c), "List:to_succ:cursor invalid");
   err_assert (c.defined_for (self), "List:to_succ:cursor invalid");

   sos_List_node node = sos_List_node::make (c.get_current());
   if (i>0)
   {  sos_Object entry;
      sos_Offset list = NIL;
      sos_Offset  prev, next;

      if (node.get_list_cursor())
      {  next = node.get_offset();
	 i++;
      }
      else
         next = node.get_next();
   
      for (;i>0 AND next != NIL; i--)
      {  sos_Int size;
         list = next;
         self.read_node (list, entry,prev,next,size);
      }

      if ( i != 0)
	 list = NIL;
   
      node.set_offset (list);
      node.set_next (next);
      node.set_prev (prev);
      node.set_entry (entry);
   }
   else
      if (i<0)
	 self.to_pred (c,-i);

   return self.is_valid (c);
} // ** List::to_succ **

// ************************************************************************** 
sos_Bool sos_Object_List::to_pred(sos_Cursor c, Index i)
// ************************************************************************** 
{  err_assert (c != NO_OBJECT, "List::to_pred:cursor invalid");
   err_assert (self.is_valid(c), "List:to_pred:cursor invalid");
   err_assert (c.defined_for (self), "List:to_pred:cursor invalid");

   sos_List_node node = sos_List_node::make (c.get_current());
   if (i > 0)
   {  sos_Offset list = NIL;
      sos_Offset prev, next;
      sos_Object entry;

      if (node.get_list_cursor())
      {  prev = node.get_offset();
	 i++;
      }
      else
         prev = node.get_prev();
   
      for (;i>0 AND prev != NIL; i--)
      {  sos_Int size;
         list = prev;
         self.read_node (list, entry, prev, next, size);
      }

      if ( i != 0)
	 list = NIL;

      node.set_offset (list);
      node.set_next (next);
      node.set_prev (prev);
      node.set_entry (entry);
  }
  else 
     if (i<0)
        self.to_succ (c, -i);

   return self.is_valid (c);
} // ** List::to_pred **
