/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module cfe_main                                           Juergen Uhl (ju)
//
// **************************************************************************
//
// tracing conventions: see trc_cfe.h

#define OBST_IMP_STDCONST
#define OBST_IMP_STREAM
#define OBST_IMP_FILE
#include "obst_stdinc.h"

#include "cfe.h"

#ifdef PROFILE
extern "C" void monitor(...);
#endif

// ---------- global variables ----------------------------------------------

EXPORT sos_Schema_module   cfe_schema;
EXPORT sos_Type_table      cfe_type_tab;
EXPORT sos_Forward_table   cfe_forward_tab;
EXPORT sos_Type_descr_List cfe_types;
EXPORT sos_Container	   cfe_cnt;
EXPORT sos_Class_type      ERROR_TYPE;
EXPORT sos_Bool		   cfe_echo_flag = FALSE;


// --------------------------------------------------------------------------

EXPORT void cfe_compile()
{
   T_PROC ("cfe_compile")
   TT (cfe_H, T_ENTER);

   /* ERROR_TYPE is the result of cfe_lookup_generic_instantiation in case
      of an incorrect instantiation. Since cfe doesn't abort execution, this
      class must be pretty initialized. In addition to this class a further
      class is needed, representing the generic class, of which ERROR_TYPE
      is an instantiation. */

   sos_Class_type error_gen_class = sos_Class_type::create(TEMP_CONTAINER);

   error_gen_class.set_formal_gen_params (
	           sos_Gen_param_List::create (TEMP_CONTAINER));
   error_gen_class.set_actual_gen_params (sos_Type_descr_List::make(NO_OBJECT));
   ERROR_TYPE = sos_Class_type::create (TEMP_CONTAINER);
   ERROR_TYPE.set_name (sos_String::create (TEMP_CONTAINER, "__error"));
   error_gen_class.set_name (ERROR_TYPE.get_name());
   ERROR_TYPE.set_create_params (sos_Param_List::make(NO_OBJECT));
   ERROR_TYPE.set_super_classes (sos_Super_class_List::create (
						     TEMP_CONTAINER, TRUE));
   ERROR_TYPE.set_super_closure (sos_Super_class_List::create (TEMP_CONTAINER,
		FALSE));
   cfe_set_super_closure (ERROR_TYPE);
   ERROR_TYPE.set_friends (sos_Type_List::make (NO_OBJECT));
   ERROR_TYPE.set_local_methods (sos_Method_List::create (TEMP_CONTAINER));
   ERROR_TYPE.set_methods (sos_Method_table::create (TEMP_CONTAINER));
   ERROR_TYPE.set_components (sos_Comp_descr_List::create (TEMP_CONTAINER));
   ERROR_TYPE.set_root_class (error_gen_class);
   ERROR_TYPE.set_formal_gen_params (sos_Gen_param_List::make(NO_OBJECT));
   ERROR_TYPE.set_actual_gen_params (sos_Type_descr_List::create(TEMP_CONTAINER));

   cfe_yyparse();

   TT (cfe_H, T_LEAVE);
}


EXPORT void cfe_init (sos_String name, sos_Import_List imp)
{
   T_PROC ("cfe_init")
   TT (cfe_H, T_ENTER);

   smg_String schema_nm     = name;
   sos_Bool   is_bootschema = FALSE;
   for (char** snm = _obst_bootschemas;  *snm;  ++ snm)
      if (schema_nm.equal (*snm))
      {  is_bootschema = TRUE;
	 break;
      }
   if (is_bootschema)
#ifndef BOOT
      cfe_error (err_USE, err_CFE_RESERVED_SCHEMA_NAME);
#else
       ;
   else
      cfe_error (err_WNG, err_CFE_NO_RESERVED_SCHEMA_NAME);
#endif

   if (NOT schema_nm.equal ("knl"))
   {  sos_String 	knlstr = sos_String::create (TEMP_CONTAINER, "knl");
      sos_Schema_module sm     = sos_Schema_module::lookup (knlstr);
      knlstr.destroy();
      if INVALID(sm)
         cfe_error (err_SYS, err_CFE_NO_KERNEL);
      else 
      {  // check, if knl is already imported; if not, import it
	 sos_Bool found = FALSE;
	 agg_iterate (imp, sos_Import imp_mdl)
	 {
	    if (imp_mdl.get_module().operator==(sm))
	    {			     // ATT2.0 QUEERNESS: explicit operator
	       found = TRUE;
	       break;
	    }
         } agg_iterate_end (imp, imp_mdl);
	 if (NOT found)
	 {  sos_Import knl_imp = sos_Import::create(cfe_cnt);
	    knl_imp.set_module (sm);
	    knl_imp.set_mode (sos_IMP_NONE);
	    imp.insert (1, knl_imp);
	    // Attention: knl can only be inserted as the first import
	    // without further effort,
	    // since it does not import schemas by itself. Otherwise these
	    // schemas imported indirect via knl would have to be appended
	    // to the list of imports.
	    sm.container().open (READING, WAITING);
	 }
      }
   }
   // not yet implemented:
   // open imported schemas (transitively) for reading
   // open depending schemas (transitively) for writing

   cfe_types       = sos_Type_descr_List::create (cfe_cnt, FALSE);
   cfe_type_tab    = sos_Type_table::create (cfe_cnt,
					     /*list_cursor =*/ FALSE,
					     /*key_based_on_equal =*/ TRUE);
   cfe_forward_tab = sos_String_sos_Class_type_List_Mapping::create (
						TEMP_CONTAINER, FALSE, TRUE);

   cfe_schema.set_has_external_types (FALSE);
   cfe_schema.set_types (cfe_types);
   cfe_schema.set_type_table (cfe_type_tab);

#ifdef BOOT
   cfe_schema.install();
#endif
   TT (cfe_H, T_LEAVE);
}


EXPORT void cfe_finalize()
{
   T_PROC ("cfe_finalize");
   TT (cfe_H, T_ENTER);

   if (err_occurred (err_SYS) + err_occurred (err_USE) != 0)
      cfe_cnt.destroy();
#ifndef BOOT
   else
      cfe_schema.install();
#endif
   cfe_cnt.close();

   sos_Container_set open_cnts = sos_Container_set::open_containers(OPEN);
   cnt_iterate (open_cnts, sos_Container c)
      c.close();
   cnt_iterate_end (open_cnts, c)
   TT (cfe_H, T_LEAVE);
}


EXPORT void cfe_check_schema ()
{
   T_PROC ("cfe_check_schema");
   TT (cfe_H, T_ENTER);

   agg_iterate_association (cfe_forward_tab, sos_String name,
			    sos_Class_type_List ctl)
   {  agg_iterate (ctl, sos_Class_type ct)
      {
	 if INVALID(ct.get_methods())
	    cfe_error (err_USE, err_CFE_NOT_DECLARED, ct.get_name());
      }
      agg_iterate_end (ctl, ct);
   }
   agg_iterate_association_end (cfe_forward_tab, name, ctl);

   TT (cfe_H, T_LEAVE);
}


EXPORT void cfe_error (err_class ec, err_msg s, char* where /* = 0 */)
{
   T_PROC ("cfe_error");
   TT (cfe_H, T_ENTER);

   smg_String line_msg = smg_String ("line ") + cfe_yylineno;

   if (where)
      line_msg += smg_String(": ") + where;
   
   err_raise (ec, s, line_msg.make_Cstring (SMG_BORROW));

   TT (cfe_H, T_LEAVE);
}


EXPORT void cfe_error (err_class ec, err_msg s, sos_String where)
{  sos_Cstring w = where.make_Cstring();
   cfe_error (ec, s, w);
   delete w;
}


EXPORT void cfe_import_module (sos_Import_List imp, sos_String name)

// Check existence of a schema module named 'name' and append it
// at the module list imp

{  T_PROC ("cfe_import_module")
   TT (cfe_H, T_ENTER);

   sos_Schema_module sm;
   sos_Import	     i = sos_Import::create(cfe_cnt);

   sm = sos_Schema_module::lookup (name);
   if INVALID(sm)
      cfe_error (err_USE, err_CFE_INVALID_IMPORT, name);
   else
   {  i.set_module(sm);
      i.set_mode(sos_IMP_NONE);
      imp.append (i);
   }

   TT (cfe_H, T_LEAVE);
}


LOCAL void cfe_imp_search_or_append (sos_Import_List il, sos_Import imp)
{
   T_PROC ("cfe_imp_search_or_append");
   TT (cfe_M, T_ENTER);

   sos_Bool found = FALSE;

   agg_iterate (il, sos_Import import_from_list)
   {
      if (import_from_list.get_module().operator== (imp.get_module()))
      {  found = TRUE;			// ATT2.0 QUEERNESS: explicit operator
	 break; // found, so do not append
      }
   } agg_iterate_end (il, import_from_list);

   if (NOT found)
   {  sos_Import imp_clone = sos_Import::clone(imp, cfe_cnt);
      imp_clone.set_mode (sos_IMP_NONE);
      il.append (imp_clone); // copy it in the current cnt
   }
   TT (cfe_M, T_LEAVE);
}


EXPORT void cfe_complete_imports (sos_Import_List imports)
{
   // Append all indirect imported schema modules to the list of imports.
   // Schema modules reachable through several paths are included only once.
   T_PROC ("cfe_complete_imports");
   TT (cfe_H, T_ENTER);

   agg_iterate_reverse (imports, sos_Import i)
   {
      sos_Import_List il = i.get_module().get_imports();
      agg_iterate (il, sos_Import indirect_import)
	 cfe_imp_search_or_append (imports, indirect_import);
      agg_iterate_end (il, indirect_import);
   } agg_iterate_reverse_end (imports, i);

   TT (cfe_H, T_LEAVE);
}


EXPORT int cfe_main (int argc, char *argv[])
{  T_INIT ("cfe.out");

   char* infile = NULL;
   int   status = 0;

#define print_version() \
   cout << "OBST version:" << obst_version << "\n"

   if (argc == 2 AND streql (argv[1], "-v"))
      print_version();
   else
   {  for (int i = 1;  i < argc;  ++ i)
      {  if (streql (argv[i], "-E"))
	    cfe_echo_flag = TRUE;
	 else if (streql (argv[i], "-v"))
	    print_version();
	 else if (i == argc-1)
	    infile = argv[argc-1];
	 else
	    status = 1;
      }
      if (status OR !infile)
      {  err_raise (err_USE, err_CFE_USAGE, NULL, FALSE);
	 status = 1;
      }
      else
      {  cfe_yyin = fopen (infile, "r");
	 if (cfe_yyin == 0)
	 {  err_raise (err_USE, err_CFE_NO_FILE, argv[argc-1], FALSE);
	    status = 1;
	 }
	 else
	 {  cfe_compile ();

	    if (err_occurred (err_SYS) + err_occurred (err_USE) != 0)
	       status = -1;
	 }
      }
   }
#ifdef PROFILE
   monitor (0);
#endif

   T_EXIT ();
   return status;
}
