/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module cfe_method                                         Juergen Uhl (ju)
//
// **************************************************************************
// OBST schema compiler front end (cfe)
// **************************************************************************
//
// tracing conventions: see trc_cfe.h
 
#define OBST_IMP_STRINGOP
#include "obst_stdinc.h"

#include "cfe.h"
 
 
LOCAL inline sos_Type_descr cfe_get_void_type()
{  return sos_Type_descr::make (_obst_knl_type (_THE_VOID_TYPE));
}
LOCAL inline sos_Type_descr cfe_get_int_type()
{  return sos_Type_descr::make (_obst_knl_type (_THE_INT_TYPE));
}
LOCAL inline sos_Type_descr cfe_get_container_type()
{  return sos_Type_descr::make (_obst_knl_type (_THE_CNT_TYPE));
}
LOCAL inline sos_Type_descr cfe_get_bool_type()
{  return sos_Type_descr::make (_obst_knl_type (_THE_BOOL_TYPE));
}
LOCAL inline sos_Type_descr cfe_get_eqkind_type()
{  return sos_Type_descr::make (_obst_knl_type (_THE_EQKIND_TYPE));
}
LOCAL inline sos_Type_descr cfe_get_object_type()
{  return sos_Type_descr::make (_obst_knl_type (_THE_OBJECT_TYPE));
}
 
LOCAL sos_Type_descr cfe_SelfTp (sos_Class_type ct)
{  T_PROC ("cfe_SelfTp")
   TT (cfe_L, T_ENTER);
 
   /* for a non-generic class type cfe_SelfTp returns the given ct,
      for generic class types an sos_Generic_instantiation object is
      created representing the root class as an instantiation. */
 
   static sos_Class_type last_ct = sos_Class_type::make (NO_OBJECT);
   static sos_Type_descr last_td;
   if (ct != last_ct)
   {  last_ct = ct;
      if (ct.is_generic_class())
      {  sos_Gen_param_List  gpl = ct.get_formal_gen_params();
	 sos_Type_descr_List tdl = sos_Type_descr_List::create (cfe_cnt);
	 agg_iterate (gpl, sos_Gen_param gp)
	    tdl.append (gp);
	 agg_iterate_end (gpl, gp);
 
	 sos_Generic_instantiation gi
	    = cfe_get_instantiation (ct, ct.get_root_class(), tdl);
 
	 if (gi.get_act_gen_params() != tdl)
	    tdl.destroy();
 
	 last_td = gi;
      }
      else
	 last_td = ct;
   }
   TT (cfe_L, T_LEAVE);
   return last_td;
}
 
 
// ------------------------------------------------------------------------
 
LOCAL void cfe_destroy_method (sos_Method m)
{  T_PROC ("cfe_destroy_method")
   TT (cfe_VL, T_ENTER);
 
   sos_Param_List pl = m.get_params();
   if (pl != NO_OBJECT)
   {  agg_iterate (pl, sos_Param p)
	 p.destroy();
      agg_iterate_end (pl, p);
      pl.destroy();
   }
   m.get_name().destroy();
   m.destroy();
 
   TT (cfe_VL, T_LEAVE);
}
 
LOCAL sos_Method cfe_remove_method(sos_Class_type ct,smg_String nm,sos_Int pcard
)
{  T_PROC ("cfe_remove_method")
   TT (cfe_L, T_ENTER);
 
   sos_String       mname  = nm.make_String (TEMP_CONTAINER);
   sos_Method_table mtable = ct.get_methods();
   sos_Method_List  ml     = mtable[mname];
   sos_Method       m;
   agg_iterate (ml, sos_Method _m)
      sos_Param_List pl = _m.get_params();
      if (pcard == ((pl == NO_OBJECT) ? 0 : pl.card()))
      {  m = _m;
	 ml.remove_at (agg_current_cursor());
	 break;
      }
   agg_iterate_end (ml, _m);
 
   mtable.remove (mname);       // The m.get_name() object might be the key
				// for ml and is hence replaced in any case.
   if (ml.card() == 0)
      ml.destroy();
   else
      mtable.insert (ml.get_nth(1).get_name(), ml);
 
   ml = ct.get_local_methods();
   ml.remove (ml.find (m));
 
   TT (cfe_L, T_LEAVE);
   return m;
}
 
 
// ------------------------------------------------------------------------
 
LOCAL sos_Bool cfe_add_predefined_method (sos_Bool        use_mtable,
					  sos_Class_type  ct,
					  smg_String      n,
					  sos_Bool        is_static,
					  sos_Method_kind kind,
					  sos_Param_List  pl,
					  sos_Type_descr  rt)
{  T_PROC ("cfe_add_predefined_method")
   TT (cfe_L, T_ENTER);
 
   sos_Method_table mt;
   sos_Method_List  ml;
   sos_Bool         name_already_defined;
   sos_Method       m   = sos_Method::create (cfe_cnt);
   sos_String       str = n.make_String(cfe_cnt);
 
   if (use_mtable)
   {  mt = ct.get_methods();
      ml = mt [str];
      name_already_defined = (ml != NO_OBJECT);
   }
   else
      name_already_defined = FALSE;
 
   m.set_name (str);
   m.set_kind (kind);
   m.set_is_abstract ((is_static) ? FALSE
				  : ct.get_is_abstract());
   m.set_is_static (is_static);
   m.set_is_definite (FALSE);
   m.set_is_operator (FALSE);
   m.set_is_generated (TRUE);
   m.set_generated_from (sos_Method::make (NO_OBJECT));
   m.set_params (pl);
   m.set_result_type (rt);
   m.set_impls (sos_Method_impl_List::make (NO_OBJECT));
 
   if (name_already_defined)
   {  // Check, if for a local method an illegal parameter list was provided
      // or more than one local_...-method exists with equal names.
      if (!strncmp (n.make_Cstring (SMG_BORROW), "local_", 6)
	  AND (mt.lookup (m) == NO_OBJECT  OR  ml.card() > 1))
	 cfe_error (err_USE, err_CFE_INVALID_LOCAL_METHOD, str);
 
      cfe_destroy_method (m);
   }
   else
   {  m.set_defined_in (ct);
 
      ct.get_local_methods().append(m);
      if (use_mtable)
	 mt.lookup_or_add (m);
   }
   TT (cfe_L, T_LEAVE);
   return name_already_defined;
}
 
LOCAL inline void cfe_init_add_method (sos_Class_type  ct,
				       smg_String      n,
				       sos_Bool        is_static,
				       sos_Method_kind kind,
				       sos_Param_List  pl,
				       sos_Type_descr  rt)
{  (void)cfe_add_predefined_method (FALSE, ct, n, is_static, kind, pl, rt);
}
 
LOCAL inline sos_Bool cfe_final_add_local_method (sos_Class_type  ct,
						  smg_String      n,
						  sos_Method_kind kind,
						  sos_Param_List  pl,
						  sos_Type_descr  rt)
{  return cfe_add_predefined_method (TRUE, ct, n, TRUE, kind, pl, rt);
}
 
LOCAL void cfe_add_param (sos_Param_List pl, smg_String n,
			  sos_Type_descr td, sos_Expr   de)
{  T_PROC ("cfe_add_param")
   TT (cfe_VL, T_ENTER);
 
   sos_Param p = sos_Param::create (cfe_cnt);
 
   p.set_name (n.make_String (cfe_cnt));
   p.set_type (td);
   p.set_default_expr (de);
   p.set_is_ref (FALSE);
 
   pl.append (p);
 
   TT (cfe_VL, T_LEAVE);
}
 
 
// --------------------------------------------------------------------------
 
EXPORT void cfe_init_methods (sos_Class_type ct)
{  T_PROC ("cfe_init_methods")
   TT (cfe_M, T_ENTER);
 
   ct.set_local_methods (sos_Method_List::create (cfe_cnt, FALSE));
 
   if (NOT ct.is_instantiation())
   {  sos_Param_List pl;
 
      // create
      pl = sos_Param_List::create (cfe_cnt);
      cfe_add_param (pl, "_ct", cfe_get_container_type(),
				sos_Expr::make(NO_OBJECT));
      sos_Param_List tail = ct.get_create_params();
      if (tail != NO_OBJECT)
	 pl += tail;
      cfe_init_add_method (ct, "create", TRUE, sos_PUBLIC, pl, cfe_SelfTp(ct));
 
      // copy
      pl = sos_Param_List::create (cfe_cnt);
      cfe_add_param (pl, "_y", cfe_SelfTp (ct), sos_Expr::make (NO_OBJECT));
      cfe_add_param (pl, "_ct", cfe_get_container_type(),
				sos_Expr::make(NO_OBJECT));
      cfe_init_add_method (ct, "copy", TRUE, sos_PUBLIC, pl, cfe_SelfTp(ct));
 
      // clone
      pl = sos_Param_List::create (cfe_cnt);
      cfe_add_param (pl, "_y", cfe_SelfTp (ct), sos_Expr::make (NO_OBJECT));
      cfe_add_param (pl, "_ct", cfe_get_container_type(),
				sos_Expr::make (NO_OBJECT));
      cfe_init_add_method (ct, "clone", TRUE, sos_PUBLIC, pl, cfe_SelfTp(ct));
 
      // destroy
      cfe_init_add_method (ct, "destroy", FALSE, sos_PUBLIC,
			   sos_Param_List::make (NO_OBJECT),
			   cfe_get_void_type());
      // assign
      pl = sos_Param_List::create (cfe_cnt);
      cfe_add_param (pl,"_y",cfe_get_object_type(),sos_Expr::make (NO_OBJECT));
      cfe_init_add_method (ct, "assign", FALSE, sos_PUBLIC,
			   pl, cfe_get_void_type());
 
      // equal
      pl = sos_Param_List::create (cfe_cnt);
      cfe_add_param (pl,"_y",cfe_get_object_type(),sos_Expr::make (NO_OBJECT));
      sos_Identifier id = sos_Identifier::create (cfe_cnt);
      id.set_id (sos_String::create (cfe_cnt, "EQ_STRONG"));
      cfe_add_param (pl, "_eq", cfe_get_eqkind_type(), id);
      cfe_init_add_method (ct, "equal", FALSE, sos_PUBLIC,
			   pl, cfe_get_bool_type());
 
      // hash_value
      cfe_init_add_method (ct, "hash_value", FALSE, sos_PUBLIC,
			   sos_Param_List::make (NO_OBJECT),
			   cfe_get_int_type());
   }
   TT (cfe_M, T_LEAVE);
}
 
 
EXPORT void cfe_complete_methods (sos_Class_type ct)
{  T_PROC ("cfe_complete_methods")
   TT (cfe_M, T_ENTER);
 
   sos_Param_List pl;
   sos_Bool       assign_defined, equal_defined, hash_value_defined,
		  finalize_defined, initialize_defined,
		  t_ass_defined, t_eq_defined, t_hash_defined,
		  t_fin_defined, t_init_defined;
   sos_Method_table mt = ct.get_methods();
   static sos_String
	  nm_t_ass  = sos_String::create(TEMP_CONTAINER,"total_assign"),
	  nm_t_eql  = sos_String::create(TEMP_CONTAINER,"total_equal"),
	  nm_t_hash = sos_String::create(TEMP_CONTAINER,"total_hash_value"),
	  nm_final  = sos_String::create(TEMP_CONTAINER,"local_finalize"),
	  nm_init   = sos_String::create(TEMP_CONTAINER,"local_initialize"),
	  nm_t_fin  = sos_String::create(TEMP_CONTAINER,"total_finalize"),
	  nm_t_init = sos_String::create(TEMP_CONTAINER,"total_initialize");
 
   t_ass_defined      = (mt[nm_t_ass]  != NO_OBJECT);
   t_eq_defined       = (mt[nm_t_eql]  != NO_OBJECT);
   t_hash_defined     = (mt[nm_t_hash] != NO_OBJECT);
   t_fin_defined      = (mt[nm_t_fin]  != NO_OBJECT);
   t_init_defined     = (mt[nm_t_init] != NO_OBJECT);
   finalize_defined   = (mt[nm_final]  != NO_OBJECT);
   initialize_defined = (mt[nm_init]   != NO_OBJECT);

   if (ct.get_is_abstract())
   {  cfe_destroy_method (cfe_remove_method (ct, "copy", 2));
 
      sos_Param_List pl = ct.get_create_params();
      sos_Method     m  = cfe_remove_method (ct, "create",
					     (pl == NO_OBJECT) ? 1
							       : 1+ pl.card());
      m.get_params().get_nth (1).destroy();
      m.set_params(sos_Param_List::make (NO_OBJECT));
      cfe_destroy_method (m);
   }
   if (ct.get_local_size() == 0)
   {  static sos_String
	     nm_assign = sos_String::create(TEMP_CONTAINER,"local_assign"),
	     nm_equal  = sos_String::create(TEMP_CONTAINER,"local_equal"),
	     nm_hashv  = sos_String::create(TEMP_CONTAINER,"local_hash_value");
 
      assign_defined     = (mt[nm_assign] != NO_OBJECT);
      equal_defined      = (mt[nm_equal ] != NO_OBJECT);
      hash_value_defined = (mt[nm_hashv ] != NO_OBJECT);
   }
   else
   {  // local_assign
      pl = sos_Param_List::create (cfe_cnt);
      cfe_add_param (pl,"_x",cfe_SelfTp (ct),sos_Expr::make (NO_OBJECT));
      cfe_add_param (pl,"_y",cfe_get_object_type(),sos_Expr::make (NO_OBJECT));
      assign_defined
	 = cfe_final_add_local_method (ct, "local_assign", sos_PROTECTED,
				       pl, cfe_get_void_type());
 
      // local_equal
      pl = sos_Param_List::create (cfe_cnt);
      cfe_add_param (pl,"_x",cfe_SelfTp (ct),sos_Expr::make (NO_OBJECT));
      cfe_add_param (pl,"_y",cfe_get_object_type(),sos_Expr::make (NO_OBJECT));
      cfe_add_param (pl,"_eq",cfe_get_eqkind_type(),sos_Expr::make(NO_OBJECT));
      equal_defined
	 = cfe_final_add_local_method (ct, "local_equal", sos_PROTECTED,
				       pl, cfe_get_bool_type());
 
      // local_hash_value
      pl = sos_Param_List::create (cfe_cnt);
      cfe_add_param (pl,"_x",cfe_SelfTp(ct),sos_Expr::make (NO_OBJECT));
      hash_value_defined
	 = cfe_final_add_local_method (ct, "local_hash_value", sos_PROTECTED,
				       pl,cfe_get_int_type());
   }
   if (assign_defined AND NOT equal_defined)
      cfe_error (err_WNG, err_CFE_ASSIGN_WITHOUT_EQUAL);
   if (equal_defined AND NOT hash_value_defined)
      cfe_error (err_WNG, err_CFE_EQUAL_WITHOUT_HASH);
   if (assign_defined && t_ass_defined)
      cfe_error (err_WNG, err_CFE_L_AND_T_ASSIGN);
   if (equal_defined && t_eq_defined)
      cfe_error (err_WNG, err_CFE_L_AND_T_EQUAL);
   if (hash_value_defined && t_hash_defined)
      cfe_error (err_WNG, err_CFE_L_AND_T_HASH);
   if (finalize_defined && t_fin_defined)
      cfe_error (err_WNG, err_CFE_L_AND_T_FINALIZE);
   if (initialize_defined && t_init_defined)
      cfe_error (err_WNG, err_CFE_L_AND_T_INIT);
 
   TT (cfe_M, T_LEAVE);
}
 
 
EXPORT void cfe_append_comp_methods (sos_Class_type  ct,
				     sos_Type_descr  tn,
				     sos_String      name,
				     sos_Expr        init,
				     sos_Method_kind kind,
				     sos_Method_kind set_kind,
				     sos_Bool        is_value,
				     sos_Bool        is_local,
				     sos_Int&        offset)
// Creates a new component method and appends it to the appropriate
// method lists.
{
   T_PROC ("cfe_append_comp_method")
   TT (cfe_M, T_ENTER);
 
   sos_Method     cm  = sos_Method::create (cfe_cnt);
   smg_String     s   = smg_String ("get_") + name;
   sos_String     str = s.make_String (cfe_cnt);
   sos_Comp_descr cd  = sos_Comp_descr::create(cfe_cnt);
 
   cm.set_name (str);
   cm.set_kind (kind);
   cm.set_is_static (FALSE);
   cm.set_is_operator (FALSE);
   cm.set_is_generated (TRUE);
   cm.set_is_abstract (FALSE);
   cm.set_is_definite (FALSE);
   cm.set_generated_from (sos_Method::make (NO_OBJECT));
   cm.set_impls (sos_Method_impl_List::make (NO_OBJECT));
   cm.set_defined_in (ct);
   cm.set_params (sos_Param_List::make (NO_OBJECT));
   cm.set_result_type (tn);
   cm.set_is_set (FALSE);
   cm.set_comp_descr (cd);
   cd.set_name (sos_String::copy (name, cfe_cnt));
   cd.set_init_expr (init);
   cd.set_is_value (is_value);
   cd.set_is_local (is_local);
   cd.set_offset (offset);
   cd.set_get_method (cm);
   ct.get_local_methods().append (cm);
 
   if (is_value)
      cd.set_set_method (sos_Method::make(NO_OBJECT));
   else
   {  cm  = sos_Method::create (cfe_cnt);
      s   = smg_String ("set_") + name;
      str = s.make_String (cfe_cnt);
 
      cd.set_set_method (cm);
 
      cm.set_name (str);
      cm.set_kind (set_kind);
      cm.set_is_static (FALSE);
      cm.set_is_operator (FALSE);
      cm.set_is_generated (TRUE);
      cm.set_is_abstract (FALSE);
      cm.set_is_definite (FALSE);
      cm.set_generated_from (sos_Method::make(NO_OBJECT));
      cm.set_impls (sos_Method_impl_List::make (NO_OBJECT));
      cm.set_defined_in (ct);
      cm.set_result_type (cfe_get_void_type());
      cm.set_is_set (TRUE);
      cm.set_comp_descr (cd);
      ct.get_local_methods().append (cm);
 
      sos_Param pa = sos_Param::create (cfe_cnt);
      pa.set_name (sos_String::make (NO_OBJECT));
      pa.set_type (tn);
      pa.set_default_expr (sos_Expr::make (NO_OBJECT));
      pa.set_is_ref (FALSE);
 
      sos_Param_List pl = sos_Param_List::create (cfe_cnt, FALSE);
      pl.append (pa);
      cm.set_params (pl);
   }
   ct.get_components().append (cd);
 
   sos_Type bt = tn.make_type();
   if (bt.is_scalar())
      offset += bt.get_object_size();
   else
      offset += (is_local) ? SOS_OFFSET_SIZE
			   : SOS_TYPED_ID_SIZE;
 
   TT (cfe_M, T_LEAVE);
}
 
 
EXPORT void cfe_complete_components (sos_Class_type ct)
{  // Adds the component-list of each user-specified superclasse to the
   // components list of the class ct.
 
   T_PROC ("cfe_complete_components")
   TT (cfe_M, T_ENTER);
 
   sos_Comp_descr_List  comp = ct.get_components();
   sos_Super_class_List scl  = ct.get_super_classes();
   agg_iterate(scl, sos_Super_class sc)
   {  sos_Class_type      sct = sos_Class_type::make(sc.make_type());
      sos_Comp_descr_List cdl = sct.get_components();
 
      agg_iterate (cdl, sos_Comp_descr cd)
	 comp.find_or_append (cd);    // comps. inherited over multiple paths
      agg_iterate_end (cdl, cd);      // are inserted only once.
   }
   agg_iterate_end(scl, sc);
 
   TT (cfe_M, T_LEAVE);
}
 
 
// --------------------------------------------------------------------------
 
LOCAL void insert_methods (sos_Method_table mt, sos_Method_List ml)
{  T_PROC ("insert_methods")
   TT (cfe_M, T_ENTER);
 
   agg_iterate (ml, sos_Method m)
   {  sos_Method m1 = mt.lookup_or_add (m);
      if (m1.operator!=(m))     //ATT2.0 QUEERNESS: explicit operator
	 cfe_error (err_USE, err_CFE_INVALID_OVERLOADING, m.get_name());
   }
   agg_iterate_end (ml, m);
 
   TT (cfe_M, T_LEAVE);
}
 
 
LOCAL sos_Bool cfe_dominates (sos_Method m, sos_Method m1)
// class of m derived from class of m1
{
   T_PROC ("cfe_dominates")
   TT (cfe_L, T_ENTER);
 
   // A method does NOT dominate another, if both are defined in
   // instantiations of the same generic class!
 
   sos_Bool result;
   sos_Class_type c  = m.get_defined_in();
   sos_Class_type c1 = m1.get_defined_in();
   sos_Class_type cg = c.get_generic_class();
   
   result = sos_Bool((cg == NO_OBJECT || cg != c1.get_generic_class())
	              && c.is_derived_from_some(c1.get_root_class()));
 
   TT (cfe_L, T_LEAVE);
   return result;
}
 
 
EXPORT void cfe_build_methodtable (sos_Class_type ct)
{  T_PROC ("cfe_build_methodtable")
   TT (cfe_M, T_ENTER);
 
   sos_Method_table mt = sos_Method_table::create (cfe_cnt);
 
   insert_methods (mt, ct.get_local_methods());
 
   sos_Super_class_List super_classes = ct.get_super_classes();
   sos_Bool             not_abstract  = (sos_Bool) NOT ct.get_is_abstract();
 
   agg_iterate (super_classes, sos_Super_class stn)
   {  if (NOT stn.get_is_direct())
	 continue;
 
      sos_Class_type   super_class = sos_Class_type::make(stn.make_type());
      sos_Method_table scmt        = super_class.get_methods();
 
      agg_iterate_association (scmt, sos_String name, sos_Method_List ml)
      {  agg_iterate (ml, sos_Method m)
	 {  if (m.get_kind() != sos_PRIVATE AND NOT m.get_is_static())
	    {  sos_Method m1 = mt.lookup_or_add (m);
	       if (m1.operator==(m))    //ATT2.0 QUEERNESS: explicit operator
	       {  if (not_abstract AND m.get_is_abstract())
		  {  not_abstract = FALSE;
		     ct.set_is_abstract (TRUE);
		  }
	       }
	       else
	       {  // m1 - a different method with the same name as m found
		  // m1 doesn't overload m!
 
		  if (cfe_dominates (m1, m))
		  {                     //ATT2.0 QUEERNESS: explicit operator
		     if (ct.operator== (m1.get_defined_in()))
		     {  if (m1.redefines (m) && m.get_kind() <= m1.get_kind())
			{  if (m.get_is_definite())
			      cfe_error (err_USE, err_CFE_DEFINITE_REDEFINED,
					 m.get_name());
			}
			else // m1 defined in the given class
			     // and no redefinition, or access right restriction
			   if (!m1.redefines (m))
			      cfe_error (err_USE, err_CFE_INVALID_REDEFINITION,
				      m.get_name());
			   else
			      cfe_error (err_USE, err_CFE_INVALID_ACCESS_RESTR,
				      m.get_name());
		     }
		  }
		  else if (cfe_dominates (m, m1))
		  {  mt.replace_or_add (m);
		     if (not_abstract AND m.get_is_abstract())
		     {  not_abstract = FALSE;
			ct.set_is_abstract (TRUE);
		     }
		  }
		  else
		  {  smg_String msg = smg_String (ct.get_name())
				      + "::" + m.get_name();
		     cfe_error (err_USE, err_CFE_AMBIGUOUS_METHODS,
				msg.make_Cstring(SMG_BORROW));
		  }
	       }
	    }
	 }
	 agg_iterate_end (ml, m);
      }
      agg_iterate_association_end (scmt, name, ml);
   }
   agg_iterate_end (super_classes, super_class);
 
   ct.set_methods (mt);
 
   TT (cfe_M, T_LEAVE);
}
 
 
#ifdef CFE_DEBUG
EXPORT void cfe_print_local_methods (sos_Class_type ct)
{  sos_Method_List ml = ct.get_local_methods();
 
   if (ml != NO_OBJECT)
   {  cout << "class " << ct.get_name() << " : \n"
	   << "local methods are:\n";
      agg_iterate (ml, sos_Method m)
	 cfe_print_method (ct, m);
      agg_iterate_end (ml, m);
   }
}
 
EXPORT void cfe_print_method (sos_Class_type ct, sos_Method m)
{  cfe_print_typedescr (m.get_result_type());
   cout << " " << ct.get_name() << "::" << m.get_name() << "\n";
   if (m.get_params() != NO_OBJECT)
   {  sos_Param_List pl = m.get_params();
      agg_iterate (pl, sos_Param p)
	 cout << "\t";
	 cfe_print_typedescr (p.get_type());
	 cout << "\n";
      agg_iterate_end (pl, p);
   }
}
#endif
