/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module cfe_super                                          Juergen Uhl (ju)
//
// **************************************************************************
// OBST schema compiler front end (cfe)
// **************************************************************************
//
// tracing conventions: see trc_cfe.h

#include "cfe.h"

// LOCAL


EXPORT void cfe_check_super_classes (sos_Class_type ct, sos_Super_class_List tnl)
// Checks the correct use of super classes given in list 'tnl'. A class
// specified as super class of any other class must be declared before.
{
   T_PROC ("cfe_check_super_classes");
   TT (cfe_H, T_ENTER);

   sos_Cursor c = tnl.open_cursor();
   for (sos_Bool valid = tnl.is_valid (c); valid;)
   {  sos_Super_class tn = tnl.get(c);
      sos_Bool error = FALSE;
      if (tn.get_super_class().has_type (sos_Gen_param_type))
      {  cfe_error (err_USE, err_CFE_UNDECL_CLASS,
		    sos_Gen_param::make(tn.get_super_class()).get_name());
	 error = TRUE;
      }
      else
      {
	 if (tn.make_type().has_type (sos_Class_type_type))
	 {
	    sos_Class_type sct=sos_Class_type::make(tn.make_type());
	    // check for "class A : A"
	    if (ct.operator==(sct))	//ATT2.0 QUEERNESS: explicit operator
	    {  cfe_error (err_USE, err_CFE_UNDECL_CLASS,
				   tn.make_type().get_name());
	       error = TRUE;
	    }
	    // check, if superclass was forward
	    if (sct.get_methods() == NO_OBJECT)
	    {
	       error = TRUE;
	       cfe_error (err_USE, err_CFE_UNKNOWN_TYPE);
	    }

	    // check create parameters
	    sos_Param_List fpl = sct.get_create_params();
	    sos_Expr_List apl;
	    sos_Int fc, ac;
	    apl = tn.get_create_params();
	    if (apl == NO_OBJECT)
	       ac = 0;
	    else
	       ac = apl.card();
	    if (fpl == NO_OBJECT)
	       fc = 0;
	    else
	       fc = fpl.card();
	    if (fc < ac)
	    {
	       smg_String s = smg_String ("(") + fc + " create params) " 
			      + tn.make_type().get_name();
	       sos_String ss = s.make_String (TEMP_CONTAINER);
	       cfe_error (err_USE, err_CFE_MANY_CREATE_PARAMS, ss);
	       error = TRUE;
	       ss.destroy();
	    }
	    else
	    {  sos_Bool has_defaults = TRUE;
	       for (ac++; ac <= fc; ac++)
	       {  if (fpl.get_nth (ac).get_default_expr() == NO_OBJECT)
		  {  has_defaults = FALSE;
		     break;
		  }
	       }
	       if (NOT has_defaults)
	       {
		  smg_String s = smg_String ("(") + fc + " create params) " 
				 + tn.make_type().get_name();
		  sos_String ss = s.make_String (TEMP_CONTAINER);
		  cfe_error (err_USE, err_CFE_FEW_CREATE_PARAMS, ss);
		  error = TRUE;
		  ss.destroy();
	       }
	    }
	 }
	 else  // superclass was no sos_Class_type
	 {
	    cfe_error (err_USE, err_CFE_CLASS_EXPECTED, 
		       tn.make_type().get_name());
	    error = TRUE;
         }
      }

      if (error)
      {  tnl.remove_at(c);
	 valid = tnl.is_valid(c);
      }
      else
	 valid = tnl.to_succ(c);
   }
   tnl.close_cursor (c);

   if (tnl.card() == 0)  // due to errors superclass list became empty
      // This causes sos_Object to be entered as sc, if ct != sos_Object
      cfe_enter_object_as_sc (ct, tnl);

   TT (cfe_H, T_LEAVE);
}


EXPORT sos_Super_class super_class_in_list (sos_Super_class_List scl,
					   sos_Type_descr tp)
{
   // If the entry looked for is contained in the given superclass list scl, then
   // it is returned. Otherwise NO_OBJECT is the result.

   T_PROC ("super_class_in_list");
   TT (cfe_H, T_ENTER);

   sos_Super_class result = sos_Super_class::make(NO_OBJECT);

   agg_iterate (scl, sos_Super_class sc)
   {
      if (sc.get_super_class().identical(tp))
      {  result = sc;
	 break;
      }
   }
   agg_iterate_end (scl, sc);

   TT (cfe_H, T_LEAVE);
   return result;
}


LOCAL sos_Expr_List formal_params_to_actuals (sos_Param_List fpl)
{
   T_PROC ("formal_params_to_actuals");
   TT (cfe_H, T_ENTER);

   sos_Expr_List result;

   if (fpl == NO_OBJECT)
      result = sos_Expr_List::make (NO_OBJECT);
   else
   {  result = sos_Expr_List::create (cfe_cnt);
      agg_iterate (fpl, sos_Param fp)
      {  sos_Identifier id = sos_Identifier::create (cfe_cnt);
	 id.set_id (fp.get_name());
	 result.append (id);
      }
      agg_iterate_end (fpl, fp);
   }

   TT (cfe_H, T_LEAVE);

   return result;
}


LOCAL sos_Expr_List expand_params (sos_Expr_List original,
				    sos_Param_List fpl,
				    sos_Expr_List apl)
// class A (f1, ..., fm) : ... C (o1, ..., on) ...
// class B : ... A (a1, ..., am) ...
// result : subst fi through ai in oi
{
   T_PROC ("expand_params");
   TT (cfe_H, T_ENTER);

   sos_Expr_List result;

   if (original == NO_OBJECT || fpl == NO_OBJECT)
      result = original;
   else
   {  result = sos_Expr_List::create (cfe_cnt);
      agg_iterate (original, sos_Expr expr)
      {  sos_Expr subst = expr;
	 if (expr.has_type (sos_Identifier_type))
	 {  sos_String id = sos_Identifier::make (expr).get_id();
	    sos_Bool av = apl != NO_OBJECT;
	    sos_Cursor c;
	    if (av)
	    {  c = apl.open_cursor();
	       av = apl.is_valid (c);
	    }

	    agg_iterate (fpl, sos_Param fp)
	    {  sos_String s = fp.get_name();
	       if (s != NO_OBJECT AND s.equal (id))
	       {  if (av)
		     subst = apl.get(c);
		  else
		     subst = fp.get_default_expr();
		  break;
	       }
	       if (av) av = apl.to_succ (c);
	    }
	    agg_iterate_end (fpl, fp);
	    if (apl != NO_OBJECT)
	       apl.close_cursor (c);
	 }
	 result.append (subst);
      }
      agg_iterate_end (original, expr);
   }

   TT (cfe_H, T_LEAVE);

   return result;
}


LOCAL sos_Super_class append_super_class (sos_Super_class_List scl,
			       		  sos_Type_descr ct,
			       		  sos_Expr_List cpl,
			       		  sos_Bool isd)
{  T_PROC ("append_super_class")
   TT (cfe_H, T_ENTER);

   sos_Super_class sc = sos_Super_class::create (cfe_cnt);
   sc.set_super_class (ct);
   sc.set_create_params (cpl);
   sc.set_is_direct (isd);
   scl.append (sc);

   TT (cfe_H, T_LEAVE);
   return sc;
}


LOCAL sos_Expr_List cfe_get_actual_create_params (sos_Super_class which,
						  sos_Super_class context)
{
   /* This function determines from a given sos_Super_class object the
      actual create parameter list for the derived class context.
      Therefore the class type is extracted from <which> and searched for
      in the super_closure of context. The actual corresponding create parameter
      list of context's super_closure is returned. */

   T_PROC ("cfe_get_actual_create_params");
   TT (cfe_H, T_ENTER);

   sos_Expr_List result;

   sos_Class_type which_class = sos_Class_type::make (which.make_type());
   sos_Super_class context_sc = super_class_in_list (
       sos_Class_type::make(context.make_type()).get_super_closure(),
       which_class);

   if (context_sc == NO_OBJECT)
   {  smg_String class_name = which_class.get_name();
      err_raise (err_SYS, err_CFE_INTERNAL,
			  class_name.make_Cstring (SMG_BORROW));
   }
   else
      result = context_sc.get_create_params();

   TT (cfe_H, T_LEAVE);
   return result;
}


LOCAL void cfe_correct_sc_order (sos_Class_type ct)
{
   /* This function establishes an order in the super_classes list of ct.
      The criteria is like in the super_closure list the class hierarchy. */

   T_PROC ("cfe_correct_sc_order");
   TT (cfe_H, T_ENTER);

   sos_Bool order_already_correct = TRUE;
   sos_Int  max_index = 0;
   sos_Super_class_List sclo = ct.get_super_closure();
   sos_Super_class_List scla = ct.get_super_classes();
   // The following mappings associate the super_closure indices with the
   // corresponding sos_Super_class objects from the super_classes list.
   sos_Object_sos_Object_Mapping pos_to_sc =
	sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);

   agg_iterate (scla, sos_Super_class sc)
   {
      /* Now find class beloging to sc in the super_closure. Class is found,
	 since superclosure contains at least all classes contained in the
	 super_classes list. */
      sos_Int i = 0; // index of class sc within super_closure

      agg_iterate (sclo, sos_Super_class s2)
      {
	 i++;
	 if (s2.get_super_class().make_type()==sc.get_super_class().make_type())
	    break;
      }
      agg_iterate_end (sclo, s2);
      // order is correct, if indices are in increasing order
      if (i < max_index)
	 order_already_correct = FALSE;
      max_index = i;
      pos_to_sc.insert (make_sos_Int_object(i), sc);
   }
   agg_iterate_end (scla, sc);

   if (!order_already_correct) // order needs to be corrected
   {
      // Now adapt positions of list elements from the gained knowledge:
      scla.clear(); // first remove all elements from the list.
      for (sos_Int i=1; i <= sclo.card(); i++)
      {
	 sos_Super_class class_to_append = 
	    sos_Super_class::make(pos_to_sc [make_sos_Int_object(i)]);

	 /* NOTE: appending to scla can cause trouble if scla is not located
	    in cfe_cnt. This could happen, if the super_classes list was
	    just copied when doing a generic instantiation. But in this case it
	    can be proposed, that this list must have been in the correct
	    order, because the schema containing the super_classes list has
	    been compiled already. So no write access outside cfe_cnt
	    will occur. */

	 if (class_to_append != NO_OBJECT)  // sc with that index was contained
	    scla.append (class_to_append);  // in the super_classes list:
      }
   }
   pos_to_sc.destroy();   // destroy temporary mapping
  
   TT (cfe_H, T_LEAVE);
}

EXPORT void cfe_set_super_closure (sos_Class_type ct)
   // compute super closure in prefix order using the already
   // computed super closure of the super classes of ct.
   // (i.e. a superclass appears always before any of its subclasses
   //       in the Super_class_List)
{
   T_PROC ("cfe_set_super_closure");
   TT (cfe_H, T_ENTER);

   /* Three temporary mappings exist:
      - sc2direct maps each object from the new constructed superclosure
	to its corresponding direct superclass of the class, for which the
	superclosure is constructed. This info yields the class,
	from which the create-parameters must be taken for initialization.
      - scl_lookup is intended to speed up the search for an element in the
	new constructed superclosure. Key is an sos_Class_type, info is
	the associated sos_Super_class element contained in the new
	superclosure, that has the sos_Class_type as its super_class entity.
      - acp_lists contains for each element of the new superclosure the
	actual create parameter list, which must be expanded later.
	E.g. if we have class C (c) {...};
			class B (b) : C (b) {...};
			class A : B (x);
        then if we construct A's superclosure, for the class C the actual
	parameter list (b) is entered, since this was the initialization
	yielded by B. Note, that this association may be changed during the
	process: If we have instead class A : B (x), C (y);
	then we must change C's actual parameter list from (b) to (y), as
	well as we have to change the direct association from B to A.        */

   sos_Object_sos_Object_Mapping sc2direct = 
	sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);
   sos_Object_sos_Object_Mapping scl_lookup =
	sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);
   sos_Object_sos_Object_Mapping acp_lists =
	sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);
   sos_Param_List       fcp     = ct.get_create_params ();
   sos_Super_class_List scl     = ct.get_super_classes ();
   sos_Super_class_List new_scl = sos_Super_class_List::create (cfe_cnt, FALSE);

   agg_iterate (scl, sos_Super_class sc)
   {
      sos_Class_type ct1 = sos_Class_type::make(sc.make_type());
      sos_Super_class prev_sc = sos_Super_class::make(scl_lookup [ct1]);
      sos_Super_class_List scl1 = ct1.get_super_closure();
      if (prev_sc == NO_OBJECT)
      {
	 agg_iterate (scl1, sos_Super_class sc2)
	 {
	    sos_Class_type ct2 = sos_Class_type::make(sc2.make_type());
	    sos_Super_class class_in_list = sos_Super_class::make(
							scl_lookup [ct2]);
	    if (class_in_list == NO_OBJECT)
	    {				 // ATT2.0 QUEERNESS: explicit operator
	       sos_Bool compare_result = (ct2.operator== (ct1));
	       sos_Super_class inserted_sc =
			append_super_class (new_scl, ct2,
				   	    sos_Expr_List::make(NO_OBJECT),
				   	    compare_result);
				   // is direct iff it is the supplied sup-cl.
	       scl_lookup.insert (ct2, inserted_sc); // insert in speedup map
					     // save actual create parameters
                  // Error in Sun CC Compiler:
		  // The code doesn't work with create_params passed directly 
		  // without a temporary variable. Thanks to [matsj]
               sos_Expr_List acp = sc2.get_create_params(); 
	       acp_lists.insert  (inserted_sc, acp);
	       sc2direct.insert  (inserted_sc, sc); // Associate the class to
					 // append with the direct superclass
	    }
	    else// class already in superclosure, but direct-state may change:
	        // if the found class is marked to be direct, it must be marked
		// to be indirect, since it is not ct1 (ct1 is asserted to be
		// not in the superclosure at this point by the check
		// prev_sc == NO_OBJECT).
	       class_in_list.set_is_direct (FALSE);
	 }
	 agg_iterate_end (scl1, sc2);
      }
      else // superclass was previously inserted in the superclosure
      {
	 if (!prev_sc.get_is_direct())
	 /* the superclass was not direct and only the direct-flag
	    in the super_classes list needs to be changed, since indirect
	    'overrules' direct.
	    NOTE: aviod access outside container through previous check  */
	    if (sc.get_is_direct())
	       sc.set_is_direct (FALSE);

	 sos_Super_class associated = sos_Super_class::make(sc2direct[prev_sc]);
	 // modify the associations of the superclasses from the constructed
	 // superclosure, which are contained in the superclosure of sc and
	 // are initialized by 'associated'
	 agg_iterate (scl1, sos_Super_class ass_sc)
	 {
	    sos_Class_type ass_ct = sos_Class_type::make (ass_sc.make_type());
	    sos_Super_class ass_sc_from_new_scl = 
		sos_Super_class::make(scl_lookup [ass_ct]);
	    if (sos_Super_class::make(sc2direct [ass_sc_from_new_scl])
		.operator== (associated))
	    {   // ATT2.0 QUEERNESS: explicit operator

	       // Associate it with direct sc.
	       sc2direct.insert (ass_sc_from_new_scl, sc);
	       // Determine the actual create parameters of ass_sc in the context
	       // of the direct superclass sc.
	       sos_Expr_List acp = cfe_get_actual_create_params (
					ass_sc_from_new_scl, sc);
	       acp_lists.insert (ass_sc_from_new_scl, acp);
	    }
         } agg_iterate_end (scl1, ass_sc);
      }
   }
   agg_iterate_end (scl, sc);

   // NOTE: the following call to formal_params_to_actuals yields
   // NO_OBJECT in case the formal create parameter list was also NO_OBJECT!
   append_super_class (new_scl, ct, formal_params_to_actuals (fcp), FALSE);
			      // the super_closure contains the class itself!

   /* Now that the super_closure is constructed with the order already
      being correct, the create parameters can be expanded using the
      created mapping sc2direct, in which for each class of the constructed
      superclosure the association with the direct superclass is registered. */
   agg_iterate (new_scl, sos_Super_class sc)
   {			 // ATT2.0 QUEERNESS: explicit operator
      if (sc.make_type().operator!= (ct)) // Don't process current class
      {  // Parameters not yet expanded (is the case for all classes but
	 // the class itself (for which the superclosure is constructed),
	 // since for that the call to formal_params_to_actuals filled them in)
	 sos_Super_class assoc_direct = sos_Super_class::make(sc2direct [sc]);
	 sos_Param_List fcp1 = sos_Class_type::make (
		assoc_direct.make_type()).get_create_params();
	 sos_Expr_List  acp1 = assoc_direct.get_create_params();
	 sos_Expr_List  acp2 = sos_Expr_List::make(acp_lists [sc]);

	 sc.set_create_params (expand_params (acp2, fcp1, acp1));
      }
   } agg_iterate_end (new_scl, sc);

   ct.set_super_closure (new_scl);
   cfe_correct_sc_order (ct); // use superclosure order to sort superclass list

   TT (cfe_H, T_LEAVE);
}
