#line 1 "./src/ext_sync/ext_sync.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : extended syncronization                                          */
/*          all that goes beyond locking (currently optimistic and timestamp)*/
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg, Michael Ranft        DATE: 29.09.1993              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* VERSION: none                                                             */
/* ========================================================================= */

#define OBST_IMP_MEMOP
#include "_obst_config.h"
#include "obst_stdinc.h"
 
#include "obst.h"
#include "obst_progstd.h"

#include "sync_obst.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "sync_trc.h"
#include "ext_sync_obst.h"

#include "trc_tsy.h"

#include "psm_util.h"


/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: ExtSyncObj                                                    */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 29.09.1993              */
/*                                                                           */
/* ========================================================================= */


/* ================================================================ */
/* Method: ExtSyncObj::get                                          */
/*                                                                  */
/* Error: SYNC_OK, SYNC_WRONG_ACCESS, SYNC_TA_ABORTED,              */
/*        SYNC_IMPLTA_ABORTED                                       */
/* ================================================================ */

sos_Bool _ExtSyncObj::get (const sos_Typed_id&_OBSThis,sos_Access_mode am, sos_Sync_mode sm)

const{ 
  T_PROC("ExtSyncObj::get");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TAM(am); TSM(sm));
  TSOBJ(ExtSyncObj::make(_OBSThis,this));

  SyncCont         syct;
  Transaction      ta; 
  LockingTA        tplta;
  sos_HostProc_Id  hpid;
  sos_HostProc_Id  tapid;
  sos_Open_result  result;
  sos_Bool         access_delay = TRUE;
  sos_Bool         okay;
  sos_Bool         impl_firstcont = FALSE;
  ta_Access_mode   ct_am;

  // control parameter 
  if ((am != READING) && (am != WRITING))
    { set_error(_OBSThis,SYNC_WRONG_ACCESS);
      TT(tsy_H, TXT("SYNC_WRONG_ACCESS"); T_LEAVE);
      return FALSE; }

  while (access_delay)
    {     
      access_delay = FALSE; 
       
      syct = _SyncCont::get_root();

      // look for transaction 
      hpid = syct.get_HP();
      ta  = syct.get_ta_tab()[hpid];
      TT(tsy_VL, TI(hpid));
      TTA(ta);

      // is transaction intern aborted ?
      if (VALID(ta) && (ta.get_status() == ABORTED))
        { // nothing to do
          set_error(_OBSThis,SYNC_OK);
          TT(tsy_H, TXT("nothing to do -> TRUE"); T_LEAVE);
          return TRUE; }
 

      if ((get_awaking(_OBSThis)) &&
          (INVALID(ta) || (ta.get_type() != OPTIMISTIC)))
        { // another ta was waiting to access this SyncObj and to this ta a
          // message was sent that it can access the SyncObj. But as long
          // as awaking is set, the ta has not accessed the SyncObj, it has not
          // awake. The other ta, which has accessed the SyncObj before has 
          // probably released all locks, so that to keep evrything in order
          // we have to wait, till the other ta has accessed the Obj, 
          // under the assumption that the process was not aborted.
          
          if (!get_queue(_OBSThis).is_empty())
            {    
              tapid = get_queue(_OBSThis).get_nth(1).get_ta().get_hostproc_id();
              okay  = syct.process_check(tapid);
              if (!okay) 
                 syct.process_remove(tapid);
              else
              { // close Sync_Container and open it to give the awaking ta
                // the possibility to access the Sync_Container
                
                SYNC_CONTAINER.close();
      
                result=SYNC_CONTAINER.open(WRITING, WAITING);
                if (result != OPENED)
                   err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE);

                access_delay = TRUE;
              } 
            }
            else
              set_awaking(_OBSThis,FALSE); 
        } 
    }  

  // if no transaction, create one implicit
  if INVALID(ta) 
    { TT(tsy_L, TXT("create implicit TA"));
      ta = ExtLockingTA::create(SYNC_CONTAINER);
      if INVALID(ta) 
        { TT(tsy_H, TXT("SYNC_NO_CREATE_IMPLICIT_TA -> FALSE"); T_LEAVE);
          err_raise(err_SYS, err_SYNC_NO_CREATE_IMPLICIT_TA, ERR, FALSE); 
        }
      ta.set_implicit(TRUE);
      impl_firstcont = TRUE;
      TTA(ta);
    }
 

  // container already in set of transaction ?
  ct_am = ta.is_accessed (ExtSyncObj::make(_OBSThis,this));
  TT(tsy_VL, TXT("CT:"); TTM(ct_am); TXT("AM:"); TAM(am));

  // change to higher access
  if ((ct_am == READ_AC) && (am == WRITING))
    {
      if (ta.get_type() == LOCKING)
        { okay = tpl_writelock(_OBSThis,LockingTA::make(ta), sm);
          if (!okay)
            { TT(tsy_H, TXT("FALSE"); T_LEAVE); 
              return FALSE; } 
        }     

      // move from readset to writeset
      ta.get_SO (ExtSyncObj::make(_OBSThis,this), am);
      set_error(_OBSThis,ta.get_error());
      TT(tsy_H, TB(get_error(_OBSThis)==SYNC_OK); T_LEAVE);
      return sos_Bool(get_error(_OBSThis) == SYNC_OK);
    }

      
  // first access to container by transaction
  if ((ct_am == NO_AC) && ((am == READING) || (am == WRITING)))
    { // call different methods for syncronization
      switch (ta.get_type()) 
        { 
          case LOCKING:
              { if (am == READING) 
                  okay = tpl_readlock(_OBSThis,LockingTA::make(ta), sm);
                 else
                  okay = tpl_writelock(_OBSThis,LockingTA::make(ta), sm); 
                break; }
          case TIMESTAMP:
              { okay = tso_read(_OBSThis,TimestampTA::make(ta), sm);
                break; }
          case OPTIMISTIC:
              { if (am == READING)
                  opt_read(_OBSThis,OptimisticTA::make(ta));
                 else
                  opt_write(_OBSThis,OptimisticTA::make(ta));
                okay = TRUE; 
                break; }
        }


      // if read failed, return
      if (!okay) 
        { // implict transaction objects have to be destroyed if the 
          // transaction was intern aborted or if the implict transaction just
          // has been started at the beginning of this method and the SyncObj,
          // which should be opened TESTING, failed to open.
          if ((ta.get_implicit()) && 
             ((impl_firstcont) || (get_error(_OBSThis) == SYNC_IMPLTA_ABORTED)))
            { TT(tsy_L, TXT("Implicit TA intern aborted -> destroy"));
              tplta = LockingTA::make(ta);
              tplta.destroy(); }
          TT(tsy_H, TXT("FALSE"); T_LEAVE);
          return FALSE; } 

      ta.get_SO (ExtSyncObj::make(_OBSThis,this), am);
      set_error(_OBSThis,ta.get_error());
      TT(tsy_H, TB(get_error(_OBSThis) == SYNC_OK); T_LEAVE);
      return sos_Bool(get_error(_OBSThis) == SYNC_OK);       
    }      



  // the other cases: ct_am = READ_AC && am == READING
  //                  ct_am = WRITE_AC && (am == READING || am == WRITING)

  // reopen if already released
  ta.get_SO(ExtSyncObj::make(_OBSThis,this), am);
  if (ta.get_error() != SYNC_OK)
    { set_error(_OBSThis,ta.get_error());
      TT(tsy_H, TXT("FALSE"); T_LEAVE);
      return FALSE; }   


  // access is already given
  set_error(_OBSThis,SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return TRUE;      
}   

/* ================================================================ */
/* Method: ExtSyncObj::release()                                    */
/*                                                                  */
/* Error: SYNC_OK, SYNC_NO_GET, SYNC_TA_ABORTED (ts),               */
/* ================================================================ */

void _ExtSyncObj::release (const sos_Typed_id&_OBSThis)

const{ 
  T_PROC("ExtSyncObj::release");
  TT(tsy_H, T_ENTER);
  TSOBJ(ExtSyncObj::make(_OBSThis,this));

  SyncCont        syct;
  Transaction     ta;
  LockingTA       tplta; 
  sos_HostProc_Id hpid;
  sos_HostProc_Id tapid;
  sos_Open_result result;
  sos_Bool        access_delay = TRUE;
  sos_Bool        okay;
  ta_Access_mode  ct_am;


  while (access_delay)
    { 
      access_delay = FALSE;

      syct = _SyncCont::get_root();

      // look for transaction 
      hpid = syct.get_HP();
      ta  = syct.get_ta_tab()[hpid];  
      TT(tsy_VL, TI(hpid));
      TTA(ta);

      // if no transaction, no container open
      if INVALID(ta) 
        { set_error(_OBSThis,SYNC_NOT_GET);
          TT(tsy_H, TXT("SYNC_NOT_GET"); T_LEAVE);
          return; }


      // is transaction intern aborted ?
      if (ta.get_status() == ABORTED)
        { // nothing to do
          set_error(_OBSThis,SYNC_OK);
          TT(tsy_H, TXT("intern aborted - nothing to do"); T_LEAVE);
          return; }


      if ((get_awaking(_OBSThis)) &&
          (ta.get_type() != OPTIMISTIC))
        { // another ta was waiting to access this SyncObj and to this ta a
          // message was sent that it can access the SyncObj. But as long
          // as awaking is set, the ta has not accessed the SyncObj, it has not
         // awake. The other ta, which has accessed the SyncObj before has
          // probably released all locks, so that to keep evrything in order
          // we have to wait, till the other ta has accessed the Obj,
          // under the assumption that the process was not aborted.
         
          if (!get_queue(_OBSThis).is_empty())
            {
              tapid = get_queue(_OBSThis).get_nth(1).get_ta().get_hostproc_id();
              okay  = syct.process_check(tapid);
              if (!okay)
                 syct.process_remove(tapid);
              else
              { // close Sync_Container and open it to give the awaking ta
                // the possibility to access the Sync_Container
 
                SYNC_CONTAINER.close();
 
                result=SYNC_CONTAINER.open(WRITING, WAITING);
                if (result != OPENED)
                   err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE);
 
                access_delay = TRUE;
              }
            }
            else
              set_awaking(_OBSThis,FALSE);
        }
    } 

  // get access mode
  ct_am = ta.is_accessed(ExtSyncObj::make(_OBSThis,this));
  TT(tsy_L, TXT("CT:"); TTM(ct_am));
  if (ct_am == NO_AC)
    { set_error(_OBSThis,SYNC_NOT_GET);
      TT(tsy_H, TXT("SYNC_NOT_GET"); T_LEAVE);
      return; }


  // call different methods for syncronisation
  if ((ta.get_type() == TIMESTAMP) && (ct_am == WRITE_AC) &&
      (!ta.is_released(ExtSyncObj::make(_OBSThis,this))))
    { // write for Timestamp
      tso_write(_OBSThis,TimestampTA::make(ta));
      if (get_error(_OBSThis) != SYNC_OK)
        { TT(tsy_H, T_LEAVE);
          return; }
    } 

  // remove from openset
  ta.release_SO(ExtSyncObj::make(_OBSThis,this));
  if (ta.get_error() != SYNC_OK)
    { set_error(_OBSThis,ta.get_error());
      TT(tsy_H, T_LEAVE);
      return; }


  // commit implicit transactions
  // !! beware of  _TA aborted implicitly somewhere before 
  TT(tsy_VL, TI(ta.get_openset().card()));
  if (ta.get_implicit())
    { if (ta.get_openset().is_empty())
	{ tplta = LockingTA::make(ta);
	  tplta.commit();
	  tplta.destroy();
	}
      else
      if (!ta.get_writeset().is_empty())
	{ sos_SOSet helpset = sos_SOSet::create(TEMP_CONTAINER);
          helpset += ta.get_openset();
          helpset *= ta.get_writeset();
          if (helpset.is_empty())
            { tplta = LockingTA::make(ta);
              tplta.commit();
            } 
        }
    }
  

  // end release
  set_error(_OBSThis,SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}  

