/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module genCC_back                                      Michael Gravenhorst 
//
// **************************************************************************
// OBST interface generator / C++ back end
// **************************************************************************
//
// tracing conventions: see trc_genCC.h

#define OBST_IMP_STREAM
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "obst.h"
#include "smg.h"
#include "genCC_err.h"
#include "trc_genCC.h"
#include "cci_use.h"
#include "dir_use.h"

#include "genCC.h"


// --------  string constants  ---------------------------------------------

char* _ID	       = "_";
char* NL               = ";\n";

char* USE_H_SUFFIX     = "_use.h";
char* IMPL_H_SUFFIX    = "_obst.h";
char* EXT_H_SUFFIX     = "_ext.h";
char* IMPL_C_SUFFIX    = "_obst.C";

char* make_TYPE_OBJ    = "_obst_type_object";
char* cvt_from_EXTERN  = "_obst_object_from_extern";
char* cvt_to_EXTERN    = "_obst_extern_from_object";
char* cvt_from_LOCAL   = "_obst_make_local_typed_id";
char* cvt_to_LOCAL     = "_obst_local_offset";

char* get_IMPL_OBJ     = "_cci_get_impl_obj(";
char* set_IMPL_OBJ     = "_cci_enter_impl_obj(";
char* set_STRING_FCT   = "cci_Schema_impl::enter_string_io(";
char* set_EXTCVT_FCT   = "cci_Schema_impl::enter_extern_cvt(";

char* nm_UNIV_PREF     = "_univ_";
char* nm_GEN_UNIV_PREF = "_cci_univ_";
char* set_UNIV_FCT     = "_cci_enter_fun(";

char* nm_ROOT	       = "obst_Root_class";
char* nm_IMPL_ROOT     = "_obst_Root_class";
char* nm_SCALAR_ROOT   = "sos_Scalar_object";

char* IMPL_comp	       = "_impl";
char* call_IMPL	       = "_impl->";

char* THIS_comp        = "_self";
char* THIS_param       = "_OBSThis";
char* THIS_param_type  = "const sos_Typed_id&";
char* THIS_param_decl  = "const sos_Typed_id&_OBSThis";
char* make_THIS        = "::make(_OBSThis,this)";

char* OBJ_container    = ".container()";
char* OBJ_offset       = ".offset()";
char* OBJ_typed_id     = "._typed_id()";
char* OBJ_type_ID      = "._type_id()";

char* INT1	       = "sos_Char";
char* INT2	       = "sos_Short";
char* INT4	       = "sos_Int";


// --------  backend: forward declarations  ---------------------------------

LOCAL void       back_univ_method ();
LOCAL smg_String convert_to_class (smg_String, smg_String);

// --------  backend: global variables  -------------------------------------

LOCAL  filebuf fb_impl_h, fb_use_h, fb_impl_c;
LOCAL ostream os_impl_h (&fb_impl_h),
	      os_use_h  (&fb_use_h),
	      os_impl_c (&fb_impl_c);

LOCAL int pass = 0;


// ======== schema handling =================================================

LOCAL smg_String nm_schema;
LOCAL smg_String inits_level0;
LOCAL smg_String inits_level1;
LOCAL smg_String inits_level2;
LOCAL int	 tpcounter = 0;
LOCAL int	 tpobjcounter = 0;
#ifdef BOOT
LOCAL sos_Bool   init_mta_level1;
LOCAL sos_Bool   is_kernel_schema;
#endif
LOCAL sos_Bool   has_external_defs;


EXPORT void back_schema_start (sos_String schema_name, sos_Bool external_defs)
{  T_PROC ("back_schema_start")
   TT (genCC_M, T_ENTER);

   nm_schema         = schema_name;
   has_external_defs = external_defs;

   smg_String nm_use_h  = nm_schema + USE_H_SUFFIX;
   smg_String nm_impl_h = nm_schema + IMPL_H_SUFFIX;
   smg_String nm_impl_c = nm_schema + IMPL_C_SUFFIX;

   if (fb_use_h.open (nm_use_h.make_Cstring  (SMG_BORROW), output) == 0 || 
       fb_impl_h.open(nm_impl_h.make_Cstring (SMG_BORROW), output) == 0 ||
       fb_impl_c.open(nm_impl_c.make_Cstring (SMG_BORROW), output) == 0)
   {  err_raise (err_USE, err_GEN_OUTPUT_ERROR, NULL, FALSE);
      return;
   }
   os_use_h << "#ifndef " << nm_schema << "_USE_H\n"
	    << "#define " << nm_schema << "_USE_H 1\n"
	    << FIXED_NAME("#include \"obst_config.h\"\n");

   os_impl_h << "#ifndef " << nm_schema << "_IMPL_H\n"
	     << "#define " << nm_schema << "_IMPL_H 1\n"
	     GCC_SPECIFIC(
	     << "#ifdef __GNUG__\n#pragma interface\n#endif\n")
	     << "#include \"" << nm_schema << USE_H_SUFFIX << "\"\n";

   os_impl_c GCC_SPECIFIC(
	     << "#ifdef __GNUG__\n#pragma implementation\n#endif\n")
      	     << "#include \"" << nm_schema << IMPL_H_SUFFIX << "\"\n"
	     << FIXED_NAME("#include \"obst_err.h\"\n")
	     << "void**_"    << nm_schema << "_obj_;\n"
	     << "sos_Id*_"   << nm_schema << "__tp_;\n"
	     << "sos_Type*_" << nm_schema <<  "_tp_;\n";

#ifdef BOOT
   init_mta_level1  = (sos_Bool)(! nm_schema.equal(FIXED_SCHEMA("mta")));
   is_kernel_schema = (sos_Bool)nm_schema.equal(FIXED_SCHEMA("knl"));
#endif
   TT (genCC_M, T_LEAVE);
}

EXPORT void back_schema_end ()
{  T_PROC ("back_schema_end")
   TT (genCC_M, T_ENTER);

   smg_String tparray_decl,objarray_decl;
   if (tpcounter)
      tparray_decl = smg_String("  static sos_Id ___tp[") + tpcounter + "];\n"
      		     + "  static sos_Type __tp[" + tpcounter + "];\n"
		     + "  _" + nm_schema + "__tp_=___tp;\n"
		     + "  _" + nm_schema + "_tp_=__tp;\n";
   if (tpobjcounter)
      objarray_decl = smg_String("  static void*__obj[") + tpobjcounter + "];\n"
	 	      + "  _" + nm_schema + "_obj_=__obj;\n";
 
   os_impl_c << "struct _obst_" << nm_schema << "_dummy\n"
             << "{  _obst_" << nm_schema << "_dummy(){_"
                                         << nm_schema << "_init_obst(999);}\n"
             << "  ~_obst_" << nm_schema << "_dummy(){}\n};\n"
             << "static _obst_" << nm_schema << "_dummy _obst_init_dummy;\n"
#ifdef BOOT
             << "extern \"C\" void _mta_init_obst(int)" << NL
	     << "void _" << nm_schema << "_init_obst"
	     		 << "(int wanted_level){\n"
	     << "  static int init_level=-1" << NL
	     << tparray_decl
	     << objarray_decl
	     << "  if (init_level<0) {\n  ++init_level" << NL
#else
             << FIXED_FCT("extern void _obst_initcore()") << NL
	     << "void _" << FIXED_NAME(nm_schema << "_init_obst(int){\n")
	     << "  static int initialized=0" << NL
	     << tparray_decl
	     << objarray_decl
	     << "  if (initialized) return" << NL 
             << FIXED_FCT("  _obst_initcore()") << NL
#endif
	     << inits_level0
#ifdef BOOT
	     << (is_kernel_schema ? FIXED_FCT("  _obst_basic_init();\n")
				  : "")
	     << "  }\n  if(init_level>=wanted_level)return" << NL
	     << "  if(init_level<1){++init_level" << NL
#endif
             << inits_level1
#ifdef BOOT
             << "  }\n  if(init_level<2&&init_level<wanted_level){"
	     << "++init_level" << NL
	     << (init_mta_level1 ? FIXED_FCT("  _mta_init_obst(1);\n")
			         : "")
#endif
             << inits_level2
#ifdef BOOT
      	     << "  }\n"
#endif
             << "}\n";

   os_use_h  << "extern \"C\" void _" << nm_schema << "_init_obst(int)" << NL;

   os_use_h  << "#endif\n";
   os_impl_h << "#endif\n";

   fb_use_h.close();
   fb_impl_h.close();
   fb_impl_c.close();

   TT (genCC_M, T_LEAVE);
}

// -------- import handling -------------------------------------------------

EXPORT void back_import (smg_String schema_name, sos_Import_mode mode)
{  T_PROC ("back_import")
   TT (genCC_M, T_ENTER);

   sos_Bool is_kernel_schema = (sos_Bool)schema_name.equal ("knl");

   // AT&T-2.1 (or maybe the SunOS4.1.3 preprocessor employed by this
   // compiler) fails with spurious error messages if the kernel schema
   // is not included in first place.
   static sos_Bool first_call = TRUE;
   if (first_call)
   {  first_call = FALSE;
      ATT_SPECIFIC(
      os_use_h << "#include \"knl" << USE_H_SUFFIX << "\"\n";)
   }
   if (mode != sos_IMP_NONE AND NOT is_kernel_schema)
      os_use_h << "#include \"" << schema_name << USE_H_SUFFIX << "\"\n";

   // The internal knl interface is always included as sos_IMP_SUPERCLASS,
   // since sos_Object is implicit superclass of all classes.
   if (mode >= sos_IMP_GENERIC OR is_kernel_schema)
      os_impl_h << "#include \"" << schema_name << IMPL_H_SUFFIX << "\"\n";

   if (mode != sos_IMP_NONE)
   {
#ifdef BOOT
      smg_String sminit = smg_String("  _") + schema_name + "_init_obst(";
      inits_level1 += sminit + "0)" + NL;
      inits_level2 += sminit + "1)" + NL;
      if (schema_name.equal (FIXED_SCHEMA("mta")))
	 init_mta_level1 = FALSE;
#else
      sos_Bool is_bootschema = FALSE;
      for (char** snm = _obst_bootschemas;  *snm;  ++ snm)
	if (schema_name.equal(*snm))
	{  is_bootschema = TRUE;
	   break;
	}

      if (NOT is_bootschema)
         inits_level1 += smg_String("  _")
		      +  schema_name + "_init_obst(999)" + NL;
#endif
   }
   TT (genCC_M, T_LEAVE);
}


// ======== type-handling ===================================================

LOCAL smg_String   nm_type;
LOCAL smg_String   impl_nm_type;
LOCAL smg_String   type_nm_type;
LOCAL sos_Typed_id tpid_type;

EXPORT sos_Bool back_types_start ()
{  T_PROC ("back_types_start")
   TT (genCC_M, T_ENTER);

   sos_Bool loop_over_types = (sos_Bool)(++ pass <= 2);

   if (pass == 1)
   {  if (gen_inline)
         os_use_h << FIXED_SCHEMA("#include \"cci") << USE_H_SUFFIX << "\"\n";
      else
         os_impl_c << FIXED_SCHEMA("#include \"cci") << USE_H_SUFFIX << "\"\n";
   }
   else if (pass == 2)
   {  if (has_external_defs)
         os_use_h << "#include \"" << nm_schema << EXT_H_SUFFIX << "\"\n";

      os_impl_h << "extern void**_"    << nm_schema << "_obj_;\n";
      os_use_h  << "extern sos_Id*_"   << nm_schema << "__tp_;\n"
	        << "extern sos_Type*_" << nm_schema <<  "_tp_;\n";
   }
   TT (genCC_M, T_LEAVE);
   return loop_over_types;
}

// -------- type_handling: type constants -----------------------------------

LOCAL void back_type_consts (smg_String type_alias)
{  T_PROC ("back_type_consts")
   TT (genCC_M, T_ENTER);

   if (type_alias.length())
      os_use_h << "#define " << nm_type << "_type " << type_alias << "_type\n";

   else
   {  smg_String tpconst_nm  = nm_type + "_type";
      smg_String tpconst_idx = smg_String("[") + tpcounter + "]";

      ++ tpcounter;
       
      os_use_h << "#define " << _ID << tpconst_nm << " _"
	    		     << nm_schema << "__tp_" << tpconst_idx << "\n";

      os_use_h << "#define " << tpconst_nm << " _"
	       		     << nm_schema << "_tp_" << tpconst_idx << "\n";
      
      inits_level0 += smg_String("  ___tp") + tpconst_idx
	           +  FIXED_FCT("=sos_Id::make(sos_Container::make(")
	           +  (sos_Int)tpid_type.container() + "),"
	           +  tpid_type.offset() + ")" + NL;
      inits_level2 += smg_String("  __tp") + tpconst_idx + "="
	 	   +  make_TYPE_OBJ + "(" + _ID +  tpconst_nm + ","
	       	   +  _ID + type_nm_type + ")" + NL;
   }
   TT (genCC_M, T_LEAVE);
}

// -------- type handling: scalar types -------------------------------------

LOCAL void back_scalar (sos_Bool is_extern)
{  T_PROC ("back_scalar")
   TT (genCC_M, T_ENTER);

   smg_String typedcl;
   FLOAT_PATCH(
   if (nm_type.equal("sos_Float"))
      typedcl = smg_String("OBST_") + "FLOATMAKEDECL(" + nm_type + ")";
   else
   )  typedcl = smg_String("const ") + nm_type;

   os_use_h  << FIXED_TYPE("sos_Object") << " make_" << nm_type << "_object("
	     << typedcl << ")" << NL;

   os_impl_c << FIXED_TYPE("sos_Object") << " make_" << nm_type << "_object("
	     << typedcl << " _x) {\n"
	     << "return " << cvt_from_EXTERN << "((void*)&_x,bcopy_from_" 
             << nm_type << "," << impl_nm_type << "_type);\n}\n";

   if (is_extern)
      os_impl_h << FIXED_TYPE("sos_Cstring make_string_from_") << nm_type 
		<< FIXED_TYPE("_object(const sos_Object&)") << NL
		<< FIXED_TYPE("sos_Object make_") << nm_type 
		<< FIXED_TYPE("_object_from_string(const sos_Cstring)") << NL;

   os_use_h  << nm_type << " make_" << nm_type
			<< "(" << FIXED_TYPE("const sos_Object&") << ");\n";
   os_impl_c << nm_type << " make_" << nm_type
			<< "(" << FIXED_TYPE("const sos_Object&") << "_y) {\n"
	     << nm_type << " _x;\n"
	     << cvt_to_EXTERN << "((void*)&_x,_y,bcopy_to_" << nm_type << ","
	     << impl_nm_type << "_type);\nreturn _x;\n}\n";

   back_type_consts("");

   TT (genCC_M, T_LEAVE);    
}

// -------- type handling: enumerations -------------------------------------

LOCAL sos_Bool first_literal;

EXPORT sos_Bool back_enum_start (smg_String   type_name, 
                                 sos_Typed_id typed_id,
                                 int          type_size)
{  T_PROC ("back_enum_start")
   TT (genCC_M, T_ENTER);

   nm_type       = type_name;
   impl_nm_type  = smg_String(_ID) + nm_type;
   type_nm_type  = FIXED_TYPE("sos_Enum_type_type");
   tpid_type     = typed_id;

   first_literal = TRUE;

   if (pass == 2)
   {   smg_String cpp_tp;
       if (type_size == 1)
          cpp_tp = INT1;
       else
       {  err_assert (type_size == 2, "back_enum_start");
          cpp_tp = INT2;
       }
       os_use_h << "inline void bcopy_from_" << nm_type 
                << FIXED_TYPE("(void*e,void*c)\n")
                << "{*(" << cpp_tp << "*)c=(" << cpp_tp << ")*(" << nm_type 
                << "*)e;}\n"
                << "inline void bcopy_to_"   << nm_type 
                << FIXED_TYPE("(void*e,void*c)\n")
                << "{*(" << nm_type  << "*)e=(" << nm_type  
                << ")*(" << cpp_tp << "*)c;}\n";
       back_scalar (FALSE);
   }
   TT (genCC_M, T_LEAVE);
   return (sos_Bool)(pass == 1);
}

EXPORT void back_enum_literal (smg_String literal)
{  T_PROC ("back_enum_literal")
   TT (genCC_M, T_ENTER);

   if (first_literal)
   {  first_literal = FALSE;
      os_use_h << "enum " << nm_type << " { ";
   }
   else 
      os_use_h << ",";
   os_use_h << literal;

   TT (genCC_M, T_LEAVE);
}

EXPORT void back_enum_end ()
{  T_PROC ("back_enum_end")
   TT (genCC_M, T_ENTER);

   os_use_h << " };\n";

   TT (genCC_M, T_LEAVE);
}


// -------- type handling: typedefs  ----------------------------------------

EXPORT sos_Bool back_typedef_start (smg_String   type_name, 
                                    sos_Typed_id typed_id)
{  T_PROC ("back_typedef_start")
   TT (genCC_M, T_ENTER);

   nm_type      = type_name;
   impl_nm_type = smg_String(_ID) + nm_type;
   tpid_type 	= typed_id;

   TT (genCC_M, T_LEAVE);
   return (sos_Bool)(pass == 1);
}

EXPORT void back_typedef_end (sos_Type   named_type,
			      smg_String nm_named_type,
                              sos_Bool   type_is_Scalar)
{  T_PROC ("back_typedef_end")
   TT (genCC_M, T_ENTER);

#ifdef BOOT
   if (named_type.has_type (sos_Class_type_type))
      type_nm_type = FIXED_TYPE("sos_Class_type_type");
   else if (named_type.has_type (sos_Enum_type_type))
      type_nm_type = FIXED_TYPE("sos_Enum_type_type");
   else if (named_type.has_type (sos_Extern_type_type))
      type_nm_type = FIXED_TYPE("sos_Extern_type_type");
   else if (named_type.has_type (sos_Union_type_type))
      type_nm_type = FIXED_TYPE("sos_Union_type_type");
   else
      err_assert (FALSE, "back_typedef_end - unexpected type");
#else
   type_nm_type = smg_String(named_type.type().get_name()) + "_type";
#endif

   os_use_h << "typedef ";
   if (NOT type_is_Scalar)
      os_use_h << "class ";

   os_use_h << nm_named_type << " " << nm_type << ";\n";

   if (type_is_Scalar)
   {  os_use_h << "#define make_" << nm_type << "_object make_" 
               << nm_named_type << "_object\n";
      os_use_h << "#define make_" << nm_type << " make_" 
               << nm_named_type << "\n";
   }
#if COMPLICATED_ARGUMENT_BUG
   else
      os_use_h << "#define _"
	       << nm_type << "_tmp _" << nm_named_type << "_tmp\n";
#endif
   back_type_consts(nm_named_type);

   TT (genCC_M, T_LEAVE);
}


//  -------- type handling: external types  ----------------------------------

EXPORT sos_Bool back_extern_start (smg_String   type_name,   
                                   sos_Typed_id typed_id)
{  T_PROC ("back_extern_start") 
   TT (genCC_M, T_ENTER); 

   sos_Bool full_decl;

   if (pass == 2)
   {  nm_type      = type_name;
      impl_nm_type = smg_String(_ID) + nm_type;
      type_nm_type = FIXED_TYPE("sos_Extern_type_type");
      tpid_type    = typed_id;

      if (full_decl = (sos_Bool)(NOT type_name.equal (FIXED_TYPE("void"))))
         back_scalar (TRUE);
      else
         back_type_consts("");
   }
   else
      full_decl = FALSE;

   TT (genCC_M, T_LEAVE);
   return full_decl;
}

EXPORT void back_scalar_superclass (smg_String nm_superclass)
{  T_PROC ("back_scalar_superclass")
   TT (genCC_M, T_ENTER);

   SCALAR_EXCEPTION(
   inits_level1 += smg_String("  ") + set_IMPL_OBJ 
            	+  _ID + nm_superclass + "_type,"
            	+  impl_nm_type + "_type,("
            	+  _ID + nm_superclass + "*)"
            	+  _ID + nm_SCALAR_ROOT + "_OBJ_)" + NL;)

   TT (genCC_M, T_LEAVE);
}

EXPORT void back_extern_end ()
{  T_PROC ("back_extern_end")
   TT (genCC_M, T_ENTER);

   inits_level1 += smg_String("  ") + set_STRING_FCT
            	+  impl_nm_type + FIXED_TYPE("_type,(cci_IO_fun)make_")
		+  nm_type + "_object_from_string,"
	    	+  FIXED_TYPE("(cci_IO_fun)make_string_from_")
            	+  nm_type + "_object)"
	    	+  NL
            	+  "  " + set_EXTCVT_FCT + impl_nm_type
	    	+  FIXED_TYPE("_type,(cci_Cvt_fun)bcopy_from_") + nm_type
	    	+  FIXED_TYPE(",(cci_Cvt_fun)bcopy_to_") + nm_type + ")" + NL;

   TT (genCC_M, T_LEAVE);
}

// -------- type handling: unions -------------------------------------------

EXPORT sos_Bool back_union_start (smg_String   type_name, 
                                  sos_Typed_id typed_id)
{  T_PROC ("back_union_start") 
   TT (genCC_M, T_ENTER); 

   if (pass == 2)
   {  nm_type      = type_name;
      impl_nm_type = smg_String(_ID) + nm_type;
      type_nm_type = FIXED_TYPE("sos_Union_type_type");
      tpid_type    = typed_id;

      os_use_h  << "class " << nm_type
		            << ":public " << FIXED_TYPE("sos_Object") 
                            << "\n{\npublic:\n"
                << "   " << nm_type << "(" << THIS_param_type << ");\n"
                         << "   " << nm_type << "(){}\n"
                         << "   static "
                         << nm_type << FIXED_TYPE(" make(sos_Object);\n");
   
      os_impl_c << nm_type << "::" << nm_type << "("
                << THIS_param_type << " _x){\n"
                << FIXED_TYPE("*(sos_Object*)this=sos_Object::make(_x);\n}\n")

                << nm_type << " " << nm_type << "::make(sos_Object _x){\n"
                << "err_assert(sos_Union_type::make(" << nm_type
                           << "_type).is_base_of(_x.type())"
                           << ",\"" << nm_type << "::make\");\n"
                << "return " << nm_type << "(_x._typed_id());\n}\n";
   }
   TT (genCC_M, T_LEAVE);
   return (sos_Bool)(pass == 2);
}

EXPORT void back_union_type (smg_String type_name, sos_Bool is_scalar)
{  T_PROC ("back_union_type") 
   TT (genCC_M, T_ENTER);

   os_use_h  << "   " << nm_type << "(" << type_name << "&);\n";
   os_impl_c << nm_type << "::" << nm_type << "(" << type_name 
             << FIXED_TYPE("&_x){\n*(sos_Object*)this=");
   if (is_scalar)
      os_impl_c << "make_" << type_name << "_object(_x)";
   else
      os_impl_c << FIXED_TYPE("(sos_Object)_x");
   os_impl_c << ";\n}\n";

   TT (genCC_M, T_LEAVE);
}
 
EXPORT void back_union_end ()
{  T_PROC ("back_union_end")
   TT (genCC_M, T_ENTER);

   os_use_h << "};\n";
   back_type_consts("");

   TT (genCC_M, T_LEAVE);
}

// -------- type handling: classes  -----------------------------------------

EXPORT sos_Bool   back_class_is_root;
LOCAL  sos_Bool	  class_enter_impl_obj;
LOCAL  int        obj_size;
LOCAL  smg_String nm_root;
LOCAL  smg_String impl_class_c;
LOCAL  smg_String impl_class_h;
LOCAL  smg_String class_c;
LOCAL  smg_String class_h;
LOCAL  smg_String univ_methods;
LOCAL  smg_String univ_inits;

LOCAL  int	  sc_direct_count;


LOCAL smg_String convert_to_class (smg_String nm_class, smg_String nm_root)
{  T_PROC ("convert_to_class")
   TT (genCC_L, T_ENTER);

   // OBST_ BASECONV hides if a compiler does not correctly handle the
   // conversion of a NULL pointer from a derived to a base class.
   // (Necessary for earlier releases of the GNU compiler.)

   // (The macro name is split to account for old style preprocessors
   //  which replace even in string literals.)

   smg_String result = nm_class + "::make(" + THIS_comp + ",";
   if (gen_inline)
      result += smg_String("(") + _ID + nm_root + "*)"
	     +  IMPL_comp + ")";
   else
      result +=	GCC_SPECIFIC(smg_String("OBST_") + "BASECONV(")
	     +  IMPL_comp + "," + _ID + nm_root + "))";

   TT (genCC_L, T_LEAVE);
   return result;
}

EXPORT sos_Bool back_class_start (smg_String   type_name, 
                                  sos_Typed_id typed_id,
                                  smg_String   root_name,
                                  sos_Bool     is_root,
				  sos_Bool     is_abstract,
                                  int          object_size)
{  T_PROC ("back_class_start") 
   TT (genCC_M, T_ENTER);

   nm_type            = type_name;
   nm_root            = root_name;
   impl_nm_type       = smg_String(_ID) + nm_root;
   type_nm_type       = FIXED_TYPE("sos_Class_type_type");
   tpid_type	      = typed_id;
   back_class_is_root = is_root;
   obj_size           = object_size;
   
   if (pass == 2)
   {  impl_class_c.empty();
      impl_class_h.empty();
      class_c.empty();
      class_h.empty();
      univ_methods.empty();

      sc_direct_count      = 0;
      class_enter_impl_obj = (sos_Bool)(
			     NOT is_abstract
	 		     SCALAR_EXCEPTION(
			     OR nm_type.equal(nm_SCALAR_ROOT))
#ifdef BOOT
			     OR nm_type.equal(FIXED_TYPE("sos_Type"))
#endif
			     );

      smg_String comment = smg_String("  // *** ") + nm_type + " ***\n";

      inits_level0 += comment;
      inits_level1 += comment;
      inits_level2 += comment;
      impl_class_c += comment;

      class_h += smg_String("class ") + nm_type 
              +  ": public " + nm_ROOT
              +  " \n{  friend class " + impl_nm_type + NL
              +  "   const class " + impl_nm_type + "*" + IMPL_comp + NL
              +  "public :\n";

      if (back_class_is_root)
         impl_class_h += smg_String("class ") + impl_nm_type;
   }
   TT (genCC_M, T_LEAVE);
   return (sos_Bool)(pass == 2);
}

EXPORT void back_class_end ()
{  T_PROC ("back_class_end")
   TT (genCC_M, T_ENTER);

   if (pass == 1)
   {  if (back_class_is_root)
         os_impl_h << "class " << impl_nm_type << NL;
      os_use_h << "class " << nm_type << NL;
   } 
   else
   {
#if COMPLICATED_ARGUMENT_BUG
      class_h 	   += smg_String("OBST_TMP") + "DEF(extern," + nm_type
	      	   +  ",_" + nm_type + "_tmp)\n";
      impl_class_c += smg_String("OBST_TMP") + "DEF(," + nm_type
	      	   +  ",_" + nm_type + "_tmp)\n";
#endif
      os_impl_h << impl_class_h;
      os_impl_c << impl_class_c;
      os_use_h  << class_h;
      os_impl_c << class_c;
      os_impl_c << univ_methods;

      back_type_consts("");
   }
   TT (genCC_M, T_LEAVE);
}

// -------- super classes  ---------------------------------------------------

LOCAL  smg_String sc_stuff;
LOCAL  smg_String sc_stuff_local;

EXPORT void back_super_class(smg_String nm_supercl,
                             smg_String nm_sc_root, 
                             int        offset,
			     sos_Bool	has_local_components,
                             sos_Bool   is_direct)
{  T_PROC ("back_super_class")
   TT (genCC_M, T_ENTER);

   sos_Bool is_this_class;

   static sos_String_sos_Object_Mapping root_name_map = 
          sos_String_sos_Object_Mapping::create
          (TEMP_CONTAINER, /*list_cursor*/ FALSE, /*key_based_on_equal*/ TRUE);
   static smg_String last_class_name;

   if (NOT last_class_name.equal(nm_type))
   {
      last_class_name = nm_type;
      root_name_map.clear();
   }

   if (NOT (is_this_class = (sos_Bool)nm_type.equal(nm_supercl)))
   {  if (gen_inline)
         class_h += smg_String("   inline operator ") + nm_supercl + "() const"
                 + "{return "
		 +  convert_to_class (nm_supercl, nm_sc_root) + ";}" + NL;
      else
      {  class_h += smg_String("   operator ") + nm_supercl + "() const" + NL;
         class_c += nm_type + "::operator " + nm_supercl + "() const"
                 +  "{\nreturn "
		 +  convert_to_class (nm_supercl, nm_sc_root) + ";\n}\n";
      }
   } 
   if (NOT nm_supercl.equal(nm_sc_root))
   {  sos_String rn;
      if (NOT root_name_map.is_key (rn = sos_String::create(
				 TEMP_CONTAINER, nm_sc_root.make_Cstring())))
      {  root_name_map.insert (rn, NO_OBJECT);
         if (gen_inline)
            class_h += smg_String("   inline operator ") + nm_sc_root + "()"
                    + " const"
                    + "{return "
	   	    +  convert_to_class (nm_sc_root, nm_sc_root) + ";}" + NL;
      	 else
         {  class_h += smg_String("   operator ") + nm_sc_root + "() const"
                    +  NL;
            class_c += nm_type + "::operator " + nm_sc_root + "() const"
                    +  "{\nreturn "
		    +  convert_to_class (nm_sc_root, nm_sc_root) + ";\n}\n";
         }
      }
   } 
   if (back_class_is_root)
   {  if (is_direct)
	 impl_class_h += smg_String((++sc_direct_count == 1) ? ":" : ",")
      		      +  "virtual public " + _ID + nm_sc_root;

      if (NO_COMPS_SHORTCUT(
	  has_local_components
	     sos_OBJECT_SHORTCUT(
	     AND NOT nm_supercl.equal("sos_Object") )))
	 *(is_this_class ? &sc_stuff_local : &sc_stuff)
             += smg_String("   virtual sos_Offset _offset_") + nm_sc_root
             +  "()const {return " + offset + ";}\n";
   }
   if (class_enter_impl_obj)
      inits_level1 += smg_String("  ") + set_IMPL_OBJ 
               	   +  _ID + nm_sc_root + "_type,"
               	   +  _ID + nm_type + "_type,(" + _ID + nm_sc_root + "*)" 
               	   +  impl_nm_type + "_OBJ_);\n";

   TT (genCC_M, T_LEAVE);
}

EXPORT void back_super_class_end ()
{  T_PROC ("back_super_class_end")
   TT (genCC_M, T_ENTER);

   if (back_class_is_root)
   {  if (sc_direct_count == 0)
	 impl_class_h += smg_String(":virtual public ") + nm_IMPL_ROOT;

      impl_class_h += smg_String("\n{  friend class ") + nm_type + NL;

      if (sc_direct_count != 1)
	 impl_class_h += sc_stuff;

      impl_class_h += sc_stuff_local + "public:\n";

      if (class_enter_impl_obj)
	 impl_class_h += smg_String("   ")
		      +  impl_nm_type + "(){_size=" + obj_size + ";}\n";
   }
   sc_stuff.empty();
   sc_stuff_local.empty();

   TT (genCC_M, T_LEAVE);
}

EXPORT void back_friend (smg_String nm_friend)
{  T_PROC ("back_friend")
   TT (genCC_M, T_ENTER);
     
   smg_String decl = smg_String("   friend class ") + _ID + nm_friend + NL;

   class_h      += decl;
   impl_class_h += decl;

   TT (genCC_M, T_LEAVE);
}

// -------- methods ---------------------------------------------------------

LOCAL sos_Method_kind current_visibility;
LOCAL sos_Method_kind impl_curr_visibility;

EXPORT void back_methods_start()
{  T_PROC ("back_methods_start")
   TT(genCC_M, T_ENTER);
    
   current_visibility   = sos_PUBLIC;	// see back_class_start
   impl_curr_visibility = sos_PUBLIC;

   TT(genCC_M, T_LEAVE);
}
 
EXPORT void back_methods_end()
{  T_PROC ("back_methods_end")
   TT (genCC_M, T_ENTER);

   class_h += smg_String ("}") + NL;

   if (back_class_is_root)
   {  impl_class_h += smg_String("}") + NL;

      if (class_enter_impl_obj)
      {  smg_String objidx = smg_String("[") + (tpobjcounter ++) + "]";

	 impl_class_h += smg_String("#define ") + impl_nm_type + "_OBJ_ (("
	    	      +  impl_nm_type + "*)_" + nm_schema + "_obj_"
		      +  objidx + ")\n";
	 inits_level0 += smg_String("  __obj") + objidx
	       	      +  "=(void*)new " + impl_nm_type + NL;
      }
   }

   TT (genCC_M, T_LEAVE);
}

//-------- method handling -------------------------------------------------- 

EXPORT sos_Bool      is_void, is_static, OBSThis_param;
EXPORT smg_String    nm_method;
EXPORT smg_String    nm_result_type;

LOCAL  sos_Bool	     decl_shadowed, impl_shadowed, overloaded_inherited;
LOCAL  sos_Bool      decl_in_impl_class, decl_in_class;
LOCAL  sos_Bool      is_local,     is_comp_method,   is_abstract,
		     is_generated, is_inline_method, gen_method_inline, 
                     has_params,   is_pure_method,
		     GENERAL_UNIV_SHORTCUT(
		     has_general_univ),
		     ATTRSET_METHOD_SHORTCUT(
		     maybe_attrset_method);
LOCAL  smg_String    nm_impl_method;
LOCAL  int           param_nr;

LOCAL  Predef_Method actual_method;
LOCAL  sos_Bool      is_gen_in_class, is_gen_in_impl_class, gen_univ_method;
LOCAL  int           mtpid_container, mtpid_offset;

EXPORT sos_Bool back_method_start (sos_Method      method,
                                   smg_String      mname,
				   smg_String      tp_result,
				   sos_Bool        is_loc,
				   sos_Bool        is_comp,
				   sos_Bool        is_abstr,
				   sos_Bool        is_stat,
				   sos_Bool        is_operator,
				   sos_Bool        is_gen,
				   sos_Bool	   is_definite,
				   sos_Method_kind visibility,
				   Predef_Method   act_method,
				   int        	   paramcount)
{  T_PROC ("back_method_start")
   TT (genCC_M, T_ENTER);

   sos_Bool is_cpp_meth      = FALSE, // Internal method of the C++ binding,
				      // generated for interface class.
      	    is_impl_cpp_meth = FALSE; // Like 'is_cpp_meth', but generated for
				      // implementation class.

   sos_Method_kind impl_visibility = visibility;

   has_general_univ     = FALSE;
   is_gen_in_class      = FALSE;
   decl_shadowed        = FALSE;
   OVERLOADED_INHERITED_PATCH(
   overloaded_inherited = (sos_Bool)(act_method == OVERLOADED_INHERITED);
   )

   switch (act_method)
   {  case OFFSET:
      case CONTAINER:
      case COMPARE_IDS:
      case OP_IDENTICAL:
      case OP_NOT_IDENTICAL:
      case IDENTICAL:	     decl_shadowed    = TRUE;
			     break;
      case _CREATE:
      case _CREATE_COMPS:
      case _COPY:
      case _COPY_COMPS:
      case _CLONE:
      case GENERIC_PATCH:    is_impl_cpp_meth = TRUE;
			     break;
      case MAKE_HANDLE:
      case MAKE_CP_HANDLE:
      case MAKE_SC_HANDLE:   is_cpp_meth      = TRUE;
      case COPY:
      case CREATE:	     is_gen_in_class  = TRUE;
			     break;

      case CLONE:	     is_gen_in_class  = TRUE;
      case ASSIGN:
      case DESTROY:
      case EQUAL:
      case HASH_VALUE:	     GENERAL_UNIV_SHORTCUT(
			     has_general_univ = TRUE);
			     break;
      default: ;
   }
   if (NOT (back_class_is_root OR is_cpp_meth OR visibility == sos_PUBLIC))
   {  TT (genCC_M, T_LEAVE);
      return FALSE;
   }
   nm_result_type       = tp_result;
   actual_method        = act_method;
   is_abstract          = is_abstr;
   is_generated         = (sos_Bool)(is_gen OR is_abstract);
   is_static            = is_stat;
   is_void              = (sos_Bool)nm_result_type.equal(FIXED_TYPE("void"));
   is_comp_method       = is_comp;
   is_local             = is_loc;
   has_params           = (sos_Bool)(paramcount > 0);
   ATTRSET_METHOD_SHORTCUT(
   maybe_attrset_method = FALSE;

   if (NOT (   is_static
	    OR paramcount != 1
	    OR act_method != NOT_PREDEF
	    OR NOT is_void
	    OR is_operator))
      maybe_attrset_method = (sos_Bool)
      			     streqln("set_",mname.make_Cstring(SMG_BORROW), 4);
   )
   if (is_operator)
   {  nm_method = smg_String("operator") + mname;

      static sos_String m_name = sos_String::create(TEMP_CONTAINER);

      m_name.assign_Cstring(mname.make_Cstring (SMG_BORROW));
      sos_String m_op_name = cci_Method_impl::operator_string(m_name);
      nm_impl_method = m_op_name;
      m_op_name.destroy();
   }
   else
   {  nm_method      =
      nm_impl_method = mname;
   }
   sos_Bool is_object_tp= (sos_Bool)nm_type.equal(FIXED_TYPE("sos_Object"));

   is_pure_method	= (sos_Bool)(
			      is_abstract
      			  AND NOT (   class_enter_impl_obj
				   OR is_static
#  ifdef BOOT
				   SCALAR_EXCEPTION(OR is_object_tp)
				   OR nm_type.equal (FIXED_TYPE("sos_Named"))
#  endif
				  ));
   // `sos_Object` is implied since there's an implementation object for
   // the derived class `sos_Scalar_object`. Note, that we expose here, that
   // `sos_Scalar_object` is a direct subclass of `sos_Object`.
   // The like holds for the base classes `sos_Object` and `sos_Named` of
   // `sos_Type`.

   impl_shadowed 	= (sos_Bool)(
			     decl_shadowed
      		      	  sos_OBJECT_SHORTCUT(
			  OR is_comp_method AND is_object_tp));
//			  OR is_pure_method;	    // EXPLICIT_ABSTRACT_BUG()

   sos_Bool decl_implcl = (sos_Bool)(
			      back_class_is_root
      			  AND is_local
			  AND NOT (is_gen_in_class OR decl_shadowed)
			  AND ABSTRACT_SHORTCUT(
			      NOT (    is_abstract
				   AND is_gen
				   AND sc_direct_count == 1
				   AND NOT is_object_tp)));

   OVERLOADED_INHERITED_PATCH(
   decl_in_impl_class   = (sos_Bool)(decl_implcl OR overloaded_inherited);
   )

   decl_in_class        = (sos_Bool)(NOT (   is_impl_cpp_meth 
                                          OR decl_shadowed
                                          OR overloaded_inherited));

   is_gen_in_impl_class = (sos_Bool)(
			      decl_implcl
      			  AND is_generated
			  AND NOT impl_shadowed);

   sos_Bool gen_impl  = (sos_Bool)(
                            is_local
                          AND NOT (is_impl_cpp_meth OR is_cpp_meth)
                          AND (back_class_is_root OR is_gen_in_class));

   gen_univ_method      = (sos_Bool)(gen_impl AND visibility == sos_PUBLIC);

   is_inline_method   = (sos_Bool)(actual_method == MAKE_HANDLE OR
                                   actual_method == MAKE_SC_HANDLE);
   gen_method_inline  = (sos_Bool)(actual_method == MAKE_CP_HANDLE OR
                                  (is_inline_method AND gen_inline));     

   OBSThis_param	= (sos_Bool)(actual_method == _CLONE          
                                     OR actual_method == GENERIC_PATCH
                                     OR NOT (is_static OR is_impl_cpp_meth));

   if (gen_impl)
   {  cci_Method_impl cpp_impl = cci_Method_impl::make_impl (method);

      mtpid_container = cpp_impl.container();
      mtpid_offset    = cpp_impl.offset();
   }
   else
   {  mtpid_container = -1;
      mtpid_offset    = -1;
   }

   OVERLOADED_INHERITED_PATCH(
   if (overloaded_inherited)
      impl_class_h += smg_String("#ifndef OBST_") + "CAN_OVERLOAD_INHERITED"
		   +  "\n";
   )

   if (decl_in_impl_class)
   {  SPLIT_CREATE_SHORTCUT(
      if (nm_method.equal(nm_local_ctor) OR nm_method.equal(nm_total_ctor))
	 impl_visibility = sos_PUBLIC;
      )
      if (impl_visibility != impl_curr_visibility)
      {  impl_curr_visibility = impl_visibility;  
         switch (impl_visibility)
         {  case sos_PRIVATE   : impl_class_h += "private:\n";   break;
            case sos_PROTECTED : impl_class_h += "protected:\n"; break;
	    case sos_PUBLIC    : impl_class_h += "public:\n";    break; 
         }
      }
      impl_class_h += smg_String("   ")
		   +  (is_static ? "static " 
	 			 : (is_definite ? ""
		       			        : "virtual "))
	           + nm_result_type + " " + nm_impl_method + "(";

      if (overloaded_inherited)
         impl_class_h += THIS_param_decl;
      else if (OBSThis_param)
	 impl_class_h += THIS_param_type;

      if (is_gen_in_impl_class)
      {  // OBST_ USEPURE specifies if abstract methods may be implemented as
	 // pure virtual methods or if an explicit body must be provided.

	 // (The macro name is split to account for old style preprocessors
         //  which replace even in string literals.)
	 EXPLICIT_ABSTRACT_BUG(
	 if (is_pure_method)
	    impl_class_c += smg_String("#ifndef OBST_") + "USEPURE\n";
	 )
	 impl_class_c += nm_result_type + " " + impl_nm_type + "::" 
	 	      +  nm_impl_method + "(";
         if (OBSThis_param)
            impl_class_c += (is_abstract) ? THIS_param_type  
	       				  : THIS_param_decl;
      }
   }
  
   if (decl_in_class)    
   {  if (visibility != current_visibility)
      {  current_visibility = visibility;  
         switch (visibility)
         {  case sos_PRIVATE   : class_h += "private:\n"; break;
            case sos_PROTECTED : class_h += "protected:\n"; break;
	    case sos_PUBLIC    : class_h += "public:\n"; break;
         }
      }
      class_h += smg_String("   ")
	      + (gen_inline AND gen_method_inline ? "inline " : "")
              + (is_static ? "static " : "") 
              + nm_result_type + " " + nm_method + "(";
      if (NOT gen_method_inline)
         class_c += nm_result_type + " " + nm_type + "::" + nm_method + "(";
      if (actual_method == MAKE_HANDLE AND NOT gen_inline)
      {  class_c += THIS_param_decl;
         class_h += THIS_param_type;
      }
      else if (actual_method == MAKE_CP_HANDLE OR actual_method == MAKE_HANDLE)
         class_h += THIS_param_decl;
   }
   param_nr = 0;

   TT (genCC_M, T_LEAVE);
   return TRUE;
}


EXPORT void back_param (smg_String param_name,
                        smg_String _param_type,
                        sos_Bool   is_ref,
                        sos_Bool   is_scalar,
                        sos_Expr   expr,
			sos_Bool   constref_macro /* = TRUE */)
{  T_PROC ("back_param")
   TT (genCC_M, T_ENTER);

   smg_String expr_str, param_use, param_type, impl_param_type;

   if (param_name.length())
      param_use = smg_String(" ") + param_name;
   else
      param_use = smg_String(" _p") + param_nr;

   FLOAT_PATCH(
   if (_param_type.equal (FIXED_TYPE("sos_Float")) AND NOT is_ref)
   {  param_type      = smg_String ("OBST_") + "FLOATDECL(" 
                      + _param_type + ")";
      impl_param_type = smg_String ("_") + param_type;
   }
   else {
   )
   sos_Bool is_pointer = (sos_Bool)(actual_method == MAKE_CP_HANDLE);

   sos_Bool impl_c_and_r, const_and_ref;
   if (is_ref OR is_pointer)
   {  impl_c_and_r  = FALSE;
      const_and_ref = FALSE;
   }
   else
   {  impl_c_and_r  = (sos_Bool)(is_scalar
				 ATTRSET_METHOD_SHORTCUT(
				 AND maybe_attrset_method));
      const_and_ref = (sos_Bool)(NOT is_scalar
				 ATTRSET_METHOD_SHORTCUT(
				 OR impl_c_and_r));
   }
   if (NOT const_and_ref)
   {  param_type = smg_String (is_pointer ? "const " : "")
                   + _param_type
	           + (is_ref ? "&" : "");
   }
   else if (constref_macro)
      param_type = smg_String ("OBST_") + "PARDECL(" + _param_type + ")";
   else
      param_type = smg_String ("const ") + _param_type + "&";

   impl_param_type = param_type;
   if (impl_c_and_r)
      param_type = _param_type;

   FLOAT_PATCH(
   }
   )
   expr_str = code_expr (expr, " =", _param_type);

   ++ param_nr;
      
   if (decl_in_impl_class AND 
       (param_nr > 1  OR  OBSThis_param OR overloaded_inherited))
   {  impl_class_h += ",";
      if (is_gen_in_impl_class)
         impl_class_c += ",";
   }
   if (decl_in_class AND param_nr > 1)
   {  class_h += ",";
      if (NOT gen_method_inline)
         class_c += ",";
   }
   if (actual_method == MAKE_CP_HANDLE)
      class_h += ",";
   
   if (decl_in_class)
   {  class_h += param_type;
      if (gen_method_inline)
	 class_h += param_use;
      else
         class_c += param_type + param_use;

      class_h += expr_str;
   }

   if (decl_in_impl_class)
   {  if (overloaded_inherited)
         impl_class_h += impl_param_type + param_use + expr_str;
      else
         impl_class_h += impl_param_type + expr_str;

      if (is_gen_in_impl_class)
         impl_class_c += impl_param_type + param_use;
   }
   TT (genCC_M, T_LEAVE);
}

EXPORT void back_param_end ()
{  T_PROC ("back_param_end")
   TT (genCC_M, T_ENTER);

   smg_String const_mark = (is_static) ? "" : "const";

   if (decl_in_impl_class)
   {  // The macro OBST_ ISPURE is expanded either to the marker for pure
      // virtual methods or to white space.

      // (The macro name is split to account for old style preprocessors
      //  which replace even in string literals.)
      if (is_pure_method)
	 impl_class_h += EXPLICIT_ABSTRACT_BUG(
			 smg_String(")OBST_") + "ISPURE\n");
      else
	 impl_class_h += smg_String(")") + const_mark 
	              +  (overloaded_inherited ? "{\n" : NL);
         
      if (is_gen_in_impl_class)
         impl_class_c += smg_String(")") + const_mark + "{\n";
   }
   if (decl_in_class)
   {  if (gen_method_inline)
	 class_h += smg_String(")") + const_mark + "{\n      ";
      else
      {  class_h += smg_String(")") + const_mark + ";\n";
         class_c += smg_String(")") + const_mark + "{\n";
      }
   }    
   TT (genCC_M, T_LEAVE);
}
  
EXPORT void back_method_end()
{  T_PROC ("back_method_end")
   TT (genCC_M, T_ENTER);

   if (NOT impl_shadowed)
   {  if (is_gen_in_class)
      {  if (NOT gen_method_inline) 
	    class_c += code_generate_body (actual_method);
	 else 
	    class_h += code_generate_body (actual_method);
      }
      if (is_gen_in_impl_class)
      {  if (is_comp_method)
	    impl_class_c += code_comp_method();
	 else if (is_abstract)
	    impl_class_c += code_abstract_method();
         else if (actual_method == GENERIC_PATCH)
            impl_class_c += code_method_patch (nm_impl_method,
                                               GENERIC_PATCH);
	 else
	    impl_class_c += code_generate_body (actual_method);
	 
	 if (decl_in_impl_class)
	 {  impl_class_c += "}\n";
	    EXPLICIT_ABSTRACT_BUG(
	    if (is_pure_method)
	    impl_class_c += "#endif\n";)
	 }
      }
   }
   OVERLOADED_INHERITED_PATCH(
   if (overloaded_inherited)
      impl_class_h += code_method_patch (nm_impl_method, 
                                         OVERLOADED_INHERITED)
                      + "}\n#endif\n";
   )

   if (gen_univ_method) 
      back_univ_method();
   if (decl_in_class)
   {  if (NOT is_gen_in_class)
	 if (NOT gen_method_inline)
	    class_c += code_method_call_impl (nm_impl_method);
         else
	    class_h += code_method_call_impl (nm_impl_method);

      if (NOT gen_method_inline)
	 class_c += "}\n";
      else
	 class_h += "   }\n";
   }
   TT (genCC_M, T_LEAVE);
}

EXPORT smg_String back_call_method (smg_String receiver,
                                    smg_String func_name,
                                    sos_Bool   is_static,
                                    sos_Bool   receiver_is_lvalue,
                                    sos_Bool   interface2impl,
                                    smg_String args)
{  T_PROC ("back_call_method")
   TT (genCC_VL, T_ENTER);

   smg_String result = receiver.clone();

   if (is_static)
      result += "::";
   else
   {  result += (receiver_is_lvalue ? "->" : ".");
      if (interface2impl)
       result += call_IMPL;
   }
   result += func_name + "(" + args + ")";

   TT (genCC_VL, T_LEAVE);
   return result;
}

//-------- universal methods --------------------------------------------------

LOCAL void back_univ_method ()
{  T_PROC ("back_univ_method");
   TT (genCC_M, T_ENTER);

   smg_String nm_univ_method;

   if (GENERAL_UNIV_SHORTCUT(has_general_univ))
      nm_univ_method = smg_String(nm_GEN_UNIV_PREF) + nm_method;
   else
   {  nm_univ_method = smg_String(nm_UNIV_PREF)
      		       + mtpid_container + "_" + mtpid_offset;

      univ_methods += smg_String(FIXED_TYPE("static sos_Object "))
   	           +  nm_univ_method + FIXED_TYPE("(const sos_Object&");

      if (NOT is_static)
         univ_methods += "_x";

      univ_methods += FIXED_TYPE(",const sos_Object_Array&"); 
      if (has_params)
         univ_methods += "_p";

      univ_methods += smg_String("){\n") + code_univ_body() + "}\n";
   }
   inits_level1 += smg_String("  ") + set_UNIV_FCT
	    	+  FIXED_FCT("sos_Id::make(")
                   + FIXED_FCT("sos_Container::make(") + mtpid_container + "),"
	           + mtpid_offset
	    	+  FIXED_TYPE("),(cci_Fun)") + nm_univ_method
	    	+  ")" + NL; 

   TT (genCC_M, T_LEAVE);
}
