/* --------------------------------------------------------------------------
 * Copyright 1992 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// OBST directory browser
// **************************************************************************

#define OBST_IMP_STREAM
#define OBST_IMP_FORMATTED_IO
#define OBST_IMP_MALLOC
#define OBST_IMP_SORT_SEARCH
#include "obst_stdinc.h" 

#include "graph_ut.h"
#include "obst_trc.h"

#include "mta_use.h"


// ---------------------------------------------------------------------------
// ************************  Classes  ****************************************
// ---------------------------------------------------------------------------

class DirKnownObject : public KnownObject
{ friend class DirItemList;

  public:
   sos_String    name;
   sos_Directory parent;
   smg_String    path;
  private:
   sos_String    *names;

  protected:
   void item_selected (int);

  public:
   DirKnownObject() { Application::enter_ptr (this, (KnownObject*)this); }
   ~DirKnownObject();
   
   smg_String elem_path (smg_String); 

   static DirKnownObject* make (KnownObject*);

   static void display (sos_Object, sos_Directory, sos_String, smg_String,
   			WdgtPosition);
   void redisplay ();

#ifndef OBST_CAN_OVERLOAD_INHERITED
   void display (sos_Bool b, WdgtPosition w) { KnownObject::display (b, w); }
#endif
  public:
   void cmd_new_directory (char*);
   void cmd_move_elem (char*);
   void cmd_delete_elem ();
   void cmd_collect_containers (sos_Object, sos_Container, sos_Container_set&);
};

class DirKnownObjectList : public KnownObjectList
{   
  public:
   DirKnownObjectList() 
 		     { Application::enter_ptr (this, (KnownObjectList*)this); }

   static DirKnownObjectList* make (KnownObjectList*);

   void insert (DirKnownObject*);
   void	change_paths (smg_String, smg_String);
   void delete_tree (DirKnownObject*);
};

class DirItemList 
{ friend class KnownObject;

  class ObjectItem
  { public:
     sos_String name;
     char       *descr;
  };

  private:
   ObjectItem  *items;
   int         size;
   int         nr_items;

   static int compare_items (_qsort_const_ void*, _qsort_const_ void*);
  public:
   DirItemList () {items = NULL;  nr_items = size = 0;}

   void	  append (const sos_String&, char*);
   void	  make_KnownObject (DirKnownObject*);
};

class DirObjectCmdMenu : public ObjectCmdMenu
{ enum DirObjectCmd
   {  ocmd_TITLE, 
      ocmd_NEW,         ocmd_DELETE, ocmd_MOVE,
      ocmd_CLOSE_TREE,  ocmd_CLOSE
   };
   
  public:
   void install();

  protected:
   void do_it (int);
};

class DirMenuSystem : public MenuSystem
{  enum MenuID     { menu_FILE, menu_CONTAINER };
   enum FileItemID { file_STD_ROOT, file_INIT_ROOT, file_QUIT };
   enum CntItemID  { cnt_RESET_ALL, cnt_COMMIT_ALL };

  protected:
   void do_it (int, int);
};

class DirApplication : public Application
{ public:
   DirApplication() { Application::enter_ptr (this, (Application*)this); }

   void init (int, char**);
   void display_root (sos_Bool);

   static DirApplication* make();
};

// ---------------------------------------------------------------------------
// ************************  Variables  **************************************
// ---------------------------------------------------------------------------

static DirItemList the_item_list;


// ---------------------------------------------------------------------------
// *************************  class: DirItemList  ****************************
// ---------------------------------------------------------------------------

void DirItemList::append (const sos_String &name, char *desc)
	// Append new item to this list
{
   if (nr_items EQ size)
   {  if (size)
         items
	   = (ObjectItem*)REALLOC ((void*)items,
				   (size_t)sizeof(ObjectItem)*(size += 10));
      else
         items = new ObjectItem [size = 10];
   }
   items[nr_items  ].descr = desc;
   items[nr_items++].name  = name;
}

int DirItemList::compare_items (_qsort_const_ void *i1, _qsort_const_ void *i2)
{  return strcmp (((ObjectItem*)i1)->descr, ((ObjectItem*)i2)->descr);
}

void DirItemList::make_KnownObject (DirKnownObject *kobj)
{  qsort (&items[1], nr_items-1, sizeof(ObjectItem),
	  (qsort_cmpfct_t*)compare_items);
	// items[0] refers to the object itself and stays the first item!
	// The other items are sorted lexically according to their path names.

   kobj->list_length = nr_items;
   kobj->descr_list  = new char* [nr_items];
   kobj->names       = new sos_String [nr_items];
   for (int i=0; i<nr_items; i++)
   {  kobj->descr_list[i] = items[i].descr;
      kobj->names[i]      = items[i].name;
   }
   nr_items = 0;
}


// ---------------------------------------------------------------------------
// *************************  class: DirKnownObject  ************************
// ---------------------------------------------------------------------------

DirKnownObject* DirKnownObject::make (KnownObject* baseptr)
{ 
   return (DirKnownObject*)Application::lookup_ptr (baseptr);
}

DirKnownObject::~DirKnownObject()
{
  delete names;
}

smg_String DirKnownObject::elem_path (smg_String name)
{  smg_String elempath = path.make_Cstring (SMG_BORROW);

   if (path.length() > 1)
      elempath += "/";

   return (elempath += name);
}

// ***************************  DirKnownObject: Display operations  *********

void DirKnownObject::display (sos_Object    o,
   			      sos_Directory parent,
			      sos_String    name,
			      smg_String    path,
			      WdgtPosition  pos)
{  if (o == NO_OBJECT) return;

   DirKnownObject *kobj = DirKnownObject::make(
			     the_known_objects->known_object (o));

   if (kobj)  				// object is already displayed
      kobj->display (TRUE, pos);	//    --> move to new position and raise

   else 		// construct new object widget and display
   {  smg_String descr;
      if (NOT o.is_some (sos_Object_Directory_type))
      {  descr = path + " (" + o.type().get_name() + ")";
         the_item_list.append (sos_String::make (NO_OBJECT),
                               descr.make_Cstring (SMG_TRANSFER));
      }
      else
      {  the_item_list.append (sos_String::make (NO_OBJECT),
                               path.make_Cstring (SMG_TRANSFER));

         sos_Directory dir = sos_Directory::make (o);
         agg_iterate_association (dir, sos_String ename, sos_Object elem)
         {  descr = smg_String(ename) + " (" + elem.type().get_name() + ")";
            the_item_list.append (ename, descr.make_Cstring (SMG_TRANSFER));
         }
         agg_iterate_association_end (dir, ename, elem);
      }

      DirKnownObject *kobj = new DirKnownObject;
      kobj->obj     = o;
      kobj->name    = name;
      kobj->parent  = parent;
      kobj->path    = path;
      the_item_list.make_KnownObject (kobj);

      kobj->display (FALSE, pos);

      DirKnownObjectList::make(the_known_objects)->insert (kobj);
   }
}

void DirKnownObject::redisplay ()		// Redisplay this object.
{
   the_known_objects->remove (this);

   DirKnownObject::display (obj, parent, name, path, WdgtPosition::at (wdgt));

   delete this;
}

// ***************************  DirKnownObject: Command handlers  *********

void DirKnownObject::cmd_new_directory (char *name_str)
	// Create a new directory named according to the given name and
	// insert it into this directory.
{  DirKnownObject *kobj = DirKnownObject::make(
			     the_known_objects->known_object(obj));
 
   err_assert (obj.is_some (sos_Object_Directory_type),
	       "DirKnownObject::new_directory");

   sos_String new_name = sos_String::create (TEMP_CONTAINER, name_str);

   sos_Container new_cnt = sos_Container::create();

   sos_Directory new_dir = sos_Object_Directory::create (new_cnt, new_name);

   new_cnt.close();

   sos_Directory this_dir = sos_Object_Directory::make (kobj->obj);
   the_application->access_ct (this_dir.container());

   this_dir.insert (new_name, new_dir);

   DirKnownObject::display (new_dir, this_dir, new_name,
			   elem_path (new_name), WdgtPosition::next_to (wdgt));
   redisplay();
}

void DirKnownObject::cmd_move_elem (char *dest_path)
	// Move this object to the given directory.
{  err_assert (parent != NO_OBJECT, "DirKnownObject::cmd_move_elem");

   smg_String dest_str  = dest_path;
   sos_String dest_name = sos_String::create (TEMP_CONTAINER, dest_path);
   sos_Object dest_o    = sos_Object_Directory::lookup (dest_name);
   dest_name.destroy();

   if (dest_o == NO_OBJECT  OR  NOT dest_o.is_some (sos_Object_Directory_type))
      the_info_line.display ("destination directory does not exist");
   else if (dest_o == obj)
      the_info_line.display ("can not move directory into itself");
   else
   {  DirKnownObject *dest_obj
        = DirKnownObject::make (the_known_objects->known_object (dest_o));
      sos_Directory  dest_dir  = sos_Object_Directory::make (dest_o);
      smg_String     old_path  = path;

      if (dest_obj)
	 path = dest_obj->elem_path (name);
      else
      {  if (dest_str.length() > 1)
	    dest_str += "/";
	 path = dest_str + name;
      }
                      // note: 'parent.remove(..)' will delete the old name
      name = sos_String::clone (name, TEMP_CONTAINER);

      the_application->access_ct (parent.container());
      parent.remove (name);

      the_application->access_ct (dest_dir.container());
      dest_dir.insert (name, obj);

      sos_Cursor c = dest_dir.open_cursor();
      dest_dir.move_cursor (c, name);
      name.destroy();                           // destroy temporary name
      name = dest_dir.get_key (c);              // name := key_of(obj)
      dest_dir.close_cursor (c);

      DirKnownObject* kobj;
      if (kobj
	   = DirKnownObject::make (the_known_objects->known_object (parent)))
         kobj->redisplay();

      parent = dest_dir;
      
      if (kobj
	   = DirKnownObject::make (the_known_objects->known_object (dest_dir)))
         kobj->redisplay();
      redisplay ();

      DirKnownObjectList::make(the_known_objects)->change_paths(old_path,path);
   }
}

void DirKnownObject::cmd_delete_elem()
{
   sos_Container_set cset;

   cmd_collect_containers (obj, parent.container(), cset);

   DirKnownObject *kobj 
      = DirKnownObject::make (the_known_objects->known_object(obj));

   sos_Directory pa = sos_Directory::make(kobj->parent);
   the_application->access_ct (pa.container());
   pa.remove (kobj->name);

   DirKnownObjectList::make(the_known_objects)->delete_tree (kobj);

   DirKnownObject::make( the_known_objects->known_object(pa) )->redisplay();

   cnt_iterate (cset, sos_Container c)
      the_application->access_ct (c);
      c.destroy();
   cnt_iterate_end (cset, c);

}

void DirKnownObject::cmd_collect_containers(sos_Object 	       o, 
                                            sos_Container      do_not_destroy,
                                            sos_Container_set& to_be_destroyed)
{  if (o.is_some (sos_Object_Directory_type))
   {  sos_Directory dir = sos_Directory::make (o);
      agg_iterate_association (dir, sos_String name, sos_Object elem)
         cmd_collect_containers (elem, do_not_destroy, to_be_destroyed);
      agg_iterate_association_end (dir, name, elem);
   }
   if (o.container() == do_not_destroy)
       o.destroy();
   else
       to_be_destroyed += o.container();
}

// ***************************  DirKnownObject: Callbacks  *****************

void DirKnownObject::item_selected (int item)
	// An item of a displayed object was selected.
	// item[0]   --> Activate object command menu for this object.
	// otherwise --> Display the selected object.
{  static char *items[] = { NULL, "new", "delete",
	 	            "move", "close tree", "close",
                            NULL
			  };
   if (item == 0)
   {  items[0] = (obj.is_some (sos_Object_Directory_type)) ? "Directory"
	                                                   : "Object";
      the_object_menu->activate (this, items);
   }
   else
   {  sos_Directory dir       = sos_Object_Directory::make (obj);
      sos_String    elem_name = names[item];
      smg_String    elem_path = this->elem_path(elem_name);

      DirKnownObject::display (dir[elem_name], dir, elem_name, elem_path,
   			       WdgtPosition::next_to (wdgt));
   }
}

// ---------------------------------------------------------------------------
// *************************  class: DirKnownObjectList  ********************
// ---------------------------------------------------------------------------

DirKnownObjectList* DirKnownObjectList::make (KnownObjectList* baseptr)
{ 
   return (DirKnownObjectList*)Application::lookup_ptr (baseptr);
}

void DirKnownObjectList::insert (DirKnownObject *kobj)
{  KnownObjectList::insert (new DirKnownObjectList, kobj);
}

void DirKnownObjectList::change_paths (smg_String oldstr, smg_String newstr)
{
   newstr  = newstr + "/";
   oldstr  = oldstr + "/";

   int  l   = oldstr.length();
   char *os = oldstr.make_Cstring (SMG_BORROW);

   for (DirKnownObjectList *ol = DirKnownObjectList::make(next);  ol; )
   {  DirKnownObject *ko = DirKnownObject::make(ol->kobj);
      char           *op = (ko->path).make_Cstring (SMG_BORROW);

      ol = DirKnownObjectList::make(ol->next);

      if (NOT strncmp (os, op, l))
      {  ko->path = newstr + (op+l);
	 ko->redisplay();
      }
   }
}

void DirKnownObjectList::delete_tree (DirKnownObject *kobj)
        // Recursive delete of all data belonging to the display of kobj.
{
   smg_String prefix;
   if (kobj->path.length() > 1)
      prefix = kobj->path + "/";
   else
      prefix = kobj->path;

   char       *pref   = prefix.make_Cstring (SMG_BORROW);
   int        plength = prefix.length();

   for (DirKnownObjectList *ol      = DirKnownObjectList::make(next),
			   *ol_pred = this;
        ol; )
   {  DirKnownObject *ko = DirKnownObject::make(ol->kobj);
      char           *op = (ko->path).make_Cstring (SMG_BORROW);

      if (ko == kobj  OR  !strncmp (op, pref, plength))
      {  ol = DirKnownObjectList::make(ol->next);
         delete ol_pred->next;
         delete ko;
         ol_pred->next = ol;
      }
      else
      {   ol_pred = ol;
          ol      = DirKnownObjectList::make(ol->next);
      }
   }
}


// ---------------------------------------------------------------------------
// **********************  class: DirObjectCmdMenu  **************************
// ---------------------------------------------------------------------------

void DirObjectCmdMenu::do_it (int current_cmd)
{
   DirKnownObject  *dkobj = DirKnownObject::make(kobj);
   sos_Object      obj    = dkobj->obj;
   sos_Directory   parent = dkobj->parent;
   sos_String      name   = dkobj->name;

   switch (current_cmd)
   {  case ocmd_TITLE:
	 break;
      case ocmd_NEW:
      {  if (obj.is_some (sos_Object_Directory_type))
	 {  the_text_dialog->prepare_start ("New Directory: ");
	    the_text_dialog->prepare_add ("");
	    if (the_text_dialog->activate(TRUE))
	       if (the_text_dialog->get_text(0))
		  dkobj->cmd_new_directory (the_text_dialog->get_text(0));
	 }
	 else
	    the_info_line.display ("no directory");
	 break;
      }
      case ocmd_DELETE:
      {	 if (parent == NO_OBJECT)
	    the_info_line.display ("root directory must not be deleted");
	 else
	 {  char *question;
            if (   obj.isa (sos_Schema_module_type)
		OR obj.isa (sos_Schema_module_Directory_type))
	       question = "*** This might be a part of the OBST meta database.\n*** Are you sure to delete?";
            else
               question = "Are you sure to delete this?";

            the_confirm_dialog.activate (question);
	    if (the_confirm_dialog.get_answer())
                  dkobj->cmd_delete_elem ();
	 }
	 break;
      }
      case ocmd_MOVE:
      {  if (parent == NO_OBJECT)
	    the_info_line.display ("root directory must not be moved");
	 else
	 {  the_text_dialog->prepare_start ("Move Directory to: ");
	    the_text_dialog->prepare_add ("");
	    if (the_text_dialog->activate(TRUE))
	       if (the_text_dialog->get_text(0))
		  dkobj->cmd_move_elem(the_text_dialog->get_text(0));
	 }
	 break;
      }
      case ocmd_CLOSE_TREE:
      {  DirKnownObjectList::make(the_known_objects)->delete_tree (dkobj);
	 break;
      }
      case ocmd_CLOSE:
      {  the_known_objects->remove(dkobj);
	 delete dkobj;
	 break;
      }
      default:
      {  the_info_line.display ("INTERNAL ERROR: unknown command");
      }
   }
}

void DirObjectCmdMenu::install()
{
   static char *items[] = { "Directory", "new", "delete",
			    "move", "close tree", "close",
			    NULL
			  };
   ObjectCmdMenu::install (items);
}

// ---------------------------------------------------------------------------
// ****************************  class: DirMenuSystem  ***********************
// ---------------------------------------------------------------------------

void DirMenuSystem::do_it (int menu, int item)
{
   switch (menu)
   {  case menu_FILE:
      {  FileItemID file_item = (FileItemID)item;
	 switch (file_item)
	 {  case file_STD_ROOT:
	    {  the_application->display_root (TRUE);  break;
	    }
	    case file_INIT_ROOT:
	    {  the_application->display_root (FALSE); break;
	    }
	    case file_QUIT:
	    {  the_application->commit_and_exit();    break;
	    }
	 }
         break;
      }
      case menu_CONTAINER:
      {  CntItemID cnt_item = (CntItemID)item;
	 switch (cnt_item)
	 {  case cnt_RESET_ALL:
	    {  the_application->reset();  break;
	    }
            case cnt_COMMIT_ALL:
	    {  the_application->commit(); break;
	    }
	 }
	 break; 
      }
   }
}


// ---------------------------------------------------------------------------
// ************************* class: DirApplication ***************************
// ---------------------------------------------------------------------------

DirApplication* DirApplication::make ()
{
   return (DirApplication*)Application::lookup_ptr (the_application);
}

void DirApplication::init (int argc, char* argv[])
{
   standard_root = sos_Object_Directory::root();

   if (argc > 3)
   {  err_raise (err_USE, err_DIR_USAGE, NULL, FALSE);
      exit (1);
   }
   else if (argc > 1)
   {  sos_Container ct = sos_Container::make (atoi (argv [1]));
      sos_Offset    os = (argc > 2) ? atoi (argv [2])
	                            : ROOT_OFFSET;
      if (ct == 0 OR os == 0)
      {  err_raise (err_USE, err_DIR_USAGE, NULL, FALSE);
	 exit (1);
      }
      err_block
         initial_root = Application::make_object (ct, os);
      err_exception
         err_raise (err_USE, err_GTOOLS_INVALID_ROOT, NULL, FALSE);
	 exit (1);
      err_block_end
   }
   else
      initial_root = standard_root;
}

void DirApplication::display_root (sos_Bool std_root)
{  DirKnownObject::display ((std_root) ? standard_root : initial_root,
   			    sos_Object_Directory::make (NO_OBJECT),
			    sos_String::make (NO_OBJECT),
			    smg_String ("/"),
			    WdgtPosition::root());
}


// ---------------------------------------------------------------------------
// *****************************  Main  **************************************
// ---------------------------------------------------------------------------

int dir_main (int argc, char *argv[])
{  T_INIT ("dir.out");

   err_block
      the_application      = new DirApplication;
      the_known_objects    = new DirKnownObjectList;
      the_object_menu      = new DirObjectCmdMenu;
      the_menu_system      = new DirMenuSystem;

      static char*  headers[]    = {"File", "Container"};
      static char*  file_items[] = {"standard root","initial root", 
				    "quit", NULL};
      static char*  cont_items[] = {"reset all", "commit all", NULL};
      static char** menulists[]  = {file_items, cont_items};

      the_application->install (argc, argv, "Dir", 2, headers, menulists);
   err_exception
      cerr << "*** ERROR <dir::main> : uncaught OBST error\n\t"
	   << err_last_raised() << " (" << err_last_origin() << ")\n";
   err_block_end
      
   return 1;
}
