/* --------------------------------------------------------------------------
 * Copyright 1992 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// OBST graphic tools - common utilities
// **************************************************************************

#ifdef __GNUG__
#pragma implementation
#endif

#include "graph_ut.h"

// ---------------------------------------------------------------------------
// ************************  Variables  **************************************
// ---------------------------------------------------------------------------

EXPORT KnownObjectList  *the_known_objects;
EXPORT VarTextDialog	*the_text_dialog;
EXPORT VarTextDialog    *the_container_dialog;
EXPORT ConfirmDialog    the_confirm_dialog;
EXPORT InfoLine		the_info_line;
EXPORT ObjectCmdMenu    *the_object_menu;
EXPORT MenuSystem       *the_menu_system;
EXPORT Application	*the_application;

EXPORT Arg al[ac_max];
EXPORT int ac;


// ---------------------------------------------------------------------------
// *************************  class: KnownObjectList  ********************
// ---------------------------------------------------------------------------

void KnownObjectList::insert (KnownObjectList *ol, KnownObject *kobj)
{  ol->next = next;
   ol->kobj = kobj;
   next = ol;
}

void KnownObjectList::remove (KnownObject *kobj)
{  for (KnownObjectList *ol = this; ol->next; ol=ol->next)
      if (ol->next->kobj == kobj)
      {
         KnownObjectList *od = ol->next;
         ol->next = ol->next->next;
         delete od;
         return;
      }
}

void KnownObjectList::remove_all ()
{
   KnownObjectList *ol = this;
   while (ol->next)
   {
      KnownObjectList *od = ol->next;
      ol->next = od->next;
      delete od->kobj;
      delete od;
   }
}

Widget KnownObjectList::widget (sos_Object o)
{  for (KnownObjectList *ol = next; ol; ol=ol->next)
      if (ol->kobj->obj == o)
         return ol->kobj->wdgt;
   return NULL;
}

KnownObject* KnownObjectList::known_object (sos_Object o)
{  for (KnownObjectList *ol = next; ol; ol=ol->next)
      if (ol->kobj->obj == o)
	 return ol->kobj;
   return NULL;
}

KnownObject* KnownObjectList::known_object (int nr)
{  for (KnownObjectList *ol = next; ol; ol=ol->next)
      if (ol->kobj->key == nr)
	 return ol->kobj;
   return 0;
}

// ---------------------------------------------------------------------------
// *************************  class: KnownObject  ****************************
// ---------------------------------------------------------------------------

KnownObject::~KnownObject()
{
  XtDestroyWidget (wdgt);
  delete descr_list;
}

void KnownObject::display (sos_Bool just_move_me, WdgtPosition pos)
{
   RESET_ARGLIST();
   ADD_ARG (XtNhorizDistance, pos.x);
   ADD_ARG (XtNvertDistance, pos.y);

   if (just_move_me)
   {  SETVALUES (wdgt);

      XtUnmanageChild (wdgt);
      XtManageChild (wdgt);  

      XtMoveWidget (wdgt, pos.x, pos.y);
      XRaiseWindow (XtDisplay (wdgt), XtWindow (wdgt));

      Application::scroll_viewport (wdgt);
   }
   else
   {  ADD_ARG (XtNnumberStrings, list_length);
      ADD_ARG (XtNlist, (XtArgVal)descr_list);
      ADD_ARG (XtNverticalList, TRUE);
      ADD_ARG (XtNforceColumns, TRUE);
      ADD_ARG (XtNdefaultColumns, 1);

      ADD_ARG (XtNtop, XtChainTop);
      ADD_ARG (XtNbottom, XtChainTop);
      ADD_ARG (XtNleft, XtChainLeft);
      ADD_ARG (XtNright, XtChainLeft);

      ADD_ARG (XtNborderWidth, 2);
      ADD_ARG (XtNborderColor, the_application->red_pxl);
      ADD_ARG (XtNforeground, the_application->blue_pxl);

      wdgt = XtCreateManagedWidget ("knownObjectList", listWidgetClass,
				    the_application->form_wdgt,
                                    ARGLIST(), ARGNR());
      XtAddCallback (wdgt, XtNcallback,
	             (XtCallbackProc) KnownObject::callback_handler,
	             (caddr_t) this);
      XtAddEventHandler(wdgt, ButtonPressMask, FALSE,
                	(XtEventHandler) Application::drag_callback_handler,
                        NULL);
      Application::scroll_viewport (wdgt);
   }
}

void KnownObject::callback_handler (Widget  w,
				    caddr_t client_data, caddr_t call_data)
	// An item of a displayed object was selected.
	// item[0]   --> Activate object command menu for this object.
	// otherwise --> Display the selected object.
{  int i = ((XawListReturnStruct*) call_data)->list_index;
   KnownObject *kobj = (KnownObject*) client_data;

   XawListUnhighlight (w);

   kobj->item_selected (i);
}

// ---------------------------------------------------------------------------
// *************************  class: ObjectCmdMenu  **************************
// ---------------------------------------------------------------------------

void ObjectCmdMenu::activate (KnownObject *Kobj, char* menulist[])
	// Activate this menu for the given object.
{  WdgtPosition pos = WdgtPosition::at (Kobj->wdgt);

   XawListChange (wdgt, menulist, 0, 0, TRUE);
   XawListHighlight (wdgt, 0);
   XtMoveWidget (wdgt, pos.x, pos.y);
   XtMapWidget (wdgt);
   XRaiseWindow (XtDisplay (wdgt), XtWindow (wdgt));
   Application::scroll_viewport(Kobj->wdgt);
   XtAddGrab (wdgt, TRUE, FALSE);

   kobj = Kobj;
}

void ObjectCmdMenu::install (char *items[])
	// Install this object menu at the user interface handler.
	// (Pre: There must be just one such menu.)
{
   RESET_ARGLIST();
   ADD_ARG (XtNverticalList, TRUE);
   ADD_ARG (XtNlist, (XtArgVal)items);
   ADD_ARG (XtNforceColumns, TRUE);
   ADD_ARG (XtNdefaultColumns, 1);
   ADD_ARG (XtNtop, XtChainTop);
   ADD_ARG (XtNbottom, XtChainTop);
   ADD_ARG (XtNleft, XtChainLeft);
   ADD_ARG (XtNright, XtChainLeft);
   ADD_ARG (XtNmappedWhenManaged, FALSE);
   ADD_ARG (XtNborderColor,the_application->red_pxl);
   ADD_ARG (XtNforeground, the_application->green_pxl);

   wdgt = XtCreateManagedWidget ("menuList",
            listWidgetClass, the_application->form_wdgt, ARGLIST(), ARGNR());
   XtAddCallback (wdgt, XtNcallback,
		  (XtCallbackProc) ObjectCmdMenu::callback_handler,
		  (caddr_t) this);
}

void ObjectCmdMenu::callback_handler (Widget  w,
				      caddr_t ocmptr, caddr_t call_data)
	// A command of an object menu has been selected.
	// (Pre: There must be just one such menu: *the_object_menu.)
{  int           cmd   = (int)( ((XawListReturnStruct*)call_data)->list_index);
   ObjectCmdMenu *menu = (ObjectCmdMenu *) ocmptr;

   XtRemoveGrab (w);
   XtUnmapWidget (w);

   err_block
      menu->do_it (cmd);
   err_exception
      the_info_line.display_OBST_error();
   err_block_end
}


// ---------------------------------------------------------------------------
// ************************  class: WdgtPosition  ****************************
// ---------------------------------------------------------------------------

const int WdgtPosition::default_distance = 4;

WdgtPosition WdgtPosition::root ()
{  WdgtPosition result;
   result.x = WdgtPosition::default_distance;
   result.y = WdgtPosition::default_distance;
   return result;
}

WdgtPosition WdgtPosition::at (Widget w)
{  WdgtPosition result;

   RESET_ARGLIST();
   ADD_ARG (XtNhorizDistance, (XtArgVal)&result.x);
   ADD_ARG (XtNvertDistance, (XtArgVal)&result.y);
   GETVALUES (w);
   return result;
}

WdgtPosition WdgtPosition::next_to (Widget w)
{  WdgtPosition result;
   Dimension    width;

   RESET_ARGLIST();
   ADD_ARG (XtNhorizDistance, (XtArgVal)&result.x);
   ADD_ARG (XtNvertDistance, (XtArgVal)&result.y);
   ADD_ARG (XtNwidth, (XtArgVal)&width);
   GETVALUES (w);
   result.x += width + WdgtPosition::default_distance;
   return result;
}

WdgtPosition WdgtPosition::center (Widget w)
{  WdgtPosition result;
   Dimension    width, height;

   RESET_ARGLIST();
   ADD_ARG (XtNhorizDistance, (XtArgVal)&result.x);
   ADD_ARG (XtNvertDistance,  (XtArgVal)&result.y);
   ADD_ARG (XtNwidth, 	      (XtArgVal)&width);
   ADD_ARG (XtNheight,	      (XtArgVal)&height);
   GETVALUES (w);

   result.x += width/2;
   result.y += height/2;

   return result;
}


// ---------------------------------------------------------------------------
// ************************** class: ModalDialog *****************************
// ---------------------------------------------------------------------------

void ModalDialog::activate (sos_Bool grab)
{
   WdgtPosition center = WdgtPosition::center (the_application->viewport_wdgt);
   Dimension    width, height;
   Position a,b;

   RESET_ARGLIST();
   ADD_ARG (XtNx, (XtArgVal)&a);
   ADD_ARG (XtNy, (XtArgVal)&b);
   GETVALUES (the_application->form_wdgt);

   XtManageChild (main_wdgt);
   RESET_ARGLIST();
   ADD_ARG (XtNwidth,  (XtArgVal)&width);
   ADD_ARG (XtNheight, (XtArgVal)&height);
   GETVALUES (main_wdgt);
   XtUnmanageChild (main_wdgt);

   RESET_ARGLIST();
   ADD_ARG (XtNhorizDistance, center.x - width/2 - a);
   ADD_ARG (XtNvertDistance,  center.y - height/2 - b);
   SETVALUES (main_wdgt);

   XtManageChild (main_wdgt);
   XRaiseWindow (XtDisplay (main_wdgt), XtWindow (main_wdgt));
   if (grab)
      XtAddGrab (main_wdgt, TRUE, FALSE);

   active = TRUE;
   do
   {  XEvent event;
      XtAppNextEvent (the_application->app_context, &event);
      XtDispatchEvent (&event);
   } while (active);
   if (grab)
      XtRemoveGrab (main_wdgt);
}

void ModalDialog::callback_handler ()
{  active = FALSE;
   XtUnmanageChild (main_wdgt);
}

// ---------------------------------------------------------------------------
// *********************** class: VarTextDialog ******************************
// ---------------------------------------------------------------------------

static VarTextDialog **vtd_pool;
static int           vtd_poolsize = 0;

VarTextDialog* VarTextDialog::alloc_dialog ()
{  VarTextDialog* result;

   for (int i = 0;  i < vtd_poolsize;  ++ i)
   {  if (vtd_pool[i])
      {  result      = vtd_pool[i];
	 vtd_pool[i] = NULL;
	 return result;
      }
   }
   if (i = vtd_poolsize)
      vtd_pool = (VarTextDialog**)
		    REALLOC ((void*)vtd_pool,
			     sizeof(VarTextDialog*) * (vtd_poolsize += 2));
   else
      vtd_pool = (VarTextDialog**)
	            malloc (sizeof(VarTextDialog*) * (vtd_poolsize = 2));
   for (; i < vtd_poolsize;  ++ i)
      vtd_pool[i] = new VarTextDialog;

   i           = vtd_poolsize - 1;
   result      = vtd_pool[i];
   vtd_pool[i] = NULL;
   return result;
}

void VarTextDialog::free_dialog (VarTextDialog* vtd)
{  for (int i = 0;  i < vtd_poolsize;  ++ i)
   {  if (!vtd_pool[i])
      {  vtd_pool[i] = vtd; 
	 return;
      }
   }
   err_assert (FALSE, "VarTextDialog::free_dialog - too many free calls");
}

void VarTextDialog::cancel_callback_handler (Widget, caddr_t vtdptr, caddr_t)
{  VarTextDialog *vtd = (VarTextDialog*)vtdptr;

   vtd->callback_handler();
   vtd->do_pressed = FALSE;
}

void VarTextDialog::do_callback_handler (Widget, caddr_t vtdptr, caddr_t)
{  VarTextDialog *vtd = (VarTextDialog*)vtdptr;

   for (int i=vtd->nr_pars; --i>=0; )
   {
      RESET_ARGLIST();
      ADD_ARG (XtNstring, (XtArgVal)&(vtd->text[i]));
      GETVALUES(vtd->par_wdgts[i]);
   }
   vtd->callback_handler();
   vtd->do_pressed = TRUE;
}

void VarTextDialog::install()
{
   RESET_ARGLIST();
   ADD_ARG (XtNtop, XtChainTop);
   ADD_ARG (XtNbottom, XtChainTop);
   ADD_ARG (XtNleft, XtChainLeft);
   ADD_ARG (XtNright, XtChainLeft);
   ADD_ARG (XtNresizable, TRUE);

   ADD_ARG (XtNborderWidth, 2);
   ADD_ARG (XtNborderColor, the_application->red_pxl);
   ADD_ARG (XtNforeground, the_application->blue_pxl);

   main_wdgt = XtCreateWidget ("textBox", formWidgetClass,
			       the_application->form_wdgt,
			       ARGLIST(), ARGNR());
   XtAddEventHandler (main_wdgt, ButtonPressMask, FALSE,
		      (XtEventHandler) Application::drag_callback_handler,
		      NULL);
		      
   RESET_ARGLIST();
   ADD_ARG (XtNborderWidth, 0);
   ADD_ARG (XtNresizable, TRUE);
   header = XtCreateManagedWidget ("label", labelWidgetClass, main_wdgt,
 				   ARGLIST(), ARGNR());
   
   do_but     =
   cancel_but = NULL;
}

void VarTextDialog::prepare_start (char* title)
{
   RESET_ARGLIST();
   ADD_ARG (XtNlabel, (XtArgVal)title); 
   SETVALUES (header);

   nr_pars = 0;
}

void VarTextDialog::add_par()
{
   if (nr_pars EQ size)
   {
      if (size)
      {  size_t bytes = sizeof(Widget)*(size += 2);
	 par_wdgts    = (Widget*)REALLOC (par_wdgts, bytes);
	 label_wdgts  = (Widget*)REALLOC (label_wdgts, bytes);
         text         = (char**) REALLOC (text, 
					 (size_t)sizeof(char*)*size);
      }
      else
      {
	 par_wdgts   = new Widget [size = 2];
	 label_wdgts = new Widget [size    ];
	 text        = new char*  [size    ];
      }
      par_wdgts [size-1]   = par_wdgts [size-2]   =
      label_wdgts [size-1] = label_wdgts [size-2] = NULL;
      text[size-1]         = text[size-2]         = NULL;
   }
}

void VarTextDialog::prepare_add (char* label)
{
   add_par();

   if (!par_wdgts [nr_pars])
   {  RESET_ARGLIST();
      ADD_ARG (XtNstring, (XtArgVal)"");
      ADD_ARG (XtNeditType, XawtextEdit);
      ADD_ARG (XtNscrollHorizontal, TRUE);
      ADD_ARG (XtNfromVert, (XtArgVal)(nr_pars ? par_wdgts [nr_pars - 1]
				               : header));

      par_wdgts [nr_pars] = 
	                XtCreateManagedWidget ("textField",
					       asciiTextWidgetClass,
					       main_wdgt, ARGLIST(), ARGNR());

      static char trans[] = "<Key>Return : end-of-line()";
      XtOverrideTranslations (par_wdgts [nr_pars] , 
			      XtParseTranslationTable(trans));

      XawTextEnableRedisplay (par_wdgts [nr_pars]);
   }
   
   RESET_ARGLIST();
   ADD_ARG (XtNlabel, (XtArgVal)label);
   if (label_wdgts [nr_pars])
      SETVALUES (label_wdgts [nr_pars]);

   else
   {  ADD_ARG (XtNresizable, TRUE);
      ADD_ARG (XtNborderWidth, 0);
      ADD_ARG (XtNfromHoriz, (XtArgVal)par_wdgts [nr_pars]);
      ADD_ARG (XtNfromVert, (XtArgVal)((nr_pars) ? par_wdgts [nr_pars - 1]
				                 : header));
      label_wdgts[nr_pars] = 
                        XtCreateManagedWidget ("label", labelWidgetClass,
					       main_wdgt ,ARGLIST(), ARGNR());
   }
   nr_pars++;
}

sos_Bool VarTextDialog::activate(sos_Bool grab)
{
   for (int i=nr_pars; i<size && 0; i++)
   {  if (par_wdgts [i])
      {
	 XtDestroyWidget (par_wdgts [i]);     par_wdgts[i]   = NULL;
	 XtDestroyWidget (label_wdgts [i]);   label_wdgts[i] = NULL;
      }
   }

   RESET_ARGLIST();
   ADD_ARG (XtNfromVert, (XtArgVal)(nr_pars ? par_wdgts[nr_pars-1] : header));
   ADD_ARG (XtNlabel, (XtArgVal)"  do  ");
   do_but = XtCreateManagedWidget ("doBut", commandWidgetClass, main_wdgt,
				   ARGLIST(), ARGNR());
   XtAddCallback (do_but, XtNcallback, (XtCallbackProc) do_callback_handler,
		  (XtPointer)this);

   RESET_ARGLIST();
   ADD_ARG (XtNlabel, (XtArgVal)"cancel");
   ADD_ARG (XtNfromHoriz, (XtArgVal)do_but);
   ADD_ARG (XtNfromVert, (XtArgVal)(nr_pars ? par_wdgts[nr_pars-1] : header));
   ADD_ARG (XtNhorizDistance, 20);
   cancel_but = XtCreateManagedWidget ("cancelBut", commandWidgetClass, 
				       main_wdgt, ARGLIST(), ARGNR());
   XtAddCallback (cancel_but, XtNcallback,
		  (XtCallbackProc) cancel_callback_handler, (XtPointer)this);

   ModalDialog::activate(grab); // GCC2.2.2 SQUIRK: must call explicitly
   return do_pressed;
}
   
void VarTextDialog::finish()
{
   for (int i=nr_pars; --i>=0; )
   {
      text[i]        = NULL;
      par_wdgts[i]   =
      label_wdgts[i] = NULL;
   }
   XtDestroyWidget (main_wdgt);
   main_wdgt =
   header    = NULL;

   VarTextDialog::free_dialog (this);
}

// ---------------------------------------------------------------------------
// *********************** class: ConfirmDialog ******************************
// ---------------------------------------------------------------------------

void ConfirmDialog::activate(char *question)
{
   RESET_ARGLIST();
   ADD_ARG (XtNlabel, (XtArgVal)question);
   SETVALUES (question_label);

   ModalDialog::activate(TRUE); // GCC2.2.2 SQUIRK: must call explicitly
}

void ConfirmDialog::install()
{  Widget do_but, cancel_but;

   RESET_ARGLIST();
   ADD_ARG (XtNtop, XtChainTop);
   ADD_ARG (XtNleft, XtChainLeft);
   ADD_ARG (XtNright, XtChainLeft);
   ADD_ARG (XtNbottom, XtChainTop); 
   ADD_ARG (XtNresizable, TRUE);
   main_wdgt = XtCreateWidget
		 ("textBox", formWidgetClass, the_application->viewport_wdgt,
		  ARGLIST(), ARGNR());
   XtAddEventHandler (main_wdgt, ButtonPressMask, FALSE,
		    (XtEventHandler) Application::drag_callback_handler, NULL);

   RESET_ARGLIST();
   ADD_ARG (XtNresizable, TRUE);
   ADD_ARG (XtNborderWidth, 0);
   question_label = XtCreateManagedWidget ("label", 
                       labelWidgetClass, main_wdgt, ARGLIST(), ARGNR());

   RESET_ARGLIST();
   ADD_ARG (XtNlabel, (XtArgVal)"  do  ");
   ADD_ARG (XtNfromVert, (XtArgVal)question_label);
   do_but = XtCreateManagedWidget ("doBut", 
                commandWidgetClass, main_wdgt, ARGLIST(), ARGNR());
   XtAddCallback (do_but, XtNcallback,
	          (XtCallbackProc) ConfirmDialog::do_callback_handler,
		  (XtPointer)this);

   RESET_ARGLIST();
   ADD_ARG (XtNlabel, (XtArgVal)"cancel");
   ADD_ARG (XtNfromVert, (XtArgVal)question_label);
   ADD_ARG (XtNfromHoriz, (XtArgVal)do_but);
   ADD_ARG (XtNhorizDistance, 20);
   cancel_but = XtCreateManagedWidget ("cancelBut",
                   commandWidgetClass, main_wdgt, ARGLIST(), ARGNR());
   XtAddCallback (cancel_but, XtNcallback,
		  (XtCallbackProc) ConfirmDialog::cancel_callback_handler, 
                  (XtPointer)this);
}

void ConfirmDialog::cancel_callback_handler (Widget, caddr_t cdptr, caddr_t)
{
   ConfirmDialog *cd = (ConfirmDialog*)cdptr;
   cd->callback_handler();
   cd->answer = FALSE;
}

void ConfirmDialog::do_callback_handler (Widget, caddr_t cdptr, caddr_t)
{
   ConfirmDialog *cd = (ConfirmDialog*)cdptr;
   cd->callback_handler();
   cd->answer = TRUE;   
}

// ---------------------------------------------------------------------------
// ************************** class: InfoLine *******************************
// ---------------------------------------------------------------------------

void InfoLine::display (const char *msg)
{
   strncpy (text, msg, InfoLine_INFO_LENGTH);
   text [InfoLine_INFO_LENGTH] = EOS;

   for (char *ch = text;  *ch;  ch ++)
      if (isspace(*ch))
         *ch = ' ';
      else if (NOT isprint(*ch))
         *ch = '?';

     RESET_ARGLIST();
     ADD_ARG (XtNlabel, (XtArgVal)text);
     SETVALUES (wdgt);
}

void InfoLine::display_OBST_error ()
{  display (err_last_raised());
}

void InfoLine::install ()
{  text[0] = EOS;

   RESET_ARGLIST();
   ADD_ARG (XtNlabel, (XtArgVal)text);
   ADD_ARG (XtNborderWidth, 0);
   ADD_ARG (XtNwidth, 800);
   ADD_ARG (XtNjustify, XtJustifyLeft);
   ADD_ARG (XtNfromVert, (XtArgVal)the_application->viewport_wdgt);
   wdgt = XtCreateManagedWidget
	     ("infoLine", labelWidgetClass, the_application->app_form_wdgt,
	      ARGLIST(), ARGNR());
}


// ---------------------------------------------------------------------------
// ************************** class: MenuSystem ******************************
// ---------------------------------------------------------------------------

const int MenuSystem::NO_MENU = -1;


void MenuSystem::install (int no, char* header_items[], char** list_items[])
{
   no_menus = no;
   headers  = new Widget[no];
   lists    = new Widget[no];

   for (int i=0; i<no_menus; i++)
   {  Widget w = ((i == 0) ? NULL
		          : headers[i-1]);
      RESET_ARGLIST();
      ADD_ARG (XtNborderWidth, 0);
      ADD_ARG (XtNlabel, (XtArgVal)header_items[i]);
      ADD_ARG (XtNfromHoriz, (XtArgVal)w);
      ADD_ARG (XtNhorizDistance, 0);
      ADD_ARG (XtNvertDistance, 0);
      ADD_ARG (XtNtop, XtChainTop);
      ADD_ARG (XtNbottom, XtChainTop);
      ADD_ARG (XtNleft, XtChainLeft);
      ADD_ARG (XtNright, XtChainLeft);
      ADD_ARG (XtNforeground, the_application->red_pxl);

      headers[i] = XtCreateManagedWidget
      			 ("menuHeader", commandWidgetClass,
			  the_application->app_form_wdgt,
			  ARGLIST(), ARGNR());
      XtAddCallback (headers[i], XtNcallback,
		     (XtCallbackProc) MenuSystem::header_callback_handler,
		     (caddr_t) i);

      RESET_ARGLIST();
      ADD_ARG (XtNlist, (XtArgVal)list_items[i]);
      ADD_ARG (XtNverticalList, TRUE);
      ADD_ARG (XtNforceColumns, TRUE);
      ADD_ARG (XtNdefaultColumns, 1);
      ADD_ARG (XtNmappedWhenManaged, FALSE);
      ADD_ARG (XtNfromHoriz, (XtArgVal)w);
      ADD_ARG (XtNfromVert, (XtArgVal)headers[i]);
      ADD_ARG (XtNhorizDistance, (XtArgVal)0);
      ADD_ARG (XtNvertDistance, (XtArgVal)0);
      ADD_ARG (XtNtop, XtChainTop);
      ADD_ARG (XtNbottom, XtChainTop);
      ADD_ARG (XtNleft, XtChainLeft);
      ADD_ARG (XtNright, XtChainLeft);

      ADD_ARG (XtNborderWidth, 2);
      ADD_ARG (XtNborderColor,the_application->red_pxl);
      ADD_ARG (XtNforeground, the_application->green_pxl);

      lists[i] = XtCreateManagedWidget
      			("menuList", listWidgetClass, 
			 the_application->app_form_wdgt, ARGLIST(), ARGNR());
      XtAddCallback (lists[i], XtNcallback,
		     (XtCallbackProc) MenuSystem::list_callback_handler,
		     (XtPointer)this);
   }
}

void MenuSystem::change_list (int no, char* list_items[])
{
   Widget lw = the_menu_system->lists[no];
   XawListChange (lw, list_items, 0, 0, TRUE);
}

void MenuSystem::header_callback_handler (Widget, caddr_t header_nr, caddr_t)
{  int    menuID = (int) header_nr;
   Widget lw     = the_menu_system->lists[menuID];

   if (menuID == the_menu_system->active_menu)
   {
      XtRemoveGrab (lw);
      for (int i = the_menu_system->no_menus;  --i >= 0; )
         XtRemoveGrab (the_menu_system->headers[i]);
      the_menu_system->active_menu = NO_MENU;
      XtUnmapWidget (lw);
   }
   else
   {  if (the_menu_system->active_menu != NO_MENU)
      {  Widget aw = the_menu_system->lists[the_menu_system->active_menu]; 
         XtRemoveGrab (aw);
	 XtUnmapWidget (aw);
      }
      else
      {  for (int j=0; j<the_menu_system->no_menus; j++)
	    XtAddGrab (the_menu_system->headers[j], FALSE, FALSE);
      }
      the_menu_system->active_menu = menuID;
      XtMapWidget (lw);
      XRaiseWindow (XtDisplay (lw), XtWindow (lw));
      XtAddGrab (lw, FALSE, FALSE);
   }
}

void MenuSystem::list_callback_handler (Widget  w, caddr_t, caddr_t call_data)
{  int item = ((XawListReturnStruct*) call_data)->list_index;
   int menu = the_menu_system->active_menu;

   XawListUnhighlight (w);
  
   for (int j = the_menu_system->no_menus;  --j >= 0; )
     XtRemoveGrab (the_menu_system->headers[j]);
   XtUnmapWidget (w);

   the_menu_system->active_menu = NO_MENU;

   the_menu_system->do_it (menu, item);
}


// ---------------------------------------------------------------------------
// ************************** class: Application *****************************
// ---------------------------------------------------------------------------

#define PTR_HTAB_SIZE  512
#define HASH_PTR(ptr)  (unsigned long)(ptr) % PTR_HTAB_SIZE

struct gtools_PtrEntry
{
   gtools_PtrEntry *next;
   void *key;
   void *info;
};

LOCAL gtools_PtrEntry *ptr_htab[PTR_HTAB_SIZE];

void Application::enter_ptr (void* k, void* i)
{  register gtools_PtrEntry **e;
   for (e = &ptr_htab[ HASH_PTR(k) ];  *e AND (*e)->key != k; e = &(*e)->next)
      ;

   if (! *e)
   {
      *e = new gtools_PtrEntry;
      (*e)->next = NULL;
      (*e)->key  = k;
      (*e)->info = i;
   }
}

void* Application::lookup_ptr (void* k)
{  register gtools_PtrEntry **e;

   for (e = &ptr_htab[ HASH_PTR(k) ];  *e AND (*e)->key != k;  e = &(*e)->next)
      ;
   return (*e) ? (*e)->info
	       : NULL;
}

// -----------

sos_Object Application::make_object (sos_Container ct, sos_Offset os)
{  sos_Typed_id tpid = sos_Typed_id::make(sos_Id::make (ct, os));
   sos_Object   o;

   if (obst_ID_valid (tpid) EQ NOT_EXISTING)
   {  err_raise (err_USE,
		 "Application::make_object", "invalid object id", FALSE);
      o = NO_OBJECT;
   }
   else
      o = sos_Object::make (tpid);

   return o;
}

void Application::access_ct (sos_Container ct)
{  sos_Container_status status    = ct.status();
   sos_Open_result	op_status;
   if (status EQ UNAVAILABLE)
      op_status = ct.open (WRITING, WAITING);
   else
      op_status = (status EQ READABLE) ? ct.access (WRITING, WAITING)
				       : OPENED;
   if (op_status != OPENED)
   {  smg_String msg = smg_String("can't open container ")
		       + (int)ct
		       + " for writing";
      err_raise (err_USE, msg.make_Cstring (SMG_BORROW));
      exit (1);
   }
}

void Application::commit ()
{  sos_Container_set modified = sos_Container_set::open_containers (WRITEABLE);
   modified += sos_Container_set::open_containers (DESTROYED);
   modified.close();
}

void Application::reset ()
{ 
   sos_Container_set modified = sos_Container_set::open_containers (WRITEABLE);
   modified += sos_Container_set::open_containers (DESTROYED);
   modified.reset();
   the_known_objects->remove_all();
   modified.close();
}

void Application::commit_and_exit ()
{  sos_Container_set modified = sos_Container_set::open_containers (WRITEABLE);
   modified += sos_Container_set::open_containers (DESTROYED);
   modified.close();
   exit (0);
}

void Application::run ()
{  XtRealizeWidget (app_shell_wdgt);

   display_root (FALSE);

   XtAppMainLoop (app_context);
}

void Application::install (int argc,     char* argv[],    char*  appl_name,
			   int no_menus, char* headers[], char** menulists[])
{
#if XtSpecificationRelease == 4
   app_shell_wdgt = XtAppInitialize (&app_context, appl_name, NULL, 0,
	 			     (Cardinal*)&argc, argv, NULL, NULL, 0);
#else
   app_shell_wdgt = XtAppInitialize (&app_context, appl_name, NULL, 0,
	 			     &argc, argv, NULL, NULL, 0);
#endif
   init (argc, argv);

#define ApplOffset(f) XtOffsetOf(Application, f)
#define ColorRes(name,class,field,default) \
           { name, class, XtRPixel, sizeof(Pixel), ApplOffset(field),\
	     XtRString, (XtPointer)(default) }

static XtResource appl_res[] = {
   ColorRes("red", "Color", red_pxl, "red"),
   ColorRes("green", "Color", green_pxl, "darkgreen"),
   ColorRes("blue",  "Color", blue_pxl,  "blue") };

   XtGetApplicationResources (app_shell_wdgt, (XtPointer)the_application,
			      (XtResourceList)appl_res, XtNumber(appl_res),
			      NULL, 0);

   Display *disp = XtDisplay (app_shell_wdgt);

   XrmDatabase db = XtDatabase (disp);
   XrmPutLineResource (&db, "*textField*Width: 200");
   XrmPutLineResource (&db, "*textField.Height: 22");
   XrmPutLineResource (&db, "*textBox*Scrollbar*Thickness: 5");
   XrmPutLineResource (&db, "*appView.Height: 500");
   XrmPutLineResource (&db, "*appView.Width: 800");

   RESET_ARGLIST();
   ADD_ARG (XtNdefaultDistance, 0);
   app_form_wdgt = XtCreateManagedWidget
	       		("appForm", formWidgetClass, app_shell_wdgt,
	       		 ARGLIST(), ARGNR());

   the_menu_system->install(no_menus, headers, menulists);

   RESET_ARGLIST();
   ADD_ARG (XtNforceBars, TRUE);
   ADD_ARG (XtNallowHoriz, TRUE);
   ADD_ARG (XtNallowVert, TRUE);
   ADD_ARG (XtNuseBottom, TRUE);
   ADD_ARG (XtNuseRight, TRUE);
   ADD_ARG (XtNfromVert, (XtArgVal)the_menu_system->headers[0]);
   ADD_ARG (XtNtop, XtChainTop);
   ADD_ARG (XtNbottom, XtChainBottom);
   ADD_ARG (XtNleft, XtChainLeft);
   ADD_ARG (XtNright, XtChainRight);
   viewport_wdgt = XtCreateManagedWidget
	       		("appView", viewportWidgetClass, app_form_wdgt,
	       		 ARGLIST(), ARGNR());
   RESET_ARGLIST();
   ADD_ARG (XtNdefaultDistance, 0);    // default_distance);
   form_wdgt = XtCreateManagedWidget
	       ("form", formWidgetClass, viewport_wdgt, ARGLIST(), ARGNR());

   the_text_dialog      = VarTextDialog::alloc_dialog();
   the_container_dialog = VarTextDialog::alloc_dialog();

   the_text_dialog->install();
   the_container_dialog->install();

   the_confirm_dialog.install ();
   the_info_line.install ();
   the_object_menu->install();

   run();
}

void Application::scroll_viewport (Widget wdgt)
{
   Dimension view_width;
   Dimension wdgt_x;
   Position form_x, form_y;
   sos_Bool changed = FALSE;
   WdgtPosition next_pos = WdgtPosition::next_to (wdgt);
   
   RESET_ARGLIST();
   ADD_ARG (XtNx, (XtArgVal)&wdgt_x);
   GETVALUES (wdgt);

   RESET_ARGLIST();
   ADD_ARG (XtNwidth, (XtArgVal)&view_width);
   GETVALUES (the_application->viewport_wdgt);

   RESET_ARGLIST();
   ADD_ARG (XtNx, (XtArgVal)&form_x);
   ADD_ARG (XtNy, (XtArgVal)&form_y);   
   GETVALUES (the_application->form_wdgt);

   if ((view_width - form_x) < next_pos.x)
   {
      form_x = view_width - next_pos.x;
      changed = TRUE;
   }
   if (wdgt_x < -form_x)
   {
      form_x = -wdgt_x + WdgtPosition::default_distance;;
      changed = TRUE;
   }
   if (next_pos.y < -form_y)
   {
      form_y = -next_pos.y + WdgtPosition::default_distance;
      changed = TRUE;
   }
   if (changed)
   {
      RESET_ARGLIST();
      ADD_ARG(XtNx, (XtArgVal)form_x);
      ADD_ARG(XtNy, (XtArgVal)form_y);
      SETVALUES(the_application->form_wdgt);

      XtMoveWidget (the_application->form_wdgt, form_x, form_y);
   }
}


// ***************************  Application: Dragging  *****************

inline void draw_frame (Display *disp, Window win, GC dragGC,
		 	int win_x, int win_y,
		 	Dimension width, Dimension height)
{  XDrawRectangle (disp, win, dragGC, win_x, win_y, width, height);
}

inline void query_pointer (Display *disp, Window win,
			   int &win_x, int &win_y,
			   int xoffs, int yoffs)
{  Window	rootwin, childwin;
   int		root_x, root_y;
   unsigned int	mask;

   XQueryPointer (disp, win,
		  &rootwin, &childwin,
		  &root_x, &root_y,
		  &win_x, &win_y, &mask);
   win_x -= xoffs; if (win_x < 0) win_x = 0;
   win_y -= yoffs; if (win_y < 0) win_y = 0;
}

void Application::drag_callback_handler (Widget w, caddr_t, XEvent *event)
{  Widget	parent = XtParent (w);
   Display	*disp  = XtDisplay (parent);
   Window 	win    = XtWindow (parent);
   Dimension	width, height, borderwidth;
   int		win_x, win_y;
   int		xoffs, yoffs;

   if (event->xbutton.button != 2)
      return;

   static GC  dragGC;
   static int initGC = TRUE;
   if (initGC)			    // initialize GraphicContext in first call
   {  initGC = FALSE;
      dragGC = XCreateGC (disp, win, 0, NULL);
      XSetForeground (disp, dragGC, BlackPixel (disp, 0));
      XSetSubwindowMode (disp, dragGC, IncludeInferiors);
      XSetFunction (disp, dragGC, GXxor);
   }

   RESET_ARGLIST();
   ADD_ARG (XtNwidth, (XtArgVal)&width);
   ADD_ARG (XtNheight, (XtArgVal)&height);
   ADD_ARG (XtNborderWidth, (XtArgVal)&borderwidth);
   GETVALUES (w);

   xoffs = event->xbutton.x + borderwidth;
   yoffs = event->xbutton.y + borderwidth;

   query_pointer (disp, win, win_x, win_y, xoffs, yoffs);
   draw_frame (disp, win, dragGC, win_x, win_y, width, height);

   XChangeActivePointerGrab (disp,
                             (unsigned int) (ButtonReleaseMask |
                                             ButtonMotionMask |
                                             OwnerGrabButtonMask),
			     XCreateFontCursor (disp, XC_fleur),
			     CurrentTime);
   XEvent ev;
   do
   {  XtAppNextEvent (the_application->app_context, &ev);
      
      if (ev.type EQ MotionNotify)
      {  draw_frame (disp, win, dragGC, win_x, win_y, width, height);
	 query_pointer (disp, win, win_x, win_y, xoffs, yoffs);
	 draw_frame (disp, win, dragGC, win_x, win_y, width, height);
      }
   } while (ev.type != ButtonRelease  ||  ev.xbutton.button != 2);

   draw_frame (disp, win, dragGC, win_x, win_y, width, height);
   query_pointer (disp, win, win_x, win_y, xoffs, yoffs);

   XtMoveWidget (w, win_x, win_y);
   XRaiseWindow (XtDisplay (w), XtWindow (w));
   RESET_ARGLIST();
   ADD_ARG (XtNhorizDistance, win_x);
   ADD_ARG (XtNvertDistance, win_y);
   SETVALUES (w);

   XtUnmanageChild (w);
   XtManageChild (w);
}
