 /* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
// **************************************************************************
// Module mta
// **************************************************************************
// implements methods of all classes declared in schema "mta" 
// **************************************************************************

#include "obst_progstd.h"
#include "obst.h"
#include "smg.h"
#include "mta_err.h"
#include "trc_mta.h"

// dir_obst.h must be included, since scp "optimizes" the call to
// sos_Object_Directory::root to _sos_Object_Directory::root.
#include "dir_obst.h"
#include "cci_use.h"

#include "mta_obst.h"


// -------------------------------------------------------------------------
// sos_Import
// -------------------------------------------------------------------------
// *************************************************************************
sos_Type_descr sos_Import::lookup_type (sos_String	name,
					sos_Import_mode imp_mode)
// *************************************************************************
{  T_PROC ("sos_Import::lookup_type")
   TT (mta_H, T_ENTER);

   sos_Type_descr result = self.get_module().get_type_table() [name];
   
   if VALID (result) 
      if (imp_mode > self.get_mode())
	 self.set_mode(imp_mode);

   TT (mta_H, T_LEAVE);
   return result;
}


// -------------------------------------------------------------------------
// sos_Schema_module
// -------------------------------------------------------------------------
// *************************************************************************
sos_Schema_module_Directory sos_Schema_module::schema_dir ()
// *************************************************************************
{  T_PROC ("sos_Schema_module::schema_dir")
   TT (mta_H, T_ENTER);

   static sos_String name = sos_String::create (TEMP_CONTAINER, "sos_schemas");

   sos_Container_status root_status = ROOT_CONTAINER.status();

   sos_Bool read_open = sos_Bool((root_status == WRITEABLE) OR
				 (root_status == READABLE));
   if (NOT read_open)
      ROOT_CONTAINER.open (READING, WAITING);
	
   sos_Directory root_dir = sos_Object_Directory::root();
   sos_Schema_module_Directory dir
      = sos_Schema_module_Directory::make (root_dir[name]);
   if (NOT read_open)
       ROOT_CONTAINER.close ();

   if INVALID(dir)
   {  sos_Container dir_cnt = sos_Container::create();
      dir = sos_Schema_module_Directory::create (dir_cnt, name);
      sos_Bool write_open = sos_Bool(root_status == WRITEABLE);
      if (NOT write_open)
	 ROOT_CONTAINER.open (WRITING, WAITING);
      dir_cnt.close();
      root_dir.insert (name, dir);
      if (NOT write_open)
      {
         ROOT_CONTAINER.close ();
	 // dir.container().close();              geaendert 10.08.93 (bs) 
	 // dir.container().open (WRITING, WAITING);
      }
   }
   TT (mta_H, T_LEAVE);
   return dir;
}

// *************************************************************************
sos_Schema_module sos_Schema_module::lookup (sos_String name)
// *************************************************************************
{  T_PROC ("sos_Schema_module::lookup")
   TT (mta_H, T_ENTER);

   static sos_Schema_module_Directory sd = sos_Schema_module::schema_dir ();

   sd.container().open (READING, WAITING);
   sos_Schema_module sm = sd [name];
   sd.container().close ();

   TT (mta_H, T_LEAVE);
   return sm;
}

// *************************************************************************
sos_Schema_module sos_Schema_module::retrieve (sos_Container schema_ct)
// *************************************************************************
{  T_PROC ("sos_Schema_module::retrieve")
   TT (mta_H, T_ENTER);

   sos_Schema_module sm = sos_Schema_module::make (schema_ct.root_object());

   TT (mta_H, T_LEAVE);
   return sm;
}

// *************************************************************************
sos_Type_descr sos_Schema_module::lookup_type (sos_String name, 
					       sos_Import_mode imp_mode)
// *************************************************************************

// search for a type declaration with 'name' in the type table
// of the schema module. If no declaration is found search for 'name'
// in the type tables of the imported modules. If the search fails
// return NO_OBJECT. If found in an import, the mode of the import is
// raised to the demanded level imp_mode, if necessary.

{  T_PROC ("sos_Schema_module::lookup_type")
   TT (mta_H, T_ENTER);

   static char 		     mode_code[2] = { '\0', '\0' };
   static sos_String         key	  = sos_String::create (TEMP_CONTAINER);
   static sos_Object_sos_Object_Mapping_Hash tmp_dir
     = sos_Object_sos_Object_Mapping_Hash::create (TEMP_CONTAINER,
						   FALSE, // list_cursor
						   TRUE,  // key_based_on_equal
						   FALSE);// info_b.o.e
   mode_code[0] = 'A' + (int)imp_mode;

   smg_String _key = smg_String(mode_code) + name;

   sos_Type_descr result;

   if (_key.length() == 1)
   {  tmp_dir.clear();
      result = sos_Type_descr::make (NO_OBJECT);
   }
   else
   {  char* keystr = _key.make_Cstring (SMG_BORROW);
      key.assign_Cstring(keystr);

      result = sos_Type_descr::make(tmp_dir[(sos_Object)key]);
      if INVALID(result)
      {  result = self.get_type_table()[name];

	 if INVALID(result) // not found in current schema
	 {
	    sos_Import_List imp_list = self.get_imports();
	    agg_iterate (imp_list, sos_Import imp)
	    {  result = imp.lookup_type (name, imp_mode);
	       if VALID(result)
		  break;
	    }
	    agg_iterate_end (imp_list, imp);
	 }

	 if VALID (result) // enter found type into speedup-mapping
	    for (sos_Import_mode im=sos_IMP_NONE; 
		 im < imp_mode; 
		 im = sos_Import_mode (sos_Int(im) +1 ))
	    {  *keystr = 'A' + (int)im;

	       tmp_dir.insert (sos_String::create(TEMP_CONTAINER, keystr),
			       result);
	    }
      }
   }
   TT (mta_H, T_LEAVE);
   return result;
}


// *************************************************************************
void sos_Schema_module::install ()
// *************************************************************************
{  T_PROC ("sos_Schema_module::install")
   TT (mta_H, T_ENTER);

   sos_Schema_module_Directory sd = sos_Schema_module::schema_dir ();

   sd.container().open (WRITING, WAITING);
   sos_String str = self.get_name ();
   if (sd.is_key (str))
   {  // not yet implemented: info that schemas must be recompiled

      sos_Container ct = sd[str].container();
      sd.remove (str);
      ct.open (WRITING, WAITING);
      ct.destroy();
      ct.close();
   }
   sd.insert (str, self);
   sd.container().close();

   TT (mta_H, T_LEAVE);
}

// -------------------------------------------------------------------------
// sos_Identifier
// -------------------------------------------------------------------------
// *************************************************************************
sos_Bool sos_Identifier::total_equal (sos_Identifier x, sos_Object o,
				      sos_Eq_kind)
// *************************************************************************
{  T_PROC ("sos_Identifier::total_equal")
   TT (mta_M, T_ENTER);

   sos_Bool result = ( x.get_id().equal (sos_Identifier::make (o).get_id()) );

   TT (mta_M, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Comp_descr
// -------------------------------------------------------------------------
// *************************************************************************
sos_Method sos_Comp_descr::comp_method (sos_Class_type ct,
				        sos_Bool       set,
					sos_Bool       redef /* = TRUE */)
// *************************************************************************
{  T_PROC("sos_Comp_descr::comp_method")
   TT(mta_H, T_ENTER);

   sos_Method result = set ? self.get_set_method()
			   : self.get_get_method();
   if (redef)
      result = ct.get_methods().lookup (result); 

   TT(mta_H, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Method
// -------------------------------------------------------------------------
// *************************************************************************
sos_Bool sos_Method::overloads (sos_Method m)
// *************************************************************************
// precondition: self.name == m.name

{  T_PROC ("sos_Method::overloads")
   TT (mta_H, T_ENTER);

   sos_Bool result;

   if (self.get_is_static() != m.get_is_static())
      result = TRUE;
   else
   {  sos_Param_List pls = self.get_params(),
		     plm = m.get_params();
      sos_Int        pls_clow, pls_chigh, // minimum and maximum number of
		     plm_clow, plm_chigh; // parameters taking default exprs.
					  // into account.

      if (INVALID(pls))
	 pls_clow = pls_chigh = 0;
      else
      {
	 pls_clow = 0;
	 agg_iterate (pls, sos_Param p)
	    if (VALID(p.get_default_expr()))
	       break;
	    pls_clow++;
	 agg_iterate_end (pls, p)
	 pls_chigh = INVALID(pls) ? 0 : pls.card();
      }
      if (INVALID(plm))
	 plm_clow = plm_chigh = 0;
      else
      {
	 plm_clow = 0;
	 agg_iterate (plm, sos_Param p)
	    if (VALID(p.get_default_expr()))
	       break;
	    plm_clow++;
	 agg_iterate_end (plm, p)
	 plm_chigh = INVALID(plm) ? 0 : plm.card();
      }
	    
      result = (sos_Bool)(pls_chigh < plm_clow  OR  pls_clow > plm_chigh);
   }
   TT (mta_H, T_LEAVE; TB(result));
   return result;
}

// *************************************************************************
sos_Bool sos_Method::redefines (sos_Method m)
// *************************************************************************
// precondition: self.get_name() == m.get_name()

{  T_PROC ("sos_Method::redefines")
   TT (mta_H, T_ENTER);

   sos_Bool	  result;
   sos_Param_List pls = self.get_params(),
		  plm =    m.get_params();

   if ((INVALID(pls) != INVALID(plm))
       || (self.get_is_static() != m.get_is_static()))
      result = FALSE;
   else
   {  sos_Type ts = self.get_result_type().make_type();
      sos_Type tm =    m.get_result_type().make_type();

      if (ts.equal (tm))
      {  result = TRUE;
	 if VALID (pls) 
         {  int comp;
	    sos_Int pos_count = 0;
	    agg_iterate_double (pls, sos_Param ps, plm, sos_Param pm, comp)
	    {  pos_count++;
	       ts = ps.get_type().make_type();
	       tm = pm.get_type().make_type();
	       sos_Expr des = ps.get_default_expr();
	       sos_Expr dem = pm.get_default_expr();
	       result = (sos_Bool)(
			   ts.equal(tm)
			   AND (INVALID(dem)
			   OR (VALID(des) AND des.equal(dem)))); // NO_OBJECT!
	       if (NOT result)
		  break;
	    }
	    agg_iterate_double_end (pls, ps, plm, pm, comp);
	    // Now self redefines m, if the types and default expressions
	    // were OK so far and if self's parameter list is longer than m's,
	    // then self's remaining parameters have default expressions.
	    // The result is not OK, if self's parameter list is shorter than
	    // m's one.
	    result = (sos_Bool)(result AND 
		     (comp == 0 OR (comp > 0 
		     AND VALID(pls.get_nth(pos_count+1).get_default_expr()))));
	 }
      }
      else
         result = FALSE;
   }
   TT (mta_H, T_LEAVE; TB(result));
   return result;
}

// *************************************************************************
sos_Object sos_Method::execute (sos_Object o, sos_Object_Array p)
// *************************************************************************
{  T_PROC ("sos_Method::execute")
   TT (mta_H, T_ENTER);

   sos_Object result;
   sos_Bool   found = FALSE;

   sos_Method_impl_List impls = self.get_impls();
   if VALID (impls) 
   {  agg_iterate (impls, sos_Method_impl impl)
         if (impl.isa (cci_Method_impl_type))
	 {  result = cci_Method_impl::make (impl).execute (o, p);
	    found = TRUE;
	    break;
	 }
      agg_iterate_end (impls, impl);
   }
   if (NOT found)
      err_raise (err_SYS, err_MTA_METHOD_NOT_EXECUTABLE,
			  self.get_name().make_Cstring());
   TT (mta_H, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Method_table
// -------------------------------------------------------------------------
// *************************************************************************
sos_Method sos_Method_table::lookup (sos_Method m)
// *************************************************************************
{  T_PROC ("sos_Method_table::lookup")
   TT (mta_H, T_ENTER);

   sos_Method result = sos_Method::make (NO_OBJECT);
   sos_String m_name = m.get_name();

   if (self.is_key (m_name))
   {  sos_Method_List ml = self[m_name];

      agg_iterate (ml, sos_Method m1)
         if (NOT m.overloads (m1))
	 {  result = m1;
	    break;
	 }
      agg_iterate_end (ml, m1);
   }
   TT (mta_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Method sos_Method_table::lookup_or_add (sos_Method m)
// *************************************************************************
{  T_PROC ("sos_Method_table::lookup_or_add")
   TT (mta_H, T_ENTER);

   sos_Method result;
   sos_Bool   found = FALSE;

   sos_String      m_name = m.get_name();
   sos_Method_List ml     = self[m_name];
   if INVALID (ml) 
   {  ml = sos_Method_List::create (self.container(), FALSE);
      self.insert (m_name, ml);
   }
   else
   {  agg_iterate (ml, sos_Method m1)
         if (NOT m.overloads (m1))
	 {  found = TRUE;
	    result = m1;
	    break;
	 }
      agg_iterate_end (ml, m1);
   }
   if (NOT found)
   {  ml.append (m);
      result = m;
   }
   TT (mta_H, T_LEAVE; TB(found));
   return result;
}

// *************************************************************************
sos_Method sos_Method_table::replace_or_add (sos_Method m)
// *************************************************************************
{  T_PROC ("sos_Method_table::replace_or_add")
   TT (mta_H, T_ENTER);

   sos_Method result;
   sos_Bool   found = FALSE;

   sos_String      m_name = m.get_name();
   sos_Method_List ml     = self[m_name];
   if INVALID (ml) 
   {  ml = sos_Method_List::create (self.container(), FALSE);
      self.insert (m_name, ml);
   }
   else
   {  sos_Cursor c = ml.open_cursor();
      for (sos_Bool valid = ml.is_valid (c);  valid;  valid = ml.to_succ (c))
      {  sos_Method m1 = ml.get (c);
	 if (NOT m.overloads (m1))
	 {  ml.set (c, m);
	    found  = TRUE;
	    result = m1;
	    break;
	 }
      }
      ml.close_cursor (c);
   }
   if (NOT found)
   {  ml.append (m);
      result = m;
   }
   TT (mta_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Method sos_Method_table::lookup_comp (sos_String name,
					  sos_Bool   is_set)
// *************************************************************************
{  T_PROC ("sos_Method_table::lookup_comp")
   TT (mta_H, T_ENTER);

   sos_Method result = sos_Method::make (NO_OBJECT);

   sos_String mn = sos_String::create (TEMP_CONTAINER,
				       (is_set ? "set_" : "get_"));
   mn += name;

   sos_Method_List ml = self[mn];
   if VALID (ml) 
   {  agg_iterate (ml, sos_Method m)
         if VALID (m.get_comp_descr() )// component method found?
	 {  result = m;
	    break;
	 }
      agg_iterate_end (ml, m);
   }
   mn.destroy();

   TT (mta_H, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Type_descr
// -------------------------------------------------------------------------
// *************************************************************************
sos_Type sos_Type_descr::make_type ()
// *************************************************************************
{  // default implementation expects self to be derived from sos_Type.
   T_PROC("sos_Type_descr::make_type")
   TT (mta_H, T_ENTER);

   sos_Type result = sos_Type::make(self);

   TT (mta_H, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Generic_instantiation
// -------------------------------------------------------------------------
// *************************************************************************
sos_Type sos_Generic_instantiation::make_type ()
// *************************************************************************
{  T_PROC ("sos_Generic_instantiation::make_type")
   TT (mta_H, T_ENTER);

   sos_Type result = self.get_instantiation();

   TT (mta_H, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Super_class
// -------------------------------------------------------------------------
// *************************************************************************
sos_Type sos_Super_class::make_type ()
// *************************************************************************
{  T_PROC ("sos_Super_class::make_type")
   TT (mta_H, T_ENTER);

   sos_Type result = self.get_super_class().make_type();

   TT (mta_H, T_LEAVE);
   return result;
}


// -------------------------------------------------------------------------
// sos_Gen_param
// -------------------------------------------------------------------------
// *************************************************************************
sos_Type sos_Gen_param::make_type ()
// *************************************************************************
{  T_PROC ("sos_Gen_param::make_type")
   TT (mta_H, T_ENTER);

   sos_Type result = INVALID(self.get_super_type())
		       ? sos_Type::make(_obst_knl_type(_THE_OBJECT_TYPE))
		       : self.get_super_type().make_type();

   TT (mta_H, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Enum_type
// -------------------------------------------------------------------------
// *************************************************************************
sos_Scalar_object sos_Enum_type::make_object (sos_String literal)
// *************************************************************************
{  T_PROC ("sos_Enum_type::make_object")
   TT (mta_H, T_ENTER);

   sos_String_List sl    = self.get_literals();
   sos_Int         idx   = 0;
   sos_Bool        found = FALSE;

   agg_iterate (sl, sos_String lit)
      if (lit.equal (literal))
      {  found = TRUE;
         break;
      }
      ++ idx;
   agg_iterate_end (sl, lit);

   sos_Scalar_object result = found ? _obst_enum_from_sos_Int (idx, (sos_Type)self)
				    : sos_Scalar_object::make (NO_OBJECT);
   TT (mta_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_String sos_Enum_type::make_string (sos_Scalar_object enum_obj)
// *************************************************************************
{  T_PROC ("sos_Enum_type::make_string")
   TT (mta_H, T_ENTER);

   int    	   lidx     = _obst_Int_from_enum ((sos_Object)enum_obj, (sos_Type)self) + 1;
   sos_String_List literals = self.get_literals();

   sos_String result
      = (1 <= lidx  AND  lidx <= literals.card())
	   ? sos_String::clone (literals.get_nth (lidx), TEMP_CONTAINER)
	   : sos_String::make (NO_OBJECT);

   TT (mta_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Bool sos_Enum_type::is_scalar ()
// *************************************************************************
{  T_PROC ("sos_Enum_type::is_scalar")
   TT (mta_H, T_ENTER);
   TT (mta_H, T_LEAVE);

   return TRUE;
}

// *************************************************************************
sos_Bool sos_Enum_type::is_derived_from (sos_Type tp)
// *************************************************************************
// note: is_derived_from_some is implemented in sos_Type as is_derived_from.
//       This relies on the fact that both relations are equivalent for
//	 sos_Scalar_object.

{  T_PROC ("sos_Enum_type::is_derived_from")
   TT (mta_H, T_ENTER);

   sos_Bool result;

   if (self._self_id() == tp._self_id()
#ifdef BOOT
       OR _obst_knl_type(_THE_SCALAR_TYPE).is_derived_from (tp)
#else
       OR sos_Scalar_object_type.is_derived_from (tp)
#endif
      )
      result = TRUE;

   else if (tp._type_id() == _sos_Union_type_type)
      result = sos_Union_type::make(tp).is_base_of (self);

   else
      result = FALSE;

   TT (mta_H, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Extern_type
// -------------------------------------------------------------------------
// *************************************************************************
sos_Bool sos_Extern_type::is_scalar ()
// *************************************************************************
{  T_PROC ("sos_Extern_type::is_scalar")
   TT (mta_H, T_ENTER);
   TT (mta_H, T_LEAVE);

   return TRUE;
}

// *************************************************************************
sos_Bool sos_Extern_type::is_derived_from (sos_Type tp)
// *************************************************************************
// note: is_derived_from_some is implemented in sos_Type as is_derived_from.
//       This relies on the fact that both relations are equivalent for
//	 sos_Scalar_object.

{  T_PROC ("sos_Extern_type::is_derived_from")
   TT (mta_H, T_ENTER);

   sos_Bool result;

   if (self._self_id() == tp._self_id()
#ifdef BOOT
       OR _obst_knl_type(_THE_SCALAR_TYPE).is_derived_from (tp)
#else
       OR sos_Scalar_object_type.is_derived_from (tp)
#endif
      )
      result = TRUE;

   else if (tp._type_id() == _sos_Union_type_type)
      result = sos_Union_type::make(tp).is_base_of (self);

   else
      result = FALSE;

   TT (mta_H, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Union_type
// -------------------------------------------------------------------------
// *************************************************************************
sos_Bool sos_Union_type::is_base_of (sos_Type tp)
// *************************************************************************
{  T_PROC ("sos_Union_type::is_base_of")
   TT (mta_H, T_ENTER);

   sos_Bool result  = FALSE;

   sos_Type_List tl = self.get_united();
   agg_iterate (tl, sos_Type utp)
      if (tp.is_derived_from (utp))
      {  result = TRUE;
	 break;
      }
   agg_iterate_end (tl, utp);

   TT (mta_H, T_LEAVE);
   return result;
}

// -------------------------------------------------------------------------
// sos_Typedef_type
// -------------------------------------------------------------------------
// *************************************************************************
sos_Type sos_Typedef_type::make_type ()
// *************************************************************************
{  T_PROC ("sos_Typedef_type::make_type")
   TT (mta_H, T_ENTER);

   sos_Type t = self.get_named_type();

   TT (mta_H, T_LEAVE);
   return t;
}

// -------------------------------------------------------------------------
// sos_Class_type
// -------------------------------------------------------------------------
// *************************************************************************
sos_Bool sos_Class_type::is_generic_class ()
// *************************************************************************
{  T_PROC ("sos_Class_type::is_generic_class")
   TT(mta_H, T_ENTER);

   sos_Bool result = VALID (self.get_formal_gen_params()) ;

   TT(mta_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Bool sos_Class_type::is_instantiation ()
// *************************************************************************
{  T_PROC ("sos_Class_type::is_instantiation")
   TT(mta_H, T_ENTER);

   sos_Bool result = VALID (self.get_actual_gen_params()); 

   TT(mta_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Type sos_Class_type::generic_param (sos_Class_type rtp, sos_Int n)
// *************************************************************************
{  T_PROC ("sos_Class_type::generic_param")
   TT(mta_H, T_ENTER);

   //PRE: self.is_derived_from_some (rtp) &&
   //     self.is_instantiation() && rtp.is_instantiation()

   sos_Type result = sos_Type::make(NO_OBJECT);

   sos_Super_class_List scl = self.get_super_closure();
   agg_iterate_reverse (scl, sos_Super_class sc)
      sos_Class_type sctp = sos_Class_type::make (sc.make_type());
      if (sctp.root() EQ rtp)
      {  result = sctp.get_actual_gen_params().get_nth (n).make_type();
	 break;
      }
   agg_iterate_reverse_end (scl, sc);

   TT(mta_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Type sos_Class_type::root ()
// *************************************************************************
{  T_PROC ("sos_Class_type::root")
   TT (mta_H, T_ENTER);

   sos_Type tp = self.get_root_class();

   TT (mta_H, T_LEAVE);
   return tp;
}

// *************************************************************************
sos_Bool sos_Class_type::is_derived_from (sos_Type tp)
// *************************************************************************
// pre: self.make_type() == self
// method should maybe be implemented by a CCI hashtable lookup.

{  T_PROC ("sos_Class_type::is_derived_from")
   TT (mta_H, T_ENTER);

   sos_Bool result  = FALSE;

   if (   tp._self_id() == self._self_id()
#ifdef BOOT
       OR tp._self_id() == _obst_knl_type (_THE_OBJECT_TYPE)._self_id()
#else
       OR tp._self_id() == _sos_Object_type
#endif
      )
      result = TRUE;

   else if (tp._type_id() == _sos_Union_type_type)
      result = sos_Union_type::make(tp).is_base_of (self);

   else
   {  sos_Super_class_List scl = self.get_super_closure();

      agg_iterate (scl, sos_Super_class sc)
         if (sc.get_super_class() ==  tp)
	 {  result = TRUE;
	    break;
	 }
      agg_iterate_end (scl, sc);
   }
   TT (mta_H, T_LEAVE; TB (result));
   return result;
}

// *************************************************************************
sos_Bool sos_Class_type::is_derived_from_some (sos_Type tp)
// *************************************************************************
// pre: self.make_type() == self

{  T_PROC ("sos_Class_type::is_derived_from_some")
   TT (mta_H, T_ENTER);

   sos_Bool result  = FALSE;

   if (   tp._self_id() == self._self_id()
#ifdef BOOT
       OR tp._self_id() == _obst_knl_type (_THE_OBJECT_TYPE)._self_id()
#else
       OR tp._self_id() == _sos_Object_type
#endif
      )
      result = TRUE;

   else if (tp._type_id() == _sos_Union_type_type)
      result = sos_Union_type::make(tp).is_base_of (self);

   else
   {  sos_Super_class_List scl = self.get_super_closure();

      agg_iterate (scl, sos_Super_class sc)
         if (sos_Class_type::make(sc.get_super_class()).root() == tp)
	 {  result = TRUE;
	    break;
	 }
      agg_iterate_end (scl, sc);
   }
   TT (mta_H, T_LEAVE; TB (result));
   return result;
}
