/* -----------------------------------------------------------------------------
* Copyright 1992 by Forschungszentrum Informatik(FZI)
*
* You can use and distribute this software under the terms of the license
* you should have received along with this software.
* If not or if you want additional information, write to
* Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
* D-76131 Karlsruhe, Germany.
* ------------------------------------------------------------------------------
*/
/* OBST LIBRARY MODULE */

// *****************************************************************************
// Module psm                  Emil Sekerinski, Oliver Spatscheck, Walter Zimmer
//
// *****************************************************************************
//  persistent storage manager: implementation 
// *****************************************************************************


// *****************************************************************************
// INCLUDE FILES 
// *****************************************************************************

// ================= system and platform files =================================

#define OBST_IMP_FILE
#define OBST_IMP_RANDOM
#define OBST_IMP_ERROR
#define OBST_IMP_MEMOP
#include "obst_stdinc.h"

// ================ obst files =================================================

#include "_obst_config.h"
#include "obst.h"
#include "obst_progstd.h"
#include "knl_use.h"
#include "psm_err.h"    // err_raise, err_SYS
#include "trc_psm.h"    // trace-module; numbers 11(H) -14(VL) + psm_CHECK
#ifdef OBST_HAVE_JOYCE
#  include "sync_use.h"   // transaction handling
#  include "sync_decl.h"  // declarations for transactions
#endif
#include "psm_util.h"


// *****************************************************************************
// PUBLIC FUNCTION DEFINITIONS 
// *****************************************************************************

// ================= class sos_Container =======================================

sos_Open_result sos_Container::access(sos_Access_mode a, sos_Sync_mode s) const
{
   // access() changes the accessmode of the container. An implicit commit ...
   // ... is done when downgrading.

   T_PROC("sos_Container::access");
   TT(psm_H, T_ENTER; TI(id); TI((int)a); TI((int)s));
    
   psm_return_if_id0(id, OPENED);
   psm_return_if_deadlock(DEADLOCK, 1);
    
   sos_Open_result result;
   psm_cnt_info &ci = psm_lookup_cnt(id, READ_PERM, TRUE);

   if (ci.stat == DESTROYED) 
      result = UNACCESSIBLE;
   else
   {
      sos_Container_status cnt_stat = ci.stat;

      if (
#ifdef OBST_HAVE_JOYCE
	  (!psm_is_syncactivated) && 
#endif
	  			     (psm_downgrading(cnt_stat, a)))
      {
         psm_commit_cnt(ci);
         if (a == READING) psm_set_last_read_time(ci);    
         // only needed when downgrading to READING ... 
         // ...(never disturbing else, but inefficient)
      }

#ifdef OBST_HAVE_JOYCE
      if ((!psm_is_syncactivated) ||
         ((psm_is_syncactivated) && ((psm_upgrading(cnt_stat, a)) || 
         (!ci.is_cnt_open))))
#endif
      {  
         result = psm_lock(ci.fd, a, s, cnt_stat, ci.id, ci.fd_lock);
         if (result == OPENED)
         {
            if (
#ifdef OBST_HAVE_JOYCE
		(!psm_is_syncactivated) &&
#endif
		(ci.stat == READABLE) && (psm_upgrading(cnt_stat, a)))
               psm_updating(ci); // see ci.filepages.ddoc(R->W, R->C)

            ci.stat = sos_Container_status(a);
            if (psm_upgrading(ci.stat_ta, a))
               ci.stat_ta = sos_Container_status(a);
         }
      }
#ifdef OBST_HAVE_JOYCE
      else result = OPENED;
#endif
      if (result == OPENED)
         ci.is_cnt_open = TRUE;
   }

   TT(psm_H, T_LEAVE; TI(result));
   return result;
}

//------------------------------------------------------------------------------

sos_Offset sos_Container::allocate(sos_Int sz) const 
{
   // FUNCTION: Allocates rounded(size) bytes in the container. 

   // PRE: Size of the allocated area must be a positive multiple of 4 ... 
   // ...(else it's uprounded).

   // POST:(1) case sz >= P : return_offset begins at page boundary ... 
   //                            ...(return_offset % P = 0).
   //(2) case sz <  P : Block does not cross page boundary.

   T_PROC("sos_Container::allocate");
   TT(psm_M, T_ENTER; TI(id); TI(sz));
   sos_Offset result;
   sz = psm_four_upround(sz);

   if (id == 0)
   {  
      if (sz == 0) 
         err_raise(err_SYS, err_PSM_ALLOC_NULL_BYTES, psm_ERR, FALSE); 
      result =(sos_Offset) new char [sz];
      if (result == 0) 
         err_raise(err_SYS, err_PSM_TEMP_FULL, psm_ERR, FALSE);
   }
   else
   {  
      psm_return_if_deadlock(0, 0);
      if (sz == 0) 
         err_raise(err_SYS, err_PSM_ALLOC_NULL_BYTES, psm_ERR, FALSE);

      psm_cnt_info &ci = psm_lookup_cnt(id, WRITE_PERM, FALSE, TRUE);
      if (ci.cnt_pgs == 0)  
      {
         // allocate page with table of free blocks
         psm_alloc_pg(ci, 1);
         ci.ub_free_sz = 0; 
      }
      if (sz <= psm_MAX_FREEBLOCK_LENGTH)
      {  
         result = psm_read_int(ci, sz);
         if (result != 0)  
            // free block of identical size exists
            psm_write_int(ci, sz, psm_read_int(ci, result));  
               // eff. psm_rm_first_from_f_list
         else             
         {
            // try to find block with larger than size(sz > size)
            int larger_sz = sz + 4;  
            while (larger_sz <= ci.ub_free_sz  &&  
                   psm_read_int(ci, larger_sz) == 0)
               larger_sz += 4;
            if (larger_sz <= ci.ub_free_sz)            
            {  
               // delete bigger block and insert rest of it ...
               // ...(readint(ci, larger_sz) != 0)
               result = psm_rm_first_from_f_list(ci, larger_sz);
               psm_insert_in_f_list(ci, larger_sz - sz, result+sz);
            }
            else    
            {  
               // no free bigger block(sz > ci.ubfreesize)
               ci.ub_free_sz = psm_min(ci.ub_free_sz, psm_max(sz-4, 0));
               result = psm_alloc_pg(ci, 1);
               // insert rest of page in table of free blocks
               psm_insert_in_f_list(ci, psm_P - sz, result+sz);
            }
         } 
      }
      else 
      {  
         // new version: allocate *exact* size of bytes
         int sz_in_pgs          = psm_uprounded_pg_number(sz); 
         result                 = psm_alloc_pg(ci, sz_in_pgs); 
         int sz_of_restblock    = sz_in_pgs*psm_P - sz; 
         int offset_of_restblock= result + sz_in_pgs*psm_P - sz_of_restblock;
         psm_insert_in_f_list(ci, sz_of_restblock, offset_of_restblock);
      }
   }
   TT(psm_M, T_LEAVE; TI(result));
   return sos_Offset(result);
}

//------------------------------------------------------------------------------

void sos_Container::clear() const 
{
   // It clears the contents of the container, so that the container ...
   // ... occupies no space of the buffer anymore and when commiting the ...
   // ... container has no contents anymore. The entry in the containertable ...
   // ... is NOT removed(but the tables are set "right", see deallocatepages)

   T_PROC("sos_Container::clear");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, nothing);
   psm_return_if_deadlock(nothing, 1);

   psm_cnt_info &ci = psm_lookup_cnt(id, WRITE_PERM, TRUE, TRUE);

   psm_return_if_destroyed(sos_Container_status(ci.stat), nothing);
   
   psm_dealloc_pg(ci, 0, ci.cnt_pgs);
   TT(psm_H, T_LEAVE); 
}

//------------------------------------------------------------------------------

void sos_Container::close() const 
{
   // The container + lockfile is closed and all corresponding information ...
   // ... is removed. If status == DESTROYED then the two files are ...
   // ... deleted, if status == WRITEABLE or CHECKEDOUT, then the ...
   // ... container is committed.

   T_PROC("sos_Container::close");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, nothing);
   psm_return_if_psm_deadlock(nothing, 1);

   psm_cnt_info &ci = psm_lookup_cnt(id, READ_PERM, TRUE, TRUE);

#ifdef OBST_HAVE_JOYCE 
   if ((psm_is_syncactivated) && (ci.id != SYNC_CONTAINER))
   { 
      if (ci.is_cnt_open)
      { 
         ci.is_cnt_open = FALSE;
         int result = psm_sync_lock_a_file(READING, TESTING, ci.id, psm_UNLOCK); 
      }
   }
   else
#endif
   { 
      if (ci.stat == WRITEABLE  || ci.stat == CHECKEDOUT)   
         psm_commit_cnt(ci);
      if (ci.stat == DESTROYED)
      {
         unlink(psm_cnt_path(id));
#ifndef LOCKD_CORRECT
         unlink(psm_cnt_path_lock(id));    // locked or not doesn't care 
#endif
      }

      psm_rm_cnt(ci);                      // closing of fd ...
   }

   TT(psm_H, T_LEAVE); 
}

//------------------------------------------------------------------------------

void sos_Container::commit() const 
{
   T_PROC("sos_Container::commit");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, nothing);

#ifdef OBST_HAVE_JOYCE
   if (!psm_is_syncactivated)
#endif
      psm_commit_cnt(psm_lookup_cnt(id, WRITE_PERM, TRUE)); 

   TT(psm_H, T_LEAVE); 
}

//------------------------------------------------------------------------------

void sos_Container::compress() const
{
   T_PROC("sos_Container::compress");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, nothing);
   psm_return_if_deadlock(nothing, 1);
    
   psm_cnt_info &ci = psm_lookup_cnt(id, WRITE_PERM, TRUE, TRUE);

   if (ci.stat == WRITEABLE)
   {
      // only allowed for WRITING
#ifdef OBST_HAVE_JOYCE
      if (psm_is_syncactivated)
         // compress will be done in write_close()
         ci.operation_at_end = COMPRESS;
      else
#endif
      { 
         psm_commit_cnt(ci);
         psm_compress_cnt(ci);
         psm_commit_cnt(ci);
         psm_squeeze_cnt(ci);
         psm_commit_cnt(ci);
      }
   } 

   TT(psm_VL, TI(ci.operation_at_end)); 

   TT(psm_H, T_LEAVE);
}

//------------------------------------------------------------------------------

void sos_Container::copy(sos_Offset o1, sos_Int sz, sos_Container c2,
        sos_Offset o2) const 
{
   T_PROC("sos_Container::copy");
   TT(psm_M, T_ENTER; TI(id); TI(o1); TI(sz); TI((int)c2); TI(o2));

   if ((id != 0) || (c2.id != 0))
      psm_return_if_deadlock(nothing, 0);

   psm_cnt_info *ci1 = psm_cnt_tbl, *ci2 = psm_cnt_tbl;
   unsigned d1, d2;
   char *a1, *a2;
   char *copy_buf;
    
   if (id==c2.id)   
      copy_buf = new char[psm_P];

   if (id != 0) 
      ci1 = &psm_lookup_cnt(id, NO_PERM, FALSE, TRUE);
   if (c2.id != 0) 
      ci2 = &psm_lookup_cnt(c2.id, WRITE_PERM, FALSE, TRUE);
   if (o1 > o2) 
      //copy from low to high
      while (sz > 0) 
      {
         if (id == 0) 
         {
            d1 = 0; 
            a1 =(char*) o1;
         }
         else 
         {
            d1 = o1 % psm_P; 
            a1 = psm_buf[psm_lookup_pg(*ci1, o1/psm_P)] + d1; 
         }

         if (c2.id == 0) 
         {
            d2 = 0; 
            a2 =(char*) o2;
         }
         else 
         {   
            d2 = o2 % psm_P;
            int destination_buf_pg = psm_lookup_pg(*ci2, o2/psm_P);
            a2 = psm_buf[destination_buf_pg] + d2;
            psm_buf_tbl[destination_buf_pg].is_mod = TRUE;
         }
         int len = psm_min(sz, psm_min(psm_P-d1, psm_P-d2));
         if (id==c2.id) 
         {   
            memcpy(copy_buf, a1, len);
            a1=copy_buf;
         }
         memcpy(a2, a1, len);
         o1   += len; 
         o2   += len; 
         sz   -= len; 
      }
      else  
      {
         //copy from high to low
         o1 += sz - 1; o2 += sz - 1;
         while (sz > 0) 
         {   
            if (id == 0) 
            {
               d1 = psm_P; 
               a1 =(char*) o1 + 1;
            }
            else 
            {
               d1 = o1%psm_P + 1; 
               a1 = psm_buf[psm_lookup_pg(*ci1, o1/psm_P)] + d1;
            }
            if (c2.id == 0) 
            {
               d2 = psm_P; 
               a2 =(char*) o2 + 1;
            }
            else 
            {
               d2 = o2%psm_P + 1;
               int destination_buf_pg = psm_lookup_pg(*ci2, o2/psm_P);
               a2 = psm_buf[destination_buf_pg] + d2;
               psm_buf_tbl[destination_buf_pg].is_mod = TRUE; 
            }
            int len = psm_min(sz, psm_min(d1, d2));
            if (id==c2.id) 
            {   
               memcpy(copy_buf, a1-len, len);
               a1=copy_buf+len;
            }
            memcpy(a2-len, a1-len, len);
            o1   -= len; 
            o2   -= len; 
            sz   -= len; 
      } 
   }

   if (id==c2.id)   
      delete copy_buf;

   TT(psm_M, T_LEAVE); 
}

//------------------------------------------------------------------------------

sos_Container sos_Container::create() 
{
   // A new container is created and opened, i.e. the two corresponding ...
   // ... files are created, the container is locked for WRITING ...
   // ...(the lockfile is closed immediately afterwards) and an entry ...
   // ... in the containertable is made.

   T_PROC("sos_Container::create");
   TT(psm_H, T_ENTER);

   int id = UNUSED_CONTAINER;
   int fd, fd_lock;

   psm_check_lock_method();

   psm_return_if_deadlock(UNUSED_CONTAINER, 1);

   do
   {
      id =(sos_Int)(random()%0x5f5e0f0)+3; // 2<id<99999987
      fd = ::open(psm_cnt_path(id), O_RDWR | O_CREAT | O_EXCL, 0664); 
         // owner:rw, group:rw others:r
   } while (fd == psm_UNDEF  &&  errno == EEXIST);

   if (fd == psm_UNDEF) 
      err_raise(err_SYS, err_PSM_CREATE_FAILED, psm_ERR, FALSE);

   fd_lock = psm_UNDEF;
#ifndef LOCKD_CORRECT
   fd_lock = ::open(psm_cnt_path_lock(id), O_RDWR | O_CREAT | O_EXCL, 0664); 
   if (fd_lock == psm_UNDEF) 
      err_raise(err_SYS, err_PSM_CREATE_FAILED, psm_ERR, FALSE);
   else 
   {
      ::close(fd_lock);
      fd_lock = psm_UNDEF;
   }
#endif LOCKD_CORRECT

   if (psm_lock(fd, WRITING, TESTING, UNAVAILABLE, id, fd_lock) != OPENED)
      err_raise(err_SYS, err_PSM_CREATE_LOCK_FAILED, psm_ERR, FALSE);

   psm_enter_cnt(id, fd, WRITEABLE, fd_lock, TRUE);

   TT(psm_H, T_LEAVE; TI((int)id));
   return sos_Container::make(id);
}

//------------------------------------------------------------------------------

void sos_Container::deallocate(sos_Offset o, sos_Int sz) const 
{
   // PRE:  Because of reasons of efficiency and simplicity the following ...
   // ... restrictions  are made:
   //(1) case size <  psm_P : Blocks must not cross page boundary.
   //(2) case size >= psm_P : The offset must begin at a page ...
   //     ... boundary(o % P = 0).
   //(3) Size of the deallocated area must be a multiple of 4 ...
   //     ...(else it's uprounded).
   //(4) You can deallocate formerly allocated blocks(as a whole or ...
   //     ... in parts), but you can NOT deallocate a large block ...
   //     ... consisting of several small, formerly allocated blocks.

   T_PROC("sos_Container::deallocate");
   TT(psm_M, T_ENTER; ; TI(id); TI(o); TI(sz));
   
   if (id != 0) psm_return_if_deadlock(nothing, 0);

   if (sz == 0) 
      err_raise(err_SYS, err_PSM_DEALLOC_NULL_BYTES, psm_ERR, FALSE);

   if (id == 0) 
      delete(char*) o;
   else 
      if (o < 1024) 
         err_raise(err_SYS, err_PSM_DEALLOCATE_WRONG_OFFSET, psm_ERR, FALSE); 
      else 
      {  
         psm_cnt_info &ci = psm_lookup_cnt(id, WRITE_PERM, FALSE, TRUE);
         TTN(psm_CHECK_WRITE, psm_check_f_list(ci, o, sz));
         sz = psm_four_upround(sz);

         if (sz <= psm_MAX_FREEBLOCK_LENGTH) 
            psm_insert_in_f_list(ci, sz, o); 
         else
         { 
            int sz_in_pgs = psm_offrounded_pg_number(sz);   
            psm_dealloc_pg(ci, o/psm_P, sz_in_pgs); 
            int sz_of_restblock = sz - sz_in_pgs*psm_P; 
            psm_insert_in_f_list(ci, sz_of_restblock, o + sz_in_pgs*psm_P);
         }
      }
   TT(psm_M, T_LEAVE) 
}

//------------------------------------------------------------------------------

void sos_Container::destroy() const 
{
   // PRE: Container is open for WRITING.
   // For the first call the container is cleared and status is set to ...
   // ... destroyed.

   T_PROC("sos_Container::destroy");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, nothing);
   psm_return_if_deadlock(nothing, 1);
 
   psm_cnt_info &ci = psm_lookup_cnt(id, WRITE_PERM, TRUE, TRUE);

   // WRITEABLE and CHECKEDOUT differ only in this operation and some other... 
   // ... operation(compress, squeeze)

   if (ci.stat != WRITEABLE  &&  ci.stat != DESTROYED)  
      err_raise(err_SYS, err_PSM_DESTROY_ONLY_PERMITTED_FOR_WRITING, psm_ERR, FALSE);

   if (ci.stat != DESTROYED)  
   {  
      clear();                // not necessary, but saves space
      ci.stat = DESTROYED;
   }

   TT(psm_H, T_LEAVE); 
}

//------------------------------------------------------------------------------

sos_Bool sos_Container::equal(sos_Offset o1, sos_Int sz, sos_Container c2,
        sos_Offset o2) const 
{
   T_PROC("sos_Container::equal");
   TT(psm_M, T_ENTER; TI(id); TI(o1); TI(sz); TI((int)c2); TI(o2));

   if ((id != 0) || (c2.id != 0)) 
      psm_return_if_deadlock(FALSE, 0);

   psm_cnt_info *ci1 = psm_cnt_tbl, *ci2 = psm_cnt_tbl;
   unsigned d1, d2;
   char *a1, *a2;
   if (id == 0) 
      d1 = 0; 
   else 
      ci1 = &psm_lookup_cnt(id, NO_PERM, FALSE, TRUE);
   if (c2.id == 0) 
      d2 = 0; 
   else 
      ci2 = &psm_lookup_cnt(c2.id, NO_PERM, FALSE, TRUE);
   while (sz > 0) 
   {   
      if (id == 0) 
         a1 =(char*) o1;
      else 
      {
         d1 = o1%psm_P;
         a1 = psm_buf[psm_lookup_pg(*ci1, o1/psm_P)] + d1; 
      }

      if (c2.id == 0) 
         a2 =(char*) o2;
      else 
      {
         d2 = o2%psm_P; 
         a2 = psm_buf[psm_lookup_pg(*ci2, o2/psm_P)] + d2; 
      }
      int len = psm_min(sz, psm_min(psm_P-d1, psm_P-d2));
      if (memcmp(a1, a2, len) != 0) 
      {  
         TT(psm_M, T_LEAVE; TI(FALSE));
         return FALSE;
      }
      o1    += len; 
      o2    += len; 
      sz    -= len; 
   }

   TT(psm_M, T_LEAVE; TI(TRUE));
   return TRUE; 
}

//------------------------------------------------------------------------------

sos_Bool sos_Container::equal_container(sos_Container c2) const 
{
   T_PROC("sos_Container::equal_container");
   TT(psm_M, T_ENTER; TI(id); TI((int)c2));

   psm_cnt_info ci1, ci2;

   ci1 = psm_lookup_cnt(id, READ_PERM);
   ci2 = psm_lookup_cnt(c2.id, READ_PERM);

   TT(psm_M, T_LEAVE; TI(TRUE));
   return psm_is_equal_cnt(ci1, ci2); 
}

//------------------------------------------------------------------------------

sos_Bool sos_Container::exists() const 
{
   // Returns whether the container exists or not("not" = "file is ...
   // ... destroyed or container has been destroyed, but the operation has ...
   // ... been not yet  committed")

   T_PROC("sos_Container::exists");
   TT(psm_H, T_ENTER; TI(id));
   sos_Bool result = FALSE;          
   int fd = psm_UNDEF;                   // not in switch because warnings
   switch (status())
   {
      case READABLE:
      case WRITEABLE:
      case CHECKEDOUT:  result = TRUE;
                        break;

      case DESTROYED:   result = FALSE;
                        break;

      case UNAVAILABLE:
#ifdef OBST_HAVE_JOYCE
			if ((psm_is_syncactivated) && (psm_is_cnt_open(id)))
                        { 
                           psm_cnt_info &ci = psm_lookup_cnt
                                              (id, READ_PERM, TRUE);
                           if (ci.stat == DESTROYED)
                              result = FALSE;
                           else
                              result = TRUE;
                        }
                        else
#endif
                        { 
                           fd = ::open(psm_cnt_path(id), O_RDONLY);
                           if (fd != psm_UNDEF) 
                           {  
                              ::close(fd);
                              result = TRUE;
                           }
                        }
                        break;
      default:          err_raise(err_SYS, err_PSM_EXISTS, psm_ERR, FALSE);
                        break;
   };
   TT(psm_H, T_LEAVE; TB(result));
   return result; 
}

//------------------------------------------------------------------------------

int sos_Container::hash_value(sos_Offset o, sos_Int sz) const 
{
   T_PROC("sos_Container::hash_value");
   TT(psm_H, T_ENTER; TI(id); TI(o); TI(sz));

   if (id != 0) 
      psm_return_if_deadlock(-1, 1);

   sos_Offset u = o + sz;
   unsigned l = psm_SZ_OF_INT;
   sos_Int result = 0;
   sos_Int data;
   for (; o < u; o += l) 
   {   
      if (l > u - o) 
      { 
         l = u - o; 
         data = 0; 
      } 
      read(o, l, &data);
      result ^= data; 
   }
   TT(psm_H, T_LEAVE; TI(result));
   return result; 
}

//------------------------------------------------------------------------------

sos_Bool  sos_Container::modified() const 
{ 
   T_PROC("sos_Container::modified");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, TRUE);
   psm_return_if_deadlock(FALSE, 1);
   
   psm_cnt_info &ci = psm_lookup_cnt(id, READ_PERM, TRUE, TRUE);

   psm_return_if_destroyed(sos_Container_status(ci.stat), TRUE);

   sos_Bool result = sos_Bool(ci.stat == READABLE) ? 
                     sos_Bool(psm_is_mod_since_last_read(ci))
                     : sos_Bool(psm_is_mod_since_last_commit(ci));
   TT(psm_H, T_LEAVE; TI(result));
   return result;
}

//------------------------------------------------------------------------------

sos_Existing_status sos_Container::object_exists(sos_Offset o,
                                                 sos_Int sz) const 
{ 
   // Returns whether the object exists(perhaps) or not(sure).

   // PRE: offset & size belongs to former objects in that container. ...
   // ... If this condition is not fulfilled then access outside allocated ...
   // ... area is possible.

    
   psm_return_if_id0_no_trc(id, PERHAPS_EXISTING);

   return psm_check_object_exists(id, o, sz);
}

//------------------------------------------------------------------------------

sos_Int sos_Container::occupied() const 
{
   T_PROC("sos_Container::occupied");
   TT(psm_H, T_ENTER; TI(id));
      
   psm_return_if_id0(id, -1);
   psm_return_if_deadlock(-1, 1);

   psm_cnt_info &ci = psm_lookup_cnt(id, READ_PERM, TRUE, TRUE);

   psm_return_if_destroyed(sos_Container_status(ci.stat), 0);

   unsigned result = psm_occupied_bytes(ci);

   TT(psm_H, T_LEAVE; TI((int)result));
   return result;
}

//------------------------------------------------------------------------------

sos_Open_result sos_Container::open(sos_Access_mode a, sos_Sync_mode s) const 
{ 
   // It opens the container with accessmode a and synchronisation ... 
   // ... strategy s. If the container is already opened then an ... 
   // ... upgrading is possible, downgrading is never possible with ...
   // ... open(only possible with access).

   T_PROC("sos_Container::open");
   TT(psm_H, T_ENTER; TI(id); TI((int)a); TI((int)s));
   
   psm_check_lock_method();
 
   psm_return_if_id0(id, OPENED);
   psm_return_if_deadlock(DEADLOCK, 1);
   
   // <af> because of the restriction that only LOCKING transaction are ...
   // ... used, checkout is replaced by writing, because the modifications ...
   // ... do not have to be stored at the end of the file at no time
#ifdef OBST_HAVE_JOYCE    
   if ((psm_is_syncactivated) && (a == CHECKOUT))
   { 
      a = WRITING;
      TT(psm_M, TXT("checkout --> writing"));
   }
#endif
   sos_Open_result result;
   int fd, fd_lock;

   if (! psm_is_cnt_open(id))  
   { 
      fd      = ::open(psm_cnt_path(id), O_RDWR, 0644); // O_RDWR for upgrading
      fd_lock = psm_UNDEF;   // only opened if necessary

      if (fd == psm_UNDEF)
         result = UNACCESSIBLE;
      else
      {
         result = psm_lock(fd, a, s, UNAVAILABLE, id, fd_lock);
         if (result == OPENED)
            psm_enter_cnt(id, fd, sos_Container_status(a), fd_lock); 
         else
            ::close(fd);
      }
   }
   else  // container is already open
   { 
      psm_cnt_info &ci = psm_lookup_cnt(id, READ_PERM, TRUE);
      if (ci.stat == DESTROYED) 
         result = UNACCESSIBLE;
      else
      {
         sos_Container_status cnt_stat = ci.stat;

         if ((psm_upgrading(cnt_stat, a))
#ifdef OBST_HAVE_JOYCE
             || ((psm_is_syncactivated) && (!ci.is_cnt_open))
#endif
	    )
         {
            result = psm_lock(ci. fd, a, s, cnt_stat, ci.id, ci.fd_lock);
               if (result == OPENED)
            {
               if (
#ifdef OBST_HAVE_JOYCE
		   (!psm_is_syncactivated) &&
#endif
		   			      (ci.stat == READABLE))
                  psm_updating(ci);   // see ci.filepages.ddoc(R->W, R-> C)
               ci.stat = sos_Container_status(a);
               if (psm_upgrading(ci.stat_ta, a))
                  ci.stat_ta = sos_Container_status(a);
            }
         }
         else
            result = OPENED;
     
         if (result == OPENED)
            ci.is_cnt_open = TRUE; 
      }
   }

   TT(psm_H, T_LEAVE; TI(result));
   return result;
}

//------------------------------------------------------------------------------

void sos_Container::print_freelists() const    
{
   T_PROC("sos_Container::print_freelists");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, nothing);
   psm_return_if_deadlock(nothing, 1);

   psm_cnt_info &ci = psm_lookup_cnt(id, READ_PERM, FALSE, TRUE);

   psm_print_f_lists(ci);

   TT(psm_H, T_LEAVE);
}

//------------------------------------------------------------------------------

void sos_Container::read_close() const
{
#ifdef OBST_HAVE_JOYCE
   T_PROC("sos_Container::read_close()");
   TT(psm_H, T_ENTER; TI(id));

   if (!psm_is_syncactivated)
#endif
      err_raise(err_SYS, err_PSM_SYNC_NOT_ACTIVATED, psm_ERR, FALSE);

#ifdef OBST_HAVE_JOYCE
   if (id == SYNC_CONTAINER)
      err_raise(err_SYS, err_PSM_SYCT_READ_CLOSE, psm_ERR, FALSE);
   if (!psm_is_cnt_open(SYNC_CONTAINER)) 
      err_raise(err_SYS, err_PSM_SYCT_OPEN_RCL, psm_ERR, FALSE);
 
   psm_return_if_id0(id, nothing);

   psm_cnt_info &ci=psm_lookup_cnt(id, READ_PERM, TRUE);

   //   if (ci.ct_open)
   //       err_raise(err_SYS, err_PSM_CONTAINER_STILL_OPEN, psm_ERR, FALSE);

   // remove container-file if container was created during transaction ...
   // ... and the transaction is aborted 
   if (ci.is_cnt_created)
      unlink(psm_cnt_path(id));
 
   psm_rm_cnt(ci);

   TT(psm_H, T_LEAVE);
#endif
}

//------------------------------------------------------------------------------

void sos_Container::read(register sos_Offset o, register sos_Int sz, 
                         register sos_Pointer data) const 
{
   T_PROC("sos_Container::read");
   TT(psm_M, T_ENTER; TI(id); TI(o); TI(sz); TP(data));

   if (id == 0) 
      memcpy(data,(char*)o, sz);
   else if (sz > 0)
   {   
      psm_return_if_psm_deadlock(nothing, 0);
       
      psm_cnt_info &ci = psm_lookup_cnt(id, NO_PERM, FALSE, TRUE);
      TTN(psm_CHECK_READ, psm_check_f_list(ci, o, sz));

      register int  len;
      while (1)
      {   
         register unsigned  disp = o % psm_P; // displacement in page

         if ((len = psm_P-disp) >= sz)
         {  
            psm_read_from_pg(ci, o/psm_P, disp, sz, data);
            break;
         }
         psm_read_from_pg(ci, o/psm_P, disp, len, data);
         o += len;        
         data = sos_Pointer((char *)(data) + len);
         sz   -= len; 
      }
   } 
   TT(psm_M, T_LEAVE); 
}

//------------------------------------------------------------------------------

sos_Int sos_Container::realsize(sos_Int sz) const 
{
   T_PROC("sos_Container::realsize");
   TT(psm_M, T_ENTER; TI(sz));
 
   sos_Int result = psm_four_upround(sz);
 
   TT(psm_M, T_LEAVE; TI(result));
   return result;
}

//------------------------------------------------------------------------------

void sos_Container::reset() const 
{ 
   T_PROC("sos_Container::reset");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, nothing);
   psm_return_if_deadlock(nothing, 1);

   psm_reset_cnt(psm_lookup_cnt(id, WRITE_PERM, TRUE, TRUE));

   TT(psm_H, T_LEAVE); 
}

//------------------------------------------------------------------------------

sos_Object sos_Container::root_object() const 
{

   psm_return_if_id0_no_trc(id, NO_OBJECT);

#ifdef OBST_HAVE_JOYCE
   // return_if_psm_deadlock without trace 
   if ((psm_is_syncactivated) && (psm_is_deadlock) && (id != SYNC_CONTAINER)) 
      return NO_OBJECT;
#endif

   psm_cnt_info *ct = psm_cnt_tbl_position(id);
   if (ct != (psm_cnt_info *) psm_UNDEF)
      psm_return_if_destroyed_no_trc(sos_Container_status((*ct).stat),
                                     NO_OBJECT);

   return sos_Object::make(sos_Typed_id::make 
                          (sos_Id::make(*this, ROOT_OFFSET))); 
}

//------------------------------------------------------------------------------

void sos_Container::squeeze() const 
{
   T_PROC("sos_Container::squeeze");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, nothing);
   psm_return_if_deadlock(nothing, 1);

   psm_cnt_info &ci = psm_lookup_cnt(id, WRITE_PERM, TRUE, TRUE);

#ifdef OBST_HAVE_JOYCE
   if (psm_is_syncactivated) 
      // squeeze will be done in write_close()
      ci.operation_at_end =(ci.operation_at_end == NOP ? SQUEEZE 
                            : ci.operation_at_end);
   else
#endif
      if (ci.stat == WRITEABLE) 
      {
       psm_commit_cnt(ci);
       psm_squeeze_cnt(ci);
       psm_commit_cnt(ci);
       }
    else 
       if (ci.stat == CHECKEDOUT)     // tries to get a WRITELOG
          {
          if (OPENED == psm_lock(ci.fd, WRITING, TESTING,
                                 CHECKEDOUT, ci.id, ci.fd_lock))
             {
             ci.stat=WRITEABLE;
             psm_commit_cnt(ci);
             psm_squeeze_cnt(ci);
             psm_commit_cnt(ci);
             ci.stat=CHECKEDOUT;
             psm_lock(ci.fd, CHECKOUT, WAITING,
                      WRITEABLE, ci.id, ci.fd_lock);
             }
          }

   //TRACE
   TT(psm_VL, TI(ci.operation_at_end));

   TT(psm_H, T_LEAVE); 
}

//------------------------------------------------------------------------------

sos_Container_status sos_Container::status() const 
{ 
   // Returns status of an opened container  or  UNAVAILABLE, if not opened.

   T_PROC("sos_Container::status");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id, WRITEABLE);
    
   psm_cnt_info *cnt = psm_cnt_tbl_position(id);

   sos_Container_status result  =(cnt ==(psm_cnt_info *) psm_UNDEF) 
                                   ?(sos_Container_status) UNAVAILABLE
                                   :(sos_Container_status) cnt->stat;

   if ((cnt != (psm_cnt_info *)psm_UNDEF) && (!cnt->is_cnt_open)) 
      // -> psm_is_syncactivated = TRUE
      result =(sos_Container_status) UNAVAILABLE;

   TT(psm_H, T_LEAVE; TI(result));
   return result;
}

//------------------------------------------------------------------------------

void sos_Container::update() const
{
   T_PROC("sos_Container::update");
   TT(psm_H, T_ENTER; TI(id));
    
   psm_return_if_id0(id,nothing);

#ifdef OBST_HAVE_JOYCE   
   if (!psm_is_syncactivated)
#endif
   { 
      psm_cnt_info &ci = psm_lookup_cnt(id, READ_PERM, TRUE);
      if (ci.stat != READABLE) 
         return;     // makes no sense 
      else 
         psm_updating(ci);
   }
   TT(psm_H, T_LEAVE);
}

//------------------------------------------------------------------------------

void sos_Container::write_close() const
{
#ifdef OBST_HAVE_JOYCE
   T_PROC("sos_Container::write_close()");
   TT(psm_H, T_ENTER; TI(id));

   if (!psm_is_syncactivated)
#endif
      err_raise(err_SYS, err_PSM_SYNC_NOT_ACTIVATED, psm_ERR, FALSE);

#ifdef OBST_HAVE_JOYCE
   if (id == SYNC_CONTAINER)
      err_raise(err_SYS, err_PSM_SYCT_WRITE_CLOSE, psm_ERR, FALSE);
   if (!psm_is_cnt_open(SYNC_CONTAINER))
      err_raise(err_SYS, err_PSM_SYCT_OPEN_WCL, psm_ERR, FALSE);
 
   psm_return_if_id0(id, nothing);

   psm_cnt_info &ci = psm_lookup_cnt(id, READ_PERM, TRUE);
   if ((ci.stat == READABLE) && (ci.stat_ta == READABLE))
   { 
   // so the container was never opened for WRITING or CHECKOUT, because ...
   // ... otherwise at least ci.stat_ta would be WRITEABLE or CHECKEDOUT. ...
   // ... ci.status can be READABLE, if the container was closed after ...
   // ... WRITING and was opened for READING afterwards during the transaction.
      err_raise(err_SYS, err_PSM_NO_WRITEOPEN_CONTAINER, psm_ERR, FALSE);
   }

   //  if (ci.ct_open)
   //     err_raise(err_SYS, err_PSM_CONTAINER_STILL_OPEN, psm_ERR, FALSE);

   if (ci.stat != DESTROYED) 
   { 
      switch (ci.operation_at_end)
      {
         case COMPRESS : psm_commit_cnt(ci);
                         psm_compress_cnt(ci);
                         psm_commit_cnt(ci);
                         psm_squeeze_cnt(ci);
                         psm_commit_cnt(ci);
                         break;
         case SQUEEZE  : if ((ci.stat_ta == WRITEABLE) ||
                            ((ci.stat_ta == CHECKEDOUT) &&
                            (psm_is_squeeze_possible(ci))))
                         { 
                            psm_commit_cnt(ci);
                            psm_squeeze_cnt(ci);
                         }
                         psm_commit_cnt(ci);
                         break;
         case NOP      : psm_commit_cnt(ci);
                         break;
         default       : err_raise(err_SYS, err_PSM_WRITE_CLOSE, psm_ERR, FALSE);
                         break;
      };
   }

   if (ci.stat == DESTROYED)
      unlink(psm_cnt_path(id));

   psm_rm_cnt(ci);         

   TT(psm_H, T_LEAVE);
#endif
}

//------------------------------------------------------------------------------

void sos_Container::write(register sos_Offset o, register sos_Int  sz,
                          register sos_Pointer data) const 
{
   T_PROC("sos_Container::write");
   TT(psm_M, T_ENTER; TI(id); TI(o); TI(sz); TP(data));

   if (id == 0) 
      memcpy((char*)o, data, sz);
   else 
      if (sz >0)
      {  
         psm_return_if_deadlock(, 0);
      
         psm_cnt_info &ci = psm_lookup_cnt(id, WRITE_PERM, FALSE, TRUE);
         TTN(psm_CHECK_WRITE, psm_check_f_list(ci, o, sz));
         register int len;
         while (1)
         {
            register unsigned disp = o % psm_P; // displacement in page

            if ((len = psm_P-disp) >= sz)
            {  
               psm_write_to_pg(ci, o/psm_P, disp, sz, data);
               break;
            }
            psm_write_to_pg(ci, o/psm_P, disp, len, data);
            o += len;        
            data = sos_Pointer((char *)(data) + len);
            sz   -= len; 
         } 
      } 
   TT(psm_M, T_LEAVE); 
}

// ================= class sos_Container_set ===================================


sos_Container_set::sos_Container_set() 
{
   T_PROC("sos_Container_set::sos_Container_set");
   TT(psm_H, T_ENTER);
     
   set= new sos_Container_list;
   set->add_ref();

   TT(psm_H, T_LEAVE);
}

//------------------------------------------------------------------------------

sos_Container_set::~sos_Container_set()
{
   set->rm_ref();
   if (!(set->is_referenced())) 
      delete set;
} 

//------------------------------------------------------------------------------

sos_Container_set::sos_Container_set(const sos_Container_set& cs)
{
   set=cs.set;
   set->add_ref();
}

//------------------------------------------------------------------------------

sos_Container_set& sos_Container_set::operator=(const sos_Container_set& cs)
{
   set->rm_ref();
   if (!(set->is_referenced())) 
      delete set;
   set=cs.set;
   set->add_ref();

   return *this;
}

//------------------------------------------------------------------------------

sos_Container_set& sos_Container_set::operator+=(sos_Container c) 
{
   T_PROC("sos_Container_set::operator+=(sos_Container)");
   TT(psm_H, T_ENTER);

   set->add_cnt(c);

   TT(psm_H, T_LEAVE);
   return *this; 
}

//------------------------------------------------------------------------------

sos_Container_set& sos_Container_set::operator-=(sos_Container c) 
{
   T_PROC("sos_Container_set::operator-=(sos_Container)");
   TT(psm_H, T_ENTER);

   set->rm_cnt(c);

   TT(psm_H, T_LEAVE);
   return *this; 
}

//------------------------------------------------------------------------------

sos_Container_set& sos_Container_set::operator+=(sos_Container_set cs)
{
   T_PROC("sos_Container_set::operator+=(sos_Container_set)")
   TT(psm_H, T_ENTER);

   for (int i = cs.set->get_cnt_n()-1; i >= 0;  -- i)
   {
      set->add_cnt(cs.set->get_cnt(i));
   }
   TT(psm_H, T_LEAVE);
   return *this;
}

//------------------------------------------------------------------------------

sos_Container_set& sos_Container_set::operator-=(sos_Container_set cs)
{  
   T_PROC("sos_Container_set::operator-=(sos_Container_set)")
   TT(psm_H, T_ENTER);

   for (int i = cs.set->get_cnt_n()-1;  i >= 0;  -- i)
   {
      set->rm_cnt(cs.set->get_cnt(i));
   }

   TT(psm_H, T_LEAVE);
   return *this;
}

//------------------------------------------------------------------------------

sos_Container_set sos_Container_set::clone() 
{
   T_PROC("sos_Container_set::clone");
   TT(psm_H, T_ENTER);

   sos_Container_set result;
   result.set = new sos_Container_list;
   int i=card()-1;
 
   if (i > 0)
   {  
      do 
      {
         result.set->add_cnt(set->get_cnt(i));
      } while (--i);
   }

   result.set->add_ref();
    
   TT(psm_H, T_LEAVE);
   return result;
}

//------------------------------------------------------------------------------

void sos_Container_set::close() const
{
   T_PROC("sos_Container_set::close"); TT(psm_H, T_ENTER);

#ifdef OBST_HAVE_JOYCE
   if ((psm_is_syncactivated) && (psm_is_deadlock)) 
   { 
      TT(psm_H, T_LEAVE); 
      return;
   }
#endif
   int i;
   if ((i = card()-1) >= 0)
   {  
      do 
      {
         sos_Container::make(set->get_cnt(i)).close();
      } while (i--);
   }
   TT(psm_H, T_LEAVE);
}

//------------------------------------------------------------------------------

void sos_Container_set::cl(int i, const sos_Container_set& rd,
                           const sos_Container_set& ch,
                           const sos_Container_set& wr) 
{
   if (i < rd.card()) 
      rd.set->get_cnt(i).close(); 
   else 
      if (i <(rd.card()+ch.card()))
         ch.set->get_cnt(i-rd.card()).close();                 
      else 
         wr.set->get_cnt(i-rd.card()-ch.card()).close();          
}

//------------------------------------------------------------------------------

void sos_Container_set::commit() const 
{
   T_PROC("sos_Container_set::commit"); TT(psm_H, T_ENTER);

#ifdef OBST_HAVE_JOYCE 
   if (!psm_is_syncactivated)
#endif
   { 
      int i;
      if ((i = card()-1) >= 0)
      {  
         do 
         {
            sos_Container::make(set->get_cnt(i)).commit();
         } while (i--);
      }
   }
   TT(psm_H, T_LEAVE);
}

//------------------------------------------------------------------------------

sos_Container_set& sos_Container_set::open_containers(sos_Container_status stat)
{
   // returns set of containers openend for WRITING/READING/CHECKOUT ...
   // ... dependent on status

   T_PROC("sos_Container_set::open_containers");
   TT(psm_H, T_ENTER);

   sos_Container_set* new_set = new sos_Container_set;
   for (int i = 0; i < psm_C; i++) 
   {
      if ((psm_cnt_tbl[i].id != psm_UNUSED) && 
         ((psm_cnt_tbl[i].stat == stat) ||
         (psm_cnt_tbl[i].stat_ta == stat)))
      { 
#ifdef OBST_HAVE_JOYCE
         if ((!psm_is_syncactivated) ||
            ((psm_is_syncactivated) && (psm_cnt_tbl[i].is_cnt_open)))
#endif
           (*new_set) += sos_Container::make(psm_cnt_tbl[i].id);
      }
   }
   TT(psm_H, T_LEAVE);
   return(*new_set);
}

//------------------------------------------------------------------------------

sos_Container_set& sos_Container_set::open_containers(sos_Container_mod_status
                                                      stat)
{
   // returns set of containers openend for OPEN, MODIFIABLE, NOT_MODIFIABLE ...
   // ... dependent on status

   T_PROC("sos_Container_set::open_containers");
   TT(psm_H, T_ENTER);

   sos_Container_status current_stat;
   sos_Container_set* new_set = new sos_Container_set;
   for (int i = 0; i < psm_C; i++) 
   {
      if (psm_cnt_tbl[i].id != psm_UNUSED) 
      {
         current_stat = psm_cnt_tbl[i].stat;
         switch (stat)
         {
            case(OPEN) :   
               // no break because open returns ...
               // ... CHECKEDOUT, WRITEABLE containers too
               if (current_stat==READABLE)
                 (*new_set) += sos_Container::make(psm_cnt_tbl[i].id);
            case(MODIFIABLE) :
               if ((current_stat==CHECKEDOUT) || (current_stat==WRITEABLE) ||
                  (current_stat==DESTROYED))
                 (*new_set) += sos_Container::make(psm_cnt_tbl[i].id);
               break;
            case(NOT_MODIFIABLE) :                 
               if (current_stat==READABLE)
                 (*new_set) += sos_Container::make(psm_cnt_tbl[i].id);
            default : 
            break;
         }   
      }
   }
   TT(psm_H, T_LEAVE);
   return(*new_set);
}

//------------------------------------------------------------------------------

sos_Open_result sos_Container_set::open(sos_Access_mode a, sos_Sync_mode s) 
                                   const 
{
   T_PROC("sos_Container_set::open"); TT(psm_H, T_ENTER);

   sos_Container_set empty;
   sos_Open_result result;
   switch (a)
   {       
      case READING :  
         result = sos_Container_set::open(*this, empty, empty, s);break;
      case CHECKOUT : 
         result = sos_Container_set::open(empty, *this, empty, s);break;
      case WRITING :  
         result = sos_Container_set::open(empty, empty, *this, s);break;
      default: 
         err_raise(err_SYS, err_PSM_INVALID_ACCESS_MODE, psm_ERR, FALSE);
   }

   TT(psm_H, T_LEAVE);
   return result;
 }

//------------------------------------------------------------------------------

sos_Open_result sos_Container_set::open(const sos_Container_set& rd,
                                        const sos_Container_set& ch,
                                        const sos_Container_set& wr,
                                        sos_Sync_mode s)
{
   T_PROC("sos_Container_set::open");
   TT(psm_H, T_ENTER);

#ifdef OBST_HAVE_JOYCE
   if ((psm_is_syncactivated) && (sync_deadlock)) 
   { 
      TT(psm_H, T_LEAVE); 
      return DEADLOCK; 
   }
#endif

   int i = 0;
   int j ;   // not in switch because of warnings
   sos_Int rd_ch_wr_card = rd.card() + ch.card() + wr.card();
   while (i < rd_ch_wr_card)
      switch (sos_Container_set::op(i, rd, ch, wr, TESTING)) 
      {
         case OPENED : 
            i++; 
            break;
         case UNACCESSIBLE :
            for (j = 0; j < i; j++) 
               sos_Container_set::cl(j, rd, ch, wr);
            TT(psm_H, T_LEAVE; TI((int)UNACCESSIBLE));
            return UNACCESSIBLE;
         case LOCKED :
            for (j = 0; j < i; j++) 
               sos_Container_set::cl(j, rd, ch, wr);
            if (s == TESTING) 
            {   
               TT(psm_H, T_LEAVE; TI((int)LOCKED));
               return LOCKED;
            }
            switch (sos_Container_set::op(i, rd, ch, wr, WAITING)) 
            {
            case UNACCESSIBLE : 
               TT(psm_H, T_LEAVE; TI((int)UNACCESSIBLE));
               return UNACCESSIBLE;
            case OPENED : 
               sos_Container_set::cl(i, rd, ch, wr);
               i = 0;
               break;
            case DEADLOCK : TT(psm_H, T_LEAVE; TI((int)DEADLOCK));
               return DEADLOCK;
               // break; 
            default : 
               err_raise(err_SYS, "switch error");
            }
         case DEADLOCK:
         default:
            err_raise(err_SYS, "switch error");
      }


   TT(psm_H, T_LEAVE; TI((int)OPENED));
   return OPENED; 
}

//------------------------------------------------------------------------------

sos_Open_result sos_Container_set::op(int i, const sos_Container_set& rd,
                                      const sos_Container_set& ch,
                                      const sos_Container_set& wr,
                                      sos_Sync_mode s) 
{  
   if (i < rd.card()) 
      return rd.set->get_cnt(i).open(READING, s);
   else 
      if (i <(rd.card()+ch.card()))
         return ch.set->get_cnt(i-rd.card()).open(CHECKOUT, s);
      else       
         return wr.set->get_cnt(i-rd.card()-ch.card()).open(WRITING, s); 
}

//------------------------------------------------------------------------------

void sos_Container_set::reset() const 
{
   T_PROC("sos_Container_set::reset"); TT(psm_H, T_ENTER);

#ifdef OBST_HAVE_JOYCE
   if ((psm_is_syncactivated) && (sync_deadlock))
   { 
      TT(psm_H, T_LEAVE); 
      return; 
   } 
#endif
   int i;
   if ((i = card()-1) >= 0)
   {  
      do 
      {
         sos_Container::make(set->get_cnt(i)).reset();
      } while (i--);
   }
   TT(psm_H, T_LEAVE);
}


// ================= class sos_Container_list ==================================

sos_Container_list::sos_Container_list() 
{
   T_PROC("sos_Container_list::sos_Container_list");
   TT(psm_H, T_ENTER);

   sz                   = 16;
   set                  = new sos_Int[sz];
   number_of_cnt        = 0;
   number_of_references = 0; 

   TT(psm_H, T_LEAVE);
}

//------------------------------------------------------------------------------

sos_Container_list::~sos_Container_list() 
{
   T_PROC("sos_Container_list::~sos_Container_list");
   TT(psm_H, T_ENTER);
    
#ifdef VECTOR_DELETE_WITH_SIZE
   delete [sz] set;
#else
   delete set;
#endif

   TT(psm_H, T_LEAVE);
}

//------------------------------------------------------------------------------

void sos_Container_list::add_cnt(sos_Container c) 
{
   T_PROC("sos_Container_list::add_cnt(sos_Container)");
   TT(psm_H, T_ENTER);

   if (number_of_cnt == sz)
   {  
      set =(sos_Int*)psm_extend_tbl(set, psm_SZ_OF_INT, sz, 2*sz);
      sz *= 2;
   }
   for (int i = number_of_cnt-1; i >= 0; --i) 
      // multiple appearance of the same container ... 
      // ... in one set(not bag!) not allowed
      if (set[i] == c)                  
      {  
         TT(psm_H, T_LEAVE);
         return;
      }
   set[number_of_cnt ++] =(sos_Int)c;
   TT(psm_H, T_LEAVE);
   return; 
}

//------------------------------------------------------------------------------

sos_Bool sos_Container_list::is_referenced()
{ 
   // implementation not in psm.h because of usage of(sos_Bool)
   return(sos_Bool)(number_of_references==0);
} 

//------------------------------------------------------------------------------

void sos_Container_list::rm_cnt(sos_Container c) 
{
   T_PROC("sos_Container_list::rm_cnt(sos_Container)");
   TT(psm_H, T_ENTER);

   sos_Bool is_first = TRUE;
   int curr_number_of_cnt=number_of_cnt;
 
   for (int i = 0; i < curr_number_of_cnt ; ++i) 
   {
      if (! is_first)
         set[i-1]=set[i];
      if (set[i] == c)              
         is_first = FALSE;
   }

   if (! is_first)
       number_of_cnt--;

   TT(psm_H, T_LEAVE);
}


// ================= sos_Container_set and sos_Container_cursor ================

void sos_Container_set::close_cursor(sos_Container_cursor &c) const 
{ 
   T_PROC("sos_Container_set::close_cursor"); TT(psm_L, T_ENTER);
   c.idx = -1;
   TT(psm_L, T_LEAVE);
}

//------------------------------------------------------------------------------

sos_Container sos_Container_set::get(sos_Container_cursor c) const 
{
   T_PROC("sos_Container_set::get"); TT(psm_L, T_ENTER);
   if (! this->is_valid(c))  
      err_raise(err_SYS, err_PSM_INVALID_CURSOR, psm_ERR, FALSE); 
   TT(psm_L, T_LEAVE);
   return sos_Container::make(set->get_cnt(c.idx));
}

//------------------------------------------------------------------------------

sos_Bool sos_Container_set::is_valid(sos_Container_cursor c) const 
{ 
   T_PROC("sos_Container_set::is_valid"); TT(psm_L, T_ENTER);
   TT(psm_L, T_LEAVE);
   return(sos_Bool)(c.idx >= 0  &&  c.idx < card());  
}

//------------------------------------------------------------------------------

sos_Container_cursor sos_Container_set::open_cursor() const 
{
   // creates a new cursor-object and positions it to the first element ...
   // ... if one exists

   T_PROC("sos_Container_set::open_cursor"); TT(psm_L, T_ENTER);
   sos_Container_cursor new_c;
   new_c.idx =(card() > 0) ? 0 : -1 ;
   TT(psm_L, T_LEAVE; TI(new_c.idx));
   return new_c;
}

//------------------------------------------------------------------------------

sos_Bool sos_Container_set::to_succ(sos_Container_cursor& c) const 
{  
   // moves cursor one position further and return FALSE, if 'end of ... 
   // ... set' is reached, else TRUE
   T_PROC("sos_Container_set::to_succ"); TT(psm_L, T_ENTER);

   if (++c.idx >= card()) 
   {  
      c.idx = -1;
      TT(psm_L, T_LEAVE; TI(FALSE));
      return FALSE;
   } 
   else 
   {  
      TT(psm_L, T_LEAVE; TI(TRUE));
      return TRUE;
   }
} 
