#line 1 "./src/trans/trans.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany. 
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */

/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : trans - schema implementation                                    */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Claus Engel                         DATE: 30.04.1992              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* VERSION: none                                                             */
/* ========================================================================= */

#define OBST_IMP_FILE
#include "obst_stdinc.h"
#include "_obst_config.h"

#include "obst.h"
#include "obst_progstd.h"
#include "sync_obst.h" // necessary to enable scp to optimize SyncCont::get_root
#include "trans_obst.h"
#include "trans_err.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "sync_trc.h"

#ifdef OBST_HAVE_XTDSYNC
#include "ext_sync_obst.h"
#endif

#include "smg.h"
#include "trc_tsy.h"
#include "psm_util.h"

/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: sos_Transaction                                               */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Claus Engel                         DATE: 30.04.1992              */
/*                                                                           */
/* ========================================================================= */

void _sos_Transaction::local_initialize (OBST_PARDECL(sos_Transaction) sos_ta)
{
    T_PROC("sos_Transaction::local_initialize");
    TT(tsy_H, T_ENTER);

    static sos_Bool instance_exists /* = FALSE */;
    smg_String      cnt_name;
    int             fd;


    if (instance_exists)
       err_raise( err_SYS, err_TRANS_CREATE_SECOND, ERR, FALSE);
      else
       instance_exists = TRUE;

    sos_ta.set_status (INACTIVE);

    // syncactivated indicates if  transaction management is turned on/off.
    // If SYNC_CONTAINER exists then ta management is turned on.

    cnt_name = smg_String (obst_getenv(ENVVAR_CONTAINER)) + "/" + 2;
    fd = ::open(cnt_name.make_Cstring (SMG_BORROW),O_RDONLY);
    if (fd != -1)
    {
       ::close(fd);
       sos_ta.set_sync_on (TRUE);
    }
    else
      sos_ta.set_sync_on (FALSE);

    sos_ta.set_error (SYNC_OK);
    
    TT(tsy_H, T_LEAVE);
    return;
}

/* ================================================================ */
/* Method: sos_Transaction::start                                   */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_NOT_STARTED, SYNC_TA_IMPLICIT,           */
/*        SYNC_TA_ALREADY_EXISTS, (Transaction::create)             */
/* ================================================================ */

void _sos_Transaction::start (const sos_Typed_id&_OBSThis,sos_Transaction_tp t)
const{
    T_PROC("sos_Transaction::start");
    TT(tsy_H, T_ENTER);

    Transaction     ta;
    SyncCont        sc;
    sos_Open_result result;
    sos_HostProc_Id hpid;


    if (!get_sync_on(_OBSThis))
      { // Dummy method
        set_error(_OBSThis,SYNC_OK);
        TT(tsy_H, TXT("Dummy, JOYCE deaktivated"); T_LEAVE);
        return;
      } 

    result = SYNC_CONTAINER.open (WRITING, WAITING);  // begin critical section
    if (result != OPENED)
     { TT(tsy_H, TXT("cannot find SYNC_CONTAINER"); T_LEAVE);
       err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE); }

    sc   = _SyncCont::get_root();
    hpid = sc.get_HP();

    if (sc.exists_TA(hpid))
    {   // if there already exists an transaction belonging to this process 
        // (implicit or explicit) => Ignore Start 

        ta = sc.get_ta_tab()[hpid];

        if (ta.get_implicit())
          { set_error(_OBSThis,SYNC_TA_IMPLICIT);
            TT(tsy_H, TXT("SYNC_TA_IMPLICIT")); }
         else 
          { set_error(_OBSThis,SYNC_TA_ALREADY_EXISTS);
            TT(tsy_H, TXT("SYNC_TA_ALREADY_EXISTS")); }

        SYNC_CONTAINER.close ();                // leave critical section
        TT (tsy_H, T_LEAVE);
        return;
    }

    // create new transaction of specified type 

#ifdef OBST_HAVE_XTDSYNC
    switch (t)
    {
    case LOCKING:
        ta = ExtLockingTA::create (SYNC_CONTAINER);
        break;
    case TIMESTAMP:
        ta = TimestampTA::create (SYNC_CONTAINER);
        break;
    case OPTIMISTIC:
        ta = OptimisticTA::create (SYNC_CONTAINER);
        break;
    };

#else
    // Currently only LOCKING is allowed <af>
    if (t != LOCKING)
      { err_raise(err_WNG, err_TRANS_ONLY_LOCKING_IMPLEMENTED, ERR, FALSE);
        t = LOCKING;
      }
  
    ta = LockingTA::create (SYNC_CONTAINER);
#endif

    TT(tsy_L, TXT("Transaction = "); TTA(ta));

    set_error ( _OBSThis,INVALID(ta) ? SYNC_TA_NOT_STARTED : SYNC_OK );

    SYNC_CONTAINER.close ();                    // leave critical section

    TT (tsy_H, T_LEAVE);
}


/* ================================================================ */
/* Method: sos_Transaction::commit                                  */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_NOT_ACTIVE, SYNC_TA_IMPLICIT (commit)    */
/* ================================================================ */

void _sos_Transaction::commit (const sos_Typed_id&_OBSThis)
const{
    T_PROC("sos_Transaction::commit");
    TT(tsy_H, T_ENTER);

    Transaction     ta;
    SyncCont        sc;
    sos_Open_result result;


    if (!get_sync_on(_OBSThis))
      { // Dummy method
        set_error(_OBSThis,SYNC_OK);
        TT(tsy_H, TXT("Dummy, JOYCE deaktivated"); T_LEAVE);
        return; 
      }

    result = SYNC_CONTAINER.open (WRITING, WAITING);  // begin critical section
    if (result != OPENED)
     { TT(tsy_H, TXT("cannot find SYNC_CONTAINER"); T_LEAVE);
       err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE); }

    sc = _SyncCont::get_root();

    ta = sc.get_ta_tab()[sc.get_HP()];
    if INVALID(ta) 
    {   // no transaction active 

        set_error (_OBSThis,SYNC_TA_NOT_ACTIVE);

        SYNC_CONTAINER.close ();                // leave critical section
        TT (tsy_H, TXT("SYNC_TA_NOT_ACTIVE"); T_LEAVE);
        return;
    }

    if (ta.get_implicit())
    {   // transaction was started implicit => no explicit commit possible

        set_error (_OBSThis,SYNC_TA_IMPLICIT);

        SYNC_CONTAINER.close ();                // leave critical section
        TT (tsy_H, TXT("SYNC_TA_IMPLICIT"); T_LEAVE);
        return;
    }

    // commit on explizit started transaction 
    ta.commit ();

    // save status for later queries
    set_error(_OBSThis,ta.get_error());
    set_status (_OBSThis,ta.get_status());
    TT(tsy_L, TXT("Status = "); TTS(get_status(_OBSThis)));

    // reset deadlock indication 
    sync_deadlock  = FALSE;
    psm_is_deadlock= FALSE;

    // delete transactions object from container 
    ta.destroy();

    SYNC_CONTAINER.close ();                    // leave critical section

    TT (tsy_H, T_LEAVE);
}

/* ================================================================ */
/* Method: sos_Transaction::abort                                   */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_NOT_ACTIVE, SYNC_TA_IMPLICIT (abort)     */
/* ================================================================ */

void _sos_Transaction::abort (const sos_Typed_id&_OBSThis)
const{
    T_PROC("sos_Transaction::abort");
    TT(tsy_H, T_ENTER);

    Transaction     ta;
    SyncCont        sc;
    sos_Open_result result;
    sos_HostProc_Id hpid;


    if (!get_sync_on(_OBSThis))
      { // Dummy method
        set_error(_OBSThis,SYNC_OK);
        TT(tsy_H, TXT("Dummy, JOYCE deaktivated"); T_LEAVE);
        return; 
      }

    result = SYNC_CONTAINER.open (WRITING, WAITING);  // begin critical section
    if (result != OPENED)
     { TT(tsy_H, TXT("cannot find SYNC_CONTAINER"); T_LEAVE);
       err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE); }

    sc   = _SyncCont::get_root();
    hpid = sc.get_HP();

    ta   = sc.get_ta_tab()[hpid];
    if INVALID(ta) 
    {   // no transaction active 

        set_error (_OBSThis,SYNC_TA_NOT_ACTIVE);

        SYNC_CONTAINER.close ();                // leave critical section
        TT (tsy_H, TXT("SYNC_TA_NOT_ACTIVE"); T_LEAVE);
        return;
    }

    if (ta.get_implicit())
    {   // implicit transaction => no explicit abort possible 

        set_error (_OBSThis,SYNC_TA_IMPLICIT);

        SYNC_CONTAINER.close ();                // leave critical section
        TT (tsy_H, TXT("SYNC_TA_IMPLICIT"); T_LEAVE);
        return;
    }

    // abort on explicit started transaction 
    ta.abort ();

    // save status for later queries 
    set_error(_OBSThis,ta.get_error());
    set_status (_OBSThis,ABORTED);
    TT(tsy_L, TXT("Status = "); TTS(get_status(_OBSThis)));

    // remove deadlock indication 
    sync_deadlock   = FALSE;
    psm_is_deadlock = FALSE;

    // delete transaction object from the container 
    ta.destroy();

    SYNC_CONTAINER.close ();                    // leave critical section

    TT (tsy_H, T_LEAVE);
}

/* ================================================================ */
/* Method: sos_Transaction::prev_transaction                        */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

sos_Transaction_status _sos_Transaction::prev_transaction (const sos_Typed_id&_OBSThis)
const{
    T_PROC ("sos_Transaction::prev_transaction");
    TT(tsy_H, T_ENTER);

    sos_Transaction_status status;


    if (get_sync_on(_OBSThis))
      { status = get_status(_OBSThis);
        TT (tsy_H, TTS(status)); }
      else
      { status = INACTIVE;
        TT(tsy_H, TXT("Dummy: JOYCE deaktivated"));
      }
 
    TT(tsy_H, T_LEAVE);
    set_error(_OBSThis,SYNC_OK);
    return status;
}

/* ================================================================ */
/* Method: sos_Transaction::act_ta_status                           */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

sos_Transaction_status _sos_Transaction::act_ta_status (const sos_Typed_id&_OBSThis)
const{
    T_PROC("sos_Transaction::act_ta_status");
    TT(tsy_H, T_ENTER);

    Transaction            ta;
    SyncCont               sc;
    sos_Open_result        result;
    sos_HostProc_Id        hpid;
    sos_Transaction_status status;


    if (!get_sync_on(_OBSThis))
      { // Dummy method
        set_error(_OBSThis,SYNC_OK);
        TT(tsy_H, TXT("Dummy, JOYCE deaktivated"); T_LEAVE);
        return INACTIVE; 
      }

    result = SYNC_CONTAINER.open (WRITING, WAITING); // begin critical section
    if (result != OPENED)
     { TT(tsy_H, TXT("cannot find SYNC_CONTAINER"); T_LEAVE);
       err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE); }

    sc   = _SyncCont::get_root();
    hpid = sc.get_HP();
 
    ta   = sc.get_ta_tab()[hpid];
    if (INVALID(ta) || (ta.get_implicit()))
       status = INACTIVE;
    else
       status = ta.get_status();
    

    SYNC_CONTAINER.close ();                    // leave critical section

    TT (tsy_H, TTS(status); T_LEAVE);
    set_error(_OBSThis,SYNC_OK);
    return status;
}

/* ================================================================ */
/* Method: sos_Transaction::act_ta_tp                               */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

sos_Transaction_tp _sos_Transaction::act_ta_tp (const sos_Typed_id&_OBSThis)
const{
    T_PROC("sos_Transaction::act_ta_tp");
    TT(tsy_H, T_ENTER);

    Transaction        ta;
    SyncCont           sc;
    sos_Open_result    result;
    sos_HostProc_Id    hpid;
    sos_Transaction_tp tp;


    if (!get_sync_on(_OBSThis))
      { // Dummy method
        set_error(_OBSThis,SYNC_OK);
        TT(tsy_H, TXT("Dummy, JOYCE deaktivated"); T_LEAVE);
        return NOTYPE;
      }

    result = SYNC_CONTAINER.open (WRITING, WAITING); // begin critical section
    if (result != OPENED)
     { TT(tsy_H, TXT("cannot find SYNC_CONTAINER"); T_LEAVE);
       err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE); }

    sc   = _SyncCont::get_root();
    hpid = sc.get_HP();

    ta = sc.get_ta_tab()[hpid];
    if (INVALID(ta) || (ta.get_implicit()))
       tp = NOTYPE;
    else
       tp = ta.get_type();


    SYNC_CONTAINER.close ();                    // leave critical section

    TT (tsy_H, TTT(tp); T_LEAVE);
    set_error(_OBSThis,SYNC_OK);
    return tp;
}

