/* Find a variable's value in memory, for GDB, the GNU debugger.
   Copyright (C) 1989 Free Software Foundation, Inc.

   MODIFIED BY CHRIS BENENATI, FOR INTEL CORPORATION, 12/88

GDB is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY.  No author or distributor accepts responsibility to anyone
for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.
Refer to the GDB General Public License for full details.

Everyone is granted permission to copy, modify and redistribute GDB,
but only under the conditions described in the GDB General Public
License.  A copy of this license is supposed to have been given to you
along with GDB so you can know your rights and responsibilities.  It
should be in a file named COPYING.  Among other things, the copyright
notice and this notice must be preserved on all copies.

In other words, go ahead and share GDB, but don't try to stop
anyone else from sharing it farther.  Help stamp out software hoarding!
*/

#include "defs.h"
#include "param.h"
#include "symtab.h"
#include "frame.h"
#include "value.h"

CORE_ADDR read_register ();

/* Return the address in which frame FRAME's value of register REGNUM
   has been saved in memory.  Or return zero if it has not been saved.
   If REGNUM specifies the SP, the value we return is actually
   the SP value, not an address where it was saved.  */

static CORE_ADDR
find_saved_register (frame, regnum)
     FRAME frame;
     int regnum;
{
  struct frame_info fi;
  struct frame_saved_regs saved_regs;

  register FRAME frame1 = 0;
  register CORE_ADDR addr = 0;

#ifdef I80960

  /* On 80960, calling function saves its OWN registers into its OWN stack
   * frame (actually, into the register cache until the cache is depleted
   * or a flushregs is done;  the latter is done by the NINDY monitor
   * when execution halts).
   *
   * While we're at it, we'll clean the code up a bit as well:
   *	o use get_prev_frame() instead of get_prev_frame_info()
   *	o call FRAME_FIND_SAVED_REGS directly, instead of going through the
   *		intermediary
   *	o make the arguments to the FRAME_FIND_SAVED_REGS more reasonable so
   *		it can be implemented as a function instead of a macro.
   */
  if ( frame == get_current_frame() ){
	return 0;
  }
  while ( frame1 != frame ){
	QUIT;
	frame1 = get_prev_frame( frame1 );
	if ( frame1 == 0 ){
		return 0;
	}
  }
  FRAME_FIND_SAVED_REGS( frame1, &saved_regs );
  if ( saved_regs.regs[regnum] ){
	addr = saved_regs.regs[regnum];
  }

#else

  while (1)
    {
      QUIT;
      fi = get_prev_frame_info (frame1);
      if (fi.frame == 0 || fi.frame == frame)
	break;
      get_frame_saved_regs (&fi, &saved_regs);
      if (saved_regs.regs[regnum])
	addr = saved_regs.regs[regnum];
      frame1 = fi.frame;
    }

#endif

  return addr;
}

/* Copy the bytes of register REGNUM, relative to the current stack frame,
   into our memory at MYADDR.
   The number of bytes copied is REGISTER_RAW_SIZE (REGNUM).  */

void
read_relative_register_raw_bytes (regnum, myaddr)
     int regnum;
     char *myaddr;
{
  register CORE_ADDR addr;

  if (regnum == FP_REGNUM)
    {
      bcopy (&selected_frame, myaddr, sizeof (CORE_ADDR));
      return;
    }

  addr = find_saved_register (selected_frame, regnum);

  if (addr)
    {
      if (regnum == SP_REGNUM)
	{
	  CORE_ADDR buffer = addr;
	  bcopy (&buffer, myaddr, sizeof (CORE_ADDR));
	}
      else
	read_memory (addr, myaddr, REGISTER_RAW_SIZE (regnum));
#ifdef I80960
	byteswap( myaddr, REGISTER_RAW_SIZE(regnum) );
			/* i960 byte-order to host byte-order */
#endif
      return;
    }
  read_register_bytes (REGISTER_BYTE (regnum),
		       myaddr, REGISTER_RAW_SIZE (regnum));
#ifdef I80960
  byteswap( myaddr, REGISTER_RAW_SIZE(regnum) );
			/* i960 byte-order to host byte-order */
#endif
}

/* Return a `value' with the contents of register REGNUM
   in its virtual format, with the type specified by
   REGISTER_VIRTUAL_TYPE.  */

value
value_of_register (regnum)
     int regnum;
{
  register CORE_ADDR addr = find_saved_register (selected_frame, regnum);
  register value val;
  char raw_buffer[MAX_REGISTER_RAW_SIZE];
  char virtual_buffer[MAX_REGISTER_VIRTUAL_SIZE];

  if (addr)
    {
      if (regnum == SP_REGNUM)
	return value_from_long (builtin_type_int, addr);
      read_memory (addr, raw_buffer, REGISTER_RAW_SIZE (regnum));
    }
  else
    read_register_bytes (REGISTER_BYTE (regnum), raw_buffer,
			 REGISTER_RAW_SIZE (regnum));
#ifdef I80960
  byteswap( raw_buffer, REGISTER_RAW_SIZE(regnum) );
			/* i960 byte-order to host byte-order */
#endif	/* I80960 */

  REGISTER_CONVERT_TO_VIRTUAL (regnum, raw_buffer, virtual_buffer);
  val = allocate_value (REGISTER_VIRTUAL_TYPE (regnum));
  bcopy (virtual_buffer, VALUE_CONTENTS (val), REGISTER_VIRTUAL_SIZE (regnum));
  VALUE_LVAL (val) = addr ? lval_memory : lval_register;
  VALUE_ADDRESS (val) = addr ? addr : REGISTER_BYTE (regnum);
  VALUE_REGNO (val) = regnum;
  return val;
}

#ifndef I80960		/* SEE remote.c FOR I80960 VERSIONS */

/* Low level examining and depositing of registers.

   Note that you must call `fetch_registers' once
   before examining or depositing any registers.  */

char registers[REGISTER_BYTES];

/* Copy LEN bytes of consecutive data from registers
   starting with the REGBYTE'th byte of register data
   into memory at MYADDR.  */

read_register_bytes (regbyte, myaddr, len)
     int regbyte;
     char *myaddr;
     int len;
{
  bcopy (&registers[regbyte], myaddr, len);
}

/* Copy LEN bytes of consecutive data from memory at MYADDR
   into registers starting with the REGBYTE'th byte of register data.  */

write_register_bytes (regbyte, myaddr, len)
     int regbyte;
     char *myaddr;
     int len;
{
  bcopy (myaddr, &registers[regbyte], len);
  if (have_inferior_p ())
    store_inferior_registers (-1);
}

/* Return the contents of register REGNO,
   regarding it as an integer.  */

CORE_ADDR
read_register (regno)
     int regno;
{
  /* This loses when REGISTER_RAW_SIZE (regno) != sizeof (int) */
  return *(int *) &registers[REGISTER_BYTE (regno)];
}

/* Store VALUE in the register number REGNO, regarded as an integer.  */

void
write_register (regno, val)
     int regno, val;
{
  /* This loses when REGISTER_RAW_SIZE (regno) != sizeof (int) */

#if defined(sun4)
  /* This is a no-op on a Sun 4.  */
  if (regno == 0)
    return;
#endif

  *(int *) &registers[REGISTER_BYTE (regno)] = val;

  if (have_inferior_p ())
    store_inferior_registers (regno);
}

/* Record that register REGNO contains VAL.
   This is used when the value is obtained from the inferior or core dump,
   so there is no need to store the value there.  */

void
supply_register (regno, val)
     int regno;
     char *val;
{
  bcopy (val, &registers[REGISTER_BYTE (regno)], REGISTER_RAW_SIZE (regno));
}
#endif	/* not I80960 */

/* Given a struct symbol for a variable,
   and a stack frame address, read the value of the variable
   and return a (pointer to a) struct value containing the value.  */

value
read_var_value (var, frame)
     register struct symbol *var;
     FRAME frame;
{
  register value v;

  struct frame_info fi;

  struct type *type = SYMBOL_TYPE (var);
  register CORE_ADDR addr = 0;
  int val = SYMBOL_VALUE (var);
  register int len;
  int numbytes;
#ifdef I80960
  /* On i80960, we might have small structures or arrays passed in
   * registers g0-g3.  Want to make sure we pick up as many bytes as
   * necessary, rather than stopping at raw size of one register.
   */
  char raw_buffer[4*MAX_REGISTER_RAW_SIZE];
  char virtual_buffer[4*MAX_REGISTER_VIRTUAL_SIZE];
#else
  char raw_buffer[MAX_REGISTER_RAW_SIZE];
  char virtual_buffer[MAX_REGISTER_VIRTUAL_SIZE];
#endif

  v = allocate_value (type);
  VALUE_LVAL (v) = lval_memory;	/* The most likely possibility.  */
  len = TYPE_LENGTH (type);

  if (frame == 0) frame = selected_frame;

  switch (SYMBOL_CLASS (var))
    {
    case LOC_CONST:
    case LOC_LABEL:
      bcopy (&val, VALUE_CONTENTS (v), len);
      VALUE_LVAL (v) = not_lval;
      return v;

    case LOC_CONST_BYTES:
      bcopy (val, VALUE_CONTENTS (v), len);
      VALUE_LVAL (v) = not_lval;
      return v;

    case LOC_STATIC:
      addr = val;
      break;

#ifdef I80960

    case LOC_ARG_BLOCK:
    case LOC_ARG:
      fi = get_frame_info (frame);
      addr = arg_address( var, &fi );
      break;

#else	/* not I80960 */

    case LOC_ARG:
      fi = get_frame_info (frame);
      addr = val + FRAME_ARGS_ADDRESS (fi);
      break;
#endif

    case LOC_LOCAL:
      fi = get_frame_info (frame);
      addr = val + FRAME_LOCALS_ADDRESS (fi);
      break;

    case LOC_TYPEDEF:
      error ("Cannot look up value of a typedef");

    case LOC_BLOCK:
      VALUE_ADDRESS (v) = BLOCK_START (SYMBOL_BLOCK_VALUE (var));
      return v;

    case LOC_REGPARM:
    case LOC_REGISTER:
      {

	VALUE_REGNO (v) = val;
#ifdef I80960
	if ( (val >= G0_REGNUM) && (val <= FP_REGNUM) ){
		/* Can't be sure if global register still
		 * contains the argument on i80960
		 */
		dunno = 1;
	}
#endif

	/* Locate the register's contents in a real register or in core;
	   read the data in raw format.  */

	numbytes = REGISTER_RAW_SIZE(val);
#ifdef I80960
	/* On i80960, we might have small structures or arrays passed in
	 * registers g0-g3.  Want to make sure we pick up as many bytes as
	 * necessary, rather than stopping at raw size of one register.
	 */
	if ( len > REGISTER_RAW_SIZE(val) ){
		numbytes = len;
	}
#endif

	addr = find_saved_register (frame, val);
	if (addr == 0)
	  {
	    /* Value is really in a register.  */

	    VALUE_LVAL (v) = lval_register;
	    VALUE_ADDRESS (v) = REGISTER_BYTE (val);
	    read_register_bytes (REGISTER_BYTE (val), raw_buffer, numbytes);
	  }
	else
	  {
	    /* Value was in a register that has been saved in memory.  */

	    read_memory (addr, raw_buffer, numbytes);
	    VALUE_ADDRESS (v) = addr;
	  }

	/* Convert the raw contents to virtual contents.
	   (Just copy them if the formats are the same.)  */

#ifdef I80960
	/* On i80960, we might have small structures or arrays passed in
	 * registers g0-g3.  Want to make sure we pick up as many bytes as
	 * necessary, rather than stopping at raw size of one register.
	 */
	bcopy( raw_buffer, virtual_buffer, numbytes );
#else
	REGISTER_CONVERT_TO_VIRTUAL (val, raw_buffer, virtual_buffer);
#endif

	if (REGISTER_CONVERTIBLE (val))
	  {
	    /* When the raw and virtual formats differ, the virtual format
	       corresponds to a specific data type.  If we want that type,
	       copy the data into the value.
	       Otherwise, do a type-conversion.  */

	    if (type != REGISTER_VIRTUAL_TYPE (val))
	      {
		/* eg a variable of type `float' in a 68881 register
		   with raw type `extended' and virtual type `double'.
		   Fetch it as a `double' and then convert to `float'.  */
		v = allocate_value (REGISTER_VIRTUAL_TYPE (val));
		bcopy (virtual_buffer, VALUE_CONTENTS (v), len);
		v = value_cast (type, v);
	      }
	    else
	      bcopy (virtual_buffer, VALUE_CONTENTS (v), len);
	  }
	else
	  {
	    /* Raw and virtual formats are the same for this register.  */

#ifndef I80960
	    /* i80960 register data is always in little-end order, regardless of
 	     * host machine.
 	     */

	    union { int i; char c; } test;
	    /* If we want less than the full size, we need to
	       test for a big-endian or little-endian machine.  */
	    test.i = 1;
	    if (test.c != 1 && len < REGISTER_RAW_SIZE (val))
	      {
		/* Big-endian, and we want less than full size.  */
		VALUE_OFFSET (v) = REGISTER_RAW_SIZE (val) - len;
	      }
#endif	/* not I80960 */

	    bcopy (virtual_buffer + VALUE_OFFSET (v),
		   VALUE_CONTENTS (v), len);
	  }

#ifdef I80960
	byteswap_val( v );	/* i960 byte-order to host byte-order */
#endif
	return v;
      }
    }

  read_memory (addr, VALUE_CONTENTS (v), len);
  VALUE_ADDRESS (v) = addr;
#ifdef I80960
  byteswap_val( v );		/* i960 byte-order to host byte-order */
#endif
  return v;
}

/* Given a struct symbol for a variable,
   and a stack frame address,
   return a (pointer to a) struct value containing the variable's address.  */

value
locate_var_value (var, frame)
     register struct symbol *var;
     FRAME frame;
{
  register CORE_ADDR addr = 0;
  int val = SYMBOL_VALUE (var);
  struct frame_info fi;
  struct type *type = SYMBOL_TYPE (var);

  if (frame == 0) frame = selected_frame;

  switch (SYMBOL_CLASS (var))
    {
    case LOC_CONST:
    case LOC_CONST_BYTES:
      error ("Address requested for identifier \"%s\" which is a constant.",
	     SYMBOL_NAME (var));

    case LOC_REGISTER:
    case LOC_REGPARM:
      addr = find_saved_register (frame, val);
      if (addr != 0)
	{
	  union { int i; char c; } test;
	  int len = TYPE_LENGTH (type);

#ifndef I80960
	    /* i80960 register data is always in little-end order, regardless of
 	     * host machine.
 	     */

	  /* If var is less than the full size of register, we need to
	     test for a big-endian or little-endian machine.  */
	  test.i = 1;
	  if (test.c != 1 && len < REGISTER_RAW_SIZE (val))
	    /* Big-endian, and we want less than full size.  */
	    addr += REGISTER_RAW_SIZE (val) - len;
#endif	/* not I80960 */

	  break;
	}
      error ("Address requested for identifier \"%s\" which is in a register.",
	     SYMBOL_NAME (var));

    case LOC_STATIC:
    case LOC_LABEL:
      addr = val;
      break;

#ifdef I80960

    case LOC_ARG_BLOCK:
    case LOC_ARG:
      fi = get_frame_info (frame);
      addr = arg_address( var, &fi );
      break;

#else	/* Not I80960 */

    case LOC_ARG:
      fi = get_frame_info (frame);
      addr = val + FRAME_ARGS_ADDRESS (fi);
      break;

#endif

    case LOC_LOCAL:
      fi = get_frame_info (frame);
      addr = val + FRAME_LOCALS_ADDRESS (fi);
      break;

    case LOC_TYPEDEF:
      error ("Address requested for identifier \"%s\" which is a typedef.",
	     SYMBOL_NAME (var));

    case LOC_BLOCK:
      addr = BLOCK_START (SYMBOL_BLOCK_VALUE (var));
      break;
    }

  return value_cast (lookup_pointer_type (type),
		     value_from_long (builtin_type_long, addr));
}
