/*
 * math.h : standard math routines.
 *
 * [atw] Inlined on a KB or MC for speed.
 *       To get a copy where all the routines are instantiated
 *       so they can be called by people who don't use headers,
 *       set the __INSTANTIATE_MATH_H flag.
 *       To suppress inlining, set the __NOINLINE__ flag.
 *       To suppress error checking in inlined routines (for benchmarking),
 *       set the __NOERRCHECK__ flag.
 */

#ifndef __MATH_H__
#define __MATH_H__

#include <errno.h>

#ifndef HUGE_VAL
#define HUGE_VAL 1e308
#endif


/* Math exception structure and function */
/* (Sys V-style) If you define _MATHERR_ when
   compiling & using the libraries, you'll get
   this facility; otherwise "errno" is set and the
   function returns, as per ANSI.  [atw]
*/

#ifdef _MATHERR_
typedef enum {DOMAIN, SING, OVERFLOW, UNDERFLOW, TLOSS, PLOSS} _MATHERRS;
struct exception {
	int type;
	char *name;
	double arg1;
	double arg2;
	double retval;
};
int matherr(struct exception *);
#endif

/* some useful constants */
#define M_E	2.7182818284590452354
#define M_LOG2E	1.4426950408889634074
#define M_LOG10_2	(M_LN2 * M_LOG10E)
#define M_LOG10E	0.43429448190325182765
#define M_LN2	0.69314718055994530942
#define M_LN10	2.30258509299404568402
#define M_PI	3.14159265358979323846
#define M_PI_2	1.57079632679489661923
#define M_PI_4	0.78539816339744830962
#define M_1_PI	0.31830988618379067154
#define M_2_PI	0.63661977236758134308
#define M_2_SQRTPI	1.12837916709551257390
#define M_SQRT2	1.41421356237309504880
#define M_SQRT1_2	0.70710678118654752440
#define MAXFLOAT	((float)1.701411733192644299e+38)
#define M_LN_MAXDOUBLE  (M_LN10*308)
#define M_INF_LOW 0x0L
#define M_INF_HIGH 0x7FF00000L
#define M_NEG_INF_HIGH 0xFFF00000L
#define HUGE	MAXFLOAT

/*
 * Define NaN's (using array initializers,
 * a useful GCC extension).
 */
#define M_QNAN	(*((const double *)((unsigned[]){0xFFFFFFFFL, 0xFFFFFFFFl})))
#define M_SNAN  (*((const double *)((unsigned[]){0xFFFFFFFFL, 0xFFF8FFFFL})))

/*
 * return 0.0 from domain errors.
 * (ANSI says this value is implementation-dependent).
 */
#define M_ERRVAL 0.0

#define _ABS(x)	((x) < 0 ? -(x) : (x))
#define _REDUCE(TYPE, X, XN, C1, C2)	{ \
	double x1 = (double)(TYPE)X, x2 = X - x1; \
	X = x1 - (XN) * (C1); X += x2; X -= (XN) * (C2); }
#define _POLY1(x, c)	((c)[0] * (x) + (c)[1])
#define _POLY2(x, c)	(_POLY1((x), (c)) * (x) + (c)[2])
#define _POLY3(x, c)	(_POLY2((x), (c)) * (x) + (c)[3])
#define _POLY4(x, c)	(_POLY3((x), (c)) * (x) + (c)[4])
#define _POLY5(x, c)	(_POLY4((x), (c)) * (x) + (c)[5])
#define _POLY6(x, c)	(_POLY5((x), (c)) * (x) + (c)[6])
#define _POLY7(x, c)	(_POLY6((x), (c)) * (x) + (c)[7])
#define _POLY8(x, c)	(_POLY7((x), (c)) * (x) + (c)[8])
#define _POLY9(x, c)	(_POLY8((x), (c)) * (x) + (c)[9])


#if defined(__GNUC__) && defined(__STDC__)
/*
 * Prototypes for out-of-line routines.
 */
double pow(double, double);
/* Hyperbolics */
double sinh(double);
double cosh(double);
double tanh(double);
double asin(double);
double acos(double);

#if (defined(__i960_KB__) || defined(__i960_MC__)) && !defined(__NOINLINE__)
/*
 * Additional inline assembly code.  Only included if
 * this is STDC and we're compiling for a KB or MC.
 */
#if !defined(__INSTANTIATE_MATH_H)
#define	INLINE __inline
#define STATIC static
#else
#define INLINE
#define STATIC
#endif

INLINE STATIC const double exp(double x) {
  double result, intpart, fracpart;
  float fintpart;
  unsigned long scalefac;
  unsigned long arith_controls, round_nearest=0, mask=3<<30;

  /*
   * check for overflow & underflow
   */
#ifndef __NOERRCHECK__
  if (x > M_LN_MAXDOUBLE)
    {
      errno = ERANGE;
      return HUGE_VAL;
    }
  else if (x < -M_LN_MAXDOUBLE)
    {
      errno = ERANGE;
      return 0.0;
    }
#endif
  /*
   * compute 2 ** (log2(e)*x)
   */
  x *= M_LOG2E;
  __asm volatile ("modac %2,%1,%0"    : "=d" (arith_controls) : "di" (round_nearest), "d" (mask));
  __asm volatile ("roundrl %1,%0 "    : "=dG" (intpart)       : "dG" (x));
  fracpart = x - intpart;
  __asm volatile ("exprl %1,%0"       : "=dG" (result)        : "dG" (fracpart));
  __asm volatile ("addrl 0f1.0,%1,%0" : "=dG" (result)        : "dG" (result));
  fintpart = intpart;
  __asm volatile ("cvtri %1,%0"       : "=d"  (scalefac)      : "d" (fintpart));
  __asm volatile ("scalerl %2,%1,%0"  : "=dG" (result)        : "dG" (result), "d" (scalefac));
  /*
   * restore rounding mode.
   */
  __asm volatile ("modac %2,%1,%0"  : "=d" (arith_controls) : "d" (arith_controls), "d" (mask));
  return result;
}

INLINE STATIC const double sin(double x) {
  double _value;
  __asm ("sinrl	%1,%0" : "=dG" (_value) : "dG" (x));
  return _value;
}

INLINE STATIC const double cos(double x) {
  double _value;
  __asm ("cosrl	%1,%0" : "=dG" (_value) : "dG" (x));
  return _value;
}

INLINE STATIC const double tan (double x)
{
  double _value;
  __asm ("tanrl	%1,%0" : "=dG" (_value) : "dG" (x));
  return _value;
}

INLINE STATIC const double atan(double x) {
  double _value;
  __asm ("atanrl 0f1.0,%1,%0" : "=dG" (_value) : "dG" (x));
  return _value;
}

INLINE STATIC const double atan2 (double y, double x) {
  double _value;
  
#ifndef __NOERRCHECK__
  if (y==0.0 && x==0.0)
    {
      errno = EDOM;
      return M_ERRVAL;
    }
#endif
  __asm ("atanrl %2,%1,%0" : "=dG" (_value) : "dG" (y), "dG" (x));
  return _value;
}

INLINE STATIC const double log10 (double x)
{
  double _value, log10=M_LOG10_2;

#ifndef __NOERRCHECK__
  if (x < 0)
    {
      errno = EDOM;
      return M_ERRVAL;
    }
  else if (x == 0.0)
    {
      errno = ERANGE;
      return -HUGE_VAL;
    }
#endif
  __asm ("logrl %1,%2,%0"
	 : "=dG" (_value)
	 : "dG" (x), "dG" (log10));
  return _value;
}

INLINE STATIC const double sqrt(double x) {
  double _value;

#ifndef __NOERRCHECK__
  if (x < 0)
    {
      errno = EDOM;
      return M_ERRVAL;
    }
#endif
  __asm("sqrtrl	%1,%0" : "=dG" (_value) : "dG" (x));
  return _value;
}

INLINE STATIC const double log (double x) {
  double _value, ln_2=M_LN2;

#ifndef __NOERRCHECK__
  if (x == 0.0)
    {
      errno = ERANGE;
      return -HUGE_VAL;
    }
  else if (x < 0)
    {
      errno = EDOM;
      return M_ERRVAL;
    }
#endif
  __asm ("logrl	%1,%2,%0" : "=dG" (_value) : "dG" (x), "dG" (ln_2));
  return _value;
}


INLINE STATIC const double ldexp (double x, int n)
{
  union {
    double _value;
    long   l[2];} d;

  __asm volatile ("scalerl %2,%1,%0"
	 	: "=fG" (d._value)
	 	: "fG" (x),
	   	  "d" (n));
#ifndef __NOERRCHECK__
  if (d.l[0]==M_INF_LOW && 
      ((d.l[1]==M_INF_HIGH) || (d.l[1]==M_NEG_INF_HIGH)))
    {
      /* we scaled ourselves off the scale...*/
      errno = ERANGE;
      return HUGE_VAL;
    }
#endif
  return d._value;
}

INLINE STATIC double frexp (double x, int *exp)
{
  double float_exponent;
  int int_exponent;
  double mantissa;
  typedef struct {
    unsigned long frac_low;
    unsigned frac_hi  : 20;
    unsigned exponent : 11;
    unsigned sign     : 1; } __DOUBLE;
  union {
    double d;
    __DOUBLE D; } un;

  if (x == 0.0)
    {
      *exp = 0;
      return 0.0;
    }
  un.d = x;
  /*
   * extract exponent, subtract bias
   */
  *exp = un.D.exponent - 1022;
  /*
   * return fractional part
   * in range 0.5 .. <1.0
   */
  un.D.exponent = 1022;
  return un.d;
}

INLINE STATIC const double fabs (double x)
{
  typedef struct {
    unsigned long frac_low;
    unsigned frac_hi  : 20;
    unsigned exponent : 11;
    unsigned sign     : 1; } __DOUBLE;
  union {
    double d;
    __DOUBLE D; } un;
  un.d = x;
  un.D.sign = 0;
  return un.d;
}

INLINE STATIC const double ceil (double x)
{
  unsigned long arith_controls;
  unsigned long round_inf=1<<31, mask=3<<30;
  double _value;

  /*
   * set the arith. controls to round toward positive infinity.
   */
  __asm volatile("modac %1,%2,%0" : "=d" (arith_controls) : "d" (mask), "d" (round_inf));
  __asm volatile("roundrl %1,%0"  : "=dG" (_value) : "dG" (x));

  /*
   * restore previous mode.
   */
  __asm volatile("modac %1,%2,%0" : "=d" (arith_controls) : "d" (mask), "d" (arith_controls));

  return _value;
}

INLINE STATIC const double floor (double x)
{
  unsigned long arith_controls;
  unsigned long round_down=0x40000000L, mask=0xc0000000L;
  double _value;

  /*
   * set the arith. controls to round toward negative infinity.
   */
  __asm volatile("modac %1,%2,%0" : "=d" (arith_controls) : "d" (mask), "d" (round_down));
  __asm volatile("roundrl %1,%0"  : "=dG" (_value) : "dG" (x));

  /*
   * restore previous mode.
   */
  __asm volatile("modac %1,%2,%0" : "=d" (arith_controls) : "d" (mask), "d" (arith_controls));

  return _value;
}

INLINE STATIC const double fmod (double x, double y)
{
  double _value;

  if (y==0.0)
    return 0.0;
  __asm ("remrl %2,%1,%0"
	 : "=dG" (_value)
	 : "dG" (x),
	   "dG" (y));
  return _value;
}

INLINE STATIC double modf (double x, double *ip)
{
  double temp, temp2;
  /*
   * Extract fractional part, then return
   * the integer part.
   */
  __asm volatile ("cvtzril %1,%0"
		  : "=dG" (temp)
		  : "fG" (x));
  __asm volatile ("cvtilr %1,%0"
		  : "=fG" (temp2)
		  : "dG" (temp));
  *ip = temp2;
  return (x-temp2);
}
#undef INLINE
#undef STATIC
#else
/*
 * Prototypes for routines that are inlined above.
 */
double atan(double);
double atan2(double, double);
double ceil(double);
double cos(double);
double exp(double);
double fabs(double);
double floor(double);
double fmod(double, double);
double frexp(double, int *);
double ldexp(double, int);
double log(double);
double log10(double);
double modf(double, double *);
double sin(double);
double sqrt(double);
double tan(double);
#endif /* KB or MC */

#else
/*
 * not STD_C ... no prototypes OR inlining ... just declare
 * types of everything.
 */
double exp();
double pow();
double sinh();
double cosh();
double tanh();
double asin();
double acos();
double atan();
double atan2();
double ceil();
double cos();
double fabs();
double floor();
double fmod();
double frexp();
double ldexp();
double log();
double log10();
double modf();
double sin();
double sqrt();
double tan();
#endif /* __GNU_C__ && __STD_C__ */

#endif /* __MATH_H__ */
