/*
 coff-convert.c - Converts M68K object files in the GNU a.out format
                  to Apollo COFF format

 Copyright (c) 1989  John Vasta
 Copyright (c) 1990  John Vasta and Michael Bloom

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 1, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software,
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 Questions about this program can be addressed to
	John Vasta	  (vasta@apollo.com)
	Apollo Computer, Inc.
	330 Billerica Rd.
	Chelmsford, MA 01824, USA

  or to
	Michael Bloom (mb@ttidca.tti.com)
	Citicorp/TTI
	3100 Ocean Park Blvd.
	Santa Monica, CA, 90405

 This module may be usable on other COFF systems, but has only been tested
 on Apollo, 386 systems (Sun, ESIX, 386/IX), and Citicorp/TTI PBB systems.
 It is designed to be built into the GNU assembler, which calls it just before
 writing out an object file.

 The current version of this file is drastically changed from the original
 Apollo version, and the most recent version has not been tested on Apollos.

 Changes for Sun 386i, Unicom/PBB, debugging symbol support, cross assembly and
 sparc by Michael Bloom, Citicorp/TTI (mb@ttidca.tti.com).

 The original version of this file used ANSI C prototypes, but this presented
 a potential chicken and egg bootstrapping problem, since some people may
 need to compile this file before they are able to get GCC running, and cant
 use prototypes until they have it running.  To avoid this problem, the
 prototypes were reluctantly removed. They may be conditionally restored at
 some future date.

*/
#ifndef COFF_OPTION

int convert_to_coff()
{
	/* do nothing */
}

#else /*  COFF_OPTION */

#include <stdio.h>

/* if stdio.h did not define L_tmpnam, then it probably did not define
 * tempnam() either.
 */
#ifndef L_tmpnam
extern char *tempnam();
#endif

#ifdef USG
#include <string.h>
#define rindex strrchr
#else
#include <strings.h>
#endif
#include <time.h>

#ifdef __GNUC__
#define alloca __builtin_alloca
#else
#ifdef sparc
#include <alloca.h>
#else
extern char * alloca();
#endif
#endif

#ifdef STANDALONE_CONV

void md_number_to_chars();
void as_fatal();
char *xmalloc(),*xrealloc();
#include "a.out.gnu.h"

#else /* not STANDALONE_CONV */

/* the GNU assembler interface */
#include "as.h"
#include "a.out.h" /* I think this will change to a.out.gnu.h with gas 1.37 */

#endif /* not STANDALONE_CONV */

extern char version_string[];

/* the a.out format specification */
#include "stab.gnu.h"

#undef	N_ABS	       /* conflicts with N_ABS in syms.h */
#define N_ABSOLUTE 2

#ifndef N_DATOFF
#define N_DATOFF(x) (N_TXTOFF(x) + (x).a_text)
#endif

#ifndef N_TRELOFF
#define N_TRELOFF(x) (N_DATOFF(x) + (x).a_data)
#endif

#ifndef N_DRELOFF
#define N_DRELOFF(x) (N_TRELOFF(x) + (x).a_trsize)
#endif

#ifndef N_TXTADDR
#define N_TXTADDR(x) 0
#endif

#ifndef N_DATADDR
#define N_DATADDR(x) (N_TXTADDR(x)+(x).a_text)
#define N_BSSADDR(x) (N_DATADDR(x) + (x).a_data)
#endif

#ifndef TARGET_68K

#define TARGET_68K	100
#define TARGET_APOLLO	150

#define TARGET_386	200
#define TARGET_SUN386	250

#define TARGET_SPARC	300
#define TARGET_NS32K	400
#define TARGET_VAX	500
#endif

#ifndef TARGET
#  ifdef sparc
#    define TARGET TARGET_SPARC
#  else
#    ifdef m68k
#      define TARGET TARGET_68K
#    else
#      ifdef sun386
#	 define TARGET TARGET_SUN386
#      else
#	 ifdef i386
#	   define TARGET TARGET_386
#	 endif
#      endif
#    endif
#  endif
#endif /* ! defined TARGET */

/* If you add a new TARGET_XXX for a machine whose cpu type is already
 * represented, set TARGET_ARCH to an existing TARGET_XXX.
 */

#if TARGET == TARGET_APOLLO
#define TARGET_ARCH TARGET_68K
#else /* not TARGET == TARGET_APOLLO */
#if TARGET == TARGET_SUN386
#define TARGET_ARCH TARGET_386
#else
#define TARGET_ARCH TARGET
#endif
#endif /* not TARGET == TARGET_APOLLO */

#define BYTEORDER_LITTLE_ENDIAN 1234
#define BYTEORDER_BIG_ENDIAN 4321

/* does anyone actually use  coff on a vax? */

#if TARGET_ARCH == TARGET_386	/* || TARGET_ARCH == TARGET_VAX */
#define TARGET_BYTEORDER BYTEORDER_LITTLE_ENDIAN
#else
#define TARGET_BYTEORDER BYTEORDER_BIG_ENDIAN
#endif /* TARGET_ARCH */

#if defined(vax) || defined(i386)
#define HOST_BYTEORDER BYTEORDER_LITTLE_ENDIAN
#else
#define HOST_BYTEORDER BYTEORDER_BIG_ENDIAN
#endif /* neither vax nor 386 */

/* This is how to increment a pointer to a symbol correctly regardless of
   host (or target!) compiler padding. */
#define SYM_INCR(p)   p = (SYMENT *) ((int)p + SYMESZ)

/* This is how to index into an arrayof symbols. Use instead of "ptr[index]" */
#define SYM_INDEX(ptr,index) (* (SYMENT *) ((int)ptr + index*SYMESZ))

/* Difference between two symbols. Ue instead of "ptrhi - ptrlow" */
#define SYM_DIFF(ptrhi,ptrlow) ((((int) ptrhi) - ( (int) ptrlow)) /SYMESZ)

#if TARGET == TARGET_SPARC
#include "sparc.h"
#define r_symbolnum r_index
#define RELOC_ADD_EXTRA(reloc) reloc->r_addend
#define relocation_info reloc_info_sparc

#undef SYM_INCR
	/* Warning: side effect of zeroing sparc SYMENT filler bytes */
#define SYM_INCR(p) { p = (SYMENT *) ((int)p + SYMESZ); ((short *)p)[-1] = 0; }

#endif /* TARGET == TARGET_SPARC */

#if TARGET_BYTEORDER == HOST_BYTEORDER
#define SWAPPING 0
int need_byteswap = 0;
/* 68K opcodes of interest */
#define BRA	0x60ff
#define JMP	0x4ef9
#define BSR	0x61ff
#define JSR	0x4eb9

#else				/* byte orders differ */
#define SWAPPING 1
int need_byteswap = 1;
/* 68K opcodes of interest */
#define BRA	0xff60
#define JMP	0xf94e
#define BSR	0xff61
#define JSR	0xb94e
#endif

#ifdef STANDALONE_CONV
int aout_is_intermediate = SWAPPING;
#if TARGET == TARGET_SPARC
#define SPARC_RELOC_DEFINED
#endif
#endif

/* the COFF format specification (for the pieces we need) */
#include <filehdr.h>
#include <scnhdr.h>
#include <reloc.h>
#include "syms.h"
#include "linenum.h"

#if TARGET != TARGET_APOLLO

#include <aouthdr.h>
#define NO_TEXT_ADDRESS_OFFSET

#else /* TARGET == TARGET_APOLLO */

#define M68K_NO_PCREL_RELOC

#ifdef apollo
#include <aouthdr.h>
#include <apollo/mir.h>
#else

#define APOLLOM68KMAGIC 0627
#define APOLLO_COFF_VERSION_NUMBER 1

typedef	struct aouthdr
{
  short	vformat;
  short	vstamp;
  long	tsize;
  long	dsize;
  long	bsize;
  long	entry;
  long	text_start;
  long	data_start;
  long	o_sri;
  long	o_inlib;
  long  vid[2];
} AOUTHDR;

#define STYP_INSTRUCTION	0x00200000
#define  _APTV ".aptv"
#define _MIR	".mir"

#define MIRHDRSZ 4
#define MIR_MAKER_SIZE 14
#define  maker_mir_kind 1

typedef struct
{
  long int nmirs;
} mirhdr;

typedef struct
{
  short kind;
  short mir_size;
  union
    {
      struct
        {
	  unsigned short int version;
	  long  int creation_time;
	  long  int _maker_name_offset;
        } mir_maker;
      struct
        {
	  long int module_name_offset;
        } mir_module_name;
    } info;
} mir;
#define maker_name_offset mir_maker._maker_name_offset

#endif /* notdef apollo */

#endif /* TARGET == TARGET_APOLLO */

#if TARGET == TARGET_SPARC
/* if we're making a cross sparc assembler, we
   want these values, not those in the host's
   headers. For a native sparc assembler these
   are also right. */

#undef SYMESZ
#undef AUXESZ
#undef LINESZ
#undef RELSZ
#define SYMESZ 20
#define AUXESZ 20
#define LINESZ 8
#define RELSZ 16
#endif

#if  TARGET == TARGET_SUN386 || TARGET == TARGET_SPARC
#define SUN_STYLE_COFF
#endif

#if  TARGET == TARGET_SUN386 || defined(DONT_HANDLE_SET_VECTORS)

#ifndef DONT_HANDLE_SET_VECTORS
#define DONT_HANDLE_SET_VECTORS
#endif

#define process_set_lists(sptr)
#define process_set_symbol(aptr,sptr,xptr,type,an)
#define build_coff_vector_sections()
#define build_coff_vector_relocs()

#else

static void process_set_lists	    ();
static void process_set_symbol	    ();
static void build_coff_vector_relocs();
static void build_coff_vector_sections ();

#endif

#ifdef n_leading_zero
#define syment_pun syment
#else
/* A pun for struct syment. */
struct syment_pun
{
  union
    {
      unsigned char _n_name[2];
    } _n;
  unsigned short n_dbx_desc;
};

#define n_leading_zero	  _n._n_name[0]
#ifdef SUN_STYLE_COFF
#define n_dbx_type _n._n_name[1]
#endif /* SUN_STYLE_COFF */

#endif

/* conflicts with nlist declaration in a.out.h */
#undef n_name

/* global a.out data structures */
struct exec	       *aout_hdr;      /* the a.out file header */
char		       *aout_text;     /* the a.out text section */
char		       *aout_data;     /* the a.out data section */
struct relocation_info *aout_trelocs;  /* array of a.out text reloc entries */
long			aout_ntrelocs; /* number of a.out text reloc entries */
struct relocation_info *aout_drelocs;  /* array of a.out data reloc entries */
long			aout_ndrelocs; /* number of a.out data reloc entries */
struct nlist	       *aout_symbols;  /* array of a.out symbol entries */
long			aout_nsymbols; /* number of a.out symbols */
char		       *aout_strings;  /* the a.out string table */

/* macro to return the string for an a.out symbol */
#define AOUT_SYM_NAME(s)	(aout_strings + (s).n_un.n_strx)

/* global COFF data structures */
#define MAX_COFF_SECTIONS   20

/* define the appropriate COFF magic number here */

#ifndef I386MAGIC
#define I386MAGIC 0514
#endif

#if TARGET == TARGET_APOLLO
#define COFF_MAGIC_NUMBER   APOLLOM68KMAGIC
#else
#if TARGET_ARCH == TARGET_68K
#define COFF_MAGIC_NUMBER   MC68MAGIC
#endif
#endif

#if TARGET_ARCH == TARGET_386
#define COFF_MAGIC_NUMBER  I386MAGIC
#endif

#if TARGET_ARCH == TARGET_VAX
#define COFF_MAGIC_NUMBER  VAXROMAGIC
#endif

#if TARGET_ARCH == TARGET_NS32K
#define COFF_MAGIC_NUMBER  0	/* I don't have this number handy */
#endif

#if TARGET == TARGET_SPARC
#ifndef SPARCMAGIC /* ugggh - gotta get John's group to stop dragging their feet */
#define SPARCMAGIC MC68MAGIC
#endif
#define COFF_MAGIC_NUMBER  SPARCMAGIC
#endif

#if TARGET==TARGET_APOLLO || TARGET_ARCH==TARGET_386 
# define DIRECT32BIT_RELOCATION R_DIR32;
#else
# define DIRECT32BIT_RELOCATION R_RELLONG;
#endif

/* base text address for COFF files */
#if TARGET == TARGET_APOLLO
#define COFF_TXTADDR	0x8000
#else
				/* other 68k, 386, or sparc */
#define COFF_TXTADDR 0

#endif

/* memory segment size */
#if TARGET == TARGET_APOLLO
#define SEGMENT_SIZE	0x8000
#define APOLLO_BSS_MAPPING
#define GENERATE_AOUTHDR
#else
#if TARGET == TARGET_68K
#define SEGMENT_SIZE	0x10000
#endif
#endif

#define GUESS_SEGMENT_SIZE 0x20000
#ifndef SEGMENT_SIZE
#define SEGMENT_SIZE GUESS_SEGMENT_SIZE
#endif
FILHDR	 coff_hdr;			    /* the COFF file header */
AOUTHDR	 coff_aouthdr;		      	    /* the COFF optional header */

SCNHDR	 coff_scnhdrs[MAX_COFF_SECTIONS];   /* array of COFF section headers */
long 	next_scnptr = 0;
char	*coff_snames[MAX_COFF_SECTIONS];    /* names of COFF sections */
RELOC	*coff_relocs[MAX_COFF_SECTIONS];    /* array of COFF reloc entries */
SYMENT	*coff_symbols;			    /* array of COFF symbol entries */
char	*coff_strings;			    /* the COFF string table */
long	 coff_strings_len;		    /* current used length of table */
long	 coff_strings_size;		  /* current allocated size of table */
char	*coff_filename;			    /* the temp output filename */
FILE	*coff_file;			    /* the temp output stream */
long	 reloc_count;		       /* total count of relocation entries */

int value_adjust_table[MAX_COFF_SECTIONS + 2] = {0};
int * value_adjust = &value_adjust_table[2];

/* Length of optional header: */
#if defined(GENERATE_AOUTHDR)
int aouthdr_length =  sizeof (coff_aouthdr);
#else
int aouthdr_length =  0;
#endif

/* a.out-to-COFF symbol index map */
long	   *symbol_map;

/* COFF section numbers */
int	    text_snum;
int	    data_snum;
int	    bss_snum;
#if TARGET == TARGET_APOLLO
int	    aptv_snum;
int	    mir_snum;
#endif

/* program command options flags */
#if TARGET == TARGET_APOLLO
int         writable_text = 1;        /* put code in a writable data section */
int	    use_aptv = 0;	      /* generate Apollo transfer vector */
#else
int	    writable_text = 0;	      /* put code in a normal text section */
#endif

/* macro to align a COFF file pointer or size value on a 4 byte boundary */
#define scn_align(a)	(((long)(a) + 3) & ~3)

/* macro to align a COFF file pointer or size value on a segment boundary */
#define seg_align(a)	(((long)(a) + SEGMENT_SIZE - 1) & ~(SEGMENT_SIZE - 1))

struct type_map {
  int sclass;
  int scnum;
  char sname;
  char line_change;
  short cofftype;
};
struct type_map type_map[256];
unsigned int  section_map[] = { -2, N_TEXT, N_DATA,  0 /* N_BSS */,0 };
static void add_type();
static void init_type_map();

/* COFF storage class */
#define CLASS_OF(type) type_map[ (unsigned) (type) ].sclass

/* COFF section number */
#define SCNUM_OF(type) type_map[ (unsigned) (type) ].scnum

/* One char name to use for unnamed stab symbol (for systems where coff
   loader would barf at zero length names) */
#define SNAME_OF(type) type_map[ (unsigned) (type) ].sname

/* coff type to set on systems where coff type and dbx type are not the
   same field of symbol structure */
#define COFFTYPE_OF(type) type_map[ (unsigned) (type) ].cofftype

/* Boolean saying this symbol may affect the scope in which line numbers are
   evaluated by the debugger. Used to determine partitioning of this object 
   file's line numbers. True for N_SOL, N_BINCL, N_EINCL, and N_FUN. */
#define LINE_SCOPE_CHANGE(type) type_map[ (unsigned) (type) ].line_change

#ifdef APOLLO_BSS_MAPPING

#define STATIC_BSS(a) (a->a_bss)
#define BSS_SECTION 2

#else

#define STATIC_BSS(a) (0)
#define BSS_SECTION 3

#endif

#define DATA_SIZE(a) (scn_align(a->a_data + STATIC_BSS(a)))
#define BSS_SIZE(a)  (scn_align((a->a_data + a->a_bss) - DATA_SIZE(a)))

int bss_reloc_flag;

#if TARGET == TARGET_SUN386 || TARGET == TARGET_SPARC

#define SET_PLAIN_TYPE(sptr , type)  sptr->n_type   = COFFTYPE_OF(type);
#define PLAINTYPE_OF(type) COFFTYPE_OF(type)

#else

#define SET_PLAIN_TYPE(sptr , type)  sptr->n_type   = (type <<8)
#define PLAINTYPE_OF(type) (type <<8)

#endif

#ifndef N_SETA
#define N_SETA  0x14            /* Absolute set element symbol */
#endif                          /* This is input to LD, in a .o file.  */

#ifndef N_SETT
#define N_SETT  0x16            /* Text set element symbol */
#endif                          /* This is input to LD, in a .o file.  */

#ifndef N_SETD
#define N_SETD  0x18            /* Data set element symbol */
#endif                          /* This is input to LD, in a .o file.  */

#ifndef N_SETB
#define N_SETB  0x1A            /* Bss set element symbol */
#endif                          /* This is input to LD, in a .o file.  */

/* Macros dealing with the set element symbols defined in a.out.h */
#define SET_ELEMENT_P(x)        ((x)>=N_SETA&&(x)<=(N_SETB|N_EXT))
#define TYPE_OF_SET_ELEMENT(x)  ((x)-N_SETA+N_ABSOLUTE)

#ifndef N_SETV
#define N_SETV  0x1C            /* Pointer to set vector in data area.  */
#endif                          /* This is output from LD.  */


/*
 * Currently Defined Coff Symbol building passes:
 */

/* In order for set symbol collection to work, STAB_PASS *MUST* come before
 * the SECTIONS_PASS, unless you have another pass that will call 
 * process_set_symbol() for each set symbol, and process_set_lists() before
 * SECTIONS_PASS.
 */

#define SECTIONS_PASS()							\
  for (sn = 0; sn < coff_hdr.f_nscns; ++sn)				\
    {									\
   /* set up symbol map - sections symbols are at end of map */		\
      symbol_map[aout_nsymbols + sn] = SYM_DIFF(sptr, coff_symbols);	\
      add_section_symbol(&sptr,sn);					\
    }									\
    process_set_lists(&sptr);

#define STAB_PASS()						\
	for (an = 0; an < aout_nsymbols; ++an)			\
	  {							\
	    aptr = &aout_symbols[an];				\
	    type = (unsigned char) aptr->n_type;		\
	    if (type &N_STAB)					\
	      add_debug_type(&aptr,&sptr,&xptr,type,an);	\
	    else if (SET_ELEMENT_P(type))			\
	      process_set_symbol (&aptr,&sptr,&xptr,type,an);	\
	  }							\

/* This is what sun separates stabs from other symbols with on the 386i: */

#if TARGET == TARGET_APOLLO
#define END_DEBUG_PASS()
#else /* not TARGET_APOLLO */
#define END_DEBUG_PASS() add_anonymous(&sptr,"-lg",T_NULL, 0, -1, C_EXT, 0)
#endif /* not TARGET_APOLLO */

#define REMAINING_SYMBOLS_PASS()				\
	for (an = 0; an < aout_nsymbols; ++an)			\
	  {							\
	    aptr = &aout_symbols[an];				\
	    type = (unsigned char) aptr->n_type;		\
	    if ((type > N_UNDF) && type <= (N_BSS|N_EXT))	\
	      add_plain_symbol (&aptr,&sptr,&xptr,type,an);	\
	  }

#define DISTINCT_PASS(condition,func)				\
	for (an = 0; an < aout_nsymbols; ++an)			\
	  {							\
	    aptr = &aout_symbols[an];				\
	    type = (unsigned char) aptr->n_type;		\
	    if (condition)					\
	      func (&aptr,&sptr,&xptr,type,an);			\
	  }

/* The following are for apollo, but I have not tested them: */

#define STATICS_PASS()			\
  DISTINCT_PASS((type == N_DATA || type == N_BSS), add_plain_symbol)

#define DEFINED_GLOBALS_PASS()		\
  DISTINCT_PASS((type == (N_DATA | N_EXT)), add_plain_symbol)

#define UNDEFINED_GLOBALS_PASS()	\
  DISTINCT_PASS((type == N_EXT), add_plain_symbol)

#if TARGET == TARGET_APOLLO 
#   define FUNCTIONS_PASS()		\
       DISTINCT_PASS(((type & ~N_EXT) == N_TEXT), add_function_symbol)
#else /* not  TARGET == TARGET_APOLLO */
#   define FUNCTIONS_PASS()		\
       DISTINCT_PASS(((type & ~N_EXT) == N_TEXT), add_plain_symbol)
#endif /* not  TARGET == TARGET_APOLLO */

/* OUTPUT SYMBOL ORDERING: */

/* The order here for the sun 386 matches that of the native sun coff tools,
 * although the USE_TTI_SYMBOL_ORDER ordering also works ok.
 *
 * For other machines, I use USE_TTI_SYMBOL_ORDER. 
 *
 * The apollo ordering is based upon the code in the original version of
 * coff-convert.c, although I have no way to test it. I suspect, but have no
 * way to prove, that USE_TTI_SYMBOL_ORDER would also work here, and would be
 * cheaper.
 *
 */

#if TARGET == TARGET_APOLLO 
#define USE_APOLLO_SYMBOL_ORDER
#else
#if TARGET == TARGET_SUN386
#define USE_SUN_SYMBOL_ORDER
#else
#define USE_TTI_SYMBOL_ORDER
#endif
#endif

#ifdef USE_SUN_SYMBOL_ORDER
#define SYMBOL_PASS1() STAB_PASS()
#define SYMBOL_PASS2() SECTIONS_PASS()
#define SYMBOL_PASS3() END_DEBUG_PASS()
#define SYMBOL_PASS4() REMAINING_SYMBOLS_PASS()
#endif

#ifdef USE_TTI_SYMBOL_ORDER
#define SYMBOL_PASS1() STAB_PASS()
#define SYMBOL_PASS2() SECTIONS_PASS()
#define SYMBOL_PASS3() REMAINING_SYMBOLS_PASS()
#endif

#ifdef USE_APOLLO_SYMBOL_ORDER
#define SYMBOL_PASS1() STAB_PASS()
#define SYMBOL_PASS2() END_DEBUG_PASS()
#define SYMBOL_PASS3() FUNCTIONS_PASS()
#define SYMBOL_PASS4() SECTIONS_PASS()
#define SYMBOL_PASS5() STATICS_PASS()
#define SYMBOL_PASS6() DEFINED_GLOBALS_PASS()
#define SYMBOL_PASS7() UNDEFINED_GLOBALS_PASS()
#endif

/* internal function prototypes */
static void open_files		    ();
static void close_files		    ();
static void setup_aout_data	    ();
static void build_coff_hdr	    ();
static void build_coff_aouthdr	    ();
static void build_coff_scnhdrs	    ();
static void build_coff_text_section ();
static void build_coff_data_section ();

#if TARGET == TARGET_APOLLO
static void build_coff_aptv_section ();
static void build_coff_mir_section  ();
static void build_coff_aptv_relocs  ();
#endif

static void reloc_section_data	    ();
static void build_coff_text_relocs  ();
static void build_coff_data_relocs  ();
static void build_coff_symbols	    ();
static void write_coff_symbols	    ();
static long map_reloc_symbol	    ();
static long add_coff_string	    ();
static void write_coff_strings	    ();
static void write_coff_hdr	    ();
static void write_coff_aouthdr	    ();
static void write_coff_scnhdrs	    ();

#ifndef __GNUC__
#define inline
#endif

/* support for set vectors */
struct set_info {
  struct nlist *set_root;	/* the root of this set */
  struct nlist *set_tail;	/* the tail of this set */
  int set_count;		/* number of elements in this set */

  char *set_name;		/* symbol name of this set - changed later ..*/
  int set_type;			/* the type of this set */
  int set_section;		/* section this set will go in */
  struct reloc *relocs;
  unsigned long *set_data;
  struct set_info* next_set;	/* the next set */
};
struct set_info *setlist_head = 0;
struct set_info *setlist_tail = 0;


/*
 * Support for text line numbers. GDB accepts either N_SLINE stabs or (with
 * support in the appropriate dep.c) a line number table, but emitting N_SLINE
 * stabs is significantly more costly in terms of file space.
 */

struct set_info line_info;

/* used for tracking first line entry */
static struct nlist head_sym;

/* used for first line count entry */
static struct nlist line_sym; 

/* points to the symbol that will next be used as a line number group header.*/
static struct nlist * line_scope_change_sym = &line_sym;

/* link SYM into a chain of line number symbols */

static void inline
link_line_nums(sym)
     struct nlist *sym;
{
  line_info.set_count++;
  sym->n_un.n_next = 0;
  if (line_info.set_root == 0)
    line_info.set_root = sym;
  if (line_info.set_tail != 0)
    line_info.set_tail->n_un.n_next = sym;
  line_info.set_tail = sym;
}

/*
 * Terminate line chain.
 */
void
terminate_line_list()
{
  if (line_info.set_tail != 0)
    line_info.set_tail->n_un.n_next = 0;
}

/* Fill array of line number structures located at LINE with
 * line number/address pairs from the global chain of line number symbols.
 */
void
fill_linebuf(line)
     LINENO *line;
{
  struct nlist *info = line_info.set_root;
  char * lp = (char *) line;
  unsigned int textsection_addr = coff_scnhdrs[text_snum - 1].s_vaddr;
  int group_line_count = 0;
  int header_symnum = 0;
  int next_header_symnum = 0;

  while (info)
    {
      line = (LINENO *) lp;
      line->l_lnno = info->n_desc;	/* 0 if paddr is really sym index... */
      line->l_addr.l_paddr = info->n_value;

      if (line->l_lnno)
	group_line_count++;      /* increment line count for group */
      else 
	{
	  header_symnum = next_header_symnum;
	  next_header_symnum = line->l_addr.l_paddr;
	  if (group_line_count)
	    {			/*  store line count for group */
	      struct syment * sym = &SYM_INDEX(coff_symbols,header_symnum);
	      sym->n_value = group_line_count;
	      group_line_count = 0;
	    }
	}

      /* Unless sun 386, make line number addresses match text
       * addresses. Sun's loader does not use the standard method for adjusting
       * line number address values as it does for adjusting symbol values
       *    WHY, WHY, WHY did sun break this?	   (!^@$&^*@$&!!?)
       *
       * 6/24/90: This should no longer actually matter now that we set the
       * text address to 0 for the sun 386i.
       */
#if TARGET != TARGET_SUN386
      if (line->l_lnno)
	line->l_addr.l_paddr += textsection_addr;
#endif
      if (need_byteswap)
	{
	  md_number_to_chars (&line->l_addr.l_paddr, line->l_addr.l_paddr,
			     sizeof(line->l_addr.l_paddr));
	  md_number_to_chars (&line->l_lnno, line->l_lnno,
			     sizeof(line->l_lnno));
	}

      /* compress line table */
      if (line->l_lnno == 0 && info != line_info.set_root)
	line_info.set_count--;
      else
	lp += LINESZ;

      info = info->n_un.n_next;
    }

  if (group_line_count)		/* store line count for final group */
    {
      struct syment * sym = &SYM_INDEX(coff_symbols, next_header_symnum);
      sym->n_value = group_line_count;
      group_line_count = 0;
    }
}

/* Output coff line number table. */

void
write_coff_line_numbers()
{
  /* assumes output file pointer is already positioned correctly */
  if (line_info.set_count)
    {

      LINENO *linebuf = (LINENO *) alloca(LINESZ * (line_info.set_count));
      int lnnoptr = ftell (coff_file);

      fill_linebuf(linebuf);
      coff_scnhdrs[text_snum-1].s_nlnno	  = line_info.set_count;

      /* set line info in COFF text header */
      coff_scnhdrs[text_snum-1].s_lnnoptr = lnnoptr;

      coff_hdr.f_flags	&= ~F_LNNO;
      fwrite ((char*) linebuf, LINESZ, line_info.set_count, coff_file);
    }
}

/*
 * Main entry point to convert an a.out memory image pointed to by
 * the char pointer located at AOUT_FILE_DATA to a coff memory image.
 * the pointer stored at AOUT_FILE_DATA is changed to refer to the new
 * coff image. The length of the output file is stored at the location
 * pointed to by AOUT_FILE_SIZE. After conversion, the new object file
 * size is stored there by close_files().
 */
int
convert_to_coff(aout_file_data, aout_file_size)
  char  **aout_file_data;
  long	*aout_file_size;
{
  int i;

  /* init type map */
  init_type_map();
  /* open the temporary files */
  open_files ();

  /* set up a.out structure info */
  setup_aout_data (*aout_file_data);

  /* initialize COFF headers */
  build_coff_hdr ();
  build_coff_aouthdr ();
  build_coff_scnhdrs ();

  /* build COFF symbol table in core */
  build_coff_symbols ();

  /* now that we know how many sections there are, we can 
     adjust the section pointers to their correct values. */

  for (i = 0; i<coff_hdr.f_nscns;i++)
    if (coff_scnhdrs[i].s_scnptr)
      coff_scnhdrs[i].s_scnptr  += sizeof (coff_scnhdrs[0]) * coff_hdr.f_nscns;

  /* build the section data in output file */
  build_coff_text_section ();
  build_coff_data_section ();

#if TARGET == TARGET_APOLLO
  if (use_aptv)
    build_coff_aptv_section ();
  build_coff_mir_section ();
#endif

  build_coff_vector_sections ();

  /* build the relocation data in output file */
  build_coff_text_relocs ();
  build_coff_data_relocs ();
#if TARGET == TARGET_APOLLO
  if (use_aptv)
    build_coff_aptv_relocs ();
#endif
  build_coff_vector_relocs();

  /* write out line numbers */
  write_coff_line_numbers();
  /* write out symbols and strings */
  write_coff_symbols ();
  write_coff_strings ();

  /* finish off the headers */
  write_coff_hdr ();
  if (aouthdr_length > 0)
    write_coff_aouthdr ();

  write_coff_scnhdrs ();

  /* finish everything */
  close_files (aout_file_data, aout_file_size);
}

/*
 *  open temporary output file
 */
static void
open_files()
{
  if ((coff_filename = tempnam (NULL, "ATOC")) == NULL)
    as_fatal ("can't construct temporary filename");

  coff_file = fopen (coff_filename, "w+");
  if (coff_file == NULL)
    as_fatal ("cannot create temporary file %s", coff_filename);
}

/*
 * Read final temp file into memory, replacing original AOUT_FILE_DATA
 * with new coff file data, adjusting AOUT_FILE_SIZE to the new file size.
 */
static void
close_files(aout_file_data,aout_file_size)
  char	**aout_file_data;
  long	 *aout_file_size;
{
  if (coff_file != NULL)
    {
      fseek (coff_file, 0, 2); /* position to end-of-file to get size */
      *aout_file_size = ftell (coff_file);
      rewind (coff_file);

      *aout_file_data = xmalloc (*aout_file_size);

      if (fread (*aout_file_data, *aout_file_size, 1, coff_file) == 0)
	as_fatal ("cannot read converted data from temp file");

      fclose (coff_file);
      unlink (coff_filename);
    }
}

/*
 * setup_aout_data - sets references to a.out data structures stored in
 * the area AOUT_FILE_DATA points to.
 */
static void
setup_aout_data(aout_file_data)
  char *aout_file_data;
{

  aout_hdr = (struct exec*) aout_file_data;

  {
    int bad_magic = (aout_hdr->a_magic&0xffff) != (OMAGIC &0xffff);
  if (bad_magic)
    as_fatal ("bad magic number %o, expected %o", aout_hdr->a_magic, OMAGIC);
  }
  if (aout_hdr->a_text)
    aout_text = aout_file_data + N_TXTOFF (*aout_hdr);

  if (aout_hdr->a_data)
    aout_data = aout_file_data + N_DATOFF (*aout_hdr);

  if (aout_hdr->a_trsize)
    {
      aout_trelocs =
	(struct relocation_info*) (aout_file_data + N_TRELOFF (*aout_hdr));
      aout_ntrelocs = aout_hdr->a_trsize / sizeof (struct relocation_info);
    }

  if (aout_hdr->a_drsize)
    {
      aout_drelocs =
	(struct relocation_info*) (aout_file_data + N_DRELOFF (*aout_hdr));
      aout_ndrelocs = aout_hdr->a_drsize / sizeof (struct relocation_info);
    }

  if (aout_hdr->a_syms)
    {
      aout_symbols = (struct nlist*) (aout_file_data + N_SYMOFF (*aout_hdr));
      aout_nsymbols = aout_hdr->a_syms / sizeof (struct nlist);
      aout_strings = aout_file_data + N_STROFF (*aout_hdr);
    }
}

/*
 *  build_coff_hdr - initializes as much of the coff file header as possible
 */
static void
build_coff_hdr()
{
  coff_hdr.f_magic  = COFF_MAGIC_NUMBER;
#if TARGET == TARGET_APOLLO
  coff_hdr.f_nscns  = use_aptv ? 5 : 4;
#else
  coff_hdr.f_nscns  = 3;
#endif
  coff_hdr.f_timdat = time (NULL);
  coff_hdr.f_symptr = 0; /* to be filled in later */
  coff_hdr.f_nsyms  = 0; /* to be filled in later */
  coff_hdr.f_opthdr = aouthdr_length;
  coff_hdr.f_flags  = F_LNNO | F_LSYMS;
}

/*
 * Write coff filehdr to output file
 */
static void
write_coff_hdr()
{
  FILHDR *newhdr = (FILHDR *) alloca (FILHSZ);

  if (need_byteswap)
    md_filhdr_to_chars(&coff_hdr, newhdr);
  else
    newhdr = &coff_hdr;

  fseek (coff_file, 0, 0);
  fwrite ((char*) newhdr, FILHSZ, 1, coff_file);
}


/*
 * Initializes the coff optional header
 */
static void
build_coff_aouthdr()
{
#ifdef NO_TEXT_ADDRESS_OFFSET
  coff_aouthdr.tsize   =  aout_hdr->a_text;
  coff_aouthdr.tsize   = scn_align (coff_aouthdr.tsize);
  coff_aouthdr.data_start =  (COFF_TXTADDR + coff_aouthdr.tsize);
#else
  coff_aouthdr.tsize   = sizeof (coff_hdr) + aouthdr_length
			 + sizeof (coff_scnhdrs[0]) * coff_hdr.f_nscns
			 + (writable_text ? 0 : aout_hdr->a_text);
  coff_aouthdr.tsize   = scn_align (coff_aouthdr.tsize);
  coff_aouthdr.data_start = seg_align (COFF_TXTADDR + coff_aouthdr.tsize);
#endif

  coff_aouthdr.dsize   = DATA_SIZE(aout_hdr)
			 + (writable_text ? aout_hdr->a_text : 0);
  coff_aouthdr.dsize   = scn_align (coff_aouthdr.dsize);

  coff_aouthdr.bsize   = 0; /* figured out later by looking at symbols */
  coff_aouthdr.entry   = 0;
  coff_aouthdr.text_start = COFF_TXTADDR;

  coff_aouthdr.vstamp  = 0;

#if TARGET == TARGET_APOLLO

  coff_aouthdr.vformat = APOLLO_COFF_VERSION_NUMBER;
  coff_aouthdr.o_sri   = 0;
  coff_aouthdr.o_inlib = 0;
  coff_aouthdr.vid[0]  = 0;
  coff_aouthdr.vid[1]  = 0;

#else /* not TARGET == TARGET_APOLLO */

  coff_aouthdr.magic   = 0407;

#endif /* not TARGET == TARGET_APOLLO */

}

/*
 *  Write coff optional header to output file.
 */
static void
write_coff_aouthdr()
{
  AOUTHDR *newhdr = (AOUTHDR *) alloca (sizeof (coff_aouthdr));
  if (need_byteswap)
    md_aouthdr_to_chars((char*)& coff_aouthdr, newhdr);
  else
    newhdr = & coff_aouthdr;

  fseek (coff_file, FILHSZ, 0);
  fwrite ((char*) newhdr, sizeof (coff_aouthdr), 1, coff_file);
}

/* Intitialize coff section NAME with section SN at addr ADDR of size SIZE
   having NRELOCS relocation entries and flags FLAGS, and *tentatively* at
   offset SCNPTR.

   If SCNPTR is non, zero, "next_scnptr" is set to the value of SCNPTR plus
   the section size.

   SCNPTR does not include the sizes of any of the section headers. The total
   size of all the section headers will be added later, when the total number
   of sections is known.

   */

build_section_header(name,sn,addr,size,nrelocs,flags,scnptr)
char *name;
{
  coff_snames[sn] = name;
  strncpy (coff_scnhdrs[sn].s_name, coff_snames[sn], SYMNMLEN);

  coff_scnhdrs[sn].s_scnptr  = scnptr;
  coff_scnhdrs[sn].s_paddr   = addr;
  coff_scnhdrs[sn].s_vaddr   = addr;
  coff_scnhdrs[sn].s_size    = size;

  coff_scnhdrs[sn].s_nreloc  = nrelocs;
  reloc_count += nrelocs;
  coff_scnhdrs[sn].s_relptr  = 0;
  coff_scnhdrs[sn].s_lnnoptr = 0;
  coff_scnhdrs[sn].s_nlnno   = 0;
  coff_scnhdrs[sn].s_flags   = flags;

  if (scnptr != 0)
    next_scnptr = coff_scnhdrs[sn].s_scnptr + coff_scnhdrs[sn].s_size;
}

/*
 *  Initialize coff section headers
 */
static void
build_coff_scnhdrs()
{
  long sn = 0;

  int textaddr;

  next_scnptr  = sizeof (coff_hdr) + aouthdr_length;
				/* will increment later when we know number
				 of sections */
  /* init header for text section */

#if defined(NO_TEXT_ADDRESS_OFFSET)
  textaddr = writable_text ? coff_aouthdr.data_start : COFF_TXTADDR;
#else
  textaddr = writable_text ? coff_aouthdr.data_start
    			   : COFF_TXTADDR + coff_scnhdrs[sn].s_scnptr;
#endif
  build_section_header(writable_text ? ".wtext" : _TEXT,
		       sn,
		       textaddr,
		       scn_align (aout_hdr->a_text),
		       aout_ntrelocs,
		       writable_text ? STYP_REG | STYP_DATA
#if TARGET == TARGET_APOLLO
				     : STYP_REG | STYP_TEXT | STYP_INSTRUCTION,
#else
				     : STYP_REG | STYP_TEXT,
#endif
		       next_scnptr);
  text_snum = ++sn;
  value_adjust[text_snum] =
	   coff_scnhdrs[text_snum - 1].s_vaddr;

  /* init header for data section */
  build_section_header(_DATA,
		       sn,
		       writable_text ? coff_scnhdrs[sn - 1].s_paddr
				       + coff_scnhdrs[sn - 1].s_size
				     : coff_aouthdr.data_start,
		       DATA_SIZE(aout_hdr),
		       aout_ndrelocs,
		       STYP_REG | STYP_DATA,
		       next_scnptr);
  data_snum = ++sn;
  value_adjust[data_snum] =
	   coff_scnhdrs[data_snum - 1].s_vaddr - N_DATADDR (*aout_hdr);


#if TARGET == TARGET_APOLLO
  /* init header for Apollo transfer vector section, if used */
  if (use_aptv)
    {
      build_section_header(_APTV,
			   sn,
			   coff_scnhdrs[sn - 1].s_paddr
			   + coff_scnhdrs[sn - 1].s_size,
			   0,
			   0,
			   STYP_REG | STYP_DATA,
			   next_scnptr);
      aptv_snum = ++sn;
    }
#endif

  /* init header for bss section */

  build_section_header(_BSS,
		       sn,
		       coff_scnhdrs[sn - 1].s_paddr
		        + coff_scnhdrs[sn - 1].s_size,
		       BSS_SIZE(aout_hdr),
		       0,
		       STYP_REG | STYP_BSS,
		       0);
  bss_snum = ++sn;
  value_adjust[bss_snum] =
	   coff_scnhdrs[bss_snum - 1].s_vaddr - N_BSSADDR (*aout_hdr);

#if TARGET != TARGET_APOLLO
  /* you might set this differently if you wanted to join static bss
   * into data, for some reason ....
   */
  bss_reloc_flag = BSS_SIZE(aout_hdr) != 0;
#endif

#if TARGET == TARGET_APOLLO

  /* init header for Apollo .mir section */
  build_section_header(_MIR, sn, 0, 0, 0, STYP_INFO, next_scnptr);
  mir_snum = ++sn;
#endif

  coff_hdr.f_nscns = sn;
}

/*
 * Write COFF section headers to the output file
 */
static void
write_coff_scnhdrs()
{
  SCNHDR *newhdr = (SCNHDR *) alloca (SCNHSZ* coff_hdr.f_nscns);
  int i;

  fseek (coff_file, FILHSZ + aouthdr_length, 0);

  for (i = 0; i<coff_hdr.f_nscns;i++)
    {
      if (need_byteswap)
	md_scnhdr_to_chars((char*) &coff_scnhdrs[i], newhdr);
      else
	newhdr = &coff_scnhdrs[i];
      fwrite ((char*) newhdr, SCNHSZ, 1, coff_file);
    }
}

/*
 * map_reloc_symbol - converts an a.out symbol index to a COFF symbol index
 * based upon values in *RPTR.
 */
static inline long
map_reloc_symbol(rptr)
     struct relocation_info *rptr;
{
  if (rptr->r_extern)
    return symbol_map[rptr->r_symbolnum];
  else
    switch (rptr->r_symbolnum)
      {
      case N_TEXT:
	return symbol_map[aout_nsymbols + text_snum - 1];
      case N_BSS:
	if (bss_reloc_flag)
	  return symbol_map[aout_nsymbols + bss_snum - 1];
      case N_DATA:
	return symbol_map[aout_nsymbols + data_snum - 1];
      default:
	as_fatal ("unknown a.out symbol reference %d",rptr->r_symbolnum);
      }
}


#if TARGET != TARGET_SPARC

/*
 * reloc_section_data - fixes up section DATA according to reloc info at RPTR
 * If IS_TEXT is true, change 68020 pc relative references to more (at least
 * on 68020, according to the motorola manual - dont know about the other 68K
 * processors) efficient non-pc relative references.
 *
 * This routine is incomplete: only 386 and 68k processors are supported here.
 * It will be nice if someone wants to provide versions for ns32k or vax.
 * For sparc, no section data is adjusted and this routine is never called.
 */
static void inline
reloc_section_data (rptr, data, is_text)
     struct relocation_info *rptr;    /* pointer to a.out reloc record */
     unsigned char	    *data;    /* pointer to section data */
     int		    is_text;  /* TRUE if this is the text section */
{
			/*   object data being relocated */
  unsigned char * scn_data = data + rptr->r_address;
				/* value of symbol referred to */
  long value = SYM_INDEX(coff_symbols,map_reloc_symbol (rptr)).n_value;
  char tmp_char;
  short tmp_short;
  long tmp_long;

  if (rptr->r_pcrel) /* adjust section data */
    {
#ifdef M68K_NO_PCREL_RELOC
	  switch (rptr->r_length)
	    {
	    case 0:
	      md_number_to_chars (((char*) (scn_data)), value, 1);
	      break;
	    case 1:
	      md_number_to_chars (((short*) (scn_data)), value, 2);
	      break;
	    case 2:
	      md_number_to_chars (((long*) (scn_data)), value, 4);
	      break;
	    }
	  /* if this is the text section,
	     change pc-relative calls to absolute */
	  if (is_text)
	    {
	      /* get pointer to instruction opcode */
	      unsigned short *optr = (unsigned short*) (scn_data - 2);

	      if (*optr == BRA) /* change to JMP */
		*optr = JMP;
	      else if (*optr == BSR) /* change to JSR */
		*optr = JSR;
	      else		/* can't handle it */
		as_fatal ("unknown PC-relative reference");
	    }
#else /* not M68K_NO_PCREL_RELOC */
	  int adjust = coff_scnhdrs[text_snum - 1].s_vaddr;
	  if (adjust)
	    {
	      switch (rptr->r_length)
		{
		case 0:
		  md_number_to_chars (&tmp_char, *((char *)scn_data), 1);
		  md_number_to_chars (scn_data, tmp_char - adjust, 1);
		  break;
		case 1:
		  md_number_to_chars (&tmp_short,*((short *)scn_data), 2);
		  md_number_to_chars (scn_data, tmp_short - adjust, 2);
		  break;
		case 2:
		  md_number_to_chars (&tmp_long,*((long *)scn_data), 4);
		  md_number_to_chars (scn_data, tmp_long - adjust, 4);
		  break;
		}
	    }
#endif
    }
  else /* change offset to absolute */
    {
      /* data section-relative references are actually relative to the base of
	 the text section, so compensate the base value if so */
      if (!rptr->r_extern)
	{
	  switch (rptr->r_symbolnum)
	    {
	    case N_BSS:
	      if (bss_reloc_flag)
		{
		  value -= N_BSSADDR (*aout_hdr);
		  break;
		}
	    case N_DATA:
	      value -= N_DATADDR (*aout_hdr);
	      break;
	    }
	}
      if (value)
	{
	  switch (rptr->r_length)
	    {
	    case 0:
	      tmp_char = *((char*) (scn_data));
	      md_number_to_chars (scn_data, tmp_char + value, 1);
	      break;
	    case 1:
	      md_number_to_chars (&tmp_short,*((short *)scn_data), 2);
	      md_number_to_chars (scn_data, tmp_short + value, 2);
	      break;
	    case 2:
	      md_number_to_chars (&tmp_long,*((long *)scn_data), 4);
	      md_number_to_chars (scn_data, tmp_long + value, 4);
	      break;
	    }
	}
    }
}

#endif /* TARGET != TARGET_SPARC */

/*
 * build .text section data in output file.
 */
static void
build_coff_text_section()
{
  long	pad_size;

  /* position output file pointer */
  fseek (coff_file, coff_scnhdrs[text_snum - 1].s_scnptr, 0);

  if (aout_hdr->a_text)
    {
#if TARGET != TARGET_SPARC
      /* modify section data to convert to COFF relocation mechanism */
      if (aout_hdr->a_trsize)
	{
	  long rn;

	  for (rn = 0; rn < aout_ntrelocs; ++rn)
	    reloc_section_data (&aout_trelocs[rn], aout_text, 1);
	}
#endif
      /* write text data out to COFF file */
      fwrite (aout_text, aout_hdr->a_text, 1, coff_file);

      /* take care of any alignment padding */
      pad_size = coff_scnhdrs[text_snum - 1].s_size - aout_hdr->a_text;
      if (pad_size > 0)
	{
	  char *padding;

	  /* allocate a zeroed chunk of memory of the right size */
	  if ((padding = xmalloc (pad_size)) == NULL)
	    as_fatal ("out of memory for text padding");

	  bzero (padding, pad_size);
	  fwrite (padding, pad_size, 1, coff_file);
	  free (padding);
	}
    }
}

/*
 *  Build .data section in output file.
 */
static void

build_coff_data_section()
{
  long	bss_size;

  /* position output file pointer */
  fseek (coff_file, coff_scnhdrs[data_snum - 1].s_scnptr, 0);

  if (aout_hdr->a_data)
    {
#if TARGET != TARGET_SPARC
      /* modify section data to convert to COFF relocation mechanism */
      if (aout_hdr->a_drsize)
	{
	  long rn;

	  for (rn = 0; rn < aout_ndrelocs; ++rn)
	    reloc_section_data (&aout_drelocs[rn], aout_data, 0);
	}
#endif
      /* write data out to COFF file */
      fwrite (aout_data, aout_hdr->a_data, 1, coff_file);
    }

  /* take care of alignment padding or
     static bss data added to the end of the COFF data section */
  if ((bss_size = coff_scnhdrs[data_snum - 1].s_size - aout_hdr->a_data) > 0)
    {
      char *padding;

      /* allocate a zeroed chunk of memory of the right size */
      if ((padding = xmalloc (bss_size)) == NULL)
	as_fatal ("out of memory for bss data");

      bzero (padding, bss_size);
      fwrite (padding, bss_size, 1, coff_file);
      free (padding);
    }
}

#if TARGET == TARGET_APOLLO
/*
 *  build_coff_aptv_section
 *  - builds Apollo transfer vector section in output file
 */
static void build_coff_aptv_section()
{
  /* not yet implemented, use writable-text instead */
}

/*
 *  build_coff_mir_section
 *  - builds Apollo transfer vector section in output file
 */
static void build_coff_mir_section()
{
  mirhdr mir_hdr;
  mir mir_rec;
  int sn = mir_snum -1;
  mir_hdr.nmirs = 1;
  fwrite ((char *) &mir_hdr, MIRHDRSZ, 1, coff_file);
  coff_scnhdrs[sn].s_size = MIRHDRSZ;

  mir_rec.kind = maker_mir_kind;
  mir_rec.mir_size = MIR_MAKER_SIZE;
  mir_rec.info.mir_maker.version = 0;
  mir_rec.info.mir_maker.creation_time = 0;
  mir_rec.info.maker_name_offset = add_coff_string (version_string);

  fwrite ((char *) &mir_rec, mir_rec.mir_size, 1, coff_file);
  coff_scnhdrs[sn].s_size += MIR_MAKER_SIZE;

  next_scnptr  = coff_scnhdrs[sn].s_scnptr + coff_scnhdrs[sn].s_size;
}
#endif

#define md_long(data) md_number_to_chars ((char *)&data, data, sizeof(data))
#define md_short(data) md_number_to_chars ((char *)&data, data, sizeof(data))

/*
 * Convert *RELOC to target byte order
 */
static void inline
md_reloc_to_chars(reloc)
     register struct reloc *reloc;
{
    /* this is easy */
  md_long (reloc->r_vaddr);
  md_long (reloc->r_symndx);
  md_short (reloc->r_type);
  /* now the fun stuff */
#ifdef	RELOC_ADD_EXTRA
  md_long (RELOC_ADD_EXTRA(reloc));
#endif

}
/* define ASCENDING_RELOCS if you have a broken loader that complains
   about relocation entries not being in ascending order. */

#ifdef ASCENDING_RELOCS
static int inline
compare_reloc_addresses(first, second)
     struct relocation_info *first;
     struct relocation_info *second;
{
  return second->r_address - first->r_address;
}
#endif

/*
 *  for the section COFF_SPTR points to, build coff relocation using the
 *  a.out relocation info at RPTR.
 */
static void
build_coff_relocs(coff_sptr,aout_rptr)
  SCNHDR *coff_sptr;
  struct relocation_info *aout_rptr;
{
  RELOC coff_reloc;
  long	rn;

  if (coff_sptr->s_nreloc)
    {
#ifdef ASCENDING_RELOCS
      qsort(&aout_rptr[0], coff_sptr->s_nreloc, sizeof(aout_rptr[0]),
	    compare_reloc_addresses);
#endif
      /* set file pointer to relocs in section header */
      coff_sptr->s_relptr = ftell (coff_file);

      /* generate all the reloc entries in ascending order */
#if defined(REGRESSION_TEST) || defined(RELOCS_ALREADY_ORDERED)
      for (rn = 0; rn < coff_sptr->s_nreloc; ++rn)
#else
      for (rn = coff_sptr->s_nreloc -1 ; rn >= 0 ; --rn)
#endif
	{
	  coff_reloc.r_vaddr  = coff_sptr->s_vaddr + aout_rptr[rn].r_address;
	  coff_reloc.r_symndx = map_reloc_symbol (&aout_rptr[rn]);

#if TARGET == TARGET_SPARC
	  coff_reloc.r_type = (unsigned short) aout_rptr[rn].r_type;
	  coff_reloc.r_addend = aout_rptr[rn].r_addend;
	  /* The AT&T loader wants the size added if common */
	  if (aout_rptr[rn].r_extern)
	    coff_reloc.r_addend += coff_symbols[coff_reloc.r_symndx].n_value;
#else /* not sparc */

#ifndef M68K_NO_PCREL_RELOC
	  if (aout_rptr[rn].r_pcrel)
	    coff_reloc.r_type	= R_PCRLONG; /* same for both m68k and 386 */
	  else
#endif
	    coff_reloc.r_type   = DIRECT32BIT_RELOCATION;

#endif /* not sparc */
	  if (need_byteswap)
	    md_reloc_to_chars((char*) &coff_reloc);
	  fwrite ((char*) &coff_reloc, RELSZ, 1, coff_file);
	}
    }
}

/*
 *  build_coff_text_relocs - builds .text relocation info in output file
 */
static void
build_coff_text_relocs()
{
  build_coff_relocs (&coff_scnhdrs[text_snum - 1], aout_trelocs);
}

/*
 *  build_coff_data_relocs - builds .data relocation info in output file
 */
static void
build_coff_data_relocs()
{
  build_coff_relocs (&coff_scnhdrs[data_snum - 1], aout_drelocs);
}

#if TARGET == TARGET_APOLLO
/*
 *  build_coff_aptv_relocs - builds .aptv relocation info in output file
 */
static void build_coff_aptv_relocs()
{
  /* not yet implemented */
}
#endif

/*
 * Add STRING to the in-core coff string table, returning its index value
 */
static long inline
add_coff_string(str)
char *str;
{
  int	strsize;
  char *st_ptr;

  strsize = strlen (str) + 1;

  /* if no room for string, expand the table */
  while (coff_strings_len + strsize > coff_strings_size)
    {
      /* if this is the first string, create the initial table */
      if (coff_strings == NULL)
	{
	  coff_strings_size = 8192;
	  coff_strings = xmalloc (coff_strings_size);
	  coff_strings_len = 4;
	}
      else /* expand the existing table */
	{
	  coff_strings_size *= 2;
	  coff_strings = xrealloc (coff_strings, coff_strings_size);
	}
    }

  st_ptr = coff_strings + coff_strings_len;
  memcpy (st_ptr, str, strsize);
  coff_strings_len += strsize;

  return st_ptr - coff_strings;
}

/*
 * Set the name of a coff symbol SYM to NAME,
 * putting it in the string table if required.
 */
static void inline
set_symbol_name(sym,name)
     SYMENT *sym; char *name;
{
  sym->n_leading_zero = 0;	/* notice we use the one byte define here */
  if (name[0] == 0)
    return;
  /* see if string needs to go in string table */
  if (strlen (name) > SYMNMLEN)
    {
      sym->_n._n_n._n_offset = add_coff_string (name);
    }
  else /* it fits in symbol entry */
    strncpy (sym->_n._n_name, name, SYMNMLEN);
}

#if TARGET == TARGET_APOLLO

static void inline
set_symbol(aptr, sptr)
     struct nlist *aptr;
     SYMENT *sptr;
{
  int type = (unsigned char)aptr->n_type;
  sptr->n_sclass = CLASS_OF(type);
  sptr->n_scnum = SCNUM_OF(type);
  sptr->n_type  = PLAINTYPE_OF(type);
  sptr->n_value = aptr->n_value + value_adjust[sptr->n_scnum];
#ifdef REGRESSION_TEST
  if (sptr->n_scnum == data_snum
      || sptr->n_scnum == bss_snum
      || (sptr->n_scnum == 0 && sptr->n_value != 0))
    sptr->n_type = T_INT;
  else
    sptr->n_type = T_NULL;
#endif
}

#else

static void inline
set_symbol(aptr, sptr)
     struct nlist *aptr;
     SYMENT *sptr;
{
	int type = (unsigned char)aptr->n_type;
	sptr->n_type  = PLAINTYPE_OF(type);
	sptr->n_sclass = CLASS_OF(type);
	sptr->n_scnum = SCNUM_OF(type);
	sptr->n_value = aptr->n_value + value_adjust[sptr->n_scnum];

#ifndef SUN_STYLE_COFF
	/* Type is set to 0 for N_EXT entries, because coff loaders never look at
	 * n_type, and (since the loader does not change it) there would otherwise
	 * be an inappropriate value when the debugger read the linked binary.
	 * With this type set to 0, the debugger gets back the N_EXT (for
	 * relocatable binaries) in dep.c, based upon the values of section number
	 * and storage class.
	 */
	if (sptr->n_scnum == 0)
	  sptr->n_type = 0;
#endif
}

#endif

#if TARGET == TARGET_APOLLO
static void inline
add_function_symbol(aptr, sptr, xptr_ignored, type, an)
     struct nlist **aptr;
     SYMENT **sptr;
     AUXENT **xptr_ignored;
     unsigned int type;
     int an;
{
  AUXENT *xptr;

  (*sptr)->_n._n_nptr[0] = 0; /* clears n_leading_zero,n_dbx_type,n_dbx_desc */
  set_symbol_name ((*sptr), AOUT_SYM_NAME (*(*aptr)));
  set_symbol(*aptr,*sptr);

  /* set up symbol map */
  symbol_map[an] = SYM_DIFF((*sptr),  coff_symbols);

  (*sptr)->n_type =  T_INT | (DT_FCN << N_BTSHFT);
  (*sptr)->n_numaux = 1;
  SYM_INCR((*sptr));
  xptr = (AUXENT*) (*sptr);
  bzero(xptr,AUXESZ);
  xptr->x_sym.x_tagndx = 0;
  xptr->x_sym.x_misc.x_fsize = 0; /* can this be figured out? */
  xptr->x_sym.x_fcnary.x_fcn.x_lnnoptr = 0;
  xptr->x_sym.x_fcnary.x_fcn.x_endndx  = (*sptr) - coff_symbols + 1;
  SYM_INCR((*sptr));
}
#endif /*  TARGET == TARGET_APOLLO */

static void inline
add_plain_symbol (aptr, sptr, xptr, type, an)
     struct nlist **aptr;
     SYMENT **sptr;
     AUXENT **xptr;
     unsigned int type;
     int an;
{
  (*sptr)->_n._n_nptr[0] = 0; /* clears n_leading_zero,n_dbx_type,n_dbx_desc */

  set_symbol_name ((*sptr), AOUT_SYM_NAME (*(*aptr)));
  set_symbol(*aptr,*sptr);

  /* set up symbol map */
  symbol_map[an] = SYM_DIFF((*sptr),  coff_symbols);

  if (type == N_EXT)
    {
      if ((*aptr)->n_value)
	  coff_aouthdr.bsize += (*aptr)->n_value;
    }
  (*sptr)->n_numaux = 0;
  SYM_INCR((*sptr));
}

static inline make_anonymous(sptr,name,type,valu,scnum,sclass,numaux)
     SYMENT **sptr;
     char *name;
     int type, valu, scnum, sclass, numaux;
{
  /* generate symbol entry for function */
/*  (*sptr)->n_zeroes = 0;*/
  (*sptr)->_n._n_nptr[0] = 0; /* clears n_leading_zero,n_dbx_type,n_dbx_desc */

  (*sptr)->n_type = type;
  (*sptr)->n_scnum = scnum;
  (*sptr)->n_sclass = sclass;
  (*sptr)->n_value = valu;
  (*sptr)->n_numaux = numaux;
}

static inline add_anonymous(sptr,name,type,valu,scnum,sclass,numaux)
     SYMENT **sptr;
     char *name;
     int type, valu, scnum, sclass, numaux;
{
  make_anonymous(sptr,name,type,valu,scnum,sclass,numaux);
  set_symbol_name ((*sptr),name);
  /* must set symbol map before call if wanted ...  */
  SYM_INCR((*sptr));
}

static void inline
add_section_symbol (sptr,sn)
     SYMENT **sptr;
     int sn;
{
  AUXENT *xptr;

  add_anonymous(sptr, coff_snames[sn], T_NULL, 
		coff_scnhdrs[sn].s_vaddr, sn+1, C_STAT, 1);		
  xptr = (AUXENT*) *sptr; 
  bzero(xptr,AUXESZ); 
  xptr->x_scn.x_scnlen = coff_scnhdrs[sn].s_size; 
  xptr->x_scn.x_nreloc = coff_scnhdrs[sn].s_nreloc; 
  SYM_INCR((*sptr));

}

#ifndef N_LINE_COUNT_TYPE
#define N_LINE_COUNT_TYPE (0xf1)
#endif

int have_name = 0;
AUXENT *file_aux;

static int have_line_header = 0;

/* aptr has symbol #, line,  pair in value/desc, forged for the header. */

/* 
  Create line count symbol. 
  
  The first time called, first create a symbol with an aux that the loader
  will adjust to point to to the portion of the coff line number table 
  that corresponds to this object file.
  We now give this symbol a desc value of N_FUN.  This tells gdb to process
  the line table entries for this file incrementally, rather than all at once. 

  When gdb sees an N_LINE_COUNT_TYPE symbol, it will process the number
  of lines specified by the symbol's value field.  This allows gdb to
  match line numbers with the correct file, if generated text was defined 
  in an included file. 

  I initially tried using the seemingly more intuitive approach of using
  multiple function auxents to provide the grouping, but I could not
  come up with a general way of doing this that was portable to all the
  coff systems I have access to, without losing compatibility with the
  native toolset, particularly on the sun386i.
 */

make_line_info(aptr,sptr,xptr,type,an)
  struct nlist **aptr;
  SYMENT **sptr;
  AUXENT **xptr;
  unsigned int type;
  int an;
{
  struct syment_pun *punptr = (struct syment_pun *) (*sptr);

  if (have_line_header == 0)
    {
      make_anonymous(sptr,"S", (DT_FCN <<N_BTSHFT) | T_INT, 0,1,C_STAT,0);
      punptr->n_dbx_desc = (unsigned short) N_FUN; 

#ifdef SUN_STYLE_COFF
      (*sptr)->n_dbx_type = (unsigned short) N_FUN;

#  if TARGET == TARGET_SUN386
      (*sptr)->n_scnum = -2;
      (*sptr)->_n._n_n._n_offset = add_coff_string ("\0177\0177.ln:F1");
#  else
      (*sptr)->_n._n_n._n_offset = add_coff_string ("_ln_ptr_");
#  endif
#else
      (*sptr)->n_type = ((unsigned short) N_FUN) << 8;
      (*sptr)->_n._n_name[0]='l';
      (*sptr)->_n._n_name[1]=0;
#endif
      head_sym.n_value = SYM_DIFF((*sptr),coff_symbols); 
      head_sym.n_desc = 0;
      link_line_nums(&head_sym);
      (*sptr)->n_numaux = 1;	/* symbol has one aux */

      /* now generate the line number auxent */

      SYM_INCR((*sptr));
      (*xptr) = (AUXENT*) (*sptr);
      bzero((*xptr), AUXESZ);
      SYM_INCR((*sptr));
#if TARGET != TARGET_SUN386
      (*xptr)->x_sym.x_fcnary.x_fcn.x_endndx = SYM_DIFF((*sptr),coff_symbols);
#endif

      have_line_header =1;
    }

  /* generate the line group header pointer: */

  /* We reuse the aout symbol that indicated a line number scope change as an
     internal line number header entry. This entry wont go into the line 
     number table, but serves as a back pointer to the symbol we will give
     a line count to (when we know what the count is).
     */
  line_scope_change_sym->n_value = SYM_DIFF((*sptr),coff_symbols) ;
  line_scope_change_sym->n_desc = 0;
  link_line_nums(line_scope_change_sym);

  /* generate the line group header: */

  /* For this special type, (except for sparc) use the non-sun method of 
     encoding the dbx type for both sun and non-sun machines. 

     dbx on the sun386 does not know about this type, and would spit out a 
     bunch of disconcerting "stab entry unrecognized" messages if this type 
     were encoded in the n_dbx_type field. dbx can't handle source lines from 
     include files anyway on the 386i, so it doesn't matter if dbx ignores
     this symbol, we just dont want it to complain. The important thing is 
     that gdb handles it properly.

     For machines that don't use SUN_STYLE_COFF, we HAVE to use the non-sun
     method, of encoding this symbol. Again, gdb will handle it.

     We do have SUN_STYLE_COFF on System V/SPARC, but we don't have dbx, so
     we're free to do it the way we'd prefer to on the other machines. */

  make_anonymous(sptr, "", N_LINE_COUNT_TYPE<<8, 0, N_DEBUG, C_TPDEF, 0);

#if defined(SUN_STYLE_COFF) && TARGET != TARGET_SUN386
  (*sptr)->_n._n_n._n_offset = add_coff_string ("");
  (*sptr)->_n._n_nptr[0] = 0; /* clears n_leading_zero,n_dbx_type,n_dbx_desc */
  (*sptr)->n_dbx_type = N_LINE_COUNT_TYPE;
#else
  strcpy((*sptr)->_n._n_name, "l.cnt");
#endif
  SYM_INCR((*sptr));
}

static void inline
add_debug_type(aptr,sptr,xptr,type,an)
  struct nlist **aptr;
  SYMENT **sptr;
  AUXENT **xptr;
  unsigned int type;
  int an;
{
  int snum;
  char sname ;
  struct syment_pun *punptr = (struct syment_pun *) (*sptr);

  if (type == (unsigned short) N_SLINE)
    {
      if (line_scope_change_sym == 0) 
	link_line_nums((*aptr));
      else
	{
	  /* A new line info header is generated whenever an N_SLINE occurs
	     after a symbol that indicated a change of line number scope 
	     */
	  make_line_info(aptr,sptr,xptr,type,an);
	  link_line_nums((*aptr));
	  line_scope_change_sym = 0;
	}

      return;
    }

  (*sptr)->n_zeroes = 0;
  (*sptr)->_n._n_n._n_offset = 0;

  set_symbol(*aptr,*sptr);
  sname = SNAME_OF(type);
  snum = (*sptr)->n_scnum;

  if (type == (unsigned int) N_SO
      && aptr[0][1].n_type != (unsigned int) N_SO)
    {				/* use this to fill in file auxent  */
      if (! have_name)
	{			/* have file name: replace old crud */
	  char *s = AOUT_SYM_NAME (*(*aptr));
	  char *rindex();
	  char *t = rindex(s,'/');
	  int n;

	  if (t)
	    s = t+1;
	  n = strlen(s);
	  if (n>FILNMLEN)
	    {
	      s = (s+n) - FILNMLEN;
	      n = FILNMLEN;
	    }
	  have_name ++;
	  strncpy (file_aux->x_file.x_fname,
		   s, FILNMLEN);
	}
    }

  /*
   * the only sparc I know of that uses COFF also uses Sun's COFF
   * extensions. before the sparc port is released, need to decide
   * on better ifdefs....
   */

#ifdef SUN_STYLE_COFF

  if (sname)
      (*sptr)->_n._n_n._n_offset = add_coff_string ("");
  else
    (*sptr)->_n._n_n._n_offset = add_coff_string (AOUT_SYM_NAME (*(*aptr)));

  /* Use punptr. Even though this field is explicitly named on the
   * target, it might not be named on the host.
   */
  punptr->n_dbx_desc = (*aptr)->n_desc;
  (*sptr)->n_dbx_type = type;
  (*sptr)->n_leading_zero = 0;

#else	/* Don't have Sun-style extensions to coff. use what's in plain coff */

  if (sname)			/* object provides context for address */
    {
      struct syment_pun *punptr = (struct syment_pun *) (*sptr);
      punptr->n_dbx_desc = (*aptr)->n_desc;

      (*sptr)->_n._n_name[0] = sname;
      (*sptr)->_n._n_name[1] = 0;
    }
  else
    if ((*aptr)->n_un.n_strx)
      set_symbol_name ((*sptr), AOUT_SYM_NAME (*(*aptr)));
#endif				/*  */

  /* set up symbol map */
  symbol_map[an] = SYM_DIFF((*sptr),  coff_symbols);
  (*sptr)->n_numaux = 0;

  if (LINE_SCOPE_CHANGE(type))
    line_scope_change_sym = (*aptr);

  SYM_INCR((*sptr));
}


/*
 * Build COFF symbol table in core, and a mapping from
 * a.out symbol numbers to COFF symbol numbers
 */
static void
build_coff_symbols()
{
  long coff_nsyms; /* initial number of COFF symbols */
  long an;	   /* a.out symbol number */
  long sn;	   /* COFF section index */
  struct nlist *aptr;
  SYMENT *sptr;
  AUXENT *xptr;

  unsigned int type;

  /* number of COFF symbols is number of a.out symbols
     + number_of_funcs (for the function symbol aux entries)
     + 3 (for the .file symbol, its auxent, and possibly an "end_debug" symbol)
     + number_of_sections * 2 (for the section name symbols and their aux 
       entries)
     since we don't know the number of function symbols yet, for now we
     overestimate by using aout_nsymbols * 2 */

  coff_nsyms = aout_nsymbols * 2 + 3 + coff_hdr.f_nscns * 2;

  coff_symbols = (SYMENT*) xmalloc (coff_nsyms * SYMESZ);

  /* allocate the a.out-to-COFF symbol map */
  symbol_map =
    (long*) xmalloc ((aout_nsymbols + coff_hdr.f_nscns) * sizeof (long));

  /* setup coff symbols pointer	 */
  sptr = coff_symbols;

  /* generate the .file symbol */

#if TARGET == TARGET_SUN386
		/* sun ld marches to the beat of a different drummer !!! */
  add_anonymous(&sptr, ".file", T_NULL, 0, N_DEBUG, C_FILE, 1);
#else
  add_anonymous(&sptr, ".file", T_NULL, 2, N_DEBUG, C_FILE, 1);
#endif

  /* generate the .file symbol's aux entry */
  xptr = (AUXENT*) sptr;
  bzero (sptr,AUXESZ);
  strncpy (xptr->x_file.x_fname, "<unknown>", FILNMLEN);
  file_aux = xptr;
  SYM_INCR(sptr);

  /* One or more passes to build the coff symbols: */

#ifdef SYMBOL_PASS1
     SYMBOL_PASS1();
#endif
#ifdef SYMBOL_PASS2
     SYMBOL_PASS2();
#endif
#ifdef SYMBOL_PASS3
     SYMBOL_PASS3();
#endif
#ifdef SYMBOL_PASS4
     SYMBOL_PASS4();
#endif
#ifdef SYMBOL_PASS5
     SYMBOL_PASS5();
#endif
#ifdef SYMBOL_PASS6
     SYMBOL_PASS6();
#endif
#ifdef SYMBOL_PASS7
     SYMBOL_PASS7();
#endif

  terminate_line_list();

  /* set final number of COFF symbols */
  coff_hdr.f_nsyms = SYM_DIFF(sptr, coff_symbols);

}
/*
 *  write_coff_symbols - to output file
 */
static void
write_coff_symbols()
{
  /* assumes output file pointer is already positioned correctly */
  if (coff_hdr.f_nsyms)
    {
      /* set symbol table pointer in COFF header */
      coff_hdr.f_symptr = ftell (coff_file);
      if (need_byteswap)
	md_syment_to_chars((char*) coff_symbols, coff_hdr.f_nsyms);

      fwrite ((char*) coff_symbols, SYMESZ, coff_hdr.f_nsyms, coff_file);
    }
}

/*
 *  Write COFF string table to output file
 */
static void write_coff_strings()
{
  /* ld seems to insist on there being a string table on apollo systems. ld on
   * other coff systems does not seem to care. mkshlib on the other hand, gets
   * very upset about the presence of a string count if there are no strings.
   */

  if (coff_strings_len == 0)
    {
#if TARGET == TARGET_APOLLO 
      coff_strings_len = 4; /* + strlen (my_name) + 1 */
      if (need_byteswap)
	md_number_to_chars((long*) coff_strings, coff_strings_len,
			   sizeof(coff_strings_len));

      fwrite ((char*) &coff_strings_len, sizeof (coff_strings_len), 1,
	      coff_file);
#endif
    }
  else
    {
      /* set the string table length */
      md_number_to_chars((long*) coff_strings, coff_strings_len,
			 sizeof(coff_strings_len));
      fwrite (coff_strings, coff_strings_len, 1, coff_file);
    }
}

static void inline
add_type(type, name,scnum,sclass, cofftype,line_scope_change)
     unsigned int type;
     char name;
     int scnum;
     int sclass;
     int cofftype;
     int line_scope_change;
{
  CLASS_OF (type) = sclass;
  SCNUM_OF (type) = scnum;
  SNAME_OF (type) = name;
  COFFTYPE_OF (type) = cofftype;
  LINE_SCOPE_CHANGE(type) = line_scope_change;
}

/* Initialize mapping from aout types to coff symbol information */
static void
init_type_map()
{
  int i;

  for (i = 0; i < 256; i++)
    {
      SCNUM_OF(i) = N_DEBUG;
      CLASS_OF(i) = C_STAT;
      COFFTYPE_OF(i) = 0;
      LINE_SCOPE_CHANGE(i) = 0;
    }

  /*	    symbol				storage	coff type on   line  */
  /*	    type	       sname  section	class	sparc/sun386i  scope */
  /*	    -----	       -----  -------	-------	------------   ----- */

  /* defined and undefined globals */

  add_type (N_UNDF,		  0,	0,	C_EXT,		0,	0);
  add_type (N_EXT,		  0,	0,	C_EXT,		0,	0);

  /* symbol types to link text relative */

  add_type (N_TEXT ,		  0,	1,	C_STAT,		0,	0);
  add_type ((int)N_TEXT | N_EXT,  0,	1,	C_EXT,		0,	0);
  add_type (N_ENTRY ,		  0,	1,	C_STAT,		T_INT,	0);
  add_type (N_SOL ,		  0,	1,	C_STAT,		T_INT,	1);
  add_type (N_SO ,		  0,	1,	C_STAT,		T_INT,	0);

  add_type (N_LBRAC ,		  'L',	1,	C_BLOCK,	0,	0);
  add_type (N_RBRAC ,		  'R',	1,	C_BLOCK,	0,	0);
  add_type (N_SETT ,		  0,	1,	C_STAT,		0,	0);

#if TARGET == TARGET_SUN386
  add_type (N_FUN ,		  0,	1,	C_EXT,
    					(DT_FCN <<N_BTSHFT) | T_INT,	1);
#else
  add_type (N_FUN ,		  0,	1,	C_STAT,		N_FUN,	1);
#endif

  /* Obsolete: N_SLINE symbols are no longer emitted now that we generate a
   * line number section.
   */
  add_type (N_SLINE ,		  'S',	1,	C_LABEL,	0,	0);

  /* symbol types to link data relative */

  add_type (N_DATA ,		  0,	2,	C_STAT,		0,	0);
  add_type ((int)N_DATA | N_EXT,  0,	2,	C_EXT,		0,	0);
  add_type (N_STSYM ,		  0,	2,	C_STAT,		T_INT,	0);
  add_type (N_DSLINE ,		'D',	2,	C_LABEL,	0,	0);
  add_type (N_SETD ,		  0,	2,	C_STAT,		0,	0);

  /* symbol types to link bss relative */

  add_type (N_BSS,		0, BSS_SECTION, C_STAT,		0,	0);
  add_type ((int)N_BSS | N_EXT, 0, BSS_SECTION, C_EXT,		0,	0);
  add_type (N_LCSYM ,		0, BSS_SECTION, C_STAT,		T_INT,	0);
  add_type (N_BSLINE ,	      'B', BSS_SECTION, C_LABEL,	0,	0);
  add_type (N_SETB ,		  0,	3,	C_STAT,		0,	0);

  /* symbol types to link absolutely */
  add_type (N_BINCL,		  0,	-2,	C_NULL,		0,	1);
  add_type (N_EINCL,		'E',	-2,	C_NULL,		0,	1);

  add_type (N_GSYM ,		  0,	-2,	C_STAT,		T_NULL,	0);
  add_type (N_PSYM ,		  0,	-1,	C_ARG,		T_INT,	0);
  add_type (N_RSYM ,		  0,	-1,	C_REG,		T_INT,	0);
  add_type (N_SSYM ,		  0,	-1,	C_MOS,		T_INT,	0);

  add_type (N_LSYM ,		  0,	-1,	C_TPDEF,	T_INT,	0);
  add_type (N_SETA ,		  0,	-1,	C_STAT,		0,	0);

}

/*
 * Store contents of FILEHEADER in NEWHEADER, using target byte order
 */
md_filhdr_to_chars(fileheader,newheader)
     register struct filehdr *fileheader,*newheader;
{
  if (newheader != fileheader)
    bcopy(fileheader, newheader, sizeof *fileheader);

  md_short (newheader->f_magic);
  md_short (newheader->f_nscns);
  md_long  (newheader->f_timdat);
  md_long  (newheader->f_symptr);
  md_long  (newheader->f_nsyms);
  md_short (newheader->f_opthdr);
  md_short (newheader->f_flags);
}

/*
 * Store contents of AOUTHEADER in NEWHEADER, using target byte order
 */
md_aouthdr_to_chars(aoutheader,newheader)
     register struct aouthdr *aoutheader,*newheader;
{
  if (newheader != aoutheader)
    bcopy(aoutheader, newheader, sizeof *aoutheader);

#if TARGET == TARGET_APOLLO
  md_short (newheader->vformat);
#else
  md_short (newheader->magic);
#endif
  md_short (newheader->vstamp);
  md_long  (newheader->tsize);
  md_long  (newheader->dsize);
  md_long  (newheader->bsize);
  md_long  (newheader->entry);
  md_long  (newheader->text_start);
  md_long  (newheader->data_start);
}

/*
 * Store contents of SCNHDR in NEWHEADER, using target byte order
 */
md_scnhdr_to_chars(scnhdr,newheader)
     register struct scnhdr *scnhdr,*newheader;
{
  if (newheader != scnhdr)
    bcopy(scnhdr, newheader, sizeof *scnhdr);

  md_long  (newheader->s_paddr);
  md_long  (newheader->s_vaddr);
  md_long  (newheader->s_size);
  md_long  (newheader->s_scnptr);
  md_long  (newheader->s_relptr);
  md_long  (newheader->s_lnnoptr);
  md_short (newheader->s_nreloc);
  md_short (newheader->s_nlnno);
  md_long  (newheader->s_flags);
}


#ifdef SUN_STYLE_COFF
#define SHOULD_SWAP_DESC(s, type) \
  ((s->n_leading_zero == 0) && (s->n_dbx_type != 0))
#else
#define SHOULD_SWAP_DESC(s, type) \
  ((nlp->_n._n_name[1] == 0) && (type & N_STAB))
#endif

/*
 * Convert symbol at *NLP to target byte order
 */
md_syment_to_chars(nlp, numents)
     register struct syment *nlp;
     unsigned numents;
{
  register struct syment *end = &SYM_INDEX(nlp,numents);

  while (nlp < end)
    {
      unsigned char type = 0;
#ifdef SUN_STYLE_COFF
      if (nlp->n_dbx_type &&nlp->_n._n_name[0] == 0)
	type = nlp->n_dbx_type;
#else
      type = nlp->n_type >> 8 ;
#endif

      md_long  (nlp->n_value);
      md_short (nlp->n_scnum);
      md_short (nlp->n_type);
      /* handle string offset */
      if (nlp->_n._n_name[0] == 0)
	md_long	 (nlp->_n._n_n._n_offset);

      /* handle desc */
      if (SHOULD_SWAP_DESC(nlp,type))
	{
	  register unsigned short * desc
	    =(unsigned short *)&nlp->_n._n_name[2];

	  md_short(*desc);
	}
      if (nlp->n_numaux)
	{
	  int class = nlp->n_sclass;
	  SYM_INCR(nlp);
	  if (class != C_FILE)
	    md_aux_to_chars(nlp,type);
	}
      SYM_INCR(nlp);
    }
}
/*
 * Convert auxent at *XPTR to target byte order. TYPE indicates the format
 * that we originally created the auxent entry in.
 */
md_aux_to_chars(xptr,type)
     union auxent *xptr;
     int type;
{
  if (type == (int) N_FUN)
    {
      md_long(xptr->x_sym.x_fcnary.x_fcn.x_endndx);
      md_long(xptr->x_sym.x_fcnary.x_fcn.x_lnnoptr);
    }
  else
    {
      /*	scn auxent; */
      md_long(xptr->x_scn.x_scnlen);
      md_short(xptr->x_scn.x_nreloc);
      md_short(xptr->x_scn.x_nlinno);
    }
}

#endif /* COFF_OPTION */
#ifdef STANDALONE_CONV
#  if TARGET_BYTEORDER == BYTEORDER_BIG_ENDIAN

/*
 * Write out big-endian.
 */
void
md_number_to_chars(buf,val,n)
    char *buf;
    long val;
    int n;
{
  switch(n) {

  case 4:
    *buf++ = val >> 24;
    *buf++ = val >> 16;
  case 2:
    *buf++ = val >> 8;
  case 1:
    *buf = val;
    break;

  default:
    abort();
  }
  return;
}
#  else /* not TARGET_BYTEORDER == BYTEORDER_BIG_ENDIAN */
#define BITS_PER_CHAR 8
void
md_number_to_chars (buf, value, nbytes)
     char       *buf;
     long int   value;
     int        nbytes;
{
  while (nbytes--)
    {
      *buf++ = value;           /* Lint wants & MASK_CHAR. */
      value >>= BITS_PER_CHAR;
    }
}

#  endif /* not TARGET_BYTEORDER == BYTEORDER_BIG_ENDIAN */
#include <varargs.h>
void
as_fatal(Format,va_alist)
char *Format;
va_dcl
{
  va_list args;

  va_start(args);
  fprintf (stderr, "FATAL:");
  vfprintf(stderr, Format, args);
  (void) putc('\n', stderr);
  va_end(args);
  exit(42);
}
/* as_warn is only used in one place so far.. */
#define as_warn(fmt,msg) fprintf(stderr,fmt,msg)
#endif /* STANDALONE_CONV */

/*
  Set handling: Collect all set symbols of the same name into a section
  whose name is derived from the set symbol name and whose data is a 
  relocatable collection of the values of the original set symbols.
  This should make the use of "collect" with g++ no longer be necessary. */

/* link aout symbol SYM into the list of symbols managed by WHICH_SET.
   Used for handling set symbols.  */

link_aout_symbol(sym,which_set)
     struct nlist *sym;
     struct set_info *which_set;
{
  which_set->set_count++;
  sym->n_un.n_next = 0;
  if (which_set->set_root == 0)
    which_set->set_root = sym;
  if (which_set->set_tail != 0)
    which_set->set_tail->n_un.n_next = sym;
  which_set->set_tail = sym;
}

#ifdef DONT_HANDLE_SET_VECTORS
/* 
  For the Sun 386i, we cannot use set vector sections.
  The sun coff loader is badly broken and cannot handle multiple text type
  sections. I strongly suspect that it is really a bsd loader that has been
  hacked to minimally understand coff. */

#else /* not DONT_HANDLE_SET_VECTORS */

struct set_info *
make_set_entry(name,sptr,type)
     char *name;
     struct syment **sptr;
{
  struct set_info * entry
    = (struct set_info *) xmalloc(sizeof (struct set_info));

  bzero(entry, sizeof(struct set_info));
  entry->set_name = name;
  entry->set_type = type; 
  return entry;
}
struct set_info *
get_set_entry(aptr,sptr)
  struct nlist *aptr;
     struct syment **sptr;
{
  struct set_info *entry = setlist_head;
  char *name = AOUT_SYM_NAME(*aptr);

  if (entry == 0)		/* first time around */
    {
      setlist_tail = setlist_head = make_set_entry(name,sptr,aptr->n_type);
      return setlist_head;
    }
  else
    while (entry != 0)		/* search for current name */
      {
	if (strcmp(entry->set_name,name) == 0)
	  {
	    return entry;
	  }
	entry = entry->next_set;
      }
  if (entry == 0)		/* not found, so make new one */
    {
      entry = make_set_entry(name,sptr,aptr->n_type);
      setlist_tail->next_set = entry;
      setlist_tail = entry;
    }
  return entry;
}

static void
process_set_symbol(aptr,sptr,xptr,type,an)
     struct nlist **aptr;
     SYMENT **sptr;
     AUXENT **xptr;
     unsigned int type;
     int an;
{
  int scnum = SCNUM_OF(type);
  struct set_info *which_set;

  switch (type &~N_EXT)
    {
      /* for now we only deal with N_SETT symbols which are what
	 g++ generates.
       */
    case N_SETT:
      which_set = get_set_entry(*aptr, sptr);
      (*aptr)->n_value += value_adjust[scnum];
      link_aout_symbol(*aptr,which_set);
      break;
    default:
      as_warn ("set type 0x%x not implemented\n",type);
    }
}

/* translate a name such as "___CTOR_LIST__" to something like ".dtor" */
char *
derive_section_name(name)
     char *name;
{
  char *buf;
  char *bufp;

  if (*name != '_' && name[1] && name[1] != '_')
    return name;
  while (*name == '_')
    name++;

  bufp = buf = xmalloc(strlen(name)+1);
  *bufp++ = '.';
  while (*name != '_' && (*bufp++ = *name++))
    if (bufp[-1] <= 'Z' && bufp[-1] >='A')
      bufp[-1] += ('a' - 'A');
    ;
  *bufp = 0;
  return buf;
}

process_set_list(entry,sptr)
     struct set_info *entry;
     SYMENT **sptr;
{
  int sn =   coff_hdr.f_nscns++;
  struct nlist *member;
  int symbol_index;
  char *crelocs;
  char *name;
  struct reloc *relocs;
  unsigned long *set_data;
  int offset;
  union auxent *xptr;

  /* TODO: when set name is name1:name2, generate symbol name2 and make
   * set the reloc entry in section name1 for this value to have the symndx
   * of name2.
   */

  /* allocate relocation and data */
  crelocs = xmalloc(entry->set_count *RELSZ);
  entry->set_data =
    (unsigned long *) xmalloc(entry->set_count *sizeof(unsigned long));
  entry->relocs = (struct reloc *) crelocs;
  entry->set_section = sn;

  /* fill in section header info */

  name = derive_section_name(entry->set_name); 

  build_section_header(name,
		       sn,
		       coff_scnhdrs[sn-1].s_paddr + coff_scnhdrs[sn-1].s_size,
		       entry->set_count *sizeof(unsigned long),
		       entry->set_count,
		       STYP_REG | STYP_DATA,
		       next_scnptr
		       );

  /* fill in section symbol info */
  add_section_symbol(sptr,sn);

  set_data = entry->set_data;
  member = entry->set_root;
  offset = 0;
  symbol_index = symbol_map[aout_nsymbols + SCNUM_OF(entry->set_type) - 1];
			     
  while(member)
    {
      relocs = (struct reloc *) crelocs;
      relocs->r_vaddr = coff_scnhdrs[sn].s_vaddr + offset;
      relocs->r_symndx = symbol_index;
      relocs->r_type = DIRECT32BIT_RELOCATION;
      offset += 4;
      crelocs += RELSZ;
      *set_data++ = member->n_value;
      member = member->n_un.n_next;
    }
}
static void
process_set_lists(sptr)
  SYMENT **sptr;
{
  struct set_info *set = setlist_head;
  
  while (set)
    {
      process_set_list(set,sptr);
      set = set->next_set;
    }
}

static void
build_coff_vector_relocs()
{
  RELOC *coff_reloc;
  long	rn;
  struct set_info *set = setlist_head;
  
  while (set)
    {
      SCNHDR *coff_sptr = &coff_scnhdrs[set->set_section];
      if (coff_sptr->s_nreloc)
	{
	  char *crelocs = (char *)set->relocs;

	  /* set file pointer to relocs in section header */
	  coff_sptr->s_relptr = ftell (coff_file);

	  /* generate all the reloc entries in ascending order */
	  for (rn = 0; rn < coff_sptr->s_nreloc; ++rn)
	    {
	      coff_reloc = (struct reloc *) &crelocs[rn *RELSZ];
	      if (need_byteswap)
		md_reloc_to_chars((char*) coff_reloc);
	      fwrite ((char*) coff_reloc, RELSZ, 1, coff_file);
	    }
	}
      set = set->next_set;
    }
}

static void
build_coff_vector_sections()
{
  struct set_info *set = setlist_head;

  while (set)
    {
      SCNHDR *coff_sptr = &coff_scnhdrs[set->set_section];
      if (coff_sptr->s_size)
	{
	  unsigned long *set_data = set->set_data;
	  int i;

	  for (i =0; i< set->set_count;i++)
	    {
	      md_long(set_data[i]);
	    }
	  /* position output file pointer */
	  fseek (coff_file, coff_sptr->s_scnptr, 0);
	  
	  /* write text data out to COFF file */
	  fwrite (set_data, coff_sptr->s_size, 1, coff_file);
	}
      set = set->next_set;
    }
}
#endif /* DONT_HANDLE_SET_VECTORS */
