/*
 coff-convert.c - Converts M68K object files in the GNU a.out format
                  to Apollo COFF format

 Copyright (c) 1989  John Vasta

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 1, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software,
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 Questions about this program can be addressed to
     John Vasta   (vasta@apollo.com)
     Apollo Computer, Inc.
     330 Billerica Rd.
     Chelmsford, MA 01824, USA

 This module may be usable on other COFF systems, but I've only tested
 it on the Apollo. It is designed to be built into the GNU assembler,
 which calls it just before writing out an object file.
*/

#define TARGET_APOLLO	150

#include <stdio.h>
#include <strings.h>
#include <time.h>

extern int bzero (char*, int);

/* the assembler interface */
#include "as.h"
extern char version_string[];

/* the a.out format specification */
#include "a.out.h"
#undef  N_ABS          /* conflicts with N_ABS in syms.h */
#define N_ABSOLUTE 2

#ifndef N_DATOFF
#define N_DATOFF(x) (N_TXTOFF(x) + (x).a_text)
#endif

#ifndef N_TRELOFF
#define N_TRELOFF(x) (N_DATOFF(x) + (x).a_data)
#endif

#ifndef N_DRELOFF
#define N_DRELOFF(x) (N_TRELOFF(x) + (x).a_trsize)
#endif

#ifndef N_TXTADDR
#define N_TXTADDR(x) 0
#endif

#ifndef N_DATADDR
#define N_DATADDR(x) (N_TXTADDR(x)+(x).a_text)
#endif

/* the COFF format specification (for the pieces we need) */
#include <filehdr.h>
#include <scnhdr.h>
#include <reloc.h>
#include <syms.h>

#if TARGET == TARGET_APOLLO
#ifdef apollo
#include <aouthdr.h>
#include <apollo/mir.h>
#else
#define apollo
#define APOLLOM68KMAGIC 0627
#define APOLLO_COFF_VERSION_NUMBER 1

typedef	struct aouthdr
{
  short	vformat;
  short	vstamp;
  long	tsize;
  long	dsize;
  long	bsize;
  long	entry;
  long	text_start;
  long	data_start;
  long	o_sri;
  long	o_inlib;
  long  vid[2];
} AOUTHDR;

#define STYP_INSTRUCTION	0x00200000
#define  _APTV ".aptv"
#define _MIR	".mir"

#define MIRHDRSZ 4
#define MIR_MAKER_SIZE 14
#define  maker_mir_kind 1

typedef struct
{
  long int nmirs;
} mirhdr;

typedef struct
{
  short kind;
  short mir_size;
  union
    {
      struct
        {
	  unsigned short int version;
	  long  int creation_time;
	  long  int _maker_name_offset;
        } mir_maker;
      struct
        {
	  long int module_name_offset;
        } mir_module_name;
    } info;
} mir;
#define maker_name_offset mir_maker._maker_name_offset

#endif /* apollo */
#endif
/* conflicts with nlist declaration in a.out.h */
#undef n_name

/* global a.out data structures */
struct exec            *aout_hdr;       /* the a.out file header */
char                   *aout_text;      /* the a.out text section */
char                   *aout_data;      /* the a.out data section */
struct relocation_info *aout_trelocs;   /* array of a.out text reloc entries */
long                    aout_ntrelocs;  /* number of a.out text reloc entries */
struct relocation_info *aout_drelocs;   /* array of a.out data reloc entries */
long                    aout_ndrelocs;  /* number of a.out data reloc entries */
struct nlist           *aout_symbols;   /* array of a.out symbol entries */
long                    aout_nsymbols;  /* number of a.out symbols */
char                   *aout_strings;   /* the a.out string table */

/* macro to return the string for an a.out symbol */
#define AOUT_SYM_NAME(s)	(aout_strings + (s).n_un.n_strx)

/* global COFF data structures */
#define MAX_COFF_SECTIONS   4

/* define the appropriate COFF magic number here */
#ifdef apollo
#define COFF_MAGIC_NUMBER   APOLLOM68KMAGIC
#endif

/* base text address for COFF files */
#ifdef apollo
#define COFF_TXTADDR    0x8000
#endif

/* memory segment size */
#ifdef apollo
#define SEGMENT_SIZE    0x8000
#endif

FILHDR      coff_hdr;                           /* the COFF file header */
AOUTHDR     coff_aouthdr;                       /* the COFF optional header */
SCNHDR      coff_scnhdrs[MAX_COFF_SECTIONS];    /* array of COFF section headers */
char       *coff_snames[MAX_COFF_SECTIONS];     /* names of COFF sections */
RELOC      *coff_relocs[MAX_COFF_SECTIONS];     /* array of COFF reloc entries */
SYMENT     *coff_symbols;                       /* array of COFF symbol entries */
char       *coff_strings;                       /* the COFF string table */
long        coff_strings_len;                   /* current used length of table */
long        coff_strings_size;                  /* current allocated size of table */
char       *coff_filename;                      /* the temp output filename */
FILE       *coff_file;                          /* the temp output stream */

/* a.out-to-COFF symbol index map */
long       *symbol_map;

/* COFF section numbers */
int         text_snum;
int         data_snum;
int         bss_snum;
#ifdef apollo
int         aptv_snum;
int	    mir_snum;
#endif

/* program command options flags */
int         writable_text = 1;        /* put code in a writable data section */
#ifdef apollo
int         use_aptv = 0;             /* generate Apollo transfer vector */
#endif

/* 68K opcodes of interest */
#define BRA     0x60ff
#define JMP     0x4ef9
#define BSR     0x61ff
#define JSR     0x4eb9

/* macro to align a COFF file pointer or size value on a 4 byte boundary */
#define scn_align(a)    (((long)(a) + 3) & ~3)

/* macro to align a COFF file pointer or size value on a segment boundary */
#define seg_align(a)    (((long)(a) + SEGMENT_SIZE - 1) & ~(SEGMENT_SIZE - 1))

/* internal function prototypes */
static void open_files              (void);
static void close_files             (char**, long*);
static void setup_aout_data         (char*);
static void build_coff_hdr          (void);
static void build_coff_aouthdr      (void);
static void build_coff_scnhdrs      (void);
static void build_coff_text_section (void);
static void build_coff_data_section (void);
#ifdef apollo
static void build_coff_aptv_section (void);
static void build_coff_mir_section  (void);
#endif
static void reloc_section_data      (struct relocation_info *rptr, char *data,
				     int is_text);
static void build_coff_text_relocs  (void);
static void build_coff_data_relocs  (void);
#ifdef apollo
static void build_coff_aptv_relocs  (void);
#endif
static void build_coff_symbols      (void);
static void write_coff_symbols      (void);
static long map_reloc_symbol        (struct relocation_info *rptr);
static long add_coff_string         (char *str);
static void write_coff_strings      (void);
static void write_coff_hdr          (void);
static void write_coff_aouthdr      (void);
static void write_coff_scnhdrs      (void);

/*
 * Main entry point to convert an a.out memory image to a COFF memory image
 */

int convert_to_coff
(
  char  **aout_file_data,
  long   *aout_file_size
)
{
  /* open the temporary files */
  open_files ();
  
  /* set up a.out structure info */
  setup_aout_data (*aout_file_data);
  
  /* initialize COFF headers */
  build_coff_hdr ();
  build_coff_aouthdr ();
  build_coff_scnhdrs ();
  
  /* build COFF symbol table in core */
  build_coff_symbols ();
  
  /* build the section data in output file */
  build_coff_text_section ();
  build_coff_data_section ();

#ifdef apollo
  if (use_aptv)
    build_coff_aptv_section ();
  build_coff_mir_section ();
#endif
 
  /* build the relocation data in output file */
  build_coff_text_relocs ();
  build_coff_data_relocs ();

#ifdef apollo
  if (use_aptv)
    build_coff_aptv_relocs ();
#endif
  
  /* write out symbols and strings */
  write_coff_symbols ();
  write_coff_strings ();
  
  /* finish off the headers */
  write_coff_hdr ();
  write_coff_aouthdr ();
  write_coff_scnhdrs ();
  
  /* finish everything */
  close_files (aout_file_data, aout_file_size);
}

/*
 *  open_files - temporary output file
 */
static void open_files(void)
{
  if ((coff_filename = tempnam (NULL, "ATOC")) == NULL)
    as_fatal ("can't construct temporary filename");

  coff_file = fopen (coff_filename, "w+");
  if (coff_file == NULL)
    as_fatal ("cannot create temporary file %s", coff_filename);
}

/*
 *  close_files - read final temp file into memory, replacing original a.out data
 */
static void close_files
(
  char  **aout_file_data,
  long   *aout_file_size
)
{
  if (coff_file != NULL)
    {
      fseek (coff_file, 0, 2); /* position to end-of-file to get size */
      *aout_file_size = ftell (coff_file);
      rewind (coff_file);

      *aout_file_data = xmalloc (*aout_file_size);

      if (fread (*aout_file_data, *aout_file_size, 1, coff_file) == 0)
	as_fatal ("cannot read converted data from temp file");

      fclose (coff_file);
      unlink (coff_filename);
    }
}

/*
 *  setup_aout_data - sets up global a.out data structures
 */
static void setup_aout_data
(
  char *aout_file_data
)
{
  aout_hdr = (struct exec*) aout_file_data;

  if (aout_hdr->a_magic != OMAGIC)
    as_fatal ("bad magic number %o, expected %o", aout_hdr->a_magic, OMAGIC);

  if (aout_hdr->a_text)
    aout_text = aout_file_data + N_TXTOFF (*aout_hdr);

  if (aout_hdr->a_data)
    aout_data = aout_file_data + N_DATOFF (*aout_hdr);

  if (aout_hdr->a_trsize)
    {
      aout_trelocs = (struct relocation_info*) (aout_file_data + N_TRELOFF (*aout_hdr));
      aout_ntrelocs = aout_hdr->a_trsize / sizeof (struct relocation_info);
    }
  
  if (aout_hdr->a_drsize)
    {
      aout_drelocs = (struct relocation_info*) (aout_file_data + N_DRELOFF (*aout_hdr));
      aout_ndrelocs = aout_hdr->a_drsize / sizeof (struct relocation_info);
    }

  if (aout_hdr->a_syms)
    {
      aout_symbols = (struct nlist*) (aout_file_data + N_SYMOFF (*aout_hdr));
      aout_nsymbols = aout_hdr->a_syms / sizeof (struct nlist);
      aout_strings = aout_file_data + N_STROFF (*aout_hdr);
    }
}

/*
 *  build_coff_hdr - initializes as much of the COFF file header as possible
 */
static void build_coff_hdr(void)
{
  coff_hdr.f_magic  = COFF_MAGIC_NUMBER;
#ifdef apollo
  coff_hdr.f_nscns  = use_aptv ? 5 : 4;
#else
  coff_hdr.f_nscns  = 3;
#endif
  coff_hdr.f_timdat = time (NULL);
  coff_hdr.f_symptr = 0; /* to be filled in later */
  coff_hdr.f_nsyms  = 0; /* to be filled in later */
  coff_hdr.f_opthdr = sizeof (AOUTHDR);
  coff_hdr.f_flags  = F_LNNO | F_LSYMS;
}

/*
 *  write_coff_hdr - writes current COFF filehdr to output file
 */
static void write_coff_hdr(void)
{
  fseek (coff_file, 0, 0);
  fwrite ((char*) &coff_hdr, FILHSZ, 1, coff_file);
}


/*
 *  build_coff_aouthdr - initializes the COFF optional header
 */
static void build_coff_aouthdr(void)
{
#ifdef apollo
  coff_aouthdr.vformat = APOLLO_COFF_VERSION_NUMBER;
#else
  coff_aouthdr.magic   = 0;
#endif
  coff_aouthdr.vstamp  = 0;

  coff_aouthdr.tsize   = sizeof (coff_hdr) + sizeof (coff_aouthdr)
                         + sizeof (coff_scnhdrs[0]) * coff_hdr.f_nscns
			 + (writable_text ? 0 : aout_hdr->a_text);
  coff_aouthdr.tsize   = scn_align (coff_aouthdr.tsize);

  coff_aouthdr.dsize   = aout_hdr->a_data + aout_hdr->a_bss
                         + (writable_text ? aout_hdr->a_text : 0);
  coff_aouthdr.dsize   = scn_align (coff_aouthdr.dsize);

  coff_aouthdr.bsize   = 0; /* figured out later by looking at symbols */
  coff_aouthdr.entry   = 0;
  coff_aouthdr.text_start = COFF_TXTADDR;
  coff_aouthdr.data_start = seg_align (COFF_TXTADDR + coff_aouthdr.tsize);

#ifdef apollo
  coff_aouthdr.o_sri   = 0;
  coff_aouthdr.o_inlib = 0;
  coff_aouthdr.vid[0]  = 0;
  coff_aouthdr.vid[1]  = 0;
#endif
}

/*
 *  write_coff_aouthdr - writes current COFF optional header to output file
 */
static void write_coff_aouthdr(void)
{
  fseek (coff_file, FILHSZ, 0);
  fwrite ((char*) &coff_aouthdr, sizeof (coff_aouthdr), 1, coff_file);
}

/*
 *  build_coff_scnhdrs - initializes all COFF section headers
 */
static void build_coff_scnhdrs(void)
{
  long sn = 0;

  /* init header for text section */
  if (writable_text)
    coff_snames[sn] = ".wtext";
  else
    coff_snames[sn] = _TEXT;

  strncpy (coff_scnhdrs[sn].s_name, coff_snames[sn], SYMNMLEN);

  coff_scnhdrs[sn].s_scnptr  = sizeof (coff_hdr) + sizeof (coff_aouthdr)
                               + sizeof (coff_scnhdrs[sn]) * coff_hdr.f_nscns;
  coff_scnhdrs[sn].s_paddr   =
    writable_text ? coff_aouthdr.data_start
                  : COFF_TXTADDR + coff_scnhdrs[sn].s_scnptr;
  coff_scnhdrs[sn].s_vaddr   = coff_scnhdrs[sn].s_paddr;
  coff_scnhdrs[sn].s_size    = scn_align (aout_hdr->a_text);
  coff_scnhdrs[sn].s_relptr  = 0;
  coff_scnhdrs[sn].s_lnnoptr = 0;
  coff_scnhdrs[sn].s_nreloc  = aout_ntrelocs;
  coff_scnhdrs[sn].s_nlnno   = 0;
  coff_scnhdrs[sn].s_flags   = writable_text ? STYP_REG | STYP_DATA
#ifdef apollo
                                             : STYP_REG | STYP_TEXT | STYP_INSTRUCTION;
#else
                                             : STYP_REG | STYP_TEXT;
#endif
  text_snum = ++sn;

  /* init header for data section */
  coff_snames[sn] = _DATA;
  strncpy (coff_scnhdrs[sn].s_name, coff_snames[sn], SYMNMLEN);

  coff_scnhdrs[sn].s_scnptr  = coff_scnhdrs[sn - 1].s_scnptr
                               + coff_scnhdrs[sn - 1].s_size;
  coff_scnhdrs[sn].s_paddr   =
    writable_text ? coff_scnhdrs[sn - 1].s_paddr + coff_scnhdrs[sn - 1].s_size
                  : coff_aouthdr.data_start;
  coff_scnhdrs[sn].s_vaddr   = coff_scnhdrs[sn].s_paddr;
  coff_scnhdrs[sn].s_size    = scn_align (aout_hdr->a_data + aout_hdr->a_bss);
  coff_scnhdrs[sn].s_relptr  = 0;
  coff_scnhdrs[sn].s_lnnoptr = 0;
  coff_scnhdrs[sn].s_nreloc  = aout_ndrelocs;
  coff_scnhdrs[sn].s_nlnno   = 0;
  coff_scnhdrs[sn].s_flags   = STYP_REG | STYP_DATA;

  data_snum = ++sn;

#ifdef apollo
  /* init header for Apollo transfer vector section, if used */
  if (use_aptv)
    {
      coff_snames[sn] = _APTV;
      strncpy (coff_scnhdrs[sn].s_name, coff_snames[sn], SYMNMLEN);

      coff_scnhdrs[sn].s_scnptr  = coff_scnhdrs[sn - 1].s_scnptr
	                           + coff_scnhdrs[sn - 1].s_size;
      coff_scnhdrs[sn].s_paddr   = coff_scnhdrs[sn - 1].s_paddr
	                           + coff_scnhdrs[sn - 1].s_size;
      coff_scnhdrs[sn].s_vaddr   = coff_scnhdrs[sn].s_paddr;
      coff_scnhdrs[sn].s_size    = 0;
      coff_scnhdrs[sn].s_relptr  = 0;
      coff_scnhdrs[sn].s_lnnoptr = 0;
      coff_scnhdrs[sn].s_nreloc  = 0;
      coff_scnhdrs[sn].s_nlnno   = 0;
      coff_scnhdrs[sn].s_flags   = STYP_REG | STYP_DATA;

      aptv_snum = ++sn;
    }
#endif

  /* init header for bss section */
  coff_snames[sn] = _BSS;
  strncpy (coff_scnhdrs[sn].s_name, coff_snames[sn], SYMNMLEN);

  coff_scnhdrs[sn].s_scnptr  = 0;
  coff_scnhdrs[sn].s_paddr   = coff_scnhdrs[sn - 1].s_paddr
                               + coff_scnhdrs[sn - 1].s_size;
  coff_scnhdrs[sn].s_vaddr   = coff_scnhdrs[sn].s_paddr;
  coff_scnhdrs[sn].s_size    = 0;
  coff_scnhdrs[sn].s_relptr  = 0;
  coff_scnhdrs[sn].s_lnnoptr = 0;
  coff_scnhdrs[sn].s_nreloc  = 0;
  coff_scnhdrs[sn].s_nlnno   = 0;
  coff_scnhdrs[sn].s_flags   = STYP_REG | STYP_BSS;

  bss_snum = ++sn;

#ifdef apollo
  /* init header for Apollo .mir section */
  coff_snames[sn] = _MIR;
  strncpy (coff_scnhdrs[sn].s_name, coff_snames[sn], SYMNMLEN);

  /* note that the location and size of this section are based on the
     section before last, because the last one was .bss which has no size */
  coff_scnhdrs[sn].s_scnptr  = coff_scnhdrs[sn - 2].s_scnptr
    + coff_scnhdrs[sn - 2].s_size;
  coff_scnhdrs[sn].s_paddr   = 0;
  coff_scnhdrs[sn].s_vaddr   = coff_scnhdrs[sn].s_paddr;
  coff_scnhdrs[sn].s_size    = 0; /* filled in later by build_coff_mir_section */
  coff_scnhdrs[sn].s_relptr  = 0;
  coff_scnhdrs[sn].s_lnnoptr = 0;
  coff_scnhdrs[sn].s_nreloc  = 0;
  coff_scnhdrs[sn].s_nlnno   = 0;
  coff_scnhdrs[sn].s_flags   = STYP_INFO;
  
  mir_snum = ++sn;
#endif
}

/*
 *  write_coff_scnhdrs - writes the current COFF section headers to the output file
 */
static void write_coff_scnhdrs(void)
{
  fseek (coff_file, FILHSZ + sizeof (coff_aouthdr), 0);
  fwrite ((char*) coff_scnhdrs, SCNHSZ, coff_hdr.f_nscns, coff_file);
}

/*
 *  build_coff_text_section - builds .text section data in output file
 */
static void build_coff_text_section(void)
{
  long  pad_size;

  /* position output file pointer */
  fseek (coff_file, coff_scnhdrs[text_snum - 1].s_scnptr, 0);

  if (aout_hdr->a_text)
    {
      /* modify section data to convert to COFF relocation mechanism */
      if (aout_hdr->a_trsize)
	{
	  long rn;

	  for (rn = 0; rn < aout_ntrelocs; ++rn)
	    reloc_section_data (&aout_trelocs[rn], aout_text, 1);
	}

      /* write text data out to COFF file */
      fwrite (aout_text, aout_hdr->a_text, 1, coff_file);

      /* take care of any alignment padding */
      if ((pad_size = coff_scnhdrs[text_snum - 1].s_size - aout_hdr->a_text) > 0)
	{
	  char *padding;

	  /* allocate a zeroed chunk of memory of the right size */
	  if ((padding = xmalloc (pad_size)) == NULL)
	    as_fatal ("out of memory for text padding");

	  bzero (padding, pad_size);
	  fwrite (padding, pad_size, 1, coff_file);
	  free (padding);
	}
    }
}

/*
 *  build_coff_data_section - builds .data section in output file
 */
static void build_coff_data_section(void)
{
  long  bss_size;

  /* position output file pointer */
  fseek (coff_file, coff_scnhdrs[data_snum - 1].s_scnptr, 0);

  if (aout_hdr->a_data)
    {
      /* modify section data to convert to COFF relocation mechanism */
      if (aout_hdr->a_drsize)
	{
	  long rn;

	  for (rn = 0; rn < aout_ndrelocs; ++rn)
	    reloc_section_data (&aout_drelocs[rn], aout_data, 0);
	}

      /* write data out to COFF file */
      fwrite (aout_data, aout_hdr->a_data, 1, coff_file);
    }

  /* take care of alignment padding or
     static bss data added to the end of the COFF data section */
  if ((bss_size = coff_scnhdrs[data_snum - 1].s_size - aout_hdr->a_data) > 0)
    {
      char *padding;

      /* allocate a zeroed chunk of memory of the right size */
      if ((padding = xmalloc (bss_size)) == NULL)
	as_fatal ("out of memory for bss data");

      bzero (padding, bss_size);
      fwrite (padding, bss_size, 1, coff_file);
      free (padding);
    }
}

#ifdef apollo
/*
 *  build_coff_aptv_section - builds Apollo transfer vector section in output file
 */
static void build_coff_aptv_section(void)
{
  /* not yet implemented, use writable-text instead */
}

/*
 *  build_coff_mir_section - builds Apollo transfer vector section in output file
 */
static void build_coff_mir_section(void)
{
  mirhdr mir_hdr;
  mir mir_rec;

  mir_hdr.nmirs = 1;
  fwrite ((char *) &mir_hdr, MIRHDRSZ, 1, coff_file);
  coff_scnhdrs[mir_snum - 1].s_size = MIRHDRSZ;

  mir_rec.kind = maker_mir_kind;
  mir_rec.mir_size = MIR_MAKER_SIZE;
  mir_rec.info.mir_maker.version = 0;
  mir_rec.info.mir_maker.creation_time = 0;
  mir_rec.info.maker_name_offset = add_coff_string (version_string);

  fwrite ((char *) &mir_rec, mir_rec.mir_size, 1, coff_file);
  coff_scnhdrs[mir_snum - 1].s_size += MIR_MAKER_SIZE;
}
#endif

/*
 *  reloc_section_data - fixes up section data according to reloc info
 */
static void reloc_section_data
  (
   struct relocation_info *rptr,    /* pointer to a.out reloc record */
   char                   *data,    /* pointer to section data */
   int                     is_text  /* TRUE if this is the text section */
  )
{
  /* get value of symbol referred to */
  long value = coff_symbols[map_reloc_symbol (rptr)].n_value;

  if (rptr->r_pcrel) /* change pc_relative to absolute */
    {
      switch (rptr->r_length)
	{
	case 0:
	  *((char*) (data + rptr->r_address)) = value;
	  break;
	case 1:
	  *((short*) (data + rptr->r_address)) = value;
	  break;
	case 2:
	  *((long*) (data + rptr->r_address)) = value;
	  break;
	}

      /* if this is the text section, change pc-relative calls to absolute */
      if (is_text)
	{
	  /* get pointer to instruction opcode */
	  unsigned short *optr = (unsigned short*) (data + rptr->r_address - 2);

	  if (*optr == BRA) /* change to JMP */
	    *optr = JMP;
	  else if (*optr == BSR) /* change to JSR */
	    *optr = JSR;
	  else /* can't handle it */
	    as_fatal ("unknown PC-relative reference");
	}
    }
  else /* change offset to absolute */
    {
      /* data section-relative references are actually relative to the base of
	 the text section, so compensate the base value if so */
      if (!rptr->r_extern)
	switch (rptr->r_symbolnum)
	  {
	  case N_DATA:
	  case N_BSS:
	    value -= N_DATADDR (*aout_hdr);
	    break;
	  }

      switch (rptr->r_length)
	{
	case 0:
	  *((char*) (data + rptr->r_address)) += value;
	  break;
	case 1:
	  *((short*) (data + rptr->r_address)) += value;
	  break;
	case 2:
	  *((long*) (data + rptr->r_address)) += value;
	  break;
	}
    }
}

/*
 *  build_coff_relocs - for the given section, in the output file;
 *                      assumes that output file position is already set
 */
static void build_coff_relocs
(
 SCNHDR *coff_sptr,
 struct relocation_info *aout_rptr
)
{
  RELOC coff_reloc;
  long  rn;

  if (coff_sptr->s_nreloc)
    {
      /* set file pointer to relocs in section header */
      coff_sptr->s_relptr = ftell (coff_file);

      /* generate all the reloc entries */
      for (rn = 0; rn < coff_sptr->s_nreloc; ++rn)
	{
	  coff_reloc.r_vaddr  = coff_sptr->s_vaddr + aout_rptr[rn].r_address;
	  coff_reloc.r_symndx = map_reloc_symbol (&aout_rptr[rn]);
	  coff_reloc.r_type   = R_DIR32;

	  fwrite ((char*) &coff_reloc, RELSZ, 1, coff_file);
	}
    }
}

/*
 *  build_coff_text_relocs - builds .text relocation info in output file
 */
static void build_coff_text_relocs(void)
{
  build_coff_relocs (&coff_scnhdrs[text_snum - 1], aout_trelocs);
}

/*
 *  build_coff_data_relocs - builds .data relocation info in output file
 */
static void build_coff_data_relocs(void)
{
  build_coff_relocs (&coff_scnhdrs[data_snum - 1], aout_drelocs);
}

#ifdef apollo
/*
 *  build_coff_aptv_relocs - builds .aptv relocation info in output file
 */
static void build_coff_aptv_relocs(void)
{
  /* not yet implemented */
}
#endif

/*
 *  set_symbol_name - sets the name for a COFF symbol, putting it in the
 *                    string table if required
 */
static void set_symbol_name(SYMENT *sym, char *name)
{
  /* see if string needs to go in string table */
  if (strlen (name) > SYMNMLEN)
    {
      sym->_n._n_n._n_zeroes = 0;
      sym->_n._n_n._n_offset = add_coff_string (name);
    }
  else /* it fits in symbol entry */
    strncpy (sym->_n._n_name, name, SYMNMLEN);
}

/*
 *  build_coff_symbols - builds COFF symbol table in core, and a mapping from
 *                       a.out symbol numbers to COFF symbol numbers
 */
static void build_coff_symbols(void)
{
  long coff_nsyms; /* initial number of COFF symbols */
  long an;         /* a.out symbol number */
  long sn;         /* COFF section index */
  struct nlist *aptr;
  SYMENT *sptr;
  AUXENT *xptr;

  /* number of COFF symbols is number of a.out symbols
     + number_of_funcs (for the function symbol aux entries)
     + 2 (for the .file symbol and its aux entry)
     + number_of_sections * 2 (for the section name symbols and their aux entries)

     since we don't know the number of function symbols yet, for now we
     overestimate by using aout_nsymbols * 2 */

  coff_nsyms = aout_nsymbols * 2 + 2 + coff_hdr.f_nscns * 2;
  coff_symbols = (SYMENT*) xmalloc (coff_nsyms * sizeof (SYMENT));

  /* allocate the a.out-to-COFF symbol map */
  symbol_map = (long*) xmalloc ((aout_nsymbols + coff_hdr.f_nscns) * sizeof (long));

  /* generate the .file symbol and its aux entry */
  sptr = coff_symbols;
  set_symbol_name (sptr, ".file");
  sptr->n_value  = 2;      /* first global symbol is #2 */
  sptr->n_scnum  = N_DEBUG; /* this is what they do */
  sptr->n_type   = T_NULL;
  sptr->n_sclass = C_FILE;
  sptr->n_numaux = 1;

  xptr = (AUXENT*) ++sptr;
  strncpy (xptr->x_file.x_fname, "<unknown>", FILNMLEN);
  ++sptr;

  /* add symbols for functions defined in this module */
  for (an = 0; an < aout_nsymbols; ++an)
    {
      aptr = &aout_symbols[an];

      if ((aptr->n_type & ~N_EXT) == N_TEXT)
	{
	  /* generate symbol entry for function */
	  set_symbol_name (sptr, AOUT_SYM_NAME (*aptr));
	  sptr->n_value  = aptr->n_value + coff_scnhdrs[text_snum - 1].s_vaddr;
	  sptr->n_scnum  = text_snum;
	  sptr->n_type   = T_INT | (DT_FCN << N_BTSHFT);
	  sptr->n_sclass = (aptr->n_type & N_EXT) ? C_EXT : C_STAT;
	  sptr->n_numaux = 1;

	  /* set up symbol map */
	  symbol_map[an] = sptr - coff_symbols;

	  /* generate aux entry for function */
	  xptr = (AUXENT*) ++sptr;
	  xptr->x_sym.x_tagndx = 0;
	  xptr->x_sym.x_misc.x_fsize = 0; /* can this be figured out? */
	  xptr->x_sym.x_fcnary.x_fcn.x_lnnoptr = 0;
	  xptr->x_sym.x_fcnary.x_fcn.x_endndx  = sptr - coff_symbols + 1;

	  ++sptr;
	}
    }

  /* add symbols for sections */
  for (sn = 0; sn < coff_hdr.f_nscns; ++sn)
    {
      set_symbol_name (sptr, coff_snames[sn]);
      sptr->n_value = coff_scnhdrs[sn].s_vaddr;
      sptr->n_scnum = sn + 1;
      sptr->n_type  = T_NULL;
      sptr->n_sclass = C_STAT;
      sptr->n_numaux = 1;

      /* set up symbol map - sections symbols are at end of map */
      symbol_map[aout_nsymbols + sn] = sptr - coff_symbols;

      /* generate section symbol aux entry */
      xptr = (AUXENT*) ++sptr;
      xptr->x_scn.x_scnlen = coff_scnhdrs[sn].s_size;
      xptr->x_scn.x_nreloc = coff_scnhdrs[sn].s_nreloc;
      xptr->x_scn.x_nlinno = 0;

      ++sptr;
    }

  /* add symbols for defined statics */
  for (an = 0; an < aout_nsymbols; ++an)
    {
      aptr = &aout_symbols[an];

      /* static bss data is put into the COFF data section */
      if (aptr->n_type == N_DATA || aptr->n_type == N_BSS)
	{
	  set_symbol_name (sptr, AOUT_SYM_NAME (*aptr));

	  /* a.out symbol value is relative to base of text section, not data! */
	  sptr->n_value  = aptr->n_value - N_DATADDR (*aout_hdr)
	                   + coff_scnhdrs[data_snum - 1].s_vaddr;
	  sptr->n_scnum  = data_snum;
	  sptr->n_type   = T_INT;
	  sptr->n_sclass = C_STAT;
	  sptr->n_numaux = 0;

	  /* set up symbol map */
	  symbol_map[an] = sptr - coff_symbols;

	  ++sptr;
	}
    }

  /* add symbols for defined globals */
  for (an = 0; an < aout_nsymbols; ++an)
    {
      aptr = &aout_symbols[an];

      if (aptr->n_type == (N_DATA | N_EXT))
	{
	  set_symbol_name (sptr, AOUT_SYM_NAME (*aptr));

	  /* a.out symbol value is relative to base of text section, not data! */
	  sptr->n_value  = aptr->n_value - N_DATADDR (*aout_hdr)
	                   + coff_scnhdrs[data_snum - 1].s_vaddr;
	  sptr->n_scnum  = data_snum;
	  sptr->n_type   = T_INT;
	  sptr->n_sclass = C_EXT;
	  sptr->n_numaux = 0;

	  /* set up symbol map */
	  symbol_map[an] = sptr - coff_symbols;

	  ++sptr;
	}
    }

  /* add symbols for undefined globals */
  for (an = 0; an < aout_nsymbols; ++an)
    {
      aptr = &aout_symbols[an];

      if (aptr->n_type == N_EXT)
	{
	  set_symbol_name (sptr, AOUT_SYM_NAME (*aptr));

	  sptr->n_value  = aptr->n_value;
	  sptr->n_scnum  = 0;
	  sptr->n_type   = aptr->n_value ? T_INT : T_NULL;
	  sptr->n_sclass = C_EXT;
	  sptr->n_numaux = 0;

	  /* set up symbol map */
	  symbol_map[an] = sptr - coff_symbols;

	  /* if this is an external bss symbol (n_value != 0), add its size
	     to the bss size for the module */
	  if (aptr->n_value)
	    coff_aouthdr.bsize += aptr->n_value;

	  ++sptr;
	}
    }

  /* set final number of COFF symbols */
  coff_hdr.f_nsyms = sptr - coff_symbols;
}

/*
 *  write_coff_symbols - to output file
 */
static void write_coff_symbols(void)
{
  /* assumes output file pointer is already positioned correctly */
  if (coff_hdr.f_nsyms)
    {
      /* set symbol table pointer in COFF header */
      coff_hdr.f_symptr = ftell (coff_file);

      fwrite ((char*) coff_symbols, SYMESZ, coff_hdr.f_nsyms, coff_file);
    }
}

/*
 *  map_reloc_symbol - converts an a.out symbol index to a COFF symbol index
 */
static long map_reloc_symbol
(
 struct relocation_info *rptr
)
{
  if (rptr->r_extern)
    return symbol_map[rptr->r_symbolnum];
  else
    switch (rptr->r_symbolnum)
      {
      case N_TEXT:
	return symbol_map[aout_nsymbols + text_snum - 1];

      case N_DATA:
      case N_BSS:
	return symbol_map[aout_nsymbols + data_snum - 1];

      default:
	as_fatal ("unknown a.out symbol reference");
      }
}

/*
 *  add_coff_string - adds a string to the in-core COFF string table,
 *                    returning its index value
 */
static long add_coff_string(char *str)
{
  int   strsize;
  char *st_ptr;

  strsize = strlen (str) + 1;

  /* if no room for string, expand the table */
  while (coff_strings_len + strsize > coff_strings_size)
    {
      /* if this is the first string, create the initial table */
      if (coff_strings == NULL)
	{
	  coff_strings_size = 8192;
	  coff_strings = xmalloc (coff_strings_size);
	  coff_strings_len = 4;
	}
      else /* expand the existing table */
	{
	  coff_strings_size *= 2;
	  coff_strings = xrealloc (coff_strings, coff_strings_size);
	}
    }

  st_ptr = coff_strings + coff_strings_len;
  memcpy (st_ptr, str, strsize);
  coff_strings_len += strsize;

  return st_ptr - coff_strings;
}

/*
 *  write_coff_strings - writes COFF string table to output file
 */
static void write_coff_strings(void)
{
  if (coff_strings_len == 0) /* ld seems to insist on there being a string table */
    {
      coff_strings_len = 4; /* + strlen (my_name) + 1 */
      fwrite ((char*) &coff_strings_len, sizeof (coff_strings_len), 1, coff_file);
/*      fwrite (my_name, coff_strings_len - 4, 1, coff_file); */
    }
  else
    {
      /* set the string table length */
      *((long*) coff_strings) = coff_strings_len;

      fwrite (coff_strings, coff_strings_len, 1, coff_file);
    }
}
