/*
 * select.c
 *
 * Selection of a style sheet
 * Copyright (c) 1988, 89, 90, 91, 92, 93 Miguel Santana
 * Copyright (c) 1995, 96, 97, 98 Akim Demaille, Miguel Santana
 * 
 */

/*
 * This file is part of a2ps.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * $Id: select.c,v 1.51 1998/03/03 11:47:10 demaille Exp $
 */

#include "a2ps.h"
#include "select.h"
#include "routines.h"
#include "xfnmatch.h"
#include "getshline.h"
#include "pathconcat.h"
#include "pathwalk.h"
#include "filtdir.h"
#include "msg.h"

extern char * style_request;

/*
 * Take priviledged access to job :)
 */
extern a2ps_job * job;
extern struct darray * sheets_map;


/*
 * The user has requested the style sheet him self.
 */
void
set_requested_style (const char * arg)
{
  xstrcpy (style_request, arg);
}

/************************************************************************/
/*			sheets.map handling				*/
/************************************************************************/
/*
 * Association of suffixes rules, and corresponding style sheet
 */
struct pattern_rule {
  const char * pattern;
  const char * command;
  int on_file_verdict;
};

static struct pattern_rule *
pattern_rule_new (const char * pattern, int on_file_verdict,
		  const char * command)
{
  struct pattern_rule * res = ALLOC (struct pattern_rule, 1);

  res->pattern = pattern;
  res->on_file_verdict = on_file_verdict;
  res->command = command;

  return res;
}

static void
pattern_rule_self_print (struct pattern_rule * item, FILE * stream)
{
  switch (item->on_file_verdict) {
  case TRUE:
    fprintf (stream, "file/%s: %s\n", item->pattern, item->command);
    break;
  case FALSE:
    fprintf (stream, "name/%s: %s\n", item->pattern, item->command);
    break;
  }
}

struct darray *
sheets_map_new (void)
{
  return da_new ("Sheets map", 200, 
		 da_linear, 20, 
		 (da_print_func_t) pattern_rule_self_print, NULL);
}

#define GET_TOKEN(from) (strtok ((from), " \t\n"))
#define GET_LINE_TOKEN(from) (strtok ((from), "\n"))

#define CHECK_TOKEN() 							\
  if (token2 == NULL) 							\
    error_at_line (1, 0, file, firstline, 				\
		   _("missing style sheet name for pattern: %s"), token);

/*
 * Read a sheets map file
 */
static int
sheets_map_load (const char *file)
{
  FILE * fp;
  char *buf = NULL;
  size_t bufsiz = 0;
  char *token, *token2;
  int firstline = 0, lastline = 0;
  
  message (msg_file, (stderr, "Loading sheets map `%s'\n", file));

  fp = xrfopen (file);
  
  while (getshline_numbered (&firstline, &lastline, &buf, &bufsiz, fp) != -1)
    {
      token = GET_TOKEN (buf);

      if (!token)
	/* Empty line */
	continue;

      if (strequ (token, "**")) {
	/* Pattern on result of file(1) */
	token = GET_TOKEN (NULL);	/* Pattern	*/
	token2 = GET_TOKEN (NULL);	/* Name		*/
	CHECK_TOKEN ();
	da_append (sheets_map, 
		   pattern_rule_new (xstrdup (token), TRUE,
				     xstrdup (token2)));
      } else if (strequ (token, "***")) {
	/* Load another sheets map */
	token2 = GET_TOKEN (NULL);	/* Sheets.map path	*/
	CHECK_TOKEN ();
	sheets_map_load (token2);
      } else {
	/* Pattern on the file's name */
	token2 = GET_TOKEN (NULL);	/* Name		*/
	CHECK_TOKEN ();
	da_append (sheets_map, 
		   pattern_rule_new (xstrdup (token), FALSE, 
				     xstrdup (token2)));
      }
    }
  
  FREE (buf);
  fclose (fp);
  return 1;
}

/*
 * Read the sheets.map file
 */
static int
sheets_map_load_main (void)
{
  char * file;

  /* System's */
  file = pw_find_file (job->common.path, "sheets.map", NULL);
  if (!file) {
    error (0, errno, _("cannot find file `%s'"), "sheets.map");
    /* sheets.map can not be found: there is no automatic prettyprinting */
    error (0, 0, _("automatic style selection cancelled"));
    return 0;
  }

  sheets_map_load (file);

  FREE (file);
  return 1;
}

/*
 * What says file about the type of a file (result is malloc'd)
 */
char *
file_verdict_on (const uchar *filename)
{
  char * res = NULL, * command;
  char buf [1024], field1 [512], field2 [512];
  FILE * file_out;
  
  if (IS_EMPTY (job->file_command))
    return NULL;

  /* Call file(1) with the correct option */
  command = ALLOCA (char, (2 
			   + strlen (job->file_command)
			   + ustrlen (filename)));
  sprintf (command, "%s %s", job->file_command, (const char *) filename);
  message (msg_tool, (stderr, "Reading pipe: `%s'\n", command));
  file_out = popen (command, "r");

  /* Check for failure */
  if (!file_out) {
    if (message_test(msg_tool))
      error (0, errno, _("cannot open a pipe on `%s'"),
             command);
    return NULL;
  }

  /* Get the answer */
  fgets (buf, sizeof (buf), file_out);
  pclose (file_out);
  message (msg_tool, (stderr, "file(1): %s", buf));

  /* Treat the answer:
   * the two first fields are of interest */
  switch (sscanf (buf, "%*[^:]: %511s%511s", field1, field2))
    {
    case 2:
      res = MALLOC (strlen (field1) + strlen (field2) + 1);
      sprintf (res, "%s%s", field1, field2);
      break;
      
    case 1:
      res = xstrdup (field1);
      break;
      
    default:
      message (msg_tool, (stderr, "sscanf failed\n"));
      return NULL;
    }
  if (res)
    message (msg_tool, (stderr, "File's verdict: `%s'\n", res));
  return res;
}

/*
 * Get style name from FILENAME, using pattern rules
 * and file(1) rules if USE_FILE.
 */
#define rule(_i_) ((struct pattern_rule *)sheets_map->content[_i_])
const char *
get_command (const uchar * name_to_match, const uchar * name_to_file)
{
  int i;
  char * file_verdict;

  /* We only want to read the sheets map if needed, hence,
   * from here (not needed if the sheet name is given by the
   * user) */
  if (da_is_empty (sheets_map))
    sheets_map_load_main ();

  file_verdict = file_verdict_on (name_to_file);

  /* We look from bottom up, so that most recently defined rules are
   * honored.  Make sure not to call fnmatch on an empty FILE_VERDICT
   * (thanks to Michael Taeschner <Michael.Taeschner@dlr.de>,
   * Christian Mondrup <scancm@biobase.dk> and Jens Henrik Leonhard
   * Jensen recjhl@mediator.uni-c.dk) */
  /* The loop is split to speed up */
  if (file_verdict && name_to_match)
    {
      for (i = sheets_map->len - 1 ; i >= 0 ; i--)
	if (rule(i)->on_file_verdict) 
	  {
	    /* Testing upon file's result */
	    if (!fnmatch (rule(i)->pattern, file_verdict, 0)) {
	      XFREE (file_verdict);
	      return rule(i)->command; 
	    }
	  } else {
	    /* Upon file name */
	    if (!fnmatch (rule(i)->pattern, (const char *) name_to_match, 0)) {
	      XFREE (file_verdict);
	      return rule(i)->command;
	    }
	  }
      FREE (file_verdict);
    }
  else if (name_to_match)
    {
      for (i = sheets_map->len - 1 ; i >= 0 ; i--)
	if (!rule(i)->on_file_verdict) 
	  /* Upon file name */
	  if (!fnmatch (rule(i)->pattern, (const char *) name_to_match, 0)) {
	    XFREE (file_verdict);
	    return rule(i)->command;
	  }
    }
  else if (file_verdict)
    {
      for (i = sheets_map->len - 1 ; i >= 0 ; i--)
	if (rule(i)->on_file_verdict) 
	  /* Testing upon file's result */
	  if (!fnmatch (rule(i)->pattern, file_verdict, 0)) {
	    XFREE (file_verdict);
	    return rule(i)->command; 
	  }
    }
  
  return "plain";
}
