/*********************************************************************
Spectral lines.
This is part of GNU Astronomy Utilities (Gnuastro) package.

Original author:
     Mohammad Akhlaghi <mohammad@akhlaghi.org>
Contributing author(s):
Copyright (C) 2019-2023 Free Software Foundation, Inc.

Gnuastro is free software: you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation, either version 3 of the License, or (at your
option) any later version.

Gnuastro is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with Gnuastro. If not, see <http://www.gnu.org/licenses/>.
**********************************************************************/
#ifndef __GAL_SPECLINES_H__
#define __GAL_SPECLINES_H__

/* Include other headers if necessary here. Note that other header files
   must be included before the C++ preparations below */



/* C++ Preparations */
#undef __BEGIN_C_DECLS
#undef __END_C_DECLS
#ifdef __cplusplus
# define __BEGIN_C_DECLS extern "C" {
# define __END_C_DECLS }
#else
# define __BEGIN_C_DECLS                /* empty */
# define __END_C_DECLS                  /* empty */
#endif
/* End of C++ preparations */



/* Actual header contants (the above were for the Pre-processor). */
__BEGIN_C_DECLS  /* From C++ preparations */


/* Spectral line internal codes (SORT BY WAVELENGTH).
   rb:  right bracket
   lb:  left  bracket
   rlb: left & right bracket

   From: http://astronomy.nmsu.edu/drewski/tableofemissionlines.html
   Added: Lyman-Gamma from Wikipeda
*/
enum gal_speclines_line_codes
{
  /* Allowing '0' to be identied as a known-non-line. */
  GAL_SPECLINES_INVALID=0,

  /* Spectral line names from blue to red. */
  GAL_SPECLINES_Ne_VIII_770,
  GAL_SPECLINES_Ne_VIII_780,
  GAL_SPECLINES_Ly_epsilon,
  GAL_SPECLINES_Ly_delta,
  GAL_SPECLINES_Ly_gamma,
  GAL_SPECLINES_C_III_977,
  GAL_SPECLINES_N_III_989,
  GAL_SPECLINES_N_III_991_51,
  GAL_SPECLINES_N_III_991_57,
  GAL_SPECLINES_Ly_beta,
  GAL_SPECLINES_O_VI_1031,
  GAL_SPECLINES_O_VI_1037,
  GAL_SPECLINES_Ar_I_1066,
  GAL_SPECLINES_Ly_alpha,
  GAL_SPECLINES_N_V_1238,
  GAL_SPECLINES_N_V_1242,
  GAL_SPECLINES_Si_II_1260,
  GAL_SPECLINES_Si_II_1264,
  GAL_SPECLINES_O_I_1302,
  GAL_SPECLINES_C_II_1334,
  GAL_SPECLINES_C_II_1335,
  GAL_SPECLINES_Si_IV_1393,
  GAL_SPECLINES_O_IV_1397,
  GAL_SPECLINES_O_IV_1399,
  GAL_SPECLINES_Si_IV_1402,
  GAL_SPECLINES_N_IV_1486,
  GAL_SPECLINES_C_IV_1548,
  GAL_SPECLINES_C_IV_1550,
  GAL_SPECLINES_He_II_1640,
  GAL_SPECLINES_O_III_1660,
  GAL_SPECLINES_O_III_1666,
  GAL_SPECLINES_N_III_1746,
  GAL_SPECLINES_N_III_1748,
  GAL_SPECLINES_Al_III_1854,
  GAL_SPECLINES_Al_III_1862,
  GAL_SPECLINES_Si_III,
  GAL_SPECLINES_C_III_1908,
  GAL_SPECLINES_N_II_2142,
  GAL_SPECLINES_O_III_2320,
  GAL_SPECLINES_C_II_2323,
  GAL_SPECLINES_C_II_2324,
  GAL_SPECLINES_Fe_XI_2648,
  GAL_SPECLINES_He_II_2733,
  GAL_SPECLINES_Mg_V_2782,
  GAL_SPECLINES_Mg_II_2795,
  GAL_SPECLINES_Mg_II_2802,
  GAL_SPECLINES_Fe_IV_2829,
  GAL_SPECLINES_Fe_IV_2835,
  GAL_SPECLINES_Ar_IV_2853,
  GAL_SPECLINES_Ar_IV_2868,
  GAL_SPECLINES_Mg_V_2928,
  GAL_SPECLINES_He_I_2945,
  GAL_SPECLINES_O_III_3132,
  GAL_SPECLINES_He_I_3187,
  GAL_SPECLINES_He_II_3203,
  GAL_SPECLINES_O_III_3312,
  GAL_SPECLINES_Ne_V_3345,
  GAL_SPECLINES_Ne_V_3425,
  GAL_SPECLINES_O_III_3444,
  GAL_SPECLINES_N_I_3466_4,
  GAL_SPECLINES_N_I_3466_5,
  GAL_SPECLINES_He_I_3487,
  GAL_SPECLINES_Fe_VII_3586,
  GAL_SPECLINES_Fe_VI_3662,
  GAL_SPECLINES_H_19,
  GAL_SPECLINES_H_18,
  GAL_SPECLINES_H_17,
  GAL_SPECLINES_H_16,
  GAL_SPECLINES_H_15,
  GAL_SPECLINES_H_14,
  GAL_SPECLINES_O_II_3726,
  GAL_SPECLINES_O_II_3728,
  GAL_SPECLINES_H_13,
  GAL_SPECLINES_H_12,
  GAL_SPECLINES_Fe_VII_3758,
  GAL_SPECLINES_H_11,
  GAL_SPECLINES_H_10,
  GAL_SPECLINES_H_9,
  GAL_SPECLINES_Fe_V_3839,
  GAL_SPECLINES_Ne_III_3868,
  GAL_SPECLINES_He_I_3888,
  GAL_SPECLINES_H_8,
  GAL_SPECLINES_Fe_V_3891,
  GAL_SPECLINES_Fe_V_3911,
  GAL_SPECLINES_Ne_III_3967,
  GAL_SPECLINES_H_epsilon,
  GAL_SPECLINES_He_I_4026,
  GAL_SPECLINES_S_II_4068,
  GAL_SPECLINES_Fe_V_4071,
  GAL_SPECLINES_S_II_4076,
  GAL_SPECLINES_H_delta,
  GAL_SPECLINES_He_I_4143,
  GAL_SPECLINES_Fe_II_4178,
  GAL_SPECLINES_Fe_V_4180,
  GAL_SPECLINES_Fe_II_4233,
  GAL_SPECLINES_Fe_V_4227,
  GAL_SPECLINES_Fe_II_4287,
  GAL_SPECLINES_Fe_II_4304,
  GAL_SPECLINES_O_II_4317,
  GAL_SPECLINES_H_gamma,
  GAL_SPECLINES_O_III_4363,
  GAL_SPECLINES_Ar_XIV,
  GAL_SPECLINES_O_II_4414,
  GAL_SPECLINES_Fe_II_4416,
  GAL_SPECLINES_Fe_II_4452,
  GAL_SPECLINES_He_I_4471,
  GAL_SPECLINES_Fe_II_4489,
  GAL_SPECLINES_Fe_II_4491,
  GAL_SPECLINES_N_III_4510,
  GAL_SPECLINES_Fe_II_4522,
  GAL_SPECLINES_Fe_II_4555,
  GAL_SPECLINES_Fe_II_4582,
  GAL_SPECLINES_Fe_II_4583,
  GAL_SPECLINES_Fe_II_4629,
  GAL_SPECLINES_N_III_4634,
  GAL_SPECLINES_N_III_4640,
  GAL_SPECLINES_N_III_4641,
  GAL_SPECLINES_C_III_4647,
  GAL_SPECLINES_C_III_4650,
  GAL_SPECLINES_C_III_5651,
  GAL_SPECLINES_Fe_III_4658,
  GAL_SPECLINES_He_II_4685,
  GAL_SPECLINES_Ar_IV_4711,
  GAL_SPECLINES_Ar_IV_4740,
  GAL_SPECLINES_H_beta,
  GAL_SPECLINES_Fe_VII_4893,
  GAL_SPECLINES_Fe_IV_4903,
  GAL_SPECLINES_Fe_II_4923,
  GAL_SPECLINES_O_III_4958,
  GAL_SPECLINES_O_III_5006,
  GAL_SPECLINES_Fe_II_5018,
  GAL_SPECLINES_Fe_III_5084,
  GAL_SPECLINES_Fe_VI_5145,
  GAL_SPECLINES_Fe_VII_5158,
  GAL_SPECLINES_Fe_II_5169,
  GAL_SPECLINES_Fe_VI_5176,
  GAL_SPECLINES_Fe_II_5197,
  GAL_SPECLINES_N_I_5200,
  GAL_SPECLINES_Fe_II_5234,
  GAL_SPECLINES_Fe_IV_5236,
  GAL_SPECLINES_Fe_III_5270,
  GAL_SPECLINES_Fe_II_5276,
  GAL_SPECLINES_Fe_VII_5276,
  GAL_SPECLINES_Fe_XIV,
  GAL_SPECLINES_Ca_V,
  GAL_SPECLINES_Fe_II_5316_6,
  GAL_SPECLINES_Fe_II_5316_7,
  GAL_SPECLINES_Fe_VI_5335,
  GAL_SPECLINES_Fe_VI_5424,
  GAL_SPECLINES_Cl_III_5517,
  GAL_SPECLINES_Cl_III_5537,
  GAL_SPECLINES_Fe_VI_5637,
  GAL_SPECLINES_Fe_VI_5677,
  GAL_SPECLINES_C_III_5697,
  GAL_SPECLINES_Fe_VII_5720,
  GAL_SPECLINES_N_II_5754,
  GAL_SPECLINES_C_IV_5801,
  GAL_SPECLINES_C_IV_5811,
  GAL_SPECLINES_He_I_5875,
  GAL_SPECLINES_O_I_6046,
  GAL_SPECLINES_Fe_VII_6087,
  GAL_SPECLINES_O_I_6300,
  GAL_SPECLINES_S_III_6312,
  GAL_SPECLINES_Si_II_6347,
  GAL_SPECLINES_O_I_6363,
  GAL_SPECLINES_Fe_II_6369,
  GAL_SPECLINES_Fe_X,
  GAL_SPECLINES_Fe_II_6516,
  GAL_SPECLINES_N_II_6548,
  GAL_SPECLINES_H_alpha,
  GAL_SPECLINES_N_II_6583,
  GAL_SPECLINES_S_II_6716,
  GAL_SPECLINES_S_II_6730,
  GAL_SPECLINES_O_I_7002,
  GAL_SPECLINES_Ar_V,
  GAL_SPECLINES_He_I_7065,
  GAL_SPECLINES_Ar_III_7135,
  GAL_SPECLINES_Fe_II_7155,
  GAL_SPECLINES_Ar_IV_7170,
  GAL_SPECLINES_Fe_II_7172,
  GAL_SPECLINES_C_II_7236,
  GAL_SPECLINES_Ar_IV_7237,
  GAL_SPECLINES_O_I_7254,
  GAL_SPECLINES_Ar_IV_7262,
  GAL_SPECLINES_He_I_7281,
  GAL_SPECLINES_O_II_7319,
  GAL_SPECLINES_O_II_7330,
  GAL_SPECLINES_Ni_II_7377,
  GAL_SPECLINES_Ni_II_7411,
  GAL_SPECLINES_Fe_II_7452,
  GAL_SPECLINES_N_I_7468,
  GAL_SPECLINES_S_XII,
  GAL_SPECLINES_Ar_III_7751,
  GAL_SPECLINES_He_I_7816,
  GAL_SPECLINES_Ar_I_7868,
  GAL_SPECLINES_Ni_III,
  GAL_SPECLINES_Fe_XI_7891,
  GAL_SPECLINES_He_II_8236,
  GAL_SPECLINES_Pa_20,
  GAL_SPECLINES_Pa_19,
  GAL_SPECLINES_Pa_18,
  GAL_SPECLINES_O_I_8446,
  GAL_SPECLINES_Pa_17,
  GAL_SPECLINES_Ca_II_8498,
  GAL_SPECLINES_Pa_16,
  GAL_SPECLINES_Ca_II_8542,
  GAL_SPECLINES_Pa_15,
  GAL_SPECLINES_Cl_II,
  GAL_SPECLINES_Pa_14,
  GAL_SPECLINES_Fe_II_8616,
  GAL_SPECLINES_Ca_II_8662,
  GAL_SPECLINES_Pa_13,
  GAL_SPECLINES_N_I_8680,
  GAL_SPECLINES_N_I_8703,
  GAL_SPECLINES_N_I_8711,
  GAL_SPECLINES_Pa_12,
  GAL_SPECLINES_Pa_11,
  GAL_SPECLINES_Fe_II_8891,
  GAL_SPECLINES_Pa_10,
  GAL_SPECLINES_S_III_9068,
  GAL_SPECLINES_Pa_9,
  GAL_SPECLINES_S_III_9531,
  GAL_SPECLINES_Pa_epsilon,
  GAL_SPECLINES_C_I_9824,
  GAL_SPECLINES_C_I_9850,
  GAL_SPECLINES_S_VIII,
  GAL_SPECLINES_He_I_10027,
  GAL_SPECLINES_He_I_10031,
  GAL_SPECLINES_Pa_delta,
  GAL_SPECLINES_S_II_10286,
  GAL_SPECLINES_S_II_10320,
  GAL_SPECLINES_S_II_10336,
  GAL_SPECLINES_Fe_XIII,
  GAL_SPECLINES_He_I_10830,
  GAL_SPECLINES_Pa_gamma,

  /* This should be the last element (to keep the total number of
     lines). */
  GAL_SPECLINES_NUMBER
};





enum gal_speclines_limit_codes
{
  /* Allowing '0' to be identied as a known-non-line. */
  GAL_SPECLINES_LIMIT_INVALID=0,

  GAL_SPECLINES_LIMIT_LYMAN=GAL_SPECLINES_NUMBER,
  GAL_SPECLINES_LIMIT_BALMER,
  GAL_SPECLINES_LIMIT_PASCHEN,

  /* Total number of limits and lines+limits. */
  GAL_SPECLINES_LIMIT_NUMBERWITHLINES,
  GAL_SPECLINES_LIMIT_NUMBER = (GAL_SPECLINES_LIMIT_NUMBERWITHLINES
                                - GAL_SPECLINES_LIMIT_LYMAN)
};





/* Spectral lines wavelengths in Angstroms (SORT BY WAVELENGTH).
   Source: http://astronomy.nmsu.edu/drewski/tableofemissionlines.html

   The following extra lines have been added:
   - Ly-gamma (https://en.wikipedia.org/wiki/Lyman_series) */
#define GAL_SPECLINES_ANGSTROM_Ne_VIII_770          770.409
#define GAL_SPECLINES_ANGSTROM_Ne_VIII_780          780.324
#define GAL_SPECLINES_ANGSTROM_Ly_epsilon           937.814
#define GAL_SPECLINES_ANGSTROM_Ly_delta             949.742
#define GAL_SPECLINES_ANGSTROM_Ly_gamma             972.536
#define GAL_SPECLINES_ANGSTROM_C_III_977            977.030
#define GAL_SPECLINES_ANGSTROM_N_III_989            989.790
#define GAL_SPECLINES_ANGSTROM_N_III_991_51         991.514
#define GAL_SPECLINES_ANGSTROM_N_III_991_57         991.579
#define GAL_SPECLINES_ANGSTROM_Ly_beta              1025.722
#define GAL_SPECLINES_ANGSTROM_O_VI_1031            1031.912
#define GAL_SPECLINES_ANGSTROM_O_VI_1037            1037.613
#define GAL_SPECLINES_ANGSTROM_Ar_I_1066            1066.660
#define GAL_SPECLINES_ANGSTROM_Ly_alpha             1215.670
#define GAL_SPECLINES_ANGSTROM_N_V_1238             1238.821
#define GAL_SPECLINES_ANGSTROM_N_V_1242             1242.804
#define GAL_SPECLINES_ANGSTROM_Si_II_1260           1260.422
#define GAL_SPECLINES_ANGSTROM_Si_II_1264           1264.730
#define GAL_SPECLINES_ANGSTROM_O_I_1302             1302.168
#define GAL_SPECLINES_ANGSTROM_C_II_1334            1334.532
#define GAL_SPECLINES_ANGSTROM_C_II_1335            1335.708
#define GAL_SPECLINES_ANGSTROM_Si_IV_1393           1393.755
#define GAL_SPECLINES_ANGSTROM_O_IV_1397            1397.232
#define GAL_SPECLINES_ANGSTROM_O_IV_1399            1399.780
#define GAL_SPECLINES_ANGSTROM_Si_IV_1402           1402.770
#define GAL_SPECLINES_ANGSTROM_N_IV_1486            1486.496
#define GAL_SPECLINES_ANGSTROM_C_IV_1548            1548.187
#define GAL_SPECLINES_ANGSTROM_C_IV_1550            1550.772
#define GAL_SPECLINES_ANGSTROM_He_II_1640           1640.420
#define GAL_SPECLINES_ANGSTROM_O_III_1660           1660.809
#define GAL_SPECLINES_ANGSTROM_O_III_1666           1666.150
#define GAL_SPECLINES_ANGSTROM_N_III_1746           1746.823
#define GAL_SPECLINES_ANGSTROM_N_III_1748           1748.656
#define GAL_SPECLINES_ANGSTROM_Al_III_1854          1854.716
#define GAL_SPECLINES_ANGSTROM_Al_III_1862          1862.790
#define GAL_SPECLINES_ANGSTROM_Si_III               1892.030
#define GAL_SPECLINES_ANGSTROM_C_III_1908           1908.734
#define GAL_SPECLINES_ANGSTROM_N_II_2142            2142.780
#define GAL_SPECLINES_ANGSTROM_O_III_2320           2320.951
#define GAL_SPECLINES_ANGSTROM_C_II_2323            2323.500
#define GAL_SPECLINES_ANGSTROM_C_II_2324            2324.690
#define GAL_SPECLINES_ANGSTROM_Fe_XI_2648           2648.710
#define GAL_SPECLINES_ANGSTROM_He_II_2733           2733.289
#define GAL_SPECLINES_ANGSTROM_Mg_V_2782            2782.700
#define GAL_SPECLINES_ANGSTROM_Mg_II_2795           2795.528
#define GAL_SPECLINES_ANGSTROM_Mg_II_2802           2802.705
#define GAL_SPECLINES_ANGSTROM_Fe_IV_2829           2829.360
#define GAL_SPECLINES_ANGSTROM_Fe_IV_2835           2835.740
#define GAL_SPECLINES_ANGSTROM_Ar_IV_2853           2853.670
#define GAL_SPECLINES_ANGSTROM_Ar_IV_2868           2868.210
#define GAL_SPECLINES_ANGSTROM_Mg_V_2928            2928.000
#define GAL_SPECLINES_ANGSTROM_He_I_2945            2945.106
#define GAL_SPECLINES_ANGSTROM_O_III_3132           3132.794
#define GAL_SPECLINES_ANGSTROM_He_I_3187            3187.745
#define GAL_SPECLINES_ANGSTROM_He_II_3203           3203.100
#define GAL_SPECLINES_ANGSTROM_O_III_3312           3312.329
#define GAL_SPECLINES_ANGSTROM_Ne_V_3345            3345.821
#define GAL_SPECLINES_ANGSTROM_Ne_V_3425            3425.881
#define GAL_SPECLINES_ANGSTROM_O_III_3444           3444.052
#define GAL_SPECLINES_ANGSTROM_N_I_3466_4           3466.497
#define GAL_SPECLINES_ANGSTROM_N_I_3466_5           3466.543
#define GAL_SPECLINES_ANGSTROM_He_I_3487            3487.727
#define GAL_SPECLINES_ANGSTROM_Fe_VII_3586          3586.320
#define GAL_SPECLINES_ANGSTROM_Fe_VI_3662           3662.500
#define GAL_SPECLINES_ANGSTROM_H_19                 3686.831
#define GAL_SPECLINES_ANGSTROM_H_18                 3691.551
#define GAL_SPECLINES_ANGSTROM_H_17                 3697.157
#define GAL_SPECLINES_ANGSTROM_H_16                 3703.859
#define GAL_SPECLINES_ANGSTROM_H_15                 3711.977
#define GAL_SPECLINES_ANGSTROM_H_14                 3721.945
#define GAL_SPECLINES_ANGSTROM_O_II_3726            3726.032
#define GAL_SPECLINES_ANGSTROM_O_II_3728            3728.815
#define GAL_SPECLINES_ANGSTROM_H_13                 3734.369
#define GAL_SPECLINES_ANGSTROM_H_12                 3750.158
#define GAL_SPECLINES_ANGSTROM_Fe_VII_3758          3758.920
#define GAL_SPECLINES_ANGSTROM_H_11                 3770.637
#define GAL_SPECLINES_ANGSTROM_H_10                 3797.904
#define GAL_SPECLINES_ANGSTROM_H_9                  3835.391
#define GAL_SPECLINES_ANGSTROM_Fe_V_3839            3839.270
#define GAL_SPECLINES_ANGSTROM_Ne_III_3868          3868.760
#define GAL_SPECLINES_ANGSTROM_He_I_3888            3888.647
#define GAL_SPECLINES_ANGSTROM_H_8                  3889.064
#define GAL_SPECLINES_ANGSTROM_Fe_V_3891            3891.280
#define GAL_SPECLINES_ANGSTROM_Fe_V_3911            3911.330
#define GAL_SPECLINES_ANGSTROM_Ne_III_3967          3967.470
#define GAL_SPECLINES_ANGSTROM_H_epsilon            3970.079
#define GAL_SPECLINES_ANGSTROM_He_I_4026            4026.190
#define GAL_SPECLINES_ANGSTROM_S_II_4068            4068.600
#define GAL_SPECLINES_ANGSTROM_Fe_V_4071            4071.240
#define GAL_SPECLINES_ANGSTROM_S_II_4076            4076.349
#define GAL_SPECLINES_ANGSTROM_H_delta              4101.742
#define GAL_SPECLINES_ANGSTROM_He_I_4143            4143.761
#define GAL_SPECLINES_ANGSTROM_Fe_II_4178           4178.862
#define GAL_SPECLINES_ANGSTROM_Fe_V_4180            4180.600
#define GAL_SPECLINES_ANGSTROM_Fe_II_4233           4233.172
#define GAL_SPECLINES_ANGSTROM_Fe_V_4227            4227.190
#define GAL_SPECLINES_ANGSTROM_Fe_II_4287           4287.394
#define GAL_SPECLINES_ANGSTROM_Fe_II_4304           4303.176
#define GAL_SPECLINES_ANGSTROM_O_II_4317            4317.139
#define GAL_SPECLINES_ANGSTROM_H_gamma              4340.471
#define GAL_SPECLINES_ANGSTROM_O_III_4363           4363.210
#define GAL_SPECLINES_ANGSTROM_Ar_XIV               4412.300
#define GAL_SPECLINES_ANGSTROM_O_II_4414            4414.899
#define GAL_SPECLINES_ANGSTROM_Fe_II_4416           4416.830
#define GAL_SPECLINES_ANGSTROM_Fe_II_4452           4452.098
#define GAL_SPECLINES_ANGSTROM_He_I_4471            4471.479
#define GAL_SPECLINES_ANGSTROM_Fe_II_4489           4489.183
#define GAL_SPECLINES_ANGSTROM_Fe_II_4491           4491.405
#define GAL_SPECLINES_ANGSTROM_N_III_4510           4510.910
#define GAL_SPECLINES_ANGSTROM_Fe_II_4522           4522.634
#define GAL_SPECLINES_ANGSTROM_Fe_II_4555           4555.893
#define GAL_SPECLINES_ANGSTROM_Fe_II_4582           4582.835
#define GAL_SPECLINES_ANGSTROM_Fe_II_4583           4583.837
#define GAL_SPECLINES_ANGSTROM_Fe_II_4629           4629.339
#define GAL_SPECLINES_ANGSTROM_N_III_4634           4634.140
#define GAL_SPECLINES_ANGSTROM_N_III_4640           4640.640
#define GAL_SPECLINES_ANGSTROM_N_III_4641           4641.850
#define GAL_SPECLINES_ANGSTROM_C_III_4647           4647.420
#define GAL_SPECLINES_ANGSTROM_C_III_4650           4650.250
#define GAL_SPECLINES_ANGSTROM_C_III_5651           4651.470
#define GAL_SPECLINES_ANGSTROM_Fe_III_4658          4658.050
#define GAL_SPECLINES_ANGSTROM_He_II_4685           4685.710
#define GAL_SPECLINES_ANGSTROM_Ar_IV_4711           4711.260
#define GAL_SPECLINES_ANGSTROM_Ar_IV_4740           4740.120
#define GAL_SPECLINES_ANGSTROM_H_beta               4861.333
#define GAL_SPECLINES_ANGSTROM_Fe_VII_4893          4893.370
#define GAL_SPECLINES_ANGSTROM_Fe_IV_4903           4903.070
#define GAL_SPECLINES_ANGSTROM_Fe_II_4923           4923.927
#define GAL_SPECLINES_ANGSTROM_O_III_4958           4958.911
#define GAL_SPECLINES_ANGSTROM_O_III_5006           5006.843
#define GAL_SPECLINES_ANGSTROM_Fe_II_5018           5018.440
#define GAL_SPECLINES_ANGSTROM_Fe_III_5084          5084.770
#define GAL_SPECLINES_ANGSTROM_Fe_VI_5145           5145.750
#define GAL_SPECLINES_ANGSTROM_Fe_VII_5158          5158.890
#define GAL_SPECLINES_ANGSTROM_Fe_II_5169           5169.033
#define GAL_SPECLINES_ANGSTROM_Fe_VI_5176           5176.040
#define GAL_SPECLINES_ANGSTROM_Fe_II_5197           5197.577
#define GAL_SPECLINES_ANGSTROM_N_I_5200             5200.257
#define GAL_SPECLINES_ANGSTROM_Fe_II_5234           5234.625
#define GAL_SPECLINES_ANGSTROM_Fe_IV_5236           5236.060
#define GAL_SPECLINES_ANGSTROM_Fe_III_5270          5270.400
#define GAL_SPECLINES_ANGSTROM_Fe_II_5276           5276.002
#define GAL_SPECLINES_ANGSTROM_Fe_VII_5276          5276.380
#define GAL_SPECLINES_ANGSTROM_Fe_XIV               5302.860
#define GAL_SPECLINES_ANGSTROM_Ca_V                 5309.110
#define GAL_SPECLINES_ANGSTROM_Fe_II_5316_6         5316.615
#define GAL_SPECLINES_ANGSTROM_Fe_II_5316_7         5316.784
#define GAL_SPECLINES_ANGSTROM_Fe_VI_5335           5335.180
#define GAL_SPECLINES_ANGSTROM_Fe_VI_5424           5424.220
#define GAL_SPECLINES_ANGSTROM_Cl_III_5517          5517.709
#define GAL_SPECLINES_ANGSTROM_Cl_III_5537          5537.873
#define GAL_SPECLINES_ANGSTROM_Fe_VI_5637           5637.600
#define GAL_SPECLINES_ANGSTROM_Fe_VI_5677           5677.000
#define GAL_SPECLINES_ANGSTROM_C_III_5697           5695.920
#define GAL_SPECLINES_ANGSTROM_Fe_VII_5720          5720.700
#define GAL_SPECLINES_ANGSTROM_N_II_5754            5754.590
#define GAL_SPECLINES_ANGSTROM_C_IV_5801            5801.330
#define GAL_SPECLINES_ANGSTROM_C_IV_5811            5811.980
#define GAL_SPECLINES_ANGSTROM_He_I_5875            5875.624
#define GAL_SPECLINES_ANGSTROM_O_I_6046             6046.440
#define GAL_SPECLINES_ANGSTROM_Fe_VII_6087          6087.000
#define GAL_SPECLINES_ANGSTROM_O_I_6300             6300.304
#define GAL_SPECLINES_ANGSTROM_S_III_6312           6312.060
#define GAL_SPECLINES_ANGSTROM_Si_II_6347           6347.100
#define GAL_SPECLINES_ANGSTROM_O_I_6363             6363.776
#define GAL_SPECLINES_ANGSTROM_Fe_II_6369           6369.462
#define GAL_SPECLINES_ANGSTROM_Fe_X                 6374.510
#define GAL_SPECLINES_ANGSTROM_Fe_II_6516           6516.081
#define GAL_SPECLINES_ANGSTROM_N_II_6548            6548.050
#define GAL_SPECLINES_ANGSTROM_H_alpha              6562.819
#define GAL_SPECLINES_ANGSTROM_N_II_6583            6583.460
#define GAL_SPECLINES_ANGSTROM_S_II_6716            6716.440
#define GAL_SPECLINES_ANGSTROM_S_II_6730            6730.810
#define GAL_SPECLINES_ANGSTROM_O_I_7002             7002.230
#define GAL_SPECLINES_ANGSTROM_Ar_V                 7005.870
#define GAL_SPECLINES_ANGSTROM_He_I_7065            7065.196
#define GAL_SPECLINES_ANGSTROM_Ar_III_7135          7135.790
#define GAL_SPECLINES_ANGSTROM_Fe_II_7155           7155.157
#define GAL_SPECLINES_ANGSTROM_Ar_IV_7170           7170.620
#define GAL_SPECLINES_ANGSTROM_Fe_II_7172           7172.000
#define GAL_SPECLINES_ANGSTROM_C_II_7236            7236.420
#define GAL_SPECLINES_ANGSTROM_Ar_IV_7237           7237.260
#define GAL_SPECLINES_ANGSTROM_O_I_7254             7254.448
#define GAL_SPECLINES_ANGSTROM_Ar_IV_7262           7262.760
#define GAL_SPECLINES_ANGSTROM_He_I_7281            7281.349
#define GAL_SPECLINES_ANGSTROM_O_II_7319            7319.990
#define GAL_SPECLINES_ANGSTROM_O_II_7330            7330.730
#define GAL_SPECLINES_ANGSTROM_Ni_II_7377           7377.830
#define GAL_SPECLINES_ANGSTROM_Ni_II_7411           7411.160
#define GAL_SPECLINES_ANGSTROM_Fe_II_7452           7452.538
#define GAL_SPECLINES_ANGSTROM_N_I_7468             7468.310
#define GAL_SPECLINES_ANGSTROM_S_XII                7611.000
#define GAL_SPECLINES_ANGSTROM_Ar_III_7751          7751.060
#define GAL_SPECLINES_ANGSTROM_He_I_7816            7816.136
#define GAL_SPECLINES_ANGSTROM_Ar_I_7868            7868.194
#define GAL_SPECLINES_ANGSTROM_Ni_III               7889.900
#define GAL_SPECLINES_ANGSTROM_Fe_XI_7891           7891.800
#define GAL_SPECLINES_ANGSTROM_He_II_8236           8236.790
#define GAL_SPECLINES_ANGSTROM_Pa_20                8392.397
#define GAL_SPECLINES_ANGSTROM_Pa_19                8413.318
#define GAL_SPECLINES_ANGSTROM_Pa_18                8437.956
#define GAL_SPECLINES_ANGSTROM_O_I_8446             8446.359
#define GAL_SPECLINES_ANGSTROM_Pa_17                8467.254
#define GAL_SPECLINES_ANGSTROM_Ca_II_8498           8498.020
#define GAL_SPECLINES_ANGSTROM_Pa_16                8502.483
#define GAL_SPECLINES_ANGSTROM_Ca_II_8542           8542.090
#define GAL_SPECLINES_ANGSTROM_Pa_15                8545.383
#define GAL_SPECLINES_ANGSTROM_Cl_II                8578.700
#define GAL_SPECLINES_ANGSTROM_Pa_14                8598.392
#define GAL_SPECLINES_ANGSTROM_Fe_II_8616           8616.950
#define GAL_SPECLINES_ANGSTROM_Ca_II_8662           8662.140
#define GAL_SPECLINES_ANGSTROM_Pa_13                8665.019
#define GAL_SPECLINES_ANGSTROM_N_I_8680             8680.282
#define GAL_SPECLINES_ANGSTROM_N_I_8703             8703.247
#define GAL_SPECLINES_ANGSTROM_N_I_8711             8711.703
#define GAL_SPECLINES_ANGSTROM_Pa_12                8750.472
#define GAL_SPECLINES_ANGSTROM_Pa_11                8862.782
#define GAL_SPECLINES_ANGSTROM_Fe_II_8891           8891.910
#define GAL_SPECLINES_ANGSTROM_Pa_10                9014.909
#define GAL_SPECLINES_ANGSTROM_S_III_9068           9068.600
#define GAL_SPECLINES_ANGSTROM_Pa_9                 9229.014
#define GAL_SPECLINES_ANGSTROM_S_III_9531           9531.100
#define GAL_SPECLINES_ANGSTROM_Pa_epsilon           9545.969
#define GAL_SPECLINES_ANGSTROM_C_I_9824             9824.130
#define GAL_SPECLINES_ANGSTROM_C_I_9850             9850.260
#define GAL_SPECLINES_ANGSTROM_S_VIII               9913.000
#define GAL_SPECLINES_ANGSTROM_He_I_10027           10027.730
#define GAL_SPECLINES_ANGSTROM_He_I_10031           10031.160
#define GAL_SPECLINES_ANGSTROM_Pa_delta             10049.368
#define GAL_SPECLINES_ANGSTROM_S_II_10286           10286.730
#define GAL_SPECLINES_ANGSTROM_S_II_10320           10320.490
#define GAL_SPECLINES_ANGSTROM_S_II_10336           10336.410
#define GAL_SPECLINES_ANGSTROM_Fe_XIII              10746.800
#define GAL_SPECLINES_ANGSTROM_He_I_10830           10830.340
#define GAL_SPECLINES_ANGSTROM_Pa_gamma             10938.086

/* Common limits of line series; from
   https://en.wikipedia.org/wiki/Hydrogen_spectral_series */
#define GAL_SPECLINES_ANGSTROM_LIMIT_LYMAN          912
#define GAL_SPECLINES_ANGSTROM_LIMIT_BALMER         3646
#define GAL_SPECLINES_ANGSTROM_LIMIT_PASCHEN        8204



/* Spectral line name strings (SORT BY WAVELENGTH). */
#define GAL_SPECLINES_NAME_Ne_VIII_770          "Ne-VIII-770"
#define GAL_SPECLINES_NAME_Ne_VIII_780          "Ne-VIII-780"
#define GAL_SPECLINES_NAME_Ly_epsilon           "Ly-epsilon"
#define GAL_SPECLINES_NAME_Ly_delta             "Ly-delta"
#define GAL_SPECLINES_NAME_Ly_gamma             "Ly-gamma"
#define GAL_SPECLINES_NAME_C_III_977            "C-III-977"
#define GAL_SPECLINES_NAME_N_III_989            "N-III-989"
#define GAL_SPECLINES_NAME_N_III_991_51         "N-III-991.51"
#define GAL_SPECLINES_NAME_N_III_991_57         "N-III-991.57"
#define GAL_SPECLINES_NAME_Ly_beta              "Ly-beta"
#define GAL_SPECLINES_NAME_O_VI_1031            "O-VI-1031"
#define GAL_SPECLINES_NAME_O_VI_1037            "O-VI-1037"
#define GAL_SPECLINES_NAME_Ar_I_1066            "Ar-I-1066"
#define GAL_SPECLINES_NAME_Ly_alpha             "Ly-alpha"
#define GAL_SPECLINES_NAME_N_V_1238             "N-V-1238"
#define GAL_SPECLINES_NAME_N_V_1242             "N-V-1242"
#define GAL_SPECLINES_NAME_Si_II_1260           "Si-II-1260"
#define GAL_SPECLINES_NAME_Si_II_1264           "Si_II-1302"
#define GAL_SPECLINES_NAME_O_I_1302             "O-I-1302"
#define GAL_SPECLINES_NAME_C_II_1334            "C-II-1334"
#define GAL_SPECLINES_NAME_C_II_1335            "C-II-1335"
#define GAL_SPECLINES_NAME_Si_IV_1393           "Si-IV-1393"
#define GAL_SPECLINES_NAME_O_IV_1397            "O-IV-1397"
#define GAL_SPECLINES_NAME_O_IV_1399            "O-IV-1399"
#define GAL_SPECLINES_NAME_Si_IV_1402           "Si-IV-1402"
#define GAL_SPECLINES_NAME_N_IV_1486            "N-IV-1486"
#define GAL_SPECLINES_NAME_C_IV_1548            "C-IV-1548"
#define GAL_SPECLINES_NAME_C_IV_1550            "C-IV-1550"
#define GAL_SPECLINES_NAME_He_II_1640           "He-II-1640"
#define GAL_SPECLINES_NAME_O_III_1660           "O-III-1660"
#define GAL_SPECLINES_NAME_O_III_1666           "O-III-1666"
#define GAL_SPECLINES_NAME_N_III_1746           "N-III-1746"
#define GAL_SPECLINES_NAME_N_III_1748           "N-III-1748"
#define GAL_SPECLINES_NAME_Al_III_1854          "Al-III-1854"
#define GAL_SPECLINES_NAME_Al_III_1862          "Al-III-1862"
#define GAL_SPECLINES_NAME_Si_III               "Si-III"
#define GAL_SPECLINES_NAME_C_III_1908           "C-III-1908"
#define GAL_SPECLINES_NAME_N_II_2142            "N-II-2142"
#define GAL_SPECLINES_NAME_O_III_2320           "O-III-2320"
#define GAL_SPECLINES_NAME_C_II_2323            "C-II-2323"
#define GAL_SPECLINES_NAME_C_II_2324            "C-II-2324"
#define GAL_SPECLINES_NAME_Fe_XI_2648           "Fe-XI-2648"
#define GAL_SPECLINES_NAME_He_II_2733           "He-II-2733"
#define GAL_SPECLINES_NAME_Mg_V_2782            "Mg-V-2782"
#define GAL_SPECLINES_NAME_Mg_II_2795           "Mg-II-2795"
#define GAL_SPECLINES_NAME_Mg_II_2802           "Mg-II-2802"
#define GAL_SPECLINES_NAME_Fe_IV_2829           "Fe-IV-2829"
#define GAL_SPECLINES_NAME_Fe_IV_2835           "Fe-IV-2835"
#define GAL_SPECLINES_NAME_Ar_IV_2853           "Ar-IV-2853"
#define GAL_SPECLINES_NAME_Ar_IV_2868           "Ar-IV-2868"
#define GAL_SPECLINES_NAME_Mg_V_2928            "Mg-V-2928"
#define GAL_SPECLINES_NAME_He_I_2945            "He-I-2945"
#define GAL_SPECLINES_NAME_O_III_3132           "O-III-3132"
#define GAL_SPECLINES_NAME_He_I_3187            "He-I-3187"
#define GAL_SPECLINES_NAME_He_II_3203           "He-II-3203"
#define GAL_SPECLINES_NAME_O_III_3312           "O-III-3312"
#define GAL_SPECLINES_NAME_Ne_V_3345            "Ne-V-3345"
#define GAL_SPECLINES_NAME_Ne_V_3425            "Ne-V-3425"
#define GAL_SPECLINES_NAME_O_III_3444           "O-III-3444"
#define GAL_SPECLINES_NAME_N_I_3466_4           "N-I-3466-4"
#define GAL_SPECLINES_NAME_N_I_3466_5           "N-I-3466-5"
#define GAL_SPECLINES_NAME_He_I_3487            "He-I-3487"
#define GAL_SPECLINES_NAME_Fe_VII_3586          "Fe-VII-3586"
#define GAL_SPECLINES_NAME_Fe_VI_3662           "Fe-VI-3662"
#define GAL_SPECLINES_NAME_H_19                 "H-19"
#define GAL_SPECLINES_NAME_H_18                 "H-18"
#define GAL_SPECLINES_NAME_H_17                 "H-17"
#define GAL_SPECLINES_NAME_H_16                 "H-16"
#define GAL_SPECLINES_NAME_H_15                 "H-15"
#define GAL_SPECLINES_NAME_H_14                 "H-14"
#define GAL_SPECLINES_NAME_O_II_3726            "O-II-3726"
#define GAL_SPECLINES_NAME_O_II_3728            "O-II-3728"
#define GAL_SPECLINES_NAME_H_13                 "H-13"
#define GAL_SPECLINES_NAME_H_12                 "H-12"
#define GAL_SPECLINES_NAME_Fe_VII_3758          "Fe-VII-3758"
#define GAL_SPECLINES_NAME_H_11                 "H-11"
#define GAL_SPECLINES_NAME_H_10                 "H-10"
#define GAL_SPECLINES_NAME_H_9                  "H-9"
#define GAL_SPECLINES_NAME_Fe_V_3839            "Fe-V-3839"
#define GAL_SPECLINES_NAME_Ne_III_3868          "Ne-III-3868"
#define GAL_SPECLINES_NAME_He_I_3888            "He-I-3888"
#define GAL_SPECLINES_NAME_H_8                  "H-8"
#define GAL_SPECLINES_NAME_Fe_V_3891            "Fe-V-3891"
#define GAL_SPECLINES_NAME_Fe_V_3911            "Fe-V-3911"
#define GAL_SPECLINES_NAME_Ne_III_3967          "Ne-III-3967"
#define GAL_SPECLINES_NAME_H_epsilon            "H-epsilon"
#define GAL_SPECLINES_NAME_He_I_4026            "He-I-4026"
#define GAL_SPECLINES_NAME_S_II_4068            "S-II-4068"
#define GAL_SPECLINES_NAME_Fe_V_4071            "Fe-V-4071"
#define GAL_SPECLINES_NAME_S_II_4076            "S-II-4076"
#define GAL_SPECLINES_NAME_H_delta              "H-delta"
#define GAL_SPECLINES_NAME_He_I_4143            "He-I-4143"
#define GAL_SPECLINES_NAME_Fe_II_4178           "Fe-II-4178"
#define GAL_SPECLINES_NAME_Fe_V_4180            "Fe-V-4180"
#define GAL_SPECLINES_NAME_Fe_II_4233           "Fe-II-4233"
#define GAL_SPECLINES_NAME_Fe_V_4227            "Fe-V-4227"
#define GAL_SPECLINES_NAME_Fe_II_4287           "Fe-II-4287"
#define GAL_SPECLINES_NAME_Fe_II_4304           "Fe-II-4304"
#define GAL_SPECLINES_NAME_O_II_4317            "O-II-4317"
#define GAL_SPECLINES_NAME_H_gamma              "H-gamma"
#define GAL_SPECLINES_NAME_O_III_4363           "O-III-4363"
#define GAL_SPECLINES_NAME_Ar_XIV               "Ar-XIV"
#define GAL_SPECLINES_NAME_O_II_4414            "O-II-4414"
#define GAL_SPECLINES_NAME_Fe_II_4416           "Fe-II-4416"
#define GAL_SPECLINES_NAME_Fe_II_4452           "Fe-II-4452"
#define GAL_SPECLINES_NAME_He_I_4471            "He-I-4471"
#define GAL_SPECLINES_NAME_Fe_II_4489           "Fe-II-4489"
#define GAL_SPECLINES_NAME_Fe_II_4491           "Fe-II-4491"
#define GAL_SPECLINES_NAME_N_III_4510           "N-III-4510"
#define GAL_SPECLINES_NAME_Fe_II_4522           "Fe-II-4522"
#define GAL_SPECLINES_NAME_Fe_II_4555           "Fe-II-4555"
#define GAL_SPECLINES_NAME_Fe_II_4582           "Fe-II-4582"
#define GAL_SPECLINES_NAME_Fe_II_4583           "Fe-II-4583"
#define GAL_SPECLINES_NAME_Fe_II_4629           "Fe-II-4629"
#define GAL_SPECLINES_NAME_N_III_4634           "N-III-4634"
#define GAL_SPECLINES_NAME_N_III_4640           "N-III-4640"
#define GAL_SPECLINES_NAME_N_III_4641           "N-III-4641"
#define GAL_SPECLINES_NAME_C_III_4647           "C-III-4647"
#define GAL_SPECLINES_NAME_C_III_4650           "C-III-4650"
#define GAL_SPECLINES_NAME_C_III_5651           "C-III-5651"
#define GAL_SPECLINES_NAME_Fe_III_4658          "Fe-III-4658"
#define GAL_SPECLINES_NAME_He_II_4685           "He-II-4685"
#define GAL_SPECLINES_NAME_Ar_IV_4711           "Ar-IV-4711"
#define GAL_SPECLINES_NAME_Ar_IV_4740           "Ar-IV-4740"
#define GAL_SPECLINES_NAME_H_beta               "H-beta"
#define GAL_SPECLINES_NAME_Fe_VII_4893          "Fe-VII-4893"
#define GAL_SPECLINES_NAME_Fe_IV_4903           "Fe-IV-4903"
#define GAL_SPECLINES_NAME_Fe_II_4923           "Fe-II-4923"
#define GAL_SPECLINES_NAME_O_III_4958           "O-III-4958"
#define GAL_SPECLINES_NAME_O_III_5006           "O-III-5006"
#define GAL_SPECLINES_NAME_Fe_II_5018           "Fe-II-5018"
#define GAL_SPECLINES_NAME_Fe_III_5084          "Fe-III-5084"
#define GAL_SPECLINES_NAME_Fe_VI_5145           "Fe-VI-5145"
#define GAL_SPECLINES_NAME_Fe_VII_5158          "Fe-VII-5158"
#define GAL_SPECLINES_NAME_Fe_II_5169           "Fe-II-5169"
#define GAL_SPECLINES_NAME_Fe_VI_5176           "Fe-VI-5176"
#define GAL_SPECLINES_NAME_Fe_II_5197           "Fe-II-5197"
#define GAL_SPECLINES_NAME_N_I_5200             "N-I-5200"
#define GAL_SPECLINES_NAME_Fe_II_5234           "Fe-II-5234"
#define GAL_SPECLINES_NAME_Fe_IV_5236           "Fe-IV-5236"
#define GAL_SPECLINES_NAME_Fe_III_5270          "Fe-III-5270"
#define GAL_SPECLINES_NAME_Fe_II_5276           "Fe-II-5276"
#define GAL_SPECLINES_NAME_Fe_VII_5276          "Fe-VII-5276"
#define GAL_SPECLINES_NAME_Fe_XIV               "Fe-XIV"
#define GAL_SPECLINES_NAME_Ca_V                 "Ca-V"
#define GAL_SPECLINES_NAME_Fe_II_5316_6         "Fe-II-5316-6"
#define GAL_SPECLINES_NAME_Fe_II_5316_7         "Fe-II-5316-7"
#define GAL_SPECLINES_NAME_Fe_VI_5335           "Fe-VI-5335"
#define GAL_SPECLINES_NAME_Fe_VI_5424           "Fe-VI-5424"
#define GAL_SPECLINES_NAME_Cl_III_5517          "Cl-III-5517"
#define GAL_SPECLINES_NAME_Cl_III_5537          "Cl-III-5537"
#define GAL_SPECLINES_NAME_Fe_VI_5637           "Fe-VI-5637"
#define GAL_SPECLINES_NAME_Fe_VI_5677           "Fe-VI-5677"
#define GAL_SPECLINES_NAME_C_III_5697           "C-III-5697"
#define GAL_SPECLINES_NAME_Fe_VII_5720          "Fe-VII-5720"
#define GAL_SPECLINES_NAME_N_II_5754            "N-II-5754"
#define GAL_SPECLINES_NAME_C_IV_5801            "C-IV-5801"
#define GAL_SPECLINES_NAME_C_IV_5811            "C-IV-5811"
#define GAL_SPECLINES_NAME_He_I_5875            "He-I-5875"
#define GAL_SPECLINES_NAME_O_I_6046             "O-I-6046"
#define GAL_SPECLINES_NAME_Fe_VII_6087          "Fe-VII-6087"
#define GAL_SPECLINES_NAME_O_I_6300             "O-I-6300"
#define GAL_SPECLINES_NAME_S_III_6312           "S-III-6312"
#define GAL_SPECLINES_NAME_Si_II_6347           "Si-II-6347"
#define GAL_SPECLINES_NAME_O_I_6363             "O-I-6363"
#define GAL_SPECLINES_NAME_Fe_II_6369           "Fe-II-6369"
#define GAL_SPECLINES_NAME_Fe_X                 "Fe-X"
#define GAL_SPECLINES_NAME_Fe_II_6516           "Fe-II-6516"
#define GAL_SPECLINES_NAME_N_II_6548            "N-II-6548"
#define GAL_SPECLINES_NAME_H_alpha              "H-alpha"
#define GAL_SPECLINES_NAME_N_II_6583            "N-II-6583"
#define GAL_SPECLINES_NAME_S_II_6716            "S-II-6716"
#define GAL_SPECLINES_NAME_S_II_6730            "S-II-6730"
#define GAL_SPECLINES_NAME_O_I_7002             "O-I-7002"
#define GAL_SPECLINES_NAME_Ar_V                 "Ar-V"
#define GAL_SPECLINES_NAME_He_I_7065            "He-I-7065"
#define GAL_SPECLINES_NAME_Ar_III_7135          "Ar-III-7135"
#define GAL_SPECLINES_NAME_Fe_II_7155           "Fe-II-7155"
#define GAL_SPECLINES_NAME_Ar_IV_7170           "Ar-IV-7170"
#define GAL_SPECLINES_NAME_Fe_II_7172           "Fe-II-7172"
#define GAL_SPECLINES_NAME_C_II_7236            "C-II-7236"
#define GAL_SPECLINES_NAME_Ar_IV_7237           "Ar-IV-7237"
#define GAL_SPECLINES_NAME_O_I_7254             "O-I-7254"
#define GAL_SPECLINES_NAME_Ar_IV_7262           "Ar-IV-7262"
#define GAL_SPECLINES_NAME_He_I_7281            "He-I-7281"
#define GAL_SPECLINES_NAME_O_II_7319            "O-II-7319"
#define GAL_SPECLINES_NAME_O_II_7330            "O-II-7330"
#define GAL_SPECLINES_NAME_Ni_II_7377           "Ni-II-7377"
#define GAL_SPECLINES_NAME_Ni_II_7411           "Ni-II-7411"
#define GAL_SPECLINES_NAME_Fe_II_7452           "Fe-II-7452"
#define GAL_SPECLINES_NAME_N_I_7468             "N-I-7468"
#define GAL_SPECLINES_NAME_S_XII                "S-XII"
#define GAL_SPECLINES_NAME_Ar_III_7751          "Ar-III-7751"
#define GAL_SPECLINES_NAME_He_I_7816            "He-I-7816"
#define GAL_SPECLINES_NAME_Ar_I_7868            "Ar-I-7868"
#define GAL_SPECLINES_NAME_Ni_III               "Ni-III"
#define GAL_SPECLINES_NAME_Fe_XI_7891           "Fe-XI-7891"
#define GAL_SPECLINES_NAME_He_II_8236           "He-II-8236"
#define GAL_SPECLINES_NAME_Pa_20                "Pa-20"
#define GAL_SPECLINES_NAME_Pa_19                "Pa-19"
#define GAL_SPECLINES_NAME_Pa_18                "Pa-18"
#define GAL_SPECLINES_NAME_O_I_8446             "O-I-8446"
#define GAL_SPECLINES_NAME_Pa_17                "Pa-17"
#define GAL_SPECLINES_NAME_Ca_II_8498           "Ca-II-8498"
#define GAL_SPECLINES_NAME_Pa_16                "Pa-16"
#define GAL_SPECLINES_NAME_Ca_II_8542           "Ca-II-8542"
#define GAL_SPECLINES_NAME_Pa_15                "Pa-15"
#define GAL_SPECLINES_NAME_Cl_II                "Cl-II"
#define GAL_SPECLINES_NAME_Pa_14                "Pa-14"
#define GAL_SPECLINES_NAME_Fe_II_8616           "Fe-II-8616"
#define GAL_SPECLINES_NAME_Ca_II_8662           "Ca-II-8662"
#define GAL_SPECLINES_NAME_Pa_13                "Pa-13"
#define GAL_SPECLINES_NAME_N_I_8680             "N-I-8680"
#define GAL_SPECLINES_NAME_N_I_8703             "N-I-8703"
#define GAL_SPECLINES_NAME_N_I_8711             "N-I-8711"
#define GAL_SPECLINES_NAME_Pa_12                "Pa-12"
#define GAL_SPECLINES_NAME_Pa_11                "Pa-11"
#define GAL_SPECLINES_NAME_Fe_II_8891           "Fe-II-8891"
#define GAL_SPECLINES_NAME_Pa_10                "Pa-10"
#define GAL_SPECLINES_NAME_S_III_9068           "S-III-9068"
#define GAL_SPECLINES_NAME_Pa_9                 "Pa-9"
#define GAL_SPECLINES_NAME_S_III_9531           "S-III-9531"
#define GAL_SPECLINES_NAME_Pa_epsilon           "Pa-epsilon"
#define GAL_SPECLINES_NAME_C_I_9824             "C-I-9824"
#define GAL_SPECLINES_NAME_C_I_9850             "C-I-9850"
#define GAL_SPECLINES_NAME_S_VIII               "S-VIII"
#define GAL_SPECLINES_NAME_He_I_10027           "He-I-10027"
#define GAL_SPECLINES_NAME_He_I_10031           "He-I-10031"
#define GAL_SPECLINES_NAME_Pa_delta             "Pa-delta"
#define GAL_SPECLINES_NAME_S_II_10286           "S-II-10286"
#define GAL_SPECLINES_NAME_S_II_10320           "S-II-10320"
#define GAL_SPECLINES_NAME_S_II_10336           "S-II-10336"
#define GAL_SPECLINES_NAME_Fe_XIII              "Fe-XIII"
#define GAL_SPECLINES_NAME_He_I_10830           "He-I-10830"
#define GAL_SPECLINES_NAME_Pa_gamma             "Pa-gamma"

/* The limits. */
#define GAL_SPECLINES_NAME_LIMIT_LYMAN          "Ly-limit"
#define GAL_SPECLINES_NAME_LIMIT_BALMER         "H-limit"
#define GAL_SPECLINES_NAME_LIMIT_PASCHEN        "Pa-limit"


/*********************************************************************/
/*************        Internal names and codes         ***************/
/*********************************************************************/
char *
gal_speclines_line_name(int linecode);

int
gal_speclines_line_code(char *name);

double
gal_speclines_line_angstrom(int linecode);





/*********************************************************************/
/*************             Redshifted lines            ***************/
/*********************************************************************/
double
gal_speclines_line_redshift(double obsline, double restline);

double
gal_speclines_line_redshift_code(double obsline, int linecode);


__END_C_DECLS    /* From C++ preparations */

#endif           /* __GAL_COSMOLOGY_H__ */
