/* -*- c++ -*- */
/*
 * Copyright 2002 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * seg_sync_corr.cc -- feed it some data and look for seg sync pulses
 */

#include <unistd.h>
#include <stdlib.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <getopt.h>


void
usage (const char *name)
{
  fprintf (stderr, "usage: %s [-F] <filename>\n", name);
  fprintf (stderr, "   -F file contains floats, else shorts\n");
  
  exit (1);
}

int 
compute_short_mean (short *v, unsigned n)
{
  long long	total = 0;

  for (unsigned i = 0; i < n; i++)
    total += v[i];

  return total / n;
}

float
compute_float_mean (float *v, unsigned n)
{
  double	total = 0;

  for (unsigned i = 0; i < n; i++)
    total += v[i];

  return total / n;
}

inline static void
output_bit (int bit, FILE *fout)
{
  fputc (bit, fout);
}

//
// look for 1001, 10001, 100001, 1000001, 10000001,
//
void
do_output (FILE *f, unsigned char bit[], int nentries)
{
  nentries -= 8;
  for (int i = 0; i < nentries; i++){

    if (bit[i] == 0)
      output_bit (0, f);
    else {				// 1
      if (bit[i+1] == 1)
	output_bit (0, f);
      else {				// 10
	if (bit[i+2] == 1)
	  output_bit (0, f);
	else {				// 100
	  if (bit[i+3] == 1)		// 1001
	    output_bit (1, f);
	  else {			// 1000
	    if (bit[i+4] == 1)		// 10001
	      output_bit (1, f);
	    else {			// 10000
	      if (bit[i+5] == 1)	// 100001
		output_bit (1, f);
	      else {			// 100000
		if (bit[i+6] == 1)	// 1000001
		  output_bit (1, f);
		else {			// 1000000
		  if (bit[i+7] == 1)	// 10000001
		    output_bit (1, f);
		  else
		    output_bit (0, f);
		}
	      }
	    }
	  }
	}
      }
    }
  }
}

int 
main (int argc, char **argv)
{
  int			fd;
  char			*base;
  struct stat		stat;
  off_t			file_size;
  unsigned long 	nentries; 	// # of data entries
  int			c;
  bool			float_p = false;
  FILE			*fout;
  

  while ((c = getopt (argc, argv, "F")) != EOF){
    switch (c){
    case 'F':	float_p = true; break;
    default:
      usage (argv[0]);
    }
  }
    
  if (argc - optind != 1)
    usage (argv[0]);

  char *filename = argv[optind];


  fd = open (filename, O_RDONLY);
  if (fd < 0){
    perror (filename);
    exit (1);
  }

  if (fstat (fd, &stat) < 0){
    perror ("stat");
    exit (1);
  }

  file_size = stat.st_size;	// file size in bytes

  if ((base = (char *) mmap (0, file_size, PROT_READ, MAP_SHARED, fd, 0)) == MAP_FAILED){
    perror ("mmap");
    exit (1);
  }

  fout = stdout;

  if (float_p){
    float *data = (float *) base;
    nentries = file_size / sizeof (float);
    unsigned char *bit = new unsigned char[nentries];
    
    float mean = compute_float_mean (data, nentries);

    for (unsigned long i = 0; i < nentries; i++)
      bit[i] = (data[i] - mean) > 0;

    do_output (fout, bit, nentries);
  }

  else {
    short *data = (short *) base;
    nentries = file_size / sizeof (short);
    unsigned char *bit = new unsigned char[nentries];

    int mean = compute_short_mean (data, nentries);

    for (unsigned long i = 0; i < nentries; i++)
      bit[i] = (data[i] - mean) > 0;

    do_output (fout, bit, nentries);
  }
  

  munmap (base, file_size);
  close (fd);
  exit (0);
}
