#!/usr/bin/env python
#
# Copyright 2004 Free Software Foundation, Inc.
# 
# This file is part of GNU Radio
# 
# GNU Radio is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2, or (at your option)
# any later version.
# 
# GNU Radio is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with GNU Radio; see the file COPYING.  If not, write to
# the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA.
# 

from gnuradio import gr, gr_unittest
import math

def reverse(taps):
    r = list(taps)
    r.reverse()
    return r

class qa_filter_delay_fc (gr_unittest.TestCase):

    def setUp (self):
        self.fg = gr.flow_graph ()

    def tearDown (self):
        self.fg = None

    def test_001_filter_delay_one_input (self):

        # expected result
        expected_result = (
            +2.3406542482007353e-08   +1.0000816583633423j,
            -0.5877840518951416       +0.80908381938934326j,
            -0.95105588436126709      +0.30904293060302734j,
            -0.95105588436126709      -0.30904296040534973j,
            -0.5877838134765625       -0.80908387899398804j,
            -2.6332360292258272e-08   -1.0000815391540527j,
            +0.58778399229049683      -0.80908381938934326j,
            +0.95105582475662231      -0.30904299020767212j,
            +0.95105588436126709      +0.30904293060302734j,
            +0.5877838134765625       +0.80908381938934326j,
            +3.218399768911695e-08    +1.0000815391540527j)
            
        fg = self.fg

        sampling_freq = 100

        ntaps = 51
        src1 = gr.sig_source_f (sampling_freq, gr.GR_SIN_WAVE,
                               sampling_freq * 0.10, 1.0)
        head = gr.head (gr.sizeof_float, int (ntaps + sampling_freq * 0.10))
        dst2 = gr.vector_sink_c ()

        # calculate taps
        taps = gr.firdes_hilbert (ntaps)
        hd = gr.filter_delay_fc (reverse(taps))

        fg.connect (src1, head)
        fg.connect (head, hd)
        fg.connect (hd,dst2)
        fg.run ()

        # get output
        result_data = dst2.data ()
        self.assertComplexTuplesAlmostEqual (expected_result, result_data, 6)

    def test_002_filter_delay_two_inputs (self):

        # giving the same signal to both the inputs should fetch the same results
        # as above

        # expected result
        expected_result = (
            +2.3406542482007353e-08  +1.0000816583633423j,
            -0.5877840518951416      +0.80908381938934326j,
            -0.95105588436126709     +0.30904293060302734j,
            -0.95105588436126709     -0.30904296040534973j,
            -0.5877838134765625      -0.80908387899398804j,
            -2.6332360292258272e-08  -1.0000815391540527j,
            +0.58778399229049683     -0.80908381938934326j,
            +0.95105582475662231     -0.30904299020767212j,
            +0.95105588436126709     +0.30904293060302734j,
            +0.5877838134765625      +0.80908381938934326j,
            +3.218399768911695e-08   +1.0000815391540527j)
        
        fg = self.fg

        sampling_freq = 100
        ntaps = 51
        src1 = gr.sig_source_f (sampling_freq, gr.GR_SIN_WAVE,
                               sampling_freq * 0.10, 1.0)
        head = gr.head (gr.sizeof_float, int (ntaps + sampling_freq * 0.10))
        dst2 = gr.vector_sink_c ()

        # calculate taps
        taps = gr.firdes_hilbert (ntaps)
        hd = gr.filter_delay_fc (reverse(taps))

        fg.connect (src1, head)
        fg.connect (head, (hd,0))
        fg.connect (head, (hd,1))
        fg.connect (hd,dst2)
        fg.run ()

        # get output
        result_data = dst2.data ()
        self.assertComplexTuplesAlmostEqual (expected_result, result_data, 6)


    def test_003_filter_delay_two_inputs (self):

        # give two different inputs

        # expected result
        expected_result = (
            +2.3406542482007353e-08   +1.1920928955078125e-07j,
            -0.5877840518951416       -0.58783555030822754j,
            -0.95105588436126709      -0.95113480091094971j,
            -0.95105588436126709      -0.95113474130630493j,
            -0.5877838134765625       -0.58783555030822754j,
            -2.6332360292258272e-08   -8.1956386566162109e-08j,
            +0.58778399229049683      +0.58783555030822754j,
            +0.95105582475662231      +0.95113474130630493j,
            +0.95105588436126709      +0.95113474130630493j,
            +0.5877838134765625       +0.58783560991287231j,
            +3.218399768911695e-08    +1.1920928955078125e-07j)

        fg = self.fg

        sampling_freq = 100
        ntaps = 51
        
        src1 = gr.sig_source_f (sampling_freq, gr.GR_SIN_WAVE,sampling_freq * 0.10, 1.0)
        src2 = gr.sig_source_f (sampling_freq, gr.GR_COS_WAVE,sampling_freq * 0.10, 1.0)
        
        head1 = gr.head (gr.sizeof_float, int (ntaps + sampling_freq * 0.10))
        head2 = gr.head (gr.sizeof_float, int (ntaps + sampling_freq * 0.10))
        
        taps = gr.firdes_hilbert (ntaps)
        hd = gr.filter_delay_fc (reverse(taps))

        dst2 = gr.vector_sink_c ()

        fg.connect (src1, head1)
        fg.connect (src2, head2)
        
        fg.connect (head1, (hd,0))
        fg.connect (head2, (hd,1))
        fg.connect (hd, dst2)
        
        fg.run ()

        # get output
        result_data = dst2.data ()
        self.assertComplexTuplesAlmostEqual (expected_result, result_data, 6)

        
if __name__ == '__main__':
    gr_unittest.main ()
