/*  gnutrition - a nutrition and diet analysis program.
 *  Copyright (C) 2000 Edgar Denny (e.denny@ic.ac.uk)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <gnome.h>
#include <glade/glade.h>
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <ctype.h>

#include "text_util.h"
#include "food.h"
#include "recipe.h"
#include "recipe_win.h"
#include "nutr_io.h"

static gboolean
open_file( gchar *);

static gboolean
open_file( gchar *);

/* Open the file dialog when "open" is selected from the recipe
 * window menu item. */
void
on_open1_activate( GtkMenuItem *menuitem,
                   gpointer     user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	GtkWidget *widget = glade_xml_get_widget( xml, "file_open_dlg");
	gtk_widget_show( widget);
}


/* Save a recipe when "save" is selected from the recipe 
 * window menu item. */
void
on_save2_activate( GtkMenuItem *menuitem,
                   gpointer     user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	Recipe_t *recipe = gnutr_ret_recipe();

	gchar *recipe_name = gtk_entry_get_text( GTK_ENTRY(
			glade_xml_get_widget( xml, "mw_recipe_entry")));
	gchar *no_serv = gtk_entry_get_text( GTK_ENTRY(
			glade_xml_get_widget( xml, "mw_no_serv_entry")));

	/* Check that the recipe name is defined. */
	if ( empty_text( recipe_name))
	{
		GtkWidget *message = 
				glade_xml_get_widget( xml, "rnnd_msgbox");
		gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
		gtk_widget_show( message);
		return;
	}

	/* Check that the number of servings is defined. */
	if ( !all_digits( no_serv))
	{
		GtkWidget *message = glade_xml_get_widget( xml, "nserv_msgbox");
		gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
		gtk_widget_show( message);
		return;
	}
	
	/* If there is a current working recipe->file_name, ask if it is be
	 * over written, else open the file dialog. */
	if ( recipe->file_name != NULL)
	{
		if ( g_file_exists( recipe->file_name))
		{
			/* warn that the file exists. Over writing existing file? */
			GtkWidget *message = glade_xml_get_widget( xml, "fe_msgbox2");
			gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
	 		gtk_widget_show( message);
		}
	}
	else
	{
		GtkWidget *widget = glade_xml_get_widget( xml, "file_save_dlg");
		gtk_widget_show( widget);
	}
}


/* Open the file dialog when "save as" is selected from the recipe
 * window menu bar. */
void
on_save_as1_activate( GtkMenuItem *menuitem,
                      gpointer     user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	GtkWidget *widget = glade_xml_get_widget( xml, "file_save_dlg");
	gtk_widget_show( widget);
}


/* When the "New" button is released on the recipe window AppBar,
 * ask if current recipe is to be saved. */
void
on_mw_new_button_released( GtkButton *button,
                           gpointer   user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	Recipe_t *recipe = gnutr_ret_recipe();
	if ( recipe->no_ingredients !=0)
	{
		GtkWidget *save_cur_rec_dlg = glade_xml_get_widget( xml, "screc_dlg");
		gnome_dialog_close_hides( GNOME_DIALOG( save_cur_rec_dlg), TRUE);
		gtk_widget_show( save_cur_rec_dlg);
	}
	else
	{
		gnutr_free_recipe();
	}
}


/* When the "Open" button is released on the AppBar of the recipe window. */
void
on_mw_open_button_released( GtkButton *button,
                            gpointer   user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	GtkWidget *widget = glade_xml_get_widget( xml, "file_open_dlg");
	gtk_widget_show( widget);
}


/* When the "Save" button is released on the AppBar of the recipe window. */
void
on_mw_save_button_released( GtkButton *button,
                            gpointer   user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	Recipe_t *recipe = gnutr_ret_recipe();
	gchar *recipe_name = gtk_entry_get_text( GTK_ENTRY(
			glade_xml_get_widget( xml, "mw_recipe_entry")));
	gchar *no_serv = gtk_entry_get_text( GTK_ENTRY(
			glade_xml_get_widget( xml, "mw_no_serv_entry")));

	/* Check that the recipe name is defined. */
	if ( empty_text( recipe_name))
	{
		GtkWidget *message = 
				glade_xml_get_widget( xml, "rnnd_msgbox");
		gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
		gtk_widget_show( message);
		return;
	}

	/* Check that the number of servings is defined. */
	if ( !all_digits( no_serv))
	{
		GtkWidget *message = glade_xml_get_widget( xml, "nserv_msgbox");
		gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
		gtk_widget_show( message);
		return;
	}

	/* If there is a current working recipe->file_name, ask if it is be
	 * over written, else open the file dialog. */
	if ( recipe->file_name != NULL)
	{
		if ( g_file_exists( recipe->file_name))
		{
			/* warn that the file exists. Over writing existing file? */
			GtkWidget *message = glade_xml_get_widget( xml, "fe_msgbox2");
			gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
	 		gtk_widget_show( message);
		}
	}
	else
	{
		GtkWidget *widget = glade_xml_get_widget( xml, "file_save_dlg");
		gtk_widget_show( widget);
	}
}

/* read recipe from file. */
static gboolean
open_file( gchar *fn)
{
	GladeXML *xml = gnutr_ret_xml();
	Recipe_t *recipe = gnutr_ret_recipe();
	GList *food_glist = gnutr_ret_food_glist();
	GtkWidget *name_entry = glade_xml_get_widget( xml, "mw_recipe_entry");
	GtkWidget *no_serv_entry =  glade_xml_get_widget( xml, "mw_no_serv_entry");

	gtk_clist_clear( recipe->ingredient_clist); 

	g_return_val_if_fail( fn !=NULL, FALSE);

	gnutr_read_recipe( fn, &recipe, food_glist);

	/* store name of the read file. */
	recipe->file_name = g_strdup( fn);

	gtk_entry_set_text( GTK_ENTRY( name_entry), recipe->name);
	gtk_entry_set_text( GTK_ENTRY( no_serv_entry), recipe->no_serv);

	return TRUE;
}

/* save recipe to file. */
static gboolean
save_file( gchar *fn)
{
	GladeXML *xml = gnutr_ret_xml();
	Recipe_t *recipe = gnutr_ret_recipe();
	GtkWidget *name_entry = glade_xml_get_widget( xml, "mw_recipe_entry");
	GtkWidget *no_serv_entry =  glade_xml_get_widget( xml, "mw_no_serv_entry");

	recipe->name = gtk_entry_get_text( GTK_ENTRY( name_entry));
	recipe->no_serv = gtk_entry_get_text( GTK_ENTRY( no_serv_entry));

	g_return_val_if_fail( fn !=NULL, FALSE);

	gnutr_write_recipe( fn, recipe);

	return TRUE;
}

/* Called when a button is clicked on the file exists message box1. 
 * The file name given in the file dialog is identical to one
 * that exists in the directory the recipe is to be saved to. */
void
on_fe_msgbox1_clicked( GnomeDialog *gnomedialog,
                       gint         button,
                       gpointer     user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	Recipe_t *recipe = gnutr_ret_recipe();
	switch( button)
	{
		/* cancel button pressed. */
		case 0:
		break;

		/* ok button pressed. */
		case 1:
		{
			GtkWidget *fdlg = glade_xml_get_widget( xml, "file_save_dlg");
			gchar *fn = gtk_file_selection_get_filename( 
						GTK_FILE_SELECTION( fdlg));

			if ( recipe->file_name != NULL) g_free( recipe->file_name);
			recipe->file_name = g_strdup( fn);
			save_file( fn);
			gtk_widget_hide( fdlg);
		}
		break;
	}
	gtk_widget_hide( GTK_WIDGET( gnomedialog));
}

/* Called when a button is clicked on the file exists message box2. 
 * The file name that the recipe is to be saved under is identical
 * to the current working recipe->file_name. */
void
on_fe_msgbox2_clicked( GnomeDialog *gnomedialog,
                       gint         button,
                       gpointer     user_data)
{
	Recipe_t *recipe = gnutr_ret_recipe();
	/* FIXME: I could merge this with on_fe_msgbox1_clicked() if
	 * I can find of way checking that "file_save_dlg" is not open. */
	switch( button)
	{
		/* cancel button pressed. */
		case 0:
		break;

		/* ok button pressed. */
		case 1:
		{
			/* Just an extra check to make sure. */
			if ( recipe->file_name != NULL)
			{
				save_file( recipe->file_name);
			} 
		}
		break;
	}
	gtk_widget_hide( GTK_WIDGET( gnomedialog));
}

/* The "OK" button is released on the file 
 * dialog - when a file is being opened. */
void
on_file_open_dlg_ok_button_released( GtkButton *button,
                                     gpointer   user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	GtkWidget *fdlg = glade_xml_get_widget( xml, "file_open_dlg");
	gchar *fn = gtk_file_selection_get_filename( GTK_FILE_SELECTION( fdlg));
	gchar *home = g_strconcat( g_get_home_dir(), "/", NULL);

	/* check that a file name has been given. */
	if ( strcmp( fn, "") == 0 || strcmp( fn, home) == 0)
	{
		GtkWidget *message = glade_xml_get_widget( xml, "nfg_msgbox");
		gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
	 	gtk_widget_show( message);
	}
	else if( !g_file_exists( fn))
	{
		/* open file not found message box. */
		GtkWidget *message = glade_xml_get_widget( xml, "fnf_msgbox");
		gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
		gtk_widget_show( message);
	}
	else
	{
		open_file( fn);
		gtk_widget_hide( fdlg);
	}
}


/* The "Cancel" button is released on the file
 * dialog - when a file is being opened. */
void
on_file_open_dlg_cancel_button_released( GtkButton *button,
                                         gpointer   user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	GtkWidget *widget = glade_xml_get_widget( xml, "file_open_dlg");
	gtk_widget_hide( widget);
}


/* The "OK" button is released on the file dialog - when a recipe is
 * being saved to file. */
void
on_file_save_dlg_ok_button_released( GtkButton *button,
                                     gpointer   user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	Recipe_t *recipe = gnutr_ret_recipe();
	GtkWidget *fdlg = glade_xml_get_widget( xml, "file_save_dlg");
	gchar *fn = gtk_file_selection_get_filename( GTK_FILE_SELECTION( fdlg));
	gchar *home = g_strconcat( g_get_home_dir(), "/", NULL);

	/* check that a file name has been given. */
	if ( strcmp( fn, "") == 0 || strcmp( fn, home) == 0)
	{
		GtkWidget *message = glade_xml_get_widget( xml, "nfg_msgbox");
		gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
	 	gtk_widget_show( message);
	}
	else if( g_file_exists( fn))
	{
		/* show the file exists message box. */
		GtkWidget *message = glade_xml_get_widget( xml, "fe_msgbox1");
		gnome_dialog_close_hides( GNOME_DIALOG( message), TRUE);
	 	gtk_widget_show( message);
	}
	else
	{
		if ( recipe->file_name != NULL) g_free( recipe->file_name);
		recipe->file_name = g_strdup( fn);
		save_file( fn);
		gtk_widget_hide( fdlg);
	}
}


/* The "Cancel" button is released - when a recipe is being saved
 * to file. */
void
on_file_save_dlg_cancel_button_released( GtkButton *button,
                                         gpointer   user_data)
{
	GladeXML *xml = gnutr_ret_xml();
	GtkWidget *widget = glade_xml_get_widget( xml, "file_save_dlg");
	gtk_widget_hide( widget);
}

/* on "save current recipe?" dialog clicked. */
void
on_screc_dlg_clicked( GnomeDialog *save_cur_rec_dlg,
                      gint         button,
                      gpointer     user_data)
{	
	GladeXML *xml = gnutr_ret_xml();
	Recipe_t *recipe = gnutr_ret_recipe();
	switch( button)
	{
		/* no button pressed. */
		case 0:
			gnutr_free_recipe();
		break;

		/* yes button pressed. */
		case 1:
			if ( recipe->file_name == NULL)
			{
				GtkWidget *fdlg = glade_xml_get_widget( xml, "file_save_dlg");
				gtk_widget_show( fdlg);
				/* FIXME: How do I call gnutr_free_recipe() after
				 * the file_save_dlg is closed? At present when I
				 * save the recipe it is not cleared. */
			}
			else
			{
				save_file( recipe->file_name);
				gnutr_free_recipe();
			}
		break;
	}
	gtk_widget_hide( GTK_WIDGET( save_cur_rec_dlg));
}
