/*
 This file is part of GNU Taler
 (C) 2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  TransactionMajorState,
  TransactionMinorState,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig } from "../harness/denomStructures.js";
import {
  createSimpleTestkudosEnvironmentV3,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

const coinCommon = {
  cipher: "RSA" as const,
  durationLegal: "3 years",
  durationSpend: "2 years",
  durationWithdraw: "7 days",
  feeDeposit: "TESTKUDOS:0",
  feeRefresh: "TESTKUDOS:0",
  feeRefund: "TESTKUDOS:0",
  feeWithdraw: "TESTKUDOS:0",
  rsaKeySize: 1024,
};

const coinConfigList: CoinConfig[] = [
  {
    ...coinCommon,
    name: "n1",
    value: "TESTKUDOS:1",
  },
];

/**
 * Test deposit with a large number of coins.
 *
 * In particular, this checks that the wallet properly
 * splits deposits into batches with <=64 coins per batch.
 *
 * Since we use an artificially large number of coins, this
 * test is a bit slower than other tests.
 */
export async function runDepositLargeTest(t: GlobalTestState) {
  // Set up test environment
  const { walletClient, bankClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(t, coinConfigList);

  // Withdraw digital cash into the wallet.
  const withdrawRes = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:200",
  });

  await withdrawRes.withdrawalFinishedCond;

  const depositResp = await walletClient.call(
    WalletApiOperation.CreateDepositGroup,
    {
      amount: "TESTKUDOS:100",
      depositPaytoUri: withdrawRes.accountPaytoUri,
    },
  );

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositResp.transactionId,
    txState: {
      major: TransactionMajorState.Finalizing,
      minor: TransactionMinorState.Track,
    },
  });
}

runDepositLargeTest.suites = ["wallet", "slow"];
